#############################################################################
# geossav.tcl
#
# This program script creates GEOSSAV,
# developed at the University of Basel, Switzerland,
# by Christian Regli and Lukas Rosenthaler
#
#############################################################################


package require Iwidgets
package require Tktable


global exedir
set exedir [file dirname $argv0]

global project
set project(name) {}

global version
set version "1.0"

global node
global borehole
global groundtype
global nodeedit
global menustatus

global sisimpar          ;# used for geossav_sisim_main (sequential indicatio simulation)
global gamvpar           ;# used for geossav_gamv_main (irregularly spaced data variogram computation)
global bicalibpar        ;# used for geossav_bicalib_main (markov-bayes calibration)
global asmpar            ;# used for geossav_export_asm_pmwin_main (data export for aquifer simulation model 
                         ;# and processing modflow)
global mofpar            ;# used for geossav_export_modflow_main (data export for modflow supported programs)
global vargpltpar        ;# used for geossav_varplt_main (variogram plotting)


set scriptdir [file dirname [info script]]
cd $scriptdir

global exedir
source $exedir/Geossav_sisim.tcl
source $exedir/Geossav_gamv.tcl
source $exedir/Geossav_bicalib.tcl
source $exedir/Geossav_export_asm_pmwin.tcl
source $exedir/Geossav_export_modflow.tcl
source $exedir/Geossav_output_visualization.tcl
source $exedir/Geossav_input_visualization.tcl
source $exedir/Geossav_vargplt.tcl

#############################################################################


set w .
wm title $w "GEOSSAV [format "%s" $version]"
wm iconname $w "menu"
wm protocol $w WM_DELETE_WINDOW {
    exit
}
wm iconbitmap . -default geossav.ico

#############################################################################


iwidgets::panedwindow .pw \
    -width 10.9i \
    -height 7i \
    -orient horizontal \
    -sashcursor arrow

pack .pw


.pw add top
set pane [.pw childsite top]

global visframe
set visframe [frame $pane.l -background white -borderwidth 2]


pack $pane.l -expand yes -fill both


.pw add "bottom"
set pane [.pw childsite "bottom"]

set logwin_sb [scrollbar $pane.scroll \
		   -orient vertical \
		   -command "$pane.l yview"]
set logwin [text $pane.l \
		-background black \
		-foreground white \
		-relief sunken \
		-borderwidth 2 \
		-yscrollcommand "$logwin_sb set"\
		-setgrid 1]

pack $logwin_sb -side right -fill both
pack $logwin -expand yes -fill both

.pw fraction 80 20

#############################################################################

set menustatus "    "
frame .statusBar
label .statusBar.label \
    -textvariable menustatus \
    -relief sunken \
    -bd 1 \
    -font "Helvetica 10" \
    -anchor w

pack .statusBar.label \
    -side left \
    -padx 2 \
    -expand yes \
    -fill both

pack .statusBar \
    -side bottom \
    -fill x \
    -pady 2

#############################################################################
proc log { str } {
    global logwin

    $logwin insert end "$str\n"
    $logwin see end
}
#============================================================================


#############################################################################
proc log_nonewline { str } {
    global logwin

    $logwin insert end "$str"
    $logwin see end
}
#============================================================================


#############################################################################
proc log_newline { } {
    global logwin

    $logwin insert end "\n"
    $logwin see end
}
#============================================================================


iwidgets::timefield .tf -command fix_time

#############################################################################
proc fix_time { } {
    if {![winfo exists .tf]} return
    .tf show
    after 1000 fix_time
}
#============================================================================

fix_time

pack .tf \
    -side right \
    -pady 0 \
    -in .statusBar

#############################################################################


iwidgets::datefield .df -command returnCmd

#############################################################################
proc returnCmd { } {
    puts [.df get]
}
#============================================================================

pack .df \
    -side right \
    -pady 0 \
    -in .statusBar

#############################################################################

menu .menu -tearoff 0


#############################################################################
#############################################################################


set m .menu.file
menu $m -tearoff 0
.menu add cascade \
    -label "File" \
    -menu $m \
    -underline 0
$m add command \
    -label "New" \
    -foreground gray50 \
    -command {}
$m add command \
    -label "Open" \
    -foreground gray50 \
    -command {}
$m add command \
    -label "Close" \
    -foreground gray50 \
    -command {}
$m add separator
$m add command \
    -label "Save" \
    -foreground gray50 \
    -command {}
$m add command \
    -label "Save as ..." \
    -foreground gray50 \
    -command {}
$m add separator
$m add command \
    -label "Print" \
    -foreground gray50 \
    -command {}
$m add separator
$m add command \
    -label "Exit" \
    -command {
	set answer [tk_messageBox \
			-title "Exit the program" \
			-icon question \
			-message "Save before quitting?" \
			-type yesnocancel]
	switch $answer {
	    yes {}             ; # gleicher Befehl wie bei "Save" einfgen
	    no {exit}
	    cancel {}
	}
	return
}

#############################################################################

set groundtypedata_filetypes {
    {{Ground type Data} {.gtd}}
    {{All} *}
}
set fielddata_filetypes {
    {{Field Data} {.hgd}}
    {{All} *}
}

#############################################################################


#############################################################################

set m .menu.data
.menu add cascade \
    -label "Data" \
    -menu $m \
    -underline 0
menu $m -tearoff 0
$m add command \
    -label "Change coordinate system" \
    -foreground gray50 \
    -command {}
$m add command \
    -label "Markov-Bayes Calibration" \
    -command {run_bicalib}

#############################################################################
proc run_bicalib { } {
    global bicalibpar

    geossav_bicalib_main
}
#============================================================================


#############################################################################

set m .menu.variograms
.menu add cascade \
    -label "Variogram" \
    -menu $m \
    -underline 0
menu $m -tearoff 0
$m add command \
    -label "Irregularly spaced data variogram computation" \
    -command {run_gamv}

#############################################################################
proc run_gamv { } {
    global gamvpar

    geossav_gamv_main
}
#============================================================================

$m add command \
    -label "Variogram plotting" \
    -command {run_vargplt}

#############################################################################
proc run_vargplt { } {
    global vargpltpar

    geossav_vargplt_main
}
#============================================================================


#############################################################################

set m .menu.simulation
.menu add cascade \
    -label "Simulation" \
    -menu $m \
    -underline 0
menu $m -tearoff 0
$m add command \
    -label "Sequential Indicator Simulation SIS" \
    -command {run_sisim}

#############################################################################
proc run_sisim { } {
    global sisimpar

    geossav_sisim_main
}
#============================================================================

$m add command \
    -label "SIS Locally varying mean" \
    -foreground gray50 \
    -command {}
$m add command \
    -label "SIS Gridded secondary variable" \
    -foreground gray50 \
    -command {}

#############################################################################

set m .menu.plot
.menu add cascade \
    -label "Visualization" \
    -menu $m \
    -underline 0
menu $m -tearoff 0
$m add command \
    -label "Input data" \
    -command {run_input_visualization}

#############################################################################
proc run_input_visualization { } {
    global inpar

    geossav_input_visualization_main
}
#============================================================================

$m add command \
    -label "Output data" \
    -command {run_output_visualization}

#############################################################################
proc run_output_visualization { } {
    global outpar

    geossav_output_visualization_main
}
#============================================================================


#############################################################################

set m .menu.export
.menu add cascade \
    -label "Export" \
    -menu $m \
    -underline 0
menu $m -tearoff 0
$m add command \
    -label "2D matrix  ( ASM / PMWIN )" \
    -command {run_export_asm}

#############################################################################
proc run_export_asm { } {
    global asmpar

    geossav_export_asm_main
}
#============================================================================

$m add command \
    -label "BCF package  ( MODFLOW )" \
    -command {run_export_modflow}

#############################################################################
proc run_export_modflow { } {
    global mofpar

    geossav_export_modflow_main
}
#============================================================================


#############################################################################

set m .menu.help
.menu add cascade \
    -label "Help" \
    -menu $m \
    -underline 0
menu $m -tearoff 0
$m add command \
    -label "Contents" \
    -foreground gray50 \
    -command {}
$m add command \
    -label "Search for Help" \
    -foreground gray50 \
    -command {}
$m add separator
$m add command \
    -label "About GEOSSAV" \
    -command {tk_messageBox \
		  -title "About GEOSSAV" \
		  -icon info \
		  -message "GEOSSAV Version $version \nGeostatistical Environment fOr Subsurface Simulation And Visualization\n\nCopyright (c) 2000-2002 Ch. Regli, L. Rosenthaler, P. Huggenberger \nAll Rights Reserved.\n________________________________________________________ \n\nGEOSSAV is a tool used for integration of hard and soft data in \nstochastic simulation and visualization of subsurface heterogeneities. \nAs an interface to selected geostatistical programs, it can be used \nfor data analysis, variogram computation, and sequential indicator \nsimulation. Simulated data can be visualized by volume rendering and \ndynamic slicing, and can be exported for use in finite-difference \ngroundwater models. \n\nhttp://www.unibas.ch/earth/pract/software" \
		  -type ok
	return
}

#############################################################################
#############################################################################

. configure -menu .menu

bind Menu <<MenuSelect>> {
    if {[catch {%W entrycget active -label} label]} {
	set label "    "
    }
    set menustatus $label
    update idletasks
}




#############################################################################
#> read_fielddata fname param_name
#>---------------------------------------------------------------------------
#> STRING fname      : [<-] Name of field data file to open
#> STRING param_name : [<-] Name of Tcl array variable which is to be filled
#>                          with the field data data. 
#>---------------------------------------------------------------------------
#> This routine reads a field data file and stores the header and the values
#> in the given Tcl array (name given by param_name). The array has the
#>                          following members:
#>                          <TABLE BORDER="1">
#>                          <TR><TD><B>Index</B></TD><TD><B>Description</B><TD></TR>
#>                          <TR><TD>labels</TD><TD>textual description of the parameter meanings<TD></TR>
#>                          <TR><TD>fname</TD><TD>Name of parameter file<TD></TR>
#>                          <TR><TD>nvalues</TD><TD>number of parameter lines<TD></TR>
#>                          <TR><TD>n,values</TD><TD>parameter value of n-th line<TD></TR>
#>                          </TABLE>
#############################################################################


#############################################################################
proc read_fielddata { fname param_name {status_label {}} } {
    upvar $param_name params

    global node
    global borehole
    global menustatus

    if { [catch { set inchan [open $fname r] } err_msg] } {
	tk_messageBox \
	    -title "Error" \
	    -icon error \
	    -message "Couldn't open $fname!\n$errmsg"
	return
    }
    #
    # read first line with field parameter labels
    #
    gets $inchan line
    set params(node_labels) [split $line {,}]

    #
    # the data has the following order:
    #
    # 0.  data source ["Georadar", "Borehole"]
    # 1.  borehole number [1..n]
    # 2.  node number [1..m]
    # 3.  X-coord
    # 4.  Y-coord
    # 5.  Z-coord
    # 6.  Depth
    # 7.  Zone number
    # 8.  Radar pattern
    # 9.  Surface node
    # 10. Groundtype probability
    # 11. Groundtype probability
    # ... Groundtype probability

    set node(min,X) 1.0e38
    set node(min,Y) 1.0e38
    set node(min,Z) 1.0e38
    set node(max,X) -1.0e38
    set node(max,Y) -1.0e38
    set node(max,Z) -1.0e38
    set lineno 0
    while { ![eof $inchan] } {
	gets $inchan line
	if { $line == {} } {
	    continue
	}
	set tmplist [split $line {,}]

	#
	# processing of parameter list
	#
	set node_number                  [lindex $tmplist 2]; # get the node number
	lappend node(list) $node_number; # append node number to list of existing node numbers

	set node($node_number,source)    [lindex $tmplist 0]; # get data source
	set node($node_number,bh_number) [lindex $tmplist 1]; # get borehole number
	lappend borehole($node($node_number,bh_number)) $node_number; # append node number to borehole
	lappend borehole(list) $node($node_number,bh_number); # append borehole number to borehole list
	set node($node_number,X)         [lindex $tmplist 3]; # get coordinates of node
	set node($node_number,Y)         [lindex $tmplist 4]
	set node($node_number,Z)         [lindex $tmplist 5]
	set node($node_number,depth)     [lindex $tmplist 6]; # get depth of node
	set node($node_number,zone)      [lindex $tmplist 7]; # get zone (or polygon)
	set node($node_number,radar)     [lindex $tmplist 8]; # get radar pattern
	set node($node_number,surf_node) [lindex $tmplist 9]; # get surface node status

	if { [expr $node($node_number,X) > $node(max,X)] } { set node(max,X) $node($node_number,X) }
	if { [expr $node($node_number,Y) > $node(max,Y)] } { set node(max,Y) $node($node_number,Y) }
	if { [expr $node($node_number,Z) > $node(max,Z)] } { set node(max,Z) $node($node_number,Z) }
	if { [expr $node($node_number,X) < $node(min,X)] } { set node(min,X) $node($node_number,X) }
	if { [expr $node($node_number,Y) < $node(min,Y)] } { set node(min,Y) $node($node_number,Y) }
	if { [expr $node($node_number,Z) < $node(min,Z)] } { set node(min,Z) $node($node_number,Z) }

	#
	# now we get the probabilities of the ground types
	#
	set ground_type_no 0
	for { set i 10 } { $i < [llength $tmplist] } { incr i } {
	    set groundtype_name [lindex $params(node_labels) $i]
	    set node($node_number,$groundtype_name) [lindex $tmplist $i]
	}

	if { [expr ($lineno % 100) == 0] } {
	    set menustatus "Reading line $lineno..."
	    update
	}

	incr lineno
    }
    if { $status_label != {} } {
	set menustatus "    "
	update
    }
    
    set params(nvalues) $lineno
    set params(fname) $fname

    close $inchan

    return
}
#============================================================================


#############################################################################
proc read_groundtypedata { fname param_name } {
    upvar $param_name params
    global groundtype
    global params

    if { [catch { set inchan [open $fname r] } err_msg] } {
	tk_messageBox \
	    -title "Error" \
	    -icon error \
	    -message "Couldn't open $fname!\n$errmsg"
	return
    }
    #
    # read first line with node parameter labels
    #
    gets $inchan line
    set params(groundtype_labels) [split $line {,}]

    set lineno 0
    while { ![eof $inchan] } {
	gets $inchan line
	if { $line == {} } {
	    continue
	}
	set tmplist [split $line {,}]
	set groundtype_name [lindex $tmplist 0]
	lappend groundtype(list) $groundtype_name

	for { set i 1 } { $i < [llength $tmplist] } { incr i } {
	    set value_name [lindex $params(groundtype_labels) $i]
	    set groundtype($groundtype_name,$value_name) [lindex $tmplist $i]
	}
	incr lineno
    }
    close $inchan
    return
}
#============================================================================

