##START#OF#BICALIB###########################################################
# geossav_bicalib.tcl
#
# Source script creating the bicalib tabnotebook in GEOSSAV,
# developed at the University of Basel, Switzerland,
# by Christian Regli and Lukas Rosenthaler
#
#############################################################################


#############################################################################
#
# The following set of parameters are required for the BICALIB
#
source Geossav_bicalib_globals.tcl

set bicalibpar(datafl) "*.dat"               ;# secondary data file, simplified Geo-EAS format
set bicalibpar(icolsec) 4                    ;# column number for the secondary variable
set bicalibpar(calibfl) "*.dat"              ;# pairs of primary and secondary data values,
                                             ;# simplified Geo-EAS format
set bicalibpar(ivru) 1                       ;# column number for the primary variable
set bicalibpar(ivrv) 2                       ;# column number for the secondary variable
set bicalibpar(iwt) 0                        ;# column number for the declustering weight
set bicalibpar(tmin) -1e+21                  ;# minimum trimming limit, all values strictly less 
                                             ;# than tmin are ignored
set bicalibpar(tmax) +1e+21                  ;# maximum trimming limit, all values strictly greater 
                                             ;# or equal than tmax are ignored
set bicalibpar(outfl) "*.out"                ;# output data file containing the prior distributions 
                                             ;# corresponding to all input secondary data
set bicalibpar(repfl) "*.txt"                ;# output data file containing the secondary data thresholds,
                                             ;# the calibration table, and the B(z) values for the 
                                             ;# Markov-Bayes model 
set bicalibpar(calfl) "*.txt"                ;# output data file containing a report of the pairs retained, 
                                             ;# some summary statistics, a bivariate distribution table, 
                                             ;# the prior distribution, and the B(z) parameter values
set bicalibpar(ncutu) 1                      ;# number of thresholds applied to the primary variable
set bicalibpar(cutu) 1                       ;# threshold values applied to the primary variable
set bicalibpar(ncutv) 1                      ;# number of thresholds applied to the secondary variable
set bicalibpar(cutv) 1                       ;# threshold values applied to the secondary variable
#
# The following set of parameters are required for GEOSSAV BICALIB, especially for the feeding of the 
# BICALIB parameters tabnotebook after running BICALIB (BICALIB parameters tabnotebook disappear after 
# start BICALIB) or after cancel and restart the BICALIB parameters tabnotebook without an exit of Geossav
#
set bicalibpar(variables) 4                  ;# number of variables (None, x, y, z, var1, var2, ...)
set bicalibpar(scatter_variables) 4          ;# number of scatterplot variables (None, var1, var2, ...)
set bicalibpar(cancel_counter_data) 0        ;# counter for the exit from the BICALIB parameters tabnotebook
                                             ;# if data were fed
set bicalibpar(cancel_counter_scatterplot) 0 ;# counter for the exit from the BICALIB parameters tabnotebook
                                             ;# if scatterplot data were fed
set bicalibpar(title) 0                      ;# flag for writing parameter title
#
# End of initialization of variables. This has to be done once at the start 
# of the programm.
#
#############################################################################


#############################################################################
#
# Start of proc section.
#

#############################################################################
proc bicalib_read_data { fname } {
    global bicalibpar
    global dim_name_data
    
    lappend dim_name_data "None"
    if { [file readable $fname] } {
	set inf [open $fname "r"]
	gets $inf
	gets $inf n
	for { set i 0 } { $i < $n } { incr i } {
	    gets $inf dum
	    lappend dim_name_data $dum
	    set tmp_parameters [expr $i + 1]
	}
	if { $bicalibpar(cancel_counter_data) == 0 } {
	    set bicalibpar(variables) $tmp_parameters
	} elseif { $bicalibpar(cancel_counter_data) != 0 } {
	    set bicalibpar(variables) $bicalibpar(variables)
	}
	close $inf
	return 1
    } else {
	return 0
    }
}
#============================================================================


#############################################################################
proc bicalib_read_scatterplot { fname } {
    global bicalibpar
    global dim_name_scatterplot
    
    lappend dim_name_scatterplot "None"
    if { [file readable $fname] } {
	set inf [open $fname "r"]
	gets $inf
	gets $inf n
	for { set i 0 } { $i < $n } { incr i } {
	    gets $inf dum
	    lappend dim_name_scatterplot $dum
	    set tmp_parameters [expr $i + 1]
	}
	if { $bicalibpar(cancel_counter_scatterplot) == 0 } {
	    set bicalibpar(scatter_variables) $tmp_parameters
	} elseif { $bicalibpar(cancel_counter_scatterplot) != 0 } {
	    set bicalibpar(scatter_variables) $bicalibpar(scatter_variables)
	}
	close $inf
	return 1
    } else {
	return 0
    }
}
#============================================================================


#############################################################################
proc regexpvalidate { char } {
    return [regexp {![ ]} $char]
}
#============================================================================


#############################################################################
proc check_number_t { entry newstr } {
    return [regexp {^[+-]?[0-9]*[\.]?[0-9]*([e][+-]?[0-9]*)?$} $newstr]
}
#============================================================================


#############################################################################
proc plus_primary_variables { } {
    global bicalibpar
    global cutu_table_w
    global cutu_vals
    global logwin

    if { $bicalibpar(datafl) == {*.dat} || $bicalibpar(calibfl) == {*.dat} } {
	tk_messageBox \
	    -title "Thresholds on primary" \
	    -icon warning \
	    -message "You should select a file first." \
	    -type ok
    }
    if { $bicalibpar(datafl) != {*.dat} && $bicalibpar(calibfl) != {*.dat} } {
	if { $bicalibpar(ncutu) < 20 } {

	    set bicalibpar(ncutu) [expr $bicalibpar(ncutu) + 1]
	    set ii $bicalibpar(ncutu) 

	    ### add an additional row in the cutu (primary variable) table
	    $cutu_table_w configure -rows [expr $ii + 1]
	    
	    set cutu_vals($ii,label) [iwidgets::labeledwidget $cutu_table_w.label$ii \
					   -labeltext "$ii" \
					   -background gray50 \
					   -foreground white]
	    update
	    
	    set bicalibpar(cutu,$ii) $ii
	    set cutu_vals($ii,entry) [iwidgets::entryfield $cutu_table_w.entry$ii \
					  -textvariable bicalibpar(cutu,$ii) \
					  -validate {check_number_thresholds %W %P %c} \
					  -width 29]
	    update
	    
	    $cutu_table_w window configure $ii,0 -window $cutu_vals($ii,label) -relief flat -background gray50
	    $cutu_table_w window configure $ii,1 -window $cutu_vals($ii,entry)

	} elseif { $bicalibpar(ncutu) >= 20 } {
	    tk_messageBox \
		-title "Thresholds on primary" \
		-icon warning \
		-message "The maximum number of cutoffs is 20." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc minus_primary_variables { } {
    global bicalibpar
    global cutu_table_w
    global cutu_vals
    global logwin
    
    if { $bicalibpar(datafl) == {*.dat} || $bicalibpar(calibfl) == {*.dat} } {
	tk_messageBox \
	    -title "Thresholds on primary" \
	    -icon warning \
	    -message "You should select a file first." \
	    -type ok
    }
    if { $bicalibpar(datafl) != {*.dat} && $bicalibpar(calibfl) != {*.dat} } {
	if { $bicalibpar(ncutu) > 2 } {

	    set ii $bicalibpar(ncutu)

	    ### destroy the last row in the cutu (primary variable) table
	    destroy $cutu_vals($ii,label)
	    destroy $cutu_vals($ii,entry)
	    $cutu_table_w configure -rows $ii

	    set bicalibpar(ncutu) [expr $bicalibpar(ncutu) - 1]

	} elseif { $bicalibpar(ncutu) <= 2 } {
	    tk_messageBox \
		-title "Thresholds on primary" \
		-icon warning \
		-message "At least 2 cutoffs are needed." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc plus_secondary_variables { } {
    global bicalibpar
    global cutv_table_w
    global cutv_vals
    global logwin
    
    if { $bicalibpar(datafl) == {*.dat} || $bicalibpar(calibfl) == {*.dat} } {
	tk_messageBox \
	    -title "Thresholds on secondary" \
	    -icon warning \
	    -message "You should select a file first." \
	    -type ok
    }
    if { $bicalibpar(datafl) != {*.dat} && $bicalibpar(calibfl) != {*.dat} } {
	if { $bicalibpar(ncutv) < 20 } {

	    set bicalibpar(ncutv) [expr $bicalibpar(ncutv) + 1]
	    set ii $bicalibpar(ncutv) 

	    ### add an additional row in the cutv (secondary variable) table
	    $cutv_table_w configure -rows [expr $ii + 1]
	    
	    set cutv_vals($ii,label) [iwidgets::labeledwidget $cutv_table_w.label$ii \
					   -labeltext "$ii" \
					   -background gray50 \
					   -foreground white]
	    update
	    
	    set bicalibpar(cutv,$ii) $ii
	    set cutv_vals($ii,entry) [iwidgets::entryfield $cutv_table_w.entry$ii \
					  -textvariable bicalibpar(cutv,$ii) \
					  -validate {check_number_thresholds %W %P %c} \
					  -width 29]
	    update
	    
	    $cutv_table_w window configure $ii,0 -window $cutv_vals($ii,label) -relief flat -background gray50
	    $cutv_table_w window configure $ii,1 -window $cutv_vals($ii,entry)

	} elseif { $bicalibpar(ncutv) >= 20 } {
	    tk_messageBox \
		-title "Thresholds on secondary" \
		-icon warning \
		-message "The maximum number of cutoffs is 20." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc minus_secondary_variables { } {
    global bicalibpar
    global cutv_table_w
    global cutv_vals
    global logwin

    if { $bicalibpar(datafl) == {*.dat} || $bicalibpar(calibfl) == {*.dat} } {
	tk_messageBox \
	    -title "Thresholds on secondary" \
	    -icon warning \
	    -message "You should select a file first." \
	    -type ok
    }
    if { $bicalibpar(datafl) != {*.dat} && $bicalibpar(calibfl) != {*.dat} } {
	if { $bicalibpar(ncutv) > 2 } {

	    set ii $bicalibpar(ncutv)

	    ### destroy the last row in the cutv (secondary variable) table
	    destroy $cutv_vals($ii,label)
	    destroy $cutv_vals($ii,entry)
	    $cutv_table_w configure -rows $ii

	    set bicalibpar(ncutv) [expr $bicalibpar(ncutv) - 1]

	} elseif { $bicalibpar(ncutv) <= 2 } {
	    tk_messageBox \
		-title "Thresholds on secondary" \
		-icon warning \
		-message "At least 2 variable are needed." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc check_number_thresholds { entry newstr char } {
    set current [$entry get]
    if {$current == {} && [string match {[0-9.]} $char]} {
	return [regexp {[0-9.]} $char]
    } elseif {$current != {} && [string match {[0-9.]} $char]} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    }
}
#============================================================================


#############################################################################
proc read_bicalib_params { fname } {
    global bicalibpar
    global dim_name_data
    global dim_name_scatterplot
    global icolsec_w
    global ivru_w
    global ivrv_w
    global iwt_w
    global cutu_vals
    global cutv_vals
    global cutu_table_w
    global cutv_table_w
    global bicalib_box_w
    global datafl_w
    global calibfl_w
    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }
    
    ### sets the cursor on watch as long as parameters are read
    . configure -cursor watch
    update
    
    ### searches the line "parameters for bicalib" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Parameters for BICALIB" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File selection error" \
		-icon error \
		-message "You have selected a wrong parameter file.\n"
	    return 0
	}
    }

    ### set file name in bicalib box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $bicalib_box_w "BICALIB parameters   -   $fname_short"

    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }

    ### destroys the content of the cutu- and the cutv_table 
    if { $bicalibpar(datafl) != {*.dat} && $bicalibpar(calibfl) != {*.dat} } {
	for { set ii 1 } { $ii <= $bicalibpar(ncutu) } { incr ii } {
	    destroy $cutu_vals($ii,label)
	    destroy $cutu_vals($ii,entry)
	}
	for { set ii 1 } { $ii <= $bicalibpar(ncutv) } { incr ii } {
	    destroy $cutv_vals($ii,label)
	    destroy $cutv_vals($ii,entry)
	}
    } elseif { $bicalibpar(datafl) == {*.dat} || $bicalibpar(calibfl) == {*.dat} } {
	for { set ii 1 } { $ii <= 2 } { incr ii } {
	    destroy $cutu_vals($ii,label)
	    destroy $cutu_vals($ii,entry)
	    destroy $cutv_vals($ii,label)
	    destroy $cutv_vals($ii,entry)
	}
    }    
    
    ### reads name of datafl: file with secondary data
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set bicalibpar(datafl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set bicalibpar(datafl) [format "%s" $line]
    }

    if { $bicalibpar(datafl) == {*.dat} } {
	$datafl_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $bicalibpar(datafl) != {*.dat} } {
	$datafl_w configure -state normal -textbackground white -foreground black
    }

    ### reads the icolsec: column for secondary variable
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_bicalibpar(icolsec) [format "%g" [string range $line 0 $space]]

    if { $bicalibpar(datafl) != {*.dat} } {
	unset dim_name_data
	if { [bicalib_read_data $bicalibpar(datafl)] != 1 } {
	    set dim_name_data {}
	    lappend dim_name_data "None"
	    for { set i 1 } { $i < 4 } { incr i } {
		lappend dim_name_data $i
	    }
	}
	
	$icolsec_w clear list
	foreach i $dim_name_data {
	    $icolsec_w insert list end $i
	}
	$icolsec_w selection set $tmp_bicalibpar(icolsec)
    }

    ### reads name of calibfl: file with calibration scatterplot
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set bicalibpar(calibfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set bicalibpar(calibfl) $line
    }

    if { $bicalibpar(calibfl) == {*.dat} } {
	$calibfl_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $bicalibpar(calibfl) != {*.dat} } {
	$calibfl_w configure -state normal -textbackground white -foreground black
    }

    ### reads the ivru, ivrv, iwt: columns for primary and secondary variable, and weight
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_bicalibpar(ivru) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_bicalibpar(ivrv) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_bicalibpar(iwt) [format "%d" [string range $line 0 [expr $space - 1]]]
    
    if { $bicalibpar(calibfl) != {*.dat} } {
	unset dim_name_scatterplot
	if { [bicalib_read_scatterplot $bicalibpar(calibfl)] != 1 } {
	    set dim_name_scatterplot {}
	    lappend dim_name_scatterplot "None"
	    for { set i 1 } { $i < 4 } { incr i } {
		lappend dim_name_scatterplot $i
	    }
	}
	
	$ivru_w clear list
	foreach i $dim_name_scatterplot {
	    $ivru_w insert list end $i
	}
	$ivru_w selection set $tmp_bicalibpar(ivru)
	
	$ivrv_w clear list
	foreach i $dim_name_scatterplot {
	    $ivrv_w insert list end $i
	}
	$ivrv_w selection set $tmp_bicalibpar(ivrv)
	
	$iwt_w clear list
	foreach i $dim_name_scatterplot {
	    $iwt_w insert list end $i
	}
	$iwt_w selection set $tmp_bicalibpar(iwt)
    }

    ### reads the tmin, tmax: trimming limits
    set line [gets $inf]
    set space [string first " " $line]
    set bicalibpar(tmin) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set bicalibpar(tmax) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the outfl: file for output data / distribution
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set bicalibpar(outfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set bicalibpar(outfl) $line
    }

    ### reads the calfl: file for output calibration (sisim)
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set bicalibpar(calfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set bicalibpar(calfl) $line
    }

    ### reads the repfl: file for calibration report
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set bicalibpar(repfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set bicalibpar(repfl) $line
    }

    ### reads the ncutu: number of thresholds on primary
    set line [gets $inf]
    set space [string first " " $line]
    set bicalibpar(ncutu) [format "%d" [string range $line 0 $space]]

    ### reads the cutu(i) and fills the cutu (primary variable) table: thresholds on primary
    set line [gets $inf]
    $cutu_table_w configure -rows [expr $bicalibpar(ncutu) + 1]
    
    for { set ii 1 } { $ii <= $bicalibpar(ncutu) } { incr ii } {
	
	set cutu_vals($ii,label) [iwidgets::labeledwidget $cutu_table_w.label$ii \
				      -labeltext "$ii" \
				      -background gray50 \
				      -foreground white]
	update
	
	set space [string first " " $line]
	set bicalibpar(cutu,$ii) [format "%g" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set cutu_vals($ii,entry) [iwidgets::entryfield $cutu_table_w.entry$ii \
				      -textvariable bicalibpar(cutu,$ii) \
				      -validate {check_number_thresholds %W %P %c} \
				      -width 28]
	update
	
	$cutu_table_w window configure $ii,0 -window $cutu_vals($ii,label) -relief flat -background gray50
	$cutu_table_w window configure $ii,1 -window $cutu_vals($ii,entry)
    }

    ### reads the ncutu: number of thresholds on secondary
    set line [gets $inf]
    set space [string first " " $line]
    set bicalibpar(ncutv) [format "%d" [string range $line 0 $space]]

    ### reads the cutv(i) and fills the cutv (secondary variable) table: thresholds on secondary
    set line [gets $inf]
    $cutv_table_w configure -rows [expr $bicalibpar(ncutv) + 1]
    
    for { set ii 1 } { $ii <= $bicalibpar(ncutv) } { incr ii } {
	
	set cutv_vals($ii,label) [iwidgets::labeledwidget $cutv_table_w.label$ii \
				      -labeltext "$ii" \
				      -background gray50 \
				      -foreground white]
	update
	
	set space [string first " " $line]
	set bicalibpar(cutv,$ii) [format "%g" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set cutv_vals($ii,entry) [iwidgets::entryfield $cutv_table_w.entry$ii \
				      -textvariable bicalibpar(cutv,$ii) \
				      -validate {check_number_thresholds %W %P %c} \
				      -width 28]
	update
	
	$cutv_table_w window configure $ii,0 -window $cutv_vals($ii,label) -relief flat -background gray50
	$cutv_table_w window configure $ii,1 -window $cutv_vals($ii,entry)
    }

    ### reset the cancel counters to zero
    set bicalibpar(cancel_counter_data) 0
    set bicalibpar(cancel_counter_scatterplot) 0

    ### reset the cursor on arrow 
    . configure -cursor arrow
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc write_bicalib_params { fname {tmpflag 0} } {
    global tcl_platform
    global version
    global bicalibpar
    global icolsec_w
    global ivru_w
    global ivrv_w
    global iwt_w
    global bicalib_box_w

    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }

    ### sets the cursor on watch as long as parameters are written
    . configure -cursor watch
    update
    
    ### set file name in bicalib box widget
    if { $bicalibpar(title) == 0 } {
	set param_dname [file dirname $fname]
	set dname_length [string length $param_dname]
	set fname_short [string range $fname [expr $dname_length + 1] end]
	wm title $bicalib_box_w "BICALIB parameters   -   $fname_short"
    }

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    
    puts $outf "Parameters for BICALIB"
    puts $outf "**********************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### datafl: file with secondary data
    if { $tmpflag } {
	catch {
	    file copy -force $bicalibpar(datafl) "datafl_in.dat"
	}
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "datafl_in.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $bicalibpar(datafl)]
    }
    set line [format "%s" $bicalibpar(datafl)]
    set line_length_total [string length $line]
    set comment "\\secondary data"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### icolsec: column for secondary variable
    puts -nonewline $outf [format "%d" [$icolsec_w curselection]]
    set line [format "%d" [$icolsec_w curselection]]
    set line_length [string length $line]
    set comment "\\column for secondary variable"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### calibfl: file with calibration scatterplot
    if { $tmpflag } {
	catch {
	    file copy -force $bicalibpar(calibfl) "calibfl_in.dat"
	}
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "calibfl_in.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $bicalibpar(calibfl)]
    }
    set line [format "%s" $bicalibpar(calibfl)]
    set line_length_total [string length $line]
    set comment "\\calibration scatterplot"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### ivru, ivrv, iwt: columns for primary and secondary variable, and weight
    puts -nonewline $outf [format "%d %d " [$ivru_w curselection] [$ivrv_w curselection]]
    puts -nonewline $outf [format "%d" [$iwt_w curselection]]
    set line [format "%d %d %d" [$ivru_w curselection] [$ivrv_w curselection] \
		  [$iwt_w curselection]]
    set line_length [string length $line]
    set comment "\\columns for primary and secondary variable, weight"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### tmin, tmax: trimming limits
    catch {scan $bicalibpar(tmin) {%g} bicalibpar(tmin)}
    catch {scan $bicalibpar(tmax) {%g} bicalibpar(tmax)}
    puts -nonewline $outf [format "%g %g" $bicalibpar(tmin) $bicalibpar(tmax)]
    set line [format "%g %g" $bicalibpar(tmin) $bicalibpar(tmax)]
    set line_length [string length $line]
    set comment "\\trimming limits"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### outfl: file for output data / distribution
    if { $tmpflag } {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "outfl_out.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $bicalibpar(outfl)]
    }
    set line [format "%s" $bicalibpar(outfl)]
    set line_length_total [string length $line]
    set comment "\\output data / distribution"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### calfl: file for output calibration (sisim)
    if { $tmpflag } {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "calfl_out.txt"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $bicalibpar(calfl)]
    }
    set line [format "%s" $bicalibpar(calfl)]
    set line_length_total [string length $line]
    set comment "\\output calibration (sisim)"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### repfl: file for calibration report
    if { $tmpflag } {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "repfl_out.txt"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $bicalibpar(repfl)]
    }
    set line [format "%s" $bicalibpar(repfl)]
    set line_length_total [string length $line]
    set comment "\\calibration report"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### ncutu: number of thresholds on primary
    set comment "\\number of thresholds on primary"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $bicalibpar(ncutu) $comment]

    ### cutu(i), i=1, ncutu: thresholds on primary
    set line_length_total 0
    for { set ii 1 } { $ii <= $bicalibpar(ncutu) } { incr ii } {
	puts -nonewline $outf "[format "%g" $bicalibpar(cutu,$ii)] "
	set line [format "%g" $bicalibpar(cutu,$ii)]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    set comment "\\thresholds on primary"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### ncutv: number of thresholds on secondary
    set comment "\\number of thresholds on secondary"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $bicalibpar(ncutv) $comment]

    ### cutv(i), i=1, ncutv: thresholds on secondary
    set line_length_total 0
    for { set ii 1 } { $ii <= $bicalibpar(ncutv) } { incr ii } {
	puts -nonewline $outf "[format "%g" $bicalibpar(cutv,$ii)] "
	set line [format "%g" $bicalibpar(cutv,$ii)]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    set comment "\\thresholds on secondary"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### reset the cursor on arrow 
    . configure -cursor arrow

    ### close the file
    close $outf
}
#============================================================================


#############################################################################
proc write_bicalib_params_run { fname {tmpflag 0} } {
    global tcl_platform
    global version
    global bicalibpar
    global icolsec_w
    global ivru_w
    global ivrv_w
    global iwt_w
    global bicalib_box_w

    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }

    ### sets the cursor on watch as long as parameters are written
    . configure -cursor watch
    update
    
    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    
    puts $outf "Parameters for BICALIB"
    puts $outf "**********************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### datafl: file with secondary data
    if { $tmpflag } {
	catch {
	    file copy -force $bicalibpar(datafl) "datafl_in.dat"
	}
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "datafl_in.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $bicalibpar(datafl)]
    }
    set line [format "%s" $bicalibpar(datafl)]
    set line_length_total [string length $line]
    set comment "\\secondary data"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### icolsec: column for secondary variable
    puts -nonewline $outf [format "%d" [$icolsec_w curselection]]
    set line [format "%d" [$icolsec_w curselection]]
    set line_length [string length $line]
    set comment "\\column for secondary variable"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### calibfl: file with calibration scatterplot
    if { $tmpflag } {
	catch {
	    file copy -force $bicalibpar(calibfl) "calibfl_in.dat"
	}
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "calibfl_in.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $bicalibpar(calibfl)]
    }
    set line [format "%s" $bicalibpar(calibfl)]
    set line_length_total [string length $line]
    set comment "\\calibration scatterplot"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### ivru, ivrv, iwt: columns for primary and secondary variable, and weight
    puts -nonewline $outf [format "%d %d " [$ivru_w curselection] [$ivrv_w curselection]]
    puts -nonewline $outf [format "%d" [$iwt_w curselection]]
    set line [format "%d %d %d" [$ivru_w curselection] [$ivrv_w curselection] \
		  [$iwt_w curselection]]
    set line_length [string length $line]
    set comment "\\columns for primary and secondary variable, weight"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### tmin, tmax: trimming limits
    catch {scan $bicalibpar(tmin) {%g} bicalibpar(tmin)}
    catch {scan $bicalibpar(tmax) {%g} bicalibpar(tmax)}
    puts -nonewline $outf [format "%g %g" $bicalibpar(tmin) $bicalibpar(tmax)]
    set line [format "%g %g" $bicalibpar(tmin) $bicalibpar(tmax)]
    set line_length [string length $line]
    set comment "\\trimming limits"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### outfl: file for output data / distribution
    if { $tmpflag } {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "outfl_out.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $bicalibpar(outfl)]
    }
    set line [format "%s" $bicalibpar(outfl)]
    set line_length_total [string length $line]
    set comment "\\output data / distribution"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### calfl: file for output calibration (sisim)
    if { $tmpflag } {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "calfl_out.txt"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $bicalibpar(calfl)]
    }
    set line [format "%s" $bicalibpar(calfl)]
    set line_length_total [string length $line]
    set comment "\\output calibration (sisim)"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### repfl: file for calibration report
    if { $tmpflag } {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "repfl_out.txt"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $bicalibpar(repfl)]
    }
    set line [format "%s" $bicalibpar(repfl)]
    set line_length_total [string length $line]
    set comment "\\calibration report"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### ncutu: number of thresholds on primary
    set comment "\\number of thresholds on primary"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $bicalibpar(ncutu) $comment]

    ### cutu(i), i=1, ncutu: thresholds on primary
    set line_length_total 0
    for { set ii 1 } { $ii <= $bicalibpar(ncutu) } { incr ii } {
	puts -nonewline $outf "[format "%g" $bicalibpar(cutu,$ii)] "
	set line [format "%g" $bicalibpar(cutu,$ii)]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    set comment "\\thresholds on primary"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### ncutv: number of thresholds on secondary
    set comment "\\number of thresholds on secondary"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $bicalibpar(ncutv) $comment]

    ### cutv(i), i=1, ncutv: thresholds on secondary
    set line_length_total 0
    for { set ii 1 } { $ii <= $bicalibpar(ncutv) } { incr ii } {
	puts -nonewline $outf "[format "%g" $bicalibpar(cutv,$ii)] "
	set line [format "%g" $bicalibpar(cutv,$ii)]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    set comment "\\thresholds on secondary"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### reset the cursor on arrow 
    . configure -cursor arrow

    ### close the file
    close $outf
}
#============================================================================

#
# End of proc section.
#
#############################################################################


#############################################################################
proc geossav_bicalib_main { } {
    
    source Geossav_bicalib_globals.tcl
    
    . configure -cursor watch
    update
    
    set bicalib_box_w [toplevel .bicalib_box -class Dialog]
    wm resizable $bicalib_box_w false false
    wm sizefrom $bicalib_box_w program
    wm group $bicalib_box_w .
    wm transient $bicalib_box_w [winfo toplevel [winfo parent $bicalib_box_w]]
    wm title $bicalib_box_w "BICALIB parameters"
    
    set geostr [wm geometry .]
    regexp {([0-9]*)x([0-9]*)(\+)([0-9]*)(\+)([0-9]*)} $geostr all width height p1 pos_x p2 pos_y
    set pos_x [expr $pos_x + $width / 2 + 212]
    set pos_y [expr $pos_y + $height / 2 + 15]
    wm geometry $bicalib_box_w "+$pos_x+$pos_y"
    
    
    #############################################################################
    # Tabnotebook
    #############################################################################
    set bicalib_tabnotebook_w [iwidgets::tabnotebook $bicalib_box_w.bicalib_tbn \
				   -width 5.4i \
				   -height 5.3i \
				   -tabpos n \
				   -tabforeground gray50 \
				   -background gray85 \
				   -bevelamount 2]
    
    pack $bicalib_tabnotebook_w -padx 0 -pady 0
    
    
    #
    # page1: Geossav BICALIB-parameters: Files (there is only one page)
    #
    set page [$bicalib_box_w.bicalib_tbn add \
		  -label "Files"]
    
    
    # initialize dimension names for data
    if { ![bicalib_read_data $bicalibpar(datafl)] } {
	set dim_name_data {}
	lappend dim_name_data "None"
	for { set i 1} { $i <= 4 } { incr i } {
	    lappend dim_name_data $i
	}
    }

    # initialize dimension names for scatter-data
    if { ![bicalib_read_scatterplot $bicalibpar(calibfl)] } {
	set dim_name_scatterplot {}
	lappend dim_name_scatterplot "None"
	for { set i 1} { $i <= 4 } { incr i } {
	    lappend dim_name_scatterplot $i
	}
    }
    
    
    set work_frame_w [frame $page.workframe \
			  -borderwidth 2 \
			  -relief flat]
    
    # fields for data
    set data_iframe_w [iwidgets::labeledframe $work_frame_w.data \
			   -labeltext "Secondary data" \
			   -labelpos nw \
			   -borderwidth 2 \
			   -relief groove]
    set data_frame_w [$data_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $data_frame_w.subfr1]
    set datafl_w [iwidgets::entryfield $subframe1_w.datafl \
		      -labeltext "File " \
		      -width 67 \
		      -state disabled \
		      -textbackground gray85 \
		      -foreground gray50 \
		      -textvariable bicalibpar(datafl)]

    if { $bicalibpar(datafl) != {*.dat} } {
	$datafl_w configure -state normal -textbackground white -foreground black
    }
    
    set datafl_browse_w [iwidgets::pushbutton $subframe1_w.datafl_browse \
			     -height 27 \
			     -width 40 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			     -command {
				 set tmp_fname [tk_getOpenFile \
						    -defaultextension ".dat" \
						    -filetypes {{{Geo-EAS Files} {.dat}} {{All Files} {.*}}} \
						    -initialfile $bicalibpar(datafl) \
						    -initialdir $exedir \
						    -title "Select BICALIB secondary data"]
				 
				 ### exit from the BICALIB parameters tabnotebook
				 set bicalibpar(cancel_counter_data) 0
				 
				 ### select file name and data
				 if { $tmp_fname != {} } {
				     set bicalibpar(datafl) $tmp_fname
				     unset dim_name_data
				     if { [bicalib_read_data $bicalibpar(datafl)] != 1 } {
					 set dim_name_data {}
					 lappend dim_name_data "None"
				     }
				     
				     $icolsec_w clear list
				     foreach i $dim_name_data {
					 $icolsec_w insert list end $i
				     }
				     if { $bicalibpar(variables) >= 4 } {
					 $icolsec_w selection set 4
					 set bicalibpar(icolsec) [lindex $dim_name_data 4]
				     } elseif { $bicalibpar(variables) < 4 } {
					 $icolsec_w selection set 1
					 set bicalibpar(icolsec) [lindex $dim_name_data 1]
				     }
				     
				     ### set datafl widget normal
				     $datafl_w configure -state normal -textbackground white -foreground black
				     
				     ### output path- and filenames
				     if { $bicalibpar(datafl) != {*.dat} } {
					 set datafl_rootname [file rootname $bicalibpar(datafl)]
					 set datafl_dirname [file dirname $bicalibpar(datafl)]
					 set datafl_dirname_lenght [string length $datafl_dirname]
					 set datafl_name \
					     [string range $datafl_rootname [expr $datafl_dirname_lenght + 1] end]
					 
					 ### output data / distribution path- and filename
					 set outfl_extension [file extension $bicalibpar(outfl)]
					 set simulation "/BICALIB_"
					 regsub {} $outfl_extension $datafl_name outputfl_1
					 regsub {} $outputfl_1 $simulation outputfl_2
					 regsub {} $outputfl_2 $datafl_dirname outputfl_3
					 set bicalibpar(outfl) $outputfl_3
					 
					 ### output calibration (sisim) path- and filename
					 set outfl_c_extension [file extension $bicalibpar(calfl)]
					 set calibration "/BICALIB_c_"
					 regsub {} $outfl_c_extension $datafl_name outputfl_c_1
					 regsub {} $outputfl_c_1 $calibration outputfl_c_2
					 regsub {} $outputfl_c_2 $datafl_dirname outputfl_c_3
					 set bicalibpar(calfl) $outputfl_c_3
					 
					 ### calibration report path- and filename
					 set outfl_r_extension [file extension $bicalibpar(repfl)]
					 set report "/BICALIB_r_"
					 regsub {} $outfl_r_extension $datafl_name outputfl_r_1
					 regsub {} $outputfl_r_1 $report outputfl_r_2
					 regsub {} $outputfl_r_2 $datafl_dirname outputfl_r_3
					 set bicalibpar(repfl) $outputfl_r_3
				     }
				 }
			     }]
    
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $datafl_w        -row 0 -column 0 -sticky we -pady 0
    grid $datafl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    
    set subframe2_w [frame $data_frame_w.subfr2]
    
    if { $bicalibpar(datafl) != {*.dat} } {
	set bicalibpar(cancel_counter_data) [expr $bicalibpar(cancel_counter_data) + 1]
    }
    
    set icolsec_w [iwidgets::combobox $subframe2_w.icolsec \
		       -labeltext "Variable " \
		       -textvariable bicalibpar(icolsec) \
		       -width 13 \
		       -listheight 78 \
		       -validate {regexpvalidate %c}]
    if { $bicalibpar(datafl) == {*.dat} } {
	foreach i $dim_name_data {
	    $icolsec_w insert list end $i
	}
	if { $bicalibpar(variables) >= 4 } {
	    $icolsec_w selection set 4
	    set bicalibpar(icolsec) [lindex $dim_name_data 4]
	} elseif { $bicalibpar(variables) < 4 } {
	    $icolsec_w selection set 1
	    set bicalibpar(icolsec) [lindex $dim_name_data 1]
	}
    } elseif { $bicalibpar(datafl) != {*.dat} } {
	set tmp_icolsec $bicalibpar(icolsec)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if bicalibpar(icolsec) is empty
	foreach i $dim_name_data {
	    $icolsec_w insert list end $i
	    $icolsec_w delete list [expr $bicalibpar(variables) + 1]
	    if { $i == $tmp_icolsec } { ;# ok, that's the value in bicalibpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($bicalibpar(cancel_counter_data) * ($bicalibpar(variables) + 1))]
	$icolsec_w selection set $sel
	set bicalibpar(icolsec) [lindex $dim_name_data $sel]
    }


    grid columnconfigure $subframe2_w 0 -minsize 154
    grid $icolsec_w -row 0 -column 0 -sticky e -pady 0
    
    
    pack \
	$subframe1_w \
	$subframe2_w \
	-anchor w -expand 1 -padx 3 -pady 1
    
    
    # fields for calibration data
    set calibration_iframe_w [iwidgets::labeledframe $work_frame_w.calibration \
				  -labeltext "Calibration scatterplot" \
				  -labelpos nw \
				  -borderwidth 2 \
				  -relief groove]
    set calibration_frame_w [$calibration_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $calibration_frame_w.subfr1]
    set calibfl_w [iwidgets::entryfield $subframe1_w.calibfl \
		       -labeltext "File " \
		       -state disabled \
		       -textbackground gray85 \
		       -foreground gray50 \
		       -width 67 \
		       -textvariable bicalibpar(calibfl)]

    if { $bicalibpar(calibfl) != {*.dat} } {
	$calibfl_w configure -state normal -textbackground white -foreground black
    }
    
    set calibfl_browse_w [iwidgets::pushbutton $subframe1_w.calibfl_browse \
			      -height 27 \
			      -width 40 \
			      -defaultringpad 0 \
			      -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			      -command {
				  set tmp_fname [tk_getOpenFile \
						     -defaultextension ".dat" \
						     -filetypes {{{Geo-EAS Files} {.dat}} {{All Files} {.*}}} \
						     -initialfile $bicalibpar(calibfl) \
						     -initialdir $exedir \
						     -title "Select BICALIB calibration data"]
				  
				  ### exit from the BICALIB parameters tabnotebook
				  set bicalibpar(cancel_counter_scatterplot) 0
				  
				  ### select file name and scatter-data
				  if { $tmp_fname != {} } {
				      set bicalibpar(calibfl) $tmp_fname
				      unset dim_name_scatterplot
				      if { [bicalib_read_scatterplot $bicalibpar(calibfl)] != 1 } {
					  set dim_name_scatterplot {}
					  lappend dim_name_scatterplot "None"
				      }
				      
				      $ivru_w clear list
				      foreach i $dim_name_scatterplot {
					  $ivru_w insert list end $i
				      }
				      if { $bicalibpar(scatter_variables) >= 6 } {
					  $ivru_w selection set 4
					  set bicalibpar(ivru) [lindex $dim_name_scatterplot 4]
				      } elseif { $bicalibpar(scatter_variables) < 6 } {
					  $ivru_w selection set 1
					  set bicalibpar(ivru) [lindex $dim_name_scatterplot 1]
				      }
				      
				      $ivrv_w clear list
				      foreach i $dim_name_scatterplot {
					  $ivrv_w insert list end $i
				      }
				      if { $bicalibpar(scatter_variables) >= 6 } {
					  $ivrv_w selection set 5
					  set bicalibpar(ivrv) [lindex $dim_name_scatterplot 5]
				      } elseif { $bicalibpar(scatter_variables) < 6 } {
					  $ivrv_w selection set 2
					  set bicalibpar(ivrv) [lindex $dim_name_scatterplot 2]
				      }
				      
				      $iwt_w clear list
				      foreach i $dim_name_scatterplot {
					  $iwt_w insert list end $i
				      }
				      if { $bicalibpar(scatter_variables) >= 6 } {
					  $iwt_w selection set 6
					  set bicalibpar(iwt) [lindex $dim_name_scatterplot 6]
				      } elseif { $bicalibpar(scatter_variables) < 6 } {
					  $iwt_w selection set 0
					  set bicalibpar(iwt) [lindex $dim_name_scatterplot 0]
				      }
				      
				      ### set datafl widget normal
				      $calibfl_w configure -state normal -textbackground white -foreground black
				  }
			      }]
    
    
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $calibfl_w        -row 0 -column 0 -sticky we -pady 0
    grid $calibfl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    
    set subframe2_w [frame $calibration_frame_w.subfr2]
    
    if { $bicalibpar(calibfl) != {*.dat} } {
	set bicalibpar(cancel_counter_scatterplot) [expr $bicalibpar(cancel_counter_scatterplot) + 1]
    }

    set ivru_w [iwidgets::combobox $subframe2_w.ivru \
		    -labeltext "Primary  " \
		    -textvariable bicalibpar(ivru) \
		    -width 13 \
		    -listheight 78 \
		    -validate {regexpvalidate %c}]
    if { $bicalibpar(calibfl) == {*.dat} } {
	foreach i $dim_name_scatterplot {
	    $ivru_w insert list end $i
	}
	if { $bicalibpar(scatter_variables) >= 6 } {
	    $ivru_w selection set 4
	    set bicalibpar(ivru) [lindex $dim_name_scatterplot 4]
	} elseif { $bicalibpar(scatter_variables) < 6 } {
	    $ivru_w selection set 1
	    set bicalibpar(ivru) [lindex $dim_name_scatterplot 1]
	}
    } elseif { $bicalibpar(calibfl) != {*.dat} } {
	set tmp_ivru $bicalibpar(ivru)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if bicalibpar(icolx) is empty
	foreach i $dim_name_scatterplot {
	    $ivru_w insert list end $i
	    $ivru_w delete list [expr $bicalibpar(scatter_variables) + 1]
	    if { $i == $tmp_ivru } { ;# ok, that's the value in bicalibpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($bicalibpar(cancel_counter_scatterplot) * ($bicalibpar(scatter_variables) + 1))]
	$ivru_w selection set $sel
	set bicalibpar(ivru) [lindex $dim_name_scatterplot $sel]
    }
    
    
    set ivrv_w [iwidgets::combobox $subframe2_w.ivrv \
		    -labeltext "Secondary " \
		    -textvariable bicalibpar(ivrv) \
		    -width 13 \
		    -listheight 78 \
		    -validate {regexpvalidate %c}]
    if { $bicalibpar(calibfl) == {*.dat} } {
	foreach i $dim_name_scatterplot {
	    $ivrv_w insert list end $i
	}
	if { $bicalibpar(scatter_variables) >= 6 } {
	    $ivrv_w selection set 5
	    set bicalibpar(ivrv) [lindex $dim_name_scatterplot 5]
	} elseif { $bicalibpar(scatter_variables) < 6 } {
	    $ivrv_w selection set 2
	    set bicalibpar(ivrv) [lindex $dim_name_scatterplot 2]
	}
    } elseif { $bicalibpar(calibfl) != {*.dat} } {
	set tmp_ivrv $bicalibpar(ivrv)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if bicalibpar(ivrv) is empty
	foreach i $dim_name_scatterplot {
	    $ivrv_w insert list end $i
	    $ivrv_w delete list [expr $bicalibpar(scatter_variables) + 1]
	    if { $i == $tmp_ivrv } { ;# ok, that's the value in bicalibpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($bicalibpar(cancel_counter_scatterplot) * ($bicalibpar(scatter_variables) + 1))]
	$ivrv_w selection set $sel
	set bicalibpar(ivrv) [lindex $dim_name_scatterplot $sel]
    }
    
    
    set iwt_w [iwidgets::combobox $subframe2_w.iwt \
		   -labeltext "Weight " \
		   -textvariable bicalibpar(iwt) \
		   -width 13 \
		   -listheight 78 \
		   -validate {regexpvalidate %c}]
    if { $bicalibpar(calibfl) == {*.dat} } {
	foreach i $dim_name_scatterplot {
	    $iwt_w insert list end $i
	}
	if { $bicalibpar(scatter_variables) >= 6 } {
	    $iwt_w selection set 6
	    set bicalibpar(iwt) [lindex $dim_name_scatterplot 6]
	} elseif { $bicalibpar(scatter_variables) < 6 } {
	    $iwt_w selection set 0
	    set bicalibpar(iwt) [lindex $dim_name_scatterplot 0]
	}
    } elseif { $bicalibpar(calibfl) != {*.dat} } {
	set tmp_iwt $bicalibpar(iwt)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if bicalibpar(iwt) is empty
	foreach i $dim_name_scatterplot {
	    $iwt_w insert list end $i
	    $iwt_w delete list [expr $bicalibpar(scatter_variables) + 1]
	    if { $i == $tmp_iwt } { ;# ok, that's the value in bicalibpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($bicalibpar(cancel_counter_scatterplot) * ($bicalibpar(scatter_variables) + 1))]
	$iwt_w selection set $sel
	set bicalibpar(iwt) [lindex $dim_name_scatterplot $sel]
    }
    
    
    grid columnconfigure $subframe2_w 0 -minsize 155
    grid columnconfigure $subframe2_w 1 -minsize 171
    grid columnconfigure $subframe2_w 2 -minsize 160
    
    grid $ivru_w -row 0 -column 0 -sticky w -pady 0
    grid $ivrv_w -row 0 -column 1 -sticky w -pady 0
    grid $iwt_w  -row 0 -column 2 -sticky w -pady 0

    
    set subframe3_w [frame $calibration_frame_w.subfr3]
    set trimlim_min_w [iwidgets::entryfield $subframe3_w.min \
			   -labeltext "Trimming limits " \
			   -width 16 \
			   -textvariable bicalibpar(tmin) \
			   -validate {check_number_t %W %P}]
    
    set trimlim_max_w [iwidgets::entryfield $subframe3_w.max \
			   -labeltext "     " \
			   -width 16 \
			   -textvariable bicalibpar(tmax) \
			   -validate {check_number_t %W %P}]
    
    
    grid columnconfigure $subframe3_w 0 -minsize 195
    grid columnconfigure $subframe3_w 1 -minsize 60
    
    grid $trimlim_min_w -row 0 -column 0 -sticky w -pady 6
    grid $trimlim_max_w -row 0 -column 1 -sticky w -pady 6
    
    
    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	-anchor w -expand 1 -padx 3 -pady 1
    
    
    # fields for output data
    set output_iframe_w [iwidgets::labeledframe $work_frame_w.output \
			     -labeltext "Output" \
			     -labelpos nw \
			     -borderwidth 2 \
			     -relief groove]
    set output_frame_w [$output_iframe_w childsite]
    
    set subframe1_w [frame $output_frame_w.subfr1]
    set outfl_w [iwidgets::entryfield $subframe1_w.outfl \
		     -labeltext "Output data / distribution " \
		     -width 52 \
		     -textvariable bicalibpar(outfl)]
    set outfl_browse_w [iwidgets::pushbutton $subframe1_w.outfl_browse \
			    -height 27 \
			    -width 27 \
			    -defaultringpad 0 \
			    -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			    -command {
				set tmp_fname [tk_getSaveFile \
						   -defaultextension ".out" \
						   -filetypes {{{Output Files} {.out}} {{All Files} {.*}}} \
						   -initialfile $bicalibpar(outfl) \
						   -initialdir $exedir \
						   -title "Save output file as ..."]

				if { $tmp_fname != {} } {
				    set bicalibpar(outfl) $tmp_fname
				}
			    }]
    
    
    grid columnconfigure $subframe1_w 0 -minsize 448
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $outfl_w        -row 0 -column 0 -sticky e -pady 0
    grid $outfl_browse_w -row 0 -column 1 -sticky e -pady 0
    
    
    set subframe2_w [frame $output_frame_w.subfr2]
    set calfl_w [iwidgets::entryfield $subframe2_w.calfl \
		     -labeltext "Output calibration (sisim) " \
		     -width 52 \
		     -textvariable bicalibpar(calfl)]
    set calfl_browse_w [iwidgets::pushbutton $subframe2_w.calfl_browse \
			    -height 27 \
			    -width 27 \
			    -defaultringpad 0 \
			    -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			    -command {
				set tmp_fname [tk_getSaveFile \
						   -defaultextension ".txt" \
						   -filetypes {{{Calibration Files} {.txt}} {{All Files} {.*}}} \
						   -initialfile $bicalibpar(calfl) \
						   -initialdir $exedir \
						   -title "Save calibration file as ..."]
				if { $tmp_fname != {} } {
				    set bicalibpar(calfl) $tmp_fname
				}
			    }]
    
    
    grid columnconfigure $subframe2_w 0 -minsize 448
    grid columnconfigure $subframe2_w 1 -minsize 25
    
    grid $calfl_w        -row 0 -column 0 -sticky e -pady 0
    grid $calfl_browse_w -row 0 -column 1 -sticky e -pady 0
    
    
    set subframe3_w [frame $output_frame_w.subfr3]
    set repfl_w [iwidgets::entryfield $subframe3_w.repfl \
		     -labeltext "Calibration report " \
		     -width 52 \
		     -textvariable bicalibpar(repfl)]
    set repfl_browse_w [iwidgets::pushbutton $subframe3_w.repfl_browse \
			    -height 27 \
			    -width 27 \
			    -defaultringpad 0 \
			    -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			    -command {
				set tmp_fname [tk_getSaveFile \
						   -defaultextension ".txt" \
						   -filetypes {{{Report Files} {.txt}} {{All Files} {.*}}} \
						   -initialfile $bicalibpar(repfl) \
						   -initialdir $exedir \
						   -title "Save report file as ..."]
				if { $tmp_fname != {} } {
				    set bicalibpar(repfl) $tmp_fname
				}
			    }]
    
    
    grid columnconfigure $subframe3_w 0 -minsize 448
    grid columnconfigure $subframe3_w 1 -minsize 25
    
    grid $repfl_w        -row 0 -column 0 -sticky e -pady 0
    grid $repfl_browse_w -row 0 -column 1 -sticky e -pady 0
    
    
    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	-anchor w -expand 1 -padx 3 -pady 0
    
    
    # fields for thresholds
    set thresholds_iframe_w [iwidgets::labeledframe $work_frame_w.thresholds \
				 -labeltext "Thresholds" \
				 -labelpos nw \
				 -borderwidth 2 \
				 -relief groove]
    set thresholds_frame_w [$thresholds_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $thresholds_frame_w.subfr1]
    set plus_primary_variables_pushbutton_w [iwidgets::pushbutton $subframe1_w.plus_pri_pbutton \
						 -height 27 \
						 -width 27 \
						 -defaultringpad 0 \
						 -image [image create photo -file [file join "$exedir/images/plus.gif"]] \
						 -command {plus_primary_variables}]
    
    set minus_primary_variables_pushbutton_w [iwidgets::pushbutton $subframe1_w.minus_pri_pbutton \
						  -height 27 \
						  -width 27 \
						  -defaultringpad 0 \
						  -image [image create photo -file \
							      [file join "$exedir/images/minus.gif"]] \
						  -command {minus_primary_variables}]
    
    set plus_secondary_variables_pushbutton_w [iwidgets::pushbutton $subframe1_w.plus_sec_pbutton \
						   -height 27 \
						   -width 27 \
						   -defaultringpad 0 \
						   -image [image create photo -file \
							       [file join "$exedir/images/plus.gif"]] \
						   -command {plus_secondary_variables}]
    
    set minus_secondary_variables_pushbutton_w [iwidgets::pushbutton $subframe1_w.minus_sec_pbutton \
						    -height 27 \
						    -width 27 \
						    -defaultringpad 0 \
						    -image [image create photo -file \
								[file join "$exedir/images/minus.gif"]] \
						    -command {minus_secondary_variables}]
    
    
    grid columnconfigure $subframe1_w 0 -minsize 10
    grid columnconfigure $subframe1_w 1 -minsize 217
    grid columnconfigure $subframe1_w 2 -minsize 10
    grid columnconfigure $subframe1_w 3 -minsize 10
    
    grid $plus_primary_variables_pushbutton_w    -row 0 -column 0 -sticky w -pady 0
    grid $minus_primary_variables_pushbutton_w   -row 0 -column 1 -sticky w -pady 0
    grid $plus_secondary_variables_pushbutton_w  -row 0 -column 2 -sticky w -pady 0
    grid $minus_secondary_variables_pushbutton_w -row 0 -column 3 -sticky w -pady 0
    
    
    set subframe2_w [frame $thresholds_frame_w.subfr2]
    set cutu_frame_w [frame $subframe2_w.cutu_frame \
			  -background grey40 \
			  -borderwidth 2 \
			  -relief sunken]
    
    set cutu_sframe_w [frame $cutu_frame_w.cutu_sframe \
			   -height 90 \
			   -width 225]
    set cutu_hsbar_w [scrollbar $cutu_sframe_w.yscroll \
			  -orient vertical \
			  -command "$cutu_sframe_w.table yview"]

    if { $bicalibpar(calibfl) == "*.dat" } {
    
	set cutu_table_w [table $cutu_sframe_w.table \
			      -variable cutu_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 2 \
			      -rows 3 \
			      -yscrollcommand "$cutu_hsbar_w set" \
			      -height 5]
	$cutu_table_w width 0 5
	$cutu_table_w width 1 28
	
	set cutu_vals(0,0) "\#"
	set cutu_vals(0,1) "Thresholds on primary"
	
	for { set ii 1 } { $ii < 3 } { incr ii } {
	    
	    set bicalibpar(ncutu) $ii
	    
	    set cutu_vals($ii,label) [iwidgets::labeledwidget $cutu_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]
	    
	    set bicalibpar(cutu,$ii) $ii
	    set cutu_vals($ii,entry) [iwidgets::entryfield $cutu_table_w.entry$ii \
					  -textvariable bicalibpar(cutu,$ii) \
					  -validate {check_number_thresholds %W %P %c} \
					  -width 28]
	    
	    $cutu_table_w window configure $ii,0 -window $cutu_vals($ii,label) -relief flat -background gray50
	    $cutu_table_w window configure $ii,1 -window $cutu_vals($ii,entry)
	}

    } elseif { $bicalibpar(calibfl) != "*.dat" } {

	set cutu_table_w [table $cutu_sframe_w.table \
			      -variable cutu_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 2 \
			      -rows [expr $bicalibpar(ncutu) + 1] \
			      -yscrollcommand "$cutu_hsbar_w set" \
			      -height 5]
	$cutu_table_w width 0 5
	$cutu_table_w width 1 28
	
	set cutu_vals(0,0) "\#"
	set cutu_vals(0,1) "Thresholds on primary"
	
	for { set ii 1 } { $ii < [expr $bicalibpar(ncutu) + 1] } { incr ii } {
	    
	    set cutu_vals($ii,label) [iwidgets::labeledwidget $cutu_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]
	    
	    set cutu_vals($ii,entry) [iwidgets::entryfield $cutu_table_w.entry$ii \
					  -textvariable bicalibpar(cutu,$ii) \
					  -validate {check_number_thresholds %W %P %c} \
					  -width 28]
	    
	    $cutu_table_w window configure $ii,0 -window $cutu_vals($ii,label) -relief flat -background gray50
	    $cutu_table_w window configure $ii,1 -window $cutu_vals($ii,entry)
	}
    }
    
    
    pack $cutu_hsbar_w  -side right -anchor w -fill y
    pack $cutu_table_w  -side left -anchor w -padx 2 -expand 1 -fill both
    pack $cutu_sframe_w -anchor nw -expand 0
    pack propagate $cutu_sframe_w 0
    
    
    set spaceframe_w [frame $thresholds_frame_w.spaceframe]
    
    
    set cutv_frame_w [frame $subframe2_w.cutv_frame \
			  -background grey40 \
			  -borderwidth 2 \
			  -relief sunken]
    
    set cutv_sframe_w [frame $cutv_frame_w.cutv_sframe \
			   -height 90 \
			   -width 225]
    set cutv_hsbar_w [scrollbar $cutv_sframe_w.yscroll \
			  -orient vertical \
			  -command "$cutv_sframe_w.table yview"]

    if { $bicalibpar(calibfl) == "*.dat" } {
    
	set cutv_table_w [table $cutv_sframe_w.table \
			      -variable cutv_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 2 \
			      -rows 3 \
			      -yscrollcommand "$cutv_hsbar_w set" \
			      -height 5]
	$cutv_table_w width 0 5
	$cutv_table_w width 1 28
	
	set cutv_vals(0,0) "\#"
	set cutv_vals(0,1) "Thresholds on secondary"
	
	for { set ii 1 } { $ii < 3 } { incr ii } {
	    
	    set bicalibpar(ncutv) $ii
	    
	    set cutv_vals($ii,label) [iwidgets::labeledwidget $cutv_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]
	    
	    set bicalibpar(cutv,$ii) $ii
	    set cutv_vals($ii,entry) [iwidgets::entryfield $cutv_table_w.entry$ii \
					  -textvariable bicalibpar(cutv,$ii) \
					  -validate {check_number_thresholds %W %P %c} \
					  -width 28]
	    
	    $cutv_table_w window configure $ii,0 -window $cutv_vals($ii,label) -relief flat -background gray50
	    $cutv_table_w window configure $ii,1 -window $cutv_vals($ii,entry)
	}

    } elseif { $bicalibpar(calibfl) != "*.dat" } {
	
	set cutv_table_w [table $cutv_sframe_w.table \
			      -variable cutv_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 2 \
			      -rows [expr $bicalibpar(ncutv) + 1] \
			      -yscrollcommand "$cutv_hsbar_w set" \
			      -height 5]
	$cutv_table_w width 0 5
	$cutv_table_w width 1 28
	
	set cutv_vals(0,0) "\#"
	set cutv_vals(0,1) "Thresholds on secondary"
	
	for { set ii 1 } { $ii < [expr $bicalibpar(ncutv) + 1] } { incr ii } {
	    
	    set cutv_vals($ii,label) [iwidgets::labeledwidget $cutv_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]
	    
	    set cutv_vals($ii,entry) [iwidgets::entryfield $cutv_table_w.entry$ii \
					  -textvariable bicalibpar(cutv,$ii) \
					  -validate {check_number_thresholds %W %P %c} \
					  -width 28]
	    
	    $cutv_table_w window configure $ii,0 -window $cutv_vals($ii,label) -relief flat -background gray50
	    $cutv_table_w window configure $ii,1 -window $cutv_vals($ii,entry)
	}
    }


    pack $cutv_hsbar_w  -side right -anchor w -fill y
    pack $cutv_table_w  -side left -anchor w -padx 2 -expand 1 -fill both
    pack $cutv_sframe_w -anchor nw -expand 0
    pack propagate $cutv_sframe_w 0
    
    
    grid columnconfigure $subframe2_w 0 -minsize 230
    grid columnconfigure $subframe2_w 1 -minsize 15
    grid columnconfigure $subframe2_w 2 -minsize 230
    
    grid $cutu_frame_w -row 0 -column 0 -sticky new -rowspan 4 -pady 0
    grid $spaceframe_w -row 0 -column 1 -sticky new -rowspan 4 -pady 0
    grid $cutv_frame_w -row 0 -column 2 -sticky new -rowspan 4 -pady 0
    
    
    pack \
	$subframe1_w \
	$spaceframe_w \
	$subframe2_w \
	-anchor w -expand 1 -padx 3 -pady 1
    
    
    pack \
	$data_iframe_w \
	$calibration_iframe_w \
	$output_iframe_w \
	$thresholds_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x
    
    
    pack \
	$work_frame_w \
	-anchor nw -side top -fill x -fill y -expand true \
	-padx 0 -pady 0
    
    
    $bicalib_box_w.bicalib_tbn view "Files"
    
    
    . configure -cursor arrow
    
    
    #############################################################################
    # Default buttons
    #############################################################################
    # write_bicalib_params fname [tmpflag]
    #----------------------------------------------------------------------------
    # This proc writes the parameter file for running bicalib
    #
    # fname   : [<-] : name of parameter file
    # tmpflag : [<-] : optional flag (0,1). If this flag is set, the
    #                  temporary data files are created in order to
    #                  circumvent a problem with the bicalib program
    #                  which forbids long filenames. The following
    #                  temporary files are created in the working
    #                  directory:
    #                    - "datafl_in.dat" (secondary data input)
    #                    - "calibfl_in.dat" (calibrated scatterplot input)
    #                    - "outfl_out.dat" (result of bicalib run, prior distribution)
    #                    - "calfl_out.txt" (result of bicalib run, calibratio)
    #                    - "repfl_out.txt" (result of bicalib run, report)
    #############################################################################
    
    set button_frame_w [frame $bicalib_box_w.button_frame]
    
    set buttons_iframe_w [iwidgets::labeledframe $button_frame_w.buttons_iframe \
			      -labeltext "Defaults" \
			      -labelpos nw \
			      -borderwidth 2 \
			      -relief groove]
    set buttons_frame_w [$buttons_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set open_button_w [iwidgets::pushbutton $buttons_iframe_w.open \
			   -height 27 \
			   -width 40 \
			   -defaultringpad 0 \
			   -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			   -command {
			       set tmp_fname [tk_getOpenFile \
						  -initialdir [file dirname $bicalibpar(datafl)] \
						  -defaultextension ".par" \
						  -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
						  -title "Select BICALIB parameter file"]
			       
			       if { $tmp_fname != {} } {
				   set param_fname $tmp_fname
				   read_bicalib_params $param_fname
			       }
			   }]
    
    
    set save_button_w [iwidgets::pushbutton $buttons_iframe_w.save \
			   -height 27 \
			   -width 27 \
			   -defaultringpad 0 \
			   -state disabled \
			   -image [image create photo -file [file join "$exedir/images/save.gif"]] \
			   -command {
			       set bicalibpar(title) 0
			       write_bicalib_params $param_fname
			   }]
    
    
    set save_as_button_w [iwidgets::pushbutton $buttons_iframe_w.save_as \
			      -height 27 \
			      -width 27 \
			      -defaultringpad 0 \
			      -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			      -command {
				  if { $bicalibpar(datafl) != {*.dat} && $bicalibpar(calibfl) != {*.dat} } {
				      set tmp_fname [tk_getSaveFile \
							 -initialdir [file dirname $bicalibpar(datafl)] \
							 -defaultextension ".par" \
							 -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
							 -title "Save BICALIB parameter file as ..."]
				      
				      $save_button_w configure -state normal
				      
				      if { $tmp_fname != {} } {
					  set bicalibpar(title) 0
					  set param_fname $tmp_fname
					  write_bicalib_params $param_fname
				      }
				  } elseif { $bicalibpar(datafl) == {*.dat} && $bicalibpar(calibfl) != {*.dat} } {
				      tk_messageBox \
					  -title "Defaults" \
					  -icon warning \
					  -message "You should select a secondary data file first." \
					  -type ok
				  } elseif { $bicalibpar(datafl) != {*.dat} && $bicalibpar(calibfl) == {*.dat} } {
				      tk_messageBox \
					  -title "Defaults" \
					  -icon warning \
					  -message "You should select a calibration scatterplot file first." \
					  -type ok
				  } elseif { $bicalibpar(datafl) == {*.dat} && $bicalibpar(calibfl) == {*.dat} } {
				      tk_messageBox \
					  -title "Defaults" \
					  -icon warning \
					  -message "You should select a secondary data file and a calibration scatterplot file first." \
					  -type ok
				  }
			      }]
    
    
    set cancel_button_w [iwidgets::pushbutton $buttons_iframe_w.cancel \
			     -height 27 \
			     -width 27 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/stop.gif"]] \
			     -command {
				 set private(button) 0
			     }]
    
    
    set run_button_w [iwidgets::pushbutton $buttons_iframe_w.run \
			  -height 27 \
			  -width 27 \
			  -defaultringpad 0 \
			  -image [image create photo -file [file join "$exedir/images/run.gif"]] \
			  -command {
			      if { $bicalibpar(datafl) == {*.dat} && $bicalibpar(calibfl) != {*.dat} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "You should select a secondary data file first." \
				      -type ok
			      } elseif { $bicalibpar(datafl) != {*.dat} && $bicalibpar(calibfl) == {*.dat} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "You should select a calibration scatterplot file first." \
				      -type ok
			      } elseif { $bicalibpar(datafl) == {*.dat} && $bicalibpar(calibfl) == {*.dat} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "You should select a secondary data file and a calibration scatterplot file first." \
				      -type ok
			      } elseif { $bicalibpar(datafl) != {*.dat} && $bicalibpar(calibfl) != {*.dat} } {
				  if { $tcl_platform(platform) == "unix" } {
				      set command "../gslib/bin/bicalib"
				  } elseif { $tcl_platform(platform) == "windows" } {
				      set command "C:/Programme/Statios/Gslib77/bin/bicalib.exe"
				  }
				  
				  set bicalibpar(title) 1
				  set paramfile "tmprun.par"
				  write_bicalib_params_run $paramfile 1
				  
				  set outf [open "BICALIB.TMP" {w}]
				  puts $outf "$paramfile"
				  close $outf
				  set logfile [open "| $command < BICALIB.TMP" {r}]
				  fconfigure $logfile -buffering none
				  while { ![eof $logfile] } {
				      gets $logfile input_line
				      log $input_line
				      update
				  }
				  close $logfile
				  
				  ### copy/delete the temporary files
				  catch {
				      file copy -force "outfl_out.dat" $bicalibpar(outfl)
				      file copy -force "calfl_out.txt" $bicalibpar(calfl)
				      file copy -force "repfl_out.txt" $bicalibpar(repfl)
				  }
				  catch {
				      file delete -force \
					  "datafl_in.dat" \
					  "calibfl_in.dat" \
					  "outfl_out.dat" \
					  "calfl_out.txt" \
					  "repfl_out.txt" \
					  "tmprun.par" \
					  "BICALIB.TMP"
				  }
				  set private(button) 1
			      }
			  }]
    
    bind $run_button_w <Return> "
	    $run_button_w configure -state active -relief sunken
	    update idletasks
	    after 100
	    $run_button_w invoke
	"
    
    grid columnconfigure $button_frame_w 0 -minsize 15
    grid columnconfigure $button_frame_w 1 -minsize 100
    grid columnconfigure $button_frame_w 2 -minsize 15
    grid columnconfigure $button_frame_w 3 -minsize 100
    grid columnconfigure $button_frame_w 4 -minsize 15
    grid columnconfigure $button_frame_w 5 -minsize 100
    grid columnconfigure $button_frame_w 6 -minsize 15
    grid columnconfigure $button_frame_w 7 -minsize 100
    grid columnconfigure $button_frame_w 8 -minsize 15
    grid columnconfigure $button_frame_w 9 -minsize 100
    grid columnconfigure $button_frame_w 10 -minsize 15
    
    grid $open_button_w    -row 0 -column 1 -sticky we -pady 8 -padx 5
    grid $save_button_w    -row 0 -column 3 -sticky we -pady 8 -padx 5
    grid $save_as_button_w -row 0 -column 5 -sticky we -pady 8 -padx 5
    grid $cancel_button_w  -row 0 -column 7 -sticky we -pady 8 -padx 5
    grid $run_button_w     -row 0 -column 9 -sticky we -pady 8 -padx 5
    
    
    pack \
	$buttons_iframe_w \
	-padx 5 -pady 5 -expand 1
    
    
    pack $button_frame_w \
	-side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    raise $bicalib_box_w
    
    tkwait visibility $bicalib_box_w
    tkwait visibility $run_button_w
    
    set oldFocus [focus]
    set oldGrab [grab current $bicalib_box_w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $bicalib_box_w
    focus $run_button_w
    
    . configure -cursor arrow
    
    update idletask
    tkwait variable private(button)
    
    catch {focus $oldFocus}
    catch {
	# It's possible that the window has already been destroyed,
	# hence this "catch".  Delete the Destroy handler so that
	# tkPriv(button) doesn't get reset by it.
	
	bind $bicalib_box_w <Destroy> {}
	destroy $bicalib_box_w
    }
    
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }   
    return
}
#============================================================================
#=END=OF=BICALIB=============================================================