##START#OF#SISIM#############################################################
# geossav_sisim.tcl
#
# Source script creating the sisim tabnotebook in GEOSSAV,
# developed at the University of Basel, Switzerland,
# by Christian Regli and Lukas Rosenthaler
#
#############################################################################


#############################################################################
#
# The following set of parameters are required for the SISIM
#
global exedir
source $exedir/Geossav_sisim_globals.tcl

set sisimpar(vartype) 0                   ;# variable type: 1=continuous (cdf), 0=categorical (pdf)
set sisimpar(ncat)  4                     ;# number of thresholds or categories
set sisimpar(cat) 1                       ;# threshold values or category codes, there should be 
                                          ;# ncat values on this line of input
set sisimpar(pdf) 0.15                    ;# global cdf or pdf values, there should be ncat 
                                          ;# values on this line of input
set sisimpar(datafl) "*.dat"              ;# input data file, hard data, simplified Geo-EAS format
set sisimpar(icolx) 1                     ;# column number for the x coordinates
set sisimpar(icoly) 2                     ;# column number for the y coordinates, icoly=0 and icolz=0 
                                          ;# indicates 1-D simulation
set sisimpar(icolz) 3                     ;# column number for the z coordinates, icolz=0 indicates 
                                          ;# 2-D simulation
set sisimpar(icolvr)  4                   ;# column number for the variable to be simulated
set sisimpar(directik) "*.dat"            ;# input data file, soft data, simplified Geo-EAS format, 
                                          ;# the cdf data should steadily increase from 0 to 1 and 
                                          ;# soft categorical probabilities must be between 0 to 1 
                                          ;# and sum to 1.0
set sisimpar(icolsx) 1                    ;# column number for the x coordinates
set sisimpar(icolsy) 2                    ;# column number for the y coordinates
set sisimpar(icolsz) 3                    ;# column number for the z coordinates
set sisimpar(icoli)  4                    ;# column number for the indicator variables
set sisimpar(imbsim) 0                    ;# considering Markov-Bayes option for cokriging with soft 
                                          ;# indicator data, 1=yes, 0=no
set sisimpar(bz) 0                        ;# if imbsim is set to 1, then the B(z) calibration values 
                                          ;# are needed
set sisimpar(tmin) -1e+21                 ;# minimum trimming limit, all values strictly less than tmin 
                                          ;# are ignored
set sisimpar(tmax) +1e+21                 ;# maximum trimming limit, all values strictly greater than 
                                          ;# tmax are ignored
set sisimpar(zmin) 0                      ;# minimum attribute value when considering a continuous 
                                          ;# variable
set sisimpar(zmax) 30                     ;# maximum attribute value when considering a continuous 
                                          ;# variable
set sisimpar(ltail) 1                     ;# extrapolation in the lower tail: 
                                          ;# 1=linear interpolation to the lower limit zmin, 
                                          ;# 2= power model interpolation to the lower limit zmin, 
                                          ;# 3= linear interpolation between tabulated quantiles 
                                          ;# (only for continuous variables)
set sisimpar(ltpar) 1                     ;# parameter for interpolation in the lower tail
set sisimpar(middle) 1                    ;# interpolation within the middle of the distribution: 
                                          ;# 1=linear interpolation to the lower limit zmin, 
                                          ;# 2= power model interpolation to the lower limit zmin, 
                                          ;# 3= linear interpolation between tabulated quantiles 
                                          ;# (only for continuous variables)
set sisimpar(midpar) 1                    ;# parameter for interpolation within the middle
set sisimpar(utail) 1                     ;# extrapolation in the upper tail of the distribution: 
                                          ;# 1=linear interpolation to the lower limit zmin, 
                                          ;# 2= power model interpolation to the lower limit zmin, 
                                          ;# 3= linear interpolation between tabulated quantiles, 
                                          ;# 4=hyperbolic model extrapolation limited by zmax 
                                          ;# (only for continuous variables)
set sisimpar(utpar) 1                     ;# parameter for interpolation in the upper tail
set sisimpar(tabfl) "*.dat"               ;# tabulated data file, used if linear interpolation is the 
                                          ;# option selected for any of the three regions, simplified 
                                          ;# Geo-EAS format, legitimate is also datafl
set sisimpar(icolvrt) 4                   ;# column number for values in tabfl
set sisimpar(icolwtt) 5                   ;# column number for declustering weights in tabfl, 
                                          ;# 0=declustering weights not used
set sisimpar(idbg) 1                      ;# integer debugging level:
                                          ;# 0=none, 1=low, 2=middle, 3=high
set sisimpar(dbgfl) "*.dbg"               ;# output debugging file
set sisimpar(outfl) "*.out"               ;# output data file containing the results, cycling fastest 
                                          ;# on x then y then z then simulation by simulation
set sisimpar(nsim) 1                      ;# number of simulations to generate
set sisimpar(nx) 100                      ;# number of grid cells (x axis)
set sisimpar(xmn) 5                       ;# x coordinate at the center of the first grid cell (x axis)
set sisimpar(xsiz) 10                     ;# size of grid cells (x axis)
set sisimpar(ny) 100                      ;# number of grid cells (y axis)
set sisimpar(ymn) 5                       ;# y coordinate at the center of the first grid cell (y axis)
set sisimpar(ysiz) 10                     ;# size of grid cells (y axis)
set sisimpar(nz) 100                      ;# number of grid cells (z axis)
set sisimpar(zmn) 5                       ;# z coordinate at the center of the first grid cell (z axis)
set sisimpar(zsiz) 10                     ;# size of grid cells (z axis)
set sisimpar(seed) 69069                  ;# random number seed, a large odd integer 
set sisimpar(ndmax) 10                    ;# maximum number of original data that will be used to 
                                          ;# simulate a grid node
set sisimpar(ncnode) 10                   ;# maximum number of previously simulated nodes to use for 
                                          ;# the simulation of another node
set sisimpar(maxsec) 3                    ;# maximum number of soft data (at node locations) that will 
                                          ;# be used for the simulation of a node, particularly useful 
                                          ;# to restrict the number of soft data when an exhaustive 
                                          ;# secondary variable informs all grid nodes
set sisimpar(sstrat) 0                    ;# search strategy of data and previously simulated grid nodes: 
                                          ;# 0=data are searched with a super block search and previously 
                                          ;# simulated nodes are searched with a spiral search, 
                                          ;# 1=data are relocated to grid nodes and a spiral search is 
                                          ;# used, ndmin and ndmax are not considered
set sisimpar(multgrid) 0                  ;# multiple grid refinement: 
                                          ;# 1= multiple grid simulation, 
                                          ;# 0=standard spiral search
set sisimpar(nmult) 0                     ;# target number of multiple grid refinements, used only if 
                                          ;# multgrid is set to 1
set sisimpar(noct) 10                     ;# number of original data to use per octant, noct=0 it is not 
                                          ;# used, otherwise the closest noct data in each octant are 
                                          ;# retained for the simulation of a grid node
set sisimpar(radius_hmax) 20              ;# maximum horizontal direction of search ellipsoid
set sisimpar(radius_hmin) 20              ;# minimum horizontal direction of search ellipsoid
set sisimpar(radius_vert) 20              ;# vertical direction of search ellipsoid
set sisimpar(sang1) 0                     ;# angle defining the search ellipsoid
set sisimpar(sang2) 0                     ;# angle defining the search ellipsoid
set sisimpar(sang3) 0                     ;# angle defining the search ellipsoid
set sisimpar(mik) 0                       ;# indicator flag specifying median indikator kriging: 
                                          ;# 0=full indicator kriging using variograms of all categories, 
                                          ;# 1=median indicator kriging using the variogram 
                                          ;# corresponding to category mikcat
set sisimpar(mikcat) 0                    ;# variogram of the mikcat category, 1..ncat
set sisimpar(ktype) 0                     ;# kriging type, 0=simple kriging, 1=ordinary kriging
#
# The following set of parameters are required for each of the ncat categories
#
set sisimpar(nst) 1                       ;# number of semivariogram structures
set sisimpar(c0) 0.1                      ;# isotropic nugget constant
set sisimpar(it) 1                        ;# type of semivariogram structure: 1=spherical, 2=exponential, 
                                          ;# 3=gaussian, 4=power, 5=hole effect
set sisimpar(cc) 0.9                      ;# difference between sill and isotropic nugget constant
set sisimpar(ang1) 0                      ;# angle defining the structure ellipsoid
set sisimpar(ang2) 0                      ;# angle defining the structure ellipsoid
set sisimpar(ang3) 0                      ;# angle defining the structure ellipsoid
set sisimpar(aa_hmax) 10                  ;# maximum horizontal range of structure ellipsoid
set sisimpar(aa_hmin) 10                  ;# minimum horizontal range of structure ellipsoid
set sisimpar(aa_vert) 10                  ;# vertical range of structure ellipsoid
#
# The following set of parameters are required for GEOSSAV SISIM, especially for the feeding of the 
# SISIM parameters tabnotebook after running SISIM (SISIM parameters tabnotebook disappear after 
# start SISIM) or after cancel and restart the SISIM parameters tabnotebook without an exit of Geossav
#
set sisimpar(soft_data) 0                 ;# soft data available (0=no, 1=yes)
set sisimpar(variables_initial) 0         ;# number of initial variables (None, x, y, z, var1, var2, ...)
set sisimpar(variables) 0                 ;# number of variables (None, x, y, z, var1, var2, ...)
set sisimpar(hard_parameters) 0           ;# number of hard data variables excluding the variable of None
set sisimpar(soft_parameters) 0           ;# number of soft data variables excluding the variable of None
set sisimpar(tabulated_parameters) 0      ;# number of tabulated data variables excluding the variable of None
set sisimpar(hard_cancel_counter) 0       ;# counter for the exit from the SISIM parameters tabnotebook
                                          ;# if hard data were fed
set sisimpar(tabulated_cancel_counter) 0  ;# counter for the exit from the SISIM parameters tabnotebook
                                          ;# if tabulated data were fed
set sisimpar(soft_cancel_counter) 0       ;# counter for the exit from the SISIM parameters tabnotebook
                                          ;# if soft data were fed
set sisimpar(soft_cancel) 0               ;# variable indicating no or 'one or more' exit of the SISIM 
                                          ;# parameters tabnotebook (0=no, 1=one or more)
set sisimpar(infofl) "*.out"              ;# info file containing selected information for sisim data export
set sisimpar(title) 0                     ;# flag for writing parameter title


set focus_variable 0
#
# End of initialization of variables. This has to be done once at the start 
# of the programm.
#
#############################################################################


#############################################################################
#
# Start of proc section.
#

#############################################################################
proc sisim_read_hard_data { fname } {
    global dim_name_hard
    global sisimpar

    lappend dim_name_hard "None"
    if { [file readable $fname] } {	
	set inf [open $fname "r"]
	gets $inf
	gets $inf n
	for { set i 0 } { $i < $n } { incr i } {
	    gets $inf dum
	    lappend dim_name_hard $dum
	    set tmp_parameters [expr $i + 1]
	}
	close $inf
	set sisimpar(hard_parameters) $tmp_parameters
	return 1
    } else {
	return 0
    }
}
#============================================================================


#############################################################################
proc sisim_read_soft_data { fname } {
    global sisimpar
    global dim_name_soft
    global max_cut_cat

    lappend dim_name_soft "None"
    if { [file readable $fname] } {
	set inf [open $fname "r"]
	gets $inf
	gets $inf n
	for { set i 0 } { $i < $n } { incr i } {
	    gets $inf dum
	    lappend dim_name_soft $dum
	    set tmp_parameters [expr $i + 1]
	}
	if { ($sisimpar(soft_cancel_counter) == 0 && $sisimpar(soft_cancel) == 0) } {
	    set sisimpar(variables) $tmp_parameters
	    set sisimpar(soft_cancel) 1
	} elseif { ($sisimpar(soft_cancel_counter) == 0 && $sisimpar(soft_cancel) == 1) || \
		       $sisimpar(soft_cancel_counter) != 0 } {
	    set sisimpar(variables) $sisimpar(variables)
	}
	set max_cut_cat $sisimpar(variables)
	close $inf
	set sisimpar(soft_parameters) $tmp_parameters
	return 1
    } else {
	return 0
    }
}
#============================================================================


#############################################################################
proc sisim_read_tabulated_values { fname } {
    global dim_name_tabulated
    global sisimpar

    lappend dim_name_tabulated "None"
    if { [file readable $fname] } {
	set inf [open $fname "r"]
	gets $inf
	gets $inf n
	for { set i 0 } { $i < $n } { incr i } {
	    gets $inf dum
	    lappend dim_name_tabulated $dum
	    set tmp_parameters [expr $i + 1]
	}
	close $inf
	set sisimpar(tabulated_parameters) $tmp_parameters
	return 1
    } else {
	return 0
    }
}
#============================================================================


#############################################################################
proc regexpvalidate { char } {
    return [regexp {![ ]} $char]
}
#============================================================================


#############################################################################
proc check_number_t { entry newstr } {
    return [regexp {^[+-]?[0-9]*[\.]?[0-9]*([e][+-]?[0-9]*)?$} $newstr]
}
#============================================================================


#############################################################################
proc soft_data_activator { } {
    global sisimpar
    global directik_w
    global directik_browse_w
    global icolsx_w
    global icolsy_w
    global icolsz_w
    global imbsim_checkbutton_w
    global icoli_table_w
    
    if { $sisimpar(soft_data) == 1 && $sisimpar(directik) == {*.dat} } {
	$directik_w configure -state disabled -textbackground gray85 -foreground gray50
	$directik_browse_w configure -state normal
    } elseif { $sisimpar(soft_data) == 1 && $sisimpar(directik) != {*.dat} } {
	$directik_w configure -state normal -textbackground white -foreground black
	$directik_browse_w configure -state normal
	$icolsx_w configure -state normal -textbackground white -foreground black
	$icolsy_w configure -state normal -textbackground white -foreground black
	$icolsz_w configure -state normal -textbackground white -foreground black
	$imbsim_checkbutton_w configure -state normal
	if { $sisimpar(imbsim) == 0 } {
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3]} { incr ii } {
		$icoli_table_w.combo$ii configure -state normal -textbackground white -foreground black
	    }
	} elseif { $sisimpar(imbsim) == 1 } {
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3]} { incr ii } {
		$icoli_table_w.combo$ii configure -state normal -textbackground white -foreground black
		$icoli_table_w.bz$ii configure -state normal -textbackground white -foreground black
	    }
	}
    } elseif { $sisimpar(soft_data) == 0 && $sisimpar(directik) != {*.dat} } {
	$directik_w configure -state disabled -textbackground gray85 -foreground gray50
	$directik_browse_w configure -state disabled
	$icolsx_w configure -state disabled -textbackground gray85 -foreground gray50
	$icolsy_w configure -state disabled -textbackground gray85 -foreground gray50
	$icolsz_w configure -state disabled -textbackground gray85 -foreground gray50
	$imbsim_checkbutton_w configure -state disabled
	for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3]} { incr ii } {
	    $icoli_table_w.combo$ii configure -state disabled -textbackground gray85 -foreground gray50
	    $icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
	}
    } elseif { $sisimpar(soft_data) == 0 && $sisimpar(directik) == {*.dat} } {
	$directik_w configure -state disabled -textbackground gray85 -foreground gray50
	$directik_browse_w configure -state disabled
	$icolsx_w configure -state disabled -textbackground gray85 -foreground gray50
	$icolsy_w configure -state disabled -textbackground gray85 -foreground gray50
	$icolsz_w configure -state disabled -textbackground gray85 -foreground gray50
	$imbsim_checkbutton_w configure -state disabled
    }
}
#============================================================================


#############################################################################
proc bz_calibration { } {
    global sisimpar
    global bz_vals
    global icoli_vals
    global icoli_table_w
    global dim_name_soft
    global variables_counter
    
    if { $sisimpar(directik) == {*.dat} } {
	if {$sisimpar(imbsim) == 1} {
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4]} { incr ii } {
		$icoli_table_w.bz$ii configure -state normal -textbackground white -foreground black
	    }
	} elseif {$sisimpar(imbsim) == 0} {
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4]} { incr ii } {
		$icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
	    }
	}
    } elseif { $sisimpar(directik) != {*.dat} } {
	if {$sisimpar(imbsim) == 1} {
	    if { $sisimpar(soft_parameters) >= 4 } {
		for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3]} { incr ii } {
		    $icoli_table_w.bz$ii configure -state normal -textbackground white -foreground black
		}
	    } elseif { $sisimpar(soft_parameters) < 4 } {
		for { set ii 1 } { $ii <= $variables_counter } { incr ii } {
		    $icoli_table_w.bz$ii configure -state normal -textbackground white -foreground black
		}	    
	    }
	} elseif {$sisimpar(imbsim) == 0} {
	    if { $sisimpar(soft_parameters) >= 4 } {
		for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3]} { incr ii } {
		    $icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
		}
	    } elseif { $sisimpar(soft_parameters) < 4 } {
		for { set ii 1 } { $ii <= $variables_counter } { incr ii } {
		    $icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
		}	    
	    }
	}
    }
    
    #############################################################################
    proc check_number_bz_values { entry newstr char } {
	if { [regexp {^[+-]?[0-9]*[\.]?[0-9]*([e][+-]?[0-9]*)?$} $newstr] } {
	    catch {scan $newstr {%f} newstr}
	    if { $newstr == "-" || $newstr == "+" || $newstr == "-." || $newstr == "+." || $newstr == "." } {
		return 1
	    }
	    return [expr ($newstr >= -1.0) && ($newstr <= 1.0) ]
	} else {
	    return 0
	}
    }
    #============================================================================
}
#============================================================================


#############################################################################
proc check_number_seed { entry newstr char} {
    set current [$entry get]
    set len [string length $current]
    if {$len < 8} {
	return [regexp {^[0-9]*$} $newstr]
    } elseif {$len == 8 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "69069"
    }
}
#============================================================================


#############################################################################
proc check_number_nsim { entry newstr char } {
    set current [$entry get]
    set len [string length $current]
    if { $len < 2 } {
	return [regexp {^[0-9]*$} $newstr]
    } elseif { $len == 2 && [string match {[0-9]} $char] } {
	$entry delete 0 end  
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc seed_generator { } {
    global sisimpar

    set sisimpar(seed) [expr (rand()*10000000)]
    catch {scan $sisimpar(seed) {%d} sisimpar(seed)}	
    return
}
#============================================================================


#############################################################################
proc extrapolation_activator { } {
    global sisimpar
    global min_attribute_value_w
    global extrapol_lower_tail_w
    global extrapol_lower_tail_parameter_w
    global interpol_middle_w
    global interpol_middle_parameter_w
    global max_attribute_value_w
    global extrapol_upper_tail_w
    global extrapol_upper_tail_parameter_w
    global tabfl_w
    global tabfl_browse_w
    global icolvrt_w
    global icolwtt_w
    
    if { $sisimpar(vartype) == "Categorical" } {
	$min_attribute_value_w           configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_lower_tail_w           configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_lower_tail_parameter_w configure -state disabled -textbackground gray85 -foreground gray50
	$interpol_middle_w               configure -state disabled -textbackground gray85 -foreground gray50
	$interpol_middle_parameter_w     configure -state disabled -textbackground gray85 -foreground gray50
	$max_attribute_value_w           configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_upper_tail_w           configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_upper_tail_parameter_w configure -state disabled -textbackground gray85 -foreground gray50
	$tabfl_w                         configure -state disabled -textbackground gray85 -foreground gray50
	$tabfl_browse_w                  configure -state disabled
	$icolvrt_w                       configure -state disabled -textbackground gray85 -foreground gray50
	$icolwtt_w                       configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { ($sisimpar(vartype) == "Continous") && \
		   ($sisimpar(ltail) == "Tabulated Quantiles Linear Interpolation" || \
			$sisimpar(middle) == "Tabulated Quantiles Linear Interpolation" || \
			$sisimpar(utail) == "Tabulated Quantiles Linear Interpolation") } {
	$min_attribute_value_w           configure -state normal -textbackground white -foreground black
	$extrapol_lower_tail_w           configure -state normal -textbackground white -foreground black
	$extrapol_lower_tail_parameter_w configure -state normal -textbackground white -foreground black
	$interpol_middle_w               configure -state normal -textbackground white -foreground black
	$interpol_middle_parameter_w     configure -state normal -textbackground white -foreground black
	$max_attribute_value_w           configure -state normal -textbackground white -foreground black
	$extrapol_upper_tail_w           configure -state normal -textbackground white -foreground black
	$extrapol_upper_tail_parameter_w configure -state normal -textbackground white -foreground black
	$tabfl_w                         configure -state normal -textbackground white -foreground black
	$tabfl_browse_w                  configure -state normal
	$icolvrt_w                       configure -state normal -textbackground white -foreground black
	$icolwtt_w                       configure -state normal -textbackground white -foreground black
    } elseif { ($sisimpar(vartype) == "Continous") && \
		   ($sisimpar(ltail) != "Tabulated Quantiles Linear Interpolation" || \
			$sisimpar(middle) != "Tabulated Quantiles Linear Interpolation" || \
			$sisimpar(utail) != "Tabulated Quantiles Linear Interpolation") } {
	$min_attribute_value_w           configure -state normal -textbackground white -foreground black
	$extrapol_lower_tail_w           configure -state normal -textbackground white -foreground black
	$extrapol_lower_tail_parameter_w configure -state normal -textbackground white -foreground black
	$interpol_middle_w               configure -state normal -textbackground white -foreground black
	$interpol_middle_parameter_w     configure -state normal -textbackground white -foreground black
	$max_attribute_value_w           configure -state normal -textbackground white -foreground black
	$extrapol_upper_tail_w           configure -state normal -textbackground white -foreground black
	$extrapol_upper_tail_parameter_w configure -state normal -textbackground white -foreground black
	$tabfl_w                         configure -state disabled -textbackground gray85 -foreground gray50
	$tabfl_browse_w                  configure -state disabled 
	$icolvrt_w                       configure -state disabled -textbackground gray85 -foreground gray50
	$icolwtt_w                       configure -state disabled -textbackground gray85 -foreground gray50
    }
}
#============================================================================


#############################################################################
proc cutoff_category_activator { } {
    global sisimpar
    global sisim_box_w
    global cutoff_category_w
    
    if {$sisimpar(mik) == "Full IK"} {
	$cutoff_category_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif {$sisimpar(mik) == "Median IK"} {
	$cutoff_category_w configure -state normal -textbackground white -foreground black
    }
}
#============================================================================


#############################################################################
proc check_number_mikcat { entry newstr char } {
    set current [$entry get]
    catch {scan $current {%g} current}
    if {$current < 10} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$current >= 10 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "100"
    } elseif {$current >= 10 && $current <= 100 && [string match {[\.]} $char]} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    }
}
#============================================================================


#############################################################################
proc check_number_ndmax { entry newstr char} {
    set current [$entry get]
    catch {scan $current {%d} current}
    if {$current < 10} {
	return [regexp {^[0-9]*$} $newstr]
    } elseif {$current >= 10 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc check_number_maxsec { entry newstr char} {
    set current [$entry get]
    catch {scan $current {%d} current}
    if {$current < 10} {
	return [regexp {^[0-9]*$} $newstr]
    } elseif {$current >= 10 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc check_number_ncnode { entry newstr char} {
    set current [$entry get]
    catch {scan $current {%d} current}
    if {$current < 10} {
	return [regexp {^[0-9]*$} $newstr]
    } elseif {$current >= 10 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc check_number_noct { entry newstr char} {
    set current [$entry get]
    catch {scan $current {%d} current}
    if {$current < 10} {
	return [regexp {^[0-9]*$} $newstr]
    } elseif {$current >= 10 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc max_org_data_number { } {
    global sisimpar
    global sisim_box_w
    global max_org_data_w
    
    if {$sisimpar(sstrat) == 0} {
	$max_org_data_w configure -state normal -foreground black
	$max_org_data_w component entry configure -background white
    } elseif {$sisimpar(sstrat) == 1} {
	$max_org_data_w configure -state disabled -foreground gray50
	$max_org_data_w component entry configure -background gray85
    }
}
#============================================================================


#############################################################################
proc check_number_radius { entry newstr char } {
    set current [$entry get]
    catch {scan $current {%g} current}
    if {$current < 100} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$current >= 100 && [string match {[0-9.]} $char]} {
	$entry delete 0 end  
	$entry insert end "1000"
    }
}
#============================================================================


#############################################################################
proc grid_refinement_number { } {
    global sisimpar
    global sisim_box_w
    global grid_refinement_number_w
    
    if {$sisimpar(multgrid) == 1} {
	$grid_refinement_number_w configure -state normal -textbackground white -foreground black
    } elseif {$sisimpar(multgrid) == 0} {
	$grid_refinement_number_w configure -state disabled -textbackground gray85 -foreground gray50
    }
}
#============================================================================


#############################################################################
proc check_number_sang1 { entry newstr char } {
    if {$newstr == {} || ($newstr >= 0 && $newstr <= 360) && [string match {[0-9.]} $char]} {
	return [regexp {^[-]?[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$newstr > 360 && [string match {[.0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "360"
    }
}
#============================================================================


#############################################################################
proc check_number_sang { entry newstr char } {
    if {$newstr == {} || ($newstr >= -90 && $newstr <= 90) || $newstr == {-} && [string match {[-0-9.]} $char]} {
	return [regexp {^[-]?[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$newstr > 90 && [string match {[0-9]} $char]} {
	$entry delete 0 endr    if {$newstr == {} || ($newstr >= -90 && $newstr <= 90) || $newstr == {-} && [string match {[-0-9.]} $char]} {
	return [regexp {^[-]?[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$newstr > 90 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "90"
    } elseif {$newstr < -90 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "-90"
    }  
	$entry insert end "90"
    } elseif {$newstr < -90 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "-90"
    }
}
#============================================================================


#############################################################################
proc check_number_multgrid { entry newstr char } {
    set current [$entry get]
    catch {scan $current {%d} current}
    if {$current < 10} {
	return [regexp {^[0-9]*$} $newstr]
    } elseif {$current >= 10 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc plus_cutoff_category { } {
    global sisimpar
    global icoli_table_w
    global pdf_cdf_table_w
    global icoli_vals
    global bz_vals
    global cut_cat_vals
    global pdf_cdf_vals
    global logwin
    global dim_name_soft
    global max_cut_cat_initial
    global max_cut_cat
    global variables_initial_counter
    global variables_counter
    global model_sframe_w
    global model_hsbar_w
    global model_table_w
    global model_vals_label
    global model_type
    global model_cc
    global model_ang1
    global model_ang2
    global model_ang3
    global model_aa_hmax
    global model_aa_hmin
    global model_aa_vert
    global models_nugget_w
    global models_sill_label_w
    global models_sill_value_w
    global cut_cat_number_label_w
    global cut_cat_number_value_w
    global cut_cat_number
    global subframe3_variogram_w
    global no
    global sum_cc
    global focus_variable
    global sill
    
    if { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat} && $focus_variable == 0 } {
	set max_hard_variables [expr $sisimpar(variables_initial) + 1]
	
	if { $sisimpar(variables_initial) <= 18 } {
	    
	    set ii [expr $variables_initial_counter + 1]
	    
	    ### pdf_cdf table
	    $pdf_cdf_table_w configure -rows [expr $ii + 1]
	    
	    set pdf_cdf_vals($ii,label) [iwidgets::labeledwidget $pdf_cdf_table_w.pdf_cdf$ii \
					     -labeltext "$ii" \
					     -background gray50 \
					     -foreground white]
	    
	    set sisimpar(ncat) [expr $sisimpar(ncat) + 1]
	    
	    set sisimpar(cat,$ii) $ii
	    set cut_cat_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
					     -textvariable sisimpar(cat,$ii) \
					     -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
					     -focuscommand "focus_proc $ii"]
	    update
	    
	    set sisimpar(pdf,$ii) 0.15
	    set pdf_cdf_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
					     -textvariable sisimpar(pdf,$ii) \
					     -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
					     -focuscommand "focus_proc $ii"]
	    update
	    
	    $pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) -relief flat -background gray50
	    $pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
	    $pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
	    
	    set sisimpar(variables_initial) [expr $sisimpar(variables_initial) + 1]
	    set variables_initial_counter [expr $variables_initial_counter + 1]
	    
	    
	    ### model table
	    set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
					-orient vertical \
					-command "$model_sframe_w.table_$ii yview"]
	    
	    set no($ii) 1               ;# number of variogram types with corresponding parameters
	    set model_table_w($ii) [table $model_sframe_w.table_$ii \
					-variable model_vals \
					-titlerows 1 \
					-titlecols 0 \
					-cols 9 \
					-rows [expr $no($ii) + 1] \
					-yscrollcommand "$model_hsbar_w($ii) set" \
					-height 4]
	    
	    $model_table_w($ii) width 0 3
	    $model_table_w($ii) width 1 15
	    $model_table_w($ii) width 2 8
	    $model_table_w($ii) width 3 8
	    $model_table_w($ii) width 4 8
	    $model_table_w($ii) width 5 8
	    $model_table_w($ii) width 6 8
	    $model_table_w($ii) width 7 8
	    $model_table_w($ii) width 8 8
	    
	    set model_vals(0,0) "No"
	    set model_vals(0,1) "Type"
	    set model_vals(0,2) "Cc"
	    set model_vals(0,3) "Azimuth"
	    set model_vals(0,4) "Dip"
	    set model_vals(0,5) "Plunge"
	    set model_vals(0,6) "hmax"
	    set model_vals(0,7) "hmin"
	    set model_vals(0,8) "vert"
	    
	    
	    set sum_cc($ii) 0                ;# sum of all cc values
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		set no_counter($ii) $i       ;# counter of the number of variogram types with corresponding parameters
		
		set model_vals_label($i,$ii) [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
						  -labeltext "$i" \
						  -background gray50 \
						  -foreground white]
		
		set sisimpar(it$ii,$i) 0
		set model_type($i,combo$ii) [iwidgets::combobox $model_table_w($ii).combo_type$i \
						 -width 15 \
						 -textvariable sisimpar(it$ii,$i) \
						 -listheight 75]
		$model_table_w($ii).combo_type$i insert list end \
		    "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
		$model_type($i,combo$ii) selection set 0
		set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i 0]
		$model_table_w($ii).combo_type$i configure -state disabled -textbackground gray85 -foreground gray50
		
		set sisimpar(cc$ii,$i) 0.9
		set model_cc($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
					       -textvariable sisimpar(cc$ii,$i) \
					       -state disabled \
					       -textbackground gray85 \
					       -foreground gray50 \
					       -validate "check_number_cc %W %S %P $i $ii"]
		
		set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]
		
		set sisimpar(ang1$ii,$i) 0
		set model_ang1($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
						 -textvariable sisimpar(ang1$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang1 %W %P %c}]
		
		set sisimpar(ang2$ii,$i) 0
		set model_ang2($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang2$i \
						 -textvariable sisimpar(ang2$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set sisimpar(ang3$ii,$i) 0
		set model_ang3($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang3$i \
						 -textvariable sisimpar(ang3$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set sisimpar(aa_hmax$ii,$i) 10
		set model_aa_hmax($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
						    -textvariable sisimpar(aa_hmax$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set sisimpar(aa_hmin$ii,$i) 10
		set model_aa_hmin($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
						    -textvariable sisimpar(aa_hmin$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set sisimpar(aa_vert$ii,$i) 10
		set model_aa_vert($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
						    -textvariable sisimpar(aa_vert$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		$model_table_w($ii) window configure $i,0 \
		    -window $model_vals_label($i,$ii) -relief flat -background gray50
		$model_table_w($ii) window configure $i,1 -window $model_type($i,combo$ii)
		$model_table_w($ii) window configure $i,2 -window $model_cc($i,entry$ii)
		$model_table_w($ii) window configure $i,3 -window $model_ang1($i,entry$ii)
		$model_table_w($ii) window configure $i,4 -window $model_ang2($i,entry$ii)	
		$model_table_w($ii) window configure $i,5 -window $model_ang3($i,entry$ii)	
		$model_table_w($ii) window configure $i,6 -window $model_aa_hmax($i,entry$ii)	
		$model_table_w($ii) window configure $i,7 -window $model_aa_hmin($i,entry$ii)	
		$model_table_w($ii) window configure $i,8 -window $model_aa_vert($i,entry$ii)
	    }
	    pack $model_hsbar_w($ii)  -side right -anchor w -fill y
	    pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
	    
	    
	    ### create nugget entry and sill label
	    set sisimpar(c0$ii) 0.1
	    set models_nugget_w($ii) [iwidgets::entryfield $subframe3_variogram_w.entry_nugget$ii \
					  -labeltext "Nugget " \
					  -width 7 \
					  -textvariable sisimpar(c0$ii) \
					  -state disabled \
					  -textbackground gray85 \
					  -foreground gray50 \
					  -validate "check_number_nugget %W %P"]
	    
	    
	    set models_sill_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
					      -state disabled \
					      -labeltext "Sill "]
	    
	    set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
	    set models_sill_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
					      -state disabled \
					      -labelvariable sill($ii)]
	    update
	    
	    set cut_cat_number_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_label$ii \
						 -state disabled \
						 -labeltext "Cut/Cat \# "]
	    
	    set cut_cat_number($ii) $ii
	    set cut_cat_number_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_value$ii \
						 -state disabled \
						 -labeltext "$cut_cat_number($ii)"]
	    
	    
	    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	    grid $models_nugget_w($ii)     -row 0 -column 2 -sticky e -pady 0
	    grid $models_sill_label_w($ii) -row 0 -column 3 -sticky e -pady 0
	    grid $models_sill_value_w($ii) -row 0 -column 4 -sticky w -pady 0
	    grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	    grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	} else {
	    tk_messageBox \
		-title "Cutoffs / Categories" \
		-icon warning \
		-message "The maximum number of cutoffs or categories is limitted on [expr $variables_initial_counter]." \
		-type ok
	}
    }
    if { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat} && $focus_variable != 0 } {
	set max_hard_variables [expr $sisimpar(variables_initial) + 1]
	if { $sisimpar(variables_initial) <= 18 } {
	    
	    set ii [expr $variables_initial_counter + 1]
	    
	    ### pdf_cdf table
	    $pdf_cdf_table_w configure -rows [expr $ii + 1]
	    
	    set pdf_cdf_vals($ii,label) [iwidgets::labeledwidget $pdf_cdf_table_w.pdf_cdf$ii \
					     -labeltext "$ii" \
					     -background gray50 \
					     -foreground white]
	    
	    set sisimpar(ncat) [expr $sisimpar(ncat) + 1]
	    
	    set sisimpar(cat,$ii) $ii
	    set cut_cat_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
					     -textvariable sisimpar(cat,$ii) \
					     -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
					     -focuscommand "focus_proc $ii"]
	    update
	    
	    set sisimpar(pdf,$ii) 0.15
	    set pdf_cdf_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
					     -textvariable sisimpar(pdf,$ii) \
					     -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
					     -focuscommand "focus_proc $ii"]
	    update
	    
	    $pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) -relief flat -background gray50
	    $pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
	    $pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
	    
	    set sisimpar(variables_initial) [expr $sisimpar(variables_initial) + 1]
	    set variables_initial_counter [expr $variables_initial_counter + 1]
	    
	    
	    ### model table
	    set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
					-orient vertical \
					-command "$model_sframe_w.table_$ii yview"]
	    
	    set no($ii) 1               ;# number of variogram types with corresponding parameters
	    set model_table_w($ii) [table $model_sframe_w.table_$ii \
					-variable model_vals \
					-titlerows 1 \
					-titlecols 0 \
					-cols 9 \
					-rows [expr $no($ii) + 1] \
					-yscrollcommand "$model_hsbar_w($ii) set" \
					-height 4]
	    
	    $model_table_w($ii) width 0 3
	    $model_table_w($ii) width 1 15
	    $model_table_w($ii) width 2 8
	    $model_table_w($ii) width 3 8
	    $model_table_w($ii) width 4 8
	    $model_table_w($ii) width 5 8
	    $model_table_w($ii) width 6 8
	    $model_table_w($ii) width 7 8
	    $model_table_w($ii) width 8 8
	    
	    set model_vals(0,0) "No"
	    set model_vals(0,1) "Type"
	    set model_vals(0,2) "Cc"
	    set model_vals(0,3) "Azimuth"
	    set model_vals(0,4) "Dip"
	    set model_vals(0,5) "Plunge"
	    set model_vals(0,6) "hmax"
	    set model_vals(0,7) "hmin"
	    set model_vals(0,8) "vert"
	    
	    
	    set sum_cc($ii) 0                ;# sum of all cc values
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		set no_counter($ii) $i       ;# counter of the number of variogram types with corresponding parameters
		
		set model_vals_label($i,$ii) [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
						  -labeltext "$i" \
						  -background gray50 \
						  -foreground white]
		
		set sisimpar(it$ii,$i) 0
		set model_type($i,combo$ii) [iwidgets::combobox $model_table_w($ii).combo_type$i \
						 -width 15 \
						 -textvariable sisimpar(it$ii,$i) \
						 -listheight 75]
		$model_table_w($ii).combo_type$i insert list end \
		    "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
		$model_type($i,combo$ii) selection set 0
		set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i 0]
		$model_table_w($ii).combo_type$i configure -state disabled -textbackground gray85 -foreground gray50
		
		set sisimpar(cc$ii,$i) 0.9
		set model_cc($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
					       -textvariable sisimpar(cc$ii,$i) \
					       -state disabled \
					       -textbackground gray85 \
					       -foreground gray50 \
					       -validate "check_number_cc %W %S %P $i $ii"]
		
		set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]
		
		set sisimpar(ang1$ii,$i) 0
		set model_ang1($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
						 -textvariable sisimpar(ang1$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang1 %W %P %c}]
		
		set sisimpar(ang2$ii,$i) 0
		set model_ang2($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang2$i \
						 -textvariable sisimpar(ang2$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set sisimpar(ang3$ii,$i) 0
		set model_ang3($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang3$i \
						 -textvariable sisimpar(ang3$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set sisimpar(aa_hmax$ii,$i) 10
		set model_aa_hmax($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
						    -textvariable sisimpar(aa_hmax$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set sisimpar(aa_hmin$ii,$i) 10
		set model_aa_hmin($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
						    -textvariable sisimpar(aa_hmin$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set sisimpar(aa_vert$ii,$i) 10
		set model_aa_vert($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
						    -textvariable sisimpar(aa_vert$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		$model_table_w($ii) window configure $i,0 \
		    -window $model_vals_label($i,$ii) -relief flat -background gray50
		$model_table_w($ii) window configure $i,1 -window $model_type($i,combo$ii)
		$model_table_w($ii) window configure $i,2 -window $model_cc($i,entry$ii)
		$model_table_w($ii) window configure $i,3 -window $model_ang1($i,entry$ii)
		$model_table_w($ii) window configure $i,4 -window $model_ang2($i,entry$ii)	
		$model_table_w($ii) window configure $i,5 -window $model_ang3($i,entry$ii)	
		$model_table_w($ii) window configure $i,6 -window $model_aa_hmax($i,entry$ii)	
		$model_table_w($ii) window configure $i,7 -window $model_aa_hmin($i,entry$ii)	
		$model_table_w($ii) window configure $i,8 -window $model_aa_vert($i,entry$ii)
	    }
	    pack $model_hsbar_w($ii)  -side right -anchor w -fill y
	    pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
	    
	    
	    ### create nugget entry and sill label
	    set sisimpar(c0$ii) 0.1
	    set models_nugget_w($ii) [iwidgets::entryfield $subframe3_variogram_w.entry_nugget$ii \
					  -labeltext "Nugget " \
					  -width 7 \
					  -textvariable sisimpar(c0$ii) \
					  -state disabled \
					  -textbackground gray85 \
					  -foreground gray50 \
					  -validate "check_number_nugget %W %P"]
	    
	    
	    set models_sill_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
					      -state disabled \
					      -labeltext "Sill "]

	    set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
	    set models_sill_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
					      -state disabled \
					      -labelvariable sill($ii)]
	    update
	    
	    set cut_cat_number_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_label$ii \
						 -state disabled \
						 -labeltext "Cut/Cat \# "]
	    
	    set cut_cat_number($ii) $ii
	    set cut_cat_number_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_value$ii \
						 -state disabled \
						 -labeltext "$cut_cat_number($ii)"]
	    
	    
	    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	    grid $models_nugget_w($ii)     -row 0 -column 2 -sticky e -pady 0
	    grid $models_sill_label_w($ii) -row 0 -column 3 -sticky e -pady 0
	    grid $models_sill_value_w($ii) -row 0 -column 4 -sticky w -pady 0
	    grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	    grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	    
	    
	    for { set ii 1 } { $ii <= [expr $variables_initial_counter - 0] } { incr ii } {
		for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		    $model_type($i,combo$ii) configure -state normal -textbackground white -foreground black
		    $model_cc($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_ang1($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_ang2($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_ang3($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_aa_hmax($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_aa_hmin($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_aa_vert($i,entry$ii) configure -state normal -textbackground white -foreground black
		}
		$models_nugget_w($ii) configure -state normal -textbackground white -foreground black
		$models_sill_label_w($ii) configure -state normal
		$models_sill_value_w($ii) configure -state normal
		$cut_cat_number_label_w($ii) configure -state normal
		$cut_cat_number_value_w($ii) configure -state normal
		
		pack forget $model_hsbar_w($ii)
		pack forget $model_table_w($ii)
		grid forget $models_nugget_w($ii)
		grid forget $models_sill_label_w($ii)
		grid forget $models_sill_value_w($ii)
		grid forget $cut_cat_number_label_w($ii)
		grid forget $cut_cat_number_value_w($ii)
	    }
	    
	    pack $model_hsbar_w($focus_variable)  -side right -anchor w -fill y
	    pack $model_table_w($focus_variable)  -side left -anchor w -padx 2 -expand 1 -fill both
	    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	    grid $models_nugget_w($focus_variable)     -row 0 -column 2 -sticky e -pady 0
	    grid $models_sill_label_w($focus_variable) -row 0 -column 3 -sticky e -pady 0
	    grid $models_sill_value_w($focus_variable) -row 0 -column 4 -sticky w -pady 0
	    grid $cut_cat_number_label_w($focus_variable) -row 0 -column 5 -sticky e -pady 0
	    grid $cut_cat_number_value_w($focus_variable) -row 0 -column 6 -sticky w -pady 0
	} else {
	    tk_messageBox \
		-title "Cutoffs / Categories" \
		-icon warning \
		-message "The maximum number of cutoffs or categories is limitted on [expr $variables_initial_counter]." \
		-type ok
	}
    }
    if { $sisimpar(directik) == {*.dat} && $sisimpar(datafl) == {*.dat} } {
	tk_messageBox \
	    -title "Cutoffs / Categories" \
	    -icon warning \
	    -message "The maximum number of cutoffs or categories is dependent on the input data. \nYou should select a file first." \
	    -type ok
    }
    if { ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) == {*.dat} && $focus_variable == 0 ) || \
	     ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) != {*.dat} && $focus_variable == 0 ) } {
	if { $sisimpar(variables) <= 4 } {
	    tk_messageBox \
		-title "Cutoffs / Categories" \
		-icon warning \
		-message "At least two cutoffs or categories are needed. The soft data file has too few data." \
		-type ok
	} elseif { $sisimpar(variables) < $max_cut_cat } {

	    set ii [expr $variables_counter + 1]
	    
	    ### icoli table
	    $icoli_table_w configure -rows [expr $ii + 1]
	    
	    set icoli_vals($ii,label) [iwidgets::labeledwidget $icoli_table_w.label$ii \
					   -labeltext "$ii" \
					   -background gray50 \
					   -foreground white]
	    
	    set icoli_vals($ii,combo) [iwidgets::combobox $icoli_table_w.combo$ii \
					   -validate {regexpvalidate %c} \
					   -listheight 100 \
					   -focuscommand "focus_proc $ii" \
					   -selectioncommand "focus_proc $ii"]
	    update
	    
	    foreach i $dim_name_soft {
		$icoli_vals($ii,combo) insert list end $i
	    }
	    if { $sisimpar(variables) < $max_cut_cat } {
		$icoli_vals($ii,combo) selection set [expr $ii + 3]
		set sisimpar(icoli) [lindex $dim_name_soft [expr $ii + 3]]
	    } else {
		$icoli_vals($ii,combo) selection set end
		set sisimpar(icoli) [lindex $dim_name_soft end]
	    }
	    
	    set sisimpar(bz,$ii) 0;	# initialize to 0
	    set bz_vals($ii,entry) [iwidgets::entryfield $icoli_table_w.bz$ii \
					-textvariable sisimpar(bz,$ii) \
					-validate {check_number_bz_values %W %P %c} \
					-focuscommand "focus_proc $ii"]
	    update
	    
	    if { $sisimpar(imbsim) == 0 } {
		$icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
	    } elseif { $sisimpar(imbsim) == 1 } {
		$icoli_table_w.bz$ii configure -state normal -textbackground white -foreground black
	    }
	    
	    $icoli_table_w window configure $ii,0 -window $icoli_vals($ii,label) -relief flat -background gray50
	    $icoli_table_w window configure $ii,1 -window $icoli_vals($ii,combo)
	    $icoli_table_w window configure $ii,2 -window $bz_vals($ii,entry)
	    
	    
	    ### pdf_cdf table
	    $pdf_cdf_table_w configure -rows [expr $ii + 1]
	    
	    set pdf_cdf_vals($ii,label) [iwidgets::labeledwidget $pdf_cdf_table_w.pdf_cdf$ii \
					     -labeltext "$ii" \
					     -background gray50 \
					     -foreground white]
	    
	    set sisimpar(ncat) [expr $sisimpar(ncat) + 1]
	    
	    set sisimpar(cat,$ii) $ii
	    set cut_cat_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
					     -textvariable sisimpar(cat,$ii) \
					     -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
					     -focuscommand "focus_proc $ii"]
	    update
	    
	    set sisimpar(pdf,$ii) 0.15
	    set pdf_cdf_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
					     -textvariable sisimpar(pdf,$ii) \
					     -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
					     -focuscommand "focus_proc $ii"]
	    update
	    
	    $pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) -relief flat -background gray50
	    $pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
	    $pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
	    
	    set sisimpar(variables) [expr $sisimpar(variables) + 1]
	    set variables_counter [expr $variables_counter + 1]
	    
	    
	    ### model table
	    set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
					-orient vertical \
					-command "$model_sframe_w.table_$ii yview"]
	    
	    set no($ii) 1               ;# number of variogram types with corresponding parameters
	    set model_table_w($ii) [table $model_sframe_w.table_$ii \
					-variable model_vals \
					-titlerows 1 \
					-titlecols 0 \
					-cols 9 \
					-rows [expr $no($ii) + 1] \
					-yscrollcommand "$model_hsbar_w($ii) set" \
					-height 4]
	    
	    $model_table_w($ii) width 0 3
	    $model_table_w($ii) width 1 15
	    $model_table_w($ii) width 2 8
	    $model_table_w($ii) width 3 8
	    $model_table_w($ii) width 4 8
	    $model_table_w($ii) width 5 8
	    $model_table_w($ii) width 6 8
	    $model_table_w($ii) width 7 8
	    $model_table_w($ii) width 8 8
	    
	    set model_vals(0,0) "No"
	    set model_vals(0,1) "Type"
	    set model_vals(0,2) "Cc"
	    set model_vals(0,3) "Azimuth"
	    set model_vals(0,4) "Dip"
	    set model_vals(0,5) "Plunge"
	    set model_vals(0,6) "hmax"
	    set model_vals(0,7) "hmin"
	    set model_vals(0,8) "vert"
	    
	    
	    set sum_cc($ii) 0                ;# sum of all cc values
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		set no_counter($ii) $i       ;# counter of the number of variogram types with corresponding parameters
		
		set model_vals_label($i,$ii) [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
						  -labeltext "$i" \
						  -background gray50 \
						  -foreground white]
		
		set sisimpar(it$ii,$i) 0
		set model_type($i,combo$ii) [iwidgets::combobox $model_table_w($ii).combo_type$i \
						 -width 15 \
						 -textvariable sisimpar(it$ii,$i) \
						 -listheight 75]
		$model_table_w($ii).combo_type$i insert list end \
		    "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
		$model_type($i,combo$ii) selection set 0
		set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i 0]
		$model_table_w($ii).combo_type$i configure -state disabled -textbackground gray85 -foreground gray50
		
		set sisimpar(cc$ii,$i) 0.9
		set model_cc($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
					       -textvariable sisimpar(cc$ii,$i) \
					       -state disabled \
					       -textbackground gray85 \
					       -foreground gray50 \
					       -validate "check_number_cc %W %S %P $i %ii"]
		
		set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]		    		    
		
		set sisimpar(ang1$ii,$i) 0
		set model_ang1($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
						 -textvariable sisimpar(ang1$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang1 %W %P %c}]
		
		set sisimpar(ang2$ii,$i) 0
		set model_ang2($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang2$i \
						 -textvariable sisimpar(ang2$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set sisimpar(ang3$ii,$i) 0
		set model_ang3($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang3$i \
						 -textvariable sisimpar(ang3$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set sisimpar(aa_hmax$ii,$i) 10
		set model_aa_hmax($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
						    -textvariable sisimpar(aa_hmax$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set sisimpar(aa_hmin$ii,$i) 10
		set model_aa_hmin($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
						    -textvariable sisimpar(aa_hmin$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set sisimpar(aa_vert$ii,$i) 10
		set model_aa_vert($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
						    -textvariable sisimpar(aa_vert$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		$model_table_w($ii) window configure $i,0 \
		    -window $model_vals_label($i,$ii) -relief flat -background gray50
		$model_table_w($ii) window configure $i,1 -window $model_type($i,combo$ii)
		$model_table_w($ii) window configure $i,2 -window $model_cc($i,entry$ii)
		$model_table_w($ii) window configure $i,3 -window $model_ang1($i,entry$ii)
		$model_table_w($ii) window configure $i,4 -window $model_ang2($i,entry$ii)	
		$model_table_w($ii) window configure $i,5 -window $model_ang3($i,entry$ii)	
		$model_table_w($ii) window configure $i,6 -window $model_aa_hmax($i,entry$ii)	
		$model_table_w($ii) window configure $i,7 -window $model_aa_hmin($i,entry$ii)	
		$model_table_w($ii) window configure $i,8 -window $model_aa_vert($i,entry$ii)
	    }
	    pack $model_hsbar_w($ii)  -side right -anchor w -fill y
	    pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
	    
	    
	    ### create nugget entry and sill label
	    set sisimpar(c0$ii) 0.1
	    set models_nugget_w($ii) [iwidgets::entryfield $subframe3_variogram_w.entry_nugget$ii \
					  -labeltext "Nugget " \
					  -width 7 \
					  -textvariable sisimpar(c0$ii) \
					  -state disabled \
					  -textbackground gray85 \
					  -foreground gray50 \
					  -validate "check_number_nugget %W %P"]
	    
	    
	    set models_sill_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
					      -state disabled \
					      -labeltext "Sill "]
	    
	    set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
	    set models_sill_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
					      -state disabled \
					      -labelvariable sill($ii)]
	    
	    
	    set cut_cat_number_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_label$ii \
						 -state disabled \
						 -labeltext "Cut/Cat \# "]
	    
	    set cut_cat_number($ii) $ii
	    set cut_cat_number_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_value$ii \
						 -state disabled \
						 -labeltext "$cut_cat_number($ii)"]
	    
	    
	    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	    grid $models_nugget_w($ii)     -row 0 -column 2 -sticky e -pady 0
	    grid $models_sill_label_w($ii) -row 0 -column 3 -sticky e -pady 0
	    grid $models_sill_value_w($ii) -row 0 -column 4 -sticky w -pady 0
	    grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	    grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0

	} else {
	    tk_messageBox \
		-title "Cutoffs / Categories" \
		-icon warning \
		-message "The maximum number of cutoffs or categories is dependent on the input data. \nIn this case only [expr $max_cut_cat - 3] cutoffs or categories are permitted for the analysis of the heterogeneity." \
		-type ok
	}
    }
    if { ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) == {*.dat} && $focus_variable != 0 ) || \
	     ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) != {*.dat} && $focus_variable != 0 ) } {
	if { $sisimpar(variables) <= 4 } {
	    tk_messageBox \
		-title "Cutoffs / Categories" \
		-icon warning \
		-message "At least two cutoffs or categories are needed. The soft data file has too few data." \
		-type ok
	} elseif { $sisimpar(variables) < $max_cut_cat } {
	    set ii [expr $variables_counter + 1]
	    
	    ### icoli table
	    $icoli_table_w configure -rows [expr $ii + 1]
	    
	    set icoli_vals($ii,label) [iwidgets::labeledwidget $icoli_table_w.label$ii \
					   -labeltext "$ii" \
					   -background gray50 \
					   -foreground white]
	    
	    set icoli_vals($ii,combo) [iwidgets::combobox $icoli_table_w.combo$ii \
					   -validate {regexpvalidate %c} \
					   -listheight 100 \
					   -focuscommand "focus_proc $ii" \
					   -selectioncommand "focus_proc $ii"]
	    update
	    
	    foreach i $dim_name_soft {
		$icoli_vals($ii,combo) insert list end $i
	    }
	    if { $sisimpar(variables) < $max_cut_cat } {
		$icoli_vals($ii,combo) selection set [expr $ii + 3]
		set sisimpar(icoli) [lindex $dim_name_soft [expr $ii + 3]]
	    } else {
		$icoli_vals($ii,combo) selection set end
		set sisimpar(icoli) [lindex $dim_name_soft end]
	    }
	    
	    set sisimpar(bz,$ii) 0;	# initialize to 0
	    set bz_vals($ii,entry) [iwidgets::entryfield $icoli_table_w.bz$ii \
					-textvariable sisimpar(bz,$ii) \
					-validate {check_number_bz_values %W %P %c} \
					-focuscommand "focus_proc $ii"]
	    update
	    
	    if { $sisimpar(imbsim) == 0 } {
		$icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
	    } elseif { $sisimpar(imbsim) == 1 } {
		$icoli_table_w.bz$ii configure -state normal -textbackground white -foreground black
	    }
	    
	    $icoli_table_w window configure $ii,0 -window $icoli_vals($ii,label) -relief flat -background gray50
	    $icoli_table_w window configure $ii,1 -window $icoli_vals($ii,combo)
	    $icoli_table_w window configure $ii,2 -window $bz_vals($ii,entry)
	    
	    
	    ### pdf_cdf table
	    $pdf_cdf_table_w configure -rows [expr $ii + 1]
	    
	    set pdf_cdf_vals($ii,label) [iwidgets::labeledwidget $pdf_cdf_table_w.pdf_cdf$ii \
					     -labeltext "$ii" \
					     -background gray50 \
					     -foreground white]
	    
	    set sisimpar(ncat) [expr $sisimpar(ncat) + 1]
	    
	    set sisimpar(cat,$ii) $ii
	    set cut_cat_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
					     -textvariable sisimpar(cat,$ii) \
					     -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
					     -focuscommand "focus_proc $ii"]
	    update
	    
	    set sisimpar(pdf,$ii) 0.15
	    set pdf_cdf_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
					     -textvariable sisimpar(pdf,$ii) \
					     -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
					     -focuscommand "focus_proc $ii"]
	    update
	    
	    $pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) -relief flat -background gray50
	    $pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
	    $pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
	    
	    set sisimpar(variables) [expr $sisimpar(variables) + 1]
	    set variables_counter [expr $variables_counter + 1]
	    
	    
	    ### model table
	    set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
					-orient vertical \
					-command "$model_sframe_w.table_$ii yview"]
	    
	    set no($ii) 1               ;# number of variogram types with corresponding parameters
	    set model_table_w($ii) [table $model_sframe_w.table_$ii \
					-variable model_vals \
					-titlerows 1 \
					-titlecols 0 \
					-cols 9 \
					-rows [expr $no($ii) + 1] \
					-yscrollcommand "$model_hsbar_w($ii) set" \
					-height 4]
	    
	    $model_table_w($ii) width 0 3
	    $model_table_w($ii) width 1 15
	    $model_table_w($ii) width 2 8
	    $model_table_w($ii) width 3 8
	    $model_table_w($ii) width 4 8
	    $model_table_w($ii) width 5 8
	    $model_table_w($ii) width 6 8
	    $model_table_w($ii) width 7 8
	    $model_table_w($ii) width 8 8
	    
	    set model_vals(0,0) "No"
	    set model_vals(0,1) "Type"
	    set model_vals(0,2) "Cc"
	    set model_vals(0,3) "Azimuth"
	    set model_vals(0,4) "Dip"
	    set model_vals(0,5) "Plunge"
	    set model_vals(0,6) "hmax"
	    set model_vals(0,7) "hmin"
	    set model_vals(0,8) "vert"
	    
	    
	    set sum_cc($ii) 0                ;# sum of all cc values
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		set no_counter($ii) $i       ;# counter of the number of variogram types with corresponding parameters
		
		set model_vals_label($i,$ii) [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
						  -labeltext "$i" \
						  -background gray50 \
						  -foreground white]
		
		set sisimpar(it$ii,$i) 0
		set model_type($i,combo$ii) [iwidgets::combobox $model_table_w($ii).combo_type$i \
						 -width 15 \
						 -textvariable sisimpar(it$ii,$i) \
						 -listheight 75]
		$model_table_w($ii).combo_type$i insert list end \
		    "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
		$model_type($i,combo$ii) selection set 0
		set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i 0]
		$model_table_w($ii).combo_type$i configure -state disabled -textbackground gray85 -foreground gray50
		
		set sisimpar(cc$ii,$i) 0.9
		set model_cc($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
					       -textvariable sisimpar(cc$ii,$i) \
					       -state disabled \
					       -textbackground gray85 \
					       -foreground gray50 \
					       -validate "check_number_cc %W %S %P $i %ii"]
		
		set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]		    		    
		
		set sisimpar(ang1$ii,$i) 0
		set model_ang1($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
						 -textvariable sisimpar(ang1$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang1 %W %P %c}]
		
		set sisimpar(ang2$ii,$i) 0
		set model_ang2($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang2$i \
						 -textvariable sisimpar(ang2$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set sisimpar(ang3$ii,$i) 0
		set model_ang3($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang3$i \
						 -textvariable sisimpar(ang3$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set sisimpar(aa_hmax$ii,$i) 10
		set model_aa_hmax($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
						    -textvariable sisimpar(aa_hmax$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set sisimpar(aa_hmin$ii,$i) 10
		set model_aa_hmin($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
						    -textvariable sisimpar(aa_hmin$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set sisimpar(aa_vert$ii,$i) 10
		set model_aa_vert($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
						    -textvariable sisimpar(aa_vert$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		$model_table_w($ii) window configure $i,0 \
		    -window $model_vals_label($i,$ii) -relief flat -background gray50
		$model_table_w($ii) window configure $i,1 -window $model_type($i,combo$ii)
		$model_table_w($ii) window configure $i,2 -window $model_cc($i,entry$ii)
		$model_table_w($ii) window configure $i,3 -window $model_ang1($i,entry$ii)
		$model_table_w($ii) window configure $i,4 -window $model_ang2($i,entry$ii)	
		$model_table_w($ii) window configure $i,5 -window $model_ang3($i,entry$ii)	
		$model_table_w($ii) window configure $i,6 -window $model_aa_hmax($i,entry$ii)	
		$model_table_w($ii) window configure $i,7 -window $model_aa_hmin($i,entry$ii)	
		$model_table_w($ii) window configure $i,8 -window $model_aa_vert($i,entry$ii)
	    }
	    pack $model_hsbar_w($ii)  -side right -anchor w -fill y
	    pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
	    
	    
	    ### create nugget entry and sill label
	    set sisimpar(c0$ii) 0.1
	    set models_nugget_w($ii) [iwidgets::entryfield $subframe3_variogram_w.entry_nugget$ii \
					  -labeltext "Nugget " \
					  -width 7 \
					  -textvariable sisimpar(c0$ii) \
					  -state disabled \
					  -textbackground gray85 \
					  -foreground gray50 \
					  -validate "check_number_nugget %W %P"]
	    
	    
	    set models_sill_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
					      -state disabled \
					      -labeltext "Sill "]
	    
	    set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
	    set models_sill_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
					      -state disabled \
					      -labelvariable sill($ii)]
	    
	    
	    set cut_cat_number_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_label$ii \
						 -state disabled \
						 -labeltext "Cut/Cat \# "]
	    
	    set cut_cat_number($ii) $ii
	    set cut_cat_number_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_value$ii \
						 -state disabled \
						 -labeltext "$cut_cat_number($ii)"]
	    
	    
	    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	    grid $models_nugget_w($ii)     -row 0 -column 2 -sticky e -pady 0
	    grid $models_sill_label_w($ii) -row 0 -column 3 -sticky e -pady 0
	    grid $models_sill_value_w($ii) -row 0 -column 4 -sticky w -pady 0
	    grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	    grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	    
	    
	    for { set ii 1 } { $ii <= [expr $variables_counter - 0] } { incr ii } {
		for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		    $model_type($i,combo$ii) configure -state normal -textbackground white -foreground black
		    $model_cc($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_ang1($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_ang2($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_ang3($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_aa_hmax($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_aa_hmin($i,entry$ii) configure -state normal -textbackground white -foreground black
		    $model_aa_vert($i,entry$ii) configure -state normal -textbackground white -foreground black
		}
		$models_nugget_w($ii) configure -state normal -textbackground white -foreground black
		$models_sill_label_w($ii) configure -state normal
		$models_sill_value_w($ii) configure -state normal
		$cut_cat_number_label_w($ii) configure -state normal
		$cut_cat_number_value_w($ii) configure -state normal
		
		pack forget $model_hsbar_w($ii)
		pack forget $model_table_w($ii)
		grid forget $models_nugget_w($ii)
		grid forget $models_sill_label_w($ii)
		grid forget $models_sill_value_w($ii)
		grid forget $cut_cat_number_label_w($ii)
		grid forget $cut_cat_number_value_w($ii)
	    }
	    
	    pack $model_hsbar_w($focus_variable)  -side right -anchor w -fill y
	    pack $model_table_w($focus_variable)  -side left -anchor w -padx 2 -expand 1 -fill both
	    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	    grid $models_nugget_w($focus_variable)     -row 0 -column 2 -sticky e -pady 0
	    grid $models_sill_label_w($focus_variable) -row 0 -column 3 -sticky e -pady 0
	    grid $models_sill_value_w($focus_variable) -row 0 -column 4 -sticky w -pady 0
	    grid $cut_cat_number_label_w($focus_variable) -row 0 -column 5 -sticky e -pady 0
	    grid $cut_cat_number_value_w($focus_variable) -row 0 -column 6 -sticky w -pady 0
	    
	} else {
	    tk_messageBox \
		-title "Cutoffs / Categories" \
		-icon warning \
		-message "The maximum number of cutoffs or categories is dependent on the input data. \nIn this case only [expr $max_cut_cat - 3] cutoffs or categories are permitted for the analysis of the heterogeneity." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc minus_cutoff_category { } {
    global sisimpar
    global icoli_table_w
    global pdf_cdf_table_w
    global icoli_vals
    global bz_vals
    global cut_cat_vals
    global pdf_cdf_vals
    global logwin
    global variables_initial_counter
    global variables_counter
    global model_table_w
    global model_hsbar_w
    global model_vals_label
    global model_type
    global model_cc
    global model_ang1
    global model_ang2
    global model_ang3
    global model_aa_hmax
    global model_aa_hmin
    global model_aa_vert
    global models_nugget_w
    global models_sill_label_w
    global models_sill_value_w
    global cut_cat_number_label_w
    global cut_cat_number_value_w
    global cut_cat_number
    global subframe3_variogram_w
    global plus_model_pushbutton_w
    global minus_model_pushbutton_w
    global no
    global focus_variable
    
    if { $sisimpar(directik) == {*.dat} && $sisimpar(datafl) == {*.dat} } {
	tk_messageBox \
	    -title "Cutoffs / Categories" \
	    -icon warning \
	    -message "The minimal number of cutoffs or categories is dependent on the input data. \nYou should select a file first." \
	    -type ok
    }
    if { $sisimpar(directik) == {*.dat} && $sisimpar(datafl) != {*.dat} && $focus_variable == 0 } {
	if {$sisimpar(variables_initial) > 6} {
	    
	    set sisimpar(ncat) [expr $sisimpar(ncat) - 1]
	    set ii $variables_initial_counter
	    
	    destroy $pdf_cdf_vals($ii,label)
	    destroy $cut_cat_vals($ii,entry)
	    destroy $pdf_cdf_vals($ii,entry)
	    $pdf_cdf_table_w configure -rows $ii
	    
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		destroy $model_type($i,combo$ii)
		destroy $model_cc($i,entry$ii)
		destroy $model_ang1($i,entry$ii)
		destroy $model_ang2($i,entry$ii)
		destroy $model_ang3($i,entry$ii)
		destroy $model_aa_hmax($i,entry$ii)
		destroy $model_aa_hmin($i,entry$ii)
		destroy $model_aa_vert($i,entry$ii)
	    }
	    destroy $model_table_w($ii)
	    destroy $model_hsbar_w($ii)
	    
	    destroy $models_nugget_w($ii)
	    destroy $models_sill_label_w($ii)
	    destroy $models_sill_value_w($ii)
	    destroy $cut_cat_number_label_w($ii)
	    destroy $cut_cat_number_value_w($ii)
	    
	    set variables_initial_counter [expr $variables_initial_counter - 1]
	    set sisimpar(variables_initial) [expr $sisimpar(variables_initial) - 1]
	    
	} else {
	    tk_messageBox \
		-title "Cutoffs / Categories" \
		-icon warning \
		-message "At least 2 cutoffs or categories are needed." \
		-type ok
	}
    }
    if { $sisimpar(directik) == {*.dat} && $sisimpar(datafl) != {*.dat} && $focus_variable != 0 } {
	if {$sisimpar(variables_initial) > 6} {
	    
	    set sisimpar(ncat) [expr $sisimpar(ncat) - 1]
	    set ii $variables_initial_counter
	    
	    destroy $pdf_cdf_vals($ii,label)
	    destroy $cut_cat_vals($ii,entry)
	    destroy $pdf_cdf_vals($ii,entry)
	    $pdf_cdf_table_w configure -rows $ii
	    
	    
	    for { set ii 1 } { $ii <= [expr $variables_initial_counter - 1] } { incr ii } {
		for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		    $model_type($i,combo$ii) configure -state normal
		    $model_cc($i,entry$ii) configure -state normal
		    $model_ang1($i,entry$ii) configure -state normal
		    $model_ang2($i,entry$ii) configure -state normal
		    $model_ang3($i,entry$ii) configure -state normal
		    $model_aa_hmax($i,entry$ii) configure -state normal
		    $model_aa_hmin($i,entry$ii) configure -state normal
		    $model_aa_vert($i,entry$ii) configure -state normal
		}
		$models_nugget_w($ii) configure -state normal
		$models_sill_label_w($ii) configure -state normal
		$models_sill_value_w($ii) configure -state normal
		$cut_cat_number_label_w($ii) configure -state normal
		$cut_cat_number_value_w($ii) configure -state normal
	    }
	    
	    
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		destroy $model_type($i,combo$ii)
		destroy $model_cc($i,entry$ii)
		destroy $model_ang1($i,entry$ii)
		destroy $model_ang2($i,entry$ii)
		destroy $model_ang3($i,entry$ii)
		destroy $model_aa_hmax($i,entry$ii)
		destroy $model_aa_hmin($i,entry$ii)
		destroy $model_aa_vert($i,entry$ii)
	    }
	    destroy $model_table_w($ii)
	    destroy $model_hsbar_w($ii)
	    
	    destroy $models_nugget_w($ii)
	    destroy $models_sill_label_w($ii)
	    destroy $models_sill_value_w($ii)
	    destroy $cut_cat_number_label_w($ii)
	    destroy $cut_cat_number_value_w($ii)
	    
	    
	    if { $variables_initial_counter > $focus_variable } {
		pack configure $model_hsbar_w($focus_variable) -side right -anchor w -fill y
		pack configure $model_table_w($focus_variable) -side left -anchor w -padx 2 -expand 1 -fill both
		
		for { set i 1 } { $i <= [expr $no($focus_variable) - 0] } { incr i } {
		    $model_table_w($focus_variable) window configure $i,0 \
			-window $model_vals_label($i,$focus_variable) -relief flat -background gray50
		    $model_table_w($focus_variable) window configure $i,1 \
			-window $model_type($i,combo$focus_variable)
		    $model_table_w($focus_variable) window configure $i,2 \
			-window $model_cc($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,3 \
			-window $model_ang1($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,4 \
			-window $model_ang2($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,5 \
			-window $model_ang3($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,6 \
			-window $model_aa_hmax($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,7 \
			-window $model_aa_hmin($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,8 \
			-window $model_aa_vert($i,entry$focus_variable)
		}
		
		grid columnconfigure $subframe3_variogram_w 2 -minsize 121
		grid columnconfigure $subframe3_variogram_w 3 -minsize 47
		grid columnconfigure $subframe3_variogram_w 4 -minsize 60
		grid columnconfigure $subframe3_variogram_w 5 -minsize 180
		grid columnconfigure $subframe3_variogram_w 6 -minsize 10
		grid $models_nugget_w($focus_variable)     -row 0 -column 2 -sticky e -pady 0
		grid $models_sill_label_w($focus_variable) -row 0 -column 3 -sticky e -pady 0
		grid $models_sill_value_w($focus_variable) -row 0 -column 4 -sticky w -pady 0
		grid $cut_cat_number_label_w($focus_variable) -row 0 -column 5 -sticky e -pady 0
		grid $cut_cat_number_value_w($focus_variable) -row 0 -column 6 -sticky w -pady 0
		
	    } elseif { $variables_initial_counter == $focus_variable } {
		pack configure $model_hsbar_w([expr $variables_initial_counter - 1]) -side right -anchor w -fill y
		pack configure $model_table_w([expr $variables_initial_counter - 1]) -side left -anchor w \
		    -padx 2 -expand 1 -fill both
		
		for { set i 1 } { $i <= [expr $no([expr $variables_initial_counter - 1]) - 0] } { incr i } {
		    $model_table_w([expr $variables_initial_counter - 1]) window configure $i,0 \
			-window $model_vals_label($i,[expr $variables_initial_counter - 1]) \
			    -relief flat -background gray50
			$model_table_w([expr $variables_initial_counter - 1]) window configure $i,1 \
			-window $model_type($i,combo[expr $variables_initial_counter - 1])
		    $model_table_w([expr $variables_initial_counter - 1]) window configure $i,2 \
			-window $model_cc($i,entry[expr $variables_initial_counter - 1])
		    $model_table_w([expr $variables_initial_counter - 1]) window configure $i,3 \
			-window $model_ang1($i,entry[expr $variables_initial_counter - 1])
		    $model_table_w([expr $variables_initial_counter - 1]) window configure $i,4 \
			-window $model_ang2($i,entry[expr $variables_initial_counter - 1])
		    $model_table_w([expr $variables_initial_counter - 1]) window configure $i,5 \
			-window $model_ang3($i,entry[expr $variables_initial_counter - 1])
		    $model_table_w([expr $variables_initial_counter - 1]) window configure $i,6 \
			-window $model_aa_hmax($i,entry[expr $variables_initial_counter - 1])
		    $model_table_w([expr $variables_initial_counter - 1]) window configure $i,7 \
			-window $model_aa_hmin($i,entry[expr $variables_initial_counter - 1])
		    $model_table_w([expr $variables_initial_counter - 1]) window configure $i,8 \
			    -window $model_aa_vert($i,entry[expr $variables_initial_counter - 1])
		}
		
		grid columnconfigure $subframe3_variogram_w 2 -minsize 121
		grid columnconfigure $subframe3_variogram_w 3 -minsize 47
		grid columnconfigure $subframe3_variogram_w 4 -minsize 60
		grid columnconfigure $subframe3_variogram_w 5 -minsize 180
		grid columnconfigure $subframe3_variogram_w 6 -minsize 10
		grid $models_nugget_w([expr $variables_initial_counter - 1])     -row 0 -column 2 -sticky e -pady 0
		grid $models_sill_label_w([expr $variables_initial_counter - 1]) -row 0 -column 3 -sticky e -pady 0
		grid $models_sill_value_w([expr $variables_initial_counter - 1]) -row 0 -column 4 -sticky w -pady 0
		grid $cut_cat_number_label_w([expr $variables_initial_counter - 1]) -row 0 -column 5 -sticky e -pady 0
		grid $cut_cat_number_value_w([expr $variables_initial_counter - 1]) -row 0 -column 6 -sticky w -pady 0
		
		$pdf_cdf_vals([expr $variables_initial_counter - 1],label) configure \
		    -background yellow -foreground black
		$pdf_cdf_table_w window configure [expr $variables_initial_counter - 1],0 \
		    -relief flat -background yellow
		
		set focus_variable [expr $focus_variable - 1]
	    }
	    set variables_initial_counter [expr $variables_initial_counter - 1]
	    set sisimpar(variables_initial) [expr $sisimpar(variables_initial) - 1]
	    
	} else {
	    tk_messageBox \
		-title "Cutoffs / Categories" \
		-icon warning \
		-message "At least 2 cutoffs or categories are needed." \
		-type ok
	}
    }
    if { ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) != {*.dat} && $focus_variable == 0 ) || \
	     ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) == {*.dat} && $focus_variable == 0 ) } {
	if {$sisimpar(variables) > 5} {
	    
	    set sisimpar(ncat) [expr $sisimpar(ncat) - 1]
	    set ii $variables_counter
	    
	    destroy $icoli_vals($ii,label)
	    destroy $icoli_vals($ii,combo)
	    destroy $bz_vals($ii,entry)
	    $icoli_table_w configure -rows $ii
	    
	    destroy $pdf_cdf_vals($ii,label)
	    destroy $cut_cat_vals($ii,entry)
	    destroy $pdf_cdf_vals($ii,entry)
	    $pdf_cdf_table_w configure -rows $ii
	    
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		destroy $model_type($i,combo$ii)
		destroy $model_cc($i,entry$ii)
		destroy $model_ang1($i,entry$ii)
		destroy $model_ang2($i,entry$ii)
		destroy $model_ang3($i,entry$ii)
		destroy $model_aa_hmax($i,entry$ii)
		destroy $model_aa_hmin($i,entry$ii)
		destroy $model_aa_vert($i,entry$ii)
	    }
	    destroy $model_table_w($ii)
	    destroy $model_hsbar_w($ii)
	    
	    destroy $models_nugget_w($ii)
	    destroy $models_sill_label_w($ii)
	    destroy $models_sill_value_w($ii)
	    destroy $cut_cat_number_label_w($ii)
	    destroy $cut_cat_number_value_w($ii)
	    
	    set variables_counter [expr $variables_counter - 1]
	    set sisimpar(variables) [expr $sisimpar(variables) - 1]

	} elseif { $sisimpar(variables) <= 5 } {
	    if { $variables_counter >= 2 } {
		tk_messageBox \
		    -title "Cutoffs / Categories" \
		    -icon warning \
		    -message "At least two cutoffs or categories are needed." \
		    -type ok
	    } elseif { $variables_counter < 2 } {
		tk_messageBox \
		    -title "Cutoffs / Categories" \
		    -icon warning \
		    -message "At least two cutoffs or categories are needed. The soft data file has too few data." \
		    -type ok
	    }
	}
    }
    if { ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) != {*.dat} && $focus_variable != 0 ) || \
	     ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) == {*.dat} && $focus_variable != 0 ) } {
	if {$sisimpar(variables) > 5} {
	    
	    set sisimpar(ncat) [expr $sisimpar(ncat) - 1]
	    set ii $variables_counter
	    
	    destroy $icoli_vals($ii,label)
	    destroy $icoli_vals($ii,combo)
	    destroy $bz_vals($ii,entry)
	    $icoli_table_w configure -rows $ii
	    
	    destroy $pdf_cdf_vals($ii,label)
	    destroy $cut_cat_vals($ii,entry)
	    destroy $pdf_cdf_vals($ii,entry)
	    $pdf_cdf_table_w configure -rows $ii
	    
	    
	    for { set ii 1 } { $ii <= [expr $variables_counter - 1] } { incr ii } {
		for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		    $model_type($i,combo$ii) configure -state normal
		    $model_cc($i,entry$ii) configure -state normal
		    $model_ang1($i,entry$ii) configure -state normal
		    $model_ang2($i,entry$ii) configure -state normal
		    $model_ang3($i,entry$ii) configure -state normal
		    $model_aa_hmax($i,entry$ii) configure -state normal
		    $model_aa_hmin($i,entry$ii) configure -state normal
		    $model_aa_vert($i,entry$ii) configure -state normal
		}
		$models_nugget_w($ii) configure -state normal
		$models_sill_label_w($ii) configure -state normal
		$models_sill_value_w($ii) configure -state normal
		$cut_cat_number_label_w($ii) configure -state normal
		$cut_cat_number_value_w($ii) configure -state normal
	    }
	    
	    
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		destroy $model_type($i,combo$ii)
		destroy $model_cc($i,entry$ii)
		destroy $model_ang1($i,entry$ii)
		destroy $model_ang2($i,entry$ii)
		destroy $model_ang3($i,entry$ii)
		destroy $model_aa_hmax($i,entry$ii)
		destroy $model_aa_hmin($i,entry$ii)
		destroy $model_aa_vert($i,entry$ii)
	    }
	    destroy $model_table_w($ii)
	    destroy $model_hsbar_w($ii)
	    
	    destroy $models_nugget_w($ii)
	    destroy $models_sill_label_w($ii)
	    destroy $models_sill_value_w($ii)
	    destroy $cut_cat_number_label_w($ii)
	    destroy $cut_cat_number_value_w($ii)
	    
	    
	    if { $variables_counter > $focus_variable } {
		pack configure $model_hsbar_w($focus_variable) -side right -anchor w -fill y
		pack configure $model_table_w($focus_variable) -side left -anchor w -padx 2 -expand 1 -fill both
		
		for { set i 1 } { $i <= [expr $no($focus_variable) - 0] } { incr i } {
		    $model_table_w($focus_variable) window configure $i,0 \
			-window $model_vals_label($i,$focus_variable) -relief flat -background gray50
		    $model_table_w($focus_variable) window configure $i,1 \
			-window $model_type($i,combo$focus_variable)
		    $model_table_w($focus_variable) window configure $i,2 \
			-window $model_cc($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,3 \
			-window $model_ang1($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,4 \
			-window $model_ang2($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,5 \
			-window $model_ang3($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,6 \
			-window $model_aa_hmax($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,7 \
			-window $model_aa_hmin($i,entry$focus_variable)
		    $model_table_w($focus_variable) window configure $i,8 \
			-window $model_aa_vert($i,entry$focus_variable)
		}
		
		grid columnconfigure $subframe3_variogram_w 2 -minsize 121
		grid columnconfigure $subframe3_variogram_w 3 -minsize 47
		grid columnconfigure $subframe3_variogram_w 4 -minsize 60
		grid columnconfigure $subframe3_variogram_w 5 -minsize 180
		grid columnconfigure $subframe3_variogram_w 6 -minsize 10
		grid $models_nugget_w($focus_variable)     -row 0 -column 2 -sticky e -pady 0
		grid $models_sill_label_w($focus_variable) -row 0 -column 3 -sticky e -pady 0
		grid $models_sill_value_w($focus_variable) -row 0 -column 4 -sticky w -pady 0
		grid $cut_cat_number_label_w($focus_variable) -row 0 -column 5 -sticky e -pady 0
		grid $cut_cat_number_value_w($focus_variable) -row 0 -column 6 -sticky w -pady 0
		
	    } elseif { $variables_counter == $focus_variable } {
		pack configure $model_hsbar_w([expr $variables_counter - 1]) -side right -anchor w -fill y
		pack configure $model_table_w([expr $variables_counter - 1]) -side left -anchor w \
		    -padx 2 -expand 1 -fill both
		
		for { set i 1 } { $i <= [expr $no([expr $variables_counter - 1]) - 0] } { incr i } {
		    $model_table_w([expr $variables_counter - 1]) window configure $i,0 \
			-window $model_vals_label($i,[expr $variables_counter - 1]) \
			-relief flat -background gray50
		    $model_table_w([expr $variables_counter - 1]) window configure $i,1 \
			-window $model_type($i,combo[expr $variables_counter - 1])
		    $model_table_w([expr $variables_counter - 1]) window configure $i,2 \
			-window $model_cc($i,entry[expr $variables_counter - 1])
		    $model_table_w([expr $variables_counter - 1]) window configure $i,3 \
			-window $model_ang1($i,entry[expr $variables_counter - 1])
		    $model_table_w([expr $variables_counter - 1]) window configure $i,4 \
			-window $model_ang2($i,entry[expr $variables_counter - 1])
		    $model_table_w([expr $variables_counter - 1]) window configure $i,5 \
			-window $model_ang3($i,entry[expr $variables_counter - 1])
		    $model_table_w([expr $variables_counter - 1]) window configure $i,6 \
			-window $model_aa_hmax($i,entry[expr $variables_counter - 1])
		    $model_table_w([expr $variables_counter - 1]) window configure $i,7 \
			-window $model_aa_hmin($i,entry[expr $variables_counter - 1])
		    $model_table_w([expr $variables_counter - 1]) window configure $i,8 \
			-window $model_aa_vert($i,entry[expr $variables_counter - 1])
		}
		
		grid columnconfigure $subframe3_variogram_w 2 -minsize 121
		grid columnconfigure $subframe3_variogram_w 3 -minsize 47
		grid columnconfigure $subframe3_variogram_w 4 -minsize 60
		grid columnconfigure $subframe3_variogram_w 5 -minsize 180
		grid columnconfigure $subframe3_variogram_w 6 -minsize 10
		grid $models_nugget_w([expr $variables_counter - 1])     -row 0 -column 2 -sticky e -pady 0
		grid $models_sill_label_w([expr $variables_counter - 1]) -row 0 -column 3 -sticky e -pady 0
		grid $models_sill_value_w([expr $variables_counter - 1]) -row 0 -column 4 -sticky w -pady 0
		grid $cut_cat_number_label_w([expr $variables_counter - 1]) -row 0 -column 5 -sticky e -pady 0
		grid $cut_cat_number_value_w([expr $variables_counter - 1]) -row 0 -column 6 -sticky w -pady 0
		
		$icoli_vals([expr $variables_counter - 1],label) configure \
		    -background yellow -foreground black
		$icoli_table_w window configure [expr $variables_counter - 1],0 \
		    -relief flat -background yellow
		$pdf_cdf_vals([expr $variables_counter - 1],label) configure \
		    -background yellow -foreground black
		$pdf_cdf_table_w window configure [expr $variables_counter - 1],0 \
		    -relief flat -background yellow
		
		set focus_variable [expr $focus_variable - 1]
	    }
	    
	    set variables_counter [expr $variables_counter - 1]
	    set sisimpar(variables) [expr $sisimpar(variables) - 1]
	    
	} elseif { $sisimpar(variables) <= 5 } {
	    if { $variables_counter >= 2 } {
		tk_messageBox \
		    -title "Cutoffs / Categories" \
		    -icon warning \
		    -message "At least two cutoffs or categories are needed." \
		    -type ok
	    } elseif { $variables_counter < 2 } {
		tk_messageBox \
		    -title "Cutoffs / Categories" \
		    -icon warning \
		    -message "At least two cutoffs or categories are needed. The soft data file has too few data." \
		    -type ok
	    }
	}
    }
}
#============================================================================


#############################################################################
proc check_number_cut_cat_pdf_cdf { entry newstr char } {
    set current [$entry get]
    if {$current == {} && [string match {[0-9.]} $char]} {
	return [regexp {[0-9.]} $char]
    } elseif {$current != {} && [string match {[0-9.]} $char]} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    }
}
#============================================================================


#############################################################################
proc focus_proc { ii } {
    global sisimpar
    global model_table_w
    global model_sframe_w
    global model_hsbar_w
    global focus_variable
    global no
    global model_vals_label
    global model_type
    global model_cc
    global model_ang1
    global model_ang2
    global model_ang3
    global model_aa_hmax
    global model_aa_hmin
    global model_aa_vert
    global minus_model_pushbutton_w
    global plus_model_pushbutton_w
    global dim_name_soft
    global models_nugget_w
    global models_sill_label_w
    global models_sill_value_w
    global cut_cat_number_label_w
    global cut_cat_number_value_w
    global cut_cat_number
    global subframe3_variogram_w
    global sill
    global pdf_cdf_table_w
    global pdf_cdf_vals
    global icoli_table_w
    global icoli_vals
    
    if { $sisimpar(directik) == {*.dat} && $sisimpar(datafl) != {*.dat} } {
	set focus_variable $ii
	
	for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
	    pack forget $model_hsbar_w($ii)
	    pack forget $model_table_w($ii)
	    grid forget $models_nugget_w($ii)
	    grid forget $models_sill_label_w($ii)
	    grid forget $models_sill_value_w($ii)
	    grid forget $cut_cat_number_label_w($ii)
	    grid forget $cut_cat_number_value_w($ii)
	}
	set ii $focus_variable
	
	pack configure $model_hsbar_w($ii) -side right -anchor w -fill y
	pack configure $model_table_w($ii) -side left -anchor w -padx 2 -expand 1 -fill both
	
	for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
	    $model_type($i,combo$ii) configure -state normal -textbackground white -foreground black
	    $model_cc($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_ang1($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_ang2($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_ang3($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_aa_hmax($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_aa_hmin($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_aa_vert($i,entry$ii) configure -state normal -textbackground white -foreground black
	}
	
	grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	grid $models_nugget_w($ii)     -row 0 -column 2 -sticky e -pady 0
	grid $models_sill_label_w($ii) -row 0 -column 3 -sticky e -pady 0
	grid $models_sill_value_w($ii) -row 0 -column 4 -sticky w -pady 0
	grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	
	$plus_model_pushbutton_w configure -state normal
	$minus_model_pushbutton_w configure -state normal
	
	$models_nugget_w($ii) configure -state normal -textbackground white -foreground black
	$models_sill_label_w($ii) configure -state normal
	$models_sill_value_w($ii) configure -state normal
	$cut_cat_number_label_w($ii) configure -state normal
	$cut_cat_number_value_w($ii) configure -state normal
	
	
	### focus_variable is highlighted in the icoli_table and the pcf_cdf_table
	for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
	    $pdf_cdf_vals($ii,label) configure -background gray50 -foreground white
	    $pdf_cdf_table_w window configure $ii,0 -relief flat -background gray50
	}
	$pdf_cdf_vals($focus_variable,label) configure -background yellow -foreground black
	$pdf_cdf_table_w window configure $focus_variable,0 -relief flat -background yellow
	
    } elseif { ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) == {*.dat}) || \
		   ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) != {*.dat}) } {
	set focus_variable $ii
	
	for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3] } { incr ii } {
	    pack forget $model_hsbar_w($ii)
	    pack forget $model_table_w($ii)
	    grid forget $models_nugget_w($ii)
	    grid forget $models_sill_label_w($ii)
	    grid forget $models_sill_value_w($ii)
	    grid forget $cut_cat_number_label_w($ii)
	    grid forget $cut_cat_number_value_w($ii)
	}
	set ii $focus_variable
	
	pack configure $model_hsbar_w($ii) -side right -anchor w -fill y
	pack configure $model_table_w($ii) -side left -anchor w -padx 2 -expand 1 -fill both
	
	for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
	    $model_type($i,combo$ii) configure -state normal -textbackground white -foreground black
	    $model_cc($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_ang1($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_ang2($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_ang3($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_aa_hmax($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_aa_hmin($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_aa_vert($i,entry$ii) configure -state normal -textbackground white -foreground black
	}
	
	grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	grid $models_nugget_w($ii)      -row 0 -column 2 -sticky e -pady 0
	grid $models_sill_label_w($ii)  -row 0 -column 3 -sticky e -pady 0
	grid $models_sill_value_w($ii)  -row 0 -column 4 -sticky w -pady 0
	grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	
	$plus_model_pushbutton_w configure -state normal
	$minus_model_pushbutton_w configure -state normal
	
	$models_nugget_w($ii) configure -state normal -textbackground white -foreground black
	$models_sill_label_w($ii) configure -state normal
	$models_sill_value_w($ii) configure -state normal
	$cut_cat_number_label_w($ii) configure -state normal
	$cut_cat_number_value_w($ii) configure -state normal
	
	
	### focus_variable is highlighted in the icoli_table and the pcf_cdf_table
	for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3] } { incr ii } {
	    $icoli_vals($ii,label) configure -background gray50 -foreground white
	    $icoli_table_w window configure $ii,0 -relief flat -background gray50
	    $pdf_cdf_vals($ii,label) configure -background gray50 -foreground white
	    $pdf_cdf_table_w window configure $ii,0 -relief flat -background gray50
	}
	$icoli_vals($focus_variable,label) configure -background yellow -foreground black
	$icoli_table_w window configure $focus_variable,0 -relief flat -background yellow
	$pdf_cdf_vals($focus_variable,label) configure -background yellow -foreground black
	$pdf_cdf_table_w window configure $focus_variable,0 -relief flat -background yellow
    }
    return
}
#============================================================================


#############################################################################
proc plus_model { } {
    global sisimpar
    global model_table_w
    global no
    global model_vals_label
    global model_type
    global model_cc
    global model_ang1
    global model_ang2
    global model_ang3
    global model_aa_hmax
    global model_aa_hmin
    global model_aa_vert
    global focus_variable
    global sill
    global sum_cc
    global models_sill_value_w
    global models_nugget_w
    
    
    if { $no($focus_variable) <= 4 } {
	set no($focus_variable) [expr $no($focus_variable) + 1]
	
	set ii $focus_variable
	
	set no($ii) $no($focus_variable)
	
	set i $no($focus_variable)
	
	$model_table_w($focus_variable) configure -rows [expr $no($focus_variable) + 1]
	
	set model_vals_label($i,$focus_variable) \
	    [iwidgets::labeledwidget $model_table_w($focus_variable).label_vals$i \
		 -labeltext "$i" \
		 -background gray50 \
		 -foreground white]
	update
	
	set sisimpar(it$focus_variable,$i) 0
	set model_type($i,combo$focus_variable) [iwidgets::combobox $model_table_w($focus_variable).combo_type$i \
						     -width 15 \
						     -textvariable sisimpar(it$focus_variable,$i) \
						     -listheight 75]
	$model_table_w($focus_variable).combo_type$i insert list end \
	    "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
	$model_type($i,combo$focus_variable) selection set 0
	set $model_type($i,combo$focus_variable) [lindex $model_table_w($focus_variable).combo_type$i 0]
	update
	
	set sisimpar(cc$focus_variable,$i) 0.9
	set model_cc($i,entry$focus_variable) [iwidgets::entryfield $model_table_w($focus_variable).entry_cc$i \
						   -textvariable sisimpar(cc$focus_variable,$i) \
						   -validate "check_number_cc %W %S %P $i %focus_variable"]
	set sum_cc($focus_variable) [expr $sum_cc($focus_variable) + $sisimpar(cc$focus_variable,$i)]
	set sill($focus_variable) [expr $sisimpar(c0$focus_variable) + $sum_cc($focus_variable)]
	update
	
	set sisimpar(ang1$focus_variable,$i) 0
	set model_ang1($i,entry$focus_variable) [iwidgets::entryfield $model_table_w($focus_variable).entry_ang1$i \
						     -textvariable sisimpar(ang1$focus_variable,$i) \
						     -validate {check_number_sang1 %W %P %c}]
	update
	
	set sisimpar(ang2$focus_variable,$i) 0
	set model_ang2($i,entry$focus_variable) [iwidgets::entryfield $model_table_w($focus_variable).entry_ang2$i \
						     -textvariable sisimpar(ang2$focus_variable,$i) \
						     -validate {check_number_sang %W %P %c}]
	update
	
	set sisimpar(ang3$focus_variable,$i) 0
	set model_ang3($i,entry$focus_variable) [iwidgets::entryfield $model_table_w($focus_variable).entry_ang3$i \
						     -textvariable sisimpar(ang3$focus_variable,$i) \
						     -validate {check_number_sang %W %P %c}]
	update
	
	set sisimpar(aa_hmax$focus_variable,$i) 10
	set model_aa_hmax($i,entry$focus_variable) [iwidgets::entryfield \
							$model_table_w($focus_variable).entry_aa_hmax$i \
							-textvariable sisimpar(aa_hmax$focus_variable,$i) \
							-validate {check_number_radius %W %P %c}]
	update
	
	set sisimpar(aa_hmin$focus_variable,$i) 10
	set model_aa_hmin($i,entry$focus_variable) [iwidgets::entryfield \
							$model_table_w($focus_variable).entry_aa_hmin$i \
							-textvariable sisimpar(aa_hmin$focus_variable,$i) \
							-validate {check_number_radius %W %P %c}]
	update
	
	set sisimpar(aa_vert$focus_variable,$i) 10
	set model_aa_vert($i,entry$focus_variable) [iwidgets::entryfield \
							$model_table_w($focus_variable).entry_aa_vert$i \
							-textvariable sisimpar(aa_vert$focus_variable,$i) \
							-validate {check_number_radius %W %P %c}]
	update
	
	
	$model_table_w($focus_variable) window configure $i,0 \
	    -window $model_vals_label($i,$focus_variable) -relief flat -background gray50
	$model_table_w($focus_variable) window configure $i,1 -window $model_type($i,combo$focus_variable)
	$model_table_w($focus_variable) window configure $i,2 -window $model_cc($i,entry$focus_variable)
	$model_table_w($focus_variable) window configure $i,3 -window $model_ang1($i,entry$focus_variable)
	$model_table_w($focus_variable) window configure $i,4 -window $model_ang2($i,entry$focus_variable)	
	$model_table_w($focus_variable) window configure $i,5 -window $model_ang3($i,entry$focus_variable)	
	$model_table_w($focus_variable) window configure $i,6 -window $model_aa_hmax($i,entry$focus_variable)	
	$model_table_w($focus_variable) window configure $i,7 -window $model_aa_hmin($i,entry$focus_variable)	
	$model_table_w($focus_variable) window configure $i,8 -window $model_aa_vert($i,entry$focus_variable)
    } else {
	tk_messageBox \
	    -title "Nested variogram models" \
	    -icon warning \
	    -message \
	    "Maximum 5 nested models are permitted for the structural analysis of a single cutoff or category." \
	    -type ok
    }
}
#============================================================================


#############################################################################
proc minus_model { } {
    global sisimpar
    global model_table_w
    global model_vals_label
    global model_type
    global model_cc
    global model_ang1
    global model_ang2
    global model_ang3
    global model_aa_hmax
    global model_aa_hmin
    global model_aa_vert
    global no
    global focus_variable
    global sum_cc
    global models_sill_value_w
    global models_nugget_w
    global subframe3_w
    global sill
    
    if { $no($focus_variable) >= 2 } {
	set i $no($focus_variable)
	
	set sum_cc($focus_variable) [expr $sum_cc($focus_variable) - $sisimpar(cc$focus_variable,$i)]
	set sill($focus_variable) [expr $sisimpar(c0$focus_variable) + $sum_cc($focus_variable)]
	
	destroy $model_vals_label($i,$focus_variable)
	destroy $model_type($i,combo$focus_variable)
	destroy $model_cc($i,entry$focus_variable)
	destroy $model_ang1($i,entry$focus_variable)
	destroy $model_ang2($i,entry$focus_variable)
	destroy $model_ang3($i,entry$focus_variable)
	destroy $model_aa_hmax($i,entry$focus_variable)
	destroy $model_aa_hmin($i,entry$focus_variable)
	destroy $model_aa_vert($i,entry$focus_variable)
	
	$model_table_w($focus_variable) configure -rows [expr $no($focus_variable) - 0]
	
	set no($focus_variable) [expr $no($focus_variable) - 1]
	update   
    } else {
	tk_messageBox \
	    -title "Nested variogram models" \
	    -icon warning \
	    -message \
	    "At least 1 variogram model is needed for the structural analysis of a single cutoff or category." \
	    -type ok
    }
}
#============================================================================


#############################################################################
proc check_number_cc { entry oldstr newstr i ii } {
    global sisimpar
    global sill
    global focus_variable
    global no
    global sum_cc
    
    if { [regexp {^[0-9]*[\.]?[0-9]*$} $newstr] } {
	catch {scan $oldstr {%f} oldstr}
	catch {scan $newstr {%f} newstr}
	catch {scan $sisimpar(c0$focus_variable) {%f} sisimpar(c0$focus_variable)}
	
	set sum_cc($focus_variable) $newstr
	if { $sum_cc($focus_variable) == 0 } {
	    set sill($focus_variable) "?"
	} else {
	    set sill($focus_variable) [expr $sum_cc($focus_variable) + $sisimpar(c0$focus_variable)]
	}
	return 1
    } else {
	return 0
    }
}
#============================================================================


#############################################################################
proc check_number_nugget { entry newstr } {
    global sisimpar
    global sill
    global focus_variable
    global no
    global sum_cc
    
    if { [regexp {^[0-9]*[\.]?[0-9]*$} $newstr] } {
	catch {scan $newstr {%f} newstr}
	catch {scan $sum_cc($focus_variable) {%f} sum_cc($focus_variable)}
	if { $sum_cc($focus_variable) == 0 } {
	    set sill($focus_variable) "?"
	} else {
	    set sill($focus_variable) [expr $newstr + $sum_cc($focus_variable)]
	}
	return 1
    } else {
	return 0
    }
}
#============================================================================

	
#############################################################################
proc check_number_z { entry newstr char} {
    set current [$entry get]
    set len [string length $current]
    if {$len < 2} {
	return [regexp {^[0-9]*$} $newstr]
    } elseif {$len == 2 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc tabulated_values_file_activator { } {
    global sisimpar
    global sisim_box_w
    global tabfl_w
    global tabfl_browse_w
    global icolvrt_w
    global icolwtt_w
    
    if {$sisimpar(ltail) == "Tabulated Quantiles Linear Interpolation" || \
	    $sisimpar(middle) == "Tabulated Quantiles Linear Interpolation" || \
	    $sisimpar(utail) == "Tabulated Quantiles Linear Interpolation"} {
	$tabfl_w configure -state disabled -textbackground gray85 -foreground gray50
	$tabfl_browse_w configure -state normal
	$icolvrt_w configure -state normal -textbackground white -foreground black
	$icolwtt_w configure -state normal -textbackground white -foreground black
    } else {
	$tabfl_w configure -state disabled -textbackground gray85 -foreground gray50
	$tabfl_browse_w configure -state disabled
	$icolvrt_w configure -state disabled -textbackground gray85 -foreground gray50
	$icolwtt_w configure -state disabled -textbackground gray85 -foreground gray50
    }
}
#============================================================================


#############################################################################
proc check_number_par { entry newstr char} {
    set current [$entry get]
    set len [string length $current]
    if {$len < 2} {
	return [regexp {^[0-9]*$} $newstr]
    } elseif {$len == 2 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc check_number_n { entry newstr char } {
    if {$newstr == {} || ($newstr >= 0 && $newstr <= 2000) && [string match {[0-9]} $char]} {
	return [regexp {[0-9]} $newstr]
    } elseif {$newstr > 2000 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "2000"
    }
}
#============================================================================


#############################################################################
proc check_number_siz { entry newstr char } {
    if {$newstr == {} || ($newstr >= 0 && $newstr <= 1000) && [string match {[0-9.]} $char]} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$newstr > 1000 && [string match {[0-9.]} $char]} {
	$entry delete 0 end  
	$entry insert end "1000"
    }
}
#============================================================================


#############################################################################
proc check_number_xmn { entry  newstr char } {
    global sisimpar
    
    if {$newstr == {} || ($newstr >= 0 && $newstr <= $sisimpar(xsiz)) && [string match {[0-9.]} $char]} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$newstr > $sisimpar(xsiz) && [string match {[0-9.]} $char]} {
	$entry delete 0 end  
	$entry insert end $sisimpar(xsiz)
    }
}
#============================================================================


#############################################################################
proc grid_definition { } {
    global grid
    global sisimpar
    
    catch {scan $sisimpar(nx) {%d} sisimpar(nx)}
    catch {scan $sisimpar(ny) {%d} sisimpar(ny)}
    catch {scan $sisimpar(nz) {%d} sisimpar(nz)}
    catch {scan $sisimpar(xsiz) {%g} sisimpar(xsiz)}
    catch {scan $sisimpar(ysiz) {%g} sisimpar(ysiz)}
    catch {scan $sisimpar(zsiz) {%g} sisimpar(zsiz)}
    catch {scan $sisimpar(xmn) {%g} sisimpar(xmn)}
    catch {scan $sisimpar(ymn) {%g} sisimpar(ymn)}
    catch {scan $sisimpar(zmn) {%g} sisimpar(zmn)}

    if { $sisimpar(nx) == {} } {
	set sisimpar(nx) 1
    }

    if { $sisimpar(ny) == {} } {
	set sisimpar(ny) 1
    }

    if { $sisimpar(nz) == {} } {
	set sisimpar(nz) 1
    }
    
    if { $sisimpar(nx) == 0 || $sisimpar(xsiz) == 0 || $sisimpar(xmn) == {} } {
	set grid(xmax) "?"
    } else {
	set grid(xmax) [expr (($sisimpar(nx)-1)*$sisimpar(xsiz))+$sisimpar(xmn)]
    }
    set grid(xtot) [expr $sisimpar(nx)*$sisimpar(xsiz)]
    if { $sisimpar(ny) == 0 || $sisimpar(ysiz) == 0 || $sisimpar(ymn) == {} } {
	set grid(ymax) "?"
    } else {
	set grid(ymax) [expr (($sisimpar(ny)-1)*$sisimpar(ysiz))+$sisimpar(ymn)]
    }
    set grid(ytot) [expr $sisimpar(ny)*$sisimpar(ysiz)]
    if { $sisimpar(nz) == 0 || $sisimpar(zsiz) == 0 || $sisimpar(zmn) == {} } {
	set grid(zmax) "?"
    } else {
	set grid(zmax) [expr (($sisimpar(nz)-1)*$sisimpar(zsiz))+$sisimpar(zmn)]
    }
    set grid(ztot) [expr $sisimpar(nz)*$sisimpar(zsiz)]
    set grid(cell) [expr $sisimpar(nx)*$sisimpar(ny)*$sisimpar(nz)]
    set grid(area) [expr $sisimpar(nx)*$sisimpar(xsiz)*$sisimpar(ny)*$sisimpar(ysiz)]
    set grid(volume) [expr $sisimpar(nx)*$sisimpar(xsiz)*$sisimpar(ny)*$sisimpar(ysiz) \
			  *$sisimpar(nz)*$sisimpar(zsiz)]
    return
}
#============================================================================


#############################################################################
proc check_number_ymn { entry  newstr char } {
    global sisimpar
    
    if {$newstr == {} || ($newstr >= 0 && $newstr <= $sisimpar(ysiz)) && [string match {[0-9.]} $char]} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$newstr > $sisimpar(ysiz) && [string match {[0-9.]} $char]} {
	$entry delete 0 end  
	$entry insert end $sisimpar(ysiz)
    }
}
#============================================================================


#############################################################################
proc check_number_zmn { entry  newstr char } {
    global sisimpar
    
    if {$newstr == {} || ($newstr >= 0 && $newstr <= $sisimpar(zsiz)) && [string match {[0-9.]} $char]} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$newstr > $sisimpar(zsiz) && [string match {[0-9.]} $char]} {
	$entry delete 0 end  
	$entry insert end $sisimpar(zsiz)
    }
}
#============================================================================


#############################################################################
proc read_sisim_params { fname } {
    global sisimpar
    global variable_type_w
    global pdf_cdf_table_w
    global directik_w
    global datafl_w
    global directik_browse_w
    global icolsx_w
    global icolsy_w
    global icolsz_w
    global icoli_table_w
    global model_hsbar_w
    global icoli_vals
    global bz_vals
    global imbsim_checkbutton_w
    global realization_w
    global max_org_data_w
    global max_prev_nodes_w
    global max_soft_nodes_w
    global max_per_octant_w
    global dbgfl_level_w
    global kriging_type_w
    global kriging_flag_w
    global cutoff_category_w
    global min_attribute_value_w
    global max_attribute_value_w
    global extrapol_lower_tail_w
    global extrapol_lower_tail_parameter_w
    global interpol_middle_w
    global interpol_middle_parameter_w
    global extrapol_upper_tail_w
    global extrapol_upper_tail_parameter_w
    global tabfl_w
    global tabfl_browse_w
    global icolvrt_w
    global icolwtt_w
    global grid_refinement_number_w
    global icolx_w
    global icoly_w
    global icolz_w
    global icolvr_w
    global dim_name_hard
    global dim_name_soft
    global dim_name_tabulated
    global grid
    global variables_initial_counter
    global variables_counter
    global pdf_cdf_vals
    global cut_cat_vals				       
    global model_hsbar_w
    global model_table_w
    global models_nugget_w
    global models_sill_label_w
    global models_sill_value_w
    global cut_cat_number_label_w
    global cut_cat_number_value_w
    global cut_cat_number
    global model_sframe_w
    global subframe3_variogram_w
    global sill
    global no
    global model_type
    global model_cc
    global model_ang1
    global model_ang2
    global model_ang3
    global model_aa_hmax
    global model_aa_hmin
    global model_aa_vert
    global model_vals_label
    global plus_model_pushbutton_w
    global minus_model_pushbutton_w
    global plus_cut_cat_pushbutton_w
    global minus_cut_cat_pushbutton_w
    global focus_variable
    global sum_cc
    global sisim_box_w
    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }
    
    ### sets the cursor on watch as long as parameters are reading
    . configure -cursor watch
    update
    
    ### searches the line "parameters for sisim" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Parameters for SISIM" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File selection error" \
		-icon error \
		-message "You have selected a wrong parameter file.\n"
	    return 0
	}
    }

    ### set file name in sisim box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $sisim_box_w "SISIM parameters   -   $fname_short"

    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }
    
    ### destroys the content of the icoli-table and the pdf-cdf-table; 
    ### destroys the nuggets, sills, cut-cat-numbers and the model-tables
    if { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat} || \
	     ($sisimpar(datafl) == {*.dat} && $sisimpar(directik) == {*.dat}) } {
	set sisimpar(variables_initial) [expr $sisimpar(ncat) + 4]
	for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
	    destroy $icoli_vals($ii,label)
	    destroy $icoli_vals($ii,combo)
	    destroy $bz_vals($ii,entry)
	    
	    destroy $pdf_cdf_vals($ii,label)
	    destroy $cut_cat_vals($ii,entry)
	    destroy $pdf_cdf_vals($ii,entry)
	    
	    destroy $model_hsbar_w($ii)
	    destroy $model_table_w($ii)
	    
	    destroy $models_nugget_w($ii)
	    destroy $models_sill_label_w($ii)
	    destroy $models_sill_value_w($ii)
	    destroy $cut_cat_number_label_w($ii)
	    destroy $cut_cat_number_value_w($ii)
	}
    } elseif { ($sisimpar(datafl) == {*.dat} && $sisimpar(directik) != {*.dat}) || \
		   ($sisimpar(datafl) != {*.dat} && $sisimpar(directik) != {*.dat}) } {
	set sisimpar(variables) [expr $sisimpar(ncat) + 3]
	for {set ii 1} {$ii <= [expr $sisimpar(variables) - 3]} {incr ii} {
	    destroy $icoli_vals($ii,label)
	    destroy $icoli_vals($ii,combo)
	    destroy $bz_vals($ii,entry)
	    
	    destroy $pdf_cdf_vals($ii,label)
	    destroy $cut_cat_vals($ii,entry)
	    destroy $pdf_cdf_vals($ii,entry)
	    
	    destroy $model_hsbar_w($ii)
	    destroy $model_table_w($ii)
	    
	    destroy $models_nugget_w($ii)
	    destroy $models_sill_label_w($ii)
	    destroy $models_sill_value_w($ii)
	    destroy $cut_cat_number_label_w($ii)
	    destroy $cut_cat_number_value_w($ii)
	}
    }
    
    ### reads the vartype: variable type (1=cont.(cdf), 0=cat.(pdf))
    ### and sets the extrapolation parameter widgets normal / disabled
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(vartype) [format "%d" [string range $line 0 $space]]
    $variable_type_w selection set $sisimpar(vartype)
    if { $sisimpar(vartype) == "Categorical" } {
	$min_attribute_value_w           configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_lower_tail_w           configure -state normal -textbackground white -foreground black
	$extrapol_lower_tail_parameter_w configure -state disabled -textbackground gray85 -foreground gray50
	$interpol_middle_w               configure -state normal -textbackground white -foreground black
	$interpol_middle_parameter_w     configure -state disabled -textbackground gray85 -foreground gray50
	$max_attribute_value_w           configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_upper_tail_w           configure -state normal -textbackground white -foreground black
	$extrapol_upper_tail_parameter_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $sisimpar(vartype) == "Continous" } {
	$min_attribute_value_w           configure -state normal -textbackground white -foreground black
	$extrapol_lower_tail_w           configure -state normal -textbackground white -foreground black
	$extrapol_lower_tail_parameter_w configure -state normal -textbackground white -foreground black
	$interpol_middle_w               configure -state normal -textbackground white -foreground black
	$interpol_middle_parameter_w     configure -state normal -textbackground white -foreground black
	$max_attribute_value_w           configure -state normal -textbackground white -foreground black
	$extrapol_upper_tail_w           configure -state normal -textbackground white -foreground black
	$extrapol_upper_tail_parameter_w configure -state normal -textbackground white -foreground black
    }	
    
    ### reads the ncat: number of thresholds / categories
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(ncat) [format "%d" [string range $line 0 $space]]

    ### reads the cat: thresholds / categories
    set line [gets $inf]
    set catline $line
    
    ### reads the pdf: global cdf's / pdf's
    set line [gets $inf]
    set pdfline $line
    
    ### reads name of datafl: file with hard data
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set sisimpar(datafl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set sisimpar(datafl) $line
    }

    if { $sisimpar(datafl) == {*.dat} } {
	$datafl_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $sisimpar(datafl) != {*.dat} } {
	$datafl_w configure -state normal -textbackground white -foreground black
    }
    
    ### reads the icolx, ..y, ..z, ..vr: columns for x, y, z, variable (hard data)
    ### and fills the icol.. comboboxes
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_sisimpar(icolx) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_sisimpar(icoly) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_sisimpar(icolz) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_sisimpar(icolvr) [format "%d" [string range $line 0 [expr $space - 1]]]
    
    if { $sisimpar(datafl) != {*.dat} } {
	unset dim_name_hard
	if { [sisim_read_hard_data $sisimpar(datafl)] != 1 } {
	    set dim_name_hard {}
	    lappend dim_name_hard "None"
	    for { set i 1 } { $i < 5 } { incr i } {
		lappend dim_name_hard $i
	    }
	}
	
	$icolx_w clear list
	foreach i $dim_name_hard {
	    $icolx_w insert list end $i
	}
	$icolx_w selection set $tmp_sisimpar(icolx)
	
	$icoly_w clear list
	foreach i $dim_name_hard {
	    $icoly_w insert list end $i
	}
	$icoly_w selection set $tmp_sisimpar(icoly)
	
	$icolz_w clear list
	foreach i $dim_name_hard {
	    $icolz_w insert list end $i
	}
	$icolz_w selection set $tmp_sisimpar(icolz)
	
	$icolvr_w clear list
	foreach i $dim_name_hard {
	    $icolvr_w insert list end $i
	}
	$icolvr_w selection set $tmp_sisimpar(icolvr)

    } elseif { $sisimpar(datafl) == {*.dat} } {
	unset dim_name_hard
	if { [sisim_read_hard_data $sisimpar(datafl)] != 1 } {
	    set dim_name_hard {}
	    lappend dim_name_hard "None"
	    for { set i 1 } { $i < 5 } { incr i } {
		lappend dim_name_hard $i
	    }
	}
	
	$icolx_w clear list
	foreach i $dim_name_hard {
	    $icolx_w insert list end $i
	}
	$icolx_w selection set 1
	set sisimpar(icolx) [lindex $dim_name_soft 1]
	
	$icoly_w clear list
	foreach i $dim_name_hard {
	    $icoly_w insert list end $i
	}
	$icoly_w selection set 2
	set sisimpar(icoly) [lindex $dim_name_soft 2]
	
	$icolz_w clear list
	foreach i $dim_name_hard {
	    $icolz_w insert list end $i
	}
	$icolz_w selection set 3
	set sisimpar(icolz) [lindex $dim_name_soft 3]
	
	$icolvr_w clear list
	foreach i $dim_name_hard {
	    $icolvr_w insert list end $i
	}
	$icolvr_w selection set 4
	set sisimpar(icolvr) [lindex $dim_name_soft 4]
    }

    ### reads the name of directik: file with soft indicator input
    ### and sets some soft input parameter widgets normal / disabled
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set sisimpar(directik) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set sisimpar(directik) $line
    }

    if { $sisimpar(directik) != {*.dat} } {
	set sisimpar(soft_data) 1
	$directik_w configure -state normal -textbackground white -foreground black
	$directik_browse_w configure -state normal
	$icolsx_w configure -state normal -textbackground white -foreground black
	$icolsy_w configure -state normal -textbackground white -foreground black
	$icolsz_w configure -state normal -textbackground white -foreground black
	$imbsim_checkbutton_w configure -state normal
    } elseif { $sisimpar(directik) == {*.dat} } {
	set sisimpar(soft_data) 0
	$directik_w configure -state disabled -textbackground gray85 -foreground gray50
	$directik_browse_w configure -state disabled
	$icolsx_w configure -state disabled -textbackground gray85 -foreground gray50
	$icolsy_w configure -state disabled -textbackground gray85 -foreground gray50
	$icolsz_w configure -state disabled -textbackground gray85 -foreground gray50
	$imbsim_checkbutton_w configure -state disabled
    }
    
    ### reads the icolsx, ..sy, ..sz, ..i: columns for x, y, z, indic. (soft data)
    ### and fills the icols.. comboboxes
    set line [gets $inf]
    set softline $line
    set space [string first " " $softline]
    set tmp_sisimpar(icolsx) [format "%d" [string range $softline 0 [expr $space - 1]]]
    set softline [string range $softline [expr $space + 1] end]
    set space [string first " " $softline]
    set tmp_sisimpar(icolsy) [format "%d" [string range $softline 0 [expr $space - 1]]]
    set softline [string range $softline [expr $space + 1] end]
    set space [string first " " $softline]
    set tmp_sisimpar(icolsz) [format "%d" [string range $softline 0 [expr $space - 1]]]
    set softline [string range $softline [expr $space + 1] end]
    
    if { $sisimpar(directik) == {*.dat} } {
	unset dim_name_soft
	if { [sisim_read_soft_data $sisimpar(directik)] != 1 } {
	    set dim_name_soft {}
	    lappend dim_name_soft "None"
	    for { set i 1 } { $i <= [expr $sisimpar(variables_initial) - 1] } { incr i } {
		lappend dim_name_soft $i
	    }
	}
	
	$icolsx_w clear list
	foreach i $dim_name_soft {
	    $icolsx_w insert list end $i
	}
	$icolsx_w selection set 1
	set sisimpar(icolsx) [lindex $dim_name_soft 1]

	$icolsy_w clear list
	foreach i $dim_name_soft {
	    $icolsy_w insert list end $i
	}
	$icolsy_w selection set 2
	set sisimpar(icolsy) [lindex $dim_name_soft 2]
	
	$icolsz_w clear list
	foreach i $dim_name_soft {
	    $icolsz_w insert list end $i
	}
	$icolsz_w selection set 3
	set sisimpar(icolsz) [lindex $dim_name_soft 3]
	
    } elseif { $sisimpar(directik) != {*.dat} } {
	unset dim_name_soft
	if { [sisim_read_soft_data $sisimpar(directik)] != 1 } {
	    set dim_name_soft {}
	    lappend dim_name_soft "None"
	    for { set i 1 } { $i <= [expr $sisimpar(variables) - 3] } { incr i } {
		lappend dim_name_soft $i
	    }
	}
	
	$icolsx_w clear list
	foreach i $dim_name_soft {
	    $icolsx_w insert list end $i
	}
	$icolsx_w selection set $tmp_sisimpar(icolsx)
	
	$icolsy_w clear list
	foreach i $dim_name_soft {
	    $icolsy_w insert list end $i
	}
	$icolsy_w selection set $tmp_sisimpar(icolsy)
	
	$icolsz_w clear list
	foreach i $dim_name_soft {
	    $icolsz_w insert list end $i
	}
	$icolsz_w selection set $tmp_sisimpar(icolsz)
    }

    ### reads the imbsim: Markov-Bayes simulation (0=no, 1=yes)
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(imbsim) [format "%d" [string range $line 0 $space]]

    ### reads the b(z): calibration B(z) values
    set line [gets $inf]
    set bzline $line
    
    ### sets the cut_cat pushbuttons normal
    $plus_cut_cat_pushbutton_w configure -state normal
    $minus_cut_cat_pushbutton_w configure -state normal

    ### fills the pdf-cdf-table
    $pdf_cdf_table_w configure -rows [expr $sisimpar(ncat) + 1]
    
    for { set ii 1 } { $ii <= $sisimpar(ncat) } { incr ii } {
	
	set pdf_cdf_vals($ii,label) [iwidgets::labeledwidget $pdf_cdf_table_w.pdf_cdf$ii \
					 -labeltext "$ii" \
					 -background gray50 \
					 -foreground white]
	update
	
	set space [string first " " $catline]
	set sisimpar(cat,$ii) [format "%g" [string range $catline 0 [expr $space - 1]]]
	set catline [string range $catline [expr $space + 1] end]
	set cut_cat_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
					 -textvariable sisimpar(cat,$ii) \
					 -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
					 -focuscommand "focus_proc $ii"]
	update
	
	set space [string first " " $pdfline]
	set sisimpar(pdf,$ii) [format "%g" [string range $pdfline 0 [expr $space - 1]]]
	set pdfline [string range $pdfline [expr $space + 1] end]
	set pdf_cdf_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
					 -textvariable sisimpar(pdf,$ii) \
					 -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
					 -focuscommand "focus_proc $ii"]
	update
	
	$pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) -relief flat -background gray50
	$pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
	$pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
    }
    
    ### fills the icoli-table
    if { $sisimpar(directik) != {*.dat} } {

	$icoli_table_w configure -rows [expr $sisimpar(ncat) + 1]
	
	for { set ii 1 } { $ii <= $sisimpar(ncat) } { incr ii } {
	    
	    set icoli_vals($ii,label) [iwidgets::labeledwidget $icoli_table_w.label$ii \
					   -labeltext "$ii" \
					   -background gray50 \
					   -foreground white]
	    update
	    
	    set icoli_vals($ii,combo) [iwidgets::combobox $icoli_table_w.combo$ii \
					   -textvariable sisimpar(icoli,$ii) \
					   -validate {regexpvalidate %c} \
					   -listheight 100 \
					   -focuscommand "focus_proc $ii" \
					   -selectioncommand "focus_proc $ii"]
	    
	    $icoli_vals($ii,combo) clear list
	    foreach i $dim_name_soft {
		$icoli_vals($ii,combo) insert list end $i
	    }
	    set space [string first " " $softline]
	    set tmp_sisimpar(icoli) [format "%d" [string range $softline 0 [expr $space - 1]]]
	    set softline [string range $softline [expr $space + 1] end]
	    $icoli_vals($ii,combo) selection set $tmp_sisimpar(icoli)
	    update
	    
	    set space [string first " " $bzline]
	    set sisimpar(bz,$ii) [format "%g" [string range $bzline 0 [expr $space - 1]]]
	    set bzline [string range $bzline [expr $space + 1] end]
	    set bz_vals($ii,entry) [iwidgets::entryfield $icoli_table_w.bz$ii \
					-textvariable sisimpar(bz,$ii) \
					-validate {check_number_bz_values %W %P %c} \
					-focuscommand "focus_proc $ii"]
	    update

	    if { $sisimpar(imbsim) == 0 } {
		$bz_vals($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	    } elseif { $sisimpar(imbsim) == 1 } {
		$bz_vals($ii,entry) configure -state normal -textbackground white -foreground black
	    }

	    $icoli_table_w window configure $ii,0 -window $icoli_vals($ii,label) -relief flat -background gray50
	    $icoli_table_w window configure $ii,1 -window $icoli_vals($ii,combo)
	    $icoli_table_w window configure $ii,2 -window $bz_vals($ii,entry)
	}
    } elseif { $sisimpar(directik) == {*.dat} } {

	$icoli_table_w configure -rows [expr $sisimpar(ncat) + 1]
	
	for { set ii 1 } { $ii <= 3 } { incr ii } {
	    
	    set icoli_vals($ii,label) [iwidgets::labeledwidget $icoli_table_w.label$ii \
					   -labeltext "$ii" \
					   -background gray50 \
					   -foreground white]
	    update
	    
	    set icoli_vals($ii,combo) [iwidgets::combobox $icoli_table_w.combo$ii \
					   -textvariable sisimpar(icoli,$ii) \
					   -validate {regexpvalidate %c} \
					   -listheight 100 \
					   -focuscommand "focus_proc $ii" \
					   -selectioncommand "focus_proc $ii"]
	    foreach i $dim_name_soft {
		$icoli_vals($ii,combo) insert list end $i
	    }
	    $icoli_vals($ii,combo) selection set [expr $ii + 3]
	    set sisimpar(icoli) [lindex $dim_name_soft [expr $ii + 3]]
	    $icoli_table_w.combo$ii configure -state disabled -textbackground gray85 -foreground gray50
	    update
	    
	    set sisimpar(bz,$ii) 0	;# initialize to 0
	    set bz_vals($ii,entry) [iwidgets::entryfield $icoli_table_w.bz$ii \
					-textvariable sisimpar(bz,$ii) \
					-state disabled \
					-textbackground gray85 \
					-foreground gray50 \
					-validate {check_number_bz_values %W %P %c} \
					-focuscommand "focus_proc $ii"]
	    update
	    
	    $icoli_table_w window configure $ii,0 -window $icoli_vals($ii,label) -relief flat -background gray50
	    $icoli_table_w window configure $ii,1 -window $icoli_vals($ii,combo)
	    $icoli_table_w window configure $ii,2 -window $bz_vals($ii,entry)
	}
    }
    
    ### sets the variables_initial / variables
    if { $sisimpar(directik) == {*.dat} && $sisimpar(datafl) != {*.dat} } {
	set sisimpar(variables_initial) [expr $sisimpar(ncat) + 4]
	set variables_initial_counter $sisimpar(ncat)
    } elseif { ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) == {*.dat}) || \
		   ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) != {*.dat}) } {
	set sisimpar(variables) [expr $sisimpar(ncat) + 3]
	set variables_counter $sisimpar(ncat)
    }
    
    ### reads the tmin, tmax: trimming limits
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(tmin) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(tmax) [format "%g" [string range $line 0 [expr $space - 1]]]
    
    ### reads the zmin, zmax: minimum and maximum data value (continous variable)
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(zmin) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(zmax) [format "%g" [string range $line 0 [expr $space - 1]]]
   
    ### reads the ltail, ltpar: extrapolation lower tail, option, param.
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(ltail) [format "%d" [string range $line 0 [expr $space - 1]]]
    $extrapol_lower_tail_w selection set [expr $sisimpar(ltail) - 1]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(ltpar) [format "%g" [string range $line 0 [expr $space - 1]]]
    
    ### reads the middle, midpar: extrapolation middle, option param.
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(middle) [format "%d" [string range $line 0 [expr $space - 1]]]
    $interpol_middle_w selection set [expr $sisimpar(middle) - 1]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(midpar) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the utail, utpar: extrapolation upper tail, option and param.
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(utail) [format "%d" [string range $line 0 [expr $space - 1]]]
    $extrapol_upper_tail_w selection set [expr $sisimpar(utail) - 1]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(utpar) [format "%g" [string range $line 0 [expr $space - 1]]]
    
    if { $sisimpar(vartype) == "Categorical" } {
	$extrapol_lower_tail_w configure -state disabled -textbackground gray85 -foreground gray50
	$interpol_middle_w     configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_upper_tail_w configure -state disabled -textbackground gray85 -foreground gray50
    }
    
    ### reads the name of tabfl: file with tabulated values
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set sisimpar(tabfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set sisimpar(tabfl) $line
    }

    if { $sisimpar(ltail) == "Tabulated Quantiles Linear Interpolation" || \
	     $sisimpar(middle) == "Tabulated Quantiles Linear Interpolation" || \
	     $sisimpar(utail) == "Tabulated Quantiles Linear Interpolation" } {
	if { $sisimpar(tabfl) != {*.dat} } {
	    $tabfl_w configure -state normal -textbackground white -foreground black
	    $tabfl_browse_w configure -state normal 
	    $icolvrt_w configure -state normal -textbackground white -foreground black
	    $icolwtt_w configure -state normal -textbackground white -foreground black
	} elseif { $sisimpar(tabfl) == {*.dat} } {
	    $tabfl_w configure -state disabled -textbackground gray85 -foreground gray50
	    $tabfl_browse_w configure -state normal 
	    $icolvrt_w configure -state normal -textbackground white -foreground black
	    $icolwtt_w configure -state normal -textbackground white -foreground black	
	}
    } elseif { $sisimpar(ltail) != "Tabulated Quantiles Linear Interpolation" || \
		   $sisimpar(middle) != "Tabulated Quantiles Linear Interpolation" || \
		   $sisimpar(utail) != "Tabulated Quantiles Linear Interpolation" } {
	$tabfl_w configure -state disabled -textbackground gray85 -foreground gray50
	$tabfl_browse_w configure -state disabled
	$icolvrt_w configure -state disabled -textbackground gray85 -foreground gray50
	$icolwtt_w configure -state disabled -textbackground gray85 -foreground gray50
    }

    ### reads the icolvrt, icolwtt: columns for var., weight (0=not used) (tabulated data)
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_sisimpar(icolvrt) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_sisimpar(icolwtt) [format "%d" [string range $line 0 [expr $space - 1]]]
    
    if { $sisimpar(tabfl) != {*.dat} } {
	unset dim_name_tabulated
	if { [sisim_read_tabulated_values $sisimpar(tabfl)] != 1 } {
	    set dim_name_tabulated {}
	    lappend dim_name_tabulated "None"
	    for { set i 1 } { $i < 6 } { incr i } {
		lappend dim_name_tabulated $i
	    }
	}
	
	$icolvrt_w clear list
	foreach i $dim_name_tabulated {
	    $icolvrt_w insert list end $i
	}
	$icolvrt_w selection set $tmp_sisimpar(icolvrt)
	
	$icolwtt_w clear list
	foreach i $dim_name_tabulated {
	    $icolwtt_w insert list end $i
	}
	$icolwtt_w selection set $tmp_sisimpar(icolwtt)

    } elseif { $sisimpar(tabfl) == {*.dat} } {
	unset dim_name_tabulated
	if { [sisim_read_tabulated_values $sisimpar(tabfl)] != 1 } {
	    set dim_name_tabulated {}
	    lappend dim_name_tabulated "None"
	    for { set i 1 } { $i < 6 } { incr i } {
		lappend dim_name_tabulated $i
	    }
	}
	
	$icolvrt_w clear list
	foreach i $dim_name_tabulated {
	    $icolvrt_w insert list end $i
	}
	$icolvrt_w selection set 4
	set sisimpar(icolvrt) [lindex $dim_name_tabulated 4]
	
	$icolwtt_w clear list
	foreach i $dim_name_tabulated {
	    $icolwtt_w insert list end $i
	}
	$icolwtt_w selection set 5
	set sisimpar(icolwtt) [lindex $dim_name_tabulated 5]
    }

    ### reads the idbg: debugging level (0=none, 1=low, 2=middle, 3=high)
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(idbg) [format "%d" [string range $line 0 $space]]
    $dbgfl_level_w selection set $sisimpar(idbg)
    
    ### reads the dbgfl: file for debugging output
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set sisimpar(dbgfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set sisimpar(dbgfl) [format "%s" $line]
    }

    ### reads the outfl: file for simulation output
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set sisimpar(outfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set sisimpar(outfl) [format "%s" $line]
    }

    ### reads the nsim: number of realizations
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(nsim) [format "%g" [string range $line 0 $space]]
    
    ### reads the nx, xmn, xsiz: number, coordinate, size of grid cells
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(nx) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(xmn) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(xsiz) [format "%g" [string range $line 0 [expr $space - 1]]]
    
    ### reads the ny, ymn, ysiz: number, coordinate, size of grid cells
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(ny) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(ymn) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(ysiz) [format "%g" [string range $line 0 [expr $space - 1]]]
    
    ### reads the nz, zmn, zsiz: number, coordinate, size of grid cells
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(nz) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(zmn) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(zsiz) [format "%g" [string range $line 0 [expr $space - 1]]]
    
    set grid(xmax) [expr (($sisimpar(nx)-1)*$sisimpar(xsiz))+$sisimpar(xmn)]
    set grid(xtot) [expr $sisimpar(nx)*$sisimpar(xsiz)]
    set grid(ymax) [expr (($sisimpar(ny)-1)*$sisimpar(ysiz))+$sisimpar(ymn)]
    set grid(ytot) [expr $sisimpar(ny)*$sisimpar(ysiz)]
    set grid(zmax) [expr (($sisimpar(nz)-1)*$sisimpar(zsiz))+$sisimpar(zmn)]
    set grid(ztot) [expr $sisimpar(nz)*$sisimpar(zsiz)]
    set grid(cell) [expr $sisimpar(nx)*$sisimpar(ny)*$sisimpar(nz)]
    set grid(area) [expr $sisimpar(nx)*$sisimpar(xsiz)*$sisimpar(ny)*$sisimpar(ysiz)]
    set grid(volume) [expr $sisimpar(nx)*$sisimpar(xsiz)*$sisimpar(ny)*$sisimpar(ysiz) \
			  *$sisimpar(nz)*$sisimpar(zsiz)]
    
    ### reads the seed: random number seed
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(seed) [format "%d" [string range $line 0 $space]]

    ### reads the ndmax: maximum original data for each kriging
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(ndmax) [format "%d" [string range $line 0 $space]]

    ### reads the ncnode: maximum previous nodes for each kriging
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(ncnode) [format "%d" [string range $line 0 $space]]

    ### reads the maxsec: soft indicator nodes for kriging
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(maxsec) [format "%d" [string range $line 0 $space]]

    ### reads the sstrat: search strat., ass. data to nodes? (0=no, 1=yes)
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(sstrat) [format "%d" [string range $line 0 $space]]
    if { $sisimpar(sstrat) == 1 } {
	$max_org_data_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $sisimpar(sstrat) == 0 } {
	$max_org_data_w configure -state normal -textbackground white -foreground black
    }
    
    ### reads the multgrid, nmult: grid refinement? (0=no, 1=yes), num.
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(multgrid) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(nmult) [format "%d" [string range $line 0 [expr $space - 1]]]
    if { $sisimpar(multgrid) == 1 } {
	$grid_refinement_number_w configure -state normal -textbackground white -foreground black
    } elseif { $sisimpar(multgrid) == 0 } {
	$grid_refinement_number_w configure -state disabled -textbackground gray85 -foreground gray50
    }

    ### reads the noct: maximum per octant (0=not used)
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(noct) [format "%d" [string range $line 0 $space]]

    ### reads the radius_hmax, radius_hmin, radius_vert: search radii
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(radius_hmax) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(radius_hmin) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(radius_vert) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the sang1, sang2, sang3: angles for search ellipsoid
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(sang1) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(sang2) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(sang3) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the mik, mikcat: 0=full IK, 1=median IK approx., cut/cat
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(mik) [format "%d" [string range $line 0 [expr $space - 1]]]
    $kriging_flag_w selection set $sisimpar(mik)
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set sisimpar(mikcat) [format "%g" [string range $line 0 [expr $space - 1]]]
    if { $sisimpar(mik) == "Full IK" } {
	$cutoff_category_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $sisimpar(mik) == "Median IK" } {
	$cutoff_category_w configure -state normal -textbackground white -foreground black
    }	

    ### reads the ktype: kriging type (0=simple kriging, 1=ordinary kriging)
    set line [gets $inf]
    set space [string first " " $line]
    set sisimpar(ktype) [format "%d" [string range $line 0 [expr $space - 1]]]
    $kriging_type_w selection set $sisimpar(ktype)
    
    ### reads the nst, c0: number of nested models, nugget effect
    for { set ii 1 } { $ii <= $sisimpar(ncat) } { incr ii } {
	set focus_variable $ii
	set line [gets $inf]
	set space [string first " " $line]
	set no($ii) [format "%d" [string range $line 0 [expr $space - 1]]]   ;# no corresponds with nst
	set no($focus_variable) $no($ii)
	set line [string range $line [expr $space + 1] end]
	set space [string first " " $line]
	set sisimpar(c0$ii) [format "%g" [string range $line 0 [expr $space - 1]]]

	### creates model_table
	set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
				    -orient vertical \
				    -command "$model_sframe_w.table_$ii yview"]
	update
	
	set model_table_w($ii) [table $model_sframe_w.table_$ii \
				    -variable model_vals \
				    -titlerows 1 \
				    -titlecols 0 \
				    -cols 9 \
				    -rows [expr $no($ii) + 1] \
				    -yscrollcommand "$model_hsbar_w($ii) set" \
				    -height 4]
	
	$model_table_w($ii) width 0 3
	$model_table_w($ii) width 1 15
	$model_table_w($ii) width 2 8
	$model_table_w($ii) width 3 8
	$model_table_w($ii) width 4 8
	$model_table_w($ii) width 5 8
	$model_table_w($ii) width 6 8
	$model_table_w($ii) width 7 8
	$model_table_w($ii) width 8 8
	
	set model_vals(0,0) "No"
	set model_vals(0,1) "Type"
	set model_vals(0,2) "Cc"
	set model_vals(0,3) "Azimuth"
	set model_vals(0,4) "Dip"
	set model_vals(0,5) "Plunge"
	set model_vals(0,6) "hmax"
	set model_vals(0,7) "hmin"
	set model_vals(0,8) "vert"
	update
	
	
	set sum_cc($ii) 0                ;# sum of all cc values
	for { set i 1 } { $i <= $no($ii) } { incr i } {
	    ### reads the it, cc, ang1, ang2, ang3: model, cc, ang. structure ellipsoid
	    set line [gets $inf]
	    set space [string first " " $line]
	    set sisimpar(nst$ii,$i) [format "%d" [string range $line 0 [expr $space - 1]]]
	    set line [string range $line [expr $space + 1] end]
	    set space [string first " " $line]
	    set sisimpar(cc$ii,$i) [format "%g" [string range $line 0 [expr $space - 1]]]
	    set line [string range $line [expr $space + 1] end]
	    set space [string first " " $line]
	    set sisimpar(ang1$ii,$i) [format "%g" [string range $line 0 [expr $space - 1]]]
	    set line [string range $line [expr $space + 1] end]
	    set space [string first " " $line]
	    set sisimpar(ang2$ii,$i) [format "%g" [string range $line 0 [expr $space - 1]]]
	    set line [string range $line [expr $space + 1] end]
	    set space [string first " " $line]
	    set sisimpar(ang3$ii,$i) [format "%g" [string range $line 0 [expr $space - 1]]]
	    
	    ### reads the aa_hmax, aa_hmin, aa_vert: radii of structure ellipsoid
	    set line [gets $inf]
	    set space [string first " " $line]
	    set sisimpar(aa_hmax$ii,$i) [format "%g" [string range $line 0 [expr $space - 1]]]
	    set line [string range $line [expr $space + 1] end]
	    set space [string first " " $line]
	    set sisimpar(aa_hmin$ii,$i) [format "%g" [string range $line 0 [expr $space - 1]]]
	    set line [string range $line [expr $space + 1] end]
	    set space [string first " " $line]
	    set sisimpar(aa_vert$ii,$i) [format "%g" [string range $line 0 [expr $space - 1]]]
	    
	    ### fills model_table
	    set no_counter($ii) $i       ;# counter of the number of variogram types with corresponding parameters
	    
	    set model_vals_label($i,$ii) [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
					      -labeltext "$i" \
					      -background gray50 \
					      -foreground white]
	    update
	    
	    set model_type($i,combo$ii) [iwidgets::combobox $model_table_w($ii).combo_type$i \
					     -width 15 \
					     -textvariable sisimpar(it$ii,$i) \
					     -listheight 75]
	    $model_table_w($ii).combo_type$i insert list end "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
	    $model_type($i,combo$ii) selection set [expr $sisimpar(nst$ii,$i) - 1]
	    set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i [expr $sisimpar(nst$ii,$i) - 1]]
	    $model_table_w($ii).combo_type$i configure -state disabled -textbackground gray85 -foreground gray50
	    update
	    
	    set model_cc($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
					   -textvariable sisimpar(cc$ii,$i) \
					   -state disabled \
					   -textbackground gray85 \
					   -foreground gray50 \
					   -validate "check_number_cc %W %S %P $i $ii"]
	    update
	    
	    set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]
	    
	    set model_ang1($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
					     -textvariable sisimpar(ang1$ii,$i) \
					     -state disabled \
					     -textbackground gray85 \
					     -foreground gray50 \
					     -validate {check_number_sang1 %W %P %c}]
	    update
	    
	    set model_ang2($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang2$i \
					     -textvariable sisimpar(ang2$ii,$i) \
					     -state disabled \
					     -textbackground gray85 \
					     -foreground gray50 \
					     -validate {check_number_sang %W %P %c}]
	    update
	    
	    set model_ang3($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang3$i \
					     -textvariable sisimpar(ang3$ii,$i) \
					     -state disabled \
					     -textbackground gray85 \
					     -foreground gray50 \
					     -validate {check_number_sang %W %P %c}]
	    update
	    
	    set model_aa_hmax($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
						-textvariable sisimpar(aa_hmax$ii,$i) \
						-state disabled \
						-textbackground gray85 \
						-foreground gray50 \
						-validate {check_number_radius %W %P %c}]
	    update
	    
	    set model_aa_hmin($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
						-textvariable sisimpar(aa_hmin$ii,$i) \
						-state disabled \
						-textbackground gray85 \
						-foreground gray50 \
						-validate {check_number_radius %W %P %c}]
	    update
	    
	    set model_aa_vert($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
						-textvariable sisimpar(aa_vert$ii,$i) \
						-state disabled \
						-textbackground gray85 \
						-foreground gray50 \
						-validate {check_number_radius %W %P %c}]
	    update
	    
	    $model_table_w($ii) window configure $i,0 -window $model_vals_label($i,$ii) -relief flat -background gray50
	    $model_table_w($ii) window configure $i,1 -window $model_type($i,combo$ii)
	    $model_table_w($ii) window configure $i,2 -window $model_cc($i,entry$ii)
	    $model_table_w($ii) window configure $i,3 -window $model_ang1($i,entry$ii)
	    $model_table_w($ii) window configure $i,4 -window $model_ang2($i,entry$ii)	
	    $model_table_w($ii) window configure $i,5 -window $model_ang3($i,entry$ii)	
	    $model_table_w($ii) window configure $i,6 -window $model_aa_hmax($i,entry$ii)	
	    $model_table_w($ii) window configure $i,7 -window $model_aa_hmin($i,entry$ii)	
	    $model_table_w($ii) window configure $i,8 -window $model_aa_vert($i,entry$ii)
	}
	pack $model_hsbar_w($ii)  -side right -anchor w -fill y
	pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
	
	### creates nugget entry and sill label
	set models_nugget_w($ii) [iwidgets::entryfield $subframe3_variogram_w.entry_nugget$ii \
				      -labeltext "Nugget " \
				      -width 7 \
				      -textvariable sisimpar(c0$ii) \
				      -state disabled \
				      -textbackground gray85 \
				      -foreground gray50 \
				      -validate "check_number_nugget %W %P"]
	update
	
	set models_sill_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
					  -labeltext "Sill " \
					  -state disabled]
	update
	set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
	set models_sill_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
					  -labelvariable sill($ii) \
					  -state disabled]
	update
	
	set cut_cat_number_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_label$ii \
					     -labeltext "Cut/Cat \# " \
					     -state disabled]
	update
	set cut_cat_number($ii) $ii
	set cut_cat_number_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_value$ii \
					     -labeltext "$cut_cat_number($ii)" \
					     -state disabled]
	update
	
	grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	grid $models_nugget_w($ii)        -row 0 -column 2 -sticky e -pady 0
	grid $models_sill_label_w($ii)    -row 0 -column 3 -sticky e -pady 0
	grid $models_sill_value_w($ii)    -row 0 -column 4 -sticky w -pady 0
	grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
    }
    
    for { set ii 1 } { $ii <= $sisimpar(ncat) } { incr ii } {
	pack forget $model_hsbar_w($ii)
	pack forget $model_table_w($ii)
	grid forget $models_nugget_w($ii)
	grid forget $models_sill_label_w($ii)
	grid forget $models_sill_value_w($ii)
	grid forget $cut_cat_number_label_w($ii)
	grid forget $cut_cat_number_value_w($ii)
    }
    set ii $sisimpar(ncat)
    
    pack configure $model_hsbar_w($ii) -side right -anchor w -fill y
    pack configure $model_table_w($ii) -side left -anchor w -padx 2 -expand 1 -fill both
    
    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
    grid $models_nugget_w($ii)     -row 0 -column 2 -sticky e -pady 0
    grid $models_sill_label_w($ii) -row 0 -column 3 -sticky e -pady 0
    grid $models_sill_value_w($ii) -row 0 -column 4 -sticky w -pady 0
    grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
    grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
    
    ### sets the model pushbuttons, the nugget, sill, cut/cat number and the model table normal
    $plus_model_pushbutton_w configure -state normal
    $minus_model_pushbutton_w configure -state normal
    
    for { set ii 1 } { $ii <= $sisimpar(ncat) } { incr ii } {
	
	$models_nugget_w($ii) configure -state normal -textbackground white -foreground black
	$models_sill_label_w($ii) configure -state normal
	$models_sill_value_w($ii) configure -state normal
	$cut_cat_number_label_w($ii) configure -state normal
	$cut_cat_number_value_w($ii) configure -state normal
	
	for { set i 1 } { $i <= $no($ii) } { incr i } {
	    
	    $model_type($i,combo$ii) configure -state normal -textbackground white -foreground black
	    $model_cc($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_ang1($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_ang2($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_ang3($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_aa_hmax($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_aa_hmin($i,entry$ii) configure -state normal -textbackground white -foreground black
	    $model_aa_vert($i,entry$ii) configure -state normal -textbackground white -foreground black
	}
    }
    
    ### the focus variable is highlighted in the icoli_table and the pcf_cdf_table
    if { $sisimpar(directik) != {*.dat} } {
	for { set ii 1 } { $ii <= $sisimpar(ncat) } { incr ii } {
	    $icoli_vals($ii,label) configure -background gray50 -foreground white
	    $icoli_table_w window configure $ii,0 -relief flat -background gray50
	    $pdf_cdf_vals($ii,label) configure -background gray50 -foreground white
	    $pdf_cdf_table_w window configure $ii,0 -relief flat -background gray50
	}
	$icoli_vals($focus_variable,label) configure -background yellow -foreground black
	$icoli_table_w window configure $focus_variable,0 -relief flat -background yellow
	$pdf_cdf_vals($focus_variable,label) configure -background yellow -foreground black
	$pdf_cdf_table_w window configure $focus_variable,0 -relief flat -background yellow
    } elseif { $sisimpar(directik) == {*.dat} } {
	for { set ii 1 } { $ii <= 3 } { incr ii } {
	    $icoli_vals($ii,label) configure -background gray50 -foreground white
	    $icoli_table_w window configure $ii,0 -relief flat -background gray50
	    $pdf_cdf_vals($ii,label) configure -background gray50 -foreground white
	    $pdf_cdf_table_w window configure $ii,0 -relief flat -background gray50
	}
	$pdf_cdf_vals($focus_variable,label) configure -background yellow -foreground black
	$pdf_cdf_table_w window configure $focus_variable,0 -relief flat -background yellow
    }

    ### reset the cancel counters to zero
    set sisimpar(hard_cancel_counter) 0
    set sisimpar(tabulated_cancel_counter) 0
    set sisimpar(soft_cancel_counter) 0

    
    ### reset the cursor on arrow 
    . configure -cursor arrow
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc write_sisim_params { fname {tmpflag 0} } {
    global tcl_platform
    global version
    global sisimpar
    global variable_type_w
    global icolx_w
    global icoly_w
    global icolz_w
    global icolvr_w
    global icolsx_w
    global icolsy_w
    global icolsz_w
    global icoli_w
    global icoli_vals
    global extrapol_lower_tail_w
    global interpol_middle_w
    global extrapol_upper_tail_w
    global icolvrt_w
    global icolwtt_w
    global dbgfl_level_w
    global realization_w
    global max_org_data_w
    global max_soft_nodes_w
    global max_prev_nodes_w
    global max_per_octant_w
    global kriging_flag_w
    global kriging_type_w
    global no
    global model_type
    global sisim_box_w
    
    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }
    
    ### set file name in sisim box widget
    if { $sisimpar(title) == 0 } {
	set param_dname [file dirname $fname]
	set dname_length [string length $param_dname]
	set fname_short [string range $fname [expr $dname_length + 1] end]
	wm title $sisim_box_w "SISISM parameters   -   $fname_short"
    }

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    
    puts $outf "Parameters for SISIM"
    puts $outf "********************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### vartype: variable type (1=cont.(cdf), 0=cat.(pdf))
    set comment "\\vartype: variable type (1=cont.(cdf), 0=cat.(pdf))"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$variable_type_w curselection] $comment]
    
    ### ncat: number of thresholds / categories
    set comment "\\ncat: number of thresholds / categories"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(ncat) $comment]
    
    ### cat: thresholds / categories
    set line_length_total 0
    if { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat} } {
	for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
	    if { $sisimpar(vartype) == "Categorical" } {
		puts -nonewline $outf "[format "%g" $sisimpar(cat,$ii)] "
		set line [format "%g" $sisimpar(cat,$ii)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
	    } elseif { $sisimpar(vartype) == "Continous" } {
		puts -nonewline $outf "[format "%g" $sisimpar(cat,$ii)] "
		set line [format "%g" $sisimpar(cat,$ii)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
	    }
	}
    } elseif { ($sisimpar(datafl) == {*.dat} && $sisimpar(directik) != {*.dat}) || \
		   ($sisimpar(datafl) != {*.dat} && $sisimpar(directik) != {*.dat}) } {
	for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3] } { incr ii } {
	    if { $sisimpar(vartype) == "Categorical" } {
		puts -nonewline $outf "[format "%g" $sisimpar(cat,$ii)] "
		set line [format "%g" $sisimpar(cat,$ii)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
	    } elseif { $sisimpar(vartype) == "Continous" } {
		puts -nonewline $outf "[format "%g" $sisimpar(cat,$ii)] "
		set line [format "%g" $sisimpar(cat,$ii)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
	    }
	}
    }
    set comment "\\cat: thresholds / categories"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### pdf: global cdf / pdf
    set line_length_total 0
    if { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat} } {
	for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
	    puts -nonewline $outf "[format "%g" $sisimpar(pdf,$ii)] "
	    set line [format "%g" $sisimpar(pdf,$ii)]
	    set line_length [string length $line]
	    set line_length_total [expr $line_length_total + $line_length + 1]
	}
    } elseif { ($sisimpar(datafl) == {*.dat} && $sisimpar(directik) != {*.dat}) || \
		   ($sisimpar(datafl) != {*.dat} && $sisimpar(directik) != {*.dat}) } {
	for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3] } { incr ii } {
	    puts -nonewline $outf "[format "%g" $sisimpar(pdf,$ii)] "
	    set line [format "%g" $sisimpar(pdf,$ii)]
	    set line_length [string length $line]
	    set line_length_total [expr $line_length_total + $line_length + 1]
	}
    }
    set comment "\\pdf: global cdf / pdf"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### datafl: file with hard data
    set comment "\\datafl: file with hard data"
    if { $tmpflag } {
	catch {
	    file copy -force $sisimpar(datafl) "sisimh_in.dat"
	}
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "sisimh_in.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $sisimpar(datafl)]
    }
    set line [format "%s" $sisimpar(datafl)]
    set line_length_total [string length $line]
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### icolx, ..y, ..z, ..vr: columns for x, y, z, variable
    puts -nonewline $outf [format "%d %d " [$icolx_w curselection] [$icoly_w curselection]]
    puts -nonewline $outf [format "%d %d" [$icolz_w curselection] [$icolvr_w curselection]]
    set line [format "%d %d %d %d" [$icolx_w curselection] [$icoly_w curselection] \
		  [$icolz_w curselection] [$icolvr_w curselection]]
    set line_length [string length $line]
    set comment "\\icolx, ..y, ..z, ..vr: columns for x, y, z, variable"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### directik: file with soft data
    set comment "\\directik: file with soft data"
    if { $tmpflag } {
	catch {
	    file copy -force $sisimpar(direcik) "sisims_in.dat"
	}
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "sisims_in.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $sisimpar(directik)]
    }
    set line [format "%s" $sisimpar(directik)]
    set line_length_total [string length $line]
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### icolsx, ..sy, ..sz, ..i: columns for x, y, z, indic.
    set line_length_total 0
    if { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat} } {
	puts -nonewline $outf [format "%d %d " $sisimpar(icolsx) $sisimpar(icolsy)]
	puts -nonewline $outf [format "%d " $sisimpar(icolsz)]
	set line [format "%d %d %d" $sisimpar(icolsx) $sisimpar(icolsy) $sisimpar(icolsz)]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
	for {set ii 1} {$ii <= 3} {incr ii} {
	    puts -nonewline $outf "[format "%d " [$icoli_vals($ii,combo) curselection]]"
	    set line [format "%d" [$icoli_vals($ii,combo) curselection]]
	    set line_length [string length $line]
	    set line_length_total [expr $line_length_total + $line_length + 1]
	}
    } elseif { ($sisimpar(datafl) == {*.dat} && $sisimpar(directik) != {*.dat}) || \
		   ($sisimpar(datafl) != {*.dat} && $sisimpar(directik) != {*.dat}) } {
	puts -nonewline $outf [format "%d %d " [$icolsx_w curselection] [$icolsy_w curselection]]
	puts -nonewline $outf [format "%d " [$icolsz_w curselection]]
	set line [format "%d %d %d" [$icolsx_w curselection] [$icolsy_w curselection] [$icolsz_w curselection]]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
	for {set ii 1} {$ii <= [expr $sisimpar(variables) - 3]} {incr ii} {
	    puts -nonewline $outf "[format "%d " [$icoli_vals($ii,combo) curselection]]"
	    set line [format "%d" [$icoli_vals($ii,combo) curselection]]
	    set line_length [string length $line]
	    set line_length_total [expr $line_length_total + $line_length + 1]
	}
    }
    set comment "\\icolsx, ..sy, ..sz, ..i: columns for x, y, z, indic."
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### imbsim: Markov-Bayes simulation (0=no, 1=yes)
    set comment "\\imbsim: Markov-Bayes simulation (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(imbsim) $comment]
    
    ### b(z): calibration B(z) values
    set line_length_total 0
    if { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat} } {
	for {set ii 1} {$ii <= 3} {incr ii} {
	    puts -nonewline $outf "[format "%g" $sisimpar(bz,$ii)] "
	    set line [format "%g" $sisimpar(bz,$ii)]
	    set line_length [string length $line]
	    set line_length_total [expr $line_length_total + $line_length + 1]
	}
    } elseif { ($sisimpar(datafl) == {*.dat} && $sisimpar(directik) != {*.dat}) || \
		   ($sisimpar(datafl) != {*.dat} && $sisimpar(directik) != {*.dat}) } {
	for {set ii 1} {$ii <= [expr $sisimpar(variables) - 3]} {incr ii} {
	    puts -nonewline $outf "[format "%g" $sisimpar(bz,$ii)] "
	    set line [format "%g" $sisimpar(bz,$ii)]
	    set line_length [string length $line]
	    set line_length_total [expr $line_length_total + $line_length + 1]
	}
    }
    set comment "\\b(z): calibration B(z) values"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### tmin, tmax: trimming limits
    catch {scan $sisimpar(tmin) {%g} sisimpar(tmin)}
    catch {scan $sisimpar(tmax) {%g} sisimpar(tmax)}
    puts -nonewline $outf [format "%g %g" $sisimpar(tmin) $sisimpar(tmax)]
    set line [format "%g %g" $sisimpar(tmin) $sisimpar(tmax)]
    set line_length [string length $line]
    set comment "\\tmin, tmax: trimming limits"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### zmin, zmax: minimum and maximum data value (continous variable)
    puts -nonewline $outf [format "%g %g" $sisimpar(zmin) $sisimpar(zmax)]
    set line [format "%g %g" $sisimpar(zmin) $sisimpar(zmax)]
    set line_length [string length $line]
    set comment "\\zmin, zmax: minimum and maximum data value"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### ltail, ltpar: extrapolation lower tail, option, param.
    puts -nonewline $outf [format "%d %g" [expr [$extrapol_lower_tail_w curselection] + 1] $sisimpar(ltpar)]
    set line [format "%d %g" [expr [$extrapol_lower_tail_w curselection] + 1] $sisimpar(ltpar)]
    set line_length [string length $line]
    set comment "\\ltail, ltpar: extrapolation lower tail, option param."
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### middle, midpar: extrapolation middle, option param.
    puts -nonewline $outf [format "%d %g" [expr [$interpol_middle_w curselection] + 1] $sisimpar(midpar)]
    set line [format "%d %g" [expr [$interpol_middle_w curselection] + 1] $sisimpar(midpar)]
    set line_length [string length $line]
    set comment "\\middle, midpar: extrapolation middle, option param."
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### utail, utpar: extrapolation upper tail, option and param.
    puts -nonewline $outf [format "%d %g" [expr [$extrapol_upper_tail_w curselection] + 1] $sisimpar(utpar)]
    set line [format "%d %g" [expr [$extrapol_upper_tail_w curselection] + 1] $sisimpar(utpar)]
    set line_length [string length $line]
    set comment "\\utail, utpar: extrapolation upper tail, option param."
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### tabfl: file with tabulated values
    set comment "\\tabfl: file with tabulated values"
    if { $tmpflag } {
	catch {
	    file copy -force $sisimpar(tabfl) "sisim_in.tab"
	}
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "sisim_in.tab"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $sisimpar(tabfl)]
    }
    set line [format "%s" $sisimpar(tabfl)]
    set line_length_total [string length $line]
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### icolvrt, icolwtt: columns for var., weight (0=not used)
    if { $sisimpar(tabfl) == {*.dat} } {
	puts -nonewline $outf [format "%d %d" $sisimpar(icolvrt) $sisimpar(icolwtt)]
	set line [format "%d %d" $sisimpar(icolvrt) $sisimpar(icolwtt)]
	set line_length [string length $line]
    } elseif { $sisimpar(tabfl) != {*.dat} } {
	puts -nonewline $outf [format "%d %d" [$icolvrt_w curselection] [$icolwtt_w curselection]]
	set line [format "%d %d" [$icolvrt_w curselection] [$icolwtt_w curselection]]
	set line_length [string length $line]
    }
    set comment "\\icolvrt, icolwtt: columns for var., weight (0=not used)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### idbg: debugging level (0=none, 1=low, 2=middle, 3=high)
    set comment "\\idbg: debugging level (0=none, 1=low, 2=middle, 3=high)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$dbgfl_level_w curselection] $comment]
    
    ### dbgfl: file for debugging output
    set comment "\\dbgfl: file for debugging output"
    if { $tmpflag } {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "sisim_out.dbg"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $sisimpar(dbgfl)]
    }
    set line [format "%s" $sisimpar(dbgfl)]
    set line_length_total [string length $line]
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### outfl: file for simulation output
    set comment "\\outfl: file for simulation output"
    if { $tmpflag } {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "sisim_out.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $sisimpar(outfl)]
    }
    set line [format "%s" $sisimpar(outfl)]
    set line_length_total [string length $line]
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### nsim: number of realizations
    set comment "\\nsim: number of realizations"
    catch {scan [$realization_w get] {%d} sisimpar(nsim)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(nsim) $comment]
    
    ### nx, xmn, xsiz: number, coordinate, size of grid cells
    puts -nonewline $outf [format "%g %g %g" $sisimpar(nx) $sisimpar(xmn) $sisimpar(xsiz)]
    set line [format "%g %g %g" $sisimpar(nx) $sisimpar(xmn) $sisimpar(xsiz)]
    set line_length [string length $line]
    set comment "\\nx, xmn, xsiz: number, coordinate, size of grid cells"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### ny, ymn, ysiz: number, coordinate, size of grid cells
    puts -nonewline $outf [format "%g %g %g" $sisimpar(ny) $sisimpar(ymn) $sisimpar(ysiz)]
    set line [format "%g %g %g" $sisimpar(ny) $sisimpar(ymn) $sisimpar(ysiz)]
    set line_length [string length $line]
    set comment "\\ny, ymn, ysiz: number, coordinate, size of grid cells"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### nz, zmn, zsiz: number, coordinate, size of grid cells
    puts -nonewline $outf [format "%g %g %g" $sisimpar(nz) $sisimpar(zmn) $sisimpar(zsiz)]
    set line [format "%g %g %g" $sisimpar(nz) $sisimpar(zmn) $sisimpar(zsiz)]
    set line_length [string length $line]
    set comment "\\nz, zmn, zsiz: number, coordinate, size of grid cells"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### seed: random number seed
    set comment "\\seed: random number seed"
    catch {scan $sisimpar(seed) {%d} sisimpar(seed)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(seed) $comment]
    
    ### ndmax: maximum original data for each kriging
    set comment "\\ndmax: maximum original data for each kriging"
    catch {scan [$max_org_data_w get] {%d} sisimpar(ndmax)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(ndmax) $comment]
    
    ### ncnode: maximum previous nodes for each kriging
    set comment "\\ncnode: maximum previous nodes for each kriging"
    catch {scan [$max_prev_nodes_w get] {%d} sisimpar(ncnode)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(ncnode) $comment]
    
    ### maxsec: soft indicator nodes for kriging
    set comment "\\maxsec: maximum soft indicator nodes for kriging"
    catch {scan [$max_soft_nodes_w get] {%d} sisimpar(maxsec)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(maxsec) $comment]
    
    ### sstrat: search strat., ass. data to nodes? (0=no, 1=yes)
    set comment "\\sstrat: search strat., ass. data to nodes? (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(sstrat) $comment]
    
    ### multgrid, nmult: grid refinement? (0=no, 1=yes), num.
    catch {scan $sisimpar(nmult) {%d} sisimpar(nmult)}
    puts -nonewline $outf [format "%d %d" $sisimpar(multgrid) $sisimpar(nmult)]
    set line [format "%g %d" $sisimpar(multgrid) $sisimpar(nmult)]
    set line_length [string length $line]
    set comment "\\multgrid, nmult: grid refinement? (0=no, 1=yes), num."
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### noct: maximum per octant (0=not used)
    set comment "\\noct: maximum per octant (0=not used)"
    catch {scan [$max_per_octant_w get] {%d} sisimpar(noct)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(noct) $comment]
    
    ### radius_hmax, radius_hmin, radius_vert: search radii
    puts -nonewline $outf [format "%g %g %g" $sisimpar(radius_hmax) $sisimpar(radius_hmin) $sisimpar(radius_vert)]
    set line [format "%g %g %g" $sisimpar(radius_hmax) $sisimpar(radius_hmin) $sisimpar(radius_vert)]
    set line_length [string length $line]
    set comment "\\radius_hmax, radius_hmin, radius_vert: search radii"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### sang1, sang2, sang3: angles for search ellipsoid
    puts -nonewline $outf [format "%g %g %g" $sisimpar(sang1) $sisimpar(sang2) $sisimpar(sang3)]
    set line [format "%g %g %g" $sisimpar(sang1) $sisimpar(sang2) $sisimpar(sang3)]
    set line_length [string length $line]
    set comment "\\sang1, sang2, sang3: angles for search ellipsoid"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### mik, mikcat: 0=full IK, 1=median IK approx., cut/cat
    if { $sisimpar(mikcat) == {} } {
	set sisimpar(mikcat) 0
    }
    puts -nonewline $outf [format "%d %g" [$kriging_flag_w curselection] $sisimpar(mikcat)]
    set line [format "%d %g" [$kriging_flag_w curselection] $sisimpar(mikcat)]
    set line_length [string length $line]
    set comment "\\mik and mikcat: 0=full IK, 1=median IK approx., cut/cat"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### ktype: kriging type (0=simple kriging, 1=ordinary kriging)
    set comment "\\ktype: kriging type (0=simple kriging, 1=ordinary kriging)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$kriging_type_w curselection] $comment]
    
    ### nst, c0: number of nested models, nugget effect
    if { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat} } {
	for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
	    set line_length_total 0
	    puts -nonewline $outf "[format "%d %g" $no($ii) $sisimpar(c0$ii)] "
	    set line [format "%d %g" $no($ii) $sisimpar(c0$ii)]
	    set line_length [string length $line]
	    set line_length_total [expr $line_length_total + $line_length + 1]
	    set comment "\\nst, c0: number of nested models, nugget effect"
	    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		### it, cc, ang1, ang2, ang3: model, cc, ang. structure ellipsoid
		set line_length_total 0
		puts -nonewline $outf "[format "%d" [expr [$model_type($i,combo$ii) curselection] + 1]] "
		puts -nonewline $outf "[format "%g %g" $sisimpar(cc$ii,$i) $sisimpar(ang1$ii,$i)] "
		puts -nonewline $outf "[format "%g %g" $sisimpar(ang2$ii,$i) $sisimpar(ang3$ii,$i)] "
		set line [format "%d %g %g %g %g" [expr [$model_type($i,combo$ii) curselection] + 1] \
			      $sisimpar(cc$ii,$i) $sisimpar(ang1$ii,$i) $sisimpar(ang2$ii,$i) $sisimpar(ang3$ii,$i)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
		set comment "\\it, cc, ang1, ang2, ang3: model, cc, ang. struct. ellips."
		if { $line_length_total < $parameter_space } {
		    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
		} elseif { $line_length_total >= $parameter_space } {
		    set parameter_space [expr $line_length_total + 1]
		    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
		    set parameter_space 24
		}
		### aa_hmax, aa_hmin, aa_vert: radii of structure ellipsoid
		set line_length_total 0
		puts -nonewline $outf "[format "%g %g" $sisimpar(aa_hmax$ii,$i) $sisimpar(aa_hmin$ii,$i)] "
		puts -nonewline $outf "[format "%g" $sisimpar(aa_vert$ii,$i)] "
		set line [format "%g %g %g" $sisimpar(aa_hmax$ii,$i) $sisimpar(aa_hmin$ii,$i) $sisimpar(aa_vert$ii,$i)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
		set comment "\\aa_hmax, aa_hmin, aa_vert: radii of structure ellipsoid"
		if { $line_length_total < $parameter_space } {
		    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
		} elseif { $line_length_total >= $parameter_space } {
		    set parameter_space [expr $line_length_total + 1]
		    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
		    set parameter_space 24
		}
	    }
	}
    } elseif { ($sisimpar(datafl) == {*.dat} && $sisimpar(directik) != {*.dat}) || \
		   ($sisimpar(datafl) != {*.dat} && $sisimpar(directik) != {*.dat}) } {
	for {set ii 1} {$ii <= [expr $sisimpar(variables) - 3]} {incr ii} {
	    set line_length_total 0
	    puts -nonewline $outf "[format "%d %g" $no($ii) $sisimpar(c0$ii)] "
	    set line [format "%d %g" $no($ii) $sisimpar(c0$ii)]
	    set line_length [string length $line]
	    set line_length_total [expr $line_length_total + $line_length + 1]
	    set comment "\\nst, c0: number of nested models, nugget effect"
	    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		### it, cc, ang1, ang2, ang3: structure type, c parameter, ang. struct. ellips.
		set line_length_total 0
		puts -nonewline $outf "[format "%d" [expr [$model_type($i,combo$ii) curselection] + 1]] "
		puts -nonewline $outf "[format "%g %g" $sisimpar(cc$ii,$i) $sisimpar(ang1$ii,$i)] "
		puts -nonewline $outf "[format "%g %g" $sisimpar(ang2$ii,$i) $sisimpar(ang3$ii,$i)] "
		set line [format "%d %g %g %g %g" [expr [$model_type($i,combo$ii) curselection] + 1] \
			      $sisimpar(cc$ii,$i) $sisimpar(ang1$ii,$i) $sisimpar(ang2$ii,$i) $sisimpar(ang3$ii,$i)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
		set comment "\\it, cc, ang1, ang2, ang3: model, cc, ang. struct. ellips."
		if { $line_length_total < $parameter_space } {
		    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
		} elseif { $line_length_total >= $parameter_space } {
		    set parameter_space [expr $line_length_total + 1]
		    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
		    set parameter_space 24
		}
		### aa_hmax, aa_hmin, aa_vert: radii of structure ellipsoid
		set line_length_total 0
		puts -nonewline $outf "[format "%g %g" $sisimpar(aa_hmax$ii,$i) $sisimpar(aa_hmin$ii,$i)] "
		puts -nonewline $outf "[format "%g" $sisimpar(aa_vert$ii,$i)] "
		set line [format "%g %g %g" $sisimpar(aa_hmax$ii,$i) $sisimpar(aa_hmin$ii,$i) $sisimpar(aa_vert$ii,$i)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
		set comment "\\aa_hmax, aa_hmin, aa_vert: radii of structure ellipsoid"
		if { $line_length_total < $parameter_space } {
		    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
		} elseif { $line_length_total >= $parameter_space } {
		    set parameter_space [expr $line_length_total + 1]
		    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
		    set parameter_space 24
		}
	    }
	}
    }
    close $outf
}
#============================================================================


#############################################################################
proc write_sisim_infofl { } {
    global tcl_platform
    global version
    global sisimpar
    global variable_type_w
    global outf

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    
    puts $outf "Information for SISIM data export"
    puts $outf "*********************************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
			      
    ### vartype: variable type (1=cont.(cdf), 0=cat.(pdf))
    set comment "\\vartype: variable type (1=cont.(cdf), 0=cat.(pdf))"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$variable_type_w curselection] $comment]

    ### nsim: number of realizations
    set comment "\\nsim: number of realizations"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(nsim) $comment]

    ### nx, xsiz: number, size of grid cells
    catch {scan $sisimpar(xsiz) {%g} sisimpar(xsiz)}
    puts -nonewline $outf [format "%g %g" $sisimpar(nx) $sisimpar(xsiz)]
    set line [format "%g %g" $sisimpar(nx) $sisimpar(xsiz)]
    set line_length [string length $line]
    set comment "\\nx, xsiz: number, size of grid cells"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### ny, ysiz: number, size of grid cells
    catch {scan $sisimpar(ysiz) {%g} sisimpar(ysiz)}
    puts -nonewline $outf [format "%g %g" $sisimpar(ny) $sisimpar(ysiz)]
    set line [format "%g %g" $sisimpar(ny) $sisimpar(ysiz)]
    set line_length [string length $line]
    set comment "\\ny, ysiz: number, size of grid cells"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    
    ### nz, zsiz: number, size of grid cells
    catch {scan $sisimpar(zsiz) {%g} sisimpar(zsiz)}
    puts -nonewline $outf [format "%g %g" $sisimpar(nz) $sisimpar(zsiz)]
    set line [format "%g %g" $sisimpar(nz) $sisimpar(zsiz)]
    set line_length [string length $line]
    set comment "\\nz, zsiz: number, size of grid cells"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### ncat: number of thresholds / categories
    set comment "\\ncat: number of thresholds / categories"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $sisimpar(ncat) $comment]

    ### cat: thresholds / categories
    set line_length_total 0
    if { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat} } {
	for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
	    if { $sisimpar(vartype) == "Categorical" } {
		puts -nonewline $outf "[format "%g" $sisimpar(cat,$ii)] "
		set line [format "%g" $sisimpar(cat,$ii)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
	    } elseif { $sisimpar(vartype) == "Continous" } {
		puts -nonewline $outf "[format "%g" $sisimpar(cat,$ii)] "
		set line [format "%g" $sisimpar(cat,$ii)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
	    }
	}
    } elseif { ($sisimpar(datafl) == {*.dat} && $sisimpar(directik) != {*.dat}) || \
		   ($sisimpar(datafl) != {*.dat} && $sisimpar(directik) != {*.dat}) } {
	for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3] } { incr ii } {
	    if { $sisimpar(vartype) == "Categorical" } {
		puts -nonewline $outf "[format "%g" $sisimpar(cat,$ii)] "
		set line [format "%g" $sisimpar(cat,$ii)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
	    } elseif { $sisimpar(vartype) == "Continous" } {
		puts -nonewline $outf "[format "%g" $sisimpar(cat,$ii)] "
		set line [format "%g" $sisimpar(cat,$ii)]
		set line_length [string length $line]
		set line_length_total [expr $line_length_total + $line_length + 1]
	    }
	}
    }
    set comment "\\cat: thresholds / categories"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }


}
#============================================================================


#
# End of proc section.
#
#############################################################################


#############################################################################
proc geossav_sisim_main { } {
    global exedir
    source $exedir/Geossav_sisim_globals.tcl


    . configure -cursor watch
    update
    
    set sisim_box_w [toplevel .sisim_box -class Dialog]
    wm resizable $sisim_box_w false false
    wm sizefrom $sisim_box_w program
    wm group $sisim_box_w .
    wm transient $sisim_box_w [winfo toplevel [winfo parent $sisim_box_w]]
    wm title $sisim_box_w "SISIM parameters"
    
    set geostr [wm geometry .]
    regexp {([0-9]*)x([0-9]*)(\+)([0-9]*)(\+)([0-9]*)} $geostr all width height p1 pos_x p2 pos_y
    set pos_x [expr $pos_x + $width / 2 + 212]
    set pos_y [expr $pos_y + $height / 2 + 15]
    wm geometry $sisim_box_w "+$pos_x+$pos_y"
    
    
    #############################################################################
    # Tabnotebook
    #############################################################################
    set sisim_tabnotebook_w [iwidgets::tabnotebook $sisim_box_w.sisim_tbn \
				 -width 5.4i \
				 -height 4.45i \
				 -tabpos n \
				 -tabforeground gray50 \
				 -background gray85 \
				 -bevelamount 2]
    
    pack $sisim_tabnotebook_w -padx 0 -pady 0
    
    
    #
    # page1: Geossav SISIM-parameters: Files
    #
    set page [$sisim_box_w.sisim_tbn add \
		  -label "Files"]
    

    # initialize dimension names for hard and soft data
    if { ![sisim_read_hard_data $sisimpar(datafl)] } {
	set dim_name_hard {}
	lappend dim_name_hard "None"
	for { set i 1} { $i < 5 } { incr i } {
	    lappend dim_name_hard $i
	}
    }
    if { ![sisim_read_soft_data $sisimpar(directik)] } {
	set dim_name_soft {}
	lappend dim_name_soft "None"
	for { set i 1} { $i < 7 } { incr i } {
	    lappend dim_name_soft $i
	}
	if { $sisimpar(variables_initial) == 0 } {
	    set sisimpar(variables_initial) $i  ;# number of initial variables (None, x, y, z, var1, var2, ...)
	}                                       ;# this number can be changed                                      
	set max_cut_cat_initial $sisimpar(variables_initial)  ;# maximal number of initial variables 
    }	                                                      ;# this number can not be changed
    
    
    set work_frame_w [frame $page.workframe \
			  -borderwidth 2 \
			  -relief flat]
    

    # here are the input filed for the hard input data
    set hard_input_iframe_w [iwidgets::labeledframe $work_frame_w.hard_input \
				 -labeltext "Hard Input Data" \
				 -labelpos nw \
				 -borderwidth 2 \
				 -relief groove]
    set hard_input_frame_w [$hard_input_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $hard_input_frame_w.subfr1]
    
    set datafl_w [iwidgets::entryfield $subframe1_w.datafl \
		      -labeltext "File " \
		      -width 67 \
		      -state disabled \
		      -textbackground gray85 \
		      -foreground gray50 \
		      -textvariable sisimpar(datafl)]

    if { $sisimpar(datafl) != {*.dat} } {
	$datafl_w configure -state normal -textbackground white -foreground black
    }
    
    set datafl_browse_w [iwidgets::pushbutton $subframe1_w.datafl_browse \
			     -height 27 \
			     -width 40 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			     -command {
				 set tmp_fname [tk_getOpenFile \
						    -defaultextension ".dat" \
						    -filetypes {{{Geo-EAS Files} {.dat}} {{All Files} {.*}}} \
						    -initialfile $sisimpar(datafl) \
						    -initialdir $exedir \
						    -title "Select SISIM input data (hard)"]
				 
				 if { $tmp_fname != {} } {
				     set sisimpar(datafl) $tmp_fname
				     unset dim_name_hard
				     if { [sisim_read_hard_data $sisimpar(datafl)] != 1 } {
					 set dim_name_hard {}
					 lappend dim_name_hard "None"
					 for { set i 1 } { $i < 5 } { incr i } {
					     lappend dim_name_hard $i
					 }
				     }
				     
				     $icolx_w clear list
				     foreach i $dim_name_hard {
					 $icolx_w insert list end $i
				     }
				     if { $sisimpar(hard_parameters) >= 4 } {
					 $icolx_w selection set 1
					 set sisimpar(icolx) [lindex $dim_name_hard 1]
				     } elseif { $sisimpar(hard_parameters) < 4 } {
					 $icolx_w selection set 1
					 set sisimpar(icolx) [lindex $dim_name_hard 1]
				     }
				     
				     $icoly_w clear list
				     foreach i $dim_name_hard {
					 $icoly_w insert list end $i
				     }
				     if { $sisimpar(hard_parameters) >= 4 } {
					 $icoly_w selection set 2
					 set sisimpar(icoly) [lindex $dim_name_hard 2]
				     } elseif { $sisimpar(hard_parameters) < 4 } {
					 $icoly_w selection set 1
					 set sisimpar(icoly) [lindex $dim_name_hard 1]
				     }
				     
				     $icolz_w clear list
				     foreach i $dim_name_hard {
					 $icolz_w insert list end $i
				     }
				     if { $sisimpar(hard_parameters) >= 4 } {
					 $icolz_w selection set 3
					 set sisimpar(icolz) [lindex $dim_name_hard 3]
				     } elseif { $sisimpar(hard_parameters) < 4 } {
					 $icolz_w selection set 1
					 set sisimpar(icolz) [lindex $dim_name_hard 1]
				     }
				     
				     $icolvr_w clear list
				     foreach i $dim_name_hard {
					 $icolvr_w insert list end $i
				     }
				     if { $sisimpar(hard_parameters) >= 4 } {
					 $icolvr_w selection set 4
					 set sisimpar(icolvr) [lindex $dim_name_hard 4]
				     } elseif { $sisimpar(hard_parameters) < 4 } {
					 $icolvr_w selection set 1
					 set sisimpar(icolvr) [lindex $dim_name_hard 1]
				     }

				     ### set datafl widget normal
				     $datafl_w configure -state normal -textbackground white -foreground black
				     
				     ### counter for the exit from the SISIM parameters tabnotebook
				     set sisimpar(hard_cancel_counter) 0
				     
				     ### output- and debugfile path- and filename
				     set datafl_rootname [file rootname $sisimpar(datafl)]
				     set datafl_dirname [file dirname $sisimpar(datafl)]
				     set datafl_dirname_lenght [string length $datafl_dirname]
				     set datafl_name [string range $datafl_rootname [expr $datafl_dirname_lenght + 1] end]
				     
				     ### output path- and filename
				     set outfl_extension [file extension $sisimpar(outfl)]
				     set simulation "/SIS_"
				     regsub {} $outfl_extension $datafl_name outputfl_1
				     regsub {} $outputfl_1 $simulation outputfl_2
				     regsub {} $outputfl_2 $datafl_dirname outputfl_3
				     set sisimpar(outfl) $outputfl_3
				     
				     ### debug path- and filename
				     set dbgfl_extension [file extension $sisimpar(dbgfl)]
				     regsub {} $dbgfl_extension $datafl_name debugfl_1
				     regsub {} $debugfl_1 $simulation debugfl_2
				     regsub {} $debugfl_2 $datafl_dirname debugfl_3
				     set sisimpar(dbgfl) $debugfl_3
				     
				     
				     if { $focus_variable == 0 } {
					 $plus_cut_cat_pushbutton_w configure -state normal
					 $minus_cut_cat_pushbutton_w configure -state normal
					 
					 $plus_model_pushbutton_w configure -state disabled
					 $minus_model_pushbutton_w configure -state disabled
					 
					 for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
					     $cut_cat_vals($ii,entry) configure \
						 -state normal \
						 -textbackground white \
						 -foreground black
					     $pdf_cdf_vals($ii,entry) configure \
						 -state normal \
						 -textbackground white \
						 -foreground black
					     
					     for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
						 $model_type($i,combo$ii) configure \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50
						 $model_cc($i,entry$ii) configure \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50
						 $model_ang1($i,entry$ii) configure \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50
						 $model_ang2($i,entry$ii) configure \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50
						 $model_ang3($i,entry$ii) configure \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50
						 $model_aa_hmax($i,entry$ii) configure \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50
						 $model_aa_hmin($i,entry$ii) configure \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50
						 $model_aa_vert($i,entry$ii) configure \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50
					     }
					     $models_nugget_w($ii) configure \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50
					     $models_sill_label_w($ii) configure -state disabled
					     $models_sill_value_w($ii) configure -state disabled
					     $cut_cat_number_label_w($ii) configure -state disabled
					     $cut_cat_number_value_w($ii) configure -state disabled
					 }
				     } elseif  { $focus_variable != 0 } {
					 $plus_cut_cat_pushbutton_w configure -state normal
					 $minus_cut_cat_pushbutton_w configure -state normal
					 
					 $plus_model_pushbutton_w configure -state normal
					 $minus_model_pushbutton_w configure -state normal
					 
					 for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
					     $cut_cat_vals($ii,entry) configure \
						 -state normal \
						 -textbackground white \
						 -foreground black
					     $pdf_cdf_vals($ii,entry) configure \
						 -state normal \
						 -textbackground white \
						 -foreground black
					     
					     for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
						 $model_type($i,combo$ii) configure \
						     -state normal \
						     -textbackground white \
						     -foreground black
						 $model_cc($i,entry$ii) configure \
						     -state normal \
						     -textbackground white \
						     -foreground black
						 $model_ang1($i,entry$ii) configure \
						     -state normal \
						     -textbackground white \
						     -foreground black
						 $model_ang2($i,entry$ii) configure \
						     -state normal \
						     -textbackground white \
						     -foreground black
						 $model_ang3($i,entry$ii) configure \
						     -state normal \
						     -textbackground white \
						     -foreground black
						 $model_aa_hmax($i,entry$ii) configure \
						     -state normal \
						     -textbackground white \
						     -foreground black
						 $model_aa_hmin($i,entry$ii) configure \
						     -state normal \
						     -textbackground white \
						     -foreground black
						 $model_aa_vert($i,entry$ii) configure \
						     -state normal \
						     -textbackground white \
						     -foreground black
					     }
					     $models_nugget_w($ii) configure \
						 -state normal \
						 -textbackground white \
						 -foreground black
					     $models_sill_label_w($ii) configure -state normal
					     $models_sill_value_w($ii) configure -state normal
					     $cut_cat_number_label_w($ii) configure -state normal
					     $cut_cat_number_value_w($ii) configure -state normal
					 }
				     }
				 }
			     }]
    
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $datafl_w        -row 0 -column 0 -sticky we -pady 0
    grid $datafl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    
    set subframe2_w [frame $hard_input_frame_w.subfr2]
    
    if { $sisimpar(datafl) != {*.dat} } {
	set sisimpar(hard_cancel_counter) [expr $sisimpar(hard_cancel_counter) + 1]
    }
    
    set icolx_w [iwidgets::combobox $subframe2_w.icolx \
		     -labeltext "X " \
		     -textvariable sisimpar(icolx) \
		     -width 17 \
		     -listheight 74 \
		     -validate {regexpvalidate %c}]
    if { $sisimpar(datafl) == {*.dat} } {
	foreach i $dim_name_hard {
	    $icolx_w insert list end $i
	}
	$icolx_w selection set 1
	set sisimpar(icolx) [lindex $dim_name_hard 1]
    } elseif { $sisimpar(datafl) != {*.dat} } {
	set tmp_icolx $sisimpar(icolx)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if sisimpar(icolx) is empty
	foreach i $dim_name_hard {
	    $icolx_w insert list end $i
	    $icolx_w delete list [expr $sisimpar(hard_parameters) + 1]
	    if { $i == $tmp_icolx } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($sisimpar(hard_cancel_counter) * ($sisimpar(hard_parameters) + 1))]
	$icolx_w selection set $sel
	set sisimpar(icolx) [lindex $dim_name_hard $sel]
    }
    
    
    set icoly_w [iwidgets::combobox $subframe2_w.icoly \
		     -labeltext "Y " \
		     -textvariable sisimpar(icoly) \
		     -width 17 \
		     -listheight 74 \
		     -validate {regexpvalidate %c}]
    if { $sisimpar(datafl) == {*.dat} } {
	foreach i $dim_name_hard {
	    $icoly_w insert list end $i
	}
	$icoly_w selection set 2
	set sisimpar(icoly) [lindex $dim_name_hard 2]
    } elseif { $sisimpar(datafl) != {*.dat} } {
	set tmp_icoly $sisimpar(icoly)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if sisimpar(icoly) is empty
	foreach i $dim_name_hard {
	    $icoly_w insert list end $i
	    $icoly_w delete list [expr $sisimpar(hard_parameters) + 1]
	    if { $i == $tmp_icoly } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($sisimpar(hard_cancel_counter) * ($sisimpar(hard_parameters) + 1))]
	$icoly_w selection set $sel
	set sisimpar(icoly) [lindex $dim_name_hard $sel]
    }
    
    
    set icolz_w [iwidgets::combobox $subframe2_w.icolz \
		     -labeltext "Z " \
		     -textvariable sisimpar(icolz) \
		     -width 17 \
		     -listheight 74 \
		     -validate {regexpvalidate %c}]
    if { $sisimpar(datafl) == {*.dat} } {
	foreach i $dim_name_hard {
	    $icolz_w insert list end $i
	}
	$icolz_w selection set 3
    set sisimpar(icolz) [lindex $dim_name_hard 3]
    } elseif { $sisimpar(datafl) != {*.dat} } {
	set tmp_icolz $sisimpar(icolz)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if sisimpar(icolz) is empty
	foreach i $dim_name_hard {
	    $icolz_w insert list end $i
	    $icolz_w delete list [expr $sisimpar(hard_parameters) + 1]
	    if { $i == $tmp_icolz } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($sisimpar(hard_cancel_counter) * ($sisimpar(hard_parameters) + 1))]
	$icolz_w selection set $sel
	set sisimpar(icolz) [lindex $dim_name_hard $sel]
    }
    
    
    grid columnconfigure $subframe2_w 0 -minsize 154
    grid columnconfigure $subframe2_w 1 -minsize 160
    grid columnconfigure $subframe2_w 2 -minsize 160
    
    grid $icolx_w -row 0 -column 0 -sticky e -pady 0
    grid $icoly_w -row 0 -column 1 -sticky e -pady 0
    grid $icolz_w -row 0 -column 2 -sticky e -pady 0
    
    
    set subframe3_w [frame $hard_input_frame_w.subfr3]    
    
    set icolvr_w [iwidgets::combobox $subframe3_w.icolvr_option \
		      -labeltext "Var " \
		      -textvariable sisimpar(icolvr) \
		      -width 17 \
		      -listheight 74 \
		      -validate {regexpvalidate %c}]
    if { $sisimpar(datafl) == {*.dat} } {
	foreach i $dim_name_hard {
	    $icolvr_w insert list end $i
	}
	$icolvr_w selection set 4
	set sisimpar(icolvr) [lindex $dim_name_hard 4]
    } elseif { $sisimpar(datafl) != {*.dat} } {
	set tmp_icolvr $sisimpar(icolvr)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if sisimpar(icolvr) is empty
	foreach i $dim_name_hard {
	    $icolvr_w insert list end $i
	    $icolvr_w delete list [expr $sisimpar(hard_parameters) + 1]
	    if { $i == $tmp_icolvr } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($sisimpar(hard_cancel_counter) * ($sisimpar(hard_parameters) + 1))]
	$icolvr_w selection set $sel
	set sisimpar(icolvr) [lindex $dim_name_hard $sel]
    }
    
    
    set trimlim_min_w [iwidgets::entryfield $subframe3_w.min \
			   -labeltext "Trimming limits " \
			   -width 16 \
			   -textvariable sisimpar(tmin) \
			   -validate {check_number_t %W %P}]
    
    
    set trimlim_max_w [iwidgets::entryfield $subframe3_w.max \
			   -labeltext "     " \
			   -width 16 \
			   -textvariable sisimpar(tmax) \
			   -validate {check_number_t %W %P}]
    
    
    grid columnconfigure $subframe3_w 0 -minsize 154
    grid columnconfigure $subframe3_w 1 -minsize 195
    grid columnconfigure $subframe3_w 2 -minsize 60
    
    grid $icolvr_w      -row 0 -column 0 -sticky e -pady 6
    grid $trimlim_min_w -row 0 -column 1 -sticky e -pady 6
    grid $trimlim_max_w -row 0 -column 2 -sticky w -pady 6
    
    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	-anchor w -expand 1 -padx 3 -pady 1
    
    
    # here are the input filed for the soft input data
    set soft_input_iframe_w [iwidgets::labeledframe $work_frame_w.soft_input \
				 -labeltext "Soft Input Data" \
				 -labelpos nw \
				 -borderwidth 2 \
				 -relief groove]
    set soft_input_frame_w [$soft_input_iframe_w childsite]
    
    set subframe1_w [frame $soft_input_frame_w.subfr1]
    
    
    set soft_checkbutton_w [checkbutton $subframe1_w.soft_cbutton \
				-text " " \
				-variable sisimpar(soft_data) \
				-command {soft_data_activator}]
    
    
    set directik_w [iwidgets::entryfield $subframe1_w.directik \
			-labeltext "File " \
			-width 61 \
			-state disabled \
			-textbackground gray85 \
			-foreground gray50 \
			-textvariable sisimpar(directik)]
    
    
    set directik_browse_w [iwidgets::pushbutton $subframe1_w.directik_browse \
			       -height 27 \
			       -width 40 \
			       -defaultringpad 0 \
			       -state disabled \
			       -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			       -command {
				   set tmp_fname [tk_getOpenFile \
						      -defaultextension ".dat" \
						      -filetypes {{{Geo-EAS Files} {.dat}} {{All Files} {.*}}} \
						      -initialfile $sisimpar(directik) \
						      -initialdir $exedir \
						      -title "Select SISIM input data (soft)"]
				   
				   if { $tmp_fname != {} } {
				       if { $sisimpar(directik) == {*.dat} } {
					   for { set ii 1 } { $ii <= 3} { incr ii } {
					       destroy $icoli_vals($ii,label)
					       destroy $icoli_vals($ii,combo)
					       destroy $bz_vals($ii,entry)
					   }
					   for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4]} { incr ii } {
					       destroy $pdf_cdf_vals($ii,label)
					       destroy $cut_cat_vals($ii,entry)
					       destroy $pdf_cdf_vals($ii,entry)
					       
					       destroy $model_hsbar_w($ii)
					       destroy $model_table_w($ii)
					       
					       destroy $models_nugget_w($ii)
					       destroy $models_sill_label_w($ii)
					       destroy $models_sill_value_w($ii)
					       destroy $cut_cat_number_label_w($ii)
					       destroy $cut_cat_number_value_w($ii)
					   }
				       }
				       if { $sisimpar(directik) != {*.dat} } {
					   if { $sisimpar(soft_parameters) >= 4 } {
					       for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3]} { incr ii } {
						   destroy $icoli_vals($ii,label)
						   destroy $icoli_vals($ii,combo)
						   destroy $bz_vals($ii,entry)
						   
						   destroy $pdf_cdf_vals($ii,label)
						   destroy $cut_cat_vals($ii,entry)
						   destroy $pdf_cdf_vals($ii,entry)
						   
						   destroy $model_hsbar_w($ii)
						   destroy $model_table_w($ii)
						   
						   destroy $models_nugget_w($ii)
						   destroy $models_sill_label_w($ii)
						   destroy $models_sill_value_w($ii)
						   destroy $cut_cat_number_label_w($ii)
						   destroy $cut_cat_number_value_w($ii)
					       }
					   } elseif { $sisimpar(soft_parameters) < 4 } {
					       for { set ii 1 } { $ii <= $variables_counter} { incr ii } {
						   destroy $icoli_vals($ii,label)
						   destroy $icoli_vals($ii,combo)
						   destroy $bz_vals($ii,entry)
						   
						   destroy $pdf_cdf_vals($ii,label)
						   destroy $cut_cat_vals($ii,entry)
						   destroy $pdf_cdf_vals($ii,entry)
						   
						   destroy $model_hsbar_w($ii)
						   destroy $model_table_w($ii)
						   
						   destroy $models_nugget_w($ii)
						   destroy $models_sill_label_w($ii)
						   destroy $models_sill_value_w($ii)
						   destroy $cut_cat_number_label_w($ii)
						   destroy $cut_cat_number_value_w($ii)
					       }
					   }
				       }
				       
				       ### exit from the SISIM parameters tabnotebook, indicator and counter 
				       set sisimpar(soft_cancel) 0
				       set sisimpar(soft_cancel_counter) 0
				       
				       set sisimpar(directik) $tmp_fname
				       unset dim_name_soft
				       if { [sisim_read_soft_data $sisimpar(directik)] != 1 } {
					   set dim_name_soft {}
					   lappend dim_name_soft "None"
					   for { set i 1 } { $i <= [expr $sisimpar(variables) - 3] } { incr i } {
					       lappend dim_name_soft $i
					   }
				       }
				       
				       $icolsx_w configure -state normal -textbackground white -foreground black
				       $icolsx_w clear list
				       foreach i $dim_name_soft {
					   $icolsx_w insert list end $i
				       }
				       if { $sisimpar(soft_parameters) >= 4 } {
					   $icolsx_w selection set 1
					   set tmp_icolsx [lindex $dim_name_soft 1]
				       } elseif { $sisimpar(soft_parameters) < 4 } {
					   $icolsx_w selection set 1
					   set tmp_icolsx [lindex $dim_name_soft 1]				       
				       }
				       
				       $icolsy_w configure -state normal -textbackground white -foreground black
				       $icolsy_w clear list
				       foreach i $dim_name_soft {
					   $icolsy_w insert list end $i
				       }
				       if { $sisimpar(soft_parameters) >= 4 } {
					   $icolsy_w selection set 2
					   set tmp_icolsy [lindex $dim_name_soft 2]
				       } elseif { $sisimpar(soft_parameters) < 4 } {
					   $icolsy_w selection set 1
					   set tmp_icolsy [lindex $dim_name_soft 1]				       
				       }
				       
				       $icolsz_w configure -state normal -textbackground white -foreground black
				       $icolsz_w clear list
				       foreach i $dim_name_soft {
					   $icolsz_w insert list end $i
				       }
				       if { $sisimpar(soft_parameters) >= 4 } {
					   $icolsz_w selection set 3
					   set tmp_icolsz [lindex $dim_name_soft 3]
				       } elseif { $sisimpar(soft_parameters) < 4 } {
					   $icolsz_w selection set 1
					   set tmp_icolsz [lindex $dim_name_soft 1]				       
				       }
				       
				       ### output and debug path- and filename, if there is no hard data
				       if { $sisimpar(datafl) == {*.dat} } {
					   set directik_rootname [file rootname $sisimpar(directik)]
					   set directik_dirname [file dirname $sisimpar(directik)]
					   set directik_dirname_lenght [string length $directik_dirname]
					   set directik_name [string range $directik_rootname \
								  [expr $directik_dirname_lenght + 1] end]
					   
					   ### output path- and filename
					   set outfl_extension [file extension $sisimpar(outfl)]
					   set simulation "/SIS_"
					   regsub {} $outfl_extension $directik_name outputfl_1
					   regsub {} $outputfl_1 $simulation outputfl_2
					   regsub {} $outputfl_2 $directik_dirname outputfl_3
					   set sisimpar(outfl) $outputfl_3
					   
					   ### debug path- and filename
					   set dbgfl_extension [file extension $sisimpar(dbgfl)]
					   regsub {} $dbgfl_extension $directik_name debugfl_1
					   regsub {} $debugfl_1 $simulation debugfl_2
					   regsub {} $debugfl_2 $directik_dirname debugfl_3
					   set sisimpar(dbgfl) $debugfl_3
				       }


				       ### set directic widget normal
				       $directik_w configure -state normal -textbackground white -foreground black
				       
				       
				       ### set Markov Bayes simulation (imbsim) off
				       set sisimpar(imbsim) 0
				       $imbsim_checkbutton_w configure -variable sisimpar(imbsim) -state normal
				       

				       ### setting up the rows of the icoli- and the pdf_cdf- tables
				       if { $sisimpar(soft_parameters) >= 4 } {
					   $icoli_table_w configure -rows [expr $sisimpar(variables) - 2]
					   $pdf_cdf_table_w configure -rows [expr $sisimpar(variables) - 2]
				       } elseif { $sisimpar(soft_parameters) < 4 } {
					   $icoli_table_w configure -rows 2
					   $pdf_cdf_table_w configure -rows 2
				       }

				       
				       ### setting the pushbuttons of the icoli- and the pdf_cdf- tables off/on
				       $plus_model_pushbutton_w configure -state disabled
				       $minus_model_pushbutton_w configure -state disabled
				       $plus_cut_cat_pushbutton_w configure -state normal
				       $minus_cut_cat_pushbutton_w configure -state normal
				       
				       
				       ### fills the icoli-, the pdf_cdf-, and the model-tables
				       if { $sisimpar(soft_parameters) >= 4 } {
					   for {set ii 1} {$ii <= [expr $sisimpar(variables) - 3]} {incr ii} {
					       set variables_counter $ii
					       
					       set icoli_vals($ii,label) [iwidgets::labeledwidget $icoli_table_w.label$ii \
									      -labeltext "$ii" \
									      -background gray50 \
									      -foreground white]
					       
					       set icoli_vals($ii,combo) [iwidgets::combobox $icoli_table_w.combo$ii \
									      -textvariable sisimpar(icoli,$ii) \
									      -validate {regexpvalidate %c} \
									      -listheight 100 \
									      -focuscommand "focus_proc $ii" \
									      -selectioncommand "focus_proc $ii"]
					       
					       $icoli_vals($ii,combo) clear list
					       foreach i $dim_name_soft {
						   $icoli_vals($ii,combo) insert list end $i
					       }
					       $icoli_vals($ii,combo) selection set [expr $ii + 3]
					       set sisimpar(icoli) [lindex $dim_name_soft [expr $ii + 3]]
					       update
					       
					       set sisimpar(bz,$ii) 0;	# initialize to 0
					       set bz_vals($ii,entry) [iwidgets::entryfield $icoli_table_w.bz$ii \
									   -textvariable sisimpar(bz,$ii) \
									   -state disabled \
									   -textbackground gray85 \
									   -foreground gray50 \
									   -validate {check_number_bz_values %W %P %c} \
									   -focuscommand "focus_proc $ii"]
					       update
					       
					       $icoli_table_w window configure $ii,0 -window $icoli_vals($ii,label) \
						   -relief flat -background gray50
					       $icoli_table_w window configure $ii,1 -window $icoli_vals($ii,combo)
					       $icoli_table_w window configure $ii,2 -window $bz_vals($ii,entry)
					       
					       set pdf_cdf_vals($ii,label) \
						   [iwidgets::labeledwidget $pdf_cdf_table_w.pdf_cdf$ii \
							-labeltext "$ii" \
							-background gray50 \
							-foreground white]
					       
					       set sisimpar(ncat) $ii
					       
					       set sisimpar(cat,$ii) $ii
					       set cut_cat_vals($ii,entry) \
						   [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
							-textvariable sisimpar(cat,$ii) \
							-validate {check_number_cut_cat_pdf_cdf %W %P %c} \
							-focuscommand "focus_proc $ii"]
					       update
					       
					       set sisimpar(pdf,$ii) 0.15
					       set pdf_cdf_vals($ii,entry) \
						   [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
							-textvariable sisimpar(pdf,$ii) \
							-validate {check_number_cut_cat_pdf_cdf %W %P %c} \
							-focuscommand "focus_proc $ii"]
					       update
					       
					       $pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) \
						   -relief flat -background gray50
					       $pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
					       $pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
					       
					       
					       ### create model table
					       set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
									   -orient vertical \
									   -command "$model_sframe_w.table_$ii yview"]
					       update
					       
					       set no($ii) 1   ;# number of variogram types with corresponding parameters
					       set model_table_w($ii) [table $model_sframe_w.table_$ii \
									   -variable model_vals \
									   -titlerows 1 \
									   -titlecols 0 \
									   -cols 9 \
									   -rows [expr $no($ii) + 1] \
									   -yscrollcommand "$model_hsbar_w($ii) set" \
									   -height 4]
					       update
					       $model_table_w($ii) width 0 3
					       $model_table_w($ii) width 1 15
					       $model_table_w($ii) width 2 8
					       $model_table_w($ii) width 3 8
					       $model_table_w($ii) width 4 8
					       $model_table_w($ii) width 5 8
					       $model_table_w($ii) width 6 8
					       $model_table_w($ii) width 7 8
					       $model_table_w($ii) width 8 8
					       
					       set model_vals(0,0) "No"
					       set model_vals(0,1) "Type"
					       set model_vals(0,2) "Cc"
					       set model_vals(0,3) "Azimuth"
					       set model_vals(0,4) "Dip"
					       set model_vals(0,5) "Plunge"
					       set model_vals(0,6) "hmax"
					       set model_vals(0,7) "hmin"
					       set model_vals(0,8) "vert"
					   
					       
					       set sum_cc($ii) 0                ;# sum of all cc values

					       for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
						   set no_counter($ii) $i \
						       ;# counts the number of variogram types with corr. parameters
						   
						   set model_vals_label($i,$ii) \
						       [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
							    -labeltext "$i" \
							    -background gray50 \
							    -foreground white]
						   update
						   
						   set model_type($i,combo$ii) [iwidgets::combobox \
										    $model_table_w($ii).combo_type$i \
										    -width 15 \
										    -textvariable sisimpar(it$ii,$i) \
										    -listheight 75]
						   $model_table_w($ii).combo_type$i insert list end \
						       "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
						   $model_type($i,combo$ii) selection set 0
						   set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i 0]
						   $model_table_w($ii).combo_type$i configure \
						       -state disabled -textbackground gray85 -foreground gray50
						   update
						   
						   set sisimpar(cc$ii,$i) 0.9
						   set model_cc($i,entry$ii) \
						       [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
							    -textvariable sisimpar(cc$ii,$i) \
							    -state disabled \
							    -textbackground gray85 \
							    -foreground gray50 \
							    -validate "check_number_cc %W %S %P $i %ii"]
						   
						   set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]
						   update
						   
						   set sisimpar(ang1$ii,$i) 0
						   set model_ang1($i,entry$ii) \
						       [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
							    -textvariable sisimpar(ang1$ii,$i) \
							    -state disabled \
							    -textbackground gray85 \
							    -foreground gray50 \
							    -validate {check_number_sang1 %W %P %c}]
						   update
						   
						   set sisimpar(ang2$ii,$i) 0
						   set model_ang2($i,entry$ii) [iwidgets::entryfield \
										    $model_table_w($ii).entry_ang2$i \
										    -textvariable sisimpar(ang2$ii,$i) \
										    -state disabled \
										    -textbackground gray85 \
										    -foreground gray50 \
										    -validate {check_number_sang %W %P %c}]
						   update
						   
						   set sisimpar(ang3$ii,$i) 0
						   set model_ang3($i,entry$ii) [iwidgets::entryfield \
										    $model_table_w($ii).entry_ang3$i \
										    -textvariable sisimpar(ang3$ii,$i) \
										    -state disabled \
										    -textbackground gray85 \
										    -foreground gray50 \
										    -validate {check_number_sang %W %P %c}]
						   update
						   
						   set sisimpar(aa_hmax$ii,$i) 10
						   set model_aa_hmax($i,entry$ii) \
						       [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
							    -textvariable sisimpar(aa_hmax$ii,$i) \
							    -state disabled \
							    -textbackground gray85 \
							    -foreground gray50 \
							    -validate {check_number_radius %W %P %c}]
						   update
						   
						   set sisimpar(aa_hmin$ii,$i) 10
						   set model_aa_hmin($i,entry$ii) \
						       [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
							    -textvariable sisimpar(aa_hmin$ii,$i) \
							    -state disabled \
							    -textbackground gray85 \
							    -foreground gray50 \
							    -validate {check_number_radius %W %P %c}]
						   update
						   
						   set sisimpar(aa_vert$ii,$i) 10
						   set model_aa_vert($i,entry$ii) \
						       [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
							    -textvariable sisimpar(aa_vert$ii,$i) \
							    -state disabled \
							    -textbackground gray85 \
							    -foreground gray50 \
							    -validate {check_number_radius %W %P %c}]
						   update
						   
						   $model_table_w($ii) window configure $i,0 \
						       -window $model_vals_label($i,$ii) -relief flat -background gray50
						   $model_table_w($ii) window configure $i,1 \
						       -window $model_type($i,combo$ii)
						   $model_table_w($ii) window configure $i,2 \
						       -window $model_cc($i,entry$ii)
						   $model_table_w($ii) window configure $i,3 \
						       -window $model_ang1($i,entry$ii)
						   $model_table_w($ii) window configure $i,4 \
						       -window $model_ang2($i,entry$ii)
						   $model_table_w($ii) window configure $i,5 \
						       -window $model_ang3($i,entry$ii)
						   $model_table_w($ii) window configure $i,6 \
						       -window $model_aa_hmax($i,entry$ii)
						   $model_table_w($ii) window configure $i,7 \
						       -window $model_aa_hmin($i,entry$ii)
						   $model_table_w($ii) window configure $i,8 \
						       -window $model_aa_vert($i,entry$ii)
					       }
					       
					       ### create nugget entry and sill label
					       set sisimpar(c0$ii) 0.1
					       set models_nugget_w($ii) [iwidgets::entryfield \
									     $subframe3_variogram_w.entry_nugget$ii \
									     -labeltext "Nugget " \
									     -width 7 \
									     -state disabled \
									     -textbackground gray85 \
									     -foreground gray50 \
									     -textvariable sisimpar(c0$ii) \
									     -validate "check_number_nugget %W %P"]
					       
					       
					       set models_sill_label_w($ii) \
						   [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
							-state disabled \
							-labeltext "Sill "]
					       
					       set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
					       set models_sill_value_w($ii) \
						   [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
							-state disabled \
							-labelvariable sill($ii)]
					       
					       
					       set cut_cat_number_label_w($ii) \
						   [iwidgets::labeledwidget \
							$subframe3_variogram_w.cut_cat_number_label$ii \
							-state disabled \
							-labeltext "Cut/Cat \# "]
					       
					       set cut_cat_number($ii) $ii
					       set cut_cat_number_value_w($ii) \
						   [iwidgets::labeledwidget \
							$subframe3_variogram_w.cut_cat_number_value$ii \
							-state disabled \
							-labelvariable cut_cat_number($ii)]
					   
					       
					       grid columnconfigure $subframe3_variogram_w 2 -minsize 121
					       grid columnconfigure $subframe3_variogram_w 3 -minsize 47
					       grid columnconfigure $subframe3_variogram_w 4 -minsize 60
					       grid columnconfigure $subframe3_variogram_w 5 -minsize 180
					       grid columnconfigure $subframe3_variogram_w 6 -minsize 10
					       grid $models_nugget_w($ii)     -row 0 -column 2 -sticky e -pady 0
					       grid $models_sill_label_w($ii) -row 0 -column 3 -sticky e -pady 0
					       grid $models_sill_value_w($ii) -row 0 -column 4 -sticky w -pady 0
					       grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
					       grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
					       
					       pack $model_hsbar_w($ii)  -side right -anchor w -fill y
					       pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
					       
					       set focus_variable 0
					   }
				       } elseif { $sisimpar(soft_parameters) < 4 } {
					   for {set ii 1} {$ii < 2} {incr ii} {
					       set variables_counter $ii
					       
					       set icoli_vals($ii,label) [iwidgets::labeledwidget $icoli_table_w.label$ii \
									      -labeltext "$ii" \
									      -background gray50 \
									      -foreground white]
					       
					       set icoli_vals($ii,combo) [iwidgets::combobox $icoli_table_w.combo$ii \
									      -textvariable sisimpar(icoli,$ii) \
									      -validate {regexpvalidate %c} \
									      -listheight 100 \
									      -focuscommand "focus_proc $ii" \
									      -selectioncommand "focus_proc $ii"]
					       
					       $icoli_vals($ii,combo) clear list
					       foreach i $dim_name_soft {
						   $icoli_vals($ii,combo) insert list end $i
					       }
					       $icoli_vals($ii,combo) selection set end
					       set sisimpar(icoli) [lindex $dim_name_soft end]
					       update
					       
					       set sisimpar(bz,$ii) 0;	# initialize to 0
					       set bz_vals($ii,entry) [iwidgets::entryfield $icoli_table_w.bz$ii \
									   -textvariable sisimpar(bz,$ii) \
									   -state disabled \
									   -textbackground gray85 \
									   -foreground gray50 \
									   -validate {check_number_bz_values %W %P %c} \
									   -focuscommand "focus_proc $ii"]
					       update
					       
					       $icoli_table_w window configure $ii,0 -window $icoli_vals($ii,label) \
						   -relief flat -background gray50
					       $icoli_table_w window configure $ii,1 -window $icoli_vals($ii,combo)
					       $icoli_table_w window configure $ii,2 -window $bz_vals($ii,entry)
					       
					       set pdf_cdf_vals($ii,label) \
						   [iwidgets::labeledwidget $pdf_cdf_table_w.pdf_cdf$ii \
							-labeltext "$ii" \
							-background gray50 \
							-foreground white]
					       
					       set sisimpar(ncat) $ii
					       
					       set sisimpar(cat,$ii) $ii
					       set cut_cat_vals($ii,entry) \
						   [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
							-textvariable sisimpar(cat,$ii) \
							-validate {check_number_cut_cat_pdf_cdf %W %P %c} \
							-focuscommand "focus_proc $ii"]
					       update
					       
					       set sisimpar(pdf,$ii) 0.15
					       set pdf_cdf_vals($ii,entry) \
						   [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
							-textvariable sisimpar(pdf,$ii) \
							-validate {check_number_cut_cat_pdf_cdf %W %P %c} \
							-focuscommand "focus_proc $ii"]
					       update
					       
					       $pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) \
						   -relief flat -background gray50
					       $pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
					       $pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
					       
					       
					       ### create model table
					       set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
									   -orient vertical \
									   -command "$model_sframe_w.table_$ii yview"]
					       update
					       
					       set no($ii) 1   ;# number of variogram types with corresponding parameters
					       set model_table_w($ii) [table $model_sframe_w.table_$ii \
									   -variable model_vals \
									   -titlerows 1 \
									   -titlecols 0 \
									   -cols 9 \
									   -rows [expr $no($ii) + 1] \
									   -yscrollcommand "$model_hsbar_w($ii) set" \
									   -height 4]
					       update
					       $model_table_w($ii) width 0 3
					       $model_table_w($ii) width 1 15
					       $model_table_w($ii) width 2 8
					       $model_table_w($ii) width 3 8
					       $model_table_w($ii) width 4 8
					       $model_table_w($ii) width 5 8
					       $model_table_w($ii) width 6 8
					       $model_table_w($ii) width 7 8
					       $model_table_w($ii) width 8 8
					       
					       set model_vals(0,0) "No"
					       set model_vals(0,1) "Type"
					       set model_vals(0,2) "Cc"
					       set model_vals(0,3) "Azimuth"
					       set model_vals(0,4) "Dip"
					       set model_vals(0,5) "Plunge"
					       set model_vals(0,6) "hmax"
					       set model_vals(0,7) "hmin"
					       set model_vals(0,8) "vert"
					   
					       
					       set sum_cc($ii) 0                ;# sum of all cc values

					       for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
						   set no_counter($ii) $i \
						       ;# counts the number of variogram types with corr. parameters
						   
						   set model_vals_label($i,$ii) \
						       [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
							    -labeltext "$i" \
							    -background gray50 \
							    -foreground white]
						   update
						   
						   set model_type($i,combo$ii) [iwidgets::combobox \
										    $model_table_w($ii).combo_type$i \
										    -width 15 \
										    -textvariable sisimpar(it$ii,$i) \
										    -listheight 75]
						   $model_table_w($ii).combo_type$i insert list end \
						       "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
						   $model_type($i,combo$ii) selection set 0
						   set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i 0]
						   $model_table_w($ii).combo_type$i configure \
						       -state disabled -textbackground gray85 -foreground gray50
						   update
						   
						   set sisimpar(cc$ii,$i) 0.9
						   set model_cc($i,entry$ii) \
						       [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
							    -textvariable sisimpar(cc$ii,$i) \
							    -state disabled \
							    -textbackground gray85 \
							    -foreground gray50 \
							    -validate "check_number_cc %W %S %P $i %ii"]
						   
						   set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]
						   update
						   
						   set sisimpar(ang1$ii,$i) 0
						   set model_ang1($i,entry$ii) \
						       [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
							    -textvariable sisimpar(ang1$ii,$i) \
							    -state disabled \
							    -textbackground gray85 \
							    -foreground gray50 \
							    -validate {check_number_sang1 %W %P %c}]
						   update
						   
						   set sisimpar(ang2$ii,$i) 0
						   set model_ang2($i,entry$ii) [iwidgets::entryfield \
										    $model_table_w($ii).entry_ang2$i \
										    -textvariable sisimpar(ang2$ii,$i) \
										    -state disabled \
										    -textbackground gray85 \
										    -foreground gray50 \
										    -validate {check_number_sang %W %P %c}]
						   update
						   
						   set sisimpar(ang3$ii,$i) 0
						   set model_ang3($i,entry$ii) [iwidgets::entryfield \
										    $model_table_w($ii).entry_ang3$i \
										    -textvariable sisimpar(ang3$ii,$i) \
										    -state disabled \
										    -textbackground gray85 \
										    -foreground gray50 \
										    -validate {check_number_sang %W %P %c}]
						   update
						   
						   set sisimpar(aa_hmax$ii,$i) 10
						   set model_aa_hmax($i,entry$ii) \
						       [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
							    -textvariable sisimpar(aa_hmax$ii,$i) \
							    -state disabled \
							    -textbackground gray85 \
							    -foreground gray50 \
							    -validate {check_number_radius %W %P %c}]
						   update
						   
						   set sisimpar(aa_hmin$ii,$i) 10
						   set model_aa_hmin($i,entry$ii) \
						       [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
							    -textvariable sisimpar(aa_hmin$ii,$i) \
							    -state disabled \
							    -textbackground gray85 \
							    -foreground gray50 \
							    -validate {check_number_radius %W %P %c}]
						   update
						   
						   set sisimpar(aa_vert$ii,$i) 10
						   set model_aa_vert($i,entry$ii) \
						       [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
							    -textvariable sisimpar(aa_vert$ii,$i) \
							    -state disabled \
							    -textbackground gray85 \
							    -foreground gray50 \
							    -validate {check_number_radius %W %P %c}]
						   update
						   
						   $model_table_w($ii) window configure $i,0 \
						       -window $model_vals_label($i,$ii) -relief flat -background gray50
						   $model_table_w($ii) window configure $i,1 \
						       -window $model_type($i,combo$ii)
						   $model_table_w($ii) window configure $i,2 \
						       -window $model_cc($i,entry$ii)
						   $model_table_w($ii) window configure $i,3 \
						       -window $model_ang1($i,entry$ii)
						   $model_table_w($ii) window configure $i,4 \
						       -window $model_ang2($i,entry$ii)
						   $model_table_w($ii) window configure $i,5 \
						       -window $model_ang3($i,entry$ii)
						   $model_table_w($ii) window configure $i,6 \
						       -window $model_aa_hmax($i,entry$ii)
						   $model_table_w($ii) window configure $i,7 \
						       -window $model_aa_hmin($i,entry$ii)
						   $model_table_w($ii) window configure $i,8 \
						       -window $model_aa_vert($i,entry$ii)
					       }
					       
					       ### create nugget entry and sill label
					       set sisimpar(c0$ii) 0.1
					       set models_nugget_w($ii) [iwidgets::entryfield \
									     $subframe3_variogram_w.entry_nugget$ii \
									     -labeltext "Nugget " \
									     -width 7 \
									     -state disabled \
									     -textbackground gray85 \
									     -foreground gray50 \
									     -textvariable sisimpar(c0$ii) \
									     -validate "check_number_nugget %W %P"]
					       
					       
					       set models_sill_label_w($ii) \
						   [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
							-state disabled \
							-labeltext "Sill "]
					       
					       set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
					       set models_sill_value_w($ii) \
						   [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
							-state disabled \
							-labelvariable sill($ii)]
					       
					       
					       set cut_cat_number_label_w($ii) \
						   [iwidgets::labeledwidget \
							$subframe3_variogram_w.cut_cat_number_label$ii \
							-state disabled \
							-labeltext "Cut/Cat \# "]
					       
					       set cut_cat_number($ii) $ii
					       set cut_cat_number_value_w($ii) \
						   [iwidgets::labeledwidget \
							$subframe3_variogram_w.cut_cat_number_value$ii \
							-state disabled \
							-labelvariable cut_cat_number($ii)]
					   
					       
					       grid columnconfigure $subframe3_variogram_w 2 -minsize 121
					       grid columnconfigure $subframe3_variogram_w 3 -minsize 47
					       grid columnconfigure $subframe3_variogram_w 4 -minsize 60
					       grid columnconfigure $subframe3_variogram_w 5 -minsize 180
					       grid columnconfigure $subframe3_variogram_w 6 -minsize 10
					       grid $models_nugget_w($ii)     -row 0 -column 2 -sticky e -pady 0
					       grid $models_sill_label_w($ii) -row 0 -column 3 -sticky e -pady 0
					       grid $models_sill_value_w($ii) -row 0 -column 4 -sticky w -pady 0
					       grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
					       grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
					       
					       pack $model_hsbar_w($ii)  -side right -anchor w -fill y
					       pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
					       
					       set focus_variable 0
					   }
				       }
				   }
			       }]
    
    
    grid columnconfigure $subframe1_w 1 -minsize 20
    grid columnconfigure $subframe1_w 2 -minsize 380
    grid columnconfigure $subframe1_w 3 -minsize 25
    
    grid $soft_checkbutton_w -row 0 -column 0 -sticky w -pady 0
    grid $directik_w         -row 0 -column 1 -sticky we -pady 2
    grid $directik_browse_w  -row 0 -column 2 -sticky w -pady 2
    
    
    set subframe2_w [frame $soft_input_frame_w.subfr2]
    
    if { $sisimpar(directik) != {*.dat} } {
	set sisimpar(soft_cancel_counter) [expr $sisimpar(soft_cancel_counter) + 1]
    }
    
    ### setting the soft input data widgets (X,Y,Z) disabled/normal depending on the input data
    if { (($sisimpar(soft_data) == 0) && ($sisimpar(directik) == "*.dat")) || \
	     (($sisimpar(soft_data) == 1) && ($sisimpar(directik) == "*.dat")) || \
	     (($sisimpar(soft_data) == 0) && ($sisimpar(directik) != "*.dat")) } {
	
	### iwidgets::combobox for X variable (soft)
	set xvar_frame_w [frame $subframe2_w.xframe]
	
	set icolsx_w [iwidgets::combobox $xvar_frame_w.icolsx \
			  -labeltext "X " \
			  -textvariable sisimpar(icolsx) \
			  -width 15 \
			  -listheight 60 \
			  -textbackground gray85 \
			  -foreground gray50 \
			  -validate {regexpvalidate %c}]
	if { $sisimpar(directik) == {*.dat} } {
	    foreach i $dim_name_soft {
		$icolsx_w insert list end $i
	    }
	    $icolsx_w selection set 1
	    set sisimpar(icolsx) [lindex $dim_name_soft 1]
	} elseif { $sisimpar(directik) != {*.dat} } {
	    set tmp_icolsx $sisimpar(icolsx)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if sisimpar(icolsx) is empty
	    foreach i $dim_name_soft {
		$icolsx_w insert list end $i
		$icolsx_w delete list [expr $sisimpar(soft_parameters) + 1]
		if { $i == $tmp_icolsx } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($sisimpar(soft_cancel_counter) * ($sisimpar(soft_parameters) + 1))]
	    $icolsx_w selection set $sel
	    set sisimpar(icolsx) [lindex $dim_name_soft $sel]
	}
	
	### iwidgets::combobox for Y variable (soft)
	set yvar_frame_w [frame $subframe2_w.yframe]
	
	set icolsy_w [iwidgets::combobox $yvar_frame_w.icolsy \
			  -labeltext "Y " \
			  -textvariable sisimpar(icolsy) \
			  -width 15 \
			  -listheight 60 \
			  -textbackground gray85 \
			  -foreground gray50 \
			  -validate {regexpvalidate %c}]
	if { $sisimpar(directik) == {*.dat} } {
	    foreach i $dim_name_soft {
		$icolsy_w insert list end $i
	    }
	    $icolsy_w selection set 2
	    set sisimpar(icolsy) [lindex $dim_name_soft 2]
	} elseif { $sisimpar(directik) != {*.dat} } {
	    set tmp_icolsy $sisimpar(icolsy)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if sisimpar(icolsy) is empty
	    foreach i $dim_name_soft {
		$icolsy_w insert list end $i
		$icolsy_w delete list [expr $sisimpar(soft_parameters) + 1]
		if { $i == $tmp_icolsy } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($sisimpar(soft_cancel_counter) * ($sisimpar(soft_parameters) + 1))]
	    $icolsy_w selection set $sel
	    set sisimpar(icolsy) [lindex $dim_name_soft $sel]
	}
	
	### iwidgets::combobox for Z variable (soft)
	set zvar_frame_w [frame $subframe2_w.zframe]
	
	set icolsz_w [iwidgets::combobox $zvar_frame_w.icolsz \
			  -labeltext "Z " \
			  -textvariable sisimpar(icolsz) \
			  -width 15 \
			  -listheight 60 \
			  -textbackground gray85 \
			  -foreground gray50 \
			  -validate {regexpvalidate %c}]
	if { $sisimpar(directik) == {*.dat} } {
	    foreach i $dim_name_soft {
		$icolsz_w insert list end $i
	    }
	    $icolsz_w selection set 3
	    set sisimpar(icolsz) [lindex $dim_name_soft 3]
	} elseif { $sisimpar(directik) != {*.dat} } {
	    set tmp_icolsz $sisimpar(icolsz)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if sisimpar(icolsz) is empty
	    foreach i $dim_name_soft {
		$icolsz_w insert list end $i
		$icolsz_w delete list [expr $sisimpar(soft_parameters) + 1]
		if { $i == $tmp_icolsz } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($sisimpar(soft_cancel_counter) * ($sisimpar(soft_parameters) + 1))]
	    $icolsz_w selection set $sel
	    set sisimpar(icolsz) [lindex $dim_name_soft $sel]
	}
	
	$icolsx_w configure -state disabled
	$icolsy_w configure -state disabled
	$icolsz_w configure -state disabled
	
    } elseif { (($sisimpar(soft_data) == 1) && ($sisimpar(directik) != "*.dat")) } {
	
	### iwidgets::combobox for X variable (soft)
	set xvar_frame_w [frame $subframe2_w.xframe]
	
	set icolsx_w [iwidgets::combobox $xvar_frame_w.icolsx \
			  -labeltext "X " \
			  -textvariable sisimpar(icolsx) \
			  -width 15 \
			  -listheight 60 \
			  -validate {regexpvalidate %c}]
	if { $sisimpar(directik) == {*.dat} } {
	    foreach i $dim_name_soft {
		$icolsx_w insert list end $i
	    }
	    $icolsx_w selection set 1
	    set sisimpar(icolsx) [lindex $dim_name_soft 1]
	} elseif { $sisimpar(directik) != {*.dat} } {
	    set tmp_icolsx $sisimpar(icolsx)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if sisimpar(icolsx) is empty
	    foreach i $dim_name_soft {
		$icolsx_w insert list end $i
		$icolsx_w delete list [expr $sisimpar(soft_parameters) + 1]
		if { $i == $tmp_icolsx } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($sisimpar(soft_cancel_counter) * ($sisimpar(soft_parameters) + 1))]
	    $icolsx_w selection set $sel
	    set sisimpar(icolsx) [lindex $dim_name_soft $sel]
	}
	
	### iwidgets::combobox for Y variable (soft)
	set yvar_frame_w [frame $subframe2_w.yframe]
	
	set icolsy_w [iwidgets::combobox $yvar_frame_w.icolsy \
			  -labeltext "Y " \
			  -textvariable sisimpar(icolsy) \
			  -width 15 \
			  -listheight 60 \
			  -validate {regexpvalidate %c}]
	if { $sisimpar(directik) == {*.dat} } {
	    foreach i $dim_name_soft {
		$icolsy_w insert list end $i
	    }
	    $icolsy_w selection set 2
	    set sisimpar(icolsy) [lindex $dim_name_soft 2]
	} elseif { $sisimpar(directik) != {*.dat} } {
	    set tmp_icolsy $sisimpar(icolsy)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if sisimpar(icolsy) is empty
	    foreach i $dim_name_soft {
		$icolsy_w insert list end $i
		$icolsy_w delete list [expr $sisimpar(soft_parameters) + 1]
		if { $i == $tmp_icolsy } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
	    incr cnt
	    }
	    set sel [expr $sel - ($sisimpar(soft_cancel_counter) * ($sisimpar(soft_parameters) + 1))]
	    $icolsy_w selection set $sel
	    set sisimpar(icolsy) [lindex $dim_name_soft $sel]
	}
	
	### iwidgets::combobox for Z variable (soft)
	set zvar_frame_w [frame $subframe2_w.zframe]
	
	set icolsz_w [iwidgets::combobox $zvar_frame_w.icolsz \
			  -labeltext "Z " \
			  -textvariable sisimpar(icolsz) \
			  -width 15 \
			  -listheight 60 \
			  -validate {regexpvalidate %c}]
	if { $sisimpar(directik) == {*.dat} } {
	    foreach i $dim_name_soft {
		$icolsz_w insert list end $i
	    }
	    $icolsz_w selection set 3
	    set sisimpar(icolsz) [lindex $dim_name_soft 3]
	} elseif { $sisimpar(directik) != {*.dat} } {
	    set tmp_icolsz $sisimpar(icolsz)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if sisimpar(icolsz) is empty
	    foreach i $dim_name_soft {
		$icolsz_w insert list end $i
		$icolsz_w delete list [expr $sisimpar(soft_parameters) + 1]
		if { $i == $tmp_icolsz } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($sisimpar(soft_cancel_counter) * ($sisimpar(soft_parameters) + 1))]
	    $icolsz_w selection set $sel
	    set sisimpar(icolsz) [lindex $dim_name_soft $sel]
	}
    }
    
    
    grid columnconfigure $subframe2_w 0 -minsize 155  
    grid columnconfigure $subframe2_w 1 -minsize 155
    grid columnconfigure $subframe2_w 2 -minsize 155 
    
    grid $icolsx_w -row 0 -column 0 -sticky w -pady 0
    grid $icolsy_w -row 0 -column 1 -sticky w -pady 0  
    grid $icolsz_w -row 0 -column 2 -sticky w -pady 0
    
    
    set imbsim_checkbutton_w [checkbutton $subframe2_w.icoli_cbutton \
				  -text "Markov Bayes simulation" \
				  -variable sisimpar(imbsim) \
				  -state disabled \
				  -command {bz_calibration}]
    
    
    set icoli_frame_w [frame $subframe2_w.icoli_frame \
			   -background grey40 \
			   -borderwidth 2 \
			   -relief sunken]
    
    set icoli_sframe_w [frame $icoli_frame_w.icoli_sframe \
			    -height 72 \
			    -width 316]
    set icoli_hsbar_w [scrollbar $icoli_sframe_w.yscroll \
			   -orient vertical \
			   -command "$icoli_sframe_w.table yview"]
    
    
    ### setting the icoli table normal/disabled depending on the input data
    if { ($sisimpar(directik) == "*.dat") && ($sisimpar(datafl) == "*.dat") || \
	     ($sisimpar(directik) == "*.dat") && ($sisimpar(datafl) != "*.dat") } {
	
	set tmp_variables_initial $sisimpar(variables_initial)
	
	set icoli_table_w [table $icoli_sframe_w.table \
			       -colwidth 25 \
			       -variable icoli_vals \
			       -titlerows 1 \
			       -titlecols 1 \
			       -cols 3 \
			       -rows 4 \
			       -yscrollcommand "$icoli_hsbar_w set" \
			       -height 4]
	
	$icoli_table_w width 0 5
	$icoli_table_w width 1 22
	$icoli_table_w width 2 22
	
	set icoli_vals(0,0) "\#"
	set icoli_vals(0,1) "Variable"
	set icoli_vals(0,2) "Calibration"
	
	for {set ii 1} {$ii <= [expr $sisimpar(variables_initial) - 4]} {incr ii} {
	    set variables_initial_counter $ii
	}
	
	for {set ii 1} {$ii <= 3} {incr ii} {
	    
	    set icoli_vals($ii,label) [iwidgets::labeledwidget $icoli_table_w.label$ii \
					   -labeltext "$ii" \
					   -background gray50 \
					   -foreground white]

	    set icoli_vals($ii,combo) [iwidgets::combobox $icoli_table_w.combo$ii \
					   -textvariable sisimpar(icoli,$ii) \
					   -validate {regexpvalidate %c} \
					   -listheight 100 \
					   -textbackground gray85 \
					   -foreground gray50 \
					   -focuscommand "focus_proc $ii" \
					   -selectioncommand "focus_proc $ii"]
	    foreach i $dim_name_soft {
		$icoli_vals($ii,combo) insert list end $i
	    }
	    $icoli_vals($ii,combo) selection set [expr $ii + 3]
	    set sisimpar(icoli) [lindex $dim_name_soft [expr $ii + 3]]
	    $icoli_table_w.combo$ii configure -state disabled -textbackground gray85 -foreground gray50
	    
	    set sisimpar(bz,$ii) 0
	    set bz_vals($ii,entry) [iwidgets::entryfield $icoli_table_w.bz$ii \
					-textvariable sisimpar(bz,$ii) \
					-state disabled \
					-textbackground gray85 \
					-foreground gray50 \
					-validate {check_number_bz_values %W %P %c} \
					-focuscommand "focus_proc $ii"]
	    
	    $icoli_table_w window configure $ii,0 -window $icoli_vals($ii,label) -relief flat -background gray50
	    $icoli_table_w window configure $ii,1 -window $icoli_vals($ii,combo)
	    $icoli_table_w window configure $ii,2 -window $bz_vals($ii,entry)
	}
	set sisimpar(variables_initial) $tmp_variables_initial 

    } elseif { ($sisimpar(directik) != "*.dat") && ($sisimpar(datafl) == "*.dat") || \
		   ($sisimpar(directik) != "*.dat") && ($sisimpar(datafl) != "*.dat") } {
	if { $sisimpar(soft_parameters) >= 4 } {
	
	    set icoli_table_w [table $icoli_sframe_w.table \
				   -colwidth 25 \
				   -variable icoli_vals \
				   -titlerows 1 \
				   -titlecols 1 \
				   -cols 3 \
				   -rows [expr $sisimpar(variables) - 2] \
				   -yscrollcommand "$icoli_hsbar_w set" \
				   -height 4]
	    
	    $icoli_table_w width 0 5
	    $icoli_table_w width 1 22
	    $icoli_table_w width 2 22
	    
	    set icoli_vals(0,0) "\#"
	    set icoli_vals(0,1) "Variable"
	    set icoli_vals(0,2) "Calibration"
	    
	    
	    for {set ii 1} {$ii <= [expr $sisimpar(variables) - 3]} {incr ii} {
		
		set variables_counter $ii
		
		set icoli_vals($ii,label) [iwidgets::labeledwidget $icoli_table_w.label$ii \
					       -labeltext "$ii" \
					       -background gray50 \
					       -foreground white]
		
		set icoli_vals($ii,combo) [iwidgets::combobox $icoli_table_w.combo$ii \
					       -textvariable sisimpar(icoli,$ii) \
					       -validate {regexpvalidate %c} \
					       -listheight 100 \
					       -focuscommand "focus_proc $ii" \
					       -selectioncommand "focus_proc $ii"]
		set tmp_icoli($ii) $sisimpar(icoli,$ii)
		set cnt 0   ;# counter starts at 0
		set sel 1   ;# that's the default selection if sisimpar(icoli) is empty
		foreach i $dim_name_soft {
		    $icoli_vals($ii,combo) insert list end $i
		    $icoli_vals($ii,combo) delete list [expr $sisimpar(soft_parameters) + 1]
		    if { $i == $tmp_icoli($ii) } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
			set sel $cnt
		    }
		    incr cnt
		}
		set sel [expr $sel - ($sisimpar(soft_cancel_counter) * ($sisimpar(soft_parameters) + 1))]
		$icoli_vals($ii,combo) selection set $sel
		set sisimpar(icoli,$ii) [lindex $dim_name_soft $sel]
		$icoli_table_w.combo$ii configure -state disabled -textbackground gray85 -foreground gray50
		
		set bz_vals($ii,entry) [iwidgets::entryfield $icoli_table_w.bz$ii \
					    -textvariable sisimpar(bz,$ii) \
					    -state disabled \
					    -validate {check_number_bz_values %W %P %c} \
					    -focuscommand "focus_proc $ii"]
		
		$icoli_table_w window configure $ii,0 -window $icoli_vals($ii,label) -relief flat -background gray50
		$icoli_table_w window configure $ii,1 -window $icoli_vals($ii,combo)
		$icoli_table_w window configure $ii,2 -window $bz_vals($ii,entry)
	    }
	} elseif { $sisimpar(soft_parameters) < 4 } {
	    set icoli_table_w [table $icoli_sframe_w.table \
				   -colwidth 25 \
				   -variable icoli_vals \
				   -titlerows 1 \
				   -titlecols 1 \
				   -cols 3 \
				   -rows 2 \
				   -yscrollcommand "$icoli_hsbar_w set" \
				   -height 4]
	    
	    $icoli_table_w width 0 5
	    $icoli_table_w width 1 22
	    $icoli_table_w width 2 22
	    
	    set icoli_vals(0,0) "\#"
	    set icoli_vals(0,1) "Variable"
	    set icoli_vals(0,2) "Calibration"
	    

	    for { set ii 1 } { $ii < 2 } { incr ii } {
		
		set variables_counter $ii
		
		set icoli_vals($ii,label) [iwidgets::labeledwidget $icoli_table_w.label$ii \
					       -labeltext "$ii" \
					       -background gray50 \
					       -foreground white]
		
		set icoli_vals($ii,combo) [iwidgets::combobox $icoli_table_w.combo$ii \
					       -textvariable sisimpar(icoli,$ii) \
					       -validate {regexpvalidate %c} \
					       -listheight 100 \
					       -focuscommand "focus_proc $ii" \
					       -selectioncommand "focus_proc $ii"]
		set tmp_icoli($ii) $sisimpar(icoli,$ii)
		set cnt 0   ;# counter starts at 0
		set sel 1   ;# that's the default selection if sisimpar(icoli) is empty
		foreach i $dim_name_soft {
		    $icoli_vals($ii,combo) insert list end $i
		    $icoli_vals($ii,combo) delete list [expr $sisimpar(soft_parameters) + 1]
		    if { $i == $tmp_icoli($ii) } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
			set sel $cnt
		    }
		    incr cnt
		}
		set sel [expr $sel - ($sisimpar(soft_cancel_counter) * ($sisimpar(soft_parameters) + 1))]
		$icoli_vals($ii,combo) selection set $sel
		set sisimpar(icoli,$ii) [lindex $dim_name_soft $sel]
		$icoli_table_w.combo$ii configure -state disabled -textbackground gray85 -foreground gray50
		
		set bz_vals($ii,entry) [iwidgets::entryfield $icoli_table_w.bz$ii \
					    -textvariable sisimpar(bz,$ii) \
					    -state disabled \
					    -validate {check_number_bz_values %W %P %c} \
					    -focuscommand "focus_proc $ii"]
		
		$icoli_table_w window configure $ii,0 -window $icoli_vals($ii,label) -relief flat -background gray50
		$icoli_table_w window configure $ii,1 -window $icoli_vals($ii,combo)
		$icoli_table_w window configure $ii,2 -window $bz_vals($ii,entry)
	    }
	}
    }
    
    
    ### setting the soft input data widgets disabled/normal depending on the input data
    if { ($sisimpar(soft_data) == 1) && ($sisimpar(directik) != "*.dat") } {
	$directik_w configure -state normal -textbackground white -foreground black
	$directik_browse_w configure -state normal
	$imbsim_checkbutton_w configure -state normal
	if { $sisimpar(imbsim) == 0 } {
	    if { $sisimpar(soft_parameters) >= 4 } {
		for {set ii 1} {$ii <= [expr $sisimpar(variables) - 3]} {incr ii} {
		    $icoli_table_w.combo$ii configure -state normal -textbackground white -foreground black
		    $icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
		}
	    } elseif { $sisimpar(soft_parameters) < 4 } {
		for {set ii 1} {$ii < 2} {incr ii} {
		    $icoli_table_w.combo$ii configure -state normal -textbackground white -foreground black
		    $icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
		}
	    }
	} elseif { $sisimpar(imbsim) == 1 } {
	    if { $sisimpar(soft_parameters) >= 4 } {
		for {set ii 1} {$ii <= [expr $sisimpar(variables) - 3]} {incr ii} {
		    $icoli_table_w.combo$ii configure -state normal -textbackground white -foreground black
		    $icoli_table_w.bz$ii configure -state normal -textbackground white -foreground black
		}
	    } elseif { $sisimpar(soft_parameters) < 4 } {
		for {set ii 1} {$ii < 2} {incr ii} {
		    $icoli_table_w.combo$ii configure -state normal -textbackground white -foreground black
		    $icoli_table_w.bz$ii configure -state normal -textbackground white -foreground black
		}
	    }
	}
    } elseif { ($sisimpar(soft_data) == 1) && ($sisimpar(directik) == "*.dat") } {
	$directik_w configure -state disabled -textbackground gray85 -foreground gray50
	$directik_browse_w configure -state normal
	$imbsim_checkbutton_w configure -state disabled
	for {set ii 1} {$ii <= 3} {incr ii} {
	    $icoli_table_w.combo$ii configure -state disabled -textbackground gray85 -foreground gray50
	    $icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
	}
    } elseif { ($sisimpar(soft_data) == 0) && ($sisimpar(directik) == "*.dat") } {
	$directik_w configure -state disabled -textbackground gray85 -foreground gray50
	$directik_browse_w configure -state disabled
	$imbsim_checkbutton_w configure -state disabled
	for {set ii 1} {$ii <= 3} {incr ii} {
	    $icoli_table_w.combo$ii configure -state disabled -textbackground gray85 -foreground gray50
	    $icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
	}
    } elseif { ($sisimpar(soft_data) == 0) && ($sisimpar(directik) != "*.dat") } {
	$directik_w configure -state disabled -textbackground gray85 -foreground gray50
	$directik_browse_w configure -state disabled
	$imbsim_checkbutton_w configure -state disabled
	for {set ii 1} {$ii <= [expr $sisimpar(variables) - 3]} {incr ii} {
	    $icoli_table_w.combo$ii configure -state disabled -textbackground gray85 -foreground gray50
	    $icoli_table_w.bz$ii configure -state disabled -textbackground gray85 -foreground gray50
	}
    }

    
    pack $icoli_hsbar_w  -side right -anchor w -fill y
    pack $icoli_table_w  -side left -anchor w -padx 2 -expand 1 -fill both
    pack $icoli_sframe_w -anchor nw -expand 0
    pack propagate $icoli_sframe_w 0
    
    
    grid $xvar_frame_w -row 0 -column 0 -sticky nw -pady 0 -padx 8
    grid $yvar_frame_w -row 1 -column 0 -sticky nw -pady 6 -padx 8
    grid $zvar_frame_w -row 2 -column 0 -sticky nw -pady 0 -padx 8
    
    grid $imbsim_checkbutton_w -row 3 -column 0 -sticky nw -pady 0
    grid $icoli_frame_w        -row 0 -column 1 -sticky new -rowspan 4 -pady 0
    
    
    set subframe3_w [frame $soft_input_frame_w.subfr3]
    
    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	-anchor w -expand 1 -padx 3 -pady 0
    
    
    # here are the output filed for the output data
    set output_iframe_w [iwidgets::labeledframe $work_frame_w.output \
			     -labeltext "Output" \
			     -labelpos nw \
			     -borderwidth 2 \
			     -relief groove]
    set output_frame_w [$output_iframe_w childsite]
    
    set subframe1_w [frame $output_frame_w.subfr1]
    set outfl_w [iwidgets::entryfield $subframe1_w.outfl_entry \
		     -labeltext "File " \
		     -width 69 \
		     -textvariable sisimpar(outfl)]
    set outfl_browse_w [iwidgets::pushbutton $subframe1_w.outfl_browse \
			    -height 27 \
			    -width 27 \
			    -defaultringpad 0 \
			    -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			    -command {
				set tmp_fname [tk_getSaveFile \
						   -defaultextension ".out" \
						   -filetypes {{{Output Files} {.out}} {{All Files} {.*}}} \
						   -initialfile $sisimpar(outfl) \
						   -initialdir $exedir \
						   -title "Save output file as ..."]

				if { $tmp_fname != {} } {
				    set sisimpar(outfl) $tmp_fname
				}
			    }]
    
    
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $outfl_w        -row 0 -column 0 -sticky we -pady 0
    grid $outfl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    
    set subframe2_w [frame $output_frame_w.subfr2]
    
    
    #
    # bad hack: We have to save the value we want the spinbox to be initialized
    # into a temporary variable, since the range option does a bad
    # initialization to the lower limit. After the spinint has been created
    # we set the textvariable back to the saved value!
    # (THIS IS A BUG IN spinit!!!)
    #
    set tmp_nsim $sisimpar(nsim)
    set realization_w [iwidgets::spinint $subframe2_w.realization \
			   -labeltext "Number of realizations " \
			   -range {1 100} \
			   -width 5 \
			   -textvariable sisimpar(nsim) \
			   -validate {check_number_nsim %W %P %c}]
    set sisimpar(nsim) $tmp_nsim
    
    set seed_w [iwidgets::entryfield $subframe2_w.seed_entry \
		    -labeltext "Seed " \
		    -width 14 \
		    -textvariable sisimpar(seed) \
		    -validate {check_number_seed %W %P %c}]
    
    set seed_pushbutton_w [iwidgets::pushbutton $subframe2_w.seed_pbutton \
			       -height 27 \
			       -width 27 \
			       -defaultringpad 0 \
			       -image [image create photo -file [file join "$exedir/images/get.gif"]] \
			       -command {seed_generator}]
    
    
    grid columnconfigure $subframe2_w 0 -minsize 118
    grid columnconfigure $subframe2_w 1 -minsize 147
    grid columnconfigure $subframe2_w 2 -minsize 50
    
    grid $realization_w     -row 0 -column 0 -sticky we -pady 0
    grid $seed_w            -row 0 -column 1 -sticky e -pady 0
    grid $seed_pushbutton_w -row 0 -column 2 -sticky w -pady 0
    
    
    set subframe3_w [frame $output_frame_w.subfr3]
    set dbgfl_w [iwidgets::entryfield $subframe3_w.dbgfl \
		     -labeltext "Debug file " \
		     -width 41 \
		     -textvariable sisimpar(dbgfl)]
    set dbgfl_browse_w [iwidgets::pushbutton $subframe3_w.dbgfl_browse \
			    -height 27 \
			    -width 27 \
			    -defaultringpad 0 \
			    -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			    -command {
				set tmp_fname [tk_getSaveFile \
						   -defaultextension ".dbg" \
						   -filetypes {{{Debug Files} {.dbg}} {{All Files} {.*}}} \
						   -initialfile $sisimpar(dbgfl) \
						   -title "Save debug file as ..."]

				if { $tmp_fname != {} } {
				    set sisimpar(dbgfl) $tmp_fname
				}
			    }]
    
    
    # initialize dimension names for debug level options
    set dbgfl_level_w [iwidgets::combobox $subframe3_w.dbgfl_level_option \
			   -labeltext "    Level " \
			   -textvariable sisimpar(idbg) \
			   -width 10 \
			   -listheight 60 \
			   -validate {regexpvalidate %c}]
    set sel 1   ;# that's the default selection if sisimpar(idbg) is empty
    $subframe3_w.dbgfl_level_option insert list end None Low Middle High
    if { $sisimpar(idbg) == "None" } {
	set sel 0
    } elseif { $sisimpar(idbg) == "Low" } {
	set sel 1
    } elseif { $sisimpar(idbg) == "Middle" } {
	set sel 2
    } elseif { $sisimpar(idbg) == "High" } {
	set sel 3
    }
    $dbgfl_level_w selection set $sel
    set $dbgfl_level_w [lindex $subframe3_w.dbgfl_level_option $sel]
    
    
    grid columnconfigure $subframe3_w 0 -minsize 250
    grid columnconfigure $subframe3_w 1 -minsize 27
    grid columnconfigure $subframe3_w 2 -minsize 105
    
    grid $dbgfl_w        -row 0 -column 0 -sticky we -pady 0
    grid $dbgfl_browse_w -row 0 -column 1 -sticky e -pady 0
    grid $dbgfl_level_w  -row 0 -column 2 -sticky e -pady 0
    
    
    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	-anchor w -expand 1 -padx 3 -pady 0
    
    
    pack \
	$hard_input_iframe_w \
	$soft_input_iframe_w \
	$output_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x
    
    
    pack \
	$work_frame_w \
	-anchor nw -side top -fill x -fill y -expand true \
	-padx 0 -pady 0
    
    
    #
    # page2: Geossav SISIM-parameters: Kriging Options
    #
    set page [$sisim_box_w.sisim_tbn add \
		  -label "Kriging Options"]
    
    set work_frame_w [frame $page.work_frame \
			  -borderwidth 2 \
			  -relief flat]
    
    
    set kriging_option_iframe_w [iwidgets::labeledframe $work_frame_w.kriging_option \
				     -labeltext "Options" \
				     -labelpos nw \
				     -borderwidth 2 \
				     -relief groove]
    set kriging_option_frame_w [$kriging_option_iframe_w childsite]; # here we have to get the actual label widget
    
    
    
    set subframe1_w [frame $kriging_option_frame_w.subfr1]
    

    # initialize variable type for kriging
    set variable_type_w [iwidgets::combobox $subframe1_w.variable_type_option \
			     -labeltext "Variable type " \
			     -width 15 \
			     -listheight 32 \
			     -textvariable sisimpar(vartype) \
			     -selectioncommand {extrapolation_activator} \
			     -validate {regexpvalidate %c}]
    set sel 0   ;# that's the default selection if sisimpar(vartype) is empty
    $subframe1_w.variable_type_option insert list end "Categorical" "Continous"
    if { $sisimpar(vartype) == "Categorical" } {
	set sel 0
    } elseif { $sisimpar(vartype) == "Continous" } {
	set sel 1
    }
    $variable_type_w selection set $sel
    set $variable_type_w [lindex $subframe1_w.variable_type_option $sel]
    
    
    grid columnconfigure $subframe1_w 0 -minsize 188
    
    grid $variable_type_w  -row 0 -column 0 -sticky e -pady 0
    
    
    set subframe2_w [frame $kriging_option_frame_w.subfr2]
    

    # initialize kriging type
    set kriging_type_w [iwidgets::combobox $subframe2_w.kriging_type \
			    -labeltext "Kriging type " \
			    -textvariable sisimpar(ktype) \
			    -width 15 \
			    -listheight 32 \
			    -validate {regexpvalidate %c}]
    set sel 0   ;# that's the default selection if sisimpar(ktype) is empty
    $subframe2_w.kriging_type insert list end "Simple Kriging" "Ordinary Kriging"
    if { $sisimpar(ktype) == "Simple Kriging" } {
	set sel 0
    } elseif { $sisimpar(ktype) == "Ordinary Kriging" } {
	set sel 1
    }
    $kriging_type_w selection set $sel
    set $kriging_type_w [lindex $$subframe2_w.kriging_type $sel]
    

    # initialize indicator flag
    set kriging_flag_w [iwidgets::combobox $subframe2_w.kriging_flag_option \
			    -labeltext "IK flag " \
			    -width 15 \
			    -listheight 32 \
			    -textvariable sisimpar(mik) \
			    -selectioncommand {cutoff_category_activator} \
			    -validate {regexpvalidate %c}]
    set sel 0   ;# that's the default selection if sisimpar(mik) is empty
    $subframe2_w.kriging_flag_option insert list end "Full IK" "Median IK"
    if { $sisimpar(mik) == "Full IK" } {
	set sel 0	
    } elseif { $sisimpar(mik) == "Median IK" } {
	set sel 1
    }
    $kriging_flag_w selection set $sel
    set $kriging_flag_w [lindex $subframe2_w.kriging_flag_option $sel]
    
    
    set cutoff_category_w [iwidgets::entryfield $subframe2_w.cutoff_category \
			       -labeltext "Cut/Cat " \
			       -width 7 \
			       -textvariable sisimpar(mikcat) \
			       -state disabled \
			       -textbackground gray85 \
			       -foreground gray50 \
			       -validate {check_number_mikcat %W %P %c}]
    
    if { $sisimpar(mik) == "Full IK" } {
	$cutoff_category_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $sisimpar(mik) == "Median IK" } {
	$cutoff_category_w configure -state normal -textbackground white -foreground black
    }
    
    
    grid columnconfigure $subframe2_w 0 -minsize 188
    grid columnconfigure $subframe2_w 1 -minsize 175
    grid columnconfigure $subframe2_w 2 -minsize 112
    
    
    grid $kriging_type_w    -row 0 -column 0 -sticky e -pady 0
    grid $kriging_flag_w    -row 0 -column 1 -sticky e -pady 0
    grid $cutoff_category_w -row 0 -column 2 -sticky e -pady 0
    
    
    pack \
	$subframe1_w \
	$subframe2_w \
	-anchor w -expand 1 -padx 3 -pady 3
    
    
    set search_parameter_iframe_w [iwidgets::labeledframe $work_frame_w.search_parameter \
				       -labeltext "Search Parameters" \
				       -labelpos nw \
				       -borderwidth 2 \
				       -relief groove]
    set search_parameter_frame_w [$search_parameter_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set subframe1_w [frame $search_parameter_frame_w.subfr1]
    
    
    set tmp_ndmax $sisimpar(ndmax)
    set max_org_data_w [iwidgets::spinint $subframe1_w.max_org_data \
			    -labeltext "Max original data for each Kriging " \
			    -range {0 100} \
			    -width 4 \
			    -textvariable sisimpar(ndmax) \
			    -validate {check_number_ndmax %W %P %c}]
    set sisimpar(ndmax) $tmp_ndmax
    
    
    set tmp_maxsec $sisimpar(maxsec)
    set max_soft_nodes_w [iwidgets::spinint $subframe1_w.max_soft_nodes \
			      -labeltext "Max soft indicator nodes for each Kriging " \
			      -range {0 100} \
			      -width 4 \
			      -textvariable sisimpar(maxsec) \
			      -validate {check_number_maxsec %W %P %c}]
    set sisimpar(maxsec) $tmp_maxsec
    
    
    grid columnconfigure $subframe1_w 0 -minsize 212
    grid columnconfigure $subframe1_w 1 -minsize 262
    
    grid $max_org_data_w   -row 0 -column 0 -sticky e -pady 0
    grid $max_soft_nodes_w -row 0 -column 1 -sticky e -pady 0
    
    
    set subframe2_w [frame $search_parameter_frame_w.subfr2]
    
    
    set tmp_ncnode $sisimpar(ncnode)
    set max_prev_nodes_w [iwidgets::spinint $subframe2_w.max_prev_nodes \
			      -labeltext "Max previous nodes for each Kriging " \
			      -range {0 100} \
			      -width 4 \
			      -textvariable sisimpar(ncnode) \
			      -validate {check_number_ncnode %W %P %c}]
    set sisimpar(ncnode) $tmp_ncnode
    

    set tmp_noct $sisimpar(noct)
    set max_per_octant_w [iwidgets::spinint $subframe2_w.max_per_octant \
			      -labeltext "Max original data per octant " \
			      -range {0 100} \
			      -width 4 \
			      -textvariable sisimpar(noct) \
			      -validate {check_number_noct %W %P %c}]
    set sisimpar(noct) $tmp_noct
    
    grid columnconfigure $subframe2_w 0 -minsize 212
    grid columnconfigure $subframe2_w 1 -minsize 262
    
    grid $max_prev_nodes_w -row 0 -column 1 -sticky e -pady 0    
    grid $max_per_octant_w -row 0 -column 0 -sticky e -pady 0
    
    
    set subframe3_w [frame $search_parameter_frame_w.subfr3]
    
    set search_strategy_checkbutton_w [checkbutton $subframe3_w.search_strategy_checkbutton \
					   -text "Assign data to nodes" \
					   -variable sisimpar(sstrat) \
					   -command {max_org_data_number}]
    
    if { $sisimpar(sstrat) == 1 } {
	$max_org_data_w configure -state disabled -foreground gray50
	$max_org_data_w component entry configure -background gray85
    } elseif { $sisimpar(sstrat) == 0 } {
	$max_org_data_w configure -state normal -foreground black
	$max_org_data_w component entry configure -background white
    }
    
    set search_radii_label_w [iwidgets::labeledwidget $subframe3_w.search_radii_label \
				  -labeltext "Search Radii" \
				  -labelfont "Helvetica 8 bold"]
    
    set search_radius_hmax_w [iwidgets::entryfield $subframe3_w.search_radius_hmax \
				  -labeltext "hmax " \
				  -width 5 \
				  -textvariable sisimpar(radius_hmax) \
				  -validate {check_number_radius %W %P %c}]
    
    set search_radius_hmin_w [iwidgets::entryfield $subframe3_w.search_radius_hmin \
				  -labeltext "hmin " \
				  -width 5 \
				  -textvariable sisimpar(radius_hmin) \
				  -validate {check_number_radius %W %P %c}]
    
    set search_radius_vert_w [iwidgets::entryfield $subframe3_w.search_radius_vert \
				  -labeltext "vert " \
				  -width 5 \
				  -textvariable sisimpar(radius_vert) \
				  -validate {check_number_radius %W %P %c}]
    
    grid columnconfigure $subframe3_w 0 -minsize 125
    grid columnconfigure $subframe3_w 1 -minsize 99
    grid columnconfigure $subframe3_w 2 -minsize 91
    grid columnconfigure $subframe3_w 3 -minsize 74
    grid columnconfigure $subframe3_w 4 -minsize 84
    
    grid $search_strategy_checkbutton_w -row 0 -column 0 -sticky w -pady 0
    grid $search_radii_label_w          -row 0 -column 1 -sticky e -pady 0
    grid $search_radius_hmax_w          -row 0 -column 2 -sticky e -pady 0
    grid $search_radius_hmin_w          -row 0 -column 3 -sticky e -pady 0
    grid $search_radius_vert_w          -row 0 -column 4 -sticky e -pady 0
    
    
    set subframe4_w [frame $search_parameter_frame_w.subfr4]
    
    set grid_refinement_checkbutton_w [checkbutton $subframe4_w.grid_refinement_checkbutton \
					   -text "Multiple grid search" \
					   -variable sisimpar(multgrid) \
					   -command {grid_refinement_number}]
        
    set search_angles_label_w [iwidgets::labeledwidget $subframe4_w.search_angles_label \
				   -labeltext "Search Angles" \
				   -labelfont "Helvetica 8 bold"]
    
    set search_angle_x_w [iwidgets::entryfield $subframe4_w.search_angle_x \
			      -labeltext "Azimuth " \
			      -textvariable sisimpar(sang1) \
			      -width 5 \
			      -validate {check_number_sang1 %W %P %c}]
    
    set search_angle_y_w [iwidgets::entryfield $subframe4_w.search_angle_y \
			      -labeltext "Dip " \
			      -textvariable sisimpar(sang2) \
			      -width 5 \
			      -validate {check_number_sang %W %P %c}]
    
    set search_angle_z_w [iwidgets::entryfield $subframe4_w.search_angle_z \
			      -labeltext "Plunge " \
			      -textvariable sisimpar(sang3) \
			      -width 5 \
			      -validate {check_number_sang %W %P %c}]
    
    
    grid columnconfigure $subframe4_w 0 -minsize 125
    grid columnconfigure $subframe4_w 1 -minsize 101
    grid columnconfigure $subframe4_w 2 -minsize 91 
    grid columnconfigure $subframe4_w 3 -minsize 74
    grid columnconfigure $subframe4_w 4 -minsize 84
    
    grid $grid_refinement_checkbutton_w -row 0 -column 0 -sticky w -pady 0
    grid $search_angles_label_w         -row 0 -column 1 -sticky e -pady 0
    grid $search_angle_x_w              -row 0 -column 2 -sticky e -pady 0
    grid $search_angle_y_w              -row 0 -column 3 -sticky e -pady 0
    grid $search_angle_z_w              -row 0 -column 4 -sticky e -pady 0
    

    set subframe5_w [frame $search_parameter_frame_w.subfr5]
    
    set grid_refinement_number_w [iwidgets::entryfield $subframe5_w.grid_refinement_number \
				      -labeltext "Number of multiple grid refinement to consider " \
				      -textvariable sisimpar(nmult) \
				      -width 5 \
				      -state disabled \
				      -textbackground gray85 \
				      -foreground gray50 \
				      -validate {check_number_multgrid %W %P %c}]
    
    if { $sisimpar(multgrid) == 0 } {
	$grid_refinement_number_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $sisimpar(multgrid) == 1 } {
	$grid_refinement_number_w configure -state normal -textbackground white -foreground black
    }
    
    
    grid columnconfigure $subframe5_w 0 -minsize 230
    
    grid $grid_refinement_number_w -row 0 -column 0 -sticky nws -pady 0
    
    
    pack \
	$subframe1_w \
	$subframe2_w \
	-anchor w -expand 1 -padx 3 -pady 3

    pack \
	$subframe3_w \
	$subframe4_w \
	-anchor w -expand 1 -padx 3 -pady 0
    
    pack \
	$subframe5_w \
	-anchor w -expand 1 -padx 3 -pady 3
    
    pack \
	$kriging_option_iframe_w \
	$search_parameter_iframe_w \
	-anchor nw -side top -padx 0 -pady 0 -expand 1 -fill x
    
    
    pack $work_frame_w \
	-anchor nw -side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    #
    # page3: Geossav SISIM-parameters: Variograms
    #
    set page [$sisim_box_w.sisim_tbn add \
		  -label "Variograms"]
    
    set work_frame_w [frame $page.workframe \
			  -borderwidth 2 \
			  -relief flat]
    
    set variogram_iframe_w [iwidgets::labeledframe $work_frame_w.variogram \
				-labeltext "Variograms" \
				-labelpos nw \
				-borderwidth 2 \
				-relief groove]
    set variogram_frame_w [$variogram_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set subframe1_w [frame $variogram_frame_w.subfr1]
    set plus_cut_cat_pushbutton_w [iwidgets::pushbutton $subframe1_w.plus_pbutton \
				       -height 27 \
				       -width 27 \
				       -state disabled \
				       -defaultringpad 0 \
				       -image [image create photo -file [file join "$exedir/images/plus.gif"]] \
				       -command {plus_cutoff_category}]
    
    set minus_cut_cat_pushbutton_w [iwidgets::pushbutton $subframe1_w.minus_pbutton \
					-height 27 \
					-width 27 \
					-state disabled \
					-defaultringpad 0 \
					-image [image create photo -file [file join "$exedir/images/minus.gif"]] \
					-command {minus_cutoff_category}]

    if { $sisimpar(datafl) != {*.dat}  || $sisimpar(directik) != {*.dat} } {
	$plus_cut_cat_pushbutton_w configure -state normal
	$minus_cut_cat_pushbutton_w configure -state normal
    }
    

    set subframe2_w [frame $variogram_frame_w.subfr2]
    set pdf_cdf_frame_w [frame $subframe2_w.pdf_cdf_frame \
			     -background grey40 \
			     -borderwidth 2 \
			     -relief sunken]
    
    set pdf_cdf_sframe_w [frame $pdf_cdf_frame_w.pdf_cdf_sframe \
			      -height 72 \
			      -width 200]
    set pdf_cdf_hsbar_w [scrollbar $pdf_cdf_sframe_w.yscroll \
			     -orient vertical \
			     -command "$pdf_cdf_sframe_w.table yview"]
    
    
    ### setting the pdf_pdf table normal/disabled depending on the input data
    if { ($sisimpar(directik) == "*.dat") && ($sisimpar(datafl) == "*.dat") || \
	     ($sisimpar(directik) == "*.dat") && ($sisimpar(datafl) != "*.dat") } {
	set pdf_cdf_table_w [table $pdf_cdf_sframe_w.table \
				 -variable pdf_cdf_vals \
				 -titlerows 1 \
				 -titlecols 1 \
				 -cols 3 \
				 -rows [expr $sisimpar(variables_initial) - 3] \
				 -yscrollcommand "$pdf_cdf_hsbar_w set" \
				 -height 4]
	
	$pdf_cdf_table_w width 0 3
	$pdf_cdf_table_w width 1 13
	$pdf_cdf_table_w width 2 13
	
	set pdf_cdf_vals(0,0) "\#"
	set pdf_cdf_vals(0,1) "Cut/Cat"
	set pdf_cdf_vals(0,2) "cdf/pdf"
	
	for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
	    set pdf_cdf_vals($ii,label) [iwidgets::labeledwidget $pdf_cdf_table_w.pdf_cdf$ii \
					     -labeltext "$ii" \
					     -background gray50 \
					     -foreground white]
	    
	    set sisimpar(ncat) $ii
	    
	    if { ($sisimpar(directik) == "*.dat") && ($sisimpar(datafl) == "*.dat") } {
		set sisimpar(cat,$ii) $ii
		set cut_cat_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
						 -textvariable sisimpar(cat,$ii) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
						 -focuscommand "focus_proc $ii"]
		
		set sisimpar(pdf,$ii) 0.15
		set pdf_cdf_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
						 -textvariable sisimpar(pdf,$ii) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
						 -focuscommand "focus_proc $ii"]
		
		$pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) -relief flat -background gray50
		$pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
		$pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
		
	    } elseif { ($sisimpar(directik) == "*.dat") && ($sisimpar(datafl) != "*.dat") } {
		set cut_cat_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
						 -textvariable sisimpar(cat,$ii) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
						 -focuscommand "focus_proc $ii"]
		
		set pdf_cdf_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
						 -textvariable sisimpar(pdf,$ii) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
						 -focuscommand "focus_proc $ii"]
		
		$pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) -relief flat -background gray50
		$pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
		$pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
	    }
	}
	if { $sisimpar(datafl) != "*.dat" } {
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
		$pdf_cdf_table_w.cut_cat$ii configure -state normal -textbackground white -foreground black
		$pdf_cdf_table_w.pdf_cdf_entry$ii configure -state normal -textbackground white -foreground black
	    }
	}
    } elseif { ($sisimpar(directik) != "*.dat") && ($sisimpar(datafl) == "*.dat") || \
		   ($sisimpar(directik) != "*.dat") && ($sisimpar(datafl) != "*.dat") } {
	if { $sisimpar(soft_parameters) >= 4 } {
	    set pdf_cdf_table_w [table $pdf_cdf_sframe_w.table \
				     -variable pdf_cdf_vals \
				     -titlerows 1 \
				     -titlecols 1 \
				     -cols 3 \
				     -rows [expr $sisimpar(variables) - 2] \
				     -yscrollcommand "$pdf_cdf_hsbar_w set" \
				     -height 4]
	    
	    $pdf_cdf_table_w width 0 3
	    $pdf_cdf_table_w width 1 13
	    $pdf_cdf_table_w width 2 13
	    
	    set pdf_cdf_vals(0,0) "\#"
	    set pdf_cdf_vals(0,1) "Cut/Cat"
	    set pdf_cdf_vals(0,2) "cdf/pdf"
	
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3] } { incr ii } {
		set pdf_cdf_vals($ii,label) [iwidgets::labeledwidget $pdf_cdf_table_w.pdf_cdf$ii \
						 -labeltext "$ii" \
						 -background gray50 \
						 -foreground white]
		
		set sisimpar(ncat) $ii
		
		set cut_cat_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
						 -textvariable sisimpar(cat,$ii) \
						 -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
						 -focuscommand "focus_proc $ii"]
		
		set pdf_cdf_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
						 -textvariable sisimpar(pdf,$ii) \
						 -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
						 -focuscommand "focus_proc $ii"]
		
		$pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) -relief flat -background gray50
		$pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
		$pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
	    }
	} elseif { $sisimpar(soft_parameters) < 4 } {
	    set pdf_cdf_table_w [table $pdf_cdf_sframe_w.table \
				     -variable pdf_cdf_vals \
				     -titlerows 1 \
				     -titlecols 1 \
				     -cols 3 \
				     -rows 2 \
				     -yscrollcommand "$pdf_cdf_hsbar_w set" \
				     -height 4]
	    
	    $pdf_cdf_table_w width 0 3
	    $pdf_cdf_table_w width 1 13
	    $pdf_cdf_table_w width 2 13
	    
	    set pdf_cdf_vals(0,0) "\#"
	    set pdf_cdf_vals(0,1) "Cut/Cat"
	    set pdf_cdf_vals(0,2) "cdf/pdf"
	
	    for { set ii 1 } { $ii < 2 } { incr ii } {
		set pdf_cdf_vals($ii,label) [iwidgets::labeledwidget $pdf_cdf_table_w.pdf_cdf$ii \
						 -labeltext "$ii" \
						 -background gray50 \
						 -foreground white]
		
		set sisimpar(ncat) $ii
		
		set cut_cat_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.cut_cat$ii \
						 -textvariable sisimpar(cat,$ii) \
						 -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
						 -focuscommand "focus_proc $ii"]
		
		set pdf_cdf_vals($ii,entry) [iwidgets::entryfield $pdf_cdf_table_w.pdf_cdf_entry$ii \
						 -textvariable sisimpar(pdf,$ii) \
						 -validate {check_number_cut_cat_pdf_cdf %W %P %c} \
						 -focuscommand "focus_proc $ii"]
		
		$pdf_cdf_table_w window configure $ii,0 -window $pdf_cdf_vals($ii,label) -relief flat -background gray50
		$pdf_cdf_table_w window configure $ii,1 -window $cut_cat_vals($ii,entry)
		$pdf_cdf_table_w window configure $ii,2 -window $pdf_cdf_vals($ii,entry)
	    }
	}
    }


    grid columnconfigure $subframe1_w 0 -minsize 10
    grid columnconfigure $subframe1_w 1 -minsize 10
    grid $plus_cut_cat_pushbutton_w  -row 0 -column 0 -sticky w -pady 0
    grid $minus_cut_cat_pushbutton_w -row 0 -column 1 -sticky w -pady 0
    
    pack $pdf_cdf_hsbar_w  -side right -anchor w -fill y
    pack $pdf_cdf_table_w  -side left -anchor w -padx 2 -expand 1 -fill both
    pack $pdf_cdf_sframe_w -anchor nw -expand 1
    pack propagate $pdf_cdf_sframe_w 0
    grid $pdf_cdf_frame_w -row 0 -column 0 -sticky news -pady 0
    
    
    set spaceframe_w [frame $variogram_frame_w.spaceframe]
    
    
    set subframe3_variogram_w [frame $variogram_frame_w.subfr3]
    
    set plus_model_pushbutton_w [iwidgets::pushbutton $subframe3_variogram_w.plus_model_pbutton \
				     -height 27 \
				     -width 27 \
				     -defaultringpad 0 \
				     -state disabled \
				     -image [image create photo -file [file join "$exedir/images/plus.gif"]] \
				     -command {plus_model}]
    
    set minus_model_pushbutton_w [iwidgets::pushbutton $subframe3_variogram_w.minus_model_pbutton \
				      -height 27 \
				      -width 27 \
				      -defaultringpad 0 \
				      -state disabled \
				      -image [image create photo -file [file join "$exedir/images/minus.gif"]] \
				      -command {minus_model}] 
    
    
    set subframe4_w [frame $variogram_frame_w.subfr4]
    set model_frame_w [frame $subframe4_w.model_frame \
			   -background grey40 \
			   -borderwidth 2 \
			   -relief sunken]
    
    set model_sframe_w [frame $model_frame_w.model_sframe \
			    -height 72 \
			    -width 480]
    
    
    ### setting the model table normal/disabled depending on the input data
    if { ($sisimpar(directik) == "*.dat") && ($sisimpar(datafl) == "*.dat") } {
	
	for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
	    set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
					-orient vertical \
					-command "$model_sframe_w.table_$ii yview"]
	    
	    set no($ii) 1               ;# number of variogram types with corresponding parameters
	    set model_table_w($ii) [table $model_sframe_w.table_$ii \
					-variable model_vals \
					-titlerows 1 \
					-titlecols 0 \
					-cols 9 \
					-rows [expr $no($ii) + 1] \
					-yscrollcommand "$model_hsbar_w($ii) set" \
					-height 4]
	    
	    $model_table_w($ii) width 0 3
	    $model_table_w($ii) width 1 15
	    $model_table_w($ii) width 2 8
	    $model_table_w($ii) width 3 8
	    $model_table_w($ii) width 4 8
	    $model_table_w($ii) width 5 8
	    $model_table_w($ii) width 6 8
	    $model_table_w($ii) width 7 8
	    $model_table_w($ii) width 8 8
	    
	    set model_vals(0,0) "No"
	    set model_vals(0,1) "Type"
	    set model_vals(0,2) "Cc"
	    set model_vals(0,3) "Azimuth"
	    set model_vals(0,4) "Dip"
	    set model_vals(0,5) "Plunge"
	    set model_vals(0,6) "hmax"
	    set model_vals(0,7) "hmin"
	    set model_vals(0,8) "vert"
	    
	    
	    set sum_cc($ii) 0                ;# sum of all cc values
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		set no_counter($ii) $i       ;# counter of the number of variogram types with corresponding parameters
		
		set model_vals_label($i,$ii) [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
						  -labeltext "$i" \
						  -background gray50 \
						  -foreground white]
		
		set sisimpar(it$ii,$i) 0
		set model_type($i,combo$ii) [iwidgets::combobox $model_table_w($ii).combo_type$i \
						 -width 15 \
						 -textvariable sisimpar(it$ii,$i) \
						 -listheight 75]
		$model_table_w($ii).combo_type$i insert list end "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
		$model_type($i,combo$ii) selection set 0
		set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i 0]
		$model_table_w($ii).combo_type$i configure \
		    -state disabled -textbackground gray85 -foreground gray50
		
		set sisimpar(cc$ii,$i) 0.9
		set model_cc($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
					       -textvariable sisimpar(cc$ii,$i) \
					       -state disabled \
					       -textbackground gray85 \
					       -foreground gray50 \
					       -validate "check_number_cc %W %S %P $i $ii"]
		
		set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]
					
		set sisimpar(ang1$ii,$i) 0
		set model_ang1($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
						 -textvariable sisimpar(ang1$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang1 %W %P %c}]
		
		set sisimpar(ang2$ii,$i) 0
		set model_ang2($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang2$i \
						 -textvariable sisimpar(ang2$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set sisimpar(ang3$ii,$i) 0
		set model_ang3($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang3$i \
						 -textvariable sisimpar(ang3$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set sisimpar(aa_hmax$ii,$i) 10
		set model_aa_hmax($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
						    -textvariable sisimpar(aa_hmax$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set sisimpar(aa_hmin$ii,$i) 10
		set model_aa_hmin($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
						    -textvariable sisimpar(aa_hmin$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set sisimpar(aa_vert$ii,$i) 10
		set model_aa_vert($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
						    -textvariable sisimpar(aa_vert$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		$model_table_w($ii) window configure $i,0 -window $model_vals_label($i,$ii) -relief flat -background gray50
		$model_table_w($ii) window configure $i,1 -window $model_type($i,combo$ii)
		$model_table_w($ii) window configure $i,2 -window $model_cc($i,entry$ii)
		$model_table_w($ii) window configure $i,3 -window $model_ang1($i,entry$ii)
		$model_table_w($ii) window configure $i,4 -window $model_ang2($i,entry$ii)	
		$model_table_w($ii) window configure $i,5 -window $model_ang3($i,entry$ii)	
		$model_table_w($ii) window configure $i,6 -window $model_aa_hmax($i,entry$ii)	
		$model_table_w($ii) window configure $i,7 -window $model_aa_hmin($i,entry$ii)	
		$model_table_w($ii) window configure $i,8 -window $model_aa_vert($i,entry$ii)
	    }
	    pack $model_hsbar_w($ii)  -side right -anchor w -fill y
	    pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
	    
	    
	    ### creates nugget entry and sill label
	    set sisimpar(c0$ii) 0.1
	    set models_nugget_w($ii) [iwidgets::entryfield $subframe3_variogram_w.entry_nugget$ii \
					  -labeltext "Nugget " \
					  -width 7 \
					  -textvariable sisimpar(c0$ii) \
					  -state disabled \
					  -textbackground gray85 \
					  -foreground gray50 \
					  -validate "check_number_nugget %W %P"]
	    
	    set models_sill_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
					      -labeltext "Sill " \
					      -state disabled]
	    
	    set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
	    set models_sill_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
					      -labelvariable sill($ii) \
					      -state disabled]
	    
	    
	    set cut_cat_number_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_label$ii \
						 -labeltext "Cut/Cat \# " \
						 -state disabled]
	    set cut_cat_number($ii) $ii
	    set cut_cat_number_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_value$ii \
						 -labeltext "$cut_cat_number($ii)" \
						 -state disabled]
	    
	    
	    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	    grid $models_nugget_w($ii)        -row 0 -column 2 -sticky e -pady 0
	    grid $models_sill_label_w($ii)    -row 0 -column 3 -sticky e -pady 0
	    grid $models_sill_value_w($ii)    -row 0 -column 4 -sticky w -pady 0
	    grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	    grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	}
    } elseif { ($sisimpar(directik) == "*.dat") && ($sisimpar(datafl) != "*.dat") } {
	
	for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
	    set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
					-orient vertical \
					-command "$model_sframe_w.table_$ii yview"]
	    
	    set model_table_w($ii) [table $model_sframe_w.table_$ii \
					-variable model_vals \
					-titlerows 1 \
					-titlecols 0 \
					-cols 9 \
					-rows [expr $no($ii) + 1] \
					-yscrollcommand "$model_hsbar_w($ii) set" \
					-height 4]
	    
	    $model_table_w($ii) width 0 3
	    $model_table_w($ii) width 1 15
	    $model_table_w($ii) width 2 8
	    $model_table_w($ii) width 3 8
	    $model_table_w($ii) width 4 8
	    $model_table_w($ii) width 5 8
	    $model_table_w($ii) width 6 8
	    $model_table_w($ii) width 7 8
	    $model_table_w($ii) width 8 8
	    
	    set model_vals(0,0) "No"
	    set model_vals(0,1) "Type"
	    set model_vals(0,2) "Cc"
	    set model_vals(0,3) "Azimuth"
	    set model_vals(0,4) "Dip"
	    set model_vals(0,5) "Plunge"
	    set model_vals(0,6) "hmax"
	    set model_vals(0,7) "hmin"
	    set model_vals(0,8) "vert"
	    
	    
	    set sum_cc($ii) 0                ;# sum of all cc values
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		
		set no_counter($ii) $i       ;# counter of the number of variogram types with corresponding parameters
		
		set model_vals_label($i,$ii) [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
						  -labeltext "$i" \
						  -background gray50 \
						  -foreground white]
		
		set model_type($i,combo$ii) [iwidgets::combobox $model_table_w($ii).combo_type$i \
						 -width 15 \
						 -textvariable sisimpar(it$ii,$i) \
						 -listheight 75]
		$model_table_w($ii).combo_type$i insert list end "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
		if { $sisimpar(it$ii,$i) == "Spherical" } {
		    set sel 0	
		} elseif { $sisimpar(it$ii,$i) == "Exponential" } {
		    set sel 1
		} elseif { $sisimpar(it$ii,$i) == "Gaussian" } {
		    set sel 2
		} elseif { $sisimpar(it$ii,$i) == "Power" } {
		    set sel 3
		} elseif { $sisimpar(it$ii,$i) == "Hole effect" } {
		    set sel 4
		}
		$model_type($i,combo$ii) selection set $sel
		set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i $sel]
		$model_table_w($ii).combo_type$i configure \
		    -state disabled -textbackground gray85 -foreground gray50
		
		set model_cc($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
					       -textvariable sisimpar(cc$ii,$i) \
					       -state disabled \
					       -textbackground gray85 \
					       -foreground gray50 \
					       -validate "check_number_cc %W %S %P $i $ii"]
		
		set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]
				
		set model_ang1($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
						 -textvariable sisimpar(ang1$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang1 %W %P %c}]
		
		set model_ang2($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang2$i \
						 -textvariable sisimpar(ang2$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set model_ang3($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang3$i \
						 -textvariable sisimpar(ang3$ii,$i) \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -validate {check_number_sang %W %P %c}]
		
		set model_aa_hmax($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
						    -textvariable sisimpar(aa_hmax$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set model_aa_hmin($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
						    -textvariable sisimpar(aa_hmin$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		set model_aa_vert($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
						    -textvariable sisimpar(aa_vert$ii,$i) \
						    -state disabled \
						    -textbackground gray85 \
						    -foreground gray50 \
						    -validate {check_number_radius %W %P %c}]
		
		$model_table_w($ii) window configure $i,0 -window $model_vals_label($i,$ii) -relief flat -background gray50
		$model_table_w($ii) window configure $i,1 -window $model_type($i,combo$ii)
		$model_table_w($ii) window configure $i,2 -window $model_cc($i,entry$ii)
		$model_table_w($ii) window configure $i,3 -window $model_ang1($i,entry$ii)
		$model_table_w($ii) window configure $i,4 -window $model_ang2($i,entry$ii)	
		$model_table_w($ii) window configure $i,5 -window $model_ang3($i,entry$ii)	
		$model_table_w($ii) window configure $i,6 -window $model_aa_hmax($i,entry$ii)	
		$model_table_w($ii) window configure $i,7 -window $model_aa_hmin($i,entry$ii)	
		$model_table_w($ii) window configure $i,8 -window $model_aa_vert($i,entry$ii)
	    }
	    pack $model_hsbar_w($ii)  -side right -anchor w -fill y
	    pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
	    
	    
	    ### creates nugget entry and sill label
	    set models_nugget_w($ii) [iwidgets::entryfield $subframe3_variogram_w.entry_nugget$ii \
					  -labeltext "Nugget " \
					  -width 7 \
					  -textvariable sisimpar(c0$ii) \
					  -state disabled \
					  -textbackground gray85 \
					  -foreground gray50 \
					  -validate "check_number_nugget %W %P"]
	    	    
	    set models_sill_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
					      -labeltext "Sill " \
					      -state disabled]
	    
	    set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
	    set models_sill_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
					      -labelvariable sill($ii) \
					      -state disabled]
	    
	    
	    set cut_cat_number_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_label$ii \
						 -labeltext "Cut/Cat \# " \
						 -state disabled]
	    set cut_cat_number($ii) $ii
	    set cut_cat_number_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_value$ii \
						 -labeltext "$cut_cat_number($ii)" \
						 -state disabled]
	    
	    
	    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	    grid $models_nugget_w($ii)        -row 0 -column 2 -sticky e -pady 0
	    grid $models_sill_label_w($ii)    -row 0 -column 3 -sticky e -pady 0
	    grid $models_sill_value_w($ii)    -row 0 -column 4 -sticky w -pady 0
	    grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	    grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	}
	if { $focus_variable != 0 } {
	    $plus_model_pushbutton_w configure -state normal
	    $minus_model_pushbutton_w configure -state normal
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
		for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		    $model_table_w($ii).combo_type$i configure \
			-state normal -textbackground white -foreground black
		    $model_table_w($ii).entry_cc$i configure \
			-state normal -textbackground white -foreground black
		    $model_table_w($ii).entry_ang1$i configure \
			-state normal -textbackground white -foreground black
		    $model_table_w($ii).entry_ang2$i configure \
			-state normal -textbackground white -foreground black
		    $model_table_w($ii).entry_ang3$i configure \
			-state normal -textbackground white -foreground black
		    $model_table_w($ii).entry_aa_hmax$i configure \
			-state normal -textbackground white -foreground black
		    $model_table_w($ii).entry_aa_hmin$i configure \
			-state normal -textbackground white -foreground black
		    $model_table_w($ii).entry_aa_vert$i configure \
			-state normal -textbackground white -foreground black
		}
		$subframe3_variogram_w.entry_nugget$ii configure -state normal -textbackground white -foreground black
		$subframe3_variogram_w.label_sill$ii configure -state normal
		$subframe3_variogram_w.value_sill$ii configure -state normal
		$subframe3_variogram_w.cut_cat_number_label$ii configure -state normal
		$subframe3_variogram_w.cut_cat_number_value$ii configure -state normal
	    }
	}
    } elseif { ($sisimpar(directik) != "*.dat") && ($sisimpar(datafl) == "*.dat") || \
		   ($sisimpar(directik) != "*.dat") && ($sisimpar(datafl) != "*.dat") } {

	if { $sisimpar(soft_parameters) >= 4 } {
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3] } { incr ii } {
		set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
					    -orient vertical \
					    -command "$model_sframe_w.table_$ii yview"]
		
		set model_table_w($ii) [table $model_sframe_w.table_$ii \
					    -variable model_vals \
					    -titlerows 1 \
					    -titlecols 0 \
					    -cols 9 \
					    -rows [expr $no($ii) + 1] \
					    -yscrollcommand "$model_hsbar_w($ii) set" \
					    -height 4]
		
		$model_table_w($ii) width 0 3
		$model_table_w($ii) width 1 15
		$model_table_w($ii) width 2 8
		$model_table_w($ii) width 3 8
		$model_table_w($ii) width 4 8
		$model_table_w($ii) width 5 8
		$model_table_w($ii) width 6 8
		$model_table_w($ii) width 7 8
		$model_table_w($ii) width 8 8
		
		set model_vals(0,0) "No"
		set model_vals(0,1) "Type"
		set model_vals(0,2) "Cc"
		set model_vals(0,3) "Azimuth"
		set model_vals(0,4) "Dip"
		set model_vals(0,5) "Plunge"
		set model_vals(0,6) "hmax"
		set model_vals(0,7) "hmin"
		set model_vals(0,8) "vert"
		
		
		set sum_cc($ii) 0                ;# sum of all cc values
		for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		    set no_counter($ii) $i       ;# counter of the number of variogram types with corresponding parameters
		    
		    set model_vals_label($i,$ii) [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
						      -labeltext "$i" \
						      -background gray50 \
						      -foreground white]
		    
		    set model_type($i,combo$ii) [iwidgets::combobox $model_table_w($ii).combo_type$i \
						     -width 15 \
						     -textvariable sisimpar(it$ii,$i) \
						     -listheight 75]
		    $model_table_w($ii).combo_type$i \
			insert list end "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
		    if { $sisimpar(it$ii,$i) == "Spherical" } {
			set sel 0	
		    } elseif { $sisimpar(it$ii,$i) == "Exponential" } {
			set sel 1
		    } elseif { $sisimpar(it$ii,$i) == "Gaussian" } {
			set sel 2
		    } elseif { $sisimpar(it$ii,$i) == "Power" } {
			set sel 3
		    } elseif { $sisimpar(it$ii,$i) == "Hole effect" } {
			set sel 4
		    }
		    $model_type($i,combo$ii) selection set $sel
		    set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i $sel]
		    $model_table_w($ii).combo_type$i configure -state disabled -textbackground gray85 -foreground gray80
		    
		    set model_cc($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
						   -textvariable sisimpar(cc$ii,$i) \
						   -state disabled \
						   -textbackground gray85 \
						   -foreground gray50 \
						   -validate "check_number_cc %W %S %P $i $ii"]
		    
		    set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]
		    
		    set model_ang1($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
						     -textvariable sisimpar(ang1$ii,$i) \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50 \
						     -validate {check_number_sang1 %W %P %c}]
		    
		    set model_ang2($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang2$i \
						     -textvariable sisimpar(ang2$ii,$i) \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50 \
						     -validate {check_number_sang %W %P %c}]
		    
		    set model_ang3($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang3$i \
						     -textvariable sisimpar(ang3$ii,$i) \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50 \
						     -validate {check_number_sang %W %P %c}]
		    
		    set model_aa_hmax($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
							-textvariable sisimpar(aa_hmax$ii,$i) \
							-state disabled \
							-textbackground gray85 \
							-foreground gray50 \
							-validate {check_number_radius %W %P %c}]
		    
		    set model_aa_hmin($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
							-textvariable sisimpar(aa_hmin$ii,$i) \
							-state disabled \
							-textbackground gray85 \
							-foreground gray50 \
							-validate {check_number_radius %W %P %c}]
		    
		    set model_aa_vert($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
							-textvariable sisimpar(aa_vert$ii,$i) \
							-state disabled \
							-textbackground gray85 \
							-foreground gray50 \
							-validate {check_number_radius %W %P %c}]
		    
		    $model_table_w($ii) window configure $i,0 \
			-window $model_vals_label($i,$ii) -relief flat -background gray50
		    $model_table_w($ii) window configure $i,1 -window $model_type($i,combo$ii)
		    $model_table_w($ii) window configure $i,2 -window $model_cc($i,entry$ii)
		    $model_table_w($ii) window configure $i,3 -window $model_ang1($i,entry$ii)
		    $model_table_w($ii) window configure $i,4 -window $model_ang2($i,entry$ii)	
		    $model_table_w($ii) window configure $i,5 -window $model_ang3($i,entry$ii)	
		    $model_table_w($ii) window configure $i,6 -window $model_aa_hmax($i,entry$ii)	
		    $model_table_w($ii) window configure $i,7 -window $model_aa_hmin($i,entry$ii)	
		    $model_table_w($ii) window configure $i,8 -window $model_aa_vert($i,entry$ii)
		}
		pack $model_hsbar_w($ii)  -side right -anchor w -fill y
		pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
		
		
		### creates nugget entry and sill label
		set models_nugget_w($ii) [iwidgets::entryfield $subframe3_variogram_w.entry_nugget$ii \
					      -labeltext "Nugget " \
					      -width 7 \
					      -textvariable sisimpar(c0$ii) \
					      -state disabled \
					      -textbackground gray85 \
					      -foreground gray50 \
					      -validate "check_number_nugget %W %P"]
		
		set models_sill_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
						  -labeltext "Sill " \
						  -state disabled]
		
		set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
		set models_sill_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
						  -labelvariable sill($ii) \
						  -state disabled]
		
		set cut_cat_number_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_label$ii \
						     -labeltext "Cut/Cat \# " \
						     -state disabled]
		set cut_cat_number($ii) $ii
		set cut_cat_number_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_value$ii \
						     -labeltext "$cut_cat_number($ii)" \
						     -state disabled]
		
		
		grid columnconfigure $subframe3_variogram_w 2 -minsize 121
		grid columnconfigure $subframe3_variogram_w 3 -minsize 47
		grid columnconfigure $subframe3_variogram_w 4 -minsize 60
		grid columnconfigure $subframe3_variogram_w 5 -minsize 180
		grid columnconfigure $subframe3_variogram_w 6 -minsize 10
		grid $models_nugget_w($ii)        -row 0 -column 2 -sticky e -pady 0
		grid $models_sill_label_w($ii)    -row 0 -column 3 -sticky e -pady 0
		grid $models_sill_value_w($ii)    -row 0 -column 4 -sticky w -pady 0
		grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
		grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	    }
	    if { $focus_variable != 0 } {
		$plus_model_pushbutton_w configure -state normal
		$minus_model_pushbutton_w configure -state normal
		for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3] } { incr ii } {
		    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
			$model_table_w($ii).combo_type$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_cc$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_ang1$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_ang2$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_ang3$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_aa_hmax$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_aa_hmin$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_aa_vert$i configure \
			    -state normal -textbackground white -foreground black
		    }
		    $subframe3_variogram_w.entry_nugget$ii configure -state normal -textbackground white -foreground black
		    $subframe3_variogram_w.label_sill$ii configure -state normal
		    $subframe3_variogram_w.value_sill$ii configure -state normal
		    $subframe3_variogram_w.cut_cat_number_label$ii configure -state normal
		    $subframe3_variogram_w.cut_cat_number_value$ii configure -state normal
		}
	    }
	} elseif { $sisimpar(soft_parameters) < 4 } {
	    for { set ii 1 } { $ii < 2 } { incr ii } {
		set model_hsbar_w($ii) [scrollbar $model_sframe_w.yscroll($ii) \
					    -orient vertical \
					    -command "$model_sframe_w.table_$ii yview"]
		
		set model_table_w($ii) [table $model_sframe_w.table_$ii \
					    -variable model_vals \
					    -titlerows 1 \
					    -titlecols 0 \
					    -cols 9 \
					    -rows [expr $no($ii) + 1] \
					    -yscrollcommand "$model_hsbar_w($ii) set" \
					    -height 4]
		
		$model_table_w($ii) width 0 3
		$model_table_w($ii) width 1 15
		$model_table_w($ii) width 2 8
		$model_table_w($ii) width 3 8
		$model_table_w($ii) width 4 8
		$model_table_w($ii) width 5 8
		$model_table_w($ii) width 6 8
		$model_table_w($ii) width 7 8
		$model_table_w($ii) width 8 8
		
		set model_vals(0,0) "No"
		set model_vals(0,1) "Type"
		set model_vals(0,2) "Cc"
		set model_vals(0,3) "Azimuth"
		set model_vals(0,4) "Dip"
		set model_vals(0,5) "Plunge"
		set model_vals(0,6) "hmax"
		set model_vals(0,7) "hmin"
		set model_vals(0,8) "vert"
		
		
		set sum_cc($ii) 0                ;# sum of all cc values
		for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		    set no_counter($ii) $i       ;# counter of the number of variogram types with corresponding parameters
		    
		    set model_vals_label($i,$ii) [iwidgets::labeledwidget $model_table_w($ii).label_vals$i \
						      -labeltext "$i" \
						      -background gray50 \
						      -foreground white]
		    
		    set model_type($i,combo$ii) [iwidgets::combobox $model_table_w($ii).combo_type$i \
						     -width 15 \
						     -textvariable sisimpar(it$ii,$i) \
						     -listheight 75]
		    $model_table_w($ii).combo_type$i \
			insert list end "Spherical" "Exponential" "Gaussian" "Power" "Hole effect"
		    if { $sisimpar(it$ii,$i) == "Spherical" } {
			set sel 0	
		    } elseif { $sisimpar(it$ii,$i) == "Exponential" } {
			set sel 1
		    } elseif { $sisimpar(it$ii,$i) == "Gaussian" } {
			set sel 2
		    } elseif { $sisimpar(it$ii,$i) == "Power" } {
			set sel 3
		    } elseif { $sisimpar(it$ii,$i) == "Hole effect" } {
			set sel 4
		    }
		    $model_type($i,combo$ii) selection set $sel
		    set $model_type($i,combo$ii) [lindex $model_table_w($ii).combo_type$i $sel]
		    $model_table_w($ii).combo_type$i configure -state disabled -textbackground gray85 -foreground gray80
		    
		    set model_cc($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_cc$i \
						   -textvariable sisimpar(cc$ii,$i) \
						   -state disabled \
						   -textbackground gray85 \
						   -foreground gray50 \
						   -validate "check_number_cc %W %S %P $i $ii"]
		    
		    set sum_cc($ii) [expr $sum_cc($ii) + $sisimpar(cc$ii,$i)]
		    
		    set model_ang1($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang1$i \
						     -textvariable sisimpar(ang1$ii,$i) \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50 \
						     -validate {check_number_sang1 %W %P %c}]
		    
		    set model_ang2($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang2$i \
						     -textvariable sisimpar(ang2$ii,$i) \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50 \
						     -validate {check_number_sang %W %P %c}]
		    
		    set model_ang3($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_ang3$i \
						     -textvariable sisimpar(ang3$ii,$i) \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50 \
						     -validate {check_number_sang %W %P %c}]
		    
		    set model_aa_hmax($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmax$i \
							-textvariable sisimpar(aa_hmax$ii,$i) \
							-state disabled \
							-textbackground gray85 \
							-foreground gray50 \
							-validate {check_number_radius %W %P %c}]
		    
		    set model_aa_hmin($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_hmin$i \
							-textvariable sisimpar(aa_hmin$ii,$i) \
							-state disabled \
							-textbackground gray85 \
							-foreground gray50 \
							-validate {check_number_radius %W %P %c}]
		    
		    set model_aa_vert($i,entry$ii) [iwidgets::entryfield $model_table_w($ii).entry_aa_vert$i \
							-textvariable sisimpar(aa_vert$ii,$i) \
							-state disabled \
							-textbackground gray85 \
							-foreground gray50 \
							-validate {check_number_radius %W %P %c}]
		    
		    $model_table_w($ii) window configure $i,0 \
			-window $model_vals_label($i,$ii) -relief flat -background gray50
		    $model_table_w($ii) window configure $i,1 -window $model_type($i,combo$ii)
		    $model_table_w($ii) window configure $i,2 -window $model_cc($i,entry$ii)
		    $model_table_w($ii) window configure $i,3 -window $model_ang1($i,entry$ii)
		    $model_table_w($ii) window configure $i,4 -window $model_ang2($i,entry$ii)	
		    $model_table_w($ii) window configure $i,5 -window $model_ang3($i,entry$ii)	
		    $model_table_w($ii) window configure $i,6 -window $model_aa_hmax($i,entry$ii)	
		    $model_table_w($ii) window configure $i,7 -window $model_aa_hmin($i,entry$ii)	
		    $model_table_w($ii) window configure $i,8 -window $model_aa_vert($i,entry$ii)
		}
		pack $model_hsbar_w($ii)  -side right -anchor w -fill y
		pack $model_table_w($ii)  -side left -anchor w -padx 2 -expand 1 -fill both
		
		
		### creates nugget entry and sill label
		set models_nugget_w($ii) [iwidgets::entryfield $subframe3_variogram_w.entry_nugget$ii \
					      -labeltext "Nugget " \
					      -width 7 \
					      -textvariable sisimpar(c0$ii) \
					      -state disabled \
					      -textbackground gray85 \
					      -foreground gray50 \
					      -validate "check_number_nugget %W %P"]
		
		set models_sill_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.label_sill$ii \
						  -labeltext "Sill " \
						  -state disabled]
		
		set sill($ii) [expr $sisimpar(c0$ii) + $sum_cc($ii)]
		set models_sill_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.value_sill$ii \
						  -labelvariable sill($ii) \
						  -state disabled]
		
		set cut_cat_number_label_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_label$ii \
						     -labeltext "Cut/Cat \# " \
						     -state disabled]
		set cut_cat_number($ii) $ii
		set cut_cat_number_value_w($ii) [iwidgets::labeledwidget $subframe3_variogram_w.cut_cat_number_value$ii \
						     -labeltext "$cut_cat_number($ii)" \
						     -state disabled]
		
		
		grid columnconfigure $subframe3_variogram_w 2 -minsize 121
		grid columnconfigure $subframe3_variogram_w 3 -minsize 47
		grid columnconfigure $subframe3_variogram_w 4 -minsize 60
		grid columnconfigure $subframe3_variogram_w 5 -minsize 180
		grid columnconfigure $subframe3_variogram_w 6 -minsize 10
		grid $models_nugget_w($ii)        -row 0 -column 2 -sticky e -pady 0
		grid $models_sill_label_w($ii)    -row 0 -column 3 -sticky e -pady 0
		grid $models_sill_value_w($ii)    -row 0 -column 4 -sticky w -pady 0
		grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
		grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	    }
	    if { $focus_variable != 0 } {
		$plus_model_pushbutton_w configure -state normal
		$minus_model_pushbutton_w configure -state normal
		for { set ii 1 } { $ii < 2 } { incr ii } {
		    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
			$model_table_w($ii).combo_type$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_cc$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_ang1$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_ang2$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_ang3$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_aa_hmax$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_aa_hmin$i configure \
			    -state normal -textbackground white -foreground black
			$model_table_w($ii).entry_aa_vert$i configure \
			    -state normal -textbackground white -foreground black
		    }
		    $subframe3_variogram_w.entry_nugget$ii configure -state normal -textbackground white -foreground black
		    $subframe3_variogram_w.label_sill$ii configure -state normal
		    $subframe3_variogram_w.value_sill$ii configure -state normal
		    $subframe3_variogram_w.cut_cat_number_label$ii configure -state normal
		    $subframe3_variogram_w.cut_cat_number_value$ii configure -state normal
		}
	    }	
	}
    }
    
    
    grid columnconfigure $subframe3_variogram_w 0 -minsize 10
    grid columnconfigure $subframe3_variogram_w 1 -minsize 10
    grid $plus_model_pushbutton_w  -row 0 -column 0 -sticky nws -pady 0
    grid $minus_model_pushbutton_w -row 0 -column 1 -sticky nws -pady 0
    
    pack $model_sframe_w -anchor nw -expand 0
    pack propagate $model_sframe_w 0
    grid $model_frame_w -row 0 -column 0 -sticky news -rowspan 4 -pady 0
    
    
    pack \
	$subframe1_w \
	$subframe2_w \
	$spaceframe_w \
	$subframe3_variogram_w \
	$subframe4_w \
	-anchor w -expand 1 -padx 3 -pady 3
    
    
    pack \
	$variogram_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x
    
    
    pack $work_frame_w \
	-anchor nw -side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    ### bei cancel und wiedereinlesen des SISIM parameters windows soll sauber gepackt werden
    if { $focus_variable != 0 } {
	if { $sisimpar(directik) == {*.dat} && $sisimpar(datafl) != {*.dat} } {
	    
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
		pack forget $model_hsbar_w($ii)
		pack forget $model_table_w($ii)
		grid forget $models_nugget_w($ii)
		grid forget $models_sill_label_w($ii)
		grid forget $models_sill_value_w($ii)
		grid forget $cut_cat_number_label_w($ii)
		grid forget $cut_cat_number_value_w($ii)
	    }
	    set ii $focus_variable
	    
	    pack configure $model_hsbar_w($ii) -side right -anchor w -fill y
	    pack configure $model_table_w($ii) -side left -anchor w -padx 2 -expand 1 -fill both
	    
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		$model_type($i,combo$ii) configure -state normal -textbackground white -foreground black
		$model_cc($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_ang1($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_ang2($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_ang3($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_aa_hmax($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_aa_hmin($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_aa_vert($i,entry$ii) configure -state normal -textbackground white -foreground black
	    }
	    
	    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	    grid $models_nugget_w($ii)     -row 0 -column 2 -sticky e -pady 0
	    grid $models_sill_label_w($ii) -row 0 -column 3 -sticky e -pady 0
	    grid $models_sill_value_w($ii) -row 0 -column 4 -sticky w -pady 0
	    grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	    grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	    
	    $plus_cut_cat_pushbutton_w configure -state normal
	    $minus_cut_cat_pushbutton_w configure -state normal
	    $plus_model_pushbutton_w configure -state normal
	    $minus_model_pushbutton_w configure -state normal
	    
	    $models_nugget_w($ii) configure -state normal -textbackground white -foreground black
	    $models_sill_label_w($ii) configure -state normal
	    $models_sill_value_w($ii) configure -state normal
	    $cut_cat_number_label_w($ii) configure -state normal
	    $cut_cat_number_value_w($ii) configure -state normal
	    
	    
	    ### focus_variable is highlighted in the icoli_table and the pcf_cdf_table
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables_initial) - 4] } { incr ii } {
		$pdf_cdf_vals($ii,label) configure -background gray50 -foreground white
		$pdf_cdf_table_w window configure $ii,0 -relief flat -background gray50
	    }
	    $pdf_cdf_vals($focus_variable,label) configure -background yellow -foreground black
	    $pdf_cdf_table_w window configure $focus_variable,0 -relief flat -background yellow
	    
	} elseif { ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) == {*.dat}) || \
		       ($sisimpar(directik) != {*.dat} && $sisimpar(datafl) != {*.dat}) } {
	    
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3] } { incr ii } {
		pack forget $model_hsbar_w($ii)
		pack forget $model_table_w($ii)
		grid forget $models_nugget_w($ii)
		grid forget $models_sill_label_w($ii)
		grid forget $models_sill_value_w($ii)
		grid forget $cut_cat_number_label_w($ii)
		grid forget $cut_cat_number_value_w($ii)
	    }
	    set ii $focus_variable
	    
	    pack configure $model_hsbar_w($ii) -side right -anchor w -fill y
	    pack configure $model_table_w($ii) -side left -anchor w -padx 2 -expand 1 -fill both
	    
	    for { set i 1 } { $i <= [expr $no($ii) - 0] } { incr i } {
		$model_type($i,combo$ii) configure -state normal -textbackground white -foreground black
		$model_cc($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_ang1($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_ang2($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_ang3($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_aa_hmax($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_aa_hmin($i,entry$ii) configure -state normal -textbackground white -foreground black
		$model_aa_vert($i,entry$ii) configure -state normal -textbackground white -foreground black
	    }
	    
	    grid columnconfigure $subframe3_variogram_w 2 -minsize 121
	    grid columnconfigure $subframe3_variogram_w 3 -minsize 47
	    grid columnconfigure $subframe3_variogram_w 4 -minsize 60
	    grid columnconfigure $subframe3_variogram_w 5 -minsize 180
	    grid columnconfigure $subframe3_variogram_w 6 -minsize 10
	    grid $models_nugget_w($ii)      -row 0 -column 2 -sticky e -pady 0
	    grid $models_sill_label_w($ii)  -row 0 -column 3 -sticky e -pady 0
	    grid $models_sill_value_w($ii)  -row 0 -column 4 -sticky w -pady 0
	    grid $cut_cat_number_label_w($ii) -row 0 -column 5 -sticky e -pady 0
	    grid $cut_cat_number_value_w($ii) -row 0 -column 6 -sticky w -pady 0
	    
	    $plus_model_pushbutton_w configure -state normal
	    $minus_model_pushbutton_w configure -state normal
	    
	    $models_nugget_w($ii) configure -state normal -textbackground white -foreground black
	    $models_sill_label_w($ii) configure -state normal
	    $models_sill_value_w($ii) configure -state normal
	    $cut_cat_number_label_w($ii) configure -state normal
	    $cut_cat_number_value_w($ii) configure -state normal
	    
	    
	    ### focus_variable is highlighted in the icoli_table and the pcf_cdf_table
	    for { set ii 1 } { $ii <= [expr $sisimpar(variables) - 3] } { incr ii } {
		$icoli_vals($ii,label) configure -background gray50 -foreground white
		$icoli_table_w window configure $ii,0 -relief flat -background gray50
		$pdf_cdf_vals($ii,label) configure -background gray50 -foreground white
		$pdf_cdf_table_w window configure $ii,0 -relief flat -background gray50
	    }
	    $icoli_vals($focus_variable,label) configure -background yellow -foreground black
	    $icoli_table_w window configure $focus_variable,0 -relief flat -background yellow
	    $pdf_cdf_vals($focus_variable,label) configure -background yellow -foreground black
	    $pdf_cdf_table_w window configure $focus_variable,0 -relief flat -background yellow
	}
    }


    #
    # page4: Geossav SISIM-parameters: Extrapolation / Grid Definition
    #
    set page [$sisim_box_w.sisim_tbn add \
		  -label "Extrapolation / Grid"]
    
    
    # initialize dimension names for tabulated values
    if { ![sisim_read_tabulated_values $sisimpar(tabfl)] } {
	set dim_name_tabulated {}
	lappend dim_name_tabulated "None"
	for { set i 1} { $i < 11 } { incr i } {
	    lappend dim_name_tabulated $i
	}
    }
    
    
    set work_frame_w [frame $page.work_frame \
			  -borderwidth 2 \
			  -relief flat]
    
    
    set extrapolation_iframe_w [iwidgets::labeledframe $work_frame_w.extrapolation \
				    -labeltext "Extrapolation" \
				    -labelpos nw \
				    -borderwidth 2 \
				    -relief groove]
    set extrapolation_frame_w [$extrapolation_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set subframe1_w [frame $extrapolation_frame_w.subfr1]
    
    set min_attribute_value_w [iwidgets::entryfield $subframe1_w.min_attribute_value \
				   -labeltext "Min data value " \
				   -width 5 \
				   -state disabled \
				   -textbackground gray85 \
				   -foreground gray50 \
				   -textvariable sisimpar(zmin) \
				   -validate {check_number_z %W %P %c}]
    
    
    # initialize variable for the extrapolation in the lower tail
    set extrapol_lower_tail_w [iwidgets::combobox $subframe1_w.extrapol_lower_tail_option \
				   -labeltext "Lower tail " \
				   -listheight 64 \
				   -width 27 \
				   -textvariable sisimpar(ltail) \
				   -selectioncommand {tabulated_values_file_activator} \
				   -validate {regexpvalidate %c}]
    set sel 0   ;# that's the default selection if sisimpar(ltail) is empty
    $subframe1_w.extrapol_lower_tail_option insert list end \
	"Linear Interpolation" "Power Model Interpolation" "Tabulated Quantiles Linear Interpolation"
    if { $sisimpar(ltail) == "Linear Interpolation" } {
	set sel 0
    } elseif { $sisimpar(ltail) == "Power Model Interpolation" } {
	set sel 1
    } elseif { $sisimpar(ltail) == "Tabulated Quantiles Linear Interpolation" } {
	set sel 2
    }
    $extrapol_lower_tail_w selection set $sel
    set $extrapol_lower_tail_w [lindex $subframe1_w.extrapol_lower_tail_option $sel]
    $extrapol_lower_tail_w configure -state disabled -textbackground gray85 -foreground gray50
    
    
    set extrapol_lower_tail_parameter_w [iwidgets::entryfield $subframe1_w.extrapol_lower_tail_parameter \
					     -labeltext "Parameter " \
					     -width 5 \
					     -state disabled \
					     -textbackground gray85 \
					     -foreground gray50 \
					     -textvariable sisimpar(ltpar) \
					     -validate {check_number_par %W %P %c}]
    
    
    grid columnconfigure $subframe1_w 0 -minsize 123
    grid columnconfigure $subframe1_w 1 -minsize 250
    grid columnconfigure $subframe1_w 2 -minsize 103
    
    grid $min_attribute_value_w            -row 0 -column 0 -sticky e -pady 0
    grid $extrapol_lower_tail_w            -row 0 -column 1 -sticky e -pady 0
    grid $extrapol_lower_tail_parameter_w  -row 0 -column 2 -sticky e -pady 0
    
  
    set subframe2_w [frame $extrapolation_frame_w.subfr2]
    
    
    # initialize variable for the extrapolation in the middle
    set interpol_middle_w [iwidgets::combobox $subframe2_w.extrapol_middle \
			       -labeltext "Middle " \
			       -listheight 64 \
			       -width 27 \
			       -textvariable sisimpar(middle) \
			       -selectioncommand {tabulated_values_file_activator} \
			       -validate {regexpvalidate %c}]
    set sel 0   ;# that's the default selection if sisimpar(middle) is empty
    $subframe2_w.extrapol_middle insert list end \
	"Linear Interpolation" "Power Model Interpolation" "Tabulated Quantiles Linear Interpolation"
    if { $sisimpar(middle) == "Linear Interpolation" } {
	set sel 0
    } elseif { $sisimpar(middle) == "Power Model Interpolation" } {
	set sel 1
    } elseif { $sisimpar(middle) == "Tabulated Quantiles Linear Interpolation" } {
	set sel 2
    }
    $interpol_middle_w selection set $sel
    set $interpol_middle_w [lindex $subframe2_w.extrapol_middle $sel]
    $interpol_middle_w configure -state disabled -textbackground gray85 -foreground gray50
    
    
    set interpol_middle_parameter_w [iwidgets::entryfield $subframe2_w.interpol_middle \
					 -labeltext "Parameter " \
					 -width 5 \
					 -state disabled \
					 -textbackground gray85 \
					 -foreground gray50 \
					 -textvariable sisimpar(midpar) \
					 -validate {check_number_par %W %P %c}]
    
    
    grid columnconfigure $subframe2_w 0 -minsize 373
    grid columnconfigure $subframe2_w 1 -minsize 103
    
    grid $interpol_middle_w           -row 0 -column 0 -sticky e -pady 0
    grid $interpol_middle_parameter_w -row 0 -column 1 -sticky e -pady 0
    
    
    set subframe3_w [frame $extrapolation_frame_w.subfr3]
    
    set max_attribute_value_w [iwidgets::entryfield $subframe3_w.max_attribute_value \
				   -labeltext "Max data value " \
				   -width 5 \
				   -state disabled \
				   -textbackground gray85 \
				   -foreground gray50 \
				   -textvariable sisimpar(zmax) \
				   -validate {check_number_z %W %P %c}]
    
    
    # initialize variable for the extrapolation in the upper tail
    set extrapol_upper_tail_w [iwidgets::combobox $subframe3_w.extrapol_upper_tail_option \
				   -labeltext "Upper tail " \
				   -listheight 78 \
				   -width 27 \
				   -textvariable sisimpar(utail) \
				   -selectioncommand {tabulated_values_file_activator} \
				   -validate {regexpvalidate %c}]
    set sel 0   ;# that's the default selection if sisimpar(utail) is empty
    $subframe3_w.extrapol_upper_tail_option insert list end "Linear Interpolation" "Power Model Interpolation" \
	"Tabulated Quantiles Linear Interpolation" "Hyperbolic Model Interpolation"
    if { $sisimpar(mik) == "Linear Interpolation" } {
	set sel 0
    } elseif { $sisimpar(utail) == "Power Model Interpolation" } {
	set sel 1
    } elseif { $sisimpar(utail) == "Tabulated Quantiles Linear Interpolation" } {
	set sel 2
    } elseif { $sisimpar(utail) == "Hyperbolic Model Interpolation" } {
	set sel 3
    }
    $extrapol_upper_tail_w selection set $sel
    set $extrapol_upper_tail_w [lindex $subframe3_w.extrapol_upper_tail_option $sel]
    $extrapol_upper_tail_w configure -state disabled -textbackground gray85 -foreground gray50
    
    
    set extrapol_upper_tail_parameter_w [iwidgets::entryfield $subframe3_w.extrapol_upper_tail_parameter \
					     -labeltext "Parameter " \
					     -width 5 \
					     -state disabled \
					     -textbackground gray85 \
					     -foreground gray50 \
					     -textvariable sisimpar(utpar) \
					     -validate {check_number_par %W %P %c}]
    
    ### setting the the extrapolation parameters disabled/normal depending on the selected variabel type
    if { $sisimpar(vartype) == "Categorical" } {
	$min_attribute_value_w           configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_lower_tail_w           configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_lower_tail_parameter_w configure -state disabled -textbackground gray85 -foreground gray50
	$interpol_middle_w               configure -state disabled -textbackground gray85 -foreground gray50
	$interpol_middle_parameter_w     configure -state disabled -textbackground gray85 -foreground gray50
	$max_attribute_value_w           configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_upper_tail_w           configure -state disabled -textbackground gray85 -foreground gray50
	$extrapol_upper_tail_parameter_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $sisimpar(vartype) == "Continous" } {
	$min_attribute_value_w           configure -state normal -textbackground white -foreground black
	$extrapol_lower_tail_w           configure -state normal -textbackground white -foreground black
	$extrapol_lower_tail_parameter_w configure -state normal -textbackground white -foreground black
	$interpol_middle_w               configure -state normal -textbackground white -foreground black
	$interpol_middle_parameter_w     configure -state normal -textbackground white -foreground black
	$max_attribute_value_w           configure -state normal -textbackground white -foreground black
	$extrapol_upper_tail_w           configure -state normal -textbackground white -foreground black
	$extrapol_upper_tail_parameter_w configure -state normal -textbackground white -foreground black
    }
    
    
    grid columnconfigure $subframe3_w 0 -minsize 123
    grid columnconfigure $subframe3_w 1 -minsize 250
    grid columnconfigure $subframe3_w 2 -minsize 103
    
    grid $max_attribute_value_w            -row 0 -column 0 -sticky e -pady 0
    grid $extrapol_upper_tail_w            -row 0 -column 1 -sticky e -pady 0
    grid $extrapol_upper_tail_parameter_w  -row 0 -column 2 -sticky e -pady 0
    
    
    set subframe4_w [frame $extrapolation_frame_w.subfr4]
    set tabfl_w [iwidgets::entryfield $subframe4_w.tabfl_entry \
		     -labeltext "Tabulated values, file " \
		     -width 53 \
		     -state disabled \
		     -textbackground gray85 \
		     -foreground gray50 \
		     -textvariable sisimpar(tabfl)]
    set tabfl_browse_w [iwidgets::pushbutton $subframe4_w.tabfl_browse \
			    -height 27 \
			    -width 40 \
			    -defaultringpad 0 \
			    -state disabled \
			    -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			    -command {
				set tmp_fname [tk_getOpenFile \
						   -defaultextension ".dat" \
						   -filetypes {{{Geo-EAS Files} {.dat}} {{All Files} {.*}}} \
						   -initialfile $sisimpar(tabfl) \
						   -title "Select tabulated data"]
				if { $tmp_fname != {} } {
				    set sisimpar(tabfl) $tmp_fname
				    unset dim_name_tabulated
				    if { [sisim_read_tabulated_values $sisimpar(tabfl)] != 1 } {
					set dim_name_tabulated {}
					lappend dim_name_tabulated "None"
					for { set i 0 } { $i < 10 } { incr i } {
					    lappend dim_name_tabulated $i
					}
				    }
				    
				    $icolvrt_w clear list
				    foreach i $dim_name_tabulated {
					$icolvrt_w insert list end $i
				    }
				    if { $sisimpar(tabulated_parameters) >= 5 } {
					$icolvrt_w selection set 4
					set tmp_icolvrt [lindex $dim_name_tabulated 4]
				    } elseif { $sisimpar(tabulated_parameters) < 5 } {
					$icolvrt_w selection set 1
					set tmp_icolvrt [lindex $dim_name_tabulated 1]
				    }
				    
				    $icolwtt_w clear list
				    foreach i $dim_name_tabulated {
					$icolwtt_w insert list end $i
				    }
				    if { $sisimpar(tabulated_parameters) >= 5 } {
					$icolwtt_w selection set 5
					set tmp_icolwtt [lindex $dim_name_tabulated 5]
				    } elseif { $sisimpar(tabulated_parameters) < 5 } {
					$icolwtt_w selection set 1
					set tmp_icolwtt [lindex $dim_name_tabulated 1]
				    }

				    ### set tabfl widget normal
				    $tabfl_w configure -state normal -textbackground white -foreground black
				    
				    ### counter for the exit from the SISIM parameters tabnotebook
				    set sisimpar(tabulated_cancel_counter) 0
				}
			    }]
    
    grid columnconfigure $subframe4_w 0 -minsize 437
    grid columnconfigure $subframe4_w 1 -minsize 25
    
    grid $tabfl_w        -row 0 -column 0 -sticky e -pady 0
    grid $tabfl_browse_w -row 0 -column 1 -sticky e -pady 0
    
    
    set subframe5_w [frame $extrapolation_frame_w.subfr5]
    
    if { $sisimpar(tabfl) != {*.dat} } {
	set sisimpar(tabulated_cancel_counter) [expr $sisimpar(tabulated_cancel_counter) + 1]
    }
    
    set icolvrt_w [iwidgets::combobox $subframe5_w.icolvrt_option \
		       -labeltext "Variable " \
		       -textvariable sisimpar(icolvrt) \
		       -listheight 75 \
		       -width 21 \
		       -validate {regexpvalidate %c}]
    if { $sisimpar(tabfl) == {*.dat} } {
	foreach i $dim_name_tabulated {
	    $icolvrt_w insert list end $i
	}
	$icolvrt_w selection set 4
	set sisimpar(icolvrt) [lindex $dim_name_tabulated 4]
    } elseif { $sisimpar(tabfl) != {*.dat} } {
	set tmp_icolvrt $sisimpar(icolvrt)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if sisimpar(icolvrt) is empty
	foreach i $dim_name_tabulated {
	    $icolvrt_w insert list end $i
	    $icolvrt_w delete list [expr $sisimpar(tabulated_parameters) + 1]
	    if { $i == $tmp_icolvrt } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($sisimpar(tabulated_cancel_counter) * ($sisimpar(tabulated_parameters) + 1))]
	$icolvrt_w selection set $sel
	set sisimpar(icolvrt) [lindex $dim_name_tabulated $sel]
    }
    $icolvrt_w configure -state disabled -textbackground gray85 -foreground gray50
    
    
    set icolwtt_w [iwidgets::combobox $subframe5_w.icolwtt_option \
		       -labeltext "Weight " \
		       -textvariable sisimpar(icolwtt) \
		       -listheight 75 \
		       -width 21 \
		       -validate {regexpvalidate %c}]
    if { $sisimpar(tabfl) == {*.dat} } {
	foreach i $dim_name_tabulated {
	    $icolwtt_w insert list end $i
	}
	$icolwtt_w selection set 5
	set sisimpar(icolwtt) [lindex $dim_name_tabulated 5]
    } elseif { $sisimpar(tabfl) != {*.dat} } {
	set tmp_icolwtt $sisimpar(icolwtt)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if sisimpar(icolwtt) is empty
	foreach i $dim_name_tabulated {
	    $icolwtt_w insert list end $i
	    $icolwtt_w delete list [expr $sisimpar(tabulated_parameters) + 1]
	    if { $i == $tmp_icolwtt } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($sisimpar(tabulated_cancel_counter) * ($sisimpar(tabulated_parameters) + 1))]
	$icolwtt_w selection set $sel
	set sisimpar(icolwtt) [lindex $dim_name_tabulated $sel]
    }
    $icolwtt_w configure -state disabled -textbackground gray85 -foreground gray50
    
    ### setting the tabulated values disabled/normal depending on the selected extrapolation
    if { ($sisimpar(ltail) == "Tabulated Quantiles Linear Interpolation" || \
	      $sisimpar(middle) == "Tabulated Quantiles Linear Interpolation" || \
	      $sisimpar(utail) == "Tabulated Quantiles Linear Interpolation") } {
	$tabfl_w        configure -state normal -textbackground white -foreground black
	$tabfl_browse_w configure -state normal 
	$icolvrt_w      configure -state normal -textbackground white -foreground black
	$icolwtt_w      configure -state normal -textbackground white -foreground black
    } elseif { ($sisimpar(ltail) != "Tabulated Quantiles Linear Interpolation" || \
		    $sisimpar(middle) != "Tabulated Quantiles Linear Interpolation" || \
		    $sisimpar(utail) != "Tabulated Quantiles Linear Interpolation") } {
	$tabfl_w        configure -state disabled -textbackground gray85 -foreground gray50
	$tabfl_browse_w configure -state disabled
	$icolvrt_w      configure -state disabled -textbackground gray85 -foreground gray50
	$icolwtt_w      configure -state disabled -textbackground gray85 -foreground gray50
    }
    
    
    grid columnconfigure $subframe5_w 0 -minsize 263
    grid columnconfigure $subframe5_w 1 -minsize 212
    
    grid $icolvrt_w  -row 0 -column 0 -sticky e -pady 0
    grid $icolwtt_w  -row 0 -column 1 -sticky e -pady 0
    
    
    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	$subframe4_w \
	$subframe5_w \
	-anchor w -expand 1 -padx 3 -pady 3
    
    
    set grid_definition_iframe_w [iwidgets::labeledframe $work_frame_w.grid_definition \
				      -labeltext "Grid Definition" \
				      -labelpos nw \
				      -borderwidth 2 \
				      -relief groove]
    set grid_definition_frame_w [$grid_definition_iframe_w childsite]; # here we have to get the actual label widget
    
    
    # set grid variables
    set grid(xmax) [expr (($sisimpar(nx)-1)*$sisimpar(xsiz))+$sisimpar(xmn)] \
	;# x coordinate at the center of the last grid cell (x axis)
    set grid(xtot) [expr $sisimpar(nx)*$sisimpar(xsiz)]   ;# total length (x axis)
    set grid(ymax) [expr (($sisimpar(ny)-1)*$sisimpar(ysiz))+$sisimpar(ymn)] \
	;# y coordinate at the center of the last grid cell (y axis)
    set grid(ytot) [expr $sisimpar(ny)*$sisimpar(ysiz)]   ;# total length (y axis)
    set grid(zmax) [expr (($sisimpar(nz)-1)*$sisimpar(zsiz))+$sisimpar(zmn)] \
	;# z coordinate at the center of the last grid axis (z axis)
    set grid(ztot) [expr $sisimpar(nz)*$sisimpar(zsiz)]   ;# total length (z axis)
    set grid(cell) [expr $sisimpar(nx)*$sisimpar(ny)*$sisimpar(nz)]  ;# total number of grid cells
    set grid(area) [expr $sisimpar(nx)*$sisimpar(xsiz)*$sisimpar(ny)*$sisimpar(ysiz)] \
	;# total area of the grid (x-y plane)
    set grid(volume) [expr $sisimpar(nx)*$sisimpar(xsiz)*$sisimpar(ny)*$sisimpar(ysiz) \
			  *$sisimpar(nz)*$sisimpar(zsiz)] ;# total volume of the grid
    
    set subframe1_w [frame $grid_definition_frame_w.subfr1]
    set grid_definition_number_label_w [iwidgets::labeledwidget $subframe1_w.grid_definition_number_label \
					    -labeltext "     Number"]
    
    set grid_definition_size_label_w [iwidgets::labeledwidget $subframe1_w.grid_definition_size_label \
					  -labeltext "Size"]
    
    set grid_definition_minimum_label_w [iwidgets::labeledwidget $subframe1_w.grid_definition_minimum_label \
					     -labeltext "Minimum"]
    
    set grid_definition_maximum_label_w [iwidgets::labeledwidget $subframe1_w.grid_definition_maximum_label \
					     -labeltext "Maximum"]
    
    set grid_definition_total_label_w [iwidgets::labeledwidget $subframe1_w.grid_definition_total_label \
					   -labeltext "Total"]
    
    grid columnconfigure $subframe1_w 0 -minsize 91
    grid columnconfigure $subframe1_w 1 -minsize 76
    grid columnconfigure $subframe1_w 2 -minsize 77
    grid columnconfigure $subframe1_w 3 -minsize 110
    grid columnconfigure $subframe1_w 4 -minsize 110
    
    grid $grid_definition_number_label_w  -row 0 -column 0 -sticky nws -pady 0
    grid $grid_definition_size_label_w    -row 0 -column 1 -sticky nws -pady 0
    grid $grid_definition_minimum_label_w -row 0 -column 2 -sticky nws -pady 0
    grid $grid_definition_maximum_label_w -row 0 -column 3 -sticky e -pady 0
    grid $grid_definition_total_label_w   -row 0 -column 4 -sticky e -pady 0
    
    
    set subframe2_w [frame $grid_definition_frame_w.subfr2]
    
    set grid_definition_number_x_w [iwidgets::entryfield $subframe2_w.grid_definition_number_x \
					-labeltext "x " \
					-textvariable sisimpar(nx) \
					-width 10 \
					-validate {check_number_n %W %P %c}]
    
    
    set grid_definition_size_x_w [iwidgets::entryfield $subframe2_w.grid_definition_size_x \
				      -textvariable sisimpar(xsiz) \
				      -width 10 \
				      -validate {check_number_siz %W %P %c}]
    
    
    set grid_definition_minimum_x_w [iwidgets::entryfield $subframe2_w.grid_definition_minimum_x \
					 -textvariable sisimpar(xmn) \
					 -width 10 \
					 -validate {check_number_xmn %W %P %c}]
    
    
    set grid_definition_maximum_x_label_w [iwidgets::labeledwidget $subframe2_w.grid_definition_maximum_x_label \
					       -labelvariable grid(xmax)]
    
    set grid_definition_total_x_label_w [iwidgets::labeledwidget $subframe2_w.grid_definition_total_x_label \
					     -labelvariable grid(xtot)]
    
    grid columnconfigure $subframe2_w 0 -minsize 93
    grid columnconfigure $subframe2_w 1 -minsize 76
    grid columnconfigure $subframe2_w 2 -minsize 75
    grid columnconfigure $subframe2_w 3 -minsize 110
    grid columnconfigure $subframe2_w 4 -minsize 110
    
    grid $grid_definition_number_x_w        -row 0 -column 0 -sticky nws -pady 0
    grid $grid_definition_size_x_w          -row 0 -column 1 -sticky nws -pady 0
    grid $grid_definition_minimum_x_w       -row 0 -column 2 -sticky nws -pady 0
    grid $grid_definition_maximum_x_label_w -row 0 -column 3 -sticky e -pady 0
    grid $grid_definition_total_x_label_w   -row 0 -column 4 -sticky e -pady 0
    
    bind [$grid_definition_number_x_w component entry]  <KeyRelease> {grid_definition}
    bind [$grid_definition_minimum_x_w component entry] <KeyRelease> {grid_definition}
    bind [$grid_definition_size_x_w component entry]    <KeyRelease> {grid_definition}
    
    
    
    set subframe3_w [frame $grid_definition_frame_w.subfr3]
    
    set grid_definition_number_y_w [iwidgets::entryfield $subframe3_w.grid_definition_number_y \
					-labeltext "y " \
					-textvariable sisimpar(ny) \
					-width 10 \
					-validate {check_number_n %W %P %c}]
    
    
    set grid_definition_size_y_w [iwidgets::entryfield $subframe3_w.grid_definition_size_y \
				      -textvariable sisimpar(ysiz) \
				      -width 10 \
				      -validate {check_number_siz %W %P %c}]
    
    
    set grid_definition_minimum_y_w [iwidgets::entryfield $subframe3_w.grid_definition_minimum_y \
					 -textvariable sisimpar(ymn) \
					 -width 10 \
					 -validate {check_number_ymn %W %P %c}]
    
    
    set grid_definition_maximum_y_label_w [iwidgets::labeledwidget $subframe3_w.grid_definition_maximum_y_label \
					       -labelvariable grid(ymax)]
    
    set grid_definition_total_y_label_w [iwidgets::labeledwidget $subframe3_w.grid_definition_total_y_label \
					     -labelvariable grid(ytot)]
    
    grid columnconfigure $subframe3_w 0 -minsize 93
    grid columnconfigure $subframe3_w 1 -minsize 76
    grid columnconfigure $subframe3_w 2 -minsize 75
    grid columnconfigure $subframe3_w 3 -minsize 110
    grid columnconfigure $subframe3_w 4 -minsize 110
    
    grid $grid_definition_number_y_w        -row 0 -column 0 -sticky nws -pady 0
    grid $grid_definition_size_y_w          -row 0 -column 1 -sticky nws -pady 0
    grid $grid_definition_minimum_y_w       -row 0 -column 2 -sticky nws -pady 0
    grid $grid_definition_maximum_y_label_w -row 0 -column 3 -sticky e -pady 0
    grid $grid_definition_total_y_label_w   -row 0 -column 4 -sticky e -pady 0
    
    bind [$grid_definition_number_y_w component entry]  <KeyRelease> {grid_definition}
    bind [$grid_definition_minimum_y_w component entry] <KeyRelease> {grid_definition}
    bind [$grid_definition_size_y_w component entry]    <KeyRelease> {grid_definition}
    
    
    set subframe4_w [frame $grid_definition_frame_w.subfr4]
    
    
    set grid_definition_number_z_w [iwidgets::entryfield $subframe4_w.grid_definition_number_z \
					-labeltext "z " \
					-textvariable sisimpar(nz) \
					-width 10 \
					-validate {check_number_n %W %P %c}] 
    
    set grid_definition_size_z_w [iwidgets::entryfield $subframe4_w.grid_definition_size_z \
				      -textvariable sisimpar(zsiz) \
				      -width 10 \
				      -validate {check_number_siz %W %P %c}]
      
    set grid_definition_minimum_z_w [iwidgets::entryfield $subframe4_w.grid_definition_minimum_z \
					 -textvariable sisimpar(zmn) \
					 -width 10 \
					 -validate {check_number_zmn %W %P %c}]   
    
    set grid_definition_maximum_z_label_w [iwidgets::labeledwidget $subframe4_w.grid_definition_maximum_z_label \
					       -labelvariable grid(zmax)]
    
    set grid_definition_total_z_label_w [iwidgets::labeledwidget $subframe4_w.grid_definition_total_z_label \
					     -labelvariable grid(ztot)]
    
    grid columnconfigure $subframe4_w 0 -minsize 93
    grid columnconfigure $subframe4_w 1 -minsize 76
    grid columnconfigure $subframe4_w 2 -minsize 75
    grid columnconfigure $subframe4_w 3 -minsize 110
    grid columnconfigure $subframe4_w 4 -minsize 110
    
    grid $grid_definition_number_z_w        -row 0 -column 0 -sticky nws -pady 0
    grid $grid_definition_size_z_w          -row 0 -column 1 -sticky nws -pady 0
    grid $grid_definition_minimum_z_w       -row 0 -column 2 -sticky nws -pady 0
    grid $grid_definition_maximum_z_label_w -row 0 -column 3 -sticky e -pady 0
    grid $grid_definition_total_z_label_w   -row 0 -column 4 -sticky e -pady 0
    
    bind [$grid_definition_number_z_w component entry]  <KeyRelease> {grid_definition}
    bind [$grid_definition_minimum_z_w component entry] <KeyRelease> {grid_definition}
    bind [$grid_definition_size_z_w component entry]    <KeyRelease> {grid_definition}
    
    
    set subframe5_w [frame $grid_definition_frame_w.subfr5]
    
    set grid_cell_number_label_w [iwidgets::labeledwidget $subframe5_w.grid_cell_number_label \
				      -labeltext "    Grid cells " \
				      -labelfont "Helvetica 8 bold"]
    
    set grid_cell_number_w [iwidgets::labeledwidget $subframe5_w.grid_cell_number \
				-labelvariable grid(cell)]
    
    
    set area_label_w [iwidgets::labeledwidget $subframe5_w.area_label \
			  -labeltext "Area " \
			  -labelfont "Helvetica 8 bold"]
    
    set area_w [iwidgets::labeledwidget $subframe5_w.area \
		    -labelvariable grid(area)]
    
    
    set volume_label_w [iwidgets::labeledwidget $subframe5_w.volume_label \
			    -labeltext "Volume " \
			    -labelfont "Helvetica 8 bold"]
    
    set volume_w [iwidgets::labeledwidget $subframe5_w.volume \
		      -labelvariable grid(volume)]
    
    
    grid columnconfigure $subframe5_w 0 -minsize 76
    grid columnconfigure $subframe5_w 1 -minsize 100
    grid columnconfigure $subframe5_w 2 -minsize 36
    grid columnconfigure $subframe5_w 3 -minsize 100
    grid columnconfigure $subframe5_w 4 -minsize 53
    grid columnconfigure $subframe5_w 5 -minsize 100

    grid $grid_cell_number_label_w -row 0 -column 0 -sticky nws -pady 0
    grid $grid_cell_number_w       -row 0 -column 1 -sticky nws -pady 0
    grid $area_label_w             -row 0 -column 2 -sticky nws -pady 0
    grid $area_w                   -row 0 -column 3 -sticky nws -pady 0
    grid $volume_label_w           -row 0 -column 4 -sticky nws -pady 0
    grid $volume_w                 -row 0 -column 5 -sticky nws -pady 0
    
    
    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	$subframe4_w \
	$subframe5_w \
	-anchor w -expand 1 -padx 3 -pady 3

    
    pack \
	$extrapolation_iframe_w \
	$grid_definition_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x
    
    
    pack $work_frame_w \
	-anchor nw -side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    $sisim_box_w.sisim_tbn view "Files"
    
    
    #############################################################################
    # Default buttons
    #############################################################################
    # write_sisim_params fname [tmpflag]
    #----------------------------------------------------------------
    # This proc writes the parameter file for running sisim
    #
    # fname   : [<-] : name of parameter file
    # tmpflag : [<-] : optional flag (0,1). If this flag is set, the
    #                  temporary data files are created in order to
    #                  circumvent a problem with the sisim program
    #                  which forbids long filenames. The following
    #                  temporary files are created in the working
    #                  directory:
    #                    - "sisimh_in.dat" (hard data input)
    #                    - "sisims_in.dat" (soft data input)
    #                    - "sisim_in.tab" (tabular data)
    #                    - "sisim_out.dbg" (debugging file)
    #                    - "sisim_out.dat" (result of sisim run)
    #
    #############################################################################
       
    set button_frame_w [frame $sisim_box_w.button_frame]
    
    set buttons_iframe_w [iwidgets::labeledframe $button_frame_w.buttons_iframe \
			      -labeltext "Defaults" \
			      -labelpos nw \
			      -borderwidth 2 \
			      -relief groove]
    set buttons_frame_w [$buttons_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set open_button_w [iwidgets::pushbutton $buttons_iframe_w.open \
			   -height 27 \
			   -width 40 \
			   -defaultringpad 0 \
			   -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			   -command {
			       set tmp_fname [tk_getOpenFile \
						  -initialdir [file dirname $sisimpar(datafl)] \
						  -defaultextension ".par" \
						  -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
						  -title "Select SISIM parameter file"]
			       
			       if { $tmp_fname != {} } {
				   set param_fname $tmp_fname
				   read_sisim_params $param_fname
			       }
			   }]
    
    
    set save_button_w [iwidgets::pushbutton $buttons_iframe_w.save \
			   -height 27 \
			   -width 27 \
			   -defaultringpad 0 \
			   -state disabled \
			   -image [image create photo -file [file join "$exedir/images/save.gif"]] \
			   -command {
			       set sisimpar(title) 0
			       write_sisim_params $param_fname
			   }]
    
    set save_as_button_w [iwidgets::pushbutton $buttons_iframe_w.save_as \
			      -height 27 \
			      -width 27 \
			      -defaultringpad 0 \
			      -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			      -command {
				  if { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) != {*.dat} } {
				      set tmp_fname [tk_getSaveFile \
							 -initialdir [file dirname $sisimpar(datafl)] \
							 -defaultextension ".par" \
							 -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
							 -title "Save SISIM parameter file as ..."]
				      
				      $save_button_w configure -state normal
				      
				      if { $tmp_fname != {} } {
					  set sisimpar(title) 0
					  set param_fname $tmp_fname
					  write_sisim_params $param_fname
				      }
				  } elseif { $sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat} } {
				      set tmp_fname [tk_getSaveFile \
							 -initialdir [file dirname $sisimpar(datafl)] \
							 -defaultextension ".par" \
							 -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
							 -title "Save SISIM parameter file as ..."]
				      
				      $save_button_w configure -state normal
				      
				      if { $tmp_fname != {} } {
					  set sisimpar(title) 0
					  set param_fname $tmp_fname
					  write_sisim_params $param_fname
				      }
				  } elseif { $sisimpar(datafl) == {*.dat} && $sisimpar(directik) != {*.dat} } {
				      set tmp_fname [tk_getSaveFile \
							 -initialdir [file dirname $sisimpar(directik)] \
							 -defaultextension ".par" \
							 -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
							 -title "Save SISIM parameter file as ..."]
				      
				      $save_button_w configure -state normal
				      
				      if { $tmp_fname != {} } {
					  set sisimpar(title) 0
					  set param_fname $tmp_fname
					  write_sisim_params $param_fname
				      }
				  } elseif { $sisimpar(datafl) == {*.dat} && $sisimpar(directik) == {*.dat} } {
				      tk_messageBox \
					  -title "Defaults" \
					  -icon warning \
					  -message "You should select an input file first." \
					  -type ok
				  }
			      }]
    
    
    set cancel_button_w [iwidgets::pushbutton $buttons_iframe_w.cancel \
			     -height 27 \
			     -width 27 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/stop.gif"]] \
			     -command {
				 set private(button) 0
			     }]
    
    
    set run_button_w [iwidgets::pushbutton $buttons_iframe_w.run \
			  -height 27 \
			  -width 27 \
			  -defaultringpad 0 \
			  -image [image create photo -file [file join "$exedir/images/run.gif"]] \
			  -command {
			      if { $sisimpar(datafl) == {*.dat} && $sisimpar(directik) == {*.dat} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "You should select an input file first." \
				      -type ok
			      } elseif { ($sisimpar(datafl) != {*.dat} && $sisimpar(directik) != {*.dat}) || \
					 ($sisimpar(datafl) != {*.dat} && $sisimpar(directik) == {*.dat}) || \
					 ($sisimpar(datafl) == {*.dat} && $sisimpar(directik) != {*.dat}) } {

				  if { $tcl_platform(platform) == "unix" } {
				      set command "../gslib/bin/sisim"
				  } elseif { $tcl_platform(platform) == "windows" } {
				      set command "C:/Programme/Statios/Gslib77/bin/sisim.exe"
				  }
				  
				  set sisimpar(title) 1
				  set paramfile "tmprun.par"
				  write_sisim_params $paramfile 1
				  
				  set outf [open "SISIM.TMP" {w}]
				  puts $outf "$paramfile"
				  close $outf
				  set logfile [open "| $command < SISIM.TMP" {r}]
				  fconfigure $logfile -buffering none
				  while { ![eof $logfile] } {
				      gets $logfile input_line
				      log $input_line
				      update
				  }
				  close $logfile
				  
				  ### copy the temporary files
				  catch {
				      file copy -force "sisim_out.dat" $sisimpar(outfl)
				      file copy -force "sisim_out.dbg" $sisimpar(dbgfl)
				  }
				  
				  ### write information file (infofl) and create infofl path- and filename
				  set infofl_rootname [file rootname $sisimpar(outfl)]
				  set infofl_dirname [file dirname $sisimpar(outfl)]
				  set infofl_dirname_lenght [string length $infofl_dirname]
				  set infofl_name [string range $infofl_rootname \
						       [expr $infofl_dirname_lenght + 1] end]
				  set infofl_extension [file extension $sisimpar(outfl)]
				  set info "/Info_"
				  regsub {} $infofl_extension $infofl_name infofl_1
				  regsub {} $infofl_1 $info infofl_2
				  regsub {} $infofl_2 $infofl_dirname infofl_3
				  set sisimpar(infofl) $infofl_3

				  ### message, if file cannot be written
				  if { [catch {set outf [open $sisimpar(infofl) {w}]} err_msg] } {
				      tk_messageBox \
					  -title "File open error" \
					  -icon error \
					  -message "The file $fname couldn't open!\n$err_msg"
				  }
				  write_sisim_infofl
				  close $outf

				  ### delete the temporary files
				  catch {
				      file delete -force \
					  "sisimh_in.dat" \
					  "sisims_in.dat" \
					  "sisim_in.tab" \
					  "sisim_out.dbg" \
					  "sisim_out.dat" \
					  "tmprun.par" \
					  "SISIM.TMP"
				  }
				  set private(button) 1
			      }
			  }]
    
    bind $run_button_w <Return> "
	    $run_button_w configure -state active -relief sunken
	    update idletasks
	    after 100
	    $run_button_w invoke
	"
    
    grid columnconfigure $button_frame_w 0 -minsize 15
    grid columnconfigure $button_frame_w 1 -minsize 100
    grid columnconfigure $button_frame_w 2 -minsize 15
    grid columnconfigure $button_frame_w 3 -minsize 100
    grid columnconfigure $button_frame_w 4 -minsize 15
    grid columnconfigure $button_frame_w 5 -minsize 100
    grid columnconfigure $button_frame_w 6 -minsize 15
    grid columnconfigure $button_frame_w 7 -minsize 100
    grid columnconfigure $button_frame_w 8 -minsize 15
    grid columnconfigure $button_frame_w 9 -minsize 100
    grid columnconfigure $button_frame_w 10 -minsize 15
    
    grid $open_button_w    -row 0 -column 1 -sticky we -pady 8 -padx 5
    grid $save_button_w    -row 0 -column 3 -sticky we -pady 8 -padx 5
    grid $save_as_button_w -row 0 -column 5 -sticky we -pady 8 -padx 5
    grid $cancel_button_w  -row 0 -column 7 -sticky we -pady 8 -padx 5
    grid $run_button_w     -row 0 -column 9 -sticky we -pady 8 -padx 5
    
    
    pack \
	$buttons_iframe_w \
	-padx 5 -pady 5 -expand 1
    
    
    pack $button_frame_w \
	-side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    raise $sisim_box_w
    
    tkwait visibility $sisim_box_w
    tkwait visibility $run_button_w
    
    set oldFocus [focus]
    set oldGrab [grab current $sisim_box_w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $sisim_box_w
    focus $run_button_w
    
    . configure -cursor arrow
    
    update idletask
    tkwait variable private(button)
    
    catch {focus $oldFocus}
    catch {
	# It's possible that the window has already been destroyed,
	# hence this "catch".  Delete the Destroy handler so that
	# tkPriv(button) doesn't get reset by it.
	
	bind $sisim_box_w <Destroy> {}
	destroy $sisim_box_w
    }
    
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    return
}
#============================================================================
#=END=OF=SISIM===============================================================