##START#OF#EXPORT_MODFLOW####################################################
# geossav_export_modflow.tcl
#
# Source script creating the export_modflow widget in GEOSSAV, and running 
# the export of data. The output files can be used in MODFLOW supported 
# programs (bcf file). The called c/c++ procs are in the geossav.c file.
# Developed at the University of Basel, Switzerland,
# by Christian Regli and Lukas Rosenthaler
#
#############################################################################


#############################################################################
#
# The following set of parameters are required for the EXPORT_MODFLOW
#
global exedir
source $exedir/Geossav_export_modflow_globals.tcl

set mofpar(datafl) "*.out"                     ;# simulated realizations file, simplified Geo-EAS format
set mofpar(nsim) 1                             ;# number of simulations to consider from simulation data
set mofpar(max_nsim) 1                         ;# maximum number of simulations to consider from simulation data
set mofpar(vartype) 0                          ;# variable type: 1=continuous (cdf), 0=categorical (pdf)
set mofpar(nx) 100                             ;# number of grid cells (x axis)
set mofpar(xsiz) 10                            ;# size of grid cell in x-direction
set mofpar(ny) 100                             ;# number of grid cells (y axis)
set mofpar(ysiz) 10                            ;# size of grid cell in y-direction
set mofpar(nz) 10                              ;# number of grid cells (z axis)
set mofpar(zsiz) 10                            ;# size of grid cell in z-direction
set mofpar(charfl) "*.gtd"                     ;# ground type data file, simplified Geo-EAS format
set mofpar(mean) 0                             ;# considering mean values from ground type data, 1=yes, 0=no
set mofpar(cmean) 1                            ;# column number for the mean
set mofpar(dist) 1                             ;# considering distribution values from ground type data, 1=yes, 0=no
set mofpar(cdist) 2                            ;# column number for the standard deviation
set mofpar(nsimchar) 1                         ;# number of simulations to consider from characterization data
set mofpar(icoli) 1                            ;# column number for the characterization variables
set mofpar(infofl) "*.out"                     ;# information file to the realization file, simplified Geo-EAS format
set mofpar(ncat) 3                             ;# number of thresholds or categories
set mofpar(cat) 1                              ;# threshold values or category codes
set mofpar(krel) 1                             ;# reletion of hydraulic conductivity (Kvertical/Khorizontal)
                                               ;# working with categorical variables
set mofpar(relk) 1                             ;# reletion of hydraulic conductivity (Kvertical/Khorizontal)
                                               ;# working with continous variables
set mofpar(dflbc) 0                            ;# counter for the activation of the datafl browse button
set mofpar(inbcffl) "*.dat"                    ;# input bcf file to overwrite
set mofpar(modnx) 1                            ;# number of modflow model cells in x-direction
set mofpar(modny) 1                            ;# number of modflow model cells in y-direction
set mofpar(modnz) 1                            ;# number of modflow model cells in z-direction
set mofpar(ltype) 0                            ;# type of modflow model layer (0,1,2,3) 
set mofpar(outfl) "*.out"                      ;# output data file/s containing the values for the import in modflow
set mofpar(noutfl) 1                           ;# number of created ouput files
set mofpar(sname) "( *.out )"                  ;# output file extension modification and explanation
set mofpar(charvar) 0                          ;# number of characterization variables (ground types)
set mofpar(oldcharvar) 0                       ;# old number of characterization variables (ground types)
                                               ;# if simulated realizations data were fed newly
set mofpar(char_parameters) 0                  ;# number of characterization variables
set mofpar(cancel_counter_characterization) 0  ;# counter for the exit from the EXPORT_MODFLOW parameters tabnotebook
                                               ;# if characterization data were fed
set mofpar(repfl) "*.out"                      ;# file with replaced values

#
# End of initialization of variables. This has to be done once at the start 
# of the programm.
#
#############################################################################


#############################################################################
#
# Start of proc section.
#

#############################################################################
proc export_modflow_read_char_data { fname } {
    global mofpar
    global dim_name_char

    if { [file readable $fname] } {
	set inf [open $fname "r"]
	gets $inf
	gets $inf n
	for { set i 0 } { $i < $n } { incr i } {
	    gets $inf dum
	    lappend dim_name_char $dum
	    set tmp_parameters $i
	}
	if { $mofpar(cancel_counter_characterization) == 0 } {
	    set mofpar(charvar) $tmp_parameters
	} elseif { $mofpar(cancel_counter_characterization) != 0 } {
	    set mofpar(charvar) $mofpar(charvar)
	}
	close $inf
	set mofpar(char_parameters) $tmp_parameters
	return 1
    } else {
	return 0
    }
}
#============================================================================


#############################################################################
proc check_modflow_number_nsim_sim { entry newstr char } {
    global mofpar

    set current [$entry get]
    catch {scan $current {%d} current}

    if { $mofpar(datafl) == {*.out} } {
	if {$current < 100} {
	    return [regexp {^[1-9]*$} $newstr]
	} elseif {$current >= 100 && [string match {[0-9]} $char]} {
	    $entry delete 0 end
	    $entry insert end "1000"
	}
    } elseif { $mofpar(datafl) != {*.out} } {
	if { $newstr < $mofpar(max_nsim) } {
	    return [regexp {^[1-9]*$} $newstr]
	} elseif { $newstr >= $mofpar(max_nsim) } {
	    $entry delete 0 end  
	    $entry insert end "$mofpar(max_nsim)"
	}
    }
}
#============================================================================


#############################################################################
proc check_modflow_number_nsim_char { entry newstr char } {
    global mofpar

    set current [$entry get]
    catch {scan $current {%d} current}

    if {$current < 10} {
	return [regexp {^[1-9]*$} $newstr]
    } elseif {$current >= 10 && [string match {[0-9]} $char]} {
	$entry delete 0 end
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc check_modflow_cell_number_x { entry newstr char } {
    global mofpar

    set current [$entry get]
    catch {scan $current {%d} current}

    if { $mofpar(datafl) == {*.out} } {
	if {$current < 1000} {
	    return [regexp {^[1-9]*$} $newstr]
	} elseif {$current >= 1000 && [string match {[0-9]} $char]} {
	    $entry delete 0 end
	    $entry insert end "10000"
	}
    } elseif { $mofpar(datafl) != {*.out} } {
	if { $newstr < $mofpar(nx) } {
	    return [regexp {^[1-9]*$} $newstr]
	} elseif { $newstr >= $mofpar(nx) } {
	    $entry delete 0 end  
	    $entry insert end "$mofpar(nx)"
	}
    }
}
#============================================================================


#############################################################################
proc check_modflow_cell_number_y { entry newstr char } {
    global mofpar

    set current [$entry get]
    catch {scan $current {%d} current}

    if { $mofpar(datafl) == {*.out} } {
	if {$current < 1000} {
	    return [regexp {^[1-9]*$} $newstr]
	} elseif {$current >= 1000 && [string match {[0-9]} $char]} {
	    $entry delete 0 end
	    $entry insert end "10000"
	}
    } elseif { $mofpar(datafl) != {*.out} } {
	if { $newstr < $mofpar(ny) } {
	    return [regexp {^[1-9]*$} $newstr]
	} elseif { $newstr >= $mofpar(ny) } {
	    $entry delete 0 end  
	    $entry insert end "$mofpar(ny)"
	}
    }
}
#============================================================================


#############################################################################
proc check_modflow_cell_number_z { entry newstr char } {
    global mofpar

    set current [$entry get]
    catch {scan $current {%d} current}

    if { $mofpar(datafl) == {*.out} } {
	if {$current < 1000} {
	    return [regexp {^[1-9]*$} $newstr]
	} elseif {$current >= 1000 && [string match {[0-9]} $char]} {
	    $entry delete 0 end
	    $entry insert end "10000"
	}
    } elseif { $mofpar(datafl) != {*.out} } {
	if { $newstr < $mofpar(nz) } {
	    return [regexp {^[1-9]*$} $newstr]
	} elseif { $newstr >= $mofpar(nz) } {
	    $entry delete 0 end  
	    $entry insert end "$mofpar(nz)"
	}
    }
}
#============================================================================


#############################################################################
proc number_modflow_output_files { } {
    global mofpar
    
    catch {scan $mofpar(nsim) {%d} mofpar(nsim)}
    catch {scan $mofpar(nz) {%d} mofpar(nz)}
    catch {scan $mofpar(nsimchar) {%d} mofpar(nsimchar)}

    
    if { $mofpar(vartype) == "Categorical" } {
	if { $mofpar(mean) == 0 } {
	    if { $mofpar(nsim) == 0 || $mofpar(nsimchar) == 0 || $mofpar(nsim) == {} || $mofpar(nsimchar) == {} } {
		set mofpar(noutfl) "?"
		set mofpar(sname) "?"
	    } else {
		if { $mofpar(datafl) == {*.out} } {
		    set mofpar(noutfl) [expr $mofpar(nsim) * $mofpar(nsimchar)]
		    set mofpar(sname) "( *.out )"
		} elseif { $mofpar(datafl) != {*.out} && $mofpar(charfl) != {*.gtd} } {
		    set mofpar(noutfl) [expr $mofpar(nsim) * $mofpar(nsimchar)]
		    set mofpar(sname) "( *.out_s1-$mofpar(nsim),_c1-$mofpar(nsimchar) )   'simulation, characterization'"
		} elseif { $mofpar(datafl) != {*.out} && $mofpar(charfl) == {*.gtd} } {
		    set mofpar(noutfl) $mofpar(nsim)
		    set mofpar(sname) "( *.out_s1-$mofpar(nsim) )   'simulation'"
		}
	    }
	} elseif { $mofpar(mean) == 1 } {
	    if { $mofpar(nsim) == 0 || $mofpar(nsimchar) == 0 || $mofpar(nsim) == {} || $mofpar(nsimchar) == {} } {
		set mofpar(noutfl) "?"
		set mofpar(sname) "?"
	    } else {
		if { $mofpar(datafl) == {*.out} } {
		    set mofpar(noutfl) $mofpar(nsim)
		    set mofpar(sname) "( *.out )"
		} elseif { $mofpar(datafl) != {*.out} && $mofpar(charfl) != {*.gtd} } {
		    set mofpar(noutfl) $mofpar(nsim)
		    set mofpar(sname) "( *.out_s1-$mofpar(nsim),_c1 )   'simulation, characterization'"
		} elseif { $mofpar(datafl) != {*.out} && $mofpar(charfl) == {*.gtd} } {
		    set mofpar(noutfl) $mofpar(nsim)
		    set mofpar(sname) "( *.out_s1-$mofpar(nsim) )   'simulation'"
		}
	    }
	}
    } elseif { $mofpar(vartype) == "Continous" } {
	if { $mofpar(nsim) == 0 || $mofpar(nsimchar) == 0 || $mofpar(nsim) == {} || $mofpar(nsimchar) == {} } {
	    set mofpar(noutfl) "?"
	    set mofpar(sname) "?"
	} else {
	    if { $mofpar(datafl) == {*.out} } {
		set mofpar(noutfl) $mofpar(nsim)
		set mofpar(sname) "( *.out )"
	    } elseif { $mofpar(datafl) != {*.out} } {
		set mofpar(noutfl) $mofpar(nsim)
		set mofpar(sname) "( *.out_s1-$mofpar(nsim) )   'simulation'"
	    }
	}
    }
}
#============================================================================


#############################################################################
proc mean_value_modflow_selection { } {
    global mofpar
    global distribution_value_combo_w
    global realization_char_w
    
    if { $mofpar(mean) == 1 } {
	set mofpar(dist) 0
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
	$realization_char_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $mofpar(mean) == 0 } {
	set mofpar(dist) 1
	$distribution_value_combo_w configure -state normal -textbackground white -foreground black
	$realization_char_w configure -state normal -textbackground white -foreground black
    }
    number_modflow_output_files
}
#============================================================================


#############################################################################
proc distribution_value_modflow_selection { } {
    global mofpar
    global distribution_value_combo_w
    global realization_char_w

    if { $mofpar(dist) == 1 } {
	set mofpar(mean) 0
	$distribution_value_combo_w configure -state normal -textbackground white -foreground black
	$realization_char_w configure -state normal -textbackground white -foreground black
    } elseif { $mofpar(dist) == 0 } {
	set mofpar(mean) 1
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
	$realization_char_w configure -state disabled -textbackground gray85 -foreground gray50
    }
    number_modflow_output_files
}
#============================================================================


#############################################################################
proc regexpvalidate { char } {
    return [regexp {![ ]} $char]
}
#============================================================================


#############################################################################
proc check_number_modflow_krel_values { entry newstr char } {
    if { [regexp {^[+]?[0-9]*[\.]?[0-9]*([e][+-]?[0-9]*)?$} $newstr] } {
	catch {scan $newstr {%f} newstr}
	if { $newstr == "+" || $newstr == "+." || $newstr == "." } {
	    return 1
	}
	return [expr ($newstr >= 0.0) && ($newstr <= 1.0) ]
    } else {
	return 0
    }
}
#============================================================================


#############################################################################
proc read_modflow_infofl_params { fname } {
    global mofpar
    global charfl_w
    global charfl_browse_w
    global realization_char_w
    global mean_value_w
    global distribution_value_w
    global mean_value_combo_w
    global distribution_value_combo_w
    global datafl_w
    global char_vals
    global char_vals_k
    global char_table_w
    global dim_name_char
    global tmp_charvar
    global relation_k_w
    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	set mofpar(datafl) "*.out"
	$datafl_w configure -textvariable mofpar(datafl)
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\nNo such file or directory."
	return 0
    }
    
    ### sets the cursor on watch as long as parameters are read
    . configure -cursor watch
    update
    
    ### searches the line "information for sisim data export" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Information for SISIM data export" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File content error" \
		-icon error \
		-message "Information to the simulated realizations file can not be found.\n"
	    return 0
	}
    }

    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }

    ### reads the vartype: variable type (1=cont.(cdf), 0=cat.(pdf))
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_vartype [format "%d" [string range $line 0 $space]]

    ### reads the nsim: number of realizations
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(nsim) [format "%d" [string range $line 0 $space]]
    set mofpar(max_nsim) $mofpar(nsim)

    ### reads the nx, xsiz: number, size of grid cells (x direction)
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(nx) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set mofpar(xsiz) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the ny, ysiz: number, size of grid cells (y direction)
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(ny) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set mofpar(ysiz) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the nz, zsiz: number, size of grid cells (z direction)
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(nz) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set mofpar(zsiz) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### destroys the content of the characterization table
    if { $mofpar(dflbc) == 1 } {
	for { set ii 1 } { $ii <= 3 } { incr ii } {
	    destroy $char_vals($ii,label)
	    destroy $char_vals($ii,entry)
	    destroy $char_vals($ii,combo)
	    destroy $char_vals_k($ii,entry)
	}
    } elseif { $mofpar(dflbc) != 1 } {
	for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	    destroy $char_vals($ii,label)
	    destroy $char_vals($ii,entry)
	    destroy $char_vals($ii,combo)
	    destroy $char_vals_k($ii,entry)
	}
    }

    ### reads the ncat: number of thresholds / categories
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(ncat) [format "%d" [string range $line 0 $space]]

    ### setting up the rows of the char table
    $char_table_w configure -rows [expr $mofpar(ncat) + 1]

    ### reads the cat: thresholds / categories and fills part of the char table
    $char_table_w configure -rows [expr $mofpar(ncat) + 1]
    set line [gets $inf]
    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	
	set char_vals($ii,label) [iwidgets::labeledwidget $char_table_w.label$ii \
				      -labeltext "$ii" \
				      -background gray50 \
				      -foreground white]
	update

	set space [string first " " $line]
	set mofpar(cat,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set char_vals($ii,entry) [iwidgets::entryfield $char_table_w.entry$ii \
				      -textvariable mofpar(cat,$ii) \
				      -validate {regexpvalidate %c} \
				      -width 10]
	update
	

	if { $mofpar(charfl) == {*.gtd} } {
 	    set mofpar(icoli,$ii) $ii
	} elseif { $mofpar(charfl) != {*.gtd} } {
	    if { $mofpar(oldcharvar) < [expr $mofpar(ncat) + 1] } {
		set mofpar(icoli,$ii) 1
	    } elseif { $mofpar(oldcharvar) >= [expr $mofpar(ncat) + 1] } {
		set mofpar(icoli,$ii) $mofpar(icoli,$ii)
	    }
	}
	set char_vals($ii,combo) [iwidgets::combobox $char_table_w.combo$ii \
				      -textvariable mofpar(icoli,$ii) \
				      -validate {regexpvalidate %c} \
				      -width 22 \
				      -listheight 100]
	$char_vals($ii,combo) clear list
	foreach i $dim_name_char {
	    $char_vals($ii,combo) insert list end $i
	}
	if { $mofpar(charfl) == {*.gtd} } {
	    $char_vals($ii,combo) selection set 0
	    set mofpar(icoli,$ii) [lindex $dim_name_char 0]
	} elseif { $mofpar(charfl) != {*.gtd} } {
	    if { $tmp_vartype == 0 } {
		if { $mofpar(charvar) >= [expr $mofpar(ncat) + 1] } {
		    $char_vals($ii,combo) selection set [expr $ii - 1]
		    set mofpar(icoli,$ii) [lindex $dim_name_char [expr $ii - 1]]
		} elseif { $mofpar(charvar) < [expr $mofpar(ncat) + 1] } {
		    $char_vals($ii,combo) selection set [expr $ii - 1]
		    set mofpar(icoli,$ii) [lindex $dim_name_char [expr $ii - 1]]
		}
	    } elseif { $tmp_vartype == 1 } {
		$char_vals($ii,combo) selection set 0
		set mofpar(icoli,$ii) [lindex $dim_name_char 0]	    
	    }
	}
	$char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	update

	set mofpar(krel,$ii) 1
	set char_vals_k($ii,entry) [iwidgets::entryfield $char_table_w.entry_k$ii \
					-textvariable mofpar(krel,$ii) \
					-validate {check_number_modflow_krel_values %W %P %c} \
					-width 10]
	update

	if { $mofpar(charfl) == {*.gtd} } {
	    $char_vals($ii,entry)   configure -state disabled -textbackground gray85 -foreground gray50
	    $char_vals($ii,combo)   configure -state disabled -textbackground gray85 -foreground gray50
	    $char_vals_k($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	} elseif { $mofpar(charfl) != {*.gtd} } {
	    $char_vals($ii,entry)   configure -state normal -textbackground white -foreground black
	    $char_vals($ii,combo)   configure -state normal -textbackground white -foreground black
	    $char_vals_k($ii,entry) configure -state normal -textbackground white -foreground black
	}
	
	$char_table_w window configure $ii,0 -window $char_vals($ii,label) -relief flat -background gray50
	$char_table_w window configure $ii,1 -window $char_vals($ii,entry)
	$char_table_w window configure $ii,2 -window $char_vals($ii,combo)
	$char_table_w window configure $ii,3 -window $char_vals_k($ii,entry)
    }
    
    ### control the selection of the mean value and distribution value comboboxes depending on the content of the char data
    if { $mofpar(charfl) != {*.gtd} } { 
	### mean value combobox
	if { $mofpar(charvar) >= [expr $mofpar(ncat) + 1] } {
	    set tmp_cmean $mofpar(cmean)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if sisimpar(icolsz) is empty
	    foreach i $dim_name_char {
		$mean_value_combo_w insert list end $i
		$mean_value_combo_w delete list [expr $mofpar(charvar) + 1]
		if { $i == $tmp_cmean } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($mofpar(cancel_counter_characterization) * ($mofpar(charvar) + 1))]
	    $mean_value_combo_w selection set $sel
	    set mofpar(cmean) [lindex $dim_name_char $sel]

	    ### distribution value combobox
	    set tmp_cdist $mofpar(cdist)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if sisimpar(icolsz) is empty
	    foreach i $dim_name_char {
		$distribution_value_combo_w insert list end $i
		$distribution_value_combo_w delete list [expr $mofpar(charvar) + 1]
		if { $i == $tmp_cdist } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($mofpar(cancel_counter_characterization) * ($mofpar(charvar) + 1))]
	    $distribution_value_combo_w selection set $sel
	    set mofpar(cdist) [lindex $dim_name_char $sel]
	} elseif { $mofpar(charvar) < [expr $mofpar(ncat) + 1] } {
	    ### mean value combobox
	    $mean_value_combo_w selection set 0
	    set mofpar(cmean) [lindex $dim_name_char 0]
	    
	    ### distribution value combobox
	    $distribution_value_combo_w selection set 0
	    set mofpar(cdist) [lindex $dim_name_char 0]
	}
    }

    ### setting the characterization input widgets normal/disabled depending on the vartype
    if { $tmp_vartype == 1 } {
	$charfl_w        configure -state disabled -textbackground gray85 -foreground gray50
	$charfl_browse_w configure -state disabled
	$relation_k_w    configure -state normal -textbackground white -foreground black
	if { $mofpar(charfl) == {*.gtd} } {
	    for { set ii 1 } { $ii <= 3 } { incr ii } {
		$char_vals($ii,entry)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals_k($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	} elseif { $mofpar(charfl) != {*.gtd} } {
	    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
		$char_vals($ii,entry)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals_k($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	}
	$mean_value_w               configure -state disabled
	$distribution_value_w       configure -state disabled
	$mean_value_combo_w         configure -state disabled -textbackground gray85 -foreground gray50
	$realization_char_w         configure -state disabled -textbackground gray85 -foreground gray50
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50

    } elseif { $tmp_vartype == 0 } {
	$relation_k_w configure -state disabled -textbackground gray85 -foreground gray50
	if { $mofpar(charfl) == {*.gtd} } {
	    $charfl_w        configure -state disabled -textbackground gray85 -foreground gray50
	    $charfl_browse_w configure -state normal
	    for { set ii 1 } { $ii <= 3 } { incr ii } {
		$char_vals($ii,entry)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals_k($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	    $mean_value_w               configure -state disabled
	    $distribution_value_w       configure -state disabled
	    $mean_value_combo_w         configure -state disabled -textbackground gray85 -foreground gray50
	    $realization_char_w         configure -state disabled -textbackground gray85 -foreground gray50
	    $distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
	} elseif { $mofpar(charfl) != {*.gtd} } {
	    $charfl_w             configure -state normal -textbackground white -foreground black
	    $charfl_browse_w      configure -state normal
	    $mean_value_w         configure -state normal
	    $distribution_value_w configure -state normal
	    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
		$char_vals($ii,entry)   configure -state normal -textbackground white -foreground black
	    	$char_vals($ii,combo)   configure -state normal -textbackground white -foreground black
		$char_vals_k($ii,entry) configure -state normal -textbackground white -foreground black
	    }
	    if { $mofpar(mean) == 1 } {
		$mean_value_combo_w         configure -state normal -textbackground white -foreground black
		$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
		$realization_char_w         configure -state disabled -textbackground gray85 -foreground gray50
	    } elseif { $mofpar(dist) == 1 } {
		$mean_value_combo_w         configure -state normal -textbackground white -foreground black
		$distribution_value_combo_w configure -state normal -textbackground white -foreground black
		$realization_char_w         configure -state normal -textbackground white -foreground black
	    }
	}
    }

    if { $tmp_vartype == 0 } {
	set mofpar(vartype) "Categorical"
    } elseif { $tmp_vartype == 1 } {
    	set mofpar(vartype) "Continous"
    }

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc read_export_modflow_params { fname } {
    global mofpar
    global dim_name_char
    global export_modflow_box_w
    global datafl_w
    global realization_sim_w
    global char_table_w
    global char_vals
    global char_vals_k
    global charfl_w
    global charfl_browse_w
    global mean_value_w
    global distribution_value_w
    global mean_value_combo_w
    global realization_char_w
    global distribution_value_combo_w
    global bcf_inputfl_w
    global relation_k_w
    global modflow_model_nx_w
    global modflow_model_ny_w
    global modflow_model_nz_w
    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }
    
    ### sets the cursor on watch as long as parameters are read
    . configure -cursor watch
    update
    
    ### searches the line "parameters for export_modflow" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Parameters for EXPORT_MODFLOW" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File selection error" \
		-icon error \
		-message "You have selected a wrong parameter file.\n"
	    return 0
	}
    }

    ### set file name in export_modflow box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $export_modflow_box_w "EXPORT MODFLOW parameters   -   $fname_short"

    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }

    ### destroys the content of the ivar-table, the vario_dir_table and the vario_type_table 
    if { $mofpar(datafl) == {*.out} } {
	for { set ii 1 } { $ii <= 3 } { incr ii } {
	    destroy $char_vals($ii,label)
	    destroy $char_vals($ii,entry)
	    destroy $char_vals($ii,combo)
	    destroy $char_vals_k($ii,entry)
	}
    } elseif { $mofpar(datafl) != {*.out} } {
	for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	    destroy $char_vals($ii,label)
	    destroy $char_vals($ii,entry)
	    destroy $char_vals($ii,combo)
	    destroy $char_vals_k($ii,entry)
	}
    }    
    
    ### reads name of datafl: file with simulated realizations
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set mofpar(datafl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set mofpar(datafl) [format "%s" $line]
    }
    $datafl_w configure -state normal -textbackground white -foreground black

    ### reads the nsim: number of realizations to consider
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(nsim) [format "%d" [string range $line 0 $space]]
    set mofpar(max_nsim) $mofpar(nsim)
    $realization_sim_w configure -state normal -textbackground white -foreground black

    ### reads the vartype: variable type (1=cont.(cdf), 0=cat.(pdf))
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_vartype [format "%d" [string range $line 0 $space]]
    if { $tmp_vartype == 1 } {
	set mofpar(vartype) "Continous"
    } elseif { $tmp_vartype == 0 } {
	set mofpar(vartype) "Categorical"
    }

    ### reads the relk: relation of hydraulic conductivity Kvertical/Khorizontal (continous variables)
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(relk) [format "%g" [string range $line 0 $space]]
    if { $mofpar(vartype) == "Categorical" } {
	$relation_k_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $mofpar(vartype) == "Continous" } {
	$relation_k_w configure -state normal -textbackground white -foreground black
    }

    ### reads the nx, ny, nz: number of grid cells (x-, y-, and z-axis)
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(nx) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set mofpar(ny) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set mofpar(nz) [format "%d" [string range $line 0 [expr $space - 1]]]

    ### reads the xsiz, ysiz, zsiz: size of grid cells (x-, y-, and z-axis)
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(xsiz) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set mofpar(ysiz) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set mofpar(zsiz) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads name of charfl: file with characterization data
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set mofpar(charfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set mofpar(calibfl) $line
    }

    ### reads the ncat: number of thresholds or categories
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(ncat) [format "%d" [string range $line 0 $space]]

    ### reads the cat(i): thresholds or categories
    set line [gets $inf]
    set catline $line
    
    ### reads the icoli(i): column number for the characterization variables
    set line [gets $inf]
    set icoliline $line

    ### reads the krel(i): Kvertical/Khorizontal
    set line [gets $inf]
    set krelline $line

    ### fills the char-table
    $char_table_w configure -rows [expr $mofpar(ncat) + 1]

    if { $mofpar(datafl) != {*.out} } {
	unset dim_name_char
	if { [export_modflow_read_char_data $mofpar(charfl)] != 1 } {
	    set dim_name_char {}
	    for { set i 1 } { $i <= $mofpar(charvar) } { incr i } {
		lappend dim_name_data $i
	    }
	}
    }

    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	
	set char_vals($ii,label) [iwidgets::labeledwidget $char_table_w.label$ii \
				      -labeltext "$ii" \
				      -background gray50 \
				      -foreground white]
	update
	
	set space [string first " " $catline]
	set mofpar(cat,$ii) [format "%g" [string range $catline 0 [expr $space - 1]]]
	set catline [string range $catline [expr $space + 1] end]
	set char_vals($ii,entry) [iwidgets::entryfield $char_table_w.entry$ii \
				      -textvariable mofpar(cat,$ii) \
				      -validate {regexpvalidate %c} \
				      -state normal \
				      -width 10 \
				      -textbackground white \
				      -foreground black]
	update

	set char_vals($ii,combo) [iwidgets::combobox $char_table_w.combo$ii \
				      -textvariable mofpar(icoli,$ii) \
				      -validate {regexpvalidate %c} \
				      -width 22 \
				      -listheight 100]
	$char_vals($ii,combo) clear list
	foreach i $dim_name_char {
	    $char_vals($ii,combo) insert list end $i
	}
	set space [string first " " $icoliline]
	set mofpar(icoli,$ii) [format "%d" [string range $icoliline 0 [expr $space - 1]]]
	set icoliline [string range $icoliline [expr $space + 1] end]
	$char_vals($ii,combo) selection set $mofpar(icoli,$ii)
	update

	set space [string first " " $krelline]
	set mofpar(krel,$ii) [format "%g" [string range $krelline 0 [expr $space - 1]]]
	set krelline [string range $krelline [expr $space + 1] end]
	set char_vals_k($ii,entry) [iwidgets::entryfield $char_table_w.entry_k$ii \
					-textvariable mofpar(krel,$ii) \
					-validate {check_number_modflow_krel_values %W %P %c} \
					-state normal \
					-width 10 \
					-textbackground white \
					-foreground black]
	update
	
	$char_table_w window configure $ii,0 -window $char_vals($ii,label) -relief flat -background gray50
	$char_table_w window configure $ii,1 -window $char_vals($ii,entry)
	$char_table_w window configure $ii,2 -window $char_vals($ii,combo)
	$char_table_w window configure $ii,3 -window $char_vals_k($ii,entry)
    }

    ### reads the mean: mean value to consider (0=no, 1=yes)
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(mean) [format "%d" [string range $line 0 $space]]

    ### reads the dist: value from the distribution to consider (0=no, 1=yes)
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(dist) [format "%d" [string range $line 0 $space]]

    ### reads the cmean: column number for the mean
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_mofpar(cmean) [format "%g" [string range $line 0 $space]]

    if { $mofpar(datafl) != {*.out} } {
	unset dim_name_char
	if { [export_modflow_read_char_data $mofpar(charfl)] != 1 } {
	    set dim_name_char {}
	    for { set i 1 } { $i <= $mofpar(charvar) } { incr i } {
		lappend dim_name_data $i
	    }
	}
	$mean_value_combo_w configure -state normal -textbackground white -foreground black
	$mean_value_combo_w clear list
	foreach i $dim_name_char {
	    $mean_value_combo_w insert list end $i
	}
	$mean_value_combo_w selection set $tmp_mofpar(cmean)
    }

    ### reads the cdist: column number for the standard deviation
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_mofpar(cdist) [format "%g" [string range $line 0 $space]]

    if { $mofpar(datafl) != {*.out} } {
	unset dim_name_char
	if { [export_modflow_read_char_data $mofpar(charfl)] != 1 } {
	    set dim_name_char {}
	    for { set i 1 } { $i <= $mofpar(charvar) } { incr i } {
		lappend dim_name_data $i
	    }
	}
	$distribution_value_combo_w configure -state normal -textbackground white -foreground black
	$distribution_value_combo_w clear list
	foreach i $dim_name_char {
	    $distribution_value_combo_w insert list end $i
	}
	$distribution_value_combo_w selection set $tmp_mofpar(cdist)
    }

    ### reads the nsimchar: number of characterization realizations to consider
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(nsimchar) [format "%d" [string range $line 0 $space]]

    ### setting the characterization input widgets normal/disabled depending on the vartype
    if { $tmp_vartype == 1 } {
	$charfl_w        configure -state disabled -textbackground gray85 -foreground gray50
	$charfl_browse_w configure -state disabled
	for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	    $char_vals($ii,entry)   configure -state disabled -textbackground gray85 -foreground gray50
	    $char_vals($ii,combo)   configure -state disabled -textbackground gray85 -foreground gray50
	    $char_vals_k($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	}
	$mean_value_w               configure -state disabled
	$distribution_value_w       configure -state disabled
	$mean_value_combo_w         configure -state disabled -textbackground gray85 -foreground gray50
	$realization_char_w         configure -state disabled -textbackground gray85 -foreground gray50
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50

    } elseif { $tmp_vartype == 0 } {
	$charfl_w             configure -state normal -textbackground white -foreground black
	$charfl_browse_w      configure -state normal
	$mean_value_w         configure -state normal
	$distribution_value_w configure -state normal
	for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	    $char_vals($ii,entry)   configure -state normal -textbackground white -foreground black
	    $char_vals($ii,combo)   configure -state normal -textbackground white -foreground black
	    $char_vals_k($ii,entry) configure -state normal -textbackground white -foreground black
	}
	if { $mofpar(mean) == 1 } {
	    $mean_value_combo_w         configure -state normal -textbackground white -foreground black
	    $distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
	    $realization_char_w         configure -state disabled -textbackground gray85 -foreground gray50
	} elseif { $mofpar(dist) == 1 } {
	    $mean_value_combo_w         configure -state normal -textbackground white -foreground black
	    $distribution_value_combo_w configure -state normal -textbackground white -foreground black
	    $realization_char_w         configure -state normal -textbackground white -foreground black
	}
    }

    ### reads the inbcffl: input bcf file
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set mofpar(inbcffl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set mofpar(inbcffl) $line
    }
    $bcf_inputfl_w configure -state normal -textbackground white -foreground black

    ### reads the modnx, modny, modnz: number of modflow model grid cells (x-, y-, and z-axis)
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(modnx) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set mofpar(modny) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set mofpar(modnz) [format "%d" [string range $line 0 [expr $space - 1]]]
    $modflow_model_nx_w configure -state normal -textbackground white -foreground black
    $modflow_model_ny_w configure -state normal -textbackground white -foreground black
    $modflow_model_nz_w configure -state normal -textbackground white -foreground black

    ### reads the outfl: file for output data
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set mofpar(outfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set mofpar(outfl) $line
    }

    ### reads the noutfl: number of output files
    set line [gets $inf]
    set space [string first " " $line]
    set mofpar(noutfl) [format "%d" [string range $line 0 $space]]

    ### reads the sname: extension of output files
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set mofpar(sname) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set mofpar(sname) $line
    }

    ### reset the cancel counter
    set mofpar(cancel_counter_characterization) 0

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc write_export_modflow_params { fname } {
    global tcl_platform
    global version
    global mofpar
    global export_modflow_box_w
    global variable_type_value_w
    global char_vals
    global char_vals_k
    global mean_value_combo_w
    global distribution_value_combo_w
    global relation_k_w

    ### message, if file cannot be open
    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }

    ### sets the cursor on watch as long as parameters are written
    . configure -cursor watch
    update
    
    ### set file name in bicalib box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $export_modflow_box_w "EXPORT MODFLOW parameters   -   $fname_short"

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    
    puts $outf "Parameters for EXPORT_MODFLOW"
    puts $outf "*****************************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### datafl: file with simulated realizations
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $mofpar(datafl)]
    set line [format "%s" $mofpar(datafl)]
    set line_length_total [string length $line]
    set comment "\\simulated realizations"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr ($parameter_space - 0) - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### nsim: number of realizations to consider
    set comment "\\number of realizations to consider"
    catch {scan [$realization_sim_w get] {%d} mofpar(nsim)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(nsim) $comment]

    ### vartype: variable type (1=cont.(cdf), 0=cat.(pdf)
    set comment "\\variable type (1=cont.(cdf), 0=cat.(pdf)"
    set line [format "%s" $mofpar(vartype)] 
    if { $line == "Categorical" } {
	set mofpar(vartype) 0
	puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(vartype) $comment]
	set mofpar(vartype) "Categorical"
    } elseif { $line == "Continous" } {
	set mofpar(vartype) 1
	puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(vartype) $comment]
	set mofpar(vartype) "Continous"
    }

    ### relk: relation of hydraulic conductivity Kvertical/Khorizontal (continous variables)
    set comment "\\Kvertical/Khorizontal (continous variables)"
    catch {scan [$relation_k_w get] {%g} mofpar(relk)}
    puts $outf [format "%-[expr $parameter_space - 0]g %s" $mofpar(relk) $comment]

    ### nx, ny, nz: number of grid cells (x-, y-, and z-axis)
    puts -nonewline $outf [format "%d %d %d" $mofpar(nx) $mofpar(ny) $mofpar(nz)]
    set line [format "%d %d %d" $mofpar(nx) $mofpar(ny) $mofpar(nz)]
    set line_length [string length $line]
    set comment "\\number of grid cells (x-, y-, and z-axis)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### xsiz, ysiz, zsiz: size of grid cells (x-, y-, and z-axis)
    puts -nonewline $outf [format "%g %g %g" $mofpar(xsiz) $mofpar(ysiz) $mofpar(zsiz)]
    set line [format "%g %g %g" $mofpar(xsiz) $mofpar(ysiz) $mofpar(zsiz)]
    set line_length [string length $line]
    set comment "\\size of grid cells (x-, y-, and z-axis)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### charfl: file with characterization data
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $mofpar(charfl)]
    set line [format "%s" $mofpar(charfl)]
    set line_length_total [string length $line]
    set comment "\\characterization data"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr ($parameter_space - 0) - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### ncat: number of thresholds or categories
    set comment "\\number of thresholds or categories"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(ncat) $comment]

    ### cat(i): threshold values or category codes
    set line_length_total 0
    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	puts -nonewline $outf "[format "%g" $mofpar(cat,$ii)] "
	set line [format "%g" $mofpar(cat,$ii)]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    set comment "\\thresholds or categories"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### icoli(i): column number for the characterization variables
    set line_length_total 0
    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	puts -nonewline $outf "[format "%d" [$char_vals($ii,combo) curselection]] "
	set line [format "%d" [$char_vals($ii,combo) curselection]]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    set comment "\\column number for the characterization variables"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### krel(i): relation of hydraulic conductivity Kvertical/Khorizontal (Kvertical/Khorizontal) 
    set line_length_total 0
    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	puts -nonewline $outf "[format "%g" $mofpar(krel,$ii)] "
	set line [format "%g" $mofpar(krel,$ii)]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    set comment "\\Kvertical/Khorizontal (categorical variables)"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### mean: mean value to consider (0=no, 1=yes)
    set comment "\\mean value to consider (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(mean) $comment]

    ### dist: value from the distribution to consider (0=no, 1=yes)
    set comment "\\value from the distribution to consider (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(dist) $comment]

    ### cmean: column number for the mean
    set comment "\\column number for the mean"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$mean_value_combo_w curselection] $comment]

    ### cdist: column number for the standard deviation
    set comment "\\column number for the standard deviation"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$distribution_value_combo_w curselection] $comment]

    ### nsimchar: number of characterization realizations to consider
    set comment "\\number of characterization realizations to consider"
    catch {scan [$realization_char_w get] {%d} mofpar(nsimchar)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(nsimchar) $comment]

    ### bcfinputfl: input bcf file
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $mofpar(inbcffl)]
    set line [format "%s" $mofpar(inbcffl)]
    set line_length_total [string length $line]
    set comment "\\input bcf file"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### modnx, modny, modnz: number of modflow model grid cells (x-, y-, and z-axis)
    puts -nonewline $outf [format "%d %d %d" $mofpar(modnx) $mofpar(modny) $mofpar(modnz)]
    set line [format "%d %d %d" $mofpar(modnx) $mofpar(modny) $mofpar(modnz)]
    set line_length [string length $line]
    set comment "\\number of modflow model grid cells (x-, y-, and z-axis)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### outfl: file for output data
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $mofpar(outfl)]
    set line [format "%s" $mofpar(outfl)]
    set line_length_total [string length $line]
    set comment "\\output data"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### noutfl: number of output files
    set comment "\\number of output files"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(noutfl) $comment]

    ### sname: extensions of output files
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $mofpar(sname)]
    set line [format "%s" $mofpar(sname)]
    set line_length_total [string length $line]
    set comment "\\extensions of output files"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update

    ### close the file
    close $outf
}
#============================================================================


#############################################################################
proc write_export_modflow_params_run { fname } {
    global tcl_platform
    global version
    global mofpar
    global export_modflow_box_w
    global variable_type_value_w
    global char_vals
    global mean_value_combo_w
    global distribution_value_combo_w

    ### message, if file cannot be open
    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }

    ### set file name in bicalib box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $export_modflow_box_w "EXPORT MODFLOW parameters   -   $fname_short"

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    log "EXPORT MODFLOW Version: $version\n"
    log "Parameters:\n"
    
    puts $outf "Parameters for EXPORT_MODFLOW"
    puts $outf "*************************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### datafl: file with simulated realizations
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $mofpar(datafl)]
    set line [format "%s" $mofpar(datafl)]
    set line_length_total [string length $line]
    set comment "\\simulated realizations"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr ($parameter_space - 0) - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    log "file with simulated realizations = $mofpar(datafl)"

    ### nsim: number of realizations to consider
    set comment "\\number of realizations to consider"
    catch {scan [$realization_sim_w get] {%d} mofpar(nsim)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(nsim) $comment]
    log "number of realizations to consider = $mofpar(nsim)"

    ### vartype: variable type (1=cont.(cdf), 0=cat.(pdf)
    set comment "\\variable type (1=cont.(cdf), 0=cat.(pdf)"
    set line [format "%s" $mofpar(vartype)] 
    if { $line == "Categorical" } {
	set mofpar(vartype) 0
	puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(vartype) $comment]
	set mofpar(vartype) "Categorical"
    } elseif { $line == "Continous" } {
	set mofpar(vartype) 1
	puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(vartype) $comment]
	set mofpar(vartype) "Continous"
    }
    log "variable type (1=continous, 0=categorical) = $mofpar(vartype)"

    ### relk: relation of hydraulic conductivity Kvertical/Khorizontal (continous variables)
    set comment "\\Kvertical/Khorizontal (continous variables)"
    catch {scan [$relation_k_w get] {%g} mofpar(relk)}
    puts $outf [format "%-[expr $parameter_space - 0]g %s" $mofpar(relk) $comment]
    log "Kvertical/Khorizontal (continous variables) = $mofpar(relk)"

    ### nx, ny, nz: number of grid cells (x-, y-, and z-axis)
    puts -nonewline $outf [format "%d %d %d" $mofpar(nx) $mofpar(ny) $mofpar(nz)]
    set line [format "%d %d %d" $mofpar(nx) $mofpar(ny) $mofpar(nz)]
    set line_length [string length $line]
    set comment "\\number of grid cells (x-, y-, and z-axis)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    log "number of grid cells (x-, y-, z-axis) = $mofpar(nx)  $mofpar(ny)  $mofpar(nz)"

    ### xsiz, ysiz, zsiz: size of grid cells (x-, y-, and z-axis)
    puts -nonewline $outf [format "%g %g %g" $mofpar(xsiz) $mofpar(ysiz) $mofpar(zsiz)]
    set line [format "%g %g %g" $mofpar(xsiz) $mofpar(ysiz) $mofpar(zsiz)]
    set line_length [string length $line]
    set comment "\\size of grid cells (x-, y-, and z-axis)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    log "size of grid cells (x-, y-, z-axis) = $mofpar(xsiz)  $mofpar(ysiz)  $mofpar(zsiz)"

    ### charfl: file with characterization data
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $mofpar(charfl)]
    set line [format "%s" $mofpar(charfl)]
    set line_length_total [string length $line]
    set comment "\\characterization data"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr ($parameter_space - 0) - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    log "file with characterization data = $mofpar(charfl)"

    ### ncat: number of thresholds or categories
    set comment "\\number of thresholds or categories"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(ncat) $comment]
    log "number of categories = $mofpar(ncat)"

    ### cat(i): threshold values or category codes
    set line_length_total 0
    log_nonewline "number of categories = "
    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	puts -nonewline $outf "[format "%g" $mofpar(cat,$ii)] "
	set line [format "%g" $mofpar(cat,$ii)]
	log_nonewline "$mofpar(cat,$ii)  "
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    log_newline
    set comment "\\thresholds or categories"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### icoli(i): column number for the characterization variables
    set line_length_total 0
    log_nonewline "column number of the characterization variables = "
    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	puts -nonewline $outf "[format "%d" [$char_vals($ii,combo) curselection]] "
	set line [format "%d" [$char_vals($ii,combo) curselection]]
	log_nonewline "[$char_vals($ii,combo) curselection]  "
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    log_newline
    set comment "\\column number for the characterization variables"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### krel(i): relation of hydraulic conductivity Kvertical/Khorizontal (categorical variables)
    set line_length_total 0
    log_nonewline "Kvertical/Khorizontal (categorical variables) = "
    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	puts -nonewline $outf "[format "%g" $mofpar(krel,$ii)] "
	set line [format "%g" $mofpar(krel,$ii)]
	log_nonewline "$mofpar(krel,$ii)  "
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    log_newline
    set comment "\\Kvertical/Khorizontal (categorical variables)"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### mean: mean value to consider (0=no, 1=yes)
    set comment "\\mean value to consider (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(mean) $comment]
    log "mean value to consider (0=no, 1=yes) = $mofpar(mean)"

    ### dist: value from the distribution to consider (0=no, 1=yes)
    set comment "\\value from the distribution to consider (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(dist) $comment]
    log "value from the distribution to consider (0=no, 1=yes) = $mofpar(dist)"

    ### cmean: column number for the mean
    set comment "\\column number for the mean"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$mean_value_combo_w curselection] $comment]
    log "column number for the mean = [$mean_value_combo_w curselection]"

    ### cdist: column number for the standard deviation
    set comment "\\column number for the standard deviation"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$distribution_value_combo_w curselection] $comment]
    log "column number for the standard deviation = [$distribution_value_combo_w curselection]"

    ### nsimchar: number of characterization realizations to consider
    set comment "\\number of characterization realizations to consider"
    catch {scan [$realization_char_w get] {%d} mofpar(nsimchar)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(nsimchar) $comment]
    log "number of characterization realizations to consider = $mofpar(nsimchar)"

    ### bcfinputfl: input bcf file
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $mofpar(inbcffl)]
    set line [format "%s" $mofpar(inbcffl)]
    set line_length_total [string length $line]
    set comment "\\input bcf file"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    log "file/s for output data = $mofpar(inbcffl)"

    ### modnx, modny, modnz: number of modflow model grid cells (x-, y-, and z-axis)
    puts -nonewline $outf [format "%d %d %d" $mofpar(modnx) $mofpar(modny) $mofpar(modnz)]
    set line [format "%d %d %d" $mofpar(modnx) $mofpar(modny) $mofpar(modnz)]
    set line_length [string length $line]
    set comment "\\number of modflow model grid cells (x-, y-, and z-axis)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    log "number of modflow model grid cells (x-, y-, z-axis) = $mofpar(modnx)  $mofpar(modny)  $mofpar(modnz)"

    ### outfl: file/s for output data
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $mofpar(outfl)]
    set line [format "%s" $mofpar(outfl)]
    set line_length_total [string length $line]
    set comment "\\output data"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    log "file/s for output data = $mofpar(outfl)"

    ### noutfl: number of output file/s
    set comment "\\number of output file/s"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $mofpar(noutfl) $comment]
    log "number of output file/s = $mofpar(noutfl)"

    ### sname: extensions of output file/s
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $mofpar(sname)]
    set line [format "%s" $mofpar(sname)]
    set line_length_total [string length $line]
    set comment "\\extensions of output files"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    log "extensions of output file/s = $mofpar(sname)"

    ### close the file
    close $outf
}
#============================================================================


#############################################################################
proc export_modflow_layer_counter { } {
    global mofpar 
    global bcff
    global nchar
    global nlayer
    global nlines
    global nx
    global ny

    ### analyzes number of layers and layer types, described in the bcf file (laycon's)
    set ii 1
    set line [gets $bcff]                             ;# reads the first line of the bcf file
    set nchar($ii) [gets $bcff line]                  ;# reads the second line of the bcf file  
    set layconline($ii) $line                         ;# (first layconline) and counts the characters

    set space [string first " " $layconline($ii)]                               ;# deletes " " at the beginning of
    set layconline($ii) [string range $layconline($ii) [expr $space + 1] end]   ;# the layconline and insert " " at 
    set layconline($ii) [format "%-[expr $nchar($ii) - 0]s " $layconline($ii)]  ;# the end of the layconline

    for { set i 1 } { $i <= [expr $nchar($ii) / 2] } { incr i } {
	set space [string first " " $layconline($ii)]                                           ;# analyzes the layconline 
	set mofpar(ltype,$i) [format "%d" [string range $layconline($ii) 0 [expr $space - 1]]]  ;# and saves the layer 
	set layconline($ii) [string range $layconline($ii) [expr $space + 1] end]               ;# types in an array
	set nlayer $i
    }

    while { $nchar($ii) <= 80 } {
	incr ii                                                   ;# the same is done for the following
	set nchar($ii) [gets $bcff line]                          ;# layconlines, if there are any
	set layconline($ii) $line

	set space [string first " " $layconline($ii)]
	set layconline($ii) [string range $layconline($ii) [expr $space + 1] end]
	set layconline($ii) [format "%-[expr $nchar($ii) - 0]s " $layconline($ii)]
	
	if { $nchar($ii) <= 80 } {
	    for { set i 1 } { $i <= [expr $nchar($ii) / 2] } { incr i } {
		set space [string first " " $layconline($ii)]
		set iii [expr $i + (40 * ($ii - 1))]
		set mofpar(ltype,$iii) [format "%d" [string range $layconline($ii) 0 [expr $space - 1]]]
		set layconline($ii) [string range $layconline($ii) [expr $space + 1] end]
		set nlayer $iii
	    }
	} 
    }

    ### analyzes number of modflow model cells in x- and y-direction
    set line [gets $bcff]
    while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		[string first "Hydr. Conductivity of layer" $line] == -1 } { ;# rejects lines up to the hyd. cond.
	set line [gets $bcff]                                                ;# or transmissivity information
    }
    set ii 1
    set ny 0
    set nchar($ii) [gets $bcff line]
    while { [string first "Calculated Vertical Leakance of layer" $line] == -1 && \
		[string first "Elevation of the bottom of the layer" $line] == -1 } {  ;# counts characters per line and
	if { $nchar($ii) != 280 } {                                                    ;# number of lines, which describe
	    set ny [expr $ny + 1]                                                      ;# one modflow layer; at the same
	}                                                                              ;# time the number of cells in y- 
	incr ii                                                                        ;# is extracted (ny); this is only
	set nlines $ii                                                                 ;# possible, if the number of cells
	set nchar($ii) [gets $bcff line]                                               ;# in x-direction is not 20 or a 
    }                                                                                  ;# multiple of 20
    if { $ny != 0  } {                                                                 ;# if it is so, the number of cells
	set nx 0                                                                       ;# in x-direction is extracted
	for { set ii 1 } { $ii <= [expr $nlines / $ny] } { incr ii } {
	    set nx [expr ($nchar($ii) / 14) + $nx]
	}
    }
    log "nx = $nx"
    log "ny = $ny"
    log "nz = $nlayer"
}
#============================================================================


#############################################################################
proc export_modflow_vertical_k_continous { } {
    global mofpar
    global inf
    global kvf

    ### searches the line "simulated value" and reject all preceded lines
    set line [gets $inf]
    puts $kvf $line
    while { $line != "Simulated Value" } {
	set line [gets $inf]
	puts $kvf $line
    }
    
    ### calculation of the vertical K value
    while { ![eof $inf] } {
	set line [gets $inf]
	catch {scan $line {%g} kh}
	set kv [expr $kh * $mofpar(relk)]
	puts $kvf $kv
    }
}
#============================================================================


#############################################################################
proc export_modflow_continous_variables { iv } {
    global mofpar 
    global outfcon
    global inf
    global bcff
    global kvf
    global ckvf
    global nchar
    global nlayer
    
    ### searches the lines "simulated value" and reject all preceded lines
    if { $iv == 1 } { ;# condition for the first simulation
	seek $inf 0
	seek $kvf 0
	seek $ckvf 0
	set simline [gets $inf]
	while { $simline != "Simulated Value" } {
	    set simline [gets $inf]
	}
	set kvline [gets $kvf]
	while { $kvline != "Simulated Value" } {
	    set kvline [gets $kvf]
	}
	set ckvline [gets $ckvf]
	while { $ckvline != "Simulated Value" } {
	    set ckvline [gets $ckvf]
	}

	### goes to the second layer of the copied vertical K values file
	for { set c 1 } { $c <= [expr $mofpar(nx) * $mofpar(ny)] } { incr c } {
	    set ckvline [gets $ckvf]
	}

	### writes all lines before the hydraulic conductivity or transmissivity, and leakance information
	seek $bcff 0
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    puts $outfcon($iv) $line
	    set line [gets $bcff]
	}
	seek $bcff 0
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information, 
	### except the information of the last line 
	for { set i 1 } { $i <= [expr $nlayer - 1] } { incr i } {
	    set line [gets $bcff]
	    while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
			[string first "Hydr. Conductivity of layer" $line] == -1 } {
		set line [gets $bcff]
	    }
	    puts $outfcon($iv) $line
	    if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $inf]
			catch {scan $simline {%g} simline}
			set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcon($iv) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcon($iv) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcon($iv) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcon($iv) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvf]
			set ckvline [gets $ckvf]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcon($iv) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcon($iv) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcon($iv) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    } elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $inf]
			catch {scan $simline {%g} simline}
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcon($iv) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcon($iv) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcon($iv) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcon($iv) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvf]
			set ckvline [gets $ckvf]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcon($iv) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcon($iv) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcon($iv) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    }
	}
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information
	### of the last layer
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    set line [gets $bcff]
	}
	puts $outfcon($iv) $line
	if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $inf]
		    catch {scan $simline {%g} simline}
		    set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcon($iv) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcon($iv) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcon($iv) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcon($iv) $line
	    }
	} elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $inf]
		    catch {scan $simline {%g} simline}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcon($iv) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcon($iv) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcon($iv) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcon($iv) $line
	    }
	}
    } elseif { $iv >= 1 } { ;# condition for the following simulations

	### writes all lines before the hydraulic conductivity or transmissivity, and leakance information
	seek $bcff 0
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    puts $outfcon($iv) $line
	    set line [gets $bcff]
	}
	seek $bcff 0

	### goes to the first layer of the next simulation (vertical K values file)
	for { set c 1 } { $c <= [expr $mofpar(nx) * $mofpar(ny)] } { incr c } {
	    set kvline [gets $kvf]
	}

	### goes to the second layer of the next simulation (copied vertical K values file)
	for { set c 1 } { $c <= [expr $mofpar(nx) * $mofpar(ny)] } { incr c } {
	    set ckvline [gets $ckvf]
	}
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information, 
	### except the information of the last line 
	for { set i 1 } { $i <= [expr $nlayer - 1] } { incr i } {
	    set line [gets $bcff]
	    while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
			[string first "Hydr. Conductivity of layer" $line] == -1 } {
		set line [gets $bcff]
	    }
	    puts $outfcon($iv) $line
	    if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $inf]
			catch {scan $simline {%g} simline}
			set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcon($iv) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcon($iv) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcon($iv) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcon($iv) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvf]
			set ckvline [gets $ckvf]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcon($iv) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcon($iv) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcon($iv) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    } elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $inf]
			catch {scan $simline {%g} simline}
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcon($iv) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcon($iv) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcon($iv) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcon($iv) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvf]
			set ckvline [gets $ckvf]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcon($iv) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcon($iv) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcon($iv) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    }
	}
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information
	### of the last layer
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    set line [gets $bcff]
	}
	puts $outfcon($iv) $line
	if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $inf]
		    catch {scan $simline {%g} simline}
		    set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcon($iv) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcon($iv) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcon($iv) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcon($iv) $line
	    }
	} elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $inf]
		    catch {scan $simline {%g} simline}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcon($iv) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcon($iv) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcon($iv) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcon($iv) $line
	    }
	}
    }
}
#============================================================================


#############################################################################
proc mean_value_modflow_replacement { } {
    global mofpar 
    global outf
    global inf
    global charf
    global repf
    global kvf
    global mean_value_combo_w
    global char_vals
    global mean_position
    global variable_position
    global mean_value
    global ln_mean_value

    ### searches the mean value position in the characterization data
    set mean_position [$mean_value_combo_w curselection]

    ### reading the mean values of the variables
    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	set variable_position($ii) [$char_vals($ii,combo) curselection]
	for { set i 0 } { $i <= [expr + $mofpar(charvar) + $variable_position($ii) + 3] } { incr i } {
	    set charline [gets $charf]
	}
	for { set iii 0 } { $iii <= $mean_position } { incr iii } {
	    set space [string first " " $charline]
	    set ln_mean_value($ii) [format "%g" [string range $charline 0 [expr $space - 1]]] 
	    ;# it is the ln-mean of the field data
	    set charline [string range $charline [expr $space + 1] end]
	}
	set mean_value($ii) [expr (0 + exp($ln_mean_value($ii)))] ;# the ln-value has to be expound to the base of e
	seek $charf 0
    }

    ### searches the line "simulated value" and reject all preceded lines
    set line [gets $inf]
    puts $repf $line
    puts $kvf $line
    while { $line != "Simulated Value" } {
	set line [gets $inf]
	puts $repf $line
	puts $kvf $line
    }
    
    ### replace the categorical variables with mean values
    while { ![eof $inf] } {
	set line [gets $inf]
	catch {scan $line {%d} line}
	for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	    if { $line == $mofpar(cat,$ii) } {
		puts $repf $mean_value($ii)
		puts $kvf [expr $mean_value($ii) * $mofpar(krel,$ii)]
	    }
	}
    }
}
#============================================================================


#############################################################################
proc export_modflow_categorical_variables_mean { iv } {
    global mofpar 
    global outfcatm
    global repf
    global kvf
    global ckvf
    global bcff
    global nchar
    global nlayer
    
    ### searches the simline "simulated value" and reject all preceded lines
    if { $iv == 1 } { ;# condition for the first simulation
	seek $repf 0
	seek $kvf 0
	seek $ckvf 0
	set simline [gets $repf]
	while { $simline != "Simulated Value" } {
	    set simline [gets $repf]
	}
	set kvline [gets $kvf]
	while { $kvline != "Simulated Value" } {
	    set kvline [gets $kvf]
	}
	set ckvline [gets $ckvf]
	while { $ckvline != "Simulated Value" } {
	    set ckvline [gets $ckvf]
	}

	### goes to the second layer of the copied vertical K values file
	for { set c 1 } { $c <= [expr $mofpar(nx) * $mofpar(ny)] } { incr c } {
	    set ckvline [gets $ckvf]
	}
	
	### writes all lines before the hydraulic conductivity or transmissivity, and leakance information
	seek $bcff 0
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    puts $outfcatm($iv) $line
	    set line [gets $bcff]
	}
	seek $bcff 0
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information, 
	### except the information of the last line
	for { set i 1 } { $i <= [expr $nlayer - 1] } { incr i } {
	    set line [gets $bcff]
	    while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
			[string first "Hydr. Conductivity of layer" $line] == -1 } {
		set line [gets $bcff]
	    }
	    puts $outfcatm($iv) $line
	    if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $repf]
			catch {scan $simline {%g} simline}
			set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatm($iv) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatm($iv) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatm($iv) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcatm($iv) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvf]
			set ckvline [gets $ckvf]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatm($iv) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatm($iv) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatm($iv) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    } elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $repf]
			catch {scan $simline {%g} simline}
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatm($iv) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatm($iv) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatm($iv) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcatm($iv) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvf]
			set ckvline [gets $ckvf]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatm($iv) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatm($iv) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatm($iv) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    }
	}
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information
	### of the last layer
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    set line [gets $bcff]
	}
	puts $outfcatm($iv) $line
	if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $repf]
		    catch {scan $simline {%g} simline}
		    set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcatm($iv) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcatm($iv) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatm($iv) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcatm($iv) $line
	    }
	} elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $repf]
		    catch {scan $simline {%g} simline}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcatm($iv) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcatm($iv) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatm($iv) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcatm($iv) $line
	    }
	}
    } elseif { $iv >= 1 } { ;# condition for the following simulations

	### writes all lines before the hydraulic conductivity or transmissivity, and leakance information
	seek $bcff 0
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    puts $outfcatm($iv) $line
	    set line [gets $bcff]
	}
	seek $bcff 0

	### goes to the first layer of the next simulation (vertical K values file)
	for { set c 1 } { $c <= [expr $mofpar(nx) * $mofpar(ny)] } { incr c } {
	    set kvline [gets $kvf]
	}

	### goes to the second layer of the next simulation (copied vertical K values file)
	for { set c 1 } { $c <= [expr $mofpar(nx) * $mofpar(ny)] } { incr c } {
	    set ckvline [gets $ckvf]
	}
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information, 
	### except the information of the last line 
	for { set i 1 } { $i <= [expr $nlayer - 1] } { incr i } {
	    set line [gets $bcff]
	    while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
			[string first "Hydr. Conductivity of layer" $line] == -1 } {
		set line [gets $bcff]
	    }
	    puts $outfcatm($iv) $line
	    if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $repf]
			catch {scan $simline {%g} simline}
			set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatm($iv) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatm($iv) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatm($iv) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcatm($iv) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvf]
			set ckvline [gets $ckvf]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatm($iv) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatm($iv) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatm($iv) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    } elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $repf]
			catch {scan $simline {%g} simline}
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatm($iv) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatm($iv) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatm($iv) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcatm($iv) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvf]
			set ckvline [gets $ckvf]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatm($iv) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatm($iv) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatm($iv) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    }
	}
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information
	### of the last layer
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    set line [gets $bcff]
	}
	puts $outfcatm($iv) $line
	if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $repf]
		    catch {scan $simline {%g} simline}
		    set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcatm($iv) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcatm($iv) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatm($iv) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcatm($iv) $line
	    }
	} elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $repf]
		    catch {scan $simline {%g} simline}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcatm($iv) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcatm($iv) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatm($iv) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcatm($iv) $line
	    }
	}
    }
}
#============================================================================


#############################################################################
proc distribution_value_modflow_replacement { i } {
    global mofpar 
    global outf
    global inf
    global charf
    global repfcatd
    global kvfcatd
    global mean_value_combo_w
    global distribution_value_combo_w
    global char_vals
    global mean_position
    global std_position
    global variable_position
    global mean_value
    global ln_mean_value
    global std_value
    global ln_std_value

    ### go to the start of the simulated realizations file
    seek $inf 0

    ### searches the mean value position in the characterization data file
    set mean_position [$mean_value_combo_w curselection]

    ### searches the standard deviation position in the characterization data file
    set std_position [$distribution_value_combo_w curselection]

    ### reading the mean values and the standard deviations of the variables
    for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	set variable_position($ii) [$char_vals($ii,combo) curselection]
	for { set v 0 } { $v <= [expr + $mofpar(charvar) + $variable_position($ii) + 3] } { incr v } {
	    set charline_1 [gets $charf]
	    set charline_2 $charline_1
	}
	for { set iii 0 } { $iii <= $mean_position } { incr iii } {
	    set space [string first " " $charline_1]
	    set mean_value($ii) [format "%g" [string range $charline_1 0 [expr $space - 1]]]
	    set charline_1 [string range $charline_1 [expr $space + 1] end] ;# it is the ln-mean of the field data
	}
	for { set iv 0 } { $iv <= $std_position } { incr iv } {
	    set space [string first " " $charline_2]
	    set std_value($ii) [format "%g" [string range $charline_2 0 [expr $space - 1]]] ;# it is the std of the ln-mean
	    set charline_2 [string range $charline_2 [expr $space + 1] end]
	}
	seek $charf 0
    }

    ### searches the line "simulated value" and reject all preceded lines
    set line [gets $inf]
    puts $repfcatd($i) $line
    puts $kvfcatd($i) $line
    while { $line != "Simulated Value" } {
	set line [gets $inf]
	puts $repfcatd($i) $line
	puts $kvfcatd($i) $line
    }
    
    ### replace the categorical variables with normally distributed values values 
    while { ![eof $inf] } {
	set line [gets $inf]
	catch {scan $line {%d} line}
	for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	    if { $line == $mofpar(cat,$ii) } {
		set krandom [expr (0 + exp([geossav_grandom -sigma $std_value($ii) -mean $mean_value($ii)]))]
		puts $repfcatd($i) $krandom
		;# the value given back from the geossav_grandom proc (g for gauss) in the file geossav.c 
		;# is a ln-value from a normal distribution, so this value has to be expound to the base of e
		puts $kvfcatd($i) [expr $krandom * $mofpar(krel,$ii)]
	    }
	}
    }
}
#============================================================================


#############################################################################
proc export_modflow_categorical_variables_distribution { vii viii } {
    global mofpar 
    global outfcatd
    global repfcatd
    global kvfcatd
    global ckvfcatd
    global bcff
    global nchar
    global nlayer
    
    ### searches the simline "simulated value" and reject all preceded lines
    if { $vii == 1 } { ;# condition for the first simulation
	seek $repfcatd($viii) 0
	seek $kvfcatd($viii) 0
	seek $ckvfcatd($viii) 0
	set simline [gets $repfcatd($viii)]
	while { $simline != "Simulated Value" } {
	    set simline [gets $repfcatd($viii)]
	}
	set kvline [gets $kvfcatd($viii)]
	while { $kvline != "Simulated Value" } {
	    set kvline [gets $kvfcatd($viii)]
	}
	set ckvline [gets $ckvfcatd($viii)]
	while { $ckvline != "Simulated Value" } {
	    set ckvline [gets $ckvfcatd($viii)]
	}

	### goes to the second layer of the copied vertical K values file
	for { set c 1 } { $c <= [expr $mofpar(nx) * $mofpar(ny)] } { incr c } {
	    set ckvline [gets $ckvfcatd($viii)]
	}

	### writes all lines before the hydraulic conductivity or transmissivity, and leakance information
	seek $bcff 0
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    puts $outfcatd($vii,$viii) $line
	    set line [gets $bcff]
	}
	seek $bcff 0
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information, 
	### except the information of the last line 
	for { set i 1 } { $i <= [expr $nlayer - 1] } { incr i } {
	    set line [gets $bcff]
	    while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
			[string first "Hydr. Conductivity of layer" $line] == -1 } {
		set line [gets $bcff]
	    }
	    puts $outfcatd($vii,$viii) $line
	    if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $repfcatd($viii)]
			catch {scan $simline {%g} simline}
			set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcatd($vii,$viii) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvfcatd($viii)]
			set ckvline [gets $ckvfcatd($viii)]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    } elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $repfcatd($viii)]
			catch {scan $simline {%g} simline}
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcatd($vii,$viii) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvfcatd($viii)]
			set ckvline [gets $ckvfcatd($viii)]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    }
	}
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information
	### of the last layer
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    set line [gets $bcff]
	}
	puts $outfcatd($vii,$viii) $line
	if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $repfcatd($viii)]
		    catch {scan $simline {%g} simline}
		    set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcatd($vii,$viii) $line
	    }
	} elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $repfcatd($viii)]
		    catch {scan $simline {%g} simline}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcatd($vii,$viii) $line
	    }
	}
    } elseif { $vii >= 1 } { ;# condition for the following simulations

	### writes all lines before the hydraulic conductivity or transmissivity, and leakance information
	seek $bcff 0
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    puts $outfcatd($vii,$viii) $line
	    set line [gets $bcff]
	}
	seek $bcff 0

	### goes to the first layer of the next simulation (vertical K values file)
	for { set c 1 } { $c <= [expr $mofpar(nx) * $mofpar(ny)] } { incr c } {
	    set kvline [gets $kvfcatd($viii)]
	}

	### goes to the second layer of the next simulation (copied vertical K values file)
	for { set c 1 } { $c <= [expr $mofpar(nx) * $mofpar(ny)] } { incr c } {
	    set ckvline [gets $ckvfcatd($viii)]
	}
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information, 
	### except the information of the last line 
	for { set i 1 } { $i <= [expr $nlayer - 1] } { incr i } {
	    set line [gets $bcff]
	    while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
			[string first "Hydr. Conductivity of layer" $line] == -1 } {
		set line [gets $bcff]
	    }
	    puts $outfcatd($vii,$viii) $line
	    if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $repfcatd($viii)]
			catch {scan $simline {%g} simline}
			set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcatd($vii,$viii) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvfcatd($viii)]
			set ckvline [gets $ckvfcatd($viii)]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    } elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity, leakance
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set simline [gets $repfcatd($viii)]
			catch {scan $simline {%g} simline}
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			    set iii 0
			}
		    }
		}
		set line [gets $bcff]
		while { [string first "Calculated Vertical Leakance of layer" $line] == -1 } {
		    set line [gets $bcff]
		}
		puts $outfcatd($vii,$viii) $line
		for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		    set iii 0
		    for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
			set kvline [gets $kvfcatd($viii)]
			set ckvline [gets $ckvfcatd($viii)]
			catch {scan $kvline {%g} kvline}
			catch {scan $ckvline {%g} ckvline}
			set vcont [expr 2 / (($mofpar(zsiz) / $kvline) + ($mofpar(zsiz) / $ckvline))] ;# leakance
			set iii [expr $iii + 1]
			if { $iii <= 19 } {
			    if { $v < $mofpar(nx) } {
				puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    } elseif { $v == $mofpar(nx) } {
				puts $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    }
			} elseif { $iii == 20 } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $vcont]
			    set iii 0
			}
		    }
		}
	    }
	}
	
	### writes the transmissivity and leakance, and the hydraulic conductivity and leakance information
	### of the last layer
	set line [gets $bcff]
	while { [string first "Transmissivity = Hydr. Conductivity * thickness of layer" $line] == -1 && \
		    [string first "Hydr. Conductivity of layer" $line] == -1 } {
	    set line [gets $bcff]
	}
	puts $outfcatd($vii,$viii) $line
	if { $mofpar(ltype,$i) == 0 || $mofpar(ltype,$i) == 2 } { ;# transmissivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $repfcatd($viii)]
		    catch {scan $simline {%g} simline}
		    set simline [expr $simline * $mofpar(zsiz)] ;# transmissivity = hydr. conduct. * thikness of layer
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcatd($vii,$viii) $line
	    }
	} elseif { $mofpar(ltype,$i) == 1 || $mofpar(ltype,$i) == 3 } { ;# hydraulic conductivity
	    for { set ii 1 } { $ii <= $mofpar(ny) } { incr ii } {
		set iii 0
		for { set v 1 } { $v <= $mofpar(nx) } { incr v } {
		    set simline [gets $repfcatd($viii)]
		    catch {scan $simline {%g} simline}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $v < $mofpar(nx) } {
			    puts -nonewline $outfcatd($vii,$viii) [format "%14.4e" $simline]
			} elseif { $v == $mofpar(nx) } {
			    puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatd($vii,$viii) [format "%14.4e" $simline]
			set iii 0
		    }
		}
	    }
	    ### the lines to be replaced are skipped
	    for { set vi 1 } { $vi <= [expr round($mofpar(nx) / 20.0) * $mofpar(ny)] } { incr vi } {
		set line [gets $bcff]
	    }
	    ### the rest of the output file is written
	    while { ![eof $bcff] } {
		set line [gets $bcff]
		puts $outfcatd($vii,$viii) $line
	    }
	}
    }
}
#============================================================================

#
# End of proc section.
#
#############################################################################


#############################################################################
proc geossav_export_modflow_main { } {
    global exedir
    source $exedir/Geossav_export_modflow_globals.tcl
    
    . configure -cursor watch
    update
    
    set export_modflow_box_w [toplevel .export_modflow_box -class Dialog]
    wm resizable $export_modflow_box_w false false
    wm sizefrom $export_modflow_box_w program
    wm group $export_modflow_box_w .
    wm transient $export_modflow_box_w [winfo toplevel [winfo parent $export_modflow_box_w]]
    wm title $export_modflow_box_w "EXPORT MODFLOW parameters"
    
    set geostr [wm geometry .]
    regexp {([0-9]*)x([0-9]*)(\+)([0-9]*)(\+)([0-9]*)} $geostr all width height p1 pos_x p2 pos_y
    set pos_x [expr $pos_x + $width / 2 + 212]
    set pos_y [expr $pos_y + $height / 2 + 15]
    wm geometry $export_modflow_box_w "+$pos_x+$pos_y"
    
    
    #############################################################################
    # Tabnotebook
    #############################################################################
    set export_modflow_tabnotebook_w [iwidgets::tabnotebook $export_modflow_box_w.export_modflow_tbn \
					  -width 5.4i \
					  -height 4.7i \
					  -tabpos n \
					  -tabforeground gray50 \
					  -background gray85 \
					  -bevelamount 2]
    
    pack $export_modflow_tabnotebook_w -padx 0 -pady 0
    
    
    #
    # page1: Geossav EXPORT_MODFLOW-parameters: Files (there is only one page)
    #
    set page [$export_modflow_box_w.export_modflow_tbn add \
		  -label "Files"]
    

    # initialize dimension names for characterization data
    if { ![export_modflow_read_char_data $mofpar(charfl)] } {
	set dim_name_char {}
	for { set i 1} { $i <= 3 } { incr i } {
	    lappend dim_name_char $i
	}
    }
    
    
    set work_frame_w [frame $page.workframe \
			  -borderwidth 2 \
			  -relief flat]
    
    # fields for simulated data
    set data_iframe_w [iwidgets::labeledframe $work_frame_w.data \
			   -labeltext "Simulation Input" \
			   -labelpos nw \
			   -borderwidth 2 \
			   -relief groove]
    set data_frame_w [$data_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $data_frame_w.subfr1]

    set datafl_w [iwidgets::entryfield $subframe1_w.datafl \
		      -labeltext "Simulated realizations, file " \
		      -width 51 \
		      -state disabled \
		      -textbackground gray85 \
		      -foreground gray50 \
		      -textvariable mofpar(datafl)]

    if { $mofpar(datafl) != {*.out} } {
	$datafl_w configure -state normal -textbackground white -foreground black
    }

    if { $mofpar(datafl) == {*.out} } {
	set mofpar(dflbc) 0
    } elseif { $mofpar(datafl) != {*.out} } {
	set mofpar(dflbc) $mofpar(dflbc)
    }
    set datafl_browse_w [iwidgets::pushbutton $subframe1_w.datafl_browse \
			     -height 27 \
			     -width 40 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			     -command {
				 set tmp_fname [tk_getOpenFile \
						    -defaultextension ".out" \
						    -filetypes {{{Simulation Files} {.out}} {{All Files} {.*}}} \
						    -initialfile $mofpar(datafl) \
						    -initialdir $exedir \
						    -title "Select simulated realizations file for EXPORT MODFLOW"]
				 
				 ### counts the activation of the datafl browse button
				 set mofpar(dflbc) [expr $mofpar(dflbc) + 1]
				 
 				 ### select file name
				 if { $tmp_fname != {} } {
				     set mofpar(datafl) $tmp_fname
				 
				     ### set datafl and realization widgets normal
				     if { $mofpar(datafl) == {*.out} } {
					 $datafl_w configure -state disabled -textbackground gray85 -foreground gray50
					 $realization_sim_w configure \
					     -state disabled -textbackground gray85 -foreground gray50
					 $charfl_browse_w configure -state disabled
					 $relation_k_w configure -state disabled -textbackground gray85 -foreground gray50
				     } elseif { $mofpar(datafl) != {*.out} } {
					 $datafl_w configure -state normal -textbackground white -foreground black
					 $realization_sim_w configure -state normal -textbackground white -foreground black
					 $charfl_browse_w configure -state normal
					 $relation_k_w configure -state normal -textbackground white -foreground black
				     }
				     
				     ### create infofl path- and filename
				     set infofl_rootname [file rootname $mofpar(datafl)]
				     set infofl_dirname [file dirname $mofpar(datafl)]
				     set infofl_dirname_lenght [string length $infofl_dirname]
				     set infofl_name [string range $infofl_rootname \
							  [expr $infofl_dirname_lenght + 1] end]
				     set infofl_extension [file extension $mofpar(datafl)]
				     set info "/Info_"
				     regsub {} $infofl_extension $infofl_name infofl_1
				     regsub {} $infofl_1 $info infofl_2
				     regsub {} $infofl_2 $infofl_dirname infofl_3
				     set mofpar(infofl) $infofl_3
				     
				     ### open infofl and read data
				     if { $mofpar(infofl) != {} } {
					 read_modflow_infofl_params $mofpar(infofl)
				     }
				     
				     ### number of output files
				     number_modflow_output_files
				 }
			     }]
    
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $datafl_w        -row 0 -column 0 -sticky we -pady 0
    grid $datafl_browse_w -row 0 -column 1 -sticky w -pady 0


    set subframe2_w [frame $data_frame_w.subfr2]

    set realization_sim_w [iwidgets::entryfield $subframe2_w.realization_sim \
			       -labeltext "Number of realizations " \
			       -width 8 \
			       -state disabled \
			       -textbackground gray85 \
			       -foreground gray50 \
			       -textvariable mofpar(nsim) \
			       -validate {check_modflow_number_nsim_sim %W %P %c}]
    if { $mofpar(datafl) != {*.out} } {
	$realization_sim_w configure -state normal -textbackground white -foreground black
    }


    set variable_type_label_w [iwidgets::labeledwidget $subframe2_w.label_variable_type \
				   -labeltext "   Variable type"]
    set tmp_vartype $mofpar(vartype)
    set variable_type_value_w [iwidgets::labeledwidget $subframe2_w.value_variable_type \
				   -labelvariable mofpar(vartype)]
    if { $tmp_vartype == 0 } {
	set mofpar(vartype) "Categorical"
    } elseif { $tmp_vartype == 1 } {
    	set mofpar(vartype) "Continous"
    }

    set relation_k_w [iwidgets::entryfield $subframe2_w.relation_k \
			  -labeltext "Kv/Kh " \
			  -width 11 \
			  -state disabled \
			  -textbackground gray85 \
			  -foreground gray50 \
			  -textvariable mofpar(relk) \
			  -validate {check_number_modflow_krel_values %W %P %c}]
    if { $mofpar(datafl) != {*.out} } {
	$relation_k_w configure -state normal -textbackground white -foreground black
    }
    if { $mofpar(vartype) == "Categorical" } {
	$relation_k_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $mofpar(vartype) == "Continous" } {
	$relation_k_w configure -state normal -textbackground white -foreground black
    }

    grid columnconfigure $subframe2_w 0 -minsize 186
    grid columnconfigure $subframe2_w 1 -minsize 80
    grid columnconfigure $subframe2_w 2 -minsize 101
    grid columnconfigure $subframe2_w 3 -minsize 140
    grid $realization_sim_w     -row 0 -column 0 -sticky e -pady 0
    grid $variable_type_label_w -row 0 -column 1 -sticky w -pady 0
    grid $variable_type_value_w -row 0 -column 2 -sticky w -pady 0
    grid $relation_k_w          -row 0 -column 3 -sticky w -pady 0
    bind [$realization_sim_w component entry]  <KeyRelease> {number_modflow_output_files}


    set subframe3_w [frame $data_frame_w.subfr3]

    set number_x_label_w [iwidgets::labeledwidget $subframe3_w.label_number_x \
			      -labeltext "Number x"]
    set number_x_value_w [iwidgets::labeledwidget $subframe3_w.value_number_x \
			      -labelvariable mofpar(nx)]


    set number_y_label_w [iwidgets::labeledwidget $subframe3_w.label_number_y \
			      -labeltext "Number y"]
    set number_y_value_w [iwidgets::labeledwidget $subframe3_w.value_number_y \
			      -labelvariable mofpar(ny)]


    set number_z_label_w [iwidgets::labeledwidget $subframe3_w.label_number_z \
			      -labeltext "Number z"]
    set number_z_value_w [iwidgets::labeledwidget $subframe3_w.value_number_z \
			      -labelvariable mofpar(nz)]


    grid columnconfigure $subframe3_w 0 -minsize 129
    grid columnconfigure $subframe3_w 1 -minsize 35
    grid columnconfigure $subframe3_w 2 -minsize 100
    grid columnconfigure $subframe3_w 3 -minsize 40
    grid columnconfigure $subframe3_w 4 -minsize 103
    grid columnconfigure $subframe3_w 5 -minsize 35
    grid $number_x_label_w -row 0 -column 0 -sticky e -pady 0
    grid $number_x_value_w -row 0 -column 1 -sticky w -pady 0
    grid $number_y_label_w -row 0 -column 2 -sticky e -pady 0
    grid $number_y_value_w -row 0 -column 3 -sticky w -pady 0
    grid $number_z_label_w -row 0 -column 4 -sticky e -pady 0
    grid $number_z_value_w -row 0 -column 5 -sticky w -pady 0


    set subframe4_w [frame $data_frame_w.subfr4]

    set number_xsiz_label_w [iwidgets::labeledwidget $subframe4_w.label_number_xsiz \
			       -labeltext "Size x"]
    set number_xsiz_value_w [iwidgets::labeledwidget $subframe4_w.value_number_xsiz \
			       -labelvariable mofpar(xsiz)]


    set number_ysiz_label_w [iwidgets::labeledwidget $subframe4_w.label_number_ysiz \
			       -labeltext "Size y"]
    set number_ysiz_value_w [iwidgets::labeledwidget $subframe4_w.value_number_ysiz \
			       -labelvariable mofpar(ysiz)]
    

    set number_zsiz_label_w [iwidgets::labeledwidget $subframe4_w.label_number_zsiz \
			       -labeltext "Size z"]
    set number_zsiz_value_w [iwidgets::labeledwidget $subframe4_w.value_number_zsiz \
			       -labelvariable mofpar(zsiz)]


    grid columnconfigure $subframe4_w 0 -minsize 129
    grid columnconfigure $subframe4_w 1 -minsize 35
    grid columnconfigure $subframe4_w 2 -minsize 100
    grid columnconfigure $subframe4_w 3 -minsize 40
    grid columnconfigure $subframe4_w 4 -minsize 103
    grid columnconfigure $subframe4_w 5 -minsize 35
    grid $number_xsiz_label_w -row 0 -column 0 -sticky e -pady 0
    grid $number_xsiz_value_w -row 0 -column 1 -sticky w -pady 0
    grid $number_ysiz_label_w -row 0 -column 2 -sticky e -pady 0
    grid $number_ysiz_value_w -row 0 -column 3 -sticky w -pady 0
    grid $number_zsiz_label_w -row 0 -column 4 -sticky e -pady 0
    grid $number_zsiz_value_w -row 0 -column 5 -sticky w -pady 0


    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	$subframe4_w \
	-anchor w -expand 1 -padx 3 -pady 1
    
    
    # fields for characterization data
    set characterization_iframe_w [iwidgets::labeledframe $work_frame_w.characterization \
				       -labeltext "Characterization Input" \
				       -labelpos nw \
				       -borderwidth 2 \
				       -relief groove]
    set characterization_frame_w [$characterization_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $characterization_frame_w.subfr1]

    set charfl_w [iwidgets::entryfield $subframe1_w.charfl \
		      -labeltext "Variables characterization, file  " \
		      -state disabled \
		      -textbackground gray85 \
		      -foreground gray50 \
		      -width 47 \
		      -textvariable mofpar(charfl)]


    set charfl_browse_w [iwidgets::pushbutton $subframe1_w.charfl_browse \
			     -state disabled \
			     -height 27 \
			     -width 40 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			     -command {
				 set tmp_fname [tk_getOpenFile \
						    -defaultextension ".gtd" \
						    -filetypes {{{Geo-EAS Files} {.gtd}} {{All Files} {.*}}} \
						    -initialfile $mofpar(charfl) \
						    -initialdir $exedir \
						    -title "Select ground type characterization file for EXPORT MODFLOW"]

				 ### exit from the BICALIB parameters tabnotebook
				 set mofpar(cancel_counter_characterization) 0
				 
				 ### set the old number of characterization variables
				 if { $mofpar(charfl) != {*.gtd} } {
				     set mofpar(oldcharvar) $mofpar(charvar)
				 }

				 ### select file name
				 if { $tmp_fname != {} } {
				     set mofpar(charfl) $tmp_fname
				     unset dim_name_char
				     if { [export_modflow_read_char_data $mofpar(charfl)] != 1 } {
					 set dim_name_char {}
					 for { set i 1 } { $i <= $mofpar(charvar) } { incr i } {
					     lappend dim_name_char $i
					 }
				     }
				 }

				 ### fills part of the char table
				 for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
				     ### column 1
				     $char_vals($ii,entry) configure -state normal -textbackground white -foreground black
				     ### column 2
				     $char_vals($ii,combo) configure -state normal -textbackground white -foreground black
				     $char_vals($ii,combo) clear list
				     foreach i $dim_name_char {
					 $char_vals($ii,combo) insert list end $i
				     }
				     $char_vals($ii,combo) selection set [expr $ii - 1]
				     set mofpar(icoli) [lindex $dim_name_char $ii]
				     update
				     $char_table_w window configure $ii,2 -window $char_vals($ii,combo)
				     ### column 3
				     $char_vals_k($ii,entry) configure \
					 -state normal \
					 -textbackground white \
					 -foreground black
				 }
				 
				 ### fills the mean_value_combobox
				 $mean_value_combo_w configure -state normal -textbackground white -foreground black
				 $mean_value_combo_w clear list
				 foreach i $dim_name_char {
				     $mean_value_combo_w insert list end $i
				 }
				 $mean_value_combo_w selection set [expr $mofpar(ncat) + 0]
				 set mofpar(cmean) [lindex $dim_name_char [expr $mofpar(ncat) + 0]]
				 update

				 ### fills the distribution_value_combobox
				 $distribution_value_combo_w configure \
				     -state normal -textbackground white -foreground black
				 $distribution_value_combo_w clear list
				 foreach i $dim_name_char {
				     $distribution_value_combo_w insert list end $i
				 }
				 $distribution_value_combo_w selection set [expr $mofpar(ncat) + 1]
				 set mofpar(cdist) [lindex $dim_name_char [expr $mofpar(ncat) + 1]]
				 update
				 if { $mofpar(mean) == 1 } {
				     $distribution_value_combo_w configure \
					 -state disabled -textbackground gray85 -foreground gray50
				     $realization_char_w configure \
					 -state disabled -textbackground gray85 -foreground gray50
				 } elseif { $mofpar(mean) == 0 } {
				     set mofpar(dist) 1
				     $distribution_value_combo_w configure \
					 -state normal -textbackground white -foreground black
				     $realization_char_w configure \
					 -state disabled -textbackground gray85 -foreground gray50
				 }

				 ### set the rest of the characterization input widgets normal/disabled
				 $charfl_w             configure -state normal -textbackground white -foreground black
				 $charfl_browse_w      configure -state normal
				 $mean_value_w         configure -state normal
				 $distribution_value_w configure -state normal
				 if { $mofpar(mean) == 1 } {
				     $mean_value_combo_w configure \
					 -state normal -textbackground white -foreground black
				     $distribution_value_combo_w configure \
					 -state disabled -textbackground gray85 -foreground gray50
				     $realization_char_w configure \
					 -state disabled -textbackground gray85 -foreground gray50
				 } elseif { $mofpar(dist) == 1 } {
				     $mean_value_combo_w configure \
					 -state normal -textbackground white -foreground black
				     $distribution_value_combo_w \
					 configure -state normal -textbackground white -foreground black
				     $realization_char_w \
					 configure -state normal -textbackground white -foreground black
				 }

				 ### number of output files
				 number_modflow_output_files

				 ### message, if the variables characterization file has to few data
				 if { $mofpar(datafl) != {*.out} && $mofpar(charfl) != {*.gtd} } {
				     if { $mofpar(charvar) < [expr $mofpar(ncat) + 1] } {
					 tk_messageBox \
					     -title "Characterization Input" \
					     -icon warning \
					     -message "The variables characterization file has to few data.\nYou should select another file." \
					     -type ok
				     }
				 }

			     }]

    ### set charfl disabled/normal
    if { $mofpar(datafl) != {*.out} } {
	$charfl_browse_w configure -state normal
    }    

    ### set charfl disabled/normal and count the exit from the tabnotebook
    if { $mofpar(charfl) != {*.gtd} } {
	$charfl_w configure -state normal -textbackground white -foreground black
	set mofpar(cancel_counter_characterization) [expr $mofpar(cancel_counter_characterization) + 1]
    }

    ### set charfl disabled/normal depending on the vartype
    if { $mofpar(vartype) == "Continous" } {
	$charfl_w configure -state disabled -textbackground gray85 -foreground gray50
	$charfl_browse_w configure -state disabled
    }


    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $charfl_w        -row 0 -column 0 -sticky we -pady 0
    grid $charfl_browse_w -row 0 -column 1 -sticky w -pady 0


    set subframe2_w [frame $characterization_frame_w.subfr2]

    ### setting the characterization table normal/disabled depending on the input data
    set char_frame_w [frame $subframe2_w.char_frame \
			  -background grey40 \
			  -borderwidth 2 \
			  -relief sunken]
    
    set char_sframe_w [frame $char_frame_w.char_sframe \
			   -height 72 \
			   -width 310]
    set char_hsbar_w [scrollbar $char_sframe_w.yscroll \
			  -orient vertical \
			  -command "$char_sframe_w.table yview"]
    
    
    if { $mofpar(datafl) == "*.out" && $mofpar(charfl) == "*.gtd" } {
	
	set char_table_w [table $char_sframe_w.table \
			      -variable char_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 4 \
			      -rows 4 \
			      -yscrollcommand "$char_hsbar_w set" \
			      -height 4]
	
	$char_table_w width 0 5
	$char_table_w width 1 10
	$char_table_w width 2 22
	$char_table_w width 3 10
	
	set char_vals(0,0) "\#"
	set char_vals(0,1) "Category"
	set char_vals(0,2) "Variable"
	set char_vals(0,3) "Kv/Kh"
	
	for {set ii 1} {$ii <= 3} {incr ii} {
	    
	    set char_vals($ii,label) [iwidgets::labeledwidget $char_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]

	    set mofpar(cat,$ii) $ii
	    set char_vals($ii,entry) [iwidgets::entryfield $char_table_w.entry$ii \
					  -textvariable mofpar(cat,$ii) \
					  -validate {regexpvalidate %c} \
					  -state disabled \
					  -width 10 \
					  -textbackground gray85 \
					  -foreground gray50]
	    
	    set char_vals($ii,combo) [iwidgets::combobox $char_table_w.combo$ii \
					  -textvariable mofpar(icoli,$ii) \
					  -validate {regexpvalidate %c} \
					  -width 22 \
					  -listheight 100]
	    foreach i $dim_name_char {
		$char_vals($ii,combo) insert list end $i
	    }
	    $char_vals($ii,combo) selection set [expr $ii - 1]
	    set mofpar(icoli) [lindex $dim_name_char $ii]
	    $char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50

	    set mofpar(krel,$ii) 1
	    set char_vals_k($ii,entry) [iwidgets::entryfield $char_table_w.entry_k$ii \
					    -textvariable mofpar(krel,$ii) \
					    -validate {check_number_modflow_krel_values %W %P %c} \
					    -state disabled \
					    -width 10 \
					    -textbackground gray85 \
					    -foreground gray50]
	    
	    $char_table_w window configure $ii,0 -window $char_vals($ii,label) -relief flat -background gray50
	    $char_table_w window configure $ii,1 -window $char_vals($ii,entry)
	    $char_table_w window configure $ii,2 -window $char_vals($ii,combo)
	    $char_table_w window configure $ii,3 -window $char_vals_k($ii,entry)

	    ### set entryfield and combobox of the characterizaton table disabled/normal depending on the vartype
	    if { $mofpar(vartype) == "Continous" } {
		$char_vals($ii,entry)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals_k($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	}
    } elseif { ($mofpar(datafl) != "*.out") && ($mofpar(charfl) != "*.gtd") || \
	     ($mofpar(datafl) == "*.out") && ($mofpar(charfl) != "*.gtd") } {
	
	set char_table_w [table $char_sframe_w.table \
			      -variable char_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 4 \
			      -rows [expr $mofpar(ncat) + 1] \
			      -yscrollcommand "$char_hsbar_w set" \
			      -height 4]
	
	$char_table_w width 0 5
	$char_table_w width 1 10
	$char_table_w width 2 22
	$char_table_w width 3 10
	
	set char_vals(0,0) "\#"
	set char_vals(0,1) "Category"
	set char_vals(0,2) "Variable"
	set char_vals(0,3) "Kv/Kh"
	    
	for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
		
	    set char_vals($ii,label) [iwidgets::labeledwidget $char_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]

	    set char_vals($ii,entry) [iwidgets::entryfield $char_table_w.entry$ii \
					  -textvariable mofpar(cat,$ii) \
					  -validate {regexpvalidate %c} \
					  -state normal \
					  -width 10 \
					  -textbackground white \
					  -foreground black]
	    
	    set char_vals($ii,combo) [iwidgets::combobox $char_table_w.combo$ii \
					  -textvariable mofpar(icoli,$ii) \
					  -validate {regexpvalidate %c} \
					  -width 22 \
					  -listheight 100]
	    set tmp_icoli($ii) $mofpar(icoli,$ii)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if mofpar(icoli) is empty
	    foreach i $dim_name_char {
		$char_vals($ii,combo) insert list end $i
		$char_vals($ii,combo) delete list [expr $mofpar(charvar) + 1]
		if { $i == $tmp_icoli($ii) } { ;# ok, that's the value in mofpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($mofpar(cancel_counter_characterization) * ($mofpar(charvar) + 1))]
	    $char_vals($ii,combo) selection set $sel
	    set mofpar(icoli,$ii) [lindex $dim_name_char $sel]
	    $char_vals($ii,combo) configure -state normal -textbackground white -foreground black

	    set char_vals_k($ii,entry) [iwidgets::entryfield $char_table_w.entry_k$ii \
					    -textvariable mofpar(krel,$ii) \
					    -validate {check_number_modflow_krel_values %W %P %c} \
					    -state normal \
					    -width 10 \
					    -textbackground white \
					    -foreground black]

	    $char_table_w window configure $ii,0 -window $char_vals($ii,label) -relief flat -background gray50
	    $char_table_w window configure $ii,1 -window $char_vals($ii,entry)
	    $char_table_w window configure $ii,2 -window $char_vals($ii,combo)
	    $char_table_w window configure $ii,3 -window $char_vals_k($ii,entry)

	    ### set entryfield and combobox of the characterizaton table disabled/normal depending on the vartype
	    if { $mofpar(vartype) == "Continous" } {
		$char_vals($ii,entry)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals_k($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	}
    } elseif { $mofpar(datafl) != "*.out" && $mofpar(charfl) == "*.gtd" } {
	
	set char_table_w [table $char_sframe_w.table \
			      -variable char_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 4 \
			      -rows [expr $mofpar(ncat) + 1] \
			      -yscrollcommand "$char_hsbar_w set" \
			      -height 4]
	
	$char_table_w width 0 5
	$char_table_w width 1 10
	$char_table_w width 2 22
	$char_table_w width 3 10
	
	set char_vals(0,0) "\#"
	set char_vals(0,1) "Category"
	set char_vals(0,2) "Variable"
	set char_vals(0,3) "Kv/Kh"
	
	for { set ii 1 } { $ii <= $mofpar(ncat) } { incr ii } {
	    
	    set char_vals($ii,label) [iwidgets::labeledwidget $char_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]

	    set char_vals($ii,entry) [iwidgets::entryfield $char_table_w.entry$ii \
					  -textvariable mofpar(cat,$ii) \
					  -validate {regexpvalidate %c} \
					  -state disabled \
					  -width 10 \
					  -textbackground gray85 \
					  -foreground gray50]
	    
	    set char_vals($ii,combo) [iwidgets::combobox $char_table_w.combo$ii \
					  -textvariable mofpar(icoli,$ii) \
					  -validate {regexpvalidate %c} \
					  -width 22 \
					  -listheight 100]
	    foreach i $dim_name_char {
		$char_vals($ii,combo) insert list end $i
	    }
	    $char_vals($ii,combo) selection set [expr $ii - 1]
	    set mofpar(icoli) [lindex $dim_name_char $ii]
	    $char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50

	    set mofpar(krel,$ii) 1
	    set char_vals_k($ii,entry) [iwidgets::entryfield $char_table_w.entry_k$ii \
					    -textvariable mofpar(krel,$ii) \
					    -validate {check_number_modflow_krel_values %W %P %c} \
					    -state disabled \
					    -width 10 \
					    -textbackground gray85 \
					    -foreground gray50]
	    
	    $char_table_w window configure $ii,0 -window $char_vals($ii,label) -relief flat -background gray50
	    $char_table_w window configure $ii,1 -window $char_vals($ii,entry)
	    $char_table_w window configure $ii,2 -window $char_vals($ii,combo)
	    $char_table_w window configure $ii,3 -window $char_vals_k($ii,entry)

	    ### set entryfield and combobox of the characterizaton table disabled/normal depending on the vartype
	    if { $mofpar(vartype) == "Continous" } {
		$char_vals($ii,entry)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo)   configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals_k($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	}
    }


    ### set the mean value checkbutton
    set mean_value_w [checkbutton $subframe2_w.mean_cbutton \
			  -text "Mean value ( ln ! )      " \
			  -variable mofpar(mean) \
			  -state disabled \
			  -command {mean_value_modflow_selection}]
    if { $mofpar(charfl) != {*.gtd} } {
	$mean_value_w configure -state normal
    }


    ### set the distribution value checkbutton
    set distribution_value_w [checkbutton $subframe2_w.distribution_cbutton \
				  -text "Value from distribution" \
				  -variable mofpar(dist) \
				  -state disabled \
				  -command {distribution_value_modflow_selection}]
    if { $mofpar(charfl) != {*.gtd} } {
	$distribution_value_w configure -state normal
    }

    ### set the mean value combobox
    set mean_value_combo_w [iwidgets::combobox $subframe2_w.mean_combo \
				-labeltext "Mean" \
				-textvariable mofpar(cmean) \
				-width 17 \
				-listheight 80 \
				-validate {regexpvalidate %c}]
    if { $mofpar(charfl) == {*.gtd} } {
	foreach i $dim_name_char {
	    $mean_value_combo_w insert list end $i
	}
	$mean_value_combo_w selection set 0
	set mofpar(cmean) [lindex $dim_name_char 0]
	$mean_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $mofpar(charfl) != {*.gtd} } {
	$mean_value_combo_w configure -state normal -textbackground white -foreground black
	set tmp_cmean $mofpar(cmean)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if sisimpar(icolsz) is empty
	foreach i $dim_name_char {
	    $mean_value_combo_w insert list end $i
	    $mean_value_combo_w delete list [expr $mofpar(charvar) + 1]
	    if { $i == $tmp_cmean } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($mofpar(cancel_counter_characterization) * ($mofpar(charvar) + 1))]
	$mean_value_combo_w selection set $sel
	set mofpar(cmean) [lindex $dim_name_char $sel]
	update
    }
    
    ### set mean- and distribution checkbutton, and mean value combobox disabled/normal depending on the vartype
    if { $mofpar(vartype) == "Continous" } {
	$mean_value_w configure -state disabled
	$distribution_value_w configure -state disabled
	$mean_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
    }

    pack $char_hsbar_w  -side right -anchor w -fill y
    pack $char_table_w  -side left -anchor w -padx 2 -expand 1 -fill both
    pack $char_sframe_w -anchor nw -expand 0
    pack propagate $char_sframe_w 0

    grid $char_frame_w -row 0 -column 0 -sticky new -rowspan 4 -pady 0
    grid columnconfigure $subframe2_w 1 -minsize 50
    grid columnconfigure $subframe2_w 2 -minsize 50
    grid columnconfigure $subframe2_w 3 -minsize 120

    grid $mean_value_w         -row 0 -column 1 -sticky e -pady 0 -padx 2
    grid $distribution_value_w -row 1 -column 1 -sticky e -pady 0 -padx 2
    grid $mean_value_combo_w   -row 2 -column 1 -sticky e -pady 3 -padx 5

    
    set subframe3_w [frame $characterization_frame_w.subfr3]

    ### set the realization widget
    set realization_char_w [iwidgets::entryfield $subframe3_w.realization_char \
				-labeltext "   Number of characterizations  " \
				-width 8 \
				-state disabled \
				-textbackground gray85 \
				-foreground gray50 \
				-textvariable mofpar(nsimchar) \
				-validate {check_modflow_number_nsim_char %W %P %c}]
    if { $mofpar(charfl) != {*.gtd} } {
	$realization_char_w configure -state normal -textbackground white -foreground black
    }

    ### set the distribution value combobox
    set distribution_value_combo_w [iwidgets::combobox $subframe3_w.distribution_combo \
					-labeltext "Standard Deviation" \
					-textvariable mofpar(cdist) \
					-width 17 \
					-listheight 80 \
					-validate {regexpvalidate %c}]
    if { $mofpar(charfl) == {*.gtd} } {
	foreach i $dim_name_char {
	    $distribution_value_combo_w insert list end $i
	}
	$distribution_value_combo_w selection set 1
	set mofpar(cdist) [lindex $dim_name_char 1]
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $mofpar(charfl) != {*.gtd} } {
	$distribution_value_combo_w configure -state normal -textbackground white -foreground black
	set tmp_cdist $mofpar(cdist)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if sisimpar(icolsz) is empty
	foreach i $dim_name_char {
	    $distribution_value_combo_w insert list end $i
	    $distribution_value_combo_w delete list [expr $mofpar(charvar) + 1]
	    if { $i == $tmp_cdist } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($mofpar(cancel_counter_characterization) * ($mofpar(charvar) + 1))]
	$distribution_value_combo_w selection set $sel
	set mofpar(cdist) [lindex $dim_name_char $sel]
	update
    }


    ### set the distribution_value_combobox disabled/normal depending on the value selection
    if { $mofpar(charfl) != {*.gtd} } {
	if { $mofpar(mean) == 1 } {
	    set mofpar(dist) 0
	    $distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
	    $realization_char_w configure -state disabled -textbackground gray85 -foreground gray50
	} elseif { $mofpar(mean) == 0 } {
	    set mofpar(dist) 1
	    $distribution_value_combo_w configure -state normal -textbackground white -foreground black
	    $realization_char_w configure -state normal -textbackground white -foreground black
	}
    }

    ### set mean- and distribution checkbutton, and mean value combobox disabled/normal depending on the vartype
    if { $mofpar(vartype) == "Continous" } {
	$realization_char_w configure -state disabled -textbackground gray85 -foreground gray50
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
    }


    grid columnconfigure $subframe3_w 0 -minsize 250
    grid columnconfigure $subframe3_w 1 -minsize 231
    grid $realization_char_w         -row 0 -column 0 -sticky w -pady 0
    grid $distribution_value_combo_w -row 0 -column 1 -sticky e -pady 0
    bind [$realization_char_w component entry]  <KeyRelease> {number_modflow_output_files}
 

    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	-anchor w -expand 1 -padx 3 -pady 1
    
    
    # fields for output data
    set output_iframe_w [iwidgets::labeledframe $work_frame_w.output \
			     -labeltext "Modflow Input / Output" \
			     -labelpos nw \
			     -borderwidth 2 \
			     -relief groove]
    set output_frame_w [$output_iframe_w childsite]
    
    set subframe1_w [frame $output_frame_w.subfr1]

    set bcf_inputfl_w [iwidgets::entryfield $subframe1_w.bcf_inputfl \
			   -labeltext "   bcf input file " \
			   -width 60 \
			   -state disabled \
			   -textbackground gray85 \
			   -foreground gray50 \
			   -textvariable mofpar(inbcffl)]

    if { $mofpar(inbcffl) != {*.dat} } {
	$bcf_inputfl_w configure -state normal -textbackground white -foreground black
    }

    set bcf_inputfl_browse_w [iwidgets::pushbutton $subframe1_w.bcf_inputfl_browse \
				  -height 27 \
				  -width 40 \
				  -defaultringpad 0 \
				  -image [image create photo -file [file join "$exedir/images/open.gif"]] \
				  -command {
				      set tmp_fname [tk_getOpenFile \
							 -defaultextension ".dat" \
							 -filetypes {{{Modflow Files} {.dat}} {{All Files} {.*}}} \
							 -initialfile $mofpar(inbcffl) \
							 -initialdir $exedir \
							 -title "Select bcf file to overwrite"]

				      ### select file name
				      if { $tmp_fname != {} } {
					  set mofpar(inbcffl) $tmp_fname
					  
					  ### set bcf input widget number of cell widgets normal
					  $bcf_inputfl_w configure \
					      -state normal -textbackground white -foreground black
					  $modflow_model_nx_w configure \
					      -state normal -textbackground white -foreground black
					  $modflow_model_ny_w configure \
					      -state normal -textbackground white -foreground black
					  $modflow_model_nz_w configure \
					      -state normal -textbackground white -foreground black
					  
					  ### output path- and filename
					  if { $mofpar(inbcffl) != {*.dat} } {
					      set inbcffl_rootname [file rootname $mofpar(inbcffl)]
					      set inbcffl_dirname [file dirname $mofpar(inbcffl)]
					      set inbcffl_dirname_lenght [string length $inbcffl_dirname]
					      set inbcffl_name [string range $inbcffl_rootname \
								    [expr $inbcffl_dirname_lenght + 1] end]
					      
					      set outfl_extension [file extension $mofpar(outfl)]
					      set export "/EXPORT_bcf_"
					      regsub {} $outfl_extension $inbcffl_name outputfl_1
					      regsub {} $outputfl_1 $export outputfl_2
					      regsub {} $outputfl_2 $inbcffl_dirname outputfl_3
					      set mofpar(outfl) $outputfl_3
					  }
				      }
				  }]


    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $bcf_inputfl_w        -row 0 -column 0 -sticky we -pady 0
    grid $bcf_inputfl_browse_w -row 0 -column 1 -sticky w -pady 0


    set subframe2_w [frame $output_frame_w.subfr2]

    set modflow_model_nx_w [iwidgets::entryfield $subframe2_w.modflow_nx \
				-labeltext "Number x " \
				-width 10 \
				-state disabled \
				-textbackground gray85 \
				-foreground gray50 \
				-textvariable mofpar(modnx) \
				-validate {check_modflow_cell_number_x %W %P %c}]
    if { $mofpar(inbcffl) != {*.dat} } {
	$modflow_model_nx_w configure -state normal -textbackground white -foreground black
    }

    set modflow_model_ny_w [iwidgets::entryfield $subframe2_w.modflow_ny \
				-labeltext "Number y " \
				-width 10 \
				-state disabled \
				-textbackground gray85 \
				-foreground gray50 \
				-textvariable mofpar(modny) \
				-validate {check_modflow_cell_number_y %W %P %c}]
    if { $mofpar(inbcffl) != {*.dat} } {
	$modflow_model_ny_w configure -state normal -textbackground white -foreground black
    }

    set modflow_model_nz_w [iwidgets::entryfield $subframe2_w.modflow_nz \
				-labeltext "Number z " \
				-width 10 \
				-state disabled \
				-textbackground gray85 \
				-foreground gray50 \
				-textvariable mofpar(modnz) \
				-validate {check_modflow_cell_number_z %W %P %c}]
    if { $mofpar(inbcffl) != {*.dat} } {
	$modflow_model_nz_w configure -state normal -textbackground white -foreground black
    }

    grid columnconfigure $subframe2_w 0 -minsize 144
    grid columnconfigure $subframe2_w 1 -minsize 150
    grid columnconfigure $subframe2_w 2 -minsize 150
    
    grid $modflow_model_nx_w -row 0 -column 0 -sticky e -pady 0
    grid $modflow_model_ny_w -row 0 -column 1 -sticky e -pady 0
    grid $modflow_model_nz_w -row 0 -column 2 -sticky e -pady 0


    set subframe3_w [frame $output_frame_w.subfr3]

    set outfl_w [iwidgets::entryfield $subframe3_w.outfl \
		     -labeltext "bcf output file " \
		     -width 62 \
		     -textvariable mofpar(outfl)]

    set outfl_browse_w [iwidgets::pushbutton $subframe3_w.outfl_browse \
			    -height 27 \
			    -width 27 \
			    -defaultringpad 0 \
			    -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			    -command {
				set tmp_fname [tk_getSaveFile \
						   -defaultextension ".out" \
						   -filetypes {{{Output Files} {.out}} {{All Files} {.*}}} \
						   -initialfile $mofpar(outfl) \
						   -initialdir $exedir \
						   -title "Save bcf file as ..."]

				if { $tmp_fname != {} } {
				    set mofpar(outfl) $tmp_fname
				}
			    }]
    
    


    grid columnconfigure $subframe3_w 0 -minsize 456
    grid columnconfigure $subframe3_w 1 -minsize 25
    
    grid $outfl_w        -row 0 -column 0 -sticky e -pady 0
    grid $outfl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    
    set subframe4_w [frame $output_frame_w.subfr4]

    set number_outfl_label_w [iwidgets::labeledwidget $subframe4_w.label_number_outfl \
				  -labeltext "Number of output files "]
    number_modflow_output_files
    set number_outfl_value_w [iwidgets::labeledwidget $subframe4_w.value_number_outfl \
				  -labelvariable mofpar(noutfl)]
    set outfl_sname_w [iwidgets::labeledwidget $subframe4_w.sname_outfl \
				  -labelvariable mofpar(sname)]


    grid columnconfigure $subframe4_w 0 -minsize 47
    grid columnconfigure $subframe4_w 1 -minsize 45
    grid columnconfigure $subframe4_w 2 -minsize 200
    grid $number_outfl_label_w -row 0 -column 0 -sticky e -pady 0
    grid $number_outfl_value_w -row 0 -column 1 -sticky w -pady 0
    grid $outfl_sname_w        -row 0 -column 2 -sticky w -pady 0

    
    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	$subframe4_w \
	-anchor w -expand 1 -padx 3 -pady 0
    
    
    pack \
	$data_iframe_w \
	$characterization_iframe_w \
	$output_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x
    
    
    pack \
	$work_frame_w \
	-anchor nw -side top -fill x -fill y -expand true \
	-padx 0 -pady 0
    
    
    $export_modflow_box_w.export_modflow_tbn view "Files"    
    
    
    #############################################################################
    # Default buttons
    #############################################################################
    
    set button_frame_w [frame $export_modflow_box_w.button_frame]
    
    set buttons_iframe_w [iwidgets::labeledframe $button_frame_w.buttons_iframe \
			      -labeltext "Defaults" \
			      -labelpos nw \
			      -borderwidth 2 \
			      -relief groove]
    set buttons_frame_w [$buttons_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set open_button_w [iwidgets::pushbutton $buttons_iframe_w.open \
			   -height 27 \
			   -width 40 \
			   -defaultringpad 0 \
			   -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			   -command {
			       set tmp_fname [tk_getOpenFile \
						  -initialdir [file dirname $mofpar(datafl)] \
						  -defaultextension ".par" \
						  -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
						  -title "Select EXPORT MODFLOW parameter file"]
			       
			       if { $tmp_fname != {} } {
				   set param_fname $tmp_fname
				   read_export_modflow_params $param_fname
			       }
			   }]
    
    
    set save_button_w [iwidgets::pushbutton $buttons_iframe_w.save \
			   -height 27 \
			   -width 27 \
			   -defaultringpad 0 \
			   -state disabled \
			   -image [image create photo -file [file join "$exedir/images/save.gif"]] \
			   -command {
			       write_export_modflow_params $param_fname
			   }]
    
    
    set save_as_button_w [iwidgets::pushbutton $buttons_iframe_w.save_as \
			      -height 27 \
			      -width 27 \
			      -defaultringpad 0 \
			      -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			      -command {
				  if { $mofpar(vartype) == "Categorical" } {
				      if { $mofpar(datafl) == {*.out} && $mofpar(inbcffl) == {*.dat} } {
					  tk_messageBox \
					      -title "Defaults" \
					      -icon warning \
					      -message "You should select a simulation input file and a bcf input file first." \
					      -type ok
				      } elseif { $mofpar(datafl) == {*.out} && $mofpar(inbcffl) != {*.dat} } {
					  tk_messageBox \
					      -title "Defaults" \
					      -icon warning \
					      -message "You should select a simulation input file first." \
					      -type ok
				      } elseif { $mofpar(datafl) != {*.out} && $mofpar(inbcffl) == {*.dat} } {
					  tk_messageBox \
					      -title "Defaults" \
					      -icon warning \
					      -message "You should select a bcf input file first." \
					      -type ok
				      } elseif { $mofpar(datafl) != {*.out} && $mofpar(inbcffl) != {*.dat} } {
					  if { $mofpar(charfl) == {*.gtd} } {
					      tk_messageBox \
						  -title "Defaults" \
						  -icon warning \
						  -message "You should select a characterization input file first." \
						  -type ok
					  } elseif { $mofpar(charfl) != {*.gtd} } {
					      if { $mofpar(charvar) < [expr $mofpar(ncat) + 1] } {
						  tk_messageBox \
						      -title "Characterization Input" \
						      -icon warning \
						      -message "The variables characterization file has to few data.\nYou should select another file." \
						      -type ok
					      } elseif { $mofpar(charvar) >= [expr $mofpar(ncat) + 1] } {
						  set tmp_fname [tk_getSaveFile \
								     -initialdir [file dirname $mofpar(datafl)] \
								     -defaultextension ".par" \
								     -filetypes {{{Parameter Files} {.par}} \
										     {{All Files} {.*}}}\
								     -title "Save EXPORT MODFLOW parameter file as ..."]
						  
						  $save_button_w configure -state normal
						  
						  if { $tmp_fname != {} } {
						      set param_fname $tmp_fname
						      write_export_modflow_params $param_fname
						  }
					      }
					  }
				      }
				  } elseif { $mofpar(vartype) == "Continous" } {
				      if { $mofpar(datafl) == {*.out} && $mofpar(inbcffl) == {*.dat} } {
					  tk_messageBox \
					      -title "Defaults" \
					      -icon warning \
					      -message "You should select a simulation input file and a bcf input file first." \
					      -type ok
				      } elseif { $mofpar(datafl) == {*.out} && $mofpar(inbcffl) != {*.dat} } {
					  tk_messageBox \
					      -title "Defaults" \
					      -icon warning \
					      -message "You should select a simulation input file first." \
					      -type ok
				      } elseif { $mofpar(datafl) != {*.out} && $mofpar(inbcffl) == {*.dat} } {
					  tk_messageBox \
					      -title "Defaults" \
					      -icon warning \
					      -message "You should select a bcf input file first." \
					      -type ok
				      } elseif { $mofpar(datafl) != {*.out} && $mofpar(inbcffl) != {*.dat} } {
					  if { $mofpar(charfl) != {*.gtd} } {
					      if { $mofpar(charvar) < [expr $mofpar(ncat) + 1] } {
						  tk_messageBox \
						      -title "Characterization Input" \
						      -icon warning \
						      -message "The variables characterization file has to few data, although the data are not used actually.\nYou should select another file." \
						      -type ok
					      } elseif { $mofpar(charvar) >= [expr $mofpar(ncat) + 1] } {
						  set tmp_fname [tk_getSaveFile \
								     -initialdir [file dirname $mofpar(datafl)] \
								     -defaultextension ".par" \
								     -filetypes {{{Parameter Files} {.par}} \
										     {{All Files} {.*}}}\
								     -title "Save EXPORT MODFLOW parameter file as ..."]
						  
						  $save_button_w configure -state normal
						  
						  if { $tmp_fname != {} } {
						      set param_fname $tmp_fname
						      write_export_modflow_params $param_fname
						  }
					      }
					  } elseif { $mofpar(charfl) == {*.gtd} } {
					      set tmp_fname [tk_getSaveFile \
								 -initialdir [file dirname $mofpar(datafl)] \
								 -defaultextension ".par" \
								 -filetypes {{{Parameter Files} {.par}} \
										 {{All Files} {.*}}}\
								 -title "Save EXPORT MODFLOW parameter file as ..."]
					      
					      $save_button_w configure -state normal
					      
					      if { $tmp_fname != {} } {
						  set param_fname $tmp_fname
						  write_export_modflow_params $param_fname
					      }
					  }
				      }
				  }
			      }]
    
    
    set cancel_button_w [iwidgets::pushbutton $buttons_iframe_w.cancel \
			     -height 27 \
			     -width 27 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/stop.gif"]] \
			     -command {
				 set private(button) 0
			     }]
    
    
    set run_button_w [iwidgets::pushbutton $buttons_iframe_w.run \
			  -height 27 \
			  -width 27 \
			  -defaultringpad 0 \
			  -image [image create photo -file [file join "$exedir/images/run.gif"]] \
			  -command {

			      ### set the cursor on watch
			      . configure -cursor watch
			      update
			      
			      ### messages, if there are no data or partly no data
			      if { $mofpar(datafl) == {*.out} && $mofpar(inbcffl) == {*.dat} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "There is no simulation input file and no bcf input file.\nYou should select the corresponding files first." \
				      -type ok
			      } elseif { $mofpar(datafl) != {*.out} && $mofpar(inbcffl) == {*.dat} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "There is no bcf input file.\nYou should select a corresponding file first." \
				      -type ok
			      } elseif { $mofpar(datafl) == {*.out} && $mofpar(inbcffl) != {*.dat} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "There is no simulation input file.\nYou should select a corresponding file first." \
				      -type ok
			      } elseif { $mofpar(datafl) != {*.out} && $mofpar(inbcffl) != {*.dat} } { ;# data available
				  
				  ### open the input bcf file for counting the number of modflow layers
				  if { [catch {set bcff [open $mofpar(inbcffl) {r}]} err_msg] } {
				      tk_messageBox \
					  -title "File open error" \
					  -icon error \
					  -message "The file $fname couldn't open!\n$err_msg"
				  }
				  export_modflow_layer_counter
				  close $bcff
				  
				  ### message, if the number of the modflow layers do not correspond 
				  ### with the number of the simulated layers
				  if { $mofpar(nz) != $nlayer || $mofpar(nz) != $mofpar(modnz) || \
					   $nlayer != $mofpar(modnz) } {
				      tk_messageBox \
					  -title "MODFLOW layers" \
					  -icon error \
					  -message "The number of MODFLOW layers is not equal to the number of simulated layers.\nYou should check the corresponding details, and, if necessary, choose other files."
				  } elseif { $mofpar(nz) == $nlayer && $mofpar(nz) == $mofpar(modnz) && \
					   $nlayer == $mofpar(modnz) } { ;# ok, the numbers of the layers are equal

				      ### check, if modflow x-y-grid and simulated x-y-grid is equal
				      if { $ny != 0 } {
					  if { $mofpar(ny) != $ny || $mofpar(ny) != $mofpar(modny) || \
						   $ny != $mofpar(modny) || $mofpar(nx) != $nx || \
						   $mofpar(nx) != $mofpar(modnx) || $nx != $mofpar(modnx) } {
					      tk_messageBox \
						  -title "MODFLOW x-y-grid" \
						  -icon error \
						  -message "The number of MODFLOW x-y-grid cells is not equal to the number of simulated x-y-grid cells.\nYou should check the corresponding details, and, if necessary, choose other files."
					  } elseif { $mofpar(ny) == $ny && $mofpar(ny) == $mofpar(modny) && \
							 $ny == $mofpar(modny) && $mofpar(nx) == $nx && \
							 $mofpar(nx) == $mofpar(modnx) && $nx == $mofpar(modnx) } {

					      ### assignment of characterization data depending on the variable type
					      if { $mofpar(vartype) == "Continous" } {
						  
						  ### test, if the characterization file has enough data
						  if { $mofpar(charfl) != {*.gtd} } {
						      if { $mofpar(charvar) < [expr $mofpar(ncat) + 1] } {
							  tk_messageBox \
							      -title "Characterization Input" \
							      -icon warning \
							      -message "The variables characterization file has to few data, although the data are not used actually.\nYou should select another file." \
							      -type ok
						      } elseif { $mofpar(charvar) >= [expr $mofpar(ncat) + 1] } {
							  ;# ok, enough data
							  
							  ### message, if the data are not 2D or 3D
							  if { $mofpar(nx) == 1 || $mofpar(ny) == 1 || $mofpar(nz) == 1 } {
							      tk_messageBox \
								  -title "Dimension of input data" \
								  -icon error \
								  -message "The input data are not 3D.\nYou should select another simulation input file."
							  } elseif { $mofpar(nx) != 1 && $mofpar(ny) != 1 \
									 && $mofpar(nz) != 1 } { ;# ok, 3D data
							      
							      ### write the temporary parameter file
							      set paramfile "tmprun.par"
							      write_export_modflow_params_run $paramfile
							      update
							      
							      ### restruct data from the simulated realizations file
							      log_newline
							      log_newline
							      log "Restructuring:\n"
							      log "Restructuring of simulated values."
							      geossav_convert $mofpar(datafl) "confl.out" \
								  $mofpar(nx) $mofpar(ny) $mofpar(nz) $mofpar(nsim)
							      update
							      
							      ### open the simulated realizations file
							      if { [catch {set inf [open "confl.out" {r}]} err_msg] } {
								  tk_messageBox \
								      -title "File open error" \
								      -icon error \
								      -message "The file $fname couldn't open!\n$err_msg"
							      }
							      
							      ### message, if vertical K values file can not be open
							      log_newline
							      log_newline
							      log "Vertical K value:\n"
							      log "Calculation of vertical K values."
							      update
							      if { [catch {set kvf \
									       [open "kvfl.out" {w+}]} err_msg] } {
								  tk_messageBox \
								      -title "File open error" \
								      -icon error \
								      -message "The file $fname couldn't open!\n$err_msg"
							      }
							      export_modflow_vertical_k_continous
							      close $kvf

							      ### file copy vertical K values
							      catch {
								  file copy -force "kvfl.out" "ckvfl.out"
							      }

							      ### open the vertical K values file again
							      if { [catch {set kvf \
									       [open "kvfl.out" {r}]} err_msg] } {
								  tk_messageBox \
								      -title "File open error" \
								      -icon error \
								      -message "The file $fname couldn't open!\n$err_msg"
							      }

							      ### open the copied vertical K values file
							      if { [catch {set ckvf \
									       [open "ckvfl.out" {r}]} err_msg] } {
								  tk_messageBox \
								      -title "File open error" \
								      -icon error \
								      -message "The file $fname couldn't open!\n$err_msg"
							      }

							      ### open the input bcf file
							      if { [catch {set bcff \
									       [open $mofpar(inbcffl) {r}]} err_msg] } {
								  tk_messageBox \
								      -title "File open error" \
								      -icon error \
								      -message "The file $fname couldn't open!\n$err_msg"
							      }
							      
							      ### message, if the output file/s cannot be open
							      log_newline
							      log_newline
							      log "Exporting data:\n"
							      for { set iv 1 } { $iv <= $mofpar(nsim) } { incr iv } {
								  if { [catch {set outfcon($iv) [open "$mofpar(outfl)_s$iv" \
												  {w}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  export_modflow_continous_variables $iv
								  log "Calculation of leakance and exporting of data: simulation $iv"
								  update
								  close $outfcon($iv)
							      }
							      close $inf
							      close $bcff
							      close $kvf
							      close $ckvf
							      log_newline
							      log_newline
							      log "EXPORT MODFLOW finished.\n\n"
							      
							      ### delete the files/temporary files
							      catch {
								  file delete -force \
								      $paramfile \
								      "confl.out" \
								      "kvfl.out" \
								      "ckvfl.out"
							      }
							      set private(button) 1
							  }
						      }
						  } elseif { $mofpar(charfl) == {*.gtd} } {
					      
						      ### message, if the data are not 2D or 3D
						      if { $mofpar(nx) == 1 || $mofpar(ny) == 1 || $mofpar(nz) == 1 } {
							  tk_messageBox \
							      -title "Dimension of input data" \
							      -icon error \
							      -message "The input data are not 3D.\nYou should select another simulation input file."
						      } elseif { $mofpar(nx) != 1 && $mofpar(ny) != 1 \
								     && $mofpar(nz) != 1 } { ;# ok, 3D data
							  
							  ### write the temporary parameter file
							  set paramfile "tmprun.par"
							  write_export_modflow_params_run $paramfile
							  update

							  ### restruct data from the simulated realizations file
							  log_newline
							  log_newline
							  log "Restructuring:\n"
							  log "Restructuring of simulated values."
							  geossav_convert $mofpar(datafl) "confl.out" \
							      $mofpar(nx) $mofpar(ny) $mofpar(nz) $mofpar(nsim)
							  update
							  
							  ### open the simulated realizations file
							  if { [catch {set inf [open "confl.out" {r}]} err_msg] } {
							      tk_messageBox \
								  -title "File open error" \
								  -icon error \
								  -message "The file $fname couldn't open!\n$err_msg"
							  }
							  
							  ### message, if the k_relation output file can not be open
							  log_newline
							  log_newline
							  log "Vertical K value:\n"
							  log "Calculation of vertical K values."
							  update
							  if { [catch {set kvf \
									   [open "kvfl.out" {w+}]} err_msg] } {
							      tk_messageBox \
								  -title "File open error" \
								  -icon error \
								  -message "The file $fname couldn't open!\n$err_msg"
							  }
							  export_modflow_vertical_k_continous
							  close $kvf

							  ### file copy vertical K values
							  catch {
							      file copy -force "kvfl.out" "ckvfl.out"
							  }
							  
							  ### open the vertical K values file again
							  if { [catch {set kvf \
									   [open "kvfl.out" {r}]} err_msg] } {
							      tk_messageBox \
								  -title "File open error" \
								  -icon error \
								  -message "The file $fname couldn't open!\n$err_msg"
							  }
							  
							  ### open the copied vertical K values file
							  if { [catch {set ckvf \
									   [open "ckvfl.out" {r}]} err_msg] } {
							      tk_messageBox \
								  -title "File open error" \
								  -icon error \
								  -message "The file $fname couldn't open!\n$err_msg"
							  }

							  ### open the input bcf file
							  if { [catch {set bcff \
									   [open $mofpar(inbcffl) {r}]} err_msg] } {
							      tk_messageBox \
								  -title "File open error" \
								  -icon error \
								  -message "The file $fname couldn't open!\n$err_msg"
							  }
							  
							  ### message, if the output file/s cannot be open
							  log_newline
							  log_newline
							  log "Exporting data:\n"
							  for { set iv 1 } { $iv <= $mofpar(nsim) } { incr iv } {
							      if { [catch {set outfcon($iv) [open "$mofpar(outfl)_s$iv" \
											      {w}]} err_msg] } {
								  tk_messageBox \
								      -title "File open error" \
								      -icon error \
								      -message "The file $fname couldn't open!\n$err_msg"
							      }
							      export_modflow_continous_variables $iv
							      log "Calculation of leakance and exporting of data: simulation $iv"
							      update
							      close $outfcon($iv)
							  }
							  close $inf
							  close $bcff
							  close $kvf
							  close $ckvf
							  log_newline
							  log_newline
							  log "EXPORT MODFLOW finished.\n\n"
							      
							  ### delete the files/temporary files
							  catch {
							      file delete -force \
								  $paramfile \
								  "confl.out" \
								  "kvfl.out" \
								  "ckvfl.out"
							  }
							  set private(button) 1
						      }
						  }
					      } elseif { $mofpar(vartype) == "Categorical" } {
				      
						  ### message, if there is no characterization input file
						  if { $mofpar(charfl) == {*.gtd} } {
						      tk_messageBox \
							  -title "Defaults" \
							  -icon warning \
							  -message "You should select a characterization input file first." \
							  -type ok
						  } elseif { $mofpar(charfl) != {*.gtd} } {
						      ### message, if the characterization file has to few data
						      if { $mofpar(charvar) < [expr $mofpar(ncat) + 1] } {
							  tk_messageBox \
							      -title "Characterization Input" \
							      -icon warning \
							      -message "The variables characterization file has to few data.\nYou should select another file." \
							      -type ok
						      } elseif { $mofpar(charvar) >= [expr $mofpar(ncat) + 1] } {
							  
							  ### message, if the data are not 2D or 3D
							  if { $mofpar(nx) == 1 || $mofpar(ny) == 1 || $mofpar(nz) == 1 } {
							      tk_messageBox \
								  -title "Dimension of input data" \
								  -icon error \
								  -message "The input data are not 3D.\n You should select another input data file."
							  } elseif { $mofpar(nx) != 1 && $mofpar(ny) != 1 && \
									 $mofpar(nz) != 1 } {
							      
							      ### replace the categorical variables with 
							      ### the mean K values
							      if { $mofpar(mean) == 1 } {
								  
								  ### write the temporary parameter file
								  set paramfile "tmprun.par"
								  write_export_modflow_params_run $paramfile
								  update

								  ### restruct data from the simulated realizations file
								  log_newline
								  log_newline
								  log "Restructuring:\n"
								  log "Restructuring of simulated values."
								  geossav_convert $mofpar(datafl) "confl.out" \
								      $mofpar(nx) $mofpar(ny) $mofpar(nz) $mofpar(nsim)
								  update
								  
								  ### open the simulated realizations file
								  if { [catch {set inf [open "confl.out" {r}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  
								  ### open the input bcf file
								  if { [catch {set bcff [open $mofpar(inbcffl) {r}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  
								  ### open the variable characterization file
								  if { [catch {set charf \
										   [open $mofpar(charfl) {r}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  
								  ### message, if vertical K values file can not be open 
								  if { [catch {set kvf \
										   [open "kvfl.out" {w+}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }

								  ### open the replacement file and replace 
								  ### the categorical variables with mean values
								  ### and calculation of vertcal K values
								  log_newline
								  log_newline
								  log "Characterization:\n"
								  log "Characterization of simulated categories with mean values and calculation of vertical K values."
								  update
								  if { [catch {set repf [open "repfl.out" {w+}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  mean_value_modflow_replacement
								  close $inf
								  close $charf
								  close $kvf

								  ### file copy vertical K values
								  catch {
								      file copy -force "kvfl.out" "ckvfl.out"
								  }
								  
								  ### open the vertical K values file again
								  if { [catch {set kvf \
										   [open "kvfl.out" {r}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  
								  ### open the copied vertical K values file
								  if { [catch {set ckvf \
										   [open "ckvfl.out" {r}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  
								  ### message, if the output file/s cannot be open
								  log_newline
								  log_newline
								  log "Exporting of data:\n"
								  for { set iv 1 } { $iv <= $mofpar(nsim) } { incr iv } {
								      if { [catch {set outfcatm($iv) \
										       [open "$mofpar(outfl)_s$iv,_c1" \
											    {w}]} err_msg] } {
									  tk_messageBox \
									      -title "File open error" \
									      -icon error \
									      -message "The file $fname couldn't open!\n$err_msg"
								      }
								      export_modflow_categorical_variables_mean $iv
								      log "Calculation of leakance and exporting of data: simulation $iv, characterization 1"
								      update
								      close $outfcatm($iv)
								  }
								  close $bcff
								  close $repf
								  close $kvf
								  close $ckvf
								  log_newline
								  log_newline
								  log "EXPORT MODFLOW finished.\n\n"
								  
								  ### delete the files/temporary files
								  catch {
								      file delete -force \
									  $paramfile \
									  "confl.out" \
									  "repfl.out" \
									  "kvfl.out" \
									  "ckvfl.out"
								  }
							      } elseif { $mofpar(mean) == 0 } {
								  ;# replace the categorical variables with values from 
								  ;# the distribution of a parameter
								  
								  ### write the temporary parameter file
								  set paramfile "tmprun.par"
								  write_export_modflow_params_run $paramfile
								  update

								  ### restruct data from the simulated realizations file
								  log_newline
								  log_newline
								  log "Restructuring:\n"
								  log "Restructuring of simulated values."
								  geossav_convert $mofpar(datafl) "confl.out" \
								      $mofpar(nx) $mofpar(ny) $mofpar(nz) $mofpar(nsim)
								  update

								  ### open the simulated realizations file
								  if { [catch {set inf \
										   [open "confl.out" {r}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  
								  ### open the input bcf file
								  if { [catch {set bcff \
										   [open \
											$mofpar(inbcffl) {r}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }

								  ### open the variable characterization file
								  if { [catch {set charf \
										   [open $mofpar(charfl) {r}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  
								  ### message, if the vertical K values files 
								  ### can not be open
								  for { set i 1 } { $i <= $mofpar(nsimchar) } { incr i } {
								      if { [catch {set kvfcatd($i) [open "kvfl.out$i" {w+}]} err_msg] } {
									  tk_messageBox \
									      -title "File open error" \
									      -icon error \
									      -message "The file $fname couldn't open!\n$err_msg"
								      }
								  }

								  ### open the replacement files and replace the catego-
								  ### rical variables with values from the distribution
								  ### and calculation of vertical K values
								  log_newline
								  log_newline
								  log "Characterization:\n"
								  log "Characterization of simulated categories with values from the distributions and calculation of vertical K values."
								  update
								  for { set i 1 } { $i <= $mofpar(nsimchar) } { incr i } {
								      if { [catch {set repfcatd($i) [open "repfl.out$i" {w+}]} err_msg] } {
									  tk_messageBox \
									      -title "File open error" \
									      -icon error \
									      -message "The file $fname couldn't open!\n$err_msg"
								      }
								      distribution_value_modflow_replacement $i
								  }
								  close $inf
								  close $charf
								  for { set i 1 } { $i <= $mofpar(nsimchar) } { incr i } {
								      close $kvfcatd($i)
								  }

								  ### files copy vertical K values
								  for { set i 1 } { $i <= $mofpar(nsimchar) } { incr i } {
								      catch {
									  file copy -force "kvfl.out$i" "ckvfl.out$i"
								      }
								  }

								  ### open the vertical K values files again
								  for { set i 1 } { $i <= $mofpar(nsimchar) } { incr i } {
								      if { [catch {set kvfcatd($i) \
										       [open "kvfl.out$i" {r}]} err_msg] } {
									  tk_messageBox \
									      -title "File open error" \
									      -icon error \
									      -message "The file $fname couldn't open!\n$err_msg"
								      }
								  }
								  
								  ### open the copied vertical K values files
								  for { set i 1 } { $i <= $mofpar(nsimchar) } { incr i } {
								      if { [catch {set ckvfcatd($i) \
										       [open "ckvfl.out$i" {r}]} err_msg] } {
									  tk_messageBox \
									      -title "File open error" \
									      -icon error \
									      -message "The file $fname couldn't open!\n$err_msg"
								      }
								  }

								  ### message, if the output file/s cannot be open
								  log_newline
								  log_newline
								  log "Exporting data:\n"
								  for { set viii 1 } { $viii <= $mofpar(nsimchar) } { incr viii }  {
								      for { set vii 1 } { $vii <= $mofpar(nsim) } { incr vii } {
									  if { [catch {set outfcatd($vii,$viii) [open "$mofpar(outfl)_s$vii,_c$viii" {w}]} err_msg] } {
									      tk_messageBox \
										  -title "File open error" \
										  -icon error \
										  -message "The file $fname couldn't open!\n$err_msg"
									  }
									  export_modflow_categorical_variables_distribution $vii $viii
									  log "Calculation of leakance and exporting of data: simulation $vii, characterization $viii"
									  update
									  close $outfcatd($vii,$viii)
								      }
								      close $repfcatd($viii)
								      close $kvfcatd($viii)
								      close $ckvfcatd($viii)
								  }
								  close $bcff
								  log_newline
								  log_newline
								  log "EXPORT MODFLOW finished.\n\n"

								  ### delete the files/temporary files
								  catch {
								      file delete -force \
									  $paramfile \
									  "confl.out"
								  }
								  for { set i 1 } { $i <= $mofpar(nsimchar) } { incr i } {
								      catch {
									  file delete -force \
									      "repfl.out$i" \
									      "kvfl.out$i" \
									      "ckvfl.out$i"
								      }
								  }
							      }
							      set private(button) 1  
							  }
						      }
						  }
					      }
					  }
				      }
				  }
			      }
			      ### reset the cursor on arrow 
			      . configure -cursor arrow
			      update
			  }]
				  
    bind $run_button_w <Return> "
	    $run_button_w configure -state active -relief sunken
	    update idletasks
	    after 100
	    $run_button_w invoke
	"
    
    grid columnconfigure $button_frame_w 0 -minsize 15
    grid columnconfigure $button_frame_w 1 -minsize 100
    grid columnconfigure $button_frame_w 2 -minsize 15
    grid columnconfigure $button_frame_w 3 -minsize 100
    grid columnconfigure $button_frame_w 4 -minsize 15
    grid columnconfigure $button_frame_w 5 -minsize 100
    grid columnconfigure $button_frame_w 6 -minsize 15
    grid columnconfigure $button_frame_w 7 -minsize 100
    grid columnconfigure $button_frame_w 8 -minsize 15
    grid columnconfigure $button_frame_w 9 -minsize 100
    grid columnconfigure $button_frame_w 10 -minsize 15
    
    grid $open_button_w    -row 0 -column 1 -sticky we -pady 8 -padx 5
    grid $save_button_w    -row 0 -column 3 -sticky we -pady 8 -padx 5
    grid $save_as_button_w -row 0 -column 5 -sticky we -pady 8 -padx 5
    grid $cancel_button_w  -row 0 -column 7 -sticky we -pady 8 -padx 5
    grid $run_button_w     -row 0 -column 9 -sticky we -pady 8 -padx 5
    
    
    pack \
	$buttons_iframe_w \
	-padx 5 -pady 5 -expand 1
    
    
    pack $button_frame_w \
	-side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    raise $export_modflow_box_w
    
#    tkwait visibility $export_modflow_box_w
    tkwait visibility $run_button_w
    
    set oldFocus [focus]
    set oldGrab [grab current $export_modflow_box_w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $export_modflow_box_w
    focus $run_button_w
    
    . configure -cursor arrow
    update
    
    update idletask
    tkwait variable private(button)
    
    catch {focus $oldFocus}
    catch {
	# It's possible that the window has already been destroyed,
	# hence this "catch".  Delete the Destroy handler so that
	# tkPriv(button) doesn't get reset by it.
	
	bind $export_modflow_box_w <Destroy> {}
	destroy $export_modflow_box_w
    }
    
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }   
    return
}
#============================================================================
#=END=OF=EXPORT_MODFLOW======================================================
