##START#OF#INPUT_VISUALIZATION###############################################
# geossav_input_visualization.tcl
#
# Source script creating the input_visualization widget in GEOSSAV, and running 
# the rendering of data. The called c/c++ procs are in the geossav.c file.
# Developed at the University of Basel, Switzerland,
# by Christian Regli and Lukas Rosenthaler
#
#############################################################################


#############################################################################
#
# The following set of parameters are required for the INPUT_VISUALIZATION
#
global exedir
source $exedir/Geossav_input_visualization_globals.tcl

set inpar(datafl) "*.dat"                     ;# input data file
set inpar(pos_x) 1                            ;# column number for the x coordinates
set inpar(pos_y) 2                            ;# column number for the y coordinates
set inpar(pos_z) 3                            ;# column number for the z coordinates
set inpar(pos_var) 4                          ;# column number for the variable
set inpar(parameters) 0                       ;# number of data variables excluding the variable of None
set inpar(cancel_counter) 0                   ;# counter for the exit from the INPUT DATA parameters tabnotebook
set inpar(nvis) 0                             ;# number of visualizations
set inpar(visvar) 1                           ;# visualization of variables, 1=yes, 0=no
set inpar(ballsize) 10

# set inpar(nx) 100                             ;# number of grid cells (x axis)
# set inpar(ny) 100                             ;# number of grid cells (y axis)
# set inpar(nz) 10                              ;# number of grid cells (y axis)
# set inpar(nsim) 1                             ;# number of simulations
# set inpar(vartype) 0                          ;# variable type: 1=continuous (cdf), 0=categorical (pdf)
# set inpar(xsiz) 10                            ;# size of grid cell in x-direction
# set inpar(ysiz) 10                            ;# size of grid cell in y-direction
# set inpar(zsiz) 10                            ;# size of grid cell in z-direction
# set inpar(ncat) 3                             ;# number of thresholds or categories
# set inpar(cat) 1                              ;# threshold values or category codes
# set inpar(infofl) "*.out"                     ;# information file to the realization file, simplified Geo-EAS format
# set inpar(xy_plane) 1                         ;# selection of xy plane: 1 yes, 0 no
# set inpar(xz_plane) 0                         ;# selection of xz plane: 1 yes, 0 no
# set inpar(yz_plane) 0                         ;# selection of yz plane: 1 yes, 0 no



# set params3d(xy_nn) 0
# set params3d(xz_nn) 0
# set params3d(yz_nn) 0
# set params3d(plane) "XY"
# set 3d_select(xy) 0
# set 3d_select(xz) 0
# set 3d_select(yz) 0
#
# End of initialization of variables. This has to be done once at the start 
# of the programm.
#
#############################################################################


#############################################################################
#
# Start of proc section.
#

#############################################################################
proc read_indata { fname } {
    global dim_name_indata
    global inpar

    lappend dim_name_indata "None"
    if { [file readable $fname] } {	
	set inf [open $fname "r"]
	gets $inf
	gets $inf n
	for { set i 0 } { $i < $n } { incr i } {
	    gets $inf dum
	    lappend dim_name_indata $dum
	    set tmp_parameters [expr $i + 1]
	}
	close $inf
	set inpar(parameters) $tmp_parameters
	return 1
    } else {
	return 0
    }
}
#============================================================================


# #############################################################################
 proc oglswap { widget } {
#     global params3d
#     global 3d_select

     $widget activate
#     set cmdstr "geossav_3dview 1"
#     if { $3d_select(xy) } {
# 	append cmdstr " XY " $params3d(xy_nn)
#     }
#     if { $3d_select(xz) } {
# 	append cmdstr " XZ " $params3d(xz_nn)
#     }
#     if { $3d_select(yz) } {
# 	append cmdstr " YZ " $params3d(yz_nn)
#     }
#     eval $cmdstr

     $widget swapbuffers
# }
# #============================================================================


#############################################################################
proc read_input_visualization_params { fname } {
    global inpar
    global input_visualization_box_w
    global datafl_w
    global pos_x_w
    global pos_y_w
    global pos_z_w
    global pos_var_w
    global dim_name_indata
    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }
    
    ### sets the cursor on watch as long as parameters are read
    . configure -cursor watch
    update
    
    ### searches the line "parameters for input_visualization" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Parameters for INPUT_DATA VISUALIZATION" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File selection error" \
		-icon error \
		-message "You have selected a wrong parameter file.\n"
	    return 0
	}
    }

    ### set file name in input_data visualization box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $input_visualization_box_w "INPUT DATA parameters   -   $fname_short"

    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }

    ### reads name of datafl: selected file for input_data visualization
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set inpar(datafl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set inpar(datafl) [format "%s" $line]
    }
    $datafl_w configure -state normal

    ### reads the pos_x, ..y, ..z, ..var: columns for x, y, z, variable (indata)
    ### and fills the comboboxes
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_inpar(pos_x) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_inpar(pos_y) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_inpar(pos_z) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_inpar(pos_var) [format "%d" [string range $line 0 [expr $space - 1]]]
    
    unset dim_name_indata
    if { [read_indata $inpar(datafl)] != 1 } {
	set dim_name_indata {}
	lappend dim_name_indata "None"
	for { set i 1 } { $i < 5 } { incr i } {
	    lappend dim_name_indata $i
	}
    }
    
    $pos_x_w configure -state normal
    $pos_x_w clear list
    foreach i $dim_name_indata {
	$pos_x_w insert list end $i
    }
    $pos_x_w selection set $tmp_inpar(pos_x)

    $pos_y_w configure -state normal
    $pos_y_w clear list
    foreach i $dim_name_indata {
	$pos_y_w insert list end $i
    }
    $pos_y_w selection set $tmp_inpar(pos_y)
    
    $pos_z_w configure -state normal
    $pos_z_w clear list
    foreach i $dim_name_indata {
	$pos_z_w insert list end $i
    }
    $pos_z_w selection set $tmp_inpar(pos_z)
    
    $pos_var_w configure -state normal
    $pos_var_w clear list
    foreach i $dim_name_indata {
	$pos_var_w insert list end $i
    }
    $pos_var_w selection set $tmp_inpar(pos_var)

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc write_input_visualization_params { fname } {
    global tcl_platform
    global version
    global inpar
    global input_visualization_box_w
    global pos_x_w
    global pos_y_w
    global pos_z_w
    global pos_var_w

    ### message, if file cannot be open
    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }

    ### sets the cursor on watch as long as parameters are written
    . configure -cursor watch
    update
    
    ### set file name in input_visualization box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $input_visualization_box_w "INPUT DATA parameters   -   $fname_short"

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    
    puts $outf "Parameters for INPUT_DATA VISUALIZATION"
    puts $outf "***************************************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### datafl: selected file for input_file visualization
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $inpar(datafl)]
    set line [format "%s" $inpar(datafl)]
    set line_length_total [string length $line]
    set comment "\\input_file visualization"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr ($parameter_space - 0) - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### pos_x, ..y, ..z, ..var: columns for x, y, z, variable
    puts -nonewline $outf [format "%d %d " [$pos_x_w curselection] [$pos_y_w curselection]]
    puts -nonewline $outf [format "%d %d" [$pos_z_w curselection] [$pos_var_w curselection]]
    set line [format "%d %d %d %d" [$pos_x_w curselection] [$pos_y_w curselection] \
		  [$pos_z_w curselection] [$pos_var_w curselection]]
    set line_length [string length $line]
    set comment "\\pos_x, ..y, ..z, ..var: columns for x, y, z, variable"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update

    ### close the file
    close $outf
}
#============================================================================


# #############################################################################
# proc select_var { plane } {
#     global inpar
#     global visframe
#     global image_item
#     global params3d
#     global sim_select
#     global 3d_select

#     set cmdstr "geossav_3dview 1"
#     set params3d(xy_nn) $plane
#     if { $3d_select(xy) } {
# 	append cmdstr " XY " $params3d(xy_nn)
#     }
#     if { $3d_select(xz) } {
# 	append cmdstr " XZ " $params3d(xz_nn)
#     }
#     if { $3d_select(yz) } {
# 	append cmdstr " YZ " $params3d(yz_nn)
#     }
#     $visframe.subfr_1.mframe.ogl activate
#     eval $cmdstr
#     $visframe.subfr_1.mframe.ogl swapbuffers

#     if { $sim_select == 1 } {
# 	geossav_photo "dplane" 1 $plane 400 "XY"
#     }
#     update
# }
# #============================================================================


#############################################################################
proc geossav_in_button1_press { x y } {
    global old_x old_y

    set old_x $x
    set old_y $y
}
#============================================================================


#############################################################################
proc geossav_in_button1_motion { x y } {
    global old_x old_y
    global sp
    global visframe
    global params3d
    global 3d_select
    global inpar

    set view_changed 0
    if { $old_x != $x } {
	set sp(longitude) [expr $sp(longitude) - (($x - $old_x)/5.0)]
	geossav_3dsp $sp(longitude) $sp(latitude) $sp(dist)
	set view_changed 1
	set old_x $x
    }
    if { $old_y != $y } {
	set sp(latitude) [expr $sp(latitude) + (($y - $old_y)/5.0)]
	geossav_3dsp $sp(longitude) $sp(latitude) $sp(dist)
	set view_changed 1
	set old_y $y
    }

    if { $view_changed } {
	$visframe.subfr_1.mframe.ogl activate
	set cmdstr "geossav_3dview_input $inpar(pos_x) $inpar(pos_y) $inpar(pos_z) $inpar(pos_var) $inpar(ballsize)"
	eval $cmdstr
	$visframe.subfr_1.mframe.ogl swapbuffers
    }
    
}
#============================================================================


#############################################################################
proc geossav_in_button3_press { x y } {
    global old3_x old3_y

    set old3_x $x
    set old3_y $y
}
#============================================================================


#############################################################################
proc geossav_in_button3_motion { x y } {
    global old3_x old3_y
    global sp
    global visframe
    global params3d
    global 3d_select
    global inpar

    set view_changed 0
    if { $old3_y != $y } {
	set diff [expr ($y - $old3_y)/50.0]
	if { $diff < -0.8 } {
	    set diff 0.8
	}
	set sp(dist) [expr $sp(dist)*(1.0 + $diff)]
	geossav_3dsp $sp(longitude) $sp(latitude) $sp(dist)
	set view_changed 1
	set old3_y $y
    }

    if { $view_changed } {
	$visframe.subfr_1.mframe.ogl activate
	set cmdstr "geossav_3dview_input $inpar(pos_x) $inpar(pos_y) $inpar(pos_z) $inpar(pos_var) $inpar(ballsize)"
	eval $cmdstr

	$visframe.subfr_1.mframe.ogl swapbuffers
    }
    
}
#============================================================================

proc geossav_in_ballsize { val } {
    global visframe
    global inpar

    geossav_3dview_input $inpar(pos_x) $inpar(pos_y) $inpar(pos_z) $inpar(pos_var) $inpar(ballsize)
    $visframe.subfr_1.mframe.ogl swapbuffers
}

#
# End of proc section.
#
#############################################################################
proc geossav_input_visualization_main { } {
    global exedir

    source $exedir/Geossav_input_visualization_globals.tcl

    set inpar(datafl) "*.dat"                     ;# input data file
    set inpar(pos_x) 1                            ;# column number for the x coordinates
    set inpar(pos_y) 2                            ;# column number for the y coordinates
    set inpar(pos_z) 3                            ;# column number for the z coordinates
    set inpar(pos_var) 4                          ;# column number for the variable
    set inpar(parameters) 0                       ;# number of data variables excluding the variable of None
    set inpar(cancel_counter) 0                   ;# counter for the exit from the INPUT DATA parameters tabnotebook
#    set inpar(nvis) 0                             ;# number of visualizations
    set inpar(visvar) 1                           ;# visualization of variables, 1=yes, 0=no
    set inpar(ballsize) 10
    
    . configure -cursor watch
    update
    
    set input_visualization_box_w [toplevel .input_visualization_box -class Dialog]
    wm resizable $input_visualization_box_w false false
    wm sizefrom $input_visualization_box_w program
    wm group $input_visualization_box_w .
    wm transient $input_visualization_box_w [winfo toplevel [winfo parent $input_visualization_box_w]]
    wm title $input_visualization_box_w "INPUT DATA parameters"
    
    set geostr [wm geometry .]
    regexp {([0-9]*)x([0-9]*)(\+)([0-9]*)(\+)([0-9]*)} $geostr all width height p1 pos_x p2 pos_y
    set pos_x [expr $pos_x + $width / 2 + 212]
    set pos_y [expr $pos_y + $height / 2 + 15]
    wm geometry $input_visualization_box_w "+$pos_x+$pos_y"
    
    
    #############################################################################
    # Tabnotebook
    #############################################################################
    set input_visualization_tabnotebook_w [iwidgets::tabnotebook $input_visualization_box_w.input_visualization_tbn \
					    -width 5.4i \
					    -height 1.3i \
					    -tabpos n \
					    -tabforeground gray50 \
					    -background gray85 \
					    -bevelamount 2]
    
    pack $input_visualization_tabnotebook_w -padx 0 -pady 0
    
    
    #
    # page1: Geossav INPUT_VISUALIZATION-parameters: Files (there is only one page)
    #
    set page [$input_visualization_box_w.input_visualization_tbn add \
		  -label "File"]


    # initialize dimension names for indata
    if { ![read_indata $inpar(datafl)] } {
	set dim_name_indata {}
	lappend dim_name_indata "None"
	for { set i 1} { $i < 5 } { incr i } {
	    lappend dim_name_indata $i
	}
    }

    
    set work_frame_w [frame $page.workframe \
			  -borderwidth 2 \
			  -relief flat]
    
    # fields for rendering data
    set data_iframe_w [iwidgets::labeledframe $work_frame_w.data \
			   -labeltext "Input" \
			   -labelpos nw \
			   -borderwidth 2 \
			   -relief groove]
    set data_frame_w [$data_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $data_frame_w.subfr1]
    
    set datafl_w [iwidgets::entryfield $subframe1_w.datafl \
		      -labeltext "File " \
		      -width 67 \
		      -state disabled \
		      -textvariable inpar(datafl)]
    
    if { $inpar(datafl) != {*.dat} } {
	$datafl_w configure -state normal -textbackground white -foreground black
    }
    
    set datafl_browse_w [iwidgets::pushbutton $subframe1_w.datafl_browse \
			     -height 27 \
			     -width 40 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			     -command {
				 set tmp_fname [tk_getOpenFile \
						    -defaultextension ".dat" \
						    -filetypes {{{Simulation Files} {.dat}} {{All Files} {.*}}} \
						    -initialfile $inpar(datafl) \
						    -initialdir $exedir \
						    -title "Select INPUT DATA for visualization"]
				 
				 ### select file name
				 if { $tmp_fname != {} } {
				     set inpar(datafl) $tmp_fname
				     
				     ### set widgets normal
				     if { $inpar(datafl) == {*.dat} } {
					 $datafl_w configure -state disabled
					 $pos_x_w configure -state disabled
					 $pos_y_w configure -state disabled
					 $pos_z_w configure -state disabled
					 $pos_var_w configure -state disabled
				     } elseif { $inpar(datafl) != {*.dat} } {
					 $datafl_w configure -state normal
					 $pos_x_w configure -state normal
					 $pos_y_w configure -state normal
					 $pos_z_w configure -state normal
					 $pos_var_w configure -state normal
				     }
				     
				     ### counter for the exit from the INPUT DATA parameters tabnotebook
				     set inpar(cancel_counter) 0

				     ### read data labels
				     set inpar(datafl) $tmp_fname
				     unset dim_name_indata
				     if { [read_indata $inpar(datafl)] != 1 } {
					 set dim_name_indata {}
					 lappend dim_name_indata "None"
					 for { set i 1 } { $i <= 5 } { incr i } {
					     lappend dim_name_indata $i
					 }
				     }
				     
				     $pos_x_w clear list
				     foreach i $dim_name_indata {
					 $pos_x_w insert list end $i
				     }
				     if { $inpar(parameters) >= 4 } {
					 $pos_x_w selection set 1
					 set inpar(pos_x) [lindex $dim_name_indata 1]
				     } elseif { $inpar(parameters) < 4 } {
					 $pos_x_w selection set 1
					 set inpar(pos_x) [lindex $dim_name_indata 1]
				     }

				     $pos_y_w clear list
				     foreach i $dim_name_indata {
					 $pos_y_w insert list end $i
				     }
				     if { $inpar(parameters) >= 4 } {
					 $pos_y_w selection set 2
					 set inpar(pos_y) [lindex $dim_name_indata 2]
				     } elseif { $inpar(parameters) < 4 } {
					 $pos_y_w selection set 1
					 set inpar(pos_y) [lindex $dim_name_indata 1]
				     }

				     $pos_z_w clear list
				     foreach i $dim_name_indata {
					 $pos_z_w insert list end $i
				     }
				     if { $inpar(parameters) >= 4 } {
					 $pos_z_w selection set 3
					 set inpar(pos_z) [lindex $dim_name_indata 3]
				     } elseif { $inpar(parameters) < 4 } {
					 $pos_z_w selection set 1
					 set inpar(pos_z) [lindex $dim_name_indata 1]
				     }
				     
				     $pos_var_w clear list
				     foreach i $dim_name_indata {
					 $pos_var_w insert list end $i
				     }
				     if { $inpar(parameters) >= 4 } {
					 $pos_var_w selection set 4
					 set inpar(pos_var) [lindex $dim_name_indata 4]
				     } elseif { $inpar(parameters) < 4 } {
					 $pos_var_w selection set 1
					 set inpar(pos_var) [lindex $dim_name_indata 1]
				     }
				 }
			     }]
    
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $datafl_w        -row 0 -column 0 -sticky we -pady 0
    grid $datafl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    pack \
	$subframe1_w \
	-anchor w -expand 1 -padx 3 -pady 1


    set subframe2_w [frame $data_frame_w.subfr2]

    if { $inpar(datafl) != {*.dat} } {
	set inpar(cancel_counter) [expr $inpar(cancel_counter) + 1]
    }

    set pos_x_w [iwidgets::combobox $subframe2_w.pos_x \
		     -labeltext "X " \
		     -textvariable inpar(pos_x) \
		     -width 11 \
		     -listheight 74 \
		     -validate {regexpvalidate %c}]
    if { $inpar(datafl) == {*.dat} } {
	foreach i $dim_name_indata {
	    $pos_x_w insert list end $i
	}
	$pos_x_w selection set 1
	set inpar(pos_x) [lindex $dim_name_indata 1]
    } elseif { $inpar(datafl) != {*.dat} } {
	set tmp_pos_x $inpar(pos_x)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if inpar(pos_x) is empty
	foreach i $dim_name_indata {
	    $pos_x_w insert list end $i
	    $pos_x_w delete list [expr $inpar(parameters) + 1]
	    if { $i == $tmp_pos_x } { ;# ok, that's the value in inpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($inpar(cancel_counter) * ($inpar(parameters) + 1))]
	$pos_x_w selection set $sel
	set inpar(pos_x) [lindex $dim_name_indata $sel]
    }

    
    set pos_y_w [iwidgets::combobox $subframe2_w.pos_y \
		     -labeltext "Y " \
		     -textvariable inpar(pos_y) \
		     -width 11 \
		     -listheight 74 \
		     -validate {regexpvalidate %c}]
    if { $inpar(datafl) == {*.dat} } {
	foreach i $dim_name_indata {
	    $pos_y_w insert list end $i
	}
	$pos_y_w selection set 2
	set inpar(pos_y) [lindex $dim_name_indata 2]
    } elseif { $inpar(datafl) != {*.dat} } {
	set tmp_pos_y $inpar(pos_y)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if inpar(pos_y) is empty
	foreach i $dim_name_indata {
	    $pos_y_w insert list end $i
	    $pos_y_w delete list [expr $inpar(parameters) + 1]
	    if { $i == $tmp_pos_y } { ;# ok, that's the value in inpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($inpar(cancel_counter) * ($inpar(parameters) + 1))]
	$pos_y_w selection set $sel
	set inpar(pos_y) [lindex $dim_name_indata $sel]
    }
    

    set pos_z_w [iwidgets::combobox $subframe2_w.pos_z \
		     -labeltext "Z " \
		     -textvariable inpar(pos_z) \
		     -width 11 \
		     -listheight 74 \
		     -validate {regexpvalidate %c}]
    if { $inpar(datafl) == {*.dat} } {
	foreach i $dim_name_indata {
	    $pos_z_w insert list end $i
	}
	$pos_z_w selection set 3
	set inpar(pos_z) [lindex $dim_name_indata 3]
    } elseif { $inpar(datafl) != {*.dat} } {
	set tmp_pos_z $inpar(pos_z)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if inpar(pos_z) is empty
	foreach i $dim_name_indata {
	    $pos_z_w insert list end $i
	    $pos_z_w delete list [expr $inpar(parameters) + 1]
	    if { $i == $tmp_pos_z } { ;# ok, that's the value in inpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($inpar(cancel_counter) * ($inpar(parameters) + 1))]
	$pos_z_w selection set $sel
	set inpar(pos_z) [lindex $dim_name_indata $sel]
    }


    set pos_var_w [iwidgets::combobox $subframe2_w.pos_var \
		       -labeltext "Var " \
		       -textvariable inpar(pos_var) \
		       -width 11 \
		       -listheight 74 \
		       -validate {regexpvalidate %c}]
    if { $inpar(datafl) == {*.dat} } {
	foreach i $dim_name_indata {
	    $pos_var_w insert list end $i
	}
	$pos_var_w selection set 4
	set inpar(pos_var) [lindex $dim_name_indata 4]
    } elseif { $inpar(datafl) != {*.dat} } {
	set tmp_pos_var $inpar(pos_var)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if inpar(pos_var) is empty
	foreach i $dim_name_indata {
	    $pos_var_w insert list end $i
	    $pos_var_w delete list [expr $inpar(parameters) + 1]
	    if { $i == $tmp_pos_var } { ;# ok, that's the value in inpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($inpar(cancel_counter) * ($inpar(parameters) + 1))]
	$pos_var_w selection set $sel
	set inpar(pos_var) [lindex $dim_name_indata $sel]
    }


    if { $inpar(datafl) != {*.dat} } {
	$pos_x_w configure -state normal
	$pos_y_w configure -state normal
	$pos_z_w configure -state normal
	$pos_var_w configure -state normal
    } elseif { $inpar(datafl) == {*.dat} } {
    	$pos_x_w configure -state disabled
	$pos_y_w configure -state disabled
	$pos_z_w configure -state disabled
	$pos_var_w configure -state disabled
    }
    
    
    grid columnconfigure $subframe2_w 0 -minsize 116
    grid columnconfigure $subframe2_w 1 -minsize 117
    grid columnconfigure $subframe2_w 2 -minsize 117
    grid columnconfigure $subframe2_w 3 -minsize 123
    
    grid $pos_x_w -row 0 -column 0 -sticky e -pady 0
    grid $pos_y_w -row 0 -column 1 -sticky e -pady 0
    grid $pos_z_w -row 0 -column 2 -sticky e -pady 0
    grid $pos_var_w -row 0 -column 3 -sticky e -pady 0

    pack \
	$subframe2_w \
	-anchor w -expand 1 -padx 3 -pady 1


    pack \
	$data_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x
    
    pack \
	$work_frame_w \
	-anchor nw -side top -fill x -fill y -expand true \
	-padx 0 -pady 0
    
    
    $input_visualization_box_w.input_visualization_tbn view "File"    
    
    
    #############################################################################
    # Default buttons
    #############################################################################
    
    set button_frame_w [frame $input_visualization_box_w.button_frame]
    
    set buttons_iframe_w [iwidgets::labeledframe $button_frame_w.buttons_iframe \
			      -labeltext "Defaults" \
			      -labelpos nw \
			      -borderwidth 2 \
			      -relief groove]
    set buttons_frame_w [$buttons_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set open_button_w [iwidgets::pushbutton $buttons_iframe_w.open \
			   -height 27 \
			   -width 40 \
			   -defaultringpad 0 \
			   -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			   -command {
			       set tmp_fname [tk_getOpenFile \
						  -initialdir [file dirname $mofpar(datafl)] \
						  -defaultextension ".par" \
						  -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
						  -title "Select INPUT DATA parameter file"]
			       
			       if { $tmp_fname != {} } {
				   set param_fname $tmp_fname
				   read_input_visualization_params $param_fname
			       }
			   }]
    
    
    set save_button_w [iwidgets::pushbutton $buttons_iframe_w.save \
			   -height 27 \
			   -width 27 \
			   -defaultringpad 0 \
			   -state disabled \
			   -image [image create photo -file [file join "$exedir/images/save.gif"]] \
			   -command {
			       write_input_visualization_params $param_fname
			   }]
    
    
    set save_as_button_w [iwidgets::pushbutton $buttons_iframe_w.save_as \
			      -height 27 \
			      -width 27 \
			      -defaultringpad 0 \
			      -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			      -command {
				  if { $inpar(datafl) == {*.dat} } {
				      tk_messageBox \
					  -title "Defaults" \
					  -icon warning \
					  -message "There is no input file.\nYou should select a corresponding file first." \
					  -type ok
				  } elseif { $inpar(datafl) != {*.dat} } {
				      set tmp_fname [tk_getSaveFile \
							 -initialdir [file dirname $inpar(datafl)] \
							 -defaultextension ".par" \
							 -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}} \
							 -title "Save INPUT DATA parameter file as ..."]
				      
				      $save_button_w configure -state normal
				      
				      if { $tmp_fname != {} } {
					  set param_fname $tmp_fname
					  write_input_visualization_params $param_fname
				      }
				  }
			      }]
    
    
    set cancel_button_w [iwidgets::pushbutton $buttons_iframe_w.cancel \
			     -height 27 \
			     -width 27 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/stop.gif"]] \
			     -command {
				 set private(button) 0
			     }]
    
    
    set run_button_w [iwidgets::pushbutton $buttons_iframe_w.run \
			  -height 27 \
			  -width 27 \
			  -defaultringpad 0 \
			  -image [image create photo -file [file join "$exedir/images/run.gif"]] \
			  -command {

			      ### set the cursor on watch
			      . configure -cursor watch
			      update

			      ### messages, if there are no data
			      if { $inpar(datafl) == {*.dat} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "There is no input file.\nYou should select a corresponding file first." \
				      -type ok
			      } elseif { $inpar(datafl) != {*.dat} } {

				  ### destroy visualization frames
				  if { $inpar(nvis) == 0 && $outpar(nvis) == 0 } {
				      set inpar(nvis) [expr $inpar(nvis) + 0]
				  } elseif { $inpar(nvis) != 0 || $outpar(nvis) != 0 } {
				      destroy $subframe1_w
				      destroy $subframe2_w
				  }

				  ### select combobox position of data and read data
				  set inpar(pos_x) [format "%d" [$pos_x_w curselection]]
				  set inpar(pos_y) [format "%d" [$pos_y_w curselection]]
				  set inpar(pos_z) [format "%d" [$pos_z_w curselection]]
				  set inpar(pos_var) [format "%d" [$pos_var_w curselection]]

 				  geossav_read_input $inpar(datafl)

				  ### set up visualization 
 				  set subframe1_w [frame $visframe.subfr_1]

				  set mframe [frame $subframe1_w.mframe]
				  pack $mframe 
				  set sp(longitude) -115.0
				  set sp(latitude) 60.0
				  set sp(dist) 3.0
				  geossav_3dsp $sp(longitude) $sp(latitude) $sp(dist)

				  opengl $mframe.ogl -width 800 -height 450 \
				      -rgb 1 \
				      -depth 1 \
				      -doublebuffer 1 \
				      -command "oglswap $mframe.ogl"
				  image create photo colorbar -height 450 -width 20
				  label $mframe.colbar -image colorbar
				  pack $mframe.ogl $mframe.colbar -side left
				  update

				  pack $subframe1_w
				  update

 				  $mframe.ogl activate

				  geossav_3dview_input \
 				      $inpar(pos_x) $inpar(pos_y) $inpar(pos_z) $inpar(pos_var) $inpar(ballsize) colorbar
				  
				  frame $mframe.f
				  label $mframe.f.vmin -text "Min = $dataval(vmin)"
				  label $mframe.f.vmax -text "Max = $dataval(vmax)"
				  pack $mframe.f -side left -expand true -fill y
				  pack $mframe.f.vmax -side top
				  pack $mframe.f.vmin -side bottom

				  ### set up control panel
				  set subframe2_w [frame $visframe.subfr_2]

				  set leftframe [frame $subframe2_w.leftframe]

# 				  set var_checkbutton_w [checkbutton $leftframe.var_cbutton \
# 							     -state disabled \
# 							     -text "Variable          Pointsize" \
# 							     -variable inpar(visvar) \
# 							     -command {}]

				  set var_checkbutton_w [label  $leftframe.var_cbutton -text "Ballsize"]
				  set middleframe [frame $subframe2_w.mframe]

				  set point_size [scale $middleframe.xy_psize \
						      -orient horizontal \
						      -showvalue true \
						      -length 300 \
						      -width 5 \
						      -from 0 -to 100 \
						      -variable inpar(ballsize) \
						      -command { geossav_in_ballsize }]


				  grid columnconfigure $leftframe 0 -minsize 420
				  grid columnconfigure $middleframe 1 -minsize 630

				  grid $var_checkbutton_w -row 1 -column 0 -sticky se -pady 0 -padx 8
				  grid $point_size -row 1 -column 1 -sticky w -pady 0 -padx 0


				  pack \
				      $leftframe \
				      $middleframe \
				      -side left -anchor w


				  bind $mframe.ogl <ButtonPress-1> {
				      geossav_in_button1_press %x %y
				  }
				  bind $mframe.ogl <B1-Motion> {
				      geossav_in_button1_motion %x %y
				  }
				  bind $mframe.ogl <ButtonPress-3> {
				      geossav_in_button3_press %x %y
				  }
				  bind $mframe.ogl <B3-Motion> {
				      geossav_in_button3_motion %x %y
				  }


				  pack $subframe2_w


				  set inpar(nvis) [expr $inpar(nvis) + 1]

				  set private(button) 1
      			      }

			      ### reset the cursor on arrow 
			      . configure -cursor arrow
			      update
			  }]
    
    bind $run_button_w <Return> "
	    $run_button_w configure -state active -relief sunken
	    update idletasks
	    after 100
	    $run_button_w invoke
	"
    
    grid columnconfigure $button_frame_w 0 -minsize 15
    grid columnconfigure $button_frame_w 1 -minsize 100
    grid columnconfigure $button_frame_w 2 -minsize 15
    grid columnconfigure $button_frame_w 3 -minsize 100
    grid columnconfigure $button_frame_w 4 -minsize 15
    grid columnconfigure $button_frame_w 5 -minsize 100
    grid columnconfigure $button_frame_w 6 -minsize 15
    grid columnconfigure $button_frame_w 7 -minsize 100
    grid columnconfigure $button_frame_w 8 -minsize 15
    grid columnconfigure $button_frame_w 9 -minsize 100
    grid columnconfigure $button_frame_w 10 -minsize 15
    
    grid $open_button_w    -row 0 -column 1 -sticky we -pady 8 -padx 5
    grid $save_button_w    -row 0 -column 3 -sticky we -pady 8 -padx 5
    grid $save_as_button_w -row 0 -column 5 -sticky we -pady 8 -padx 5
    grid $cancel_button_w  -row 0 -column 7 -sticky we -pady 8 -padx 5
    grid $run_button_w     -row 0 -column 9 -sticky we -pady 8 -padx 5
    
    
    pack \
	$buttons_iframe_w \
	-padx 5 -pady 5 -expand 1
    
    
    pack $button_frame_w \
	-side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    raise $input_visualization_box_w
    
#    tkwait visibility $input_visualization_box_w
    tkwait visibility $run_button_w
    
    set oldFocus [focus]
    set oldGrab [grab current $input_visualization_box_w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $input_visualization_box_w
    focus $run_button_w
    
    . configure -cursor arrow
    update
    
    update idletask
    tkwait variable private(button)
    
    catch {focus $oldFocus}
    catch {
	# It's possible that the window has already been destroyed,
	# hence this "catch".  Delete the Destroy handler so that
	# tkPriv(button) doesn't get reset by it.
	
	bind $input_visualization_box_w <Destroy> {}
	destroy $input_visualization_box_w
    }
    
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }   
    return
}
#============================================================================
#=END=OF=INPUT_VISUALIZATION=================================================
