##START#OF#VARGPLT###########################################################
# geossav_vargplt.tcl
#
# Source script creating the vargplt tabnotebook in GEOSSAV,
# developed at the University of Basel, Switzerland,
# by Christian Regli and Lukas Rosenthaler
#
#############################################################################
global exedir

#############################################################################
#
# The following set of parameters are required for the VARGPLT
#
source $exedir/Geossav_vargplt_globals.tcl

set vargpltpar(title) "Variogram plot"      ;# title for variogram plot
set vargpltpar(dislimmin) 0                 ;# minimum distance limit
set vargpltpar(dislimmax) 50                ;# maximum distance limit
set vargpltpar(varlimmin) 0                 ;# minimum variogram limit
set vargpltpar(varlimmax) 1.2               ;# maximum variogram limit
set vargpltpar(outfl) "*.ps"                ;# output data file containing the results, cycling fastest
set vargpltpar(datafl) "*.out"              ;# input data file, (output of gamv)
set vargpltpar(varionum) 1                  ;# variogram number of the input data file
set vargpltpar(dashnum) 0                   ;# dash weight (None,1,2,...,9,10), column number of combobox
set vargpltpar(points) 1                    ;# plotting points (yes,no), column number of combobox
set vargpltpar(line) 1                      ;# plotting line (yes,no), column number of combobox
set vargpltpar(color) 0                     ;# color of points and line (Red,...,Gray), column number of combobox
set vargpltpar(nvarg) 1                     ;# number of variograms to consider
set vargpltpar(partitle) 0                  ;# flag for writing parameter title

#
# End of initialization of variables. This has to be done once at the start 
# of the programm.
#
#############################################################################


#############################################################################
#
# Start of proc section.
#

#############################################################################
proc check_dislim { entry newstr } {
    return [regexp {^[+]?[0-9]*[\.]?[0-9]*([e][+-]?[0-9]*)?$} $newstr]
}
#============================================================================


#############################################################################
proc check_varlim { entry newstr } {
    return [regexp {^[+]?[0-9]*[\.]?[0-9]*([e][+-]?[0-9]*)?$} $newstr]
}
#============================================================================


#############################################################################
proc check_variogram_number { entry newstr } {
    return [regexp {^[0-9]*$} $newstr]
}
#============================================================================


#############################################################################
proc regexpvalidate { char } {
    return [regexp {![ ]} $char]
}
#============================================================================


#############################################################################
proc plus_variogram { } {
    global vargpltpar
    global variogram_table_w
    global variogram_vals
    global variogram_datafl
    global variogram_browse
    global variogram_number
    global variogram_dash
    global variogram_points
    global variogram_line
    global variogram_color
    global exedir

    set i $vargpltpar(nvarg)

    if { $vargpltpar(datafl,$i) == {*.out} } {
	tk_messageBox \
	    -title "Number of variogram types" \
	    -icon warning \
	    -message "You should select an input file first." \
	    -type ok
    } elseif { $vargpltpar(datafl,$i) != {*.out} } {
	if { $vargpltpar(nvarg) < 6 } {
	    
	    set vargpltpar(nvarg) [expr $vargpltpar(nvarg) + 1]
	    set ii $vargpltpar(nvarg)
	    
	    ### add an additional row in the variogram table
	    $variogram_table_w configure -rows [expr $vargpltpar(nvarg) + 1]
	    
	    set variogram_vals($ii,label) [iwidgets::labeledwidget $variogram_table_w.vals$ii \
					       -labeltext "$ii" \
					       -background gray50 \
					       -foreground white]
	    update
	    
	    set vargpltpar(datafl,$ii) "*.out"
	    set variogram_datafl($ii,entry) [iwidgets::entryfield $variogram_table_w.datafl$ii \
						 -width 24 \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -textvariable vargpltpar(datafl,$ii)]
	    update
	    
	    set variogram_browse($ii,browse) [iwidgets::pushbutton $variogram_table_w.browse$ii \
						  -height 27 \
						  -width 40 \
						  -defaultringpad 0 \
						  -image [image create photo -file [file join "$exedir/images/open.gif"]] \
						  -command {
						      set ii $vargpltpar(nvarg)
						      set tmp_fname [tk_getOpenFile \
									 -defaultextension ".out" \
									 -filetypes {{{Geo-EAS Files} {.out}} \
											 {{All Files} {.*}}} \
									 -initialfile $vargpltpar(datafl,$ii) \
									 -initialdir $exedir \
									 -title "Select VARGPLT input data"]
						      
						      if { $tmp_fname != {} } {
							  set vargpltpar(datafl,$ii) $tmp_fname
							  $variogram_datafl($ii,entry) configure \
							      -state normal -textbackground white -foreground black
							  $outfl_w configure \
							      -state normal -textbackground white -foreground black
						      }
						      
						      ### output path- and filename
						      if { $vargpltpar(outfl) == {*.ps} } {
							  if { $vargpltpar(datafl,$ii) != {*.out} } {
							      set datafl_rootname [file rootname $vargpltpar(datafl,$ii)]
							      set datafl_dirname [file dirname $vargpltpar(datafl,$ii)]
							      set datafl_dirname_lenght [string length $datafl_dirname]
							      set datafl_name [string range $datafl_rootname \
										   [expr $datafl_dirname_lenght + 1] end]
							      
							      set outfl_extension [file extension $vargpltpar(outfl)]
							      set plot "/VP_"
							      regsub {} $outfl_extension $datafl_name outputfl_1
							      regsub {} $outputfl_1 $plot outputfl_2
							      regsub {} $outputfl_2 $datafl_dirname outputfl_3
							      set vargpltpar(outfl) $outputfl_3
							  }
						      }
						  }]
	    update
	    
	    set vargpltpar(varionum,$ii) 1
	    set variogram_number($ii,entry) [iwidgets::entryfield $variogram_table_w.varionum$ii \
						 -textvariable vargpltpar(varionum,$ii) \
						 -validate {check_variogram_number %W %P}]
	    update

	    set vargpltpar(dashnum,$ii) 0
	    set variogram_dash($ii,combo) [iwidgets::combobox $variogram_table_w.dashnum$ii \
					       -textvariable vargpltpar(dashnum,$ii) \
					       -listheight 160 \
					       -validate {regexpvalidate %c}]
	    set sel 0   ;# that's the default selection if vargpltpar(dashnum,$ii) is empty
	    $variogram_table_w.dashnum$ii insert list end \
		"None" "1" "2" "3" "4" "5" "6" "7" "8" "9" "10"
	    if { $vargpltpar(dashnum,$ii) == "None" } {
		set sel 0
	    } elseif { $vargpltpar(dashnum,$ii) == "1" } {
		set sel 1
	    } elseif { $vargpltpar(dashnum,$ii) == "2" } {
		set sel 2
	    } elseif { $vargpltpar(dashnum,$ii) == "3" } {
		set sel 3
	    } elseif { $vargpltpar(dashnum,$ii) == "4" } {
		set sel 4
	    } elseif { $vargpltpar(dashnum,$ii) == "5" } {
		set sel 5
	    } elseif { $vargpltpar(dashnum,$ii) == "6" } {
		set sel 6
	    } elseif { $vargpltpar(dashnum,$ii) == "7" } {
		set sel 7
	    } elseif { $vargpltpar(dashnum,$ii) == "8" } {
		set sel 8
	    } elseif { $vargpltpar(dashnum,$ii) == "9" } {
		set sel 9
	    } elseif { $vargpltpar(dashnum,$ii) == "10" } {
		set sel 10
	    }
	    $variogram_dash($ii,combo) selection set $sel
	    set $variogram_dash($ii,combo) [lindex $variogram_table_w.dashnum$ii $sel]
	    update

	    set vargpltpar(points,$ii) 1
	    set variogram_points($ii,combo) [iwidgets::combobox $variogram_table_w.points$ii \
						 -textvariable vargpltpar(points,$ii) \
						 -listheight 35 \
						 -validate {regexpvalidate %c}]
	    set sel 1   ;# that's the default selection if vargpltpar(points,$ii) is empty
	    $variogram_table_w.points$ii insert list end \
		"No" "Yes"
	    if { $vargpltpar(points,$ii) == "No" } {
		set sel 0
	    } elseif { $vargpltpar(points,$ii) == "Yes" } {
		set sel 1
	    }
	    $variogram_points($ii,combo) selection set $sel
	    set $variogram_points($ii,combo) [lindex $variogram_table_w.points$ii $sel]
	    update

	    set vargpltpar(line,$ii) 1
	    set variogram_line($ii,combo) [iwidgets::combobox $variogram_table_w.line$ii \
					       -textvariable vargpltpar(line,$ii) \
					       -listheight 35 \
					       -validate {regexpvalidate %c}]
	    set sel 1   ;# that's the default selection if vargpltpar(line,$ii) is empty
	    $variogram_table_w.line$ii insert list end \
		"No" "Yes"
	    if { $vargpltpar(line,$ii) == "No" } {
		set sel 0
	    } elseif { $vargpltpar(line,$ii) == "Yes" } {
		set sel 1
	    }
	    $variogram_line($ii,combo) selection set $sel
	    set $variogram_line($ii,combo) [lindex $variogram_table_w.line$ii $sel]
	    update
	    
	    set vargpltpar(color,$ii) 0
	    set variogram_color($ii,combo) [iwidgets::combobox $variogram_table_w.color$ii \
						-textvariable vargpltpar(color,$ii) \
						-listheight 135 \
						-validate {regexpvalidate %c}]
	    set sel 0   ;# that's the default selection if vargpltpar(color,$ii) is empty
	    $variogram_table_w.color$ii insert list end \
		"Red" "Yellow" "Green" "Light blue" "Dark blue" "Violet" "White" "Black"
	    if { $vargpltpar(color,$ii) == "Red" } {
		set sel 0
	    } elseif { $vargpltpar(color,$ii) == "Yellow" } {
		set sel 1
	    } elseif { $vargpltpar(color,$ii) == "Green" } {
		set sel 2
	    } elseif { $vargpltpar(color,$ii) == "Light blue" } {
		set sel 3
	    } elseif { $vargpltpar(color,$ii) == "Dark blue" } {
		set sel 4
	    } elseif { $vargpltpar(color,$ii) == "Violet" } {
		set sel 5
	    } elseif { $vargpltpar(color,$ii) == "White" } {
		set sel 6
	    } elseif { $vargpltpar(color,$ii) == "Black" } {
		set sel 7
	    }
	    $variogram_color($ii,combo) selection set $sel
	    set $variogram_color($ii,combo) [lindex $variogram_table_w.color$ii $sel]
	    update
	    
	    
	    $variogram_table_w window configure $ii,0 -window $variogram_vals($ii,label) -relief flat -background gray50
	    $variogram_table_w window configure $ii,1 -window $variogram_datafl($ii,entry)
	    $variogram_table_w window configure $ii,2 -window $variogram_browse($ii,browse)
	    $variogram_table_w window configure $ii,3 -window $variogram_number($ii,entry)
	    $variogram_table_w window configure $ii,4 -window $variogram_dash($ii,combo)
	    $variogram_table_w window configure $ii,5 -window $variogram_points($ii,combo)
	    $variogram_table_w window configure $ii,6 -window $variogram_line($ii,combo)
	    $variogram_table_w window configure $ii,7 -window $variogram_color($ii,combo)
	    
	} elseif { $vargpltpar(nvarg) >= 6 } {
	    tk_messageBox \
		-title "Number of variogram types" \
		-icon warning \
		-message "Maximum 6 variograms are permitted in a singel pass." \
		-type ok
	}  
    }
}
#============================================================================


#############################################################################
proc minus_variogram { } {
    global vargpltpar
    global variogram_table_w
    global variogram_vals
    global variogram_datafl
    global variogram_browse
    global variogram_number
    global variogram_dash
    global variogram_points
    global variogram_line
    global variogram_color

    if { $vargpltpar(datafl,1) == {*.out} } {
	tk_messageBox \
	    -title "Number of variogram types" \
	    -icon warning \
	    -message "You should select an input file first." \
	    -type ok
    } elseif { $vargpltpar(datafl,1) != {*.out} } {
	if { $vargpltpar(nvarg) > 1 } {
	    
	    set ii $vargpltpar(nvarg)
	    
	    ### destroy the last row in the variogram table
	    destroy $variogram_vals($ii,label)
	    destroy $variogram_datafl($ii,entry)
	    destroy $variogram_browse($ii,browse)
	    destroy $variogram_number($ii,entry)
	    destroy $variogram_dash($ii,combo)
	    destroy $variogram_points($ii,combo)
	    destroy $variogram_line($ii,combo)
	    destroy $variogram_color($ii,combo)
	    $variogram_table_w configure -rows $ii
	    
	    set vargpltpar(nvarg) [expr $vargpltpar(nvarg) - 1]
	    
	} else {
	    tk_messageBox \
		-title "Number of variograms" \
		-icon warning \
		-message "At least 1 variogram is needed for a variogram plot." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc read_vargplt_params { fname } {
    global vargpltpar
    global variogram_table_w
    global variogram_vals
    global variogram_datafl
    global variogram_browse
    global variogram_number
    global variogram_dash
    global variogram_points
    global variogram_line
    global variogram_color
    global vargplt_box_w
    global outfl_w
    global exedir

    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }
    
    ### searches the line "parameters for vargplt" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Parameters for VARGPLT" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File selection error" \
		-icon error \
		-message "You have selected a wrong parameter file.\n"
	    return 0
	}
    }

    ### set file name in vargplt box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $vargplt_box_w "VARGPLT parameters   -   $fname_short"
    
    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }

    ### destroys the content of the variogram-table
    if { $vargpltpar(outfl) != {*.ps} } {
	for { set ii 1 } { $ii <= $vargpltpar(nvarg) } { incr ii } {
	    destroy $variogram_vals($ii,label)
	    destroy $variogram_datafl($ii,entry)
	    destroy $variogram_browse($ii,browse)
	    destroy $variogram_number($ii,entry)
	    destroy $variogram_dash($ii,combo)
	    destroy $variogram_points($ii,combo)
	    destroy $variogram_line($ii,combo)
	    destroy $variogram_color($ii,combo)
	}
    } elseif { $vargpltpar(outfl) == {*.ps} } {
	for { set ii 1 } { $ii <= 1} { incr ii } {
	    destroy $variogram_vals($ii,label)
	    destroy $variogram_datafl($ii,entry)
	    destroy $variogram_browse($ii,browse)
	    destroy $variogram_number($ii,entry)
	    destroy $variogram_dash($ii,combo)
	    destroy $variogram_points($ii,combo)
	    destroy $variogram_line($ii,combo)
	    destroy $variogram_color($ii,combo)
	}
    }
    
    ### reads file name of postscript output: file for postscript output
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set vargpltpar(outfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set vargpltpar(outfl) $line
    }

    if { $vargpltpar(outfl) == {*.dat} } {
	$outfl_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $vargpltpar(outfl) != {*.dat} } {
	$outfl_w configure -state normal -textbackground white -foreground black
    }

    ### reads the nvarg: number of variograms to plot
    set line [gets $inf]
    set space [string first " " $line]
    set vargpltpar(nvarg) [format "%d" [string range $line 0 $space]]

    ### reads the dislimmin, dislimmax: distance limits (from data if max<min)
    set line [gets $inf]
    set space [string first " " $line]
    set vargpltpar(dislimmin) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set vargpltpar(dislimmax) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the varlimmin, varlimmax: variogram limits (from data if max<min)
    set line [gets $inf]
    set space [string first " " $line]
    set vargpltpar(varlimmin) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set vargpltpar(varlimmax) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the title: variogram title
    set line [gets $inf]
    set vargpltpar(title) $line

    ### reads the datafl, varionum, dashnum, points, line, color:
    ### file with variogram data, variogram #, dash #, pts?, line?, color
    $variogram_table_w configure -rows [expr $vargpltpar(nvarg) + 1]
    for { set ii 1 } { $ii <= $vargpltpar(nvarg) } { incr ii } {

	set variogram_vals($ii,label) [iwidgets::labeledwidget $variogram_table_w.vals$ii \
					   -labeltext "$ii" \
					   -background gray50 \
					   -foreground white]
	update

	set line [gets $inf]
	set backslash [string first "\\" $line]
	if { $backslash >= 0 } {
	    set vargpltpar(datafl,$ii) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
	} elseif { $backslash < 0 } {
	    set vargpltpar(datafl,$ii) $line
	}
	set variogram_datafl($ii,entry) [iwidgets::entryfield $variogram_table_w.datafl$ii \
					     -width 24 \
					     -state normal \
					     -textbackground white \
					     -foreground black \
					     -textvariable vargpltpar(datafl,$ii)]
	if { $vargpltpar(datafl,$ii) == {*.out} } {
	    $variogram_datafl($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	} elseif { $vargpltpar(datafl,$ii) != {*.out} } {
	    $variogram_datafl($ii,entry) configure -state normal -textbackground white -foreground black
	}
	update

	set variogram_browse($ii,browse) [iwidgets::pushbutton $variogram_table_w.browse$ii \
					      -height 27 \
					      -width 40 \
					      -defaultringpad 0 \
					      -image [image create photo -file [file join "$exedir/images/open.gif"]] \
					      -command {
						  set ii $vargpltpar(nvarg)
						  set tmp_fname [tk_getOpenFile \
								     -defaultextension ".out" \
								     -filetypes {{{Geo-EAS Files} {.out}} \
										     {{All Files} {.*}}} \
								     -initialfile $vargpltpar(datafl,$ii) \
								     -initialdir $exedir \
								     -title "Select VARGPLT input data"]
						  
						  if { $tmp_fname != {} } {
						      set vargpltpar(datafl,$ii) $tmp_fname
						  }
					      }]
	update

	set line [gets $inf]

	set space [string first " " $line]
	set vargpltpar(varionum,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set variogram_number($ii,entry) [iwidgets::entryfield $variogram_table_w.varionum$ii \
					     -textvariable vargpltpar(varionum,$ii) \
					     -validate {check_variogram_number %W %P}]
	update

	set space [string first " " $line]
	set vargpltpar(dashnum,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set variogram_dash($ii,combo) [iwidgets::combobox $variogram_table_w.dashnum$ii \
					   -textvariable vargpltpar(dashnum,$ii) \
					   -listheight 160 \
					   -validate {regexpvalidate %c}]
	$variogram_table_w.dashnum$ii insert list end \
	    "None" "1" "2" "3" "4" "5" "6" "7" "8" "9" "10"
	$variogram_dash($ii,combo) selection set $vargpltpar(dashnum,$ii)
	update
	
	set space [string first " " $line]
	set vargpltpar(points,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set variogram_points($ii,combo) [iwidgets::combobox $variogram_table_w.points$ii \
					     -textvariable vargpltpar(points,$ii) \
					     -listheight 35 \
					     -validate {regexpvalidate %c}]
	$variogram_table_w.points$ii insert list end \
	    "No" "Yes"
	$variogram_points($ii,combo) selection set $vargpltpar(points,$ii)
	update

	set space [string first " " $line]
	set vargpltpar(line,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set variogram_line($ii,combo) [iwidgets::combobox $variogram_table_w.line$ii \
					   -textvariable vargpltpar(line,$ii) \
					   -listheight 35 \
					   -validate {regexpvalidate %c}]
	$variogram_table_w.line$ii insert list end \
	    "No" "Yes"
	$variogram_line($ii,combo) selection set $vargpltpar(line,$ii)
	update

	set space [string first " " $line]
	set vargpltpar(color,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set variogram_color($ii,combo) [iwidgets::combobox $variogram_table_w.color$ii \
					    -textvariable vargpltpar(color,$ii) \
					    -listheight 135 \
					    -validate {regexpvalidate %c}]
	$variogram_table_w.color$ii insert list end \
	    "Red" "Yellow" "Green" "Light blue" "Dark blue" "Violet" "White" "Black"
	if { $vargpltpar(color,$ii) == 1 } {
	    set sel 0
	} elseif { $vargpltpar(color,$ii) == 3 } {
	    set sel 1
	} elseif { $vargpltpar(color,$ii) == 5 } {
	    set sel 2
	} elseif { $vargpltpar(color,$ii) == 6 } {
	    set sel 3
	} elseif { $vargpltpar(color,$ii) == 7 } {
	    set sel 4
	} elseif { $vargpltpar(color,$ii) == 8 } {
	    set sel 5
	} elseif { $vargpltpar(color,$ii) == 9 } {
	    set sel 6
	} elseif { $vargpltpar(color,$ii) == 10 } {
	    set sel 7
	}
	$variogram_color($ii,combo) selection set $sel
	update
	
	
	$variogram_table_w window configure $ii,0 -window $variogram_vals($ii,label) -relief flat -background gray50
	$variogram_table_w window configure $ii,1 -window $variogram_datafl($ii,entry)
	$variogram_table_w window configure $ii,2 -window $variogram_browse($ii,browse)
	$variogram_table_w window configure $ii,3 -window $variogram_number($ii,entry)
	$variogram_table_w window configure $ii,4 -window $variogram_dash($ii,combo)
	$variogram_table_w window configure $ii,5 -window $variogram_points($ii,combo)
	$variogram_table_w window configure $ii,6 -window $variogram_line($ii,combo)
	$variogram_table_w window configure $ii,7 -window $variogram_color($ii,combo)
    }
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc write_vargplt_params { fname {tmpflag 0} } {
    global tcl_platform
    global version
    global vargpltpar
    global tmp_vargpltpar
    global vargplt_box_w
    
    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }

    ### set file name in vargplt box widget
    if { $vargpltpar(partitle) == 0 } {
	set param_dname [file dirname $fname]
	set dname_length [string length $param_dname]
	set fname_short [string range $fname [expr $dname_length + 1] end]
	wm title $vargplt_box_w "VARGPLT parameters   -   $fname_short"
    }

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    
    puts $outf "Parameters for VARGPLT"
    puts $outf "**********************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### outfl: file for postscript output
    set comment "\\file for postscript output"
    if { $tmpflag } {
	catch {
	    file copy -force $vargpltpar(outfl) "vargplt_out.ps"
	}
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "vargplt_out.ps"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $vargpltpar(outfl)]
    }
    set line [format "%s" $vargpltpar(outfl)]
    set line_length_total [string length $line]
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### nvarg: number of variograms to plot
    set comment "\\number of variograms to plot"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $vargpltpar(nvarg) $comment]

    ### dislimmin, dislimmax: distance limits (from data if max<min)
    catch {scan $vargpltpar(dislimmin) {%g} vargpltpar(dislimmin)}
    catch {scan $vargpltpar(dislimmax) {%g} vargpltpar(dislimmax)}
    puts -nonewline $outf [format "%g %g" $vargpltpar(dislimmin) $vargpltpar(dislimmax)]
    set line [format "%g %g" $vargpltpar(dislimmin) $vargpltpar(dislimmax)]
    set line_length [string length $line]
    set comment "\\distance limits (from data if max<min)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### varlimmin, varlimmax: variogram limits (from data if max<min)
    catch {scan $vargpltpar(varlimmin) {%g} vargpltpar(varlimmin)}
    catch {scan $vargpltpar(varlimmax) {%g} vargpltpar(varlimmax)}
    puts -nonewline $outf [format "%g %g" $vargpltpar(varlimmin) $vargpltpar(varlimmax)]
    set line [format "%g %g" $vargpltpar(varlimmin) $vargpltpar(varlimmax)]
    set line_length [string length $line]
    set comment "\\variogram limits (from data if max<min)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### title: title for variogram
    puts $outf [format "%-[expr $parameter_space - 0]s " $vargpltpar(title)]

    ### datafl, varionum, dashnum, points, line, color: file with variogram data, variogram #, dash #, pts?, line?, color 
    for { set ii 1 } { $ii <= $vargpltpar(nvarg) } { incr ii } {

	if { $vargpltpar(dashnum,$ii) == "None" } {
	    set tmp_vargpltpar(dashnum,$ii) 0
	} elseif { $vargpltpar(dashnum,$ii) == "1" } {
	    set tmp_vargpltpar(dashnum,$ii) 1
	} elseif { $vargpltpar(dashnum,$ii) == "2" } {
	    set tmp_vargpltpar(dashnum,$ii) 2
	} elseif { $vargpltpar(dashnum,$ii) == "3" } {
	    set tmp_vargpltpar(dashnum,$ii) 3
	} elseif { $vargpltpar(dashnum,$ii) == "4" } {
	    set tmp_vargpltpar(dashnum,$ii) 4
	} elseif { $vargpltpar(dashnum,$ii) == "5" } {
	    set tmp_vargpltpar(dashnum,$ii) 5
	} elseif { $vargpltpar(dashnum,$ii) == "6" } {
	    set tmp_vargpltpar(dashnum,$ii) 6
	} elseif { $vargpltpar(dashnum,$ii) == "7" } {
	    set tmp_vargpltpar(dashnum,$ii) 7
	} elseif { $vargpltpar(dashnum,$ii) == "8" } {
	    set tmp_vargpltpar(dashnum,$ii) 8
	} elseif { $vargpltpar(dashnum,$ii) == "9" } {
	    set tmp_vargpltpar(dashnum,$ii) 9
	} elseif { $vargpltpar(dashnum,$ii) == "10" } {
	    set tmp_vargpltpar(dashnum,$ii) 10
	}

	if { $vargpltpar(points,$ii) == "No" } {
	    set tmp_vargpltpar(points,$ii) 0
	} elseif { $vargpltpar(points,$ii) == "Yes" } {
	    set tmp_vargpltpar(points,$ii) 1
	}

	if { $vargpltpar(line,$ii) == "No" } {
	    set tmp_vargpltpar(line,$ii) 0
	} elseif { $vargpltpar(line,$ii) == "Yes" } {
	    set tmp_vargpltpar(line,$ii) 1
	}

	if { $vargpltpar(color,$ii) == "Red" } {
	    set tmp_vargpltpar(color,$ii) 1
	} elseif { $vargpltpar(color,$ii) == "Yellow" } {
	    set tmp_vargpltpar(color,$ii) 3
	} elseif { $vargpltpar(color,$ii) == "Green" } {
	    set tmp_vargpltpar(color,$ii) 5
	} elseif { $vargpltpar(color,$ii) == "Light blue" } {
	    set tmp_vargpltpar(color,$ii) 6
	} elseif { $vargpltpar(color,$ii) == "Dark blue" } {
	    set tmp_vargpltpar(color,$ii) 7
	} elseif { $vargpltpar(color,$ii) == "Violet" } {
	    set tmp_vargpltpar(color,$ii) 8
	} elseif { $vargpltpar(color,$ii) == "White" } {
	    set tmp_vargpltpar(color,$ii) 9
	} elseif { $vargpltpar(color,$ii) == "Black" } {
	    set tmp_vargpltpar(color,$ii) 10
	}

	set comment "\\file with variogram data"
	if { $tmpflag } {
	    catch {
		file copy -force $vargpltpar(datafl,$ii) "vargplt_in($ii).dat"
	    }
	    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "vargplt_in($ii).dat"]
	} else {
	    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $vargpltpar(datafl,$ii)]
	}
	set line [format "%s" $vargpltpar(datafl,$ii)]
	set line_length_total [string length $line]
	if { $line_length_total < $parameter_space } {
	    puts $outf [format "%s" $comment]
	} elseif { $line_length_total >= $parameter_space } {
	    set parameter_space [expr $line_length_total + 1]
	    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	    set parameter_space 24
	}
	set line_length_total 0
	puts -nonewline $outf "[format "%d" $vargpltpar(varionum,$ii)] "
	puts -nonewline $outf "[format "%d" $tmp_vargpltpar(dashnum,$ii)] "
	puts -nonewline $outf "[format "%d" $tmp_vargpltpar(points,$ii)] "
	puts -nonewline $outf "[format "%d" $tmp_vargpltpar(line,$ii)] "
	puts -nonewline $outf "[format "%d" $tmp_vargpltpar(color,$ii)] "
	set line [format "%d %d %d %d %d" $vargpltpar(varionum,$ii) $tmp_vargpltpar(dashnum,$ii) \
		      $tmp_vargpltpar(points,$ii) $tmp_vargpltpar(line,$ii) $tmp_vargpltpar(color,$ii)]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
	set comment "\\variogram #, dash #, pts?, line?, color"
	if { $line_length_total < $parameter_space } {
	    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	} elseif { $line_length_total >= $parameter_space } {
	    set parameter_space [expr $line_length_total + 1]
	    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	    set parameter_space 24
	}	
    }

    ### color codes for variogram lines/points:
    puts $outf "\n\nColor Codes for Variogram Lines/Points:"
    puts $outf "     1=red, 3=yellow, 5=green, 6=light blue,"
    puts $outf "     7=dark blue, 8=violet, 9=white, 10=black"

    ### close the file
    close $outf
}
#============================================================================

#
# End of proc section.
#
#############################################################################


#############################################################################
proc geossav_vargplt_main { } {
    global exedir

    source $exedir/Geossav_vargplt_globals.tcl
    
    . configure -cursor watch
    update
    
    set vargplt_box_w [toplevel .vargplt_box -class Dialog]
    wm resizable $vargplt_box_w false false
    wm sizefrom $vargplt_box_w program
    wm group $vargplt_box_w .
    wm transient $vargplt_box_w [winfo toplevel [winfo parent $vargplt_box_w]]
    wm title $vargplt_box_w "VARGPLT parameters"
    
    set geostr [wm geometry .]
    regexp {([0-9]*)x([0-9]*)(\+)([0-9]*)(\+)([0-9]*)} $geostr all width height p1 pos_x p2 pos_y
    set pos_x [expr $pos_x + $width / 2 + 212]
    set pos_y [expr $pos_y + $height / 2 + 15]
    wm geometry $vargplt_box_w "+$pos_x+$pos_y"
    
    
    #############################################################################
    # Tabnotebook
    #############################################################################
    set vargplt_tabnotebook_w [iwidgets::tabnotebook $vargplt_box_w.vargplt_tbn \
				-width 5.4i \
				-height 3.3i \
				-tabpos n \
				-tabforeground gray50 \
				-background gray85 \
				-bevelamount 2]
    
    pack $vargplt_tabnotebook_w -padx 0 -pady 0
    
    
    #
    # page1: Geossav VARGPLT-parameters: Files, there is only one file
    #
    set page [$vargplt_box_w.vargplt_tbn add \
		  -label "Files"]
    
    
    set work_frame_w [frame $page.workframe \
			  -borderwidth 2 \
			  -relief flat]
    
    
    # fields for input data
    set input_iframe_w [iwidgets::labeledframe $work_frame_w.input \
			    -labeltext "Input" \
			    -labelpos nw \
			    -borderwidth 2 \
			    -relief groove]
    set input_frame_w [$input_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $input_frame_w.subfr1]
    set title_w [iwidgets::entryfield $subframe1_w.title \
		     -labeltext "Plot title  " \
		     -width 70 \
		     -textvariable vargpltpar(title)]

    grid columnconfigure $subframe1_w 0 -minsize 480
    grid $title_w -row 0 -column 0 -sticky e -pady 0


    set subframe2_w [frame $input_frame_w.subfr2]
    set dislim_label_w [iwidgets::labeledwidget $subframe2_w.dislim_label \
			    -labeltext "Distance limits"]

    set dislimmin_w [iwidgets::entryfield $subframe2_w.dislimmin \
			 -labeltext "Minimum " \
			 -width 15 \
			 -validate {check_dislim %W %P} \
			 -textvariable vargpltpar(dislimmin)]
    
    set dislimmax_w [iwidgets::entryfield $subframe2_w.dislimmax \
			 -labeltext "Maximum " \
			 -width 15 \
			 -validate {check_dislim %W %P} \
			 -textvariable vargpltpar(dislimmax)]

    grid columnconfigure $subframe2_w 0 -minsize 154
    grid columnconfigure $subframe2_w 1 -minsize 163
    grid columnconfigure $subframe2_w 2 -minsize 163
    grid $dislim_label_w -row 0 -column 0 -sticky e -pady 5
    grid $dislimmin_w    -row 0 -column 1 -sticky e -pady 0
    grid $dislimmax_w    -row 0 -column 2 -sticky e -pady 0


    set subframe3_w [frame $input_frame_w.subfr3]
    set varlim_label_w [iwidgets::labeledwidget $subframe3_w.varlim_label \
			    -labeltext "Variogram limits"]

    set varlimmin_w [iwidgets::entryfield $subframe3_w.varlimmin \
			 -labeltext "Minimum " \
			 -width 15 \
			 -validate {check_varlim %W %P} \
			 -textvariable vargpltpar(varlimmin)]
    
    set varlimmax_w [iwidgets::entryfield $subframe3_w.varlimmax \
			 -labeltext "Maximum " \
			 -width 15 \
			 -validate {check_varlim %W %P} \
			 -textvariable vargpltpar(varlimmax)]

    grid columnconfigure $subframe3_w 0 -minsize 154
    grid columnconfigure $subframe3_w 1 -minsize 163
    grid columnconfigure $subframe3_w 2 -minsize 163
    grid $varlim_label_w -row 0 -column 0 -sticky e -pady 0
    grid $varlimmin_w    -row 0 -column 1 -sticky e -pady 0
    grid $varlimmax_w    -row 0 -column 2 -sticky e -pady 0


    set subframe4_w [frame $input_frame_w.subfr4]
    set plus_variogram_pushbutton_w [iwidgets::pushbutton $subframe4_w.plus_variogram_pbutton \
					 -height 27 \
					 -width 27 \
					 -defaultringpad 0 \
					 -image [image create photo -file [file join "$exedir/images/plus.gif"]] \
					 -command {plus_variogram}]
    
    
    set minus_variogram_pushbutton_w [iwidgets::pushbutton $subframe4_w.minus_variogram_pbutton \
					 -height 27 \
					 -width 27 \
					 -defaultringpad 0 \
					 -image [image create photo -file [file join "$exedir/images/minus.gif"]] \
					 -command {minus_variogram}]
    
    
    grid columnconfigure $subframe4_w 0 -minsize 10
    grid columnconfigure $subframe4_w 1 -minsize 152
    grid $plus_variogram_pushbutton_w  -row 0 -column 0 -sticky w -pady 0
    grid $minus_variogram_pushbutton_w -row 0 -column 1 -sticky w -pady 0
    
    
    set spaceframe1_w [frame $input_frame_w.spaceframe1 \
			   -height 6]
    
    
    set subframe5_w [frame $input_frame_w.subfr5]
    set variogram_frame_w [frame $subframe5_w.variogram_frame \
			       -background grey40 \
			       -borderwidth 2 \
			       -relief sunken]
    
    set variogram_sframe_w [frame $variogram_frame_w.variogram_sframe \
				-height 72 \
				-width 480]
    
    set variogram_hsbar_w [scrollbar $variogram_sframe_w.yscroll \
			       -orient vertical \
			       -command "$variogram_sframe_w.table yview"]
    
    if { $vargpltpar(outfl) == "*.ps" } {

	### setting up the variogram table
	set variogram_table_w [table $variogram_sframe_w.table \
				   -variable variogram_vals \
				   -titlerows 1 \
				   -titlecols 1 \
				   -cols 8 \
				   -rows 2 \
				   -yscrollcommand "$variogram_hsbar_w set" \
				   -height 4]
	
	$variogram_table_w width 0 4
	$variogram_table_w width 1 24
	$variogram_table_w width 2 6
	$variogram_table_w width 3 6
	$variogram_table_w width 4 9
	$variogram_table_w width 5 7
	$variogram_table_w width 6 7
	$variogram_table_w width 7 11

	set variogram_vals(0,0) "\#"
	set variogram_vals(0,1) "File"
	set variogram_vals(0,2) ""
	set variogram_vals(0,3) "Vario \#"
	set variogram_vals(0,4) "Dash \#"
	set variogram_vals(0,5) "Points"
	set variogram_vals(0,6) "Line"
	set variogram_vals(0,7) "Color"

	
	for { set ii 1 } { $ii < 2 } { incr ii } {
	    set variogram_vals($ii,label) [iwidgets::labeledwidget $variogram_table_w.vals$ii \
					       -labeltext "$ii" \
					       -background gray50 \
					       -foreground white]
	    
	    set vargpltpar(datafl,$ii) "*.out"
	    set variogram_datafl($ii,entry) [iwidgets::entryfield $variogram_table_w.datafl$ii \
						 -width 24 \
						 -state disabled \
						 -textbackground gray85 \
						 -foreground gray50 \
						 -textvariable vargpltpar(datafl,$ii)]

	    set variogram_browse($ii,browse) [iwidgets::pushbutton $variogram_table_w.browse$ii \
						  -height 27 \
						  -width 40 \
						  -defaultringpad 0 \
						  -image [image create photo -file [file join "$exedir/images/open.gif"]] \
						  -command {
						      set ii $vargpltpar(nvarg)
						      set tmp_fname [tk_getOpenFile \
									 -defaultextension ".out" \
									 -filetypes {{{Geo-EAS Files} {.out}} \
											 {{All Files} {.*}}} \
									 -initialfile $vargpltpar(datafl,$ii) \
									 -initialdir $exedir \
									 -title "Select VARGPLT input data"]

						      if { $tmp_fname != {} } {
							  set vargpltpar(datafl,$ii) $tmp_fname
							  $variogram_datafl($ii,entry) configure \
							      -state normal -textbackground white -foreground black
							  $outfl_w configure \
							      -state normal -textbackground white -foreground black
						      }

						      ### output path- and filename
						      if { $vargpltpar(outfl) == {*.ps} } {
							  if { $vargpltpar(datafl,$ii) != {*.out} } {
							      set datafl_rootname [file rootname $vargpltpar(datafl,$ii)]
							      set datafl_dirname [file dirname $vargpltpar(datafl,$ii)]
							      set datafl_dirname_lenght [string length $datafl_dirname]
							      set datafl_name [string range $datafl_rootname \
										   [expr $datafl_dirname_lenght + 1] end]
							      
							      set outfl_extension [file extension $vargpltpar(outfl)]
							      set plot "/VP_"
							      regsub {} $outfl_extension $datafl_name outputfl_1
							      regsub {} $outputfl_1 $plot outputfl_2
							      regsub {} $outputfl_2 $datafl_dirname outputfl_3
							      set vargpltpar(outfl) $outputfl_3
							  }
						      }
						  }]

	    set vargpltpar(varionum,$ii) 1
	    set variogram_number($ii,entry) [iwidgets::entryfield $variogram_table_w.varionum$ii \
						 -textvariable vargpltpar(varionum,$ii) \
						 -validate {check_variogram_number %W %P}]

	    set vargpltpar(dashnum,$ii) 0
	    set variogram_dash($ii,combo) [iwidgets::combobox $variogram_table_w.dashnum$ii \
					       -textvariable vargpltpar(dashnum,$ii) \
					       -listheight 160 \
					       -validate {regexpvalidate %c}]
	    set sel 0   ;# that's the default selection if vargpltpar(dashnum,$ii) is empty
	    $variogram_table_w.dashnum$ii insert list end \
		"None" "1" "2" "3" "4" "5" "6" "7" "8" "9" "10"
	    if { $vargpltpar(dashnum,$ii) == "None" } {
		set sel 0
	    } elseif { $vargpltpar(dashnum,$ii) == "1" } {
		set sel 1
	    } elseif { $vargpltpar(dashnum,$ii) == "2" } {
		set sel 2
	    } elseif { $vargpltpar(dashnum,$ii) == "3" } {
		set sel 3
	    } elseif { $vargpltpar(dashnum,$ii) == "4" } {
		set sel 4
	    } elseif { $vargpltpar(dashnum,$ii) == "5" } {
		set sel 5
	    } elseif { $vargpltpar(dashnum,$ii) == "6" } {
		set sel 6
	    } elseif { $vargpltpar(dashnum,$ii) == "7" } {
		set sel 7
	    } elseif { $vargpltpar(dashnum,$ii) == "8" } {
		set sel 8
	    } elseif { $vargpltpar(dashnum,$ii) == "9" } {
		set sel 9
	    } elseif { $vargpltpar(dashnum,$ii) == "10" } {
		set sel 10
	    }
	    $variogram_dash($ii,combo) selection set $sel
	    set $variogram_dash($ii,combo) [lindex $variogram_table_w.dashnum$ii $sel]

	    set vargpltpar(points,$ii) 1
	    set variogram_points($ii,combo) [iwidgets::combobox $variogram_table_w.points$ii \
						 -textvariable vargpltpar(points,$ii) \
						 -listheight 35 \
						 -validate {regexpvalidate %c}]
	    set sel 1   ;# that's the default selection if vargpltpar(points,$ii) is empty
	    $variogram_table_w.points$ii insert list end \
		"No" "Yes"
	    if { $vargpltpar(points,$ii) == "No" } {
		set sel 0
	    } elseif { $vargpltpar(points,$ii) == "Yes" } {
		set sel 1
	    }
	    $variogram_points($ii,combo) selection set $sel
	    set $variogram_points($ii,combo) [lindex $variogram_table_w.points$ii $sel]

	    set vargpltpar(line,$ii) 1
	    set variogram_line($ii,combo) [iwidgets::combobox $variogram_table_w.line$ii \
					       -textvariable vargpltpar(line,$ii) \
					       -listheight 35 \
					       -validate {regexpvalidate %c}]
	    set sel 1   ;# that's the default selection if vargpltpar(line,$ii) is empty
	    $variogram_table_w.line$ii insert list end \
		"No" "Yes"
	    if { $vargpltpar(line,$ii) == "No" } {
		set sel 0
	    } elseif { $vargpltpar(line,$ii) == "Yes" } {
		set sel 1
	    }
	    $variogram_line($ii,combo) selection set $sel
	    set $variogram_line($ii,combo) [lindex $variogram_table_w.line$ii $sel]
	    
	    set vargpltpar(color,$ii) 0
	    set variogram_color($ii,combo) [iwidgets::combobox $variogram_table_w.color$ii \
						-textvariable vargpltpar(color,$ii) \
						-listheight 135 \
						-validate {regexpvalidate %c}]
	    set sel 0   ;# that's the default selection if vargpltpar(color,$ii) is empty
	    $variogram_table_w.color$ii insert list end \
		"Red" "Yellow" "Green" "Light blue" "Dark blue" "Violet" "White" "Black"
	    if { $vargpltpar(color,$ii) == "Red" } {
		set sel 0
	    } elseif { $vargpltpar(color,$ii) == "Yellow" } {
		set sel 1
	    } elseif { $vargpltpar(color,$ii) == "Green" } {
		set sel 2
	    } elseif { $vargpltpar(color,$ii) == "Light blue" } {
		set sel 3
	    } elseif { $vargpltpar(color,$ii) == "Dark blue" } {
		set sel 4
	    } elseif { $vargpltpar(color,$ii) == "Violet" } {
		set sel 5
	    } elseif { $vargpltpar(color,$ii) == "White" } {
		set sel 6
	    } elseif { $vargpltpar(color,$ii) == "Black" } {
		set sel 7
	    }
	    $variogram_color($ii,combo) selection set $sel
	    set $variogram_color($ii,combo) [lindex $variogram_table_w.color$ii $sel]
	    
	    
	    $variogram_table_w window configure $ii,0 -window $variogram_vals($ii,label) -relief flat -background gray50
	    $variogram_table_w window configure $ii,1 -window $variogram_datafl($ii,entry)
	    $variogram_table_w window configure $ii,2 -window $variogram_browse($ii,browse)
	    $variogram_table_w window configure $ii,3 -window $variogram_number($ii,entry)
	    $variogram_table_w window configure $ii,4 -window $variogram_dash($ii,combo)
	    $variogram_table_w window configure $ii,5 -window $variogram_points($ii,combo)
	    $variogram_table_w window configure $ii,6 -window $variogram_line($ii,combo)
	    $variogram_table_w window configure $ii,7 -window $variogram_color($ii,combo)
	}

    } elseif { $vargpltpar(outfl) != "*.ps" } {

	### setting up the variogram table
	set variogram_table_w [table $variogram_sframe_w.table \
				   -variable variogram_vals \
				   -titlerows 1 \
				   -titlecols 1 \
				   -cols 8 \
				   -rows [expr $vargpltpar(nvarg) + 1] \
				   -yscrollcommand "$variogram_hsbar_w set" \
				   -height 4]
	
	$variogram_table_w width 0 4
	$variogram_table_w width 1 24
	$variogram_table_w width 2 6
	$variogram_table_w width 3 6
	$variogram_table_w width 4 9
	$variogram_table_w width 5 7
	$variogram_table_w width 6 7
	$variogram_table_w width 7 11

	set variogram_vals(0,0) "\#"
	set variogram_vals(0,1) "File"
	set variogram_vals(0,2) ""
	set variogram_vals(0,3) "Vario \#"
	set variogram_vals(0,4) "Dash \#"
	set variogram_vals(0,5) "Points"
	set variogram_vals(0,6) "Line"
	set variogram_vals(0,7) "Color"

	
	for { set ii 1 } { $ii < [expr $vargpltpar(nvarg) + 1] } { incr ii } {
	    set variogram_vals($ii,label) [iwidgets::labeledwidget $variogram_table_w.vals$ii \
					       -labeltext "$ii" \
					       -background gray50 \
					       -foreground white]				       
	    
	    set variogram_datafl($ii,entry) [iwidgets::entryfield $variogram_table_w.datafl$ii \
						 -width 24 \
						 -state normal \
						 -textbackground white \
						 -foreground black \
						 -textvariable vargpltpar(datafl,$ii)]
	    
	    set variogram_browse($ii,browse) [iwidgets::pushbutton $variogram_table_w.browse$ii \
						  -height 27 \
						  -width 40 \
						  -defaultringpad 0 \
						  -image [image create photo -file [file join "$exedir/images/open.gif"]] \
						  -command {
						      set ii $vargpltpar(nvarg)
						      set tmp_fname [tk_getOpenFile \
									 -defaultextension ".out" \
									 -filetypes {{{Geo-EAS Files} {.out}} \
											 {{All Files} {.*}}} \
									 -initialfile $vargpltpar(datafl,$ii) \
									 -initialdir $exedir \
									 -title "Select VARGPLT input data"]

						      if { $tmp_fname != {} } {
							  set vargpltpar(datafl,$ii) $tmp_fname
						      }
						  }]

	    set variogram_number($ii,entry) [iwidgets::entryfield $variogram_table_w.varionum$ii \
						 -textvariable vargpltpar(varionum,$ii) \
						 -validate {check_variogram_number %W %P}]

	    set variogram_dash($ii,combo) [iwidgets::combobox $variogram_table_w.dashnum$ii \
					       -textvariable vargpltpar(dashnum,$ii) \
					       -listheight 160 \
					       -validate {regexpvalidate %c}]
	    set sel 0   ;# that's the default selection if vargpltpar(dashnum,$ii) is empty
	    $variogram_table_w.dashnum$ii insert list end \
		"None" "1" "2" "3" "4" "5" "6" "7" "8" "9" "10"
	    if { $vargpltpar(dashnum,$ii) == "None" } {
		set sel 0
	    } elseif { $vargpltpar(dashnum,$ii) == "1" } {
		set sel 1
	    } elseif { $vargpltpar(dashnum,$ii) == "2" } {
		set sel 2
	    } elseif { $vargpltpar(dashnum,$ii) == "3" } {
		set sel 3
	    } elseif { $vargpltpar(dashnum,$ii) == "4" } {
		set sel 4
	    } elseif { $vargpltpar(dashnum,$ii) == "5" } {
		set sel 5
	    } elseif { $vargpltpar(dashnum,$ii) == "6" } {
		set sel 6
	    } elseif { $vargpltpar(dashnum,$ii) == "7" } {
		set sel 7
	    } elseif { $vargpltpar(dashnum,$ii) == "8" } {
		set sel 8
	    } elseif { $vargpltpar(dashnum,$ii) == "9" } {
		set sel 9
	    } elseif { $vargpltpar(dashnum,$ii) == "10" } {
		set sel 10
	    }
	    $variogram_dash($ii,combo) selection set $sel
	    set $variogram_dash($ii,combo) [lindex $variogram_table_w.dashnum$ii $sel]

	    set variogram_points($ii,combo) [iwidgets::combobox $variogram_table_w.points$ii \
						 -textvariable vargpltpar(points,$ii) \
						 -listheight 35 \
						 -validate {regexpvalidate %c}]
	    set sel 0   ;# that's the default selection if vargpltpar(points,$ii) is empty
	    $variogram_table_w.points$ii insert list end \
		"No" "Yes"
	    if { $vargpltpar(points,$ii) == "No" } {
		set sel 0
	    } elseif { $vargpltpar(points,$ii) == "Yes" } {
		set sel 1
	    }
	    $variogram_points($ii,combo) selection set $sel
	    set $variogram_points($ii,combo) [lindex $variogram_table_w.points$ii $sel]

	    set variogram_line($ii,combo) [iwidgets::combobox $variogram_table_w.line$ii \
					       -textvariable vargpltpar(line,$ii) \
					       -listheight 35 \
					       -validate {regexpvalidate %c}]
	    set sel 0   ;# that's the default selection if vargpltpar(line,$ii) is empty
	    $variogram_table_w.line$ii insert list end \
		"No" "Yes"
	    if { $vargpltpar(line,$ii) == "No" } {
		set sel 0
	    } elseif { $vargpltpar(line,$ii) == "Yes" } {
		set sel 1
	    }
	    $variogram_line($ii,combo) selection set $sel
	    set $variogram_line($ii,combo) [lindex $variogram_table_w.line$ii $sel]
	    
	    set variogram_color($ii,combo) [iwidgets::combobox $variogram_table_w.color$ii \
						-textvariable vargpltpar(color,$ii) \
						-listheight 135 \
						-validate {regexpvalidate %c}]
	    set sel 0   ;# that's the default selection if vargpltpar(color,$ii) is empty
	    $variogram_table_w.color$ii insert list end \
		"Red" "Yellow" "Green" "Light blue" "Dark blue" "Violet" "White" "Black"
	    if { $vargpltpar(color,$ii) == "Red" } {
		set sel 0
	    } elseif { $vargpltpar(color,$ii) == "Yellow" } {
		set sel 1
	    } elseif { $vargpltpar(color,$ii) == "Green" } {
		set sel 2
	    } elseif { $vargpltpar(color,$ii) == "Light blue" } {
		set sel 3
	    } elseif { $vargpltpar(color,$ii) == "Dark blue" } {
		set sel 4
	    } elseif { $vargpltpar(color,$ii) == "Violet" } {
		set sel 5
	    } elseif { $vargpltpar(color,$ii) == "White" } {
		set sel 6
	    } elseif { $vargpltpar(color,$ii) == "Black" } {
		set sel 7
	    }
	    $variogram_color($ii,combo) selection set $sel
	    set $variogram_color($ii,combo) [lindex $variogram_table_w.color$ii $sel]
	    
	    
	    $variogram_table_w window configure $ii,0 -window $variogram_vals($ii,label) -relief flat -background gray50
	    $variogram_table_w window configure $ii,1 -window $variogram_datafl($ii,entry)
	    $variogram_table_w window configure $ii,2 -window $variogram_browse($ii,browse)
	    $variogram_table_w window configure $ii,3 -window $variogram_number($ii,entry)
	    $variogram_table_w window configure $ii,4 -window $variogram_dash($ii,combo)
	    $variogram_table_w window configure $ii,5 -window $variogram_points($ii,combo)
	    $variogram_table_w window configure $ii,6 -window $variogram_line($ii,combo)
	    $variogram_table_w window configure $ii,7 -window $variogram_color($ii,combo)
	}
    }

    pack $variogram_hsbar_w -side right -anchor w -fill y
    pack $variogram_table_w -side left -anchor w -padx 2 -expand 1 -fill both
    pack $variogram_sframe_w -anchor nw -expand 1
    pack propagate $variogram_sframe_w 0
    grid $variogram_frame_w -row 0 -column 0 -sticky news -pady 0
    
    set spaceframe2_w [frame $input_frame_w.spaceframe2 \
			   -height 4]

    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	$subframe4_w \
	$spaceframe1_w \
	$subframe5_w \
	$spaceframe2_w \
	-anchor w -expand 1 -padx 3 -pady 0


    # fields for output data
    set output_iframe_w [iwidgets::labeledframe $work_frame_w.output \
			     -labeltext "Output" \
			     -labelpos nw \
			     -borderwidth 2 \
			     -relief groove]
    set output_frame_w [$output_iframe_w childsite]
    
    set subframe1_w [frame $output_frame_w.subfr1]
    set outfl_w [iwidgets::entryfield $subframe1_w.outfl_entry \
		     -labeltext "File  " \
		     -width 69 \
		     -state disabled \
		     -textbackground gray85 \
		     -foreground gray50 \
		     -textvariable vargpltpar(outfl)]
    set outfl_browse_w [iwidgets::pushbutton $subframe1_w.outfl_browse \
			    -height 27 \
			    -width 27 \
			    -defaultringpad 0 \
			    -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			    -command {
				set tmp_fname [tk_getSaveFile \
						   -defaultextension ".ps" \
						   -filetypes {{{Output Files} {.ps}} {{All Files} {.*}}} \
						   -initialfile $vargpltpar(outfl) \
						   -initialdir $exedir \
						   -title "Save postscript file as ..."]

				if { $tmp_fname != {} } {
				    set vargpltpar(outfl) $tmp_fname
				    $outfl_w configure -state normal -textbackground white -foreground black
				}
			    }]


    if { $vargpltpar(outfl) != {*.ps} } {
	$outfl_w configure -state normal -textbackground white -foreground black
    }
    
    
    grid columnconfigure $subframe1_w 0 -minsize 455
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $outfl_w        -row 0 -column 0 -sticky e -pady 0
    grid $outfl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    
    pack \
	$subframe1_w \
	-anchor w -expand 1 -padx 3 -pady 0
    
    
    pack \
	$input_iframe_w \
	$output_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x

    
    pack \
	$work_frame_w \
	-anchor nw -side top -fill x -fill y -expand true \
	-padx 0 -pady 0

    
    $vargplt_box_w.vargplt_tbn view "Files"
    
    
    . configure -cursor arrow
    
    
    #############################################################################
    # Default buttons
    #############################################################################
    #
    # write_vargplt_params fname [tmpflag]
    #----------------------------------------------------------------------------
    # This proc writs the parameter file for running vargplt
    #
    # fname   : [<-] : name of parameter file
    # tmpflag : [<-] : optional flag (0,1). If this flag is set, the
    #                  temporary data files are created in order to
    #                  circumvent a problem with the vargplt program
    #                  which forbids long filenames. The following
    #                  temporary files are created in the working
    #                  directory:
    #                    - "vargplt_in.dat" (data input)
    #                    - "vargplt_out.ps" (result of vargplt run)
    #############################################################################
    
    set button_frame_w [frame $vargplt_box_w.button_frame]
    
    set buttons_iframe_w [iwidgets::labeledframe $button_frame_w.buttons_iframe \
			      -labeltext "Defaults" \
			      -labelpos nw \
			      -borderwidth 2 \
			      -relief groove]
    set buttons_frame_w [$buttons_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set open_button_w [iwidgets::pushbutton $buttons_iframe_w.open \
			   -height 27 \
			   -width 40 \
			   -defaultringpad 0 \
			   -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			   -command {
			       set tmp_fname [tk_getOpenFile \
						  -initialdir [file dirname $vargpltpar(datafl)] \
						  -defaultextension ".par" \
						  -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
						  -title "Select VARGPLT parameter file"]
			       
			       if { $tmp_fname != {} } {
				   set param_fname $tmp_fname
				   read_vargplt_params $param_fname
			       }
			   }]
    
    
    set save_button_w [iwidgets::pushbutton $buttons_iframe_w.save \
			   -height 27 \
			   -width 27 \
			   -defaultringpad 0 \
			   -state disabled \
			   -image [image create photo -file [file join "$exedir/images/save.gif"]] \
			   -command {
			       set vargpltpar(partitle) 0
			       write_vargplt_params $param_fname
			   }]
    
    
    set save_as_button_w [iwidgets::pushbutton $buttons_iframe_w.save_as \
			      -height 27 \
			      -width 27 \
			      -defaultringpad 0 \
			      -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			      -command {
				  if { ($vargpltpar(datafl,1) != {*.out}) } {
				      set tmp_fname [tk_getSaveFile \
							 -initialdir [file dirname $vargpltpar(datafl,1)] \
							 -defaultextension ".par" \
							 -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
							 -title "Save VARGPLT parameter file as ..."]
				      
				      $save_button_w configure -state normal
				  
				      if { $tmp_fname != {} } {
					  set vargpltpar(partitle) 0
					  set param_fname $tmp_fname
					  write_vargplt_params $param_fname
				      }
				  } elseif { $vargpltpar(datafl,1) == {*.out} } {
				      tk_messageBox \
					  -title "Defaults" \
					  -icon warning \
					  -message "You should select an output file first." \
					  -type ok
				  }
			      }]
    
    
    set cancel_button_w [iwidgets::pushbutton $buttons_iframe_w.cancel \
			     -height 27 \
			     -width 27 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/stop.gif"]] \
			     -command {
				 set private(button) 0
			     }]
    
    
    set run_button_w [iwidgets::pushbutton $buttons_iframe_w.run \
			  -height 27 \
			  -width 27 \
			  -defaultringpad 0 \
			  -image [image create photo -file [file join "$exedir/images/run.gif"]] \
			  -command {
			      if { $vargpltpar(datafl,1) == {*.out} && $vargpltpar(outfl) == {*.ps} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "You should select an input and an output file first." \
				      -type ok
			      } elseif { $vargpltpar(datafl,1) == {*.out} && $vargpltpar(outfl) != {*.ps} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "You should select an input file first." \
				      -type ok
			      } elseif { $vargpltpar(datafl,1) != {*.out} && $vargpltpar(outfl) != {*.ps} } {
				  if { $tcl_platform(platform) == "unix" } {
				      set command "../gslib/bin/vargplt"
				  } elseif { $tcl_platform(platform) == "windows" } {
				      set command "C:/Programme/Statios/Gslib77/bin/vargnew.exe"
				  }
				  
				  set vargpltpar(partitle) 1
				  set paramfile "tmprun.par"
				  write_vargplt_params $paramfile 1
				  
				  set outf [open "VARGPLT.TMP" {w}]
				  puts $outf "$paramfile"
				  close $outf
				  set logfile [open "| $command < VARGPLT.TMP" {r}]
				  fconfigure $logfile -buffering none
				  while { ![eof $logfile] } {
				      gets $logfile input_line
				      log $input_line
				      update
				  }
				  close $logfile
				  
				  ### copy/delete the temporary files
				  catch {
				      file copy -force "vargplt_out.ps" $vargpltpar(outfl)
				  }
				  catch {
				      file delete -force \
					  "vargplt_out.ps" \
					  "tmprun.par" \
					  "VARGPLT.TMP"
				  }
				  for { set ii 1 } { $ii <= $vargpltpar(nvarg) } { incr ii } {
				      catch {
					  file delete -force \
					      "vargplt_in($ii).dat"
				      }
				  }
				  set private(button) 1
			      }
			  }]
    
    bind $run_button_w <Return> "
	    $run_button_w configure -state active -relief sunken
	    update idletasks
	    after 100
	    $run_button_w invoke
	"
    
    grid columnconfigure $button_frame_w 0 -minsize 15
    grid columnconfigure $button_frame_w 1 -minsize 100
    grid columnconfigure $button_frame_w 2 -minsize 15
    grid columnconfigure $button_frame_w 3 -minsize 100
    grid columnconfigure $button_frame_w 4 -minsize 15
    grid columnconfigure $button_frame_w 5 -minsize 100
    grid columnconfigure $button_frame_w 6 -minsize 15
    grid columnconfigure $button_frame_w 7 -minsize 100
    grid columnconfigure $button_frame_w 8 -minsize 15
    grid columnconfigure $button_frame_w 9 -minsize 100
    grid columnconfigure $button_frame_w 10 -minsize 15
    
    grid $open_button_w    -row 0 -column 1 -sticky we -pady 8 -padx 5
    grid $save_button_w    -row 0 -column 3 -sticky we -pady 8 -padx 5
    grid $save_as_button_w -row 0 -column 5 -sticky we -pady 8 -padx 5
    grid $cancel_button_w  -row 0 -column 7 -sticky we -pady 8 -padx 5
    grid $run_button_w     -row 0 -column 9 -sticky we -pady 8 -padx 5
    
    
    pack \
	$buttons_iframe_w \
	-padx 5 -pady 5 -expand 1
    
    
    pack $button_frame_w \
	-side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    raise $vargplt_box_w
    
    tkwait visibility $vargplt_box_w
    tkwait visibility $run_button_w
    
    set oldFocus [focus]
    set oldGrab [grab current $vargplt_box_w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $vargplt_box_w
    focus $run_button_w
    
    . configure -cursor arrow
    
    update idletask
    tkwait variable private(button)
    
    catch {focus $oldFocus}
    catch {
	# It's possible that the window has already been destroyed,
	# hence this "catch".  Delete the Destroy handler so that
	# tkPriv(button) doesn't get reset by it.
    
	bind $vargplt_box_w <Destroy> {}
	destroy $vargplt_box_w
    }
    
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    return
}
#============================================================================
#=END=OF=VARGPLT=============================================================