function Q = quadtriangle(d,varargin)

%% QUADTRIANGLE Quadrature weights and points for a triangle
%    Q = QUADTRIANGLE(d) returns the weights and points of a Gauss-Jacobi 
%    product rule of degree d for numerically integrating over a (default) 
%    triangle defined by the vertices (-1,-1),(1,-1) and (-1,1). The output 
%    is stored as a structure Q with the following fields:
%       »  Q.Points = the n quadrature points in an array of size n×2, 
%          where Q.Points(:,1) and Q.Points(:,2) are the x and y 
%          coordinates, respectively.
%       »  Q.Weights = the corresponding quadrature weights stored in a 
%          column vector of length n.
%       »  Q.Properties = contains (sub)fields describing the .Degree (the 
%          input value d), the .Type (see description below) and the 
%          .Domain of integration of the quadrature rule.
%  
%    Q = QUADTRIANGLE(d,Name,Value) same as above but with additional 
%    control over the type of rules returned. Options include specification 
%    of the following Name-Value pair arguments: 
%    ______________________________________________________________________
%    'Domain' -- Triangular domain
%    [ -1 -1; 1 -1; -1 1 ] (default) | [ x1 y2; x2 y2; x3 y3 ] | [ ] 
%    ----------------------------------------------------------------------     
%    The triangular domain over which the weights and points are defined,
%    specified as a 3×2 array of the (x,y) vertices of the triangle, i.e.,
%    T = [x1 y2; x2 y2; x3 y3], or as an empty array. 
% 
%    In the case of the latter option, the returned quadrature points 
%    consist of the first two barycentric (or area) coordinates, and the 
%    returned quadrature weights are scaled such that they sum to 1. (Note: 
%    The third barycentric coordinate for the points can easily be computed 
%    as 1-Q.Points(:,1)-Q.Points(:,2).)
%    ______________________________________________________________________
%    'Type' -- Type of quadrature rule
%    'product' (default) | 'nonproduct'
%    ----------------------------------------------------------------------  
%    The type of quadrature rule may be specified as 'product' (the
%    default), in which case the points and weights of a Gauss-Jacobi
%    product rule are returned, or as 'nonproduct'. Product rules can be of
%    arbitrary degree, while nonproduct rules are currently available up to
%    degree 25.
%
%    The following additional Name-Value pairs are available for nonproduct 
%    rules:
%    ______________________________________________________________________
%    'Symmetry' -- Symmetry of the quadrature rule
%    'full' (default) | 'allowAsymmetric'
%    ----------------------------------------------------------------------  
%    The 'full' (default) value requires the quadrature rule to be fully 
%    symmetric. The value 'allowAsymmetric' allows asymmetric quadrature 
%    rules of lower point count than the full symmetry rules to be returned 
%    when available. (Note: Some quadrature rules possess so-called partial
%    symmetry, though no distinction is made here between asymmetric and
%    partially symmetric rules, i.e., a rule is classified as either fully
%    symmetric or asymmetric.)
%    ______________________________________________________________________
%    'Weights' -- Quadrature weight conditions
%    'positive' (default) | 'allowNegative'
%    ---------------------------------------------------------------------- 
%    The 'postive' (default) value requires all quadrature weights to be 
%    positive. The value 'allowNegative' allows quadrature rules with nega-
%    tive weight(s) of lower point count than the positive weight rules to
%    be returned when available.
%    ______________________________________________________________________
%    'Points' -- Location of quadrature points
%    'inside' (default) | 'allowOutside'
%    ---------------------------------------------------------------------- 
%    The 'inside' (default) value requires all points to be inside the tri-
%    angle. The value 'allowOutside' allows quadrature rules with point(s)  
%    outside the triangle of lower point count than the quadrature rules 
%    with all points inside to be returned when available.
%    ______________________________________________________________________                 
%
%    Note: The default Name-Value pairs for nonproduct rules are set such 
%    that the minimal-point, fully symmetric so-called PI rule (that is,  
%    all weights are Positive and all points are Inside the triangle) of 
%    degree d is returned. Fully symmetric PI rules are generally preferred 
%    among practitioners.
%
%    See also quadsquare, quadGaussJacobi, quadGaussLobatto
%

%% Validate data passed to function

vd = @(x)validateattributes(x,{'numeric'},{'scalar','integer','>=',0});
vD = @(x)validateattributes(x,{'numeric'},{}); 
vT = @(x)validateattributes(x,{'char'},{'nonempty'});
ip = inputParser;
ip.addRequired('d',vd);
ip.addParameter('Domain',[ -1,-1; 1,-1; -1, 1 ],vD);
ip.addParameter('Type','product',vT);
ip.addParameter('Symmetry','full')
ip.addParameter('Weights','positive'),
ip.addParameter('Points','inside');
ip.parse(d,varargin{:}); ip.Results; 
Domain = ip.Results.Domain; Type = ip.Results.Type; 
Symmetry = ip.Results.Symmetry; Weights  = ip.Results.Weights; Points = ip.Results.Points; 

%% Compute quadrature weights and points
switch Type
    case {'product','Product'}
        %% Product Rules
        Qn = quadTriangleProduct(d);
    case {'nonproduct','Nonproduct'}
        %% Nonproduct rules
        if ~any(strcmpi(Symmetry,{'full','allowAsymmetric'}))
            error(['The value of ''Symmetry'' is invalid. Expected input to be:',...
                ' ''full'' (default) or ''allowAsymmetric''.']);
        end
        if ~any(strcmpi(Weights,{'positive','allowNegative'}))
            error(['The value of ''Weights'' is invalid. Expected input to be:',...
                ' ''positive'' (default) or ''allowNegative''.']);
        end
        if ~any(strcmpi(Points,{'inside','allowOutside'}))
            error(['The value of ''Points'' is invalid. Expected input to be:',...
                ' ''inside'' (default) or ''allowOutside''.']);
        end
        Qn = quadTriangleNonproduct(d,Symmetry,Weights,Points);
    otherwise
        error(['The value of ''Type'' is invalid. Expected input to be:',...
            ' ''product'' or ''nonproduct''.']);
end
Q = quadTrianglePointsAndWeights(Qn,Domain);
Q.Properties.Degree = d;
Q.Properties.Type   = Type;
Q.Properties.Domain = Domain;
end

%% Supporting function(s) =================================================

%% Function for product quadrature rules  
function Qn = quadTriangleProduct(d)

N = ceil((d+1)/2);
Q1 = quadGaussJacobi(N,0,0);
Q2 = quadGaussJacobi(N,1,0);
[eta1,eta2] = meshgrid(Q1.Points,Q2.Points);
Qn.m = zeros(1,N^2);
TR = triangulation([1 2 3],[-1 -1; 1 -1; -1 1]);
B = cartesianToBarycentric(TR,ones(N^2,1),[ 1/2*(1+eta1(:)).*(1-eta2(:))-1,eta2(:)]);
Qn.b1 = B(:,1); Qn.b2 = B(:,2); Qn.b3 = B(:,3);
Qn.w  = 1/2*1/2*kron(Q1.Weights,Q2.Weights); % Note additional factor of 1/2 corrected later

end

%% Function for Jacobi quadrature
function Q = quadGaussJacobi(n,alpha,beta)

%% QUADGAUSSJACOBI Gauss-Jacobi quadrature weights and points
%   Q = QUADGAUSSJACOBI(n,alpha,beta), where n is a positive integer and 
%   alpha and beta are real constants > -1, returns the n-point, 
%   (alpha,beta)-weighted Gauss-Jacobi quadrature rule Q as a stucture with 
%   fields Q.Points and Q.Weights, which store the n points (in ascending 
%   order), and corresponding weights, respectively, of the quadrature 
%   rule. An additional field, Q.Properties, stores the degree, type and 
%   interval, or domain, of the quadrature rule in subfields .Degree, .Type 
%   and .Domain, respectively. 
%
%   Note: An n-point Gauss-Jacobi quadrature rule is of degree 2n-1; that 
%   is, it integrates all polynomial up to degree 2n-1 exactly.
%
%   Functional dependencies: quadGaussLegendre
%
%   See also quadGaussLegendre, quadGaussLobatto 
%

%% Validate input

vn  = @(x)validateattributes(x,{'numeric'},{'scalar','integer','positive'});
vab = @(x)validateattributes(x,{'numeric'},{'scalar','real','>',-1});
ip  = inputParser;
ip.addRequired('n',vn); 
ip.addRequired('alpha',vab); ip.addRequired('beta',vab);
ip.parse(n,alpha,beta);
ip.Results;

%% Compute the weights and points

% Check for the Legendre case (alpha
if isequal([alpha,beta],[0,0])
    Q = quadGaussLegendre(n); 
    return
else
    N = 1:n-1; ab = alpha + beta;
    % Define the coefficients of the three-term recurrence relationship
    a(1) = 1/2*ab+1;
    b(1) = 1/2*(alpha-beta);
    a(2:n) = (2*N+ab+1).*(2*N+ab+2)./(2*(N+1).*(N+ab+1)); 
    b(2:n) = (alpha^2-beta^2)*(2*N+ab+1)./(2*(N+1).*(N+ab+1).*(2*N+ab)); 
    c(2:n) = (N+alpha).*(N+beta).*(2*N+ab+2)./((N+1).*(N+ab+1).*(2*N+ab));
    % Constructe the symmetric tridiagonal matrix 
    A = -b(1:n)./a(1:n); B = sqrt(c(2:n)./(a(1:n-1).*a(2:n)));
    J = diag(B,1) + diag(A) + diag(B,-1);    
    % Compute the eigenvalues and eigenvectors
    [V,D] = eig(J,'vector');    
    % Save (sorted) points and weights
    [Q.Points,I] = sort(D);
    Q.Weights = (V(1,I).^2*2^(alpha+beta+1).*exp(gammaln(alpha+1)+...
        gammaln(beta+1)-gammaln(alpha+beta+2)))';
end

%% Assign properties
Q.Properties.Degree = 2*n-1;
Q.Properties.Type = ['Gauss-Jacobi (',...
    num2str(alpha),',',num2str(beta),')'];
Q.Properties.Domain = [-1 1];

end

%% Function for Legendre quadrature
function Q = quadGaussLegendre(n,varargin)

%% QUADGAUSSLEGENDRE Gauss-Legendre quadrature weights and points
%    Q = quadGaussLegendre(n),  where n is a positive integer, returns the 
%    n-point Gauss-Legendre quadrature rule Q as a stucture with fields 
%    Q.Points and Q.Weights, which store the n points (in ascending order) 
%    and corresponding weights, respectively, of the quadrature rule. An 
%    additional field, Q.Properties, stores the degree, type and interval, 
%    or domain, of the quadrature rule (the default domain is [-1,1]) in 
%    subfields .Degree, .Type and .Domain, respectively. 
%
%    Q = quadGaussLegendre(n,'Domain',[a,b]), same as above, but where the
%    Name, Value pair 'Domain',[a,b] specifies the domain of integration
%    for the quadrature rule. The default domain is the bi-unit interval
%    [-1,1].
%
%    Note: An n-point Gauss-Legendre quadrature rule is of degree 2n-1;  
%    that is, it integrates all polynomial up to degree 2n-1 exactly.
%
%    See also quadGaussJacobi, quadGaussLobatto 
%

%% Validate input

vn = @(x)validateattributes(x,{'numeric'},{'scalar','integer','>=',1});
vD = @(x)validateattributes(x,{'numeric'},{'numel',2,'increasing'}); 
ip = inputParser;
ip.addRequired('n',vn); ip.addParameter('Domain',[-1,1],vD)
ip.parse(n,varargin{:}); ip.Results; Domain = ip.Results.Domain;

%% Compute the weights and points

% Define the coefficients of the three-term recurrence relationship
a = @(n)(2*n+1)./(n+1);
b = @(n)0;
c = @(n)n./(n+1);
% Constructe the symmetric tridiagonal matrix
A = -b(0:n-1)./a(0:n-1); B = sqrt(c(1:n-1)./(a(0:n-2).*a(1:n-1)));
J = diag(B,1) + diag(A) + diag(B,-1);     
% Compute the eigenvalues and eigenvectors
[V,D] = eig(J,'vector');
% Save (sorted) points and weights
[Q.Points,I] = sort(D);
Q.Weights = (2*V(1,I).^2)'; 
% Note: The next three lines insure zero is zero and the points and weights 
% are perfectly symmetric
Q.Points(abs(Q.Points)<10*eps) = 0; 
Q.Points(ceil(end/2)+1:end) = -flipud(Q.Points(1:floor(end/2)));
Q.Weights(ceil(end/2)+1:end) = flipud(Q.Weights(1:floor(end/2)));
% Transformation of points and weights if [a,b]~=[-1,1]
if ~isequal(Domain,[-1,1])
    Q.Points  = (Domain(2)-Domain(1))/2*Q.Points + (Domain(1)+Domain(2))/2;
    Q.Weights = (Domain(2)-Domain(1))/2*Q.Weights;
end

%% Assign properties

Q.Properties.Degree = 2*n-1;
Q.Properties.Type   = 'Gauss-Legendre';
Q.Properties.Domain = Domain;

end

%% Functions for non-product quadrature rules
function Qn = quadTriangleNonproduct(d,Symmetry,Weights,Points) 

vd = @(x)validateattributes(x,{'numeric'},{'scalar','nonnegative',...
    'integer','<=',25});
ip = inputParser;
ip.addRequired('d',vd);
ip.parse(d); ip.Results; 

Qf = str2func(['quadTriangleNonProductDegree',num2str(d)]);
Qn = Qf(Symmetry,Weights,Points);

end

%%  
%
%  Notes:
% 
%  The quadrature rules are described in the comments below by the follow-
%  ing designations:
%    (1) The degree of the quadrature rule. A quadrature rule is said to be 
%        of degree d if it can integrate all polynomials up to degree d 
%        exactly.
%    (2) The number of points. An asterik, e.g., 3* Points, indicates that 
%        the rule is a minimal-point rule, that is, it achieves the known
%        lower bound for the number of points of a quadrature rule of 
%        degree d over the triangle. This lower bound is given by
%
%            { (d+2)(d+4)/8                    if d is even
%        N = { 
%            { (d+1)(d+3)/8 + floor((d+1)/4)   if d is odd
%           
%        see Theorem 2.1 of page 131 of [L2] and the references therein.
%    (3) A two-letter designation indicating the "quality" of the weights 
%        and points: PI, NI, PO, or NO, where 'P' indicates the positivity 
%        of all weights, 'N' indicates the presence of a negative weight
%        (or weights), 'I' indicates all points are inside the triangle, 
%        and 'O' indicates a point (or points) are outside of the triangle.
%    (4) The symmetry of the quadrature rule: Fully symmetric, rotationally 
%        symmetric, or asymmetric.
%    (5) Reference from which the quadrature points and weights were
%        obtained. (Note: These references are not necessarily the first 
%        reference to publish the particular rule, though great effort has 
%        been made to identify the first reference.)
%
%  For example: Degree 3, 4* Points, NI, Fully symmetric, Reference: [H1]
%                (1)       (2)    (3)        (4)            (5)
%
%
%  Table summarizing the number of points in the available nonnproduct 
%  triangular rules (* denotes the rule achieves the lower bound):        
%
%  ========================================================================
%  |      |     ||           R  U  L  E      Q  U  A  L  I  T  Y          |
%  |      |     ||________________________________________________________|
%  |Degree|     ||    Not fully symmetric    ||      Fully symmetric      |
%  |  of  |Lower||___________________________||___________________________|       
%  | Rule |Bound||  NO  |  PO  |  NI  |  PI  ||  NO  |  PO  |  NI  |  PI  |
%  ========================================================================
%  |  1  |   1  ||      |      |      |      ||      |      |      |   1* |
%  |  2  |   3  ||      |      |      |      ||      |      |      |   3* |
%  |  3  |   4  ||      |      |      |   4* ||      |      |   4* |   6  |
%  |  4  |   6  ||      |      |      |      ||      |      |      |   6* |
%  |  5  |   7  ||      |      |      |      ||      |      |      |   7* |
%  |  6  |  10  ||      |  10* |      |  11  ||      |      |      |  12  |
%  |  7  |  12  ||      |      |      |  12* ||      |      |  13  |  15  |
%  |  8  |  15  ||      |  15* |      |      ||      |      |      |  16  |
%  |  9  |  17  ||      |      |      |      ||      |      |      |  19  |
%  | 10  |  21  ||      |  22  |      |  24  ||      |      |      |  25  |
%  | 11  |  24  ||      |  26  |      |  27  ||      |  27  |      |  28  |
%  | 12  |  28  ||      |  31  |      |  32  ||      |      |      |  33  |
%  | 13  |  31  ||      |      |      |  36  ||  36  |      |      |  37  |
%  | 14  |  36  ||      |  40  |      |      ||      |      |      |  42  |
%  | 15  |  40  ||      |      |      |  46  ||      |      |  48  |  49  |
%  | 16  |  45  ||      |  51  |      |  52  ||      |      |      |  55  |
%  | 17  |  49  ||      |      |      |  57  ||      |  58  |      |  60  |
%  | 18  |  55  ||      |      |  64  |  66  ||      |      |      |  67  |
%  | 19  |  60  ||      |      |      |  70  ||      |      |      |  73  |
%  | 20  |  66  ||      |      |      |  78  ||      |      |      |  79  |
%  | 21  |  71  ||      |      |      |  85  ||      |      |      |  87  |
%  | 22  |  78  ||      |      |      |  93  ||      |  94  |      |  96  |
%  | 23  |  84  ||      |      |      | 100  ||      |      |      | 102  |    
%  | 24  |  91  ||      |      |      | 109  ||      |      |      | 112  |
%  | 25  |  97  ||      |      |      | 117  ||      | 118  | 120  | 126  |
%  ========================================================================                    
%
%  References for rules:
%
%  [A1] J. Albrecht and L. Collatz, "Zur numerischen auswertung
%       mehrdimensionaler integrale, Zeitschrift fur Angewandte Mathematik
%       und Mechanik, 38, 1--15, 1958.
%  [B1] J. Berntsen and T.O. Espelid, "Degree 13 symmetric quadrature rules
%       for the triangle," Reports in Informatics 44, Department of Infor-
%       matics, University of Bergen, 1990.
%  [C1] G.R. Cowper, "Gaussian quadrature formulas for triangles," Inter-
%       national Journal of Numerical Methods in Engineering, 7, 405--408, 
%       1973.
%  [C2] R. Cools and A. Haegemans, "Construction of minimal cubature 
%       formulae for the square and the triangle using invariant theory,"
%       Report TW 96, Dept. of Computer Science, K.U. Leuven, 1987.
%  [C3] R. Cools, "An encyclopaedia of cubature formulas," Journal of 
%       Complexity, 19, 445--453,2003. Online database at: 
%       http://nines.cs.kuleuven.be/ecf/
%  [D1] D.M. Day and M.A. Taylor, "A new 11 point degree 6 cubature formula
%       for the triangle," Proceedings in Applied Mathematics & Mechanics,
%       7, 1022501--1022502, 2007.
%  [D2] D.A. Dunavant, "High degree efficient symmetrical Gaussian quad-
%       rature rules for the triangle," International Journal for Numerical
%       Methods in Engineering, 21, 1129--1148.
%  [G1] K. Gatermann, "The construction of symmetric cubature formulas for
%       the square and the triangle," Computing, 40, 229--240, 1988.
%  [G2] B. Griener and H.J. Schmid, "An interactive tool to visualize
%       common zeros of two-dimensional polynomials," Journal of
%       Computational and Applied Mathematics, 112, 83--94, 1999.
%  [H1] P.C. Hammer and A.H. Stroud, "Numerical integration over
%       simplexes," Mathematical Tables and Other Aids to Computation,
%       10(55): 133--139, 1956.
%  [L1] M.E. Laursen and M. Gellert, "Some criteria for numerically
%       integrated matrices and quadrature formulas for triangles,"
%       International Journal for Numerical Methods in Engineering, 12,
%       67--76, 1978.
%  [L2] J.N. Lyness and D. Jespersen, "Moderate degree symmetric quadrature
%       rules for the triangle," Journal of the Institute of Mathematics
%       and Its Applications (now the IMA Journal of Applied Mathematics),
%       15(1), 19--32, 1975.
%  [L3] J.N. Lyness and R. Cools, "A survey of numerical cubature over tri-
%       angles," in: Mathematics of Computation 1943-1993: a half-century
%       of computational mathematics (Vancouver, BC, 1993), volume 48 of
%       Proceedings of Symposia in Applied Mathematics, American Mathemati-
%       cal Society, Providence, RI, 1995, pp. 127-150.
%  [P1] S-A. Papanicolopulos, "Computation of moderate-degree fully-
%       symmetric cubature rules on the triangle using symmetric polynomi-
%       als and algebraic solving," Computers and Mathematics with Applica-
%       tions, 69, 650--666, 2015.
%  [P2] S-A. Papanicolopulos, "New fully symmetric and rotationally
%       symmetric cubature rules on the triangle using minimal orthonormal
%       bases," Journal of Computational and Applied Mathematics, 294,
%       39--48, 2016. (Note: Text files of the rules available at
%       https://arxiv.org/src/1411.5631v2/anc/rotationalsymmetry.txt)
%  [P3] S-A. Papanicolopulos, "Efficient computation of cubature rules with
%       application to new asymmetric rules on the triangle," Journal of
%       Computational and Applied Mathematics, 204, 73--83, 2016.
%  [R1] J. Radon, "Zur mechanischen Kubatur," Monatshefte für Mathematik,
%       52,286--300, 1948.
%  [R2] G.G. Rasputin, "Construction of cubature formulas containing pre-
%       specified knots," Metody Vychisl., 13, 1983 (in Russian).
%  [S1] A.H. Stroud, "Approximate calculation of multiple integrals,"
%       Prentice-Hall, Inc., 1971.
%  [T1] M.A. Taylor, "Asymmetric cubature formulas for polynomial
%       integration in the triangle and square," Journal of Computational
%       and Applied Mathematics 218, 184--191, 2008.
%  [W1] F.D. Witherden and P.E. Vincent, "On the identification of symmet-
%       ric quadrature rules for finite element methods," Computers and
%       Mathematics with Applications, 69, 1232--1241, 2015.
%       Online supplmemental material available at:
%       http://dx.doi.org/10.1016/j.camwa.2015.03.017
%  [W2] S. Weikl (simone.weikl@googlemail.com), Diploma Thesis, Technische
%       Universiat Munchen, Zentrum Mathematik, 2011 (?)
%  [W3] S. Wadzura and H. Xiao, Symmetric quadrature rules on a triangle,
%       Computers and Mathematics with Applications, 45 (2003), 1829--1840.
%  [X1] H. Xiao and Z. Gimbutas, "A numerical algorithm for the construc-
%       tion of efficient quadrature rules in two and higher dimensions,"
%       Computers and Mathematics with Applications, 59, 663--676, 2010.
%  [Z1] L. Zhang, T. Cui, and Hui Liu, "A set of symmetric quadrature rule
%       on triangles and tetrahedra", Journal of Computational Mathematics,
%       27, 89--96, 2009.
%  [Z2] L.B. Zhang, Parallel Hierarchical Grid, 
%       http://lsec.cc.ac.cn/phg/index_en.htm, (Note: Quadrature rules are
%       provided in the quad.c subroutine.)
%       
%
%==========================================================================
%%
function Qn = quadTriangleNonProductDegree1(varargin)
% Lower bound = 1
% _________________________________________________________________________
% Degree 1, 1* Point, PI, Fully symmetric, Reference: [Just think about it]
% -------------------------------------------------------------------------
Qn.m = 1; Qn.b1 = 1/3; Qn.w = 1;

end%*
%%
function Qn = quadTriangleNonProductDegree2(varargin)
% Lower bound = 3
% _________________________________________________________________________
% Degree 2, 3* Points, PI, Fully symmetric, References: First appeared in
% the English literature in Ref [H1], more convenient formulas for the rule
% later appeared in [S1], see page 307, formulas 2-1. (Note: This rule
% likely appeared earlier in the work of G.I. Georgiev (in Bulgarian).)
% -------------------------------------------------------------------------
Qn.m = 3; Qn.b1 = 1/6; Qn.b2 = 2/3; Qn.w = 1/3;  % Using upper signs of 2-1
%Qn.m = 3; Qn.b1 =   0; Qn.b2 = 1/2; Qn.w = 1/3; % Using lower signs of 2-1

end%*
%%
function Qn = quadTriangleNonProductDegree3(varargin)
% Lower bound = 4

switch varargin{2}
    case 'allowNegative'
        % _________________________________________________________________
        % Degree 3, 4* Points, NI, Fully symmetric, Reference: [H1], see
        % formulas of Theorem 1 on page 137.
        % -----------------------------------------------------------------
        Qn.m  = [ 1, 3 ];
        Qn.b1 = [ 1/3, 1/5 ];
        Qn.b2 = [ 1/3, 1/5 ];
        Qn.w  = [ -9/16, 25/48 ];
    case 'positive'
        switch varargin{1}
            case 'allowAsymmetric'
                % _________________________________________________________
                % Degree 3, 4* Points, PI, Asymmetric, Reference: This is 
                % just the degree 3 product rule.
                % ---------------------------------------------------------
                Qn = quadTriangleProduct(3);
            case 'full'
                % _________________________________________________________
                % Degree 3, 6 Points, PI, Fully symmetric, Reference: [S1],
                % see page 314, *T2: 3-1 (Note: Appeared earlier in [A1]
                % (in German), see Table 2 on Page 7 of that work.)
                % ---------------------------------------------------------
                Qn.m  = [ 3 3 ];
                Qn.b1 = [ 1/2, 1/6 ];
                Qn.b2 = Qn.b1;
                Qn.w  = [ 1/30, 3/10 ];
        end
end

end%*
%%
function Qn = quadTriangleNonProductDegree4(varargin)
% Lower bound = 6
% _________________________________________________________________________
% Degree 4, 6* Points, PI, Fully symmetric, Reference: [C1], see Table I on
% page 407. (Note: Analytic expressions obtained from [Z2])
% -------------------------------------------------------------------------
Qn.m  = [ 3 3 ];
Qn.b1 = [ (8 - sqrt(10) + sqrt(38 - 44*sqrt(2/5)))/18,...
    (8 - sqrt(10) - sqrt(38 - 44*sqrt(2/5)))/18 ];
Qn.b2 = Qn.b1;
Qn.w  = [ (620 + sqrt(213125 - 53320 * sqrt(10))) / 3720,...
    (620 - sqrt(213125 - 53320 * sqrt(10))) / 3720 ];
end%*
%%
function Qn = quadTriangleNonProductDegree5(varargin)
% Lower bound = 7
% _________________________________________________________________________
% Degree 5, 7* Points, PI, Fully symmetric, Reference: [R1], see page 295.
% -------------------------------------------------------------------------
Qn.m  = [ 1, 3, 3 ];
Qn.b1 = [  1/3, (6 - sqrt(15))/21, (6 + sqrt(15))/21  ];
Qn.b2 = Qn.b1;
Qn.w  = [ 9/40, (155 - sqrt(15))/1200, (155 + sqrt(15))/1200 ];

end%*
%%
function Qn = quadTriangleNonProductDegree6(varargin)
% Lower bound = 10

switch varargin{1}
    case 'allowAsymmetric'
        switch varargin{3}
            case 'allowOutside'
                % _________________________________________________________
                % Degree 6, 10* Points, PO (2 points outside), Asymmetric,
                % Reference:[G2], see Example 1 starting on page 88. This
                % rule was published earlier in [R2] (in Russian).
                % ---------------------------------------------------------
                Qn.m = [ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ];
                Qn.b1 = [  1.28867990757340236072, -0.00879199714420631034,...
                    0.78822920956686404839,  0.06468880605601090940,...
                    0.40172877323475981682,  0.07682571684908210504,...
                    0.64933214716985020413,  0.29502511936941515372,...
                    0.08533316117031069729,  0.35369054666996447962 ];
                Qn.b2 = [ -0.00879199714420631034,  1.28867990757340236072,...
                    0.06468880605601090940,  0.78822920956686404839,...
                    0.07682571684908210504,  0.40172877323475981682,...
                    0.29502511936941515372,  0.64933214716985020413,...
                    0.08533316117031069729,  0.35369054666996447962 ];
                Qn.w  = [  0.00051542875928455448,  0.00051542875928455448,...
                    0.10479531313284680990,  0.10479531313284680990,...
                    0.14855110930104331913,  0.14855110930104331913,...
                    0.09497723917756742321,  0.09497723917756742321,...
                    0.09099352359044946853,  0.21132829566806631803 ];
                % _________________________________________________________
                % Degree 6, 10* Points, PO (1 point WAY outside),
                % Asymmetric, Ref [G2], see Example 2 starting on page 89
                % ---------------------------------------------------------
                %                 Qn.m = [ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ];
                %                 Qn.b1 = [ 0.62261842170067743793,  0.29681206443213099158,...
                %                     0.86560137104657306697,  0.05848274947077690073,...
                %                     0.30783381768828903704,  0.56391552905457826463,...
                %                     0.06887470996336075494, -0.17255363572833024944,...
                %                     0.19794325221245567307,  0.04812894161425963481 ];
                %                 Qn.b2 = [ 0.29681206443213099158,  0.62261842170067743793,...
                %                     0.05848274947077690073,  0.86560137104657306697,...
                %                     0.30783381768828903704,  0.06887470996336075494,...
                %                     0.56391552905457826463, -0.17255363572833024944,...
                %                     0.04812894161425963489,  0.19794325221245567307 ];
                %                 Qn.w  = [ 0.12493284637910931301,  0.12493284637910931301,...
                %                     0.05736898913621279825,  0.05736898913621279825,...
                %                     0.21037615383745954659,  0.12547963681681714946,...
                %                     0.12547963681681714946,  0.00042783665681296109,...
                %                     0.08681653242072448544,  0.08681653242072448544 ];
            case 'inside'
                % _________________________________________________________
                % Degree 6, 11 Points, PI, Asymmetric, Reference: [D1]
                % (Note: Numerical values obtained from [Z2])
                % ---------------------------------------------------------
                Qn.m  = [ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ];
                Qn.b1 = [ 0.05725498667747685254386401337490528,...
                    0.89536264002457947703053711751108844,...
                    0.68447574845651404000000000000000000,...
                    0.06874625591502945506138017629748138,...
                    0.61567620557584050877805946045395354,...
                    0.62794614119778946000000000000000000,...
                    0.06290913834186374714863309341182919,...
                    0.06837821192050999512347955121812458,...
                    0.28752945837439246626570898085589649,...
                    0.32878355641313448865307686080999177,...
                    0.31229040501364539638767631759216157 ];
                Qn.b2 = [ 0.89549814678987956946437486806257514,...
                    0.06182822125032174107784815639931401,...
                    0.02334373849768321288645813914257559,...
                    0.06003027574726303099667885025487684,...
                    0.33346180834137626773374382178966904,...
                    0.15918918599215148000000000000000000,...
                    0.65529509370545247000000000000000000,...
                    0.30911768542826747346974929569953382,...
                    0.63642650917962018000000000000000000,...
                    0.07702400564246333220945276738749036,...
                    0.35234478644589918332680050733851911 ];
                Qn.w  = [ 0.03806807185295551439063192528703396,...
                    0.03837935530775265554764109222240791,...
                    0.04620045674456241859455515980217997,...
                    0.05346758944419901689402679518609906,...
                    0.08375582696574588108371037842585680,...
                    0.10164483302551605906571229513115369,...
                    0.10186152446136717571450065716620939,...
                    0.11142183166000185684914076857474210,...
                    0.11200945026294611573462286906312035,...
                    0.12478757143755821156418716664966669,...
                    0.18840348883739509456127089249153007 ];
        end
    case 'full'
        % _________________________________________________________________
        % Degree 6, 12 Points, PI, Fully symmetric, Reference: [C1], see 
        % Table 1 on page 407. (Note: Numerical values obtained from [Z2])
        % -----------------------------------------------------------------
        Qn.m   = [ 3,3,6 ];
        Qn.b1  = [ 0.0630890144915022283403316028708192,...
            0.2492867451709104212916385531070191,...
            0.0531450498448169473532496716313981 ];
        Qn.b2 = Qn.b1;
        Qn.b2(3) = 0.3103524510337844054166077339565522;
        Qn.w   = [ 0.0508449063702068169209368091068690,...
            0.1167862757263793660252896113855794,...
            0.0828510756183735751935534564204425 ];
end

end%*
%%
function Qn = quadTriangleNonProductDegree7(varargin)
% Lower bound = 12

switch varargin{1}
    case 'allowAsymmetric'
        % _________________________________________________________________
        % Degree 7, 12* Points, PI, Rotationally symmetric, References:
        % [G1], see Table 5 on page 238. Numerical values obtained from
        % [C3].
        % -----------------------------------------------------------------
        Qn.m  = [ 3, 3, 3, 3];
        Qn.b1 = [ 0.0623822650944021181736830009963499,...
            0.0552254566569266117374791902756449,...
            0.0343243029450971464696306424839376,...
            0.515842334353591779257463386826430 ];
        Qn.b2 = [ 0.0675178670739160854425571310508685,...
            0.321502493851981822666307849199202,...
            0.660949196186735657611980310197799,...
            0.277716166976391782569581871393723 ];
        Qn.w = [ 0.0265170281574362514287541804607391,...
            0.0438814087144460550367699031392875,...
            0.0287750427849815857384454969002185,...
            0.0674931870098027744626970861664214 ]*2;
    case 'full'
        switch varargin{2}
            case 'allowNegative'
                % _________________________________________________________
                % Degree 7, 13 Points, NI, Fully symmetric, References:
                % [C1], see Table I on page 407. Numerical values obtained
                %  from [C3].
                % ---------------------------------------------------------
                Qn.m  = [ 1, 3, 3, 6 ];
                Qn.b1 = [ 0.333333333333333333333333333333333,...
                    0.260345966079039826926242469139236,...
                    0.0651301029022158115380259063119754,...
                    0.0486903154253164117930215585284131 ];
                Qn.b2 = Qn.b1;
                Qn.b2(4) = 0.312865496004873861406644476768401;
                Qn.w  = [ -0.0747850222338408753148556277365486,...
                    0.0878076287166039058767597055781635,...
                    0.0266736178044192456349936444495335,...
                    0.0385568804451285701299325962755762 ]*2;
            case 'positive'
                % _________________________________________________________                
                % Degree 7, 15 Points, PI, Fully Symmetric, References:
                % [L1], see Table II on page 74. Numerical values obtained
                % from [C3].
                % ---------------------------------------------------------
                Qn.m  = [ 3, 6, 6 ];
                Qn.b1 = [ 0.0649305131591648630783797760303965,...
                    0.198384476681506719179876598633329,...
                    0.642577343822696020532349401830075 ];
                Qn.b2(1) = Qn.b1(1);
                Qn.b2(2:3) = [ 0.517039939069322945622707344016890,...
                    0.0438634717923724715117986959712959 ];
                Qn.w = [ 0.0265389008951162058359774874998477,...
                    0.0354265418460667836592062916232018,...
                    0.0346373410397084467561382979602076 ]*2;
        end
end

end%*
%%
function Qn = quadTriangleNonProductDegree8(varargin)
% Lower bound = 15

if strcmp(varargin{1},'allowAsymmetric') && strcmp(varargin{3},'allowOutside')
    % _____________________________________________________________________
    % Degree 8, 15* Points, PO, Rotationally symmetric, References:
    % Reported to be published in [C2]. Numerical values obtained from
    % [C3]. (Note: This rule is of little practical value as there are 3
    % points that are a significant distance outside the triangle.)
    % ---------------------------------------------------------------------
    Qn.m     = [ 3, 3, 3, 3, 3 ];
    Qn.w     = [ 1.60583438566812187984002319217607e-10,...
        0.0265306244347803793468998583175327,...
        0.0292857176401658921592602923421211,...
        0.0439095567912207824018635604112394,...
        0.0669407676399161741918307676117709 ]*2;
    Qn.b1    = [0.345792011168269028821402703032657,...
        0.0651019934589391663281047918600097,...
        0.651775303648795707537235440171623,...
        0.313251210671725306955957434773977,...
        0.513346920639454149493588964941289 ];
    Qn.b2    = [ 3.62316822156926166667912853628633,...
        0.870165101563563060777479924328995,...
        0.313477887523733007173578085409864,...
        0.630621434318956140102957434914989,...
        0.281041247315110390572737910356218 ];
else
    % _____________________________________________________________________    
    % Degree 8, 16 Points, PI, Fully symmetric, References: [L2], see Table
    % 4 on page 30. Numerical values obtained from [Z2].
    % ---------------------------------------------------------------------
    Qn.m     = [ 1, 3, 3, 3, 6 ];
    Qn.b1    = [ 0.3333333333333333333333333333333333, ...
        0.1705693077517602066222935014914645, ...
        0.0505472283170309754584235505965989, ...
        0.4592925882927231560288155144941693, ...
        0.2631128296346381134217857862846436 ];
    Qn.b2    = Qn.b1;
    Qn.b2(5) =   0.0083947774099576053372138345392944;
    Qn.w     = [ 0.1443156076777871682510911104890646, ...
        0.1032173705347182502817915502921290, ...
        0.0324584976231980803109259283417806, ...
        0.0950916342672846247938961043885843, ...
        0.0272303141744349942648446900739089 ];
end

end%*
%%
function Qn = quadTriangleNonProductDegree9(varargin)
% Lower bound = 17
% _________________________________________________________________________
% Degree 9, 19 Points, PI, Fully symmetric, References: [L2], see Table
% 4(-continued) on page 31. Numerical values obtained from [Z2].
% -------------------------------------------------------------------------
Qn.m = [ 1, 3, 3, 3, 3, 6 ];
Qn.b1    = [ 0.3333333333333333333333333333333333, ...
    0.4896825191987376277837069248361928, ...
    0.0447295133944527098651065899662764, ...
    0.4370895914929366372699303644353550, ...
    0.1882035356190327302409612804673356, ...
    0.7411985987844980206900798735234238 ];
Qn.b2    = Qn.b1;
Qn.b2(6) = [ 0.2219629891607656956751025276931911 ];
Qn.w     = [ 0.0971357962827988338192419825072886, ...
    0.0313347002271390705368548312872093, ...
    0.0255776756586980312616787985589998, ...
    0.0778275410047742793167393562994040, ...
    0.0796477389272102530328917742640453, ...
    0.0432835393772893772893772893772894 ];

end%
%%
function Qn = quadTriangleNonProductDegree10(varargin)
% Lower bound = 21

switch varargin{1}
    case 'allowAsymmetric'
        switch varargin{3}
            case 'allowOutside'
                % _________________________________________________________
                % Degree 10, 22 Points, PO, Rotationally symmetric,
                % References: Reported to be published in [C2]. Numerical
                % values obtained from [C3]. (Note: This rule is of little
                % practical value as there are 3 points that are a
                % significant distance outside the triangle.)
                % ---------------------------------------------------------
                Qn.m = [ 1 3 3 3 3 3 3 3 ];
                Qn.b1 = [0.333333333333333333333333333333333,...
                    0.0584692016835845130308403998963054,...
                    0.0508492850640314107046235532660065,...
                    0.515867324199496744865198209456030,...
                    0.243110331917390482295130382657376,...
                    0.753977659209226601343723665603381,...
                    0.422092079108469602936002757847740,...
                    0.198238788466633540678495453957414 ];
                Qn.b2 = [ 0.333333333333333333333333333333333,...
                    -0.548877787725275193169739005015091,...
                    0.907990597949578134392896949515435,...
                    0.463124528429270629021981209809870,...
                    0.721805951823719594670258558507149,...
                    0.206475698391323976329969663821696,...
                    0.126895334134111273268582501784675,...
                    0.621244125663933197445421376130889 ];
                Qn.w = [0.0479105348615200606662587952673170,...
                    1.53191300367585576302242186471823e-7,...
                    0.0132605262279287852213668114840399,...
                    0.0156464393445390421361828023561944,...
                    0.0217042582248073233107744043493244,...
                    0.0217976136001299223673954090061025,...
                    0.0385879135081934594680296573313320,...
                    0.0396995842825944130219216814750477]*2;
            case 'inside'
                % WARNING: Error slightly high on this one ~1.0e-09
                % _________________________________________________________                
                % Degree 10, 24 Points, PI, Rotationally symmetric,
                % References: Not sure who originally found it. Numerical
                % values obtained from online supplemental material of
                % [P2].
                % ---------------------------------------------------------
                Qn.m = [ 3 3 3 3 3 3 3 3 ];
                Qn.b1 = [1.4583457698929406e-01,...
                    4.8063935754816450e-01,...
                    2.8384362149153980e-03,...
                    2.7382456096225797e-02,...
                    2.9036991669239250e-01,...
                    1.0257910434962711e-01,...
                    8.9138718639893450e-02,...
                    2.8930618917096717e-01 ];
                Qn.b2 = [1.1398571160097378e-02,...
                    3.9643706994449101e-03,...
                    2.3534170704269680e-01,...
                    4.3399288238870900e-02,...
                    5.9484989766216527e-02,...
                    1.4552326855517441e-01,...
                    3.6893474593389925e-01,...
                    2.2719614311462367e-01 ];
                Qn.w = [1.4711143125265022e-02,...
                    1.4928593150681595e-02,...
                    1.4947323548874366e-02,...
                    1.5357000172646521e-02,...
                    5.1440501966811599e-02,...
                    5.5187392917754130e-02,...
                    7.1717980709822582e-02,...
                    9.5043397741477517e-02 ];
        end
    case 'full'
        % _________________________________________________________________
        % Degree 10, 25 Points, PI, Fully symmetric, References: [Z1] (not
        % listed in print but numerical values obtained from [Z2])
        % -----------------------------------------------------------------
        Qn.m  = [ 1, 3, 3, 3, 3, 6, 6 ];
        Qn.b1 = [ 0.3333333333333333333333333333333333, ...
            0.4272731788467755380904427175154472, ...
            0.1830992224486750205215743848502200, ...
            0.4904340197011305874539712223768484, ...
            0.0125724455515805327313290850210413, ...
            0.6542686679200661406665700955876279, ...
            0.1228045770685592734301298174812812 ];
        Qn.b2 = Qn.b1;
        Qn.b2(6:7) = [ 0.3080460016852477000000000000000000, ...
            0.0333718337393047862408164417747804 ];
        Qn.w = [ 0.0809374287976228802571131238165019, ...
            0.0772985880029631216825069823803434, ...
            0.0784576386123717313680939208343967, ...
            0.0174691679959294869176071632906781, ...
            0.0042923741848328280304804020901319, ...
            0.0374688582104676429790207654850445, ...
            0.0269493525918799596454494795810967 ];
end

end%
%%
function Qn = quadTriangleNonProductDegree11(varargin)
% Lower bound = 24

switch varargin{3}
    case 'allowOutside'
        switch varargin{1}
            case 'allowAsymmetric'
                Qn.m = zeros(1,26);
                % _________________________________________________________
                % Degree 11, 26 Points, PO, Asymmetric, Reference: [P3] see
                % Table A.1 page 80.
                % ---------------------------------------------------------
                Qn.m = zeros(26,1);
                Qn.b1 = [ 0.6258738065923632,...
                    0.0290632953572617,...
                    0.9330583391486744,...
                    0.0318590326985022,...
                    0.1624826332186683,...
                    0.0132184617612812,...
                    0.5918294793334099,...
                    0.8055758967896875,...
                    0.8043214954660826,...
                    0.1517977741136217,...
                    0.0290692872577444,...
                    0.0342125596440218,...
                    0.0248467017693192,...
                    0.3532703142982785,...
                    0.3617812618781889,...
                    0.7255128905173974,...
                    0.1526877583775490,...
                    0.5767609187090251,...
                    0.0957812464558656,...
                    0.1662869182873340,...
                    0.5809311067210347,...
                    0.1331546134128777,...
                    0.3613509750869149,...
                    0.3263421583773203,...
                    0.4933483827105916,...
                    0.2608387984712898 ];
                Qn.b2 = [ -0.0444983835604058,...
                    0.9408484762789769,...
                    0.0334630956930928,...
                    0.0355146752201139,...
                    0.0139846877219226,...
                    0.6427888268154324,...
                    0.3931416942741072,...
                    0.0246651880397084,...
                    0.1701217153423755,...
                    0.8180708557984559,...
                    0.1779293919407720,...
                    0.8141179370835360,...
                    0.3973211985920437,...
                    0.6159167605516321,...
                    0.0348462422413524,...
                    0.1342004546749753,...
                    0.1082745708776674,...
                    0.3299095455530797,...
                    0.5537082795511952,...
                    0.6799902364777285,...
                    0.0593349142682010,...
                    0.2982865414345181,...
                    0.4889662400184307,...
                    0.1749834270680896,...
                    0.2373606333716019,...
                    0.4018889231575169 ];
                Qn.w  = [ 1.7703429351573682e-03,...
                    1.1407181542799245e-02,...
                    1.4450148832022789e-02,...
                    1.4571713721870356e-02,...
                    1.5082717050370174e-02,...
                    1.7381504665411469e-02,...
                    1.9788048913795569e-02,...
                    2.3034628317738500e-02,...
                    2.4074295611862854e-02,...
                    2.5782293138424881e-02,...
                    2.7854536597135951e-02,...
                    2.8027408557939566e-02,...
                    3.0133750535085932e-02,...
                    3.5325613244976901e-02,...
                    3.9626273680208260e-02,...
                    4.7049077672300709e-02,...
                    4.7626634579448043e-02,...
                    4.9123786306245618e-02,...
                    5.5892517086728270e-02,...
                    5.6926272805302974e-02,...
                    5.7326951870023362e-02,...
                    6.3471175918900887e-02,...
                    6.3806180979645840e-02,...
                    7.4255945293930314e-02,...
                    7.7589779768121364e-02,...
                    7.8621220374552806e-02 ];
            case 'full'
                % _________________________________________________________
                % Degree 11, 27 Points, PO (3 points slightly outside),
                % Fully symmetric, References: [L2], see Table
                % 4(-continued) on page 31. Numerical values obtained from
                % [C3].
                % ---------------------------------------------------------
                Qn.m = [ 3 3 3 3 3 6 6 ];
                Qn.b1 = [ 0.0323649481112758931588480911328593,...
                    0.119350912282581309581102091581736,...
                    0.534611048270758309358680864963778,...
                    0.203309900431282473351326588944569,...
                    0.398969302965855222611381867187058,...
                    0.593201213428212752488840882179699,...
                    0.807489003159792153166724890348745 ];
                Qn.b2 = [ 0.0323649481112758931588480911328593,...
                    0.119350912282581309581102091581736,...
                    0.534611048270758309358680864963778,...
                    0.203309900431282473351326588944569,...
                    0.398969302965855222611381867187058,...
                    0.0501781383104946650738269077613887,...
                    0.0210220165361662971236385570923633 ];
                Qn.w = [ 6.82986550133893097935968864728942e-3,...
                    0.0180922702517090396691439677721401,...
                    4.63503164480338025330042487596786e-4,...
                    0.0296614886903870365579061947507576,...
                    0.0385747674574065614307555344183042,...
                    0.0261685559811020355585694685067805,...
                    0.0103538298195703444435161507885086 ]*2;
        end
    case 'inside'
        switch varargin{1}
            case 'allowAsymmetric'
                % WARNING: Error slightly high on this one ~1.0e-09!!!
                % _________________________________________________________                
                % Degree 11, 27 Points, PI, Rotationally symmetric,
                % References: Reported as new in [X1], see Table 1 on page
                % 671 (but no numerical values provided). Numerical values
                % obtained from online supplemental material of [P2].
                % ---------------------------------------------------------
                Qn.m = [ 3 3 3 3 3 3 3 3 3 ];
                Qn.b1 = [ 0.031732722526594,...
                    0.159837355915729,...
                    0.029507164313468,...
                    0.022822576496584,...
                    0.360374166927903,...
                    0.152649488567569,...
                    0.348698923498339,...
                    0.121263846601003,...
                    0.279902470374427 ];
                Qn.b2 = [ 0.934451908131695,...
                    0.815437813690059,...
                    0.801088959545196,...
                    0.594273309625116,...
                    0.614470889589723,...
                    0.716359493454929,...
                    0.530318579856843,...
                    0.559180146890916,...
                    0.450423215511194 ];
                Qn.w = [   0.013807279155821,...
                    0.021703235273358,...
                    0.027034127752203,...
                    0.027331241653936,...
                    0.028336285167031,...
                    0.048543667126133,...
                    0.052887299088776,...
                    0.055162253899304,...
                    0.058527944216771 ];
            case 'full'
                % _________________________________________________________                
                % Degree 11, 28 Points, PI, Fully symmetric, References:
                % [Z1] (not listed in print but numerical values obtained
                % from [Z2]).
                % ---------------------------------------------------------
                Qn.m       = [ 1, 3, 3, 3, 3, 3, 6, 6 ];
                Qn.b1      = [ 0.3333333333333333333333333333333333, ...
                    0.0309383552454307848951950149913047, ...
                    0.4364981811341288419176152765599732, ...
                    0.4989847637025932662879869838313909, ...
                    0.2146881979585943366068758138782509, ...
                    0.1136831040421133902052931562283618, ...
                    0.8256187661648629043588062003083580, ...
                    0.6404723101348652676770365908189668 ];
                Qn.b2      = Qn.b1;
                Qn.b2(7:8) = [ 0.1597423045918501898008607882250075, ...
                    0.3117837157095990000000000000000000 ];
                Qn.w       = [ 0.0811779602968671595154759687498236, ...
                    0.0123240435069094941184739010162328, ...
                    0.0628280097444101072833394281602940, ...
                    0.0122203790493645297552122150039379, ...
                    0.0677013489528115099209888618232256, ...
                    0.0402196936288516904235668896075687, ...
                    0.0147622727177161013362930655877821, ...
                    0.0407279964582990396603369584816179 ];
        end
end

end%
%%
function Qn = quadTriangleNonProductDegree12(varargin)
% Lower bound = 28

switch varargin{1}
    case 'allowAsymmetric'
        switch varargin{3}
            case 'allowOutside'
                % WARNING: Error slightly high on this one ~1.0e-09!!!
                % _________________________________________________________
                % Degree 12, 31 Points, PO (3 points outside), Rotationally
                % symmetric, Reference: [P2] reported as new in Table 4 on
                % page 46. Numerical values obtained from online supple-
                % mental material of [P2].
                % ---------------------------------------------------------
                Qn.m = [ 1 3 3 3 3 3 3 3 3 3 3 ];
                Qn.b1 = [    0.333333333333333,...
                    0.030113195801208,...
                    0.328665303214460,...
                    0.048124921152851,...
                    0.142850841527365,...
                    0.208191925148008,...
                    0.425825822203966,...
                    0.323868233593083,...
                    0.154176405413344,...
                    0.349383618039355,...
                    0.299315867642759 ];
                Qn.b2 = [     0.333333333333333,...
                    -0.094716062682498,...
                    0.008045645209261,...
                    0.923861926809378,...
                    0.031019238692721,...
                    0.764831246745195,...
                    0.550395696311637,...
                    0.079851341494144,...
                    0.707647956311328,...
                    0.524861100312631,...
                    0.229249615293796 ];
                Qn.w = [ 0.021409425726798,...
                    0.000083668178724,...
                    0.012413637067985,...
                    0.016239945629217,...
                    0.024194039324984,...
                    0.026967365586006,...
                    0.027774427657215,...
                    0.047016778451346,...
                    0.048593414405827,...
                    0.058505280711051,...
                    0.064408301078713 ];
            case 'inside'
                % _________________________________________________________                
                % Degree 12, 32 Points, PI, Asymmetric, Reference: [T1] see
                % Appendix page 190.
                % ---------------------------------------------------------
                Qn.m = zeros(32,1);
                Qn.b1 = [ 0.379860210934020,...
                    0.301417093209093,...
                    0.558025289531203,...
                    0.125122995058104,...
                    0.211179399098049,...
                    0.854314749475804,...
                    0.717881858980523,...
                    0.466317874623231,...
                    0.250155003353392,...
                    0.079955384841381,...
                    0.710081259568365,...
                    0.497320633777966,...
                    0.260770682565629,...
                    0.089602705800587,...
                    0.023088148766116,...
                    0.129532969004336,...
                    0.093448087604441,...
                    0.955269193570060,...
                    0.845935398373144,...
                    0.616009296172675,...
                    0.393165103196048,...
                    0.189206330617159,...
                    0.043010560106405,...
                    0.858158884215331,...
                    0.627315319232412,...
                    0.363846604460775,...
                    0.155570668968980,...
                    0.029754117496842,...
                    0,...
                    0.025716283623694,...
                    0.024506286636990,...
                    0.009229690905965 ];
                Qn.b2 = [ 0.210785259391404,...
                    0.409786577770025,...
                    0.213777432530060,...
                    0.619381257362556,...
                    0.244982965093490,...
                    0.071871496101589,...
                    0.203768481077730,...
                    0.408963804491245,...
                    0.627682615680314,...
                    0.826003314017560,...
                    0.064413220382261,...
                    0.070566724344037,...
                    0.095428585810585,...
                    0.116386499067277,...
                    0.749189739790679,...
                    0.422605657433460,...
                    0.243458133948800,...
                    0.023551733249579,...
                    0.154064601626856,...
                    0.361181591189672,...
                    0.581689214740147,...
                    0.788601719223132,...
                    0.945475073220971,...
                    0,...
                    0,...
                    0.014566514788347,...
                    0.021152223383122,...
                    0.027110971356256,...
                    0.927348974483950,...
                    0.544446676271925,...
                    0.332129083947645,...
                    0.146044961672176 ];
                Qn.w = [ 0.118875667902271,...
                    0.150444125206649,...
                    0.126329092845313,...
                    0.101929849753575,...
                    0.094999150650614,...
                    0.044981492398316,...
                    0.079147211585944,...
                    0.119979414654212,...
                    0.106704166097642,...
                    0.061058344824145,...
                    0.082563774790925,...
                    0.096297610073815,...
                    0.091875684331583,...
                    0.061150555208078,...
                    0.043370170834023,...
                    0.108293745226335,...
                    0.055887468639760,...
                    0.013351800054735,...
                    0.015428984747250,...
                    0.050198346855370,...
                    0.056291117210427,...
                    0.041240008239364,...
                    0.014239502872161,...
                    0.013691069308687,...
                    0.019309417484873,...
                    0.037090960843213,...
                    0.036967371622462,...
                    0.021018653471205,...
                    0.009776099629320,...
                    0.056339308919460,...
                    0.049808146403015,...
                    0.021361687315257 ]*0.5;
        end
    case 'full'
        % _________________________________________________________________
        % Degree 12, 33 Points, PI, Fully symmetric, References: [D2], see
        % Appendix II page 1141. Numerical values obtained from [Z2]
        % -----------------------------------------------------------------
        Qn.m =[ 3,3,3,3,3,6,6,6 ];
        Qn.b1 = [ 0.0213173504532103702468569755157282,...
            0.2712103850121159223459513403968947,...
            0.1275761455415859246738963251542836,...
            0.4397243922944602729797366234843611,...
            0.4882173897738048825646620652588110,...
            0.6958360867878034221416355232360725,...
            0.8580140335440726305905366166261782,...
            0.6089432357797878068561924377637101 ];
        Qn.b2(1:5) = Qn.b1(1:5);
        Qn.b2(6:8) = [ 0.2813255809899395482481306929745527,...
            0.1162519159075971412413541478426018,...
            0.2757132696855141939747963460797640 ];
        Qn.w = [ 0.0061662610515590172338664837852304,...
            0.0628582242178851003542705130928825,...
            0.0347961129307089429893283972949994,...
            0.0436925445380384021354572625574750,...
            0.0257310664404553354177909230715644,...
            0.0223567732023034457118390767023200,...
            0.0173162311086588923716421008110341,...
            0.0403715577663809295178286992522368 ];
end

end%
%%
function Qn = quadTriangleNonProductDegree13(varargin)
% Lower bound = 31

switch varargin{1}
    case 'allowAsymmetric'
        % _________________________________________________________________
        % Degree 13, 35 Points, PI, Reflection symmetry, Reference:  A
        % 35-point rule with reflection symmetry is reported as new in
        % [X1], see Table 1 on page 671 (but no numerical values provided).
        % Unsure where to obtain numerical values.
        % -----------------------------------------------------------------
        % WARNING: Error slightly high on this one ~1.0e-09!!!   
        % _________________________________________________________________
        % Degree 13, 36 Points, PI, Rotationally symmetric, References: A
        % 36-point, rotationally symmetric rule is reported as new in [X1],
        % see Table 1 on page 671 (but no numerical values provided).
        % Numerical values obtained from online supplemental material of
        % [P2]. (Note: It is unclear if this is the same rule reported in
        % [X1] but likely is, as it is not reported as new by the author.)
        % -----------------------------------------------------------------     
        Qn.m = [ 3 3 3 3 3 3 3 3 3 3 3 3 ];
        Qn.b1 = [ 0.026016776102192,...
            0.018883631298673,...
            0.131639171218806,...
            0.020731528553135,...
            0.476642895184470,...
            0.302386954450794,...
            0.101539779723999,...
            0.249614427797335,...
            0.102029240543062,...
            0.226789276022037,...
            0.437409517911078,...
            0.380758579162245 ];
        Qn.b2 = [ 0.024022213893301,...
            0.122080831671692,...
            0.022874705099002,...
            0.280737796608390,...
            0.504849456195797,...
            0.019975752788321,...
            0.119028215975855,...
            0.103589966905477,...
            0.276888257363516,...
            0.233728374217555,...
            0.095992093320340,...
            0.227855357118669 ];
        Qn.w = [ 0.008079971892246,...
            0.012892029137773,...
            0.016560237922093,...
            0.019023135868897,...
            0.019359657480323,...
            0.019583798359224,...
            0.030281493825969,...
            0.038164066479861,...
            0.038199338317111,...
            0.040847152891851,...
            0.041085695986565,...
            0.049256755171419 ];
    case 'full'
        if strcmp(varargin{2},'allowNegative') && ...
                strcmp(varargin{3},'allowOutside')
            % _____________________________________________________________
            % Degree 13, 36 Points, NO, Fully symmetric, Reference: [B1],
            % see Table 3 on page 9. Numerical values obtained from [C3].
            % (Note: This rule is of little practical value as there are 3
            % points that are a significant distance outside the triangle.)
            % -------------------------------------------------------------
            Qn.m = [ 3 3 3 3 3 3 6 6 6 ];
            Qn.b1 = [-1.09732124710628115928776639808403,...
                0.488287850733405315708960132445380,...
                0.271000295524474716503595034895827,...
                0.0247884310336613610583520745170274,...
                0.107120353118147709346761788996197,...
                0.440323874478061332339068542230318,...
                0.850459062644356742678494400238734,...
                0.683758575887968213394629726245143,...
                0.631364930935447484201224024482597 ];
            Qn.b2 = [ -1.09732124710628115928776639808403,...
                0.488287850733405315708960132445380,...
                0.271000295524474716503595034895827,...
                0.0247884310336613610583520745170274,...
                0.107120353118147709346761788996197,...
                0.440323874478061332339068542230318,...
                0.0208215208466316169587306885458108,...
                0.0229194828048128099474800958462720,...
                0.115458022821994138042223119054549 ];
            Qn.w = [ -2.21945896962485538956436213495318e-15,...
                0.0119375420275846679217718138313380,...
                0.0315948917993914165647154983903209,...
                4.02253490826229479991501578944327e-3,...
                0.0139280485567761024760117619804009,...
                0.0253425305335128838728212926073213,...
                7.43354416099169030524698401193993e-3,...
                0.0107878499081378862592389379193987,...
                0.0216991653514411836807145325302771 ]*2;
        else
            % _____________________________________________________________            
            % Degree 13, 37 Points, PI, Fully symmetric, Reference: [B1],
            % see Table 2 on page 9. Numerical values obtained from [C3].
            % -------------------------------------------------------------
            Qn.m  = [ 1,3,3,3,3,3,3,6,6,6 ];
            Qn.b1 = [ 0.3333333333333333333333333333333333,...
                0.5000000000000000000000000000000000,...
                0.0246071886432302181878499494124643,...
                0.420308753101194683716920537182100,...
                0.227900255506160619646298948153592,...
                0.116213058883517905247155321839271,...
                0.476602980049079152951254215211496,...
                0.851775587145410469734660003794168,...
                0.692797317566660854594116289398433,...
                0.637955883864209538412552782122039 ];
            Qn.b2(1:7) = Qn.b1(1:7);
            Qn.b2(8:10) = [ 0.0227978945382486125477207592747430,...
                0.0162757709910885409437036075960413,...
                0.0897330604516053590796290561145196 ];
            Qn.w  = [ 0.0293480398063595158995969648597808,...
                2.67845189554543044455908674650066e-3,...
                3.92538414805004016372590903990464e-3,...
                0.0253344765879434817105476355306468,...
                0.0250401630452545330803738542916538,...
                0.0158235572961491595176634480481793,...
                0.0157462815379843978450278590138683,...
                7.90126610763037567956187298486575e-3,...
                7.99081889046420266145965132482933e-3,...
                0.0182757511120486476280967518782978 ]*2;
        end
end

end%
%%
function Qn = quadTriangleNonProductDegree14(varargin)
% Lower bound = 36

switch varargin{1}
    case 'allowAsymmetric'
        switch varargin{3}
            case 'allowOutside'
                % WARNING: Error slightly high on this one ~1.0e-09!!!
                % _________________________________________________________
                % Degree 14, 40 Points, PO (3 points slightly outside),
                % Rotationally symmetric, Reference: [P2], reported as new
                % in Table 4 on page 46. Numerical values obtained from
                % online supplemental material of [P2].
                % ---------------------------------------------------------
                Qn.m = [ 1 3 3 3 3 3 3 3 3 3 3 3 3 3 ];
                Qn.b1 = [ 0.333333333333333,...
                    0.122379515529847,...
                    0.256919929576504,...
                    0.022412698148715,...
                    0.107983615865477,...
                    0.454399148709709,...
                    0.260936998850154,...
                    0.108651813403199,...
                    0.363151184998713,...
                    0.188070088519435,...
                    0.393963047077245,...
                    0.204286427343951,...
                    0.315511922686270,...
                    0.324023300923113 ];
                Qn.b2 = [ 0.333333333333333,...
                    -0.022116463863325,...
                    0.742961945176952,...
                    0.021296338767865,...
                    0.873605481594920,...
                    0.008402118890483,...
                    0.025735252088221,...
                    0.047317979503788,...
                    0.600340345913398,...
                    0.741933521291307,...
                    0.083219241365617,...
                    0.137280339301186,...
                    0.533376567358603,...
                    0.228102726523557 ];
                Qn.w = [ 0.017219601997331,...
                    0.001223437032443,...
                    0.005625806065308,...
                    0.006180344872468,...
                    0.010957697357973,...
                    0.011462343010640,...
                    0.022295784039866,...
                    0.023382863187460,...
                    0.026558067295112,...
                    0.030850488270756,...
                    0.040681051448570,...
                    0.043652397747231,...
                    0.050893743280553,...
                    0.053829442392510 ];
            case 'inside'
                % _________________________________________________________
                % Degree 14, 41 Points, PI, Reflection symmetry, Reference:
                % A 41-point rule with reflection symmetry is reported as
                % new in [X1], see Table 1 on page 671 (but no numerical
                % values provided). Unsure where to obtain numerical
                % values.
                % ---------------------------------------------------------
        end
    case 'full'
        % _________________________________________________________________
        % Degree 14, 42 Points, PI, Fully Symmetric, References: [D2], see
        % Appendix II page 1141. Numerical values obtained from [Z2].
        % -----------------------------------------------------------------
        Qn.m = [ 3,3,3,3,3,3,6,6,6,6 ];
        Qn.b1 = [ 0.17720553241254343695661069046505908,...
            0.01939096124870104817825009505452951,...
            0.06179988309087260126747882843693579,...
            0.41764471934045392250944082218564344,...
            0.48896391036217863867737602045239024,...
            0.27347752830883865975494428326269856,...
            0.17226668782135557837528960161365733,...
            0.57022229084668317349769621336235426,...
            0.29837288213625775297083151805961273,...
            0.11897449769695684539818196192990548 ];
        Qn.b2(1:6)  = Qn.b1(1:6);
        Qn.b2(7:10) = [ 0.05712475740364793903567712421891471,...
            0.09291624935697182475824858954872035,...
            0.01464695005565440967054132792007421,...
            0.00126833093287202508724640109549269 ];
        Qn.w = [ 0.04216258873699301753823043732418613,...
            0.00492340360240008168182602350904215,...
            0.01443369966977666760170992148065332,...
            0.03278835354412535064131097873862534,...
            0.02188358136942889064084494596332597,...
            0.05177410450729158631478491016639640,...
            0.02466575321256367396287524518363623,...
            0.03857151078706068322848902781041086,...
            0.01443630811353384049608869199901580,...
            0.00501022883850067176986009308248912 ];
end

end%
%%
function Qn = quadTriangleNonProductDegree15(varargin)
% Lower bound = 40

switch varargin{1}
    case 'allowAsymmetric'
        % WARNING: Error slightly high on this one ~1.0e-09!!!
        % _________________________________________________________________
        % Degree 15, 46 Points, PI, Rotationally symmetric, Reference:  A 
        % 46-point rule with rotation symmetry is reported as new in [X1], 
        % see Table 1 on page 671 (but no numerical values provided). 
        % Numerical values obtained from online supplemental material of 
        % [P2]. (Note: It is unclear if the rule below from [P2] is the 
        % same as that reported earlier in [X1].)
        % -----------------------------------------------------------------
        Qn.m = [ 1 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ];
        Qn.b1 = [   0.333333333333333,...
            0.018434795476157,...
            0.094469391541666,...
            0.221059537257313,...
            0.018316125580224,...
            0.384111820811407,...
            0.017609576799085,...
            0.016211104266108,...
            0.094583522597998,...
            0.223020541186686,...
            0.091368633556671,...
            0.384530936051927,...
            0.215038741798252,...
            0.085318970181729,...
            0.204230412205111,...
            0.363890898963924 ];
        Qn.b2 = [    0.333333333333333,...
            0.021020578568987,...
            0.014008492164339,...
            0.014600157192749,...
            0.106831432538477,...
            0.013506565318992,...
            0.246869440424026,...
            0.420027323435868,...
            0.078193785237664,...
            0.069024652212796,...
            0.197441620058398,...
            0.073097762126704,...
            0.155893061854053,...
            0.359007553448067,...
            0.286157214943029,...
            0.181085973861799 ];
        Qn.w = [    0.046749612087783,...
            0.005027856097248,...
            0.007598681940105,...
            0.010732119583648,...
            0.010830851965745,...
            0.012248727250487,...
            0.014397839583791,...
            0.014983807152202,...
            0.019232683397058,...
            0.020834684882606,...
            0.028576965678339,...
            0.029031677885589,...
            0.030240283270454,...
            0.032511521853897,...
            0.040187126563652,...
            0.041315302199252 ];
    case 'full'
        switch varargin{2}
            case 'allowNegative'
                % _________________________________________________________
                % Degree 15, 48 Points, NI, Fully Symmetric, Reference: 
                % [P2], reported as new in Table 4 on page 46. Numerical 
                % values obtained from online supplemental material of 
                % [P2].
                % ---------------------------------------------------------
                Qn.m = [ 3 3 3 3 3 3 6 6 6 6 6 ];
                Qn.b1 = [    0.295932339143481,...
                    0.110411372943988,...
                    0.055556777591098,...
                    0.018926277277298,...
                    0.232611403922657,...
                    0.258126987267139,...
                    0.001770793087525,...
                    0.017499897301342,...
                    0.018423065211738,...
                    0.091205272592313,...
                    0.097542934824371 ];
                Qn.b2 = [ 0.295932339143481,...
                    0.110411372943988,...
                    0.055556777591098,...
                    0.490536861361351,...
                    0.232611403922657,...
                    0.370936506366430,...
                    0.038231237353261,...
                    0.149819041618279,...
                    0.307452940137402,...
                    0.203858752868168,...
                    0.363255400132046 ];
                Qn.w = [ -0.087456325826448,...
                    0.011825359733916,...
                    0.013633359800591,...
                    0.018152749803642,...
                    0.060151078120269,...
                    0.122180120594431,...
                    0.002561371980868,...
                    0.012198751372067,...
                    0.016436943369444,...
                    0.028386552060758,...
                    0.037839876770329 ];
            case 'positive'
                % _________________________________________________________                
                % Degree 15, 49 Points, PI, Fully Symmetric, References:
                % [W1], reported in Table 1 on page 1240 (without numerical
                % values and not reported as new). Numerical values
                % obtained from [Z2], which credits the first author of
                % [W1]. (Note: A 49-point PI rule with full symmetry is
                % reported (as new) earlier in [X1], see Table 1 on page
                % 671, though it is unknown if this is the same rule from
                % [W1].)
                % ---------------------------------------------------------
                Qn.m = [ 1 3 3 3 3 6 6 6 6 6 6 ];
                Qn.b1 = [ 0.33333333333333333333333333333333333,...
                    0.11022229622834687297855264132259850,...
                    0.05197643301003435047003197947889073,...
                    0.49114565807532554119014945122395425,...
                    0.39315718888435884048226809785071794,...
                    0.03737440487572919066543605209836625,...
                    0.24824877798467321198263980694374938,...
                    0.20699402274830217740486528153682148,...
                    0.14854110526954708137688902238435510,...
                    0.30674237923596382376588728350286621,...
                    0.36703198754220473278855469116984882 ];
                Qn.b2(1:5)  = Qn.b1(1:5);
                Qn.b2(6:11) = [ 0.96251835223001214880811969560396873,...
                    0.19316669854521416819773100288721521,...
                    0.08689590883549962551575259619781217,...
                    0.01743682539845430796259020511767948,...
                    0.01749251095825766163254977051260599,...
                    0.09034802175864556044634095119222305 ];
                Qn.w = [ 0.02357126703190634206659321140821418,...
                    0.01517314955721170450311858877690239,...
                    0.01297600128392884154979521077280757,...
                    0.01706629596800615670942600046160914,...
                    0.04576001946273760698482638108892258,...
                    0.00222757447282223154006065426298478,...
                    0.02701014165986947101315702212247500,...
                    0.02608377963958756403057720483642768,...
                    0.01211015327702828337230795926322736,...
                    0.01564785059680444573399007149035058,...
                    0.03417088937929479242522512890637806 ];
        end
end

end%
%%
function Qn = quadTriangleNonProductDegree16(varargin)
% Lower bound = 45

switch varargin{1}
    case 'allowAsymmetric'
        switch varargin{3}
            case 'allowOutside'
                % WARNING: Error slightly high on this one ~1.0e-09!!!
                % _________________________________________________________
                % Degree 16, 51 Points, PO (3 points outside), Rotationally
                % symmetric, Reference: [P2], reported as new in Table 4 on
                % page 46. Numerical values obtained from online supple-
                % mental material of [P2].
                % ---------------------------------------------------------
                Qn.m = [ 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ];
                Qn.b1 = [ 0.260234434945475,...
                    0.087654435676473,...
                    0.300673047577842,...
                    0.018849925360919,...
                    0.098416684250053,...
                    0.460403016961182,...
                    0.073829911003046,...
                    0.186109933337396,...
                    0.383202318286473,...
                    0.186107887529837,...
                    0.329435670084688,...
                    0.272057325895407,...
                    0.141235380655938,...
                    0.449015635136170,...
                    0.266798367513169,...
                    0.282382457106112,...
                    0.349747321380105 ];
                Qn.b2 = [ -0.007796075391171,...
                    0.909443871530900,...
                    0.700388097083946,...
                    0.016472917542655,...
                    0.011941006711738,...
                    0.011787969349286,...
                    0.868681210960494,...
                    0.787962853056885,...
                    0.588557349843886,...
                    0.042129359576000,...
                    0.038398059357352,...
                    0.651051709131074,...
                    0.729399317013244,...
                    0.097621841960192,...
                    0.137399270890935,...
                    0.534917330044391,...
                    0.238132150850676 ];
                Qn.w = [ 0.002658404255859,...
                    0.003399622303254,...
                    0.003519109003499,...
                    0.004044140411709,...
                    0.006829471151259,...
                    0.010440167558510,...
                    0.014730712491920,...
                    0.014997188989449,...
                    0.016563214492523,...
                    0.019748195843966,...
                    0.021203128918975,...
                    0.024448645222118,...
                    0.028149869751439,...
                    0.036902116480955,...
                    0.037047736203275,...
                    0.041332532979016,...
                    0.047319077275608 ];
            case 'inside'
                % WARNING: Error slightly high on this one ~1.0e-09!!!
                % _________________________________________________________                
                % Degree 16, 52 Points, PI, Rotationally Symmetric, 
                % References: [X1], reported as new in Table 1 on page 671. 
                % Numerical values obtained from online supplemental 
                % material of [P2]. (Note: It is unclear if the rule below 
                % from [P2] is the same as that reported earlier in [X1].)
                % ---------------------------------------------------------
                Qn.m = [ 1 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ];
                Qn.b1 = [ 0.333333333333333,...
                    0.015961011713107,...
                    0.003821930841719,...
                    0.092621730805489,...
                    0.009809849705384,...
                    0.234453555493240,...
                    0.051657391379326,...
                    0.015979112409302,...
                    0.396856475222831,...
                    0.140368863416206,...
                    0.048646064695552,...
                    0.067545698927945,...
                    0.160476192471739,...
                    0.280462713730914,...
                    0.147356925256594,...
                    0.424740746647569,...
                    0.451724383070898,...
                    0.288318735917362 ];
                Qn.b2 = [ 0.333333333333333,...
                    0.016883125432530,...
                    0.099349672744069,...
                    0.004717750478119,...
                    0.246738918291676,...
                    0.012521221333541,...
                    0.057775785013585,...
                    0.413442936726308,...
                    0.016762640234767,...
                    0.050400029219950,...
                    0.153602469291318,...
                    0.293562718665031,...
                    0.276119726602890,...
                    0.077279834362079,...
                    0.148012388839050,...
                    0.208953636106142,...
                    0.087033324580191,...
                    0.190229119570889 ];
                Qn.w = [    0.036632403806727,...
                    0.003544304877043,...
                    0.004017588337041,...
                    0.004190720091296,...
                    0.008562782685159,...
                    0.009979007817050,...
                    0.010212563539779,...
                    0.014081474710640,...
                    0.014629924655641,...
                    0.016056466268480,...
                    0.017024341971023,...
                    0.024955767639172,...
                    0.028199116717009,...
                    0.028270578986067,...
                    0.029226110917024,...
                    0.034006909002989,...
                    0.034235623800026,...
                    0.039929250048986 ];
        end
    case 'full'
        % _________________________________________________________________
        % Degree 16, 55 Points, PI, Fully Symmetric, Reference: [Z1],
        % reported in Table 4.1 on page 93. Numerical values obtained from
        % [Z2].
        % -----------------------------------------------------------------
        Qn.m = [ 1 3 3 3 3 3 3 6 6 6 6 6 6 ];
        Qn.b1 = [ ...
            0.3333333333333333333333333333333333,...
            0.0817949831313738726414655931188610,...
            0.1653006019697796506267619329335566,...
            0.4685921053494613866946028972966056,...
            0.0144388134454166826141089566956602,...
            0.2417842853917833534068944592932077,...
            0.4953103429877699640654950868774055,...
            0.6505134026613522994311446848416867,...
            0.6040112814959970398494041030359670,...
            0.8021682575747416636168619478116671,...
            0.7565056064428283965511540757580608,...
            0.4659384387141181848838107335915464,...
            0.9063948439920415013624996618653400 ];
        Qn.b2(1:7) = Qn.b1(1:7);
        Qn.b2(8:13) = [ ...
            0.3313997445370895565813231681825939,...
            0.3032471627499421850415521780783469,...
            0.1880280595212371734441821142939888,...
            0.1835046685222968636823802774370004,...
            0.3596459487975046000000000000000100,...
            0.0771943712957554322825152250527139 ];
        Qn.w = [ ...
            0.0480221886803770905518394045805199,...
            0.0147091003068019271034036428618692,...
            0.0295445865493192559953097267964641,...
            0.0261250173510883774985975654917156,...
            0.0027803873523900069750030161386621,...
            0.0318217730005366495034272900559496,...
            0.0086458343495096599011737341698489,...
            0.0143003329044953651466164253682521,...
            0.0278497772036008299522298734239535,...
            0.0070416734066360975623701880892807,...
            0.0178998382599337286017702090758108,...
            0.0274582003843497630724700381009172,...
            0.0072997969394317620841125440877777 ];
end

end%
%%
function Qn = quadTriangleNonProductDegree17(varargin)
% Lower bound = 49

switch varargin{1}
    case 'allowAsymmetric'
        % WARNING: Error slightly high on this one ~1.0e-09!!!
        % _________________________________________________________________
        % Degree 17, 57 Points, PI, Rotationally Symmetric, Reference: 
        % [P2], reported as new in Table 4 on page 46. Numerical values ob-
        % tained from online supplemental material of [P2].
        % -----------------------------------------------------------------
        Qn.m  = [ 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ];
        Qn.b1 = [ 0.014720966195627,...
            0.075827872337700,...
            0.310585516558464,...
            0.014488314624940,...
            0.178143852253020,...
            0.466133550795935,...
            0.014736566166282,...
            0.014332535547880,...
            0.075303349390642,...
            0.317339393501728,...
            0.075354265490355,...
            0.182570774440877,...
            0.179710032722075,...
            0.175807941715677,...
            0.458731499688784,...
            0.075319692074374,...
            0.163305177925385,...
            0.311813568699615,...
            0.306048259112028 ];
        Qn.b2 = [ 0.017639984855507,...
            0.007697475843772,...
            0.008320091951046,...
            0.090164039580071,...
            0.016423858046516,...
            0.012335581768738,...
            0.209325171157458,...
            0.358602315795424,...
            0.055968132994742,...
            0.051694878418799,...
            0.154598211665730,...
            0.266530059635147,...
            0.081586227754906,...
            0.180353640520786,...
            0.474126007719327,...
            0.293754105362968,...
            0.384251187765942,...
            0.139501525996608,...
            0.267720653545962 ];
        Qn.w  = [ 0.003377036008100,...
            0.003819925670194,...
            0.006624026081441,...
            0.007284900734873,...
            0.009965002473405,...
            0.010276345098596,...
            0.010309491303777,...
            0.011602229136800,...
            0.012469743787482,...
            0.019632479515532,...
            0.020101507075145,...
            0.020906045568565,...
            0.021332247671604,...
            0.023749377189185,...
            0.024777024008132,...
            0.025585570242273,...
            0.031310858803301,...
            0.031590362549438,...
            0.038619160415490 ];
    case 'full'
        switch varargin{3}
            case 'allowOutside'
                % _________________________________________________________
                % Degree 17, 58 Points, PO (6 points slighlty outside),
                % full Symmetry,  Reference: [P2], reported as new in Table
                % 3 on page 46. Numerical values obtained from online
                % supplemental material of [P2].
                % ---------------------------------------------------------
                Qn.m  = [ 1 3 3 3 3 3 3 3 6 6 6 6 6 6 ];
                Qn.b1 = [ 0.333333333333333,...
                    0.014939063814611,...
                    0.009805934151716,...
                    0.080176519909013,...
                    0.066547723826800,...
                    0.159149357451283,...
                    0.180837956069870,...
                    0.244597319860783,...
                    -0.017099279594795,...
                    0.015606149478885,...
                    0.012499472547638,...
                    0.024893796930193,...
                    0.060954488983763,...
                    0.110946632046320 ];
                Qn.b2 = [ 0.333333333333333,...
                    0.014939063814611,...
                    0.495097032924142,...
                    0.080176519909013,...
                    0.466726138086600,...
                    0.159149357451283,...
                    0.409581021965065,...
                    0.244597319860783,...
                    0.304698462180812,...
                    0.077467710451681,...
                    0.183007377411325,...
                    0.331361033435097,...
                    0.190839791501631,...
                    0.303713312450755 ];
                Qn.w  = [ 0.042349520256915,...
                    0.002919669034578,...
                    0.009333819351488,...
                    0.015579210009133,...
                    0.025862383995094,...
                    0.027157513575493,...
                    0.038226206597189,...
                    0.038609391683444,...
                    0.000525996151839,...
                    0.006836980512764,...
                    0.007740298196833,...
                    0.017306871686136,...
                    0.017633848414215,...
                    0.030720321205517 ];
            case 'inside'
                % _________________________________________________________                
                % Degree 17, 60 Points, PI, Fully Symmetric, Reference:
                % [X1], reported in Table 4.1 on page 93. Numerical values
                % obtained from [Z2].
                % ---------------------------------------------------------
                Qn.m  = [ 3 3 3 3 3 3 3 3 6 6 6 6 6 6 ];
                Qn.b1 = [ 0.24056306963626902977934166278860247,...
                    0.08092323589766073062004798772340524,...
                    0.01001414912499135088254841140047604,...
                    0.15437652078663289107430782196727737,...
                    0.41716986201996268598941663596983268,...
                    0.47086974573840098186867398532866671,...
                    0.49811803384542204444865152799034832,...
                    0.36473840565291924199871629076775930,...
                    0.10986590708262616153720966373050601,...
                    0.20493227462918790108024139159058423,...
                    0.05813921564266244000000000000000000,...
                    0.13859554086776482539309659376771751,...
                    0.34660546952009260087829868774027952,...
                    0.24821986889585591697209834974065293 ];
                Qn.b2(1:8) = Qn.b1(1:8);
                Qn.b2(9:14) = [ 0.30466576969866569523225839525499357,...
                    0.05248758390645425414786013344982922,...
                    0.01500053995225954378593128753997425,...
                    0.01501023347973182500884052064335399,...
                    0.02336212893314653752768977049783837,...
                    0.00000099999999999965762180770907324 ];
                Qn.w  = [ 0.038292540080035687494251688894918170,...
                    0.01669528699775339594318472807122019,...
                    0.00143512454359061224492929722268097,...
                    0.02864276849185053630399044294140648,...
                    0.03408569078206214964786810427776196,...
                    0.02467274200053089056925349793140004,...
                    0.00586679757537134154263246190805349,...
                    0.02321859500422896151112767944153052,...
                    0.03084965458251406099116307348593810,...
                    0.01881398544005420038782109445200127,...
                    0.00512343450397285555007197439694996,...
                    0.00701239348475201777118052342883162,...
                    0.01538229443504461311363086994295179,...
                    0.00303013148261713122418018061550803 ];
        end
end

end%
%%
function Qn = quadTriangleNonProductDegree18(varargin)
% Lower bound = 55

switch varargin{1}
    case 'allowAsymmetric'
        switch varargin{2}
            case 'allowNegative'
                % _________________________________________________________
                % Degree 18, 64 Points, NI, Rotationally Symmetric, 
                % Reference: [P2], reported as new in Table 4 on page 46. 
                % Numerical values obtained from online supplemental 
                % material of [P2].
                % ---------------------------------------------------------
                Qn.m = [ 1 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ];
                Qn.b1 = [ 0.333333333333333,...
                    0.009224371770164,...
                    0.011588174868223,...
                    0.137109449232487,...
                    0.054074658130232,...
                    0.012356208531742,...
                    0.254443010786652,...
                    0.012200654548168,...
                    0.395592717135403,...
                    0.013989490384595,...
                    0.061242622253631,...
                    0.140187294281714,...
                    0.065953460657672,...
                    0.062413944208808,...
                    0.074301715278995,...
                    0.255758429076781,...
                    0.389748898002750,...
                    0.155168432990030,...
                    0.170472927860750,...
                    0.286469196549182,...
                    0.155079032148931,...
                    0.306409473651319 ];
                Qn.b2 = [ 0.333333333333333,...
                    0.009540376389022,...
                    0.059910234445417,...
                    0.008134232223840,...
                    0.015964034865851,...
                    0.156321458214785,...
                    0.013228036080598,...
                    0.438518077301825,...
                    0.012905266057921,...
                    0.287578430196017,...
                    0.082131031407760,...
                    0.050988464618208,...
                    0.182306457551239,...
                    0.411715469349846,...
                    0.293703944612865,...
                    0.068183146863573,...
                    0.066997893569429,...
                    0.140086682350670,...
                    0.261254002060855,...
                    0.163053894264229,...
                    0.408496920155351,...
                    0.304333597697523  ];
                Qn.w = [ -0.125845221584873,...
                    0.001297271105247,...
                    0.004404344515906,...
                    0.004553802778772,...
                    0.005252211687483,...
                    0.007374637872210,...
                    0.008834786872593,...
                    0.009647776424883,...
                    0.009867099780421,...
                    0.010260713189232,...
                    0.011796695983077,...
                    0.012985507535111,...
                    0.016318205824573,...
                    0.017525616506223,...
                    0.018411831442871,...
                    0.019068970594065,...
                    0.020582184685953,...
                    0.023589141249305,...
                    0.031090952263188,...
                    0.032039966987225,...
                    0.032660417772955,...
                    0.077719605456997 ];
            case 'positive'
                % _________________________________________________________
                % Degree 18, 65 Points, PI, Reflection Symmetry, Reference:
                % A 65-point rule with reflection symmetry is reported as
                % new in [X1], see Table 1 on page 671 (but no numerical
                % values provided). Unsure where to obtain numerical
                % values.
                % ---------------------------------------------------------
                % _________________________________________________________
                % Degree 18, 66 Points, PI, Rotationally Symmetric, 
                % References: [X1], reported as new in Table 1 on page 671. 
                % Numerical values obtained from online supplemental 
                % material of [P2]. (Note: It is unclear if the rule below 
                % from [P2] is the same as that reported earlier in [X1].)
                % ---------------------------------------------------------
                Qn.m = [ 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ];
                Qn.b1 = [ 0.003697809640551,...
                    0.014914399227999,...
                    0.001697399962193,...
                    0.080662083946625,...
                    0.198747901402311,...
                    0.346702719147276,...
                    0.048634633387104,...
                    0.012831862984945,...
                    0.029359538994221,...
                    0.490788394693759,...
                    0.138618784059789,...
                    0.263897606352201,...
                    0.045591174493061,...
                    0.394010358537068,...
                    0.106867206940965,...
                    0.220575279425483,...
                    0.123653835041299,...
                    0.071334879410098,...
                    0.224126910324972,...
                    0.351619049955643,...
                    0.360762340567732,...
                    0.311300128982519 ];
                Qn.b2 = [ 0.074357923508124,...
                    0.013690913073363,...
                    0.208913690615461,...
                    0.008243532690291,...
                    0.006976595096183,...
                    0.009059771353480,...
                    0.055791052971747,...
                    0.344692548477970,...
                    0.139964110318392,...
                    0.495822517789665,...
                    0.044707679385502,...
                    0.043672095251755,...
                    0.245218573506196,...
                    0.059276933781645,...
                    0.128325526947980,...
                    0.113275821301685,...
                    0.242720437216092,...
                    0.387908478942487,...
                    0.213289800009619,...
                    0.137937509976457,...
                    0.465125819964767,...
                    0.270073629388671 ];
                Qn.w = [   0.002637791126883,...
                    0.002674906650997,...
                    0.003193038273764,...
                    0.004170698437610,...
                    0.005259203237679,...
                    0.007188445007987,...
                    0.008958782692065,...
                    0.009248725387126,...
                    0.010074246248491,...
                    0.010230074886449,...
                    0.012737216187365,...
                    0.014126981551441,...
                    0.015332774415553,...
                    0.019150988304468,...
                    0.019581375851107,...
                    0.022519214872788,...
                    0.023744088053619,...
                    0.024512580234751,...
                    0.024681473602269,...
                    0.026967662919953,...
                    0.032346210873791,...
                    0.033996854517175 ];
        end
    case 'full'
        % _________________________________________________________________
        % Degree 18, 67 Points, PI, full Symmetry, References: [X1],
        % reported in Table 4.1 on page 93. Numerical values obtained from
        % [Z2].
        % -----------------------------------------------------------------
        Qn.m = [ 1 3 3 3 3 3 3 6 6 6 6 6 6 6 6 ];
        Qn.b1 = [ 0.333333333333333333333333333333333333,...
            0.15163850697260486492387353795772074,...
            0.07243870556733287047426206374480081,...
            0.00375894434106834585702462733286887,...
            0.41106710187591949855469549486746318,...
            0.26561460990537421478430796115175039,...
            0.47491821132404573588789755091754023,...
            0.06612245802840338770053947185398348,...
            0.26857330639601384733212028806856623,...
            0.30206195771287080772484323648551723,...
            0.13277883027138932992144407050471004,...
            0.25650615977424154068897765977748937,...
            0.41106566867461836291309677848250996,...
            0.04727614183265178252228403898505622,...
            0.38504403441316367334400254247436861 ];
        Qn.b2(1:7) = Qn.b1(1:7);
        Qn.b2(8:15)= [ 0.17847912556588763355267204638676643,...
            0.14906691012577383920019113944789784,...
            0.05401173533902423468044436247084948,...
            0.01433152477894195356844867129563809,...
            0.01050501881924193559868603344210775,...
            0.01169182467466708527042342649785763,...
            0.01249893248349544012804819357953175,...
            0.52452892523249571422861434426430408 ];
        Qn.w = [ 0.03074852123911585539935333820159969,...
            0.02031833884545839730521676856098738,...
            0.01379028660476693880147269080330003,...
            0.00053200561694778056109294261721746,...
            0.03347199405984789811876973462144190,...
            0.03111639660200613119689389250158563,...
            0.01310702749173875567860153100348528,...
            0.01691165391748007879456553323826843,...
            0.02759288648857947802009593334620683,...
            0.01636590841398656595815221611374510,...
            0.00764170497271963595084711372125680,...
            0.00772983528000622700809279634102600,...
            0.00958612447436150376044024017260990,...
            0.00421751677474444290984387716007124,...
            0.01532825819455314086704628681920691 ];
end

end%
%%
function Qn = quadTriangleNonProductDegree19(varargin)
% Lower bound = 60

switch varargin{1}
    case 'allowAsymmetric'
        % _________________________________________________________________
        % Degree 19, 70 Points, PI, Rotationally Symmetric, Reference:
        % [P2], reported as new in Table 4 on page 46. Numerical
        % values obtained from online supplemental material of
        % [P2].
        % -----------------------------------------------------------------
        Qn.m = [ 1 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ];
        Qn.b1 = [ 0.333333333333333,...
            0.010376656545702,...
            0.052895640212959,...
            0.130940571465279,...
            0.012668186551090,...
            0.246613960104767,...
            0.012433000096284,...
            0.009458708679260,...
            0.382677127432901,...
            0.010722190231133,...
            0.063458887217336,...
            0.149585341667801,...
            0.065938374993168,...
            0.419360344474694,...
            0.051285971562744,...
            0.059216171712594,...
            0.157744353652535,...
            0.276356371451293,...
            0.402925432504103,...
            0.144741209510670,...
            0.128002196170715,...
            0.267334020957130,...
            0.380640391337434,...
            0.241019074102766 ];
        Qn.b2 = [ 0.333333333333333,...
            0.014481047265900,...
            0.008927117342565,...
            0.006396063561708,...
            0.072908235059904,...
            0.011321028448272,...
            0.171507203857275,...
            0.454553083656949,...
            0.010304127159605,...
            0.302819418977863,...
            0.047982682558827,...
            0.041535364284311,...
            0.127098301306500,...
            0.048413394484358,...
            0.385451519447596,...
            0.244263378740914,...
            0.107039102999094,...
            0.059395829650746,...
            0.111259697463849,...
            0.207337086661877,...
            0.338803971814337,...
            0.147872285803578,...
            0.207421341282565,...
            0.267083117991291 ];
        Qn.w = [ 0.030462961760147,...
            0.001950181958066,...
            0.002761944617173,...
            0.003533179431733,...
            0.005152029077690,...
            0.007311710166271,...
            0.007389473051762,...
            0.007547786052676,...
            0.007615468406676,...
            0.007900758133129,...
            0.008210439532924,...
            0.010631666580804,...
            0.014518063122052,...
            0.014694113731912,...
            0.016869911193357,...
            0.017219169524638,...
            0.017491674145471,...
            0.018291477800298,...
            0.021089355502457,...
            0.023029171331063,...
            0.024934188451325,...
            0.025956134949103,...
            0.028915516306229,...
            0.030165599679809 ];
    case 'full'
        % _________________________________________________________________        
        % Degree 19, 73 Points, PI, Fully Symmetric, References: [D2], see
        % Appendix II page 1143. Numerical values obtained from [Z2].
        % -----------------------------------------------------------------
        Qn.m = [ 1,3,3,3,3,3,3,3,3,6,6,6,6,6,6,6,6 ];
        Qn.b1 = [ ...
            0.3333333333333333333333333333333333,...
            0.4896099870730063319661310657482982,...
            0.4545368926978926620467593905357283,...
            0.4014166806494311873939956238106886,...
            0.2555516544030976113221817681092679,...
            0.1770779421521295516426752065159011,...
            0.1100610532279518613000849516773740,...
            0.0555286242518396712486784124713557,...
            0.0126218637772286684902347667787060,...
            0.6006337947946450000000000000000000,...
            0.1344667545307797856120431989326469,...
            0.7209870258173650552166529023382789,...
            0.5945270689558709246138892880265067,...
            0.8393314736808385786174900771484052,...
            0.2238614240979156913033693895065364,...
            0.8229313240698566316274715591605332,...
            0.9243442526207840294558591379015631 ];
        Qn.b2(1:9) = Qn.b1(1:9);
        Qn.b2(10:17) = [ ...
            0.3957547873569428623047946940658279,...
            0.5576032615887839683639532425011810,...
            0.2645669484065202080403017349012149,...
            0.3585393522059505884249269906459009,...
            0.1578074059685947447376736033595065,...
            0.7010879789261733673232883365595116,...
            0.1424216011133834373155747568772374,...
            0.0654946280829377033923265249859256 ];
        Qn.w = [ ...
            0.0329063313889186520836143448464750,...
            0.0103307318912720533670399635717483,...
            0.0223872472630163925291845560351627,...
            0.0302661258694680708652801909825912,...
            0.0304909678021977810000315865785204,...
            0.0241592127416409049118480309866400,...
            0.0160508035868008752916227702764295,...
            0.0080845802617840604818056732421944,...
            0.0020793620274847807513475016743984,...
            0.0038848769049813897567049919927727,...
            0.0255741606120219038929297019526003,...
            0.0088809035733380577455259247035175,...
            0.0161245467617313912197852693278377,...
            0.0024919418174906754405846475759496,...
            0.0182428401189505783776657132097361,...
            0.0102585637361985213080480700423581,...
            0.0037999288553019139790731537136397 ];
end

end%
%%
function Qn = quadTriangleNonProductDegree20(varargin)
% Lower bound = 66

switch varargin{1}
    case 'allowAsymmetric'
        % _________________________________________________________________
        % Degree 20, 78 Points, PI, Rotationally symmetric, References: 
        % [X1], reported as new in Table 1 on page 671. Numerical values 
        % obtained from online supplemental material of [P2]. (Note: It is 
        % unclear if the rule below from [P2] is the same as that reported 
        % earlier in [X1].)
        % -----------------------------------------------------------------
        Qn.m = [ 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ];
        Qn.b1 = [ 0.007903681341623,...
            0.003702614214050,...
            0.109337390150228,...
            0.042881075376943,...
            0.001774732573073,...
            0.311814570601137,...
            0.002990006604696,...
            0.032214232134941,...
            0.012779960332488,...
            0.195430723381915,...
            0.097766916990502,...
            0.016600662844178,...
            0.413395082180740,...
            0.068329374634159,...
            0.039215307843232,...
            0.282851404387686,...
            0.163149078355705,...
            0.436290583906622,...
            0.123069087330846,...
            0.083257905081074,...
            0.376028943583989,...
            0.239866290944955,...
            0.137221148823994,...
            0.189932130608381,...
            0.324158256395639,...
            0.330862912147052 ];
        Qn.b2 = [ 0.010892617666330,...
            0.056835913941913,...
            0.003038592573344,...
            0.009805057025499,...
            0.236693018982774,...
            0.002545519795639,...
            0.463119441155937,...
            0.052485418254686,...
            0.131379538178000,...
            0.014095340079167,...
            0.035000751662895,...
            0.338676752265838,...
            0.018202587826948,...
            0.108675089954567,...
            0.214338430054550,...
            0.043250599724609,...
            0.074632420343532,...
            0.516537077780689,...
            0.173946464302243,...
            0.299185227797304,...
            0.086437944469364,...
            0.130115610897139,...
            0.391013201302958,...
            0.250321346008893,...
            0.194724488881112,...
            0.407445324376853 ];
        Qn.w = [ 0.001138990614898,...
            0.001535767087458,...
            0.001924685591787,...
            0.002500437312356,...
            0.002630751236188,...
            0.003195625614028,...
            0.003651768994079,...
            0.005324537891217,...
            0.005844456028389,...
            0.007308005980142,...
            0.007906335353292,...
            0.009655957870889,...
            0.010314589003860,...
            0.012209838544263,...
            0.012838682322099,...
            0.014666797697724,...
            0.015405053862366,...
            0.016575593874878,...
            0.019989220381195,...
            0.020297417477507,...
            0.021636690505780,...
            0.023025477362952,...
            0.026323493747773,...
            0.027048097627035,...
            0.028994339729607,...
            0.031390721621573 ];
    case 'full'
        % _________________________________________________________________
        % Degree 20, 79 Points, PI, Fully symmetric, References: 
        % [X1], reported as new in Table 1 on page 671. Numerical values 
        % obtained from online supplemental material of [W1]. (Note 1: It 
        % is unclear if the rule below from [W1] is the same as that 
        % reported earlier in [X1]. Note 2: The points are given over the 
        % reference triangle with vertices [-1 -1; 1 -1; -1 1] in the 
        % supplemental material of [W1], so these are converted to
        % barycentric coordinates here.)
        % ----------------------------------------------------------------- 
        Qn.m = [ 1 3 3 3 3 3 3 3 3 6 6 6 6 6 6 6 6 6 ];
        eta1 = [ -0.33333333333333333333333333333333333333,...
            -0.49084146465332177523029470374523072315,...
            -0.97804771794320447192070397628224037507,...
            -0.78123280657657080596625505020033275589,...
            -0.62741000451091811445653498388475129105,...
            -0.10889788608815036962555544313157118467,...
            -0.92537823880223061201299256742556755612,...
            -0.21314930436580028264207251299344481704,...
            -0.047508776919001973579296255534818088061,...
            -0.98485843899060694347353308294287690802,...
            -0.90687927018467136697246089078946388707,...
            -0.87181882878313187990026866476433962457,...
            -0.89002504171402637873154550296837045792,...
            -0.8000954074237226892341843533057676038,...
            -0.78754559055945991527489139589221510388,...
            -0.1599524823675518407071141661054621128,...
            -0.36427975232845596804806465673948070356,...
            -0.97852557428797782533895087798420844581 ];
        eta2 = [ -0.33333333333333333333333333333333333333,...
            -0.018317070693356449539410592509538553699,...
             0.95609543588640894384140795256448075015,...
             0.56246561315314161193251010040066551178,....
             0.2548200090218362289130699677695025821,...
            -0.78220422782369926074888911373685763067,...
             0.85075647760446122402598513485113511224,...
            -0.57370139126839943471585497401311036593,...
            -0.90498244616199605284140748893036382388,...
             0.66659102367647249434430238008617319241,...
             0.5098430057270950102124174847737658903,...
             0.86210895356788437332353836160651207109,...
             0.22375540709485139987629530764328543032,...
             0.72336803787297349848896164841146460112,...
             0.35633147577927110921599545999123193124,...
             0.1402893857819467195500916946480886286,...
             0.084663608344856170859761688802646336641,...
             0.41736275144064735566397781689608203229 ];
         Qn.w = [ 0.055640442805812463631566788253898051532,...
             0.05633280523008099012864178412473069936,...
             0.0031953631642664794478945847330657770722,...
             0.031320923104298133684407003648894203447,...
             0.036693851897011657919252854118138043537,...
             0.037809599732929790747314162640799596721,...
             0.008645101642662310104692722157806774508,...
             0.055152202516281836053550890863277393079,...
             0.028407301213633761967680465924479290382,...
             0.0088115896742339902561312708157853846482,...
             0.023945594315818760080585256144999165493,...
             0.0045194784085034621179235150583929560485,...
             0.034668902268877332248730626058035734221,...
             0.016582846110455430851049450635630476598,...
             0.030890431288396919377481650650853417183,...
             0.01478272600102119191287687445277038169,...
             0.046766982927310947730184310937781849814,...
             0.014312800953830741459725346431171736663 ]./2;
         TR = triangulation([1 2 3],[-1 -1; 1 -1; -1 1]);
         B = cartesianToBarycentric(TR,ones(18,1),[eta1',eta2']);
         Qn.b1 = B(:,1); Qn.b2 = B(:,2); Qn.b3 = B(:,3);
end

end%
%%
function Qn = quadTriangleNonProductDegree21(varargin)
% Lower bound = 71

switch varargin{1}
    case 'allowAsymmetric'
        % _________________________________________________________________
        % Degree 21, 85 Points, PI, Rotationally symmetric, Reference: 
        % [P2], see Table 4 page 46.  Numerical values obtained from online 
        % supplemental material of [P2].
        % -----------------------------------------------------------------
        Qn.m = [1 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3];
        Qn.b1 = [ 0.333333333333333,...
            0.010883381125829,...
            0.134483082013301,...
            0.056069791445608,...
            0.056230797163852,...
            0.495492250545261,...
            0.133413544221437,...
            0.237788853273795,...
            0.236675822687570,...
            0.359237431543588,...
            0.362123795542132,...
            0.054706462526631,...
            0.128497845564647,...
            0.132045621485988,...
            0.480002887465516,...
            0.235698652486184,...
            0.121235453505121,...
            0.226921312534570,...
            0.352725956002230,...
            0.346394760980635,...
            0.208545376179994,...
            0.116641181787499,...
            0.301681731659511,...
            0.359766505094364,...
            0.317727894779721,...
            0.450599237708166,...
            0.266780208878215,...
            0.214913322085512,...
            0.341820828914986 ];
        Qn.b2 = [ 0.333333333333333,...
            0.010843736020091,...
            0.005894017914461,...
            0.934072029515549,...
            0.010411168900746,...
            0.005301393539291,...
            0.856502202113258,...
            0.753553830034851,...
            0.010431599868322,...
            0.009774413119085,...
            0.627482731394553,...
            0.893376258603483,...
            0.036812842147871,...
            0.816750542291406,...
            0.034623316765578,...
            0.717639780294608,...
            0.098717289213208,...
            0.053382695673251,...
            0.593956020138616,...
            0.051134367191087,...
            0.126661874088773,...
            0.193583513707743,...
            0.579485561285727,...
            0.465449578776416,...
            0.121922765864561,...
            0.098026080230645,...
            0.309304671639406,...
            0.223615271231608,...
            0.203059575102732 ];
        Qn.w = [ 0.015927243921705,...
            0.001540024810414,...
            0.002964038211235,...
            0.003142570479380,...
            0.003322557729511,...
            0.004070942127675,...
            0.004706580389141,...
            0.005185880239627,...
            0.006053172272946,...
            0.006497373154781,...
            0.006973324165434,...
            0.007205875589346,...
            0.008365012828871,...
            0.010226596336838,...
            0.011740928736597,...
            0.012183249444134,...
            0.012865489474875,...
            0.013095298262017,...
            0.014086719053342,...
            0.014184030719574,...
            0.017220831567602,...
            0.017413623959356,...
            0.018083675543648,...
            0.019149138766673,...
            0.019227819270613,...
            0.020031676761422,...
            0.020765122861493,...
            0.023815052706389,...
            0.023907646563162 ];
    case 'full'
        % _________________________________________________________________        
        % Degree 21, 87 Points, PI, Fully Symmetric, References: [X1], see
        % Table 1 page 671. Numerical values obtained from [Z2], which
        % references [P2]. (Note: It is unclear if the rule below from [P2]
        % is the same as that reported earlier in [X1].)
        % -----------------------------------------------------------------
        Qn.m = [ 3 3 3 3 3 3 3 3 3 6 6 6 6 6 6 6 6 6 6 ];
        Qn.b1 = [ 0.01083064791389613439818531875282892,...
            0.49868116928006762187511807510022292,...
            0.05320630259370221372937428477043579,...
            0.48436880679907196058128682726473251,...
            0.11717932446409532061312493414817476,...
            0.18619987453160044065356184245386411,...
            0.23770504697050237030777008826766300,...
            0.45111809686333143558277239278860940,...
            0.29890161661837960265935712860196744,...
            0.01017037551397023833029242013076474,...
            0.00740148001912942691985111704851672,...
            0.00948888021003903598802889438007338,...
            0.01078708528953292731070298166591233,...
            0.04888641015641196213015335363186247,...
            0.04079047843492138579001393428298806,...
            0.10225476525427212998164710587805099,...
            0.05619510372649343472752614615138017,...
            0.19392111402204940534236787859220491,...
            0.13139693759954925634229231531863966 ];
        Qn.b2(1:9) = Qn.b1(1:9);
        Qn.b2(10:19) = [ 0.05610607874044113073901605353578031,...
            0.23806090893927819107377783880076336,...
            0.13371423612729252503412567154133090,...
            0.36087827036133643028378433162192850,...
            0.12856000707434670174510612755634180,...
            0.23192922019831618515287729451119134,...
            0.21318238024685212224076314392042087,...
            0.34615363893081412151163375832691108,...
            0.35741779412597604193763009688130316,...
            0.30964180048539107980510336108955521 ];
        Qn.w = [ 0.00153252214157029927832972284942362,...
            0.00295964208580089055557643244435822,...
            0.00710507195198322705473634640429004,...
            0.01242202545015119261392701182790142,...
            0.01315385309054860400530165567389379,...
            0.01610711993355940881429849025111080,...
            0.01955699832742153003791726026064669,...
            0.02114714580341115240235728776162335,...
            0.02148724961809553292499438055586582,...
            0.00324321603629715843164230514808509,...
            0.00444606682456729614112175291076191,...
            0.00444713701276898568121864572732943,...
            0.00719283008979440275755795626382300,...
            0.00968818276967735751464842894718971,...
            0.01077186725474600415287816641481312,...
            0.01537644879227419490362875554570800,...
            0.01541820162505099647808429434299382,...
            0.01872799408182348912155361034660658,...
            0.01961890797839586264061345700479913 ];
end

end%
%%
function Qn = quadTriangleNonProductDegree22(varargin)
% Lower bound = 78

switch varargin{1}
    case 'allowAsymmetric'
        % _________________________________________________________________
        % Degree 22, 93 Points, PI, Rotationally symmetric, References: 
        % [X1], reported as new in Table 1 on page 671. Numerical values 
        % obtained from online supplemental material of [P2]. (Note: It is 
        % unclear if the rule below from [P2] is the same as that reported 
        % earlier in [X1].)
        % -----------------------------------------------------------------
        Qn.m = [ 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ...
            3 3 3 3 ];
        Qn.b1 = [ 0.001170338483789,...
            0.010994365574026,...
            0.001068169252253,...
            0.002212283505921,...
            0.057202598716253,...
            0.353024211615277,...
            0.136788957839154,...
            0.241209563760038,...
            0.011673747786688,...
            0.030114899634799,...
            0.456500154387454,...
            0.011807594618902,...
            0.096150558730147,...
            0.030637657499208,...
            0.194815244430806,...
            0.060365503070241,...
            0.315090899482059,...
            0.411358367215817,...
            0.144020350630489,...
            0.420757709102122,...
            0.060949899580790,...
            0.096935106445587,...
            0.253783465620647,...
            0.348867101402163,...
            0.190581558653074,...
            0.096988114334813,...
            0.141266334630632,...
            0.412018101979952,...
            0.276209339065321,...
            0.218610614672497,...
            0.336799046311846 ];
        Qn.b2 = [ 0.047853691880759,...
            0.009108671753720,...
            0.202709063387926,...
            0.421769070671465,...
            0.008560870999210,...
            0.006094241250556,...
            0.007849980628480,...
            0.007027263551720,...
            0.112641216429535,...
            0.046055896520382,...
            0.011437907707031,...
            0.304996921690560,...
            0.042635231869026,...
            0.196990580025709,...
            0.038664167437872,...
            0.106610220689148,...
            0.036089820174339,...
            0.556390353961856,...
            0.097715590184513,...
            0.055528837489453,...
            0.292056016141073,...
            0.184224156891908,...
            0.092228684901440,...
            0.116235552070710,...
            0.172148246666745,...
            0.396552131508627,...
            0.275379650245183,...
            0.168573884290088,...
            0.199199356373184,...
            0.305685651047730,...
            0.269814345781991 ];
        Qn.w = [ 0.000975412040940,...
            0.001310445221150,...
            0.001819384084649,...
            0.002485389717898,...
            0.002794305762427,...
            0.003548148572011,...
            0.003778626656318,...
            0.004038266285040,...
            0.004633213593023,...
            0.004930726340302,...
            0.006415703990777,...
            0.006648425044147,...
            0.007937354455688,...
            0.009391825671203,...
            0.009766584623791,...
            0.009939312236419,...
            0.010785717950341,...
            0.011198713817120,...
            0.013592449250674,...
            0.014107287313163,...
            0.014345498985920,...
            0.015033144681114,...
            0.015837677803806,...
            0.017114505548267,...
            0.017971291100639,...
            0.018545033196902,...
            0.019398111246358,...
            0.019523670586402,...
            0.020625531969637,...
            0.021313822420183,...
            0.023527753167023 ];
    case 'full'
        switch varargin{3}
            case 'allowOutside'
                % WARNING: Error slightly high on this one ~1.0e-09!!!
                % _________________________________________________________
                % Degree 22, 94 Points, PO (6 points relatively far out),
                % Fully symmetric, Reference: [P2], reported as new in 
                % Table 3 on page 46. Numerical values obtained from online
                % supplemental material of [P2].
                % ---------------------------------------------------------
                Qn.m = [ 1 3 3 3 3 3 3 3 3 3 6 6 6 6 6 6 6 6 6 6 6 ];
                Qn.b1 = [ 0.333333333333333,...
                    0.010496492318533,...
                    0.008932709622497,...
                    0.056049615132866,...
                    0.046669588271567,...
                    0.131716048354042,...
                    0.113395390491673,...
                    0.210431968711084,...
                    0.211601327317755,...
                    0.280558889938601,...
                    -0.365758201288991,...
                    0.010866084504842,...
                    0.006986410207246,...
                    0.011088449518327,...
                    0.009499130105971,...
                    0.037046616048003,...
                    0.056598109252543,...
                    0.049307086657457,...
                    0.091696755829685,...
                    0.118093857071423,...
                    0.172252200982416 ];
                Qn.b2 = [ 0.333333333333333,...
                    0.010496492318533,...
                    0.495533645188752,...
                    0.056049615132866,...
                    0.476665205864217,...
                    0.131716048354042,...
                    0.443302304754163,...
                    0.394784015644458,...
                    0.211601327317755,...
                    0.280558889938601,...
                    0.121775779289980,...
                    0.054699006820644,...
                    0.360846088228242,...
                    0.131476833810283,...
                    0.235799841504739,...
                    0.353558464226123,...
                    0.132527176274670,...
                    0.233455411836818,...
                    0.340216421113572,...
                    0.228110979662495,...
                    0.319532176347332 ];
                Qn.w = [ 0.018046065005519,...
                    0.001444055579532,...
                    0.006266466449560,...
                    0.007651744360123,...
                    0.013279847784101,...
                    0.015110120003823,...
                    0.017591885388871,...
                    0.018843708393925,...
                    0.019848233599481,...
                    0.020867162478629,...
                    0.000000000002401,...
                    0.003391698725945,...
                    0.004742676361869,...
                    0.005171989107037,...
                    0.005628119911213,...
                    0.010573950573404,...
                    0.011086612482237,...
                    0.012134510421606,...
                    0.015154378475876,...
                    0.016798674545460,...
                    0.018524766539675 ];
            case 'inside'
                % _________________________________________________________                
                % Degree 22, 96 Points, PI, Fully symmetric, References:
                % [X1], see Table 1 page 671. Numerical values obtained
                % from [Z2], which references [P2]. (Note: It is unclear if
                % the rule below from [P2] is the same as that reported
                % earlier in [X1].)
                % ---------------------------------------------------------
                Qn.m = [ 3 3 3 3 3 3 3 3 3 3 6 6 6 6 6 6 6 6 6 6 6 ];
                Qn.b1 = [ 0.00722823551590805269240186985118743,...
                    0.49971491656223188463531154910591767,...
                    0.04195252727280408802705553682804348,...
                    0.31413219152050094565940577630042994,...
                    0.48001365161711654879356466888729030,...
                    0.12308745423016114898448687333895172,...
                    0.44571963043911530549256371899531770,...
                    0.18986241218389367642303052024426076,...
                    0.26033923338175577283598939801891875,...
                    0.39636684767162598240715439210378845,...
                    0.00773958468860307799587341381217818,...
                    0.00057706413021840388680988197995853,...
                    0.00979577185352314036856347829609728,...
                    0.00918868020649896490029828421437019,...
                    0.01346708756840181797029316077776570,...
                    0.05062631246733689522786952921020785,...
                    0.04838811776183765022487372116607709,...
                    0.02749143585326034673258117826660649,...
                    0.06713397444770439007586079560898384,...
                    0.09670430361686034561495436311127678,...
                    0.15193045482670229286438888320778608 ];
                Qn.b2(1:10) = Qn.b1(1:10);
                Qn.b2(11:21) = [ 0.03910016782732837999359532547054414,...
                    0.29956301031600608000000000000000000,...
                    0.09916758085225246952608204467401148,...
                    0.18619068155280570806618576756346898,...
                    0.40293889185510044331704310704540997,...
                    0.10358808540929187702277517444504398,...
                    0.18082300688832890010580439799931826,...
                    0.28194341507949187524318191991162303,...
                    0.35071285339799134366628146898304461,...
                    0.23017640440360132139166340852745074,...
                    0.31159323586348644974639923203001594 ];
                Qn.w = [ 0.00069714569625074030369942481362133,...
                    0.00173365376198036236244040089465219,...
                    0.00466131304304557368006266059260170,...
                    0.01158457676559956471190194225470023,...
                    0.01187897403302190998142733919159934,...
                    0.01533880365566670016085333491517060,...
                    0.01971637694027233703178135471037246,...
                    0.02057215149173475351932757540044250,...
                    0.02384066027813703919931347525917147,...
                    0.02446119708987487225534932817494781,...
                    0.00182862802731120927204804019418561,...
                    0.00188044970396283839898767788993490,...
                    0.00370410864001110188501003141229496,...
                    0.00466834037183480579057625840302880,...
                    0.00721134117192958564362665357306283,...
                    0.00796743335988734266364119940269571,...
                    0.00903580036943713630598503580960509,...
                    0.00948064111155901466263790694355664,...
                    0.01561737012835320857159572546229795,...
                    0.01624531918105698418898193939757981,...
                    0.02178480822353151268049778007478454 ];
        end
end

end%
%%
function Qn = quadTriangleNonProductDegree23(varargin)
% Lower bound = 84

switch varargin{1}
    case 'allowAsymmetric'
        % _________________________________________________________________
        % Degree 23, 100 Points, PI, Rotationally symmetric, Reference: 
        % [P2], reported as new in Table 4 on page 46. Numerical values 
        % obtained from online supplemental material of [P2].
        % -----------------------------------------------------------------
        Qn.m = [ 1 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ...
            3 3 3 3 3 3 3 ];
        Qn.b1 = [ 0.333333333333333,...
            0.009413949033005,...
            0.000482410218909,...
            0.496335565629436,...
            0.008614527590243,...
            0.049100663468776,...
            0.002660416077354,...
            0.330984469269412,...
            0.118154505921019,...
            0.214073097879029,...
            0.044577412877334,...
            0.191431395153491,...
            0.010585098456769,...
            0.025420226429006,...
            0.591292919761096,...
            0.104903656520941,...
            0.427217446018735,...
            0.177899296750880,...
            0.080548215582572,...
            0.031317311279855,...
            0.295446758742554,...
            0.054113217353948,...
            0.042740065428049,...
            0.145187675973378,...
            0.074713978478764,...
            0.381968043130448,...
            0.221783549821815,...
            0.256870811513147,...
            0.112751303426960,...
            0.210323057441034,...
            0.161575209179477,...
            0.331656221355575,...
            0.212294763316891,...
            0.274113742189351 ];
        Qn.b2 = [ 0.333333333333333,...
            0.008646444451894,...
            0.108821420580561,...
            0.503294735930982,...
            0.044964858940933,...
            0.008242497082430,...
            0.294704482021721,...
            0.003820668491799,...
            0.008413157414661,...
            0.007205397576502,...
            0.043026357023085,...
            0.031169269889860,...
            0.191864903048277,...
            0.105407606480097,...
            0.012107289245252,...
            0.043370975391950,...
            0.016697078855471,...
            0.064511068023700,...
            0.101976993969516,...
            0.285342978045656,...
            0.032881012535433,...
            0.183859582702944,...
            0.443558986603905,...
            0.129795450451007,...
            0.329913961425315,...
            0.067362527515130,...
            0.668980984358150,...
            0.096049121135375,...
            0.423674182024582,...
            0.187001109672333,...
            0.299012948455699,...
            0.150616235510019,...
            0.384271485252512,...
            0.258726723002178 ];
        Qn.w = [ 0.025138871460294,...
            0.001066117682506,...
            0.001109873150555,...
            0.001629623955646,...
            0.002205291693138,...
            0.002317554319797,...
            0.002321638701615,...
            0.003003661511396,...
            0.003547621923537,...
            0.003802651636547,...
            0.004734324702657,...
            0.004782045529448,...
            0.005054920167300,...
            0.006045206134484,...
            0.006312702445690,...
            0.006776405487138,...
            0.007884792427633,...
            0.008736907640747,...
            0.009434405853993,...
            0.009490879427819,...
            0.010210962656218,...
            0.010519253335485,...
            0.011689120375278,...
            0.013700400279401,...
            0.014460451299083,...
            0.015231395783589,...
            0.015485873319446,...
            0.015988957436554,...
            0.018929035905892,...
            0.019748081356734,...
            0.020667781559666,...
            0.020798342664795,...
            0.023575574840744,...
            0.023691854308705 ];
    case 'full'
        % _________________________________________________________________        
        % Degree 23, 102 Points, PI, Fully symmetric, References: [P2], re-
        % ported as new in Table 3 on page 46. Numerical values obtained
        % from [Z2], which references [P2].
        % -----------------------------------------------------------------
        Qn.m = [ 3 3 3 3 3 3 3 3 3 3 6 6 6 6 6 6 6 6 6 6 6 6 ];
        Qn.b1 = [ 0.00884309098564000781144468419871075,...
            0.49904231799536473576318688106778703,...
            0.04191849463192120115455956274854522,...
            0.08384395940074188671519383182016759,...
            0.35156602225687808437472423305189879,...
            0.48142250008209757218970635991224348,...
            0.13603683917665686011746620285416514,...
            0.44588621023292908434421744174344724,...
            0.19833791969077397732302473244075886,...
            0.39764873352592070572902821621712745,...
            0.00215375088982479143593969096276917,...
            0.00797567048343379974299672675667051,...
            0.00222521306870228962165445623994274,...
            0.01056799023421332981938693231799616,...
            0.02639015796005238716769753666365164,...
            0.01372661478891633442412470482374058,...
            0.03293167071711912354883994110324428,...
            0.05645836928678800749584878194121292,...
            0.24905869950604636107649169253437782,...
            0.06710022884559872615982757386133093,...
            0.10095848404745636455551318905522403,...
            0.15208140311691476836002821658866124,...
            ];
        Qn.b2(1:10) = Qn.b1(1:10);
        Qn.b2(11:22) = [ 0.11147822164436514858831541609060182,...
            0.04599876191192738898335879375393121,...
            0.29991704500621364115075519706889153,...
            0.19450617255148173513923298602622080,...
            0.10487601375183195540194359053325350,...
            0.39359274155812221677807114606384301,...
            0.27745010154284908600010796990105572,...
            0.17041930960722382214358721748321713,...
            0.28048464594062623291197233102900262,...
            0.35940684385217631018244572992741760,...
            0.23967897332407877845308433492438781,...
            0.31660538238914823066544783358734965 ];
        Qn.w  = [ 0.00102338493393064825880875292048368,...
            0.00234953285611607429820752203919566,...
            0.00442272294831746458300712854673892,...
            0.00935939493458907159588059346569364,...
            0.01122330490795551618339838001860665,...
            0.01145539386071866483451404196187953,...
            0.01485152819027043031788493609445202,...
            0.01873399097752303244589658710992199,...
            0.02009544889817501015648393973138171,...
            0.02186952160246302231865049077371942,...
            0.00139037413890586280792572004623041,...
            0.00209361317170022977809045175923150,...
            0.00226272789116058493926098904342501,...
            0.00509887607853182356576648513174798,...
            0.00580279248634163562097040517303610,...
            0.00699047412016323154662825915726588,...
            0.01010544741268549508636466193618720,...
            0.01094068682011104235992829469791130,...
            0.01269774746720014444245594705387966,...
            0.01493690521473593308948584780344384,...
            0.01624594488896314707519973030797813,...
            0.02040896492113806885822368822529305 ];
end

end%
%%
function Qn = quadTriangleNonProductDegree24(varargin)
% Lower bound = 91

switch varargin{1}
    case 'allowAsymmetric'
        % _________________________________________________________________            
        % Degree 24, 109 Points, PI, Rotationally symmetric, Reference: 
        % [P2], reported as new in Table 4 on page 46. Numerical values 
        % obtained from online supplemental material of [P2].
        % -----------------------------------------------------------------
        Qn.m = [ 1 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ...
            3 3 3 3 3 3 3 3 3 ];
        Qn.b1 = [ 0.333333333333333,...
            0.007277442182286,...
            0.093762141740160,...
            0.027549956725482,...
            0.038849156576576,...
            0.084582464342299,...
            0.169711974261767,...
            0.261114259888929,...
            0.167919705131084,...
            0.366562445874965,...
            0.271431359408183,...
            0.483415122297192,...
            0.387957374245692,...
            0.052053611601208,...
            0.229018909213726,...
            0.094914195571636,...
            0.123228014736413,...
            0.174126507274860,...
            0.272180746005687,...
            0.099006351507515,...
            0.218932787550637,...
            0.385457856195294,...
            0.453931532973698,...
            0.331414067992492,...
            0.175418478700221,...
            0.299424606258227,...
            0.164438284214270,...
            0.369724758939287,...
            0.406369061777283,...
            0.267420664299268,...
            0.383204107909751,...
            0.178724492320245,...
            0.282829128593355,...
            0.270606479254229,...
            0.370874346744923,...
            0.363840811889283,...
            0.275735321862771 ];
        Qn.b2 = [ 0.333333333333333,...
            0.001871780631611,...
            0.002225841776308,...
            0.964825831627800,...
            0.009992214684680,...
            0.907583350374655,...
            0.008550947694910,...
            0.007355235311152,...
            0.824005083539038,...
            0.007005660169500,...
            0.720628297746086,...
            0.007361193566840,...
            0.604329908418651,...
            0.907672123610023,...
            0.089345256498108,...
            0.026796719091792,...
            0.835184452859200,...
            0.044130628886304,...
            0.037468721939443,...
            0.081968796781302,...
            0.739314412421514,...
            0.037011409874506,...
            0.507124120567042,...
            0.627872100319266,...
            0.109788999485381,...
            0.089517741605651,...
            0.735720413124064,...
            0.463127807453673,...
            0.090716786111306,...
            0.632902725169850,...
            0.521623720050690,...
            0.192031135757774,...
            0.537642437490313,...
            0.161339950052654,...
            0.164548573538222,...
            0.383367191770048,...
            0.252760449512090 ];
        Qn.w = [ 0.010681776933387,...
            0.000343072138434,...
            0.001222999702037,...
            0.001638160273649,...
            0.002237225945870,...
            0.002857501868278,...
            0.003685829616327,...
            0.003694795442052,...
            0.003902032006208,...
            0.004028069096931,...
            0.004524880720162,...
            0.004539602926267,...
            0.004745695447312,...
            0.005180536573374,...
            0.005311923232723,...
            0.005364195042967,...
            0.007840965945189,...
            0.008699402266213,...
            0.008840203633368,...
            0.009629132219890,...
            0.009803252609884,...
            0.009942817375829,...
            0.010748240736700,...
            0.010775272663604,...
            0.011783097752825,...
            0.012156767654105,...
            0.012945705236133,...
            0.013554899747479,...
            0.014675832493070,...
            0.015420454458400,...
            0.015515997794155,...
            0.015662589306010,...
            0.015927537510848,...
            0.016103533030080,...
            0.016658080425731,...
            0.019353079492363,...
            0.020459358637735 ];
    case 'full'
        % _________________________________________________________________          
        % Degree 24, 112 Points, PI, Fully symmetric, References: [X1], see
        % Table 1 page 671. Numerical values obtained from [Z2], which
        % references [P2]. (Note: It is unclear if the rule below from [P2]
        % is the same as that reported earlier in [X1].)
        % -----------------------------------------------------------------
        Qn.m = [ 1 3 3 3 3 3 3 3 3 3 6 6 6 6 6 6 6 6 6 6 6 6 6 6 ];
        Qn.b1 = [ 0.33333333333333333333333333333333333,...
            0.00764481587209140357148961957868539,...
            0.49512326693489419346010327445636948,...
            0.08533571470774055253577908615445653,...
            0.47540435757943002732921306991884855,...
            0.14361312600254966163076940146917127,...
            0.44000137550057370651541171540277806,...
            0.20438684787130391661942173912629914,...
            0.39155520703268199132653695865910252,...
            0.26936069184956821890376700734821267,...
            0.00128915765322182138873877223282987,...
            0.00804615544134750962080148406751868,...
            0.00397463904079080448292596145774045,...
            0.00565339683877998034115515308702144,...
            0.03654164634700071865936106746572505,...
            0.00847203125278924422001665909111615,...
            0.03186878555848486368527331611156268,...
            0.02507448575695932327989859470575696,...
            0.02495464920723698331110869690511693,...
            0.07180692748992666960708660113790562,...
            0.04649536832522560731056542252010275,...
            0.07944565832892777248067300362002874,...
            0.11210496839891452618514886571897182,...
            0.16288704432089312482891199962593472 ];
        Qn.b2(1:10) = Qn.b1(1:10);
        Qn.b2(11:24) = [ 0.38930032117370399940994358985608046,...
            0.03994551899851089671071946872000290,...
            0.17756755351844720149250759536631588,...
            0.09730437316339079594262256407299041,...
            0.04737913646955582393784813835746318,...
            0.27654233490622630027478984616955668,...
            0.10186004800841345175103882567179717,...
            0.17990041142989784948776341885611339,...
            0.37476148829145151703198672023817762,...
            0.16285371365641890528167019069210662,...
            0.26299994692532693286696577835230479,...
            0.35387462268603241581179193663132046,...
            0.24045272001109686579097131221331152,...
            0.31632883636136616749409848630342490 ];
        Qn.w = [ 0.02293794348096749067720820639821200,...
            0.00076737746111581423045153600120047,...
            0.00552357844475169754997038457424443,...
            0.00838499530275631071273403550487762,...
            0.01221738017800235239788392469826439,...
            0.01307039060561178310890784719893180,...
            0.01788122775842957825015047766363903,...
            0.01822174413909730956979133198261050,...
            0.02164149721756761699449458590855473,...
            0.02168831292692260197279166115848309,...
            0.00174618458266172170313203275603945,...
            0.00183857160374278724340785925075360,...
            0.00199893663336212883772988094673110,...
            0.00206369617133590779327617815189614,...
            0.00229539508715841568165484400462521,...
            0.00457952328890823144675590528548258,...
            0.00529323009909540365054590996973504,...
            0.00557874699599088171390666100209646,...
            0.00861933948725664134146173130325259,...
            0.00999235399224706921212863098136616,...
            0.01030561620768130769226672677282150,...
            0.01460454612625024725227714201539116,...
            0.01499850067625046347206078267251109,...
            0.01923078311743667878627312147552622 ];
end

end%
%%
function Qn = quadTriangleNonProductDegree25(varargin)
% Lower bound = 97

switch varargin{1}
    case 'allowAsymmetric'
        % _________________________________________________________________         
        % Degree 25, 117 Points, PI, Rotationally symmetric, Reference: 
        % [P2], reported as new in Table 4 on page 46. Numerical values 
        % obtained from online supplemental material of [P2].
        % -----------------------------------------------------------------
        Qn.m = [ 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 3 ...
            3 3 3 3 3 3 3 3 3 3 3 3];
        Qn.b1 = [ 0.013188177147405,...
            0.002725962136131,...
            0.062984002092479,...
            0.052023800673154,...
            0.220200206141671,...
            0.030439556156815,...
            0.113993174377526,...
            0.342295210188022,...
            0.130473991950560,...
            0.200179308343867,...
            0.307372363215434,...
            0.276217326622710,...
            0.422935483077427,...
            0.453227003366671,...
            0.190195202693624,...
            0.036171829980613,...
            0.076079438865904,...
            0.149945977487858,...
            0.150955341179003,...
            0.244783310976575,...
            0.170674433667943,...
            0.371942611691943,...
            0.351792010432224,...
            0.098440061147215,...
            0.260865026298108,...
            0.366582144327067,...
            0.468735987743839,...
            0.257617931613541,...
            0.184517952412834,...
            0.281209118651228,...
            0.173591491675633,...
            0.442362027995822,...
            0.359368226555798,...
            0.182134268926742,...
            0.267343042736282,...
            0.365290320906563,...
            0.345064826601385,...
            0.265562227862875,...
            0.362956385501777 ];
        Qn.b2 = [ 0.003079563840119,...
            0.017104304759574,...
            0.931371984168811,...
            0.007437914960862,...
            0.776414334424490,...
            0.944856451527789,...
            0.007471029315779,...
            0.652613563397589,...
            0.861524966076300,...
            0.006507818683030,...
            0.006352407094592,...
            0.706370557783410,...
            0.007562754695338,...
            0.538998045885384,...
            0.784036511910399,...
            0.084456400529243,...
            0.040507632612956,...
            0.035835058933034,...
            0.796913702682935,...
            0.032689779930829,...
            0.078755869338189,...
            0.593428569491770,...
            0.036352873144138,...
            0.096959138605558,...
            0.684089653184229,...
            0.550781902551757,...
            0.041136822171029,...
            0.082486517285323,...
            0.712575925803732,...
            0.605257308197521,...
            0.142209318699191,...
            0.454988327657976,...
            0.090795315699794,...
            0.228667042140730,...
            0.156184413818651,...
            0.173792802318537,...
            0.482299125389173,...
            0.256802629703844,...
            0.371550947373691 ];
        Qn.w = [ 0.000526947810971,...
            0.000538378571249,...
            0.001712557293882,...
            0.001912332866971,...
            0.001989744514007,...
            0.002703001751612,...
            0.002823762142065,...
            0.002915149172473,...
            0.003146228782300,...
            0.003254397748868,...
            0.003723065781498,...
            0.004306314739282,...
            0.004525134158962,...
            0.004562499221318,...
            0.004680276620610,...
            0.005640555981544,...
            0.005778423050775,...
            0.006579141350715,...
            0.006729653165843,...
            0.007575542435868,...
            0.008145835046334,...
            0.009081835239627,...
            0.009400895304308,...
            0.009524004544071,...
            0.010107208145267,...
            0.010859126807715,...
            0.011168984186288,...
            0.011772296865232,...
            0.012142404008477,...
            0.012552698926700,...
            0.012781441276098,...
            0.013246841387625,...
            0.013828241563512,...
            0.015928212162150,...
            0.016745957138504,...
            0.018774610553793,...
            0.019738005080187,...
            0.020286284358686,...
            0.021625343577948 ];        
    case 'full'
        switch varargin{3}
            case 'allowOutside'
                % _________________________________________________________
                % Degree 25, 118 Points, PO, (6 points just outside), Fully
                % symmetric, Reference: [P2], reported as new in Table 3 on
                % page 46. Numerical values obtained from online supple-
                % mental material of [P2].
                % ---------------------------------------------------------
                Qn.m = [ 1 3 3 3 3 3 3 3 3 3 3 3 6 6 6 6 6 6 6 6 6 6 6 ...
                    6 6 6 ];
                Qn.b1 = [ 0.333333333333333,...
                    0.007802219997900,...
                    0.001599897468272,...
                    0.039745976060169,...
                    0.021031495068615,...
                    0.090694417901959,...
                    0.151224281357815,...
                    0.065573869583164,...
                    0.211913294950495,...
                    0.134475020424120,...
                    0.273760447593868,...
                    0.225268296939059,...
                    -0.000175486429018,...
                    0.007627235424357,...
                    0.007174459818221,...
                    0.006367454172095,...
                    0.007487515753257,...
                    0.037463934879854,...
                    0.018103532201924,...
                    0.033495910148005,...
                    0.039206711918798,...
                    0.082016261697039,...
                    0.060505783675033,...
                    0.125482119096222,...
                    0.095367159491401,...
                    0.174659411566861 ];
                Qn.b2 = [ 0.333333333333333,...
                    0.007802219997900,...
                    0.499200051265864,...
                    0.039745976060169,...
                    0.489484252465692,...
                    0.090694417901959,...
                    0.151224281357815,...
                    0.467213065208418,...
                    0.211913294950495,...
                    0.432762489787940,...
                    0.273760447593868,...
                    0.387365851530470,...
                    0.275059108819841,...
                    0.040675651667318,...
                    0.098073197818081,...
                    0.177108966725932,...
                    0.381611685159126,...
                    0.095814659140258,...
                    0.270608545540926,...
                    0.173144453351436,...
                    0.369875713683459,...
                    0.164905597423194,...
                    0.259760411703464,...
                    0.241438155348013,...
                    0.348637419542873,...
                    0.316050027630207 ];
                Qn.w = [ 0.019730145564171,...
                    0.000797849780090,...
                    0.001654055755910,...
                    0.003948306781559,...
                    0.007305930441956,...
                    0.008127855879641,...
                    0.011970371685576,...
                    0.012299765626204,...
                    0.016112719176053,...
                    0.016232869701501,...
                    0.018798881038162,...
                    0.018817300377345,...
                    0.001005841963635,...
                    0.001776834195195,...
                    0.002526376804661,...
                    0.002890321334426,...
                    0.004318817750982,...
                    0.005628037242900,...
                    0.006238035108432,...
                    0.006524270655411,...
                    0.009575674298140,...
                    0.009671725378103,...
                    0.010525535644239,...
                    0.013760331394148,...
                    0.013860745244640,...
                    0.017042808935730 ];
            case 'inside'
                switch varargin{2}
                    case 'allowNegative'
                        % _________________________________________________
                        % Degree 25, 120 Points, NI (6 negative weights),
                        % Fully symmetric, Reference: [P2], not reported as
                        % new but couldn't find any other reference.
                        % Numerical values obtained from online supple-
                        % mental material of [P2].
                        % -------------------------------------------------
                        Qn.m = [ 3 3 3 3 3 3 3 3 3 3 3 3 6 6 6 6 6 6 6 ...
                            6 6 6 6 6 6 6 ];
                        Qn.b1 = [ 0.241705564997226,...
                            0.066841145410301,...
                            0.007912137402099,...
                            0.038682500223883,...
                            0.008565148339804,...
                            0.096693451684011,...
                            0.046407894153638,...
                            0.151363116253387,...
                            0.110897126313796,...
                            0.276176997047182,...
                            0.196228695583230,...
                            0.229157170026698,...
                            0.000303302471091,...
                            0.007285066072018,...
                            0.002098190552789,...
                            0.008249355402163,...
                            0.009198448968891,...
                            0.022789435125159,...
                            0.043868460650469,...
                            0.024839829916622,...
                            0.046355430068864,...
                            0.071944171977123,...
                            0.077170587472223,...
                            0.114037668606329,...
                            0.153156702364265,...
                            0.239691112819589 ];
                        Qn.b2 = [ 0.241705564997226,...
                            0.066841145410301,...
                            0.007912137402099,...
                            0.038682500223883,...
                            0.495717425830098,...
                            0.096693451684011,...
                            0.476796052923181,...
                            0.151363116253387,...
                            0.444551436843102,...
                            0.361911501476409,...
                            0.401885652208385,...
                            0.229157170026698,...
                            0.178730929732424,...
                            0.041091962617909,...
                            0.380405002447554,...
                            0.098451598163584,...
                            0.271278993067566,...
                            0.173042709639872,...
                            0.091754132899196,...
                            0.376811651708434,...
                            0.265561584563093,...
                            0.170539121421737,...
                            0.355127335899842,...
                            0.244672816769111,...
                            0.328189451188988,...
                            0.245045069544506 ];
                        Qn.w = [ -2.695496122366103,...
                            -0.001805308728273,...
                            0.000818660001776,...
                            0.003872408022467,...
                            0.005195627289094,...
                            0.008275825603630,...
                            0.011494692925103,...
                            0.014289758700353,...
                            0.015595032733898,...
                            0.017062959875834,...
                            0.017127072397519,...
                            0.143068617453157,...
                            0.001137141814965,...
                            0.001713551941995,...
                            0.001968153757799,...
                            0.002898061980865,...
                            0.004776072712695,...
                            0.006489897974571,...
                            0.006841043477628,...
                            0.008264732280864,...
                            0.010224364896269,...
                            0.010244154062961,...
                            0.013671900203659,...
                            0.015057862935695,...
                            0.015806425931879,...
                            1.297823690740593 ];
                    case 'positive'
                        % _________________________________________________
                        % Degree 25, 120 Points, PI, Fully symmetric,
                        % Reference: [X1], see Table 1 on page 671. Not
                        % sure where to obtain numerical values.
                        % -------------------------------------------------
                        
                        % _________________________________________________
                        % Degree 25, 126 Points, PI, Fully symmetric,
                        % Reference: [W3], see Table 7 on page 1837. Numer-
                        % ical values obtained from [Z2].
                        % -------------------------------------------------
                        Qn.m = [ 3 3 3 3 3 3 3 3 3 3 6 6 6 6 6 6 6 6 6 ...
                            6 6 6 6 6 6 6 ];
                        Qn.b1 = [ 0.48602675846341286632108494106576852,...
                            0.43441069933617422686934557925604870,...
                            0.38988913524396382753969636952404462,...
                            0.29844323401980449068329287355490048,...
                            0.23404417233737183658023656970596734,...
                            0.15146833460901760000000000000000000,...
                            0.11273389354599349281729186486594851,...
                            0.07771569209152620223369817320827966,...
                            0.03489309361429690849533026177733858,...
                            0.00725818462093238916117013110606335,...
                            0.00129235270444219584099302931665302,...
                            0.00539970127211620182189732363132860,...
                            0.00638400303397498941895749776744540,...
                            0.00502821150199308271328582097413442,...
                            0.00682675862178184570668051931419220,...
                            0.01001619963992951145867446496978043,...
                            0.02575781317339004661287123576293491,...
                            0.03022789811991582345896227492462806,...
                            0.03050499010716207795935512861230592,...
                            0.04595654736256931928622033280840303,...
                            0.06744280054027761427028802732221226,...
                            0.07004509141591061747706842947324321,...
                            0.08391152464011664050042868280051014,...
                            0.12037553567715270000000000000000000,...
                            0.14806689915736669746097550576436943,...
                            0.19177186586732510675561189736742768 ];
                        Qn.b2(1:10) = Qn.b1(1:10);
                        Qn.b2(11:26) = [ 0.22721445215336410000000000000000000,...
                            0.43501055485357173125113362379450230,...
                            0.32030959927220447113000026445341979,...
                            0.09175032228000531039287201063543536,...
                            0.03801083585872443391784793002157854,...
                            0.15742521848531178511320753380098114,...
                            0.23988965977853325914298591466950265,...
                            0.36194311812606053179871201193912355,...
                            0.08355196095482845510691464351287596,...
                            0.14844322073241812268238560824628123,...
                            0.28373970872753497227148332506495505,...
                            0.40689937511878760390382808292080011,...
                            0.19411398702489250643935208871855679,...
                            0.32413434700070320631848835964861372,...
                            0.22927748355598104616627200389359665,...
                            0.32561812259598383120936369037062376 ];                        
                        Qn.w = [ 0.00800558188002042313368589293942171,...
                            0.01594707683239049085408554937789681,...
                            0.01310914123079550369839487163291327,...
                            0.01958300096563560520647726105123795,...
                            0.01647088544153726359394636789647424,...
                            0.00854727907409214131204679403754348,...
                            0.00816188585722650120293135168156145,...
                            0.00612114653998375062550303141747500,...
                            0.00290849826493666035989604760244737,...
                            0.00069227524566199872959231564688970,...
                            0.00124828919927740405463308732687347,...
                            0.00340475290880302630290932517884509,...
                            0.00335965432606405469389982622818046,...
                            0.00171615653949675776835927257382091,...
                            0.00148085631671560275966191325505238,...
                            0.00351131261072868028754024041382735,...
                            0.00739355014970648910670774728325426,...
                            0.00798308747737656611293770143977919,...
                            0.00435596261315803802370299459187773,...
                            0.00736505670141783108374850348433179,...
                            0.01096357284641954589593589943582605,...
                            0.01174996174354112327739198111865119,...
                            0.01001560071379857640896706453029824,...
                            0.01330964078762866755490152722632760,...
                            0.01415444650522613072733705129605375,...
                            0.01488137956116800324975278964173671 ];
                end                
        end
end

end%
%%
function Q  = quadTrianglePointsAndWeights(Qn,Domain) 
T = Domain;
if isempty(T)
    a = [0 0]; b = [1 0]; c = [0 1]; area = 1;
elseif size(T,1)==3 && size(T,2)==2
    a = [T(1,1),T(1,2)]; b = [T(2,1),T(2,2)]; c = [T(3,1),T(3,2)];
    area = 1/2*abs(det([a(1), a(2), 1; b(1), b(2), 1; c(1), c(2), 1]));
else
    error('''Domain'' should be an array of size 3×2 or an empty array.');
end
k = 1;
ng = sum(Qn.m);
Q.Points = zeros(ng,2);
Q.Weights = zeros(ng,1);
for i = 1:length(Qn.m)    
    if Qn.m(i) == 0 % No symmetry group
        Qn.b3(i) = 1 - (Qn.b1(i) + Qn.b2(i));
        Q.Points(k,1) = a(1)*Qn.b1(i) + b(1)*Qn.b2(i) + c(1)*Qn.b3(i);
        Q.Points(k,2) = a(2)*Qn.b1(i) + b(2)*Qn.b2(i) + c(2)*Qn.b3(i);
        Q.Weights(k,1) = area*Qn.w(i);
        k = k+1;
    end
    if Qn.m(i) == 1 % Center point    
        Qn.b2(i) = Qn.b1(i);
        Qn.b3(i) = 1 - (Qn.b1(i) + Qn.b2(i));
        Q.Points(k,1) = a(1)*Qn.b1(i) + b(1)*Qn.b2(i) + c(1)*Qn.b3(i);
        Q.Points(k,2) = a(2)*Qn.b1(i) + b(2)*Qn.b2(i) + c(2)*Qn.b3(i);   
        Q.Weights(k:k+Qn.m(i)-1) = area*Qn.w(i);        
        k = k+1;        
    end   
    if Qn.m(i) == 2 % Reflection symmetries
        Qn.b3(i) = 1 - (Qn.b1(i) + Qn.b2(i));
        Q.Points(k,1)   = a(1)*Qn.b1(i) + b(1)*Qn.b2(i) + c(1)*Qn.b3(i);
        Q.Points(k,2)   = a(2)*Qn.b1(i) + b(2)*Qn.b2(i) + c(2)*Qn.b3(i);        
        Q.Points(k+1,1) = a(1)*Qn.b2(i) + b(1)*Qn.b3(i) + c(1)*Qn.b1(i);
        Q.Points(k+1,2) = a(2)*Qn.b2(i) + b(2)*Qn.b3(i) + c(2)*Qn.b1(i);               
        Q.Weights(k:k+Qn.m(i)-1) = area*Qn.w(i);        
        k = k+2;                
    end
    if Qn.m(i) == 3 % Rotation symmetries    
        Qn.b3(i) = 1 - (Qn.b1(i) + Qn.b2(i));
        Q.Points(k,1)   = a(1)*Qn.b1(i) + b(1)*Qn.b2(i) + c(1)*Qn.b3(i);
        Q.Points(k,2)   = a(2)*Qn.b1(i) + b(2)*Qn.b2(i) + c(2)*Qn.b3(i);        
        Q.Points(k+1,1) = a(1)*Qn.b2(i) + b(1)*Qn.b3(i) + c(1)*Qn.b1(i);
        Q.Points(k+1,2) = a(2)*Qn.b2(i) + b(2)*Qn.b3(i) + c(2)*Qn.b1(i);        
        Q.Points(k+2,1) = a(1)*Qn.b3(i) + b(1)*Qn.b1(i) + c(1)*Qn.b2(i);
        Q.Points(k+2,2) = a(2)*Qn.b3(i) + b(2)*Qn.b1(i) + c(2)*Qn.b2(i);        
        Q.Weights(k:k+Qn.m(i)-1) = area*Qn.w(i);        
        k = k+3;        
    end  
    if Qn.m(i) == 6 % Reflection + rotation symmetry
        Qn.b3(i) = 1 - (Qn.b1(i) + Qn.b2(i));
        Q.Points(k,1)   = a(1)*Qn.b1(i) + b(1)*Qn.b2(i) + c(1)*Qn.b3(i);
        Q.Points(k,2)   = a(2)*Qn.b1(i) + b(2)*Qn.b2(i) + c(2)*Qn.b3(i);        
        Q.Points(k+1,1) = a(1)*Qn.b2(i) + b(1)*Qn.b3(i) + c(1)*Qn.b1(i);
        Q.Points(k+1,2) = a(2)*Qn.b2(i) + b(2)*Qn.b3(i) + c(2)*Qn.b1(i);        
        Q.Points(k+2,1) = a(1)*Qn.b3(i) + b(1)*Qn.b1(i) + c(1)*Qn.b2(i);
        Q.Points(k+2,2) = a(2)*Qn.b3(i) + b(2)*Qn.b1(i) + c(2)*Qn.b2(i);        
        Q.Points(k+3,1) = a(1)*Qn.b1(i) + b(1)*Qn.b3(i) + c(1)*Qn.b2(i);
        Q.Points(k+3,2) = a(2)*Qn.b1(i) + b(2)*Qn.b3(i) + c(2)*Qn.b2(i);        
        Q.Points(k+4,1) = a(1)*Qn.b2(i) + b(1)*Qn.b1(i) + c(1)*Qn.b3(i);
        Q.Points(k+4,2) = a(2)*Qn.b2(i) + b(2)*Qn.b1(i) + c(2)*Qn.b3(i);        
        Q.Points(k+5,1) = a(1)*Qn.b3(i) + b(1)*Qn.b2(i) + c(1)*Qn.b1(i);
        Q.Points(k+5,2) = a(2)*Qn.b3(i) + b(2)*Qn.b2(i) + c(2)*Qn.b1(i);        
        Q.Weights(k:k+Qn.m(i)-1) = area*Qn.w(i);        
        k = k+6;
    end
end
end




    
    
    
    
