function cmap = cmocean(ColormapName,varargin) 
% cmocean returns perceptually-uniform colormaps created by Kristen Thyng. 
% 
%% Syntax 
% 
%  cmocean 
%  cmap = cmocean('ColormapName') 
%  cmap = cmocean('-ColormapName') 
%  cmap = cmocean(...,NLevels)
%  cmap = cmocean(...,'pivot',PivotValue) 
%  cmap = cmocean(...,'negative') 
%  cmocean(...)
% 
%% Description 
% 
% cmocean without any inputs displays the options for colormaps. 
% 
% cmap = cmocean('ColormapName') returns a 256x3 colormap. ColormapName can be any of 
% of the following: 
% 
%          SEQUENTIAL:                DIVERGING: 
%          'thermal'                  'balance'
%          'haline'                   'delta'
%          'solar'                    'curl'
%          'ice'                      'diff'
%          'gray'                     'tarn'
%          'oxy' 
%          'deep'                     CONSTANT LIGHTNESS:
%          'dense'                    'phase'
%          'algae'                
%          'matter'                   OTHER:
%          'turbid'                   'topo'
%          'speed'                    
%          'amp'
%          'tempo'
%          'rain'
%
% cmap = cmocean('-ColormapName') a minus sign preceeding any ColormapName flips the
% order of the colormap. 
%
% cmap = cmocean(...,NLevels) specifies a number of levels in the colormap.  Default
% value is 256. 
%
% cmap = cmocean(...,'pivot',PivotValue) centers a diverging colormap such that white 
% corresponds to a given value and maximum extents are set using current caxis limits. 
% If no PivotValue is set, 0 is assumed. Early versions of this function used 'zero'
% as the syntax for 'pivot',0 and the old syntax is still supported. 
%
% cmap = cmocean(...,'negative') inverts the lightness profile of the colormap. This can be 
% useful particularly for divergent colormaps if the default white point of divergence
% gets lost in a white background. 
% 
% cmocean(...) without any outputs sets the current colormap to the current axes.  
% 
%% Examples 
% Using this sample plot: 
% 
%   imagesc(peaks(1000)+1)
%   colorbar
% 
% Set the colormap to 'algae': 
% 
%   cmocean('algae') 
% 
% Same as above, but with an inverted algae colormap: 
% 
%   cmocean('-algae')
% 
% Set the colormap to a 12-level 'solar': 
% 
%   cmocean('solar',12)
% 
% Get the RGB values of a 5-level thermal colormap: 
% 
%   RGB = cmocean('thermal',5)
% 
% Some of those values are below zero and others are above. If this dataset represents
% anomalies, perhaps a diverging colormap is more appropriate: 
% 
%   cmocean('balance') 
% 
% It's unlikely that 1.7776 is an interesting value about which the data values 
% diverge.  If you want to center the colormap on zero using the current color 
% axis limits, simply include the 'pivot' option:  
% 
%   cmocean('balance','pivot',0) 
%
%% Author Info 
% This function was written by Chad A. Greene of the Institute for Geophysics at the 
% University of Texas at Austin (UTIG), June 2016, using colormaps created by Kristen
% Thyng of Texas A&M University, Department of Oceanography. More information on the
% cmocean project can be found at http://matplotlib.org/cmocean/. 
% 
%% Citing this colormap: 
% If you find an occasion to cite these colormaps for any reason, or if you just want
% some nice beach reading, check out the following paper from the journal Oceanography: 
% 
% Thyng, K.M., C.A. Greene, R.D. Hetland, H.M. Zimmerle, and S.F. DiMarco. 2016. True 
% colors of oceanography: Guidelines for effective and accurate colormap selection. 
% Oceanography 29(3):9?13, http://dx.doi.org/10.5670/oceanog.2016.66.
% 
% See also colormap and caxis.  

%% Display colormap options: 

if nargin==0
   figure('menubar','none','numbertitle','off','Name','cmocean options:')
   
   if license('test','image_toolbox')
      imshow(imread('cmocean.png')); 
   else
      axes('pos',[0 0 1 1])
      image(imread('cmocean.png')); 
      axis image off
   end
   
   return
end
%% Error checks: 

assert(isnumeric(ColormapName)==0,'Input error: ColormapName must be a string.') 

%% Set defaults: 

NLevels = 256; 
autopivot = false; 
PivotValue = 0; 
InvertedColormap = false; 

%% Parse inputs: 

% Does user want to flip the colormap direction? 
dash = regexp(ColormapName,'-'); 
if any(dash) 
   InvertedColormap = true; 
   ColormapName(dash) = []; 
end

% Forgive the British: 
if strncmpi(ColormapName,'grey',4)
   ColormapName = 'gray'; 
end

% Does the user want a "negative" version of the colormap (with an inverted lightness profile)? 
tmp = strncmpi(varargin,'negative',3); 
if any(tmp) 
   negativeColormap = true; 
   varargin = varargin(~tmp); 
else
   negativeColormap = false; 
end

% Does the user want to center a diverging colormap on a specific value? 
% This parsing support original 'zero' syntax and current 'pivot' syntax. 
 tmp = strncmpi(varargin,'pivot',3) | strncmpi(varargin,'zero',3); % Thanks to Phelype Oleinik for this suggestion. 
 if any(tmp) 
   autopivot = true; 
   try
      if isscalar(varargin{find(tmp)+1})
         PivotValue = varargin{find(tmp)+1}; 
         tmp(find(tmp)+1) = 1; 
      end
   end
   varargin = varargin(~tmp); 
end

% Has user requested a specific number of levels? 
tmp = isscalar(varargin); 
if any(tmp) 
   NLevels = varargin{tmp}; 
end


%% Load RGB values and interpolate to NLevels: 

cmap = cmoceanRawRGB(ColormapName); % a subfunction provided below with RGB values of all maps. 

if negativeColormap
   
   % Convert RGB to LAB colorspace: 
   LAB = colorspace('RGB->LAB',cmap); 

   % Operate on the lightness profile: 
   L = LAB(:,1); 

   % Flip the lightness profile and set the lowest point to black:
   L = max(L) - L; 

   % Stretch the lightness profile to make the lightest bits 95% white. (Going 100% white
   % would make the ends of a divergent profile impossible to distinguish.)
   L = L*(95/max(L)); 

   % Make a new LAB matrix: 
   LAB = [L LAB(:,2:3)]; 
   
   % Convert LAB back to RGB: 
   cmap = colorspace('LAB->RGB',LAB); 
end

%% Invert the colormap if requested by user: 

if InvertedColormap
   cmap = flipud(cmap); 
end

%% Adjust values to current caxis limits? 

if autopivot
   CL = clim; 
    
   % Hilmar: I got tired of this error message, anyhow if the user wants a pivot value, give the user a pivot value and just
   % extend the limits to include it.
   if PivotValue < CL(1)
       PivotValue=CL(1)-100*eps(CL(1)); 
   end
   if PivotValue > CL(2)
       PivotValue=CL(2)+100*eps(CL(2)); 
   end

   assert(PivotValue>=CL(1) & PivotValue<=CL(2),'Error: pivot value must be within the current color axis limits.') 
   maxval = max(abs(CL-PivotValue)); 
   cmap = interp1(linspace(-maxval,maxval,size(cmap,1))+PivotValue, cmap, linspace(CL(1),CL(2),size(cmap,1)),'linear');
end

%% Interpolate if necessary: 

if NLevels~=size(cmap,1)
   cmap = interp1(1:size(cmap,1), cmap, linspace(1,size(cmap,1),NLevels),'linear');
end

%% Clean up 

if nargout==0
   colormap(gca,cmap) 
   clear cmap  
end



%%  S U B F U N C T I O N S 


function RGB = cmoceanRawRGB(cmapName) 


switch lower(cmapName(1:3))
   case {'dee'} 
      RGB = [9.928371765383620096e-01 9.943734553013935384e-01 8.001361955494933342e-01
9.849374457410008388e-01 9.913545172197536504e-01 7.953271573982337861e-01
9.770418482420034634e-01 9.883418276673759939e-01 7.905717472557142189e-01
9.691488355955474310e-01 9.853357520972024775e-01 7.858697059005903540e-01
9.612382061888059548e-01 9.823439661710483550e-01 7.812181455049906909e-01
9.533233708513803029e-01 9.793607550577946297e-01 7.766197776697654209e-01
9.454069776374501854e-01 9.763847481120443428e-01 7.720753845210462929e-01
9.374799177499437697e-01 9.734191074102575003e-01 7.675843461643131471e-01
9.295353395208965086e-01 9.704660061861888343e-01 7.631467470398153319e-01
9.215850348376466439e-01 9.675205198521025229e-01 7.587646801057830181e-01
9.136274030668488644e-01 9.645828701859077148e-01 7.544386825530340346e-01
9.056514559384564178e-01 9.616567079910524063e-01 7.501688017241657791e-01
8.976569922484586295e-01 9.587415918264282633e-01 7.459562027416458685e-01
8.896512126622908578e-01 9.558344072358454513e-01 7.418023698547455691e-01
8.816325974742758032e-01 9.529352564923027069e-01 7.377082218108865774e-01
8.735986658557798323e-01 9.500445573198598170e-01 7.336747582572585857e-01
8.655355018678880796e-01 9.471666805105979359e-01 7.297031220835451526e-01
8.574559441871676402e-01 9.442965454208677167e-01 7.257948454644865821e-01
8.493586413080321806e-01 9.414341108550433601e-01 7.219511807036801398e-01
8.412422934868647451e-01 9.385792958890862847e-01 7.181734629573161000e-01
8.331056602292961077e-01 9.357319779845385543e-01 7.144631077114116380e-01
8.249438571064254822e-01 9.328932140048489252e-01 7.108218561594963347e-01
8.167536485550084269e-01 9.300634438432798801e-01 7.072516022427790539e-01
8.085407925860420564e-01 9.272401607530683654e-01 7.037535913222745521e-01
8.003043988510905038e-01 9.244230580301131539e-01 7.003294929842703853e-01
7.920436855303579771e-01 9.216117772405966191e-01 6.969810377273069069e-01
7.837579923068553889e-01 9.188059058340631857e-01 6.937100117646464170e-01
7.754467945117471395e-01 9.160049747102424478e-01 6.905182510191696377e-01
7.671097184662585278e-01 9.132084557727712104e-01 6.874076341947011892e-01
7.587465580267132026e-01 9.104157595099175992e-01 6.843800748081978469e-01
7.503572923166539343e-01 9.076262326497597233e-01 6.814375120698907828e-01
7.419421046029871514e-01 9.048391559449654453e-01 6.785819005039074314e-01
7.335014022414247936e-01 9.020537421501462205e-01 6.758151982106467281e-01
7.250358375800118882e-01 8.992691342626382145e-01 6.731393536848668813e-01
7.165463296678448168e-01 8.964844041051395207e-01 6.705562911206142118e-01
7.080340865696405084e-01 8.936985513356989763e-01 6.680678941563225059e-01
6.995006280355556827e-01 8.909105029766873907e-01 6.656759880411459163e-01
6.909478082204841831e-01 8.881191135591984809e-01 6.633823202371271766e-01
6.823778380886152961e-01 8.853231659823359578e-01 6.611885395113676900e-01
6.737933070789577927e-01 8.825213731875580780e-01 6.590961736178440056e-01
6.651972035471812594e-01 8.797123807461469935e-01 6.571066057195337207e-01
6.565929334411800822e-01 8.768947704523611941e-01 6.552210497573525139e-01
6.479843366143701600e-01 8.740670650055498703e-01 6.534405250319036407e-01
6.393757001353049807e-01 8.712277338509245572e-01 6.517658303256151919e-01
6.307693113306602761e-01 8.683757593089920235e-01 6.501986071208971651e-01
6.221681809999368706e-01 8.655099786991408140e-01 6.487403091120448329e-01
6.135828162115329887e-01 8.626276255310663110e-01 6.473888753222819537e-01
6.050194699551968425e-01 8.597270135144634562e-01 6.461439080647282118e-01
5.964848373853082197e-01 8.568064537576968176e-01 6.450046593443751197e-01
5.879840953182007279e-01 8.538646600753627691e-01 6.439710905713145195e-01
5.795218127572832056e-01 8.509005270271372545e-01 6.430435569229606685e-01
5.711115213696185133e-01 8.479112775740021979e-01 6.422171169895777298e-01
5.627614837100215484e-01 8.448953489705807174e-01 6.414893854468299850e-01
5.544783665092802849e-01 8.418515860497588488e-01 6.408587965640016870e-01
5.462695843545278818e-01 8.387787842708503971e-01 6.403232223573597226e-01
5.381481232862911357e-01 8.356748924828519831e-01 6.398763669277579558e-01
5.301228248165350543e-01 8.325387580839158641e-01 6.395142187190840932e-01
5.221997330825246530e-01 8.293697946738645133e-01 6.392346216632595057e-01
5.143908532894702068e-01 8.261665755050031645e-01 6.390304309659169402e-01
5.067050118087177424e-01 8.229283397233977393e-01 6.388963162980869637e-01
4.991495064662723746e-01 8.196546689563362076e-01 6.388277739100475250e-01
4.917330703220189059e-01 8.163450533427777378e-01 6.388184799685107107e-01
4.844633050708381794e-01 8.129992646664957467e-01 6.388624455765857801e-01
4.773469739538908629e-01 8.096172922533247940e-01 6.389538165444101914e-01
4.703904182695624603e-01 8.061992786917829834e-01 6.390864379965265352e-01
4.635986539031755060e-01 8.027456073395251579e-01 6.392548734619343254e-01
4.569766618393767410e-01 7.992567478306369377e-01 6.394529319719780558e-01
4.505287643111087204e-01 7.957333261733031682e-01 6.396742566585853496e-01
4.442569334258392177e-01 7.921762343010834151e-01 6.399148201157260907e-01
4.381634653667463852e-01 7.885863788123392837e-01 6.401694292235835526e-01
4.322513883314121341e-01 7.849647158245569578e-01 6.404304190942151642e-01
4.265195639210723755e-01 7.813124732411278472e-01 6.406960731925235297e-01
4.209679602144861810e-01 7.776308288475485275e-01 6.409622276506293792e-01
4.155968407812616339e-01 7.739210344346504344e-01 6.412227317683711902e-01
4.104045242070183952e-01 7.701844253858858291e-01 6.414742347181462412e-01
4.053883482522956383e-01 7.664223146281371468e-01 6.417151578859574546e-01
4.005459328058830759e-01 7.626360316845265386e-01 6.419424749255564500e-01
3.958750262046074608e-01 7.588270315806920907e-01 6.421508285001665817e-01
3.913718273240804346e-01 7.549966666219802836e-01 6.423387337484185444e-01
3.870322615152646528e-01 7.511462095764531721e-01 6.425055472082883412e-01
3.828523536620307977e-01 7.472769743775978801e-01 6.426493131526664904e-01
3.788278449989629926e-01 7.433902546660990929e-01 6.427683061328856029e-01
3.749542811367644335e-01 7.394876286687900313e-01 6.428573615340694714e-01
3.712266123106295335e-01 7.355700686179795778e-01 6.429189618774799886e-01
3.676399929257251897e-01 7.316387718079003788e-01 6.429520487195032885e-01
3.641894512092864744e-01 7.276949011320993366e-01 6.429557270674552960e-01
3.608699207518579755e-01 7.237395816489646805e-01 6.429292709767442382e-01
3.576762479926827720e-01 7.197739138067411613e-01 6.428719883285997083e-01
3.546027248366774298e-01 7.157992458648064771e-01 6.427810759039616073e-01
3.516446005650431528e-01 7.118162697085735902e-01 6.426589560156208414e-01
3.487967412412643076e-01 7.078259414050183107e-01 6.425054728664090220e-01
3.460540467004642462e-01 7.038291754672099110e-01 6.423205839967465192e-01
3.434114694865876838e-01 6.998268449671132263e-01 6.421043512357870187e-01
3.408640318201912600e-01 6.958197818437616977e-01 6.418569321473704958e-01
3.384068406811324148e-01 6.918087773726725453e-01 6.415785719753461791e-01
3.360351011011302735e-01 6.877945827659479594e-01 6.412695960850323118e-01
3.337441277691141628e-01 6.837779098758932639e-01 6.409304028912300444e-01
3.315291322901112725e-01 6.797595093162658308e-01 6.405610222271946874e-01
3.293855731203558790e-01 6.757400463409192204e-01 6.401618521317685717e-01
3.273095072934468774e-01 6.717199943156246800e-01 6.397341828681051279e-01
3.252967835603527980e-01 6.676999151191671533e-01 6.392786349130804568e-01
3.233433949344308722e-01 6.636803359579184214e-01 6.387958727683986648e-01
3.214454825944703664e-01 6.596617502568347113e-01 6.382866005148117861e-01
3.195993388733257556e-01 6.556446185422359907e-01 6.377515576836865208e-01
3.178014094235615539e-01 6.516293693095515094e-01 6.371915154212616228e-01
3.160482946464280851e-01 6.476163998706647718e-01 6.366072729215248582e-01
3.143367504654775990e-01 6.436060771766987099e-01 6.359996541044424800e-01
3.126636885203824545e-01 6.395987386132337971e-01 6.353695045172913503e-01
3.110261758511674857e-01 6.355946927658273626e-01 6.347176884379418516e-01
3.094214341373541788e-01 6.315942201545601264e-01 6.340450861601970578e-01
3.078468385510386152e-01 6.275975739369872297e-01 6.333525914425652825e-01
3.062999162775806861e-01 6.236049805794583456e-01 6.326411091031564071e-01
3.047783447524610168e-01 6.196166404972232034e-01 6.319115527447410896e-01
3.032799496578738041e-01 6.156327286641396501e-01 6.311648425953061414e-01
3.018027027180503197e-01 6.116533951930802626e-01 6.304019034507418739e-01
3.003447193279823457e-01 6.076787658883653354e-01 6.296236627075371128e-01
2.989042560461285802e-01 6.037089427717098333e-01 6.288310484745529561e-01
2.974797079780254205e-01 5.997440045832753697e-01 6.280249877540684533e-01
2.960696060742640245e-01 5.957840072594756675e-01 6.272064046833601969e-01
2.946726143633343065e-01 5.918289843891730850e-01 6.263762188290394883e-01
2.932875271368584058e-01 5.878789476499071132e-01 6.255353435272825724e-01
2.919132661023912667e-01 5.839338872257091584e-01 6.246846842638799080e-01
2.905488775167375803e-01 5.799937722079637759e-01 6.238251370887518688e-01
2.891935293105973304e-01 5.760585509806988025e-01 6.229575870601862242e-01
2.878465082138687570e-01 5.721281515915097593e-01 6.220829067145717817e-01
2.865071747347951447e-01 5.682024945496734203e-01 6.212019079466474247e-01
2.851749861947763254e-01 5.642814857106783766e-01 6.203153697409601319e-01
2.838497050469647731e-01 5.603649539888247988e-01 6.194242689076802089e-01
2.825310677598035780e-01 5.564527485481920444e-01 6.185294158631611250e-01
2.812189151446758406e-01 5.525446999518639490e-01 6.176316008625786225e-01
2.799131896111074491e-01 5.486406204645675189e-01 6.167315925457064196e-01
2.786139324558262742e-01 5.447403043452749838e-01 6.158301364812402978e-01
2.773212811865227168e-01 5.408435281299215358e-01 6.149279537054621603e-01
2.760354668808263079e-01 5.369500509042623992e-01 6.140257392505246159e-01
2.747568115804877587e-01 5.330596145668072827e-01 6.131241606570347891e-01
2.734857257205313141e-01 5.291719440816851083e-01 6.122238564648385672e-01
2.722227055926091377e-01 5.252867477212617153e-01 6.113254346750160995e-01
2.709683308416185876e-01 5.214037172983150281e-01 6.104294711750373192e-01
2.697232619941725140e-01 5.175225283875982685e-01 6.095365081178055755e-01
2.684882380171457750e-01 5.136428405367017280e-01 6.086470522439324515e-01
2.672640739041831637e-01 5.097642974662250914e-01 6.077615731349965689e-01
2.660516582874621339e-01 5.058865272594796902e-01 6.068805013837572648e-01
2.648519510715739433e-01 5.020091425421505660e-01 6.060042266652778675e-01
2.636659810857353015e-01 4.981317406526836744e-01 6.051330956906521008e-01
2.624948437498288434e-01 4.942539038045714039e-01 6.042674100224807443e-01
2.613396987489819967e-01 4.903751992421883643e-01 6.034074237284018372e-01
2.602017811880152354e-01 4.864951750164123179e-01 6.025533553474511361e-01
2.590825816132030779e-01 4.826133005480903182e-01 6.017055804809932074e-01
2.579832374637801018e-01 4.787291639776031782e-01 6.008639770499764055e-01
2.569051367797046126e-01 4.748422758923547815e-01 6.000285749995881712e-01
2.558497193309375306e-01 4.709521329962904068e-01 5.991993407256713811e-01
2.548184740794410263e-01 4.670582183399953347e-01 5.983761730113198452e-01
2.538129365659009817e-01 4.631600015854966945e-01 5.975588985257048735e-01
2.528346862021874641e-01 4.592569393176054171e-01 5.967472668238706923e-01
2.518853434473967146e-01 4.553484754161292725e-01 5.959409447786497838e-01
2.509665668417216944e-01 4.514340415062205181e-01 5.951395103673859932e-01
2.500800498681828299e-01 4.475130575075850214e-01 5.943424457267542094e-01
2.492275176071283571e-01 4.435849323073600692e-01 5.935491293785424283e-01
2.484107231427581941e-01 4.396490645862106139e-01 5.927588275177814170e-01
2.476314436738927260e-01 4.357048438328084417e-01 5.919706842419948378e-01
2.468914762732849488e-01 4.317516515883728090e-01 5.911837105866416531e-01
2.461926332303804310e-01 4.277888629705645096e-01 5.903967722170088139e-01
2.455367369014649914e-01 4.238158485348740845e-01 5.896085756110598375e-01
2.449256139784778408e-01 4.198319765418777605e-01 5.888176525512366366e-01
2.443611399380560822e-01 4.158365980702596332e-01 5.880223891276964432e-01
2.438461792188084676e-01 4.118287207942545325e-01 5.872218507211212080e-01
2.433817108226277726e-01 4.078080077755985022e-01 5.864131553289283483e-01
2.429695407602900925e-01 4.037738393736669540e-01 5.855939512690606641e-01
2.426114394907277760e-01 3.997256137994158465e-01 5.847615988448080504e-01
2.423091294682429564e-01 3.956627542073260506e-01 5.839131390484204598e-01
2.420642707838099872e-01 3.915847171721096864e-01 5.830452587571144374e-01
2.418784445905285962e-01 3.874910027793410650e-01 5.821542521747482546e-01
2.417531339648231747e-01 3.833811665857123074e-01 5.812359783131098023e-01
2.416897018162703636e-01 3.792548337300427064e-01 5.802858143751018494e-01
2.416907565718227624e-01 3.751112253980636857e-01 5.792995847596160708e-01
2.417595622535183009e-01 3.709493751547713325e-01 5.782729391945388153e-01
2.418940089237943680e-01 3.667702659833258494e-01 5.771972660307567171e-01
2.420946965383475313e-01 3.625739944720036134e-01 5.760653956197256953e-01
2.423619198524908369e-01 3.583608468185064400e-01 5.748693688323588402e-01
2.426978313653707087e-01 3.541305556383861908e-01 5.736016010478480753e-01
2.431056167697961956e-01 3.498826161637011989e-01 5.722540340252517677e-01
2.435790408918375727e-01 3.456199709880173332e-01 5.708127541670769967e-01
2.441164681518231960e-01 3.413441041408621368e-01 5.692658625959313712e-01
2.447220255685461088e-01 3.370546900675063240e-01 5.676029832475817383e-01
2.453897790312707938e-01 3.327551263503047418e-01 5.658082210414479007e-01
2.461121481740833894e-01 3.284495884603169102e-01 5.638648470620498676e-01
2.468912126448456479e-01 3.241391993124722593e-01 5.617586120246663706e-01
2.477157002362878613e-01 3.198299095156284522e-01 5.594704462476630669e-01
2.485761162021524751e-01 3.155272777788998839e-01 5.569822129976372826e-01
2.494682375867288693e-01 3.112353605277175528e-01 5.542766318947622839e-01
2.503729202057394798e-01 3.069632816204772574e-01 5.513352372309007210e-01
2.512817651002536290e-01 3.027167370222472731e-01 5.481422664464320471e-01
2.521753740527091225e-01 2.985048904754492582e-01 5.446839245346258851e-01
2.530378388279626023e-01 2.943355684042912035e-01 5.409499814033098541e-01
2.538514672799476179e-01 2.902167617845944347e-01 5.369345375383627328e-01
2.545978636988186494e-01 2.861560770732184955e-01 5.326370186681820273e-01
2.552597334794323158e-01 2.821600341185547811e-01 5.280624026249696179e-01
2.558254018044066047e-01 2.782328077909240194e-01 5.232189983935346955e-01
2.562737222001675308e-01 2.743799915974786674e-01 5.181261586082565040e-01
2.566051651711689918e-01 2.706008890121900934e-01 5.127963599343589030e-01
2.568071232679735028e-01 2.668971447705413280e-01 5.072541841677337127e-01
2.568729082739400482e-01 2.632680806498599591e-01 5.015251413221865073e-01
2.568088786232387566e-01 2.597098435540172168e-01 4.956253550777797723e-01
2.566151212870083631e-01 2.562195453869526296e-01 4.895770754997314511e-01
2.562879598478859378e-01 2.527944349049762174e-01 4.834087259028412853e-01
2.558322794870629413e-01 2.494300496133808887e-01 4.771399728195321877e-01
2.552572079534596305e-01 2.461214417402471932e-01 4.707834286138410929e-01
2.545675711557803811e-01 2.428643843006248471e-01 4.643557571030281772e-01
2.537686502971923108e-01 2.396546885092911139e-01 4.578718126285494239e-01
2.528659964243425984e-01 2.364882912834416206e-01 4.513446141213755536e-01
2.518652755598765891e-01 2.333613180085921113e-01 4.447853910070804773e-01
2.507721436866522935e-01 2.302701237479483076e-01 4.382036814126420432e-01
2.495921493408267966e-01 2.272113165747541852e-01 4.316074651476269897e-01
2.483306606950595463e-01 2.241817667402334346e-01 4.250033167249620547e-01
2.469928136315570621e-01 2.211786051187170643e-01 4.183965667275956202e-01
2.455817950644241798e-01 2.181991605590304917e-01 4.117955160707330586e-01
2.440963016438912891e-01 2.152407079180826410e-01 4.052187485307188752e-01
2.425474613143962510e-01 2.123011804696942062e-01 3.986531809377649171e-01
2.409394174709658110e-01 2.093786088479052676e-01 3.921004515047561423e-01
2.392760085101002243e-01 2.064712127546999287e-01 3.855616338648150121e-01
2.375559510487991743e-01 2.035768990151702318e-01 3.790513023869773179e-01
2.357791582066215419e-01 2.006936504843463420e-01 3.725810406048237766e-01
2.339566833930070699e-01 1.978208439176197264e-01 3.661282774694271658e-01
2.320913707908846546e-01 1.949572807641995476e-01 3.596924727656225507e-01
2.301734316844865069e-01 1.921000566693645550e-01 3.533131710447778295e-01
2.282155151615900546e-01 1.892493734782149939e-01 3.469585312854190362e-01
2.262221013532286218e-01 1.864046213456004575e-01 3.406203233993785884e-01
2.241821302892643142e-01 1.835626593748518609e-01 3.343431537784405938e-01
2.221083620693462546e-01 1.807243380946543521e-01 3.280900612914751102e-01
2.200027865362182422e-01 1.778889096627078448e-01 3.218593457818205161e-01
2.178554139765243036e-01 1.750533261852178779e-01 3.156932341101472139e-01
2.156819223383368844e-01 1.722195412332822306e-01 3.095395593647118360e-01
2.134730781003147948e-01 1.693846918206770857e-01 3.034377234324792671e-01
2.112350828916275125e-01 1.665490510713014960e-01 2.973690772834624574e-01
2.089702734855372057e-01 1.637121940546225618e-01 2.913286801723345421e-01
2.066728560428846562e-01 1.608719311731062473e-01 2.853437530392856081e-01
2.043551063834350701e-01 1.580301135357929931e-01 2.793689192405632293e-01
2.020028382233076125e-01 1.551826890051333785e-01 2.734655011163185101e-01
1.996333580091984583e-01 1.523327614839000976e-01 2.675659276139747411e-01
1.972319620276792862e-01 1.494761585027865047e-01 2.617344119314569673e-01
1.948137533226730334e-01 1.466155558553730864e-01 2.559107021731873988e-01
1.923665760685599468e-01 1.437473901381048913e-01 2.501489823140411461e-01
1.899031789417767457e-01 1.408738040042367690e-01 2.443975812138057813e-01
1.874125186315797886e-01 1.379915476831030108e-01 2.387062880680187460e-01
1.849071752981109873e-01 1.351026683367795023e-01 2.330231779340583564e-01
1.823751828761793481e-01 1.322038111876145949e-01 2.274020989868827392e-01
1.798308834661569988e-01 1.292972741720705698e-01 2.217828373102059270e-01
1.772595702951434704e-01 1.263792859047284667e-01 2.162309361702130506e-01
1.746769766714075522e-01 1.234522307754042925e-01 2.106797904214151584e-01
1.720703275074147443e-01 1.205129174047056828e-01 2.051860710532931176e-01
1.694509066570421274e-01 1.175626362090423926e-01 1.997023294326863430e-01
1.668117590877864764e-01 1.145994139402238821e-01 1.942594655847655616e-01
1.641571774416621943e-01 1.116231764992316466e-01 1.888399155026514453e-01
1.614878059346168959e-01 1.086331782717616379e-01 1.834416540200605461e-01
1.587995102818766935e-01 1.056281331759973130e-01 1.780821476813713722e-01
1.561019746507273376e-01 1.026082525875711138e-01 1.727215696232307918e-01]; 

   case 'mat' 
      RGB = [9.942936149611202312e-01 9.303277953232079733e-01 6.910969022498407721e-01
9.938034737629607429e-01 9.248067799917484288e-01 6.856470896792071779e-01
9.932915386396504731e-01 9.192993747690529904e-01 6.802308895677181555e-01
9.927922245331627371e-01 9.137917294903226129e-01 6.748329403835298113e-01
9.922937701498860674e-01 9.082882194233706796e-01 6.694590190525575579e-01
9.917843550306121303e-01 9.027933977435592672e-01 6.641148019075449049e-01
9.912962746539998315e-01 8.972938428980343772e-01 6.587860434151366906e-01
9.907825925330567829e-01 8.918085346471608110e-01 6.534941055176841651e-01
9.902949549053171596e-01 8.863158938146130650e-01 6.482161486636766057e-01
9.897868190815676259e-01 8.808350065840212517e-01 6.429731121246738956e-01
9.892857473867685547e-01 8.753541152485579957e-01 6.377529594546893499e-01
9.887800796370288525e-01 8.698778821960219121e-01 6.325610007226144527e-01
9.882646997282734658e-01 8.644083076900085372e-01 6.273995656371403884e-01
9.877584964502673648e-01 8.589369989413550011e-01 6.222606017222616082e-01
9.872280289134517384e-01 8.534782966339129473e-01 6.171585665698279266e-01
9.867183884350317902e-01 8.480122040334344691e-01 6.120742768604477968e-01
9.861766767190066618e-01 8.425618846460892764e-01 6.070302342911766402e-01
9.856563017997251874e-01 8.371033300354000506e-01 6.020039838864913451e-01
9.851070454224936102e-01 8.316588814371468352e-01 5.970165676525449605e-01
9.845690242211674326e-01 8.262101763023093071e-01 5.920513492549713819e-01
9.840096682146298734e-01 8.207718909127237339e-01 5.871217557569952117e-01
9.834535848293927129e-01 8.153324961500217904e-01 5.822177451775890633e-01
9.828817074848651414e-01 8.099006555998068402e-01 5.773470477348076058e-01
9.823072425148250408e-01 8.044699893992296458e-01 5.725043678439867278e-01
9.817205500240108185e-01 7.990448613324698801e-01 5.676935526547516320e-01
9.811274651130897917e-01 7.936222997819741831e-01 5.629123141708799460e-01
9.805237822250761903e-01 7.882041368744077126e-01 5.581623151179201381e-01
9.799119019116716567e-01 7.827890166019746410e-01 5.534426550185235216e-01
9.792891597791347769e-01 7.773780574795166043e-01 5.487543863553446810e-01
9.786583517021393286e-01 7.719696800001862869e-01 5.440965033742034551e-01
9.780145739702225116e-01 7.665661517278330450e-01 5.394708896034823287e-01
9.773647283145959763e-01 7.611637891820339785e-01 5.348750765091280224e-01
9.766980163585675667e-01 7.557679109997390565e-01 5.303130790083464552e-01
9.760290252537030531e-01 7.503708130006796484e-01 5.257797515465254534e-01
9.753375433133614214e-01 7.449828010040938642e-01 5.212823917018596376e-01
9.746492807368433153e-01 7.395902023503717615e-01 5.168121142252199984e-01
9.739312415395592337e-01 7.342102748450782812e-01 5.123804929986149892e-01
9.732235441351503313e-01 7.288214038955377339e-01 5.079740009039053206e-01
9.724836441494412176e-01 7.234462681298939879e-01 5.036071894033782304e-01
9.717498445504565430e-01 7.180638747370174935e-01 4.992675340324776445e-01
9.709912116410467364e-01 7.126909352035095679e-01 4.949654270232705655e-01
9.702261620325377534e-01 7.073170976907954266e-01 4.906951514278061199e-01
9.694484453171449134e-01 7.019455954348422511e-01 4.864590138179895606e-01
9.686504017535663147e-01 6.965805969226914751e-01 4.822596297437949375e-01
9.678531538881393059e-01 6.912098001765796251e-01 4.780909555078191597e-01
9.670272977476158660e-01 6.858499112812684873e-01 4.739621016861336744e-01
9.662030356838596790e-01 6.804831678592252464e-01 4.698646253054341027e-01
9.653577621840926382e-01 6.751225363106635458e-01 4.658054627874114728e-01
9.644956606754987449e-01 6.697653999656224544e-01 4.617837747090557943e-01
9.636302007256901669e-01 6.644033993552483919e-01 4.577964421810854501e-01
9.627382975013459854e-01 6.590502602174439506e-01 4.538500459526182973e-01
9.618419029037778012e-01 6.536923650498338567e-01 4.499394348167297664e-01
9.609293304705296412e-01 6.483365953124501369e-01 4.460681718992942080e-01
9.599915754015638791e-01 6.429884241455728899e-01 4.422388634417756537e-01
9.590555927894280908e-01 6.376305748866927248e-01 4.384459238793448344e-01
9.580962561892508722e-01 6.322786777663680358e-01 4.346957499886874854e-01
9.571138853949342495e-01 6.269324224641481536e-01 4.309887354294504314e-01
9.561315297025726467e-01 6.215764963092997863e-01 4.273209635732272416e-01
9.551250405128054455e-01 6.162263913356859080e-01 4.236980325915365997e-01
9.540950546856811210e-01 6.108815734129118269e-01 4.201203681538954182e-01
9.530626716897264705e-01 6.055275479248746207e-01 4.165851980241140895e-01
9.520078397406573911e-01 6.001775213709300560e-01 4.130968208871531600e-01
9.509298744158470873e-01 5.948318045431840728e-01 4.096560342134981103e-01
9.498400194890849191e-01 5.894823244090308112e-01 4.062621760942275451e-01
9.487347976692349638e-01 5.841311256224190895e-01 4.029168199567615405e-01
9.476064491016543689e-01 5.787834319880815759e-01 3.996218296412110127e-01
9.464553073333200617e-01 5.734388133376047136e-01 3.963780340202668340e-01
9.452939757532162757e-01 5.680877777926379713e-01 3.931855149232086899e-01
9.441119180634874875e-01 5.627376402983688131e-01 3.900463532360756713e-01
9.429065439155045469e-01 5.573900694724827076e-01 3.869617715840133476e-01
9.416779791961638058e-01 5.520447400757862999e-01 3.839328326459904850e-01
9.404312902807343555e-01 5.466974929733746658e-01 3.809606687756254551e-01
9.391670109637508812e-01 5.413475099598963336e-01 3.780467610439202097e-01
9.378784499903906058e-01 5.359996531278976573e-01 3.751924384200013285e-01
9.365655059652620018e-01 5.306537435419416138e-01 3.723989770682498146e-01
9.352280123612883855e-01 5.253096443760549850e-01 3.696677100588824927e-01
9.338657357609021492e-01 5.199672647340076725e-01 3.670000247302842578e-01
9.324834513518638346e-01 5.146223341056469502e-01 3.643980039003675286e-01
9.310764112648535207e-01 5.092783920261573227e-01 3.618628439391275986e-01
9.296425700359164379e-01 5.039368520229430271e-01 3.593958605013685692e-01
9.281814529069224440e-01 4.985978383651590851e-01 3.569985751712403399e-01
9.266925163734759385e-01 4.932615377693159719e-01 3.546725399001493528e-01
9.251751483021848355e-01 4.879282029910456120e-01 3.524193303048753223e-01
9.236286685007274455e-01 4.825981562592266960e-01 3.502405381564949738e-01
9.220523297832619036e-01 4.772717924986036864e-01 3.481377630517493160e-01
9.204453195710341484e-01 4.719495822836188847e-01 3.461126032707011468e-01
9.188067620643404210e-01 4.666320744633321582e-01 3.441666458384411431e-01
9.171357210169962526e-01 4.613198983953663013e-01 3.423014558239780292e-01
9.154312031382285664e-01 4.560137657258883093e-01 3.405185649261683878e-01
9.136921621394280546e-01 4.507144716528461159e-01 3.388194594142976412e-01
9.119175034344646491e-01 4.454228956112940563e-01 3.372055675093833527e-01
9.101060894923883593e-01 4.401400013227078634e-01 3.356782463109665438e-01
9.082567458304244834e-01 4.348668361548626016e-01 3.342387683926176800e-01
9.063682676234201541e-01 4.296045297451883127e-01 3.328883082070119848e-01
9.044394268935215253e-01 4.243542918485532223e-01 3.316279284576256203e-01
9.024689802311073317e-01 4.191174093801583456e-01 3.304585666082093254e-01
9.004556769853120368e-01 4.138952426354805536e-01 3.293810217125931472e-01
8.983982678499889962e-01 4.086892206819806583e-01 3.283959417554291327e-01
8.962955137593517918e-01 4.035008359311561543e-01 3.275038116986458414e-01
8.941481104554649395e-01 3.983293983456084875e-01 3.267070561329722400e-01
8.919536525149722728e-01 3.931778253066712803e-01 3.260048269152195366e-01
8.897101199351982181e-01 3.880487567649602565e-01 3.253961697320843505e-01
8.874163795567694413e-01 3.829439054205242554e-01 3.248807846244438635e-01
8.850713585473568568e-01 3.778650063033766049e-01 3.244581619630766411e-01
8.826755435924777959e-01 3.728118797277990004e-01 3.241299144458608672e-01
8.802274289557096010e-01 3.677869127615048805e-01 3.238946418203613176e-01
8.777249655021830410e-01 3.627933891756242590e-01 3.237493930733041925e-01
8.751673985732256744e-01 3.578330531078428023e-01 3.236927558288430484e-01
8.725555139394461923e-01 3.529055754992642124e-01 3.237261048951861064e-01
8.698877548666956727e-01 3.480139364384642886e-01 3.238460395212981457e-01
8.671629330678980452e-01 3.431608117920707524e-01 3.240490255671465425e-01
8.643813155550574834e-01 3.383468700732272794e-01 3.243343280815126350e-01
8.615430100192384977e-01 3.335730372253879472e-01 3.247007528957156497e-01
8.586467469801843944e-01 3.288425164717434512e-01 3.251427866013465451e-01
8.556930023091509074e-01 3.241559097884965657e-01 3.256590726700412941e-01
8.526821206317460877e-01 3.195140519652634459e-01 3.262477399527137223e-01
8.496135249080434271e-01 3.149194771604064691e-01 3.269031319055852869e-01
8.464880691147448344e-01 3.103723490573296884e-01 3.276240607685041994e-01
8.433059903582461603e-01 3.058739837233550030e-01 3.284066774658670473e-01
8.400675201697129779e-01 3.014258630212182100e-01 3.292462412802209526e-01
8.367737994501905918e-01 2.970277335155548926e-01 3.301419279880662416e-01
8.334250817653976462e-01 2.926812202808505847e-01 3.310879308798391207e-01
8.300223167271181257e-01 2.883865925403408248e-01 3.320815261966803544e-01
8.265664528479066409e-01 2.841441353717970020e-01 3.331197789552536870e-01
8.230582869411739999e-01 2.799545862699168719e-01 3.341980671944378423e-01
8.194989991910379690e-01 2.758177426460852733e-01 3.353144156579274116e-01
8.158895345625488682e-01 2.717340841090945536e-01 3.364643892127716640e-01
8.122310472907531276e-01 2.677035350490168386e-01 3.376451421817873721e-01
8.085246874446361254e-01 2.637260312682816465e-01 3.388536188130562565e-01
8.047716060121338222e-01 2.598015433395364782e-01 3.400863817867416095e-01
8.009730135703670983e-01 2.559297758719957794e-01 3.413409259751189473e-01
7.971300939112275774e-01 2.521105612652171368e-01 3.426140174094376434e-01
7.932440533013958017e-01 2.483435687707094552e-01 3.439030220163640239e-01
7.893160890135294538e-01 2.446284526280468330e-01 3.452052771464595993e-01
7.853473877765537736e-01 2.409648192333603833e-01 3.465182972816244766e-01
7.813391260814397388e-01 2.373522835108513029e-01 3.478394483472491694e-01
7.772924488739767490e-01 2.337903640474600642e-01 3.491667244042649942e-01
7.732084983091196406e-01 2.302786572731657100e-01 3.504975234338518764e-01
7.690883651827444822e-01 2.268166676306541119e-01 3.518300145036632465e-01
7.649331213342656088e-01 2.234039396519906795e-01 3.531621316810459876e-01
7.607438261029081383e-01 2.200400345294166726e-01 3.544916653625779235e-01
7.565214435019435024e-01 2.167244640001846911e-01 3.558173116773682421e-01
7.522670051172567485e-01 2.134568325306841485e-01 3.571365659018996275e-01
7.479813874637180060e-01 2.102366867865238520e-01 3.584483801868486030e-01
7.436655059621508634e-01 2.070636399021255625e-01 3.597509060488300325e-01
7.393202434785964838e-01 2.039373173302497233e-01 3.610423849932817841e-01
7.349463482724380992e-01 2.008573725915569486e-01 3.623219106278985358e-01
7.305447136853985279e-01 1.978234802458921360e-01 3.635871804824805653e-01
7.261159828724578214e-01 1.948353685346151387e-01 3.648375781480875379e-01
7.216608959662019762e-01 1.918927853792166127e-01 3.660715129191122186e-01
7.171801670483409774e-01 1.889955002148961394e-01 3.672874263418225982e-01
7.126743300531552805e-01 1.861433788319636073e-01 3.684846837309949108e-01
7.081441223914051175e-01 1.833362215388179839e-01 3.696612595209343710e-01
7.035900061356136215e-01 1.805739799909275023e-01 3.708165722616917348e-01
6.990125222046897902e-01 1.778565964180273684e-01 3.719494030784631367e-01
6.944122804123707970e-01 1.751839824866724127e-01 3.730580800830735622e-01
6.897896069146784992e-01 1.725562544195547443e-01 3.741422132560422997e-01
6.851450759297335047e-01 1.699733803139570343e-01 3.752000809033200213e-01
6.804790620490438480e-01 1.674354838987148208e-01 3.762308077572429355e-01
6.757918752796293616e-01 1.649427652357627616e-01 3.772336599439788940e-01
6.710841190288990843e-01 1.624951923227635209e-01 3.782065903997158807e-01
6.663559221262009835e-01 1.600931455660249692e-01 3.791494395940231965e-01
6.616076837858697601e-01 1.577367904785384745e-01 3.800608417430486607e-01
6.568398281175149567e-01 1.554262673949660112e-01 3.809393122355112515e-01
6.520524648278914759e-01 1.531620260338200923e-01 3.817844598609113071e-01
6.472460502131879290e-01 1.509441834093387669e-01 3.825945806932325444e-01
6.424208086222070735e-01 1.487730846896873349e-01 3.833687631504908433e-01
6.375768667154826375e-01 1.466491860640934397e-01 3.841063458261376740e-01
6.327147719614083510e-01 1.445724818775720732e-01 3.848052877974663111e-01
6.278345360543612363e-01 1.425435417995506715e-01 3.854652342077181104e-01
6.229363449371848604e-01 1.405627442689544315e-01 3.860852182514847852e-01
6.180208267587814497e-01 1.386299275595349323e-01 3.866630046493146899e-01
6.130878172636732293e-01 1.367458410848005068e-01 3.871986373954375282e-01
6.081376300127646628e-01 1.349106539902725221e-01 3.876907660008953038e-01
6.031708343692542273e-01 1.331241752511138077e-01 3.881374204452801013e-01
5.981872594198538451e-01 1.313871076769131119e-01 3.885385758335542783e-01
5.931872993271581906e-01 1.296994119208923768e-01 3.888927420384044042e-01
5.881715194167826954e-01 1.280607661499650052e-01 3.891980996789912162e-01
5.831397292746328676e-01 1.264718001616958465e-01 3.894546477166113685e-01
5.780923809284296278e-01 1.249322477491876249e-01 3.896608882675748342e-01
5.730301484941315859e-01 1.234414650841386962e-01 3.898149438393606059e-01
5.679527785339369972e-01 1.220000281824430710e-01 3.899169825800337663e-01
5.628606992123619257e-01 1.206075179177948653e-01 3.899657247446918218e-01
5.577549128735556083e-01 1.192626131188577743e-01 3.899588733885684388e-01
5.526350045348186191e-01 1.179659461688005728e-01 3.898969895870120772e-01
5.475013145908723677e-01 1.167170132781017999e-01 3.897791516609607765e-01
5.423552229242829537e-01 1.155136939391362971e-01 3.896026927627281311e-01
5.371963110518728213e-01 1.143564329775472455e-01 3.893682721938278024e-01
5.320248625085529648e-01 1.132445825251853777e-01 3.890752921818558807e-01
5.268418790923193873e-01 1.121763322989662304e-01 3.887220732362928199e-01
5.216480344214139420e-01 1.111502746625857019e-01 3.883076350820581779e-01
5.164430850451627864e-01 1.101663332257847294e-01 3.878324664548168932e-01
5.112275102681053118e-01 1.092232673876926685e-01 3.872959651352503863e-01
5.060029073959358970e-01 1.083180273650512782e-01 3.866960834588916152e-01
5.007692304925555060e-01 1.074500336043514026e-01 3.860332364294135066e-01
4.955265944792267008e-01 1.066184092659498428e-01 3.853076049214476662e-01
4.902755524910634155e-01 1.058215397414334347e-01 3.845188368245686661e-01
4.850176459901380244e-01 1.050561979453382144e-01 3.836654992000479436e-01
4.797532909040905236e-01 1.043208730187463462e-01 3.827476922012114091e-01
4.744823785889252243e-01 1.036148463575309320e-01 3.817661365680148355e-01
4.692055171112920475e-01 1.029362343090417242e-01 3.807207588669155873e-01
4.639233255037821801e-01 1.022831037225876705e-01 3.796115541526619008e-01
4.586377341242697248e-01 1.016514176559336347e-01 3.784373263419291700e-01
4.533486056644123185e-01 1.010403969450098804e-01 3.771990792167705386e-01
4.480561070398016432e-01 1.004487677966155457e-01 3.758974928385367953e-01
4.427608762820601784e-01 9.987450133495995308e-02 3.745328419993724234e-01
4.374635532679027050e-01 9.931556251859710582e-02 3.731054660236349796e-01
4.321647778868571432e-01 9.876991822512792840e-02 3.716157670899542520e-01
4.268658599998730319e-01 9.823449013050672418e-02 3.700637078536853086e-01
4.215676735765455097e-01 9.770690573986753891e-02 3.684497084664269395e-01
4.162699168504104819e-01 9.718665004270135577e-02 3.667750666255454317e-01
4.109731987208080639e-01 9.667179308527124038e-02 3.650404160810349907e-01
4.056781184785120398e-01 9.616044175324228727e-02 3.632464399813756240e-01
4.003852641200214557e-01 9.565074509543119996e-02 3.613938677771775798e-01
3.950952107257716950e-01 9.514089908082037916e-02 3.594834719896101149e-01
3.898085189133677075e-01 9.462915077538672226e-02 3.575160648711968037e-01
3.845257333758309581e-01 9.411380193352703039e-02 3.554924949866632988e-01
3.792473815136121473e-01 9.359321200643175298e-02 3.534136437413167853e-01
3.739739721678181361e-01 9.306580057655630678e-02 3.512804218838474490e-01
3.687059944607640194e-01 9.253004923332719400e-02 3.490937660095801975e-01
3.634439167485821742e-01 9.198450291035600856e-02 3.468546350890427399e-01
3.581881856892511484e-01 9.142777070873894796e-02 3.445640070453340198e-01
3.529392254280559471e-01 9.085852623447668308e-02 3.422228754021731101e-01
3.476974369011708865e-01 9.027550748073234765e-02 3.398322460227614084e-01
3.424631972568040195e-01 8.967751628755202264e-02 3.373931339577017074e-01
3.372368593921474811e-01 8.906341741289613978e-02 3.349065604182628331e-01
3.320187516032832020e-01 8.843213724940904297e-02 3.323735498892629314e-01
3.268091773441717529e-01 8.778266222136388297e-02 3.297951273938287686e-01
3.216084150899517491e-01 8.711403689571997622e-02 3.271723159202782338e-01
3.164167182989683913e-01 8.642536184028648538e-02 3.245061340194448363e-01
3.112343154672587153e-01 8.571579126067202514e-02 3.217975935788711661e-01
3.060614102686342042e-01 8.498453044606019136e-02 3.190476977785482449e-01
3.008981817730306263e-01 8.423083305194795090e-02 3.162574392312175187e-01
2.957447847354051640e-01 8.345399824589613824e-02 3.134277983087430108e-01
2.906013499472219208e-01 8.265336774002721154e-02 3.105597416546946876e-01
2.854679846423482936e-01 8.182832273165288606e-02 3.076542208820723934e-01
2.803477329331294232e-01 8.097431979192776241e-02 3.047120362305891783e-01
2.752380681858879186e-01 8.009433346301736423e-02 3.017343316299965217e-01
2.701389341759609652e-01 7.918798082745703848e-02 2.987220235093637766e-01
2.650503484302856316e-01 7.825477700181371343e-02 2.956760032243300196e-01
2.599723067663477494e-01 7.729426121855428877e-02 2.925971426566268407e-01
2.549047838884308526e-01 7.630599343423161152e-02 2.894862939132400448e-01
2.498493147092451516e-01 7.528752506190716787e-02 2.863443687052725228e-01
2.448074110283037785e-01 7.423648337865965119e-02 2.831723508316192905e-01
2.397761080979832760e-01 7.315619369923848092e-02 2.799709378403843485e-01
2.347552980534894362e-01 7.204627593897616755e-02 2.767409087277542534e-01
2.297448542911243452e-01 7.090635374932227619e-02 2.734830227534650882e-01
2.247460435545195478e-01 6.973430487024689928e-02 2.701981884636781017e-01
2.197617914493367186e-01 6.852599251980440176e-02 2.668875888727837431e-01
2.147876871428340828e-01 6.728630515821912295e-02 2.635514822299450111e-01
2.098235281541684927e-01 6.601485651366462148e-02 2.601905585020159450e-01
2.048690954628696881e-01 6.471124480576789795e-02 2.568054914489818485e-01
1.999302337340111424e-01 6.336777961956377436e-02 2.533980282310328569e-01
1.950016002461211762e-01 6.199020593164828591e-02 2.499680455203335816e-01
1.900821894571196047e-01 6.057894133574073109e-02 2.465160535124999996e-01
1.851717128353368158e-01 5.913348735199071976e-02 2.430426744218359136e-01];

   case 'alg'
      RGB = [8.429022637670927631e-01 9.769128443086748659e-01 8.146495714674897304e-01
8.379898654100343958e-01 9.732407342044756549e-01 8.088430556678033456e-01
8.330792003033266058e-01 9.695822214541992556e-01 8.030532648620265501e-01
8.281700574948731575e-01 9.659372187629376993e-01 7.972802422761141594e-01
8.232591263824388106e-01 9.623071583180456967e-01 7.915177499867750432e-01
8.183491723383330418e-01 9.586904682961984170e-01 7.857720506350698297e-01
8.134400361234097598e-01 9.550870346415619716e-01 7.800433040617480440e-01
8.085287334688994187e-01 9.514980955891142456e-01 7.743261813916382241e-01
8.036173420648491383e-01 9.479224436104144447e-01 7.686253533397395810e-01
7.987060409170214648e-01 9.443598003220095016e-01 7.629416545120045745e-01
7.937925122347354590e-01 9.408110588423104215e-01 7.572712382873506565e-01
7.888774721308737803e-01 9.372756747279427092e-01 7.516160461408240012e-01
7.839617672758154576e-01 9.337530574201664546e-01 7.459781551153893409e-01
7.790441380861869991e-01 9.302435915337518013e-01 7.403557861728765621e-01
7.741231638352710220e-01 9.267477183327949009e-01 7.347469917710997001e-01
7.692007373711554630e-01 9.232643756857683570e-01 7.291556698393132363e-01
7.642766080322838107e-01 9.197934849334901131e-01 7.235818576277038838e-01
7.593478268473826676e-01 9.163361556599034508e-01 7.180210830197953920e-01
7.544162681443719043e-01 9.128913482866579665e-01 7.124771068936172069e-01
7.494821658995169944e-01 9.094587642780913583e-01 7.069508171108205286e-01
7.445446445348200548e-01 9.060385878865820919e-01 7.014412864903698530e-01
7.396014349798241128e-01 9.026315871837650162e-01 6.959454639490867400e-01
7.346547940498792117e-01 8.992365867115311717e-01 6.904675150024160990e-01
7.297044319156630321e-01 8.958535131435325649e-01 6.850074822882207259e-01
7.247490312042899063e-01 8.924827183233180472e-01 6.795638773235905816e-01
7.197871536503424039e-01 8.891245874731169563e-01 6.741351421211713157e-01
7.148205906315451275e-01 8.857781679670407859e-01 6.687245323963695309e-01
7.098490222489788337e-01 8.824433898084057537e-01 6.633320978710508520e-01
7.048715598638584101e-01 8.791204089748886341e-01 6.579571019276890809e-01
6.998858807424097606e-01 8.758099421807559182e-01 6.525968888447420957e-01
6.948941393454671767e-01 8.725109084606608167e-01 6.472550946180131159e-01
6.898959788510803381e-01 8.692232412884600690e-01 6.419317800561787912e-01
6.848910317118895863e-01 8.659468751996841629e-01 6.366270089015387823e-01
6.798769490247473790e-01 8.626825033431277934e-01 6.313382981547220885e-01
6.748542718513756977e-01 8.594296871845804597e-01 6.260670342676678546e-01
6.698236059410382914e-01 8.561879728828601932e-01 6.208146206652157550e-01
6.647845365512859983e-01 8.529572984927470403e-01 6.155811388293935815e-01
6.597366350463846896e-01 8.497376032066983331e-01 6.103666749315915796e-01
6.546778498073401176e-01 8.465294186026240952e-01 6.051694131498632778e-01
6.496081377794251654e-01 8.433325185775547572e-01 5.999900369339508099e-01
6.445282101635998462e-01 8.401464057748564418e-01 5.948300803069692666e-01
6.394375788791967219e-01 8.369710226136972686e-01 5.896896565661723377e-01
6.343357381511850468e-01 8.338063126244976697e-01 5.845688863102624921e-01
6.292221635729954299e-01 8.306522204479636073e-01 5.794678980553558123e-01
6.240950633957039750e-01 8.275091254279797193e-01 5.743855059995368606e-01
6.189539477625067843e-01 8.243769361676358542e-01 5.693220109772474391e-01
6.137994351057367570e-01 8.212551782044565929e-01 5.642788688985792556e-01
6.086309233573077293e-01 8.181437984076060932e-01 5.592562516207922885e-01
6.034477870757418705e-01 8.150427445292480755e-01 5.542543429870632199e-01
5.982493762730075604e-01 8.119519651518264380e-01 5.492733398149425295e-01
5.930350151926494506e-01 8.088714096235919415e-01 5.443134529660692555e-01
5.878040010377325597e-01 8.058010279811619325e-01 5.393749085029397872e-01
5.825556026472022975e-01 8.027407708577772860e-01 5.344579489389233995e-01
5.772876663840377232e-01 7.996910315164064142e-01 5.295616097386441901e-01
5.720007307801391327e-01 7.966513294788654109e-01 5.246873967172444031e-01
5.666941554135628278e-01 7.936215585273497242e-01 5.198357685218710778e-01
5.613670944828393905e-01 7.906016675911841096e-01 5.150070514423125134e-01
5.560186672658552487e-01 7.875916053777574088e-01 5.102015958622602154e-01
5.506479565076832783e-01 7.845913201632197520e-01 5.054197781680845880e-01
5.452540067541281621e-01 7.816007595558720489e-01 5.006620028015225099e-01
5.398358226320055797e-01 7.786198702294978569e-01 4.959287044657145205e-01
5.343923670778303325e-01 7.756485976236309199e-01 4.912203504944597232e-01
5.289225595174025241e-01 7.726868856074639025e-01 4.865374433949326005e-01
5.234252739997397041e-01 7.697346761037856533e-01 4.818805235744546001e-01
5.178993372898965664e-01 7.667919086690104802e-01 4.772501722621809717e-01
5.123435269266533032e-01 7.638585200249727869e-01 4.726470146367432457e-01
5.067565692524964582e-01 7.609344435378136984e-01 4.680717231709499715e-01
5.011371374253174027e-01 7.580196086388175658e-01 4.635250212045694540e-01
4.954838494233231860e-01 7.551139401816627794e-01 4.590076867559155782e-01
4.897952660572904571e-01 7.522173577300403924e-01 4.545205565824696481e-01
4.840681573071378696e-01 7.493302232220337977e-01 4.500635797773255287e-01
4.783013360124501179e-01 7.464523294656015828e-01 4.456380221845723244e-01
4.724940506024824516e-01 7.435833409938815697e-01 4.412454477696442501e-01
4.666445818616462016e-01 7.407231498818752646e-01 4.368869875723455642e-01
4.607511431720994755e-01 7.378716377692448036e-01 4.325638579641650772e-01
4.548118791551377105e-01 7.350286745174200442e-01 4.282773665493129212e-01
4.488246916697360978e-01 7.321941577075956609e-01 4.240288438198308585e-01
4.427816707495262905e-01 7.293693063882581429e-01 4.198173776667764034e-01
4.366861225292572590e-01 7.265526435148784712e-01 4.156470546985623349e-01
4.305358706593315765e-01 7.237439753686812915e-01 4.115196426107185501e-01
4.243286683688641259e-01 7.209430867475091764e-01 4.074370373866784134e-01
4.180533691692684961e-01 7.181516616727471325e-01 4.033983548745580516e-01
4.117152151566086937e-01 7.153676995586070175e-01 3.994086748894248862e-01
4.053120447326638009e-01 7.125908305255362896e-01 3.954704830581987074e-01
3.988310338161633051e-01 7.098228382481286403e-01 3.915835586978753668e-01
3.922773504125326993e-01 7.070616628058336017e-01 3.877531715317056316e-01
3.856453036392904488e-01 7.043074656954597668e-01 3.839816779215288745e-01
3.789264219900637110e-01 7.015608428874934299e-01 3.802713204645660205e-01
3.721249180146385394e-01 6.988198403254852753e-01 3.766273561291236249e-01
3.652256292309092323e-01 6.960861432527269965e-01 3.730515342232093579e-01
3.582353316782660446e-01 6.933572072574926137e-01 3.695499404331015203e-01
3.511405088327756996e-01 6.906341612001636321e-01 3.661256768699447939e-01
3.439420657842418572e-01 6.879154082481935273e-01 3.627843734908121065e-01
3.366376245587633931e-01 6.851998988322398620e-01 3.595313212824273741e-01
3.292131284244469436e-01 6.824884139177085363e-01 3.563714602024023459e-01
3.216744583078648412e-01 6.797783170101999728e-01 3.533114882245381727e-01
3.140172671350762723e-01 6.770686201449667152e-01 3.503575531959788325e-01
3.062375560076174841e-01 6.743581777161754554e-01 3.475162179105460436e-01
2.983302406892556213e-01 6.716458949569478198e-01 3.447945234485624288e-01
2.903003733928296581e-01 6.689291557488895590e-01 3.421993121220623379e-01
2.821497842888552321e-01 6.662058429849556651e-01 3.397373505300674279e-01
2.738829008222237738e-01 6.634735172067438569e-01 3.374150438308454736e-01
2.655071740281889081e-01 6.607294316207971141e-01 3.352380978850171278e-01
2.570334211300853156e-01 6.579705722659228151e-01 3.332111341597794874e-01
2.484760249031935930e-01 6.551937268253881230e-01 3.313372827145077970e-01
2.398529309513116603e-01 6.523955825377709683e-01 3.296177905380952566e-01
2.311841779376898609e-01 6.495729702230954583e-01 3.280520259104468539e-01
2.224959725553208312e-01 6.467225401811796948e-01 3.266360450956267703e-01
2.138146081063950210e-01 6.438414682793780486e-01 3.253637783252084081e-01
2.051671661243615885e-01 6.409273967897467505e-01 3.242268145630948784e-01
1.965805108315318850e-01 6.379785280095539024e-01 3.232146947371194456e-01
1.880802465071397533e-01 6.349936851788878789e-01 3.223153652533572999e-01
1.796913740541281890e-01 6.319722298814686168e-01 3.215149361924393712e-01
1.714330705005376321e-01 6.289143429606669500e-01 3.208003886075819211e-01
1.633223574014183166e-01 6.258206603139722102e-01 3.201583938437198018e-01
1.553730387230779220e-01 6.226922583196569105e-01 3.195761743639152774e-01
1.475958738836526396e-01 6.195305528827301789e-01 3.190418452926268023e-01
1.399989467346693939e-01 6.163371978378411331e-01 3.185446364378627382e-01
1.325881576814804952e-01 6.131139916341550311e-01 3.180750084472180883e-01
1.253677831749437921e-01 6.098627973272553460e-01 3.176246802424351201e-01
1.183415125249481503e-01 6.065854884488630638e-01 3.171858497741395500e-01
1.115126074344926499e-01 6.032839586583695901e-01 3.167505890039074568e-01
1.048819394515066727e-01 5.999599650973242992e-01 3.163157773881625778e-01
9.845229630728663528e-02 5.966151576836506987e-01 3.158774289651596345e-01
9.222723505152297108e-02 5.932511684885474201e-01 3.154305788787992171e-01
8.620976681204781111e-02 5.898696713297170158e-01 3.149695047280966498e-01
8.040588003861268152e-02 5.864717566016173222e-01 3.144963594445834287e-01
7.481986998062370442e-02 5.830590393336594346e-01 3.140045327658092522e-01
6.946029534260433902e-02 5.796325615836696032e-01 3.134943864901328370e-01
6.433627993915527754e-02 5.761934589668912254e-01 3.129640951373787172e-01
5.945721644433194647e-02 5.727428791151661924e-01 3.124113492075187293e-01
5.483676528890226581e-02 5.692816942618841303e-01 3.118359557718813901e-01
5.049036852106828649e-02 5.658107385911853582e-01 3.112374314935371089e-01
4.642777154917742538e-02 5.623311839573432724e-01 3.106124662721929663e-01
4.267873894906259319e-02 5.588431316231524670e-01 3.099653177208470112e-01
3.923368215696795835e-02 5.553481890828847467e-01 3.092892996314257625e-01
3.621581017331615415e-02 5.518464055245291267e-01 3.085882846661225365e-01
3.367174260746116921e-02 5.483382191382968340e-01 3.078629031746034084e-01
3.155357134880961562e-02 5.448249609075901390e-01 3.071084590117025281e-01
2.984163194432063598e-02 5.413067537046710731e-01 3.063273483583937029e-01
2.851433193286296089e-02 5.377838583018560437e-01 3.055207233916585885e-01
2.754542481457317171e-02 5.342567759013212569e-01 3.046882377538095987e-01
2.689570440508957244e-02 5.307265908415563782e-01 3.038266898313282116e-01
2.655111347348209846e-02 5.271933424208224972e-01 3.029380668195990611e-01
2.649479114279441189e-02 5.236572235511334217e-01 3.020233215230865298e-01
2.670582470265941977e-02 5.201186270434556835e-01 3.010823500978198064e-01
2.716440603849906016e-02 5.165779238412415708e-01 3.001150958048797168e-01
2.785177445187870041e-02 5.130354641701924123e-01 2.991215440838268513e-01
2.874507672231698188e-02 5.094917938725799234e-01 2.981008701035369746e-01
2.982666070550293522e-02 5.059472607886396078e-01 2.970530677865693692e-01
3.108771284944302979e-02 5.024018586614166226e-01 2.959794192966550552e-01
3.251322799624088711e-02 4.988558619272636663e-01 2.948800270305382831e-01
3.408899261646199802e-02 4.953095294092992318e-01 2.937550173740440806e-01
3.580154506882658044e-02 4.917631050765793321e-01 2.926045381193648209e-01
3.763813815230969417e-02 4.882168187516792712e-01 2.914287561365453416e-01
3.958670383768114059e-02 4.846708867690439626e-01 2.902278552740369943e-01
4.160606223970295808e-02 4.811255125865287474e-01 2.890020344652265427e-01
4.364369199371629510e-02 4.775808873526497522e-01 2.877515060197786689e-01
4.569661527765144643e-02 4.740371904320783147e-01 2.864764940804359616e-01
4.775452753988133209e-02 4.704945898919368763e-01 2.851772332276043542e-01
4.980869291606532939e-02 4.669532429514431926e-01 2.838539672156359384e-01
5.185173623572528895e-02 4.634132963974310626e-01 2.825069478261687528e-01
5.387745614388812082e-02 4.598748869682287022e-01 2.811364338252513306e-01
5.588065991543698235e-02 4.563381417083428038e-01 2.797426900122344517e-01
5.785701915204972262e-02 4.528031782963246044e-01 2.783259863495758379e-01
5.979876706363549538e-02 4.492703547749640758e-01 2.768859613477449022e-01
6.170040530833437870e-02 4.457399494079358759e-01 2.754224910913550817e-01
6.356556527046725025e-02 4.422117069661303246e-01 2.739367706711952621e-01
6.539233256679152784e-02 4.386857116546215574e-01 2.724290903132413622e-01
6.717918154984042767e-02 4.351620395957427334e-01 2.708997417620619963e-01
6.892491848697404611e-02 4.316407590421982854e-01 2.693490177429860299e-01
7.062863315353248850e-02 4.281219305768591554e-01 2.677772114552923188e-01
7.228965757688332605e-02 4.246056073010452958e-01 2.661846160919238424e-01
7.389641781918890318e-02 4.210926586635973523e-01 2.645697649893180015e-01
7.545434928417493747e-02 4.175827498520321979e-01 2.629338431200398118e-01
7.696833442757364252e-02 4.140755522595082616e-01 2.612779353323396725e-01
7.843838726966873010e-02 4.105710916270851607e-01 2.596023362544704338e-01
7.986463095393955824e-02 4.070693869948522892e-01 2.579073383225120586e-01
8.124183059024486786e-02 4.035709045576263421e-01 2.561923603053372633e-01
8.256190184629638718e-02 4.000763929725326684e-01 2.544563479235497083e-01
8.383876927413885793e-02 3.965847495529768452e-01 2.527017537470714892e-01
8.507286345646175585e-02 3.930959685837552287e-01 2.509288633450958983e-01
8.626466273198904466e-02 3.896100382985579480e-01 2.491379583258400143e-01
8.739775523971837767e-02 3.861285039195596069e-01 2.473266151547285352e-01
8.848513420336723279e-02 3.826502629618194207e-01 2.454971009362341139e-01
8.953167205157094855e-02 3.791748988832845391e-01 2.436503732426347768e-01
9.053797836732038751e-02 3.757023773832682267e-01 2.417867000353674523e-01
9.148554097617969671e-02 3.722345634210443843e-01 2.399033173203807268e-01
9.239032630846283345e-02 3.687699624551818989e-01 2.380029261610703828e-01
9.325665476520708652e-02 3.653081665793141419e-01 2.360863630636709232e-01
9.407947178296760526e-02 3.618497205076261491e-01 2.341529827160940824e-01
9.484677589858042657e-02 3.583959460277540421e-01 2.322010660237076030e-01
9.557752330564703303e-02 3.549448939617083076e-01 2.302337247418529409e-01
9.627239441888107985e-02 3.514964944168565975e-01 2.282511932335974936e-01
9.690948221035081134e-02 3.480532009152630946e-01 2.262502344543431132e-01
9.750943240499498899e-02 3.446127832007709890e-01 2.242341737774653510e-01
9.807545896746669434e-02 3.411748780303069384e-01 2.222036193428468254e-01
9.858840445742730885e-02 3.377417205454144589e-01 2.201557955312036796e-01
9.906254148928930747e-02 3.343117038364873950e-01 2.180929998845686502e-01
9.950470237088998582e-02 3.308840188006390015e-01 2.160163648855405460e-01
9.989442828501457483e-02 3.274611474658599142e-01 2.139229716850828411e-01
1.002478051840671458e-01 3.240411780581063939e-01 2.118153315535568071e-01
1.005711171338506127e-01 3.206233202298787721e-01 2.096944585359068469e-01
1.008389569492894877e-01 3.172107795653626439e-01 2.075567848977897256e-01
1.010766013830992904e-01 3.138003866895044958e-01 2.054060788674454963e-01
1.012797610659880443e-01 3.103926683916618523e-01 2.032418103374508123e-01
1.014336128495174072e-01 3.069895754355959072e-01 2.010620024754192769e-01
1.015604574719611575e-01 3.035881742672912331e-01 1.988698806638831140e-01
1.016439043068387282e-01 3.001906378105418383e-01 1.966632618336682237e-01
1.016900146818882356e-01 2.967961587009928515e-01 1.944430867217927517e-01
1.017093270307037889e-01 2.934032641989809398e-01 1.922108533595217328e-01
1.016765444870794677e-01 2.900154576395235773e-01 1.899632762562097343e-01
1.016196069311545991e-01 2.866288210415605109e-01 1.877041478451552947e-01
1.015219485974463154e-01 2.832456655164396486e-01 1.854313688320031739e-01
1.013890307775781596e-01 2.798651977505757227e-01 1.831457349733398854e-01
1.012281681325793714e-01 2.764863132581988348e-01 1.808482324484150805e-01
1.010210261960676481e-01 2.731116796865848406e-01 1.785366459436490416e-01
1.007923256501605802e-01 2.697375956289435606e-01 1.762140933897481665e-01
1.005175174628768486e-01 2.663676978648497062e-01 1.738776494133498218e-01
1.002173006867299876e-01 2.629988175863644528e-01 1.715298624270491512e-01
9.988042668340707531e-02 2.596326158562400344e-01 1.691694475105532114e-01
9.951032875855422843e-02 2.562685171837931764e-01 1.667968626726523129e-01
9.911165965335702599e-02 2.529057275734052368e-01 1.644126697900448464e-01
9.867326641808749077e-02 2.495459516150641543e-01 1.620156747879235859e-01
9.821309319240137392e-02 2.461862546952748865e-01 1.596078670948603284e-01
9.770793688346479655e-02 2.428303047023268602e-01 1.571867977376794623e-01
9.718522833757223256e-02 2.394735637952865592e-01 1.547553696066038820e-01
9.661611889067361902e-02 2.361206836681891130e-01 1.523106424045232721e-01
9.602890382911671852e-02 2.327668939738092857e-01 1.498554790261657776e-01
9.539953637497836092e-02 2.294161153913850115e-01 1.473875249443533098e-01
9.474917991081285851e-02 2.260646807246796119e-01 1.449088125658586357e-01
9.405984876891312907e-02 2.227155420665202223e-01 1.424176598027472040e-01
9.334764966575742617e-02 2.193658224428529091e-01 1.399155213806154985e-01
9.259864172169812724e-02 2.160178161752643877e-01 1.374011522998845325e-01
9.182582886012352619e-02 2.126691252719348779e-01 1.348756437215534176e-01
9.101741805439966804e-02 2.093216946605545581e-01 1.323379906767623293e-01
9.018514688041953664e-02 2.059732968223858540e-01 1.297890966546490499e-01
8.931758876675888192e-02 2.026258321763433345e-01 1.272280374767779110e-01
8.842693764273418244e-02 1.992769388914110706e-01 1.246556670387520271e-01
8.750046390510707317e-02 1.959287732623697376e-01 1.220710201130164874e-01
8.655243027600306727e-02 1.925785390204607095e-01 1.194750015988692016e-01
8.556724308545121671e-02 1.892289432653286863e-01 1.168665204409958802e-01
8.456273936118519075e-02 1.858764606951441301e-01 1.142465958957942507e-01
8.351900544036733320e-02 1.825246377933723574e-01 1.116139631166048474e-01
8.245885447713624528e-02 1.791689319540810954e-01 1.089697819472404405e-01
8.135669872091313981e-02 1.758140104483569555e-01 1.063126024658284652e-01
8.023864908973366017e-02 1.724546122292074379e-01 1.036435837789304870e-01
7.908112723260085630e-02 1.690950585274078044e-01 1.009615075211427460e-01
7.790439520871850210e-02 1.657311325223233545e-01 9.826708487284271931e-02
7.669293821383046939e-02 1.623656063191788734e-01 9.555954478117026363e-02
7.545752224924831553e-02 1.589960594221086487e-01 9.283909520333205601e-02
7.419260617097681032e-02 1.556232855365611012e-01 9.010535811327979872e-02
7.289843379575938753e-02 1.522469176001574609e-01 8.735814807810177163e-02
7.158041455970551303e-02 1.488655123209624287e-01 8.459734502497687214e-02
7.022733114464504989e-02 1.454810031009998728e-01 8.182251708486179553e-02
6.885643403782271132e-02 1.420894601159049808e-01 7.903362825094448207e-02]; 

   case 'den' 
      RGB = [9.022021640633741679e-01 9.441797977915000750e-01 9.438027309131502562e-01
8.954445384876381642e-01 9.409578903665054561e-01 9.410648759794842944e-01
8.886855788874895579e-01 9.377403820634679921e-01 9.384098650760986926e-01
8.819275100483495722e-01 9.345262546442011375e-01 9.358372814047579702e-01
8.751724774625697645e-01 9.313145521613219735e-01 9.333465385891809296e-01
8.684225465562396273e-01 9.281043788756372370e-01 9.309368983040341439e-01
8.616797015669134252e-01 9.248948973050555855e-01 9.286074874996731454e-01
8.549458444869071361e-01 9.216853262599870034e-01 9.263573151615366319e-01
8.482227943572994144e-01 9.184749387758168737e-01 9.241852885114790750e-01
8.415122870990738857e-01 9.152630598938195083e-01 9.220902285443438595e-01
8.348159759916423672e-01 9.120490642716416740e-01 9.200708847923535494e-01
8.281354328507813944e-01 9.088323736256948004e-01 9.181259492188009741e-01
8.214721499137223049e-01 9.056124540224106401e-01 9.162540691578000551e-01
8.148275424063925465e-01 9.023888130447765832e-01 9.144538592359539031e-01
8.082029517444304645e-01 8.991609968660188024e-01 9.127239122325879750e-01
8.015996493038702875e-01 8.959285872646755022e-01 9.110628088561002480e-01
7.950188742001107478e-01 8.926911903474541443e-01 9.094690972400388818e-01
7.884617893745216044e-01 8.894484461080551796e-01 9.079413399350437786e-01
7.819294709222469608e-01 8.862000288332800846e-01 9.064781367807944745e-01
7.754229463217646723e-01 8.829456358148736195e-01 9.050781031537078469e-01
7.689431984424601740e-01 8.796849849240929720e-01 9.037398757679527828e-01
7.624911712284807574e-01 8.764178119047121296e-01 9.024621162700895427e-01
7.560677754202349554e-01 8.731438677954835859e-01 9.012435140344750018e-01
7.496738942657009686e-01 8.698629164913183054e-01 9.000827882281617898e-01
7.433103891809360597e-01 8.665747324492172332e-01 8.989786892140232410e-01
7.369781053261772463e-01 8.632790985421132657e-01 8.979299993594229701e-01
7.306778770702586634e-01 8.599758040613866283e-01 8.969355333152028154e-01
7.244105333220340892e-01 8.566646428667109570e-01 8.959941378263615031e-01
7.181769027125698424e-01 8.533454116802825506e-01 8.951046911317568355e-01
7.119778186163850942e-01 8.500179085211370111e-01 8.942661020057923738e-01
7.058142285268640403e-01 8.466819129479339345e-01 8.934771781429077242e-01
6.996868754737947116e-01 8.433372425635760061e-01 8.927370216147950677e-01
6.935966262281186845e-01 8.399836925869731408e-01 8.920446369253884900e-01
6.875443783072674453e-01 8.366210531289836050e-01 8.913990435590287698e-01
6.815310569542375463e-01 8.332491095447560614e-01 8.907992825434407624e-01
6.755576190661271019e-01 8.298676415986731003e-01 8.902444142113377090e-01
6.696250569512002260e-01 8.264764227239829175e-01 8.897335158509319664e-01
6.637344019158556430e-01 8.230752193711802223e-01 8.892656792634947571e-01
6.578867276821276366e-01 8.196637904396377738e-01 8.888400082431817673e-01
6.520831536351689994e-01 8.162418867874922102e-01 8.884556159917510465e-01
6.463248478988037338e-01 8.128092508153051954e-01 8.881116224784991742e-01
6.406130302355731443e-01 8.093656161195516008e-01 8.878071517537653445e-01
6.349489747657169891e-01 8.059107072125802906e-01 8.875413292226665973e-01
6.293340124974293737e-01 8.024442393062223289e-01 8.873132788842952312e-01
6.237695336583948258e-01 7.989659181568090629e-01 8.871221205404720145e-01
6.182569898160699129e-01 7.954754399699002221e-01 8.869669669772310971e-01
6.127978529026838483e-01 7.919724939438572697e-01 8.868470036964815062e-01
6.073937143892588209e-01 7.884567552174953642e-01 8.867613059311603152e-01
6.020462362204239692e-01 7.849278896215748924e-01 8.867089225926839680e-01
5.967571335484213035e-01 7.813855546115021644e-01 8.866889106706612456e-01
5.915281898823177009e-01 7.778293984625483937e-01 8.867003055015370006e-01
5.863612575196054388e-01 7.742590605302298590e-01 8.867421178165009188e-01
5.812582575983973321e-01 7.706741715732086107e-01 8.868133307743576443e-01
5.762211797340099917e-01 7.670743541409147381e-01 8.869128969829140896e-01
5.712520812007981785e-01 7.634592230284988901e-01 8.870397355132993988e-01
5.663530856178774497e-01 7.598283858019500014e-01 8.871927289126763094e-01
5.615263810949739920e-01 7.561814433965411419e-01 8.873707202219829338e-01
5.567742177930775638e-01 7.525179907919229416e-01 8.875725100066919060e-01
5.520989048532665144e-01 7.488376177673574663e-01 8.877968534100384446e-01
5.475028066465008614e-01 7.451399097406838923e-01 8.880424572398104566e-01
5.429883038774732107e-01 7.414244411965056347e-01 8.883081102246548344e-01
5.385578037806126872e-01 7.376907727350942023e-01 8.885926739767254778e-01
5.342139074650678054e-01 7.339384990781333551e-01 8.888943435724941944e-01
5.299591516928192636e-01 7.301671976158840005e-01 8.892116013883853975e-01
5.257960979209626018e-01 7.263764470411206986e-01 8.895428673545512366e-01
5.217273234975481344e-01 7.225658287807643632e-01 8.898864965652431014e-01
5.177554119038457747e-01 7.187349285333118898e-01 8.902407770595432979e-01
5.138829420298184347e-01 7.148833379130644650e-01 8.906039278005388748e-01
5.101124764807559719e-01 7.110106562012711295e-01 8.909740968830553998e-01
5.064465489258500597e-01 7.071164922032951994e-01 8.913493600019622987e-01
5.028876505140544850e-01 7.032004662097455228e-01 8.917277192147863296e-01
4.994382152582386714e-01 6.992621674331673809e-01 8.921074350832908229e-01
4.961006347215984325e-01 6.953012224940204877e-01 8.924864661543192579e-01
4.928771955532800786e-01 6.913173542206699773e-01 8.928621485317782547e-01
4.897700668589555772e-01 6.873102514205022828e-01 8.932321753875414050e-01
4.867812943786045676e-01 6.832796259123631311e-01 8.935941653626673364e-01
4.839127827295982009e-01 6.792252150037093594e-01 8.939456639464463672e-01
4.811662774244846452e-01 6.751467840099739659e-01 8.942841454160184167e-01
4.785433468471342322e-01 6.710441287994942661e-01 8.946070153685665716e-01
4.760453643890875663e-01 6.669170783452496032e-01 8.949116138746746607e-01
4.736736428294658352e-01 6.627653768113491717e-01 8.951957859716279664e-01
4.714290262412598742e-01 6.585890168305761350e-01 8.954562723025643045e-01
4.693121574935592011e-01 6.543879757664989860e-01 8.956900818434682110e-01
4.673234528945732769e-01 6.501622438680934035e-01 8.958943277799077398e-01
4.654630534005658737e-01 6.459118568983728270e-01 8.960660842179661856e-01
4.637308126788352580e-01 6.416368981971656282e-01 8.962023935259622043e-01
4.621262870085202645e-01 6.373375005637914592e-01 8.963002743257548754e-01
4.606487857322991153e-01 6.330138155630400387e-01 8.963568414436460241e-01
4.592973162762510886e-01 6.286660463835676005e-01 8.963691878774373567e-01
4.580703339287382492e-01 6.242945769689846047e-01 8.963339924500021150e-01
4.569661322748618804e-01 6.198997574323485971e-01 8.962482674873512023e-01
4.559826907629682125e-01 6.154819930493959923e-01 8.961090514006179175e-01
4.551176793504218554e-01 6.110417444047528956e-01 8.959134196000665407e-01
4.543684658154031886e-01 6.065795271800727972e-01 8.956584955889488331e-01
4.537321103072295414e-01 6.020959187639063348e-01 8.953414439077735931e-01
4.532053056792195167e-01 5.975915899334859338e-01 8.949594061016020730e-01
4.527845804849874312e-01 5.930672153918072897e-01 8.945097368602321630e-01
4.524662055084718859e-01 5.885235252040261766e-01 8.939898705469653262e-01
4.522462243009037208e-01 5.839612972830612314e-01 8.933973456284665104e-01
4.521204742039108271e-01 5.793813548677185787e-01 8.927298144348156939e-01
4.520846088132207119e-01 5.747845636677534342e-01 8.919850523005500298e-01
4.521340640278729839e-01 5.701718535137081378e-01 8.911609204213972735e-01
4.522636556547669495e-01 5.655443967339887079e-01 8.902550664458118712e-01
4.524690821672368579e-01 5.609029696513448959e-01 8.892660746141660688e-01
4.527455061379137002e-01 5.562485728577384325e-01 8.881923021329055645e-01
4.530880320659124716e-01 5.515822303923638703e-01 8.870322627966300555e-01
4.534917310405354729e-01 5.469049855617650335e-01 8.857846304506885593e-01
4.539516647179116515e-01 5.422178966900855768e-01 8.844482413776627583e-01
4.544629083695410077e-01 5.375220328430567740e-01 8.830220956021650469e-01
4.550204443687518308e-01 5.328185230036192044e-01 8.815052933071011454e-01
4.556185611092698484e-01 5.281088097066267695e-01 8.798967638851667994e-01
4.562534204266542326e-01 5.233935859930700651e-01 8.781964868264527935e-01
4.569203709084596610e-01 5.186739169401044514e-01 8.764041195297782583e-01
4.576148706477833339e-01 5.139508576774439730e-01 8.745194730542404926e-01
4.583325010790182952e-01 5.092254498718804534e-01 8.725425074817807491e-01
4.590689790493137079e-01 5.044987184411230396e-01 8.704733265502938577e-01
4.598201671313326133e-01 4.997716685175846441e-01 8.683121716406742019e-01
4.605820822069772169e-01 4.950452826784434435e-01 8.660594152048119998e-01
4.613509023726869995e-01 4.903205184540396222e-01 8.637155537234258995e-01
4.621229722350553293e-01 4.855983061224402042e-01 8.612812002829223212e-01
4.628948066804312034e-01 4.808795467939734336e-01 8.587570768592888149e-01
4.636621617804746465e-01 4.761655004408036906e-01 8.561438461496413410e-01
4.644228483348319947e-01 4.714566096078480206e-01 8.534426324219575033e-01
4.651739386269664878e-01 4.667536635703780079e-01 8.506544391441875907e-01
4.659126485884828583e-01 4.620574294334566789e-01 8.477803402103136765e-01
4.666363628095652749e-01 4.573686402108441684e-01 8.448214774526084936e-01
4.673426313321316639e-01 4.526879945693308982e-01 8.417790529251527598e-01
4.680291657078233247e-01 4.480161567955825430e-01 8.386543213914651185e-01
4.686938344285310198e-01 4.433537569692765912e-01 8.354485830591045215e-01
4.693346578320167217e-01 4.387013913255956021e-01 8.321631765974399908e-01
4.699498025791529199e-01 4.340596227897761117e-01 8.287994724682681280e-01
4.705375757926853475e-01 4.294289816663295900e-01 8.253588665927664714e-01
4.710964189403291091e-01 4.248099664657251084e-01 8.218427743723751844e-01
4.716249015377835252e-01 4.202030448517347638e-01 8.182526250758133113e-01
4.721217147398471536e-01 4.156086546932221681e-01 8.145898565994108553e-01
4.725856648806062155e-01 4.110272052049120939e-01 8.108559106034937125e-01
4.730156670165072685e-01 4.064590781625461724e-01 8.070522280235292722e-01
4.734107385193607187e-01 4.019046291787961578e-01 8.031802449512829289e-01
4.737699927598160721e-01 3.973641890273297284e-01 7.992413888782301523e-01
4.740926329157582608e-01 3.928380650035104282e-01 7.952370752909331264e-01
4.743779459344564242e-01 3.883265423112774450e-01 7.911687046060347228e-01
4.746252966720311828e-01 3.838298854668622528e-01 7.870376594308572393e-01
4.748341222291213026e-01 3.793483397110574140e-01 7.828453021343189100e-01
4.750039264972926722e-01 3.748821324227855634e-01 7.785929727119723642e-01
4.751342749269319987e-01 3.704314745277388354e-01 7.742819869283568135e-01
4.752247895239137265e-01 3.659965618967846446e-01 7.699136347195338903e-01
4.752751440794077964e-01 3.615775767297375043e-01 7.654891788385849161e-01
4.752846577346016566e-01 3.571748688576533159e-01 7.610100397104845316e-01
4.752531583445195884e-01 3.527885695695618384e-01 7.564774234071200976e-01
4.751807975405666906e-01 3.484186721552497978e-01 7.518923528283415481e-01
4.750674206653115461e-01 3.440653140404167365e-01 7.472559685686853692e-01
4.749129042193905303e-01 3.397286256677667371e-01 7.425693796885003417e-01
4.747171524182526858e-01 3.354087317676910929e-01 7.378336635067138660e-01
4.744800939620510971e-01 3.311057526021228270e-01 7.330498655303070432e-01
4.742016790105380575e-01 3.268198051821941674e-01 7.282189995072744226e-01
4.738818763543148349e-01 3.225510044607445836e-01 7.233420475907058611e-01
4.735206707734627707e-01 3.182994645010271961e-01 7.184199606025195584e-01
4.731180605743212086e-01 3.140652996233325722e-01 7.134536583862346459e-01
4.726740552951489982e-01 3.098486255314443216e-01 7.084440302390293542e-01
4.721886735713431427e-01 3.056495604211318384e-01 7.033919354141809910e-01
4.716613727297020442e-01 3.014684828248803128e-01 6.982986813805610593e-01
4.710926963645384880e-01 2.973052870266812420e-01 6.931646771793222861e-01
4.704828136289659901e-01 2.931600441838846383e-01 6.879905802658634606e-01
4.698317639381846544e-01 2.890328918495519428e-01 6.827771264871377310e-01
4.691395866583317198e-01 2.849239757466777712e-01 6.775250248474340431e-01
4.684063197381936328e-01 2.808334508365700755e-01 6.722349583426855402e-01
4.676319984415455155e-01 2.767614823885942155e-01 6.669075848245407112e-01
4.668166541724163010e-01 2.727082470546194348e-01 6.615435378909206854e-01
4.659603133859185342e-01 2.686739339515746283e-01 6.561434278004181220e-01
4.650628643816827057e-01 2.646588038897796924e-01 6.507079958832593380e-01
4.641242580497152992e-01 2.606631006122758221e-01 6.452378997676680994e-01
4.631447877428095938e-01 2.566869305701654502e-01 6.397333527350717031e-01
4.621244494694312643e-01 2.527305449360881529e-01 6.341948722101476976e-01
4.610632293393826520e-01 2.487942133948352064e-01 6.286229575018943416e-01
4.599611028558598380e-01 2.448782253548376642e-01 6.230180909006084455e-01
4.588180342503813125e-01 2.409828911918876493e-01 6.173807387973655469e-01
4.576339758554896497e-01 2.371085435285009702e-01 6.117113528277585699e-01
4.564088675103588622e-01 2.332555385522245284e-01 6.060103710422047874e-01
4.551426114844220883e-01 2.294242672454404608e-01 6.002782575500510420e-01
4.538352109035017068e-01 2.256151009415963693e-01 5.945152849064858636e-01
4.524865766439374881e-01 2.218284716492017750e-01 5.887218272221994564e-01
4.510965881023942248e-01 2.180648477346619485e-01 5.828982757017970862e-01
4.496651090864017264e-01 2.143247287775243981e-01 5.770450152504881247e-01
4.481919873439485502e-01 2.106086471651167358e-01 5.711624259770210488e-01
4.466770541000937844e-01 2.069171697362102713e-01 5.652508847626541710e-01
4.451201235978615167e-01 2.032508994724643858e-01 5.593107669047254760e-01
4.435210649059648236e-01 1.996104535682982628e-01 5.533422989220041499e-01
4.418797550887742509e-01 1.959964848006353622e-01 5.473456323671032075e-01
4.401958303868417355e-01 1.924097603751145913e-01 5.413213827196030614e-01
4.384690252866834115e-01 1.888510510844303159e-01 5.352699392743940354e-01
4.366990547337928352e-01 1.853211730457038908e-01 5.291917009058230148e-01
4.348856137062026561e-01 1.818209895007495414e-01 5.230870785018020275e-01
4.330283767917184612e-01 1.783514125901697334e-01 5.169564975681201213e-01
4.311269977716754020e-01 1.749134050778043403e-01 5.108004010215621005e-01
4.291811092153304252e-01 1.715079819968361174e-01 5.046192521919092844e-01
4.271904344659923636e-01 1.681361967924985268e-01 4.984132106571174670e-01
4.251546522673581574e-01 1.647991755952243342e-01 4.921824740309181379e-01
4.230731286286127379e-01 1.614981359784699588e-01 4.859281380753826540e-01
4.209454004136206628e-01 1.582343200442791198e-01 4.796507952549767251e-01
4.187709818950797191e-01 1.550090272321758555e-01 4.733510791258234707e-01
4.165493646911141434e-01 1.518236143519533232e-01 4.670296690947607909e-01
4.142800178080069395e-01 1.486794951032471834e-01 4.606872955350938548e-01
4.119623878145514673e-01 1.455781389789607694e-01 4.543247452773370720e-01
4.095958991778662628e-01 1.425210694369028197e-01 4.479428674896456797e-01
4.071799547953330878e-01 1.395098612110991787e-01 4.415425799574690391e-01
4.047139367624922324e-01 1.365461366216895456e-01 4.351248757648299992e-01
4.021972936019284628e-01 1.336316105494000928e-01 4.286903779176353124e-01
3.996293630849273582e-01 1.307680150350410408e-01 4.222401836338272596e-01
3.970093606178293211e-01 1.279570274853844003e-01 4.157761120045768699e-01
3.943365976380542870e-01 1.252003942696350014e-01 4.092995513119635498e-01
3.916103749033576498e-01 1.224998688123591362e-01 4.028120041341317625e-01
3.888299858840064682e-01 1.198571984110099742e-01 3.963150954999720699e-01
3.859947208059889556e-01 1.172741089547249538e-01 3.898105809973525515e-01
3.831038714159066827e-01 1.147522874417901684e-01 3.833003546699703667e-01
3.801567365353383798e-01 1.122933622374599760e-01 3.767864564999491295e-01
3.771526284658990869e-01 1.098988810729940035e-01 3.702710792321893263e-01
3.740908802950018708e-01 1.075702868621947472e-01 3.637565742534386581e-01
3.709708541354928557e-01 1.053088915040240603e-01 3.572454561957943420e-01
3.677919503089708275e-01 1.031158479486348478e-01 3.507404058933542013e-01
3.645536174519507511e-01 1.009921209275927434e-01 3.442442712851704334e-01
3.612553634855836804e-01 9.893845688361260771e-02 3.377600658314386384e-01
3.578967673433453012e-01 9.695535377529579391e-02 3.312909639976103771e-01
3.544774912975176551e-01 9.504303157033086591e-02 3.248402933678479765e-01
3.509972936657820841e-01 9.320140436662560646e-02 3.184115229802063629e-01
3.474560416161082133e-01 9.143005518316707492e-02 3.120082475359999274e-01
3.438537237247893397e-01 8.972821452832538403e-02 3.056341672288407363e-01
3.401904735486728781e-01 8.809468358171337887e-02 2.992931815755145442e-01
3.364665724714709372e-01 8.652789261304066892e-02 2.929892038941975252e-01
3.326824502647063309e-01 8.502592614588003195e-02 2.867260839053818455e-01
3.288387024976534012e-01 8.358647051086640078e-02 2.805076847277695462e-01
3.249360934730275985e-01 8.220683878873383255e-02 2.743378151941179288e-01
3.209755584101655623e-01 8.088399100741089365e-02 2.682201871550352612e-01
3.169582022236226426e-01 7.961456284429593855e-02 2.621583713198606391e-01
3.128852946012805614e-01 7.839490235071094881e-02 2.561557528820557206e-01
3.087582612248004899e-01 7.722111381434071387e-02 2.502154883125917717e-01
3.045786711401687885e-01 7.608910749927882966e-02 2.443404647714875755e-01
3.003482204670568367e-01 7.499465368295166190e-02 2.385332635717280492e-01
2.960687128198414286e-01 7.393343917254721620e-02 2.327961290245241788e-01
2.917420369856660312e-01 7.290112435224016529e-02 2.271309438018545013e-01
2.873701425511966390e-01 7.189339879884681928e-02 2.215392116816457535e-01
2.829550142767416343e-01 7.090603360902006380e-02 2.160220482113159313e-01
2.784999277173711985e-01 6.993210653137302280e-02 2.105832735994622174e-01
2.740068130913843047e-01 6.896804327168698512e-02 2.052225548327365479e-01
2.694770905593398824e-01 6.801140076763184661e-02 1.999383122503714527e-01
2.649127155611373241e-01 6.705861363954487842e-02 1.947301490341072916e-01
2.603155555465970217e-01 6.610638194498252851e-02 1.895973275559961757e-01
2.556873713350471533e-01 6.515168315524647036e-02 1.845388003264406274e-01
2.510311382384792789e-01 6.418928564209233634e-02 1.795555007506528800e-01
2.463515264850693609e-01 6.321115386076697762e-02 1.746505568409140452e-01
2.416465428487372114e-01 6.222162722501616700e-02 1.698160042316500806e-01
2.369174512696113344e-01 6.121879219616855466e-02 1.650498029374760922e-01
2.321653556071113234e-01 6.020099184329732317e-02 1.603497740495725687e-01
2.273975644358762205e-01 5.915624173942467950e-02 1.557218456563352615e-01
2.226121431237116921e-01 5.808842465569693386e-02 1.511590815358170026e-01
2.178074381899826051e-01 5.700049790743180050e-02 1.466558187034970040e-01
2.129839422000848193e-01 5.589168645699472276e-02 1.422095068240733784e-01]; 

   case 'bal' 
      RGB = [9.317630180115785143e-02 1.111733294776027225e-01 2.615123885530547532e-01
9.697151501690241815e-02 1.168702109792841837e-01 2.730963071061036085e-01
1.009688451686782534e-01 1.223931506799195018e-01 2.849103610759459171e-01
1.049927013864766501e-01 1.278243708004132007e-01 2.968738052891420898e-01
1.089874020283561201e-01 1.331935038256579495e-01 3.089538370897204067e-01
1.129223008178065757e-01 1.385131449459734432e-01 3.211528356563003173e-01
1.167787372671460905e-01 1.437907235567953967e-01 3.334763137669404798e-01
1.205463368759411846e-01 1.490330498991488395e-01 3.459192712822865556e-01
1.242159653236002137e-01 1.542448960991174289e-01 3.584824902114073786e-01
1.277778433781233958e-01 1.594292997650425259e-01 3.711737182907450250e-01
1.312243830518974863e-01 1.645899274655008293e-01 3.839938980982077199e-01
1.345490385720651272e-01 1.697307784244891371e-01 3.969402498465131601e-01
1.377440294143187915e-01 1.748552489081271477e-01 4.100132193954066917e-01
1.407996250839473606e-01 1.799660943911219058e-01 4.232173781877683894e-01
1.437072840727513789e-01 1.850671827584920992e-01 4.365502201613832844e-01
1.464560785040432134e-01 1.901619264187031366e-01 4.500130967746718835e-01
1.490348239402249919e-01 1.952544786288682443e-01 4.636036238706138235e-01
1.514309258266629821e-01 2.003493745808981319e-01 4.773185296806868871e-01
1.536266468153191511e-01 2.054503782969492598e-01 4.911623505831314018e-01
1.556073044691906326e-01 2.105638989188344801e-01 5.051241984185815825e-01
1.573534941852142433e-01 2.156962129560367203e-01 5.191977165014443063e-01
1.588411077773489166e-01 2.208540630983729658e-01 5.333780986311171812e-01
1.600396310676128475e-01 2.260448054488821412e-01 5.476626851250718797e-01
1.609205321230856855e-01 2.312792188936386995e-01 5.620307968352061812e-01
1.614455266589641669e-01 2.365687572657042548e-01 5.764661948566720540e-01
1.615687582201552897e-01 2.419271731798259828e-01 5.909472809712390529e-01
1.612352769386722617e-01 2.473711867632350236e-01 6.054446786315181850e-01
1.603793369556453796e-01 2.529213001868846900e-01 6.199179403497330210e-01
1.589165703217830516e-01 2.586022090180682964e-01 6.343183928801525706e-01
1.567387388150081051e-01 2.644444177118183137e-01 6.485832602021305293e-01
1.537422741701028883e-01 2.704875236386605764e-01 6.625942914402649375e-01
1.497539850217274870e-01 2.767772906869140348e-01 6.762428866538948702e-01
1.446190280141932405e-01 2.833703371099008383e-01 6.893303155405390292e-01
1.381506719438430897e-01 2.903290266080994497e-01 7.016166706228140759e-01
1.301937481102784511e-01 2.977137475582261605e-01 7.127928058673724809e-01
1.207056294812082625e-01 3.055623316038361126e-01 7.225101391519437311e-01
1.098226938414934850e-01 3.138651882979107688e-01 7.304694442052537262e-01
9.795843445627248902e-02 3.225431024585883599e-01 7.365271649499161022e-01
8.570023827886968926e-02 3.314661612751826358e-01 7.407718073497575606e-01
7.367688675533522191e-02 3.404967458048910878e-01 7.434693436710165804e-01
6.252232597310111717e-02 3.495197377754474255e-01 7.449562973790243570e-01
5.283258403911513662e-02 3.584600781237165523e-01 7.455451847703780111e-01
4.520938539971628561e-02 3.672749347230809258e-01 7.454889092261348660e-01
4.023631323597558207e-02 3.759418522020857023e-01 7.449841371058433248e-01
3.831576485736640919e-02 3.844536826035842014e-01 7.441746283479093726e-01
3.948349528027724625e-02 3.928137536670130436e-01 7.431581338661337188e-01
4.345843888396401511e-02 4.010237097170425424e-01 7.420243047032398787e-01
4.962431895510652918e-02 4.090953160508887798e-01 7.408166831290733390e-01
5.739800117053886486e-02 4.170346954168047682e-01 7.395877539875740370e-01
6.626017539288729663e-02 4.248521500099918247e-01 7.383650122111450331e-01
7.582789138192139178e-02 4.325572249230948407e-01 7.371697936237424642e-01
8.583710398799437868e-02 4.401583433039914506e-01 7.360206802189788178e-01
9.611023898027164225e-02 4.476635745571445613e-01 7.349312516211791158e-01
1.065303883460570755e-01 4.550804241299538089e-01 7.339117735870001047e-01
1.170216465020320479e-01 4.624159111924117660e-01 7.329693696980987827e-01
1.275357983203295742e-01 4.696767197731893662e-01 7.321074968220891988e-01
1.380432121650685962e-01 4.768686794858796318e-01 7.313295149248254523e-01
1.485266661140711986e-01 4.839970747487774561e-01 7.306374469250058734e-01
1.589774367328510296e-01 4.910666599003393196e-01 7.300322634722052895e-01
1.693927097070233589e-01 4.980816767007848478e-01 7.295140990156090410e-01
1.797738575018875684e-01 5.050458714090990675e-01 7.290824177866462863e-01
1.901252883653588299e-01 5.119625097219066001e-01 7.287361439246872186e-01
2.004536741991692628e-01 5.188343886055641896e-01 7.284737667569484154e-01
2.107674308545827713e-01 5.256638445778923918e-01 7.282934299212738827e-01
2.210763664417479957e-01 5.324527583974503209e-01 7.281930113726282627e-01
2.313914398990180310e-01 5.392025564688637251e-01 7.281702001646749300e-01
2.417245888287989919e-01 5.459142096302009861e-01 7.282225751002735503e-01
2.520885959217774586e-01 5.525882304010224511e-01 7.283476897674903139e-01
2.624969693582706598e-01 5.592246702775622857e-01 7.285431679898072277e-01
2.729638158446638374e-01 5.658231192947545951e-01 7.288068131774058100e-01
2.835036864649327915e-01 5.723827108458280355e-01 7.291367343046032401e-01
2.941313761905648416e-01 5.789021356407471064e-01 7.295314900628433463e-01
3.048616586033888742e-01 5.853796696249742304e-01 7.299902509334000866e-01
3.157089391891977348e-01 5.918132215225327952e-01 7.305129762764809298e-01
3.266868146870496870e-01 5.982004061606387424e-01 7.311005998876300982e-01
3.378075337747806772e-01 6.045386495071959354e-01 7.317552128195412564e-01
3.490818000871770965e-01 6.108253115449155946e-01 7.324796800093837934e-01
3.605169723334935572e-01 6.170578982409773428e-01 7.332792190500567742e-01
3.721163789840524760e-01 6.232343167456089184e-01 7.341612307067191256e-01
3.838798790902256397e-01 6.293530561660979350e-01 7.351337469132933622e-01
3.958028137607393915e-01 6.354134382185918639e-01 7.362060213183361235e-01
4.078755353103907799e-01 6.414158575290970221e-01 7.373884640869047269e-01
4.200825599604326444e-01 6.473620651543986471e-01 7.386930839012650907e-01
4.324066537782813580e-01 6.532548504780409937e-01 7.401294470548486215e-01
4.448270147115060968e-01 6.590982595876608841e-01 7.417069328868316491e-01
4.573210929073768805e-01 6.648973878622411737e-01 7.434335263117878290e-01
4.698658823662555939e-01 6.706581495075844002e-01 7.453153847182404368e-01
4.824391638200489774e-01 6.763869948243975694e-01 7.473565606883062484e-01
4.950205546820442004e-01 6.820906132919950515e-01 7.495589040396917202e-01
5.075922656244505893e-01 6.877756591216436233e-01 7.519221394187723950e-01
5.201395209800695474e-01 6.934485275105084501e-01 7.544440922042097153e-01
5.326506556783058288e-01 6.991151975856019218e-01 7.571210204550441469e-01
5.451151059190834092e-01 7.047815278674717243e-01 7.599492971931861574e-01
5.575181558550945660e-01 7.104543187762520917e-01 7.629291415211207905e-01
5.698635397781144363e-01 7.161365136199150383e-01 7.660488334775965580e-01
5.821485860154738123e-01 7.218322109491160932e-01 7.693023567855580280e-01
5.943602368020499682e-01 7.275477994362923306e-01 7.726916325476315128e-01
6.065080899095934841e-01 7.332844479726540188e-01 7.762041114816310428e-01
6.185864665812587093e-01 7.390466790641196937e-01 7.798383670352863062e-01
6.305963554035781682e-01 7.448373397672364282e-01 7.835892177406345027e-01
6.425403291413834816e-01 7.506587685331359561e-01 7.874510644025978223e-01
6.544185977825304201e-01 7.565137308188258913e-01 7.914204294842384080e-01
6.662315287143806275e-01 7.624048800378464552e-01 7.954941819359623301e-01
6.779850393157120791e-01 7.683332855369554570e-01 7.996659972294575258e-01
6.896755013217630292e-01 7.743024253591546113e-01 8.039360994237751967e-01
7.013107513266423343e-01 7.803126348729136907e-01 8.082975524678305268e-01
7.128885314512861671e-01 7.863668558670167119e-01 8.127502175082055302e-01
7.244112448085553435e-01 7.924667152464015540e-01 8.172911293271547528e-01
7.358836989688380958e-01 7.986130654557248576e-01 8.219158971644889844e-01
7.473044386446877629e-01 8.048084510111997991e-01 8.266243361574353576e-01
7.586759860087319840e-01 8.110542493572184819e-01 8.314137476978961105e-01
7.700007022560961811e-01 8.173518180665149124e-01 8.362815657919246970e-01
7.812789490175476859e-01 8.237030526155880716e-01 8.412265533654796901e-01
7.925116214732782494e-01 8.301096652946458043e-01 8.462470956714009951e-01
8.036997510351909790e-01 8.365733017623517842e-01 8.513414099007207136e-01
8.148431081888104499e-01 8.430959697406819053e-01 8.565084542610862384e-01
8.259421593641688153e-01 8.496794646156250463e-01 8.617465433522450979e-01
8.369967539127278755e-01 8.563257710703112702e-01 8.670541605111352634e-01
8.480038340013845710e-01 8.630377874332209043e-01 8.724315628924932398e-01
8.589628398368314155e-01 8.698176873531217046e-01 8.778768407867350021e-01
8.698718779534676537e-01 8.766681056859056964e-01 8.833884311630415542e-01
8.807225950585577667e-01 8.835937613774932364e-01 8.889689808541467730e-01
8.915125639987498962e-01 8.905976582376062822e-01 8.946157932523199907e-01
9.022321856472953483e-01 8.976851819173250480e-01 9.003303399902121695e-01
9.128629530771473766e-01 9.048646775143293075e-01 9.061201920398328502e-01
9.233906674988954233e-01 9.121434538185491103e-01 9.119872701021832784e-01
9.337690699519377580e-01 9.195389121458987791e-01 9.179604265763980919e-01
9.438768578707728008e-01 9.270905817099515112e-01 9.241478407896098757e-01
9.450241336950316873e-01 9.267273985243987822e-01 9.232017297254778709e-01
9.401771503305338396e-01 9.175010969420787088e-01 9.127353023021862466e-01
9.357788176131993652e-01 9.081849602977229985e-01 9.019958970504049489e-01
9.316195516453253944e-01 8.988455936080792519e-01 8.911247259869568005e-01
9.276366903952597553e-01 8.895015346080178409e-01 8.801625091446737548e-01
9.237963723082156520e-01 8.801619939913720714e-01 8.691306632065185500e-01
9.200767818087643990e-01 8.708323115565800299e-01 8.580426479951916985e-01
9.164622665155582881e-01 8.615158650696722598e-01 8.469079246742292622e-01
9.129409700010679973e-01 8.522148302070590153e-01 8.357335125969563849e-01
9.095033897880691054e-01 8.429306461313940124e-01 8.245249535921664874e-01
9.061413505699446036e-01 8.336643536120030840e-01 8.132870275157044748e-01
9.028482401464246188e-01 8.244164969479621519e-01 8.020234992942889551e-01
8.996183458250350817e-01 8.151873435674267254e-01 7.907375926293952473e-01
8.964464017336835067e-01 8.059770371403549571e-01 7.794323253122593664e-01
8.933280984986637918e-01 7.967854062668440207e-01 7.681100636805964221e-01
8.902585423162623357e-01 7.876125240246927284e-01 7.567737971546929510e-01
8.872341531517662361e-01 7.784580066111700392e-01 7.454255275413548265e-01
8.842515223931736168e-01 7.693214239645329577e-01 7.340672060990318659e-01
8.813075692568397290e-01 7.602022354826803996e-01 7.227005864751234743e-01
8.783989821496545058e-01 7.511000050313838550e-01 7.113277185683902770e-01
8.755220238171187441e-01 7.420144833045193566e-01 6.999511354610528091e-01
8.726749843514796101e-01 7.329446408023255755e-01 6.885716411869530207e-01
8.698547215416584377e-01 7.238900130170270453e-01 6.771913952144653637e-01
8.670596624933959440e-01 7.148495108503407636e-01 6.658111988100340328e-01
8.642855405275950975e-01 7.058231520382152180e-01 6.544344321908397433e-01
8.615312235931757989e-01 6.968096810882964398e-01 6.430616521710770250e-01
8.587945613193220806e-01 6.878082471826545419e-01 6.316944172411573799e-01
8.560737697641845889e-01 6.788178389092562881e-01 6.203340096274330140e-01
8.533647514077651319e-01 6.698384520573987810e-01 6.089840337795546787e-01
8.506671813652186831e-01 6.608684511065675560e-01 5.976445430267850467e-01
8.479803760976425409e-01 6.519062997089123401e-01 5.863159553827915760e-01
8.452992890124730874e-01 6.429524288345007665e-01 5.750030931030669645e-01
8.426233261680482478e-01 6.340052727647743636e-01 5.637064935396088883e-01
8.399526706457652869e-01 6.250628349958033958e-01 5.524259435430849408e-01
8.372823606159069953e-01 6.161255232979981900e-01 5.411665126797099434e-01
8.346115013722664733e-01 6.071918418882036317e-01 5.299292934604437066e-01
8.319416971332159738e-01 5.982589838531851001e-01 5.187128588454869016e-01
8.292658393187906096e-01 5.893284247613858051e-01 5.075248627530006829e-01
8.265865191963318592e-01 5.803968068112989043e-01 4.963630791145096643e-01
8.239007716165898110e-01 5.714634777707676694e-01 4.852311209906218226e-01
8.212047236427109098e-01 5.625283035002011101e-01 4.741337367172645534e-01
8.185028441784485409e-01 5.535866371885466153e-01 4.630669974760600605e-01
8.157857974447502158e-01 5.446412013022012832e-01 4.520417208585162938e-01
8.130598755024439628e-01 5.356861530869529986e-01 4.410523109242584505e-01
8.103157295614864530e-01 5.267242547499885186e-01 4.301100025814295069e-01
8.075600763067809496e-01 5.177491168092126506e-01 4.192090517728162546e-01
8.047829983640772955e-01 5.087638530967844019e-01 4.083617627821872209e-01
8.019913643823087801e-01 4.997616265469864705e-01 3.975626288212918968e-01
7.991776069093209367e-01 4.907441098507538402e-01 3.868219510427612362e-01
7.963412506983238437e-01 4.817086347924898759e-01 3.761426799347312722e-01
7.934853391750592566e-01 4.726500967551418020e-01 3.655242898980974875e-01
7.906020861628313412e-01 4.635701843716890092e-01 3.549783159786080722e-01
7.876922078438237662e-01 4.544650164439674178e-01 3.445075235304189132e-01
7.847584433242353885e-01 4.453290095612297272e-01 3.341130019968927001e-01
7.817943763481589592e-01 4.361626378195018194e-01 3.238062118505327658e-01
7.787986023472407426e-01 4.269628178617235204e-01 3.135938243777520174e-01
7.757700306772615795e-01 4.177259860608659170e-01 3.034828822113839197e-01
7.727073763310282617e-01 4.084484227667745659e-01 2.934814547614346680e-01
7.696090956246998127e-01 3.991262801626034307e-01 2.835988305456279002e-01
7.664733140591651894e-01 3.897556205902359405e-01 2.738457322824414675e-01
7.632977462662783319e-01 3.803324674237734127e-01 2.642345539395078435e-01
7.600796083228246180e-01 3.708528707479538111e-01 2.547796171102231777e-01
7.568155232311498670e-01 3.613129902046505748e-01 2.454974414736316723e-01
7.535014210471865370e-01 3.517091973617619272e-01 2.364070204678299647e-01
7.501324359996863755e-01 3.420381997805444496e-01 2.275300884680255264e-01
7.467083678923864820e-01 3.322917332197801166e-01 2.188868097223075626e-01
7.432206214135255173e-01 3.224689513722865386e-01 2.105072965395320406e-01
7.396626352495903056e-01 3.125667176366879740e-01 2.024226257801388651e-01
7.360276592028781595e-01 3.025817058357924139e-01 1.946669682951213953e-01
7.323075054306246168e-01 2.925115926125090304e-01 1.872788656593587786e-01
7.284886793139783157e-01 2.823599622393618280e-01 1.803025816545419935e-01
7.245597795551689257e-01 2.721283507524605572e-01 1.737830391183990686e-01
7.205052554891407945e-01 2.618237699935750395e-01 1.677681078164939554e-01
7.163077524522897255e-01 2.514568470079543427e-01 1.623050371038979034e-01
7.119477511381474555e-01 2.410431671523047270e-01 1.574376915741905747e-01
7.074046106878864038e-01 2.306029576903746436e-01 1.532027782639572566e-01
7.026577763956233236e-01 2.201603247696940491e-01 1.496264194674281345e-01
6.976856380676357272e-01 2.097462458983016809e-01 1.467193374830310926e-01
6.924700993857928477e-01 1.993923480969033712e-01 1.444757571309586708e-01
6.869956304794752056e-01 1.891326217641629281e-01 1.428720296451389260e-01
6.812509389222788370e-01 1.790005719286615893e-01 1.418684919672574540e-01
6.752290404511711586e-01 1.690283951735306323e-01 1.414126182563193168e-01
6.689272347019628029e-01 1.592460113853382819e-01 1.414436246187784352e-01
6.623462930327670417e-01 1.496814992323431404e-01 1.418966965532167945e-01
6.554894621920681619e-01 1.403622590219790744e-01 1.427060758527507467e-01
6.483618361407726960e-01 1.313154491044398742e-01 1.438093776439538507e-01
6.409690900196458596e-01 1.225708994915949840e-01 1.451466621124405387e-01
6.333171868548845840e-01 1.141616381299322275e-01 1.466635532234118466e-01
6.254117682555888624e-01 1.061263207405454406e-01 1.483083323541042609e-01
6.172580943089347461e-01 9.850955940169045522e-02 1.500342818055006577e-01
6.088612137441752337e-01 9.136271116346880716e-02 1.517946748017894032e-01
6.002260075568476294e-01 8.474268845605749390e-02 1.535471889644061949e-01
5.913580665313180607e-01 7.870937974751122945e-02 1.552475962263118736e-01
5.822640724597829553e-01 7.332110668924821106e-02 1.568528524628541587e-01
5.729523242155735163e-01 6.862870182587510470e-02 1.583209522546889236e-01
5.634337248699358147e-01 6.466696013382158825e-02 1.596097119552595534e-01
5.537224099506778963e-01 6.144626935987178989e-02 1.606779590153868675e-01
5.438351395676241928e-01 5.894914800814857886e-02 1.614895452047331870e-01
5.337912133736735232e-01 5.712858102881844535e-02 1.620138116792186611e-01
5.236119994512118403e-01 5.591095129374230172e-02 1.622265257547594042e-01
5.133214710039554207e-01 5.519900277454217047e-02 1.621082697921440163e-01
5.029415119053961547e-01 5.489268743951671026e-02 1.616521201270082198e-01
4.924920560465744224e-01 5.489497800522934179e-02 1.608594568967110505e-01
4.819970419875420631e-01 5.509874164212420766e-02 1.597294678457327477e-01
4.714685312956432006e-01 5.543752958786138385e-02 1.582803570073196830e-01
4.609298385556733768e-01 5.581817450175154821e-02 1.565161535498545142e-01
4.503882306974699712e-01 5.620093240974462917e-02 1.544603609580012249e-01
4.398550445667638309e-01 5.653967482715421822e-02 1.521296397207116124e-01
4.293395854267749168e-01 5.679892905513854451e-02 1.495413040881575784e-01
4.188488052114383020e-01 5.695351275720081374e-02 1.467131019928892832e-01
4.083874753510262079e-01 5.698679735961746651e-02 1.436626150429021476e-01
3.979584206972520133e-01 5.688899115379054960e-02 1.404067946849407167e-01
3.875661805934746962e-01 5.664744510513378128e-02 1.369596232193354413e-01
3.772162562473511671e-01 5.624858727791105101e-02 1.333332882463785507e-01
3.669009710312429173e-01 5.571027577451325569e-02 1.295456315830630090e-01
3.566289201162299305e-01 5.501236188184371184e-02 1.256041268313332904e-01
3.463946141954395985e-01 5.416713051157899528e-02 1.215221064681567403e-01
3.362018241761821069e-01 5.316577293008726418e-02 1.173074037138401304e-01
3.260424402261492549e-01 5.202497987647033972e-02 1.129713889339670624e-01
3.159278573165507087e-01 5.071934147936141973e-02 1.085175153831856448e-01
3.058440467352613878e-01 4.927677960771512794e-02 1.039569210271578392e-01
2.957906289005873823e-01 4.769542811326495796e-02 9.929501197500162357e-02
2.857761694878986902e-01 4.595516903310007534e-02 9.453469664208309642e-02
2.757883929056377248e-01 4.407729543414404955e-02 8.968329380554104779e-02
2.658251433737037206e-01 4.206161274965801444e-02 8.474472186508703875e-02
2.558840557281191197e-01 3.990046570244704105e-02 7.972236454569985031e-02
2.459622220783502511e-01 3.762595141506584057e-02 7.461913567560218841e-02
2.360563646646140490e-01 3.529747994604028744e-02 6.943744239412558139e-02]; 

   case 'gra'
      RGB = [5.119113838889112324e-07 2.052270195661655352e-06 5.982577941045682640e-06
2.598440803609315740e-04 2.625944024536107477e-04 2.770622839605119624e-04
8.656156394278343835e-04 8.570638031131799393e-04 8.794207369001087564e-04
1.769403238430108027e-03 1.740368058158140796e-03 1.768607096399604827e-03
2.949472160671299040e-03 2.891903379202926793e-03 2.924180394823831797e-03
4.392323531342203803e-03 4.298904740810657703e-03 4.333431047933433700e-03
6.088553115957890012e-03 5.952519318106651129e-03 5.987527380406317858e-03
8.031191715623246449e-03 7.846222922619310730e-03 7.879950296964649201e-03
1.021487523527209347e-02 9.975026089550139557e-03 1.000570608701620810e-02
1.263537226121889262e-02 1.233502163300135256e-02 1.236087717148891089e-02
1.528929074309834900e-02 1.492310361152201538e-02 1.494234288239030227e-02
1.817388406754013447e-02 1.773678153559082871e-02 1.774759483393608533e-02
2.128691657088564032e-02 2.077405157770592994e-02 2.077460893832101113e-02
2.462656669738432988e-02 2.403330390048570228e-02 2.402175331762853791e-02
2.819135511163322477e-02 2.751325392935317568e-02 2.748772001065160481e-02
3.198008997843642537e-02 3.121289009539909495e-02 3.117147304311678954e-02
3.599182442355883943e-02 3.513143325966101155e-02 3.507220810044759524e-02
4.022582286584987926e-02 3.926830463975648777e-02 3.918932064337803806e-02
4.449007920559793633e-02 4.350122490641009781e-02 4.340640025397465157e-02
4.870367428407672977e-02 4.767846692745423415e-02 4.756790269604229543e-02
5.287362099388332598e-02 5.181334368492381687e-02 5.168705396373681621e-02
5.700314746954803641e-02 5.590901252301124641e-02 5.576700147480036923e-02
6.109513723484618491e-02 5.996829329212761267e-02 5.981055680351737153e-02
6.515218007540821143e-02 6.399371820125457355e-02 6.382024521975740439e-02
6.917661353844734018e-02 6.798757248509634810e-02 6.779834611081525519e-02
7.317055710313111194e-02 7.195192788781537563e-02 7.174692627103734788e-02
7.713594055593872567e-02 7.588867046670466632e-02 7.566786755046528423e-02
8.107452773767656606e-02 7.979952385914274027e-02 7.956288999694530184e-02
8.498793655960165672e-02 8.368606889245686076e-02 8.343357136470588853e-02
8.887765598636060416e-02 8.754976022063193364e-02 8.728136366834671200e-02
9.274506053351619372e-02 9.139194052486757092e-02 9.110760731542752455e-02
9.659142271361736976e-02 9.521385270343013518e-02 9.491354324016601507e-02
1.004179237775132127e-01 9.901665039071003149e-02 9.870032337586748250e-02
1.042256630300536779e-01 1.028014070791676282e-01 1.024690197379714285e-01
1.080156659465895397e-01 1.065691240661386929e-01 1.062206323382679229e-01
1.117888912751462127e-01 1.103207374067532442e-01 1.099560961104033285e-01
1.155462372762081913e-01 1.140571240219185878e-01 1.136762869947187948e-01
1.192885472257254331e-01 1.177791070845015386e-01 1.173820273048238605e-01
1.230166142857740674e-01 1.214874607860790545e-01 1.210740904776794746e-01
1.267311858301593541e-01 1.251829145698133972e-01 1.247532052922631640e-01
1.304329672982797728e-01 1.288661569013136721e-01 1.284200596282795426e-01
1.341226256391360172e-01 1.325378386381312334e-01 1.320753038252369949e-01
1.378007923979476801e-01 1.361985760492960551e-01 1.357195536930293212e-01
1.414680664900422602e-01 1.398489535286627783e-01 1.393533932175645418e-01
1.451250167001958413e-01 1.434895260394738747e-01 1.429773769986621423e-01
1.487721839401931478e-01 1.471208213222370831e-01 1.465920324521616802e-01
1.524100832928171334e-01 1.507433418935591463e-01 1.501978618037517477e-01
1.560392058666605075e-01 1.543575668598215767e-01 1.537953438982961329e-01
1.596600204829017322e-01 1.579639535664100758e-01 1.573849358452757696e-01
1.632729752124383871e-01 1.615629391005102000e-01 1.609670745182808227e-01
1.668784987794287578e-01 1.651549416631863942e-01 1.645421779242018656e-01
1.704770018452845537e-01 1.687403618244931736e-01 1.681106464558155544e-01
1.740688781854339839e-01 1.723195836736857167e-01 1.716728640397789662e-01
1.776545057697057339e-01 1.758929758751424655e-01 1.752291991906090307e-01
1.812342477558830656e-01 1.794608926393625181e-01 1.787800059799695696e-01
1.848084534049017069e-01 1.830236746173136975e-01 1.823256249295160947e-01
1.883774589251684672e-01 1.865816497254700890e-01 1.858663838346062414e-01
1.919415882526751271e-01 1.901351339080520064e-01 1.894025985253704192e-01
1.955011537728245652e-01 1.936844318422726863e-01 1.929345735709248333e-01
1.990564569892651070e-01 1.972298375917654889e-01 1.964626029318853884e-01
2.026077891444517665e-01 2.007716352128197101e-01 1.999869705657932872e-01
2.061554317961845073e-01 2.043100993175656077e-01 2.035079509895849070e-01
2.096996573539102737e-01 2.078454955978322194e-01 2.070258098028130989e-01
2.132407295782156598e-01 2.113780813130166081e-01 2.105408041749530779e-01
2.167789040465809003e-01 2.149081057449790388e-01 2.140531832997963901e-01
2.203144285881817321e-01 2.184358106226793050e-01 2.175631888196422614e-01
2.238475436902406246e-01 2.219614305190086845e-01 2.210710552217338698e-01
2.273784828782031697e-01 2.254851932220399702e-01 2.245770102091559361e-01
2.309074730717983193e-01 2.290073200827110744e-01 2.280812750482046902e-01
2.344347349188553031e-01 2.325280263407684678e-01 2.315840648940526436e-01
2.379604831085705419e-01 2.360475214306362868e-01 2.350855890963675088e-01
2.414849266657837501e-01 2.395660092687221798e-01 2.385860514863973880e-01
2.450082692276646656e-01 2.430836885235431177e-01 2.420856506468980207e-01
2.485307093041047577e-01 2.466007528699289408e-01 2.455845801661603045e-01
2.520524405229936926e-01 2.501173912284589962e-01 2.490830288772885037e-01
2.555736518614571962e-01 2.536337879911821358e-01 2.525811810837814142e-01
2.590945278640408023e-01 2.571501232345917431e-01 2.560792167723802693e-01
2.626152488487566439e-01 2.606665729207388593e-01 2.595773118140702884e-01
2.661359911018202906e-01 2.641833090872994227e-01 2.630756381540483857e-01
2.696569270618426883e-01 2.677005000273448010e-01 2.665743639914045504e-01
2.731782254941877985e-01 2.712183104595063399e-01 2.700736539492059607e-01
2.767000516561451962e-01 2.747369016891675342e-01 2.735736692356186550e-01
2.802225674535130939e-01 2.782564317612734595e-01 2.770745677966527931e-01
2.837459315891543987e-01 2.817770556052956454e-01 2.805765044610715742e-01
2.872702997040328943e-01 2.852989251728569320e-01 2.840796310779650780e-01
2.907958245112020612e-01 2.888221895684777163e-01 2.875840966474506044e-01
2.943226559231933992e-01 2.923469951738719153e-01 2.910900474449292785e-01
2.978509411732042089e-01 2.958734857661936557e-01 2.945976271392962698e-01
3.013808249304671394e-01 2.994018026306028979e-01 2.981069769054734420e-01
3.049124494101519978e-01 3.029320846674894985e-01 3.016182355316066044e-01
3.084459544781194551e-01 3.064644684946828401e-01 3.051315395212462755e-01
3.119814777508492787e-01 3.099990885449340761e-01 3.086470231908091666e-01
3.155191546907993305e-01 3.135360771589531526e-01 3.121648187625948867e-01
3.190591186974824911e-01 3.170755646742604017e-01 3.156850564536164416e-01
3.226015011944919908e-01 3.206176795100827426e-01 3.192078645604823350e-01
3.261464317127154011e-01 3.241625482485306353e-01 3.227333695405575908e-01
3.296940379699381807e-01 3.277102957122579041e-01 3.262616960896098095e-01
3.332444459470546017e-01 3.312610450388014383e-01 3.297929672161359349e-01
3.367977799610606948e-01 3.348149177517827924e-01 3.333273043125560831e-01
3.403541627350049614e-01 3.383720338291492657e-01 3.368648272234392449e-01
3.439137154650676598e-01 3.419325117686051185e-01 3.404056543109294508e-01
3.474765578849234871e-01 3.454964686503906313e-01 3.439499025175156155e-01
3.510428083275172195e-01 3.490640201975497847e-01 3.474976874262884485e-01
3.546125837844137374e-01 3.526352808338130695e-01 3.510491233188203775e-01
3.581859999628214664e-01 3.562103637392293742e-01 3.546043232307861115e-01
3.617631713404313443e-01 3.597893809036549762e-01 3.581633990054478578e-01
3.653442112181757850e-01 3.633724431782132114e-01 3.617264613451083854e-01
3.689292317710045621e-01 3.669596603248330147e-01 3.652936198606414053e-01
3.725183440967992365e-01 3.705511410639528158e-01 3.688649831191926354e-01
3.761116582634963512e-01 3.741469931204933763e-01 3.724406586901452543e-01
3.797092833545231416e-01 3.777473232681791138e-01 3.760207531894346200e-01
3.833113275126273645e-01 3.813522373722905701e-01 3.796053723222968523e-01
3.869178979821737419e-01 3.849618404309278485e-01 3.831946209245254420e-01
3.905291011499888443e-01 3.885762366148534097e-01 3.867886030023093280e-01
3.941450425848224692e-01 3.921955293059884440e-01 3.903874217707223315e-01
3.977658270754869663e-01 3.958198211346236617e-01 3.939911796909281172e-01
4.013915586677523906e-01 3.994492140154045079e-01 3.975999785061617997e-01
4.050223407000290154e-01 4.030838091821594138e-01 4.012139192765472040e-01
4.086582758379366798e-01 4.067237072216087768e-01 4.048331024128048461e-01
4.122994661077632461e-01 4.103690081060247796e-01 4.084576277089013163e-01
4.159460129289094255e-01 4.140198112248798523e-01 4.120875943736929670e-01
4.195980171453375096e-01 4.176762154155350126e-01 4.157231010616072031e-01
4.232555790560795228e-01 4.213383189930122175e-01 4.193642459024058966e-01
4.269187984448540862e-01 4.250062197788932461e-01 4.230111265300784962e-01
4.305877746088247471e-01 4.286800151293829186e-01 4.266638401108986955e-01
4.342626063865396291e-01 4.323598019625811695e-01 4.303224833706847940e-01
4.379433921850948153e-01 4.360456767849914539e-01 4.339871526213026076e-01
4.416302300065532149e-01 4.397377357173059553e-01 4.376579437864426825e-01
4.453232174736505011e-01 4.434360745195019549e-01 4.413349524267043966e-01
4.490224518548273092e-01 4.471407886152713451e-01 4.450182737640205333e-01
4.527280300886077913e-01 4.508519731158264210e-01 4.487080027054508036e-01
4.564400488073677153e-01 4.545697228430990999e-01 4.524042338663719076e-01
4.601586043605019882e-01 4.582941323523668786e-01 4.561070615930931660e-01
4.638837928370361241e-01 4.620252959543280080e-01 4.598165799849223689e-01
4.676157100876950357e-01 4.657633077366521435e-01 4.635328829157047692e-01
4.713544517464566819e-01 4.695082615850322827e-01 4.672560640548638067e-01
4.751001132516106673e-01 4.732602512037566433e-01 4.709862168879611066e-01
4.788527898663595406e-01 4.770193701358209082e-01 4.747234347367983998e-01
4.826125766989533550e-01 4.807857117826079274e-01 4.784678107790839019e-01
4.863795687224105602e-01 4.845593694231487558e-01 4.822194380676812497e-01
4.901538607938191183e-01 4.883404362329857862e-01 4.859784095494587564e-01
4.939355476732588124e-01 4.921290053026548539e-01 4.897448180837610798e-01
4.977247240423329777e-01 4.959251696558080735e-01 4.935187564605149624e-01
5.015214845223622264e-01 4.997290222669866222e-01 4.973003174179899122e-01
5.053259236922266195e-01 5.035406560790653963e-01 5.010895936602286493e-01
5.091381361058908706e-01 5.073601640203813101e-01 5.048866778741611938e-01
5.129582163096103598e-01 5.111876390215642774e-01 5.086916627464180829e-01
5.167862588588553185e-01 5.150231740320770912e-01 5.125046409798591496e-01
5.206223583349409711e-01 5.188668620364874640e-01 5.163257053098251337e-01
5.244666093614033242e-01 5.227187960704765546e-01 5.201549485201331091e-01
5.283191066201067265e-01 5.265790692366016934e-01 5.239924634588227770e-01
5.321799448671195032e-01 5.304477747198240722e-01 5.278383430536651710e-01
5.360492189483508918e-01 5.343250058028127247e-01 5.316926803274503177e-01
5.399270238149752599e-01 5.382108558810341226e-01 5.355555684130592908e-01
5.438134545386418273e-01 5.421054184776434859e-01 5.394271005683370923e-01
5.477086063264958726e-01 5.460087872581790291e-01 5.433073701907695785e-01
5.516125745359978794e-01 5.499210560450835672e-01 5.471964708319829729e-01
5.555254546895829204e-01 5.538423188320419355e-01 5.510944962120672974e-01
5.594473424891353019e-01 5.577726697981703063e-01 5.550015402337400428e-01
5.633783338303157695e-01 5.617122033220391941e-01 5.589176969963529640e-01
5.673185248167353922e-01 5.656610139955613459e-01 5.628430608097579890e-01
5.712680117739895591e-01 5.696191966377359694e-01 5.667777262080311296e-01
5.752268912635538634e-01 5.735868463082712809e-01 5.707217879630733792e-01
5.791952600965620812e-01 5.775640583210815970e-01 5.746753410980864896e-01
5.831732153474624702e-01 5.815509282576770689e-01 5.786384809009386121e-01
5.871608543675723713e-01 5.855475519804423934e-01 5.826113029374229146e-01
5.911582747985197894e-01 5.895540256458231543e-01 5.865939030644199415e-01
5.951655745855983781e-01 5.935704457174205695e-01 5.905863774429681579e-01
5.991828519910353812e-01 5.975969089789966437e-01 5.945888225512460101e-01
6.032102056071760865e-01 6.016335125474120415e-01 5.986013351974859287e-01
6.072477343695864560e-01 6.056803538854835800e-01 6.026240125328050645e-01
6.112955375701087934e-01 6.097375308147855444e-01 6.066569520639827351e-01
6.153537148698353665e-01 6.138051415283823919e-01 6.107002516661682590e-01
6.194223663120388412e-01 6.178832846035217097e-01 6.147540095955469353e-01
6.235015923350435418e-01 6.219720590142719940e-01 6.188183245019491796e-01
6.275914937850678443e-01 6.260715641441202406e-01 6.228932954414247991e-01
6.316921719290113924e-01 6.301818997985384474e-01 6.269790218887789646e-01
6.358037284672281020e-01 6.343031662175184771e-01 6.310756037500810933e-01
6.399262655462556459e-01 6.384354640880801623e-01 6.351831413751425348e-01
6.440598857715362779e-01 6.425788945567643129e-01 6.393017355699824877e-01
6.482046922201168959e-01 6.467335592421092905e-01 6.434314876092731561e-01
6.523607884533356716e-01 6.508995602471195907e-01 6.475724992487805753e-01
6.565282785295067924e-01 6.550770001717296509e-01 6.517248727377952244e-01
6.607072670166060924e-01 6.592659821252698782e-01 6.558887108315696324e-01
6.648978590049534620e-01 6.634666097389374517e-01 6.600641168037573170e-01
6.691001601199109139e-01 6.676789871782776720e-01 6.642511944588598283e-01
6.733142765345891956e-01 6.719032191556818523e-01 6.684500481446935538e-01
6.775403149825812710e-01 6.761394109429005317e-01 6.726607827648712901e-01
6.817783827707052380e-01 6.803876683835863304e-01 6.768835037913099040e-01
6.860285877917917663e-01 6.846480979058627847e-01 6.811183172767648619e-01
6.902910385374814162e-01 6.889208065349254895e-01 6.853653298673966221e-01
6.945658441110833525e-01 6.932059019056816540e-01 6.896246488153768839e-01
6.988531142404552643e-01 6.975034922754330680e-01 6.938963819915323628e-01
7.031529592909376847e-01 7.018136865365992572e-01 6.981806378980350836e-01
7.074654902783347188e-01 7.061365942294985931e-01 7.024775256811436330e-01
7.117908188819498383e-01 7.104723255551799177e-01 7.067871551439977029e-01
7.161290574576848478e-01 7.148209913883080135e-01 7.111096367594695877e-01
7.204803190511824829e-01 7.191827032901243477e-01 7.154450816830777438e-01
7.248447174110647095e-01 7.235575735214593296e-01 7.197936017659686270e-01
7.292223670022126569e-01 7.279457150558282796e-01 7.241553095679664764e-01
7.336133830191388094e-01 7.323472415925961210e-01 7.285303183706990371e-01
7.380178813994349163e-01 7.367622675702208968e-01 7.329187421908015532e-01
7.424359788373007252e-01 7.411909081795812204e-01 7.373206957932009198e-01
7.468677927971572039e-01 7.456332793773956524e-01 7.417362947044913168e-01
7.513134415273599043e-01 7.500894978997243445e-01 7.461656552263956632e-01
7.557730440739957034e-01 7.545596812755732685e-01 7.506088944493225501e-01
7.602467202947890135e-01 7.590439478405959228e-01 7.550661302660249818e-01
7.647345908731115749e-01 7.635424167508956250e-01 7.595374813853584817e-01
7.692367773320856195e-01 7.680552079969388268e-01 7.640230673461460009e-01
7.737534020488213704e-01 7.725824424175786742e-01 7.685230085311559600e-01
7.782845882687531303e-01 7.771242417141914771e-01 7.730374261811926440e-01
7.828304601200990476e-01 7.816807284649387455e-01 7.775664424093062799e-01
7.873911426284513349e-01 7.862520261391466869e-01 7.821101802151280147e-01
7.919667617314893882e-01 7.908382591118159333e-01 7.866687634993275724e-01
7.965574442938269106e-01 7.954395526782646053e-01 7.912423170782062476e-01
8.011633181219949273e-01 8.000560330689072686e-01 7.958309666984236808e-01
8.057845119795647992e-01 8.046878274641713347e-01 8.004348390518634115e-01
8.104211556024298879e-01 8.093350640095594573e-01 8.050540617906442042e-01
8.150733797142104553e-01 8.139978718308641392e-01 8.096887635422753693e-01
8.197413160418366429e-01 8.186763810495288896e-01 8.143390739249705135e-01
8.244250973312831077e-01 8.233707227981730270e-01 8.190051235631107263e-01
8.291248573634558694e-01 8.280810292362735803e-01 8.236870441028736334e-01
8.338407309702662307e-01 8.328074335660153871e-01 8.283849682280266524e-01
8.385728540508587958e-01 8.375500700483110572e-01 8.330990296758882252e-01
8.433213635880250081e-01 8.423090740189962400e-01 8.378293632534637991e-01
8.480863976647909919e-01 8.470845819052075232e-01 8.425761048537626641e-01
8.528680954812019044e-01 8.518767312419356363e-01 8.473393914722943121e-01
8.576665973712844782e-01 8.566856606887783832e-01 8.521193612237554227e-01
8.624820448202135115e-01 8.615115100468785370e-01 8.569161533589082502e-01
8.673145804816755344e-01 8.663544202760624646e-01 8.617299082816555211e-01
8.721643481954295396e-01 8.712145335121869172e-01 8.665607675663183906e-01
8.770314930050886471e-01 8.760919930846851056e-01 8.714088739751210122e-01
8.819161611761232589e-01 8.809869435343266053e-01 8.762743714758871594e-01
8.868185002140476225e-01 8.858995306312075213e-01 8.811574052599523421e-01
8.917386588828791760e-01 8.908299013929442678e-01 8.860581217602967463e-01
8.966767872237935144e-01 8.957782041031091547e-01 8.909766686699112093e-01
9.016330365740294894e-01 9.007445883298949019e-01 8.959131949603859058e-01
9.066075595860336733e-01 9.057292049450164084e-01 9.008678509007446156e-01
9.116005102468445198e-01 9.107322061428585469e-01 9.058407880765154685e-01
9.166120438977335416e-01 9.157537454598729809e-01 9.108321594090563744e-01
9.216423172541110542e-01 9.207939777942257820e-01 9.158421191751267010e-01
9.266914884256760576e-01 9.258530594257129431e-01 9.208708230267246275e-01
9.317597169368545540e-01 9.309311480359353519e-01 9.259184280111854015e-01
9.368471637474911073e-01 9.360284027287524289e-01 9.309850925915528208e-01
9.419539912738585929e-01 9.411449840510025533e-01 9.360709766672253851e-01
9.470803634098973189e-01 9.462810540135232484e-01 9.411762415948876637e-01
9.522264455488126389e-01 9.514367761124457035e-01 9.463010502097274346e-01
9.573924046049070435e-01 9.566123153508007126e-01 9.514455668469502525e-01
9.625784090357906164e-01 9.618078382604079435e-01 9.566099573635956643e-01
9.677846288648154216e-01 9.670235129241012695e-01 9.617943891606574036e-01
9.730112357039150117e-01 9.722595089982409844e-01 9.669990312055217752e-01
9.782584027767170509e-01 9.775159977355695196e-01 9.722240540547290033e-01
9.835263049420009951e-01 9.827931520083860173e-01 9.774696298770532144e-01
9.888151187175039381e-01 9.880911463320615207e-01 9.827359324769276983e-01
9.941250223040635214e-01 9.934101568888956679e-01 9.880231373182063459e-01
9.994561956101176703e-01 9.987503615523224410e-01 9.933314215482736964e-01]; 

   case 'oxy' 
      RGB = [2.503217690585841648e-01 2.046237300762866404e-02 1.966891524096342492e-02
2.555648459031592545e-01 2.111591456205731687e-02 2.090007960568026485e-02
2.608015512876060149e-01 2.176436656689777205e-02 2.215967228611414111e-02
2.660446264484087608e-01 2.238303577293145399e-02 2.344016212259009482e-02
2.712886712210940687e-01 2.298148595687718801e-02 2.474301784763054818e-02
2.765326364024714989e-01 2.356133594445880552e-02 2.606762144216200061e-02
2.817861616671273883e-01 2.410204070052599998e-02 2.740685289115278589e-02
2.870367982396965223e-01 2.462864448899236905e-02 2.876669328790133415e-02
2.922964729271802509e-01 2.511521382538545177e-02 3.013810573921637051e-02
2.975597737008193189e-01 2.557236440082744783e-02 3.152275139605174165e-02
3.028221259572578816e-01 2.600980779972315513e-02 3.292220687889551439e-02
3.080989498969098905e-01 2.639179365308667174e-02 3.432344572619091400e-02
3.133766829953620880e-01 2.674829090811269017e-02 3.573422597599480721e-02
3.186553157006936643e-01 2.707912721943054463e-02 3.715280555634475207e-02
3.239494929539773471e-01 2.734824710730623595e-02 3.856493364156053649e-02
3.292456607259481305e-01 2.758740066123798601e-02 3.997915613902540410e-02
3.345444278514146941e-01 2.779482795061332528e-02 4.137064065412290187e-02
3.398547648028715784e-01 2.794706430858015858e-02 4.271748371599067701e-02
3.451724938219263294e-01 2.805368502448264342e-02 4.402857557145471029e-02
3.504942702458693216e-01 2.812284384388611183e-02 4.530588662285887219e-02
3.558206849295221796e-01 2.815263330374409914e-02 4.654771396733792949e-02
3.611591248942810362e-01 2.812226066677377817e-02 4.774545304821291997e-02
3.665053860525750440e-01 2.804229091411641200e-02 4.890135088859193929e-02
3.718571787846235432e-01 2.791864422174392124e-02 5.001583572284549334e-02
3.772149143594046850e-01 2.774987601297047857e-02 5.108646119902058441e-02
3.825789453973626464e-01 2.753471831366087474e-02 5.211053476335369278e-02
3.879495474114582620e-01 2.727215754453635524e-02 5.308509691226611510e-02
3.933299882018631566e-01 2.695211659028963827e-02 5.400332569218738860e-02
3.987172740322774689e-01 2.658339014111724477e-02 5.486495007360833037e-02
4.041100983203390062e-01 2.617015416328878283e-02 5.566751194225177252e-02
4.095080947848049324e-01 2.571397721177636972e-02 5.640694245975855919e-02
4.149106638808665037e-01 2.521741655377478589e-02 5.707880838445476140e-02
4.203169224276966931e-01 2.468426285194311523e-02 5.767829102284769555e-02
4.257256449760880579e-01 2.411983059354547848e-02 5.820016787208186909e-02
4.311355423694091527e-01 2.353014014799948567e-02 5.863827924234723404e-02
4.365491859863961932e-01 2.290868330200904537e-02 5.897909145270988779e-02
4.419598927187284310e-01 2.228071155950197418e-02 5.922145534252384186e-02
4.473641324377782280e-01 2.166211126655111924e-02 5.935827807305734533e-02
4.527575481284770720e-01 2.107281505394200879e-02 5.938213194820316648e-02
4.581418547792654450e-01 2.051270615727579169e-02 5.927134198778213209e-02
4.635070947573724509e-01 2.002404746597910568e-02 5.902266227273281901e-02
4.688449749746688711e-01 1.964556257714445905e-02 5.862776439317262528e-02
4.741516624279997205e-01 1.940234058919213661e-02 5.806380572763907705e-02
4.794118268811014549e-01 1.936335100061424216e-02 5.733101890860898414e-02
4.846131761746131361e-01 1.958983233892800865e-02 5.642063175373675937e-02
4.897434510076072445e-01 2.014613479230887780e-02 5.532001973850570603e-02
4.947824246224634126e-01 2.112807697320261829e-02 5.404462487713374702e-02
4.997137287427875774e-01 2.261949505617124809e-02 5.260678769207213085e-02
5.045221796142561610e-01 2.470021881498402766e-02 5.103037173727244941e-02
5.091962933960687554e-01 2.743497840685808764e-02 4.934858848197953146e-02
5.137301598248652512e-01 3.086487576009750225e-02 4.759930860912273826e-02
5.181239197849397682e-01 3.500462903556054994e-02 4.582042542223261489e-02
5.223830134996687580e-01 3.984501254619848137e-02 4.404711326285257883e-02
5.265167824786487483e-01 4.511655072324017440e-02 4.230178743284147141e-02
5.305361428192386652e-01 5.056764255442364880e-02 4.059909753035081253e-02
5.344521625028915146e-01 5.613230116634507910e-02 3.893707769140806324e-02
5.382767330694612218e-01 6.174820344635036790e-02 3.738224682749225619e-02
5.420185347623404093e-01 6.738177887118912412e-02 3.591446197845812871e-02
5.456872157339706098e-01 7.300021292187203192e-02 3.454024418762975668e-02
5.492896320812740152e-01 7.858770433534650879e-02 3.324672306620826689e-02
5.528333528391162766e-01 8.412806121949528704e-02 3.203530971705802199e-02
5.563218795889670609e-01 8.962115486157959388e-02 3.088637914922974861e-02
5.597632862141976862e-01 9.505189218342754987e-02 2.981373160271517594e-02
5.631590542003003241e-01 1.004271762153811642e-01 2.879663766133408118e-02
3.124295956605860902e-01 3.104471192643529220e-01 3.090926227255514358e-01
3.153354478588585863e-01 3.133524060708574033e-01 3.119821443435121244e-01
3.182430598922476594e-01 3.162596157133579222e-01 3.148735451052371292e-01
3.211522197125276645e-01 3.191685075353222212e-01 3.177665914743306730e-01
3.240629875467020971e-01 3.220791403920337248e-01 3.206613422482794129e-01
3.269754230998080380e-01 3.249915726322321063e-01 3.235578557174154657e-01
3.298895855632509311e-01 3.279058621062435619e-01 3.264561896730452384e-01
3.328058950511413872e-01 3.308224519789740858e-01 3.293567801282767382e-01
3.357242465615515870e-01 3.337412216005199994e-01 3.322595102624680830e-01
3.386445426634580036e-01 3.366620612448586458e-01 3.351642734426954484e-01
3.415668398051758814e-01 3.395850261019035532e-01 3.380711247962143173e-01
3.444911940093728364e-01 3.425101709489803414e-01 3.409801190370675417e-01
3.474177171638533901e-01 3.454376095238633759e-01 3.438913688818530656e-01
3.503469833885533569e-01 3.483679413950338533e-01 3.468054653691610145e-01
3.532785135690075240e-01 3.513006555881026949e-01 3.497219068304560641e-01
3.562123614381408343e-01 3.542358046622061107e-01 3.526407457578920046e-01
3.591485803768168505e-01 3.571734408356729285e-01 3.555620343021813068e-01
3.620872234195166572e-01 3.601136159915630519e-01 3.584858242781328586e-01
3.650286939883333193e-01 3.630567477596149395e-01 3.614125280750120628e-01
3.679730835603966521e-01 3.660029262684281748e-01 3.643422355711175009e-01
3.709200935456273829e-01 3.689518358458999114e-01 3.672746363207957110e-01
3.738697751287151094e-01 3.719035265860498596e-01 3.702097803460631664e-01
3.768221792091647204e-01 3.748580483073254976e-01 3.731477173927215119e-01
3.797773564061294738e-01 3.778154505573166899e-01 3.760884969350717366e-01
3.827360799167192629e-01 3.807765314450937688e-01 3.790329075084533650e-01
3.856977984917309143e-01 3.837407150183698712e-01 3.819803814059496427e-01
3.886624779865983093e-01 3.867079629882915204e-01 3.849308813833784426e-01
3.916301675373726754e-01 3.896783234711069799e-01 3.878844554799001365e-01
3.946009160512599645e-01 3.926518443624523180e-01 3.908411515134125325e-01
3.975750328941100031e-01 3.956288421370338648e-01 3.938012827103044011e-01
4.005529526089701697e-01 3.986097611264988183e-01 3.967652888788774646e-01
4.035341133137877456e-01 4.015940188237794861e-01 3.997325949836182390e-01
4.065185627017746106e-01 4.045816619557017213e-01 4.027032476697736119e-01
4.095063482817839495e-01 4.075727370718815190e-01 4.056772934047228052e-01
4.124975173818349883e-01 4.105672905481647095e-01 4.086547784814160145e-01
4.154927954346919750e-01 4.135660647292051517e-01 4.116364375734101744e-01
4.184918779167747238e-01 4.165687434984514104e-01 4.146219586956935155e-01
4.214945204326190908e-01 4.195750736283389015e-01 4.176110918403175276e-01
4.245007693662425363e-01 4.225851005893354051e-01 4.206038823909587498e-01
4.275106709591262044e-01 4.255988697155072553e-01 4.236003755942250870e-01
4.305243725018209622e-01 4.286165297439478161e-01 4.266007190310782127e-01
4.335427826643468152e-01 4.316390076472571558e-01 4.296058306451354114e-01
4.365650183856102373e-01 4.346653972573469860e-01 4.326148141166248351e-01
4.395911251655309493e-01 4.376957432004656146e-01 4.356277139803868548e-01
4.426211483952462444e-01 4.407300899994052723e-01 4.386445746671270296e-01
4.456551333598003373e-01 4.437684820761266802e-01 4.416654405060406741e-01
4.486936890393982691e-01 4.468115388000440946e-01 4.446909251918003680e-01
4.517368917688296959e-01 4.498593352768152642e-01 4.477211036271617761e-01
4.547842258698517881e-01 4.529113434703738172e-01 4.507554532964671012e-01
4.578357360933586473e-01 4.559676072973268823e-01 4.537940180195341733e-01
4.608914671138729657e-01 4.590281706024677000e-01 4.568368415436758201e-01
4.639514635319542113e-01 4.620930771611265198e-01 4.598839675460500431e-01
4.670168649608225642e-01 4.651634846458155281e-01 4.629365433533288998e-01
4.700867768827386439e-01 4.682384803878612001e-01 4.659936652121621781e-01
4.731611215843889440e-01 4.713179837040911146e-01 4.690552534918061012e-01
4.762399433154002049e-01 4.744020380433919271e-01 4.721213515388643089e-01
4.793232862779120418e-01 4.774906868110462277e-01 4.751920026556298859e-01
4.824115798157278534e-01 4.805843644813184157e-01 4.782676377512943278e-01
4.855054065192261903e-01 4.836836604074202794e-01 4.813488413805353527e-01
4.886039206405141666e-01 4.867877139846760737e-01 4.844347609063475724e-01
4.917071661727567822e-01 4.898965684306006430e-01 4.875254394390606261e-01
4.948151870857158130e-01 4.930102669426385531e-01 4.906209200682110083e-01
4.979280273277140179e-01 4.961288527000837223e-01 4.937212458644600344e-01
5.010466910650995809e-01 4.992533419227356273e-01 4.968274246952877893e-01
5.041707018503041038e-01 5.023832495766817896e-01 4.999389760567756191e-01
5.072996974715958673e-01 5.055182071194114224e-01 5.030555348110034020e-01
5.104337218186610903e-01 5.086582576893806662e-01 5.061771439834861219e-01
5.135728187816994694e-01 5.118034444283539042e-01 5.093038466023047661e-01
5.167171776492902602e-01 5.149539576128636664e-01 5.124358316223290855e-01
5.198680090904711504e-01 5.181110205469275920e-01 5.155743126495593298e-01
5.230240786607408543e-01 5.212733823701987124e-01 5.187180494068470704e-01
5.261854303107180719e-01 5.244410863000215084e-01 5.218670849934367029e-01
5.293521080119285571e-01 5.276141755769954145e-01 5.250214625310645200e-01
5.325241557584816654e-01 5.307926934666159502e-01 5.281812251655980672e-01
5.357023822572596350e-01 5.339774558564434059e-01 5.313471825523317094e-01
5.388868512651764364e-01 5.371685253040656693e-01 5.345193971027960567e-01
5.420768564731617278e-01 5.403651867966881017e-01 5.376971597230371103e-01
5.452724420645790326e-01 5.435674838014370458e-01 5.408805137559964393e-01
5.484736522643940404e-01 5.467754598289500390e-01 5.440695025873386603e-01
5.516805313407564526e-01 5.499891584349245743e-01 5.472641696469991990e-01
5.548945759211054707e-01 5.532100885206570062e-01 5.504660124980879088e-01
5.581145670784500856e-01 5.564370183489838473e-01 5.536738091851505228e-01
5.613403945534086725e-01 5.596698355312789142e-01 5.568874483927889196e-01
5.645721029241054234e-01 5.629085839430609806e-01 5.601069738660463537e-01
5.678097368298002934e-01 5.661533075224889355e-01 5.633324294117918463e-01
5.710538502837967378e-01 5.694045636227955676e-01 5.665643684189753948e-01
5.743051694640114446e-01 5.726630826009092567e-01 5.698035160864106663e-01
5.775625831679842559e-01 5.759277431077964859e-01 5.730487596836393305e-01
5.808261364437535823e-01 5.791985894982362160e-01 5.763001434292895908e-01
5.840958744176996120e-01 5.824756662065079249e-01 5.795577116206488411e-01
5.873718422959868235e-01 5.857590177478061433e-01 5.828215086350755358e-01
5.906553283643478647e-01 5.890499401540877367e-01 5.860928212875072818e-01
5.939456863496749284e-01 5.923477811162676243e-01 5.893710023062436187e-01
5.972424456095442036e-01 5.956520656886060694e-01 5.926555803959596691e-01
6.005456518693509382e-01 5.989628389111977258e-01 5.959466004538667150e-01
6.038553509508688411e-01 6.022801459213279474e-01 5.992441074734932149e-01
6.071717820521228548e-01 6.056042263837068118e-01 6.025483395942317077e-01
6.104964593110322468e-01 6.089366020846007643e-01 6.058608080471198365e-01
6.138278031916972743e-01 6.122756828005896379e-01 6.091799341380006449e-01
6.171658601273857236e-01 6.156215142843693977e-01 6.125057634705911580e-01
6.205106766639508775e-01 6.189741424017308757e-01 6.158383417607966592e-01
6.238622994612079697e-01 6.223336131329110232e-01 6.191777148380593898e-01
6.272217495526114517e-01 6.257009516987572795e-01 6.225249009896979269e-01
6.305890773753839440e-01 6.290762073460191006e-01 6.258799493857725160e-01
6.339633886626701020e-01 6.324584802535679895e-01 6.292419666367617959e-01
6.373447307909330117e-01 6.358478171202772700e-01 6.326109992853028441e-01
6.407331512665517259e-01 6.392442647751326312e-01 6.359870940032011921e-01
6.441286977271819536e-01 6.426478701785689474e-01 6.393702975927646470e-01
6.475332468520634821e-01 6.460605168712778568e-01 6.427624810073024886e-01
6.509452405110124351e-01 6.494806387834576311e-01 6.461620890521664329e-01
6.543645413031805580e-01 6.529080970404893547e-01 6.495689839578402403e-01
6.577911976868459076e-01 6.563429394235856806e-01 6.529832133417282636e-01
6.612252582674151835e-01 6.597852138609858752e-01 6.564048249672772561e-01
6.646674156910871156e-01 6.632356145596635111e-01 6.598345085785605191e-01
6.681185471134806209e-01 6.666950204492557708e-01 6.632731374913205524e-01
6.715772676011302345e-01 6.701620407069336371e-01 6.667193303275074090e-01
6.750436266693425891e-01 6.736367241694728447e-01 6.701731357532019207e-01
6.785176739964410286e-01 6.771191198362991326e-01 6.736346025961090778e-01
6.819994594251277409e-01 6.806092768708275065e-01 6.771037798468937385e-01
6.854905830673579947e-01 6.841087989309356798e-01 6.805822608627878356e-01
6.889902152593658791e-01 6.876168527353629933e-01 6.840692180364433828e-01
6.924977753545575876e-01 6.911328551431065170e-01 6.875640721906080000e-01
6.960133142099522718e-01 6.946568563288202380e-01 6.910668733210059145e-01
6.995368828630013081e-01 6.981889066469423311e-01 6.945776716020720665e-01
7.030687808865530020e-01 7.017293055249038680e-01 6.980967646835468665e-01
7.066108508007371558e-01 7.052798983395075494e-01 7.016259863087929149e-01
7.101611448678148975e-01 7.088387320594561336e-01 7.051633961675597417e-01
7.137197152305658765e-01 7.124058581399919099e-01 7.087090455286524371e-01
7.172866142285043711e-01 7.159813282321918493e-01 7.122629858555886706e-01
7.208618943992978201e-01 7.195651941843620447e-01 7.158252688079900228e-01
7.244468165907456125e-01 7.231587179839266133e-01 7.193971485631489582e-01
7.280414152493157598e-01 7.267619329076518975e-01 7.229786584068481625e-01
7.316445953770737143e-01 7.303737413779746168e-01 7.265687078387530473e-01
7.352564107249326408e-01 7.339941964502680882e-01 7.301673497187975981e-01
7.388769152586618372e-01 7.376233513935839259e-01 7.337746371194190376e-01
7.425061631603386836e-01 7.412612596920828967e-01 7.373906233269835742e-01
7.461464606086873896e-01 7.449102288687008411e-01 7.410176017144567417e-01
7.497958701755464972e-01 7.485683184324932737e-01 7.446536440345916663e-01
7.534542299137884092e-01 7.522353655092588465e-01 7.482985885600491605e-01
7.571215953297079082e-01 7.559114248995535412e-01 7.519524898864893281e-01
7.607980221631284179e-01 7.595965516360115855e-01 7.556154028404162837e-01
7.644843645869688897e-01 7.632915995427993483e-01 7.592881761602847668e-01
7.681816678947460675e-01 7.669976132758333787e-01 7.629718483168250742e-01
7.718882453269261257e-01 7.707129043795911683e-01 7.666647413348012252e-01
7.756041540460802963e-01 7.744375293002100369e-01 7.703669114468018053e-01
7.793294514661021077e-01 7.781715447334471847e-01 7.740784151337410934e-01
7.830641952537596984e-01 7.819150076262105786e-01 7.777993091263859515e-01
7.868103454866590685e-01 7.856698771694257122e-01 7.815315409597441976e-01
7.905668625004234062e-01 7.894351125938581237e-01 7.852740763344663311e-01
7.943330461869060333e-01 7.932100130488932699e-01 7.890262187563090723e-01
7.981089557544192026e-01 7.969946370120729595e-01 7.927880264788568221e-01
8.018946506826831744e-01 8.007890432303661488e-01 7.965595580237599949e-01
8.056905050124041345e-01 8.045936048548131270e-01 8.003411844486416626e-01
8.094987355104209525e-01 8.084105365473570615e-01 8.041351074326460457e-01
8.133169785258600459e-01 8.122374748676888245e-01 8.079389777476483347e-01
8.171452951864401903e-01 8.160744801984102237e-01 8.117528555427496739e-01
8.209837469106039398e-01 8.199216132106035282e-01 8.155768012541027012e-01
8.248323954092128263e-01 8.237789348655052413e-01 8.194108756065791033e-01
8.286927840915763177e-01 8.276479863786950775e-01 8.232566109078919281e-01
8.325649270796637857e-01 8.315287807573953360e-01 8.271140202815062681e-01
8.364475027756378411e-01 8.354199968604651971e-01 8.309817904707684777e-01
8.403405746677916621e-01 8.393216974124767527e-01 8.348599839552471868e-01
8.442442065571257315e-01 8.432339454481773133e-01 8.387486635231791166e-01
8.481584625591254145e-01 8.471568043142441162e-01 8.426478922732166810e-01
8.520861550384412064e-01 8.510930816852287917e-01 8.465604617719046932e-01
8.560249024716682920e-01 8.550403982451367790e-01 8.504840066608007065e-01
8.599745194151316996e-01 8.589985681043337262e-01 8.544183422480284840e-01
8.639350718981159138e-01 8.629676565074151373e-01 8.583635335210283435e-01
8.679066262857828562e-01 8.669477290320206642e-01 8.623196457986929131e-01
8.718902314626645333e-01 8.709398320337272681e-01 8.662877195686166099e-01
8.758871280259100667e-01 8.749452028944590953e-01 8.702689853121506891e-01
8.798952803818768098e-01 8.789618087644507360e-01 8.742614219551688715e-01
8.839147569330807475e-01 8.829897172412359829e-01 8.782650968268708169e-01
8.879456264404302912e-01 8.870289962776199966e-01 8.822800776100528930e-01
8.919879580252079165e-01 8.910797141836338531e-01 8.863064323430556168e-01
8.960441427301344408e-01 8.951442561280015253e-01 8.903465328361804820e-01
9.001128904953542564e-01 8.992213337528862649e-01 8.943990982004267343e-01
9.041933650584489390e-01 9.033101118731188262e-01 8.984632980967039995e-01
9.082856377297909845e-01 9.074106609684426950e-01 9.025392027229122149e-01
9.123897802040108473e-01 9.115230518994157372e-01 9.066268826559935601e-01
9.165062601210144377e-01 9.156477504829055869e-01 9.107268012198581619e-01
9.206377372962850636e-01 9.197874089604549663e-01 9.148415959800745290e-01
9.247813586963474775e-01 9.239391805138976732e-01 9.189684361991556916e-01
9.289371983431667923e-01 9.281031383100254439e-01 9.231073947496013510e-01
9.331053306682361992e-01 9.322793559214043180e-01 9.272585449078167841e-01
9.372858305148153990e-01 9.364679073285836486e-01 9.314219603563131944e-01
9.414805849385761150e-01 9.406706735323028257e-01 9.355995118038626934e-01
9.456895815303881792e-01 9.448876411191396985e-01 9.397911860891489111e-01
9.499112324285693409e-01 9.491172259914930631e-01 9.439954080097928690e-01
9.541456149743491322e-01 9.533595046046891008e-01 9.482122537117517735e-01
9.583928069476742584e-01 9.576145538486544595e-01 9.524417997736036590e-01
9.626528865695892501e-01 9.618824510502640424e-01 9.566841232088870717e-01
9.717386353800526733e-01 9.973777600608618732e-01 4.115658643585359822e-01
9.662006602777347686e-01 9.937266459819259490e-01 4.023251811958163393e-01
9.607627521945550919e-01 9.900572267174857499e-01 3.929085929665734889e-01
9.554415272451376451e-01 9.863634202187735456e-01 3.833020851902597670e-01
9.502559385734440367e-01 9.826387048868144847e-01 3.734797610708937010e-01
9.452247574758879667e-01 9.788787426033022099e-01 3.633572044040755400e-01
9.403922547690045652e-01 9.750651532690909340e-01 3.529658674736868007e-01
9.358012781940933111e-01 9.711846436929117976e-01 3.422125309180296115e-01
9.315219006576069827e-01 9.672115998149192206e-01 3.310553617148976335e-01
9.276515136764329483e-01 9.631091768055900504e-01 3.194750358456411865e-01
9.243262891934023173e-01 9.588257749438183764e-01 3.074483567011621621e-01
9.217020058646493430e-01 9.542951650631289473e-01 2.951579532336194300e-01
9.198887370469149838e-01 9.494613758630886524e-01 2.829933059330041467e-01
9.188014423728452229e-01 9.443342663663614189e-01 2.716012421636955776e-01
9.181820869339853974e-01 9.389970698605141219e-01 2.614202290916152749e-01
9.177841550998647735e-01 9.335426933786861170e-01 2.524800001657211412e-01
9.174401026070885257e-01 9.280407871647589069e-01 2.446155106635521936e-01
9.170895513231805962e-01 9.225221346800714928e-01 2.375965624445583457e-01
9.167020828694254497e-01 9.170043397013648567e-01 2.312467382449614473e-01
9.162674970718558409e-01 9.114958340077099486e-01 2.254318760810480882e-01
9.157691900066246005e-01 9.060063912564605415e-01 2.200571360818813216e-01
9.152197058922081352e-01 9.005338027659569589e-01 2.150417998700874245e-01
9.146241645334555193e-01 8.950779708690549397e-01 2.103289435929298068e-01
9.139711070576973517e-01 8.896449914470077047e-01 2.058761237663177535e-01
9.132755157040669536e-01 8.842301350859270714e-01 2.016464740967732761e-01
9.125407126803214419e-01 8.788330083046588248e-01 1.976125145460150279e-01
9.117530169460994482e-01 8.734600806229881886e-01 1.937497985711440218e-01
9.109311963239964394e-01 8.681043338417845368e-01 1.900403276536027364e-01
9.100752049629435847e-01 8.627663273889324413e-01 1.864683841517677521e-01
9.091854095601151764e-01 8.574463833033296734e-01 1.830203816986048226e-01
9.082549005100644113e-01 8.521478984780772592e-01 1.796821188488442467e-01
9.072929210165844305e-01 8.468673217860986924e-01 1.764454502969030336e-01
9.063013792870692198e-01 8.416041433916346959e-01 1.733019473598792493e-01
9.052811273728736952e-01 8.363582655293054779e-01 1.702439219249277103e-01
9.042330703505969680e-01 8.311295331123278451e-01 1.672645945644082588e-01
9.031581378443196417e-01 8.259177493137719317e-01 1.643579680291452627e-01
9.020572628789119696e-01 8.207226877358885231e-01 1.615187200622499042e-01
9.009313663687085194e-01 8.155441018959304067e-01 1.587421124067343314e-01
8.997813459322432372e-01 8.103817325993176723e-01 1.560239134014781981e-01
8.986080680213675009e-01 8.052353136482993667e-01 1.533603319950107013e-01
8.974123625824063089e-01 8.001045762385368132e-01 1.507479613684808151e-01
8.961950196441476058e-01 7.949892523212133133e-01 1.481837306596257842e-01
8.949567873642767513e-01 7.898890771497076857e-01 1.456648635287154658e-01
8.936983711718720080e-01 7.848037911839249592e-01 1.431888425141153720e-01
8.924204337256359532e-01 7.797331414892141321e-01 1.407533782964209279e-01
8.911216151255343387e-01 7.746778361440714855e-01 1.383548030403051743e-01
8.898020181760183389e-01 7.696378889985142635e-01 1.359906608477614021e-01
8.884645911096110682e-01 7.646119206511362565e-01 1.336610605915157513e-01
8.871098774237282658e-01 7.595996914637694886e-01 1.313644302926826601e-01
8.857383791003278217e-01 7.546009727066200767e-01 1.290993233143932217e-01
8.843505587935225343e-01 7.496155463925207041e-01 1.268644048815629477e-01
8.829468419986222782e-01 7.446432050409976799e-01 1.246584402943621550e-01
8.815249085514379468e-01 7.396851204434677918e-01 1.224776444651401730e-01
8.800852120157934833e-01 7.347410842227017902e-01 1.203208501022957344e-01
8.786308879175975806e-01 7.298095076593623665e-01 1.181897480034229675e-01
8.771622516330287445e-01 7.248902111580254326e-01 1.160834456649006141e-01
8.756795889485279316e-01 7.199830249122411985e-01 1.140011085329386831e-01
8.741818688473920185e-01 7.150884618420272343e-01 1.119405415999485953e-01
8.726664701310860028e-01 7.102078845126149620e-01 1.098977185964423176e-01
8.711380365425036576e-01 7.053388457490094021e-01 1.078767213026572136e-01
8.695967618761146767e-01 7.004812035381728919e-01 1.058769266693721234e-01
8.680428176626151515e-01 6.956348241211850469e-01 1.038977462027144416e-01
8.664728647912948167e-01 6.908014665312249836e-01 1.019343675562955354e-01
8.648889520799001307e-01 6.859800077414488495e-01 9.998838514326469085e-02];

   case 'sol' 
      RGB = [2.014250997833959556e-01 7.730778455372402935e-02 9.342024025258441333e-02
2.062319592710875060e-01 7.906207768979725548e-02 9.541606071998920413e-02
2.110428892884436691e-01 8.079536357043612393e-02 9.737846202556904585e-02
2.158580822956807643e-01 8.250806203226229707e-02 9.930725029816905858e-02
2.206777103783343352e-01 8.420058748958639261e-02 1.012022063667477878e-01
2.255019582885565144e-01 8.587332122435217818e-02 1.030630633154644982e-01
2.303310909623697555e-01 8.752655129320324745e-02 1.048894558390352783e-01
2.351651244914957362e-01 8.916077192669072393e-02 1.066811567589351084e-01
2.400041538413338893e-01 9.077639059472936145e-02 1.084378476867565166e-01
2.448482542469172518e-01 9.237381813988079782e-02 1.101591880324293282e-01
2.496974811012195872e-01 9.395347052392380438e-02 1.118448155598189275e-01
2.545518697677066067e-01 9.551577058896976169e-02 1.134943469265998206e-01
2.594114353201673606e-01 9.706114983944272301e-02 1.151073782157037984e-01
2.642761722125430146e-01 9.859005025060077476e-02 1.166834854654446307e-01
2.691460538812255332e-01 1.001029261086328892e-01 1.182222252052740896e-01
2.740210322821139965e-01 1.016002458867145131e-01 1.197231350040377507e-01
2.789010373645949281e-01 1.030824941607314804e-01 1.211857340375411263e-01
2.837859764845531707e-01 1.045501735676682975e-01 1.226095236822268009e-01
2.886757337585150185e-01 1.060038068088898222e-01 1.239939881417738143e-01
2.935705596266552364e-01 1.074435592047325860e-01 1.253382180210721730e-01
2.984708123016127645e-01 1.088694832683588798e-01 1.266411225341654012e-01
3.033754705455499190e-01 1.102829776040219478e-01 1.279029157767591629e-01
3.082843160981594277e-01 1.116846670615513903e-01 1.291230138435368602e-01
3.131971029671865581e-01 1.130752078289798124e-01 1.303008196341493008e-01
3.181135564546491157e-01 1.144552897453308959e-01 1.314357238709698839e-01
3.230337594016901570e-01 1.158252511924781902e-01 1.325266762725714476e-01
3.279584359813157990e-01 1.171847867104950369e-01 1.335718110238658662e-01
3.328858414856893377e-01 1.185360731157172887e-01 1.345719725768937669e-01
3.378155708245871303e-01 1.198799635036237599e-01 1.355265097514237183e-01
3.427471842250678358e-01 1.212173573516503733e-01 1.364347662233350855e-01
3.476817676568914606e-01 1.225476080130979417e-01 1.372941239039429107e-01
3.526179187878998866e-01 1.238726093833323050e-01 1.381049156392651633e-01
3.575544570498519481e-01 1.251940557912539775e-01 1.388672386602565201e-01
3.624908511464776661e-01 1.265129959708437557e-01 1.395803418854749678e-01
3.674287950126607916e-01 1.278281902872998865e-01 1.402403192837867185e-01
3.723652245912653647e-01 1.291433283978542790e-01 1.408496559025098349e-01
3.772994043575393919e-01 1.304597230378949646e-01 1.414077190879147183e-01
3.822328244976108125e-01 1.317763828069311027e-01 1.419104659088263887e-01
3.871627501840784191e-01 1.330967329073056149e-01 1.423599677282273157e-01
3.920880611619059208e-01 1.344225444091708133e-01 1.427559829693639537e-01
3.970102845366911826e-01 1.357528321160993978e-01 1.430939047058796265e-01
4.019257908029647552e-01 1.370920335458912787e-01 1.433773252895977068e-01
4.068349561616130816e-01 1.384404082220755672e-01 1.436033049025308150e-01
4.117365731263126216e-01 1.397998954736640875e-01 1.437714012378476514e-01
4.166285955020519460e-01 1.411733354490503911e-01 1.438826947863026673e-01
4.215115464927484679e-01 1.425608842716513391e-01 1.439334793454142836e-01
4.263818625509953053e-01 1.439670074541691269e-01 1.439276912848104217e-01
4.312408181144407604e-01 1.453911021832318962e-01 1.438598606513883382e-01
4.360843175580979003e-01 1.468382080621888075e-01 1.437352070106750757e-01
4.409133399041038626e-01 1.483080544589621963e-01 1.435484892033262738e-01
4.457240620502335715e-01 1.498053889889406909e-01 1.433047202671754727e-01
4.505169861034231116e-01 1.513304477119505886e-01 1.429993558897900696e-01
4.552886678338638471e-01 1.528875528172430887e-01 1.426369447211624508e-01
4.600387072749128614e-01 1.544778583065680166e-01 1.422148901612839134e-01
4.647645669051368444e-01 1.561046932382532393e-01 1.417359354431781127e-01
4.694647696847664142e-01 1.577702615986552870e-01 1.412003046328143552e-01
4.741377460736882488e-01 1.594768244179092798e-01 1.406085235372486197e-01
4.787811635810699795e-01 1.612273475797827982e-01 1.399635953741071304e-01
4.833940392196432456e-01 1.630233960258756498e-01 1.392646788568434324e-01
4.879740080405346458e-01 1.648678210719416448e-01 1.385155272254653958e-01
4.925195693223419213e-01 1.667625387560449646e-01 1.377175480485133274e-01
4.970292850743541213e-01 1.687093212494573113e-01 1.368722045288005285e-01
5.015013418271374590e-01 1.707101850535491239e-01 1.359828524777431658e-01
5.059344608615837791e-01 1.727665378729273671e-01 1.350513348829507354e-01
5.103272555076001638e-01 1.748797680652766917e-01 1.340804814887887886e-01
5.146784999821305551e-01 1.770509926066398676e-01 1.330731030668735604e-01
5.189871207659747521e-01 1.792810660321904059e-01 1.320320170625302603e-01
5.232521628013230286e-01 1.815706081303557207e-01 1.309602397868673829e-01
5.274729145399480457e-01 1.839199144444812750e-01 1.298602350351042156e-01
5.316485781433754054e-01 1.863291944433481395e-01 1.287360215107217765e-01
5.357788119794710813e-01 1.887982003317690283e-01 1.275896151617726404e-01
5.398632980453529351e-01 1.913265486380361924e-01 1.264236887032485768e-01
5.439016755531911329e-01 1.939137144580994065e-01 1.252427388260835273e-01
5.478941342086204314e-01 1.965587857098634594e-01 1.240472431523338970e-01
5.518406011876045847e-01 1.992608233247842864e-01 1.228416287224595171e-01
5.557413604777560190e-01 2.020186434106095441e-01 1.216276788300265921e-01
5.595967825163993270e-01 2.048309740311727856e-01 1.204067641590413351e-01
5.634071785471536087e-01 2.076963248023981912e-01 1.191839172318373274e-01
5.671732069794859221e-01 2.106133447776186718e-01 1.179569341314256059e-01
5.708953502475510033e-01 2.135801749668952421e-01 1.167331750031382620e-01
5.745743340603017835e-01 2.165955297740416174e-01 1.155086834950974917e-01
5.782108774062354462e-01 2.196573476346848075e-01 1.142903383669127693e-01
5.818056681759461446e-01 2.227643917275112795e-01 1.130745297734544186e-01
5.853596324247829497e-01 2.259144368785390156e-01 1.118679346243981187e-01
5.888733531729506421e-01 2.291064977832514282e-01 1.106657061174652901e-01
5.923480147332496060e-01 2.323380142637169188e-01 1.094762267036840098e-01
5.957840241257977842e-01 2.356083643285451501e-01 1.082930207415330470e-01
5.991826510896112179e-01 2.389150535758420846e-01 1.071233989765111305e-01
6.025444033584952397e-01 2.422572079085022478e-01 1.059638000704014038e-01
6.058701491001592387e-01 2.456332409492081315e-01 1.048158025498839019e-01
6.091610396494541169e-01 2.490411636600449730e-01 1.036832083023569995e-01
6.124172967183565408e-01 2.524805655496659695e-01 1.025613962727195683e-01
6.156402174450263942e-01 2.559493015306805175e-01 1.014554136320146294e-01
6.188304674185852727e-01 2.594462946423264360e-01 1.003647858381197522e-01
6.219883549579363624e-01 2.629709881912258296e-01 9.928701461787994842e-02
6.251150949354874475e-01 2.665215196417168309e-01 9.822607331815666476e-02
6.282114705640686747e-01 2.700967151118268128e-01 9.718275325572381385e-02
6.312775931157122988e-01 2.736963322450938207e-01 9.615407260814298751e-02
6.343140968307461325e-01 2.773194022482146082e-01 9.514062524056160486e-02
6.373225940474543938e-01 2.809637253536654833e-01 9.414820920814792604e-02
6.403029049301643960e-01 2.846294803262617856e-01 9.317287627515624671e-02
6.432554503146770131e-01 2.883160464262430889e-01 9.221435672465166933e-02
6.461807870545831500e-01 2.920226445998113096e-01 9.127322088316058846e-02
6.490798649799141007e-01 2.957480457349839220e-01 9.035190862912848009e-02
6.519534890851086395e-01 2.994912417888541123e-01 8.945207804511770555e-02
6.548014840508611378e-01 3.032523985397311828e-01 8.857115779395227650e-02
6.576243271404891289e-01 3.070309030666096284e-01 8.770973119670061324e-02
6.604224780876138956e-01 3.108261780844247535e-01 8.686839073952234980e-02
6.631963792788750922e-01 3.146376804982159436e-01 8.604773901491818977e-02
6.659464950830153995e-01 3.184648571182398524e-01 8.524854019761693436e-02
6.686740068110078594e-01 3.223063950997894223e-01 8.447431908919650345e-02
6.713784685540787889e-01 3.261627733896342862e-01 8.372242463404078183e-02
6.740602498156743616e-01 3.300335755312562847e-01 8.299350184989945367e-02
6.767197045307955516e-01 3.339184117562906673e-01 8.228820834254704786e-02
6.793571714405546302e-01 3.378169176731946055e-01 8.160721412236313088e-02
6.819729744807380145e-01 3.417287529939577184e-01 8.095120122856311329e-02
6.845674231809769639e-01 3.456536003014257785e-01 8.032086315939043764e-02
6.871408130714994345e-01 3.495911638591582271e-01 7.971690410726239850e-02
6.896934260947393813e-01 3.535411684651399988e-01 7.914003799874758105e-02
6.922255310194272981e-01 3.575033583501715517e-01 7.859098734029026923e-02
6.947373838550107150e-01 3.614774961213378379e-01 7.807048187183049381e-02
6.972292282645524697e-01 3.654633617505798626e-01 7.757925703187978916e-02
6.997012959744630667e-01 3.694607516080961052e-01 7.711805223920600860e-02
7.021538071796381564e-01 3.734694775400307920e-01 7.668760899803153674e-02
7.045869709427744487e-01 3.774893659897066711e-01 7.628866883556534306e-02
7.070009855867870341e-01 3.815202571614760840e-01 7.592197108269088668e-02
7.093960390794303850e-01 3.855620042261392877e-01 7.558825051079040569e-02
7.117723094093527658e-01 3.896144725667428643e-01 7.528823483980665032e-02
7.141299649529336824e-01 3.936775390635336991e-01 7.502264213486334321e-02
7.164691648313716854e-01 3.977510914167320943e-01 7.479217811087204848e-02
7.187900592575979797e-01 4.018350275057843701e-01 7.459753336661339995e-02
7.210927898726463559e-01 4.059292547837213827e-01 7.443938057163165811e-02
7.233774900712311995e-01 4.100336897052239138e-01 7.431837163097509968e-02
7.256442853163093121e-01 4.141482571870220286e-01 7.423513485418420377e-02
7.278932934424962031e-01 4.182728900992377374e-01 7.419027215599127700e-02
7.301246249482277184e-01 4.224075287863344741e-01 7.418435631686179366e-02
7.323383832766364732e-01 4.265521206163164214e-01 7.421792833176479864e-02
7.345353882716817440e-01 4.307060349567022883e-01 7.429319537753659164e-02
7.367156646423576039e-01 4.348693003618859243e-01 7.441039153688730479e-02
7.388787494489402752e-01 4.390423308641819844e-01 7.456862859096566321e-02
7.410247212778052761e-01 4.432250991934941209e-01 7.476828631581519669e-02
7.431536528151155840e-01 4.474175832571065103e-01 7.500970147240951236e-02
7.452656110665121236e-01 4.516197658271001170e-01 7.529316628527241151e-02
7.473606575657232298e-01 4.558316342481479322e-01 7.561892717684162712e-02
7.494388485722311977e-01 4.600531801646510766e-01 7.598718377039928584e-02
7.515014494005359813e-01 4.642834853101814319e-01 7.640057068894035019e-02
7.535482386381918696e-01 4.685227724793568638e-01 7.685851802683341116e-02
7.555784292647326206e-01 4.727716755322179387e-01 7.735934440269953694e-02
7.575920584303713623e-01 4.770302023787819645e-01 7.790304647924226056e-02
7.595891587998102601e-01 4.812983641144596425e-01 7.848957351759905388e-02
7.615697586917992146e-01 4.855761748409669898e-01 7.911882810663906085e-02
7.635351559772435293e-01 4.898627402128037933e-01 7.979298836587264687e-02
7.654855818523200739e-01 4.941579525945941076e-01 8.051210547168902165e-02
7.674196884953543574e-01 4.984628195279873775e-01 8.127340285315723389e-02
7.693374880099380642e-01 5.027773670563000508e-01 8.207659805554132215e-02
7.712389886872909051e-01 5.071016234993879213e-01 8.292136945254757752e-02
7.731249880551478437e-01 5.114350748098054344e-01 8.380866202872513937e-02
7.749971634096607387e-01 5.157766247025432627e-01 8.474069791771285387e-02
7.768531656759695148e-01 5.201279334240793695e-01 8.571310136491594456e-02
7.786929885023088360e-01 5.244890387928055064e-01 8.672540635712724932e-02
7.805166220614638828e-01 5.288599804136482341e-01 8.777711918476124864e-02
7.823261306937436821e-01 5.332394286426104246e-01 8.887078595429290240e-02
7.841211261331867410e-01 5.376277019226174403e-01 9.000512625428583324e-02
7.859000063657322066e-01 5.420258982280999893e-01 9.117718483670700369e-02
7.876627478111991598e-01 5.464340650106903619e-01 9.238637445837943885e-02
7.894103755328435446e-01 5.508515783238705499e-01 9.363351104642655964e-02
7.911449489936349666e-01 5.552771776345648558e-01 9.492063714767667859e-02
7.928634178435132185e-01 5.597128604871853819e-01 9.624295938665472505e-02
7.945657449982463927e-01 5.641586812836011378e-01 9.759983714409603550e-02
7.962529317489702718e-01 5.686140495102688375e-01 9.899190462101559174e-02
7.979274645055595139e-01 5.730774789890704657e-01 1.004214627980108965e-01
7.995858497679293464e-01 5.775511834408552092e-01 1.018835148330231588e-01
8.012280368308171141e-01 5.820352236202099849e-01 1.033774013447087969e-01
8.028560740659145267e-01 5.865283957952628358e-01 1.049048199319871921e-01
8.044707409966546097e-01 5.910302911313597418e-01 1.064658666440542745e-01
8.060691629501393063e-01 5.955426815778210869e-01 1.080566579062864629e-01
8.076512751425731773e-01 6.000656338781130694e-01 1.096765425113208070e-01
8.092212933019986565e-01 6.045967120149606799e-01 1.113292481825315650e-01
8.107760556043207556e-01 6.091378327200270837e-01 1.130107512069984665e-01
8.123144185459451050e-01 6.136896959567099685e-01 1.147193037693588491e-01
8.138381395305365196e-01 6.182513237945769236e-01 1.164560306092000397e-01
8.153492559398581863e-01 6.228216087115111543e-01 1.182221687055251269e-01
8.168438497700267753e-01 6.274028329685673588e-01 1.200133807424188237e-01
8.183218295752512361e-01 6.319950736315883555e-01 1.218290715633482957e-01
8.197890431479778472e-01 6.365951069307501653e-01 1.236738150663118929e-01
8.212397496268510899e-01 6.412061948148843893e-01 1.255419007966243339e-01
8.226736710596913582e-01 6.458285162251576894e-01 1.274326125224898576e-01
8.240954168836683857e-01 6.504595882304867738e-01 1.293492244766023536e-01
8.255020135816337756e-01 6.551010940717902908e-01 1.312886736975442947e-01
8.268916161790003105e-01 6.597540661653059635e-01 1.332490015865780419e-01
8.282680782241739204e-01 6.644164751020815718e-01 1.352327076170389830e-01
8.296302321797530688e-01 6.690890043064162684e-01 1.372383190289331867e-01
8.309751426979781197e-01 6.737732479628292248e-01 1.392631732853782667e-01
8.323064093623322446e-01 6.784673500649637257e-01 1.413094448639914957e-01
8.336236702662669362e-01 6.831715585604193341e-01 1.433763438952303237e-01
8.349233956754664732e-01 6.878877450853391196e-01 1.454609632889922177e-01
8.362094522088989734e-01 6.926139541357441143e-01 1.475655370255327115e-01
8.374812451634525701e-01 6.973505478666088830e-01 1.496891818583828970e-01
8.387351652304512184e-01 7.020993985062000675e-01 1.518291366884736149e-01
8.399758860500933233e-01 7.068581852631129481e-01 1.539880238042938276e-01
8.412015039419409312e-01 7.116279196951651453e-01 1.561642027825314583e-01
8.424088634362330019e-01 7.164102051502189150e-01 1.583553901645263495e-01
8.436040043728154636e-01 7.212020966440341185e-01 1.605648786739153067e-01
8.447825998855326146e-01 7.260057762544317450e-01 1.627897017014220959e-01
8.459425004297633777e-01 7.308223161702667170e-01 1.650283355910636407e-01
8.470916909310660659e-01 7.356478948636396842e-01 1.672849872513207936e-01
8.482222682093125687e-01 7.404863725972353761e-01 1.695548732398838021e-01
8.493354922705429466e-01 7.453371669715372905e-01 1.718384513727829743e-01
8.504363950839656239e-01 7.501979378252857655e-01 1.741381147288197651e-01
8.515178010463546610e-01 7.550721145067320617e-01 1.764497766633903175e-01
8.525841071331127230e-01 7.599576641399760080e-01 1.787754909642916834e-01
8.536351063891680635e-01 7.648547324583224727e-01 1.811148665949305070e-01
8.546660216706046809e-01 7.697655561558608417e-01 1.834652953721074564e-01
8.556846647048813592e-01 7.746865865799383855e-01 1.858303883039418847e-01
8.566843283862169978e-01 7.796209322090990046e-01 1.882066457578640828e-01
8.576654697805085048e-01 7.845684046728831351e-01 1.905941213680029112e-01
8.586333372669123776e-01 7.895266856020336510e-01 1.929950487363375766e-01
8.595800514528921799e-01 7.944993343355386539e-01 1.954056082908571246e-01
8.605118919067250571e-01 7.994835879345245644e-01 1.978284835821235155e-01
8.614257659190140970e-01 8.044808542070145396e-01 2.002621028413892890e-01
8.623187318174559968e-01 8.094924397675259398e-01 2.027050514589471275e-01
8.631990866135879070e-01 8.145147714844941378e-01 2.051607456378260474e-01
8.640570316077024193e-01 8.195521241156307202e-01 2.076248627732380503e-01
8.648985892446832136e-01 8.246019286985761809e-01 2.100998205615737802e-01
8.657217542057287218e-01 8.296650789343075205e-01 2.125846498551076103e-01
8.665227359996962031e-01 8.347431960854363453e-01 2.150777232526754901e-01
8.673102956545263309e-01 8.398326248425269647e-01 2.175824369796515168e-01
8.680739840584389411e-01 8.449377672354075886e-01 2.200945521598048904e-01
8.688209397597312922e-01 8.500556721057834775e-01 2.226167862486261861e-01
8.695475257414353454e-01 8.551878730451877297e-01 2.251476450914057237e-01
8.702518466222823879e-01 8.603351573784762119e-01 2.276863618988967386e-01
8.709403456886846140e-01 8.654949292658645765e-01 2.302352780671254040e-01
8.716032871795772463e-01 8.706711460791707324e-01 2.327907403748207260e-01
8.722504095804027857e-01 8.758599106230183784e-01 2.353562666294088945e-01
8.728735089177709350e-01 8.810645242778994968e-01 2.379288443348342996e-01
8.734755480931443161e-01 8.862838169179700909e-01 2.405095194908647427e-01
8.740576468844427627e-01 8.915173698391286594e-01 2.430986533749082690e-01
8.746134519571161503e-01 8.967676725126847437e-01 2.456940353088591378e-01
8.751533192048812637e-01 9.020307187484648548e-01 2.482992135687792645e-01
8.756648875509137619e-01 9.073112946820588443e-01 2.509099777331614822e-01
8.761579961220332669e-01 9.126056308170312770e-01 2.535296558873600947e-01
8.766251762717257590e-01 9.179165870428791507e-01 2.561557673694155324e-01
8.770689942806796369e-01 9.232431892037369359e-01 2.587891932203917889e-01
8.774903570045399226e-01 9.285851079338152125e-01 2.614302497362179234e-01
8.778834706355089779e-01 9.339444999568264905e-01 2.640771174844134261e-01
8.782575035190057777e-01 9.393179743031147000e-01 2.667327577149610218e-01
8.785999460636340075e-01 9.447101339348284998e-01 2.693932729818908633e-01
8.789235199162210854e-01 9.501163253664915986e-01 2.720626987422674059e-01
8.792160143882136181e-01 9.555409855460589297e-01 2.747373107314689533e-01
8.794851333774609259e-01 9.609813222108515296e-01 2.774195521504189688e-01
8.797259382267708094e-01 9.664391027363387066e-01 2.801079938733134767e-01
8.799388863582034981e-01 9.719141474258771174e-01 2.828028665891310078e-01
8.801261631980005218e-01 9.774056784469543624e-01 2.855049064351531940e-01
8.802811281654138176e-01 9.829160047653974219e-01 2.882122575277183407e-01
8.804129372223431504e-01 9.884419266545670935e-01 2.909276978215925569e-01
8.805080058500511786e-01 9.939881188401472611e-01 2.936474048368232226e-01]; 

   case 'pha' 
      RGB = [6.583083928922510708e-01 4.699391690315133929e-01 4.941288203988051381e-02
6.643374189373471017e-01 4.662019008569991407e-01 5.766473450402211792e-02
6.702086925052345157e-01 4.624801381219734719e-01 6.534560309537773559e-02
6.760429905334627287e-01 4.586983759956768103e-01 7.273174322210870790e-02
6.817522846284524984e-01 4.549140651836585669e-01 7.979261956680192003e-02
6.874028047282801923e-01 4.510841669914616436e-01 8.667102950867147659e-02
6.929504980948593129e-01 4.472389296506211198e-01 9.335868960148416273e-02
6.984261912087648128e-01 4.433576785927097474e-01 9.992839268327721736e-02
7.038122981036579739e-01 4.394532762349419586e-01 1.063871045640915336e-01
7.091206923190102041e-01 4.355176525107516405e-01 1.127717449252612913e-01
7.143452449012380745e-01 4.315557562030358230e-01 1.190934789448820086e-01
7.194928861674689813e-01 4.275627171694253437e-01 1.253760590955056708e-01
7.245561927479047259e-01 4.235446971269447580e-01 1.316232516304018385e-01
7.295489482895208821e-01 4.194909849233816046e-01 1.378630483492892522e-01
7.344517242247444733e-01 4.154177405103107734e-01 1.440803933719045082e-01
7.392949550641365608e-01 4.112997327023164562e-01 1.503221736968965161e-01
7.440383351241327547e-01 4.071715772870146410e-01 1.565433461845777419e-01
7.487369523694534790e-01 4.029851907815384382e-01 1.628228161295872112e-01
7.533231937521204236e-01 3.988010690198531272e-01 1.690756638056752914e-01
7.578808294584169492e-01 3.945424511336589335e-01 1.754217924072464518e-01
7.623326022156785564e-01 3.902809567197497165e-01 1.817591538530497208e-01
7.667320478995347521e-01 3.859654943537603744e-01 1.881681875043557384e-01
7.710524721820763983e-01 3.816214148216601210e-01 1.946153193576832252e-01
7.752952778457107286e-01 3.772473246523442292e-01 2.011065208945331251e-01
7.794866593781552000e-01 3.728150850100374614e-01 2.076872953652798559e-01
7.835853362619955575e-01 3.683677247727679127e-01 2.142973641992331202e-01
7.876376313731141554e-01 3.638539950288945946e-01 2.210164757476915653e-01
7.916113385801130109e-01 3.593080404226161040e-01 2.277974012097270795e-01
7.955060552511121763e-01 3.547298979176854994e-01 2.346435260251332755e-01
7.993539838133075781e-01 3.500795929482780067e-01 2.416183193481767355e-01
8.031167084780931331e-01 3.454015163918872089e-01 2.486589162182153978e-01
8.068103261859892461e-01 3.406745225101673324e-01 2.558007514596662979e-01
8.104452043001210138e-01 3.358824841311982556e-01 2.630722168071251699e-01
8.139968010634672790e-01 3.310553831960054150e-01 2.704318260711638389e-01
8.174768947816700715e-01 3.261752637674419919e-01 2.779109619017897659e-01
8.208941481936247175e-01 3.212262889577221503e-01 2.855384604486080891e-01
8.242271261613036692e-01 3.162361985850312696e-01 2.932761677775464482e-01
8.274766137520660481e-01 3.112015433453460544e-01 3.011338788484150264e-01
8.306639856730879679e-01 3.060845940504859919e-01 3.091757851828964565e-01
8.337630658888238733e-01 3.009224359536462057e-01 3.173492086215453090e-01
8.367728587568070697e-01 2.957134612590129330e-01 3.256619937531243236e-01
8.396969301670653696e-01 2.904472328682123905e-01 3.341366497709652439e-01
8.425387334318170662e-01 2.851115076990556330e-01 3.427996193346179443e-01
8.452829693062083871e-01 2.797291658336165110e-01 3.516207809735176215e-01
8.479270414576970394e-01 2.743004518168249417e-01 3.606068061808916925e-01
8.504679257105771661e-01 2.688262350691851821e-01 3.697639476988339724e-01
8.529105574510703613e-01 2.632885874038991547e-01 3.791311611530527870e-01
8.552420044048544279e-01 2.577088839643039142e-01 3.886821696092926381e-01
8.574567263211506640e-01 2.520936658861107627e-01 3.984160108673813205e-01
8.595502320723124035e-01 2.464473708549377862e-01 4.083362533435646036e-01
8.615176695665305306e-01 2.407756349505418836e-01 4.184455712413106543e-01
8.633539245616504987e-01 2.350852138633325872e-01 4.287460573240959860e-01
8.650568452189218993e-01 2.293728768111193417e-01 4.392600781097817930e-01
8.666160631462880293e-01 2.236630759123471868e-01 4.499612660389528673e-01
8.680257787196800079e-01 2.179678524813165597e-01 4.608475800597766070e-01
8.692800281403405549e-01 2.123013227323138907e-01 4.719155444862356830e-01
8.703727414475561641e-01 2.066798807505955682e-01 4.831601540690036445e-01
8.712978068076887572e-01 2.011223984119477337e-01 4.945747939602412324e-01
8.720491402648608004e-01 1.956504128743852822e-01 5.061511790861612514e-01
8.726207597696474805e-01 1.902882889012951495e-01 5.178793172539123413e-01
8.730068619526127893e-01 1.850633392111822872e-01 5.297474998646168887e-01
8.732018998073337590e-01 1.800058813906028066e-01 5.417423233741458510e-01
8.732006592202997686e-01 1.751492049740893675e-01 5.538487436526324803e-01
8.729983321570818910e-01 1.705294177324829519e-01 5.660501641838993070e-01
8.725905843033689990e-01 1.661851370964102237e-01 5.783285576821829421e-01
8.719736126526835829e-01 1.621569796196943858e-01 5.906646606873059424e-01
8.711441430703839028e-01 1.584866680756561452e-01 6.030388085538075371e-01
8.700996606911847175e-01 1.552168662477456385e-01 6.154284378630663355e-01
8.688382315166471859e-01 1.523889228568965915e-01 6.278117548576560569e-01
8.673585803491290491e-01 1.500419870434277214e-01 6.401665059420836856e-01
8.656600953609360216e-01 1.482114935188816873e-01 6.524702171100530412e-01
8.637428203708292784e-01 1.469276207331668138e-01 6.647004334019646077e-01
8.616074355850228406e-01 1.462138565700954185e-01 6.768349518289993316e-01
8.592552279602999610e-01 1.460858181738819150e-01 6.888520417478360969e-01
8.566880526588371847e-01 1.465504601378146143e-01 7.007306474947510022e-01
8.539082940810565070e-01 1.476057634824317899e-01 7.124505416663439172e-01
8.509188132430276497e-01 1.492409439214724132e-01 7.239924974090546916e-01
8.477228732160004832e-01 1.514371672020923265e-01 7.353384896921373315e-01
8.443240935674275471e-01 1.541686506679828816e-01 7.464717393223647690e-01
8.407263939629991967e-01 1.574040314553000475e-01 7.573767829039294019e-01
8.369339386719339968e-01 1.611078555520196187e-01 7.680395184197363889e-01
8.329510832270686782e-01 1.652420527841307607e-01 7.784472273410004695e-01
8.287823240288615390e-01 1.697672910052525075e-01 7.885885755886976600e-01
8.244322514509230260e-01 1.746441391323516057e-01 7.984535959358352031e-01
8.199055067926670493e-01 1.798340046929527702e-01 8.080336545530442116e-01
8.152067432395495583e-01 1.852998414406232253e-01 8.173214043862717659e-01
8.103405908369269994e-01 1.910066437572490727e-01 8.263107279415592421e-01
8.053117579051806141e-01 1.969216010609510237e-01 8.349964504885767358e-01
8.001246695316600599e-01 2.030146524204510805e-01 8.433748621071933682e-01
7.947836730093801316e-01 2.092582614636481764e-01 8.514431956464665330e-01
7.892930180385833161e-01 2.156273737159832282e-01 8.591995655348122485e-01
7.836568069717744223e-01 2.220993550443281506e-01 8.666429444356050782e-01
7.778789796122015376e-01 2.286538551089048465e-01 8.737730828797295457e-01
7.719633002465959848e-01 2.352726496465096795e-01 8.805904299224873721e-01
7.659133466040326521e-01 2.419394735315277267e-01 8.870960555269299386e-01
7.597325004651533931e-01 2.486398530394249295e-01 8.932915751926430170e-01
7.534239396834068181e-01 2.553609429642716422e-01 8.991790771963389384e-01
7.469906314200031039e-01 2.620913721366889826e-01 9.047610526867282399e-01
7.404353264352245834e-01 2.688210993418901351e-01 9.100403287788533246e-01
7.337605543192734503e-01 2.755412805372748908e-01 9.150200047193097763e-01
7.269686195850667554e-01 2.822441475132592692e-01 9.197033911407090923e-01
7.200615985827201193e-01 2.889228976436723495e-01 9.240939523881464002e-01
7.130413372306516617e-01 2.955715940634778272e-01 9.281952518796100504e-01
7.059094495911268918e-01 3.021850754377659598e-01 9.320109004535763741e-01
6.986673173487639721e-01 3.087588744057577217e-01 9.355445076582962205e-01
6.913160902792203633e-01 3.152891437666022756e-01 9.387996359465969887e-01
6.838566878221142842e-01 3.217725894980000279e-01 9.417797577558794098e-01
6.762898018976056802e-01 3.282064097483991527e-01 9.444882154741288671e-01
6.686159011300095711e-01 3.345882390078672719e-01 9.469281843183131597e-01
6.608352366648267973e-01 3.409160967342457216e-01 9.491026381806494383e-01
6.529478497874137144e-01 3.471883397852293385e-01 9.510143185305323099e-01
6.449535815726188392e-01 3.534036180803885596e-01 9.526657064947888776e-01
6.368520848146350666e-01 3.595608329884161236e-01 9.540589982761437104e-01
6.286428385050449874e-01 3.656590980031973470e-01 9.551960841088614762e-01
6.203251651440072623e-01 3.716977013375789007e-01 9.560785309910512231e-01
6.118982511841425387e-01 3.776760701263490727e-01 9.567075694743777392e-01
6.033611709179768079e-01 3.835937359904362243e-01 9.570840848332117234e-01
5.947129141266446206e-01 3.894503017735960748e-01 9.572086129751365968e-01
5.859524178083830304e-01 3.952454093215529429e-01 9.570813414919564499e-01
5.770786022984990549e-01 4.009787082326203289e-01 9.567021162826109260e-01
5.680904120756430364e-01 4.066498255689147689e-01 9.560704542043091392e-01
5.589868615201727398e-01 4.122583365789312393e-01 9.551855622225308151e-01
5.497670858464057675e-01 4.178037365457583641e-01 9.540463635305430623e-01
5.404303973688802110e-01 4.232854139404694238e-01 9.526515310905910860e-01
5.309763471805271084e-01 4.287026251268353794e-01 9.509995290068892215e-01
5.214047922154096959e-01 4.340544709302417981e-01 9.490886620701871612e-01
5.117159675380545947e-01 4.393398754490472347e-01 9.469171337096092822e-01
5.019105635439048418e-01 4.445575675481795996e-01 9.444831124449268867e-01
4.919898075708288299e-01 4.497060655293146358e-01 9.417848067474301477e-01
4.819555492105473404e-01 4.547836655159373520e-01 9.388205479873396042e-01
4.718103483745911819e-01 4.597884341201938230e-01 9.355888808698555881e-01
4.615575649166388517e-01 4.647182059669867082e-01 9.320886604427106592e-01
4.511980082251949020e-01 4.695721776839077433e-01 9.283178636903177683e-01
4.407385246316099514e-01 4.743468819302843476e-01 9.242766906682317041e-01
4.301872191964954406e-01 4.790386406828319177e-01 9.199661970019400448e-01
4.195516590423077341e-01 4.836443962749994996e-01 9.153875906397229700e-01
4.088406331006777528e-01 4.881609386110861704e-01 9.105429315654867128e-01
3.980642083018061106e-01 4.925849423947825101e-01 9.054352272162426996e-01
3.872337717548847147e-01 4.969130108713233906e-01 9.000685170373873278e-01
3.763620564988861550e-01 5.011417254753421924e-01 8.944479427537497251e-01
3.654612663762428770e-01 5.052684034325813922e-01 8.885787674719856089e-01
3.545465439790672080e-01 5.092897968880314430e-01 8.824681923036010733e-01
3.436377931849474154e-01 5.132015795889635079e-01 8.761266360928694485e-01
3.327530939137686161e-01 5.170008198204416594e-01 8.695640864822369309e-01
3.219116583925260011e-01 5.206848733578030020e-01 8.627916585616799416e-01
3.111337198622898259e-01 5.242514414427805747e-01 8.558215206665023000e-01
3.004403986093719392e-01 5.276986238157412856e-01 8.486667948769511804e-01
2.898532566310526581e-01 5.310250549915590534e-01 8.413412305522999235e-01
2.793961594244663282e-01 5.342293148180886631e-01 8.338605094653681604e-01
2.690918119759744820e-01 5.373109871573702456e-01 8.262398418700288572e-01
2.589630009926549015e-01 5.402702010433201307e-01 8.184947475296583397e-01
2.490323931623479869e-01 5.431076290262466522e-01 8.106408952154365855e-01
2.393222899575573326e-01 5.458244839403564308e-01 8.026939223701164972e-01
2.298566351816503373e-01 5.484218890000336355e-01 7.946712158983202379e-01
2.206551002944838191e-01 5.509024141956824216e-01 7.865870626923798792e-01
2.117364094415158937e-01 5.532690129113547739e-01 7.784553296976853831e-01
2.031184306485467883e-01 5.555248911915373622e-01 7.702897347110843063e-01
1.948172015035146420e-01 5.576736467032974431e-01 7.621031776295352778e-01
1.868465966290397962e-01 5.597192229650957973e-01 7.539076339343501187e-01
1.792179884783634825e-01 5.616658613090964591e-01 7.457140662049784874e-01
1.719421959020442647e-01 5.635174702399022850e-01 7.375349843171632447e-01
1.650229512673002386e-01 5.652791533635259658e-01 7.293775439873284583e-01
1.584611602813638387e-01 5.669560017699032395e-01 7.212481886261494779e-01
1.522549918213680353e-01 5.685529672520610589e-01 7.131532051881548373e-01
1.463987618681510117e-01 5.700750569206646245e-01 7.050976866560473288e-01
1.408828405908224835e-01 5.715272914520931336e-01 6.970855353349069139e-01
1.356936625367466953e-01 5.729146658465781305e-01 6.891194758268863740e-01
1.308138525417579801e-01 5.742421127377883572e-01 6.812010763134157543e-01
1.262224734293325157e-01 5.755144682058210837e-01 6.733307769330967307e-01
1.218953938035638729e-01 5.767364399826563348e-01 6.655079242329828837e-01
1.178065439715534068e-01 5.779123523036159282e-01 6.577323260140637284e-01
1.139261298009009160e-01 5.790467957802227783e-01 6.499998366451625875e-01
1.102234782385779766e-01 5.801439808367224726e-01 6.423063681141680803e-01
1.066673235183970281e-01 5.812078215483343913e-01 6.346473314887408623e-01
1.032263067644274834e-01 5.822419757773840132e-01 6.270172879951815270e-01
9.986970116659238395e-02 5.832498248631478033e-01 6.194100074982554771e-01
9.656813269866712512e-02 5.842344549012296051e-01 6.118185305845139643e-01
9.329429206118067253e-02 5.851986396650793454e-01 6.042352341768219004e-01
9.002364276931534848e-02 5.861448252640329981e-01 5.966519005456781821e-01
8.673514013646366205e-02 5.870751166646858143e-01 5.890597894857422245e-01
8.341198598223345528e-02 5.879912662222103181e-01 5.814497133016774955e-01
8.004245400348602990e-02 5.888946643728962815e-01 5.738121141059205899e-01
7.662083060590932360e-02 5.897863326271246542e-01 5.661371427838456372e-01
7.314852485397654869e-02 5.906669189727291602e-01 5.584147388414824054e-01
6.963540714651897390e-02 5.915366957529474279e-01 5.506347102299997687e-01
6.610143501147561218e-02 5.923955600227664986e-01 5.427868121514007882e-01
6.257860760091599195e-02 5.932430363153763375e-01 5.348608238014407323e-01
5.911303759975024968e-02 5.940783314685343930e-01 5.268461376019544229e-01
5.576765010285392871e-02 5.949002975004614724e-01 5.187321979872163702e-01
5.262510565424809855e-02 5.957073200832398996e-01 5.105097808257538228e-01
4.978880680939656161e-02 5.964975038111670624e-01 5.021693595319391967e-01
4.738319269394732774e-02 5.972686215557058143e-01 4.937017361708459506e-01
4.555066847662456869e-02 5.980181252952141424e-01 4.850980867717663014e-01
4.444396189302631667e-02 5.987431566697054564e-01 4.763500004038906943e-01
4.421322948900235222e-02 5.994405566877633040e-01 4.674495124401222834e-01
4.498917867710291313e-02 6.001068740129568146e-01 4.583891327826852824e-01
4.686604485069371245e-02 6.007383712762072170e-01 4.491618701849053319e-01
4.988979024235475762e-02 6.013310288983726437e-01 4.397612541729841729e-01
5.405573006313083712e-02 6.018805543938667846e-01 4.301812015738862294e-01
5.932208540209162745e-02 6.023828876312081748e-01 4.204054325617814780e-01
6.560773880422715587e-02 6.028325831369402144e-01 4.104377156032376628e-01
7.281962363892094392e-02 6.032244155970721833e-01 4.002736262351211383e-01
8.086176781346332554e-02 6.035528335053932381e-01 3.899094149354802585e-01
8.964365767756551917e-02 6.038119404209635332e-01 3.793420777858104165e-01
9.908952486975769469e-02 6.039955435016706176e-01 3.685641184021282157e-01
1.091461686387718844e-01 6.040969460876761676e-01 3.575579876882719055e-01
1.197411868235576382e-01 6.041085844094219448e-01 3.463409605502589250e-01
1.308274635712054768e-01 6.040228036354146068e-01 3.349141605174003056e-01
1.423800347596975990e-01 6.038311914501962585e-01 3.232669997463486489e-01
1.543847008629261053e-01 6.035242532835117801e-01 3.113882295486853913e-01
1.667909278439370091e-01 6.030930063069259717e-01 2.993102878563658198e-01
1.795975741928860503e-01 6.025266800569213377e-01 2.870237044475537069e-01
1.927996570161342460e-01 6.018136381688595771e-01 2.745296424849442141e-01
2.063446461864067438e-01 6.009446614046377588e-01 2.618794002040416569e-01
2.202728729725795254e-01 5.999042993738278318e-01 2.490425136645111337e-01
2.344983329249494819e-01 5.986859114745567423e-01 2.361102156981361166e-01
2.490441577314199684e-01 5.972745984023483112e-01 2.230778046167814499e-01
2.638200588853058526e-01 5.956665601719514092e-01 2.100467332369221340e-01
2.788103972435367339e-01 5.938520960352462463e-01 1.970548426870546432e-01
2.939149436021079032e-01 5.918334783865150106e-01 1.842162055743405413e-01
3.090633958032663053e-01 5.896130197890373514e-01 1.716194213262129398e-01
3.241557701146778325e-01 5.872013194176191053e-01 1.593775348465770181e-01
3.391058987925449908e-01 5.846116417760637285e-01 1.475901239657965436e-01
3.537962403050050608e-01 5.818679304916246631e-01 1.363773405186440579e-01
3.681790539934771678e-01 5.789861015095073560e-01 1.258005415905540103e-01
3.821596573789931561e-01 5.759951191731197406e-01 1.159503977992311363e-01
3.957282445230750900e-01 5.729092809436007183e-01 1.068503820674790161e-01
4.088192648913888116e-01 5.697572697905517458e-01 9.855521202608327758e-02
4.214810649035653500e-01 5.665415862717396722e-01 9.104002246571415990e-02
4.336495334558000403e-01 5.632929561600926727e-01 8.434116239111955071e-02
4.453890783516751273e-01 5.600085906382493706e-01 7.841305443909030171e-02
4.567242130853677584e-01 5.566942951475172263e-01 7.322913012508130981e-02
4.676501707681009479e-01 5.533637269053324204e-01 6.876762134759795142e-02
4.781913781821610088e-01 5.500213008191600084e-01 6.498435721988443659e-02
4.883968612384750885e-01 5.466619506976900800e-01 6.182162837413415074e-02
4.982892398787829857e-01 5.432873953595327432e-01 5.922725775013665955e-02
5.078911374422868663e-01 5.398982668933685058e-01 5.714465998451528223e-02
5.172247456455653092e-01 5.364942872197172585e-01 5.551476288204022086e-02
5.263115025893324583e-01 5.330744300746458331e-01 5.427792556635833293e-02
5.351718645250624906e-01 5.296370658817842747e-01 5.337566853443905662e-02
5.438251536372012973e-01 5.261800887043684982e-01 5.275207547378853862e-02
5.522894663786476199e-01 5.227010255649303661e-01 5.235479075452074277e-02
5.605816294780567866e-01 5.191971290183284848e-01 5.213559944466891055e-02
5.687171933168381210e-01 5.156654540879882509e-01 5.205062439450063722e-02
5.767104547724090091e-01 5.121029206313181259e-01 5.206020204364426168e-02
5.845745037500035268e-01 5.085063619725217476e-01 5.212850674796504907e-02
5.923212894447653643e-01 5.048725602928031408e-01 5.222298774900729218e-02
5.999617038945115333e-01 5.011982688438658684e-01 5.231366903560310394e-02
6.075056816322426112e-01 4.974802205790658793e-01 5.237234455720112675e-02
6.149623152735116394e-01 4.937151222923196747e-01 5.237168182411356537e-02
6.223399877334323538e-01 4.898996328220020513e-01 5.228422613133864444e-02
6.296465225194588511e-01 4.860303233135512824e-01 5.208127381405584094e-02
6.368893542334475022e-01 4.821036169426133333e-01 5.173155232549151578e-02
6.440757220432393737e-01 4.781157049081125598e-01 5.119960076823739520e-02
6.512128893528150719e-01 4.740624350126631525e-01 5.044367478760234530e-02
6.583083928921535932e-01 4.699391690315524728e-01 4.941288204103298082e-02];

   case 'hal' 
      RGB = [1.629529545569048110e-01 9.521591660747855124e-02 4.225729247643043585e-01
1.648101130638113809e-01 9.635115909727909322e-02 4.318459659833655540e-01
1.666161667445505146e-01 9.744967053737302320e-02 4.412064832719169161e-01
1.683662394047173716e-01 9.851521320092249123e-02 4.506510991070378780e-01
1.700547063176806595e-01 9.955275459284393391e-02 4.601751103492678907e-01
1.716750780810941956e-01 1.005687314559364776e-01 4.697722208210775574e-01
1.732198670017069397e-01 1.015713570251385311e-01 4.794342308257477092e-01
1.746804342417165035e-01 1.025709733421875103e-01 4.891506793097686878e-01
1.760433654254164593e-01 1.035658402770499587e-01 4.989416012077843576e-01
1.772982333235153807e-01 1.045802467658180357e-01 5.087715885336102639e-01
1.784322966250933284e-01 1.056380265564063059e-01 5.186108302832771466e-01
1.794226692010022772e-01 1.067416562108134404e-01 5.284836071020164727e-01
1.802542327126359922e-01 1.079356346679062328e-01 5.383245681077661882e-01
1.808975365813079994e-01 1.092386640641496154e-01 5.481352134375515606e-01
1.813298273265454008e-01 1.107042924622455293e-01 5.578435355461390799e-01
1.815069308605478937e-01 1.123613365530294061e-01 5.674471854200233700e-01
1.813959559086370799e-01 1.142804413027345978e-01 5.768505865319291104e-01
1.809499433760710929e-01 1.165251530113385336e-01 5.859821014031293407e-01
1.801166524094891808e-01 1.191682999758127970e-01 5.947494236872948870e-01
1.788419557731087683e-01 1.222886104999623413e-01 6.030366129604394221e-01
1.770751344832933727e-01 1.259620672997293078e-01 6.107077426144936760e-01
1.747764954226868062e-01 1.302486445940692350e-01 6.176174300439590814e-01
1.719255883800615836e-01 1.351768519397535118e-01 6.236290832033221099e-01
1.685302279919113078e-01 1.407308818346016399e-01 6.286357211183263294e-01
1.646373543798159977e-01 1.468433194330099889e-01 6.325796572366660930e-01
1.603141656593721487e-01 1.534074847391770635e-01 6.354701889106297852e-01
1.556539455727427579e-01 1.602911795924207572e-01 6.373742153046678682e-01
1.507373567977903506e-01 1.673688895313445446e-01 6.383989700654711941e-01
1.456427577979826360e-01 1.745293312408868480e-01 6.386687569056349600e-01
1.404368075255880977e-01 1.816841459042554952e-01 6.383089542091028301e-01
1.351726504089350855e-01 1.887688275072176014e-01 6.374350053971095109e-01
1.298906561807787186e-01 1.957398580438490798e-01 6.361469852044080442e-01
1.246205125693149729e-01 2.025703385486158914e-01 6.345282558695404251e-01
1.193859004780570554e-01 2.092446623034395214e-01 6.326478270215730726e-01
1.142294912197052703e-01 2.157456284251405010e-01 6.305768690676523125e-01
1.091404911375367659e-01 2.220831206181900774e-01 6.283455167242665285e-01
1.041438584244326337e-01 2.282546518282705383e-01 6.259979600528258192e-01
9.926304855671816418e-02 2.342609767388125763e-01 6.235717761795677161e-01
9.449512580805050077e-02 2.401139958170242505e-01 6.210816676451920149e-01
8.986951574154733446e-02 2.458147193889331228e-01 6.185591936304666305e-01
8.539285840535987271e-02 2.513729453557033144e-01 6.160166295227810229e-01
8.106756674391193962e-02 2.567997050291829786e-01 6.134596708213713168e-01
7.694418932732069449e-02 2.620915612909199277e-01 6.109238301118911085e-01
7.300703739422578775e-02 2.672655035330154250e-01 6.083965011432549419e-01
6.927650669442811382e-02 2.723273008096985248e-01 6.058873223830183452e-01
6.578801445169751849e-02 2.772789491245566951e-01 6.034141752438300088e-01
6.255595479554787453e-02 2.821282390686886132e-01 6.009787922718963227e-01
5.959205181913470456e-02 2.868831717247524726e-01 5.985797542127682114e-01
5.691772151374491912e-02 2.915488716281217640e-01 5.962214962176209943e-01
5.455347307306369214e-02 2.961302536799313989e-01 5.939076044300871660e-01
5.251889627870443694e-02 3.006318409387543356e-01 5.916414807294642086e-01
5.083877347247430650e-02 3.050562292020492783e-01 5.894315315373579445e-01
4.951454037014189208e-02 3.094102218220906031e-01 5.872714908845412252e-01
4.855490408104565919e-02 3.136977658751046727e-01 5.851627302038014955e-01
4.796369156225028380e-02 3.179225992994973993e-01 5.831062484926557987e-01
4.773946305380068894e-02 3.220882581897950847e-01 5.811027291021745311e-01
4.787545181415154422e-02 3.261980852298422273e-01 5.791525884087008746e-01
4.835984720504159923e-02 3.302552388254387794e-01 5.772560174768572860e-01
4.917638757411300215e-02 3.342627026038174076e-01 5.754130176762238813e-01
5.030518671680884318e-02 3.382232950310170572e-01 5.736234310853615126e-01
5.172369283691292258e-02 3.421396789632700219e-01 5.718869664041401624e-01
5.340767549062541003e-02 3.460143709989857430e-01 5.702032209969470911e-01
5.533215100674954839e-02 3.498497505368459159e-01 5.685716996038682192e-01
5.747218306369886870e-02 3.536480684754851334e-01 5.669918301827847618e-01
5.980352430527090951e-02 3.574114555130643578e-01 5.654629772812437283e-01
6.230309069250609261e-02 3.611419300223894235e-01 5.639844532816007394e-01
6.494927925026378057e-02 3.648414054902228698e-01 5.625555278152573058e-01
6.772215122553368327e-02 3.685116975190721456e-01 5.611754356007422340e-01
7.060350747784929770e-02 3.721545303967777607e-01 5.598433829250933913e-01
7.357840396611611822e-02 3.757710087912914942e-01 5.585612898846836760e-01
7.663101364217325684e-02 3.793629991309988569e-01 5.573268773673928367e-01
7.974739830752586300e-02 3.829322364932883360e-01 5.561380319387883020e-01
8.291580548873803136e-02 3.864801413799028307e-01 5.549938581786431069e-01
8.612580955679122185e-02 3.900080689665953448e-01 5.538934528024703763e-01
8.936817980172057085e-02 3.935173134989205512e-01 5.528359060062189023e-01
9.263475301781654014e-02 3.970091123550867351e-01 5.518203023495191761e-01
9.591831391237073956e-02 4.004846497947374129e-01 5.508457212473490960e-01
9.921323508992196949e-02 4.039446967979293257e-01 5.499133654313189679e-01
1.025139614653171327e-01 4.073902646845309894e-01 5.490228475752887416e-01
1.058144853522852702e-01 4.108228877965505177e-01 5.481703859285301794e-01
1.091104131658914012e-01 4.142435690118807523e-01 5.473550236322454188e-01
1.123978495089298091e-01 4.176532725696484594e-01 5.465757987301745890e-01
1.156733362160069500e-01 4.210529263321469151e-01 5.458317432175192607e-01
1.189341701380315364e-01 4.244432161011251203e-01 5.451231873866256850e-01
1.221781892974011241e-01 4.278246691926565481e-01 5.444513010684173260e-01
1.254018943745988379e-01 4.311987106338182607e-01 5.438113725374379426e-01
1.286031296249826039e-01 4.345661396549306832e-01 5.432023919026625070e-01
1.317799711665625373e-01 4.379277275711909168e-01 5.426233385794481112e-01
1.349307019221451520e-01 4.412842189714309415e-01 5.420731800699918335e-01
1.380549051543558114e-01 4.446356900078314855e-01 5.415550926551251365e-01
1.411501069188544899e-01 4.479834819017672332e-01 5.410638066180113448e-01
1.442150210041465985e-01 4.513283116704150388e-01 5.405979268760919831e-01
1.472485820103837661e-01 4.546708245755168298e-01 5.401563599016087069e-01
1.502499341655997578e-01 4.580115969309521140e-01 5.397383042732707414e-01
1.532192022679761678e-01 4.613507130792227628e-01 5.393460827274304537e-01
1.561545863558880531e-01 4.646893652629560667e-01 5.389744586257710912e-01
1.590554825896313418e-01 4.680281092699005163e-01 5.386222668134232894e-01
1.619213854747377779e-01 4.713674796485894380e-01 5.382883230992108192e-01
1.647524478987731911e-01 4.747077026242289555e-01 5.379733478625169374e-01
1.675482630437002962e-01 4.780493319379570116e-01 5.376757027790604049e-01
1.703080688452488778e-01 4.813931150452205876e-01 5.373922894579649112e-01
1.730317245949949956e-01 4.847395013664480001e-01 5.371218460871988176e-01
1.757193664968157432e-01 4.880888303994977417e-01 5.368636833242294015e-01
1.783716503259393793e-01 4.914412289641479359e-01 5.366183612997760255e-01
1.809878167761821144e-01 4.947975030047193079e-01 5.363817525707026412e-01
1.835680719416625251e-01 4.981580161432020426e-01 5.361525222751042374e-01
1.861127112291278973e-01 5.015231101060642072e-01 5.359293193132266264e-01
1.886230405888700001e-01 5.048927132825591357e-01 5.357133548543864254e-01
1.910985251486422565e-01 5.082675669534003626e-01 5.355003101591436776e-01
1.935397227717326196e-01 5.116479477164066481e-01 5.352887824024628038e-01
1.959472883340568350e-01 5.150341080561433582e-01 5.350773667248226451e-01
1.983227076664061950e-01 5.184259856373716335e-01 5.348665525352744865e-01
2.006660342507454731e-01 5.218241099237964642e-01 5.346528031121534630e-01
2.029781360478647434e-01 5.252286912572143862e-01 5.344345169358406533e-01
2.052600444742044838e-01 5.286398903414566419e-01 5.342102605335536936e-01
2.075134652380221101e-01 5.320576287402744020e-01 5.339799959048032729e-01
2.097389494614402272e-01 5.354822793223580346e-01 5.337406085215398166e-01
2.119377691272176234e-01 5.389139515314046447e-01 5.334905459074957834e-01
2.141113437844118228e-01 5.423527133674995726e-01 5.332283775266504211e-01
2.162615771757636085e-01 5.457984712571943842e-01 5.329535750363618707e-01
2.183895286205785324e-01 5.492514494924778390e-01 5.326634150194080597e-01
2.204969036245257863e-01 5.527116487772890663e-01 5.323564832742772035e-01
2.225855272635121618e-01 5.561790393438251767e-01 5.320314302436226495e-01
2.246573660062902156e-01 5.596535532346759156e-01 5.316870266023980829e-01
2.267141352300188206e-01 5.631352211554988552e-01 5.313212748929951879e-01
2.287579175696445311e-01 5.666239548700177098e-01 5.309328290550865415e-01
2.307908679682200148e-01 5.701196510565367248e-01 5.305203252817071169e-01
2.328150701112509102e-01 5.736222405040750649e-01 5.300820599240353426e-01
2.348328561421904603e-01 5.771315766359990107e-01 5.296167282704775658e-01
2.368466495707550745e-01 5.806474896434283828e-01 5.291230766564496424e-01
2.388588283644081933e-01 5.841698333340915594e-01 5.285995915811123602e-01
2.408716822240541400e-01 5.876984999166237067e-01 5.280443921862176815e-01
2.428880608722543410e-01 5.912331932277818947e-01 5.274567814051942527e-01
2.449106759672304290e-01 5.947736703172152861e-01 5.268356212312692577e-01
2.469420290965465559e-01 5.983197676291379663e-01 5.261791312000029253e-01
2.489846702882144713e-01 6.018713111492172141e-01 5.254854952988164962e-01
2.510418446331669773e-01 6.054278820406324702e-01 5.247545535679302153e-01
2.531164830585315162e-01 6.089891735215487989e-01 5.239852980594518206e-01
2.552111567013787274e-01 6.125550130731924892e-01 5.231756545447472373e-01
2.573286340449815746e-01 6.161251617120727664e-01 5.223239185167128928e-01
2.594724447386158594e-01 6.196990932330638246e-01 5.214304767886038805e-01
2.616456589963800927e-01 6.232764459181282524e-01 5.204944566826074093e-01
2.638509342960791981e-01 6.268570181766053295e-01 5.195136536074571598e-01
2.660910828965943331e-01 6.304405546707460006e-01 5.184861377534477622e-01
2.683698467163787016e-01 6.340264147511259774e-01 5.174130230514244477e-01
2.706903509949471487e-01 6.376141889650659422e-01 5.162935692788781505e-01
2.730554221838172868e-01 6.412035896296301996e-01 5.151259285643695618e-01
2.754676310512871873e-01 6.447944545228798674e-01 5.139069764892589820e-01
2.779309010023177096e-01 6.483859905965968506e-01 5.126390269795514376e-01
2.804483318408275694e-01 6.519777450281342146e-01 5.113214639163841113e-01
2.830229969716622218e-01 6.555692556652558123e-01 5.099537040511894492e-01
2.856569925022096612e-01 6.591606030439277619e-01 5.085297306531970651e-01
2.883543502639873135e-01 6.627507929293073863e-01 5.070537569679475220e-01
2.911180907975667309e-01 6.663393219020087299e-01 5.055253556302098383e-01
2.939511790083492726e-01 6.699256847308838747e-01 5.039440536705714901e-01
2.968562131418951422e-01 6.735096490751359966e-01 5.023062065413991251e-01
2.998362114418811064e-01 6.770906951012128916e-01 5.006109626120457401e-01
3.028943961763405635e-01 6.806680059292820051e-01 4.988609220555944579e-01
3.060335830069556007e-01 6.842410278074210206e-01 4.970557164988521071e-01
3.092565373453909361e-01 6.878091952372648032e-01 4.951950035800954386e-01
3.125659486948474952e-01 6.913723055472413836e-01 4.932732004330610542e-01
3.159647522698377231e-01 6.949295261702347348e-01 4.912927476409480465e-01
3.194556489243873809e-01 6.984800979280558764e-01 4.892553378582480961e-01
3.230412442793148542e-01 7.020233920397538352e-01 4.871607422199746851e-01
3.267240985578743762e-01 7.055587631720373620e-01 4.850087606010107799e-01
3.305070751488592418e-01 7.090857414439620809e-01 4.827955626459811689e-01
3.343929739199408280e-01 7.126036449951604901e-01 4.805201317683439055e-01
3.383839618515475101e-01 7.161115318028217214e-01 4.781864627637871235e-01
3.424824761777380822e-01 7.196086676710338192e-01 4.757945201032026117e-01
3.466909265050957534e-01 7.230942938245445983e-01 4.733443136156250675e-01
3.510117003671430203e-01 7.265676248366644829e-01 4.708359034900377327e-01
3.554477481971078934e-01 7.300279237012574640e-01 4.682667163182038794e-01
3.600022066505755847e-01 7.334743741555846963e-01 4.656343331651458528e-01
3.646764457841936702e-01 7.369059239634064840e-01 4.629441295765394648e-01
3.694728210602395979e-01 7.403216514270205550e-01 4.601965063235068931e-01
3.743936915522128039e-01 7.437205957454258165e-01 4.573919681273960758e-01
3.794414259131371203e-01 7.471017539541063845e-01 4.545311394783269621e-01
3.846184079557829483e-01 7.504640777072076885e-01 4.516147832933739559e-01
3.899270416022538321e-01 7.538064699246833644e-01 4.486438228496626990e-01
3.953697549145612777e-01 7.571277813375660859e-01 4.456193674826627871e-01
4.009508646485598904e-01 7.604267477869489644e-01 4.425380639654961090e-01
4.066714019599443342e-01 7.637021069222051928e-01 4.394056497009877216e-01
4.125334807152798988e-01 7.669525443587899005e-01 4.362249727525224774e-01
4.185395667527040398e-01 7.701766751192415938e-01 4.329983295596441795e-01
4.246921350385852723e-01 7.733730477083216037e-01 4.297284080553480656e-01
4.309936593663836191e-01 7.765401418648604226e-01 4.264183470604332449e-01
4.374465975304094312e-01 7.796763669997491819e-01 4.230718037095967943e-01
4.440533711015541840e-01 7.827800615723168320e-01 4.196930295353452078e-01
4.508163388346139722e-01 7.858494937119429036e-01 4.162869557148224930e-01
4.577377626480225170e-01 7.888828634531382944e-01 4.128592877810690620e-01
4.648197650471433406e-01 7.918783070193569085e-01 4.094166097874233912e-01
4.720642768256655963e-01 7.948339036614172626e-01 4.059664974607166132e-01
4.794729738954752185e-01 7.977476856267914362e-01 4.025176392507668344e-01
4.870472021865835388e-01 8.006176519006481529e-01 3.990799633442549399e-01
4.947878897554374711e-01 8.034417864099652196e-01 3.956647676266520364e-01
5.026954455748450235e-01 8.062180814071850943e-01 3.922848482216537147e-01
5.107722312752203120e-01 8.089441566688712060e-01 3.889513459863399025e-01
5.190175807229889804e-01 8.116180108735555621e-01 3.856804341377490508e-01
5.274270476594079549e-01 8.142382455983395717e-01 3.824934902796895964e-01
5.359974387485314518e-01 8.168032862890818313e-01 3.794106529717772847e-01
5.447242959015131669e-01 8.193118012377970105e-01 3.764539238160731771e-01
5.536017493685388979e-01 8.217627673204914718e-01 3.736470734604069865e-01
5.626223858732353200e-01 8.241555398979113489e-01 3.710154595070918049e-01
5.717801732913297963e-01 8.264893011624766528e-01 3.685830453573430421e-01
5.810619798273547465e-01 8.287648131945639651e-01 3.663798607459672341e-01
5.904522695833789303e-01 8.309836316507100973e-01 3.644363382969363352e-01
5.999363056699199559e-01 8.331474672067843423e-01 3.627802030453921023e-01
6.094978370184862548e-01 8.352587020450518152e-01 3.614380620192426119e-01
6.191178753985615568e-01 8.373207419267220120e-01 3.604354982890065617e-01
6.287753075069072439e-01 8.393379672623436649e-01 3.597956834322972863e-01
6.384515865712356852e-01 8.413146218129586851e-01 3.595361765343614291e-01
6.481275660781142811e-01 8.432555569199260415e-01 3.596707668706327632e-01
6.577845458065525452e-01 8.451659780810962808e-01 3.602086612732601778e-01
6.674047070379289792e-01 8.470513147981415525e-01 3.611542742755425861e-01
6.769617561788032756e-01 8.489198363378159806e-01 3.625096347273936148e-01
6.864487597795673191e-01 8.507748949282539774e-01 3.642665641101618390e-01
6.958544314564799604e-01 8.526212799355240568e-01 3.664154684319869681e-01
7.051685608468114541e-01 8.544637256207057163e-01 3.689436786046771388e-01
7.143830272263600456e-01 8.563065614925247093e-01 3.718358172740615641e-01
7.234917624309317175e-01 8.581536540348341235e-01 3.750744951817871486e-01
7.324906484647774052e-01 8.600083717860309562e-01 3.786409937540124448e-01
7.413773645706981386e-01 8.618735721244006331e-01 3.825158976261986421e-01
7.501511992657796668e-01 8.637516069265696039e-01 3.866796514151401021e-01
7.588128421172509741e-01 8.656443435632366068e-01 3.911130257986148440e-01
7.673641682613402404e-01 8.675531974405399360e-01 3.957974875667232828e-01
7.758080262912036007e-01 8.694791724028596569e-01 4.007154759905482422e-01
7.841480375461038488e-01 8.714229056785223193e-01 4.058505933213660266e-01
7.923777293356836227e-01 8.733886508286130557e-01 4.111824877427081026e-01
8.004976303968860396e-01 8.753781889640523950e-01 4.166935560611597644e-01
8.085242857272323391e-01 8.773871783186646400e-01 4.223760515178233144e-01
8.164630734789560806e-01 8.794151844938148388e-01 4.282186311869483064e-01
8.243194501554683695e-01 8.814616081475756815e-01 4.342112747863317024e-01
8.320860387263339097e-01 8.835308362945183402e-01 4.403358497380424619e-01
8.397544323444476877e-01 8.856278479633188372e-01 4.465723185371322512e-01
8.473543986252204396e-01 8.877421380157632935e-01 4.529306634208433713e-01
8.548913210362114601e-01 8.898726582646793171e-01 4.594053245849991085e-01
8.623409541601502193e-01 8.920307544658760968e-01 4.659650403812060637e-01
8.697191661117472661e-01 8.942111604773197442e-01 4.726125804953009157e-01
8.770479480763140323e-01 8.964055876253053112e-01 4.793596015320920611e-01
8.843061388708378656e-01 8.986242192828276520e-01 4.861771826919926709e-01
8.914967901846199139e-01 9.008669432468144889e-01 4.930589607663434237e-01
8.986506618699680038e-01 9.031210853874221955e-01 5.000298200873778409e-01
9.057328617844134788e-01 9.054032588350297006e-01 5.070444917818385244e-01
9.127681739145864226e-01 9.077032841253237505e-01 5.141229319104883011e-01
9.197719823668246697e-01 9.100148294768658497e-01 5.212774789419143406e-01
9.266999503758117651e-01 9.123594905222979223e-01 5.284479861571415027e-01
9.336093927737403320e-01 9.147111822755604749e-01 5.356989519972219504e-01
9.404610328906413130e-01 9.170893351552455997e-01 5.429753047678268496e-01
9.472803518326599059e-01 9.194825361628593541e-01 5.503044468166357062e-01
9.540659681238262690e-01 9.218921210725944393e-01 5.576799909240343078e-01
9.608049809199471492e-01 9.243252266483533708e-01 5.650790057480892248e-01
9.675287370768704820e-01 9.267668902399696096e-01 5.725413863443260531e-01
9.741967269037244970e-01 9.292382142036349491e-01 5.800041593344547053e-01
9.808627042040826138e-01 9.317124815536732552e-01 5.875425838151492330e-01
9.874684104099172854e-01 9.342202886448683907e-01 5.950648878797101249e-01
9.940805805099582892e-01 9.367275819156850591e-01 6.026699962989522374e-01]; 

   case 'spe' 
      RGB = [9.996253193176977137e-01 9.913711226010460953e-01 8.041012438578545307e-01
9.969312990878144154e-01 9.865865913107011442e-01 7.958196545688069889e-01
9.942533588637104680e-01 9.818135789307643746e-01 7.875317815897165952e-01
9.915896776086415842e-01 9.770525904709529419e-01 7.792374356109948996e-01
9.889384786221749879e-01 9.723041153469224041e-01 7.709364896057565586e-01
9.862980251266783016e-01 9.675686302753326862e-01 7.626288656679628408e-01
9.836666169060123144e-01 9.628466015967408476e-01 7.543145233681930462e-01
9.810425876106124710e-01 9.581384871880828102e-01 7.459934495167190871e-01
9.784237290846492519e-01 9.534448589527805273e-01 7.376670490866494845e-01
9.758091741853186507e-01 9.487660072025493330e-01 7.293335612360094533e-01
9.731976797213667263e-01 9.441023023821585314e-01 7.209921595340745837e-01
9.705876565172376624e-01 9.394541905537218129e-01 7.126429103405369503e-01
9.679775344953384097e-01 9.348221172710475813e-01 7.042858810456844587e-01
9.653657609756586266e-01 9.302065285603877687e-01 6.959211353452218196e-01
9.627508763245108403e-01 9.256078555762781157e-01 6.875485248304887831e-01
9.601317913231469658e-01 9.210264571207541495e-01 6.791669211618720503e-01
9.575068348330096901e-01 9.164628209777989643e-01 6.707767168399573210e-01
9.548744491996995487e-01 9.119174176425877132e-01 6.623780173986024700e-01
9.522330808045905703e-01 9.073907239128325974e-01 6.539709177027834830e-01
9.495811770290348841e-01 9.028832240271246201e-01 6.455555018977681137e-01
9.469171829214290126e-01 8.983954108674836458e-01 6.371318441574638225e-01
9.442402190659517913e-01 8.939276490907958062e-01 6.286979645113369708e-01
9.415486437855169477e-01 8.894804711503382366e-01 6.202539985663386712e-01
9.388403761261178149e-01 8.850545069722992597e-01 6.118014161078325630e-01
9.361137658947894513e-01 8.806503085987659185e-01 6.033403762527677072e-01
9.333671427931780062e-01 8.762684428171652051e-01 5.948710444843188228e-01
9.305988122871574619e-01 8.719094924143574454e-01 5.863935979844183688e-01
9.278070514355759579e-01 8.675740573997056115e-01 5.779082318269497254e-01
9.249901047433721768e-01 8.632627561697012730e-01 5.694151660419812799e-01
9.221469800733509414e-01 8.589760860685742294e-01 5.609116482547054083e-01
9.192753048560652340e-01 8.547148108589553983e-01 5.523997440377725887e-01
9.163728560349155838e-01 8.504796735679325259e-01 5.438810466983201586e-01
9.134376867190158178e-01 8.462713800742609482e-01 5.353560484875481418e-01
9.104678046010745707e-01 8.420906570028113824e-01 5.268253096213930675e-01
9.074611700430375016e-01 8.379382516634488187e-01 5.182894693241438810e-01
9.044156948810763152e-01 8.338149316350844664e-01 5.097492574011085464e-01
9.013292420875533839e-01 8.297214839374013051e-01 5.012055062134753713e-01
8.981996264328737656e-01 8.256587137313211588e-01 4.926591628979440363e-01
8.950246162923013449e-01 8.216274424893472705e-01 4.841113016411668357e-01
8.918019367410795484e-01 8.176285055788921063e-01 4.755631357855747976e-01
8.885292740744069606e-01 8.136627492060259925e-01 4.670160295098501613e-01
8.852042818763621312e-01 8.097310266740884721e-01 4.584715087957387802e-01
8.818245887426444662e-01 8.058341939216442373e-01 4.499312713647861117e-01
8.783878077356160885e-01 8.019731043176190344e-01 4.413971952459587733e-01
8.748915476159051519e-01 7.981486027081815537e-01 4.328713456201330745e-01
8.713334258529761289e-01 7.943615187300834268e-01 4.243559795823733105e-01
8.677110833677242896e-01 7.906126594284177411e-01 4.158535484698712703e-01
8.640222009043423412e-01 7.869028012426558805e-01 4.073666974243692063e-01
8.602645168678531018e-01 7.832326814525909509e-01 3.988982618942717440e-01
8.564358463998196225e-01 7.796029892044549214e-01 3.904512608343593816e-01
8.525341014005127782e-01 7.760143562656642846e-01 3.820288864300384613e-01
8.485573111443696082e-01 7.724673476829161389e-01 3.736344902575098326e-01
8.445036430802366212e-01 7.689624525410815314e-01 3.652715658890242079e-01
8.403714233625397823e-01 7.655000750378188057e-01 3.569437280604416673e-01
8.361591566273345322e-01 7.620805260995172636e-01 3.486546886323707573e-01
8.318655445115968883e-01 7.587040157667198637e-01 3.404082296912668837e-01
8.274895024172935765e-01 7.553706465705579687e-01 3.322081742472747790e-01
8.230301740454328829e-01 7.520804081054507373e-01 3.240583550855133943e-01
8.184868212302832680e-01 7.488332616294635091e-01 3.159617825674137515e-01
8.138590110822148116e-01 7.456289896604103573e-01 3.079221377655331771e-01
8.091468181242912339e-01 7.424670871619818424e-01 2.999443469091602199e-01
8.043504122379883103e-01 7.393470563716729727e-01 2.920319686576324791e-01
7.994702184716161453e-01 7.362682828578653860e-01 2.841884012704181672e-01
7.945069109411802000e-01 7.332300399833050486e-01 2.764168555307422448e-01
7.894614033518293494e-01 7.302314948629916591e-01 2.687203309843908539e-01
7.843348364120679150e-01 7.272717156577747089e-01 2.611015959667907227e-01
7.791285260147496894e-01 7.243496994364320152e-01 2.535630684860531447e-01
7.738440801480908071e-01 7.214643092911102729e-01 2.461071963221510561e-01
7.684833208817296590e-01 7.186143199358207001e-01 2.387362039122769009e-01
7.630481315977482026e-01 7.157984953860191402e-01 2.314517734491695622e-01
7.575405230514773436e-01 7.130155515836559266e-01 2.242553190375982108e-01
7.519626112766101267e-01 7.102641669372077304e-01 2.171479941172158035e-01
7.463165956980535309e-01 7.075429925763011552e-01 2.101307026608347228e-01
7.406047378440419049e-01 7.048506621488696000e-01 2.032041138079452858e-01
7.348297350913484127e-01 7.021856197788983733e-01 1.963692968886621149e-01
7.289936844492124202e-01 6.995466025992004289e-01 1.896260481458106884e-01
7.230988483362374986e-01 6.969322773880018973e-01 1.829743149302962002e-01
7.171475322086006132e-01 6.943412998823550453e-01 1.764139943240976283e-01
7.111420143928013360e-01 6.917723467781027313e-01 1.699448658068684892e-01
7.050845338608371371e-01 6.892241206236302542e-01 1.635666199792324693e-01
6.989772797030316953e-01 6.866953538838792559e-01 1.572788888364401172e-01
6.928223822517928232e-01 6.841848122132092591e-01 1.510812776309110872e-01
6.866220524799201419e-01 6.816912363109779438e-01 1.449735254345622115e-01
6.803786701652846380e-01 6.792133090629258740e-01 1.389555620195359609e-01
6.740936196655853418e-01 6.767501306945983286e-01 1.330266252977832520e-01
6.677687244983904202e-01 6.743006234739316040e-01 1.271865644636745452e-01
6.614057275351008514e-01 6.718637487365417549e-01 1.214353970802317662e-01
6.550062900453353931e-01 6.694385066322930955e-01 1.157733573283348805e-01
6.485719915801509972e-01 6.670239355272904458e-01 1.102009495253571669e-01
6.421043305819508218e-01 6.646191111168818777e-01 1.047190080781650323e-01
6.356047256163376291e-01 6.622231453008022850e-01 9.932876523540395963e-02
6.290745171295046845e-01 6.598351848668921882e-01 9.403192821288078318e-02
6.225149696438274649e-01 6.574544100249212208e-01 8.883076747185855715e-02
6.159272743135092432e-01 6.550800328272597950e-01 8.372821811088401733e-02
6.093126663293514378e-01 6.527112545185101977e-01 7.872799706657532259e-02
6.026723170612737768e-01 6.503473055396404856e-01 7.383469856963406630e-02
5.960069280976730832e-01 6.479875866853281874e-01 6.905400752442764079e-02
5.893174254692321590e-01 6.456314203776034599e-01 6.439287527349069062e-02
5.826046784844106652e-01 6.432781520370909334e-01 5.985968924725062340e-02
5.758695032242261425e-01 6.409271483905801814e-01 5.546448455059219823e-02
5.691126660872921628e-01 6.385777958070207871e-01 5.121916775709036557e-02
5.623348873683923221e-01 6.362294986703330713e-01 4.713774156788400754e-02
5.555368448583254404e-01 6.338816777957024806e-01 4.323651096469569716e-02
5.487191774565705060e-01 6.315337688945173999e-01 3.952487864258176498e-02
5.418824887917018662e-01 6.291852210919098853e-01 3.612051121019213551e-02
5.350273508472259687e-01 6.268354954999130202e-01 3.311399967360150604e-02
5.281543075929241438e-01 6.244840638484698836e-01 3.049096132752013993e-02
5.212638786236020172e-01 6.221304071760309640e-01 2.823775650583685778e-02
5.143565628087404251e-01 6.197740145810481938e-01 2.634144880489198981e-02
5.074328419576341620e-01 6.174143820354919265e-01 2.478976657062584646e-02
5.004931845054064743e-01 6.150510112614009373e-01 2.357106560875196072e-02
4.935380492257968599e-01 6.126834086714986194e-01 2.267429306479169446e-02
4.865677337149327264e-01 6.103111278318402722e-01 2.208918519158591109e-02
4.795827833081557912e-01 6.079336535225863258e-01 2.180564619063158835e-02
4.725836555273193462e-01 6.055504983495709759e-01 2.181419194008693205e-02
4.655708088649732068e-01 6.031611773523495312e-01 2.210580170001537684e-02
4.585447116306416437e-01 6.007652058968021569e-01 2.267187969798983502e-02
4.515058459835388782e-01 5.983620990282547680e-01 2.350422445657735990e-02
4.444547120266517104e-01 5.959513709030562767e-01 2.459499875403014374e-02
4.373918319618326778e-01 5.935325343022211930e-01 2.593670014651517156e-02
4.303177543037495223e-01 5.911051002310937497e-01 2.752213198538535840e-02
4.232330581488202292e-01 5.886685776092354105e-01 2.934437486930205341e-02
4.161383574931313278e-01 5.862224730549979723e-01 3.139675847884770832e-02
4.090343055913983616e-01 5.837662907693373926e-01 3.367283375013734037e-02
4.019215993467592507e-01 5.812995325235387201e-01 3.616634535426579283e-02
3.948009837190709082e-01 5.788216977554323517e-01 3.887120446073646235e-02
3.876732561372571162e-01 5.763322837785355146e-01 4.174672181203834681e-02
3.805386564470196742e-01 5.738309171411589693e-01 4.468566878513501733e-02
3.733986391498164137e-01 5.713169711328536238e-01 4.768049787246867594e-02
3.662542351622480874e-01 5.687899279044026368e-01 5.071835758395210753e-02
3.591065040473656045e-01 5.662492786415918022e-01 5.378828392041421630e-02
3.519565800474088735e-01 5.636945152858550134e-01 5.688085721725138350e-02
3.448056760545124555e-01 5.611251315161035480e-01 5.998797256299469999e-02
3.376550874229136134e-01 5.585406238589376571e-01 6.310263770132876204e-02
3.305061956006355439e-01 5.559404929241524851e-01 6.621879773242350664e-02
3.233604715607960034e-01 5.533242447607230607e-01 6.933118466356372189e-02
3.162194790161148017e-01 5.506913923265978061e-01 7.243518928216058361e-02
3.090846039651472532e-01 5.480415027343880086e-01 7.552721746757809496e-02
3.019574288899705139e-01 5.453741321890830385e-01 7.860394456462099777e-02
2.948403194287022577e-01 5.426887379559618418e-01 8.166128713902018332e-02
2.877352499045566225e-01 5.399848761881764769e-01 8.469633286585029341e-02
2.806442994214729536e-01 5.372621189537871711e-01 8.770643445761908130e-02
2.735696535538801322e-01 5.345200561570193631e-01 9.068916512095565041e-02
2.665136059049172390e-01 5.317582974656462902e-01 9.364228167891575083e-02
2.594785595977833204e-01 5.289764742223507232e-01 9.656369418956520234e-02
2.524670287821700332e-01 5.261742413169625543e-01 9.945144107016240520e-02
2.454813557062531792e-01 5.233513142453691813e-01 1.023041162767710510e-01
2.385245828710040317e-01 5.205073603004944927e-01 1.051194831525161522e-01
2.315996737216303170e-01 5.176421037021866622e-01 1.078956959800008442e-01
2.247096070811669399e-01 5.147553123030387257e-01 1.106311405318080310e-01
2.178574824667257048e-01 5.118467857547527311e-01 1.133242655958566492e-01
2.110465244393363582e-01 5.089163566221597268e-01 1.159735773093806543e-01
2.042800881777394606e-01 5.059638913007615812e-01 1.185776351459725819e-01
1.975616665475327660e-01 5.029892907239723598e-01 1.211350492651094846e-01
1.908950317685597087e-01 4.999924798399172921e-01 1.236442618572499708e-01
1.842839642352289697e-01 4.969734333710090213e-01 1.261039978961912833e-01
1.777324020667393756e-01 4.939321620495928378e-01 1.285130506256617899e-01
1.712444848635372441e-01 4.908687102327232155e-01 1.308702368677320538e-01
1.648245567449286297e-01 4.877831568788963401e-01 1.331744245804179216e-01
1.584771896665797541e-01 4.846756148389082530e-01 1.354245338303780855e-01
1.522072108246106115e-01 4.815462299379932865e-01 1.376195378692431359e-01
1.460197345452351747e-01 4.783951798632230523e-01 1.397584642177332193e-01
1.399204939260773328e-01 4.752226638142488802e-01 1.418397751955981501e-01
1.339151022806790436e-01 4.720289313790105301e-01 1.438629155677119409e-01
1.280096392738992173e-01 4.688142535267130762e-01 1.458273227721058329e-01
1.222107627052576584e-01 4.655789223639808516e-01 1.477322650340249788e-01
1.165256528622910237e-01 4.623232537320298152e-01 1.495770677617926092e-01
1.109620746853187123e-01 4.590475853271560047e-01 1.513611132825743999e-01
1.055284430115649430e-01 4.557522747771929894e-01 1.530838402977779955e-01
1.002338888313226428e-01 4.524376976954341267e-01 1.547447430577089666e-01
9.508832342249537439e-02 4.491042457325262194e-01 1.563433702619193288e-01
9.010250937510871916e-02 4.457523324675090604e-01 1.578790892676192326e-01
8.528801833375249108e-02 4.423823989146767888e-01 1.593510674113139958e-01
8.065727940861416867e-02 4.389948528858526600e-01 1.607597457256386420e-01
7.622371216564308161e-02 4.355901307746828932e-01 1.621048834368656322e-01
7.200158271604989446e-02 4.321686757028083692e-01 1.633862841279137557e-01
6.800589086138220107e-02 4.287309359317861834e-01 1.646037933778843332e-01
6.425218103335716968e-02 4.252773633715483670e-01 1.657572962977645059e-01
6.075626148835750612e-02 4.218084121914437157e-01 1.668467149892300661e-01
5.753381975108700502e-02 4.183245375382101949e-01 1.678720059529140995e-01
5.459992971435845971e-02 4.148261943636451510e-01 1.688331574715942474e-01
5.196845773512390881e-02 4.113138363632788397e-01 1.697301869925023354e-01
4.965139124510086627e-02 4.077879150260529384e-01 1.705631385315065085e-01
4.765813208284918473e-02 4.042488787938245398e-01 1.713320801202671551e-01
4.599481449154869256e-02 4.006971723285012166e-01 1.720371013157572515e-01
4.466371969486728627e-02 3.971332358837184051e-01 1.726783107897287561e-01
4.366274483561478209e-02 3.935575133766276990e-01 1.732557671739120286e-01
4.298497468310508857e-02 3.899704687794856572e-01 1.737693705957654433e-01
4.262017645185838671e-02 3.863724827023667929e-01 1.742196255705478480e-01
4.255352467768171859e-02 3.827639728113220174e-01 1.746066936456434071e-01
4.276673184831410873e-02 3.791453506945430818e-01 1.749307448926006592e-01
4.323878239983177524e-02 3.755170217599948512e-01 1.751919563188165385e-01
4.394675081101364483e-02 3.718793851987791665e-01 1.753905104335057585e-01
4.486662442701292580e-02 3.682328340080037177e-01 1.755265939730583369e-01
4.597406436707644067e-02 3.645777550668193867e-01 1.756003967899375517e-01
4.724505801823831314e-02 3.609145292591857124e-01 1.756121109084000098e-01
4.865643856140024898e-02 3.572435316368365310e-01 1.755619297495205344e-01
5.018626632042253594e-02 3.535651316158259783e-01 1.754500475272437465e-01
5.181408104443557122e-02 3.498796931999255677e-01 1.752766588164567374e-01
5.352104295989750654e-02 3.461875752240315962e-01 1.750419582933170903e-01
5.528998425553303259e-02 3.424891316106106198e-01 1.747461406473147794e-01
5.710539287663336100e-02 3.387847116320403806e-01 1.743894006636908245e-01
5.895334846203271334e-02 3.350746601715560713e-01 1.739719334739151524e-01
6.082142709649813322e-02 3.313593179753057116e-01 1.734939349708606859e-01
6.269858808390260663e-02 3.276390218878380001e-01 1.729556023841174184e-01
6.457505267241087088e-02 3.239141050631572094e-01 1.723571350094970367e-01
6.644218183629055363e-02 3.201848971432968427e-01 1.716987350852014205e-01
6.829235792686130790e-02 3.164517243962254311e-01 1.709806088053268391e-01
7.011887323479765177e-02 3.127149098048046527e-01 1.702029674593475428e-01
7.191582719453681882e-02 3.089747730984767071e-01 1.693660286839586415e-01
7.367740139700959534e-02 3.052317018390342529e-01 1.684698187582648887e-01
7.539908659075802988e-02 3.014860094828903936e-01 1.675145933863426417e-01
7.707730571895973770e-02 2.977379606831319081e-01 1.665007348086736949e-01
7.870852955781817983e-02 2.939878648964006636e-01 1.654284970811750188e-01
8.028965953863020921e-02 2.902360278178472974e-01 1.642981478645798299e-01
8.181797774419891089e-02 2.864827509313960796e-01 1.631099700093367744e-01
8.329110360766753263e-02 2.827283309501391617e-01 1.618642630930951787e-01
8.470695659424687385e-02 2.789730591425138573e-01 1.605613448745624727e-01
8.606372418387428502e-02 2.752172205414125661e-01 1.592015526248838075e-01
8.735983452008261319e-02 2.714610930351273321e-01 1.577852442954886525e-01
8.859393314120664331e-02 2.677049463410028363e-01 1.563127994796906506e-01
8.976486326081492551e-02 2.639490408648579312e-01 1.547846201243904263e-01
9.087164911293293956e-02 2.601936264515474218e-01 1.532011309481255412e-01
9.191348192284298779e-02 2.564389410344570241e-01 1.515627795225681362e-01
9.288704312629730842e-02 2.526856315239155992e-01 1.498693699931691048e-01
9.379283018859288501e-02 2.489337717503207204e-01 1.481216905842226428e-01
9.463163644057562274e-02 2.451833792964967507e-01 1.463205652098015508e-01
9.540318566942779244e-02 2.414346401225993533e-01 1.444665315338557743e-01
9.610732643393554708e-02 2.376877206774311024e-01 1.425601463139360425e-01
9.674402718144511915e-02 2.339427661170380146e-01 1.406019829499289553e-01
9.731337192794831115e-02 2.301998985206034631e-01 1.385926284969124511e-01
9.781527030012060475e-02 2.264592671684997338e-01 1.365326231802845014e-01
9.824358966693200190e-02 2.227221381992704474e-01 1.344213498047777955e-01
9.860476008591259611e-02 2.189874851640599696e-01 1.322606854144605382e-01
9.889926458171927059e-02 2.152553323101172311e-01 1.300512525669721697e-01
9.912768720083675600e-02 2.115256719866073221e-01 1.277936680974917916e-01
9.929070945217863264e-02 2.077984630697710111e-01 1.254885372707029156e-01
9.938536693535210409e-02 2.040743841747782450e-01 1.231359093599569132e-01
9.940987528000957973e-02 2.003539028956775048e-01 1.207361104943659447e-01
9.937074432198822471e-02 1.966357733029605592e-01 1.182905659078862248e-01
9.926899950314096999e-02 1.929198155529378012e-01 1.157998086675786908e-01
9.910574746809375224e-02 1.892058087816931022e-01 1.132643300609986470e-01
9.887273365069987330e-02 1.854955476839638961e-01 1.106837415179452166e-01
9.857702759168757156e-02 1.817875303344013982e-01 1.080592207487474365e-01
9.822269753736714848e-02 1.780808239249605796e-01 1.053914161709541830e-01
9.781113633355650872e-02 1.743750377566490040e-01 1.026806351620476176e-01
9.733238005935909709e-02 1.706723687329834982e-01 9.992670564763897478e-02
9.679537330416121410e-02 1.669706979907544520e-01 9.713043093317094701e-02
9.620476860769888727e-02 1.632687911401897729e-01 9.429209466525781402e-02
9.555840111219859878e-02 1.595669718044100127e-01 9.141177435708905397e-02
9.484851695793727888e-02 1.558669291806107915e-01 8.848983773912671991e-02
9.408911153954402362e-02 1.521650578561620781e-01 8.552628708305054506e-02
9.328192094338655371e-02 1.484606091474970080e-01 8.252094903627574252e-02
9.241015243151923242e-02 1.447574737795673805e-01 7.947496936453349314e-02
9.149313879389489590e-02 1.410504572025015335e-01 7.638732537156053826e-02
9.053276383981978537e-02 1.373386075843833487e-01 7.325761429945673586e-02]; 

   case 'the' 
      RGB = [1.555601333154079877e-02 1.382442454646408414e-01 2.018108864558305071e-01
1.620183633850513089e-02 1.410507428866217272e-01 2.089765125440807836e-01
1.685648942708358952e-02 1.438270143621834252e-01 2.162386804476043589e-01
1.752640064782528401e-02 1.465717250667996630e-01 2.235996996833259920e-01
1.821871873545745021e-02 1.492834638238061673e-01 2.310618693528040390e-01
1.894137836902154426e-02 1.519607349643580241e-01 2.386274839825403005e-01
1.969967580211434005e-02 1.546014513385217670e-01 2.463049741539924953e-01
2.050331512714091350e-02 1.572037794856419868e-01 2.540971092002056730e-01
2.136720981691051999e-02 1.597664500864496573e-01 2.619991459036808412e-01
2.230340677460975612e-02 1.622875536944013708e-01 2.700132114112391291e-01
2.332520459934065912e-02 1.647650548255009395e-01 2.781413941166567261e-01
2.444727823684482437e-02 1.671967824349823439e-01 2.863857270216258466e-01
2.568581629165060318e-02 1.695804203313354408e-01 2.947481661145329168e-01
2.705867185137392564e-02 1.719134976584987262e-01 3.032305630924797546e-01
2.858552764394725262e-02 1.741933796245943022e-01 3.118346316168387755e-01
3.028807626872338093e-02 1.764172587161002559e-01 3.205619061457981589e-01
3.219021609777513587e-02 1.785821467123839268e-01 3.294136922283489310e-01
3.431826321592867240e-02 1.806848679100377109e-01 3.383910069732901649e-01
3.670117943457189280e-02 1.827220540829147533e-01 3.474945082255425644e-01
3.937081594703384368e-02 1.846901418460877020e-01 3.567244107935257369e-01
4.230474116360182640e-02 1.865853732643254770e-01 3.660803878810173217e-01
4.544128431656369732e-02 1.884038007524381497e-01 3.755614556934170345e-01
4.879889459256544354e-02 1.901412975600472455e-01 3.851658390246698871e-01
5.238564999909287728e-02 1.917935754201579024e-01 3.948908155057742619e-01
5.620896927989652708e-02 1.933562112707872260e-01 4.047325361348014794e-01
6.027560971317643540e-02 1.948246853300575621e-01 4.146858197468099028e-01
6.459519137924149557e-02 1.961877478810097886e-01 4.247714594384868758e-01
6.917293885235362150e-02 1.974458334676339466e-01 4.349572770994127868e-01
7.401397924875777190e-02 1.985943734629224688e-01 4.452322549865626589e-01
7.912632580773251711e-02 1.996251445642635014e-01 4.555965555444902448e-01
8.452074659970570947e-02 2.005284212595413451e-01 4.660508732253974551e-01
9.019392390291772199e-02 2.013079448693051998e-01 4.765478779135125520e-01
9.616430834359415702e-02 2.019472529514862447e-01 4.871044455010897223e-01
1.024253987466881288e-01 2.024520213616345932e-01 4.976646226688278829e-01
1.089944270249512126e-01 2.028088855832538839e-01 5.082270885561682716e-01
1.158597351626668159e-01 2.030273547727350913e-01 5.187245256305992314e-01
1.230424263443430921e-01 2.030937970306834206e-01 5.291483757425788914e-01
1.305276694613609623e-01 2.030217816424806365e-01 5.394201195013325068e-01
1.383099125215292158e-01 2.028195613734770086e-01 5.494767764101642360e-01
1.463797060150685281e-01 2.025001769277504637e-01 5.592461312000727158e-01
1.547186326280874380e-01 2.020850680852618320e-01 5.686422620965899677e-01
1.632970499077517346e-01 2.016054641193363861e-01 5.775676861911989146e-01
1.720728231749732162e-01 2.011027291278179585e-01 5.859183845130113699e-01
1.809917622181062835e-01 2.006270524284321510e-01 5.935918186078640302e-01
1.899902199574210471e-01 2.002341583564870575e-01 6.004970669758479263e-01
1.989997441940680178e-01 1.999802927301375655e-01 6.065651516258429021e-01
2.079529779081165097e-01 1.999164256738651391e-01 6.117570630961892686e-01
2.167895160216459782e-01 2.000830340237900185e-01 6.160673924886361785e-01
2.254604257150365498e-01 2.005067081654643424e-01 6.195227918364496489e-01
2.339306302023402284e-01 2.011991984057693306e-01 6.221760825969698816e-01
2.421790683949211487e-01 2.021587162723321729e-01 6.240979346009031259e-01
2.501971277034634733e-01 2.033727314506766359e-01 6.253682433520739714e-01
2.579861074778473928e-01 2.048213417591974728e-01 6.260687943604991146e-01
2.655544225081862275e-01 2.064804694302365684e-01 6.262779744769025880e-01
2.729150389729214088e-01 2.083244557603688429e-01 6.260675767301320249e-01
2.800833915559314269e-01 2.103279194298604549e-01 6.255013265069014894e-01
2.870675074033041674e-01 2.124691386407265292e-01 6.246420009551492125e-01
2.938851360627945386e-01 2.147254304183257023e-01 6.235380183242871244e-01
3.005577213461362307e-01 2.170756514619715527e-01 6.222251903623999825e-01
3.070843773107185259e-01 2.195059802015210115e-01 6.207554984633752992e-01
3.134916285549604886e-01 2.219983100391352271e-01 6.191452159520036691e-01
3.197798380866583856e-01 2.245420356602937373e-01 6.174343435010863912e-01
3.259695527216341926e-01 2.271241237863139140e-01 6.156329324235468858e-01
3.320579089479565038e-01 2.297373625924510887e-01 6.137771467682858750e-01
3.380660052498126178e-01 2.323715331127139128e-01 6.118660583919267593e-01
3.439917420525137604e-01 2.350215076527270575e-01 6.099280657295180763e-01
3.498460685618723365e-01 2.376808945869600675e-01 6.079694872801727490e-01
3.556384580181179977e-01 2.403444043084654869e-01 6.059953497614897211e-01
3.613686295558476425e-01 2.430089153836439420e-01 6.040241385974032262e-01
3.670434468015162932e-01 2.456707832167414618e-01 6.020607616318822686e-01
3.726708785615932551e-01 2.483266964722255499e-01 6.001059364008434205e-01
3.782547954216434749e-01 2.509743622345630976e-01 5.981656274234778969e-01
3.837960822688916696e-01 2.536122834643630419e-01 5.962498766549472196e-01
3.892988345534371120e-01 2.562387736058999166e-01 5.943618049976240325e-01
3.947690991030736729e-01 2.588520700720948198e-01 5.924996351953921714e-01
4.002098946086877773e-01 2.614510659770345469e-01 5.906660913059953444e-01
4.056240909379103532e-01 2.640348396856425084e-01 5.888633104220148962e-01
4.110144223076953041e-01 2.666026246434279878e-01 5.870929096766683841e-01
4.163832172801885667e-01 2.691538344975527575e-01 5.853565951388559618e-01
4.217313231007376317e-01 2.716882525666767800e-01 5.836583632684495537e-01
4.270634345309600177e-01 2.742050306264073312e-01 5.819941042354170868e-01
4.323818104897113601e-01 2.767038027770706843e-01 5.803639256565888971e-01
4.376886243188534142e-01 2.791842625143060586e-01 5.787676418773346487e-01
4.429859696643737577e-01 2.816461519867726748e-01 5.772048027503156042e-01
4.482758654902720408e-01 2.840892531259122666e-01 5.756747186760452495e-01
4.535602602863915700e-01 2.865133804028727749e-01 5.741764823913096949e-01
4.588410355259065487e-01 2.889183750000149931e-01 5.727089879419176022e-01
4.641200084236148382e-01 2.913041002126984247e-01 5.712709472315609105e-01
4.693989340425177015e-01 2.936704379215548943e-01 5.698609044983501404e-01
4.746795067933508583e-01 2.960172859965477521e-01 5.684772490345112450e-01
4.799633613697112389e-01 2.983445565121622955e-01 5.671182264323321176e-01
4.852520731601697723e-01 3.006521746683436525e-01 5.657819486102849682e-01
4.905471581781303825e-01 3.029400783247205298e-01 5.644664028469250638e-01
4.958500725502145712e-01 3.052082180664184574e-01 5.631694600262432404e-01
5.011622116043696895e-01 3.074565577287760587e-01 5.618888822762294621e-01
5.064849085998188727e-01 3.096850753156841218e-01 5.606223301621769961e-01
5.118194331420761189e-01 3.118937642524056697e-01 5.593673695773668797e-01
5.171669893275794294e-01 3.140826349187781363e-01 5.581214784559522801e-01
5.225287136638799845e-01 3.162517164128481606e-01 5.568820534159194535e-01
5.279056728126164666e-01 3.184010584984696690e-01 5.556464164236831760e-01
5.332988612036093645e-01 3.205307336933521101e-01 5.544118215561065766e-01
5.387091985692399332e-01 3.226408394566178117e-01 5.531754619204082291e-01
5.441375274486375258e-01 3.247315004373090841e-01 5.519344767774559957e-01
5.495846107110635703e-01 3.268028707475315597e-01 5.506859588994191812e-01
5.550511291471261766e-01 3.288551362262008837e-01 5.494269621786239677e-01
5.605376791749926424e-01 3.308885166617086537e-01 5.481545094908906179e-01
5.660447559653561944e-01 3.329032737201366166e-01 5.468656290966650291e-01
5.715723572408055730e-01 3.348998704862335973e-01 5.455581199822507887e-01
5.771213104289429907e-01 3.368784486432290226e-01 5.442280104688511644e-01
5.826918609018115758e-01 3.388393806259411556e-01 5.428722728323092106e-01
5.882841610050709713e-01 3.407830824811499126e-01 5.414878910241607279e-01
5.938982692324897839e-01 3.427100153754324974e-01 5.400718694374621043e-01
5.995341497366565298e-01 3.446206869260000083e-01 5.386212416598933350e-01
6.051916721861749782e-01 3.465156523493675422e-01 5.371330791454219655e-01
6.108706119725975103e-01 3.483955154267821541e-01 5.356044997333838653e-01
6.165706507631435462e-01 3.502609292895154658e-01 5.340326759432469927e-01
6.222913773881607602e-01 3.521125970311436149e-01 5.324148429745680922e-01
6.280322890453092777e-01 3.539512721578863541e-01 5.307483063447554494e-01
6.337927927958686425e-01 3.557777588917046541e-01 5.290304491020394462e-01
6.395721854014132512e-01 3.575929224264804973e-01 5.272587829694529438e-01
6.453697498455218673e-01 3.593976455700171879e-01 5.254307632896803026e-01
6.511846984697030605e-01 3.611928549527803622e-01 5.235439096149940852e-01
6.570160963803246545e-01 3.629795579684892415e-01 5.215959730192853971e-01
6.628629283562820218e-01 3.647588132682739737e-01 5.195848109675399451e-01
6.687241012633005077e-01 3.665317306708469891e-01 5.175083912494201632e-01
6.745984464659754432e-01 3.682994710922303794e-01 5.153647953022981731e-01
6.804847221871188623e-01 3.700632465224873990e-01 5.131522209386339961e-01
6.863816157645579175e-01 3.718243200760849021e-01 5.108689845040222943e-01
6.922877457570544291e-01 3.735840061411623281e-01 5.085135225034096429e-01
6.982016638532075881e-01 3.753436706510840937e-01 5.060843927435032530e-01
7.041218565401857754e-01 3.771047314992945210e-01 5.035802750493091340e-01
7.100468845018954589e-01 3.788685911907690995e-01 5.009996451582281463e-01
7.159752518731292703e-01 3.806367017978663503e-01 4.983410633235164089e-01
7.219050254760910335e-01 3.824107548202065887e-01 4.956041045679995816e-01
7.278344332015346252e-01 3.841923879692217270e-01 4.927879483140180095e-01
7.337616407032357957e-01 3.859832960516177969e-01 4.898918977085310877e-01
7.396847510296571393e-01 3.877852324461281697e-01 4.869153794944616753e-01
7.456018038508585022e-01 3.896000107579556393e-01 4.838579440403712462e-01
7.515107742785041012e-01 3.914295066368485010e-01 4.807192656291401911e-01
7.574095712835625660e-01 3.932756597383701980e-01 4.774991431076147097e-01
7.632960357235062387e-01 3.951404758009958162e-01 4.741975009994948143e-01
7.691679379984673881e-01 3.970260288041012053e-01 4.708143911830061090e-01
7.750229753641669772e-01 3.989344631636088656e-01 4.673499952331199858e-01
7.808587689384376418e-01 4.008679959130103665e-01 4.638046275250513051e-01
7.866728604480874854e-01 4.028289188075652172e-01 4.601787391913814695e-01
7.924627087737056153e-01 4.048196002786729752e-01 4.564729230191116316e-01
7.982256863620216247e-01 4.068424871537119070e-01 4.526879193650572009e-01
8.039590755885861473e-01 4.089001060440320967e-01 4.488246231577860956e-01
8.096600651680347926e-01 4.109950642903736351e-01 4.448840920414381395e-01
8.153258420561760866e-01 4.131300072582320126e-01 4.408672005707100494e-01
8.209533846301936277e-01 4.153077124896901728e-01 4.367753702901882029e-01
8.265394587133882975e-01 4.175310847974926798e-01 4.326106584748535266e-01
8.320808476618816174e-01 4.198030498289601065e-01 4.283748912056176694e-01
8.375742298072511582e-01 4.221266063894242304e-01 4.240701127101388912e-01
8.430161781421201539e-01 4.245048216497959159e-01 4.196985997897492715e-01
8.484031611801808870e-01 4.269408247931844591e-01 4.152628771038663902e-01
8.537315452228415591e-01 4.294377989032187592e-01 4.107657331305429871e-01
8.589975982777037222e-01 4.319989708985695898e-01 4.062102365684474026e-01
8.641974958823694930e-01 4.346275993272235572e-01 4.015997528869852951e-01
8.693273290890853877e-01 4.373269598520185819e-01 3.969379606684441675e-01
8.743831148591909574e-01 4.401003282876529976e-01 3.922288673203608855e-01
8.793608090992056647e-01 4.429509610905758565e-01 3.874768236699652202e-01
8.842563225402819693e-01 4.458820732584555802e-01 3.826865368883226592e-01
8.890655396175203284e-01 4.488968136663485375e-01 3.778630811333164030e-01
8.937843693689974112e-01 4.519982404973394430e-01 3.730116014425700621e-01
8.984086715918495614e-01 4.551892900487020666e-01 3.681382748140558103e-01
9.029343899688814234e-01 4.584727367754222738e-01 3.632494215164466245e-01
9.073575586033109097e-01 4.618511663085561048e-01 3.583516614288673741e-01
9.116743277656134126e-01 4.653269421603676848e-01 3.534519722899168159e-01
9.158809992051784032e-01 4.689021723136149178e-01 3.485576699619067353e-01
9.199740625655835613e-01 4.725786767270368505e-01 3.436763818638891022e-01
9.239502320744639174e-01 4.763579567194722864e-01 3.388160133952918263e-01
9.278064825504417357e-01 4.802411672570275347e-01 3.339847073943023048e-01
9.315400836682339314e-01 4.842290931875970483e-01 3.291907969273191181e-01
9.351486313601243827e-01 4.883221304365225612e-01 3.244427519772630775e-01
9.386300752174523421e-01 4.925202730905588466e-01 3.197491208755643965e-01
9.419827407976890665e-01 4.968231071526066356e-01 3.151184675888816233e-01
9.452053458453827384e-01 5.012298115494860928e-01 3.105593062092800172e-01
9.482970095990234105e-01 5.057391667267315816e-01 3.060800341879720277e-01
9.512572545755481057e-01 5.103495708806106146e-01 3.016888659817511531e-01
9.540860004899643920e-01 5.150590635745326828e-01 2.973937688352701891e-01
9.567835502644689294e-01 5.198653561843505910e-01 2.932024023936338208e-01
9.593505683914355098e-01 5.247658683349885056e-01 2.891220637265379811e-01
9.617880522174245828e-01 5.297577692490467172e-01 2.851596391521897811e-01
9.640972969908582213e-01 5.348380227429672118e-01 2.813215639873033469e-01
9.662798557460373639e-01 5.400034344900018768e-01 2.776137910349112392e-01
9.683374952663609259e-01 5.452507001277228094e-01 2.740417682745930894e-01
9.702721494708056449e-01 5.505764528210163045e-01 2.706104258619907443e-01
9.720889605050128113e-01 5.559744240639272750e-01 2.673323057285149629e-01
9.737882391976736551e-01 5.614431075234689317e-01 2.642051375197135843e-01
9.753722517954447335e-01 5.669791681305653697e-01 2.612321043607351290e-01
9.768433321430202154e-01 5.725792962307277856e-01 2.584161030939570725e-01
9.782037550763590383e-01 5.782403184217296266e-01 2.557595328984585970e-01
9.794556966363962003e-01 5.839592236834146854e-01 2.532643241222839459e-01
9.806011989875685897e-01 5.897331838357899869e-01 2.509319710696591987e-01
9.816445119805891073e-01 5.955576445021050214e-01 2.487674614053176358e-01
9.825957966513861885e-01 6.014235301335163486e-01 2.467838886220111161e-01
9.834484089207010671e-01 6.073354334336248384e-01 2.449669167474177733e-01
9.842037798764916579e-01 6.132913094304376367e-01 2.433164315861336413e-01
9.848630901763529844e-01 6.192893396808167861e-01 2.418320393357376030e-01
9.854335855591603854e-01 6.253232095344845032e-01 2.405207443125048083e-01
9.859266054527725531e-01 6.313839151833324781e-01 2.393924955683320310e-01
9.863290788081210403e-01 6.374805817839866995e-01 2.384285005271009616e-01
9.866412575432023102e-01 6.436122274808632193e-01 2.376272705850792089e-01
9.868743031654042541e-01 6.497702193582076680e-01 2.369977788720153966e-01
9.870387195949218428e-01 6.559468364085242476e-01 2.365460008515470891e-01
9.871160198111447182e-01 6.621544902577213287e-01 2.362516434857291903e-01
9.871054832314208882e-01 6.683929123881801049e-01 2.361125093170109435e-01
9.870452158568817635e-01 6.746361467497961062e-01 2.361551640855951706e-01
9.869008071972434903e-01 6.809069196703931848e-01 2.363489608363473771e-01
9.866692878063002548e-01 6.872064512263130753e-01 2.366900769944741967e-01
9.863929517876753872e-01 6.935072010148546351e-01 2.372025181179537867e-01
9.860358081545986808e-01 6.998320059248218650e-01 2.378591308273384497e-01
9.855936903889445100e-01 7.061828277187244263e-01 2.386553467137359497e-01
9.851166760681809853e-01 7.125285611733751523e-01 2.396137846497558566e-01
9.845515404596055786e-01 7.189016348706930293e-01 2.407027465239837682e-01
9.839219208542443473e-01 7.252873073536283410e-01 2.419304971970357987e-01
9.832411577231264799e-01 7.316776102841332508e-01 2.432981102176038357e-01
9.824686360853925882e-01 7.380960010628512258e-01 2.447853973017022899e-01
9.816675490572134288e-01 7.445058075421049359e-01 2.464118597749720974e-01
9.807811261674898029e-01 7.509394073137781733e-01 2.481527624504580309e-01
9.798377198237012697e-01 7.573804700508821597e-01 2.500137146942527089e-01
9.788392654586509645e-01 7.638279008720277874e-01 2.519905378906656113e-01
9.777574675384567149e-01 7.702969047505190403e-01 2.540716600569991046e-01
9.766479350869351483e-01 7.767573771217611833e-01 2.562664070834250185e-01
9.754356977304895482e-01 7.832491288133777152e-01 2.585534361426328198e-01
9.742107025732685832e-01 7.897246981741067318e-01 2.609472103669700505e-01
9.728835679779427315e-01 7.962305102434532600e-01 2.634261983272434549e-01
9.715297353896644728e-01 8.027276438790406088e-01 2.659994820979044161e-01
9.700861348373609472e-01 8.092479763321948072e-01 2.686528986078252079e-01
9.686057099481873989e-01 8.157648637457888263e-01 2.713899293880046582e-01
9.670430691608270513e-01 8.223006128282058791e-01 2.742007328066633498e-01
9.654377156641301694e-01 8.288358297838601674e-01 2.770858440538391254e-01
9.637524161478506768e-01 8.353882936883590959e-01 2.800376262549509332e-01
9.620231537308009395e-01 8.419407901977008502e-01 2.830554448577937698e-01
9.602104885023859948e-01 8.485116373841136150e-01 2.861325388392048086e-01
9.583576301078738924e-01 8.550807238511934916e-01 2.892681513706822360e-01
9.564117528707279936e-01 8.616719638797333269e-01 2.924557053792812833e-01
9.544348419615947821e-01 8.682572954460299197e-01 2.956947937778980906e-01
9.523487086523741985e-01 8.748712522095133393e-01 2.989788164145827376e-01
9.502464564071694264e-01 8.814728114137494464e-01 3.023077648025994102e-01
9.480141307284709606e-01 8.881110895508603775e-01 3.056751963404329420e-01
9.457819798963852387e-01 8.947301765899248194e-01 3.090811208408512645e-01
9.434198646152930356e-01 9.013849121338212145e-01 3.125200216900483885e-01
9.410286159691669816e-01 9.080328518084608280e-01 3.159906396320988908e-01
9.385331275251967975e-01 9.147047487143539213e-01 3.194893313796646206e-01
9.359711083432949996e-01 9.213848126149789541e-01 3.230138415459749002e-01
9.333375985220991877e-01 9.280748341844049509e-01 3.265615469169362850e-01
9.305915654403119630e-01 9.347905093445134650e-01 3.301299810381577715e-01
9.278142496620381818e-01 9.414998557883345054e-01 3.337168327083647745e-01
9.248899165263079203e-01 9.482470197415310276e-01 3.373199677733378365e-01
9.219411401959984875e-01 9.549849147932994997e-01 3.409370492608514991e-01
9.188613878011584468e-01 9.617532494963948464e-01 3.445662997232426528e-01
9.156931782520092433e-01 9.685354904254356301e-01 3.482056900726151483e-01
9.124490701578419349e-01 9.753266872784461805e-01 3.518533597970244786e-01
9.090418416674036495e-01 9.821574063216705897e-01 3.555078064299531104e-01]; 

   case 'tur' 
      RGB = [9.128247827303703765e-01 9.639053479101408195e-01 6.723488894068933019e-01
9.105541439463002984e-01 9.592872094872512134e-01 6.663907644186453094e-01
9.082861583229014935e-01 9.546805250318116665e-01 6.604691147501789983e-01
9.060452334346256187e-01 9.500783067250512248e-01 6.545538618591559832e-01
9.038150489008132116e-01 9.454849862587190179e-01 6.486649327173524826e-01
9.016037749037892901e-01 9.408980925645457072e-01 6.427923205892623892e-01
8.994100526612220925e-01 9.363178430715828338e-01 6.369376634451241470e-01
8.972283335855062436e-01 9.317456838096750404e-01 6.311075587918794083e-01
8.950696728809198754e-01 9.271782394664687121e-01 6.252888115672152747e-01
8.929173737868275618e-01 9.226202315494650419e-01 6.195011901939884158e-01
8.907923116926522722e-01 9.180653547565199579e-01 6.137200788543796248e-01
8.886692522817520867e-01 9.135209242332905655e-01 6.079750370293484085e-01
8.865762636352767512e-01 9.089784066893161762e-01 6.022333770048255985e-01
8.844822115233714754e-01 9.044469910654970857e-01 5.965311430834745465e-01
8.824197090505849772e-01 8.999166548769963470e-01 5.908308404756164034e-01
8.803543805345617201e-01 8.953977033216384829e-01 5.851717778972109762e-01
8.783207074075700671e-01 8.908794045328807254e-01 5.795148392301145979e-01
8.762837291194094380e-01 8.863723903264537629e-01 5.738994936029852001e-01
8.742771906432560414e-01 8.818660105043367725e-01 5.682879912596860983e-01
8.722681388310039585e-01 8.773704198332279436e-01 5.627170493987142530e-01
8.702869566172963811e-01 8.728758815872331711e-01 5.571531747195710427e-01
8.683053185700908561e-01 8.683912265313378231e-01 5.516275115467151879e-01
8.663476627856837586e-01 8.639084851022803546e-01 5.461135394992869818e-01
8.643928371591906856e-01 8.594343049131268897e-01 5.406342206369587622e-01
8.624568202066339451e-01 8.549633517080399425e-01 5.351725180271732496e-01
8.605281096749168857e-01 8.504992167172934492e-01 5.297408107341389227e-01
8.586117879987910095e-01 8.460400804193519697e-01 5.243338350869022335e-01
8.567083921128474389e-01 8.415855961633933457e-01 5.189512185545770429e-01
8.548097683783190126e-01 8.371383437933764826e-01 5.136015164012288636e-01
8.529307767408889074e-01 8.326931553267608033e-01 5.082696912104300857e-01
8.510506087990278301e-01 8.282569318779919865e-01 5.029770800265680464e-01
8.491921882796159560e-01 8.238216896030918779e-01 4.977007922336022516e-01
8.473354714528938958e-01 8.193941476796476886e-01 4.924611248003303854e-01
8.454893810507302376e-01 8.149710832043290942e-01 4.872494055687389136e-01
8.436548921629284381e-01 8.105519834307682858e-01 4.820650051378080891e-01
8.418197380538857688e-01 8.061410299518165790e-01 4.769199747107543685e-01
8.400053197367541857e-01 8.017304745289870471e-01 4.717941759727964368e-01
8.381909179373602248e-01 7.973275360052080041e-01 4.667076177307682427e-01
8.363830265089613469e-01 7.929297607375542789e-01 4.616544156277210820e-01
8.345877920103869085e-01 7.885347495856604993e-01 4.566293264683093933e-01
8.327919089080556558e-01 7.841472126590459668e-01 4.516448413369297810e-01
8.310063135234079246e-01 7.797629900031863848e-01 4.466914503101058753e-01
8.292275089883606176e-01 7.753832051359705879e-01 4.417727747802380756e-01
8.274479714210091208e-01 7.710105395274703399e-01 4.368957512003675547e-01
8.256785592470122781e-01 7.666407630966191045e-01 4.320514550589309999e-01
8.239135967944541949e-01 7.622758578672201857e-01 4.272453035219374029e-01
8.221475640634913207e-01 7.579178098056661428e-01 4.224823211006376589e-01
8.203879070453153899e-01 7.535636173491093714e-01 4.177568735562176006e-01
8.186335117171099629e-01 7.492135470811618347e-01 4.130705335781763021e-01
8.168774297351566460e-01 7.448701849739395309e-01 4.084293549358841147e-01
8.151208269598283485e-01 7.405329639445877854e-01 4.038328842986566580e-01
8.133730042127403914e-01 7.361980289394938204e-01 3.992747817294182155e-01
8.116225817355652294e-01 7.318697849631130570e-01 3.947642928691043052e-01
8.098694933912269356e-01 7.275481580762414024e-01 3.903019868836329342e-01
8.081161421647475862e-01 7.232320558774254504e-01 3.858867812020280175e-01
8.063663538470418057e-01 7.189197605534187741e-01 3.815168716651668457e-01
8.046125622407066524e-01 7.146142683294098852e-01 3.771981838442269308e-01
8.028545343811845925e-01 7.103155798131343124e-01 3.729315169481128289e-01
8.010919901003339394e-01 7.060237163039633224e-01 3.687177325408004802e-01
7.993300055097595225e-01 7.017364019961147559e-01 3.645545873969476269e-01
7.975634994838864955e-01 6.974556802865253813e-01 3.604460667960328046e-01
7.957906047709230046e-01 6.931822615073386373e-01 3.563940392198953200e-01
7.940108757837693876e-01 6.889162546167795220e-01 3.523995277667577586e-01
7.922238209776574225e-01 6.846577925964051348e-01 3.484635956840971271e-01
7.904293753935112132e-01 6.804068218754933950e-01 3.445871200261476641e-01
7.886299371669703850e-01 6.761621751401072355e-01 3.407699499113064912e-01
7.868208547545553211e-01 6.719258532155115704e-01 3.370151794848580962e-01
7.850015016585555339e-01 6.676980777850795024e-01 3.333239143922641090e-01
7.831712152651345571e-01 6.634790940463787257e-01 3.296972647086952590e-01
7.813292995863233559e-01 6.592691702526117803e-01 3.261363364073748827e-01
7.794750283869652518e-01 6.550685970372218669e-01 3.226422224503434077e-01
7.776076486673221266e-01 6.508776865233487641e-01 3.192159935965748763e-01
7.757263844650232887e-01 6.466967712234903409e-01 3.158586890319579621e-01
7.738304409335882150e-01 6.425262027384857078e-01 3.125713069333992955e-01
7.719190086491207747e-01 6.383663502685200664e-01 3.093547950848793415e-01
7.699919228068800026e-01 6.342172763144198200e-01 3.062099646060428282e-01
7.680475840592058123e-01 6.300797563557358760e-01 3.031377731717001534e-01
7.660850744742599971e-01 6.259542495514858196e-01 3.001389551573931946e-01
7.641036016288820232e-01 6.218411653176698639e-01 2.972141439455896483e-01
7.621023798026289597e-01 6.177409214840726692e-01 2.943638837738908332e-01
7.600806343242360041e-01 6.136539422033919777e-01 2.915886230831420400e-01
7.580376057793873912e-01 6.095806558220696614e-01 2.888887086342720734e-01
7.559725540291885038e-01 6.055214927394206859e-01 2.862643804685457427e-01
7.538847619930633126e-01 6.014768832813116584e-01 2.837157677714013393e-01
7.517735391550399715e-01 5.974472556137478962e-01 2.812428856843471325e-01
7.496385821697367779e-01 5.934328401156596655e-01 2.788457594157026098e-01
7.474790247249349928e-01 5.894341791373612915e-01 2.765241346674250367e-01
7.452941571551999766e-01 5.854517449359323278e-01 2.742776443121395791e-01
7.430834135426160891e-01 5.814859431780246002e-01 2.721058338160937673e-01
7.408462686359136296e-01 5.775371675495697410e-01 2.700081319859798934e-01
7.385822390693008721e-01 5.736057983656746018e-01 2.679838537450121017e-01
7.362908841899132861e-01 5.696922013455091305e-01 2.660322037116858995e-01
7.339718065076965559e-01 5.657967265562297010e-01 2.641522805055408485e-01
7.316246517864810617e-01 5.619197075270785380e-01 2.623430816971141777e-01
7.292491087989990683e-01 5.580614605321673194e-01 2.606035093141773062e-01
7.268449087719649482e-01 5.542222840379915638e-01 2.589323758133949549e-01
7.244119971505563749e-01 5.504023530996886571e-01 2.573286095051418587e-01
7.219503397818252122e-01 5.466018316756788842e-01 2.557911025974954899e-01
7.194594088198653647e-01 5.428211930643935812e-01 2.543180251101427314e-01
7.169390970998404944e-01 5.390606612983260826e-01 2.529078914226555730e-01
7.143893337983732161e-01 5.353204418367376594e-01 2.515591605975798783e-01
7.118100825627617922e-01 5.316007218689121627e-01 2.502702433238302993e-01
7.092013395547599464e-01 5.279016707079033921e-01 2.490395087311264022e-01
7.065632761648270588e-01 5.242233436125348645e-01 2.478655721838471937e-01
7.038960056028840118e-01 5.205658319141027723e-01 2.467469222620103930e-01
7.011993601253575514e-01 5.169294192806536126e-01 2.456813814978507926e-01
6.984734491802424561e-01 5.133142072042380377e-01 2.446672141412145063e-01
6.957184043559518916e-01 5.097202822466132544e-01 2.437026738057547215e-01
6.929343773945654261e-01 5.061477167394171639e-01 2.427860085204449625e-01
6.901216982990902027e-01 5.025964514073363310e-01 2.419159412674971588e-01
6.872804633072713276e-01 4.990665929836670123e-01 2.410905423858894503e-01
6.844107686966893755e-01 4.955582518851005536e-01 2.403077709414531138e-01
6.815128311469199618e-01 4.920714492259936068e-01 2.395658950635757289e-01
6.785868785067301623e-01 4.886061954699182919e-01 2.388631970725307307e-01
6.756332326667190413e-01 4.851624213327032087e-01 2.381983436213906957e-01
6.726521288960277678e-01 4.817401209176369048e-01 2.375696985886235346e-01
6.696437188359446457e-01 4.783393588339723279e-01 2.369751854422654791e-01
6.666082635296274317e-01 4.749601080715157297e-01 2.364131671114735878e-01
6.635460289039749604e-01 4.716023339061982678e-01 2.358820311548202042e-01
6.604573673748913576e-01 4.682659130292578520e-01 2.353807610155446706e-01
6.573424797961275878e-01 4.649508592384817285e-01 2.349074379402631418e-01
6.542016086091390070e-01 4.616571492422122946e-01 2.344603126221481149e-01
6.510350327978842166e-01 4.583847209004847101e-01 2.340378795902509079e-01
6.478430515953373936e-01 4.551334824230311438e-01 2.336388774485689268e-01
6.446259633611755024e-01 4.519033296020167900e-01 2.332622032236857934e-01
6.413839940492124247e-01 4.486942500114745047e-01 2.329058642884746511e-01
6.381174276440877424e-01 4.455061586981217681e-01 2.325684660088059297e-01
6.348265500932614991e-01 4.423389620941622913e-01 2.322486831842977550e-01
6.315116605695774155e-01 4.391925158791953887e-01 2.319458531661305334e-01
6.281730104223729461e-01 4.360667833034455043e-01 2.316579279478628295e-01
6.248108867019236401e-01 4.329616581049553492e-01 2.313836187886625928e-01
6.214255770673621226e-01 4.298770280576157399e-01 2.311216752923600515e-01
6.180173538237878628e-01 4.268127440894281532e-01 2.308715541128374960e-01
6.145865084649400067e-01 4.237687269357882092e-01 2.306313089541623396e-01
6.111333338376842006e-01 4.207448478787260138e-01 2.303997483300764260e-01
6.076581198732585731e-01 4.177409699743503957e-01 2.301757903742778777e-01
6.041611292458852756e-01 4.147569442408376994e-01 2.299587629276800271e-01
6.006426901866908086e-01 4.117926430199113641e-01 2.297469460659757323e-01
5.971031071270087587e-01 4.088479120598013661e-01 2.295392371763493311e-01
5.935426680352169360e-01 4.059225898260850895e-01 2.293347588094398759e-01
5.899616653081597439e-01 4.030165124281641087e-01 2.291325734316880802e-01
5.863604503326285133e-01 4.001295067793814719e-01 2.289312648345211421e-01
5.827393462725786177e-01 3.972613890222796984e-01 2.287298138873416486e-01
5.790986411902667719e-01 3.944119738157387811e-01 2.285275119372468522e-01
5.754386920094946012e-01 3.915810608797961612e-01 2.283231701874827158e-01
5.717598595710956522e-01 3.887684389212705538e-01 2.281156877135243621e-01
5.680624888217489232e-01 3.859738913687448258e-01 2.279041499218198430e-01
5.643468987836690598e-01 3.831972016166527162e-01 2.276878398717704366e-01
5.606135014398240246e-01 3.804381236067138072e-01 2.274655664744977823e-01
5.568626686009898741e-01 3.776964124183660454e-01 2.272364567122592827e-01
5.530947703612836275e-01 3.749718169415875435e-01 2.269997025952595338e-01
5.493102061694418170e-01 3.722640707136085636e-01 2.267544107214977123e-01
5.455093844474366849e-01 3.695728971647626593e-01 2.264997151134776621e-01
5.416927086216569709e-01 3.668980144097471752e-01 2.262348350480499204e-01
5.378606217947595747e-01 3.642391205981750923e-01 2.259588836659274236e-01
5.340135354481294616e-01 3.615959178208362768e-01 2.256711679956127647e-01
5.301518754638107067e-01 3.589680973433583278e-01 2.253709864348782954e-01
5.262761094769513592e-01 3.563553292394111560e-01 2.250575362145477709e-01
5.223867172868102982e-01 3.537572724565644089e-01 2.247300369556790578e-01
5.184841017313481792e-01 3.511736091955309780e-01 2.243880288453043437e-01
5.145687147834636654e-01 3.486039985632924942e-01 2.240309158662728284e-01
5.106411440519553757e-01 3.460480410893026493e-01 2.236577184811773256e-01
5.067017881423235837e-01 3.435054056903972253e-01 2.232681227570667559e-01
5.027510796252681047e-01 3.409757445575836710e-01 2.228617265136721426e-01
4.987895070653633467e-01 3.384586834157125024e-01 2.224379910665530979e-01
4.948177548651969127e-01 3.359537625951555251e-01 2.219958720410014630e-01
4.908360971463728295e-01 3.334606870942352086e-01 2.215355358785864315e-01
4.868450056329213793e-01 3.309790783774143597e-01 2.210565950309411609e-01
4.828450375686827445e-01 3.285085184570587513e-01 2.205584694191147777e-01
4.788368359138705510e-01 3.260485487451708631e-01 2.200404115431305319e-01
4.748206472212548879e-01 3.235988770790072522e-01 2.195027100981302715e-01
4.707969362808618885e-01 3.211591169778729160e-01 2.189450971836008897e-01
4.667662693795122109e-01 3.187288356425135860e-01 2.183670869460304087e-01
4.627293116701075015e-01 3.163075541790801304e-01 2.177680072916463594e-01
4.586862340695841422e-01 3.138950081558932736e-01 2.171483462825098965e-01
4.546374811311343911e-01 3.114908131915107847e-01 2.165079490118044792e-01
4.505834924512453488e-01 3.090945859801553230e-01 2.158466824363790559e-01
4.465251651502277208e-01 3.067057384379430762e-01 2.151634647284257906e-01
4.424624800332205288e-01 3.043240886593481798e-01 2.144591818414338102e-01
4.383958498551490668e-01 3.019492632236420726e-01 2.137337799541155214e-01
4.343256784780819557e-01 2.995808923379119637e-01 2.129872269867968959e-01
4.302525491830334059e-01 2.972185251864662980e-01 2.122191462426318287e-01
4.261769801706936645e-01 2.948617395754311588e-01 2.114293130588631442e-01
4.220990387565126678e-01 2.925103169029278360e-01 2.106183606559913768e-01
4.180190866768337399e-01 2.901639064725308748e-01 2.097863381928281035e-01
4.139374748855114139e-01 2.878221627482029921e-01 2.089333114395358071e-01
4.098546564000305481e-01 2.854846942476605975e-01 2.080591616116443943e-01
4.057712419033978057e-01 2.831510381788182595e-01 2.071635039021454405e-01
4.016871580256169971e-01 2.808210427866469350e-01 2.062471500335373853e-01
3.976027082875563945e-01 2.784943860456884357e-01 2.053102247884384113e-01
3.935181846712299536e-01 2.761707517515646915e-01 2.043528642900468983e-01
3.894338675796663596e-01 2.738498295626687340e-01 2.033752150347126197e-01
3.853501485284500094e-01 2.715312592611229259e-01 2.023772355615125473e-01
3.812675049241057712e-01 2.692146423201499661e-01 2.013587500039031863e-01
3.771858334106182320e-01 2.668998447434733912e-01 2.003204988638387918e-01
3.731053664536302938e-01 2.645865804728991244e-01 1.992626621753412208e-01
3.690263252725575205e-01 2.622745690957888343e-01 1.981854256387428070e-01
3.649489199677356521e-01 2.599635357439662453e-01 1.970889798104242807e-01
3.608733496668369289e-01 2.576532109746325627e-01 1.959735193223784977e-01
3.567998026886545215e-01 2.553433306346174492e-01 1.948392421320772061e-01
3.527288062385475209e-01 2.530334781151396539e-01 1.936858449182176090e-01
3.486601920186409020e-01 2.507235512004114542e-01 1.925140265725779620e-01
3.445940882588859333e-01 2.484133136793199026e-01 1.913240314183973223e-01
3.405306410460574029e-01 2.461025213487854080e-01 1.901160637330516767e-01
3.364699867832531277e-01 2.437909343510991644e-01 1.888903276956251931e-01
3.324122524192171801e-01 2.414783169649092898e-01 1.876470268273142949e-01
3.283575556825768516e-01 2.391644373879673879e-01 1.863863634598928731e-01
3.243060053196711312e-01 2.368490675124892975e-01 1.851085382313104599e-01
3.202577013347079893e-01 2.345319826939236685e-01 1.838137496073975197e-01
3.162127352310978301e-01 2.322129615138344705e-01 1.825021934285724068e-01
3.121713481873937823e-01 2.298917159046080516e-01 1.811738589048320913e-01
3.081335147958294551e-01 2.275680750195018809e-01 1.798290697596431065e-01
3.040992288572926250e-01 2.252418581754896398e-01 1.784681064582642751e-01
3.000685503066236048e-01 2.229128543385760497e-01 1.770911493801810288e-01
2.960415318405872354e-01 2.205808544095470558e-01 1.756983743063298686e-01
2.920182191423447704e-01 2.182456509647304199e-01 1.742899520962044868e-01
2.879986511013646333e-01 2.159070379936562922e-01 1.728660483803914072e-01
2.839828600283449411e-01 2.135648106336759944e-01 1.714268232671206371e-01
2.799708718647850314e-01 2.112187649015158653e-01 1.699724310614039724e-01
2.759627063869375951e-01 2.088686974216710790e-01 1.685030199953476415e-01
2.719583774039363577e-01 2.065144051514603007e-01 1.670187319682087390e-01
2.679578929499804030e-01 2.041556851024882158e-01 1.655197022947889340e-01
2.639612554705231817e-01 2.017923340581795344e-01 1.640060594607524180e-01
2.599684620025124460e-01 1.994241482869618665e-01 1.624779248834790635e-01
2.559795043487851918e-01 1.970509232505948671e-01 1.609354126770728421e-01
2.519943692468365470e-01 1.946724533070429697e-01 1.593786294201690046e-01
2.480130385322499298e-01 1.922885314071993146e-01 1.578076739252100846e-01
2.440354892972040890e-01 1.898989487846577950e-01 1.562226370078938775e-01
2.400616940445691738e-01 1.875034946376172018e-01 1.546236012555405048e-01
2.360916208382524140e-01 1.851019558018774935e-01 1.530106407931814128e-01
2.321252334505864434e-01 1.826941164137449169e-01 1.513838210462468969e-01
2.281624915077672400e-01 1.802797575614995262e-01 1.497431984988283338e-01
2.242033506345226801e-01 1.778586569239049842e-01 1.480888204466117852e-01
2.202477625994911925e-01 1.754305883940150690e-01 1.464207247437575854e-01
2.162956754630863121e-01 1.729953216862958332e-01 1.447389395432186476e-01
2.123470337300349953e-01 1.705526219247719688e-01 1.430434830302904636e-01
2.084017785092666109e-01 1.681022492095620435e-01 1.413343631495822261e-01
2.044598476844710633e-01 1.656439581587243193e-01 1.396115773261282222e-01
2.005211760994463077e-01 1.631774974218037078e-01 1.378751121820577796e-01
1.965856988864305155e-01 1.607026079587815515e-01 1.361249399201315824e-01
1.926533421135208923e-01 1.582190261881047944e-01 1.343610282823611279e-01
1.887240316449029232e-01 1.557264800392861304e-01 1.325833310557416600e-01
1.847976928565723820e-01 1.532246885529702785e-01 1.307917884786300444e-01
1.808742494403644818e-01 1.507133617751950094e-01 1.289863286797117148e-01
1.769536238533599426e-01 1.481921999716426241e-01 1.271668674796427312e-01
1.730357378708484994e-01 1.456608927537091369e-01 1.253333082519507424e-01
1.691205132766324948e-01 1.431191180966186194e-01 1.234855418704042807e-01
1.652078727370975830e-01 1.405665412234658185e-01 1.216234467798024410e-01
1.612977409237918991e-01 1.380028133201234297e-01 1.197468892401933882e-01
1.573900459762036241e-01 1.354275700331004917e-01 1.178557238119506412e-01
1.534847214366352741e-01 1.328404296837557008e-01 1.159497941721605863e-01
1.495817088501062986e-01 1.302409911046356894e-01 1.140289343827514401e-01
1.456809613162400874e-01 1.276288309622561901e-01 1.120929707687035870e-01
1.417824484273444985e-01 1.250035003675231404e-01 1.101417246101454861e-01
1.378861632611661503e-01 1.223645204774909678e-01 1.081750159008208478e-01
1.339921324751868759e-01 1.197113766395997425e-01 1.061926684632616136e-01]; 

   case 'del'
      RGB = [6.597738601379860013e-02 1.238600499381984077e-01 2.494811599712867811e-01
6.865757658541371544e-02 1.266324956800233548e-01 2.555762447808356264e-01
7.132312021900141796e-02 1.293951489111998809e-01 2.616639066260320057e-01
7.396584278365506138e-02 1.321405798946303223e-01 2.677947952348482819e-01
7.658629323761953489e-02 1.348691618055914976e-01 2.739690401973952083e-01
7.919241629978698849e-02 1.375884267331394517e-01 2.801420635219376565e-01
8.176925318412744947e-02 1.402841919021537431e-01 2.864049855987979565e-01
8.433406612792618273e-02 1.429731754896261531e-01 2.926562894753481636e-01
8.687299291862357609e-02 1.456421509856402619e-01 2.989801852647154812e-01
8.939564312621503528e-02 1.483008215820366127e-01 3.053194113218902772e-01
9.189721302857964402e-02 1.509448355999061520e-01 3.117030007960572280e-01
9.437767086491094526e-02 1.535746031892240571e-01 3.181307112978130625e-01
9.684032408298676176e-02 1.561940161218882128e-01 3.245815662969073756e-01
9.927796547462605647e-02 1.587964393005784070e-01 3.310975577575396844e-01
1.016993930631445442e-01 1.613915446377360008e-01 3.376235806086802516e-01
1.040929763606517067e-01 1.639684016677392386e-01 3.442267209045115073e-01
1.064700341792961802e-01 1.665394889067980111e-01 3.508360219569039429e-01
1.088176547667218386e-01 1.690928617928972866e-01 3.575240510639015601e-01
1.111462403295262191e-01 1.716403013777595965e-01 3.642247640300367850e-01
1.134452622240499642e-01 1.741726760902385374e-01 3.709940519253031033e-01
1.157201477866866002e-01 1.766970342921256598e-01 3.777940670973845005e-01
1.179671104159276951e-01 1.792114205608784372e-01 3.846393215051533421e-01
1.201817230210445731e-01 1.817136143480379551e-01 3.915459113873127617e-01
1.223722204783651013e-01 1.842136886386882277e-01 3.984598273467554463e-01
1.245183567208241193e-01 1.866951930885134003e-01 4.054791326498645709e-01
1.266365389448252943e-01 1.891762365105242871e-01 4.125046892271796994e-01
1.287143460345438750e-01 1.916485918438771141e-01 4.195883119105888182e-01
1.307469548160320461e-01 1.941117894471887095e-01 4.267380618771365319e-01
1.327408430973604225e-01 1.965752427165133320e-01 4.339060811804626439e-01
1.346828243039070450e-01 1.990323873082787132e-01 4.411366760117488295e-01
1.365667394636138066e-01 2.014836986376687955e-01 4.484350095699139449e-01
1.383960163689361422e-01 2.039374403932631141e-01 4.557622019510663702e-01
1.401636755967774206e-01 2.063944663974329741e-01 4.631231418544201062e-01
1.418540865672596740e-01 2.088512567417482035e-01 4.705504424282677678e-01
1.434598643146114128e-01 2.113116730887077965e-01 4.780359058662089766e-01
1.449764155851626657e-01 2.137823044275377615e-01 4.855569960092488979e-01
1.463910004146321830e-01 2.162663403203769485e-01 4.931143386027379560e-01
1.476886377469379952e-01 2.187680022007208458e-01 5.007063827574367298e-01
1.488517278855215897e-01 2.212928540992544768e-01 5.083284550145572567e-01
1.498596541692162332e-01 2.238481772327865493e-01 5.159715446979967757e-01
1.506883975687731414e-01 2.264434023743985280e-01 5.236207927150543506e-01
1.513102256748371788e-01 2.290905757285599531e-01 5.312536911875026524e-01
1.516842522456439690e-01 2.318030303219635324e-01 5.388545690381982833e-01
1.517570170754133085e-01 2.345981922008114062e-01 5.464014367414956608e-01
1.515076197810979464e-01 2.375036749636269540e-01 5.537960248334482527e-01
1.508527167578624095e-01 2.405440732043884755e-01 5.610095372512794443e-01
1.497779955302820376e-01 2.437529689807156341e-01 5.679039092169722025e-01
1.482412957217069116e-01 2.471597695361844038e-01 5.743778793359424206e-01
1.462677401356168583e-01 2.507850610764148502e-01 5.802953371129300209e-01
1.439355627545357286e-01 2.546290083660951442e-01 5.855507465326762473e-01
1.413540769625598603e-01 2.586695514989866829e-01 5.901033497204623002e-01
1.386407864858154870e-01 2.628687111034194723e-01 5.939785160116007878e-01
1.358917366388572578e-01 2.671843825843605580e-01 5.972486725556511722e-01
1.331956301615234428e-01 2.715742302472205494e-01 6.000063534918327335e-01
1.305811350178066321e-01 2.760102973743540078e-01 6.023428893290788677e-01
1.280671927186234904e-01 2.804711462805791200e-01 6.043358297286517411e-01
1.256919911974779258e-01 2.849349146429925872e-01 6.060559435528747319e-01
1.234353694092780451e-01 2.893960415099317007e-01 6.075503879105464966e-01
1.213123684738988406e-01 2.938438245262708359e-01 6.088643085732736715e-01
1.193314037280503725e-01 2.982710543730990871e-01 6.100344852684282948e-01
1.174733533629070403e-01 3.026784028742006138e-01 6.110815570440700784e-01
1.157410513288365805e-01 3.070626658343806881e-01 6.120282423760157187e-01
1.141526088065836497e-01 3.114180400183615416e-01 6.129002396427197796e-01
1.126916634208409429e-01 3.157472412451859389e-01 6.137061650582799066e-01
1.113553786510952659e-01 3.200503527841166984e-01 6.144572065766925606e-01
1.101437722183151724e-01 3.243271585465217766e-01 6.151637952792941011e-01
1.090564770800171446e-01 3.285777842283311712e-01 6.158346267162948529e-01
1.080928143315710299e-01 3.328026048887181565e-01 6.164769721944358682e-01
1.072518450875257212e-01 3.370021743369620570e-01 6.170969325182951160e-01
1.065324069530126649e-01 3.411771714425693713e-01 6.176996445247795453e-01
1.059331395146229648e-01 3.453283595087877078e-01 6.182894491232916456e-01
1.054525023039967757e-01 3.494565556897754055e-01 6.188700280655170527e-01
1.050887878891080540e-01 3.535626081054415448e-01 6.194445153697799578e-01
1.048401320993113395e-01 3.576473788404232468e-01 6.200155882238653771e-01
1.047045228648097182e-01 3.617117314301156461e-01 6.205855412748909616e-01
1.046798087267215294e-01 3.657565217601439489e-01 6.211563474636030424e-01
1.047637077327939481e-01 3.697825915556448573e-01 6.217297079493111500e-01
1.049538171612311999e-01 3.737907638293131996e-01 6.223070931774267178e-01
1.052476243004594747e-01 3.777818398051627224e-01 6.228897767433481114e-01
1.056425183464732021e-01 3.817565969485811617e-01 6.234788633863599383e-01
1.061358033549051061e-01 3.857157878203879009e-01 6.240753121902191669e-01
1.067247120960378159e-01 3.896601395393941014e-01 6.246799558608577829e-01
1.074064206019045564e-01 3.935903536891926513e-01 6.252935167860671495e-01
1.081780631610760601e-01 3.975071065441324047e-01 6.259166204489629015e-01
1.090367475038964695e-01 4.014110495195324368e-01 6.265498066600397875e-01
1.099820192670150909e-01 4.053023125398828030e-01 6.271951607472914247e-01
1.110081883132479630e-01 4.091820642340333603e-01 6.278512759575444191e-01
1.121122195896028517e-01 4.130509161123952500e-01 6.285183732165030568e-01
1.132912658654391114e-01 4.169094262530252948e-01 6.291967191746730137e-01
1.145425291517750410e-01 4.207581307564207673e-01 6.298865324820219769e-01
1.158632717089364272e-01 4.245975446528400532e-01 6.305879875238334931e-01
1.172508261265178542e-01 4.284281627655473490e-01 6.313012175820684746e-01
1.187026044305651562e-01 4.322504605241775932e-01 6.320263175158151725e-01
1.202170218035645832e-01 4.360647043028175740e-01 6.327640308933485391e-01
1.217903323311140817e-01 4.398716102802121553e-01 6.335133964865322653e-01
1.234201965028123016e-01 4.436716084893622125e-01 6.342743647718233069e-01
1.251044458168622253e-01 4.474650989352985664e-01 6.350469034989063566e-01
1.268410231661077081e-01 4.512524653607909442e-01 6.358309517935993860e-01
1.286279862353579828e-01 4.550340757553759663e-01 6.366264210299322768e-01
1.304635102569667859e-01 4.588102828035847680e-01 6.374331955697576380e-01
1.323456332495375476e-01 4.625814799269595823e-01 6.382509181299536039e-01
1.342726301857550264e-01 4.663480223415985004e-01 6.390792893119938700e-01
1.362431133778941039e-01 4.701102048314840798e-01 6.399181612927934415e-01
1.382557801777440920e-01 4.738683163523436659e-01 6.407673303740510917e-01
1.403094518927116008e-01 4.776226319622609018e-01 6.416265687705522414e-01
1.424030742552521711e-01 4.813734129303259279e-01 6.424956248215036858e-01
1.445357177858233311e-01 4.851209067888150872e-01 6.433742231671184530e-01
1.467060787915918652e-01 4.888654617396300250e-01 6.442615910488664888e-01
1.489131851676200047e-01 4.926073688646966375e-01 6.451570946472290347e-01
1.511571091366653297e-01 4.963466938037996434e-01 6.460609578935421204e-01
1.534374568113274773e-01 5.000836224936368035e-01 6.469728308538383876e-01
1.557539606313337044e-01 5.038183265492227614e-01 6.478923408808500151e-01
1.581064802328551455e-01 5.075509629447428894e-01 6.488190927082100323e-01
1.604950035510952222e-01 5.112816736254882644e-01 6.497526685562985405e-01
1.629196481906860228e-01 5.150105850470044766e-01 6.506926282565553832e-01
1.653806563676609720e-01 5.187378093247897448e-01 6.516385018000171447e-01
1.678770878978581016e-01 5.224637768298545648e-01 6.525882706071722827e-01
1.704107945748090658e-01 5.261882349061756114e-01 6.535428899563763272e-01
1.729824594318147002e-01 5.299112360468477556e-01 6.545018545237717422e-01
1.755929057918094727e-01 5.336328137580849118e-01 6.554646382342502742e-01
1.782431006664247641e-01 5.373529815601206794e-01 6.564306947203235598e-01
1.809341585475496006e-01 5.410717318764055594e-01 6.573994578824640111e-01
1.836673455941881250e-01 5.447890348032889962e-01 6.583703425703313350e-01
1.864440842101210971e-01 5.485048367520860557e-01 6.593427454071065785e-01
1.892659579998643982e-01 5.522190589552509188e-01 6.603160457822994100e-01
1.921347170798090864e-01 5.559315958284012371e-01 6.612896070418273764e-01
1.950522837092731887e-01 5.596423131801995243e-01 6.622627779079297561e-01
1.980207581910957138e-01 5.633510462626485360e-01 6.632348941655241692e-01
2.010424249735310309e-01 5.670575976553166031e-01 6.642052806559789468e-01
2.041197588637720939e-01 5.707617349784390726e-01 6.651732536238323945e-01
2.072554312381834074e-01 5.744631884318825987e-01 6.661381234667468343e-01
2.104523161046616408e-01 5.781616481597345869e-01 6.670991979438091191e-01
2.137134958380619842e-01 5.818567614439515978e-01 6.680557859020190836e-01
2.170422663699277943e-01 5.855481297352551628e-01 6.690072015853253395e-01
2.204421415686979580e-01 5.892353055354571101e-01 6.699527695945174388e-01
2.239168564955728025e-01 5.929177891529354705e-01 6.708918305694538953e-01
2.274703691651913662e-01 5.965950253621812305e-01 6.718237476670261277e-01
2.311068603787596043e-01 6.002664000095238039e-01 6.727479139084245885e-01
2.348307311320150803e-01 6.039312366204897531e-01 6.736637604671461554e-01
2.386465970323070063e-01 6.075887930799791503e-01 6.745707659639463838e-01
2.425592790908820962e-01 6.112382584746600678e-01 6.754684668258670310e-01
2.465737901908315322e-01 6.148787502077949219e-01 6.763564687525340791e-01
2.506953164731830497e-01 6.185093115199363778e-01 6.772344593131256474e-01
2.549296306833234715e-01 6.221293277287609502e-01 6.780979140255732895e-01
2.592822774336291380e-01 6.257372346725570411e-01 6.789506539234843041e-01
2.637588638514070660e-01 6.293318265564150638e-01 6.797927409484227912e-01
2.683650223312705752e-01 6.329118213522958447e-01 6.806243803889014954e-01
2.731063499524681304e-01 6.364758627686015746e-01 6.814459485792905280e-01
2.779906740187598757e-01 6.400228375684388071e-01 6.822520389699539001e-01
2.830218658907085461e-01 6.435508363165360901e-01 6.830490580497422526e-01
2.882048880496008714e-01 6.470582889109053326e-01 6.838383084593062655e-01
2.935458083139746988e-01 6.505436502020417455e-01 6.846186591669699562e-01
2.990503892938236596e-01 6.540052313531840023e-01 6.853899295935115266e-01
3.047196724248331101e-01 6.574412743642752410e-01 6.861585944188198782e-01
3.105583191416750322e-01 6.608501423712110912e-01 6.869242652383956704e-01
3.165691502323280671e-01 6.642301693361829518e-01 6.876887756715628353e-01
3.227495057337754214e-01 6.675799504508616034e-01 6.884592417367911832e-01
3.291028263869451576e-01 6.708979305506697077e-01 6.892345854413708395e-01
3.356230778965553774e-01 6.741831128414313978e-01 6.900227469259155866e-01
3.423073210907732200e-01 6.774345078849072221e-01 6.908268664523685709e-01
3.491497562850178760e-01 6.806514618485436374e-01 6.916515776970183493e-01
3.561412724149042863e-01 6.838337923407922236e-01 6.925029303903535993e-01
3.632736859780383298e-01 6.869814453814443445e-01 6.933849146347551562e-01
3.705348353526126681e-01 6.900949744532235419e-01 6.943034006383217438e-01
3.779122643387600178e-01 6.931752240960002975e-01 6.952632699407041983e-01
3.853938489959773395e-01 6.962232375904670034e-01 6.962682692318554745e-01
3.929633165309669440e-01 6.992408157519413026e-01 6.973239952904880523e-01
4.006094014150666793e-01 7.022293785228771457e-01 6.984322280579721154e-01
4.083159356621580693e-01 7.051912169731731073e-01 6.995972410351208870e-01
4.160701459275524816e-01 7.081283984755933902e-01 7.008208963157127602e-01
4.238620058191452378e-01 7.110427636723841704e-01 7.021034222566255867e-01
4.316725578542053299e-01 7.139377080072604187e-01 7.034496090272048807e-01
4.394994714751406240e-01 7.168142170649683953e-01 7.048555730206492731e-01
4.473284906527433269e-01 7.196752689920701274e-01 7.063236640655090604e-01
4.551495147015194864e-01 7.225233370762427221e-01 7.078541344797719681e-01
4.629611457318438261e-01 7.253594732048370686e-01 7.094433480260243785e-01
4.707534852399093972e-01 7.281862590349524877e-01 7.110918876863899785e-01
4.785169352473264692e-01 7.310063519486531547e-01 7.128002486509635860e-01
4.862541175206331334e-01 7.338201861642941193e-01 7.145637643240034809e-01
4.939610178122765816e-01 7.366294113512334985e-01 7.163810048013540266e-01
5.016310591672418218e-01 7.394362226143922356e-01 7.182517850494093414e-01
5.092596479997313352e-01 7.422424836790947333e-01 7.201752147893440981e-01
5.168502920324100636e-01 7.450484101459822206e-01 7.221473089909000720e-01
5.244013381888714687e-01 7.478552285559243451e-01 7.241664310755349110e-01
5.319115488689606375e-01 7.506640750099188297e-01 7.262309445597783242e-01
5.393800515498496928e-01 7.534759997433319034e-01 7.283392250016084146e-01
5.468044438041703703e-01 7.562923850920999502e-01 7.304903735115424457e-01
5.541839357415714318e-01 7.591142604468823496e-01 7.326829886087474764e-01
5.615214997302344635e-01 7.619417788590748808e-01 7.349143832677361710e-01
5.688172658612942190e-01 7.647756983657992835e-01 7.371830791535139982e-01
5.760715228891868378e-01 7.676167208435804579e-01 7.394876421854478243e-01
5.832846906737159109e-01 7.704654971547720832e-01 7.418266842236616032e-01
5.904572957555410673e-01 7.733226320830066669e-01 7.441988638633715292e-01
5.975899498408560051e-01 7.761886890212207346e-01 7.466028864742004778e-01
6.046833309596043593e-01 7.790641943891402077e-01 7.490375036038013912e-01
6.117381670592270115e-01 7.819496417678519773e-01 7.515015118481881418e-01
6.187552218007135174e-01 7.848454957475322624e-01 7.539937512752844517e-01
6.257352823331345792e-01 7.877521954909707524e-01 7.565131034734732252e-01
6.326791488358145532e-01 7.906701580203755464e-01 7.590584892833606157e-01
6.395876256322543529e-01 7.935997812385013894e-01 7.616288662585695146e-01
6.464615136961924247e-01 7.965414466975205832e-01 7.642232258900538699e-01
6.533016043869401823e-01 7.994955221304983484e-01 7.668405906180205678e-01
6.601086742679306285e-01 8.024623637610888149e-01 7.694800106460160105e-01
6.668834808789664281e-01 8.054423184071207720e-01 7.721405605628438584e-01
6.736267593487160754e-01 8.084357253933542875e-01 7.748213357696444037e-01
6.803392197495884419e-01 8.114429182878516444e-01 7.775214487015286169e-01
6.870215451120499361e-01 8.144642264752101068e-01 7.802400248254467430e-01
6.936743900300211818e-01 8.174999765783720340e-01 7.829761983884004906e-01
7.002974950415432609e-01 8.205507454139520096e-01 7.857294359175189813e-01
7.068916498938800919e-01 8.236168074562510988e-01 7.884988108515800231e-01
7.134582338528936418e-01 8.266982574818713125e-01 7.912831565844938853e-01
7.199978016303433259e-01 8.297954164160811219e-01 7.940815874530331442e-01
7.265108777000938156e-01 8.329086081072485381e-01 7.968932005645313899e-01
7.329979578454549616e-01 8.360381599792610086e-01 7.997170690214260302e-01
7.394595112267420278e-01 8.391844035374615984e-01 8.025522344938926800e-01
7.458959830242087863e-01 8.423476747120355324e-01 8.053976990501692246e-01
7.523077977331221744e-01 8.455283140156318877e-01 8.082524161470656665e-01
7.586951667693938584e-01 8.487267253083131680e-01 8.111153620935316333e-01
7.650577483984875027e-01 8.519434804026106978e-01 8.139856782226038145e-01
7.713970926011407547e-01 8.551785852225586293e-01 8.168616329686009259e-01
7.777135915287802792e-01 8.584323939893921951e-01 8.197418712821458175e-01
7.840076454752769042e-01 8.617052627327101977e-01 8.226249282969148036e-01
7.902796732960170045e-01 8.649975472537146937e-01 8.255092131742095551e-01
7.965301248451132077e-01 8.683096003971411125e-01 8.283929915814087774e-01
8.027594957590121760e-01 8.716417684881572203e-01 8.312743668855639978e-01
8.089680559912207913e-01 8.749944751017242339e-01 8.341514052269850543e-01
8.151567019468018982e-01 8.783679611555336164e-01 8.370217075803022544e-01
8.213264334358614249e-01 8.817624446744987132e-01 8.398826380716230000e-01
8.274781243204414327e-01 8.851781910186152791e-01 8.427314427493727278e-01
8.336128134834004388e-01 8.886154231077250110e-01 8.455651044543932571e-01
8.397317396975054749e-01 8.920743090766621863e-01 8.483803273271105505e-01
8.458363795503203164e-01 8.955549480225133419e-01 8.511735264537213519e-01
8.519285410708211659e-01 8.990573377468378258e-01 8.539407899678711500e-01
8.580104271310651232e-01 9.025813506938162867e-01 8.566778627755414766e-01
8.640844411003687497e-01 9.061267859275130565e-01 8.593803204948170515e-01
8.701533853242351402e-01 9.096933004913555498e-01 8.620435066748899366e-01
8.762204735799774546e-01 9.132803958502141439e-01 8.646625969293003644e-01
8.822893382459134903e-01 9.168874047650978909e-01 8.672326784644417419e-01
8.883640160698558219e-01 9.205134840158590848e-01 8.697488518860556628e-01
8.944490897569347121e-01 9.241575627749747390e-01 8.722061884634622064e-01
9.005494862654843669e-01 9.278183935417740891e-01 8.745999527261957285e-01
9.066697518672085510e-01 9.314947526919318266e-01 8.769263428234910229e-01
9.128147816848455331e-01 9.351852279261616552e-01 8.791818556054207257e-01
9.189893388971293042e-01 9.388883614412766310e-01 8.813636793355511534e-01
9.251978653035416444e-01 9.426027146413026303e-01 8.834697874077058755e-01
9.314442876100802460e-01 9.463269403919151168e-01 8.854989876415102490e-01
9.377318363007438595e-01 9.500598571817585603e-01 8.874509187201842231e-01
9.440634086681402026e-01 9.538003778631106711e-01 8.893253867710815275e-01
9.504401533195278029e-01 9.575479274112385086e-01 8.911237616446540111e-01
9.568622268534309194e-01 9.613022612892393459e-01 8.928479377657454474e-01
9.633289971376283178e-01 9.650634347497447640e-01 8.944999044139121391e-01
9.698388693863758681e-01 9.688318688917866295e-01 8.960818313086871267e-01
9.763893802744164629e-01 9.726083363283349881e-01 8.975959074248138769e-01
9.829773205072419584e-01 9.763939321161484441e-01 8.990442065973827113e-01
9.895988716426143972e-01 9.801900344288262401e-01 9.004285886132605832e-01
9.962497442478134291e-01 9.839982596313375796e-01 9.017506391251587372e-01
9.996253193176977137e-01 9.913711226010460953e-01 8.041012438578545307e-01
9.969312990878144154e-01 9.865865913107011442e-01 7.958196545688069889e-01
9.942533588637104680e-01 9.818135789307643746e-01 7.875317815897165952e-01
9.915896776086415842e-01 9.770525904709529419e-01 7.792374356109948996e-01
9.889384786221749879e-01 9.723041153469224041e-01 7.709364896057565586e-01
9.862980251266783016e-01 9.675686302753326862e-01 7.626288656679628408e-01
9.836666169060123144e-01 9.628466015967408476e-01 7.543145233681930462e-01
9.810425876106124710e-01 9.581384871880828102e-01 7.459934495167190871e-01
9.784237290846492519e-01 9.534448589527805273e-01 7.376670490866494845e-01
9.758091741853186507e-01 9.487660072025493330e-01 7.293335612360094533e-01
9.731976797213667263e-01 9.441023023821585314e-01 7.209921595340745837e-01
9.705876565172376624e-01 9.394541905537218129e-01 7.126429103405369503e-01
9.679775344953384097e-01 9.348221172710475813e-01 7.042858810456844587e-01
9.653657609756586266e-01 9.302065285603877687e-01 6.959211353452218196e-01
9.627508763245108403e-01 9.256078555762781157e-01 6.875485248304887831e-01
9.601317913231469658e-01 9.210264571207541495e-01 6.791669211618720503e-01
9.575068348330096901e-01 9.164628209777989643e-01 6.707767168399573210e-01
9.548744491996995487e-01 9.119174176425877132e-01 6.623780173986024700e-01
9.522330808045905703e-01 9.073907239128325974e-01 6.539709177027834830e-01
9.495811770290348841e-01 9.028832240271246201e-01 6.455555018977681137e-01
9.469171829214290126e-01 8.983954108674836458e-01 6.371318441574638225e-01
9.442402190659517913e-01 8.939276490907958062e-01 6.286979645113369708e-01
9.415486437855169477e-01 8.894804711503382366e-01 6.202539985663386712e-01
9.388403761261178149e-01 8.850545069722992597e-01 6.118014161078325630e-01
9.361137658947894513e-01 8.806503085987659185e-01 6.033403762527677072e-01
9.333671427931780062e-01 8.762684428171652051e-01 5.948710444843188228e-01
9.305988122871574619e-01 8.719094924143574454e-01 5.863935979844183688e-01
9.278070514355759579e-01 8.675740573997056115e-01 5.779082318269497254e-01
9.249901047433721768e-01 8.632627561697012730e-01 5.694151660419812799e-01
9.221469800733509414e-01 8.589760860685742294e-01 5.609116482547054083e-01
9.192753048560652340e-01 8.547148108589553983e-01 5.523997440377725887e-01
9.163728560349155838e-01 8.504796735679325259e-01 5.438810466983201586e-01
9.134376867190158178e-01 8.462713800742609482e-01 5.353560484875481418e-01
9.104678046010745707e-01 8.420906570028113824e-01 5.268253096213930675e-01
9.074611700430375016e-01 8.379382516634488187e-01 5.182894693241438810e-01
9.044156948810763152e-01 8.338149316350844664e-01 5.097492574011085464e-01
9.013292420875533839e-01 8.297214839374013051e-01 5.012055062134753713e-01
8.981996264328737656e-01 8.256587137313211588e-01 4.926591628979440363e-01
8.950246162923013449e-01 8.216274424893472705e-01 4.841113016411668357e-01
8.918019367410795484e-01 8.176285055788921063e-01 4.755631357855747976e-01
8.885292740744069606e-01 8.136627492060259925e-01 4.670160295098501613e-01
8.852042818763621312e-01 8.097310266740884721e-01 4.584715087957387802e-01
8.818245887426444662e-01 8.058341939216442373e-01 4.499312713647861117e-01
8.783878077356160885e-01 8.019731043176190344e-01 4.413971952459587733e-01
8.748915476159051519e-01 7.981486027081815537e-01 4.328713456201330745e-01
8.713334258529761289e-01 7.943615187300834268e-01 4.243559795823733105e-01
8.677110833677242896e-01 7.906126594284177411e-01 4.158535484698712703e-01
8.640222009043423412e-01 7.869028012426558805e-01 4.073666974243692063e-01
8.602645168678531018e-01 7.832326814525909509e-01 3.988982618942717440e-01
8.564358463998196225e-01 7.796029892044549214e-01 3.904512608343593816e-01
8.525341014005127782e-01 7.760143562656642846e-01 3.820288864300384613e-01
8.485573111443696082e-01 7.724673476829161389e-01 3.736344902575098326e-01
8.445036430802366212e-01 7.689624525410815314e-01 3.652715658890242079e-01
8.403714233625397823e-01 7.655000750378188057e-01 3.569437280604416673e-01
8.361591566273345322e-01 7.620805260995172636e-01 3.486546886323707573e-01
8.318655445115968883e-01 7.587040157667198637e-01 3.404082296912668837e-01
8.274895024172935765e-01 7.553706465705579687e-01 3.322081742472747790e-01
8.230301740454328829e-01 7.520804081054507373e-01 3.240583550855133943e-01
8.184868212302832680e-01 7.488332616294635091e-01 3.159617825674137515e-01
8.138590110822148116e-01 7.456289896604103573e-01 3.079221377655331771e-01
8.091468181242912339e-01 7.424670871619818424e-01 2.999443469091602199e-01
8.043504122379883103e-01 7.393470563716729727e-01 2.920319686576324791e-01
7.994702184716161453e-01 7.362682828578653860e-01 2.841884012704181672e-01
7.945069109411802000e-01 7.332300399833050486e-01 2.764168555307422448e-01
7.894614033518293494e-01 7.302314948629916591e-01 2.687203309843908539e-01
7.843348364120679150e-01 7.272717156577747089e-01 2.611015959667907227e-01
7.791285260147496894e-01 7.243496994364320152e-01 2.535630684860531447e-01
7.738440801480908071e-01 7.214643092911102729e-01 2.461071963221510561e-01
7.684833208817296590e-01 7.186143199358207001e-01 2.387362039122769009e-01
7.630481315977482026e-01 7.157984953860191402e-01 2.314517734491695622e-01
7.575405230514773436e-01 7.130155515836559266e-01 2.242553190375982108e-01
7.519626112766101267e-01 7.102641669372077304e-01 2.171479941172158035e-01
7.463165956980535309e-01 7.075429925763011552e-01 2.101307026608347228e-01
7.406047378440419049e-01 7.048506621488696000e-01 2.032041138079452858e-01
7.348297350913484127e-01 7.021856197788983733e-01 1.963692968886621149e-01
7.289936844492124202e-01 6.995466025992004289e-01 1.896260481458106884e-01
7.230988483362374986e-01 6.969322773880018973e-01 1.829743149302962002e-01
7.171475322086006132e-01 6.943412998823550453e-01 1.764139943240976283e-01
7.111420143928013360e-01 6.917723467781027313e-01 1.699448658068684892e-01
7.050845338608371371e-01 6.892241206236302542e-01 1.635666199792324693e-01
6.989772797030316953e-01 6.866953538838792559e-01 1.572788888364401172e-01
6.928223822517928232e-01 6.841848122132092591e-01 1.510812776309110872e-01
6.866220524799201419e-01 6.816912363109779438e-01 1.449735254345622115e-01
6.803786701652846380e-01 6.792133090629258740e-01 1.389555620195359609e-01
6.740936196655853418e-01 6.767501306945983286e-01 1.330266252977832520e-01
6.677687244983904202e-01 6.743006234739316040e-01 1.271865644636745452e-01
6.614057275351008514e-01 6.718637487365417549e-01 1.214353970802317662e-01
6.550062900453353931e-01 6.694385066322930955e-01 1.157733573283348805e-01
6.485719915801509972e-01 6.670239355272904458e-01 1.102009495253571669e-01
6.421043305819508218e-01 6.646191111168818777e-01 1.047190080781650323e-01
6.356047256163376291e-01 6.622231453008022850e-01 9.932876523540395963e-02
6.290745171295046845e-01 6.598351848668921882e-01 9.403192821288078318e-02
6.225149696438274649e-01 6.574544100249212208e-01 8.883076747185855715e-02
6.159272743135092432e-01 6.550800328272597950e-01 8.372821811088401733e-02
6.093126663293514378e-01 6.527112545185101977e-01 7.872799706657532259e-02
6.026723170612737768e-01 6.503473055396404856e-01 7.383469856963406630e-02
5.960069280976730832e-01 6.479875866853281874e-01 6.905400752442764079e-02
5.893174254692321590e-01 6.456314203776034599e-01 6.439287527349069062e-02
5.826046784844106652e-01 6.432781520370909334e-01 5.985968924725062340e-02
5.758695032242261425e-01 6.409271483905801814e-01 5.546448455059219823e-02
5.691126660872921628e-01 6.385777958070207871e-01 5.121916775709036557e-02
5.623348873683923221e-01 6.362294986703330713e-01 4.713774156788400754e-02
5.555368448583254404e-01 6.338816777957024806e-01 4.323651096469569716e-02
5.487191774565705060e-01 6.315337688945173999e-01 3.952487864258176498e-02
5.418824887917018662e-01 6.291852210919098853e-01 3.612051121019213551e-02
5.350273508472259687e-01 6.268354954999130202e-01 3.311399967360150604e-02
5.281543075929241438e-01 6.244840638484698836e-01 3.049096132752013993e-02
5.212638786236020172e-01 6.221304071760309640e-01 2.823775650583685778e-02
5.143565628087404251e-01 6.197740145810481938e-01 2.634144880489198981e-02
5.074328419576341620e-01 6.174143820354919265e-01 2.478976657062584646e-02
5.004931845054064743e-01 6.150510112614009373e-01 2.357106560875196072e-02
4.935380492257968599e-01 6.126834086714986194e-01 2.267429306479169446e-02
4.865677337149327264e-01 6.103111278318402722e-01 2.208918519158591109e-02
4.795827833081557912e-01 6.079336535225863258e-01 2.180564619063158835e-02
4.725836555273193462e-01 6.055504983495709759e-01 2.181419194008693205e-02
4.655708088649732068e-01 6.031611773523495312e-01 2.210580170001537684e-02
4.585447116306416437e-01 6.007652058968021569e-01 2.267187969798983502e-02
4.515058459835388782e-01 5.983620990282547680e-01 2.350422445657735990e-02
4.444547120266517104e-01 5.959513709030562767e-01 2.459499875403014374e-02
4.373918319618326778e-01 5.935325343022211930e-01 2.593670014651517156e-02
4.303177543037495223e-01 5.911051002310937497e-01 2.752213198538535840e-02
4.232330581488202292e-01 5.886685776092354105e-01 2.934437486930205341e-02
4.161383574931313278e-01 5.862224730549979723e-01 3.139675847884770832e-02
4.090343055913983616e-01 5.837662907693373926e-01 3.367283375013734037e-02
4.019215993467592507e-01 5.812995325235387201e-01 3.616634535426579283e-02
3.948009837190709082e-01 5.788216977554323517e-01 3.887120446073646235e-02
3.876732561372571162e-01 5.763322837785355146e-01 4.174672181203834681e-02
3.805386564470196742e-01 5.738309171411589693e-01 4.468566878513501733e-02
3.733986391498164137e-01 5.713169711328536238e-01 4.768049787246867594e-02
3.662542351622480874e-01 5.687899279044026368e-01 5.071835758395210753e-02
3.591065040473656045e-01 5.662492786415918022e-01 5.378828392041421630e-02
3.519565800474088735e-01 5.636945152858550134e-01 5.688085721725138350e-02
3.448056760545124555e-01 5.611251315161035480e-01 5.998797256299469999e-02
3.376550874229136134e-01 5.585406238589376571e-01 6.310263770132876204e-02
3.305061956006355439e-01 5.559404929241524851e-01 6.621879773242350664e-02
3.233604715607960034e-01 5.533242447607230607e-01 6.933118466356372189e-02
3.162194790161148017e-01 5.506913923265978061e-01 7.243518928216058361e-02
3.090846039651472532e-01 5.480415027343880086e-01 7.552721746757809496e-02
3.019574288899705139e-01 5.453741321890830385e-01 7.860394456462099777e-02
2.948403194287022577e-01 5.426887379559618418e-01 8.166128713902018332e-02
2.877352499045566225e-01 5.399848761881764769e-01 8.469633286585029341e-02
2.806442994214729536e-01 5.372621189537871711e-01 8.770643445761908130e-02
2.735696535538801322e-01 5.345200561570193631e-01 9.068916512095565041e-02
2.665136059049172390e-01 5.317582974656462902e-01 9.364228167891575083e-02
2.594785595977833204e-01 5.289764742223507232e-01 9.656369418956520234e-02
2.524670287821700332e-01 5.261742413169625543e-01 9.945144107016240520e-02
2.454813557062531792e-01 5.233513142453691813e-01 1.023041162767710510e-01
2.385245828710040317e-01 5.205073603004944927e-01 1.051194831525161522e-01
2.315996737216303170e-01 5.176421037021866622e-01 1.078956959800008442e-01
2.247096070811669399e-01 5.147553123030387257e-01 1.106311405318080310e-01
2.178574824667257048e-01 5.118467857547527311e-01 1.133242655958566492e-01
2.110465244393363582e-01 5.089163566221597268e-01 1.159735773093806543e-01
2.042800881777394606e-01 5.059638913007615812e-01 1.185776351459725819e-01
1.975616665475327660e-01 5.029892907239723598e-01 1.211350492651094846e-01
1.908950317685597087e-01 4.999924798399172921e-01 1.236442618572499708e-01
1.842839642352289697e-01 4.969734333710090213e-01 1.261039978961912833e-01
1.777324020667393756e-01 4.939321620495928378e-01 1.285130506256617899e-01
1.712444848635372441e-01 4.908687102327232155e-01 1.308702368677320538e-01
1.648245567449286297e-01 4.877831568788963401e-01 1.331744245804179216e-01
1.584771896665797541e-01 4.846756148389082530e-01 1.354245338303780855e-01
1.522072108246106115e-01 4.815462299379932865e-01 1.376195378692431359e-01
1.460197345452351747e-01 4.783951798632230523e-01 1.397584642177332193e-01
1.399204939260773328e-01 4.752226638142488802e-01 1.418397751955981501e-01
1.339151022806790436e-01 4.720289313790105301e-01 1.438629155677119409e-01
1.280096392738992173e-01 4.688142535267130762e-01 1.458273227721058329e-01
1.222107627052576584e-01 4.655789223639808516e-01 1.477322650340249788e-01
1.165256528622910237e-01 4.623232537320298152e-01 1.495770677617926092e-01
1.109620746853187123e-01 4.590475853271560047e-01 1.513611132825743999e-01
1.055284430115649430e-01 4.557522747771929894e-01 1.530838402977779955e-01
1.002338888313226428e-01 4.524376976954341267e-01 1.547447430577089666e-01
9.508832342249537439e-02 4.491042457325262194e-01 1.563433702619193288e-01
9.010250937510871916e-02 4.457523324675090604e-01 1.578790892676192326e-01
8.528801833375249108e-02 4.423823989146767888e-01 1.593510674113139958e-01
8.065727940861416867e-02 4.389948528858526600e-01 1.607597457256386420e-01
7.622371216564308161e-02 4.355901307746828932e-01 1.621048834368656322e-01
7.200158271604989446e-02 4.321686757028083692e-01 1.633862841279137557e-01
6.800589086138220107e-02 4.287309359317861834e-01 1.646037933778843332e-01
6.425218103335716968e-02 4.252773633715483670e-01 1.657572962977645059e-01
6.075626148835750612e-02 4.218084121914437157e-01 1.668467149892300661e-01
5.753381975108700502e-02 4.183245375382101949e-01 1.678720059529140995e-01
5.459992971435845971e-02 4.148261943636451510e-01 1.688331574715942474e-01
5.196845773512390881e-02 4.113138363632788397e-01 1.697301869925023354e-01
4.965139124510086627e-02 4.077879150260529384e-01 1.705631385315065085e-01
4.765813208284918473e-02 4.042488787938245398e-01 1.713320801202671551e-01
4.599481449154869256e-02 4.006971723285012166e-01 1.720371013157572515e-01
4.466371969486728627e-02 3.971332358837184051e-01 1.726783107897287561e-01
4.366274483561478209e-02 3.935575133766276990e-01 1.732557671739120286e-01
4.298497468310508857e-02 3.899704687794856572e-01 1.737693705957654433e-01
4.262017645185838671e-02 3.863724827023667929e-01 1.742196255705478480e-01
4.255352467768171859e-02 3.827639728113220174e-01 1.746066936456434071e-01
4.276673184831410873e-02 3.791453506945430818e-01 1.749307448926006592e-01
4.323878239983177524e-02 3.755170217599948512e-01 1.751919563188165385e-01
4.394675081101364483e-02 3.718793851987791665e-01 1.753905104335057585e-01
4.486662442701292580e-02 3.682328340080037177e-01 1.755265939730583369e-01
4.597406436707644067e-02 3.645777550668193867e-01 1.756003967899375517e-01
4.724505801823831314e-02 3.609145292591857124e-01 1.756121109084000098e-01
4.865643856140024898e-02 3.572435316368365310e-01 1.755619297495205344e-01
5.018626632042253594e-02 3.535651316158259783e-01 1.754500475272437465e-01
5.181408104443557122e-02 3.498796931999255677e-01 1.752766588164567374e-01
5.352104295989750654e-02 3.461875752240315962e-01 1.750419582933170903e-01
5.528998425553303259e-02 3.424891316106106198e-01 1.747461406473147794e-01
5.710539287663336100e-02 3.387847116320403806e-01 1.743894006636908245e-01
5.895334846203271334e-02 3.350746601715560713e-01 1.739719334739151524e-01
6.082142709649813322e-02 3.313593179753057116e-01 1.734939349708606859e-01
6.269858808390260663e-02 3.276390218878380001e-01 1.729556023841174184e-01
6.457505267241087088e-02 3.239141050631572094e-01 1.723571350094970367e-01
6.644218183629055363e-02 3.201848971432968427e-01 1.716987350852014205e-01
6.829235792686130790e-02 3.164517243962254311e-01 1.709806088053268391e-01
7.011887323479765177e-02 3.127149098048046527e-01 1.702029674593475428e-01
7.191582719453681882e-02 3.089747730984767071e-01 1.693660286839586415e-01
7.367740139700959534e-02 3.052317018390342529e-01 1.684698187582648887e-01
7.539908659075802988e-02 3.014860094828903936e-01 1.675145933863426417e-01
7.707730571895973770e-02 2.977379606831319081e-01 1.665007348086736949e-01
7.870852955781817983e-02 2.939878648964006636e-01 1.654284970811750188e-01
8.028965953863020921e-02 2.902360278178472974e-01 1.642981478645798299e-01
8.181797774419891089e-02 2.864827509313960796e-01 1.631099700093367744e-01
8.329110360766753263e-02 2.827283309501391617e-01 1.618642630930951787e-01
8.470695659424687385e-02 2.789730591425138573e-01 1.605613448745624727e-01
8.606372418387428502e-02 2.752172205414125661e-01 1.592015526248838075e-01
8.735983452008261319e-02 2.714610930351273321e-01 1.577852442954886525e-01
8.859393314120664331e-02 2.677049463410028363e-01 1.563127994796906506e-01
8.976486326081492551e-02 2.639490408648579312e-01 1.547846201243904263e-01
9.087164911293293956e-02 2.601936264515474218e-01 1.532011309481255412e-01
9.191348192284298779e-02 2.564389410344570241e-01 1.515627795225681362e-01
9.288704312629730842e-02 2.526856315239155992e-01 1.498693699931691048e-01
9.379283018859288501e-02 2.489337717503207204e-01 1.481216905842226428e-01
9.463163644057562274e-02 2.451833792964967507e-01 1.463205652098015508e-01
9.540318566942779244e-02 2.414346401225993533e-01 1.444665315338557743e-01
9.610732643393554708e-02 2.376877206774311024e-01 1.425601463139360425e-01
9.674402718144511915e-02 2.339427661170380146e-01 1.406019829499289553e-01
9.731337192794831115e-02 2.301998985206034631e-01 1.385926284969124511e-01
9.781527030012060475e-02 2.264592671684997338e-01 1.365326231802845014e-01
9.824358966693200190e-02 2.227221381992704474e-01 1.344213498047777955e-01
9.860476008591259611e-02 2.189874851640599696e-01 1.322606854144605382e-01
9.889926458171927059e-02 2.152553323101172311e-01 1.300512525669721697e-01
9.912768720083675600e-02 2.115256719866073221e-01 1.277936680974917916e-01
9.929070945217863264e-02 2.077984630697710111e-01 1.254885372707029156e-01
9.938536693535210409e-02 2.040743841747782450e-01 1.231359093599569132e-01
9.940987528000957973e-02 2.003539028956775048e-01 1.207361104943659447e-01
9.937074432198822471e-02 1.966357733029605592e-01 1.182905659078862248e-01
9.926899950314096999e-02 1.929198155529378012e-01 1.157998086675786908e-01
9.910574746809375224e-02 1.892058087816931022e-01 1.132643300609986470e-01
9.887273365069987330e-02 1.854955476839638961e-01 1.106837415179452166e-01
9.857702759168757156e-02 1.817875303344013982e-01 1.080592207487474365e-01
9.822269753736714848e-02 1.780808239249605796e-01 1.053914161709541830e-01
9.781113633355650872e-02 1.743750377566490040e-01 1.026806351620476176e-01
9.733238005935909709e-02 1.706723687329834982e-01 9.992670564763897478e-02
9.679537330416121410e-02 1.669706979907544520e-01 9.713043093317094701e-02
9.620476860769888727e-02 1.632687911401897729e-01 9.429209466525781402e-02
9.555840111219859878e-02 1.595669718044100127e-01 9.141177435708905397e-02
9.484851695793727888e-02 1.558669291806107915e-01 8.848983773912671991e-02
9.408911153954402362e-02 1.521650578561620781e-01 8.552628708305054506e-02
9.328192094338655371e-02 1.484606091474970080e-01 8.252094903627574252e-02
9.241015243151923242e-02 1.447574737795673805e-01 7.947496936453349314e-02
9.149313879389489590e-02 1.410504572025015335e-01 7.638732537156053826e-02
9.053276383981978537e-02 1.373386075843833487e-01 7.325761429945673586e-02];

   case 'cur'
      RGB = [8.225559928700268419e-02 1.149244079727295142e-01 2.647901677800857390e-01
8.312616532498406929e-02 1.190383729463048712e-01 2.668628892216621806e-01
8.400180885962132971e-02 1.231074880892656653e-01 2.689526699064171411e-01
8.487294239495335457e-02 1.271387529060027943e-01 2.710541708402016137e-01
8.574385298640457842e-02 1.311333174761502018e-01 2.731691209373900975e-01
8.661249189260347703e-02 1.350944971238551839e-01 2.752961432065319514e-01
8.747533041314431435e-02 1.390258052165279645e-01 2.774332852121961235e-01
8.833858505105957049e-02 1.429270910011002649e-01 2.795831537842536352e-01
8.919012906146844832e-02 1.468043594975814992e-01 2.817400195447572475e-01
9.004099984169053328e-02 1.506555099870153513e-01 2.839086654207542693e-01
9.088231952195491292e-02 1.544850037627045203e-01 2.860850125083750362e-01
9.171714479257989105e-02 1.582931942356169963e-01 2.882702798507874586e-01
9.254607948203208423e-02 1.620811665705463311e-01 2.904645825457518593e-01
9.336173420340779239e-02 1.658523274558137695e-01 2.926648040593683997e-01
9.417284157369981701e-02 1.696050767223750977e-01 2.948744162588470830e-01
9.496899572502048859e-02 1.733434725855039771e-01 2.970892156823352059e-01
9.575619444438937533e-02 1.770666290075273708e-01 2.993115284087380368e-01
9.653316478613682694e-02 1.807757562460599599e-01 3.015407883893915231e-01
9.729328810023782359e-02 1.844734436312414905e-01 3.037745103921251633e-01
9.804493118338306057e-02 1.881580898967480098e-01 3.060157402408537064e-01
9.877832247043097369e-02 1.918329889556127654e-01 3.082609202522414993e-01
9.949803783218733044e-02 1.954975058870322413e-01 3.105116763778897337e-01
1.002054858430543316e-01 1.991518634625495388e-01 3.127684262427183892e-01
1.008900241588315538e-01 2.027992883367119026e-01 3.150275553737127421e-01
1.015620277820400430e-01 2.064376400065636719e-01 3.172925125709148420e-01
1.022150454689789434e-01 2.100689913462399083e-01 3.195611195376787395e-01
1.028468845169810686e-01 2.136942776676366007e-01 3.218326666201724029e-01
1.034637486817424901e-01 2.173124203806320875e-01 3.241090419310820314e-01
1.040556247342821483e-01 2.209261472703716311e-01 3.263871052200472134e-01
1.046272664909374817e-01 2.245346804307608024e-01 3.286682704446855507e-01
1.051814556007013568e-01 2.281377376539149293e-01 3.309532535866355762e-01
1.057055066316897329e-01 2.317384538919968207e-01 3.332383062494437276e-01
1.062093640276061124e-01 2.353348932541681759e-01 3.355262248019345583e-01
1.066927192498384747e-01 2.389274243968986799e-01 3.378167764830257158e-01
1.071428862568302165e-01 2.425189920877232619e-01 3.401063918909076889e-01
1.075713140889580088e-01 2.461073946949010050e-01 3.423980926413837667e-01
1.079763885418836000e-01 2.496932209057829977e-01 3.446912766824791197e-01
1.083460004297124302e-01 2.532791299120676354e-01 3.469826761312421182e-01
1.086913329090623825e-01 2.568630402753477870e-01 3.492750549670107785e-01
1.090116287119651528e-01 2.604453159192556266e-01 3.515680214913912693e-01
1.092932808921200649e-01 2.640287615628073015e-01 3.538580573547516761e-01
1.095478763802813504e-01 2.676112773022403801e-01 3.561478615212890775e-01
1.097748910210358808e-01 2.711931376089152246e-01 3.584370856671010852e-01
1.099635342633853707e-01 2.747764822051754763e-01 3.607229892872333421e-01
1.101198294511886444e-01 2.783603021914852760e-01 3.630068099573608986e-01
1.102460152182262176e-01 2.819443225282238785e-01 3.652888319764409086e-01
1.103361572207214036e-01 2.855297163446686715e-01 3.675674754730824945e-01
1.103867349279119559e-01 2.891171759115234718e-01 3.698417464033080804e-01
1.104047114339987423e-01 2.927055780381122019e-01 3.721129484539063559e-01
1.103893448894039397e-01 2.962951553198432397e-01 3.743806401368951486e-01
1.103290785873510815e-01 2.998879083652261635e-01 3.766420819050353419e-01
1.102317288825286901e-01 3.034825829921489193e-01 3.788986922133656954e-01
1.100986315611906241e-01 3.070790351009552999e-01 3.811504545068737926e-01
1.099286556573810802e-01 3.106775192369479188e-01 3.833968205472055302e-01
1.097092559883022234e-01 3.142800415550652815e-01 3.856349205402823110e-01
1.094518269158459012e-01 3.178848431435130073e-01 3.878667847148202785e-01
1.091556968975302966e-01 3.214920812428978536e-01 3.900919340676701208e-01
1.088202096165073185e-01 3.251019032868823211e-01 3.923098843692563453e-01
1.084337294201929702e-01 3.287160421828903556e-01 3.945179917556369542e-01
1.080052802278752000e-01 3.323331728550899533e-01 3.967176863054378000e-01
1.075355258379605550e-01 3.359532282299735328e-01 3.989087127976621017e-01
1.070239159802940931e-01 3.395763147211236510e-01 4.010905648833596460e-01
1.064666158592602885e-01 3.432029795356145718e-01 4.032620996037361571e-01
1.058582376509708545e-01 3.468339423292918222e-01 4.054218838433404359e-01
1.052065591856250482e-01 3.504681464569319171e-01 4.075709632165070984e-01
1.045112201334900126e-01 3.541056562885774861e-01 4.097088092956981398e-01
1.037719209811681642e-01 3.577465259466254266e-01 4.118348877541359032e-01
1.029884320307208612e-01 3.613907991323849767e-01 4.139486582680417803e-01
1.021513134274243950e-01 3.650396452261166491e-01 4.160478671051655586e-01
1.012689167529539358e-01 3.686920150348446112e-01 4.181335233490640069e-01
1.003423119365809690e-01 3.723477936835040136e-01 4.202052548968338574e-01
9.937169919107982641e-02 3.760069784468507703e-01 4.222625006231348066e-01
9.835741884413440328e-02 3.796695548525758634e-01 4.243046936775203282e-01
9.729997123759509536e-02 3.833354963232004087e-01 4.263312615677605777e-01
9.620003933222870396e-02 3.870047637840662302e-01 4.283416262899890081e-01
9.505172587236093706e-02 3.906780327832828914e-01 4.303339883746660766e-01
9.386233813144639893e-02 3.943545459798681874e-01 4.323088724355508838e-01
9.263427266200571775e-02 3.980341179549674036e-01 4.342658626828024837e-01
9.136927887187012987e-02 4.017166535612498035e-01 4.362043674792987491e-01
9.006945702453716951e-02 4.054020428715323643e-01 4.381237915805220595e-01
8.873730500447504776e-02 4.090901606132017476e-01 4.400235366709063789e-01
8.737577058920215078e-02 4.127808655736769361e-01 4.419030019956885491e-01
8.598830961259482097e-02 4.164739999785817548e-01 4.437615850971945997e-01
8.457895032121595658e-02 4.201693888446985103e-01 4.455986826648858368e-01
8.315236408743081897e-02 4.238668393102046350e-01 4.474136915088433031e-01
8.171394242970148047e-02 4.275661399451892164e-01 4.492060096666721791e-01
8.026987997778653461e-02 4.312670600459782566e-01 4.509750376540915817e-01
7.882726258105521300e-02 4.349693489173894201e-01 4.527201798696426915e-01
7.739415915996109008e-02 4.386727351476746306e-01 4.544408461640829233e-01
7.597971511267428979e-02 4.423769258816079852e-01 4.561364535850265800e-01
7.459424408415230023e-02 4.460816060979165276e-01 4.578064283072901808e-01
7.324931366840245484e-02 4.497864378980456768e-01 4.594502077591564038e-01
7.195781915786156335e-02 4.534910598140922677e-01 4.610672429543411499e-01
7.073403782879061907e-02 4.571950861446216208e-01 4.626570010388615928e-01
6.959365457471067273e-02 4.608981063279830592e-01 4.642189680611767399e-01
6.855374817251533304e-02 4.645996843636931994e-01 4.657526519729214276e-01
6.763272639280798471e-02 4.682993582933911436e-01 4.672575858662252890e-01
6.685019795786503738e-02 4.719966397538147285e-01 4.687333314519904204e-01
6.622677049492370349e-02 4.756910136151853430e-01 4.701794827815656830e-01
6.577673858968982601e-02 4.793824370720338179e-01 4.715941633832017588e-01
6.552566064603559948e-02 4.830700650743400826e-01 4.729777438223161101e-01
6.549820807888259711e-02 4.867530881918184504e-01 4.743305399824718216e-01
6.571558023976076246e-02 4.904308747270616498e-01 4.756523204648347991e-01
6.619782080433814220e-02 4.941027658684106760e-01 4.769429100625798834e-01
6.696293828126623215e-02 4.977680989992983585e-01 4.782021118153178540e-01
6.801610562745827315e-02 5.014269919387287500e-01 4.794267081520149909e-01
6.938287907938911481e-02 5.050778255979795350e-01 4.806197953737798012e-01
7.107372065756567547e-02 5.087198306495862576e-01 4.817814896929000779e-01
7.309574065371191032e-02 5.123522156200341904e-01 4.829119931483520367e-01
7.544312997722565917e-02 5.159750173542567708e-01 4.840076729265158639e-01
7.812721320542403980e-02 5.195865216476734938e-01 4.850725749743695636e-01
8.114572394888117102e-02 5.231858403590214923e-01 4.861073630469811557e-01
8.448832972524200624e-02 5.267726392289812098e-01 4.871097239159773440e-01
8.815110303261089464e-02 5.303457163263245455e-01 4.880817131293228583e-01
9.212706115366336990e-02 5.339038791645212001e-01 4.890257976801760109e-01
9.640117372078826907e-02 5.374467824904106683e-01 4.899392661237967350e-01
1.009651127375016111e-01 5.409730397620680087e-01 4.908260102866368046e-01
1.058058868411170528e-01 5.444817357603629615e-01 4.916873206428047927e-01
1.109091719419828259e-01 5.479722539284174188e-01 4.925220286879308795e-01
1.162631754482068569e-01 5.514432364311973034e-01 4.933355557142476422e-01
1.218535592637135234e-01 5.548942058020517321e-01 4.941256571865651481e-01
1.276672732366210816e-01 5.583239554235875923e-01 4.948978636889357907e-01
1.336912106530599997e-01 5.617318570854707982e-01 4.956519172255148820e-01
1.399119531735926458e-01 5.651170059000043544e-01 4.963918396953513335e-01
1.463171276735238113e-01 5.684787369209851615e-01 4.971190242696297834e-01
1.528937099908951325e-01 5.718163393279495077e-01 4.978371217540351057e-01
1.596300036906863895e-01 5.751292412901443107e-01 4.985481743532478860e-01
1.665133003434038084e-01 5.784169089899582339e-01 4.992561455145328453e-01
1.735328783097749850e-01 5.816789154432452369e-01 4.999631315071642601e-01
1.806761031013389140e-01 5.849149331180006905e-01 5.006736479698321585e-01
1.879334378838529440e-01 5.881246912104457492e-01 5.013896313522919757e-01
1.952924636272905801e-01 5.913080959200207598e-01 5.021159025525618880e-01
2.027443124327554802e-01 5.944650539898058694e-01 5.028546409645925364e-01
2.102776619158149840e-01 5.975956842983582984e-01 5.036102222376415138e-01
2.178835011739876371e-01 6.007001216855215597e-01 5.043855354959408954e-01
2.255521696872271886e-01 6.037786523436265984e-01 5.051841613954050070e-01
2.332749143253212143e-01 6.068316286127126702e-01 5.060092633925864503e-01
2.410430130087069522e-01 6.098595200479223211e-01 5.068641518721540562e-01
2.488489441844971561e-01 6.128628192186393875e-01 5.077515718650474907e-01
2.566840197606024554e-01 6.158422311486778655e-01 5.086750547256451149e-01
2.645425291505308918e-01 6.187983086880044503e-01 5.096365792929340444e-01
2.724162323640093031e-01 6.217319406932720893e-01 5.106395912364674050e-01
2.803003746086680792e-01 6.246437759477415641e-01 5.116857725517640620e-01
2.881881143962810587e-01 6.275347641630057982e-01 5.127779478270290126e-01
2.960748324121407205e-01 6.304057046767445049e-01 5.139178767894679867e-01
3.039555214161791530e-01 6.332575133378499643e-01 5.151075596956975478e-01
3.118254758119217707e-01 6.360911463304157465e-01 5.163488722475468862e-01
3.196814435511296515e-01 6.389074409416231060e-01 5.176430585409529384e-01
3.275188103490194735e-01 6.417074699067215615e-01 5.189919719540163623e-01
3.353355552299603914e-01 6.444920106677175520e-01 5.203963563323180663e-01
3.431277348645550562e-01 6.472621506436356809e-01 5.218577640245062321e-01
3.508936232465371674e-01 6.500187029053001719e-01 5.233768303957324619e-01
3.586308928051362699e-01 6.527625972609284455e-01 5.249544352005411918e-01
3.663368289249827603e-01 6.554948605761221625e-01 5.265915743478537525e-01
3.740119682194762429e-01 6.582160193662682790e-01 5.282880419647097980e-01
3.816510562742471135e-01 6.609275747235534570e-01 5.300456830101799577e-01
3.892580517875900981e-01 6.636294950347043642e-01 5.318631054022345817e-01
3.968261890078971788e-01 6.663236061846555813e-01 5.337425455110953454e-01
4.043593499172888350e-01 6.690098785171258999e-01 5.356826761355167887e-01
4.118554481683104340e-01 6.716893377862352965e-01 5.376841277217632165e-01
4.193119538935562440e-01 6.743631277083362852e-01 5.397475699351562684e-01
4.267325988888523436e-01 6.770311866202418649e-01 5.418718349209995511e-01
4.341129901223799714e-01 6.796950319734580415e-01 5.440580596655568701e-01
4.414547339722453279e-01 6.823550079810395408e-01 5.463056567639269501e-01
4.487598229028739172e-01 6.850113432670585922e-01 5.486140038956078824e-01
4.560240745098563253e-01 6.876655632831355502e-01 5.509839689679422170e-01
4.632498634530364812e-01 6.903178139102768007e-01 5.534147775766539157e-01
4.704389351347622594e-01 6.929683367461523247e-01 5.559058726658481220e-01
4.775890398856753039e-01 6.956182556045346077e-01 5.584575291026864230e-01
4.846997162615552246e-01 6.982683108816961637e-01 5.610695620964348818e-01
4.917741270603506742e-01 7.009183842407572529e-01 5.637411222277634026e-01
4.988124547076915882e-01 7.035690215604037956e-01 5.664719533702552434e-01
5.058115298480653221e-01 7.062215879970826782e-01 5.692622613027222833e-01
5.127738759690677606e-01 7.088760718122417703e-01 5.721113127309581659e-01
5.197010257328686933e-01 7.115326679134365007e-01 5.750185876848199484e-01
5.265932673521412921e-01 7.141918646847531527e-01 5.779837442967735717e-01
5.334494816269396145e-01 7.168545129510710545e-01 5.810065526719623286e-01
5.402690647877386176e-01 7.195213413684249382e-01 5.840866494502513495e-01
5.470548736207613283e-01 7.221921468817737999e-01 5.872233804977959881e-01
5.538072551957783363e-01 7.248673679611095100e-01 5.904163410524770894e-01
5.605265661533928023e-01 7.275474322700724583e-01 5.936651120515976654e-01
5.672131712764029166e-01 7.302327570205698892e-01 5.969692609600730782e-01
5.738649923321349489e-01 7.329244536181049874e-01 6.003283620189617809e-01
5.804845702488933279e-01 7.356223182652517067e-01 6.037418650719156288e-01
5.870727421002424062e-01 7.383266077479672118e-01 6.072092971993915400e-01
5.936298933201957784e-01 7.410376988478269977e-01 6.107301851240671819e-01
6.001564124519274124e-01 7.437559595200373685e-01 6.143040456187923715e-01
6.066526905401021796e-01 7.464817491552263595e-01 6.179303860708607044e-01
6.131191206125891080e-01 7.492154188267443615e-01 6.216087050224498034e-01
6.195560972423146406e-01 7.519573115237726535e-01 6.253384926911926822e-01
6.259635861936422296e-01 7.547078996712067722e-01 6.291191822939783407e-01
6.323414409254812796e-01 7.574676913865331374e-01 6.329501570682624090e-01
6.386911927456534466e-01 7.602366470910791874e-01 6.368310037415121361e-01
6.450132383997043695e-01 7.630150778921287458e-01 6.407611911364364810e-01
6.513079749051275957e-01 7.658032878094622742e-01 6.447401822997557153e-01
6.575757994324034073e-01 7.686015739346664377e-01 6.487674349657708284e-01
6.638171092147175933e-01 7.714102265818907345e-01 6.528424020163275943e-01
6.700323014813579503e-01 7.742295294309987641e-01 6.569645319377305226e-01
6.762217734101813038e-01 7.770597596640953508e-01 6.611332692747456941e-01
6.823859220949484161e-01 7.799011880964143995e-01 6.653480550814426797e-01
6.885251445237092760e-01 7.827540793025653532e-01 6.696083273682165160e-01
6.946398375648411561e-01 7.856186917391158042e-01 6.739135215439332471e-01
7.007303979576740005e-01 7.884952778644843674e-01 6.782630708517513041e-01
7.067972223051004477e-01 7.913840842570615264e-01 6.826564067967675342e-01
7.128407070658809852e-01 7.942853517324697243e-01 6.870929595632513376e-01
7.188612485448663270e-01 7.971993154607720511e-01 6.915721584188375681e-01
7.248592428796667431e-01 8.001262050844080154e-01 6.960934321026465144e-01
7.308350860228653989e-01 8.030662448375227580e-01 7.006562091939169123e-01
7.367891737192681090e-01 8.060196536672460388e-01 7.052599184573171698e-01
7.427219014782563411e-01 8.089866453573647531e-01 7.099039891607071828e-01
7.486327256130643759e-01 8.119678017125729896e-01 7.145873998348200029e-01
7.545223076817887398e-01 8.149632306197714948e-01 7.193096566367255251e-01
7.603915796412942241e-01 8.179729213555224643e-01 7.240704295566047222e-01
7.662409381414175824e-01 8.209970689783633313e-01 7.288691437189614986e-01
7.720707796269933310e-01 8.240358639329949941e-01 7.337052258319404219e-01
7.778815003305534770e-01 8.270894921786368092e-01 7.385781042557458820e-01
7.836734962813408645e-01 8.301581353160575327e-01 7.434872090029538416e-01
7.894471633387081244e-01 8.332419707110174656e-01 7.484319716624078245e-01
7.952028972601549173e-01 8.363411716110775718e-01 7.534118252377455249e-01
8.009401766621723207e-01 8.394563042330350777e-01 7.584255820860429376e-01
8.066584706735069332e-01 8.425879497641356464e-01 7.634719632765348818e-01
8.123599093553812711e-01 8.457355276105180675e-01 7.685515296147656938e-01
8.180448941186928558e-01 8.488991950022984900e-01 7.736637090147417961e-01
8.237138279483190439e-01 8.520791051833870311e-01 7.788079287668441264e-01
8.293671160911614271e-01 8.552754073408689317e-01 7.839836146965878383e-01
8.350048677827189847e-01 8.584883831868801440e-01 7.891899443847590900e-01
8.406246724049463159e-01 8.617194786977677712e-01 7.944239257301034529e-01
8.462299834685677036e-01 8.649674562084326279e-01 7.996873827745283325e-01
8.518212329614019973e-01 8.682324447773152043e-01 8.049797193798620132e-01
8.573988637074910768e-01 8.715145671024431273e-01 8.103003320898716222e-01
8.629633327895804840e-01 8.748139384163003962e-01 8.156486084917724533e-01
8.685112937853189941e-01 8.781325055194139084e-01 8.210202040569095638e-01
8.740469282893844616e-01 8.814686228402575097e-01 8.264178662740626624e-01
8.795708463666609411e-01 8.848223352046364898e-01 8.318410153090299852e-01
8.850836254841629724e-01 8.881937049832523412e-01 8.372889893254339411e-01
8.905836610629007666e-01 8.915838848352710677e-01 8.427587078675609078e-01
8.960718329388820402e-01 8.949928217018373600e-01 8.482495190363266158e-01
9.015509263802756745e-01 8.984195016922670307e-01 8.537628451897162352e-01
9.070218374764418279e-01 9.018638497314315217e-01 8.592980228813623667e-01
9.124832972967743538e-01 9.053269113370198129e-01 8.648517190341429295e-01
9.179365336943248188e-01 9.088084827838653901e-01 8.704232606428425889e-01
9.233851804191220980e-01 9.123071100087158936e-01 8.760148319228351355e-01
9.288308811012930821e-01 9.158223472192428272e-01 8.816263177427966502e-01
9.342716413123769437e-01 9.193556495997508016e-01 8.872531405900644375e-01
9.397124373542273812e-01 9.229047657151164819e-01 8.928998260600508052e-01
9.451563753217823161e-01 9.264682865796181055e-01 8.985697730985615639e-01
9.506035559047821826e-01 9.300461763422669392e-01 9.042646579991682199e-01
9.560531046363628382e-01 9.336385357088663461e-01 9.099886803187530182e-01
9.615066616129493982e-01 9.372434217616608665e-01 9.157560566822336989e-01
9.669573847273637002e-01 9.408624710997687268e-01 9.215795307640300971e-01
9.723870692594612786e-01 9.445024597948724621e-01 9.274670258562995873e-01
9.777785730890226068e-01 9.481687534308861354e-01 9.334364948680430318e-01
9.831050718338244510e-01 9.518727670560837018e-01 9.394860306213083101e-01
9.883417388454437402e-01 9.556282921109976458e-01 9.455836323325411685e-01
9.934918422996558141e-01 9.594375624216472387e-01 9.516983192548315040e-01
9.985763296811461798e-01 9.632965417140263442e-01 9.577895036430327247e-01
9.942114721489739848e-01 9.649414783718816002e-01 9.591713509300946461e-01
9.916915526798163460e-01 9.600677293546330260e-01 9.527406681900515428e-01
9.892073759214962125e-01 9.552017644060696311e-01 9.462702365737246657e-01
9.867719407557972167e-01 9.503380654950176476e-01 9.397586228881678050e-01
9.843739071729306067e-01 9.454788135288768602e-01 9.332265558186634280e-01
9.820182926871906526e-01 9.406217084851765664e-01 9.266730991029579201e-01
9.797019478013845317e-01 9.357670195072623764e-01 9.201050706729160256e-01
9.774207980730996725e-01 9.309154126689619391e-01 9.135293740478817037e-01
9.751815609868391688e-01 9.260643851639969171e-01 9.069399044850446900e-01
9.729704910473376822e-01 9.212176417076594070e-01 9.003536220502811327e-01
9.708034919455280631e-01 9.163699401531895106e-01 8.937532865598888376e-01
9.686635749176782939e-01 9.115260705110270756e-01 8.871590238431372732e-01
9.665594650012595546e-01 9.066829842770773862e-01 8.805615104307024099e-01
9.644872784946666444e-01 9.018415117599095643e-01 8.739656672915365743e-01
9.624420742140847862e-01 8.970028582796596428e-01 8.673774626543144795e-01
9.604345584612016262e-01 8.921632903093835720e-01 8.607852860310053478e-01
9.584498054598472594e-01 8.873272041947983801e-01 8.542061390733795001e-01
9.564989754176022041e-01 8.824906944948268661e-01 8.476279025304513937e-01
9.545749598026603833e-01 8.776557062005478915e-01 8.410587414202306267e-01
9.526745457712032517e-01 8.728229738617154787e-01 8.345024009748316374e-01
9.508087846256844111e-01 8.679885222040847337e-01 8.279470532549256800e-01
9.489631025192423186e-01 8.631568397645421609e-01 8.214088147188137734e-01
9.471457599180185261e-01 8.583248540920686009e-01 8.148789280451533834e-01
9.453550305765402451e-01 8.534927996091392632e-01 8.083595091982623826e-01
9.435830323389927665e-01 8.486630445127123501e-01 8.018591800395635794e-01
9.418425662160879730e-01 8.438308651791852633e-01 7.953645386207840451e-01
9.401224124677813876e-01 8.389997911592477209e-01 7.888877269212305476e-01
9.384205871436177571e-01 8.341702205035793627e-01 7.824309818729748844e-01
9.367501034134483318e-01 8.293372224294222050e-01 7.759809608693934990e-01
9.350962812812430025e-01 8.245056601392387607e-01 7.695531958095181979e-01
9.334610667718560295e-01 8.196745424298020888e-01 7.631458380069808811e-01
9.318539060108851357e-01 8.148400979617762552e-01 7.567494905851217535e-01
9.302622555140654947e-01 8.100065567310004155e-01 7.503772218901773039e-01
9.286876056094349741e-01 8.051730724177085241e-01 7.440277117015546837e-01
9.271395831502474705e-01 8.003357056772891776e-01 7.376916134646933632e-01
9.256059253327618697e-01 7.954987025858116789e-01 7.313814866564718464e-01
9.240863437601837260e-01 7.906618776065421628e-01 7.250978248201844778e-01
9.225925658427436282e-01 7.858203843156221780e-01 7.188294627149656169e-01
9.211126169365579930e-01 7.809784744974170856e-01 7.125884635347393692e-01
9.196455684792974594e-01 7.761362045971830215e-01 7.063759865046427278e-01
9.181979022265810420e-01 7.712906810954774928e-01 7.001861662941214481e-01
9.167672001571998130e-01 7.664424894801199484e-01 6.940217357022171463e-01
9.153481567945904729e-01 7.615934195093969628e-01 6.878880649909780987e-01
9.139404639768551331e-01 7.567432836340363123e-01 6.817857667786046960e-01
9.125539294301500126e-01 7.518876928783747582e-01 6.757062380855892725e-01
9.111780957235272593e-01 7.470305845086899765e-01 6.696595752391263368e-01
9.098122501649290594e-01 7.421719339914916169e-01 6.636468136699643638e-01
9.084563128768237128e-01 7.373114504136536462e-01 6.576684342801734084e-01
9.071185731273107011e-01 7.324452193667679856e-01 6.517176237314421527e-01
9.057893354554222842e-01 7.275770233727173464e-01 6.458034937673092779e-01
9.044682125236334080e-01 7.227067015228565428e-01 6.399268508324190696e-01
9.031548045364521382e-01 7.178340969900336432e-01 6.340885324865254136e-01
9.018536281154535539e-01 7.129568373829493488e-01 6.282852974032566706e-01
9.005618093543485969e-01 7.080758191596225881e-01 6.225202235043866272e-01
8.992759701121831872e-01 7.031922144373208283e-01 6.167967177972875081e-01
8.979956379512032960e-01 6.983059000713781606e-01 6.111157531581823399e-01
8.967203212860314077e-01 6.934167616666734313e-01 6.054783383481375791e-01
8.954526708976771054e-01 6.885231926405954717e-01 5.998830635410449252e-01
8.941915566016080952e-01 6.836253365962329243e-01 5.943315847613384051e-01
8.929334172678807802e-01 6.787245284319090022e-01 5.888273795844850556e-01
8.916776705242244194e-01 6.738207055230612808e-01 5.833715948784413685e-01
8.904237095992360018e-01 6.689138192653601989e-01 5.779654133729836829e-01
8.891709022410747565e-01 6.640038362746348843e-01 5.726100532447696567e-01
8.879187220887899690e-01 6.590906724758603952e-01 5.673066752652388134e-01
8.866700510547862457e-01 6.541724962989774461e-01 5.620541403257021118e-01
8.854200814778028228e-01 6.492513861023391231e-01 5.568566631651752363e-01
8.841680786914359880e-01 6.443273831269403784e-01 5.517155748042765762e-01
8.829132805112128723e-01 6.394005490891804255e-01 5.466322335097351104e-01
8.816548969914349554e-01 6.344709672901969189e-01 5.416080224154796730e-01
8.803921103664992254e-01 6.295387436822825755e-01 5.366443467601184070e-01
8.791240751896489680e-01 6.246040078804957485e-01 5.317426307330340718e-01
8.778499186810901911e-01 6.196669141071349252e-01 5.269043139249947050e-01
8.765687412960717628e-01 6.147276420564155019e-01 5.221308473833028430e-01
8.752796175219744734e-01 6.097863976665317542e-01 5.174236892760738504e-01
8.739815969115529715e-01 6.048434137862945814e-01 5.127843001752147023e-01
8.726737941588253999e-01 5.998988989392898263e-01 5.082140946168248741e-01
8.713557496267209102e-01 5.949528226638209905e-01 5.037142772058214035e-01
8.700256052408116281e-01 5.900059904309469250e-01 4.992866995883939452e-01
8.686823445912698061e-01 5.850587401058766623e-01 4.949327569364067592e-01
8.673249354213553586e-01 5.801114372307474287e-01 4.906538189940125583e-01
8.659523319452930856e-01 5.751644748762787529e-01 4.864512237593011101e-01
8.645634773771747605e-01 5.702182733234352208e-01 4.823262709991545383e-01
8.631573066573615671e-01 5.652732795695735168e-01 4.782802156484906031e-01
8.617327493598900823e-01 5.603299666552966629e-01 4.743142611496265482e-01
8.602887327613949475e-01 5.553888328102728478e-01 4.704295527914130193e-01
8.588244189007790963e-01 5.504502458314576296e-01 4.666271134383851993e-01
8.573388771464994784e-01 5.455146532392860514e-01 4.629079425847138496e-01
8.558307110590109845e-01 5.405828402802138610e-01 4.592730689337449212e-01
8.542988567054966564e-01 5.356554029582069054e-01 4.557233480219839428e-01
8.527422657992969057e-01 5.307329553519699594e-01 4.522595444303549317e-01
8.511599092771463537e-01 5.258161276783985816e-01 4.488823265086418490e-01
8.495507808257551918e-01 5.209055642285482790e-01 4.455922615760174454e-01
8.479139003246871642e-01 5.160019211927755478e-01 4.423898116530999292e-01
8.462483171732270160e-01 5.111058643932878676e-01 4.392753297751230135e-01
8.445531134701587117e-01 5.062180669436533442e-01 4.362490569289804720e-01
8.428274070171406507e-01 5.013392068558483183e-01 4.333111196492664408e-01
8.410703541185753362e-01 4.964699646160714575e-01 4.304615283001137493e-01
8.392811521535089581e-01 4.916110207509353791e-01 4.277001760608312164e-01
8.374593644075595256e-01 4.867627987429819503e-01 4.250269048992098009e-01
8.356043139975503076e-01 4.819259312622626301e-01 4.224414243859783702e-01
8.337148868611683472e-01 4.771014353948369036e-01 4.199431953989596344e-01
8.317904658009995789e-01 4.722899701137897588e-01 4.175316375108732991e-01
8.298304844983901418e-01 4.674921821164159108e-01 4.152060584430907197e-01
8.278344282915189867e-01 4.627087037890093568e-01 4.129656573155296440e-01
8.258018346451376779e-01 4.579401513029328075e-01 4.108095284678619508e-01
8.237322933175019735e-01 4.531871228543432051e-01 4.087366658020525345e-01
8.216256878784896633e-01 4.484499819155622347e-01 4.067461112065531847e-01
8.194815690724320811e-01 4.437294051177347876e-01 4.048366204140803060e-01
8.172995050178583076e-01 4.390260854893064946e-01 4.030068099238640067e-01
8.150792968857805132e-01 4.343405321271661679e-01 4.012553165972070901e-01
8.128207950280410543e-01 4.296732294003547947e-01 3.995807052170325946e-01
8.105238975108269850e-01 4.250246362540385792e-01 3.979814747698801614e-01
8.081885484630557670e-01 4.203951856851556590e-01 3.964560648094507256e-01
8.058148470586988799e-01 4.157851737769109879e-01 3.950029765120905423e-01
8.034026962411985329e-01 4.111951046689115152e-01 3.936204324142628108e-01
8.009521378981909745e-01 4.066253627118514569e-01 3.923066947182650144e-01
7.984632874676697023e-01 4.020762706733001512e-01 3.910600267933756480e-01
7.959362958998934534e-01 3.975481251157440554e-01 3.898786646276527490e-01
7.933713473226599033e-01 3.930411968130258504e-01 3.887608225579017307e-01
7.907686574783574507e-01 3.885557303926693296e-01 3.877046999689159890e-01
7.881284544156301752e-01 3.840919640902587529e-01 3.867084595587381712e-01
7.854510012103226302e-01 3.796501032751679605e-01 3.857702673246294345e-01
7.827365958532629397e-01 3.752303177293505598e-01 3.848883050999601374e-01
7.799855573279415033e-01 3.708327552306592834e-01 3.840607581802352732e-01
7.771982233463431422e-01 3.664575425267851405e-01 3.832858192234025463e-01
7.743749481448563010e-01 3.621047863617611329e-01 3.825616918198675998e-01
7.715160518337467188e-01 3.577746368683014655e-01 3.818864831434901075e-01
7.686219524815265380e-01 3.534671195520193154e-01 3.812584982442814296e-01
7.656930517948008497e-01 3.491822750013692245e-01 3.806760122706670524e-01
7.627297518608332494e-01 3.449201381780978570e-01 3.801373059981723590e-01
7.597324610418284552e-01 3.406807299395016586e-01 3.796406843015277532e-01
7.567015922742318379e-01 3.364640582281425152e-01 3.791844777860022275e-01
7.536375467881040180e-01 3.322701411099439062e-01 3.787669970190922220e-01
7.505407196227874556e-01 3.280990002753491619e-01 3.783865433211569540e-01
7.474115712082280982e-01 3.239505493622616417e-01 3.780416634360889150e-01
7.442505183508412170e-01 3.198247550352249502e-01 3.777308063416947026e-01
7.410579750279425726e-01 3.157215772518807695e-01 3.774524509829353947e-01
7.378343512384677449e-01 3.116409704467773545e-01 3.772051064266770948e-01
7.345800519344474200e-01 3.075828847103421748e-01 3.769873118279194468e-01
7.312954622460766663e-01 3.035472925968438762e-01 3.767975677519167510e-01
7.279809779377566237e-01 2.995341347142527755e-01 3.766344783074295766e-01
7.246370003939071047e-01 2.955433232127043786e-01 3.764967560209065978e-01
7.212639047983969709e-01 2.915748020657008555e-01 3.763830604225979481e-01
7.178620569041518351e-01 2.876285169108695472e-01 3.762920784118068407e-01
7.144318123927204667e-01 2.837044162122266955e-01 3.762225233422813453e-01
7.109735162895217675e-01 2.798024524280844916e-01 3.761731340017399616e-01
7.074875003639726767e-01 2.759225885803616163e-01 3.761426556123988463e-01
7.039740902088845731e-01 2.720647802144139371e-01 3.761299015349165442e-01
7.004335983291459788e-01 2.682289908573779469e-01 3.761337091569335600e-01
6.968663212773178461e-01 2.644152012227610760e-01 3.761529095527339495e-01
6.932725426766019883e-01 2.606234021780075572e-01 3.761863526116585033e-01
6.896525329109274294e-01 2.568535960113472738e-01 3.762329055321002591e-01
6.860065488492232966e-01 2.531057977294109973e-01 3.762914512548127810e-01
6.823348363761195801e-01 2.493800214422073891e-01 3.763609500820121467e-01
6.786376242649662105e-01 2.456763038128263466e-01 3.764403535110733556e-01
6.749151231751110425e-01 2.419947210999891518e-01 3.765285264384131692e-01
6.711675324616499516e-01 2.383353539955441192e-01 3.766243994612780699e-01
6.673950369964636309e-01 2.346983031289386346e-01 3.767269100578239382e-01
6.635978070649097837e-01 2.310836906390836831e-01 3.768350007100279009e-01
6.597759982863891093e-01 2.274916618065601082e-01 3.769476169820946132e-01
6.559297473209481089e-01 2.239223513030393353e-01 3.770639116649144307e-01
6.520591754729223588e-01 2.203759654266244650e-01 3.771828077558652681e-01
6.481643941443696599e-01 2.168527541906234979e-01 3.773031206968150975e-01
6.442454984979625321e-01 2.133529749202845993e-01 3.774237883397635329e-01
6.403025689859566105e-01 2.098769174022531714e-01 3.775437410683403772e-01
6.363356714241054091e-01 2.064249059420060206e-01 3.776618996147099727e-01
6.323448551853009247e-01 2.029972985879547887e-01 3.777771986896097389e-01
6.283301122985771592e-01 1.995944416686266654e-01 3.778890664877021521e-01
6.242914965761068302e-01 1.962168403108488501e-01 3.779958773221168133e-01
6.202290151835444521e-01 1.928649814733044143e-01 3.780964890449534654e-01
6.161426615427991749e-01 1.895393981082279800e-01 3.781897379927290359e-01
6.120324156302930918e-01 1.862406715933664081e-01 3.782744366000001524e-01
6.078982443238682976e-01 1.829694341941491276e-01 3.783493709986866516e-01
6.037399990506459035e-01 1.797263486622488471e-01 3.784140428470601503e-01
5.995576965897967403e-01 1.765121975913181707e-01 3.784665259488561584e-01
5.953512720943007208e-01 1.733277858871770660e-01 3.785054621982753553e-01
5.911206437729070728e-01 1.701739770998706713e-01 3.785295035265708874e-01
5.868657195674668037e-01 1.670516976788867514e-01 3.785372630540221883e-01
5.825863435035250060e-01 1.639619511220119508e-01 3.785276019365447775e-01
5.782823203682633251e-01 1.609058321765788335e-01 3.784994295515334839e-01
5.739536466014196758e-01 1.578844538430557720e-01 3.784505787767488694e-01
5.696001981012460691e-01 1.548990153349074916e-01 3.783794775494400686e-01
5.652218458222221242e-01 1.519507861175737884e-01 3.782845039525286057e-01
5.608184279263467298e-01 1.490411220014642157e-01 3.781641038386275300e-01
5.563896048179566289e-01 1.461715605343842650e-01 3.780173288805740439e-01
5.519354537512982661e-01 1.433434802585686063e-01 3.778414852793369194e-01
5.474558435278772395e-01 1.405584279152288785e-01 3.776347269993366451e-01
5.429506470730138812e-01 1.378180135121435668e-01 3.773951523619960002e-01
5.384196501144510316e-01 1.351239813932839096e-01 3.771211030874267456e-01
5.338626135422905872e-01 1.324781762459926737e-01 3.768109124380045194e-01
5.292796498492590151e-01 1.298822067930652524e-01 3.764618033826110377e-01
5.246706783487381509e-01 1.273378756852094063e-01 3.760716446399895441e-01
5.200356363922209457e-01 1.248470120733735367e-01 3.756382551974452033e-01
5.153742367301422656e-01 1.224117140591406694e-01 3.751600258018771838e-01
5.106866581816573714e-01 1.200336468026920456e-01 3.746341300161435961e-01
5.059729862980435477e-01 1.177145617740232852e-01 3.740580772636119544e-01
5.012332745984564575e-01 1.154562389937715539e-01 3.734295174784395543e-01
4.964674957217509177e-01 1.132605442683873864e-01 3.727463195168757570e-01
4.916758190194626121e-01 1.111290906599490258e-01 3.720059818290880060e-01
4.868585609140311798e-01 1.090632523851999269e-01 3.712058226357082269e-01
4.820159589014604840e-01 1.070644243738244350e-01 3.703434438399521023e-01
4.771482837998479165e-01 1.051338742417683159e-01 3.694164970204523168e-01
4.722559296975721854e-01 1.032725993184717139e-01 3.684225499867860298e-01
4.673393879073711177e-01 1.014813305146687883e-01 3.673591735272957459e-01
4.623991454541829804e-01 9.976065343840129218e-02 3.662241176536292220e-01
4.574358041489324234e-01 9.811082191626963045e-02 3.650151416090168244e-01
4.524502157463570762e-01 9.653152558968838837e-02 3.637298817526759542e-01
4.474429556959144683e-01 9.502266265951053725e-02 3.623665563347658880e-01
4.424148033564082039e-01 9.358361785401031474e-02 3.609233190012756665e-01
4.373666093617918915e-01 9.221344371517425920e-02 3.593984619211612608e-01
4.323000061476657274e-01 9.090969474799345806e-02 3.577897503619592023e-01
4.272154852828886629e-01 8.967151542383772211e-02 3.560963875772367726e-01
4.221140984792258188e-01 8.849693359303587026e-02 3.543172112905775273e-01
4.169969758462759302e-01 8.738362154375492463e-02 3.524512375858371849e-01
4.118658301965831270e-01 8.632803919827600203e-02 3.504973239811765007e-01
4.067221669675408213e-01 8.532676500724681312e-02 3.484548346345710534e-01
4.015666852977432533e-01 8.437757766440442952e-02 3.463238990387739746e-01
3.964007130297593218e-01 8.347701754004596686e-02 3.441044235724556866e-01
3.912256097683191602e-01 8.262141527259461715e-02 3.417965388380971303e-01
3.860430941874281596e-01 8.180633277581816909e-02 3.394004639140563162e-01
3.808555339288969832e-01 8.102605665725876039e-02 3.369164884312597086e-01
3.756631594887540060e-01 8.027857713191724476e-02 3.343459517649038371e-01
3.704673590324250032e-01 7.955977986332424257e-02 3.316898655204278401e-01
3.652695090362573227e-01 7.886552703397090025e-02 3.289494343067977944e-01
3.600709660701238435e-01 7.819169210462800779e-02 3.261260438198446687e-01
3.548730588141333908e-01 7.753419260275734581e-02 3.232212473818363851e-01
3.496770804198256477e-01 7.688902041627435069e-02 3.202367511798585031e-01
3.444853557545813350e-01 7.625034119537774102e-02 3.171743822649770728e-01
3.392982604990861795e-01 7.561586038621423422e-02 3.140362188799208365e-01
3.341166082970875029e-01 7.498254313274954619e-02 3.108243324765390669e-01
3.289414559917514524e-01 7.434697155739058982e-02 3.075408588184366798e-01
3.237737954171283072e-01 7.370590341662106026e-02 3.041879989569576392e-01
3.186145498482620964e-01 7.305628171756148315e-02 3.007680026557595920e-01
3.134645711821229530e-01 7.239524117563511663e-02 2.972831523530131137e-01
3.083246378402036969e-01 7.172011173017686647e-02 2.937357478389135412e-01
3.031954533707552635e-01 7.102841937235576664e-02 2.901280917978547591e-01
2.980776457172774063e-01 7.031788456468926474e-02 2.864624763353166292e-01
2.929717671102379239e-01 6.958641854495320467e-02 2.827411705802333475e-01
2.878782945311796904e-01 6.883211781082057557e-02 2.789664094251954607e-01
2.827976306923836725e-01 6.805325707674123037e-02 2.751403834400568682e-01
2.777301054710668571e-01 6.724828098296936618e-02 2.712652299698696257e-01
2.726759777346169922e-01 6.641579481976991883e-02 2.673430254060571443e-01
2.676354374924288515e-01 6.555455450916861104e-02 2.633757786005420098e-01
2.626097591395918363e-01 6.466149844681601255e-02 2.593660533804174051e-01
2.575992689989206608e-01 6.373522644476575794e-02 2.553159580518031269e-01
2.526028775756383737e-01 6.277674824557366584e-02 2.512267303676423147e-01
2.476205211412589313e-01 6.178532806314660647e-02 2.471000835041496368e-01
2.426520784951842757e-01 6.076032925135082391e-02 2.429376426307544024e-01
2.376973739685813714e-01 5.970120319443159712e-02 2.387409437758167274e-01
2.327561803770203386e-01 5.860747847771281133e-02 2.345114333677875140e-01
2.278285229664566147e-01 5.747825339035907838e-02 2.302506881609430733e-01
2.229180662380075839e-01 5.630663136815370479e-02 2.259630047830259447e-01
2.180204522985965676e-01 5.509893160286010588e-02 2.216467626153181270e-01
2.131352588343927157e-01 5.385491895770589538e-02 2.173030574920574165e-01
2.082620235717315138e-01 5.257438772048273617e-02 2.129328977906284059e-01
2.034002463374002811e-01 5.125715285178860520e-02 2.085372063771265272e-01]; 

   case 'amp' 
      RGB = [9.463470914425774483e-01 9.290101343908121478e-01 9.257532417012246384e-01
9.437115115548888600e-01 9.244624965319422349e-01 9.206701514370421169e-01
9.413263164430620833e-01 9.198473615841167295e-01 9.154240958990897958e-01
9.390547090113202655e-01 9.152093965940418796e-01 9.101090387889319011e-01
9.368634575300728295e-01 9.105592179043572321e-01 9.047473920040983719e-01
9.347358946162058757e-01 9.059020202913443676e-01 8.993502288257003707e-01
9.326614522721177192e-01 9.012410121503533489e-01 8.939244930449518067e-01
9.306333250978273686e-01 8.965781745298962990e-01 8.884745648022901454e-01
9.286461606427048876e-01 8.919150224616214651e-01 8.830038737434344753e-01
9.266960476646858291e-01 8.872526027444326280e-01 8.775148739503740858e-01
9.247795820079757201e-01 8.825918040419664656e-01 8.720097084454925263e-01
9.228944807039246578e-01 8.779331456742857087e-01 8.664897344910690302e-01
9.210380342808240917e-01 8.732773006486703737e-01 8.609566713320465636e-01
9.192087851735465387e-01 8.686245228987975464e-01 8.554113284148980867e-01
9.174048150925064871e-01 8.639752297661511538e-01 8.498548926033711037e-01
9.156244455582711606e-01 8.593297631172468476e-01 8.442883976261035262e-01
9.138671371679708555e-01 8.546880756641417332e-01 8.387120160257571788e-01
9.121305737300281491e-01 8.500507404321120397e-01 8.331273220967562176e-01
9.104149075499773369e-01 8.454174764656698926e-01 8.275339791691540547e-01
9.087182868982504047e-01 8.407887046755945226e-01 8.219332381316957203e-01
9.070398066265998871e-01 8.361645202367321561e-01 8.163256242481420344e-01
9.053796160871283583e-01 8.315446471258203243e-01 8.107108259382036497e-01
9.037352419915254398e-01 8.269297416714002091e-01 8.050906531516163200e-01
9.021080083908009639e-01 8.223191128622155954e-01 7.994638601040017223e-01
9.004956244333240933e-01 8.177133605075642686e-01 7.938321412777887831e-01
8.988977241110835958e-01 8.131124010808855607e-01 7.881956419539296599e-01
8.973148627806266653e-01 8.085158161746068828e-01 7.825537446961500221e-01
8.957442417382828204e-01 8.039244072828748422e-01 7.769086245337948338e-01
8.941881096565635900e-01 7.993371392915321616e-01 7.712582600059587623e-01
8.926437234223070227e-01 7.947548018176209261e-01 7.656048088583016220e-01
8.911109613594879741e-01 7.901772295732085727e-01 7.599482475326526654e-01
8.895909134152744091e-01 7.856038035950861920e-01 7.542875143411862382e-01
8.880802871606852111e-01 7.810355387672826000e-01 7.486252986030703660e-01
8.865819248532246233e-01 7.764711589075004028e-01 7.429590373479496579e-01
8.850930260657449145e-01 7.719115034698935673e-01 7.372910251414285243e-01
8.836131402967402071e-01 7.673565360639147404e-01 7.316215502737657417e-01
8.821444223454562028e-01 7.628052217660783452e-01 7.259486135652994943e-01
8.806827953756125593e-01 7.582589101067193083e-01 7.202756895379377466e-01
8.792311851397603961e-01 7.537162649935950087e-01 7.146000961578288235e-01
8.777875677079440830e-01 7.491778548550656058e-01 7.089235227883947665e-01
8.763501988878248383e-01 7.446441541668697983e-01 7.032474436364752890e-01
8.749232756241082098e-01 7.401133023432254765e-01 6.975679937396324082e-01
8.735014600514631189e-01 7.355871888710872053e-01 6.918898915870100863e-01
8.720870217729678187e-01 7.310647072768980959e-01 6.862110066741609060e-01
8.706800348126626510e-01 7.265456103843925817e-01 6.805311805702053407e-01
8.692770022519556994e-01 7.220310948598716028e-01 6.748535454654478460e-01
8.678827146915726320e-01 7.175190094979648769e-01 6.691736166956238074e-01
8.664927921052477666e-01 7.130109157552886323e-01 6.634953613259193528e-01
8.651068719926267025e-01 7.085067580022798017e-01 6.578190657407729791e-01
8.637289803901067042e-01 7.040046558915140640e-01 6.521408976142408775e-01
8.623534679961476490e-01 6.995067341461493893e-01 6.464660893001026309e-01
8.609830606578612322e-01 6.950116457163946215e-01 6.407920274183934728e-01
8.596180478330472940e-01 6.905190440208218705e-01 6.351183727606340979e-01
8.582544825018789680e-01 6.860303869186620274e-01 6.294488360238388314e-01
8.568970944227636277e-01 6.815434400545420379e-01 6.237788543057761759e-01
8.555425925651013452e-01 6.770593865574592307e-01 6.181115291897923969e-01
8.541886800910097888e-01 6.725790118919016125e-01 6.124490539806487499e-01
8.528418735895523239e-01 6.680992506239646911e-01 6.067850786519378703e-01
8.514956700712879023e-01 6.636227191543088155e-01 6.011258775897345696e-01
8.501500355486913962e-01 6.591492264653023847e-01 5.954714806881288292e-01
8.488106382652110815e-01 6.546760150904927800e-01 5.898162979999663769e-01
8.474706581846417341e-01 6.502059048227525340e-01 5.841670108646530579e-01
8.461313452075343022e-01 6.457381204225883797e-01 5.785223911431566224e-01
8.447969154607379849e-01 6.412705018379197819e-01 5.728782339090506825e-01
8.434612056654142709e-01 6.368056357299208825e-01 5.672406638938772838e-01
8.421258352164873173e-01 6.323425572542190620e-01 5.616080798917625350e-01
8.407944708783339216e-01 6.278793105389963713e-01 5.559767932481035624e-01
8.394611847956942041e-01 6.234184331258648681e-01 5.503527766345649441e-01
8.381273683455257029e-01 6.189590462186138620e-01 5.447346560426542528e-01
8.367972635859407537e-01 6.144988712509784623e-01 5.391181340824646728e-01
8.354646463246349075e-01 6.100406459813725313e-01 5.335095625632335636e-01
8.341299717058989760e-01 6.055839279215072812e-01 5.279085280772741751e-01
8.327994021627661558e-01 6.011254425561419756e-01 5.223087380798424606e-01
8.314657754399900069e-01 5.966684488411785336e-01 5.167175843922774403e-01
8.301288592609736838e-01 5.922128365915518833e-01 5.111353769747372100e-01
8.287951011685089631e-01 5.877551092176589442e-01 5.055554913593263144e-01
8.274588476811213233e-01 5.832978393600145584e-01 4.999838390239015884e-01
8.261188250087038165e-01 5.788414339387950580e-01 4.944218283197156505e-01
8.247786369803286055e-01 5.743837798031475872e-01 4.888657757771856516e-01
8.234381838800088893e-01 5.699246359812502050e-01 4.833158537977020885e-01
8.220935149852323098e-01 5.654657957469002572e-01 4.777762994077761616e-01
8.207444566974767541e-01 5.610071021581608530e-01 4.722474096096786478e-01
8.193981033425531413e-01 5.565444762341525964e-01 4.667217962382390062e-01
8.180472685091408902e-01 5.520814646564632389e-01 4.612071590527028753e-01
8.166916431769154494e-01 5.476179715812223847e-01 4.557039631596026541e-01
8.153328745786677656e-01 5.431528161626910656e-01 4.502105997079735578e-01
8.139743540529152943e-01 5.386837997184832361e-01 4.447235925743368568e-01
8.126106579688792131e-01 5.342136249170785778e-01 4.392488743242201066e-01
8.112416585089723409e-01 5.297420805368263652e-01 4.337867573467541482e-01
8.098689150112462487e-01 5.252679783528519941e-01 4.283357546423488538e-01
8.084956135344389949e-01 5.207891399756702233e-01 4.228926326123328416e-01
8.071166424763304148e-01 5.163081809730883931e-01 4.174630770396351442e-01
8.057318945579315939e-01 5.118248549122679236e-01 4.120474263101407963e-01
8.043412708255589516e-01 5.073389039757838503e-01 4.066460228640024210e-01
8.029498733805645605e-01 5.028469116894594970e-01 4.012536130705923743e-01
8.015531888402489535e-01 4.983512104818560995e-01 3.958753217358975118e-01
8.001502657810105612e-01 4.938520308181937879e-01 3.905124577505094119e-01
7.987410157112331266e-01 4.893490771437783438e-01 3.851654167818974739e-01
7.973253554907255847e-01 4.848420416222078422e-01 3.798346066585175373e-01
7.959075773999774173e-01 4.803278114438007118e-01 3.745157291750967898e-01
7.944838857609586302e-01 4.758083650556605426e-01 3.692132449197628175e-01
7.930533836294737515e-01 4.712838795689627269e-01 3.639285059842314873e-01
7.916159847493996482e-01 4.667540085796573757e-01 3.586620124504662499e-01
7.901716040813833164e-01 4.622183930878590030e-01 3.534142884263108408e-01
7.887201567691978221e-01 4.576766614569128255e-01 3.481858850345221357e-01
7.872638957915846225e-01 4.531268310219068174e-01 3.429748729037092048e-01
7.858016971109998972e-01 4.485691650403910713e-01 3.377829921447058070e-01
7.843319196766078694e-01 4.440043051111859929e-01 3.326125640359084423e-01
7.828544529945841157e-01 4.394318439857143876e-01 3.274643000606006771e-01
7.813691807701973469e-01 4.348513628187685720e-01 3.223389566893520852e-01
7.798759793897394044e-01 4.302624315207902450e-01 3.172373399087716761e-01
7.783747162977331380e-01 4.256646092089760858e-01 3.121603100704657408e-01
7.768652482649208713e-01 4.210574447687118194e-01 3.071087870719240720e-01
7.753474195428932125e-01 4.164404775377301138e-01 3.020837558793280642e-01
7.738210599017090185e-01 4.118132381262810671e-01 2.970862723999735389e-01
7.722859825475274498e-01 4.071752493875019363e-01 2.921174697090146433e-01
7.707419819180442166e-01 4.025260275530667675e-01 2.871785646313976814e-01
7.691888313545882649e-01 3.978650835500125438e-01 2.822708646751007633e-01
7.676262806509283054e-01 3.931919245154482101e-01 2.773957753058901021e-01
7.660540534803422785e-01 3.885060555265225091e-01 2.725548075466420750e-01
7.644718447042359033e-01 3.838069815635994364e-01 2.677495858756240299e-01
7.628793175676447103e-01 3.790942097250059595e-01 2.629818563878072446e-01
7.612761007893139586e-01 3.743672517119065457e-01 2.582534951710902527e-01
7.596617855567490141e-01 3.696256266018116965e-01 2.535665168349998111e-01
7.580359224396768791e-01 3.648688639288839575e-01 2.489230831128432242e-01
7.563980182387591844e-01 3.600965070884658559e-01 2.443255114391649219e-01
7.547475327902162245e-01 3.553081170821107415e-01 2.397762833825934359e-01
7.530838757510928128e-01 3.505032766177708092e-01 2.352780527895846907e-01
7.514064033944252152e-01 3.456815945775675858e-01 2.308336534671754370e-01
7.497144154482878742e-01 3.408427108627715696e-01 2.264461062026797977e-01
7.480093931719870026e-01 3.359841374696425076e-01 2.221167453210010634e-01
7.462904219476443890e-01 3.311055722690471881e-01 2.178492138304993775e-01
7.445546708594931173e-01 3.262086086142511610e-01 2.136488350090644528e-01
7.428011522578348291e-01 3.212930797652773784e-01 2.095194932063288440e-01
7.410289292300998865e-01 3.163587556773898579e-01 2.054651883130465539e-01
7.392422371017411953e-01 3.113999367010875097e-01 2.014864526930904010e-01
7.374344355234756510e-01 3.064221768888288899e-01 1.975919663992890540e-01
7.356041908206575330e-01 3.014256456718681365e-01 1.937864841719280640e-01
7.337546411878117514e-01 2.964055318272263428e-01 1.900722381207737011e-01
7.318814451569350954e-01 2.913651536558918287e-01 1.864564514321402355e-01
7.299813549163196580e-01 2.863068267986909055e-01 1.829454454677189867e-01
7.280589696708802405e-01 2.812237436541007995e-01 1.795417572234761505e-01
7.261058295804370122e-01 2.761243371138519809e-01 1.762545002124356330e-01
7.241247102013246284e-01 2.710038520778982329e-01 1.730875525793737491e-01
7.221106804386585587e-01 2.658668602953934146e-01 1.700480125370592843e-01
7.200637636280236009e-01 2.607119201605753722e-01 1.671411340183387162e-01
7.179797241720924372e-01 2.555430923618877692e-01 1.643731951674216596e-01
7.158585660201038925e-01 2.503590497486887223e-01 1.617494125495184398e-01
7.136946458601574061e-01 2.451661844345556784e-01 1.592752687470095629e-01
7.114894074734618989e-01 2.399613853901303440e-01 1.569555549892431845e-01
7.092368038757341786e-01 2.347521795395665634e-01 1.547943364998323190e-01
7.069352804131667778e-01 2.295401028324536297e-01 1.527950884695459255e-01
7.045831993890990796e-01 2.243269505217069248e-01 1.509608633015488421e-01
7.021764206472544956e-01 2.191186683316911910e-01 1.492928085088980894e-01
6.997128684389145592e-01 2.139184608731344861e-01 1.477916611410667047e-01
6.971911916356166028e-01 2.087286324898059431e-01 1.464577951274126655e-01
6.946080256946649545e-01 2.035552197097931226e-01 1.452888805599515110e-01
6.919615635162069678e-01 1.984021504802455094e-01 1.442823442764895880e-01
6.892502621260994111e-01 1.932732898312458647e-01 1.434346916240710201e-01
6.864728173440260983e-01 1.881724668500568132e-01 1.427415573021981465e-01
6.836276497375878280e-01 1.831044861638990162e-01 1.421968539256191488e-01
6.807138936899735926e-01 1.780732451543534933e-01 1.417942947939738185e-01
6.777309286205458472e-01 1.730825547177755885e-01 1.415269568275668299e-01
6.746783856387729150e-01 1.681360921159456845e-01 1.413874528291547972e-01
6.715560313350845689e-01 1.632376302346780217e-01 1.413677792054587046e-01
6.683639211377226941e-01 1.583906644890515358e-01 1.414598772073469568e-01
6.651023056640176234e-01 1.535985670708633066e-01 1.416556412086291405e-01
6.617715794982882427e-01 1.488646699200451562e-01 1.419469640184388703e-01
6.583721731985413550e-01 1.441925388486675619e-01 1.423253876471278490e-01
6.549047613747174257e-01 1.395853626420375526e-01 1.427832909133252004e-01
6.513700233144809060e-01 1.350465677963595956e-01 1.433130440041158193e-01
6.477686724727176326e-01 1.305797355595543496e-01 1.439072462489923432e-01
6.441014028993071738e-01 1.261888040474711814e-01 1.445583185074540422e-01
6.403689740006455189e-01 1.218777495533199251e-01 1.452593821489098769e-01
6.365721433135939078e-01 1.176507620626412731e-01 1.460041387809807023e-01
6.327116135602410818e-01 1.135124987936143026e-01 1.467862349801432043e-01
6.287880597206695343e-01 1.094681045314993273e-01 1.475989566519263774e-01
6.248021600743940418e-01 1.055229431116234684e-01 1.484364140781085362e-01
6.207545450778898521e-01 1.016827736772349666e-01 1.492933020681679601e-01
6.166458126074587653e-01 9.795388132578869422e-02 1.501640386258630555e-01
6.124766416655831325e-01 9.434293988806133346e-02 1.510418818628477822e-01
6.082475860694411818e-01 9.085689252672626837e-02 1.519224424106922766e-01
6.039592307964848361e-01 8.750309254173271878e-02 1.528005492303129986e-01
5.996123270243045589e-01 8.428892340087804080e-02 1.536699242631612283e-01
5.952075850399283219e-01 8.122188415501030434e-02 1.545253736600938343e-01
5.907456431322212209e-01 7.830966313722748096e-02 1.553624952709374007e-01
5.862275380722600238e-01 7.555913263658070589e-02 1.561749022891834038e-01
5.816543085194019191e-01 7.297679500504322680e-02 1.569571524895727321e-01
5.770267606447557762e-01 7.056924923284621509e-02 1.577054842373717958e-01
5.723466647463710810e-01 6.834024207619035507e-02 1.584124407437495219e-01
5.676151929412465158e-01 6.629402087598951221e-02 1.590741634973840690e-01
5.628336935771917071e-01 6.443348056543876656e-02 1.596865619581833706e-01
5.580050003714496221e-01 6.275633232081309631e-02 1.602413164863800144e-01
5.531297300263749994e-01 6.126496600367018625e-02 1.607383872225261745e-01
5.482116402359448193e-01 5.995167246350390639e-02 1.611686284256901580e-01
5.432516061451815315e-01 5.881547808076787592e-02 1.615323341086055964e-01
5.382535120903690906e-01 5.784532671126727671e-02 1.618221724924289173e-01
5.332186175206464762e-01 5.703660923770780683e-02 1.620384025483199708e-01
5.281513818389996784e-01 5.637398150597446728e-02 1.621742451273615881e-01
5.230520756324539278e-01 5.585383743496476899e-02 1.622329744665089168e-01
5.179273401770114749e-01 5.545235617321788574e-02 1.622049347095048388e-01
5.127760391281988017e-01 5.516900039838320419e-02 1.620968226308776017e-01
5.076034324141470711e-01 5.498326606891081741e-02 1.619030677004743290e-01
5.024122108869354397e-01 5.488240340299161552e-02 1.616236282627719067e-01
4.972028565812885437e-01 5.486041454189729411e-02 1.612621553479370584e-01
4.919800125101927990e-01 5.489873490341760226e-02 1.608156853259256336e-01
4.867469976088732442e-01 5.498319810891529741e-02 1.602839538682285792e-01
4.815035401048063934e-01 5.511074616505500651e-02 1.596718589083305773e-01
4.762519241645499224e-01 5.527092475320084103e-02 1.589804323039821221e-01
4.709943156232950234e-01 5.545404644646995812e-02 1.582109529908402590e-01
4.657362887691403608e-01 5.564106402926658618e-02 1.573608184133624133e-01
4.604760106385220042e-01 5.583489737898188893e-02 1.564363997859181399e-01
4.552150906427448462e-01 5.602873466021515009e-02 1.554396278095509509e-01
4.499550509587917912e-01 5.621638605393316362e-02 1.543724780354263082e-01
4.446972616844994119e-01 5.639243657216582578e-02 1.532370205267698793e-01
4.394429409822757093e-01 5.655220574247237647e-02 1.520353945457279254e-01
4.341931570608791313e-01 5.669170022106548995e-02 1.507697850177304177e-01
4.289488317782668703e-01 5.680756188500035025e-02 1.494424009624717165e-01
4.237107456338875533e-01 5.689701352302534848e-02 1.480554560097830374e-01
4.184795439135307604e-01 5.695780383827742793e-02 1.466111510548973595e-01
4.132557437526688804e-01 5.698815310778364979e-02 1.451116590521219496e-01
4.080397418937663501e-01 5.698670052046273665e-02 1.435591118993865545e-01
4.028318229277383922e-01 5.695245393890951274e-02 1.419555893285963655e-01
3.976321678282654926e-01 5.688474259913823411e-02 1.403031096878759043e-01
3.924408626085833518e-01 5.678317307342812398e-02 1.386036224809996942e-01
3.872615129675306966e-01 5.663896210711539397e-02 1.368568732654815434e-01
3.820928251575063661e-01 5.645532265747305739e-02 1.350656214594591342e-01
3.769329700865952648e-01 5.623657775378711893e-02 1.332326547817182794e-01
3.717817181595340914e-01 5.598316066775241989e-02 1.313596470261583493e-01
3.666387821359218258e-01 5.569561666867810928e-02 1.294481844724836295e-01
3.615063297404105258e-01 5.536890534251808632e-02 1.274985831214487109e-01
3.563880603817025094e-01 5.499475479736596478e-02 1.255105734342905621e-01
3.512776042297060530e-01 5.458755525687735560e-02 1.234884411194991649e-01
3.461744793988128510e-01 5.414817693699586904e-02 1.214334602338749125e-01
3.410813090262644343e-01 5.367068674831452363e-02 1.193455923090092352e-01
3.360026615292399654e-01 5.314516420281268499e-02 1.172242123426087412e-01
3.309302966511110111e-01 5.258939733233552322e-02 1.150735180048746420e-01
3.258636162106021694e-01 5.200433045735335796e-02 1.128944817728364247e-01
3.208133832347068171e-01 5.136699574271335472e-02 1.106843415414836551e-01
3.157696216592825178e-01 5.069836482305852682e-02 1.084473347788683095e-01
3.107301266925714400e-01 5.000263599039595636e-02 1.061847548036680688e-01
3.057061331880795430e-01 4.925629172594595678e-02 1.038940386221758327e-01
3.006879268630521240e-01 4.847913366710451116e-02 1.015787046192155041e-01
2.956726217207689689e-01 4.767654364208809975e-02 9.923998323658683729e-02
2.906746306466351792e-01 4.681900231021129261e-02 9.687492161154248604e-02
2.856782002647242358e-01 4.593789236755790178e-02 9.448789214269404102e-02
2.806890017083342181e-01 4.502128372335665457e-02 9.207800935320636926e-02
2.757102955542003464e-01 4.406225179606895054e-02 8.964513100796869804e-02
2.707312541226000180e-01 4.308157177910076213e-02 8.719176105781661912e-02
2.657667372933599781e-01 4.204947038025994704e-02 8.471559044612611555e-02
2.608025053402583393e-01 4.099314121499578883e-02 8.221947676879418077e-02
2.558451747035172530e-01 3.989182795135969711e-02 7.970262452812598708e-02
2.508938137991353901e-01 3.876029441108298085e-02 7.716551375026653448e-02
2.459428063629143790e-01 3.762148549775875400e-02 7.460911056507216199e-02
2.410021600694670640e-01 3.645746689804394564e-02 7.203245689473128377e-02
2.360563646646140490e-01 3.529747994604028744e-02 6.943744239412558139e-02]; 

   case 'tem' 
      RGB = [9.985763296811461798e-01 9.632965417140263442e-01 9.577895036430327247e-01
9.934918422996558141e-01 9.594375624216472387e-01 9.516983192548315040e-01
9.883417388454437402e-01 9.556282921109976458e-01 9.455836323325411685e-01
9.831050718338244510e-01 9.518727670560837018e-01 9.394860306213083101e-01
9.777785730890226068e-01 9.481687534308861354e-01 9.334364948680430318e-01
9.723870692594612786e-01 9.445024597948724621e-01 9.274670258562995873e-01
9.669573847273637002e-01 9.408624710997687268e-01 9.215795307640300971e-01
9.615066616129493982e-01 9.372434217616608665e-01 9.157560566822336989e-01
9.560531046363628382e-01 9.336385357088663461e-01 9.099886803187530182e-01
9.506035559047821826e-01 9.300461763422669392e-01 9.042646579991682199e-01
9.451563753217823161e-01 9.264682865796181055e-01 8.985697730985615639e-01
9.397124373542273812e-01 9.229047657151164819e-01 8.928998260600508052e-01
9.342716413123769437e-01 9.193556495997508016e-01 8.872531405900644375e-01
9.288308811012930821e-01 9.158223472192428272e-01 8.816263177427966502e-01
9.233851804191220980e-01 9.123071100087158936e-01 8.760148319228351355e-01
9.179365336943248188e-01 9.088084827838653901e-01 8.704232606428425889e-01
9.124832972967743538e-01 9.053269113370198129e-01 8.648517190341429295e-01
9.070218374764418279e-01 9.018638497314315217e-01 8.592980228813623667e-01
9.015509263802756745e-01 8.984195016922670307e-01 8.537628451897162352e-01
8.960718329388820402e-01 8.949928217018373600e-01 8.482495190363266158e-01
8.905836610629007666e-01 8.915838848352710677e-01 8.427587078675609078e-01
8.850836254841629724e-01 8.881937049832523412e-01 8.372889893254339411e-01
8.795708463666609411e-01 8.848223352046364898e-01 8.318410153090299852e-01
8.740469282893844616e-01 8.814686228402575097e-01 8.264178662740626624e-01
8.685112937853189941e-01 8.781325055194139084e-01 8.210202040569095638e-01
8.629633327895804840e-01 8.748139384163003962e-01 8.156486084917724533e-01
8.573988637074910768e-01 8.715145671024431273e-01 8.103003320898716222e-01
8.518212329614019973e-01 8.682324447773152043e-01 8.049797193798620132e-01
8.462299834685677036e-01 8.649674562084326279e-01 7.996873827745283325e-01
8.406246724049463159e-01 8.617194786977677712e-01 7.944239257301034529e-01
8.350048677827189847e-01 8.584883831868801440e-01 7.891899443847590900e-01
8.293671160911614271e-01 8.552754073408689317e-01 7.839836146965878383e-01
8.237138279483190439e-01 8.520791051833870311e-01 7.788079287668441264e-01
8.180448941186928558e-01 8.488991950022984900e-01 7.736637090147417961e-01
8.123599093553812711e-01 8.457355276105180675e-01 7.685515296147656938e-01
8.066584706735069332e-01 8.425879497641356464e-01 7.634719632765348818e-01
8.009401766621723207e-01 8.394563042330350777e-01 7.584255820860429376e-01
7.952028972601549173e-01 8.363411716110775718e-01 7.534118252377455249e-01
7.894471633387081244e-01 8.332419707110174656e-01 7.484319716624078245e-01
7.836734962813408645e-01 8.301581353160575327e-01 7.434872090029538416e-01
7.778815003305534770e-01 8.270894921786368092e-01 7.385781042557458820e-01
7.720707796269933310e-01 8.240358639329949941e-01 7.337052258319404219e-01
7.662409381414175824e-01 8.209970689783633313e-01 7.288691437189614986e-01
7.603915796412942241e-01 8.179729213555224643e-01 7.240704295566047222e-01
7.545223076817887398e-01 8.149632306197714948e-01 7.193096566367255251e-01
7.486327256130643759e-01 8.119678017125729896e-01 7.145873998348200029e-01
7.427219014782563411e-01 8.089866453573647531e-01 7.099039891607071828e-01
7.367891737192681090e-01 8.060196536672460388e-01 7.052599184573171698e-01
7.308350860228653989e-01 8.030662448375227580e-01 7.006562091939169123e-01
7.248592428796667431e-01 8.001262050844080154e-01 6.960934321026465144e-01
7.188612485448663270e-01 7.971993154607720511e-01 6.915721584188375681e-01
7.128407070658809852e-01 7.942853517324697243e-01 6.870929595632513376e-01
7.067972223051004477e-01 7.913840842570615264e-01 6.826564067967675342e-01
7.007303979576740005e-01 7.884952778644843674e-01 6.782630708517513041e-01
6.946398375648411561e-01 7.856186917391158042e-01 6.739135215439332471e-01
6.885251445237092760e-01 7.827540793025653532e-01 6.696083273682165160e-01
6.823859220949484161e-01 7.799011880964143995e-01 6.653480550814426797e-01
6.762217734101813038e-01 7.770597596640953508e-01 6.611332692747456941e-01
6.700323014813579503e-01 7.742295294309987641e-01 6.569645319377305226e-01
6.638171092147175933e-01 7.714102265818907345e-01 6.528424020163275943e-01
6.575757994324034073e-01 7.686015739346664377e-01 6.487674349657708284e-01
6.513079749051275957e-01 7.658032878094622742e-01 6.447401822997557153e-01
6.450132383997043695e-01 7.630150778921287458e-01 6.407611911364364810e-01
6.386911927456534466e-01 7.602366470910791874e-01 6.368310037415121361e-01
6.323414409254812796e-01 7.574676913865331374e-01 6.329501570682624090e-01
6.259635861936422296e-01 7.547078996712067722e-01 6.291191822939783407e-01
6.195560972423146406e-01 7.519573115237726535e-01 6.253384926911926822e-01
6.131191206125891080e-01 7.492154188267443615e-01 6.216087050224498034e-01
6.066526905401021796e-01 7.464817491552263595e-01 6.179303860708607044e-01
6.001564124519274124e-01 7.437559595200373685e-01 6.143040456187923715e-01
5.936298933201957784e-01 7.410376988478269977e-01 6.107301851240671819e-01
5.870727421002424062e-01 7.383266077479672118e-01 6.072092971993915400e-01
5.804845702488933279e-01 7.356223182652517067e-01 6.037418650719156288e-01
5.738649923321349489e-01 7.329244536181049874e-01 6.003283620189617809e-01
5.672131712764029166e-01 7.302327570205698892e-01 5.969692609600730782e-01
5.605265661533928023e-01 7.275474322700724583e-01 5.936651120515976654e-01
5.538072551957783363e-01 7.248673679611095100e-01 5.904163410524770894e-01
5.470548736207613283e-01 7.221921468817737999e-01 5.872233804977959881e-01
5.402690647877386176e-01 7.195213413684249382e-01 5.840866494502513495e-01
5.334494816269396145e-01 7.168545129510710545e-01 5.810065526719623286e-01
5.265932673521412921e-01 7.141918646847531527e-01 5.779837442967735717e-01
5.197010257328686933e-01 7.115326679134365007e-01 5.750185876848199484e-01
5.127738759690677606e-01 7.088760718122417703e-01 5.721113127309581659e-01
5.058115298480653221e-01 7.062215879970826782e-01 5.692622613027222833e-01
4.988124547076915882e-01 7.035690215604037956e-01 5.664719533702552434e-01
4.917741270603506742e-01 7.009183842407572529e-01 5.637411222277634026e-01
4.846997162615552246e-01 6.982683108816961637e-01 5.610695620964348818e-01
4.775890398856753039e-01 6.956182556045346077e-01 5.584575291026864230e-01
4.704389351347622594e-01 6.929683367461523247e-01 5.559058726658481220e-01
4.632498634530364812e-01 6.903178139102768007e-01 5.534147775766539157e-01
4.560240745098563253e-01 6.876655632831355502e-01 5.509839689679422170e-01
4.487598229028739172e-01 6.850113432670585922e-01 5.486140038956078824e-01
4.414547339722453279e-01 6.823550079810395408e-01 5.463056567639269501e-01
4.341129901223799714e-01 6.796950319734580415e-01 5.440580596655568701e-01
4.267325988888523436e-01 6.770311866202418649e-01 5.418718349209995511e-01
4.193119538935562440e-01 6.743631277083362852e-01 5.397475699351562684e-01
4.118554481683104340e-01 6.716893377862352965e-01 5.376841277217632165e-01
4.043593499172888350e-01 6.690098785171258999e-01 5.356826761355167887e-01
3.968261890078971788e-01 6.663236061846555813e-01 5.337425455110953454e-01
3.892580517875900981e-01 6.636294950347043642e-01 5.318631054022345817e-01
3.816510562742471135e-01 6.609275747235534570e-01 5.300456830101799577e-01
3.740119682194762429e-01 6.582160193662682790e-01 5.282880419647097980e-01
3.663368289249827603e-01 6.554948605761221625e-01 5.265915743478537525e-01
3.586308928051362699e-01 6.527625972609284455e-01 5.249544352005411918e-01
3.508936232465371674e-01 6.500187029053001719e-01 5.233768303957324619e-01
3.431277348645550562e-01 6.472621506436356809e-01 5.218577640245062321e-01
3.353355552299603914e-01 6.444920106677175520e-01 5.203963563323180663e-01
3.275188103490194735e-01 6.417074699067215615e-01 5.189919719540163623e-01
3.196814435511296515e-01 6.389074409416231060e-01 5.176430585409529384e-01
3.118254758119217707e-01 6.360911463304157465e-01 5.163488722475468862e-01
3.039555214161791530e-01 6.332575133378499643e-01 5.151075596956975478e-01
2.960748324121407205e-01 6.304057046767445049e-01 5.139178767894679867e-01
2.881881143962810587e-01 6.275347641630057982e-01 5.127779478270290126e-01
2.803003746086680792e-01 6.246437759477415641e-01 5.116857725517640620e-01
2.724162323640093031e-01 6.217319406932720893e-01 5.106395912364674050e-01
2.645425291505308918e-01 6.187983086880044503e-01 5.096365792929340444e-01
2.566840197606024554e-01 6.158422311486778655e-01 5.086750547256451149e-01
2.488489441844971561e-01 6.128628192186393875e-01 5.077515718650474907e-01
2.410430130087069522e-01 6.098595200479223211e-01 5.068641518721540562e-01
2.332749143253212143e-01 6.068316286127126702e-01 5.060092633925864503e-01
2.255521696872271886e-01 6.037786523436265984e-01 5.051841613954050070e-01
2.178835011739876371e-01 6.007001216855215597e-01 5.043855354959408954e-01
2.102776619158149840e-01 5.975956842983582984e-01 5.036102222376415138e-01
2.027443124327554802e-01 5.944650539898058694e-01 5.028546409645925364e-01
1.952924636272905801e-01 5.913080959200207598e-01 5.021159025525618880e-01
1.879334378838529440e-01 5.881246912104457492e-01 5.013896313522919757e-01
1.806761031013389140e-01 5.849149331180006905e-01 5.006736479698321585e-01
1.735328783097749850e-01 5.816789154432452369e-01 4.999631315071642601e-01
1.665133003434038084e-01 5.784169089899582339e-01 4.992561455145328453e-01
1.596300036906863895e-01 5.751292412901443107e-01 4.985481743532478860e-01
1.528937099908951325e-01 5.718163393279495077e-01 4.978371217540351057e-01
1.463171276735238113e-01 5.684787369209851615e-01 4.971190242696297834e-01
1.399119531735926458e-01 5.651170059000043544e-01 4.963918396953513335e-01
1.336912106530599997e-01 5.617318570854707982e-01 4.956519172255148820e-01
1.276672732366210816e-01 5.583239554235875923e-01 4.948978636889357907e-01
1.218535592637135234e-01 5.548942058020517321e-01 4.941256571865651481e-01
1.162631754482068569e-01 5.514432364311973034e-01 4.933355557142476422e-01
1.109091719419828259e-01 5.479722539284174188e-01 4.925220286879308795e-01
1.058058868411170528e-01 5.444817357603629615e-01 4.916873206428047927e-01
1.009651127375016111e-01 5.409730397620680087e-01 4.908260102866368046e-01
9.640117372078826907e-02 5.374467824904106683e-01 4.899392661237967350e-01
9.212706115366336990e-02 5.339038791645212001e-01 4.890257976801760109e-01
8.815110303261089464e-02 5.303457163263245455e-01 4.880817131293228583e-01
8.448832972524200624e-02 5.267726392289812098e-01 4.871097239159773440e-01
8.114572394888117102e-02 5.231858403590214923e-01 4.861073630469811557e-01
7.812721320542403980e-02 5.195865216476734938e-01 4.850725749743695636e-01
7.544312997722565917e-02 5.159750173542567708e-01 4.840076729265158639e-01
7.309574065371191032e-02 5.123522156200341904e-01 4.829119931483520367e-01
7.107372065756567547e-02 5.087198306495862576e-01 4.817814896929000779e-01
6.938287907938911481e-02 5.050778255979795350e-01 4.806197953737798012e-01
6.801610562745827315e-02 5.014269919387287500e-01 4.794267081520149909e-01
6.696293828126623215e-02 4.977680989992983585e-01 4.782021118153178540e-01
6.619782080433814220e-02 4.941027658684106760e-01 4.769429100625798834e-01
6.571558023976076246e-02 4.904308747270616498e-01 4.756523204648347991e-01
6.549820807888259711e-02 4.867530881918184504e-01 4.743305399824718216e-01
6.552566064603559948e-02 4.830700650743400826e-01 4.729777438223161101e-01
6.577673858968982601e-02 4.793824370720338179e-01 4.715941633832017588e-01
6.622677049492370349e-02 4.756910136151853430e-01 4.701794827815656830e-01
6.685019795786503738e-02 4.719966397538147285e-01 4.687333314519904204e-01
6.763272639280798471e-02 4.682993582933911436e-01 4.672575858662252890e-01
6.855374817251533304e-02 4.645996843636931994e-01 4.657526519729214276e-01
6.959365457471067273e-02 4.608981063279830592e-01 4.642189680611767399e-01
7.073403782879061907e-02 4.571950861446216208e-01 4.626570010388615928e-01
7.195781915786156335e-02 4.534910598140922677e-01 4.610672429543411499e-01
7.324931366840245484e-02 4.497864378980456768e-01 4.594502077591564038e-01
7.459424408415230023e-02 4.460816060979165276e-01 4.578064283072901808e-01
7.597971511267428979e-02 4.423769258816079852e-01 4.561364535850265800e-01
7.739415915996109008e-02 4.386727351476746306e-01 4.544408461640829233e-01
7.882726258105521300e-02 4.349693489173894201e-01 4.527201798696426915e-01
8.026987997778653461e-02 4.312670600459782566e-01 4.509750376540915817e-01
8.171394242970148047e-02 4.275661399451892164e-01 4.492060096666721791e-01
8.315236408743081897e-02 4.238668393102046350e-01 4.474136915088433031e-01
8.457895032121595658e-02 4.201693888446985103e-01 4.455986826648858368e-01
8.598830961259482097e-02 4.164739999785817548e-01 4.437615850971945997e-01
8.737577058920215078e-02 4.127808655736769361e-01 4.419030019956885491e-01
8.873730500447504776e-02 4.090901606132017476e-01 4.400235366709063789e-01
9.006945702453716951e-02 4.054020428715323643e-01 4.381237915805220595e-01
9.136927887187012987e-02 4.017166535612498035e-01 4.362043674792987491e-01
9.263427266200571775e-02 3.980341179549674036e-01 4.342658626828024837e-01
9.386233813144639893e-02 3.943545459798681874e-01 4.323088724355508838e-01
9.505172587236093706e-02 3.906780327832828914e-01 4.303339883746660766e-01
9.620003933222870396e-02 3.870047637840662302e-01 4.283416262899890081e-01
9.729997123759509536e-02 3.833354963232004087e-01 4.263312615677605777e-01
9.835741884413440328e-02 3.796695548525758634e-01 4.243046936775203282e-01
9.937169919107982641e-02 3.760069784468507703e-01 4.222625006231348066e-01
1.003423119365809690e-01 3.723477936835040136e-01 4.202052548968338574e-01
1.012689167529539358e-01 3.686920150348446112e-01 4.181335233490640069e-01
1.021513134274243950e-01 3.650396452261166491e-01 4.160478671051655586e-01
1.029884320307208612e-01 3.613907991323849767e-01 4.139486582680417803e-01
1.037719209811681642e-01 3.577465259466254266e-01 4.118348877541359032e-01
1.045112201334900126e-01 3.541056562885774861e-01 4.097088092956981398e-01
1.052065591856250482e-01 3.504681464569319171e-01 4.075709632165070984e-01
1.058582376509708545e-01 3.468339423292918222e-01 4.054218838433404359e-01
1.064666158592602885e-01 3.432029795356145718e-01 4.032620996037361571e-01
1.070239159802940931e-01 3.395763147211236510e-01 4.010905648833596460e-01
1.075355258379605550e-01 3.359532282299735328e-01 3.989087127976621017e-01
1.080052802278752000e-01 3.323331728550899533e-01 3.967176863054378000e-01
1.084337294201929702e-01 3.287160421828903556e-01 3.945179917556369542e-01
1.088202096165073185e-01 3.251019032868823211e-01 3.923098843692563453e-01
1.091556968975302966e-01 3.214920812428978536e-01 3.900919340676701208e-01
1.094518269158459012e-01 3.178848431435130073e-01 3.878667847148202785e-01
1.097092559883022234e-01 3.142800415550652815e-01 3.856349205402823110e-01
1.099286556573810802e-01 3.106775192369479188e-01 3.833968205472055302e-01
1.100986315611906241e-01 3.070790351009552999e-01 3.811504545068737926e-01
1.102317288825286901e-01 3.034825829921489193e-01 3.788986922133656954e-01
1.103290785873510815e-01 2.998879083652261635e-01 3.766420819050353419e-01
1.103893448894039397e-01 2.962951553198432397e-01 3.743806401368951486e-01
1.104047114339987423e-01 2.927055780381122019e-01 3.721129484539063559e-01
1.103867349279119559e-01 2.891171759115234718e-01 3.698417464033080804e-01
1.103361572207214036e-01 2.855297163446686715e-01 3.675674754730824945e-01
1.102460152182262176e-01 2.819443225282238785e-01 3.652888319764409086e-01
1.101198294511886444e-01 2.783603021914852760e-01 3.630068099573608986e-01
1.099635342633853707e-01 2.747764822051754763e-01 3.607229892872333421e-01
1.097748910210358808e-01 2.711931376089152246e-01 3.584370856671010852e-01
1.095478763802813504e-01 2.676112773022403801e-01 3.561478615212890775e-01
1.092932808921200649e-01 2.640287615628073015e-01 3.538580573547516761e-01
1.090116287119651528e-01 2.604453159192556266e-01 3.515680214913912693e-01
1.086913329090623825e-01 2.568630402753477870e-01 3.492750549670107785e-01
1.083460004297124302e-01 2.532791299120676354e-01 3.469826761312421182e-01
1.079763885418836000e-01 2.496932209057829977e-01 3.446912766824791197e-01
1.075713140889580088e-01 2.461073946949010050e-01 3.423980926413837667e-01
1.071428862568302165e-01 2.425189920877232619e-01 3.401063918909076889e-01
1.066927192498384747e-01 2.389274243968986799e-01 3.378167764830257158e-01
1.062093640276061124e-01 2.353348932541681759e-01 3.355262248019345583e-01
1.057055066316897329e-01 2.317384538919968207e-01 3.332383062494437276e-01
1.051814556007013568e-01 2.281377376539149293e-01 3.309532535866355762e-01
1.046272664909374817e-01 2.245346804307608024e-01 3.286682704446855507e-01
1.040556247342821483e-01 2.209261472703716311e-01 3.263871052200472134e-01
1.034637486817424901e-01 2.173124203806320875e-01 3.241090419310820314e-01
1.028468845169810686e-01 2.136942776676366007e-01 3.218326666201724029e-01
1.022150454689789434e-01 2.100689913462399083e-01 3.195611195376787395e-01
1.015620277820400430e-01 2.064376400065636719e-01 3.172925125709148420e-01
1.008900241588315538e-01 2.027992883367119026e-01 3.150275553737127421e-01
1.002054858430543316e-01 1.991518634625495388e-01 3.127684262427183892e-01
9.949803783218733044e-02 1.954975058870322413e-01 3.105116763778897337e-01
9.877832247043097369e-02 1.918329889556127654e-01 3.082609202522414993e-01
9.804493118338306057e-02 1.881580898967480098e-01 3.060157402408537064e-01
9.729328810023782359e-02 1.844734436312414905e-01 3.037745103921251633e-01
9.653316478613682694e-02 1.807757562460599599e-01 3.015407883893915231e-01
9.575619444438937533e-02 1.770666290075273708e-01 2.993115284087380368e-01
9.496899572502048859e-02 1.733434725855039771e-01 2.970892156823352059e-01
9.417284157369981701e-02 1.696050767223750977e-01 2.948744162588470830e-01
9.336173420340779239e-02 1.658523274558137695e-01 2.926648040593683997e-01
9.254607948203208423e-02 1.620811665705463311e-01 2.904645825457518593e-01
9.171714479257989105e-02 1.582931942356169963e-01 2.882702798507874586e-01
9.088231952195491292e-02 1.544850037627045203e-01 2.860850125083750362e-01
9.004099984169053328e-02 1.506555099870153513e-01 2.839086654207542693e-01
8.919012906146844832e-02 1.468043594975814992e-01 2.817400195447572475e-01
8.833858505105957049e-02 1.429270910011002649e-01 2.795831537842536352e-01
8.747533041314431435e-02 1.390258052165279645e-01 2.774332852121961235e-01
8.661249189260347703e-02 1.350944971238551839e-01 2.752961432065319514e-01
8.574385298640457842e-02 1.311333174761502018e-01 2.731691209373900975e-01
8.487294239495335457e-02 1.271387529060027943e-01 2.710541708402016137e-01
8.400180885962132971e-02 1.231074880892656653e-01 2.689526699064171411e-01
8.312616532498406929e-02 1.190383729463048712e-01 2.668628892216621806e-01
8.225559928700268419e-02 1.149244079727295142e-01 2.647901677800857390e-01]; 

   case 'ice'
      RGB = [1.531167435543729846e-02 2.252059388699531942e-02 7.272873735907764425e-02
1.800549591959003243e-02 2.544551608389769570e-02 7.841879116825511975e-02
2.090133006203173313e-02 2.852652245071044673e-02 8.407771577420969367e-02
2.399818650587986005e-02 3.176264327218273481e-02 8.970750416351327972e-02
2.729775749004114890e-02 3.514909496851363613e-02 9.532647230499821656e-02
3.080428146052393429e-02 3.867708164670261017e-02 1.009662673440025749e-01
3.450984373415445089e-02 4.229955636714784889e-02 1.065828165743578915e-01
3.841359784116892689e-02 4.587404239998996852e-02 1.121776340643418912e-01
4.245789616397358662e-02 4.939963679832150983e-02 1.177960773094180458e-01
4.645777532344794875e-02 5.288489849108807955e-02 1.234176036286754041e-01
5.041621762087678676e-02 5.633788824565918313e-02 1.290216830927681801e-01
5.433779516873311205e-02 5.975737503280462853e-02 1.346239246672657763e-01
5.823000255128838593e-02 6.313585840823013329e-02 1.402651739202840087e-01
6.208647351275981691e-02 6.648865889594307577e-02 1.458921724026617794e-01
6.590890827576056932e-02 6.981759532833076154e-02 1.515056185497818952e-01
6.970669712558172360e-02 7.310909804078333241e-02 1.571701055775910905e-01
7.347360805650121618e-02 7.637815581467355397e-02 1.628308265129313204e-01
7.720982185882321880e-02 7.962820406611839652e-02 1.684800394051645944e-01
8.092190572547824923e-02 8.284856571837034833e-02 1.741697304708636207e-01
8.460722520622676601e-02 8.604719820880801784e-02 1.798725773914293391e-01
8.826407983557263415e-02 8.923087230589588081e-02 1.855652548242135713e-01
9.189748856643911723e-02 9.238991627825218766e-02 1.912953251540190358e-01
9.550605637353798416e-02 9.552936548231369396e-02 1.970466344723810770e-01
9.908760199831237458e-02 9.865735598928837558e-02 2.027885080271309981e-01
1.026464276979975609e-01 1.017632832609889626e-01 2.085729006866037794e-01
1.061807481293115807e-01 1.048533987828409453e-01 2.143771619600378098e-01
1.096889023760894977e-01 1.079351702016392300e-01 2.201721871249775475e-01
1.131748168296289048e-01 1.109951338044804781e-01 2.260236443434878173e-01
1.166355654717963208e-01 1.140448383112346031e-01 2.318836318580280165e-01
1.200706770784045196e-01 1.170883693387409774e-01 2.377372345137335197e-01
1.234831126917021182e-01 1.201095785266100835e-01 2.436640854600494177e-01
1.268694769189046001e-01 1.231271973707054601e-01 2.495809234356071160e-01
1.302307935855599175e-01 1.261348273230401829e-01 2.555217697948003464e-01
1.335669536543798996e-01 1.291293381943951213e-01 2.615060473838256017e-01
1.368768735891793542e-01 1.321228390190112012e-01 2.674792268238664894e-01
1.401607222303006828e-01 1.351007751267265800e-01 2.735164638309879881e-01
1.434177197831119077e-01 1.380764808543884503e-01 2.795564348972031099e-01
1.466474902267766722e-01 1.410468359199091026e-01 2.856183790033576808e-01
1.498489657362017669e-01 1.440082826282148287e-01 2.917246687573430419e-01
1.530229361070791771e-01 1.469729643095797900e-01 2.978170527656492372e-01
1.561658891044421793e-01 1.499237839873007738e-01 3.039884751881400948e-01
1.592803696822101267e-01 1.528796824181726244e-01 3.101441051875748478e-01
1.623627513097627983e-01 1.558287612260276789e-01 3.163512211275710251e-01
1.654142439360292427e-01 1.587796212856360245e-01 3.225680108254420086e-01
1.684327965090609003e-01 1.617293084063355368e-01 3.288148930781844004e-01
1.714174532656698446e-01 1.646790560305917694e-01 3.350897686326997915e-01
1.743682562241114509e-01 1.676319105902342177e-01 3.413798820958920399e-01
1.772818953778146633e-01 1.705846382439054620e-01 3.477088809140311265e-01
1.801603634233710782e-01 1.735435075165488450e-01 3.540447583078508709e-01
1.829984949769703495e-01 1.765034834295459432e-01 3.604229741636628126e-01
1.857993799242090571e-01 1.794715454820642320e-01 3.668060399217007994e-01
1.885572370561371114e-01 1.824432159549585208e-01 3.732275719698777694e-01
1.912746410827655397e-01 1.854240146696699842e-01 3.796579628109393312e-01
1.939472187446417972e-01 1.884119984733164943e-01 3.861158710011389772e-01
1.965746218202921169e-01 1.914094687319768118e-01 3.925922579541862301e-01
1.991567237202455654e-01 1.944185425759841768e-01 3.990785276243757895e-01
2.016870280617926170e-01 1.974370231299186207e-01 4.055979452145534458e-01
2.041733231959791395e-01 2.004720971993999568e-01 4.121034638774260794e-01
2.065989175964332847e-01 2.035163285945146838e-01 4.186611530978023854e-01
2.089769981124778853e-01 2.065792251897101139e-01 4.252032229400294350e-01
2.112968541690966595e-01 2.096575161864902004e-01 4.317649751357993670e-01
2.135580062034990179e-01 2.127534028876823524e-01 4.383374070580626225e-01
2.157654857120719361e-01 2.158705959834085197e-01 4.448946358553702574e-01
2.179027467031688925e-01 2.190056896370822792e-01 4.514821666358075913e-01
2.199801557481101399e-01 2.221640817576028826e-01 4.580565150394946827e-01
2.219970327313536274e-01 2.253470663668500351e-01 4.646129330923238210e-01
2.239359400161920477e-01 2.285528452407099564e-01 4.711900504070519191e-01
2.258093253298022463e-01 2.317859657097744996e-01 4.777445723729306093e-01
2.276150386234590539e-01 2.350474802331274371e-01 4.842749179389621017e-01
2.293411129842480300e-01 2.383375691596468227e-01 4.908001225098672093e-01
2.309905102772738528e-01 2.416584626823640725e-01 4.973037406011600603e-01
2.325653899945122616e-01 2.450118207268658921e-01 5.037733153263704855e-01
2.340634217066668299e-01 2.483988011172711396e-01 5.102061969115925244e-01
2.354739391855478203e-01 2.518205609488499142e-01 5.166133649735229483e-01
2.368010268106580107e-01 2.552786891057904350e-01 5.229787612622226467e-01
2.380459056014049835e-01 2.587743537954566575e-01 5.292929958163873350e-01
2.392069166726323859e-01 2.623086175634496420e-01 5.355517448643010159e-01
2.402826487140223288e-01 2.658824888638082751e-01 5.417502762699162311e-01
2.412690379036540600e-01 2.694971451746652202e-01 5.478870668064361737e-01
2.421646534827242569e-01 2.731536075359688454e-01 5.539567283353189486e-01
2.429737663975502504e-01 2.768521950965758815e-01 5.599478852923390759e-01
2.436964271327693443e-01 2.805934002738960653e-01 5.658549936523274981e-01
2.443331406488099544e-01 2.843775500649446952e-01 5.716723986670918523e-01
2.448849019911867875e-01 2.882047861966390290e-01 5.773944132373918237e-01
2.453532261044997220e-01 2.920750479307156477e-01 5.830154006066263772e-01
2.457401707423141346e-01 2.959880581924350662e-01 5.885298594478888257e-01
2.460483514874983180e-01 2.999433136125404520e-01 5.939325092217595525e-01
2.462809480744551638e-01 3.039400789577370587e-01 5.992183735671519074e-01
2.464417014302806019e-01 3.079773862803035778e-01 6.043828594654379049e-01
2.465349011179689409e-01 3.120540389489538935e-01 6.094218299971551067e-01
2.465653631589233563e-01 3.161686205399514837e-01 6.143316686920637926e-01
2.465383985194727345e-01 3.203195083804638021e-01 6.191093337508785099e-01
2.464597728499572371e-01 3.245048913568406301e-01 6.237524007769893464e-01
2.463356583484295759e-01 3.287227914401089635e-01 6.282590930787467220e-01
2.461725788681820570e-01 3.329710882494058555e-01 6.326282990625464731e-01
2.459773495859159942e-01 3.372475458788758984e-01 6.368595767065934332e-01
2.457570126857825388e-01 3.415498411593856920e-01 6.409531455573861392e-01
2.455187705883344895e-01 3.458755925146314025e-01 6.449098671004442895e-01
2.452699182619971774e-01 3.502223885999685149e-01 6.487312147029243858e-01
2.450177761015216449e-01 3.545878159769456084e-01 6.524192345935843074e-01
2.447696247503935441e-01 3.589694851701816236e-01 6.559764995268344556e-01
2.445326430925611194e-01 3.633650545677170052e-01 6.594060568707836856e-01
2.443138504548736933e-01 3.677722517524962265e-01 6.627113728687089589e-01
2.441200538578628954e-01 3.721888919827969766e-01 6.658962747586536501e-01
2.439578009410017234e-01 3.766128936655925852e-01 6.689648923098728828e-01
2.438333389800677597e-01 3.810422907829071337e-01 6.719216001624749302e-01
2.437527166088169217e-01 3.854752048525493247e-01 6.747709927211151815e-01
2.437215868067584834e-01 3.899098965838322384e-01 6.775178104933975431e-01
2.437449691034339061e-01 3.943448306666547665e-01 6.801668228779059744e-01
2.438275942988586409e-01 3.987785785872524635e-01 6.827228592773180171e-01
2.439737834704199804e-01 4.032098481987659855e-01 6.851907482426946583e-01
2.441874434968405727e-01 4.076374812199010655e-01 6.875752826204606372e-01
2.444720671645514987e-01 4.120604493040472271e-01 6.898811893971661391e-01
2.448307373736109405e-01 4.164778489638755743e-01 6.921131040949622948e-01
2.452661349520961487e-01 4.208888956165576789e-01 6.942755494834951246e-01
2.457805495927352646e-01 4.252929169900244166e-01 6.963729183108620102e-01
2.463758934401020784e-01 4.296893461032649797e-01 6.984094597147249006e-01
2.470537168787759197e-01 4.340777140051458871e-01 7.003892689515083259e-01
2.478152261001809742e-01 4.384576424279668649e-01 7.023162800736911793e-01
2.486613020564674703e-01 4.428288364849563008e-01 7.041942611893784454e-01
2.495925204424470634e-01 4.471910775158167151e-01 7.060268119517866259e-01
2.506091723801374682e-01 4.515442161617032046e-01 7.078173629464002969e-01
2.517112855138260996e-01 4.558881657309074575e-01 7.095691766682096224e-01
2.528986452564736531e-01 4.602228958990267071e-01 7.112853498087290394e-01
2.541708159598920491e-01 4.645484267725444316e-01 7.129688166009963135e-01
2.555271618114321464e-01 4.688648233323279846e-01 7.146223529992972168e-01
2.569668672886404326e-01 4.731721902633665988e-01 7.162485814980906751e-01
2.584889570301312500e-01 4.774706671689665227e-01 7.178499764208978728e-01
2.600923150060586719e-01 4.817604241612176152e-01 7.194288695343957762e-01
2.617757028946650633e-01 4.860416578147523370e-01 7.209874558653309728e-01
2.635377775926647237e-01 4.903145874672710236e-01 7.225277996180770046e-01
2.653771078065290112e-01 4.945794518478790480e-01 7.240518401086443179e-01
2.672921896891591875e-01 4.988365060127348261e-01 7.255613976468211490e-01
2.692814615020625024e-01 5.030860185666778950e-01 7.270581793119333947e-01
2.713433172968592322e-01 5.073282691492603247e-01 7.285437845796602918e-01
2.734761196220429347e-01 5.115635461637654258e-01 7.300197107675363561e-01
2.756782112712242161e-01 5.157921447283394523e-01 7.314873582754417569e-01
2.779479260979498267e-01 5.200143648290899145e-01 7.329480356046209621e-01
2.802835989294344965e-01 5.242305096559424227e-01 7.344029641448335255e-01
2.826835746175559994e-01 5.284408841030631132e-01 7.358532827242095786e-01
2.851462162701056124e-01 5.326457934167407871e-01 7.373000519204292447e-01
2.876699127088762631e-01 5.368455419747276691e-01 7.387442581351216786e-01
2.902530852036724895e-01 5.410404321821302709e-01 7.401868174359758079e-01
2.928941935330091062e-01 5.452307634699987693e-01 7.416285791730816701e-01
2.955917414230262996e-01 5.494168313837994866e-01 7.430703293775968721e-01
2.983442814164117829e-01 5.535989267498891975e-01 7.445127939520038707e-01
3.011504192226619470e-01 5.577773349090259236e-01 7.459566416621068452e-01
3.040088176001567444e-01 5.619523350067550105e-01 7.474024869415085703e-01
3.069181998192390681e-01 5.661241993312904341e-01 7.488508925197547850e-01
3.098773527540035766e-01 5.702931926901545490e-01 7.503023718855809099e-01
3.128851296485514188e-01 5.744595718174968502e-01 7.517573915968616127e-01
3.159404526017212111e-01 5.786235848045079289e-01 7.532163734489466522e-01
3.190423148119757579e-01 5.827854705458642703e-01 7.546796965130810886e-01
3.221897826221614136e-01 5.869454581955374506e-01 7.561476990566410317e-01
3.253819974015382255e-01 5.911037666256608869e-01 7.576206803568952264e-01
3.286181773003176154e-01 5.952606038824642676e-01 7.590989024200323065e-01
3.318976189096972118e-01 5.994161666335501293e-01 7.605825916172537227e-01
3.352196988582024639e-01 6.035706396009887786e-01 7.620719402498039585e-01
3.385838753729217276e-01 6.077241949749224714e-01 7.635671080549891743e-01
3.419896898320652912e-01 6.118769918024981047e-01 7.650682236654324786e-01
3.454367683330621941e-01 6.160291753470886755e-01 7.665753860341149029e-01
3.489248232981658759e-01 6.201808764128988738e-01 7.680886658381395060e-01
3.524536551372777216e-01 6.243322106301375518e-01 7.696081068746097875e-01
3.560231539852872773e-01 6.284832776960781464e-01 7.711337274625911231e-01
3.596333015286877766e-01 6.326341605674558055e-01 7.726655218657797475e-01
3.632841729334435055e-01 6.367849245998262742e-01 7.742034617512983941e-01
3.669759388831845826e-01 6.409356166297031088e-01 7.757474977008770312e-01
3.707088677332940896e-01 6.450862639955934341e-01 7.772975607916928764e-01
3.744833277827505080e-01 6.492368734944050646e-01 7.788535642652039126e-01
3.782997896612683153e-01 6.533874302701627723e-01 7.804154053034927374e-01
3.821588288242804832e-01 6.575378966326091978e-01 7.819829669338883571e-01
3.860611281425825880e-01 6.616882108040403887e-01 7.835561200839490370e-01
3.900074788490647260e-01 6.658383310313947812e-01 7.851343555476577585e-01
3.939988161696960089e-01 6.699882294160507401e-01 7.867167620565498343e-01
3.980361943268375668e-01 6.741376428249236108e-01 7.883041180552029514e-01
4.021207666563880734e-01 6.782863970632155848e-01 7.898962603978111341e-01
4.062538088695973326e-01 6.824342860499125196e-01 7.914930279577012673e-01
4.104367215482290221e-01 6.865810701317770492e-01 7.930942653099937178e-01
4.146710323688317379e-01 6.907264743609855540e-01 7.946998267750712275e-01
4.189583979456862339e-01 6.948701867544626598e-01 7.963095808534970121e-01
4.233007615496491849e-01 6.990119462600449252e-01 7.979223720379219342e-01
4.277003131396773239e-01 7.031514402219964932e-01 7.995369318160842065e-01
4.321588772737992579e-01 7.072880516513097016e-01 8.011551464379941256e-01
4.366786594847170133e-01 7.114212963902242226e-01 8.027769966740599950e-01
4.412619934972313862e-01 7.155506445548852623e-01 8.044025095123854552e-01
4.459117519235324401e-01 7.196756076604832186e-01 8.060302209280784114e-01
4.506310977664646500e-01 7.237956107627850910e-01 8.076586979980320269e-01
4.554220533607292176e-01 7.279097909870729799e-01 8.092910180447286939e-01
4.602874296132978826e-01 7.320174174690982083e-01 8.109274975540664565e-01
4.652312929955615961e-01 7.361178015074247849e-01 8.125654680881110314e-01
4.702567132124945704e-01 7.402100378643735601e-01 8.142058631482380626e-01
4.753658389170350440e-01 7.442931579112735951e-01 8.158518248191980460e-01
4.805629989896973986e-01 7.483662565565446512e-01 8.175014285494428545e-01
4.858519412014848382e-01 7.524283052190463561e-01 8.191547180510726500e-01
4.912340123610521858e-01 7.564782715855419282e-01 8.208164230788980165e-01
4.967149567998503934e-01 7.605150334733405959e-01 8.224828836016547795e-01
5.022958597070651399e-01 7.645375459922270078e-01 8.241590313627700226e-01
5.079798024989584659e-01 7.685447202464326111e-01 8.258458806918314021e-01
5.137703056538149848e-01 7.725354056237694333e-01 8.275437918161917539e-01
5.196672668780889515e-01 7.765087356564679411e-01 8.292580362563429786e-01
5.256748803701343231e-01 7.804635126371105569e-01 8.309874864714956733e-01
5.317909415132198170e-01 7.843991695298314637e-01 8.327389435984492438e-01
5.380183963455860141e-01 7.883146743347385632e-01 8.345120959574837682e-01
5.443539980151446134e-01 7.922097498034269547e-01 8.363133084981756449e-01
5.507978944105677011e-01 7.960838061951434064e-01 8.381442735434428970e-01
5.573464847507825226e-01 7.999368256977259506e-01 8.400099169225816453e-01
5.639964685183691540e-01 8.037688733081099768e-01 8.419139099857662067e-01
5.707439504576503619e-01 8.075802108116317823e-01 8.438596871365160457e-01
5.775827793152761291e-01 8.113715803382598457e-01 8.458518352418533670e-01
5.845080419432422403e-01 8.151436572177973572e-01 8.478930504633812593e-01
5.915124079987543748e-01 8.188976525325087907e-01 8.499872730184475644e-01
5.985892343607183141e-01 8.226347979507546704e-01 8.521371651406997039e-01
6.057310635784716180e-01 8.263566000070049489e-01 8.543453938014868854e-01
6.129312041392928068e-01 8.300645419348708920e-01 8.566135671784518291e-01
6.201810118833709362e-01 8.337606033851390208e-01 8.589441737734759830e-01
6.274752226888651307e-01 8.374461999354567698e-01 8.613371235909390577e-01
6.348052546879014990e-01 8.411235111538640785e-01 8.637941988526561810e-01
6.421662450761786989e-01 8.447940336859747212e-01 8.663146822269915948e-01
6.495514829538971968e-01 8.484597208451428729e-01 8.688988750843767983e-01
6.569557263949438175e-01 8.521222755106645508e-01 8.715461474797051578e-01
6.643745794481599187e-01 8.557832593714448377e-01 8.742554091588357057e-01
6.718027324561179903e-01 8.594444859176019191e-01 8.770260761340837874e-01
6.792376399817107169e-01 8.631071655185147407e-01 8.798561771298264444e-01
6.866744996748753715e-01 8.667730629390760777e-01 8.827449654904524490e-01
6.941118238786820882e-01 8.704431765687035139e-01 8.856901351703097003e-01
7.015458707010223671e-01 8.741190620646153153e-01 8.886905723641257415e-01
7.089754732757992395e-01 8.778016574148082007e-01 8.917440383282461136e-01
7.163982330943452492e-01 8.814922113687924110e-01 8.948489665590808606e-01
7.238129051546876580e-01 8.851916892315870866e-01 8.980033539675886800e-01
7.312183003174410612e-01 8.889010427774228784e-01 9.012052612065797330e-01
7.386131643998257168e-01 8.926212432189428725e-01 9.044528466250297827e-01
7.459969493141647146e-01 8.963530775374040083e-01 9.077440285167398537e-01
7.533688986576650981e-01 9.000973872698678768e-01 9.110768604175860652e-01
7.607282142588290830e-01 9.038550280130126513e-01 9.144494382489702922e-01
7.680754104857114850e-01 9.076264985347726189e-01 9.178593407704264129e-01
7.754086380463782735e-01 9.114129489376284754e-01 9.213050832886523489e-01
7.827298656902449414e-01 9.152144780056820084e-01 9.247836307542580681e-01
7.900362445563153813e-01 9.190325332798994218e-01 9.282937907772289554e-01
7.973305798287271262e-01 9.228669845876559252e-01 9.318320585938411060e-01
8.046093603619434154e-01 9.267195025159972177e-01 9.353972761089011101e-01
8.118762521869845594e-01 9.305897288106829146e-01 9.389853083637165199e-01
8.191262403047088192e-01 9.344798221317863751e-01 9.425952041745783161e-01
8.263649511242520118e-01 9.383888950299160703e-01 9.462215400179357916e-01
8.335857899787124659e-01 9.423196480542496145e-01 9.498633823775902707e-01
8.407925617377357552e-01 9.462718000641171523e-01 9.535152044674145566e-01
8.479820749115867251e-01 9.502471776949116267e-01 9.571737430642662803e-01
8.551509499643907830e-01 9.542477482079624318e-01 9.608352118692636834e-01
8.623025919332756306e-01 9.582735696851335527e-01 9.644920284238761576e-01
8.694298092423108359e-01 9.623279528450340292e-01 9.681401921277593692e-01
8.765283703934805271e-01 9.664134798374283131e-01 9.717733036205912223e-01
8.835932645286533882e-01 9.705331352089504593e-01 9.753839343072715495e-01
8.906174074108087479e-01 9.746907136098538205e-01 9.789640951138071090e-01
8.975917485137434593e-01 9.788908041054119602e-01 9.825051528035059212e-01
9.045013427138774986e-01 9.831399005223971921e-01 9.860005773650694083e-01
9.113300542301955298e-01 9.874449459956177177e-01 9.894442642623689776e-01
9.180592960081255249e-01 9.918135358838490179e-01 9.928328638314803944e-01]; 

   case 'rai' % rain
      RGB = [9.345899218079473103e-01 9.308468535401923649e-01 9.527121598234155053e-01
9.317427384147130009e-01 9.265559169859566291e-01 9.461412847112190549e-01
9.288323151212234396e-01 9.222903185306413620e-01 9.397258851930316848e-01
9.259014656554976908e-01 9.180422099085187027e-01 9.333811647328228434e-01
9.229490910874459386e-01 9.138118271011793636e-01 9.271087656276950639e-01
9.199777474389575493e-01 9.095987388769878335e-01 9.209029155030628022e-01
9.169834318717224875e-01 9.054037550731951489e-01 9.147707495523473842e-01
9.139687227512254264e-01 9.012264798371935060e-01 9.087060404880972220e-01
9.109285940266572679e-01 8.970679577253415360e-01 9.027176878689184836e-01
9.078584183128727281e-01 8.929291295619988800e-01 8.968142426813552337e-01
9.047531063535361184e-01 8.888107271516731966e-01 8.910081776220351024e-01
9.016111241776750829e-01 8.847109979588536621e-01 8.853222084733923802e-01
8.984773600020180551e-01 8.806072441106347348e-01 8.798005863890318023e-01
8.957156578807984326e-01 8.763733055795923654e-01 8.742439953591307766e-01
8.934429895784954390e-01 8.720376849990935098e-01 8.679132827136474271e-01
8.911862289430210193e-01 8.677553071034195264e-01 8.611451293770118198e-01
8.889359451831089221e-01 8.635014856893418189e-01 8.542095914675652546e-01
8.867116938006094351e-01 8.592609509355686459e-01 8.471741066544885568e-01
8.845266087736632921e-01 8.550269405833543779e-01 8.400488385945689140e-01
8.823737459191345334e-01 8.507991274319443020e-01 8.328611900775441113e-01
8.802551767793974635e-01 8.465759973509723313e-01 8.256144073735601774e-01
8.781723156295014876e-01 8.423565112404021171e-01 8.183096182593350143e-01
8.761185057025142608e-01 8.381415609718929627e-01 8.109618862038079357e-01
8.740930473614691998e-01 8.339308508975332712e-01 8.035739688319240015e-01
8.720952068860511330e-01 8.297241543120810192e-01 7.961481162736230299e-01
8.701280658536796331e-01 8.255204305809771270e-01 7.886787238448926818e-01
8.681860361945246130e-01 8.213205960844648379e-01 7.811769076946536439e-01
8.662688819092574377e-01 8.171244030607883735e-01 7.736431720476375506e-01
8.643746025935312716e-01 8.129320197586197283e-01 7.660813810727764572e-01
8.625034879855904002e-01 8.087431068747142904e-01 7.584907829855329631e-01
8.606546147206539654e-01 8.045576097831967921e-01 7.508729509413730741e-01
8.588266729702948021e-01 8.003755742930238615e-01 7.432301838409289818e-01
8.569623869913148839e-01 7.962191366375533930e-01 7.355859431728856146e-01
8.550504938518290743e-01 7.920970771858082404e-01 7.278998746292554278e-01
8.530849581648412006e-01 7.880123772917803082e-01 7.201683837184543746e-01
8.510553489556228479e-01 7.839691860217244956e-01 7.123959661845170599e-01
8.489330228346508855e-01 7.799787143731991002e-01 7.045972788504087925e-01
8.467217087867040526e-01 7.760402896946996254e-01 6.967642500138357953e-01
8.444094675146763818e-01 7.721590919106836592e-01 6.888992560450724056e-01
8.419826946126204303e-01 7.683408791881058963e-01 6.810067521686843373e-01
8.393858369850137890e-01 7.646065720377731578e-01 6.731281946153443441e-01
8.366346261960659891e-01 7.609512703952114876e-01 6.652451363576846743e-01
8.336844747046829873e-01 7.573915193569474846e-01 6.573955009267002936e-01
8.305100496696308232e-01 7.539367518776275423e-01 6.496025429014220531e-01
8.270795595522559829e-01 7.505979386186598656e-01 6.419039537235261550e-01
8.233795339766563082e-01 7.473786068829111340e-01 6.343290132703457429e-01
8.193527781515338448e-01 7.442973494709732574e-01 6.269543660928612594e-01
8.150187381418861898e-01 7.413436869430644061e-01 6.197992186956774452e-01
8.103671933740643762e-01 7.385171999523609809e-01 6.129154791955783166e-01
8.053792270659873020e-01 7.358201042585071905e-01 6.063662921639977332e-01
8.001091487009975856e-01 7.332281055474541009e-01 6.001460682932164836e-01
7.945789023747689139e-01 7.307291829728559396e-01 5.942715305031909256e-01
7.888006960769626819e-01 7.283155171589156263e-01 5.887591680450797726e-01
7.828500773466386953e-01 7.259575374530320424e-01 5.835578074632412626e-01
7.767061505166655833e-01 7.236609437130246958e-01 5.786949863565322705e-01
7.704515166608618681e-01 7.213952841209190225e-01 5.740985082128049477e-01
7.640735967706561160e-01 7.191644341873529855e-01 5.697767469661422224e-01
7.576101573082332230e-01 7.169548408227417458e-01 5.656900597267850994e-01
7.510868163729491620e-01 7.147576249864890929e-01 5.618076267722970085e-01
7.445073291928260284e-01 7.125716412063957117e-01 5.581177337866775057e-01
7.378726341804960898e-01 7.103967525656292858e-01 5.546105609434751615e-01
7.312079622811540336e-01 7.082247740235839695e-01 5.512528292120718598e-01
7.245223864166319139e-01 7.060527915800670629e-01 5.480303444985812344e-01
7.178179126910454455e-01 7.038807344069457628e-01 5.449299707841194218e-01
7.111005957174848513e-01 7.017068393561574080e-01 5.419396078954504814e-01
7.043748082863551252e-01 6.995299965106114293e-01 5.390480504976655762e-01
6.976428841540770476e-01 6.973498542230505137e-01 5.362451349156007741e-01
6.909082389019516324e-01 6.951656049847189101e-01 5.335213672589386169e-01
6.841729694681116802e-01 6.929769158960692454e-01 5.308681129719833303e-01
6.774389689374403778e-01 6.907834967201277321e-01 5.282774232028493167e-01
6.707084440130518521e-01 6.885849153944831880e-01 5.257418638258782861e-01
6.639830918601468124e-01 6.863809064730829190e-01 5.232546086946955333e-01
6.572628938239016838e-01 6.841717926007404582e-01 5.208097057295569821e-01
6.505310175956583452e-01 6.819627798840676158e-01 5.184181252103758908e-01
6.438052208637921048e-01 6.797486679084566719e-01 5.160591646972285673e-01
6.370832823378153043e-01 6.775304469295674314e-01 5.137285370282342889e-01
6.303705085075785863e-01 6.753064791862326555e-01 5.114205507376848869e-01
6.236618961603839217e-01 6.730786409287115024e-01 5.091324121853675333e-01
6.169335288716588650e-01 6.708537547863058226e-01 5.068827462564762243e-01
6.102104324218552422e-01 6.686246254359063945e-01 5.046462757174120517e-01
6.034898306929903367e-01 6.663922665431509795e-01 5.024201930501221991e-01
5.967676345670763771e-01 6.641578140003113750e-01 5.002054482667135371e-01
5.900234330755329548e-01 6.619267690502547152e-01 4.980175294001410458e-01
5.832786395653163369e-01 6.596933136673572839e-01 4.958328661006338733e-01
5.765340663187056292e-01 6.574571405303393234e-01 4.936489952759946509e-01
5.697582662671993869e-01 6.552266499575032377e-01 4.914884851056534054e-01
5.629730101905379147e-01 6.529958318145387963e-01 4.893302200215370878e-01
5.561847270851512093e-01 6.507627829419350141e-01 4.871678208383450337e-01
5.493551097307179942e-01 6.485375215149538075e-01 4.850265872147694890e-01
5.425256305514831734e-01 6.463076874918682879e-01 4.828887171920029364e-01
5.356612977693596678e-01 6.440835181737865067e-01 4.807599721985225671e-01
5.287922197832997107e-01 6.418536560641270317e-01 4.786600904179345028e-01
5.218850344197893953e-01 6.396291731218471943e-01 4.765733407479101902e-01
5.149376539505706729e-01 6.374099915253575999e-01 4.745006244225868364e-01
5.079766080089844760e-01 6.351862479551418916e-01 4.724595195428232253e-01
5.009755431565769968e-01 6.329662155198152451e-01 4.704378162646060679e-01
4.939285311779693655e-01 6.307509252620584483e-01 4.684356727521749586e-01
4.868490205117793068e-01 6.285349533886431805e-01 4.664642845323079823e-01
4.797346535241335252e-01 6.263183759111613513e-01 4.645235778249863778e-01
4.725684462540446495e-01 6.241056609282084056e-01 4.626084796977160685e-01
4.653487483821732718e-01 6.218962626011749206e-01 4.607211778296827487e-01
4.580958304602617548e-01 6.196828431511897106e-01 4.588752179328909331e-01
4.507924598809112671e-01 6.174698402959368781e-01 4.570641185207933721e-01
4.434287999890613730e-01 6.152589986197224414e-01 4.552877084880677105e-01
4.360045483020283386e-01 6.130492733714320019e-01 4.535480340189612103e-01
4.285254867993349426e-01 6.108377350391323013e-01 4.518515677783579432e-01
4.209986373257934011e-01 6.086212623350139017e-01 4.502044135688367255e-01
4.134050226321455135e-01 6.064040523424363283e-01 4.486012173551829352e-01
4.057442294957287476e-01 6.041849412176459877e-01 4.470442374703665345e-01
3.980103804981384719e-01 6.019638983605486438e-01 4.455375274544488007e-01
3.902063406549050040e-01 5.997389567937310151e-01 4.440824403586798308e-01
3.823330219069303038e-01 5.975084478594807624e-01 4.426827692274404313e-01
3.743994820234466392e-01 5.952685939479632760e-01 4.413464089373288490e-01
3.663895342858319859e-01 5.930221303804210642e-01 4.400695063624517345e-01
3.583027331816978078e-01 5.907676584219552218e-01 4.388545840675231458e-01
3.501387968837356146e-01 5.885037242420276815e-01 4.377041322664636525e-01
3.418974318117430355e-01 5.862288538856508247e-01 4.366206692939140765e-01
3.335788357145262895e-01 5.839414671596280249e-01 4.356065575056938810e-01
3.251833166933908448e-01 5.816399536413853211e-01 4.346641338993996184e-01
3.167127839080131069e-01 5.793224392963164382e-01 4.337951239114763990e-01
3.081690890879107969e-01 5.769870856598143805e-01 4.330012470667766733e-01
2.995554017471688812e-01 5.746318865559506550e-01 4.322836619258556556e-01
2.908749326626684506e-01 5.722548869131893756e-01 4.316434590630243706e-01
2.821317603555529852e-01 5.698540727656976612e-01 4.310813189213763552e-01
2.733327132702973450e-01 5.674271544637526921e-01 4.305966557525391369e-01
2.644844543662756564e-01 5.649720080402284017e-01 4.301888298931589305e-01
2.555934335426425785e-01 5.624866759386734083e-01 4.298572085857526592e-01
2.466705752105852745e-01 5.599688410786282100e-01 4.295989836123562111e-01
2.377250969065505815e-01 5.574166090031631438e-01 4.294119794031988069e-01
2.287974288505327247e-01 5.548237349096721838e-01 4.292976581628650257e-01
2.198778468774868489e-01 5.521921840191843511e-01 4.292473634163982976e-01
2.109782428379559649e-01 5.495207384038700571e-01 4.292558006274433957e-01
2.021129920761091103e-01 5.468081421831656463e-01 4.293174959448664008e-01
1.932970840289524594e-01 5.440534055393755342e-01 4.294264526279761851e-01
1.845458994968318112e-01 5.412558313792091846e-01 4.295762645205056240e-01
1.758747242186788073e-01 5.384150471574241648e-01 4.297604082769522771e-01
1.672992650076766952e-01 5.355309682751767664e-01 4.299719066013750202e-01
1.588637533307953875e-01 5.326007152638789766e-01 4.302077664091210063e-01
1.505683016211286918e-01 5.296267950764182997e-01 4.304586487604788458e-01
1.424122559523727238e-01 5.266116064108816719e-01 4.307170632116699704e-01
1.344085890794944338e-01 5.235562755642496624e-01 4.309766346935359205e-01
1.265690627070247265e-01 5.204621553185740934e-01 4.312317121888489257e-01
1.189042023704675322e-01 5.173307669769749984e-01 4.314776038001592595e-01
1.114246335252623010e-01 5.141637863856259871e-01 4.317091179175960858e-01
1.041396630107440102e-01 5.109629971472293697e-01 4.319221604647225932e-01
9.705780332103111641e-02 5.077302369436778040e-01 4.321134999997184867e-01
9.018737534593762595e-02 5.044674058535019157e-01 4.322799148264460101e-01
8.353621797943017180e-02 5.011764305824176757e-01 4.324186435951988816e-01
7.711177607780977938e-02 4.978592446938872595e-01 4.325273326417442554e-01
7.092118659398483071e-02 4.945177606318418850e-01 4.326041097483195319e-01
6.497112946544478240e-02 4.911539160136639270e-01 4.326469613219918742e-01
5.926836252985791947e-02 4.877695081001428012e-01 4.326551664213393877e-01
5.381952845223608034e-02 4.843663141865421351e-01 4.326279059471102584e-01
4.863121397792258965e-02 4.809460446320443228e-01 4.325646545440974822e-01
4.370962682231178259e-02 4.775103648435110038e-01 4.324649641775900610e-01
3.905112341714592900e-02 4.740608794693825234e-01 4.323285604649402813e-01
3.496888770654411893e-02 4.705940487017962748e-01 4.321582669839647561e-01
3.143475353593022659e-02 4.671148853349996188e-01 4.319569510186779815e-01
2.822939081579376591e-02 4.636310861634273528e-01 4.317228454236091695e-01
2.532316922257733458e-02 4.601429344885760719e-01 4.314611817794063997e-01
2.271621196496529868e-02 4.566502451321696188e-01 4.311704916368828688e-01
2.041021909707332535e-02 4.531526976651161776e-01 4.308500436842361836e-01
1.840750332198089895e-02 4.496499166924270985e-01 4.304993286811824582e-01
1.672948316457980103e-02 4.461408970806849950e-01 4.301165746135667267e-01
1.538958862814012937e-02 4.426249994353490536e-01 4.296990539873294934e-01
1.435413070644788770e-02 4.391028750771387434e-01 4.292495981309878972e-01
1.362474263231045098e-02 4.355741563391084537e-01 4.287674174473046218e-01
1.320277209836126682e-02 4.320384817616971795e-01 4.282516463471394697e-01
1.308981022524788491e-02 4.284954582700414294e-01 4.277015087056333931e-01
1.328604261598065328e-02 4.249447744648222614e-01 4.271158498149437155e-01
1.379162556189269216e-02 4.213861028127094399e-01 4.264935743343106211e-01
1.460601943539846688e-02 4.178191447440461337e-01 4.258334705527410113e-01
1.572829683901560624e-02 4.142436086986017174e-01 4.251343095806048300e-01
1.715690277980158954e-02 4.106592256461835122e-01 4.243947975574930975e-01
1.888987211594707338e-02 4.070657340951429060e-01 4.236136390627659454e-01
2.092392860184388512e-02 4.034629389123669529e-01 4.227893563025114654e-01
2.325551642690237841e-02 3.998506428553240677e-01 4.219205349032726371e-01
2.587976864011615197e-02 3.962287140215077774e-01 4.210056239775798459e-01
2.879190433075056593e-02 3.925969951003727698e-01 4.200432223313694258e-01
3.198535565039070661e-02 3.889554251322902556e-01 4.190317351172953564e-01
3.545196060096311025e-02 3.853040266354982313e-01 4.179694561300110189e-01
3.918323414610008770e-02 3.816428245764514893e-01 4.168548033419602339e-01
4.307413522281567514e-02 3.779718375548203335e-01 4.156863315515785251e-01
4.697176598438539824e-02 3.742911911603020725e-01 4.144624688814931845e-01
5.086969232374114608e-02 3.706011159074634187e-01 4.131815676575790186e-01
5.475506718563923764e-02 3.669018229746837667e-01 4.118421970922833131e-01
5.861718450656796392e-02 3.631935427420149298e-01 4.104430479071176641e-01
6.244486447080809660e-02 3.594767157385404532e-01 4.089825820635137332e-01
6.623026614707036575e-02 3.557516605834906143e-01 4.074596950461942813e-01
6.998642301215993178e-02 3.520178004091850665e-01 4.058682797037249657e-01
7.368349105094931795e-02 3.482766831924774542e-01 4.042119944889063232e-01
7.731558236370641990e-02 3.445288253637411868e-01 4.024900881431115462e-01
8.087676210173380675e-02 3.407748586458358986e-01 4.007018083483494530e-01
8.436223121714692130e-02 3.370153951525706182e-01 3.988466436336530196e-01
8.776750670667690657e-02 3.332510976549885595e-01 3.969242240138116662e-01
9.108869400118912996e-02 3.294826406371675898e-01 3.949343703704463748e-01
9.432201977980475549e-02 3.257107550258697137e-01 3.928770441587735585e-01
9.746469555428810549e-02 3.219361155964274857e-01 3.907524596946260753e-01
1.005137065660947904e-01 3.181594785366577693e-01 3.885609392070024093e-01
1.034667829102825787e-01 3.143815541170876804e-01 3.863030299565164416e-01
1.063234262084384596e-01 3.106029224571804637e-01 3.839789892033815266e-01
1.090886774331554510e-01 3.068237947954853828e-01 3.815863917579840892e-01
1.117516829166099457e-01 3.030455295972052654e-01 3.791303642957770670e-01
1.143116567201993838e-01 2.992687663919490482e-01 3.766121406045498943e-01
1.167670145114603486e-01 2.954942819265780063e-01 3.740330281243106070e-01
1.191180972380510850e-01 2.917225736922783308e-01 3.713945692144456912e-01
1.213648735453208283e-01 2.879541937006656616e-01 3.686983734293431958e-01
1.235072626972245069e-01 2.841897016932966036e-01 3.659461400710485868e-01
1.255450172977832790e-01 2.804296856157670326e-01 3.631396556775801088e-01
1.274793353662031781e-01 2.766744826581494787e-01 3.602808188600382100e-01
1.293108779906596839e-01 2.729245043006408022e-01 3.573715653564341621e-01
1.310404714307908669e-01 2.691801184313953876e-01 3.544138839210129732e-01
1.326690608188904053e-01 2.654416544155482338e-01 3.514098053356367601e-01
1.341977870856449484e-01 2.617093858425083575e-01 3.483613905337877847e-01
1.356276601953987637e-01 2.579835900091309586e-01 3.452707288070074876e-01
1.369600353052619901e-01 2.542644558590421155e-01 3.421399178935836116e-01
1.381965371841252788e-01 2.505520924183904374e-01 3.389710465513224302e-01
1.393387446776407912e-01 2.468465884786986142e-01 3.357661977904279893e-01
1.403881322096931505e-01 2.431480256484742086e-01 3.325274517113971373e-01
1.413463644264434660e-01 2.394564173026069165e-01 3.292568607367298839e-01
1.422150204492807768e-01 2.357717647764656088e-01 3.259564633033559256e-01
1.430241411872627821e-01 2.320878753057155075e-01 3.226241875019154048e-01
1.437206808568328997e-01 2.284105260214233102e-01 3.193108148264102164e-01
1.443167183567656553e-01 2.247376722352139033e-01 3.160160321612740519e-01
1.448172234122950819e-01 2.210687136739122871e-01 3.127401980544551319e-01
1.452270337553944191e-01 2.174029799543943375e-01 3.094836465796388381e-01
1.455503303753529210e-01 2.137398650159642033e-01 3.062467330039463920e-01
1.457916124736327868e-01 2.100785929129088681e-01 3.030297495827782850e-01
1.459553844105018761e-01 2.064182886269750883e-01 2.998329600953320573e-01
1.460450738341802501e-01 2.027582683855788659e-01 2.966566757283218836e-01
1.460639705951699008e-01 1.990978105534875198e-01 2.935011720342979302e-01
1.460132039806032178e-01 1.954367333080272440e-01 2.903667875231270012e-01
1.458975819392981654e-01 1.917737643090239308e-01 2.872535927880979223e-01
1.457218019293262112e-01 1.881075422549224607e-01 2.841616643549851884e-01
1.454887645921259243e-01 1.844371298576089335e-01 2.810911343097461934e-01
1.452032543446299950e-01 1.807609202544633309e-01 2.780420590388025248e-01
1.448676044101755434e-01 1.770779511005421691e-01 2.750145417929056313e-01
1.444780219943526678e-01 1.733892008568175547e-01 2.720086027887532176e-01
1.440413431328422811e-01 1.696921848838350289e-01 2.690241039839223469e-01
1.435643975093518765e-01 1.659841914110155581e-01 2.660610427935709565e-01
1.430436238988290198e-01 1.622659427289601941e-01 2.631191819624335571e-01
1.424820569142489501e-01 1.585358695252926109e-01 2.601982981359077240e-01
1.418869430340855831e-01 1.547906985932924473e-01 2.572984244859990999e-01
1.412504255878399562e-01 1.510326235840303566e-01 2.544186879929528633e-01
1.405847553781910952e-01 1.472562100690701925e-01 2.515594367076940396e-01
1.398866955406879442e-01 1.434617145934828086e-01 2.487199683062358835e-01
1.391555552990901834e-01 1.396484041206631221e-01 2.458995874149805250e-01
1.383972370289156262e-01 1.358126929549585971e-01 2.430982825013351944e-01
1.376119899169210015e-01 1.319530974585303440e-01 2.403154706816485464e-01
1.368012914537950486e-01 1.280674462139133607e-01 2.375506853082148262e-01
1.359688085852648887e-01 1.241523347647442155e-01 2.348038243218179000e-01
1.351079666263714507e-01 1.202088638287985289e-01 2.320731097859522474e-01
1.342294527149119343e-01 1.162298274455028368e-01 2.293596493763065958e-01
1.333340174081290574e-01 1.122121767595207487e-01 2.266630565550546428e-01
1.324249399956005380e-01 1.081511805079039545e-01 2.239835024348004189e-01
1.315721561940036699e-01 1.040055590640546201e-01 2.213363669373892839e-01];

   case {'dem','top'} % dem or topo
      RGB = [1.561019746507273376e-01 1.026082525875711138e-01 1.727215696232307918e-01
1.614878059346168959e-01 1.086331782717616379e-01 1.834416540200605461e-01
1.668117590877864764e-01 1.145994139402238821e-01 1.942594655847655616e-01
1.720703275074147443e-01 1.205129174047056828e-01 2.051860710532931176e-01
1.772595702951434704e-01 1.263792859047284667e-01 2.162309361702130506e-01
1.823751828761793481e-01 1.322038111876145949e-01 2.274020989868827392e-01
1.874125186315797886e-01 1.379915476831030108e-01 2.387062880680187460e-01
1.923665760685599468e-01 1.437473901381048913e-01 2.501489823140411461e-01
1.972319620276792862e-01 1.494761585027865047e-01 2.617344119314569673e-01
2.020028382233076125e-01 1.551826890051333785e-01 2.734655011163185101e-01
2.066728560428846562e-01 1.608719311731062473e-01 2.853437530392856081e-01
2.112350828916275125e-01 1.665490510713014960e-01 2.973690772834624574e-01
2.156819223383368844e-01 1.722195412332822306e-01 3.095395593647118360e-01
2.200027865362182422e-01 1.778889096627078448e-01 3.218593457818205161e-01
2.241821302892643142e-01 1.835626593748518609e-01 3.343431537784405938e-01
2.282155151615900546e-01 1.892493734782149939e-01 3.469585312854190362e-01
2.320913707908846546e-01 1.949572807641995476e-01 3.596924727656225507e-01
2.357791582066215419e-01 2.006936504843463420e-01 3.725810406048237766e-01
2.392760085101002243e-01 2.064712127546999287e-01 3.855616338648150121e-01
2.425474613143962510e-01 2.123011804696942062e-01 3.986531809377649171e-01
2.455817950644241798e-01 2.181991605590304917e-01 4.117955160707330586e-01
2.483306606950595463e-01 2.241817667402334346e-01 4.250033167249620547e-01
2.507721436866522935e-01 2.302701237479483076e-01 4.382036814126420432e-01
2.528659964243425984e-01 2.364882912834416206e-01 4.513446141213755536e-01
2.545675711557803811e-01 2.428643843006248471e-01 4.643557571030281772e-01
2.558322794870629413e-01 2.494300496133808887e-01 4.771399728195321877e-01
2.566151212870083631e-01 2.562195453869526296e-01 4.895770754997314511e-01
2.568729082739400482e-01 2.632680806498599591e-01 5.015251413221865073e-01
2.566051651711689918e-01 2.706008890121900934e-01 5.127963599343589030e-01
2.558254018044066047e-01 2.782328077909240194e-01 5.232189983935346955e-01
2.545978636988186494e-01 2.861560770732184955e-01 5.326370186681820273e-01
2.530378388279626023e-01 2.943355684042912035e-01 5.409499814033098541e-01
2.512817651002536290e-01 3.027167370222472731e-01 5.481422664464320471e-01
2.494682375867288693e-01 3.112353605277175528e-01 5.542766318947622839e-01
2.477157002362878613e-01 3.198299095156284522e-01 5.594704462476630669e-01
2.461121481740833894e-01 3.284495884603169102e-01 5.638648470620498676e-01
2.447220255685461088e-01 3.370546900675063240e-01 5.676029832475817383e-01
2.435790408918375727e-01 3.456199709880173332e-01 5.708127541670769967e-01
2.426978313653707087e-01 3.541305556383861908e-01 5.736016010478480753e-01
2.420946965383475313e-01 3.625739944720036134e-01 5.760653956197256953e-01
2.417595622535183009e-01 3.709493751547713325e-01 5.782729391945388153e-01
2.416897018162703636e-01 3.792548337300427064e-01 5.802858143751018494e-01
2.418784445905285962e-01 3.874910027793410650e-01 5.821542521747482546e-01
2.423091294682429564e-01 3.956627542073260506e-01 5.839131390484204598e-01
2.429695407602900925e-01 4.037738393736669540e-01 5.855939512690606641e-01
2.438461792188084676e-01 4.118287207942545325e-01 5.872218507211212080e-01
2.449256139784778408e-01 4.198319765418777605e-01 5.888176525512366366e-01
2.461926332303804310e-01 4.277888629705645096e-01 5.903967722170088139e-01
2.476314436738927260e-01 4.357048438328084417e-01 5.919706842419948378e-01
2.492275176071283571e-01 4.435849323073600692e-01 5.935491293785424283e-01
2.509665668417216944e-01 4.514340415062205181e-01 5.951395103673859932e-01
2.528346862021874641e-01 4.592569393176054171e-01 5.967472668238706923e-01
2.548184740794410263e-01 4.670582183399953347e-01 5.983761730113198452e-01
2.569051367797046126e-01 4.748422758923547815e-01 6.000285749995881712e-01
2.590825816132030779e-01 4.826133005480903182e-01 6.017055804809932074e-01
2.613396987489819967e-01 4.903751992421883643e-01 6.034074237284018372e-01
2.636659810857353015e-01 4.981317406526836744e-01 6.051330956906521008e-01
2.660516582874621339e-01 5.058865272594796902e-01 6.068805013837572648e-01
2.684882380171457750e-01 5.136428405367017280e-01 6.086470522439324515e-01
2.709683308416185876e-01 5.214037172983150281e-01 6.104294711750373192e-01
2.734857257205313141e-01 5.291719440816851083e-01 6.122238564648385672e-01
2.760354668808263079e-01 5.369500509042623992e-01 6.140257392505246159e-01
2.786139324558262742e-01 5.447403043452749838e-01 6.158301364812402978e-01
2.812189151446758406e-01 5.525446999518639490e-01 6.176316008625786225e-01
2.838497050469647731e-01 5.603649539888247988e-01 6.194242689076802089e-01
2.865071747347951447e-01 5.682024945496734203e-01 6.212019079466474247e-01
2.891935293105973304e-01 5.760585509806988025e-01 6.229575870601862242e-01
2.919132661023912667e-01 5.839338872257091584e-01 6.246846842638799080e-01
2.946726143633343065e-01 5.918289843891730850e-01 6.263762188290394883e-01
2.974797079780254205e-01 5.997440045832753697e-01 6.280249877540684533e-01
3.003447193279823457e-01 6.076787658883653354e-01 6.296236627075371128e-01
3.032799496578738041e-01 6.156327286641396501e-01 6.311648425953061414e-01
3.062999162775806861e-01 6.236049805794583456e-01 6.326411091031564071e-01
3.094214341373541788e-01 6.315942201545601264e-01 6.340450861601970578e-01
3.126636885203824545e-01 6.395987386132337971e-01 6.353695045172913503e-01
3.160482946464280851e-01 6.476163998706647718e-01 6.366072729215248582e-01
3.195993388733257556e-01 6.556446185422359907e-01 6.377515576836865208e-01
3.233433949344308722e-01 6.636803359579184214e-01 6.387958727683986648e-01
3.273095072934468774e-01 6.717199943156246800e-01 6.397341828681051279e-01
3.315291322901112725e-01 6.797595093162658308e-01 6.405610222271946874e-01
3.360351011011302735e-01 6.877945827659479594e-01 6.412695960850323118e-01
3.408640318201912600e-01 6.958197818437616977e-01 6.418569321473704958e-01
3.460540467004642462e-01 7.038291754672099110e-01 6.423205839967465192e-01
3.516446005650431528e-01 7.118162697085735902e-01 6.426589560156208414e-01
3.576762479926827720e-01 7.197739138067411613e-01 6.428719883285997083e-01
3.641894512092864744e-01 7.276949011320993366e-01 6.429557270674552960e-01
3.712266123106295335e-01 7.355700686179795778e-01 6.429189618774799886e-01
3.788278449989629926e-01 7.433902546660990929e-01 6.427683061328856029e-01
3.870322615152646528e-01 7.511462095764531721e-01 6.425055472082883412e-01
3.958750262046074608e-01 7.588270315806920907e-01 6.421508285001665817e-01
4.053883482522956383e-01 7.664223146281371468e-01 6.417151578859574546e-01
4.155968407812616339e-01 7.739210344346504344e-01 6.412227317683711902e-01
4.265195639210723755e-01 7.813124732411278472e-01 6.406960731925235297e-01
4.381634653667463852e-01 7.885863788123392837e-01 6.401694292235835526e-01
4.505287643111087204e-01 7.957333261733031682e-01 6.396742566585853496e-01
4.635986539031755060e-01 8.027456073395251579e-01 6.392548734619343254e-01
4.773469739538908629e-01 8.096172922533247940e-01 6.389538165444101914e-01
4.917330703220189059e-01 8.163450533427777378e-01 6.388184799685107107e-01
5.067050118087177424e-01 8.229283397233977393e-01 6.388963162980869637e-01
5.221997330825246530e-01 8.293697946738645133e-01 6.392346216632595057e-01
5.381481232862911357e-01 8.356748924828519831e-01 6.398763669277579558e-01
5.544783665092802849e-01 8.418515860497588488e-01 6.408587965640016870e-01
5.711115213696185133e-01 8.479112775740021979e-01 6.422171169895777298e-01
5.879840953182007279e-01 8.538646600753627691e-01 6.439710905713145195e-01
6.050194699551968425e-01 8.597270135144634562e-01 6.461439080647282118e-01
6.221681809999368706e-01 8.655099786991408140e-01 6.487403091120448329e-01
6.393757001353049807e-01 8.712277338509245572e-01 6.517658303256151919e-01
6.565929334411800822e-01 8.768947704523611941e-01 6.552210497573525139e-01
6.737933070789577927e-01 8.825213731875580780e-01 6.590961736178440056e-01
6.909478082204841831e-01 8.881191135591984809e-01 6.633823202371271766e-01
7.080340865696405084e-01 8.936985513356989763e-01 6.680678941563225059e-01
7.250358375800118882e-01 8.992691342626382145e-01 6.731393536848668813e-01
7.419421046029871514e-01 9.048391559449654453e-01 6.785819005039074314e-01
7.587465580267132026e-01 9.104157595099175992e-01 6.843800748081978469e-01
7.754467945117471395e-01 9.160049747102424478e-01 6.905182510191696377e-01
7.920436855303579771e-01 9.216117772405966191e-01 6.969810377273069069e-01
8.085407925860420564e-01 9.272401607530683654e-01 7.037535913222745521e-01
8.249438571064254822e-01 9.328932140048489252e-01 7.108218561594963347e-01
8.412422934868647451e-01 9.385792958890862847e-01 7.181734629573161000e-01
8.574559441871676402e-01 9.442965454208677167e-01 7.257948454644865821e-01
8.735986658557798323e-01 9.500445573198598170e-01 7.336747582572585857e-01
8.896512126622908578e-01 9.558344072358454513e-01 7.418023698547455691e-01
9.056514559384564178e-01 9.616567079910524063e-01 7.501688017241657791e-01
9.215850348376466439e-01 9.675205198521025229e-01 7.587646801057830181e-01
9.374799177499437697e-01 9.734191074102575003e-01 7.675843461643131471e-01
9.533233708513803029e-01 9.793607550577946297e-01 7.766197776697654209e-01
9.691488355955474310e-01 9.853357520972024775e-01 7.858697059005903540e-01
9.849374457410008388e-01 9.913545172197536504e-01 7.953271573982337861e-01
5.237510688652501772e-02 1.452546228317073418e-01 7.751950190923809214e-02
5.622628551680566161e-02 1.523457500722602831e-01 8.040279400454641845e-02
6.007072166263575236e-02 1.593963705834963718e-01 8.324827330002768089e-02
6.385205670255669763e-02 1.664232541404700172e-01 8.599651952193751447e-02
6.756460338718012215e-02 1.734310951493893138e-01 8.864073081078854832e-02
7.121409992638502717e-02 1.804216402085259963e-01 9.118655316732873772e-02
7.479859509186639888e-02 1.873981595163995706e-01 9.363114185870033412e-02
7.832196854697648369e-02 1.943623428359516903e-01 9.597819921125996800e-02
8.178249510428708957e-02 2.013169715395964343e-01 9.822505767280115263e-02
8.518219102467458614e-02 2.082637805063026204e-01 1.003733079455831900e-01
8.852454157738801066e-02 2.152040105951080751e-01 1.024263253707545229e-01
9.181013978863614144e-02 2.221394247672267563e-01 1.043841327744766678e-01
9.503809838243432173e-02 2.290719902760253168e-01 1.062449430282469132e-01
9.821010670676499910e-02 2.360029705984670323e-01 1.080100113958255836e-01
1.013286574784400540e-01 2.429333482333665417e-01 1.096815914631605327e-01
1.043934471701543076e-01 2.498646410896294690e-01 1.112585290437959340e-01
1.074478296880792549e-01 2.567911108934768372e-01 1.127437202480449374e-01
1.106260709129789077e-01 2.636946096457899458e-01 1.140997527427590474e-01
1.139531706355643714e-01 2.705743606077580798e-01 1.152896296272559462e-01
1.174768136636618332e-01 2.774250521408649361e-01 1.162908535730605153e-01
1.213000971217181034e-01 2.842316842811704602e-01 1.170941428738193346e-01
1.255340801183247312e-01 2.909784812573655843e-01 1.176678790007481823e-01
1.303876420091158728e-01 2.976323416853482451e-01 1.179910028329904936e-01
1.361775080109454139e-01 3.041399107394244794e-01 1.180416347026809198e-01
1.434815209152097981e-01 3.103930959766884601e-01 1.179050440739785321e-01
1.530522840464018097e-01 3.162194219215394564e-01 1.180658427800789778e-01
1.647443020935283886e-01 3.215541086259602332e-01 1.195833209716834211e-01
1.771224001134716619e-01 3.266051714744359624e-01 1.228383029974784990e-01
1.893186680349762951e-01 3.315543486926041949e-01 1.273271390832197980e-01
2.011470216295637714e-01 3.364693524047602802e-01 1.325809211223410999e-01
2.126201126913634942e-01 3.413717760208538898e-01 1.383257062913032798e-01
2.237814929443558420e-01 3.462703085838705896e-01 1.444001987716135027e-01
2.346826504126838242e-01 3.511667839317458850e-01 1.507119862068962424e-01
2.453653984352425765e-01 3.560619397292786315e-01 1.572020472843533856e-01
2.558610883411705506e-01 3.609568875025493395e-01 1.638280010474283122e-01
2.662540618029768935e-01 3.658464720551902194e-01 1.703646574946249270e-01
2.766375890957337713e-01 3.707202928157689592e-01 1.766042606681927363e-01
2.870120792807265286e-01 3.755815421723879277e-01 1.825448147300349488e-01
2.973798701367175723e-01 3.804331662296118188e-01 1.881754975600970214e-01
3.077385714853347887e-01 3.852782482569529487e-01 1.935061166107706343e-01
3.180883154404229307e-01 3.901193934855016199e-01 1.985393196990216658e-01
3.284295845809043213e-01 3.949589033692603168e-01 2.032788009798182638e-01
3.387636524478269684e-01 3.997986013442899611e-01 2.077297610727937283e-01
3.490920713533000597e-01 4.046399832445768951e-01 2.118988339848660862e-01
3.594164000962936090e-01 4.094843090462684243e-01 2.157940430809284771e-01
3.697392907451491073e-01 4.143322752680254073e-01 2.194241817700406383e-01
3.800629535453087238e-01 4.191845097543299148e-01 2.227990620346612660e-01
3.903897648146955057e-01 4.240414062699767728e-01 2.259290789901750585e-01
4.007222760346346169e-01 4.289031445516293117e-01 2.288249032905495528e-01
4.110629393231473583e-01 4.337698031247444463e-01 2.314973656174672545e-01
4.214141346107081465e-01 4.386413758769234783e-01 2.339572260458537833e-01
4.317776944235289238e-01 4.435179532313937023e-01 2.362153631413001498e-01
4.421554582272910761e-01 4.483995629988481446e-01 2.382822640902987343e-01
4.525491728091692312e-01 4.532862202325000367e-01 2.401679621083726013e-01
4.629617793452492358e-01 4.581777295849584486e-01 2.418766613523773423e-01
4.733964081044912953e-01 4.630738873787103027e-01 2.434097398165305792e-01
4.838512407114513025e-01 4.679753825260214994e-01 2.447870115640884969e-01
4.943280333549550654e-01 4.728822194350857377e-01 2.460147775573224282e-01
5.048329973069679566e-01 4.777936355059679285e-01 2.470786486003125892e-01
5.153613297132805249e-01 4.827109910092828859e-01 2.480074378391989298e-01
5.259177888805488532e-01 4.876337443949864681e-01 2.487920646168808037e-01
5.365028531047130178e-01 4.925622747803105050e-01 2.494384427855986242e-01
5.471168004411860464e-01 4.974969337464568153e-01 2.499544447192391661e-01
5.577636102245655536e-01 5.024372790562059432e-01 2.503320304175396527e-01
5.684402705004985012e-01 5.073846875194365502e-01 2.505862169320990929e-01
5.791517884291441653e-01 5.123382474891868821e-01 2.507078127142677859e-01
5.898977118592134694e-01 5.172986884163369714e-01 2.507019666506383193e-01
6.006780609928230596e-01 5.222664305265023454e-01 2.505758843768319810e-01
6.114977839232860202e-01 5.272408795639659251e-01 2.503116806147138718e-01
6.223533151135658414e-01 5.322233326069301107e-01 2.499311738427514862e-01
6.332678499493179514e-01 5.372050254585409856e-01 2.494034423308610915e-01
6.442492302385145475e-01 5.421770814367210534e-01 2.488390322915944863e-01
6.553235853574364000e-01 5.471283166651099705e-01 2.482154543042252026e-01
6.664663550324848584e-01 5.520678623990600276e-01 2.475959486163559209e-01
6.777036161655425328e-01 5.569846941304811283e-01 2.469507035732461664e-01
6.890381603900364027e-01 5.618763691265282745e-01 2.463040749474359470e-01
7.004711944468632323e-01 5.667403668940831363e-01 2.456991151603213352e-01
7.120126265469914895e-01 5.715693590145963787e-01 2.451873584080898616e-01
7.236929482815777082e-01 5.763444689764196660e-01 2.448364505953754544e-01
7.355168651190714391e-01 5.810504769862679941e-01 2.449163925191181757e-01
7.475258033710173722e-01 5.856296804733571726e-01 2.462013043800762579e-01
7.577499494457186069e-01 5.908503548553694085e-01 2.547905571379186496e-01
7.635824979182088690e-01 5.981540408918001317e-01 2.695060335496625714e-01
7.685008057442868079e-01 6.059656669084559910e-01 2.841339736308181596e-01
7.731069113227515555e-01 6.139683821427790456e-01 2.985534874705023101e-01
7.775652489236080100e-01 6.220751351043045663e-01 3.127742014086093980e-01
7.819223609772602002e-01 6.302590951035305089e-01 3.268597741416817137e-01
7.862053950616281206e-01 6.385051507786967395e-01 3.408479839175493908e-01
7.904426037822046558e-01 6.467997885569605199e-01 3.547436291978552925e-01
7.946574580110756791e-01 6.551326370117361853e-01 3.685463547975237342e-01
7.988337796012735526e-01 6.635096250178373900e-01 3.823162323281250607e-01
8.030088001380106810e-01 6.719161948372565085e-01 3.960118210517307724e-01
8.071531032877933276e-01 6.803638144055433878e-01 4.097046504341926854e-01
8.113041001849377043e-01 6.888386707654255980e-01 4.233460558902438220e-01
8.154602754828877975e-01 6.973419981167042758e-01 4.369537331203900976e-01
8.196058627154122478e-01 7.058802025749554288e-01 4.505666902424197429e-01
8.237568818874844156e-01 7.144480574336203871e-01 4.641691373089507633e-01
8.279197418246522222e-01 7.230440823922363869e-01 4.777602014010458586e-01
8.320913582455962132e-01 7.316701160638459100e-01 4.913552497157864241e-01
8.362534254444016213e-01 7.403317836360435722e-01 5.050123599879642322e-01
8.404445466425872757e-01 7.490163102116410565e-01 5.186797759271363217e-01
8.446648066780128028e-01 7.577248034134459465e-01 5.323633003798176055e-01
8.489177724746317377e-01 7.664572638824626027e-01 5.460621193928290040e-01
8.532123511454875464e-01 7.752120992980529035e-01 5.597655879096986586e-01
8.575441658298394998e-01 7.839918355417494489e-01 5.734875036821195371e-01
8.619146064360614368e-01 7.927971784487287676e-01 5.872309291049341295e-01
8.663250853209770730e-01 8.016288174204664330e-01 6.009988751772729065e-01
8.707820725499390013e-01 8.104859794311163323e-01 6.147852119944903215e-01
8.752928841127598503e-01 8.193677275097041024e-01 6.285824125979521115e-01
8.798495959466368088e-01 8.282774528639023082e-01 6.424105990843714808e-01
8.844600743726268588e-01 8.372140407050960853e-01 6.562614074750376947e-01
8.891318088448872947e-01 8.461765685160849149e-01 6.701273264224787418e-01
8.938558100770686021e-01 8.551685709847908212e-01 6.840307207014526547e-01
8.986544949798678239e-01 8.641851727809914951e-01 6.979374582474833222e-01
9.035103927619371200e-01 8.732320822602240851e-01 7.118855904188222672e-01
9.084448340107732500e-01 8.823048446731206473e-01 7.258432784575933328e-01
9.134526188903524524e-01 8.914059055188421343e-01 7.398268438202468822e-01
9.185353828258727704e-01 9.005358905803305669e-01 7.538405390559674846e-01
9.237018976170942031e-01 9.096936780016002810e-01 7.678758238575987827e-01
9.289597403884261029e-01 9.188785152907725795e-01 7.819264008239902308e-01
9.343121189309133712e-01 9.280907276838803455e-01 7.959940980655445530e-01
9.397648981125013012e-01 9.373300331729016444e-01 8.100759042872343052e-01
9.453185269797627077e-01 9.465973949550759992e-01 8.241789901926560580e-01
9.509801771850346919e-01 9.558922732509941289e-01 8.382978631896051969e-01
9.567560362262098606e-01 9.652144306850017896e-01 8.524284880105463813e-01
9.626518136622860267e-01 9.745638255706342568e-01 8.665671328953633568e-01
9.686647318914142213e-01 9.839422330504176140e-01 8.807260550903986962e-01
9.747892173640951841e-01 9.933519398287798952e-01 8.949228432580742520e-01];

   case 'dif' % diff
      RGB = [3.080165225110909760e-02 1.368487040065790861e-01 2.498464445599150041e-01
3.427654989146505099e-02 1.437127028532935447e-01 2.573239494162229413e-01
3.787813715751361943e-02 1.505402390894255427e-01 2.647902048135927777e-01
4.156812010333730406e-02 1.573357824203455158e-01 2.722473736730874894e-01
4.517179532760850352e-02 1.641035092921928340e-01 2.796979818302940402e-01
4.870180895529663961e-02 1.708454227759541588e-01 2.871401983091831367e-01
5.214717289956722485e-02 1.775659175978645810e-01 2.945787303179320804e-01
5.551364141943334468e-02 1.842674761732781552e-01 3.020142881481510666e-01
5.878267145509050856e-02 1.909546841623958602e-01 3.094531346302050734e-01
6.203222237219494645e-02 1.976272549553735325e-01 3.168486096722107348e-01
6.521944756836103863e-02 2.042923786219034143e-01 3.242073651148437707e-01
6.833928990190613062e-02 2.109538539883098474e-01 3.315240779733869547e-01
7.142525636138788436e-02 2.176109417067573770e-01 3.387875021305949974e-01
7.444813366291075374e-02 2.242694950814881905e-01 3.459978826653878348e-01
7.742364240945528997e-02 2.309308575473027481e-01 3.531440825838942366e-01
8.038349666926730697e-02 2.375958135712563091e-01 3.602034285272879832e-01
8.335927002300280719e-02 2.442657843736594225e-01 3.671480689795030838e-01
8.636609239069936717e-02 2.509437864845217581e-01 3.739531368808389766e-01
8.948430357745071340e-02 2.576279992007458053e-01 3.805657110274591748e-01
9.284058523033036914e-02 2.643149709823168769e-01 3.868990078988859826e-01
9.668345778728781870e-02 2.709875997226547928e-01 3.928398106800405909e-01
1.013388564145758508e-01 2.776155815858447617e-01 3.982760426582682145e-01
1.072606416994524026e-01 2.841399259508246011e-01 4.031326823215036770e-01
1.144021508332261350e-01 2.905306843282992602e-01 4.075704938470787742e-01
1.224943240061644456e-01 2.967876633109050588e-01 4.117594183512530703e-01
1.310866885733582565e-01 3.029446515900964254e-01 4.158631835078765437e-01
1.399259448624195767e-01 3.090284381668210734e-01 4.199407082636338884e-01
1.488880808179499637e-01 3.150548708758020844e-01 4.240166515702193939e-01
1.578509724192573571e-01 3.210422517899584882e-01 4.281197299059529837e-01
1.668002459411131178e-01 3.269969805380745775e-01 4.322421588278035354e-01
1.756791094481522930e-01 3.329300099876690844e-01 4.364047384684796027e-01
1.845067584338563049e-01 3.388436977327556332e-01 4.405895801655569377e-01
1.932823885204988379e-01 3.447414431949823999e-01 4.447966430522279913e-01
2.019870459778479455e-01 3.506290556082184984e-01 4.490357338061347625e-01
2.106258150884706692e-01 3.565090549124025343e-01 4.533046032331278785e-01
2.192099721057354511e-01 3.623829850143509002e-01 4.575959954859014078e-01
2.277275445301833456e-01 3.682546023486205078e-01 4.619233127548696971e-01
2.361939130992141700e-01 3.741245762230066552e-01 4.662760464071296629e-01
2.446124256489054516e-01 3.799946234495278352e-01 4.706542744097454434e-01
2.529874638889413330e-01 3.858660993963552444e-01 4.750577339190826254e-01
2.613341048773878406e-01 3.917383910736120800e-01 4.794796637798960925e-01
2.696394299537646644e-01 3.976153747948966699e-01 4.839303534081906277e-01
2.779049881050036919e-01 4.034985563305431566e-01 4.884111449080716372e-01
2.861363993124647065e-01 4.093887667782184492e-01 4.929196308795206760e-01
2.943389940220551004e-01 4.152865778556348864e-01 4.974548265548736636e-01
3.025334250389247748e-01 4.211897194688838386e-01 5.020046120542877022e-01
3.106982442651501364e-01 4.271034039757660716e-01 5.065852469354352738e-01
3.188368260273997667e-01 4.330284571606364818e-01 5.111959566876937977e-01
3.269657343217935996e-01 4.389628772920392552e-01 5.158284147455556301e-01
3.350816436419064015e-01 4.449085811856242079e-01 5.204856000258065718e-01
3.431739289326510289e-01 4.508688325370560634e-01 5.251766674651570099e-01
3.512656069646695189e-01 4.568402506918622374e-01 5.298868083741347101e-01
3.593506132834142774e-01 4.628250736129231879e-01 5.346217853574988244e-01
3.674172478556836929e-01 4.688267836740107053e-01 5.393913336663840319e-01
3.754973469427250743e-01 4.748398228270848676e-01 5.441739055327718955e-01
3.835625614368970981e-01 4.808709517650425203e-01 5.489922576544932209e-01
3.916201861820892138e-01 4.869196696786777800e-01 5.538424587653016928e-01
3.996888560566008164e-01 4.929829902359726401e-01 5.587113837570589769e-01
4.077452976164708254e-01 4.990668992329919118e-01 5.636172499463452112e-01
4.158238363550112449e-01 5.051647578473101863e-01 5.685356950340605398e-01
4.238869442834262147e-01 5.112855069066921665e-01 5.734980302852201728e-01
4.319617699154935098e-01 5.174241819464946435e-01 5.784833267853278782e-01
4.400436018491948320e-01 5.235826893170786311e-01 5.834960386464431714e-01
4.481312796407512233e-01 5.297621053719542283e-01 5.885383546494945550e-01
4.562263279894743229e-01 5.359626809225542798e-01 5.936121659102355785e-01
4.643239808635050703e-01 5.421865887597332456e-01 5.987202656129063660e-01
4.724486870344450362e-01 5.484289164378903791e-01 6.038441893417568762e-01
4.805822297704738788e-01 5.546948405817618832e-01 6.089980501349065989e-01
4.887172410474664441e-01 5.609869159623338541e-01 6.141889435023867305e-01
4.968703980230008699e-01 5.673022611108096136e-01 6.194017678807255400e-01
5.050230443352835552e-01 5.736457002325791033e-01 6.246566996141469374e-01
5.132020014296254651e-01 5.800120364495556791e-01 6.299281926614284099e-01
5.213837697997298903e-01 5.864072076681272616e-01 6.352414056079356275e-01
5.295840791882617804e-01 5.928284223714805901e-01 6.405818144094436173e-01
5.377985006631785803e-01 5.992773867217912054e-01 6.459553074236056291e-01
5.460310568089291605e-01 6.057538951510261782e-01 6.513591302487201640e-01
5.542803602235417681e-01 6.122589708601943181e-01 6.567960616850564426e-01
5.625427909572455754e-01 6.187941687382881861e-01 6.622712272448196824e-01
5.708318382965930082e-01 6.253570561449082188e-01 6.677719395169281480e-01
5.791326606499368479e-01 6.319518087895650282e-01 6.733150010032186161e-01
5.874594162608378634e-01 6.385758077347706285e-01 6.788869295566190010e-01
5.958021954354665306e-01 6.452321052037962579e-01 6.844994504225663245e-01
6.041669493667392032e-01 6.519200031142581286e-01 6.901476288957911764e-01
6.125605719486232337e-01 6.586385467570500252e-01 6.958254420726305289e-01
6.209702599139194090e-01 6.653915355400340514e-01 7.015478000189213637e-01
6.294086543235715148e-01 6.721766263526947061e-01 7.073025514622064414e-01
6.378691230472761653e-01 6.789961486811091351e-01 7.130980827587862780e-01
6.463529402277414793e-01 6.858505203730194122e-01 7.189343580950778856e-01
6.548667336608775535e-01 6.927388303718031715e-01 7.248055449230874636e-01
6.634047329440709850e-01 6.996632326198537477e-01 7.307191763991576217e-01
6.719706356478575282e-01 7.066235438274829361e-01 7.366725743067246146e-01
6.805691273316444301e-01 7.136193210595811465e-01 7.426619469578672472e-01
6.891901686164448870e-01 7.206538297392898196e-01 7.486995567627365844e-01
6.978417441976546565e-01 7.277258099214434228e-01 7.547780354337728648e-01
7.065266827690414031e-01 7.348352863051689221e-01 7.608955968203486853e-01
7.152395659488615109e-01 7.419843878319358765e-01 7.670595168287617227e-01
7.239821407052556834e-01 7.491734241473727574e-01 7.732692164928602896e-01
7.327590970387688474e-01 7.564019312463790001e-01 7.795208278459410112e-01
7.415714112445037642e-01 7.636703716037079870e-01 7.858145260879668692e-01
7.504112152902991939e-01 7.709814701087125410e-01 7.921602549877707622e-01
7.592858091842216162e-01 7.783339526866095426e-01 7.985510124613616201e-01
7.681958083667864701e-01 7.857281516038946423e-01 8.049869935018724165e-01
7.771415943289318173e-01 7.931642615485536840e-01 8.114683168046802342e-01
7.861210234635496175e-01 8.006428249211288151e-01 8.179973940806203325e-01
7.951301023947144886e-01 8.081643845947440452e-01 8.245778955860991744e-01
8.041728469938760337e-01 8.157266105617665408e-01 8.312032233576642781e-01
8.132456568435891819e-01 8.233280673585187115e-01 8.378734408880845752e-01
8.223487919825404058e-01 8.309646782111167473e-01 8.445815347091445435e-01
8.314739162371332926e-01 8.386323940000459665e-01 8.513263697699344767e-01
8.406130156021551780e-01 8.463239863842642041e-01 8.581012298825883011e-01
8.497544923553936869e-01 8.540290636742366992e-01 8.648964939043677358e-01
8.588848363304528721e-01 8.617324046759308187e-01 8.716956826827835236e-01
8.679835640163260368e-01 8.694139530371515212e-01 8.784788921095023628e-01
8.770201283066206832e-01 8.770482020713447069e-01 8.852238656983413279e-01
8.859648522194258913e-01 8.845997703979764371e-01 8.918910598190278316e-01
8.947687008355051930e-01 8.920276102541201402e-01 8.984445530744430419e-01
9.033850238565973578e-01 8.992781133601027710e-01 9.048255526813022698e-01
9.117454267355319386e-01 9.062912781484097069e-01 9.109792984365244761e-01
9.197772674046712504e-01 9.129962371365842877e-01 9.168355520637951894e-01
9.273989381574117008e-01 9.193137785263337802e-01 9.223160826098045773e-01
9.345265056154450356e-01 9.251567923877340727e-01 9.273308757297369365e-01
9.410568294999761552e-01 9.304383120413204367e-01 9.318033109548278237e-01
9.469054517188711939e-01 9.350653892052174232e-01 9.356338452806639561e-01
9.519782306847331954e-01 9.389521024465230514e-01 9.387414682758670192e-01
9.561855219785392324e-01 9.420212349770757942e-01 9.410568432390962190e-01
9.594583919916889192e-01 9.442053410129402913e-01 9.425129916890260251e-01
9.617359608985257546e-01 9.454553544501299589e-01 9.430685634745854529e-01
9.629655575123797773e-01 9.457446361853111272e-01 9.427146073591580189e-01
9.629717304152120017e-01 9.451630117299317790e-01 9.411078740469980275e-01
9.619086196808096512e-01 9.436969537999753133e-01 9.379427077261371926e-01
9.599163272304170880e-01 9.412870435035790573e-01 9.337158672983336682e-01
9.570384080931425563e-01 9.379846152444032414e-01 9.285135912753641474e-01
9.533354457428268036e-01 9.338571023316933895e-01 9.224353821916627671e-01
9.488865060103991445e-01 9.289840890976138743e-01 9.155685277230518615e-01
9.437794827762752137e-01 9.234529155688914193e-01 9.080043258769950887e-01
9.381021433775925678e-01 9.173543171347293690e-01 8.998514836828582775e-01
9.319492013404071518e-01 9.107783993290641256e-01 8.911882560099286810e-01
9.254038313424192141e-01 9.038112443270013285e-01 8.821180550321400249e-01
9.185496533096065841e-01 8.965322509059643341e-01 8.727092681908217298e-01
9.114540331246678839e-01 8.890124126067980859e-01 8.630519170031795140e-01
9.041820632910790856e-01 8.813132734312176808e-01 8.531979805537907025e-01
8.967869578435456734e-01 8.734867291901790010e-01 8.431978773211421530e-01
8.893098149893794435e-01 8.655753876269207669e-01 8.331030513350975442e-01
8.817848418192094639e-01 8.576132802667703059e-01 8.229492517613266056e-01
8.742401985712474621e-01 8.496268650289525715e-01 8.127582978377587697e-01
8.666945370128648074e-01 8.416362324606945222e-01 8.025572922175608914e-01
8.591642949649661576e-01 8.336561546444334336e-01 7.923557697274130618e-01
8.516606462490278195e-01 8.256972216872984216e-01 7.821629896098961643e-01
8.441881675648532646e-01 8.177668584523517525e-01 7.719978747167733912e-01
8.367546003836178192e-01 8.098699298493980958e-01 7.618548556771399527e-01
8.293628053720738524e-01 8.020095841729102393e-01 7.517385870991303287e-01
8.220122974103521996e-01 7.941877657343777708e-01 7.416604820361886174e-01
8.147055831041186691e-01 7.864054338886661277e-01 7.316158935999518276e-01
8.074428695046120819e-01 7.786629991741689238e-01 7.216062017816766705e-01
8.002239536861533997e-01 7.709604828947951294e-01 7.116324410842282955e-01
7.930486047248805903e-01 7.632976487972781277e-01 7.016945092401379869e-01
7.859155444098100407e-01 7.556741407493390295e-01 6.917956122566603083e-01
7.788239785472398369e-01 7.480894917424205648e-01 6.819365996758520732e-01
7.717761666842225532e-01 7.405430431500328314e-01 6.721061511286441359e-01
7.647695126504651109e-01 7.330343101062176681e-01 6.623117248985945782e-01
7.578024759469249583e-01 7.255627582077430748e-01 6.525565861539863732e-01
7.508747011746044198e-01 7.181277940669236193e-01 6.428393514066714776e-01
7.439860296366358483e-01 7.107288105275537671e-01 6.331578659286328792e-01
7.371380352230317845e-01 7.033650997305714858e-01 6.235032380461981161e-01
7.303267295539673798e-01 6.960362660858002704e-01 6.138882457032197593e-01
7.235516370761116978e-01 6.887417344158411892e-01 6.043120048450656423e-01
7.168136574616641443e-01 6.814808447026716731e-01 5.947682975771849678e-01
7.101132300652239770e-01 6.742529573780132734e-01 5.852527166373640011e-01
7.034464832580180627e-01 6.670577263264326762e-01 5.757776499898806799e-01
6.968134241386333416e-01 6.598945644190796767e-01 5.663404055906162693e-01
6.902175511682563380e-01 6.527626219057073298e-01 5.569247166393624937e-01
6.836530241251399520e-01 6.456617329968893371e-01 5.475506786771059398e-01
6.771191923792275746e-01 6.385913765706551226e-01 5.382182031186726334e-01
6.706213406334817773e-01 6.315505441676552145e-01 5.289041000996647091e-01
6.641534350911151297e-01 6.245391367090507018e-01 5.196292578451316979e-01
6.577148635275871236e-01 6.175566394334415232e-01 5.103934681036723653e-01
6.513106015597026621e-01 6.106020304081083427e-01 5.011747507363846221e-01
6.449339394177714402e-01 6.036753299959042307e-01 4.919966626063084214e-01
6.385861657554016135e-01 5.967758543247165814e-01 4.828515990278902659e-01
6.322678284337878152e-01 5.899029749242099552e-01 4.737348355247164577e-01
6.259767255318442469e-01 5.830569263656453227e-01 4.646452891563762067e-01
6.197016579332955688e-01 5.762398779913381341e-01 4.556052788341879434e-01
6.134474028946137469e-01 5.694504347953359691e-01 4.465979423650075497e-01
6.072178659159134240e-01 5.626871366850844103e-01 4.376111326566481941e-01
6.010012505217495749e-01 5.559520936787821777e-01 4.286719487404299644e-01
5.948055640046073789e-01 5.492429546962157572e-01 4.197575498195730836e-01
5.886326075840858651e-01 5.425588558149744278e-01 4.108591465488395378e-01
5.824702130202078498e-01 5.359018268492113934e-01 4.020079470557531565e-01
5.763248263841350694e-01 5.292699281179926718e-01 3.931837482598241618e-01
5.701972311820494577e-01 5.226623693222695044e-01 3.843824114977716366e-01
5.640831336223245396e-01 5.160796590123359895e-01 3.756116269725132129e-01
5.579774433964941327e-01 5.095224539424874077e-01 3.668822207066750885e-01
5.518860265359687434e-01 5.029888698004121306e-01 3.581753826709989652e-01
5.458070559295251645e-01 4.964787819395978796e-01 3.494931200704951557e-01
5.397391255491313933e-01 4.899920506546056598e-01 3.408352402134943726e-01
5.336730625437067221e-01 4.835304037810426725e-01 3.322231435985951165e-01
5.276152444526773788e-01 4.770918383518058525e-01 3.236356057173125356e-01
5.215648631466205387e-01 4.706762498171280229e-01 3.150682292920859440e-01
5.155189794095078604e-01 4.642837707070938680e-01 3.065265223389798677e-01
5.094751216031231378e-01 4.579145264501231494e-01 2.980134272377937266e-01
5.034310278715931064e-01 4.515686733480185899e-01 2.895302566325898552e-01
4.973847636608849654e-01 4.452464140846874030e-01 2.810755531259728768e-01
4.913352410442906604e-01 4.389480201490584821e-01 2.726410128949737222e-01
4.852774212045994351e-01 4.326740360757653225e-01 2.642407735080369302e-01
4.792095319460407121e-01 4.264249820482975961e-01 2.558674925325885585e-01
4.731283857839347351e-01 4.202016073968968812e-01 2.475197805445442101e-01
4.670296949396188224e-01 4.140048440647509653e-01 2.392012072518971966e-01
4.609095096289907434e-01 4.078358796897652017e-01 2.309089502347338452e-01
4.547601620003904332e-01 4.016973741434559098e-01 2.226435585187958033e-01
4.485682425668168771e-01 3.955931855786054552e-01 2.144296211245620976e-01
4.423342563272031902e-01 3.895240159345254582e-01 2.062390835191822425e-01
4.360296791131279548e-01 3.834992742811840771e-01 1.981208646399195139e-01
4.296458451439821302e-01 3.775230805131955525e-01 1.900605950657147936e-01
4.231474637241839920e-01 3.716087794574343128e-01 1.820957501112273613e-01
4.164868167567738477e-01 3.657737180218046391e-01 1.742972729554080058e-01
4.095973903281860951e-01 3.600427549828760787e-01 1.667592074668843294e-01
4.023838616513769062e-01 3.544471779032521419e-01 1.597054883781360846e-01
3.947814112345453541e-01 3.489995705485997579e-01 1.534550340717726336e-01
3.868766298266183568e-01 3.436490149527227644e-01 1.482247271514051945e-01
3.788659234446942747e-01 3.383142918487055395e-01 1.438533685673206441e-01
3.708789643993818941e-01 3.329519259272508136e-01 1.400354190966675849e-01
3.629572903164399733e-01 3.275541151937279016e-01 1.365461363519130944e-01
3.551085081353937412e-01 3.221231228094754706e-01 1.332624917481727012e-01
3.473247934253401170e-01 3.166654234722757755e-01 1.301119558630651207e-01
3.396071981367110304e-01 3.111828368128046196e-01 1.270354172742856058e-01
3.319394021514575632e-01 3.056815149128923048e-01 1.240384908643338280e-01
3.243282276863680424e-01 3.001605128554065693e-01 1.210640172441826978e-01
3.167622056152286647e-01 2.946237048061004504e-01 1.181174627342822525e-01
3.092325069255885128e-01 2.890737927737177526e-01 1.152025540049777530e-01
3.017324462072743518e-01 2.835140481503690690e-01 1.122895803439803164e-01
2.942527745713884313e-01 2.779485572093680079e-01 1.093519738743563285e-01
2.867898062904237211e-01 2.723775024827345681e-01 1.063941382240281286e-01
2.793462654071084406e-01 2.667988809657529936e-01 1.034093116060303130e-01
2.719194579070843831e-01 2.612124593378840620e-01 1.004000094727539039e-01
2.645072257635520119e-01 2.556177666826844330e-01 9.736769715746559917e-02
2.571104715415328812e-01 2.500132794565106398e-01 9.430849274229180512e-02
2.497278694500691398e-01 2.443981535173898045e-01 9.122222486885092629e-02
2.423563418593929208e-01 2.387720412133358949e-01 8.811155590734742749e-02
2.350006052093227549e-01 2.331320446993035422e-01 8.496581087820043177e-02
2.276523216303719677e-01 2.274793513535450784e-01 8.179625154164907319e-02
2.203179455088669636e-01 2.218103948803600289e-01 7.858901255132699770e-02
2.129902634426870667e-01 2.161258230519468304e-01 7.535305204258735401e-02
2.056701553920205483e-01 2.104236534304168016e-01 7.208351221952713495e-02
1.983573399507774226e-01 2.047021786467214111e-01 6.877728527916648904e-02
1.910498721129175737e-01 1.989600929322370426e-01 6.543378475478747736e-02
1.837444391314300429e-01 1.931963704923697345e-01 6.205437817912465986e-02
1.764424601043844409e-01 1.874084358013911600e-01 5.863230865340122305e-02
1.691401736478186091e-01 1.815951106612115895e-01 5.516899293098562890e-02
1.618355962290499162e-01 1.757545129863619104e-01 5.166248843687707565e-02
1.545273276784393246e-01 1.698844060286744673e-01 4.810942800782175982e-02
1.472136064507999498e-01 1.639824533449356636e-01 4.450649296503813440e-02
1.398915640220567136e-01 1.580463968635001659e-01 4.085151092791106803e-02
1.325571921668454445e-01 1.520740186438245822e-01 3.716809665985362082e-02
1.252084444586500644e-01 1.460622428214301272e-01 3.362553317766520805e-02
1.178384206042670801e-01 1.400089579293633535e-01 3.023931012520724576e-02
1.104210154536264532e-01 1.339166670121798297e-01 2.703939689144004010e-02]; 

   case 'tar' % tarn, the rain anomaly map
      RGB = [8.982325470083904473e-02 1.386884202488073425e-01 5.339634747542102572e-02
9.490477059882479471e-02 1.456373382629968793e-01 5.489581825948507826e-02
9.996820923335580922e-02 1.525451879246892684e-01 5.635425405965687612e-02
1.049932629133125961e-01 1.594260027028096272e-01 5.767943871793178995e-02
1.099844886715972414e-01 1.662815862271170841e-01 5.888156855042592924e-02
1.149326133871393096e-01 1.731199934404946961e-01 5.990900196219744317e-02
1.198471368660654901e-01 1.799400614735769122e-01 6.079328815375505818e-02
1.247299637162895547e-01 1.867439924225139936e-01 6.153299800422255134e-02
1.295805228510759355e-01 1.935349340052088807e-01 6.211473969231569997e-02
1.343996539632437981e-01 2.003151607373700460e-01 6.253114410253807209e-02
1.391929081922307909e-01 2.070845521371987852e-01 6.279848839795282300e-02
1.439554110569007950e-01 2.138477306592140859e-01 6.288007868082229335e-02
1.487348911403759133e-01 2.205914714224685436e-01 6.284098523281120285e-02
1.535498540172954285e-01 2.273157827161538247e-01 6.260141657412987559e-02
1.584099704346516035e-01 2.340203331404673293e-01 6.214047826160282867e-02
1.633293887964821223e-01 2.407033599878570240e-01 6.143905222634648416e-02
1.683325411333271293e-01 2.473603962447930571e-01 6.047333430287041983e-02
1.734516487390746486e-01 2.539840110977820697e-01 5.924078177592304734e-02
1.787290092053028800e-01 2.605653696981221068e-01 5.769398812038967900e-02
1.842226903108559743e-01 2.670914347726459082e-01 5.577103201204060973e-02
1.900442575104325516e-01 2.735335271652731270e-01 5.343902039527659992e-02
1.964012889997470146e-01 2.798383207240314197e-01 5.059091721484253873e-02
2.038180981569160388e-01 2.858582487774559699e-01 4.734796165359243109e-02
2.134803580554236468e-01 2.912137742752328173e-01 4.542335962452376946e-02
2.246379258326957618e-01 2.959985720226946948e-01 4.791967673999841110e-02
2.357078843306291693e-01 3.007243105772285929e-01 5.139927647716215769e-02
2.466644952616644515e-01 3.054215218481723948e-01 5.513668228793693754e-02
2.575461415550749367e-01 3.100881502728962680e-01 5.896503676513652897e-02
2.683585986429155579e-01 3.147286848554897709e-01 6.282164605955981029e-02
2.791275572039774722e-01 3.193399608455813055e-01 6.667246926393996520e-02
2.898358184319489994e-01 3.239318024263264095e-01 7.050980744893517449e-02
3.004984152503225037e-01 3.285032192279036534e-01 7.432550653130615137e-02
3.111553190646883515e-01 3.330438874028457952e-01 7.810231939881484564e-02
3.218034422925408755e-01 3.375573450510525597e-01 8.184304787716339957e-02
3.324333011718812458e-01 3.420495081780127733e-01 8.555280814984053683e-02
3.430769162896195046e-01 3.465110261365142996e-01 8.922112292532158317e-02
3.537299359050021241e-01 3.509453072047716837e-01 9.285241010232825332e-02
3.643894401193512600e-01 3.553552732120655588e-01 9.645119893644621412e-02
3.750453108858616824e-01 3.597467221513847013e-01 1.000256142638818735e-01
3.857100245562226637e-01 3.641166297482588132e-01 1.035734980914473635e-01
3.963858021351523986e-01 3.684657518577320601e-01 1.070973667641381966e-01
4.071213221517259173e-01 3.727752094149718864e-01 1.105826082064781946e-01
4.178758479312351115e-01 3.770629074315066109e-01 1.140472314573759138e-01
4.286423372903069851e-01 3.813332614724057046e-01 1.174974726735356911e-01
4.394433776892578969e-01 3.855777866767184925e-01 1.209275883437508081e-01
4.502962927333878373e-01 3.897896541113713975e-01 1.243344105962937429e-01
4.611581205683302209e-01 3.939890872463367444e-01 1.277374475879362037e-01
4.720563913472620166e-01 3.981648558562365103e-01 1.311281116044327733e-01
4.830210005713382881e-01 4.023038157279973936e-01 1.344963783804615232e-01
4.939986962481786592e-01 4.064316868089741797e-01 1.378665047316342263e-01
5.050443453644228864e-01 4.105232839216421126e-01 1.412177516567701130e-01
5.161305471961733504e-01 4.145923196985767945e-01 1.445638179668259637e-01
5.272226932670311950e-01 4.186568461649191053e-01 1.479227047630213288e-01
5.384231737920478489e-01 4.226667334740543680e-01 1.512491796667267130e-01
5.496735303963581343e-01 4.266509012024514158e-01 1.545811460866533815e-01
5.608924552760781168e-01 4.306494794221744082e-01 1.580220317035593569e-01
5.721888701475891237e-01 4.346087953039819429e-01 1.614907462658664583e-01
5.835602689975579738e-01 4.385299786869343297e-01 1.649946574397075372e-01
5.949141655760096237e-01 4.424610896723609743e-01 1.686236738948057867e-01
6.063474969101482204e-01 4.463528192152232399e-01 1.722929976151095499e-01
6.178322186277155348e-01 4.502196887286237792e-01 1.760393057804460759e-01
6.293072576484797231e-01 4.540947330149664452e-01 1.799322501799610063e-01
6.408487929862468624e-01 4.579376406332680838e-01 1.839101076250921896e-01
6.524478209436700427e-01 4.617530075872406381e-01 1.879952778619465859e-01
6.640840089637001231e-01 4.655519164846439462e-01 1.922225103058495532e-01
6.757077110120256469e-01 4.693624324344119469e-01 1.966573342842017347e-01
6.873619660726970615e-01 4.731606777592323732e-01 2.012786215471898954e-01
6.990381205121832808e-01 4.769509785672518265e-01 2.061259388500985001e-01
7.107469254673900450e-01 4.807250920628522439e-01 2.112467599398395457e-01
7.224230526312156453e-01 4.845218253861356961e-01 2.167123076670600113e-01
7.340988344738968996e-01 4.883178461040553198e-01 2.226023859070605515e-01
7.456126256093577043e-01 4.922084125716105762e-01 2.291293878090427394e-01
7.569950884983198680e-01 4.961693298634363702e-01 2.364236914878575235e-01
7.680178179592376253e-01 5.003309692977765399e-01 2.449057676522546911e-01
7.783432998143535730e-01 5.048913767328344626e-01 2.551145881355684764e-01
7.872937751334541101e-01 5.102708429127015277e-01 2.676604181065562749e-01
7.941838547011116356e-01 5.169436808272552808e-01 2.821513140071189585e-01
7.994756138083272123e-01 5.246737693689591531e-01 2.970255774031809182e-01
8.039010002452274817e-01 5.330024415850749264e-01 3.115266985520254717e-01
8.078274978677594254e-01 5.416811012734407127e-01 3.255924452137653469e-01
8.115188822314253203e-01 5.505337325024400874e-01 3.392264225271887645e-01
8.149565405173153643e-01 5.595497555166275561e-01 3.527260919316282384e-01
8.184200486555966991e-01 5.685470713353225625e-01 3.661765482687758810e-01
8.218871005380722350e-01 5.775395855002405376e-01 3.796232120570560142e-01
8.253524104820136875e-01 5.865320478639367563e-01 3.930813348872184143e-01
8.288195980598317414e-01 5.955244594650943579e-01 4.065541730450397684e-01
8.322893500409468404e-01 6.045186250474157141e-01 4.200438297618293571e-01
8.357658919205435133e-01 6.135146821751165103e-01 4.335463882212042819e-01
8.392529397355924514e-01 6.225129167459599877e-01 4.470604854774085646e-01
8.427490015153409342e-01 6.315159424794672960e-01 4.605928597504284627e-01
8.462593242877498589e-01 6.405233492783437566e-01 4.741391831670908608e-01
8.497829397434528698e-01 6.495374502823044738e-01 4.877048504982988697e-01
8.533243184659659031e-01 6.585581699790050703e-01 5.012868094865198243e-01
8.568819326669031566e-01 6.675878581525517275e-01 5.148924377083944348e-01
8.604592156604728981e-01 6.766270650074718285e-01 5.285174957354723535e-01
8.640624159744237920e-01 6.856750887735367783e-01 5.421555222813166930e-01
8.676874794762717835e-01 6.947353397221285309e-01 5.558159820355237368e-01
8.713433578641075483e-01 7.038060956465443940e-01 5.694881225731345253e-01
8.750269237065777528e-01 7.128903502475765208e-01 5.831797789138876142e-01
8.787417347404306023e-01 7.219884934452005520e-01 5.968884207063740455e-01
8.824929186507559642e-01 7.311003788594004904e-01 6.106091378045375162e-01
8.862780839217170303e-01 7.402286277522096558e-01 6.243484010896452885e-01
8.901025319225768229e-01 7.493730430239593510e-01 6.381009856311857797e-01
8.939696421167129259e-01 7.585341679611918853e-01 6.518645964033017437e-01
8.978784332841047711e-01 7.677140679252760780e-01 6.656434485771917098e-01
9.018326061682566674e-01 7.769131645138506181e-01 6.794347199822515782e-01
9.058380869535479496e-01 7.861311730497033690e-01 6.932323184728660381e-01
9.098936269941250155e-01 7.953702376174893729e-01 7.070408050591605598e-01
9.140018319746877618e-01 8.046311927664194785e-01 7.208589990654702406e-01
9.181667424043119530e-01 8.139144268565462470e-01 7.346836216449236234e-01
9.223937440297114154e-01 8.232199407988282092e-01 7.485094577713424790e-01
9.266329848317796936e-01 8.325649094757640034e-01 7.624119242822527953e-01
9.309587852706298072e-01 8.419294436897262202e-01 7.762663931757451952e-01
9.353308095703787295e-01 8.513275998621367968e-01 7.901346256764255616e-01
9.397666205989465560e-01 8.607552896628867245e-01 8.039940207918337967e-01
9.442852447524148207e-01 8.702076963082445715e-01 8.178194726640143353e-01
9.488564272552421075e-01 8.796932993686320534e-01 8.316532922424306751e-01
9.535132490174164088e-01 8.891984622715306541e-01 8.454417866111457736e-01
9.582183040236339489e-01 8.987227701762496856e-01 8.592166709390754997e-01
9.629681108746700469e-01 9.082408064338201026e-01 8.729337059863773174e-01
9.677369708112718572e-01 9.177061780026050108e-01 8.865286101605559521e-01
9.724470492478461958e-01 9.270421466879262828e-01 8.999251032877214618e-01
9.769577532404692954e-01 9.361234679319003771e-01 9.130099659367507670e-01
9.810936756020608440e-01 9.447503559760432879e-01 9.255607060964068378e-01
9.846187877628681528e-01 9.526495821380999152e-01 9.372697834100743863e-01
9.872037589208276787e-01 9.594994318553785595e-01 9.478188041909386685e-01
9.885417842181484227e-01 9.649430355113016722e-01 9.568049207415391111e-01
9.884013930268252812e-01 9.686440358418687557e-01 9.637998740972807399e-01
9.884162623276403492e-01 9.700501042547021724e-01 9.646078752920302923e-01
9.882481988456427446e-01 9.691949840717196674e-01 9.590849806168054714e-01
9.862202098271878326e-01 9.664497051565109631e-01 9.513022380762837793e-01
9.825317256976796587e-01 9.620307807128735123e-01 9.416222408221766038e-01
9.774953228979595954e-01 9.562506189604750295e-01 9.303982168749544979e-01
9.714512581163280425e-01 9.494527517929901572e-01 9.180386343083645206e-01
9.647328782040082151e-01 9.419601909858620337e-01 9.048410781832560978e-01
9.576029833882070408e-01 9.340364327380904497e-01 8.911071594609687452e-01
9.502540208567341606e-01 9.258747025552572785e-01 8.770597493210310347e-01
9.428203850658012364e-01 9.176036605916164657e-01 8.628094676385467121e-01
9.353802079259059266e-01 9.093000877222664480e-01 8.484369289828109784e-01
9.279731512595491560e-01 9.010054158372772237e-01 8.340022711999641736e-01
9.206162430864484048e-01 8.927397568877493139e-01 8.195491149365338179e-01
9.133281854005529388e-01 8.845128541542209843e-01 8.050384948242300664e-01
9.061049708588290175e-01 8.763271129727117081e-01 7.905141123067558340e-01
8.988722526026153847e-01 8.682095990978259126e-01 7.760244884018141498e-01
8.913604989190128114e-01 8.602593575538733939e-01 7.616936251859220963e-01
8.834947766491649812e-01 8.525076069101691356e-01 7.475141723430621665e-01
8.751348356324997191e-01 8.450011895094009517e-01 7.335968964229397926e-01
8.660362122016208586e-01 8.378139590610478304e-01 7.202229230673017346e-01
8.560628191678395504e-01 8.309783491388803567e-01 7.076282794978067114e-01
8.450850551605781913e-01 8.245041095837670753e-01 6.962540521527236237e-01
8.331237773418240788e-01 8.183393744081108867e-01 6.865239180889655124e-01
8.205576283471390786e-01 8.123351983566962087e-01 6.783182715666855600e-01
8.077229876914296947e-01 8.063801435632871328e-01 6.713153485176399649e-01
7.947461515019460521e-01 8.004373108778479740e-01 6.653252445665102099e-01
7.818234645252460924e-01 7.944593749272963468e-01 6.599830880041622772e-01
7.690043949000203716e-01 7.884424060393956379e-01 6.551042701424648618e-01
7.562981288404810876e-01 7.823918885846237181e-01 6.505794347324633797e-01
7.437166688251096724e-01 7.763107155814537030e-01 6.463185559404119873e-01
7.312864875465606707e-01 7.701962491575811143e-01 6.422364303657139839e-01
7.189850527082707332e-01 7.640604119000086181e-01 6.382908369799579207e-01
7.068142314422918293e-01 7.579064471112740842e-01 6.344344908612202794e-01
6.947742510783049275e-01 7.517375363827888402e-01 6.306282821287318985e-01
6.828537141789142728e-01 7.455586837934410349e-01 6.268596706266046370e-01
6.710011548564486228e-01 7.393836600422335481e-01 6.231736409952126632e-01
6.592657914925743601e-01 7.332038660876558644e-01 6.194650744353706884e-01
6.476362335539276316e-01 7.270240901258274713e-01 6.157213868460473805e-01
6.360362802101585666e-01 7.208613432160450030e-01 6.120373156027114625e-01
6.245080854916185142e-01 7.147082392942302187e-01 6.083234269572357356e-01
6.130488547076046180e-01 7.085668274101213360e-01 6.045591744160895287e-01
6.015892322420162142e-01 7.024496994714828357e-01 6.008489625961975777e-01
5.901987677923722364e-01 6.963445147045469463e-01 5.970628836865409239e-01
5.787942193155654058e-01 6.902683578483006510e-01 5.932973990580240331e-01
5.673843957655613224e-01 6.842194608428608937e-01 5.895220675517639508e-01
5.559759005656629283e-01 6.781954870319548689e-01 5.857224142025272418e-01
5.445452293547278222e-01 6.721991418209628533e-01 5.819309981473995697e-01
5.330284999192810291e-01 6.662471887845208274e-01 5.781518614893066399e-01
5.216118831320554206e-01 6.602860177867492242e-01 5.743450969986969579e-01
5.100461930662858467e-01 6.543801579393034862e-01 5.705892298512411642e-01
4.985462196160774240e-01 6.484680127740434230e-01 5.668522387113605898e-01
4.869212854485653330e-01 6.426004706277155254e-01 5.631503671835994540e-01
4.753175948425168440e-01 6.367342039857054603e-01 5.594833076156262575e-01
4.635884121260610002e-01 6.309044987522948178e-01 5.558810414566710545e-01
4.517937357110975438e-01 6.250933232803836948e-01 5.523287210439038475e-01
4.399781445380013811e-01 6.192864046496112662e-01 5.488272822016192487e-01
4.280176465668665831e-01 6.135117804844451017e-01 5.453889520686707737e-01
4.160613175390713292e-01 6.077287406576958873e-01 5.420126140975655149e-01
4.039857150007630238e-01 6.019638930796089582e-01 5.387130326489297794e-01
3.917475183228279478e-01 5.962226288699170595e-01 5.354998204993742794e-01
3.795396260545044753e-01 5.904576990603345177e-01 5.323562706006824685e-01
3.671980489650730761e-01 5.847026249394290387e-01 5.292995421656493393e-01
3.547169039649492039e-01 5.789540227003182604e-01 5.263324098770649773e-01
3.421991518101316077e-01 5.731858973997910889e-01 5.234530196411122382e-01
3.296321613618491964e-01 5.673978264587494769e-01 5.206643628468334839e-01
3.169539434232791497e-01 5.615982678332219757e-01 5.179739584214747561e-01
3.041848831902358996e-01 5.557790841809171489e-01 5.153814622924314248e-01
2.913580245785495904e-01 5.499302942311713460e-01 5.128866030358065764e-01
2.786210115475045712e-01 5.440237902965463501e-01 5.104770287408624263e-01
2.658767813620477316e-01 5.380746002424073859e-01 5.081635613728290313e-01
2.531728049963842264e-01 5.320730905152252221e-01 5.059419345415738789e-01
2.405801746352321802e-01 5.260084224989675095e-01 5.037958873725133513e-01
2.281731742156575815e-01 5.198710287686101328e-01 5.017115180274238639e-01
2.160357564123031038e-01 5.136521477693153370e-01 4.996732344467323395e-01
2.042553268175122949e-01 5.073441845910725556e-01 4.976709295334373340e-01
1.929455786157068808e-01 5.009402502860922368e-01 4.956743078037426087e-01
1.822159137211172564e-01 4.944360671084672698e-01 4.936576912144489682e-01
1.721720047662203268e-01 4.878296612331804449e-01 4.915973661753361701e-01
1.629121950514905992e-01 4.811217814834132800e-01 4.894661845895681984e-01
1.545167219577190942e-01 4.743156260955230796e-01 4.872413962325574111e-01
1.470425142510033700e-01 4.674166983864977420e-01 4.849028953714263346e-01
1.405113128544070999e-01 4.604338355815827399e-01 4.824271757570474661e-01
1.349217530726786463e-01 4.533748092665628726e-01 4.798101955391982920e-01
1.302797342394179658e-01 4.462455116423036938e-01 4.770426757084247904e-01
1.265939811024292538e-01 4.390496902560914738e-01 4.741166274104288703e-01
1.236843852121248255e-01 4.318074599363258548e-01 4.710274926158040665e-01
1.214646610875383670e-01 4.245259430194671113e-01 4.677892044131184979e-01
1.198403481002758841e-01 4.172122902240388842e-01 4.644123639304529871e-01
1.184609210323855077e-01 4.098925950294425857e-01 4.609615952184937249e-01
1.168754986210283897e-01 4.026022451761210874e-01 4.575160089880971337e-01
1.151007877850314109e-01 3.953385852310392079e-01 4.540805121238136732e-01
1.131413084296974680e-01 3.881005310447923073e-01 4.506558243985225309e-01
1.110221391582094097e-01 3.808834919340156611e-01 4.472532258515609649e-01
1.087511055351796929e-01 3.736854209969548424e-01 4.438760147968137115e-01
1.063262941094024749e-01 3.665057054845847206e-01 4.405230161394063093e-01
1.037537167610996236e-01 3.593424059991345287e-01 4.371963158215718681e-01
1.010396012430455071e-01 3.521934333571229425e-01 4.338979957908608021e-01
9.820785452879465804e-02 3.450535925989220432e-01 4.306390199340546787e-01
9.524886735285809092e-02 3.379231538671472745e-01 4.274136452659025309e-01
9.216708027427711336e-02 3.307999936437661659e-01 4.242225680363120865e-01
8.898719812499519821e-02 3.236783223748895821e-01 4.210773055644454477e-01
8.570498821294705860e-02 3.165572544150943024e-01 4.179737087144154706e-01
8.232861945095013012e-02 3.094336626170069993e-01 4.149138402418848237e-01
7.888091832597513009e-02 3.023016385852106969e-01 4.119079586942359095e-01
7.535426196281005962e-02 2.951606915265346798e-01 4.089483103763945637e-01
7.178216232425585486e-02 2.880027122692483954e-01 4.060506695454764170e-01
6.815855459758968227e-02 2.808269749834076956e-01 4.032054679392391150e-01
6.448898508226078019e-02 2.736310489100942100e-01 4.004060238485768752e-01
6.082018328964860360e-02 2.664035583373736138e-01 3.976774815474886093e-01
5.716124112921574379e-02 2.591406963233486849e-01 3.950170320465338780e-01
5.351466965820251415e-02 2.518414268825251989e-01 3.924068662312853450e-01
4.993186694626653571e-02 2.444935102031051688e-01 3.898697442701583027e-01
4.643877317377426844e-02 2.370912873691644052e-01 3.873999194228605614e-01
4.308214165698193848e-02 2.296247589871482364e-01 3.850043813960570827e-01
3.988193032745831340e-02 2.220895846072232227e-01 3.826648500695101207e-01
3.696588053809187618e-02 2.144711117753294594e-01 3.803951582141073540e-01
3.503963119120004355e-02 2.067382236169432175e-01 3.779948138188178319e-01
3.478421889488789165e-02 1.988679039642712310e-01 3.751470684509792819e-01
3.605446405610013977e-02 1.908764917041337605e-01 3.717394054831296146e-01
3.860378491452313293e-02 1.827911805345073526e-01 3.676697392369102646e-01
4.204158194669544574e-02 1.746484839257387567e-01 3.628639802125444569e-01
4.580769332477269834e-02 1.664868235304900745e-01 3.572935027519454909e-01
4.953046775643197425e-02 1.583391239953714225e-01 3.509783278645146609e-01
5.294794616753940170e-02 1.502287847236943308e-01 3.439780107867805858e-01
5.589746873554810730e-02 1.421689980048106428e-01 3.363784527482869824e-01
5.832364005801801005e-02 1.341588757852459068e-01 3.282715328574966618e-01
6.018929987904993867e-02 1.261942031427702726e-01 3.197538950865533303e-01
6.152266558088163168e-02 1.182670459661474271e-01 3.108793933905304452e-01];
      
   otherwise 
      error('Unrecognized colormap name.') 
     

end



function varargout = colorspace(Conversion,varargin)
%COLORSPACE  Transform a color image between color representations.
%   B = COLORSPACE(S,A) transforms the color representation of image A
%   where S is a string specifying the conversion.  The input array A 
%   should be a real full double array of size Mx3 or MxNx3.  The output B 
%   is the same size as A.
%
%   S tells the source and destination color spaces, S = 'dest<-src', or 
%   alternatively, S = 'src->dest'.  Supported color spaces are
%
%     'RGB'              sRGB IEC 61966-2-1
%     'YCbCr'            Luma + Chroma ("digitized" version of Y'PbPr)
%     'JPEG-YCbCr'       Luma + Chroma space used in JFIF JPEG
%     'YDbDr'            SECAM Y'DbDr Luma + Chroma
%     'YPbPr'            Luma (ITU-R BT.601) + Chroma 
%     'YUV'              NTSC PAL Y'UV Luma + Chroma
%     'YIQ'              NTSC Y'IQ Luma + Chroma
%     'HSV' or 'HSB'     Hue Saturation Value/Brightness
%     'HSL' or 'HLS'     Hue Saturation Luminance
%     'HSI'              Hue Saturation Intensity
%     'XYZ'              CIE 1931 XYZ
%     'Lab'              CIE 1976 L*a*b* (CIELAB)
%     'Luv'              CIE L*u*v* (CIELUV)
%     'LCH'              CIE L*C*H* (CIELCH)
%     'CAT02 LMS'        CIE CAT02 LMS
%
%  All conversions assume 2 degree observer and D65 illuminant.
%
%  Color space names are case insensitive and spaces are ignored.  When 
%  sRGB is the source or destination, it can be omitted. For example 
%  'yuv<-' is short for 'yuv<-rgb'.
%
%  For sRGB, the values should be scaled between 0 and 1.  Beware that 
%  transformations generally do not constrain colors to be "in gamut."  
%  Particularly, transforming from another space to sRGB may obtain 
%  R'G'B' values outside of the [0,1] range.  So the result should be 
%  clamped to [0,1] before displaying:
%     image(min(max(B,0),1));  % Clamp B to [0,1] and display
%
%  sRGB (Red Green Blue) is the (ITU-R BT.709 gamma-corrected) standard
%  red-green-blue representation of colors used in digital imaging.  The 
%  components should be scaled between 0 and 1.  The space can be 
%  visualized geometrically as a cube.
%  
%  Y'PbPr, Y'CbCr, Y'DbDr, Y'UV, and Y'IQ are related to sRGB by linear
%  transformations.  These spaces separate a color into a grayscale
%  luminance component Y and two chroma components.  The valid ranges of
%  the components depends on the space.
%
%  HSV (Hue Saturation Value) is related to sRGB by
%     H = hexagonal hue angle   (0 <= H < 360),
%     S = C/V                   (0 <= S <= 1),
%     V = max(R',G',B')         (0 <= V <= 1),
%  where C = max(R',G',B') - min(R',G',B').  The hue angle H is computed on
%  a hexagon.  The space is geometrically a hexagonal cone.
%
%  HSL (Hue Saturation Lightness) is related to sRGB by
%     H = hexagonal hue angle                (0 <= H < 360),
%     S = C/(1 - |2L-1|)                     (0 <= S <= 1),
%     L = (max(R',G',B') + min(R',G',B'))/2  (0 <= L <= 1),
%  where H and C are the same as in HSV.  Geometrically, the space is a
%  double hexagonal cone.
%
%  HSI (Hue Saturation Intensity) is related to sRGB by
%     H = polar hue angle        (0 <= H < 360),
%     S = 1 - min(R',G',B')/I    (0 <= S <= 1),
%     I = (R'+G'+B')/3           (0 <= I <= 1).
%  Unlike HSV and HSL, the hue angle H is computed on a circle rather than
%  a hexagon. 
%
%  CIE XYZ is related to sRGB by inverse gamma correction followed by a
%  linear transform.  Other CIE color spaces are defined relative to XYZ.
%
%  CIE L*a*b*, L*u*v*, and L*C*H* are nonlinear functions of XYZ.  The L*
%  component is designed to match closely with human perception of
%  lightness.  The other two components describe the chroma.
%
%  CIE CAT02 LMS is the linear transformation of XYZ using the MCAT02 
%  chromatic adaptation matrix.  The space is designed to model the 
%  response of the three types of cones in the human eye, where L, M, S,
%  correspond respectively to red ("long"), green ("medium"), and blue
%  ("short").

% Pascal Getreuer 2005-2010


%%% Input parsing %%%
if nargin < 2, error('Not enough input arguments.'); end
[SrcSpace,DestSpace] = parse(Conversion);

if nargin == 2
   Image = varargin{1};
elseif nargin >= 3
   Image = cat(3,varargin{:});
else
   error('Invalid number of input arguments.');
end

FlipDims = (size(Image,3) == 1);

if FlipDims, Image = permute(Image,[1,3,2]); end
if ~isa(Image,'double'), Image = double(Image)/255; end
if size(Image,3) ~= 3, error('Invalid input size.'); end

SrcT = gettransform(SrcSpace);
DestT = gettransform(DestSpace);

if ~ischar(SrcT) && ~ischar(DestT)
   % Both source and destination transforms are affine, so they
   % can be composed into one affine operation
   T = [DestT(:,1:3)*SrcT(:,1:3),DestT(:,1:3)*SrcT(:,4)+DestT(:,4)];      
   Temp = zeros(size(Image));
   Temp(:,:,1) = T(1)*Image(:,:,1) + T(4)*Image(:,:,2) + T(7)*Image(:,:,3) + T(10);
   Temp(:,:,2) = T(2)*Image(:,:,1) + T(5)*Image(:,:,2) + T(8)*Image(:,:,3) + T(11);
   Temp(:,:,3) = T(3)*Image(:,:,1) + T(6)*Image(:,:,2) + T(9)*Image(:,:,3) + T(12);
   Image = Temp;
elseif ~ischar(DestT)
   Image = rgb(Image,SrcSpace);
   Temp = zeros(size(Image));
   Temp(:,:,1) = DestT(1)*Image(:,:,1) + DestT(4)*Image(:,:,2) + DestT(7)*Image(:,:,3) + DestT(10);
   Temp(:,:,2) = DestT(2)*Image(:,:,1) + DestT(5)*Image(:,:,2) + DestT(8)*Image(:,:,3) + DestT(11);
   Temp(:,:,3) = DestT(3)*Image(:,:,1) + DestT(6)*Image(:,:,2) + DestT(9)*Image(:,:,3) + DestT(12);
   Image = Temp;
else
   Image = feval(DestT,Image,SrcSpace);
end

%%% Output format %%%
if nargout > 1
   varargout = {Image(:,:,1),Image(:,:,2),Image(:,:,3)};
else
   if FlipDims, Image = permute(Image,[1,3,2]); end
   varargout = {Image};
end

return;


function [SrcSpace,DestSpace] = parse(Str)
% Parse conversion argument

if ischar(Str)
   Str = lower(strrep(strrep(Str,'-',''),'=',''));
   k = find(Str == '>');
   
   if length(k) == 1         % Interpret the form 'src->dest'
      SrcSpace = Str(1:k-1);
      DestSpace = Str(k+1:end);
   else
      k = find(Str == '<');
      
      if length(k) == 1      % Interpret the form 'dest<-src'
         DestSpace = Str(1:k-1);
         SrcSpace = Str(k+1:end);
      else
         error(['Invalid conversion, ''',Str,'''.']);
      end   
   end
   
   SrcSpace = alias(SrcSpace);
   DestSpace = alias(DestSpace);
else
   SrcSpace = 1;             % No source pre-transform
   DestSpace = Conversion;
   if any(size(Conversion) ~= 3), error('Transformation matrix must be 3x3.'); end
end
return;


function Space = alias(Space)
Space = strrep(strrep(Space,'cie',''),' ','');

if isempty(Space)
   Space = 'rgb';
end

switch Space
case {'ycbcr','ycc'}
   Space = 'ycbcr';
case {'hsv','hsb'}
   Space = 'hsv';
case {'hsl','hsi','hls'}
   Space = 'hsl';
case {'rgb','yuv','yiq','ydbdr','ycbcr','jpegycbcr','xyz','lab','luv','lch'}
   return;
end
return;


function T = gettransform(Space)
% Get a colorspace transform: either a matrix describing an affine transform,
% or a string referring to a conversion subroutine
switch Space
case 'ypbpr'
   T = [0.299,0.587,0.114,0;-0.1687367,-0.331264,0.5,0;0.5,-0.418688,-0.081312,0];
case 'yuv'
   % sRGB to NTSC/PAL YUV
   % Wikipedia: http://en.wikipedia.org/wiki/YUV
   T = [0.299,0.587,0.114,0;-0.147,-0.289,0.436,0;0.615,-0.515,-0.100,0];
case 'ydbdr'
   % sRGB to SECAM YDbDr
   % Wikipedia: http://en.wikipedia.org/wiki/YDbDr
   T = [0.299,0.587,0.114,0;-0.450,-0.883,1.333,0;-1.333,1.116,0.217,0];
case 'yiq'
   % sRGB in [0,1] to NTSC YIQ in [0,1];[-0.595716,0.595716];[-0.522591,0.522591];
   % Wikipedia: http://en.wikipedia.org/wiki/YIQ
   T = [0.299,0.587,0.114,0;0.595716,-0.274453,-0.321263,0;0.211456,-0.522591,0.311135,0];
case 'ycbcr'
   % sRGB (range [0,1]) to ITU-R BRT.601 (CCIR 601) Y'CbCr
   % Wikipedia: http://en.wikipedia.org/wiki/YCbCr
   % Poynton, Equation 3, scaling of R'G'B to Y'PbPr conversion
   T = [65.481,128.553,24.966,16;-37.797,-74.203,112.0,128;112.0,-93.786,-18.214,128];
case 'jpegycbcr'
   % Wikipedia: http://en.wikipedia.org/wiki/YCbCr
   T = [0.299,0.587,0.114,0;-0.168736,-0.331264,0.5,0.5;0.5,-0.418688,-0.081312,0.5]*255;
case {'rgb','xyz','hsv','hsl','lab','luv','lch','cat02lms'}
   T = Space;
otherwise
   error(['Unknown color space, ''',Space,'''.']);
end
return;


function Image = rgb(Image,SrcSpace)
% Convert to sRGB from 'SrcSpace'
switch SrcSpace
case 'rgb'
   return;
case 'hsv'
   % Convert HSV to sRGB
   Image = huetorgb((1 - Image(:,:,2)).*Image(:,:,3),Image(:,:,3),Image(:,:,1));
case 'hsl'
   % Convert HSL to sRGB
   L = Image(:,:,3);
   Delta = Image(:,:,2).*min(L,1-L);
   Image = huetorgb(L-Delta,L+Delta,Image(:,:,1));
case {'xyz','lab','luv','lch','cat02lms'}
   % Convert to CIE XYZ
   Image = xyz(Image,SrcSpace);
   % Convert XYZ to RGB
   T = [3.2406, -1.5372, -0.4986; -0.9689, 1.8758, 0.0415; 0.0557, -0.2040, 1.057];
   R = T(1)*Image(:,:,1) + T(4)*Image(:,:,2) + T(7)*Image(:,:,3);  % R
   G = T(2)*Image(:,:,1) + T(5)*Image(:,:,2) + T(8)*Image(:,:,3);  % G
   B = T(3)*Image(:,:,1) + T(6)*Image(:,:,2) + T(9)*Image(:,:,3);  % B
   % Desaturate and rescale to constrain resulting RGB values to [0,1]   
   AddWhite = -min(min(min(R,G),B),0);
   R = R + AddWhite;
   G = G + AddWhite;
   B = B + AddWhite;
   % Apply gamma correction to convert linear RGB to sRGB
   Image(:,:,1) = gammacorrection(R);  % R'
   Image(:,:,2) = gammacorrection(G);  % G'
   Image(:,:,3) = gammacorrection(B);  % B'
otherwise  % Conversion is through an affine transform
   T = gettransform(SrcSpace);
   temp = inv(T(:,1:3));
   T = [temp,-temp*T(:,4)];
   R = T(1)*Image(:,:,1) + T(4)*Image(:,:,2) + T(7)*Image(:,:,3) + T(10);
   G = T(2)*Image(:,:,1) + T(5)*Image(:,:,2) + T(8)*Image(:,:,3) + T(11);
   B = T(3)*Image(:,:,1) + T(6)*Image(:,:,2) + T(9)*Image(:,:,3) + T(12);
   Image(:,:,1) = R;
   Image(:,:,2) = G;
   Image(:,:,3) = B;
end

% Clip to [0,1]
Image = min(max(Image,0),1);
return;


function Image = xyz(Image,SrcSpace)
% Convert to CIE XYZ from 'SrcSpace'
WhitePoint = [0.950456,1,1.088754];  

switch SrcSpace
case 'xyz'
   return;
case 'luv'
   % Convert CIE L*uv to XYZ
   WhitePointU = (4*WhitePoint(1))./(WhitePoint(1) + 15*WhitePoint(2) + 3*WhitePoint(3));
   WhitePointV = (9*WhitePoint(2))./(WhitePoint(1) + 15*WhitePoint(2) + 3*WhitePoint(3));
   L = Image(:,:,1);
   Y = (L + 16)/116;
   Y = invf(Y)*WhitePoint(2);
   U = Image(:,:,2)./(13*L + 1e-6*(L==0)) + WhitePointU;
   V = Image(:,:,3)./(13*L + 1e-6*(L==0)) + WhitePointV;
   Image(:,:,1) = -(9*Y.*U)./((U-4).*V - U.*V);                  % X
   Image(:,:,2) = Y;                                             % Y
   Image(:,:,3) = (9*Y - (15*V.*Y) - (V.*Image(:,:,1)))./(3*V);  % Z
case {'lab','lch'}
   Image = lab(Image,SrcSpace);
   % Convert CIE L*ab to XYZ
   fY = (Image(:,:,1) + 16)/116;
   fX = fY + Image(:,:,2)/500;
   fZ = fY - Image(:,:,3)/200;
   Image(:,:,1) = WhitePoint(1)*invf(fX);  % X
   Image(:,:,2) = WhitePoint(2)*invf(fY);  % Y
   Image(:,:,3) = WhitePoint(3)*invf(fZ);  % Z
case 'cat02lms'
    % Convert CAT02 LMS to XYZ
   T = inv([0.7328, 0.4296, -0.1624;-0.7036, 1.6975, 0.0061; 0.0030, 0.0136, 0.9834]);
   L = Image(:,:,1);
   M = Image(:,:,2);
   S = Image(:,:,3);
   Image(:,:,1) = T(1)*L + T(4)*M + T(7)*S;  % X 
   Image(:,:,2) = T(2)*L + T(5)*M + T(8)*S;  % Y
   Image(:,:,3) = T(3)*L + T(6)*M + T(9)*S;  % Z
otherwise   % Convert from some gamma-corrected space
   % Convert to sRGB
   Image = rgb(Image,SrcSpace);
   % Undo gamma correction
   R = invgammacorrection(Image(:,:,1));
   G = invgammacorrection(Image(:,:,2));
   B = invgammacorrection(Image(:,:,3));
   % Convert RGB to XYZ
   T = inv([3.2406, -1.5372, -0.4986; -0.9689, 1.8758, 0.0415; 0.0557, -0.2040, 1.057]);
   Image(:,:,1) = T(1)*R + T(4)*G + T(7)*B;  % X 
   Image(:,:,2) = T(2)*R + T(5)*G + T(8)*B;  % Y
   Image(:,:,3) = T(3)*R + T(6)*G + T(9)*B;  % Z
end
return;


function Image = hsv(Image,SrcSpace)
% Convert to HSV
Image = rgb(Image,SrcSpace);
V = max(Image,[],3);
S = (V - min(Image,[],3))./(V + (V == 0));
Image(:,:,1) = rgbtohue(Image);
Image(:,:,2) = S;
Image(:,:,3) = V;
return;


function Image = hsl(Image,SrcSpace)
% Convert to HSL 
switch SrcSpace
case 'hsv'
   % Convert HSV to HSL   
   MaxVal = Image(:,:,3);
   MinVal = (1 - Image(:,:,2)).*MaxVal;
   L = 0.5*(MaxVal + MinVal);
   temp = min(L,1-L);
   Image(:,:,2) = 0.5*(MaxVal - MinVal)./(temp + (temp == 0));
   Image(:,:,3) = L;
otherwise
   Image = rgb(Image,SrcSpace);  % Convert to sRGB
   % Convert sRGB to HSL
   MinVal = min(Image,[],3);
   MaxVal = max(Image,[],3);
   L = 0.5*(MaxVal + MinVal);
   temp = min(L,1-L);
   S = 0.5*(MaxVal - MinVal)./(temp + (temp == 0));
   Image(:,:,1) = rgbtohue(Image);
   Image(:,:,2) = S;
   Image(:,:,3) = L;
end
return;


function Image = lab(Image,SrcSpace)
% Convert to CIE L*a*b* (CIELAB)
WhitePoint = [0.950456,1,1.088754];

switch SrcSpace
case 'lab'
   return;
case 'lch'
   % Convert CIE L*CH to CIE L*ab
   C = Image(:,:,2);
   Image(:,:,2) = cos(Image(:,:,3)*pi/180).*C;  % a*
   Image(:,:,3) = sin(Image(:,:,3)*pi/180).*C;  % b*
otherwise
   Image = xyz(Image,SrcSpace);  % Convert to XYZ
   % Convert XYZ to CIE L*a*b*
   X = Image(:,:,1)/WhitePoint(1);
   Y = Image(:,:,2)/WhitePoint(2);
   Z = Image(:,:,3)/WhitePoint(3);
   fX = f(X);
   fY = f(Y);
   fZ = f(Z);
   Image(:,:,1) = 116*fY - 16;    % L*
   Image(:,:,2) = 500*(fX - fY);  % a*
   Image(:,:,3) = 200*(fY - fZ);  % b*
end
return;


function Image = luv(Image,SrcSpace)
% Convert to CIE L*u*v* (CIELUV)
WhitePoint = [0.950456,1,1.088754];
WhitePointU = (4*WhitePoint(1))./(WhitePoint(1) + 15*WhitePoint(2) + 3*WhitePoint(3));
WhitePointV = (9*WhitePoint(2))./(WhitePoint(1) + 15*WhitePoint(2) + 3*WhitePoint(3));

Image = xyz(Image,SrcSpace); % Convert to XYZ
Denom = Image(:,:,1) + 15*Image(:,:,2) + 3*Image(:,:,3);
U = (4*Image(:,:,1))./(Denom + (Denom == 0));
V = (9*Image(:,:,2))./(Denom + (Denom == 0));
Y = Image(:,:,2)/WhitePoint(2);
L = 116*f(Y) - 16;
Image(:,:,1) = L;                        % L*
Image(:,:,2) = 13*L.*(U - WhitePointU);  % u*
Image(:,:,3) = 13*L.*(V - WhitePointV);  % v*
return;  


function Image = lch(Image,SrcSpace)
% Convert to CIE L*ch
Image = lab(Image,SrcSpace);  % Convert to CIE L*ab
H = atan2(Image(:,:,3),Image(:,:,2));
H = H*180/pi + 360*(H < 0);
Image(:,:,2) = sqrt(Image(:,:,2).^2 + Image(:,:,3).^2);  % C
Image(:,:,3) = H;                                        % H
return;


function Image = cat02lms(Image,SrcSpace)
% Convert to CAT02 LMS
Image = xyz(Image,SrcSpace);
T = [0.7328, 0.4296, -0.1624;-0.7036, 1.6975, 0.0061; 0.0030, 0.0136, 0.9834];
X = Image(:,:,1);
Y = Image(:,:,2);
Z = Image(:,:,3);
Image(:,:,1) = T(1)*X + T(4)*Y + T(7)*Z;  % L
Image(:,:,2) = T(2)*X + T(5)*Y + T(8)*Z;  % M
Image(:,:,3) = T(3)*X + T(6)*Y + T(9)*Z;  % S
return;


function Image = huetorgb(m0,m2,H)
% Convert HSV or HSL hue to RGB
N = size(H);
H = min(max(H(:),0),360)/60;
m0 = m0(:);
m2 = m2(:);
F = H - round(H/2)*2;
M = [m0, m0 + (m2-m0).*abs(F), m2];
Num = length(m0);
j = [2 1 0;1 2 0;0 2 1;0 1 2;1 0 2;2 0 1;2 1 0]*Num;
k = floor(H) + 1;
Image = reshape([M(j(k,1)+(1:Num).'),M(j(k,2)+(1:Num).'),M(j(k,3)+(1:Num).')],[N,3]);
return;


function H = rgbtohue(Image)
% Convert RGB to HSV or HSL hue
[M,i] = sort(Image,3);
i = i(:,:,3);
Delta = M(:,:,3) - M(:,:,1);
Delta = Delta + (Delta == 0);
R = Image(:,:,1);
G = Image(:,:,2);
B = Image(:,:,3);
H = zeros(size(R));
k = (i == 1);
H(k) = (G(k) - B(k))./Delta(k);
k = (i == 2);
H(k) = 2 + (B(k) - R(k))./Delta(k);
k = (i == 3);
H(k) = 4 + (R(k) - G(k))./Delta(k);
H = 60*H + 360*(H < 0);
H(Delta == 0) = nan;
return;


function Rp = gammacorrection(R)
Rp = zeros(size(R));
i = (R <= 0.0031306684425005883);
Rp(i) = 12.92*R(i);
Rp(~i) = real(1.055*R(~i).^0.416666666666666667 - 0.055);
return;


function R = invgammacorrection(Rp)
R = zeros(size(Rp));
i = (Rp <= 0.0404482362771076);
R(i) = Rp(i)/12.92;
R(~i) = real(((Rp(~i) + 0.055)/1.055).^2.4);
return;


function fY = f(Y)
fY = real(Y.^(1/3));
i = (Y < 0.008856);
fY(i) = Y(i)*(841/108) + (4/29);
return;


function Y = invf(fY)
Y = fY.^3;
i = (Y < 0.008856);
Y(i) = (fY(i) - 4/29)*(108/841);
return;
