// log_harvest.C
// 
// Copyright 1996-2001 Per Abrahamsen and Sren Hansen
// Copyright 2000-2001 KVL.
//
// This file is part of Daisy.
// 
// Daisy is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser Public License as published by
// the Free Software Foundation; either version 2.1 of the License, or
// (at your option) any later version.
// 
// Daisy is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser Public License for more details.
// 
// You should have received a copy of the GNU Lesser Public License
// along with Daisy; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#define BUILD_DLL
#include "log.h"
#include "daisy.h"
#include "harvest.h"
#include "dlf.h"
#include "vcheck.h"
#include "version.h"
#include "assertion.h"
#include "librarian.h"
#include "treelog.h"
#include "frame.h"
#include <sstream>
#include <fstream>
#include <time.h>

struct LogHarvest : public Log
{
  // Filter function.
  bool check_leaf (symbol) const
  { return false; }
  bool check_interior (symbol) const
  { return false; }
  bool check_derived (symbol, symbol, const symbol) const
  { return false; }

  // Content.
  unsigned int last_size;
  const symbol file;            // Filename.
  std::ofstream out;			// Output stream.
  DLF print_header;		// How much header should be printed?
  bool print_tags;		// Set if tags should be printed.
  bool print_dimension;		// Set if dimensions should be printed.
  const bool print_N;		// Set if nitrogen content should be printed.
  const bool print_C;		// Set if carbon content should be printed.

  // Checking to see if we should log this time step.
  bool match (const Daisy& daisy, Treelog&)
  {
    print_header.finish (out, metalib (), daisy.frame ());

    if (print_tags)
      {
	out << "year\tmonth\tday\tcolumn\tcrop"
	    << "\tstem_DM\tdead_DM\tleaf_DM\tsorg_DM";
	if (print_N)
	  out << "\tstem_N\tdead_N\tleaf_N\tsorg_N";
	if (print_C)
	  out << "\tstem_C\tdead_C\tleaf_C\tsorg_C";
	out << "\tWStress\tNStress\tWP_ET\n";
	print_tags = false;
      }
    if (print_dimension)
      {
	out << "\t\t\t\t"
	    << "\tMg DM/ha\tMg DM/ha\tMg DM/ha\tMg DM/ha";
	if (print_N)
	  out << "\tkg N/ha\tkg N/ha\tkg N/ha\tkg N/ha";
	if (print_C)
	  out << "\tkg C/ha\tkg C/ha\tkg C/ha\tkg C/ha";
	out << "\td\td\tkg/m^3\n";
	print_dimension = false;
      }
    for (; last_size < daisy.harvest ().size (); last_size++)
      {
	const Harvest& harvest = *(daisy.harvest ()[last_size]);
	out << harvest.time.year ()
	    << "\t" << harvest.time.month ()
	    << "\t" << harvest.time.mday ()
	    << "\t" << harvest.column
	    << "\t" << harvest.crop
	    << "\t" << harvest.stem_DM * 0.01
	    << "\t" << harvest.dead_DM * 0.01
	    << "\t" << harvest.leaf_DM * 0.01
	    << "\t" << harvest.sorg_DM * 0.01;
	if (print_N)
	  out << "\t" << harvest.stem_N * 10.0
	      << "\t" << harvest.dead_N * 10.0
	      << "\t" << harvest.leaf_N * 10.0
	      << "\t" << harvest.sorg_N * 10.0;
	if (print_C)
	  out << "\t" << harvest.stem_C * 10.0
	      << "\t" << harvest.dead_C * 10.0
	      << "\t" << harvest.leaf_C * 10.0
	      << "\t" << harvest.sorg_C * 10.0;
	out << "\t" << harvest.water_stress_days
            << "\t" << harvest.nitrogen_stress_days
            << "\t" << harvest.water_productivity
            << "\n";
	out.flush ();
      }
    return false;
  }

  void done (const std::vector<Time::component_t>& time_columns,
	     const Time&, const double, Treelog&)
  { daisy_notreached (); }

  bool initial_match (const Daisy&, const Time& previous, Treelog&)
  { return false; }
  void initial_done (const std::vector<Time::component_t>& time_columns,
		     const Time&, Treelog&)
  { daisy_notreached (); }

  // Normal items.
  void open (symbol)
  { daisy_notreached (); }
  void close ()
  { daisy_notreached (); }

  // Unnamed items.
  void open_unnamed ()
  { daisy_notreached (); }
  void close_unnamed ()
  { daisy_notreached (); }

  // Derived items.
  void open_derived (symbol, symbol, const symbol)
  { daisy_notreached (); }
  void close_derived ()
  { daisy_notreached (); }

  // Derived items with their own alist
  void open_object (symbol, symbol, const Frame&, const symbol)
  { daisy_notreached (); }
  void close_object ()
  { daisy_notreached (); }

  // Derived items in a list.
  void open_entry (symbol, const Frame&, const symbol)
  { daisy_notreached (); }
  void close_entry ()
  { daisy_notreached (); }

  // Named derived items in a list.
  void open_named_entry (symbol, symbol, const Frame&)
  { daisy_notreached (); }
  void close_named_entry ()
  { daisy_notreached (); }

  // Named object
  void open_shallow (symbol, const symbol)
  { daisy_notreached (); }
  void close_shallow ()
  { daisy_notreached (); }

  void output_entry (symbol, bool)
  { }
  void output_entry (symbol, double)
  { }
  void output_entry (symbol, int)
  { }
  void output_entry (symbol, symbol)
  { }
  void output_entry (symbol, const std::vector<double>&)
  { }
  void output_entry (symbol, const PLF&)
  { }

  // Create and Destroy.
  void initialize (const symbol log_dir, Treelog&)
  { 
    const std::string fn = log_dir.name () + file.name ();
    out.open (fn.c_str ()); 

    // Header.
    print_header.start (out, objid, file, "");
    out.flush ();
  }

  bool check (const Border&, Treelog& msg) const
  { 
    TREELOG_MODEL (msg);
    bool ok = true;
    if (!out.good ())
      {
	std::ostringstream tmp;
	tmp << "Write error for '" << file << "'";
	msg.error (tmp.str ());
	ok = false;
      }
    return ok; 
  }

  LogHarvest (const BlockModel& al)
    : Log (al),
      last_size (0),
      file (al.name ("where")),
      print_header (al.name ("print_header")),
      print_tags (al.flag ("print_tags")),
      print_dimension (al.flag ("print_dimension")),
      print_N (al.flag ("print_N")),
      print_C (al.flag ("print_C"))
  { }

  ~LogHarvest ()
  {
    if (!out.good ())
      Assertion::error  ("Problems writing to '" + file + "'");
  }
};

static struct LogHarvestSyntax : public DeclareModel
{
  Model* make (const BlockModel& al) const
  { return new LogHarvest (al); }

  LogHarvestSyntax ()
    : DeclareModel (Log::component, "harvest", "Create a log of all harvests.")
  { }
  void load_frame (Frame& frame) const
  {  
    frame.declare_string ("where", Attribute::Const,
		"Name of the log file to create.");
    frame.set ("where", "harvest.dlf");
    frame.declare_string ("print_header", Attribute::Const,
                "If this is set to 'false', no header is printed.\n\
If this is set to 'true', a full header is printer.\n\
If this is set to 'fixed', a small fixed size header is printed.");
    static VCheck::Enum check_header ("false", "true", "fixed");
    frame.set_check ("print_header", check_header);
    frame.set ("print_header", "true");
    frame.declare_boolean ("print_tags", Attribute::Const,
		"Print a tag line in the file.");
    frame.set ("print_tags", true);
    frame.declare_boolean ("print_dimension", Attribute::Const,
		"Print a line with units after the tag line.");
    frame.declare_boolean ("print_N", Attribute::Const,
		"Print nitrogen content of harvest.");
    frame.set ("print_N", true);
    frame.declare_boolean ("print_C", Attribute::Const,
		"Print carbon content of harvest.");
    frame.set ("print_C", false);
    frame.set ("print_dimension", true);
  }
} LogHarvest_syntax;

