// =============================================================================
// Internal code for the CMB8.2 application.                           -TCoulter

unit CMB82Internals;

interface

uses
  Classes,
  Windows,
  SysUtils,
  Messages,
  Controls,
  Dialogs;

const
  // Public constants.
  CMB_VERSION_NUMBER      = '04xxx'; // Not used ...

  ERROR_MEMORY_ALLOC      = -1;
  ERROR_FILE_OPEN         = -2;
  ERROR_SPECIES_SET       = -3;
  ERROR_SOURCES_SET       = -4;
  ERROR_CMBSVD_1          = -5; // TC shut down 11/21/04; this error is handled in MainForm.pas' func VerifyVerifyCMBRunInputsValid
  ERROR_CMBSVD_2          = -6;
//  ERROR_CMBSVD_3          = -7;   *** this ERROR is never set in the DLL!
  ERROR_CMBSVD_4          = -8;
  ERROR_CMBSVD_5          = -9;
  ERROR_CMBSVD_6          = -10; // iterations maxed out
  ERROR_FILE_READ         = -11;
  ERROR_MISC              = -12;
  ERROR_FILE_CLOSE        = -13;
  ERROR_FILE_WRITE        = -14;
  AMBIENT_SET_ERROR       = -15;
  PROFILE_SET_ERROR       = -16;
//  ERROR_READ_SPECIES_SEL  = -17; // Vacant
//  ERROR_READ_SOURCES_SEL  = -18; // Vacant
//  ERROR_READ_AMBDATA_SEL  = -19; // Vacant
//  ERROR_READ_AMBDATA      = -20; // Vacant
  ERROR_READ_PROFILE      = -21;   // TC activated 11/06/04
  ERROR_MEMORY_FREE       = -22;
  NUM_ERRORS              =  22;

  LENSIZE                 = 6; // Conforms with 6-char size frac. field; critical to eliminate the Runtime error 216 @... on shutdown
  MAXSIZES                = 4;
  SIZEPOS                 = 28;
  lenSPcode               = 24;
  lenPRcode               = 24;

  lenADcode               = 42; // 1 space between strings; TC expanded from 41 on 11/07/04 to accomodate 6-char size frac. field
  AMBFLAGPOS              = 35; // counting from 0, hence 35 is equivalent to an actual position of 36 for * in ADcode

  WriteDT2_TXT            = 1;
  WriteDT2_CSV            = 2;
//  WriteDT2_DBF            = 3; // removed 10.02.03 -TC
//  WriteDT2_WKS            = 4; // removed 10.01.03 -TC

  DIMSHOWFITUNITS         = 5;

  OUTBUFFIT               = 1;
  OUTBUFSSCONT            = 2;
//  OUTBUFPCONT             = 3;
  OUTBUFPROFILE           = 4;
  OUTBUFRECEPTOR          = 5;
  OUTBUFMPIN              = 6;

  LENDEFSEL               = 19;
  LENDEFSEL1              = 16;
  LENDEFSEL2              = 35;

  FILETYPE_INVALID        = -1;
  FILETYPE_TXT            =  0;
  FILETYPE_CSV            =  1;
  FILETYPE_DBF            =  2;
  FILETYPE_WKS            =  3;
  FILETYPE_CAR            =  4;

  DIMCURRENTSAMPLE        = 33;
  MAXOUTPUTS              = 25;

type
// 03/2003 - JS, added for handling presentation of SCEs
  TSCEState = (stApproachingSCEs, stNotSCEs,
                      stJustCompletedSCEs, stSCEs);
  TCMB82c = function(
    arg0,
    arg1,
    arg2,
    arg3,
    arg4,
    arg5,
    arg6,
    arg7,
    arg8,
    arg9,
    arg10,
    arg11,
    arg12: Pointer): Integer; stdcall;

var
  // Public globals.
  g_CMB82c:                 TCMB82c;
  g_ContributionsShownFlag: Boolean; // records that source contributions have been shown since last fit
  g_BestFitFlag:            Boolean;
  g_mPINShownFlag:          Boolean; // records that mPIN matrix has been shown since last fit
  g_DisplayDecimalPlaces:   Integer;
  g_nSPtot:                 Integer;
  g_ADCode:                 PChar;
  g_PRCodeUser:             PChar;
  g_SPCodeUser:             PChar;
  g_CMB8ExeDir:             string;  // directory containing executable at Startup

  g_CMB82initOK : Boolean; // 06/03/03 - JS added
  g_Header: string;        // Added 10/23/03

  // Public function and procedure declarations.
  procedure SetMeasurementType(const MeasurementType: string);
  procedure UpdateADCode(Star: Char);
  procedure UpdateSPCodeUser(Star: Char);
  procedure UpdatePRCodeUser(Star: Char);
  procedure LoadTalkMem;
  procedure InitCMB82;
  procedure TerminateCMB82;  // 2003 added.
  procedure ShowErrorMessage(const ErrorMessage: string);
  function LookupEInOutErrorName (const aErrCode: integer): string; // Added 9/6/03 to improve I/O error handling.

  //-js 091104 - added new param, so code 'knows' when it's at the first record
  function WriteSFit2TempBuffer (const aAtFirstRecord: Boolean): Integer;
  function  DoBestFit: Integer;
  // 6-10-03 added by TC;  SetBatchFlag is called by procedure TMainForm.GetSampleCount in MainFrm.pas.
  // It resides here because its arguments are declared here.
  procedure SetBatchFlag;
  function  CalculateSourceContributions: Integer;
  procedure PrepareCMB82Input;
  procedure PresentContributionsBySpecies;
  procedure PresentNormalizedMPINMatrix;
  procedure SetDisplayDecimals;
  procedure SetDichotFlag (const DichotFlag: integer); // Added 10/24/04; added int argument on 10//26/04
// Next proc added by Scalco; 11/04/2004
  procedure Create_TotalReport_Section (var astr1: string; var astr2: string; aSavedDF: integer);
  procedure Create_RSquare_Line (aStr1: string; var aStr2: string);
  procedure Create_ChiSquare_Line(aStr1: string; var aStr2: string; var aSavedDF: integer);
  procedure Create_Best_Fit_Line(aStrBuf:PChar; var aStr: string);
  procedure Create_SourceContribution_Line;
  procedure SCEStateChecker (aStr: string; var aSCEstate:TSCEstate);
  function SkipLine (aStr: string): Boolean;

implementation

uses
  DB,
  Forms,
  MainFrm;

const
  // Internal constants.
  DIM_STRINGS        = 129;
  LEN_ERR_BUF        =  80;

// The following parameters MUST correspond exactly with those listed in
// the Enumerations (@113) in CMB82.c, as well as
// the fRun declarations (@65) in CMB82a.for:
  DO_CMB82init       =   1;
  DO_SendCodes       =   2;
  DO_SetDisplayDecPl =   3;
  DO_SetBatchFlag    =   4;
  DO_SetDichotFlag   =   5; // TC Added 10/24/04
  DO_CMB             =   6;
  DO_SendFracEst     =   7;
  DO_SendInfo        =   8; // g_NspFit, g_NprFit, g_CMBFitCount
  DO_SendSNGVAL      =   9;
  DO_PData           =  10;
  DO_SScont          =  11; // Present Source Contributions
  DO_mPIN            =  12;
  DO_Write_SFitOut   =  13;
  DO_CSV2TXT         =  14;
  DO_DBF2TXT         =  15;
  DO_WKS2TXT         =  16;
  DO_CAR2TXT         =  17;
  DO_CMB_Terminate   =  18;

var
  // Internal globals.
  g_MeasurementType: string;  // for measurement
  g_ErrorStrings: array[1..NUM_ERRORS] of string;
  g_SizeArray: PChar;
  g_SourceElimFlag: Integer;
  g_OutBuf: PChar;
  g_DimOutBuf: Integer;  // number of records - set on invocation
  g_OutBufStatus: Integer;
  g_LenOutBuf: Integer;  // length of record - set on invocation
  g_SizOutBuf: Integer;
  g_ShowFitUnits: string;
  g_DeltaIter: Integer;
  g_SPDefSel, g_PRDefSel: PChar;
  g_SizeIndex: Integer;
  g_NspFit: Integer;
  g_nPRtot: Integer;
  g_NprFit: Integer;
  g_nADtot: Integer;
  g_DimErrBuf: Integer;
  g_WtChiSquare,
  g_ChiSquare,
  g_WtRSquare,
  g_RSquare,
  g_WtPCMass,
  g_PCMass,
  g_sumSCEs,  // Added 10/27/04
  g_sumSCEs1, // Added 11/05/04
  g_WtFracEstim,
  g_FracEstim,
  g_FitMeasure: Single;
  g_FitMeasureCount: Integer;
  g_MaximumSourceUncertainty, // maximum acceptable estimable source uncertainty
  g_MinimumSourceProjection: Single; // minimum acceptable estimable source projection
  g_NumOutputs: Integer;      // counts number of calls to LoadOutBufWithPData
  g_OutputCounts: array[0..MAXOUTPUTS] of Integer; // number of lines written during each call to LoadOutBuf
  g_TotalOutputCharacters: Integer; // number of characters written to output string list
  g_CMBFitCount: Integer;
  g_LArg0,
  g_LArg1,
  g_LArg2,
  g_LArg3,
  g_LArg4: Integer;
  g_PArg0,
  g_PArg1,
  g_PArg2,
  g_PArg3,
  g_PArg4,
  g_PArg5,
  g_PArg6,
  g_PArg7,
  g_PArg8,
  g_PArg9,
  g_PArg10,
  g_PArg11,
  g_PArg12: Pointer;
  g_SArg1,
  g_SArg2,
  g_SArg3,
  g_SArg4,
  g_SArg5,
  g_SArg6,
  g_SArg7,
  g_SArg8,
  g_SArg9,
  g_SArg10: PChar;
  g_DArg: Double;
  g_ErrorBuffer: PChar;
  g_ErrorCode: Integer;

  // Imported functions.
  function CMB82c(
    arg0,
    arg1,
    arg2,
    arg3,
    arg4,
    arg5,
    arg6,
    arg7,
    arg8,
    arg9,
    arg10,
    arg11,
    arg12: Pointer): Integer; stdcall; external 'CMB82.dll' name 'CMB82c' ;

// Internal functions and procedures.
  function  ChangeMeasurementInfo(const Line: string): string; forward;
  procedure ShowOutput; forward;
  procedure LoadOutBuf; forward;
  procedure LoadOutBufWithPData; forward;
  function IsNumber (AStr: string): Boolean; forward; // Companion for proc LoadOutBufWithPData
  procedure LoadOutBufSSCont; forward;
  procedure LoadOutBufReceptor; forward;
  function  GetCMBFileType(var FileName: string): Integer; forward;
  function  GetPDataInfo: Integer; forward;
  procedure NilPointers; forward;
  function  CallCMB82dll: Integer; forward;

//  Task 1; Control comes here as soon as Control File is opened:
procedure LoadTalkMem;
begin
  StrDispose(g_SArg1);
  g_SArg1 := StrAlloc(DIM_STRINGS);
  StrDispose(g_SArg2);
  g_SArg2 := StrAlloc(DIM_STRINGS);
  StrDispose(g_SArg3);
  g_SArg3 := StrAlloc(DIM_STRINGS);
  StrDispose(g_SArg4);
  g_SArg4 := StrAlloc(DIM_STRINGS);
  StrDispose(g_SArg5);
  g_SArg5 := StrAlloc(DIM_STRINGS);
  StrDispose(g_SArg6);
  g_SArg6 := StrAlloc(DIM_STRINGS);
  StrDispose(g_SArg7);
  g_SArg7 := StrAlloc(DIM_STRINGS);
  StrDispose(g_SArg8);
  g_SArg8 := StrAlloc(DIM_STRINGS);
  StrDispose(g_SArg9);
  g_SArg9 := StrAlloc(DIM_STRINGS);
  StrDispose(g_SArg10);
  g_SArg10 := StrAlloc(DIM_STRINGS);
end;

//  Task 2
procedure PrepareCMB82Input;
var
  CMBFileType: Integer;
  NewFileName: string;
begin
//  In the 1st Session (1st pass), files are QC'd in proc ReadIN8 in MainFrm.pas
//  In subsequent sessions, files not directed by the Control File that are selected
//  'on the fly' must be checked here, as routing to proc StartSession is directed
//  by proc PageControlMainChanging.
//  At this point, all SEL filenames will either be null or hold valid names (not
//  dummy names from Control File).  We check that file names exist (in the Current
//  Directory; should = FCurrentControlFolder) before loading the argument string.
//  CMB82.dll is only called if data file isn't ASCII (and thus a conversion is necessary).

  // Source selection.
  if (g_PRSelFullPathName <> '') and (FileExists(g_PRSelFullPathName)) then // This check is placed here cuz it'll NEVER raise an exception
    StrPCopy(g_SArg4, g_PRSelFullPathName)  // Set Arg4 string directly!
  else
    StrPCopy(g_SArg4, 'dummy');             // set SArg with a dummy file name

  // Species selection.
  if (g_SPSelFullPathName <> '') and (FileExists(g_SPSelFullPathName)) then   // This check is placed here cuz it'll NEVER raise an exception
    StrPCopy(g_SArg5, g_SPSelFullPathName)  // Set Arg5 string directly!
  else
    StrPCopy(g_SArg5, 'dummy');             // set SArg with a dummy file name

  // Ambient data selection.
  if (g_ADSelFullPathName <> '') and (FileExists(g_ADSelFullPathName)) then   // This check is placed here cuz it'll NEVER raise an exception
    StrPCopy(g_SArg6, g_ADSelFullPathName)  // Set Arg6 string directly!
  else
    StrPCopy(g_SArg6, 'dummy');             // set SArg with a dummy file name

  // Ambient data file.  Check file type and create temporary counterpart TXT file name:
  try
    if FileExists(g_ADFullPathName) then // This check is placed here cuz it'll NEVER raise an exception
      begin
        NewFileName := g_ADFullPathName;
        CMBFileType := GetCMBFileType(NewFileName); // on return, NewFileName contains suffix TXT
      end;
    if CMBFileType <> FILETYPE_TXT then
    begin
      case CMBFileType of
      FILETYPE_INVALID:
        begin
          ShowErrorMessage(
            'PLEASE CHECK THE AMBIENT DATA FILE SUFFIX.'
            + #10
            + 'It should be one of TXT, DAT, WKS, DBF, or CSV.');
            Exit;
        end;
      // Set appropriate conversion routines to pass to C dll if NOT already TXT format.
      // DO_CSV2TXT = 13; DO_DBF2TXT = 14; DO_WKS2TXT = 15; DO_CAR2TXT =  16
      FILETYPE_CSV:
        g_LArg0 := DO_CSV2TXT;
      FILETYPE_DBF:
        g_LArg0 := DO_DBF2TXT;
      FILETYPE_WKS:
        g_LArg0 := DO_WKS2TXT;
      FILETYPE_CAR:
        g_LArg0 := DO_CAR2TXT;
      end;
      g_PArg0 := @g_LArg0;
      StrPCopy(g_SArg9, g_ADFullPathName);
      g_PArg9 := g_SArg9;
      StrPCopy(g_SArg10, NewFileName);
      g_PArg10 := g_SArg10;
      g_ErrorCode := CallCMB82dll;
      if g_ErrorCode <> 0 then
      begin
        ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
        Exit;
      end;
    end;
  finally
    NilPointers;
  end;
  StrPCopy(g_SArg7, NewFileName); // Set Arg7 string directly with NewFileName (.TXT).

  // Source profiles data file.  Check file type and create temporary counterpart TXT file name:
  try
    if FileExists(g_PRFullPathName) then // This check is placed here cuz it'll NEVER raise an exception
      begin
        NewFileName := g_PRFullPathName;
        CMBFileType := GetCMBFileType(NewFileName); // on return, NewFileName contains suffix TXT
      end;
    if CMBFileType <> FILETYPE_TXT then
    begin
      case CMBFileType of
      FILETYPE_INVALID:
        begin
          ShowErrorMessage(
            'PLEASE CHECK THE SOURCE PROFILES DATA FILE SUFFIX.'
            + #10
            + 'It should be one of TXT, DAT, WKS, DBF, or CSV.');
            Exit;
        end;
      // Set appropriate conversion routines to pass to C dll if NOT already TXT format.
      // DO_CSV2TXT = 13; DO_DBF2TXT = 14; DO_WKS2TXT = 15; DO_CAR2TXT =  16
      FILETYPE_CSV:
        g_LArg0 := DO_CSV2TXT;
      FILETYPE_DBF:
        g_LArg0 := DO_DBF2TXT;
      FILETYPE_WKS:
        g_LArg0 := DO_WKS2TXT;
      FILETYPE_CAR:
        g_LArg0 := DO_CAR2TXT;
      end;
      g_PArg0 := @g_LArg0;
      StrPCopy(g_SArg9, g_PRFullPathName);
      g_PArg9 := g_SArg9;
      StrPCopy(g_SArg10, NewFileName);
      g_PArg10 := g_SArg10;
      g_ErrorCode := CallCMB82dll;
      if g_ErrorCode <> 0 then
      begin
        ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
        Exit;
      end;
    end;
  finally
    NilPointers;
  end;
  StrPCopy(g_SArg8, NewFileName); // Set Arg8 string directly with NewFileName (.TXT).
end;

//  Task 3
procedure InitCMB82;
begin
  try // Prepare for call to DLL (CMB82.c's Step I):
    g_LArg1 := DO_CMB82init;
    g_PArg0 := @g_LArg1;
    g_PArg1 := g_SArg1;   // Unused.
    g_PArg2 := g_SArg2;   // Unused.
    g_PArg3 := g_SArg3;   // Unused.
    g_PArg4 := g_SArg4;   // Full Path for PR*.sel
    g_PArg5 := g_SArg5;   // Full Path for SP*.sel
    g_PArg6 := g_SArg6;   // Full Path for AD*.sel
    g_PArg7 := g_SArg7;   // Full Path for AD*.* (data input file)
    g_PArg8 := g_SArg8;   // Full Path for PR*.* (data input file)
    StrPCopy(g_SArg9, 'CMBOUTDB.TXT');
    g_PArg9 := g_SArg9;   // string: CMBoutDB.txt
    g_LArg2 := g_CMBFitCount;
    g_PArg10 := @g_LArg2;
    g_CMB8ExeDir := ExtractFilePath(Application.ExeName);
    StrPCopy(g_SArg10, g_CMB8ExeDir); // Full Path for executable directory
    g_PArg11 := g_SArg10;             // a strange "transfer"

    g_ErrorCode := CallCMB82dll;
    if g_ErrorCode <> 0 then
    begin
      ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
      Exit;
    end;

    // Now that setSPcode, setPRcode, & setADcode have been completed in CMB82.c's
    // Step I, load codes via direct calls to the DLL:
    g_nSPtot := Integer(g_PArg1^);  // Direct call to DLL; gets TOT # SPecies
      StrDispose(g_SPCodeUser);
      g_SPCodeUser := StrAlloc((g_nSPtot * lenSPcode) + 1);
    g_nPRtot := Integer(g_PArg2^);  // Direct call to DLL; gets TOT # PRofiles
      StrDispose(g_PRCodeUser);
      g_PRCodeUser := StrAlloc((g_nPRtot * lenPRcode) + 1);
    g_nADtot := Integer(g_PArg3^); // Direct call to DLL; gets TOT # AD samples
      StrDispose(g_ADCode);
      g_ADCode := StrAlloc((g_nADtot * lenADcode) + 1);

    NilPointers;
    g_DimErrBuf := 4 + g_nPRtot;

    StrDispose(g_ErrorBuffer);
    g_ErrorBuffer := StrAlloc((g_DimErrBuf * LEN_ERR_BUF) + 1);

    StrDispose(g_SPDefSel);
    g_SPDefSel := StrAlloc((g_nSPtot + 1) * LENDEFSEL);

    StrDispose(g_PRDefSel);
    g_PRDefSel := StrAlloc((g_nPRtot + 1) * LENDEFSEL);

// Prepare for call to DLL; send these codes to CMB82.c's Step II:
    g_LArg1 := DO_SendCodes; // These are sent TO DLL:
    g_PArg0 := @g_LArg1;
    g_PArg1 := g_SPCodeUser;
    g_PArg2 := g_PRCodeUser;
    g_PArg3 := g_ADCode;
//    g_PArg4 := g_SizeArray; // SizeArray isn't used ANYWHERE in Delphi.  Removed 11/15/04,
                              // w/ conforming shut down of send in Step II of CMB82.c
    g_PArg5 := g_SPDefSel;
    g_PArg6 := g_PRDefSel;

    g_ErrorCode := CallCMB82dll;
    if g_ErrorCode <> 0 then
    begin
      ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
      Exit;
    end;
    g_CMB82initOK := true;
  finally
    NilPointers;
  end;
end;

//  Task 4; was part of Task 3; extracted so that it can be caled separately from
//          MainFrm; only used in CMB82.c (not sent on to CMB82a.for)
procedure SetDisplayDecimals;
begin
  g_LArg0 := DO_SetDisplayDecPl;
  g_PArg0 := @g_LArg0;
  g_LArg1 := g_DisplayDecimalPlaces;
  g_PArg1 := @g_LArg1;
  g_ErrorCode := CallCMB82dll;

  if g_ErrorCode <> 0 then
    begin
      ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
    end;
end;

// Before a calculation is attempted in Task 12, the following proc is called by
// procedure MainFrm.pas' RunCMB.  (SetBatchFlag resides here because its arguments
// are declared here.)  The call passes a flag to DLL sends a flag to the DLL to
// reflect whether the latest sample selection(s) represent(s) Batch Mode (>1 sample
// tagged or not).

// Task 5
procedure SetBatchFlag;
var
  error: integer;
Begin
  If MainForm.SampleCount >1 then
    begin

      g_LArg0 := DO_SetBatchFlag; {Set flag in CMB82.c}
      g_PArg0 := @g_LArg0;
      g_LArg1 := 1;
      g_PArg1 := @g_LArg1;

      Error := CallCMB82dll;
      if error < 0 then
      begin
        ShowErrorMessage( g_ErrorStrings[-error] );
      end;
    end
    else

// If no batch mode, LArg0 gets value of DO_SetBatchFlag (= 4), which is then passed to PArg0;
// LArg1 is set to 0, which is then passed to PArg1.

    begin
      g_LArg0 := DO_SetBatchFlag; {Set flag in CMB82.c}
      g_PArg0 := @g_LArg0;
      g_LArg1 := 0;
      g_PArg1 := @g_LArg1;

      Error := CallCMB82dll;
      if error < 0 then
      begin
        ShowErrorMessage( g_ErrorStrings[-error] );
      end;
    end;
end;

// Task 6; After Run is clicked, control continues below here:
procedure SetDichotFlag(const DichotFlag: integer); // Need to pass a Dichot. flag to Step V in CMB82.c
begin
  g_LArg0 := DO_SetDichotFlag; // Run this module in CMB82.c
  g_PArg0 := @g_LArg0;
  g_LArg1 := DichotFlag; // Added 10/26/04
  g_PArg1 := @g_LArg1;
  CallCMB82dll;
end;

//  Task 7: Select ambient sample(s); this proc strips '*' from sample each pass if present
//                                      "    "  adds   '*'   "     "    "     "  if absent
procedure UpdateADCode(Star: Char);
var
  offset: Integer;
begin
  if MainForm.ClientDataSetSamples.Filtered = false then
  begin
    offset := MainForm.ClientDataSetSamples.RecNo - 1;
  end
  else
  begin
    //   We are using a hidden field.  This field never appears in
    //   the grid, but it is there.  When the dataset of samples is being loaded,
    //   the 'real' record number is placed in '_Record_Number_', so even if we
    //   filter (case where RecNo will be wrong), by using _Record_Number_, we
    //   should always be right because we store the 'real' number while it's
    //   being loaded.  It's only an issue if you explicitely delete a sample from the row.
    //
    offset := MainForm.ClientDataSetSamples.FieldByName('_RECORD_NUMBER_').AsInteger; // -2;
  end;

  if (((Offset * lenADcode) + AMBFLAGPOS) <= (g_nADtot * lenADcode)) then
    g_ADCode[(Offset * lenADcode) + AMBFLAGPOS] := Star
  else
  begin
    // Sanity check here - but after testing, this should never happen and can
    // probably be removed.
    ShowMessage ('Error updating g_ADcode; value is too large.');
  end;
end;

//  Task 8: Select fitting species
procedure UpdateSPCodeUser(Star: Char);
var
  Offset: Integer;
begin
  Offset := MainForm.ClientDataSetSpecies.RecNo;
  // Need to update number of fitting species selected:
  g_SPCodeUser[(Offset * lenSPcode) + 16 + (2 * g_SizeIndex)] := Star;
end;

//  Task 9: Select fitting sources
procedure UpdatePRCodeUser(Star: Char);
var
  Offset: Integer;
begin
  Offset := MainForm.ClientDataSetSources.RecNo - 1;
  // Need to update number of fitting sources selected:
  g_PRCodeUser[(Offset * lenPRcode) + 16 + (2 * g_SizeIndex)] := Star;
end;

//  Task 10
procedure SetMeasurementType(const MeasurementType: string);
begin
  g_MeasurementType := MeasurementType;
end;

//  Task 11; If Best Fit is selected in Options, control will execute this function:
function DoBestFit: Integer;
var
  I, NOuter, NBest, NFitSpecies, NFitSources: Integer;
  MaxFitMeasure: Double;
  FieldName: string;
begin
  MaxFitMeasure := -99999.;
  NBest := 0;
  // Save SPCode and PRCode.
  // Do this 10 times for each default selection:
  g_FitMeasureCount := 0;
  for NOuter := 1 to 10 do
  begin
    FieldName := 'FIT' + IntToStr(NOuter);

    with MainForm.ClientDataSetSpecies do
    begin
      DisableControls;
      First;
      while not EoF do
      begin
        if FieldByName(FieldName).Text = '*' then
          UpdateSPCodeUser('*')
        else
          UpdateSPCodeUser(' ');
        Next;
      end;
      EnableControls;
    end;

    with MainForm.ClientDataSetSources do
    begin
      DisableControls;
      First;
      while not EoF do
      begin
        if FieldByName(FieldName).Text = '*' then
          UpdatePRCodeUser('*')
        else
          UpdatePRCodeUser(' ');
        Next;
      end;
      EnableControls;
    end;

    // Check for OK SPCode and PRCode.
    NFitSpecies := 0;
    for I := 1 to g_nSPtot - 1 do
      if g_SPCodeUser[(I * lenSPcode) + 16 + (2 * g_SizeIndex)] = '*' then
        Inc(NFitSpecies);
    NFitSources := 0;
    for I := 0 to g_nPRtot - 1 do
      if g_PRCodeUser[(I * lenPRcode) + 16 + (2 * g_SizeIndex)] = '*' then
        Inc(NFitSources);

    if (NFitSources > 0) and (NFitSpecies >= NFitSources) then
    begin
      // Do fit.
      g_ErrorCode := CalculateSourceContributions;                    // Task 12
      Inc(g_FitMeasureCount); // OK.
      if g_FitMeasure > MaxFitMeasure then
      begin
        MaxFitMeasure := g_FitMeasure;
        NBest := NOuter;
      end;
    end;
  end;
  Result := NBest;
end;

//  Task 12  This controls the fit calculation:
function CalculateSourceContributions: Integer;
var
  CMB82error, OldLArg2: Integer;       // In this function, error is 'CMB8error'
  Line: string;
  PCmassTerm, ChiSqrTerm: Single;
begin
  g_LArg1 := DO_CMB;
  g_PArg0 := @g_LArg1;
  g_PArg1 := g_SPCodeUser;
  g_PArg2 := g_PRCodeUser;
  g_PArg3 := g_ADCode;
  g_PArg5 := g_ErrorBuffer;
  if MainForm.CheckBoxBritt.Checked then
    g_LArg2 := 2  // Action.
  else
    g_LArg2 := 0; // Action.

  g_PArg6 := @g_LArg2;
  g_DeltaIter := StrToInt(MainForm.MaskEditIteration.Text);
  g_LArg3 := g_DeltaIter; // Action1.
  g_PArg7 := @g_LArg3;
  if MainForm.CheckBoxSource.checked then
   g_SourceElimFlag := 1
  else
   g_SourceElimFlag :=  0;

  g_LArg4   := g_SourceElimFlag;
  g_PArg8   := @g_LArg4;
  g_PArg9   := @g_ChiSquare;
  g_PArg10  := @g_RSquare;
  g_PArg11  := @g_PCMass;
  g_PArg12  := @g_sumSCEs;    // Added 10/27/04 -TC  See notes @1500 in proc LoadOutBufWithPData...
  CMB82error := CallCMB82dll; // NB: This call invokes CMB82.c's Step V. -TC (10/23/04)
                              // At this point, cHat(1) has just been set in s.CMBSVD.
                              // So at this time would be good to pass the value in and hand it off
                              // to some Delphi var for use in reporting (i.e., LoadOutFufWithPData)
  g_sumSCEs1 := g_sumSCEs;    // Back up for use in 2nd report of the Dichot. pair as it'll be overwritten
                              // with TOTAL sumSCEs when DO_PData is called from func GetPDataInfo:

  IF (CMB82error < 0)  AND (not g_BestFitFlag) then
//  IF (CMB82error < 0) then
  Begin
    ShowErrorMessage (g_ErrorStrings[-CMB82error]); // Display error message to screen; TC added 11/21/04
    if CMB82error = ERROR_CMBSVD_2 then                   // ERROR_CMBSVD_2 = -6
    begin
      Line := g_ErrorStrings[-CMB82error];        // Insert the error string for addition to Main Report
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line :=  '';
      MainForm.DBMemoSummary.Lines.Add(Line); // TC's not sure if/why we need to add this 4 times!
      MainForm.DBMemoSummary.Lines.Add(Line);
      MainForm.DBMemoSummary.Lines.Add(Line);
      MainForm.DBMemoSummary.Lines.Add(Line);
    end

    else if CMB82error = ERROR_CMBSVD_4 then              // ERROR_CMBSVD_4 = -8
    begin
      Line := Copy(
        g_ErrorStrings[-CMB82error],
        1,
        Pos('.', g_ErrorStrings[-CMB82error]));
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line := Copy(
        g_ErrorStrings[-CMB82error],
        Pos('.', g_ErrorStrings[-CMB82error]) + 1,
        Length(g_ErrorStrings[-CMB82error]));
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line :=  '';
      MainForm.DBMemoSummary.Lines.Add(Line);
      MainForm.DBMemoSummary.Lines.Add(Line);
      MainForm.DBMemoSummary.Lines.Add(Line);
    end

    else if CMB82error = ERROR_CMBSVD_5 then              // ERROR_CMBSVD_5 = -9
    begin
      Line :=  'AKT*VeffIn*AK matrix needs improvement.';
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line :=  'Change fitting sources or fitting species.';
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line := Copy(
        g_ErrorStrings[-CMB82error],
        1,
        Pos('.', g_ErrorStrings[-CMB82error]));
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line := Copy(
        g_ErrorStrings[-CMB82error],
        Pos('.', g_ErrorStrings[-CMB82error]) + 1,
        Length(g_ErrorStrings[-CMB82error]));
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line :=  '';
      MainForm.DBMemoSummary.Lines.Add(Line);
    end

    else if CMB82error = ERROR_CMBSVD_6 then             // ERROR_CMBSVD_6 = -10
    begin
      Line := '        ****    NO CONVERGENCE AFTER '
        + MainForm.MaskEditIteration.Text
        + ' ITERATIONS    ****';
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line := ' ';
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line := Copy( g_ErrorStrings[-CMB82error],
                      1, Pos('.', g_ErrorStrings[-CMB82error]));
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line := Copy( g_ErrorStrings[-CMB82error],
                    Pos('.', g_ErrorStrings[-CMB82error]) + 1,
                    Length(g_ErrorStrings[-CMB82error]));
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line := ' ';
      MainForm.DBMemoSummary.Lines.Add(Line);

//      OldLArg2 := g_LArg2;       // NB:  In earlier version logic, these 4 lines
//      g_LArg2 := 8; // Action.   // (which simply send action = 8 to DLL via a
//      CallCMB82dll;              // 2nd call, then reset the flag) are executed
//      g_LArg2 := OldLArg2;       // for case of ERROR_CMBSVD_6 & BestFit <ON>
    end

    else if CMB82error = ERROR_READ_PROFILE then // (-21); Added 11/06/04 for ProRec error -TC
    begin
      Line := Copy(
        g_ErrorStrings[-CMB82error],
        1,
        Pos('.', g_ErrorStrings[-CMB82error]));
      MainForm.DBMemoSummary.Lines.Add(Line);
      Line := Copy(
        g_ErrorStrings[-CMB82error],
        Pos('.', g_ErrorStrings[-CMB82error]) + 1,
        Length(g_ErrorStrings[-CMB82error]));
      MainForm.DBMemoSummary.Lines.Add(Line);
    end;

    Result := CMB82error;
    Exit;
  End
  ELSE                                // CMB82error > 0; there is no problem ...
  begin
    g_WtChiSquare := StrToFloat(MainForm.MaskEditChiSquare.Text);
    g_WtRSquare   := StrToFloat(MainForm.MaskEditRSquare.Text);
    g_WtPCMass    := StrToFloat(MainForm.MaskEditPercentMass.Text);
    g_WtFracEstim := StrToFloat(MainForm.MaskEditFraction.Text); // TC added 02-01-05

    if g_ChiSquare < 0.0001 then                      // Caution if ChiSqr ==> 0
       ChiSqrTerm := 1.0 / 0.0001
    else
    ChiSqrTerm := 1.0 / g_ChiSquare;

    if g_PCMass <= 100.0 then             // per Section 6.3 of the Users Manual
      PCmassTerm := (g_PCMass / 100.0)
    else
    begin
      PCmassTerm := (100.0 / g_PCMass);
    end;

    g_LArg0 := DO_SendFracEst;
    g_PArg0 := @g_LArg0;
    g_FracEstim := StrToFloat(MainForm.MaskEditFraction.Text);
    g_PArg1 := @g_FracEstim;
    g_MaximumSourceUncertainty := StrToInt(MainForm.MaskEditUncertainty.Text);
    g_PArg2 := @g_MaximumSourceUncertainty;
    g_MinimumSourceProjection := StrToFloat(MainForm.MaskEditProjection.Text);
    g_PArg3 := @g_MinimumSourceProjection;
    g_ErrorCode := CallCMB82dll;

    // On 02/01/05, TC revised and consolidated the following expression for FM;
    // note normalization to sum of Fit Measure Weights, per Peer Review preference:
    g_FitMeasure := ((g_WtChiSquare * ChiSqrTerm) + (g_WtRSquare * g_RSquare)
                 + (g_WtPCMass * PCmassTerm) +  (g_WtFracEstim * g_FracEstim))/
                 (g_WtChiSquare + g_WtRSquare + g_WtPCMass + g_WtFracEstim);

    // Clamp the fit measure.
    if g_FitMeasure > 9999 then
      g_FitMeasure := 9999;

    g_ErrorCode := GetPDataInfo;
    if (g_ErrorCode = 0) and (not g_BestFitFlag) then
      ShowOutput;
  end;
  NilPointers;
  Result := CMB82error;
end;

//  Task 13;  From s.SScont, generates source contributions-by-species report:
procedure PresentContributionsBySpecies;
begin
  if g_ContributionsShownFlag then
  begin
    if MessageDlg(
      'Do you want to do this again?',
      mtConfirmation,
      [mbYes, mbNo],
      0) = mrNo then
    begin
      Exit;
    end;
  end;
  g_LArg0 := DO_SScont;
  g_PArg0 := @g_LArg0;
  g_LenOutBuf := 7 + (7 * g_nPRtot);
  if g_LenOutBuf < 80 then
    g_LenOutBuf := 80;
  g_DimOutBuf := 6 + g_nSPtot + 5 + 1;
  g_SizOutBuf := g_LenOutBuf * g_DimOutBuf;
  g_LArg1 := g_LenOutBuf;
  g_PArg1 := @g_LArg1;
  g_LArg2 := g_DimOutBuf;
  g_PArg2 := @g_LArg2;
  g_LArg3 := g_SizOutBuf;
  g_PArg3 := @g_LArg3;
  StrDispose(g_OutBuf);
  g_OutBuf := StrAlloc(g_SizOutBuf);
  g_PArg4 := g_OutBuf;
  g_OutBufStatus := OutBufSSCont;
  g_ErrorCode := CallCMB82dll;
  if g_ErrorCode = 0 then
  begin
    if not g_BestFitFlag then
      ShowOutput;
    g_ContributionsShownFlag := True;
  end
  else
    ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
  NilPointers;
end;

//  Task 14: Using s.PIN, generates modified PIN report:
procedure PresentNormalizedMPINMatrix;
begin
  if g_mPINShownFlag then
  begin
    if MessageDlg(
      'Do you want to do this again?',
      mtConfirmation,
      [mbYes, mbNo],
      0) = mrNo then
    begin
      Exit;
    end;
  end;
  g_LArg0 := DO_mPIN;
  g_PArg0 := @g_LArg0;
  g_LenOutBuf := 10 + (7 * g_nPRtot);
  if g_LenOutBuf < 80 then
    g_LenOutBuf := 80;
  g_DimOutBuf := 5 + g_nSPtot + 3 + 1;
  g_SizOutBuf := g_LenOutBuf * g_DimOutBuf;
  g_LArg1 := g_LenOutBuf;
  g_PArg1 := @g_LArg1;
  g_LArg2 := g_DimOutBuf;
  g_PArg2 := @g_LArg2;
  g_LArg3 := g_SizOutBuf;
  g_PArg3 := @g_LArg3;
  StrDispose(g_OutBuf);
  g_OutBuf := StrAlloc(g_SizOutBuf);
  g_PArg4 := g_OutBuf;
  g_OutBufStatus := OUTBUFMPIN;
  g_ErrorCode := CallCMB82dll;
  if g_ErrorCode = 0 then
  begin
    if not g_BestFitFlag then
      ShowOutput;
    g_mPINShownFlag := True;
  end
  else
    ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
  NilPointers;
end;

//  Task 15;  This routine is called once for each sample tagged & writes a temporary
//  output file.  The output file format (WriteDT2Type; default = TXT) must be set
//  upstream in Options.
//  In this version (EPA-CMB8.2), SFitBuf is written to $TEMPOUT.txt for ASCII-type
//  formats, updated per sample in the queue.  $TEMPOUT.txt is then transferred to
//  DBRichCMBoutDB, which is necessary in order to enable the user to save
//  particular results to output.  -TC 10/03

// JS 09.11.04 - added new param so code 'knows' when it it sees the 1st tagged sample:
function WriteSFit2TempBuffer (const aAtFirstRecord: Boolean): Integer;
var
  OutputFileFormatIndex, WriteDT2Type, Write2DT2Status: Integer;
  FileContents : TStringList;
  line: string;
  InputFile: TextFile;
begin
  WriteDT2Type := WriteDT2_TXT;  // set this to a default in case it's not 0 or 1 in case below
  // Always write the output in text format with the header included and then
  // copy all this into the temporary result table, from which the user can save
  // particular results later.

  OutputFileFormatIndex := MainForm.ComboBoxOutputFileFormat.ItemIndex;

  try
    case OutputFileFormatIndex of
      0: WriteDT2Type := WriteDT2_TXT; // 1
      1: WriteDT2Type := WriteDT2_CSV; // 2
    end;

// Set Write2DT2Status.
//     Write2DT2Status := 1 => DLL returns SFit data with header; writes over itself each pass
//                             Causes header to be written to ToF for each $TEMPOUT.txt
//     Write2DT2Status := 0 => DLL returns SFit data without header and APPENDS to this file
//                             Used for subsequent sample(s) in a batch mode; $TEMPOUT.txt is cumulative

    Write2DT2Status := 1; // Hard-wired; Delphi's DBRichCMBoutDB is loaded (below)
                          // once/pass and as such, expects a single sample's results
                          // each pass.  Don't create a cumulative $TEMPOUT.txt

    g_LArg0 := DO_Write_SFitOut;
    g_PArg0 := @g_LArg0;
    g_LArg1 := WriteDT2Type;
    g_PArg1 := @g_LArg1;
    StrPCopy(g_SArg2, g_CMB8ExeDir + '$TEMPOUT.txt');
    g_PArg2 := g_SArg2;
    g_LArg3 := Write2DT2Status;
    g_PArg3 := @g_LArg3;
    g_ErrorCode := CallCMB82dll; // populates output (results) file

    if g_ErrorCode <> 0 then
    begin
      ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
      g_ErrorCode := 0;
    end;
  finally
    NilPointers;
    WriteSFit2TempBuffer := g_ErrorCode;
  end;

  // N.B.:  This final block takes the output contained in $TEMPOUT.txt and, line
  // by line, appends results for the sample for which SFit was just calculated
  // to DBRichCMBoutDB.  This routine ONLY works for an ASCII-type (i.e., TXT or
  // CSV) format; binary formats (e.g., DBF or WKS) are not amenable and cannot
  // be successfully transferred; these have been disabled in the DLL call. -TC 10/03

  begin
      AssignFile(InputFile,  g_CMB8ExeDir + '$TEMPOUT.txt');
      if aAtFirstRecord = True then  // It's the 1st tagged sample in a batch run.
      begin
        Reset (InputFile);           // open InputFile
        ReadLn(InputFile, g_Header); // Read the 1st line into Header; used in SaveResultsFrm
        CloseFile(InputFile);
      end;
 // Create the string list object:
    FileContents := TStringList.Create;
 // Load InputFile
    FileContents.LoadFromFile(g_CMB8ExeDir + '$TEMPOUT.txt');
 // Remove the first line (header)
    FileContents.Delete(0);
 // Save InputFile w/o its header
    FileContents.SaveToFile(g_CMB8ExeDir + '$TEMPOUT.txt');

    Reset(InputFile); // reopen InputFile
    while not EoF(InputFile) do                // Load DBRichCMBoutDB
    begin
      ReadLn(InputFile, line);                 // Read a line from $TEMPOUT.ext
      MainForm.DBRichCMBoutDB.Lines.Add(line); // Write this to the next line in DBRichCMBoutDB
    end;
    CloseFile(InputFile);
  end;
end;

//  Task 16; added June 3rd 2003 (RFA)
procedure TerminateCMB82;
var
  error : integer;
begin
  if g_CMB82initOK = true then
  begin
    g_LArg0 := DO_CMB_Terminate;
    g_PArg0 := @g_LArg0;

    Error := CallCMB82dll;
    if error < 0 then
    begin
      ShowErrorMessage( g_ErrorStrings[-error] );
    end;
  end;
end;

procedure ShowErrorMessage(const ErrorMessage: string);
begin
  Beep;
  MessageDlg(ErrorMessage, mtError, [mbOK], 0);
end;

function LookupEInOutErrorName (const aErrCode :integer) : string;
// Added 9/6/03 to improve I/O error handling; JScalco.
// This function was built by JScalco using info & parameters described in Delphi.
// 'Help' under 'Exception|SysUtils'.  It adds specificity to exceptions on file input.
begin
  case aErrCode of
    2   : result := 'File not found';
    3   : result := 'Path not found';
    5   : result := 'Access denied';
    32  : result := 'Sharing violation';
    100 : result := 'End of File.';
    101 : result := 'Disk Full';
    102 : result := 'File variable not assigned.';
    103 : result := 'File not open.';
    104 : result := 'File not open for input.' ;
    105 : result := 'File not open for output' ;
    106 : result := 'Error in formatted input' ;
    107 : result := 'File is already open.';
    else
      result := 'Unknown File Error (' + IntToStr(aErrCode) + ')';
  end;
end;

// ==================================
// Internal functions and procedures.
// ==================================

//  NB:  a text string passed from the DLL is read by the following function.
//  The goal is to find the line with 'SCE(UG/M3)' because it requires special
//  handling to present the correct units in the ()'s -  e.g., SCE(ug/m3) or it
//  could be SCE(ug/ppbc) and so on.  So we use the global var g_MeasurementType,
//  which is set in the Options on another form in the application.
//  If, however, the line does not contain a match, the original line which was
//  passed in is returned to the calling code UNaltered.  So nothing is done to
//  the line - it's simply ignored.

function ChangeMeasurementInfo(const Line: string): string;
var
  Offset: Integer;
  LinePart1, LinePart2: string;
begin
  Result := Line; // By default, return what was passed in.

  Offset := Pos('SCE(UG/M3)', Line);

// We only do this if we match! Otherwise we return exactly what was passed in
// originally.
//
  if Offset > 0 then
  begin
    // Get beginning of line before SCE(...)
    LinePart1 := Copy(Line, 1, Offset - 1);
    // Get ending of line
    LinePart2 := Copy(Line, Offset + 10, Length(Line));
    // Add in the current measurement type to the ()'s
    Result := Format('%sSCE(%s)%s', [LinePart1, g_MeasurementType, LinePart2]);
  end;
end;

procedure ShowOutput;
begin
  case g_OutBufStatus of
  OUTBUFFIT:
    LoadOutBufWithPData;
  OUTBUFSSCONT:
    LoadOutBufSSCont;
  OUTBUFPROFILE:
    LoadOutBuf;
  OUTBUFRECEPTOR:
    LoadOutBufReceptor;
  OUTBUFMPIN:
    LoadOutBuf;
  else
    ShowErrorMessage('Undefined output status');
  end;
end;

procedure LoadOutBuf;
var
  Offset, Count, I: Integer;
  StrBuf: PChar;
  Str1: string;
begin
  StrBuf := StrAlloc(g_LenOutBuf + 1);
  try
    StrBuf[g_LenOutBuf] := #0;
    Offset := 0;
    Count := 0;
    while True do
    begin
      StrLCopy(StrBuf, g_OutBuf + Offset, g_LenOutBuf);
      Inc(Count);
      if StrBuf[0] = #0 then
      begin
        Break;
      end;
      Str1 := StrBuf;
      g_TotalOutputCharacters := g_TotalOutputCharacters + Length(Str1) + 1;
      if Count > 2 then
        MainForm.DBMemoMPIN.Lines.Add(Str1);
      Offset := Offset + g_LenOutBuf;
    end;
    for I := MAXOUTPUTS downto 1 do
      g_OutputCounts[I] := g_OutputCounts[I - 1];
    g_OutputCounts[0] := Count;
    if g_NumOutputs < MAXOUTPUTS then
      Inc(g_NumOutputs);
  finally
    StrDispose(StrBuf);
  end;
end;

procedure LoadOutBufWithPData;
const
  kBlank = ' ';
var
  LineNumber, Offset, I, CycleCount, SavedDF: Integer;
  StrBuf: PChar;
  NumStr, Str1, Str2: string;
  TotalReport: Boolean;
  StringList: TStringList;
  SCEState: TSCEState;
  PercentPos, FooterLine, iPlusPos: integer;
  ConcValStrs: TStringList;

  procedure AddLine (aStr: string=' '); // Default, if no param, astr = ' ' - so we get a blank line!
  begin
    MainForm.DBMemoSummary.Lines.Add(aStr);
  end;
begin
  SCEstate := stNotSCEs; // JS added for handling presentation of SCEs; 03/2003
  LineNumber  := 0;
  CycleCount  := 0;
  SavedDF     := 0;
  TotalReport := False;
  StrBuf := StrAlloc(g_LenOutBuf + 1);
  StrBuf[g_LenOutBuf] := #0;
  try
    try
      // Force the format of the output to be the same with R square starting on
      // the 6th line.
      AddLine('FITTING STATISTICS:');
      AddLine (kblank);
      Offset := 0;

      while True do
      begin
        Inc(LineNumber);
        StrLCopy(StrBuf, g_OutBuf + Offset, g_LenOutBuf);

        // Find the beginning of the Main Report:
        if (LineNumber > 0) and (Pos('CMB 8.2', StrBuf) > 0) then
        begin
          Inc(CycleCount);
          if (CycleCount = 2) then // This is the top of the Total Report for the Coarse/Fine pair: !was if CycleCount = 3
          begin // We've seen this same line twice ==> there must be a Coarse/Fine pair
            TotalReport := True;
            Offset := Offset + g_LenOutBuf;
            Continue;
          end;
        end;

        if TotalReport and (StrBuf = 'SOURCE                                                                          ') then // SOURCE + 74 spaces!
        begin
        // Create a custom header that only gets stamped at the top of the TOTAL
        // report which follows a Fine/Coarse pair:
           Create_TotalReport_Section (str1, str2, SavedDF);
        end; // do Total Report

        if StrBuf[0] = #0 then // This Break is critical !!
        begin
          Break;
        end;

        Str1 := StrBuf;
        g_TotalOutputCharacters := g_TotalOutputCharacters + Length(Str1) +1;

        if LineNumber = 3 then  Create_RSquare_Line (Str1, Str2);

        if LineNumber = 4 then
        begin
          Create_ChiSquare_Line(Str1, Str2, SavedDF);
          Create_Best_Fit_Line(StrBuf, Str1);
          Create_SourceContribution_Line;
        end;

        if LineNumber > 6 then
        begin
          // Check and manage SCE state
          SCEstateChecker (Str1, SCEstate);

          if SCEstate = stApproachingSCEs then // we should be on the line with Measurement
          begin
            Str1 := ChangeMeasurementInfo(Str1);
            AddLine (Str1);
            Offset := Offset + g_LenOutBuf;
            Continue;
          end;

          if (Trim(Str1) = '----------------------------------------------------')
                and (SCEstate = stJustCompletedSCEs) then
          begin
            AddLine(kBlank);
            AddLine('----------------------------------------------------');
            Str1 := StringOfChar(' ', 22); // Spaces padded before the sumSCEs value

            // Format the total with the same number of decimal places as
            // the rest of the output.

            // sumSCEs, retrieved from DLL, is presented here:
            if TotalReport = False then
              Str1 := Str1 + FloatToStrF(g_sumSCEs1, ffFixed, 15, StrToInt(MainForm.MaskEditDecimals.Text))
            else
              Str1 := Str1 + FloatToStrF(g_sumSCEs,  ffFixed, 15, StrToInt(MainForm.MaskEditDecimals.Text));

            // Decimal align the total with the rest of the output.
            if Pos('.', Str1) = 0 then  Str1 := Str1 + '.';
            AddLine(str1);
            SCEstate := stNotSCEs ;
          end
          else if Pos('REDUNDANT HEADER', Str1) > 0 then
          begin
          // We find CMB8.0's redundant header coming from DLL.
            AddLine ('SPECIES CONCENTRATIONS:');
            Str1 := '                                                         CALCULATED    RESIDUAL';
            AddLine(str1);

            Str1 := '                                                         ----------  -----------';
            AddLine(str1);
            Str1 := 'SPECIES      FIT       MEASURED          CALCULATED       MEASURED   UNCERTAINTY';
            AddLine (Str1);
            Str1 := '--------------------------------------------------------------------------------';
            AddLine (Str1);

         // This (above) creates the upper (dashed) line in the Main Report.
         // The (dashed) bottom line in the Main Report is created in the DLL (s.PData in CMB82a.for @2170)
          end
          else
          begin
            if SkipLine (Str1) = false then
              AddLine(str1);
          end

        end;  // Line number > 6

        Offset := Offset + g_LenOutBuf;
      end; // end While true do
    except
      on E: Exception do
        ShowMessage('Exception at line ' + IntToStr(LineNumber));
    end;
  finally
    StrDispose(StrBuf);
  end;
end;

function SkipLine (aStr: string): Boolean;
begin
  // The report we get from DLL has some lines in it, which we want to skip over -
  // they match the conditions below.  This is the 'bleeding' at the end; to suppress
  // it, we see if the string we are looking at has one of the following.  If it does,
  // then we skip

  Result := false;
  if (Pos('REDUNDANT HEADER starts here ...', aStr) > 0) or        // We don't see CMB8.0's redundant header coming from the DLL:
     (Pos('MEASURED CONCENTRATION for SIZE: COARSE', aStr) > 0) or // <- changed COARS to COARSE here
     (Pos('SAMPLE DURATION', aStr) > 0) or
     (Pos('R SQUARE', aStr) > 0) or
     (Pos('CHI SQUARE', aStr) > 0) or
     (Pos('SPECIES-------', aStr) > 0) then
  begin
    Result := true; // should skip this line ...
  end;
end;

procedure SCEstateChecker (aStr: string; var aSCEstate:TSCEstate);
begin
  if pos('SCE(', astr) > 0 then
  begin
    aSCEstate := stApproachingSCEs;
    exit;
  end;

  // Next line should be with dashed lines; we now are at the SCE presentation
  // section after this line:
  if (aSCEstate = stApproachingSCEs) and
     (pos('-------', astr) > 0)  then
  begin
    aSCEstate := stSCEs;
    exit;
  end;

  // We are presenting SCEs and see the Bottom Line:
  if (aSCEstate = stSCEs) and
      (pos ('-----', astr) > 0 ) then
  begin
    aSCEstate := stJustCompletedSCEs;
    exit;
  end;

  if (aSCEstate = stJustCompletedSCEs) then
  begin
    // Insert the (dashed) Bottom Line:
    aSCEstate := stNotSCEs ;
    exit;
  end;
end;

procedure Create_SourceContribution_Line;
var
  Str1: string;
begin
  Str1 := ' ';
  MainForm.DBMemoSummary.Lines.Add(Str1);
  Str1 := 'SOURCE CONTRIBUTION ESTIMATES:';
  MainForm.DBMemoSummary.Lines.Add(Str1);
end;

procedure Create_Best_Fit_Line(aStrBuf:PChar; var aStr: string);
begin
  if MainForm.ClientDataSetResults.FieldByName('BEST').AsBoolean then // Best Fit mode only
  begin
    StrLCopy(aStrBuf + 54, 'FIT MEASURE ', 12);
    Str(g_FitMeasure:6:2, aStr);
    StrPCopy(aStrBuf + 66, aStr);
    aStrBuf[g_LenOutBuf] := #0;
    aStr := '    FIT MEASURE    ' + aStr;
// FIT INDEX
  end
  else
    aStr := '';

  MainForm.DBMemoSummary.Lines.Add(aStr);
end;

procedure Create_ChiSquare_Line (aStr1: string; var aStr2: string; var aSavedDF: integer);
begin
  aStr2 := Copy(aStr1, 1, Pos('DF', aStr1) - 1) + 'DEGREES FREEDOM';
  aStr2 := aStr2 + Copy(aStr1, Pos('DF', aStr1) + 2, Length(aStr1));
  aSavedDF := StrToInt(Trim(Copy(aStr1, Pos('DF', aStr1) + 2, Length(aStr1))));
  MainForm.DBMemoSummary.Lines.Add(aStr2);
end;

procedure Create_RSquare_Line (aStr1: string; var aStr2: string);
begin
  // R square line.
  aStr2 :=  Copy(aStr1, 1, Pos('PERCENT', aStr1) - 1)
    + '                   % ';
  aStr2 := aStr2 + Copy(aStr1, Pos('MASS', aStr1), Length(aStr1));
  MainForm.DBMemoSummary.Lines.Add(aStr2);
end;

procedure Create_TotalReport_Section (var astr1: string; var astr2: string; aSavedDF: integer);
var
  percentMassStr: string;
begin                                                        // Do total report.
  MainForm.DBMemoSummary.Lines.Add('                     Chemical Mass Balance Version EPA-CMB8.2');
  MainForm.DBMemoSummary.Lines.Add('                              Report Date: ' + DateToStr(Date));
  MainForm.DBMemoSummary.Lines.Add('');
  MainForm.DBMemoSummary.Lines.Add('SAMPLE:                       OPTIONS: INPUT FILES:');
  MainForm.DBMemoSummary.Lines.Add('');
  if MainForm.ClientDataSetResults.FieldByName('BRITT').AsBoolean then
    aStr2 := 'Yes'
  else
    aStr2 := 'No ';
  MainForm.DBMemoSummary.Lines.Add(
    'SITE:          '
    + Format('%-15.15s', [MainForm.ClientDataSetResults.FieldByName('SITE').Text])
    + 'BRITT & LUECKE:          '
    + aStr2
    + '            '
    + MainForm.EditControlFile.Text);         // Echo Control File name.
  if MainForm.ClientDataSetResults.FieldByName('ELIM').AsBoolean then
    aStr2 := 'Yes'
  else
    aStr2 := 'No ';
  MainForm.DBMemoSummary.Lines.Add(
    'SAMPLE DATE:   '
    + Format('%-15.15s', [MainForm.ClientDataSetResults.FieldByName('DATE').Text])
    + 'SOURCE ELIMINATION:      '
    + aStr2
    + '            '
    + ExtractFileName(g_PRSelFullPathName)); // Echo PR selection file name.
  if MainForm.ClientDataSetResults.FieldByName('BEST').AsBoolean then
    aStr2 := 'Yes'
  else
    aStr2 := 'No ';
  MainForm.DBMemoSummary.Lines.Add(
    'DURATION:      '
    + Format('%-15.15s', [MainForm.ClientDataSetResults.FieldByName('DUR').Text])
    + 'BEST FIT:                '
    + aStr2
    + '            '
    + ExtractFileName(g_SPSelFullPathName)); // Echo SP selection file name.
  MainForm.DBMemoSummary.Lines.Add(
    'START HOUR:    '
    + Format('%-15.15s', [MainForm.ClientDataSetResults.FieldByName('START').Text])
    + '                                        '
    + ExtractFileName(g_ADSelFullPathName)); // Echo AD selection file name.
  MainForm.DBMemoSummary.Lines.Add(            'SIZE:          TOTAL'
    + '                                                  ' // <- 50 spaces
    + ExtractFileName(g_ADFullPathName));          // Echo AD file name.
  MainForm.DBMemoSummary.Lines.Add('                                                                      ' // <- 70 spaces
    + ExtractFileName(g_PRFullPathName));          // Echo PR file name.
  MainForm.DBMemoSummary.Lines.Add('FITTING STATISTICS:');
  MainForm.DBMemoSummary.Lines.Add('');

// %Mass, retrieved from DLL, is presented here:
  percentMassStr := format ('%5.1f', [g_pcMass]);
  MainForm.DBMemoSummary.Lines.Add('             % MASS     ' + percentMassStr);
  MainForm.DBMemoSummary.Lines.Add('    DEGREES FREEDOM     ' + Format('%5s',[IntToStr(aSavedDF)]));
  MainForm.DBMemoSummary.Lines.Add('');
  MainForm.DBMemoSummary.Lines.Add('');
end; // do TotalReport

// Used by March 10, 2003 rev. of LoadOutBufWithPData
function IsNumber (AStr: string): Boolean;
var
  i : integer;
  ch: char;
  LenOfNum: integer;
begin
  Result := true;
  Astr := trim(AStr); // Remove any blanks in front and at end:
  LenOfNum := Length(Astr);

  Astr := Trim(Astr); // Remove blanks:
  for i := 1 to LenOfNum-1 do
  begin
    ch := Astr[i];

    if (i = 1) and (ch = '-') then
      result := true
    else
       if not (ch in ['0'..'9','.']) then
         Result := false;
  end;
end;

procedure LoadOutBufSSCont;
var
  I, J, N, LineCount, Count, Offset: Integer;
  StrBuf: PChar;
  Str1: string;
  Units: array[0..DIMSHOWFITUNITS] of Char;
  CalculatedValuesList: TStringList;
  MeasuredValuesList: TStringList;
begin
  StrBuf := StrAlloc(g_LenOutBuf + 1);
  CalculatedValuesList := TStringList.Create;
  MeasuredValuesList   := TStringList.Create;

  try
    StrPCopy(Units, g_ShowFitUnits);
    StrBuf[g_LenOutBuf] := #0;
    Offset := 0;
    Count := 0;
    J := 0;
    LineCount := 0;

    while ( Pos(
      'SPECIES CONCENTRATIONS:',
      MainForm.DBMemoSummary.Lines[LineCount]) = 0) do
    begin
      Inc(LineCount);
    end;
    LineCount := LineCount + 6;

    // Copy all the lines in dbMemoSummary from lineCount to end.  This
    // includes only data rows (no headings).  Copy them into 2 lists:

    for I := LineCount to MainForm.DBMemoSummary.Lines.Count - 3 do
    begin
      Str1 := Copy(MainForm.DBMemoSummary.Lines[I], 36, 8);
      CalculatedValuesList.Add (str1);

      Str1 := Copy(MainForm.DBMemoSummary.Lines[I], 17, 8);
      MeasuredValuesList.Add (str1);
    end;

    while True do
    begin
      StrLCopy(StrBuf, g_OutBuf + Offset, g_LenOutBuf);
      Inc(Count);

      if Count = 1 then
      begin
        // Insert units in output.
        for N := 0 to DIMSHOWFITUNITS - 1 do
          StrBuf[22 + N] := Units[N];
      end;

      if StrBuf[0] = #0 then
      begin
        Break;
      end;

      Str1 := StrBuf;
      g_TotalOutputCharacters := g_TotalOutputCharacters + Length(Str1) + 1;

      // 08.19.2004 A blank line between title here would be nice.. but it's not that straight forward :(
      // TC is playing with the Fortran to effect this <g>
      if Count > 6 then // Calculated & Measured values in the report facilitate confirmation of Concentration-by-Species calcs.
      begin
        // Insert the calculated and measured mass values from the MainForm.DBMemoSummary.Lines
        // for the corresponding species.                     // JS - 08.19.2004
        if Pos('SPECIES', Str1) <> 0 then
        begin
          Str1 := 'SPECIES  CALCULATED  MEASURED   ' + Copy(Str1, 9, Length(Trim(Str1)));
        end;

        if (Count > 9)and (J < (CalculatedValuesList.Count - 1)) then
        begin
          Str1 :=
            Copy(Str1, 1, 8)
            + '   '
            + CalculatedValuesList[j] // PES initieated November 2000
            + '  '
            + MeasuredValuesList[j]   // Ideal Software initiated August 2003
            + '  '
            + Copy(Str1, 9, Length(Trim(Str1)));
          Inc(J);
        end;
        MainForm.DBMemoContributions.Lines.Add(Str1);
      end;
      Offset := Offset + g_LenOutBuf;
    end;

    for N := MAXOUTPUTS downto 1 do
      g_OutputCounts[N] := g_OutputCounts[N - 1];

    g_OutputCounts[0] := Count;

    if g_NumOutputs < MAXOUTPUTS then
      Inc(g_NumOutputs);

  finally
    CalculatedValuesList.Free;
    MeasuredValuesList.Free;
    StrDispose(StrBuf);
  end;
end;

procedure LoadOutBufReceptor;
var
  I, Offset, Count: Integer;
  StrBuf: PChar;
  Str1: string;
  Units: array[0..DIMSHOWFITUNITS] of Char;
begin
  StrBuf := StrAlloc(g_LenOutBuf + 1);
  try
    StrPCopy(Units, g_ShowFitUnits);
    StrBuf[g_LenOutBuf] := #0;
    Offset := 0;
    Count := 0;
    while True do
    begin
      StrLCopy(StrBuf, g_OutBuf + Offset, g_LenOutBuf);
      Inc(Count);
      if Count = 1 then
      begin
        // Insert units in output.
        for I := 0 to DIMSHOWFITUNITS - 1 do
          StrBuf[33 + I] := Units[I];
      end;
      if StrBuf[0] = #0 then
      begin
        Break;
      end;
      Str1 := StrBuf;
      g_TotalOutputCharacters := g_TotalOutputCharacters + Length(Str1) + 1;
      if Count > 2 then
        MainForm.DBMemoSummary.Lines.Add(Str1);
      Offset := Offset + g_LenOutBuf;
    end;
    for I := MAXOUTPUTS downto 1 do
      g_OutputCounts[I] := g_OutputCounts[I - 1];
    g_OutputCounts[0] := Count;
    if g_NumOutputs < MAXOUTPUTS then
      Inc(g_NumOutputs);
  finally
    StrDispose(StrBuf);
  end;
end;

function GetCMBFileType(var FileName: string): Integer;
// This function changes the suffix of the file name passed to it to .txt <IF> it is
// one on the allowed list.  It then returns the file type to the calling routine.

var
  Extension: string;
begin
  Extension := LowerCase(ExtractFileExt(FileName));
  if ('.txt' = Extension) or ('.dat' = Extension) then
    Result := FILETYPE_TXT      //  0
  else if '.csv' = Extension then
    Result := FILETYPE_CSV      //  1
  else if '.dbf' = Extension then
    Result := FILETYPE_DBF      //  2
  else if ('.wks' = Extension) or ('.wk1' = Extension) then
    Result := FILETYPE_WKS      //  3
  else if '.car' = Extension then
    Result := FILETYPE_CAR      //  4 - row & column reversed
  else
    Result := FILETYPE_INVALID; // -1
  if (Result <> FILETYPE_INVALID) and (Result <> FILETYPE_TXT) then
    FileName := ChangeFileExt(FileName, '.txt');
end;

function GetPDataInfo: Integer;                           // Called from Task 12
var
  I: Integer;
  TotalMeasuredMass: Double;
  ResultBuffer: Pointer;
  PValue: ^Single;
begin
  Result := 0;
  ResultBuffer := nil;
  try // Load values from CMB82.c's Step VII (DO_SendInfo):
    g_LArg0 := DO_SendInfo;
    g_PArg0 := @g_LArg0;
    g_PArg1 := @g_LArg1; // g_NspFit
    g_PArg2 := @g_LArg2; // g_NprFit
    g_PArg3 := @g_LArg3; // g_CMBFitCount
    g_PArg4 := @g_DArg;
    g_ErrorCode := CallCMB82dll;
    if g_ErrorCode <> 0 then
    begin
      ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
      Exit;
    end;
    g_NspFit := g_LArg1;
    g_NprFit := g_LArg2;
    g_CMBFitCount := g_LArg3;
    TotalMeasuredMass := g_DArg;

    ResultBuffer := StrAlloc(SizeOf(Single) * g_nPRtot);

// Load value from CMB82.c's Step VIII (DO_SendSNGVAL):
    g_LArg0 := DO_SendSNGVAL;
    g_PArg0 := @g_LArg0;
    g_PArg1 := ResultBuffer;
    g_ErrorCode := CallCMB82dll;
    if g_ErrorCode <> 0 then
    begin
      ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
      Exit;
    end;

    // Determine the size of the inestimable space.
    for I := 0 to g_NprFit do
    begin
      if I < g_NprFit then
      begin
        PValue := Ptr(Integer(ResultBuffer) + (SizeOf(Single) * I));
        if (1.0 / PValue^)
          > ((g_MaximumSourceUncertainty * TotalMeasuredMass) / 100) then
        begin
          Break;
        end;
      end;
    end;

    g_LArg0 := DO_PData;
    g_PArg0 := @g_LArg0;

    g_LenOutBuf := 80;

    // Lines required for non-inestimable source output, including Total...
    g_DimOutBuf := (2 * (24 + g_NprFit + g_nSPtot))
      // ...plus constant lines...
      + 16
      // ...plus singular values...
      + (g_NprFit div 8) + 1
      // ...plus estimable sources...
      + (g_NprFit div 5) + 1
      // ...plus inestimable source lc's.
      + (g_NprFit * (g_NprFit div 4)) + 1;

    Inc(g_DimOutBuf);
    g_SizOutBuf := g_LenOutBuf * g_DimOutBuf;
    g_LArg1 := g_LenOutBuf;
    g_PArg1 := @g_LArg1;

    g_LArg2 := g_DimOutBuf;
    g_PArg2 := @g_LArg2;

    g_LArg3 := g_SizOutBuf;
    g_PArg3 := @g_LArg3;

    StrDispose(g_OutBuf);
    g_OutBuf := StrAlloc(g_SizOutBuf);

    g_PArg4 := g_OutBuf;
    g_PArg5 := @g_MaximumSourceUncertainty;
    g_PArg6 := @g_MinimumSourceProjection;

    g_OutBufStatus := OUTBUFFIT;
    g_PArg7 := @g_pcMass;  // TC added 10/20/04 to get values for Total Report only
    g_PArg8 := @g_sumSCEs; // TC added 10/20/04 to get values for Total Report only {here, = work(ncHat+1) in s.PData}

    g_ErrorCode := CallCMB82dll;
    if g_ErrorCode <> 0 then
    begin
      ShowErrorMessage(g_ErrorStrings[-g_ErrorCode]);
      Exit;
    end;
  finally
    StrDispose(ResultBuffer);
    NilPointers;
    Result := g_ErrorCode;
  end;
end;

procedure NilPointers;
begin
  g_PArg0  := nil;
  g_PArg1  := nil;
  g_PArg2  := nil;
  g_PArg3  := nil;
  g_PArg4  := nil;
  g_PArg5  := nil;
  g_PArg6  := nil;
  g_PArg7  := nil;
  g_PArg8  := nil;
  g_PArg9  := nil;
  g_PArg10 := nil;
  g_PArg11 := nil;
  g_PArg12 := nil;
end;

function CallCMB82dll: Integer;
begin
  Result := CMB82c(
    g_PArg0,
    g_PArg1,
    g_PArg2,
    g_PArg3,
    g_PArg4,
    g_PArg5,
    g_PArg6,
    g_PArg7,
    g_PArg8,
    g_PArg9,
    g_PArg10,
    g_PArg11,
    g_PArg12);
end;

initialization
  // JS 06/03/03
  g_CMB82initOK := false;
  g_ErrorStrings[-ERROR_MEMORY_ALLOC]
    := 'Memory allocation error.  Try shutting down other applications or increasing RAM.';
  g_ErrorStrings[-ERROR_FILE_OPEN]
    := 'File Open Error.  Check that the file exists or try rebooting.';
  g_ErrorStrings[-ERROR_SPECIES_SET]
    := 'Species Set Error.  Please see Section 4 of the Users Manual regarding input file construction.';
  g_ErrorStrings[-ERROR_SOURCES_SET]
    := 'Sources Set Error.  Please see Section 4 of the Users Manual regarding input file construction.';
//  g_ErrorStrings[-ERROR_CMBSVD_1] TC shut down 11/21/04; no longer needed
//    := 'The number of fitting sources must be positive and <= number of fitting species.';
  g_ErrorStrings[-ERROR_CMBSVD_2]
    := 'Positive uncertainties are required for all fitting species and sources.';
//  g_ErrorStrings[-ERROR_CMBSVD_3] is never set in the DLL.
  g_ErrorStrings[-ERROR_CMBSVD_4]
    := 'The fitting source profiles matrix (Afit) contained a column of all zeros.  Check the source profiles data file or change the fitting sources.';
  g_ErrorStrings[-ERROR_CMBSVD_5]
    := 'AKT*VeffIn*AK matrix (s.CMBSVD) needs improvement.  The choices for fitting species and sources have produced a near-singular matrix.  There is most likely collinearity between two or more fitting sources.  Change fitting sources or fitting species.';
  g_ErrorStrings[-ERROR_CMBSVD_6]
    := 'The fitting algorithm failed to converge.  There is most likely collinearity between two or more of the fitting sources.';
  g_ErrorStrings[-ERROR_FILE_READ]
    := 'File Read Error.  Please see Section 4 of the Users Manual regarding input file construction.';
  g_ErrorStrings[-ERROR_MISC]
    := 'Unknown error.';
  g_ErrorStrings[-ERROR_FILE_CLOSE]
    := 'File Close Error.  If your disk is full, try freeing up some space, or try rebooting.';
  g_ErrorStrings[-ERROR_FILE_WRITE]
    := 'File Write Error.  If your disk is full, try freeing up some space, or try rebooting.';
  g_ErrorStrings[-AMBIENT_SET_ERROR]
    := 'Ambient Data Error.  Please see Section 4 of the Users Manual regarding input file construction.';
  g_ErrorStrings[-PROFILE_SET_ERROR]
    := 'Source Profiles Error.  Please see Section 4 of the Users Manual regarding input file construction.';

// These 5 errors are never assigned to any condition in CMB82.c: // TC activated ERROR_READ_PROFILE 11/06/04
(*  g_ErrorStrings[-ERROR_READ_SPECIES_SEL]
    := 'Error reading species selection file.  Please see Section 4 of the Users Manual regarding input file construction.';
  g_ErrorStrings[-ERROR_READ_SOURCES_SEL]
    := 'Error reading source selection file.  Please see Section 4 of the Users Manual regarding input file construction.';
  g_ErrorStrings[-ERROR_READ_AMBDATA_SEL]
    := 'Error reading ambient data selection file.  Please see Section 4 of the Users Manual regarding input file construction.';
  g_ErrorStrings[-ERROR_READ_AMBDATA]
    := 'Error reading ambient data file.  Please see Section 4 of the Users Manual regarding input file construction.';
*)
  g_ErrorStrings[-ERROR_READ_PROFILE]
    := 'Error reading source profiles file.  Please see Section 4 of the Users Manual regarding input file construction.';
  g_ErrorStrings[-ERROR_MEMORY_FREE]
    := 'Memory free error.  Try shutting down other applications or increasing RAM.';

  g_SizeArray := StrAlloc((MAXSIZES * LENSIZE) + 1); // Misallocation of LENSIZE (it must now be 6; not 5) will trigger Runtime error 216 @ error from Windows -TC

finalization
  if Assigned(g_SizeArray) then
    StrDispose(g_SizeArray);
end.

