"""Routines for retrieving data from the PyWofost database.
 
Implements the following functions:
- fetch_cropdata()
- fetch_sitedata()
- fetch_soildata()
- fetch_timerdata()
- fetch_assimdata()

Implements the following classes:
- MeteoFetcher

Implements the following exceptions:
- MeteodataError
- CropdataError
- SitedataError
- SoildataError
- TimerdataError
- DfaError
"""
import sys, os
import datetime
import logging

from numpy import zeros, float32
from sqlalchemy import create_engine, MetaData, select, Table, and_
from sqlalchemy.sql.expression import func

from . import util

#-------------------------------------------------------------------------------
class MeteodataError(Exception):
    """Exception for EnsembleMeteoCl."""
    def __init__(self, value):
        self.value = value
    def __str__(self):
         return repr(self.value)

#----------------------------------------------------------------------------
class CropdataError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
         return repr(self.value)

#-------------------------------------------------------------------------------
class SitedataError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
         return repr(self.value)

#-------------------------------------------------------------------------------
class SoildataError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
         return repr(self.value)

#-------------------------------------------------------------------------------
class TimerdataError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
         return repr(self.value)

#-------------------------------------------------------------------------------
class DfaError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
         return repr(self.value)

#-------------------------------------------------------------------------------
def fetch_cropdata(metadata, grid, year, crop):
    """Retrieve crop parameter values for given grid, year, crop from DB.
    
    Parameter values are pulled from tables 'crop_parameter_value'
    and 'variety_parameter_value'. Metadata is an SQLAlchemy metadata object.
    
    Returns a dictionary with WOFOST crop parameter name/value pairs.
    """
    
    # Define a logger for the PyWofost db_util routines
    logger = logging.getLogger('PyWofost.db_util')

    # Create initial dictionary 
    cropdata={"GRID_NO": grid, "CROP_NO":crop, "YEAR":year}    

    # Get crop variety from crop_calendar;
    table_cc = Table('crop_calendar', metadata, autoload=True)
    s = select([table_cc],
               and_(table_cc.c.grid_no==grid,
                    table_cc.c.crop_no==crop,
                    table_cc.c.year==year))
    r = s.execute()
    rows = r.fetchall()
    r.close()
    if (rows is not None) and (len(rows) == 1):
        variety = rows[0].variety_no
    else:
        logger.error(("None or multiple crop definitions found for grid: %7i and crop " + \
                            "no: %3i and year: %5i") % (cgmsgrid, crop, year))
        raise CropdataError(("No crop definition found for grid: %7i and crop " + \
                            "no: %3i and year: %5i") % (cgmsgrid, crop, year))
    
    # Define crop parameter values
    parameter_codes_sngl = ("CFET","CVL","CVO","CVR","CVS","DEPNR","DLC",
                            "DLO","DVSEND","EFF","IAIRDU","IDSL","KDIF",
                            "LAIEM","PERDL","Q10","RDI","RDMCR","RGRLAI",
                            "RML","RMO","RMR","RMS","RRI","SPA","SPAN","SSA",
                            "TBASE","TBASEM","TDWI","TEFFMX","TSUM1","TSUM2",
                            "TSUMEM")
    parameter_codes_mltp = ("AMAXTB","DTSMTB","FLTB","FOTB","FRTB","FSTB",
                            "RDRRTB","RDRSTB","RFSETB","SLATB","TMNFTB",
                            "TMPFTB")
    
    # Pull single value parameters from CROP_PARAMETER_VALUE first
    table_crop_pv = Table('crop_parameter_value', metadata, autoload=True)
    for paramcode in parameter_codes_sngl:
        s = select([table_crop_pv], and_(table_crop_pv.c.crop_no==crop,
                                         table_crop_pv.c.parameter_code==paramcode))
        r = s.execute()
        rows = r.fetchall()
        if (len(rows) == 0):
            logger.error(("No crop parameter value found for:" + \
                          "crop %s, parameter: %s") % (crop, paramcode))
            raise CropdataError(("No crop parameter value found for:" + \
                                "crop %s, parameter: %s") % (crop, paramcode))
        elif (len(rows) == 1):
            cropdata[paramcode] = float(rows[0].parameter_xvalue)
        else:
            logger.error(("Multiple crop parameter values found for:" + \
                          "crop %s, parameter: %s") % (crop, paramcode))
            raise CropdataError(("Multiple crop parameter values found for:" + \
                                "crop %s, parameter: %s") % (crop, paramcode))
    logger.debug("Succesfully retrieved single value parameters "+\
                 "from CROP_PARAMETER_VALUE TABLE")
    
    # Pull array parameter values from CROP_PARAMETER_VALUE
    # note the change in the mask value and the use of "LIKE" in the SQL query
    for paramcode in parameter_codes_mltp:
        pattern = paramcode + r'%'
        s = select([table_crop_pv],
                   and_(table_crop_pv.c.crop_no == crop,
                        table_crop_pv.c.parameter_code.like(pattern)),
                   order_by=[table_crop_pv.c.parameter_code])
        r = s.execute()
        rows = r.fetchall()
        c = len(rows)
        if (c == 0):
            logger.error(("No crop parameter value found for:" + \
                          "crop %s, parameter: %s") % (crop, paramcode))
            raise CropdataError(("No crop parameter value found for:" + \
                                "crop %s, parameter: %s") % (crop, paramcode))
        elif (c == 1):
            logger.error(("Single crop parameter value found for:" + \
                          "crop %s, parameter: %s") % (crop, paramcode))
            raise CropdataError(("Single crop parameter value found for:" + \
                                "crop %s, parameter: %s") % (crop, paramcode))
        else:
            value = zeros(c*2, dtype=float32)
            for i in range(0,c):
                value[i*2] = rows[i].parameter_xvalue
                value[(i*2)+1]= rows[i].parameter_yvalue
            cropdata[paramcode] = value
    logger.debug("Succesfully retrieved array parameters "+\
                 "from CROP_PARAMETER_VALUE TABLE")

    # Pull same parameter values from VARIETY_PARAMETER_VALUES
    # if they are defined for that variety.
    # Pull single value parameters first
    table_var_pv = Table('variety_parameter_value', metadata, autoload=True)
    for paramcode in parameter_codes_sngl:
        s = select([table_var_pv],
                   and_(table_var_pv.c.variety_no==variety,
                        table_var_pv.c.crop_no==crop,
                        table_var_pv.c.parameter_code==paramcode))
        r = s.execute()
        rows = r.fetchall()
        c = len(rows)
        if (c == 0):
            pass
        elif (c == 1):
            cropdata[paramcode] = float(rows[0].parameter_xvalue)
        else:
            errstr = "Multiple values found for: crop: %s, variety: %s, " + \
                     "parameter: %s which is supposed to be a single value."
            logger.error(errstr % (crop, paramcode))
            raise CropdataError(errstr % (crop, paramcode))
    logger.debug("Succesfully retrieved single value parameters "+\
                 "from VARIETY_PARAMETER_VALUE TABLE")
            
    # pull array value parameters - note the change in the mask value and
    # the use of "LIKE" in the SQL query
    for paramcode in parameter_codes_mltp:
        pattern = paramcode + r'%'
        s = select([table_var_pv],
                   and_(table_var_pv.c.crop_no==crop,
                        table_var_pv.c.parameter_code.like(pattern),
                        table_var_pv.c.variety_no==variety),
                   order_by=[table_var_pv.c.parameter_code])
        r = s.execute()
        rows = r.fetchall();
        c = len(rows)
        if (c == 0):
            pass
        elif (c == 1):
            errstr = "Single value found for: crop: %s, variety: %s, " + \
                     "parameter: %s which is supposed to be a single value."
            logger.error(errstr % (crop, paramcode))
            raise CropdataError(errstr % (crop, paramcode))
        else:
            value = zeros(c*2, dtype=float32)
            for i in range(0,c):
                value[i*2] = rows[i].parameter_xvalue
                value[(i*2)+1]= rows[i].parameter_yvalue
            cropdata[paramcode] = value
    logger.debug("Succesfully retrieved array parameters "+\
                 "from VARIETY_PARAMETER_VALUE TABLE")

    # Make some specific changes for FORTRAN wofost with regard to variables
    # SSA, KDIF and EFF. This is needed because the FORTRAN code expects a
    # parameter array, while these parameters have been defined in CGMS as
    # single values. DVSI does not exist in CGMS, therefore set DVSI to zero.
    
    # SSA convert to SSATB:
    SSA = cropdata["SSA"]
    SSATB = zeros(4, dtype=float32)
    SSATB[1] = SSA
    SSATB[2] = 2.0
    SSATB[3] = SSA
    cropdata.update({"SSATB":SSATB})
    # KDIF convert to KDIFTB:
    KDIF = cropdata["KDIF"]
    KDIFTB = zeros(4, dtype=float32)
    KDIFTB[1] = KDIF
    KDIFTB[2] = 2.0
    KDIFTB[3] = KDIF
    cropdata.update({"KDIFTB":KDIFTB})
    # EFF convert to EFFTB
    EFF = cropdata["EFF"]
    EFFTB = zeros(4, dtype=float32)
    EFFTB[1] = EFF
    EFFTB[2] = 40.0
    EFFTB[3] = EFF
    cropdata.update({"EFFTB":EFFTB})
    # DVSI set to 0
    cropdata.update({"DVSI":0})
    
    logger.info("Succesfully retrieved crop parameter values from database")
    return cropdata

#-------------------------------------------------------------------------------
def fetch_soiltype(metadata, grid):
    """Retrieve soil parameters for given grid from DB.
    
    Retrieves majority soil type from the table SOIL_TYPE and associated rooting
    depth and soil physical data from tables ROOTING_DEPTH and
    SOIL_PHYSICAL_GROUP. If more then one majority soil exists, then
    select the one with the large water holding capacity (whc).
    
    Returns a dictionary with WOFOST soil parameter name/value pairs.
    """
    
    # Define a logger for the PyWofost db_util routines
    logger = logging.getLogger('PyWofost.db_util')

    try:
        # Select majority soil(s) from the table SOIL_TYPE
        table_soiltype = Table('soil_type', metadata, autoload=True)
        r = select([table_soiltype.c.grid_no,
                    table_soiltype.c.rooting_depth_class,
                    table_soiltype.c.soil_group_no],
                    table_soiltype.c.grid_no==grid).execute()
        rows = r.fetchall()
        if len(rows) == 0:
            raise RuntimeError("No record found!")

        # Define soil physical variable parameter codes
        soil_variables = ["CRITICAL_AIR_CONTENT", "HYDR_CONDUCT_SATUR", 
                          "MAX_PERCOL_ROOT_ZONE", "MAX_PERCOL_SUBSOIL",
                          "SOIL_MOISTURE_CONTENT_FC", "SOIL_MOISTURE_CONTENT_SAT",
                          "SOIL_MOISTURE_CONTENT_WP"]

        # Select soil properties and soil depth from tables. In case of
        # two soils select the one with largest whc.
        whc = 0
        table_rd = Table('rooting_depth',metadata, autoload=True)
        table_soil_pg = Table('soil_physical_group',metadata, autoload=True)
        for [grid, rootd_class, soilgroup] in rows:
            tmp_soildata = {}
            # Select rooting depth
            r = select([table_rd.c.maximum_rootable_depth],
                       table_rd.c.rooting_depth_class==rootd_class).execute()
            [rootdepth,] = r.fetchall()[0]
            # Select soil physical properties
            for soil_var in soil_variables:
                r = select([table_soil_pg], 
                           and_(table_soil_pg.c.soil_group_no==soilgroup,
                                table_soil_pg.c.parameter_code==soil_var)).execute()
                row = r.fetchall()
                tmp_soildata[soil_var] = row[0].parameter_xvalue
            # Determine water holding capacity
            whc_new = rootdepth * (tmp_soildata["SOIL_MOISTURE_CONTENT_FC"] - 
                                   tmp_soildata["SOIL_MOISTURE_CONTENT_WP"])
            if whc_new > whc:
                soildata={"GRID_NO": grid, "SOIL_GROUP_NO":soilgroup,
                          "ROOT_DEPTH_CLASS":rootd_class}
                soildata["K0"]     = float(tmp_soildata["HYDR_CONDUCT_SATUR"])
                soildata["SOPE"]   = float(tmp_soildata["MAX_PERCOL_ROOT_ZONE"])
                soildata["KSUB"]   = float(tmp_soildata["MAX_PERCOL_SUBSOIL"])
                soildata["CRAIRC"] = float(tmp_soildata["CRITICAL_AIR_CONTENT"])
                soildata["SM0"]    = float(tmp_soildata["SOIL_MOISTURE_CONTENT_SAT"])
                soildata["SMW"]    = float(tmp_soildata["SOIL_MOISTURE_CONTENT_WP"])
                soildata["SMFCF"]  = float(tmp_soildata["SOIL_MOISTURE_CONTENT_FC"])
                soildata["RDMSOL"] = float(rootdepth)
                whc = whc_new
    except Exception, e:
        errstr = "Failed to select majority soil type for grid %i: " + str(e)
        logger.error(errstr % grid)
        raise SoildataError(errstr % grid)

    logger.info("Succesfully retrieved soil parameter values from database")
    return soildata

    
#-------------------------------------------------------------------------------
def fetch_timerdata(metadata, grid, year, crop, wb_early_start=30):
    """Retrieve timerdata from DB for given year, crop, grid.
    
    Routine retrieves the data related to the timer of the system (sowing data,
    emergence date, start date of waterbalance) from the CROP_CALENDAR. Default
    is to start the water balance 30 days before the sowing date or emergence
    date.
    
    Returns a dictionary with WOFOST timer parameter name/value pairs.
    """

    # Define a logger for the PyWofost db_util routines
    logger = logging.getLogger('PyWofost.db_util')

    # Make initial output dictionary
    timerdata = {"GRID_NO":grid, "CROP_NO":crop, "CAMPAIGNYEAR": year}

    # Create and execute SQL query
    try:
        table_crop_calendar = Table('crop_calendar', metadata, autoload=True)
        r = select([table_crop_calendar],
                   and_(table_crop_calendar.c.grid_no==grid,
                        table_crop_calendar.c.year==year,
                        table_crop_calendar.c.crop_no==crop)).execute()
        row = r.fetchone()
        r.close()
        if row is not None:
            # Set start type
            timerdata.update({"CROP_START_TYPE": row.start_type})
    
            # Set start day, it wil be used as emergence or sowing, depending
            # on START_TYPE
            crop_start_date = row.start_day
            timerdata["CROP_START_DATE"] = crop_start_date
    
            # Start the waterbalance 30 days before the crop starts (default)
            start_date_wb = crop_start_date - datetime.timedelta(days=wb_early_start)
            timerdata["WB_START_DATE"] = start_date_wb
    
            # Set end of simulation type: fixed harvest or maturity
            timerdata["CROP_END_TYPE"] = row.end_type
            if (row.end_type == "MATURITY"):
                timerdata["MAX_DURATION"] = row.max_duration
                timerdata["CROP_END_DATE"] = crop_start_date + \
                                             datetime.timedelta(days=row.max_duration)
            elif (row.end_type == "HARVEST"):
            
                crop_end_date = row.end_day
                timerdata["CROP_END_DATE"] = crop_end_date
            else:
                errstr =  "Unknown end option: not maturity or harvest!"
                raise RuntimeError(errstr)
        else:
            raise RuntimeError
    except Exception, e:
        errstr = "Failed to retrieve timer data from crop_calendar for "+\
                 "grid %i, year %i, crop %i: " + str(e)
        logger.error(errstr % (cgmsgrid, year, crop))
        raise TimerdataError(errstr % (cgmsgrid, year, crop))

    logger.info("Succesfully retrieved timerdata from crop calendar from database")
    return timerdata

#-------------------------------------------------------------------------------
def fetch_sitedata(metadata, grid, year):
    """Retrieve site data from DB for given grid, year.
    
    Pulls sitedata from the PyWofost database 'SITE' table,
    
    Returns a dictionary with site parameter name/value pairs.
    """

    # Define a logger for the PyWofost db_util routines
    logger = logging.getLogger('PyWofost.db_util')

    try:
        #Get all settings from table 'SITE'
        table_site = Table('site', metadata, autoload=True)
        r = select([table_site],
                   and_(table_site.c.grid_no==grid,
                        table_site.c.year==year)
                   ).execute()
        row = r.fetchone()
        r.close()
        if row is not None:
            sitedata = {}
            sitedata['IFUNRN'] = row.ifunrn
            sitedata['SSMAX'] = row.max_surface_storage
            sitedata['NOTINF'] = row.not_infiltrating_fraction
            sitedata['SSI'] = row.initial_surface_storage
            sitedata['WAV'] = row.inital_water_availability
        else:
            raise RuntimeError("No rows found")
    except Exception, e:
        errstr = "Failed to get site data for year %i and grid %i: " + str(e)
        logger.error(errstr % (year, grid))
        raise SitedataError(errstr % (year, grid))

    logger.info("Succesfully retrieved site variables from database")
    return sitedata

#-------------------------------------------------------------------------------
def fetch_assimdata(metadata, grid_no, crop_no, startdate, enddate, params):
    """Retrieves data for assimilation from table DATA_FOR_ASSIMILATION.
    
    Retrieves observations for given grid_no, crop_no, startdate, enddata
    from table DATA_FOR_ASSIMILATION that should be assimilated into the model.
    
    metadata is an SQLAlchemy metadata instance.
    
    Params should be a dictionary that can be used to pass additional
    information for converting observations into the proper units. For example
    the soil physical properties for converting Soil Water Index into an
    estimate of volumetric soil moisture.
    """
    
    # Define a logger for the PyWofost db_util routines
    logger = logging.getLogger('PyWofost.db_util')

    try:
        table_dfa = Table('data_for_assimilation',metadata, autoload=True)
        # Get the data first
        s = select([table_dfa.c.day, table_dfa.c.observed_state,
                    table_dfa.c.value, table_dfa.c.variance],
                   and_(table_dfa.c.grid_no==grid_no,
                        table_dfa.c.crop_no==crop_no,
                        table_dfa.c.day>=startdate,
                        table_dfa.c.day<=enddate),
                   order_by=[table_dfa.c.day])
        datarows = s.execute()
        
        # Get the counts per day for processing
        s = select([table_dfa.c.day, func.count(table_dfa.c.day)],
                   and_(table_dfa.c.grid_no==grid_no,
                        table_dfa.c.crop_no==crop_no,
                        table_dfa.c.day>=startdate,
                        table_dfa.c.day<=enddate),
                   group_by=[table_dfa.c.day],
                   order_by=[table_dfa.c.day])
        countbyday = s.execute().fetchall()
        
    
        # Reprocess database output in order to have multiple observations on one
        # day under a single dictionary key: data_for_assimilation[<date1>,
        # <date2>, <date3>, etc] where <date?> should be unique and in ascending
        # order (as already requested from the database). 
        data_for_assimilation = []
        for [day, count] in countbyday:
            observations_per_day = []
            for dummy in range(count):
                [day, observed_state, value, variance] = datarows.fetchone()
                value = float(value)
                variance = float(variance)
                observations_per_day.append((day, observed_state, value, variance))
            #process observations_per_day containing data of a previous day    
            d = process_dfa(day, observations_per_day, params)
            data_for_assimilation.append(d)
    except Exception, e:
        errstr = "Failed to retrieve data for assimilation for grid %i and "+\
                 "period %s to %s: " + str(e)
        logger.error(errstr % (grid_no, startdate, enddate))
        raise DfaError(errstr % (grid_no, startdate, enddate))

    logger.info("Succesfully retrieved data for assimilation from database")
    return data_for_assimilation

#-------------------------------------------------------------------------------
def process_dfa(day, observations, params):
    """Restructure the observations from table DATA_FOR_ASSIMILATION.
    
    Converts the observations into the right structure and does some additional
    processing according to the parameters provided in params.
    """

    # Define a logger for the PyWofost db_util routines
    logger = logging.getLogger('PyWofost.db_util')

    obs = {}
    for [day, observed_state, value, variance] in observations:
        if observed_state == 'soil_water_index':
            par = params[observed_state]
            SMcf = (par["SMFCF"]+par["SM0"])/2. - par["SMW"]
            volSM = (value/100.)*SMcf + par["SMW"]
            volvariance = variance/10000.
            obs['profile_moisture_content'] = (volSM, volvariance)
        elif observed_state == 'leaf_area_index':
            obs[observed_state] = (value, variance)
        else:
            errstr = "Unknown type of observation: '%s' in table 'data_for_assimilation'"
            logger.error(errstr % observed_state)
            raise DfaError(errstr % observed_state)
    return (day, obs)

#----------------------------------------------------------------------------
class MeteoFetcher:
    """Retrieves meteodata from '[ensemble]_grid_weather' table(s).
    
    Arguments:
    metadata -- SqlAlchemy metadata object providing DB access
    grid_no -- Grid ID of PyWofost run
    startdate -- Retrieve meteo data starting with startdate (datetime.date
                 object)
    enddate -- Retrieve meteo data up to and including enddate (datetime.date
               object)
    
    Keyword arguments:
    ensemble_mode -- Switch on ensemble_mode (default False)
    ensemble_table -- DB table for retrieving meteo ensemble data
                      (default 'ensemble_grid_weather')
                      
    The following keyword arguments are only necessary if strict Wofost7.1
    compatibility is necessary, see also _make_dailymeteo2_dict():
    AngstA -- Angstrom A parameter
    AngstB -- Angstrom B parameter
    Elevation -- Elevation of site [m]
                      
    Public methods:
    __call__ -- Class instance can be called for returning meteo data.
    
     
    Additional details:
    If the keyword ensemble_mode is set to True, the class will switch to
    ensemble mode. Note that some edits may be necessary to adapt the function
    '_fetch_ensemble_weather_from_db()' for specific applications. Currently
    this function reads ensemble values for rainfall from a table called
    'ensemble_grid_weather', with the same structure as 'grid_weather' except
    for the ensemble_id.
    
    Note that all meteodata is first retrieved from the DB and stored
    internally. Therefore, no DB connections are stored within the class
    instance. This makes that class instances of MeteoFetcher can be pickled.
    """
    
    def __init__(self, metadata, grid_no, startdate, enddate, ensemble_mode=False,
                 ensemble_table="ensemble_grid_weather", AngstA=None,
                 AngstB=None, Elevation=None):

        # Define a logger for the PyWofost db_util routines
        self.logger = logging.getLogger('PyWofost.meteo')
        
        self.grid_no = grid_no
        self.startdate = startdate
        self.enddate = enddate
        self.timeinterval = enddate - startdate
        self.ensemble_mode = ensemble_mode
        self.ensemble_table = ensemble_table
        
        # Wofost 7.1 compabibility parameters
        self.AngstA = AngstA
        self.AngstB = AngstB
        self.Elevation = Elevation
        
        # Retrieved meteo data.
        self.latitude = self._fetch_latitude_from_db(metadata)
        self.grid_weather = self._fetch_grid_weather_from_db(metadata)
        if self.ensemble_mode is True:
            self.ensemble_weather = self._fetch_ensemble_weather_from_db(metadata)
            
#------------------------------------------------------------------------------
    def _fetch_latitude_from_db(self, metadata):
        """Retrieves latitude from 'grid' table."""

        # Get logger for the meteo routine
        logger = self.logger

        # Pull Latitude value for grid nr from database
        try:
            table_grid = Table('grid', metadata, autoload=True)
            r = select([table_grid.c.latitude],
                       table_grid.c.grid_no==self.grid_no).execute()
            row = r.fetchone()
            r.close()
            if row is None:
                raise RuntimeError("Grid %s not found" % self.grid_no)
        except Exception, e:
            errstr = "Could not derive latitude from table GRID for " + \
                     "grid %i:" + str(e)
            raise MeteodataError(errstr % self.grid_no)

        return row.latitude
        
#------------------------------------------------------------------------------
    def _fetch_grid_weather_from_db(self, metadata):
        """Retrieves the meteo data from table 'grid_weather'.
        
        Meteo data are stored internally in a hashtable."""
        
        # Get logger for the meteo routine
        logger = self.logger

        try:
            table_gw = Table('grid_weather', metadata, autoload=True)
            r = select([table_gw],and_(table_gw.c.grid_no==self.grid_no,
                                       table_gw.c.day>=self.startdate,
                                       table_gw.c.day<self.enddate)
                       ).execute()
            rows = r.fetchall()
            c = len(rows)
            if c < self.timeinterval.days:
                errstr =  "Only %i records selected from table grid_weather "+\
                          "for grid %i, period %s -- %s."
                raise RuntimeError(errstr % (c, self.grid_no, self.startdate,
                                               self.enddate))
            
            # Set meteopackager, if AngstA, AngstB en Elevation are set then 
            # choose the non-default packager.
            meteopackager = self._make_dailymeteo_dict
            if None not in (self.AngstA, self.AngstB, self.Elevation):
                meteopackager = self._make_dailymeteo2_dict

            # Store the results as a dictionary of dictionaries and return it
            grid_weather = {}
            for row in rows:
                grid_weather[row.day] = meteopackager(row)
        except Exception, e:
            errstr = "Failure reading meteodata: " + str(e)
            logger.error(errstr)
            raise MeteodataError(errstr)

        return grid_weather
    
#------------------------------------------------------------------------------
    def _fetch_ensemble_weather_from_db(self, metadata):
        """Retrieves the ensemble meteo data from table 'ensemble_grid_weather'.
        
        Meteo ensembles are stored internally in a hashtable."""

        # Get logger for the meteo routine
        logger = self.logger

        # Now get the ensemble meteodata from the database.
        try:
            table_egw = Table(self.ensemble_table, metadata, autoload=True)
            r = select([table_egw],
                       and_(table_egw.c.grid_no==self.grid_no,
                            table_egw.c.day>=self.startdate,
                            table_egw.c.day<=self.enddate)
                       ).execute()
            rows = r.fetchall()
            ensemble_weather = {}
            for row in rows:
                ensemble_weather[(row.day, row.ensemble_id)] = {"RAIN" : float(row.rainfall)/10.}
        except Exception, e:
            errstr = ("Failure reading ensemble grid weater from %s:" +\
                     str(e)) % self.ensemble_table
            logger.error(errstr)
            raise MeteodataError(errstr)

        return ensemble_weather
            
#------------------------------------------------------------------------------
    def _make_dailymeteo_dict(self, row):
        """Process record from grid_weather including unit conversion."""

        dmeteo={
        "TMAX"   : float(row.maximum_temperature),
        "TMIN"   : float(row.minimum_temperature),
        "VAP"    : float(row.vapour_pressure),
        "WIND"   : util.wind10to2(float(row.windspeed)),
        "RAIN"   : float(row.rainfall)/10.,
        "E0"     : float(row.e0)/10.,
        "ES0"    : float(row.es0)/10.,
        "ET0"    : float(row.et0)/10.,
        "IRRAD"  : float(row.calculated_radiation)*1000., 
        "LAT"    : float(self.latitude)}
        return dmeteo

#------------------------------------------------------------------------------
    def _make_dailymeteo2_dict(self, row):
        """As _make_dailymeteo_dict() but make ET compatible with Wofost 7.1

        Routine only added to make calculations compatible with Wofost 7.1,
        this only applies to E0, ET0, ES0 which are calculated slightly
        different in the grid_weather table of CGMS.
        
        Note that the AngstA, AngstB, Elevation parameters have to be provided
        when instancing MeteoEnsembleCl
        """
        if sys.platform == 'darwin':
            from py_w60lib_darwin import penman
        elif sys.platform == 'linux2':
            from py_w60lib_linux2 import penman
        elif sys.platform == 'win32':
            from py_w60lib_win32 import penman
        
        [ANGSTA,ANGSTB] = util.check_angstAB(self.AngstA, self.AngstB)
        WIND = util.wind10to2(row.windspeed)
        LAT = self.latitude
        ELEV = self.Elevation
        TMIN = float(row.minimum_temperature)
        TMAX = float(row.maximum_temperature)
        IRRAD = float(row.calculated_radiation)*1000.
        VAP = float(row.vapour_pressure)
        RAIN = float(row.rainfall)/10.
        [E0,ES0,ET0] = penman(doy,LAT,ELEV,ANGSTA,ANGSTB,TMIN,TMAX,
                              IRRAD,VAP,WIND)

        dmeteo={
        "TMAX"   : TMAX,
        "TMIN"   : TMIN,
        "VAP"    : VAP,
        "WIND"   : WIND,
        "RAIN"   : RAIN,
        "E0"     : E0/10.,
        "ES0"    : ES0/10.,
        "ET0"    : ET0/10.,
        "IRRAD"  : IRRAD, 
        "LAT"    : LAT}

        return dmeteo

#------------------------------------------------------------------------------
    def __call__(self, day, ensemble_id=0):
        """Fetches meteodata from the internal dictionaries and returns it.
        
        Arguments:
        day -- datetime.date object
        
        Keyword Arguments:
        ensemble_id -- ensemble ID for which meteo should be retrieved,
                       (default 0)
        """
        
        # Get logger for the meteo routine
        logger = self.logger
        
        try:
            try:
                daily_meteo = self.grid_weather[day]
                dbgstr = "Fetching grid weather for day %s. "+\
                         "Values: " + str(daily_meteo)
                logger.debug(dbgstr % day)
            except KeyError:
                errstr = "Grid weather for specified day (%s) not found"
                raise RuntimeError(errstr % day)
            if (self.ensemble_mode is True):
                try:
                    ensemble_meteo = self.ensemble_weather[(day, ensemble_id)]
                    dbgstr = "Fetching ensemble grid weather for "+\
                             "day %s, ensemble_id %s. Values: " + \
                             str(ensemble_meteo)
                    logger.debug((dbgstr) %(day, ensemble_id))
                    for key in ensemble_meteo.keys():
                        daily_meteo[key] = ensemble_meteo[key]
                except KeyError:
                    errstr = "Ensemble weather for specified (%s, %s)" +\
                            "not found!"
                    raise RuntimeError(errstr % (day,ensemble_id))
        except Exception, e:
            errstr = "Failure returning meteo data: " + str(e)
            logger.error(errstr)
            raise MeteodataError(errstr)
            
        return daily_meteo
