import os, os.path, sys
import wx
import ConfigParser

import matplotlib
matplotlib.interactive(False)
matplotlib.use("WXAgg")
import matplotlib.pyplot as plt
from matplotlib.figure import Figure
from matplotlib.dates import date2num, MonthLocator, WeekdayLocator, \
    DateFormatter
from matplotlib.backends.backend_wxagg import \
    FigureCanvasWxAgg as FigCanvas, \
    NavigationToolbar2WxAgg as NavigationToolbar
from sqlalchemy import select, create_engine, MetaData, Table, func

from ..util import is_data_column
from ..pyfortran import w60_variable_description as w60_var_desc
from .pywofostoutputprocessor import PyWofostOutputProcessor

#-------------------------------------------------------------------------------
class PyWofostAbout(wx.Dialog):
    text = '''
<html>
<body bgcolor="#009900">
<center><table bgcolor="#003366" width="100%" cellspacing="0"
cellpadding="0" border="1">
<tr>
    <td align="center"><h1><font color="sky blue">PyWofost GUIPlot</font></h1></td>
</tr>
</table>
</center>
<p><b>PyWofost GUIPlot</b> is a plotting program for <b>PyWofost</b>.
The plotting tools are derived from the matplotlib library and 
the graphical user interface is based on wxPython.
</p>
<p><b>PyWofost</b> and <b>PyWofost GuiPlot</b> were developed by
<b>Allard de Wit</b>. You can contact me at:
allard.dewit@wur.nl<br />
<br />
Copyright &copy; 2010.<br />

PyWofost and related tools are made available under the GNU
Public License v3, see http://www.gnu.org/licenses/gpl-3.0-standalone.html
for more information.

</p>
</body>
</html>
'''

    def __init__(self, parent):
        wx.Dialog.__init__(self, parent, -1, 'About Sketch',
                          size=(440, 400) )

        html = wx.html.HtmlWindow(self)
        html.SetPage(self.text)
        button = wx.Button(self, wx.ID_OK, "Close")
        sizer = wx.BoxSizer(wx.VERTICAL)
        sizer.Add(html, 1, wx.EXPAND|wx.ALL, 5)
        sizer.Add(button, 0, wx.ALIGN_CENTER|wx.ALL, 5)
        self.SetSizer(sizer)
        self.Layout()

#-------------------------------------------------------------------------------
class EmptyPanel(wx.Panel):
    def __init__(self, parent=None, ID=wx.ID_ANY, label="DBSelector", 
                 pos=wx.DefaultPosition, size=wx.DefaultSize):
        wx.Panel.__init__(self, parent, ID, pos, size, wx.NO_BORDER)

#-------------------------------------------------------------------------------
class MPLPlotPanel(wx.Panel):

    def __init__(self, parent=None):
        wx.Panel.__init__(self, parent, wx.ID_ANY, size=(600,400),
                          style=wx.NO_BORDER)

        # Since we have only one plot, we can use add_axes 
        # instead of add_subplot, but then the subplot
        # configuration tool in the navigation toolbar wouldn't
        # work.      
        self.dpi = 100
        self.fig = Figure((5.0, 4.0), dpi=self.dpi)
        self.canvas = FigCanvas(self, wx.ID_ANY, self.fig)
        self.axes = self.fig.add_subplot(111)
        
        # Bind the 'pick' event for clicking on one of the bars
        self.canvas.mpl_connect('pick_event', self.OnPick)

        # Create the navigation toolbar, tied to the canvas
        self.toolbar = NavigationToolbar(self.canvas)
    
        self.sizer = wx.FlexGridSizer(rows=2, cols=1,hgap=0, vgap=0)
        self.SetSizer(self.sizer)
        self.sizer.Add(self.canvas, border=0, flag=wx.EXPAND)
        self.sizer.Add(self.toolbar, border=0, flag=wx.EXPAND)
        self.sizer.AddGrowableCol(0)
        self.sizer.AddGrowableRow(0)
        self.sizer.SetMinSize((600,400))
        self.Fit()

    def OnPick(self): pass
    def DrawFigure(self, dc, varname):
        """Draws variable (varname) based on data in datacontainer (dc)."""

        # Get some metadata for plotting
        fullvar = w60_var_desc.get_full_name(varname)
        title = w60_var_desc.get_title_name(fullvar)
        ylabel = w60_var_desc.get_unit(fullvar)

        # Locator and formatter for dates on X axis
        ticklocator = MonthLocator(range(1,13), bymonthday=1, interval=1)
        tickformatter = monthsFmt = DateFormatter("%b '%y")

        axes = self.axes
        axes.clear()        
        axes.grid(True)
        lines = []
        if hasattr(dc, "mean"): # data consists of an ensemble:
            for i in range(dc.values.shape[1]):
                lines.append(axes.plot_date(dc.days, dc.values[:, i]))

            # Set line type, width, color for ensemble members
            plt.setp(lines, linewidth=0.25, color='k', linestyle=":", marker='')

            # Plot ensemble mean and stdev
            axes.plot_date(dc.days, dc.mean, '-', color='r',
                                   linewidth=1.5,
                                   linestyle="-", marker="")
            lines = []
            lines += axes.plot_date(dc.days, (dc.mean+dc.stdev))
            lines += axes.plot_date(dc.days, (dc.mean-dc.stdev))
            plt.setp(lines, linewidth=1.0, color='r',
                     linestyle="--", marker="")
            pass
        else: # data does consist of a single run.
            lines.append(axes.plot_date(dc.days, dc.values))
            # Set line type, width, color for single run
            plt.setp(lines, linewidth=1.5, color='k',
                     linestyle='-', marker="")

        self.axes.xaxis.set_major_locator(ticklocator)
        self.axes.xaxis.set_major_formatter(tickformatter)
        self.axes.set_ylabel(ylabel)
        self.axes.set_title(title)
        self.fig.autofmt_xdate()
        self.canvas.draw()

#-------------------------------------------------------------------------------
class DBSelectorPanel(wx.Panel):

    def __init__(self, parent=None, tbl=None, ID=wx.ID_ANY, label="DBSelector", 
                 pos=wx.DefaultPosition, size=wx.DefaultSize):
        
        # Set up the panel and sizer
        wx.Panel.__init__(self, parent, ID, pos, size, wx.NO_BORDER, label)
        vbox = wx.BoxSizer(wx.VERTICAL)
        self.SetSizer(vbox)

        if not isinstance(tbl, Table):
            msg = "DBSelectorPanel initiated without providing an SQLAlchemy "+\
                  "Table instance."
            raise RuntimeError(msg)

        self.create_grid_selector(vbox, tbl)
        self.create_crop_selector(vbox, tbl)
        self.create_year_selector(vbox, tbl)
        self.create_productionlvl_radiobox(vbox)
        self.tbl_pw_output = tbl
        
        # Add query button
        self.queryDBbutton = wx.Button(self, wx.ID_ANY, "Query DB")
        vbox.Add(self.queryDBbutton, flag=wx.ALL | wx.ALIGN_CENTER_HORIZONTAL,
                 border=5)
        
        self.Fit()

    def create_grid_selector(self, vbox, tbl):
        s = select([func.distinct(tbl.c.grid_no)])
        r = s.execute().fetchall()
        choices = [t[0] for t in r]
        selector = SelectorPanel(parent=self, text="Select grid",
                                 choices=choices)
        #vbox.Add(selector, flag=wx.ALL | wx.ALIGN_LEFT, border=5)
        vbox.Add(selector, flag=wx.ALL | wx.ALIGN_CENTER_HORIZONTAL, border=5)
        self.Bind(wx.EVT_CHOICE, self.OnSelect, selector.selector)
        self.grid_selector=selector

    def create_crop_selector(self, vbox, tbl):
        s = select([func.distinct(tbl.c.crop_no)])
        r = s.execute().fetchall()
        choices = [t[0] for t in r]
        selector = SelectorPanel(parent=self, text="Select crop",
                                 choices=choices)
        vbox.Add(selector, flag=wx.ALL | wx.ALIGN_CENTER_HORIZONTAL, border=5)
        self.Bind(wx.EVT_CHOICE, self.OnSelect, selector.selector)
        self.crop_selector = selector

    def create_year_selector(self, vbox, tbl):
        s = select([func.distinct(tbl.c.year)])
        r = s.execute().fetchall()
        choices = [t[0] for t in r]
        selector = SelectorPanel(parent=self, text="Select year",
                                 choices=choices)
        vbox.Add(selector, flag=wx.ALL | wx.ALIGN_CENTER_HORIZONTAL, border=5)
        self.Bind(wx.EVT_CHOICE, self.OnSelect, selector.selector)
        self.year_selector = selector
        
    def create_productionlvl_radiobox(self, vbox):
        radiobox = RadioBoxPanel(parent=self, label="Prod lvl",
                                 choices=["pp","wlp"])
        vbox.Add(radiobox, flag=wx.ALL | wx.ALIGN_CENTER_HORIZONTAL, border=5)
        self.Bind(wx.EVT_RADIOBOX, self.OnSelect, radiobox.radiobox)
        self.prodlvl = radiobox
    
    def OnSelect(self, event):
        print "OnSelect executed."
        event.Skip()
        
    def GetSelectorStates(self):
        states = (self.grid_selector.state,
                  self.crop_selector.state,
                  self.year_selector.state,
                  self.prodlvl.state)
        return states
        
#-------------------------------------------------------------------------------
class VariableSelectorPanel(wx.Panel):

    def __init__(self, parent=None, ID=wx.ID_ANY, text="", choices=None, disable=False,
                 label="VariableSelector", pos=wx.DefaultPosition,
                 size=wx.DefaultSize):
        
        # Set up the panel and sizer
        wx.Panel.__init__(self, parent, ID, pos, size, wx.NO_BORDER, label)
        self.label = label
        sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.SetSizer(sizer)
          
        # Check choices and convert to list of strings
        if not isinstance(choices, list):
            raise RuntimeError("Choices should be list")
        self.choices = choices
        strchoices = [str(s) for s in choices]

        # setup the text label, pull-down selector and plot button
        label_txt = wx.StaticText(self, wx.ID_ANY, text)
        self.selector = wx.Choice(self, wx.ID_ANY, choices=strchoices)
        self.PlotButton = wx.Button(self, wx.ID_ANY, "Plot Variable")

        # Bind selector for setting state
        self.selector.Bind(wx.EVT_CHOICE, self.OnSelect)
                
        # Set first option as default state
        if len(choices) >= 1:
            self.state = choices[0]
            self.selector.SetSelection(0)

        # Add to sizers
        sizer.Add(label_txt, flag=wx.ALIGN_CENTER_VERTICAL)
        sizer.Add((5,0))
        sizer.Add(self.selector)
        sizer.Add((5,0))
        sizer.Add(self.PlotButton)
        self.Fit()
        
        if disable is True:
            self.Disable()

    def OnSelect(self, event):
        n = self.selector.GetSelection()
        self.state = self.choices[n]
        print "Selected from pulldown: %s" % self.state

    def GetState(self):
        return self.state
#-------------------------------------------------------------------------------
class RadioBoxPanel(wx.Panel):

    def __init__(self, parent=None, label="", choices=None, disable=False,
                 **kwargs):
        wx.Panel.__init__(self, parent, wx.ID_ANY, style=wx.NO_BORDER)
        # Set up the panel and sizer
        sizer = wx.BoxSizer(wx.VERTICAL)
        self.SetSizer(sizer)

        # Check choices and convert to list of strings
        if not isinstance(choices, list):
            raise RuntimeError("Choices should be list")
        self.choices = choices
        strchoices = [str(s) for s in choices]

        self.radiobox = wx.RadioBox(self, wx.ID_ANY, label="Prod. lvl",
                            choices=strchoices, style=wx.RA_SPECIFY_COLS,
                            majorDimension=1, **kwargs)
        sizer.Add((0,5))
        sizer.Add(self.radiobox, flag=wx.ALIGN_CENTER_HORIZONTAL)
        self.Fit()
        
        self.Bind(wx.EVT_RADIOBOX, self.OnSelect, self.radiobox)
        
        # Select first option by default.
        self.state = choices[0]
        self.radiobox.SetSelection(0)

        if disable is True:
            self.Disable()
        
    def OnSelect(self, event):
        n = self.radiobox.GetSelection()
        self.state = self.choices[n]
        print self.state
        event.Skip()

#-------------------------------------------------------------------------------
class SelectorPanel(wx.Panel):
    """Pull-down list with text label for making a selection."""
    
    def __init__(self, parent=None, text="", choices=None, disable=False,
                 **kwargs):
        wx.Panel.__init__(self, parent, wx.ID_ANY, style=wx.NO_BORDER)

        # Set up the panel and sizer
        sizer = wx.BoxSizer(wx.VERTICAL)
        self.SetSizer(sizer)
          
        # Check choices and convert to list of strings
        if not isinstance(choices, list):
            raise RuntimeError("Choices should be list")
        self.choices = choices
        strchoices = [str(s) for s in choices]

        # setup the text label and pull-down selector
        label_txt = wx.StaticText(self, wx.ID_ANY, text)
        self.selector = wx.Choice(self, wx.ID_ANY, choices=strchoices, **kwargs)
        self.selector.Bind(wx.EVT_CHOICE, self.OnSelect)
        
        # Add to sizers
        sizer.Add(label_txt, flag=wx.ALIGN_CENTER_HORIZONTAL)
        sizer.Add(self.selector, flag=wx.ALIGN_CENTER_HORIZONTAL)
        sizer.Add((0,5))
        self.Fit()

        # Set first option as default state
        if len(choices) >= 1:
            self.state = choices[0]
            self.selector.SetSelection(0)
        
        if disable is True:
            self.Disable()

    def OnSelect(self, event):
        n = self.selector.GetSelection()
        self.state = self.choices[n]
        print "Selected from pulldown: %s" % self.state
        event.Skip()
    
#-------------------------------------------------------------------------------
class PyWofostFrame(wx.Frame):
    """Frame for plotting PyWofost results."""

    def __init__(self, parent=None, id=wx.ID_ANY, pos=wx.DefaultPosition,
                 title='PyWofost Plotting Tool'):
        wx.Frame.__init__(self, parent, id, title, pos)
        self.sizer = wx.FlexGridSizer(rows=2, cols=2, vgap=2, hgap=2)
        self.SetSizer(self.sizer)
        
        self.create_menubar()
        self.create_statusbar()

        # Empty placeholders to hold and cache plot data
        self.plot_data_cache = {}
        self.plot_data = None
        
        # Open connection to data source
        self.open_data_source()
        
        # DB selector panel
        self.DBSelPanel = DBSelectorPanel(self, tbl=self.tbl_pw_output)
        self.sizer.Add(self.DBSelPanel, flag=wx.EXPAND)

        # Bind button event on QueryDB button
        self.Bind(wx.EVT_BUTTON, self.OnDBQuery, self.DBSelPanel.queryDBbutton)

        # Bind choice events from selectors to function for disable/enable 
        # variable selection panel
        self.Bind(wx.EVT_CHOICE, self.OnSelectionChanged,
                  self.DBSelPanel.grid_selector.selector)
        self.Bind(wx.EVT_CHOICE, self.OnSelectionChanged,
                  self.DBSelPanel.crop_selector.selector)
        self.Bind(wx.EVT_CHOICE, self.OnSelectionChanged,
                  self.DBSelPanel.year_selector.selector)
        self.Bind(wx.EVT_RADIOBOX, self.OnSelectionChanged,
                  self.DBSelPanel.prodlvl.radiobox)

        # Add panel where the plot will go
        self.MPLpanel = MPLPlotPanel(self)
        self.sizer.Add(self.MPLpanel, flag=wx.EXPAND)
        
        # Add empty panel in first column, second row
        self.sizer.Add(EmptyPanel(self), flag=wx.EXPAND)
        
        # Add variable selector in row 1, spanning 3 columns
        data_cols = []
        tbl = self.tbl_pw_output
        for col in tbl.columns:
            if is_data_column(col):
                data_cols.append(col.name)
        self.VarSelector = VariableSelectorPanel(self, choices=data_cols,
                                                 text="Select:", disable=True)
        self.sizer.Add(self.VarSelector, flag=wx.EXPAND)
        self.Bind(wx.EVT_BUTTON, self.OnPlotVariable, self.VarSelector.PlotButton)
        
        # Make first row, second column growable
        self.sizer.AddGrowableRow(0)
        self.sizer.AddGrowableCol(1)

        # Fit the frame to its content
        self.Fit()

        # Bind the close event
        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow)
    
    def open_data_source(self):
        """Returns Data Source Name -DSN- for SQLAlchemy
        """
        dsn = None
        # Read config file first for default DSN
        config_file = os.path.expanduser("~/.pywofost.cnf")
        if os.path.exists(config_file):
            config = ConfigParser.RawConfigParser()
            config.read(config_file)
            t = config.get('Database', 'DSN')
            msg = "Use default DSN:\n %s" % t
            dlg = wx.MessageDialog(None, msg, style=wx.YES_NO)
            retcode = dlg.ShowModal()
            if retcode == wx.ID_YES:
                dsn = t

        # Open database connection, retry if DSN fails
        tbl_pw_output = None
        msg = "Specify data source name"
        while not isinstance(tbl_pw_output, Table):
            # Get SQLAlchemy DSN string
            if dsn is None:
                installdir = os.path.dirname(os.path.abspath(__file__))
                db_location = os.path.join(installdir, "..", "database","pywofost.db")
                defaultdsn = "sqlite:///" + os.path.abspath(db_location)
                dialog = wx.TextEntryDialog(None, msg,
                         caption="Database connection",
                         defaultValue=defaultdsn, style=wx.OK|wx.CANCEL)
                if dialog.ShowModal() == wx.ID_OK:
                    dsn = dialog.GetValue()
                else:
                    wx.MessageBox("DSN Selection Cancelled: Exit",
                        caption="Database connection",
                        style=wx.OK|wx.ICON_EXCLAMATION)
                    sys.exit()
            # Open database connection with selected DSN
            try:
                engine = create_engine(dsn)
                metadata = MetaData(engine)
                tbl_pw_output = Table('pywofost_output', metadata, autoload=True)
            except:
                dsn = None
                msg = "Connection failed, specify DSN"
        
        self.dsn = dsn
        self.tbl_pw_output = tbl_pw_output

    def create_statusbar(self):
        self.statusbar = self.CreateStatusBar()

    def menuData(self):
        return (("&File",
                    ("&Open DB", "Open database connection.", self.OnOpenDB),
                    ("&Save as Default","Save as default database.",self.OnSaveDBDefault),
                    ("&Quit","Quit tool", self.OnCloseWindow)),
                ("&Help",
                    ("&About", "About plotting tool.", self.OnAbout)))

    def create_menubar(self):
        menuBar = wx.MenuBar()
        for eachMenuData in self.menuData():
            menuLabel = eachMenuData[0]
            menuItems = eachMenuData[1:]
            print "Creating menu: %s" % menuLabel
            menuBar.Append(self.createMenu(menuItems), menuLabel)
        self.SetMenuBar(menuBar)
        
    def createMenu(self, menuItems):
        menu = wx.Menu()
        for eachLabel, eachStatus, eachHandler in menuItems:
            if not eachLabel:
                menu.ApppendSeparator()
                continue
            print "Create menu entry: %s" % eachLabel
            menuItem = menu.Append(wx.ID_ANY, eachLabel, eachStatus)
            self.Bind(wx.EVT_MENU, eachHandler, menuItem)
        return menu
    
    # Grouping the event handlers
    def OnOpenDB(self, event): pass
    
    def OnDBQuery(self, event):
        k = self.DBSelPanel.GetSelectorStates()
        if k in self.plot_data_cache:
            self.statusbar.SetStatusText("Data retrieved from cache")
            self.plot_data = self.plot_data_cache[k]
            self.VarSelector.Enable()
        else:
            self.statusbar.SetStatusText("Quering database")
            try:
                p = PyWofostOutputProcessor(grid_no=k[0], year=k[2],
                        crop_no=k[1], simulation_mode=k[3],
                        tbl=self.tbl_pw_output)
                self.plot_data = p
                self.plot_data_cache[k] = p
                self.VarSelector.Enable()
                msg = "Succesfully retrieved data from database"
                self.statusbar.SetStatusText(msg)
            except Exception, e:
                msg = "No data in database for given selection"
                self.statusbar.SetStatusText(msg)

    def OnSelectionChanged(self, event):
        #print "OnSelectionChanged Executed."
        k = self.DBSelPanel.GetSelectorStates()
        if k not in self.plot_data_cache:
            self.VarSelector.Disable()
        else:
            self.statusbar.SetStatusText("Data retrieved from cache")
            self.plot_data = self.plot_data_cache[k]
            self.VarSelector.Enable()
        
    def OnPlotVariable(self, event):
        #print "running OnPlotVariable."
        varname = self.VarSelector.GetState()
        dc = self.plot_data.get_variable_data(varname)
        self.MPLpanel.DrawFigure(dc, varname)

    def OnSaveDBDefault(self, event):
        """Saves the current DSN as default one.
        """
        config_file = os.path.expanduser("~/.pywofost.cnf")
        config = ConfigParser.RawConfigParser()
        config.add_section('Database')
        config.set('Database', 'DSN', self.dsn)
        config.write(open(config_file,'wb'))

    def OnAbout(self, event):
        """Opens the About box
        """
        print "Opening About box."
        dlg = PyWofostAbout(self)
        dlg.ShowModal()
        dlg.Destroy()

    def OnCloseWindow(self, event): 
        self.Destroy()        

class PyWofostGuiPlot(wx.App):
    
    def __init__(self, redirect=True, filename=None):
        #print "App __init__"
        wx.App.__init__(self, redirect, filename)
        
    def OnInit(self):
        #print "OnInit"
        self.frame = PyWofostFrame()
        self.frame.Show()
        self.SetTopWindow(self.frame)
        return True
    
    def OnExit(self):
        pass
        #print "OnExit"

def start_guiplot():
    "Starts the pywofost GUI plotting app."
    app = PyWofostGuiPlot(filename='log.txt')
    app.MainLoop()

if __name__ == '__main__':
    start_guiplot()
