"""Module for storing information about PyWofost variables.

Functions defined here:
* get_full_name()
* get_FORTRAN_name()
* get_title_name()
* get_unit()
* check_variable()
* print_variables()
* export_as_LaTeX()

Attributes defined here:
* variable_descriptions - describes attributes of a full variable name.
* short_variable_descriptions - maps abbreviate names to full variable names.
"""

short_variable_descriptions = {
    "lai":"leaf_area_index",
    "dvs":"development_stage",
    "tagp":"total_aboveground_biomass",
    "twso":"total_weight_storage_organs",
    "twl":"total_weight_leaves",
    "tws":"total_weight_stems",
    "twr":"total_weight_roots",
    "rmc":"rootzone_moisture_content",
    "pmc":"profile_moisture_content",
    "rd":"rooting_depth",
    "tra":"transpiration",
    "ptra":"potential_transpiration",
    "danth":"day_of_anthesis",
    "dtsum":"temperature_sum_increase",
    "dvr":"development_rate",
    "pgass":"pot_gross_assimilation_rate",
    "gass":"gross_assimilation_rate",
    "mres":"maintenance_respiration",
    "dmi":"dry_matter_increase",
    "admi":"aboveground_dry_matter_increase",
    "grrt":"growth_rate_roots",
    "drrt":"death_rate_roots",
    "gwrt":"net_growth_roots",
    "grlv":"growth_rate_leaves",
    "drlv":"death_rate_leaves",
    "grst":"growth_rate_stems",
    "drst":"death_rate_stems",
    "gwst":"net_growth_stems",
    "gwso":"growth_rate_storage_organs",
    "tsum":"temperature_sum",
    "wrt":"weight_living_roots",
    "wlv":"weight_living_leaves",
    "wst":"weight_living_stems",
    "wso":"weight_living_storage_organs",
    "tadw":"total_aboveground_living_biomass",
    "dwrt":"weight_dead_roots",
    "dwlv":"weight_dead_leaves",
    "dwst":"weight_dead_stems",
    "dwso":"weight_dead_storage_organs",
    "gasst":"total_gross_assimilation",
    "mrest":"total_maintainance_respiration",
    "trat":"total_transpiration",
    "idwst":"total_water_stress_days",
    "wwlow":"profile_water_depth",
    "w":"rootzone_water_depth",
    "wlow":"subsoil_water_depth"
}

variable_descriptions = {
    "development_stage":("dvs", "Development Stage", "-"),
    "leaf_area_index":("lai", "Leaf Area Index", "-"),
    "total_aboveground_biomass":("tagp", "Total Aboveground Biomass", "kg ha-1"),
    "total_weight_storage_organs":("twso", "Total Weight Storage Organs", "kg ha-1"),
    "total_weight_leaves":("twlv", "Total Weight Leaves", "kg ha-1"),
    "total_weight_stems":("twst", "Total Weight Stems", "kg ha-1"),
    "total_weight_roots":("twrt", "Total Weight Roots", "kg ha-1"),
    "rootzone_moisture_content":("rmc", "Rootzone Moisture Content", "-"),
    "profile_moisture_content":("pmc", "Profile Moisture Content", "-"),
    "rooting_depth":("rd", "Rooting Depth", "cm"),
    "transpiration":("tra", "Transpiration", "cm"),
    "potential_transpiration":("tramx", "Potential Transpiration", "cm"),
    "day_of_anthesis":("idanth", "Day of Anthesis", "-"),
    "temperature_sum_increase":("dtsum", "Temperature Sum Increase", "degree"),
    "development_rate":("dvr", "Development Rate", "day-1"),
    "pot_gross_assimilation_rate":("pgass", "Potential Gross Assimilation Rate", "kg CH2O day-1"),
    "gross_assimilation_rate":("gass", "Gross Asssimilation Rate", "kg CH2O day-1"),
    "maintenance_respiration":("mres", "Maintenance Respiration", "kg CH2O day-1"),
    "dry_matter_increase":("dmi", "Dry Matter Increase", "kg ha-1 day-1"),
    "aboveground_dry_matter_increase":("admi", "Aboveground Dry Matter Increase", "kg ha-1 day-1"),
    "growth_rate_roots":("grrt", "Growth Rate Roots", "kg ha-1 day-1"),
    "death_rate_roots":("drrt", "Death Rate Roots", "kg ha-1 day-1"),
    "net_growth_roots":("gwrt", "Net Growth Roots", "kg ha-1 day-1"),
    "growth_rate_leaves":("grlv", "Growth Rate Leaves", "kg ha-1 day-1"),
    "death_rate_leaves":("drlv", "Death Rate Leaves", "kg ha-1 day-1"),
    "growth_rate_stems":("grst", "Growth Rate Stems", "kg ha-1 day-1"),
    "death_rate_stems":("drst", "Death Rate Stems", "kg ha-1 day-1"),
    "net_growth_stems":("gwst", "Net Growth Stems", "kg ha-1 day-1"),
    "growth_rate_storage_organs":("gwso", "Growth Rate Storage Organs", "kg ha-1 day-1"),
    "temperature_sum":("tsum", "Temperature Sum", "degree day"),
    "weight_living_roots":("wrt", "Weight Living Roots", "kg ha-1"),
    "weight_living_leaves":("wlv", "Weight Living Leaves", "kg ha-1"),
    "weight_living_stems":("wst", "Weight Living Stems", "kg ha-1"),
    "weight_living_storage_organs":("wso", "Weight Living Storage Organs", "kg ha-1"),
    "total_aboveground_living_biomass":("tadw", "Total Aboveground Living Biomass", "kg ha-1"),
    "weight_dead_roots":("dwrt", "Weight Dead Roots", "kg ha-1"),
    "weight_dead_leaves":("dwlv", "Weight Dead Leaves", "kg ha-1"),
    "weight_dead_stems":("dwst", "Weight Dead Stems", "kg ha-1"),
    "weight_dead_storage_organs":("dwso", "Weight Dead Storage Organs", "kg ha-1"),
    "total_gross_assimilation":("gasst", "Total Gross Assimilation", "kg CH2O"),
    "total_maintainance_respiration":("mrest", "Total Maintenance Respiration", "kg CH2O"),
    "total_transpiration":("trat", "Total Transpiration", "cm"),
    "total_water_stress_days":("idwst", "Total Water Stress Days", "-"),
    "profile_water_depth":("wwlow", "Profile Water Depth", "cm"),
    "rootzone_water_depth":("w", "Rootzone Water Depth", "cm"),
    "subsoil_water_depth":("wlow", "Subsoil Water Depth", "cm")
    }

def get_full_name(varname):
    """Returns the fully qualified variable name.
    """
    if varname in short_variable_descriptions:
        return short_variable_descriptions[varname]
    elif varname in variable_descriptions:
        return varname
    else:
        msg = "Unknown Wofost variable"
        raise KeyError(msg)

def get_FORTRAN_name(varname):
    """Returns FORTRAN name for PyWofost variable."""
    if varname in short_variable_descriptions:
        t = short_variable_descriptions[varname]
        return variable_descriptions[t][0]
    elif varname in variable_descriptions:
        return variable_descriptions[varname][0]
    else:
        msg = "Unknown Wofost variable"
        raise KeyError(msg)

def get_title_name(varname):
    """Returns title for PyWofost variable."""
    if varname in short_variable_descriptions:
        t = short_variable_descriptions[varname]
        return variable_descriptions[t][1]
    elif varname in variable_descriptions:
        return variable_descriptions[varname][1]
    else:
        msg = "Unknown Wofost variable"
        raise KeyError(msg)

def get_unit(varname):
    """Get units for PyWofost variable."""
    if varname in short_variable_descriptions:
        t = short_variable_descriptions[varname]
        return variable_descriptions[t][2]
    elif varname in variable_descriptions:
        return variable_descriptions[varname][2]
    else:
        msg = "Unknown Wofost variable"
        raise KeyError(msg)

def check_variable(varname):
    """Returns True if variable name is valid"""
    if varname in variable_descriptions:
        return True
    elif varname in short_variable_descriptions:
        return True
    else:
        return False    

def print_variables():
    """Print short and long names of available PyWofost variables to screen.
    """
    keys = short_variable_descriptions.keys()
    keys.sort()
    print "Available variables in PyWofost (short - long):"
    for shortvar in keys:
        longvar = short_variable_descriptions[shortvar]
        print "%5s - %35s" % (shortvar, longvar)

def export_as_LaTeX():
    "prints information over variables as LaTeX ctable to console"
    
    print """\\ctable[
caption = Overview of PyWofost variables.,
label = table:pywofost_variables
]{lll}{}{\FL
\emph{Short name} & \emph{Long name} & \emph{Unit}\\\\
\hline"""
    keys = short_variable_descriptions.keys()
    keys.sort()
    for shortname in keys:
        longname = get_full_name(shortname)
        plongname = longname.replace("_","\_")
        unit = get_unit(longname)
        print "%s & %s & %s\\\\" % (shortname, plongname, unit)
    print "\LL}\n"

if __name__ == '__main__':
    export_as_LaTeX()
    