"""Module for running and ensemble of PyWofost models.

Classes defined here:
* PyWofostEnsemble
"""
import sys, os
import types
import logging
import sqlalchemy

from .pywofost import PyWofost

class PyWofostEnsemble:
    """PyWOFOST ensemble class for running an ensemble of WOFOST models.
    
    Arguments:
    sitedata  -- Dictionary containing site variables
    timerdata -- Dictionary containing time variables
    soildata  -- Dictionary containing soil variables
    cropdata  -- Dictionary containing crop variables
    meteof    -- Instance of MeteoFetcher, which returns meteodata:
                 r = meteof(day [, ensemble_id=sim_id])
    ensemble_size -- number of members of ensemble
                 
    Keyword arguments:
    mode      -- Initialise for potential (mode='PP') or water-limited
                 (mode='WLP') production. Defaults to 'WLP'.
    E****data -- dictionaries that can be used for providing ensemble
                 members with different start values for parameters.
                 The E****data variables consist of a tuple of ensemble_size
                 containing dictionaries of same structure as the ****data
                 dictionaries.
    metadata -- SQLAlchemy metadata object. Is used for deriving the column
                names in the table 'pywofost_output'. The column names are
                used to derive the WOFOST state/rate variables during the
                simulation. If metadata is not provide then a default
                set of variables is kept during the simulation.
    
    Public methods:
    - ensemble_grow()
    - ensemble_results_to_output()
    """

    def __init__(self, sitedata, timerdata, soildata, cropdata, meteof,
                 ensemble_size, mode="WLP", Esitedata=None, Etimerdata=None,
                 Esoildata=None, Ecropdata=None, metadata=None):

        # Get logging object
        self.logger = logging.getLogger("PyWofost.Ensemble")
        
        self.sitedata = sitedata
        self.timerdata = timerdata
        self.soildata = soildata
        self.cropdata = cropdata
        self.meteof = meteof
        self.ensemble_size = ensemble_size
        self.mode = mode

        self.Esitedata = Esitedata
        self.Etimerdata = Etimerdata
        self.Esoildata = Esoildata
        self.Ecropdata = Ecropdata
        
        # Counter for iterator
        self.i = 0

        #Initialize ensemble
        self.ensemble = self._initialize_wofost_objects(ensemble_size, metadata)
        self.logger.info("PyWofost ensemble succesfully initialized")
    
    #--------------------------------------------------------------------------
    def _get_ensemble_date(self):
        """Returns the date of the ensemble state.
        
        Throws and error if the date is not equal over the ensemble.
        """
        day = None
        for member in self:
            if day is None:
                day = member.timer.today()
            else:
                if day <> member.timer.today():
                    msg = "date on member %i deviates from ensemble."
                    raise RuntimeError(msg % member.ensemble_id)
        return day

    #--------------------------------------------------------------------------    
    def _initialize_wofost_objects(self, ensemble_size, metadata):
        """Initializes an ensemble of wofost objects with given size.
        """
        
        ensemble = []
        for i in range(ensemble_size):
            self.logger.debug("Initialising ensemble member %i." % i)
            # First update the parameter dictionaries if variability in
            # initial parmeters was specified.
            if (self.Esitedata is not None):
                for key in self.Esitedata[i].keys():
                    self.sitedata[key] = self.Esitedata[i][key]
                    logstr = "Overwriting value %s for %s from Esitedata."
                    self.logger.debug(logstr % (key, self.Esitedata[i][key]))
            if (self.Etimerdata is not None):
                for key in self.Etimerdata[i].keys():
                    self.timerdata[key] = self.Etimerdata[i][key]
                    logstr = "Overwriting value %s for %s from Etimerdata."
                    self.logger.debug(logstr % (key, self.Etimerdata[i][key]))
            if (self.Esoildata is not None):
                for key in self.Esoildata[i].keys():
                    self.soildata[key] = self.Esoildata[i][key]
                    logstr = "Overwriting value %s for %s from Esoildata."
                    self.logger.debug(logstr % (key, self.Esoildata[i][key]))
            if (self.Ecropdata is not None):
                for key in self.Ecropdata[i].keys():
                    self.cropdata[key] = self.Ecropdata[i][key]
                    logstr = "Overwriting value %s for %s from Ecropdata."
                    self.logger.debug(logstr % (key, self.Ecropdata[i][key]))

            # Now initialise a PyWOFOST class
            ensemble += [PyWofost(self.sitedata, self.timerdata,
                                  self.soildata, self.cropdata, self.meteof,
                                  self.mode, ensemble_id=i, metadata=metadata)]
            self.logger.debug("Succesfully initialized ensemble member %i" % i)
        return ensemble
    
    #--------------------------------------------------------------------------
    def ensemble_grow(self, days=1, termnl_limit=5):
        """Advances the state of the ensemble with specified days.
        
        Keyword variables:
        days -- nr of days to run, defaults to 1
        termnl_limit -- if nr of live ensemble members comes below this value
          then the ensemble terminates."""
        
        self.logger.info("Advancing ensemble state with %i days." % days)
        TERMNL= [] #Used for keeping track which ensemble members terminate
        for member in self.ensemble:
            TERMNL += [member.grow(days=days)]
        if TERMNL == [0]*(self.ensemble_size): # Full ensemble is alive
            return 0
        elif TERMNL == [1]*(self.ensemble_size): # Full ensemble has terminated
            return 1
        else:
            alive = self.ensemble_size - sum(TERMNL)
            self.logger.warning("Only %i ensemble members 'alive'." % alive)
            if alive < termnl_limit:
                logstr = "Terminating ensemble with %i members 'alive'" % alive
                self.logger.debug(logstr)
                return 1
            else:
                return 0
        
    #--------------------------------------------------------------------------
    def ensemble_results_to_output(self, outputfile=None, database=None,
                                   pad_results=None):
        """Sends output for all ensemble members to an output device.

        Keyword variables:
        database -- Tuple (DBmetadata, tablename) containing an SQLAlchemy
                    metadata object and the name of the table for output.
        outputfile -- filename or a file object. If the keyword is a filename
                  then different output files will be created with the ensemble
                  id appended to the filename. If file is a file object, the
                  results will be written directly to the file.
        pad_results -- can be used to copy the final results up to a certain date.
                       This is useful when the results have to be inserted in a
                       database and aggregated to larger spatial regions which
                       differ in crop calendar. Pad_results should be a date
                       object representing the date up to which the results
                       should be copied.
        """

        # Sends output to a DB table
        if database is not None:
            if isinstance(database, sqlalchemy.schema.MetaData):
                for member in self.ensemble:
                    member.results_to_output_device(database=database,
                                                    pad_results=pad_results)
            else:
                msg = "Database in ensemble_results_to_output() not "+\
                      "SQLAlchemy metadata object."
                raise RuntimeError(msg)
        
        # Send output to a file. First check if the file argument is of type
        # string which is then assumed to be a filename. This filename is first
        # modified to contain the ensemble id.
        # Otherwise the file argument is assume to be a file object and the
        # results are simply appended.
        if outputfile is not None:
            if isinstance(outputfile, str):
                for i in range(0, self.ensemble_size, 1):
                    [filename, ext] = os.path.splitext(outputfile)
                    ofile = "%s_%i%s" % (filename, i, ext)
                    member = self.ensemble[i]
                    member.results_to_output_device(outputfile=ofile)
            elif isinstance(outputfile, file):
                for member in self.ensemble:
                    member.results_to_output_device(outputfile=outputfile)
                outputfile.flush()
            else:
                msg = "Outputfile in ensemble_results_to_output() not string "+\
                      "nor file object."
                raise RuntimeError(msg)

    #--------------------------------------------------------------------------
    def next(self):
        if self.i < len(self.ensemble):
            self.i += 1
            return self.ensemble[(self.i-1)]
        else:
            self.i = 0
            raise StopIteration

    #--------------------------------------------------------------------------
    def __iter__(self):
        return self

    #--------------------------------------------------------------------------
    def __len__(self):
        return len(self.ensemble)
