"""Provides a convenient interface for starting PyWofost

If started with no arguments, the routine will connnect to the
demo database and initialize PyWofost for winter-wheat (cropno=1)
in Spain (grid_no=31031) for the year 2000 in water-limited
production (mode='wlp')

Usage:
wofsim = start_pywofost(grid_no, crop_no, year, mode, dsn)

Parameters:
dsn - PyWofost DB as SQLAlchemy data source name
       defaults to the demo database
crop - crop number, defaults to 1 (winter-wheat)
grid - grid number, defaults to 31031
year - year to start, defaults to 2000
mode - production mode ('pp' or 'wlp'), defaults to 'wlp'
"""
import sys, os
import datetime
import logging

from sqlalchemy import create_engine, MetaData, Table

from . import db_input
from . import util
from .pywofost import PyWofost

def start_pywofost(grid=31031, crop=1, year=2000, mode='wlp',
                   dsn=None):
    """Initializes a PyWOFOST object and return it to the caller.
    """

    installdir = os.path.dirname(os.path.abspath(__file__))

    if dsn is None: # Assume SQlite demo DB
        db_location = os.path.join(installdir, "database","pywofost.db")
        dsn = "sqlite:///" + db_location
    
    # Open database connections
    pywofost_engine = create_engine(dsn)
    pywofost_metadata = MetaData(pywofost_engine)
    
    # Get input data from database
    sitedata  = db_input.fetch_sitedata(pywofost_metadata, grid, year)
    timerdata = db_input.fetch_timerdata(pywofost_metadata,grid, year, crop,
                                        wb_early_start=0)
    cropdata = db_input.fetch_cropdata(pywofost_metadata, grid, year, crop)
    soildata = db_input.fetch_soiltype(pywofost_metadata, grid)

    startdate = timerdata["WB_START_DATE"]
    enddate = startdate + datetime.timedelta(days=365)
    meteof = db_input.MeteoFetcher(pywofost_metadata, grid_no=grid,
                                   startdate=startdate, enddate=enddate)
                             
    # Initialize PyWofost
    wofsim = PyWofost(sitedata, timerdata, soildata, cropdata, meteof,
                      mode=mode)
    return wofsim
