"""Miscelaneous utilities for PyWofost

Functions defined here:
* make_len30()
* check_angstAB()
* wind10to2()
* doy()
* is_data_column()
"""
from numpy import zeros, float32
import datetime
from math import log10
from sqlalchemy import Column

#-------------------------------------------------------------------------------
def make_len30(inp):
    'Converts a list of floats to a numpy array with length 30'
    l = len(inp)
    tmp = zeros(30, dtype=float32) 
    tmp[0:l] = inp
    return tmp

#-------------------------------------------------------------------------------
def check_angstAB(xA, xB):
    """Routine checks validity of Angstrom coefficients. Is python version
    of the FORTRAN routine 'WSCAB' in 'weather.for'."""
    MIN_A = 0.1
    MAX_A = 0.4
    MIN_B = 0.3
    MAX_B = 0.7
    MIN_SUM_AB = 0.6
    MAX_SUM_AB = 0.9
    A = xA
    B = xB
    if (A == 0 and B == 0):
        check = 0
    elif (A < 0 and B < 0):
        check = 1
    elif (A > 0 and B > 0):
        check = 1
    else:
        print "Invalid Angstrom A and/or B value!"
        raise
    if (check == 1):
        A = abs(A)
        if (A < MIN_A or A > MAX_A):
            print "invalid Angstrom A value!"
            raise
        B = abs(B)
        if (B < MIN_B or B > MAX_B):
            print "invalid Angstrom B value!"
            raise
        SUM_AB = A+B
        if (SUM_AB < MIN_SUM_AB or SUM_AB > MAX_SUM_AB):
            print "invalid sum of Angstrom A & B values!"
            raise

    return [A,B]

#-------------------------------------------------------------------------------
def wind10to2(wind10):
    'Converts windspeed at 10m to windspeed at 2m using log. wind profile'
    wind2 = wind10 * (log10(2./0.033) / log10(10/0.033))
    return wind2

#-------------------------------------------------------------------------------
def doy(day):
    'Converts a date object to day-of-year (Jan 1st = doy 1)'
    return (day-datetime.date(day.year,1,1)).days + 1

#-------------------------------------------------------------------------------
def is_data_column(col):
    """Returns True if given column is not part of the run descriptors
    
    Parameters:
    * col - An SQLAlchemy column object
    """
    if not isinstance(col, Column):
        msg = "Provided variable is not an SQLAlchemy column object"
        raise RuntimeError(msg)

    run_descriptors = ["grid_no","crop_no","year","day","simulation_mode", 
                       "ensemble_id"]
    if col.name not in run_descriptors:
        return True
    else:
        return False

