"""Basic Timer for PyWofost

Classes implemented here:
  TimerCl
"""
import datetime
from util import doy

class Timer:
    """This class implements a basic timer for use with the WOFOST crop model

1. Initialize the object:
timerobj = TimerCl(StartDate, EndDate [, OutputInterval=<value>])
    StartDate = Date object representing the start time of the simulation
    EndDate = Date object representing the final date when simulation
              should finish.
    CropStartDate = Date object representing day of emergence or day of sowing,
                    the timer will signal when this date is reached
    OutputInterval = Specifies the time between successive outputs,
                     defaults to 1.

2. Calling the object:
[CurrentDate, DayOfYear, Year, Output, Terminate] = timer()
    CurrentDate = date object reflecting the current simulation date
    Year = year as integer number
    DayOfYear = day number since 1st of January (1/Jan = 1)
    Output = boolean signaling whether output should be generated
    Terminate = boolean (0/1) signaling if EndDate was reached
  """
  
  
    def __init__(self, StartDate, FinalDate, CropStartDate, OutputInterval=1):
        
        self.StartDate = StartDate
        self.FinalDate = FinalDate
        self.CropStartDate = CropStartDate
        self.CurrentDate = StartDate
        self.DaysSinceStart = 0
        self.OutputInterval = OutputInterval
        self.timestep = datetime.timedelta(days=1)
        self.firstcall = True
        
    def __call__(self):
        
        if self.firstcall is True:
            Year = self.CurrentDate.year
            DayOfYear = doy(self.CurrentDate)
            # Check if output should be generated.
            Output = False
            if (self.DaysSinceStart % self.OutputInterval) == 0: Output = True
            # Check if end date is reached.
            Terminate = 0
            if (self.CurrentDate >= self.FinalDate): Terminate = 1
            self.firstcall = False
        else:
            self.CurrentDate += self.timestep
            self.DaysSinceStart += 1
            Year = self.CurrentDate.year
            DayOfYear = doy(self.CurrentDate)
            Output = False
            if (self.DaysSinceStart % self.OutputInterval) == 0: Output = True
            Terminate = 0
            if (self.CurrentDate >= self.FinalDate): Terminate = 1
        return [self.CurrentDate, self.DaysSinceStart, DayOfYear, Year,
                Output, Terminate]
    
    def check_for_sowing_or_emergence(self):
        "Returns True if crop has been sown or emerged."
        if (self.CurrentDate >= self.CropStartDate):
            return True
        else:
            return False
        
    def today(self):
        "Returns the current date."
        return self.CurrentDate

def main():
    "Only used for testing timer routine"
    
    Start = datetime.date(2000,1,1)
    End = datetime.date(2000,2,1)
    CropStart = datetime.date(2000,1,20)
    timer = TimerCl(Start, End, CropStart, OutputInterval=10)
    for i in range(150):
        [CurrentDate, day, doy, year, O, T] = timer()
        print "%s, %4i, %4i, %s %s" % (str(CurrentDate), doy, year, O, T)

if __name__ == '__main__':
    main()
