#!/usr/bin/env python
"""Implements a plotting module for PyWofost output

Classes implemented here:
* PyWofost_tsplot

Exceptions implemented here:
* TSPlotError
* TSPlotDBError(TSPlotError):
* TSPlotNoVarError(TSPlotError):
* TSPlotInvalidVarError(TSPlotError):
* TSPlotNoDataError(TSPlotError):
"""

import os
import sys
import time
import datetime
from optparse import OptionParser

import matplotlib
matplotlib.use('Agg')
import matplotlib.pyplot as plt
from matplotlib.dates import date2num, MonthLocator, WeekdayLocator, \
                             DateFormatter
from sqlalchemy.exceptions import SQLAlchemyError

from ..pyfortran.w60_variable_description import w60_var_desc
from .pywofostoutputprocessor import PyWofostOutputProcessor

#-------------------------------------------------------------------------------
class TSPlotError(Exception):
    pass

#-------------------------------------------------------------------------------
class TSPlotDBError(TSPlotError):
    pass

#-------------------------------------------------------------------------------
class TSPlotNoVarError(TSPlotError):
    pass

#-------------------------------------------------------------------------------
class TSPlotInvalidVarError(TSPlotError):
    pass

#-------------------------------------------------------------------------------
class TSPlotNoDataError(TSPlotError):
    pass

#-------------------------------------------------------------------------------
class PyWofost_tsplot(PyWofostOutputProcessor):
    """Object for plotting variables in a PyWofost database.

    Public methods:
    - save_figures()
    - plot_variable()
    - get_data_column_names()
    """
    
    def __init__(self,  dsn=None, grid_no=None, year=None, crop_no=None,
                 simulation_mode=None, subplots=None, figsize=None, dpi=300,
                 basename=None):
        """Constructor for pywofost_tsplot.
        
        Keywords:
        * dsn - SQLAlchemy data source name
        * grid_no - grid id of spatial entity
        * crop_no - crop id of crop type
        * year - year as specified in crop_calendar
        * simulation_mode - wlp|pp for selecting water-limited|potential mode
        * subplots - specify (rows, columns) to plot on page
        * dpi - dpi for raster output
        * figsize - (xsize, ysize) to specify size of figure
        """
        
        PyWofostOutputProcessor.__init__(self, dsn=dsn, grid_no=grid_no,
            year=year, crop_no=crop_no, simulation_mode=simulation_mode)

        # Set dpi
        self.dpi = dpi
        # Define subplots
        self._define_subplots(subplots)

        # Calculate size of figure given nr of subplots
        if figsize is None:
            figsize = self._calc_figsize()
        self.figsize = figsize

        # Create a matplotlib figure instance
        self.figure = plt.figure(dpi=self.dpi, figsize=self.figsize)

        # Create placeholder in case of multiple pages
        self.figures = {}
        self.figure_id = 1
        
        # Create basename for figures
        if basename is None:
            basename = "PyWofost_plot_"
        self.basename = basename + r"%i"
    
   #----------------------------------------------------------------------------
    def _define_subplots(self, subplots):
        """Defines the subplots that are used for plotting"""
        
        if subplots is not None:
            self.prows = subplots[0]
            self.pcols = subplots[1]
            self.pmax = subplots[0] * subplots[1]
        else:
            self.prows = 1
            self.pcols = 1
            self.pmax = 1
        self.plot_id = 1
    
   #----------------------------------------------------------------------------
    def _append_active_figure(self):
        """Appends active figure to self.figures
        """
        key = self.basename % self.figure_id
        self.figures[key] = self.figure

   #----------------------------------------------------------------------------
    def _calc_figsize(self):
        "Calculates the figure size based on nr of subplots rows/colums"
        default_xsize = 6
        default_ysize = 4.5
        return (self.pcols * default_xsize, self.pcols * default_xsize)
    
   #----------------------------------------------------------------------------
    def save_figures(self, type='pdf'):
        """Save figures to a graphics file.
        
        Keywords:
        * type - graphics output format (depends on matplotlib support).
        """
        #Add active figure to self.figures"""
        if self.figure is not None:
            self._append_active_figure()
            self.figure = None
            
        for figkey, fig in self.figures.items():
            fname = "%s%s%s" % (figkey, os.path.extsep, type)
            fig.savefig(fname)
    
   #----------------------------------------------------------------------------
    def plot_variable(self, varname, linewidth=(0.25,1.5,1), marker=('','',''),
                      color=('k','r','r'), linestyle=(':','-','--'),
                      ticklocator=None, tickformatter=None):
        """Plot data available for the specified variable.
        
        If the data consists of an ensemble then the mean, and mean +/- 1
        stdev are also plotted.
        
        Graphics keywords specified as a tuple (member, mean, stdev):
        * linewidth - width of lines 
        * marker - marker of line
        * color - color of line
        
        Other keywords:
        * ticklocator - locates the plot ticks
        * tickformatter - formatting of ticks
        """
        
        # Run some checks on variables, formatters, etc.
        fullvar = w60_var_desc.get_full_name(varname)

        if fullvar not in self.pywofost_output:
            msg = "Variable '%s' (%s)  does not exist in table 'pywofost_output'."
            raise TSPlotNoVarError(msg % (varname, fullvar))
        
        # if the number of plots is larger then the number of plots that fit
        # on a page, the store the figure and generate an empty one.
        if self.plot_id > self.pmax:
            self._append_active_figure()
            self.figure = plt.figure(dpi=self.dpi, figsize=self.figsize)
            self.figure_id += 1
            self.plot_id = 1
        
        if ticklocator is None:
            ticklocator = MonthLocator(range(1,13), bymonthday=1, interval=2)

        if tickformatter is None:
            tickformatter = monthsFmt = DateFormatter("%b '%y")
        
        # Start creating the plot
        axes = self.figure.add_subplot(self.prows, self.pcols, self.plot_id)
        axes.xaxis.set_major_locator(ticklocator)
        axes.xaxis.set_major_formatter(tickformatter)
        try:
            title = w60_var_desc.get_title_name(fullvar)
            ylabel = w60_var_desc.get_unit(fullvar)
        except KeyError:
            print "Warning: title/Ylabel not found for variable '%s'" % varname
            print "Labels for non-default variables can be set using the "
            print "newvar keyword. See help for pywofost_tsplot."
            ylabel = fullvar
        axes.set_ylabel(ylabel)
        axes.set_title(title)
        
        # Plot (ensemble) variable trajectories
        lines = []
        x = self.pywofost_output[fullvar]
        if self.ensemble_size == 1:
            lines += axes.plot_date(x.days, x.values, 'k-')
            # Set line type, width, color for single run
            plt.setp(lines, linewidth=1.5, color='k',
                     linestyle='-', marker=marker[0])
        else:
            for i in range(x.values.shape[1]):
                lines += axes.plot_date(x.days, x.values[:, i])

            # Set line type, width, color for ensembles
            plt.setp(lines, linewidth=linewidth[0], color=color[0],
                     linestyle=linestyle[0], marker='')

            # Plot ensemble mean and stdev
            axes.plot_date(x.days, x.mean, '-', color=color[1],
                                   linewidth=linewidth[1],
                                   linestyle=linestyle[1], marker=marker[1])
            lines = []
            lines += axes.plot_date(x.days, (x.mean+x.stdev))
            lines += axes.plot_date(x.days, (x.mean-x.stdev))
            plt.setp(lines, linewidth=linewidth[2], color=color[2],
                     linestyle=linestyle[2], marker=marker[2])

        # Turn grid on and autoformat the dates on the x axis
        axes.grid(True)
        self.figure.autofmt_xdate()
        
        self.plot_id += 1

#----------------------------------------------------------------------------
def create_parser():
    """Creates and populates the instance parser.
    """
    usage = "usage: %prog [options] grid_no crop_no year mode"
    description = """PyWofost_TSPlot is tool to plot PyWofost output which is
stored in a PyWofost database. It needs four arguments (in this order):
grid_no, crop_no, year and mode which represent the grid id of
the spatial entity, the crop id, the year (as defined in the crop_calendar table) and
the simulation mode (e.g. pp or wlp for potential or water-limited production).
By default PyWofost_TSPlot will plot all variables that are present in the
pywofost_output table, but specific variables can be plotted with the
-v|--var option. Plotting multiple variables can be specified as -v lai
-v twso -v tagp etc. Number of figures on a page can be specified as well as
the output graphics type, see options below.
    """
    parser = OptionParser(usage=usage, description=description)
    
    parser.add_option("-f", "--file", dest="filename",
                      help="write graphics to FILE", metavar="FILE")
    parser.add_option("--dpi", dest="dpi", type="int",
                      help="DPI for raster formats", default=600)
    parser.add_option("-r","--rows", dest="rows", type="int",
                      help="Number of rows in output page", default=1)
    parser.add_option("-c","--cols", dest="cols", type="int",
                      help="Number of cols in output page", default=1)
    parser.add_option("--type", dest="ext", type="string",
                      help="Output graphics format: png|pdf|eps|svg|emf|raw",
                      default="pdf")
    parser.add_option("--dsn", dest="dsn", type="string",
                      help="SQLAlchemy data source name (DSN)",
                      default="sqlite:///../wofost/pywofost.db")
    parser.add_option("-v","--var", dest="vars", type="string",
                      help="Create plot of variable VAR", action="append",
                      metavar="VAR")
    return parser


#-------------------------------------------------------------------------------
def main():
    
    parser = create_parser()
    (options, args) = parser.parse_args()

    if len(args) != 4:
        parser.print_help()
        return

    try:
        grid_no = int(args[0])
        crop_no = int(args[1])
        year = int(args[2])
        mode = args[3].lower()
    except Exception, e:
        print "Error: Invalid argument(s) provided."
        parser.print_help()
        return
    
    # Check mode argument
    if mode not in ('wlp', 'pp'):
        print "Error: Valid values for mode are: wlp|pp"
        return
    
    # check variable names specified with -v/--var
    if options.vars is not None:
        options.vars = [s.lower() for s in options.vars]
        for varname in options.vars:
            if not w60_var_desc.check_variable(varname):
                print "Error: Variable '%s' is not a valid PyWofost variable." \
                      % varname
                return
    
    # Check ext argument
    valid_ext = ('png','pdf','eps','svg','emf','raw')
    if options.ext not in valid_ext:
        print "Error: Valid grapics formats are: ", valid_ext
        return
    
    # Start plotting data
    try:
        # Instantiate plotting objects
        tsplot = PyWofost_tsplot(dsn=options.dsn, grid_no=grid_no, year=year,
                    crop_no=crop_no, simulation_mode=mode, dpi=options.dpi,
                    basename=options.filename, subplots=(options.rows,options.cols))
        if options.vars is None:
            vars_to_plot = tsplot.get_data_column_names()
        else:
            vars_to_plot = options.vars
    
        for varname in vars_to_plot:
            print "Plotting %s" % varname
            tsplot.plot_variable(varname)
        tsplot.save_figures(type=options.ext)
    except TSPlotError, e:
        print e
        return
    except SQLAlchemyError, e:
        print "Failure to find table or connect to DB."
        print "DSN used to connect: %s" % options.dsn
        return
    except Exception, e:
        print "General error: %s" % e
        return

if __name__ == '__main__':
    main()
