"""Provides a convenient interface for running PyWofost

Starting run_pywofost.main() will simpy run PyWofost for 300 days for the
given grid_no, crop_no, year and mode.

Positional parameters:
dsn - PyWofost DB as SQLAlchemy data source name
crop - crop number
grid - grid number
year - year to start
mode - production mode ('pp' or 'wlp')

Keyword parameters:
start_stop - Runs PyWofost in 6 batches of 50 days (default False)
clear_table - Deletes everything from the table pywofost_output (default False)
"""

import sys, os
import datetime
import logging

from sqlalchemy import create_engine, MetaData, Table

from . import db_input
from . import util
from .pywofost import PyWofost

def main(dsn, crop, grid, year, mode, start_stop=False, clear_table=False):

    # Open database connection and empty output table
    pywofost_engine = create_engine(dsn)
    pywofost_metadata = MetaData(pywofost_engine)
    table_pywofost_output = Table('pywofost_output', pywofost_metadata,
                                  autoload=True)
    if clear_table is True:
        table_pywofost_output.delete().execute()
    
    # Get input data from database
    sitedata  = db_input.fetch_sitedata(pywofost_metadata, grid, year)
    timerdata = db_input.fetch_timerdata(pywofost_metadata,grid, year, crop,
                                        wb_early_start=0)
    cropdata = db_input.fetch_cropdata(pywofost_metadata, grid, year, crop)
    soildata = db_input.fetch_soiltype(pywofost_metadata, grid)

    startdate = timerdata["WB_START_DATE"]
    enddate = startdate + datetime.timedelta(days=365)
    meteof = db_input.MeteoFetcher(pywofost_metadata, grid_no=grid,
                                   startdate=startdate, enddate=enddate)
                             
    # Run simulation
    wofsim = PyWofost(sitedata, timerdata, soildata, cropdata, meteof,
                      mode=mode, metadata=pywofost_metadata)
    
    # If start_stop is true then run the simulation in steps of 50 days. This is
    # only for testing whether the switching back and forth from runtime to
    # idle state goes well.
    if start_stop is True:
        wofsim.grow(days=50)
        wofsim.grow(days=50)
        wofsim.grow(days=50)
        wofsim.grow(days=50)
        wofsim.grow(days=50)
        wofsim.grow(days=50)        
    else:
        wofsim.grow(days=300)
    wofsim.results_to_output_device(database=pywofost_metadata)
    output_file = "%s_%s_%s_%s.out" % (grid, crop, year, mode)
    wofsim.results_to_output_device(outputfile=output_file)

