package edu.iris.Fissures.seed.container;

import edu.iris.Fissures.seed.exception.*;
import com.opensymphony.oscache.base.*;
import com.opensymphony.oscache.general.*;
import com.opensymphony.oscache.base.persistence.*;
import java.util.*;

/**
 * Builder Container for Seed Blockettes.
 * <p>
 * Container that implements the interface of ObjectContainer and provides
 * a facility for storing, retrieving, and cross-referencing blockettes.
 * This container provides the implementer with the ability to ingest
 * multiple SEED volumes and integrate the information in single location
 * for merging and integration purposes.  From this container, new SEED 
 * volumes (and even other formats) can be generated by creational classes, 
 * using it as a temporary database to retrieve all needed information.
 * Note that this container is not meant for permanent storage, but the use
 * of the disk persistence option allows you to re-load a previous container
 * state from disk if so flagged.
 * <p>
 * Makes use of disk cache persistence in order to accomodate
 * a large volume of Blockettes while minimizing memory consumption.
 * Blockette's retrieved are placed in a Least Recently Used
 * queue of some preconfigured size to accomodate repeated access of certain blockettes.
 * All contents are persisted to disk as they are created and updated, and removed
 * items are promtly eliminted.  If so flagged, a new Container can access a previous
 * session's persisted objects for immediate use.
 * <p>
 * Blockettes are stored in a parent/child heirarchy with regards to stations and channels,
 * since their references to each other are positional in the SEED volume.  Child blockettes
 * are placed within a parent blockette to maintain this association.  Stations are parents
 * to Channels, and Channels are parents to response information.  If this container has
 * been flagged to indicate that a filter is being used by the Builder it is connected to,
 * then the case of running into child blockettes with no parent, which had been filtered out,
 * is treated as a normal condition, where the child blockettes are simply 'orphaned' and
 * not stored in the container.  If a filter is not being used, then this is considered an
 * error condition, and an exception is thrown.
 * 
 * @author Robert Casey, IRIS DMC
 * @version 4/19/2006
 */

public class SeedObjectContainer implements ObjectContainer {
    
    /**
     * Default constructor that does not implement serialization
     */
    public SeedObjectContainer () {
        // initialize various data structures
        try {
            initialize(false);
        } catch (ContainerException e) {
            System.err.println("ERROR: an exception was thrown while initializing the SeedObjectContainer: " + e);
        }
    }
    
    /**
     * Constructor specifying a directory for disk cache persistence
     * and bootstrap recall of previous session from disk if so flagged.
     * @param cacheDir the base directory where the disk cache will reside
     * @param capacity the maximum number of items to be cached in memory at one time
     * @param loadPrevious true if previous disk persistence cache should be referenced.
     */
    // REC
    public SeedObjectContainer (String cacheDir, int capacity, boolean loadPrevious) throws Exception {
        // flag that we are caching data
        isCaching = true;
        // remember the cache path
        this.cacheDir = cacheDir;
        // remember the capacity setting
        cacheSize = capacity;
        // initialize various data structures and possibly pre-load from disk cache
        initialize(isCaching, loadPrevious);
    }
    
    /**
     * Constructor specifying a directory for disk cache persistence.
     * All previous disk persistence entries are deleted upon startup.
     * @param cacheDir the base directory where the disk cache will reside
     * @param capacity the maximum number of items to be cached in memory at one time
     */
    public SeedObjectContainer (String cacheDir, int capacity) throws Exception {
        this (cacheDir, capacity, false);
    }
    
    /**
     * Constructor specifying a directory for disk cache persistence
     * and bootstrap recall of previous session from disk if so flagged.
     * Uses the default cache capacity value.
     * @param cacheDir the base directory where the disk cache will reside
     * @param loadPrevious true if previous disk persistence cache should be referenced.
     */
    // REC
    public SeedObjectContainer (String cacheDir, boolean loadPrevious) throws Exception {
        this(cacheDir, defaultCacheSize, loadPrevious);
    }
    
    /**
     * Contructor specifying a directory for disk cache persistence.
     * Uses default capacity specified in this class.
     * @param cacheDir the base directory where the disk cache will reside
     */
    public SeedObjectContainer (String cacheDir) throws Exception {
        this (cacheDir, defaultCacheSize);  // use default size
    }
    
    /**
     * Initialize data structures based on whether disk caching is being used or not.
     * Default behavior is to delete any previous cache entries at established cacheDir
     * to prevent aliasing effects of pre-existing disk persistence entries.
     * @param isCaching true if disk persistence caching is to be implemented
     */
    private void initialize(boolean isCaching) throws ContainerException {
        initialize(isCaching,false);  // false flag for loadPrevious
    }
    
    /**
     * Initialize data structures based on whether disk caching is being used or not.
     * Default behavior is to delete any previous cache entries at established cacheDir
     * to prevent aliasing effects of pre-existing disk persistence entries.
     * @param isCaching true if disk persistence caching is to be implemented
     * @param loadPrevious true if previous disk persistence cache should be referenced.
     */
    private void initialize(boolean isCaching, boolean loadPrevious)  throws ContainerException {
        //System.err.println("DEBUG: SeedObjectContainer.initialize() START");
        try {
            // set up hash maps
            activeHashMaps = new Vector(8,8);
            volume = new CachedHashMap("volume",cacheSize,isCaching,loadPrevious);
            activeHashMaps.add(volume);
            dictionary = new CachedHashMap("dictionary",cacheSize,isCaching,loadPrevious);
            activeHashMaps.add(dictionary);
            station = new CachedHashMap("station",cacheSize,isCaching,loadPrevious);
            activeHashMaps.add(station);
            timespan = new CachedHashMap("timespan",cacheSize,isCaching,loadPrevious);
            activeHashMaps.add(timespan);
            data = new CachedHashMap("data",cacheSize,isCaching,loadPrevious);
            activeHashMaps.add(data);
            encodingMap = new CachedHashMap("encoding",cacheSize,isCaching,loadPrevious);
            activeHashMaps.add(encodingMap);
            parentChildMap = new CachedHashMap("parentChild",cacheSize,isCaching,loadPrevious);
            activeHashMaps.add(parentChildMap);
            // set up rank lattice
            rankLattice = new BlocketteVector(8,8);
        } catch (CachePersistenceException e) {
            throw new ContainerException("CachePersistenceException thrown during map initialization: " + e);
        }
        //System.err.println("DEBUG: SeedObjectContainer.initialize() END");
    }
    
    // interface methods
    
    /**
     * Identify ourselves through this method.
     * 
     */
    public String toString() {
        if (isCaching) return new String("SeedObjectContainer with disk persistence cache set to: " + cacheDir);
        return new String("SeedObjectContainer using in-memory objects without disk caching");
    }
    
    /**
     * return the path to the base of the cache tree
     * @return base pathname of the cache
     */
    public String getCachePath() {
        return cacheDir;
    }
    
    /**
     * Clear all contents in this container and start fresh.
     * Use with caution, since this erases all previous entries, including the disk cache
     * at cacheDir.
     */
    public void clear() throws ContainerException {
        initialize(isCaching);
    }
    
    /**
     * Add a Blockette object to the container.
     */
    public void add(Object addThis) throws Exception {
        // the Object here is by necessity a Blockette
        // let's ensure this (exception will be thrown otherwise)
        try {
            // place the blockette in the addRegister for processing -- make sure this is a pure Blockette, not a decorated one
            if (addThis instanceof CachedBlocketteDecorator)
                addRegister = ((CachedBlocketteDecorator)addThis).getBlockette();  // extract Blockette from Decorator
            else
                addRegister = (Blockette) addThis;  // this is a plain ol Blockette
        } catch (Exception e) {
            String errStr = new String ("Failure to cast added object as a Blockette");
            throw new ContainerException(errStr);
        }
        // check to see that this handle is not attached to a null value
        if (addRegister == null) throw new ContainerException("add method was passed a null value");
        int addRank = SeedBlocketteRankMap.getRank(addRegister);
        //
        // check to see whether the parent was filtered by the builder...
        if (filterRank > 0) {
            if (addRank >= filterRank) {
                // any children of the filtered out blockette are rejected
                addRegister = null;  // throw out this child blockette
                return;              // just exit from this function quietly
            } else {
                // else we are placing a ranking blockette in the rank tree, so reset the filter flag
                filterRank = 0;
            }
        }
        // get blockette type
        int blkType = addRegister.getType();
        // get the lookup id of this blockette
        int lookupId = addRegister.getLookupId();
        // extract the volume number
        int currentVolumeNumber = getVolumeNumberFromId(lookupId);
        // get object version of the lookup id
        Integer lookupIdObj = new Integer(lookupId);
        // get category value
        int addCategory = SeedBlocketteRankMap.getHeaderCode(addRegister);
        //
        // write to the rank lattice
        rankLattice.setSize(addRank+1);
        // REC -- change this to adding decorated Blockette to lattice
        //rankLattice.set(addRank,addRegister);
        CachedBlocketteDecorator decBlk = BlocketteDecoratorFactory.createCachedDecorator(addRegister,isCaching());
        rankLattice.set(addRank,decBlk);
        //
        // Branch our behavior below based on whether we are a top-level parent blockette or a lower level child blockette
        if (addRank > 0) {  // we are a CHILD blockette
            // get our immediate parent - we assume it is already in the rank lattice (setParent())
            Blockette parent = rankLattice.getBlockette(addRank-1);  // this *should* be of type CachedBlocketteDecorator
            if (parent == null) {
                // otherwise, it's abnormal to see this situation, so throw an exception
                throw new ContainerException("added child blockette (type " + 
                        blkType + ") does not have a parent blockette");
            }
            // register parent with child
            addRegister.attachParent(parent);
            // add Decorator of child to parent
            parent.addChildBlockette(decBlk);
            // DEBUG
            //int numchild = parent.numberofChildBlockettes();
            //System.err.println("DEBUG: parent " + parent.getType() + " number of children: " + numchild);
            //
            // get parent type
            int parentType = parent.getType();
            // get parent lookupId
            int parentId = parent.getLookupId();
            // map child to parent map:(child,parent)
            // this will help us to quickly find child blockettes with respect to their parents
            parentChildMap.put(lookupIdObj, new Integer(parentId));
            //
            // DATA ENCODING METADATA
            // while we are here, see if this child blockette is type 52
            // if so, then take type 50 and type 52 id values to form a key object, then lookup
            // the blockette 30 dictionary reference for the associated data type.  Store the data
            // type in a HashKey for later reference.
            // if the data encoding format cannot be found, then do not store a HashKey reference.
            if (blkType == 52) {
                EncodeKey currentEncodeKey = new EncodeKey();
                currentEncodeKey.location = addRegister.toString(3);   // blk 52
                currentEncodeKey.channel = addRegister.toString(4);    // blk 52
                Btime encodeStartTime = new Btime(addRegister.toString(22));      // blk 52
                // our parent must be blockette 50
                if (parentType != 50)
                    throw new ContainerException(
                            "Blockette 52 detected without Blockette 50 as parent. It was type: " +
                            parentType);
                currentEncodeKey.station = parent.toString(3);         // blk 50
                currentEncodeKey.network = parent.toString(16);        // blk 50
                //
                // get the dictionary reference to blockette 30
                Blockette blk30 = getDictionaryBlockette(addRegister,16);
                //System.err.println("DEBUG: resolve encoding for blk 30: " + blk30);
                if (blk30 != null) {
                    String encodingStr = SeedEncodingResolver.resolve(blk30);  // get the encoding keyword
                    ///String encodingStr = "Steim1";  // temporary
                    // encoding map takes the format:
                    // key: EncodeKey                                         [station.network.channel.location]
                    // value: Vector {time0,string0,time1,string1,...,timeN,stringN}  [effective time, encoding]
                    //System.err.println("DEBUG: encoding: " + encodingStr);
                    Vector encodingVec = null;
                    if (encodingMap.containsKey(currentEncodeKey.toString())) {  // check to see if this key has already been started
                        //System.err.println("DEBUG: found existing encode value for key " + currentEncodeKey);
                        encodingVec = (Vector) encodingMap.get(currentEncodeKey.toString());  // get existing encoding vector for append
                    } else {
                        encodingVec = new Vector(2,2);  // get new encoding vector for append
                    }
                    encodingVec.add(encodeStartTime);  // add time value (even [n] position)
                    encodingVec.add(encodingStr);  // add string value (odd [n+1] position)
                    //System.err.println("DEBUG: adding " + currentEncodeKey + ", " + encodeStartTime + ", " + encodingStr);
                    encodingMap.put(currentEncodeKey.toString(),encodingVec);  // put the encoding vector into the hash map
                }
            }
            // DATA ENCODING METADATA
            // if the child blockette is type 1000, then we apply a local encoding assignment to the parent
            // blockette 999's waveform to be the string equivalent of field 3, blockette 1000.
            if (blkType == 1000) {
                if (parentType != 999)
                    throw new ContainerException("Blockette 1000 detected without Blockette 999 as parent");
                Waveform parentWaveform = parent.getWaveform();
                if (parentWaveform != null) parentWaveform.setEncoding(SeedEncodingResolver.translate(addRegister.toString(3))); // assign encoding
            }
        } else {     // we are a PARENT blockette
            // map child to parent map:(child,parent) -- for topmost parent, we
            // map to a value of zero.
            parentChildMap.put(lookupIdObj,new Integer(0));
            // DATA ENCODING METADATA
            // if this is an FSDH blockette, then make initial waveform encoding assignment based on
            // header lookup map, but only if it's UNKNOWN.
            if (blkType == 999) {  // data record FSDH
                Waveform tempWaveform = addRegister.getWaveform();
                if (tempWaveform != null && tempWaveform.getEncoding().equals("UNKNOWN"))
                    tempWaveform.setEncoding(getMetaEncoding(addRegister));  // set encoding
            }
        }  // END parent blockette block
        //System.err.println("DEBUG: parentChildMap " + lookupIdObj + " to " + parentChildMap.get(lookupIdObj).toString());
        //
        // for ALL blockettes...
        // write the blockette to the hash map
        Object addObj = addRegister;       // addObj is a pure Blockette
        //System.err.println("DEBUG: container.add(): id=" + lookupIdObj + ": " + addObj);
        // select which map based on the category number
        switch (addCategory) {
        case 1:
            volume.put(lookupIdObj,addObj);
            break;
        case 2:
            dictionary.put(lookupIdObj,addObj);
            break;
        case 3:
            station.put(lookupIdObj,addObj);
            break;
        case 4:
            timespan.put(lookupIdObj,addObj);
            break;
        case 5:
            data.put(lookupIdObj,addObj);	
            break;
        default:
            throw new ContainerException("attempted to add unknown category number: " + addCategory);
        }
        //
        // make sure that the Decorator Factory is aware of the Container Mapping
        BlocketteDecoratorFactory.setContainerByVol(this,currentVolumeNumber);
        /////////////////////////////////////////////////////////
        // remember the highest lookup ID value for this category
        int idMapIndex = addCategory;  // idMapIndex is a copy that may get modified later
        // special lookup case for dictionary blockettes.
        // index = (<blockette type> - 20)
        if (addCategory == 2) {
            idMapIndex = blkType - 20;
        }
        if (lastIdMap[idMapIndex] < addRegister.getLookupId())
            lastIdMap[idMapIndex] = addRegister.getLookupId(); // only update if higher
        /////////////////////////////////////////////////////////
        // empty the add register (blockette is still stored in the Rank Lattice)
        addRegister = null;
        // empty the locate register, since an alteration has been made to the container
        locateRegister = null;
    }
    
    
    /**
     * Get the referenced dictionary blockette.
     * Utility method to allow the specified blockette to pull back the dictionary blockette
     * referenced by the indicated field number.  Returns null if dictionary not found.
     */
    public Blockette getDictionaryBlockette (Blockette blk, int fieldNum) throws Exception {
        return getDictionaryBlockette(blk,fieldNum,0);
    }
    
    
    /**
     * Get the referenced dictionary blockette.
     * Utility method to allow the specified blockette to pull back the dictionary blockette
     * referenced by the indicated field number and repeat group field index.
     * Does not work for List fields, such as Blockette 60, which requires multiple
     * returns.  Use the listIndex variant of this method for that.
     * Returns null if dictionary not found.
     */
    public Blockette getDictionaryBlockette (Blockette blk, int fieldNum,
            int fieldIndex) throws Exception {
        // get field value
        String refStr = blk.toString(fieldNum,fieldIndex);
        // set up a default response of zero if the string value is blank
        if (refStr.length() == 0) refStr = "0";
        // generate int value from string
        int refVal = Integer.parseInt(refStr);
        // get lookup id based on field value
        int lookupId = blk.getDictionaryLookup(refVal);
        //System.err.println("DEBUG: getDictionaryBlockette " + lookupId + " from refVal " + refVal);
        // ask the object container for the blockette matching this id
        if (lookupId > 0) {
            return (Blockette) this.get(lookupId);
        } else {
            return null;
        }
    }
    
    /**
     * Get the referenced dictionary blockette.  This variant allows access to
     * dictionary references in List fields, such as in blockette 60.  The listIndex
     * parameter is the index number, starting at 0, of the desired reference in that
     * field's list.
     */
    public Blockette getDictionaryBlockette (Blockette blk, int fieldNum,
            int fieldIndex, int listIndex) throws Exception {
        // get string representation of referencing field
        String refStr = blk.toString(fieldNum,fieldIndex);
        // get value at list index of refStr
        refStr = BlocketteFactory.getListValue(refStr,listIndex);
        // trim string for leading/trailing spaces
        refStr = refStr.trim();
        // set up a default response of zero if the string value is blank
        if (refStr.length() == 0) refStr = "0";
        // generate int value from string
        int refVal = Integer.parseInt(refStr);
        // get lookup id based on field value
        int lookupId = blk.getDictionaryLookup(refVal);
        // ask the object container for the blockette matching this id
        if (lookupId > 0) {
            return (Blockette) this.get(lookupId);
        } else {
            return null;
        }
    }
    
    /**
     * Return blockette with the listed lookup ID.
     * Return object is actually a CachedBlocketteDecorator from locate().
     */
    public Object get(int lookupId) throws ContainerException {
        if (locate(lookupId)) {
            return locateRegister;
        } else {
            return null;
        }
    }
    
    /**
     * Delete the blockette object matching to lookupId from the container
     * Return that object upon removal.  Remove all mapping entries.  Remove its children as well as
     * itself from its parent.
     */
    public Object remove(int lookupId) throws ContainerException {
        // get the Blockette first, before initiating the removal
        //Blockette blockette = (Blockette) get(lookupId);
        Blockette blockette = (Blockette) lookup(lookupId);  // get Blockettte not CachedBlocketteDecorator
        // proceeed to remove this object's children first
        for (int i = 0; blockette != null && i < blockette.numberofChildBlockettes(); i++) {
            Blockette childBlk = blockette.getChildBlockette(i);
            if (childBlk != null) {
                remove(childBlk.getLookupId());  // recursive call
            }
        }
        // get object form of the lookupId
        Integer lookupIdObj = new Integer(lookupId);  // Object Integer of lookupId
        // get parent Id
        Integer parentIdObj = (Integer) parentChildMap.get(lookupIdObj);
        // remove child entry from the parent
        if (parentIdObj.intValue() > 0) {   // if parent Id is zero, then we are root of parent tree
            Blockette parentBlk = (Blockette) get(parentIdObj.intValue());
            int childCount = parentBlk.numberofChildBlockettes();
            for (int i = 0; i < childCount; i++) {
                SeedObject childObj = parentBlk.getChildBlockette(i);
                if (childObj != null && childObj.getLookupId() == lookupId) {
                    parentBlk.removeChildBlockette(i);
                }
            }
        }
        // remove the entry from any/all category maps
        volume.remove(lookupIdObj);  // volume map
        dictionary.remove(lookupIdObj); // dictionary map
        station.remove(lookupIdObj);  // station map
        timespan.remove(lookupIdObj); // timespan map
        data.remove(lookupIdObj);      // waveform map
        // continue removal from associative maps
        parentChildMap.remove(lookupIdObj);   // parent-child map
        // empty the locate register, since an alteration has been made to the
        // container
        locateRegister = null;
        //
        return blockette;
    }
    
    
    /**
     * Find the blockette with the listed lookup ID and set a handle to it
     * in locateRegister.  Return true if successful, false if not.
     * Returned Blockette is actually a CachedBlocketteDecorator, serving as the SeedObjectContainer's
     * 'external face' for Blockettes.
     */
    public boolean locate(int lookupId) throws ContainerException {
        locateRegister = null;  // reset the register
        // we have found the blockette in the HashMap, so generate a Decorator class for it and
        // place it in the locateReigster
        Blockette locateBlk = lookup(lookupId);
        try {
            if (locateBlk != null) locateRegister = BlocketteDecoratorFactory.createCachedDecorator(locateBlk,isCaching());
        } catch (SeedException e) {
            throw new ContainerException("SeedException thrown by CachedBlocketteDecorator constructor: " + e +
                    " while performing a locate on lookupId=" + lookupId);
        }
        return (locateRegister != null); // return whether locateRegister contains a blockette or not
    }
    
    /**
     * Set up Blockette interation.
     * Allows user to iterate through a set of *Parent* Blockettes for a 
     * particular category and volume.
     * This method returns the number of Blockettes to be iterated over.
     * getNext() will resolve lookups for the Blockettes encountered.
     * Category numbers for SEED are 1 through 5.  See SeedObjectBuilder for details.
     */
    public int iterate(int volNum, int catNum) {
        CachedHashMap selectedMap = null;  // this will be the object map relating to the provided catNum
        Vector pickVector = new Vector(8,8);  // establish blank iterator vector
        int category,target;
        if (catNum == -1) {
            category = 1;
            target = 5;
        } else {
            category = catNum;
            target = catNum;
        }
        for (int curCat = category; curCat <= target; curCat++) {
            switch (curCat) {
            case 1:
                selectedMap = volume;
                break;
            case 2:
                selectedMap = dictionary;
                break;
            case 3:
                selectedMap = station;
                break;
            case 4:
                selectedMap = timespan;
                break;
            case 5:
                selectedMap = data;
                break;
            default:
                currentIterator = null;
            return 0;
            }
            //Set entries = selectedMap.entrySet();
            // REC -- DEBUG here
            Collection entries = selectedMap.entrySet();
            //System.err.println("DEBUG: number of entries for category " + category + ": " + entries.size());
            Iterator tempIterator = entries.iterator();
            // iterate through the selected HashMap and take only blockettes that are
            // from the specified volume and belong to the specified header category
            Blockette blockette = null;
            while (tempIterator.hasNext()) {
                // code based on Core Java, Vol 2, pg. 106
                Object iterNext = tempIterator.next();
                //System.err.println("DEBUG: iterator next = " + iterNext);
                Map.Entry entry = (Map.Entry) iterNext;  // get next id/blockette pair
                Integer lookupIdObj = new Integer(entry.getKey().toString());
                //System.err.println("DEBUG: iterate.next, category=" + category + ": " + lookupIdObj);
                boolean meetsCriteria = true;  // assume we have the right category
                //
                if (volNum > -1) {
                    // compare volNum plus header category to lookup ID
                    // can alter meetsCriteria to false
                    meetsCriteria = ((volNum * 10) + category == lookupIdObj.intValue() / (1000 * 1000));
                }
                if (meetsCriteria) {  // if the lookup ID meets the criteria for correct volume and category number
                    try {
                        // check to see if we need to set the current volume number and map
                        // our container to it -- needed for cache pre-loading which bypasses add()
                        int lookupId = lookupIdObj.intValue();
                        if (currentVolumeNumber < 1) {
                            currentVolumeNumber = getVolumeNumberFromId(lookupId);   // get the volume from our currently sampled lookupId
                            BlocketteDecoratorFactory.setContainerByVol(this,currentVolumeNumber);  // make sure the BDFactory knows of this association
                            cachePreload = true;  // mark that we are iterating from a previous cache
                        }
                        blockette = (Blockette) get(lookupId); // make sure we get Blockette Decorator
                        // cache preload: we have to update our lookupId increment maps when bypassing add().
                        // grab the highest id number, even from blockettes that are not going into the
                        // iteration lists (i.e. child blockettes)
                        if (cachePreload) {
                            int idMapIndex = curCat;  // map to category number
                            int blkType = blockette.getType();
                            if (curCat == 2) {   // special lookup case for dictionary blockettes
                                idMapIndex = blkType - 20;   // index = (<blockette type> - 20)
                            }
                            if (lastIdMap[idMapIndex] < lookupId)
                                lastIdMap[idMapIndex] = lookupId; // only update if higher
                        }
                        // make sure that only top-level parent Blockettes are added to the iteration list
                        if (blockette.hasParent()) continue;
                    } catch (Exception e) {
                        System.err.println("Non-fatal exception thrown while building iteration list: " + e);
                        continue;   // non fatal exception...if we run into problems, don't add it to iteration list
                    }
                    //
                    pickVector.add(blockette);    // add the Blockette object to the iterator vector
                }
            }
        }  // end loop each category
        // let's sort the contents of the Vector before moving on
        Collections.sort(pickVector);
        currentIterator = pickVector.iterator();  // get an Iterator interface from assembled vector
        //
        return pickVector.size();   // return the number of elements
    }
    
    /**
     * Set up an iterator for all parent Blockettes of a specific category number, all volumes.
     */
    public int iterate(int catNum) {
        return iterate(-1,catNum);
    }
    
    /**
     * Set up an iterator for all parent Blockettes.
     */
    public int iterate() {
        return iterate(-1,-1);
    }
    
    /**
     * Set up an iterator for *Parent* Blockettes matching stations and channels.
     * Also confines to a specific category number.
     * Station and channel parameters are Vectors of string values. Each 
     * parameter can also be null.  The null vector will simply be treated as 'all'.
     * Returns the number of elements in the iterator.
     */
    public int iterate(Vector stations, Vector channels, int catNum) throws ContainerException {
        if (catNum < 1 || catNum > 5) {
            throw new ContainerException("iterate() passed an illegal category number: " + catNum);
        }
        iterate(catNum);  // set up iterator for the indicated catalog -- passes back top-level parent
        //
        // STATION FILTER
        Blockette nextBlk = null;
        Vector stationVec = new Vector(8,8);
        while ((nextBlk = (Blockette) getNext()) != null) {  // foreach blockette
            if (stations == null) {
                stations = new Vector(1);
                stations.add(null);  // add a single null entry
            }
            boolean found = false;
            for (int i = 0; i < stations.size(); i++) {   // foreach station listing
                String nextStation = (String) stations.get(i);
                switch(catNum) {
                case 3:
                    // blockette 50
                    if (nextStation == null || nextStation.equals(nextBlk.toString(3))) found = true;
                    break;
                case 4:
                    int blkType = nextBlk.getType();
                    // blockette 70-71
                    if (blkType == 70 || blkType == 71) found = true;  // just let these pass
                    // blockette 72 && 74
                    if (blkType == 72 || blkType == 74) {
                        if (nextStation == null || nextStation.equals(nextBlk.toString(3))) found = true;
                    }
                    // blockette 73
                    // TODO - postponed for now
                    break;
                case 5:
                    blkType = nextBlk.getType();
                    // Blockette 999 -- FSDH
                    if (blkType == 999) {
                        if (nextStation == null || nextStation.equals(nextBlk.toString(4))) found = true;
                    }
                    break;
                case 1:
                case 2:
                default:
                    // do nothing, let it pass through
                }
                if (found) break;
            }
            if (! found) continue;  // no station found?  skip this blockette
            stationVec.add(nextBlk);  // remember this station match
        }
        //
        // CHANNEL FILTER
        for (int i = 0; i < stationVec.size(); i++) {  // for each blockette with a matching station...
            if (channels == null) break;
            nextBlk = (Blockette) stationVec.get(i);
            boolean found = false;
            for (int j = 0; j < channels.size(); j++) {   // foreach channel listing
                String nextChannel = (String) channels.get(j);
                switch(catNum) {
                case 3:
                    // must look at child blockettes
                    for (int k = 0; k < nextBlk.numberofChildBlockettes(); k++) {
                        // get next child
                        Blockette childBlk = nextBlk.getChildBlockette(k);
                        if (childBlk == null) continue;
                        // blockette 52
                        if (nextChannel.equals(childBlk.toString(4))) {
                            found = true;
                            break;
                        }
                    }
                    break;
                case 4:
                    int blkType = nextBlk.getType();
                    // blockette 70-71
                    if (blkType == 70 || blkType == 71) found = true;  // just let these pass
                    // blockette 72 && 74
                    if (blkType == 72 || blkType == 74) {
                        if (nextChannel.equals(nextBlk.toString(5))) found = true;
                    }
                    // blockette 73
                    // TODO - postponed for now
                    break;
                case 5:
                    blkType = nextBlk.getType();
                    // Blockette 999 -- FSDH
                    if (blkType == 999) {
                        if (nextChannel.equals(nextBlk.toString(6))) found = true;
                    }
                    break;
                case 1:
                case 2:
                default:
                    // do nothing, let it pass through
                }
                if (found) break;
            }  // end for each channel
            if (! found) stationVec.remove(i);   // no channel match, eliminate blockette from list
        }  // end for each blockette matching station
        //
        // SET UP ITERATOR
        Collections.sort(stationVec);  // sort the elements of the vector
        currentIterator = stationVec.iterator(); // get an Iterator interface from vector
        return stationVec.size();   // return the number of elements
    }
    
    
    /**
     *	Get the next BlocketteDecorator in the established iterator, else return null.
     */
    public Object getNext() throws ContainerException {
        // get the next object in the established iterator, else return null
        if (currentIterator == null) return null;
        if (currentIterator.hasNext()) {
            return (Blockette) currentIterator.next();  // get next object
        } else {
            return null;
        }
    }
    
    // utility methods
    
    /**
     * Notify the container that the indicated blockette has been filtered.
     * Use this method to signal the container that the provided blockette
     * has been filtered out by the builder.  This will help indicate to the
     * container what rank of Blockette was filtered out so that the child
     * blockettes do not try to store into the Blockette rank lattice.
     */
    public void setFiltered(Blockette blk) throws Exception {
        // do not modify the filterRank value unless this Blockette is of dominant rank
        // (lower number) than the current setting OR if filterRank is 0 (dormant)
        int localRank = SeedBlocketteRankMap.getRank(blk);  // get this blockette's ranking
        localRank++;  // increment the value by 1
        if (localRank < filterRank || filterRank == 0) filterRank = localRank; // reset to the new setting
    }
    
    /**
     * Tell the container to assume that the indicated filter rank has been
     * filtered.
     * Explicitly set the filterRank value with this method.
     * Can be used to externally reset the filterRank to dormant
     * by using a value of 0
     */
    public void setFiltered(int value) {
        filterRank = value;
    }
    
    /**
     * Return the current filter rank value assigned to the container.
     * Return the filterRank value, which if non-zero, indicates the rank
     * value and higher of blockette types that are to be rejected until
     * filterRank is later reset to 0.
     */
    public int getFiltered() {
        return filterRank;
    }
    
//  DEACTIVATED
//  /**
//  * Set the number of objects that can be stored in the cache queue at any given time.
//  * Only applies to cases where serialization is used.
//  */
//  public void setCacheSize(GeneralCacheAdministrator cacheAdmin, int size) {
//  cacheSize = size;
//  if (isCaching && cacheAdmin != null) {
//  cacheAdmin.setCacheCapacity(size);
//  }
//  }
    
    /**
     * Flush all Blockettes from the memory cache.  This is typically called when attempting
     * to preserve all memory contents to disk before closing the container.
     */
    public void flushMemory() {
        if (isCaching && activeHashMaps != null) {
            for (int i = 0; i < activeHashMaps.size(); i++) {  // for each cached map
                ((CachedHashMap)activeHashMaps.get(i)).flush();  // flush memory to disk
            }
        }
    }
    
    /**
     * Return a string word representing the encoding type of data represented
     * in the indicated data blockette.  Does NOT resolve Blockette 1000 indicator,
     * ONLY METADATA REFERENCES.  Returns UNKNOWN if no match found.
     */
    public String getMetaEncoding(Blockette dataBlk) throws Exception {
        if (dataBlk == null) throw new ContainerException("attempted to resolve encoding for null dataBlk");
        if (dataBlk.getType() != 999) throw new ContainerException("can only resolve encoding for an FSDH blockette");
        EncodeKey dataKey = new EncodeKey();
        dataKey.station = dataBlk.toString(4);
        dataKey.location = dataBlk.toString(5);
        dataKey.channel = dataBlk.toString(6);
        dataKey.network = dataBlk.toString(7);
        Vector encodingVec = (Vector) encodingMap.get(dataKey.toString());  // try to find a match
        String curEncoding = "UNKNOWN";
        if (encodingVec != null) {
            Btime dataTime = (Btime) dataBlk.getFieldVal(8);
            long prevDiff = -1L;
            long curDiff = 0L;
            for (int i=0;i<encodingVec.size();i+=2) {
                curDiff = dataTime.diffSeconds((Btime) encodingVec.get(i));
                if (curDiff > 0 && curDiff < prevDiff || prevDiff == -1) { // data time greater than encoding epoch and closest time difference
                    prevDiff = curDiff;
                    curEncoding = (String) encodingVec.get(i+1);  // currently the best match
                }
            }
        }
        return curEncoding;
    }
    
    
    /**
     * Establish parent/child heirarchy for given lookup ID in the rank lattice.
     * Used typically for add() context.
     * @param lookupId the parent we want to establish in the rank lattice
     */
    public void setParent(int lookupId) throws ContainerException {
        rankLattice.setSize(0);  // clear the rank lattice
        // follow through parent-child map -- start with our parameter lookupId
        Vector parentVec = new Vector(3);
        int curLookupId = lookupId;  // start lookup id loop and work up through the parents
        while (curLookupId > 0) {
            // add current lookup ID's blockette to the parent vector
            //System.err.println("DEBUG: setParent(" + curLookupId + ")");
            Object blkGet = get(curLookupId);
            if (blkGet == null) break;  // exit loop if nothing comes back
            parentVec.add(blkGet);  // add the blockette to the vector
            Object parentLookupObj = parentChildMap.get(new Integer(curLookupId));  // get parent's lookupId -- root will map to 0
            if (parentLookupObj == null) curLookupId = 0;  // a null return will simply map to 0
            else curLookupId = ((Integer) parentLookupObj).intValue();  // otherwise, get the int equivalent of the id
        }
        int parentSize = parentVec.size();
        for (int i = 0; i < parentSize; i++) {
            rankLattice.add(parentVec.get(parentSize - i - 1)); 
        }
    }
    
    /**
     * Get the top level parent Blockette for the indicated lookupID.  Return a null if
     * no such parent is found.  A top level parent ID will simply return itself.  This
     * sets the global rankLattice vector to the current lookupID parent/child relationship.
     *
     */
    public Blockette getTopParent(int lookupId) {
        try {
            setParent(lookupId);
        } catch (ContainerException e) {
            System.err.println("Container Exception thrown: " + e);
            return null;
        }
        if (rankLattice.size() > 0) return (Blockette) rankLattice.get(0);
        else return null;
    }
    
    /**
     * Deprecated.  Use locate() instead.
     */
    public boolean improvedLocate(int lookupId) throws ContainerException {
        return locate(lookupId);
    }
    
    /**
     * Deprecated.  Use get() instead.
     */
    public Object improvedGet(int lookupId) throws ContainerException {
        return get(lookupId);
    }
    
    /**
     * return an integer that represents a new lookupID number to be assigned to a
     * new blockette, just before it is placed into the add() call.
     * Note that the id returned might will contain a volume number that is either
     * from that last id, or derived from the current volume number.  If the current volume
     * number is 0, it will ask the BlocketteDecoratorFactory for a new value.
     * Flag true for useRefNum to use already provided dictionary reference numbers
     * when the source of data is an internally consistent source such as a SEED file.
     */
    public int getNewId(Blockette blk, boolean useRefNum) throws Exception {
        int category = SeedBlocketteRankMap.getHeaderCode(blk);  // gets category number
        int blkType = blk.getType();  // get blockette type number
        int lastId = 0;
        if (category == 2) {
            // special array mapping for dictionary blockettes
            lastId = lastIdMap[blkType-20];
        } else {
            lastId = lastIdMap[category];
        }
        //System.err.println("DEBUG: getNewId.lastId=" + lastId);
        if (lastId == 0) {
            // we are the first entry, generate the first proper ID number
            lastId = category * 1000 * 1000;  // right-shifted category number
            // for dictionary blockettes, also add the blockette number
            // right-shifted into the ID
            if (category == 2) lastId += blkType * 1000 * 10;
        }
        if (lastId < 10000000) {  // our ID number is lacking a volume number
            if (currentVolumeNumber < 1) {  // we have no volume number, so get a new one
                currentVolumeNumber = BlocketteDecoratorFactory.getNewVolumeNumber();
                if (currentVolumeNumber > 214)
                    throw new BuilderException("getNewId(): new volumeNumber is too high.  must be value 000-214");
            }
            // add in the volume number to the ID
            lastId += (currentVolumeNumber * 1000 * 1000 * 10); // multipliers left-shift the volume value in the integer representation);
        }
        // do we use a reference number in the dictionary blockette?  only in cases of reading from
        // an internally consistent file, like SEED
        if (category == 2 && useRefNum) {
            int refField = SeedDictionaryReferenceMap.lookupDestFld(blkType);  // tells me which field of the blockette is a referenced field
            int refIndexNum = Integer.parseInt(blk.toString(refField));  // get the value at that referenced field
            return (lastId/1000) * 1000 + refIndexNum; // don't use incrementation, but instead use the reference number directly
        } else {
            // all other cases...
            return lastId + 1;  // return an incremented ID value
        }
    }
    
    /**
     * DEPRECATED -- does not handle predictive numbering for dictionary blockettes
     * 
     * Return a new lookupID number that is an incrementation of the previous one added to this container.
     * Each category has its own series.
     */
    public int getNewId(int category) throws Exception {
        if (category == 2) throw new Exception("getNewId(category) unable to process category 2 objects");
        int lastId = lastIdMap[category];
        if (lastId == 0) {
            // we are the first entry, generate the first proper ID number
            lastId = category * 1000 * 1000;  // right-shifted category number
        }
        if (lastId < 10000000) {  // our ID number is lacking a volume number
            if (currentVolumeNumber < 1) {  // we have no volume number, so get a new one
                currentVolumeNumber = BlocketteDecoratorFactory.getNewVolumeNumber();
                if (currentVolumeNumber > 214)
                    throw new BuilderException("getNewId(): new volumeNumber is too high.  must be value 000-214");
            }
            // add in the volume number to the ID
            lastId += (currentVolumeNumber * 1000 * 1000 * 10); // multipliers left-shift the volume value in the integer representation);
        }
        return lastId + 1;  // return an incremented ID value
    }
    
    /**
     * Return an integer that represents the lookupId of the
     * last added dictionary blockette to this container matching the
     * provided blockette delimited string. Return -1 if there is no matching
     * lookupId...which means that this dictionary reference has not
     * been added yet.
     * This check can be used to prevent writing identical dictionary blockettes
     * to the container, and consolidating other blockettes pointing to the same
     * information around one dictionary blockette.
     * NOTE: doesn't take into account multiple volumes, but treats all volumes as being
     * in the same dictionary pool, so be aware of this when checking for duplicates.
     * 
     */
    public int findMatchingDictionary(String blocketteString) {
        Iterator saveIterator = currentIterator;  // remember a current iterate() in progress
        //System.err.println("DEBUG: findMatchingDictionary iterate");
        iterate(2);  // will the return be consistent enough?
        Blockette nextBlk;
        try {
            while ((nextBlk = (Blockette) getNext()) != null) {  // foreach blockette
                //System.err.println("DEBUG: findMatchingDictionary getNext");
                if (nextBlk.toString().equals(blocketteString)) {
                    return nextBlk.getLookupId();  // matching blockette...return the lookupId
                }
            }
        } catch (ContainerException e) {
            System.err.println("Container exception encountered: " + e);
        }
        currentIterator = saveIterator;  // reconstitute the iterate() in progress
        return -1;
    }
    
    /**
     * return true if this container is caching objects to disk persistence.
     * 
     */
    public boolean isCaching() {
        return isCaching;
    }
    
    /**
     * pseudonym for isCaching() -- deprecated
     */
    public boolean isUsingSerialization() {
        return isCaching;
    }
    
    
    /**
     * force the current volume number of this container to be the
     * indicated value -- this number can get changed via the add()
     * command, which polls for the current volume.  Note that the volume number
     * is always assigned externally, either by an ObjectBuilder or other class.
     * New volume numbers can be grabbed from the BlocketteDecoratorFactory to prevent
     * duplication.
     * @param volNum volume number to assign to this container
     */
    public void setVolumeNumber(int volNum) {
        currentVolumeNumber = volNum;
    }
    
    /**
     * Return the current volume number this container is set to
     *
     */
    public int getVolumeNumber() {
        return currentVolumeNumber;
    }
    
    /**
     * utility to determine the volume number based on the provided lookupId.
     * @param lookupId
     * @return value representing the volume number of this id
     */
    public static int getVolumeNumberFromId(int lookupId) {
        return lookupId / (1000 * 1000 * 10);
    }
    
    /**
     * Return the category number encoded in the lookupId.
     */
    public static int getCategoryNumber(int lookupId) {
        return (lookupId / (1000 * 1000)) % 10;
    }
    
    /////////////////////
    // protected methods
    /////////////////////
    
    
    /**
     * Find the blockette in the hashmap that matches the blockette ID.  Returns an undecorated Blockette,
     * the raw data object that is stored to the CachedHashMap.
     */
    protected Blockette lookup(int lookupId) {
        Integer lookupIdObj = new Integer(lookupId);
        Object hashGet = null;
        // COMMENT OUT SLOW IMPLEMENTATION
//      hashGet = volume.get(lookupIdObj);
//      if (hashGet == null) hashGet = dictionary.get(lookupIdObj);
//      if (hashGet == null) hashGet = station.get(lookupIdObj);
//      if (hashGet == null) hashGet = timespan.get(lookupIdObj);
//      if (hashGet == null) hashGet = data.get(lookupIdObj);
        int category = getCategoryNumber(lookupId);
        switch (category) {
        case 1:
            hashGet = volume.get(lookupIdObj);
            break;
        case 2:
            hashGet = dictionary.get(lookupIdObj);
            break;
        case 3:
            hashGet = station.get(lookupIdObj);
            break;
        case 4:
            hashGet = timespan.get(lookupIdObj);
            break;
        case 5:
            hashGet = data.get(lookupIdObj);
            break;
        default:
            hashGet = null;
        }
        //
        // if our attempt to get the parent blockette failed, return a null Blockette
        if (hashGet == null) return null;
        // else return the found Blockette
        else return (Blockette) hashGet;
    }
    
    //////////////////
    // private methods
    //////////////////
    
    /////////////////
    // inner classes
    /////////////////
    
    /**
     * Inner class is used as a HashMap key to store an association
     * to a data encoding type.
     */
    class EncodeKey {
        String station = null;
        String network = null;
        String channel = null;
        String location = null;
        
        public String toString() {
            return "" + station + "." + network + "." + channel + "." + location;
        }
    }
    
    /**
     * Inner class used as a HashMap but triggers use of the persistence cache if cacheAdmin is not null.  A generic HashMap
     * is used if cacheAdmin is null.
     * @param group the unique group name assigned to this map
     * @param capacity the maximum number of in-memory entities allowed in the cache at one time
     * @param isCaching flagged to true if using disk persistent caching, false to use in-memory HashMap
     * @param loadPrevious if true, accesses already existing disk entries, if false deletes those disk entries
     */
    // REC
    class CachedHashMap {
        
        public CachedHashMap(String group, int capacity, boolean isCaching, boolean loadPrevious) throws CachePersistenceException {
            this.isCaching = isCaching;
            this.group = group;
            if (isCaching) {
                // set up OSCache properties
                cacheProps = new Properties();
                cacheProps.setProperty("cache.memory","true");  // use memory caching
                cacheProps.setProperty("cache.capacity",Integer.toString(capacity)); // max items in memory cache
                cacheProps.setProperty("cache.algorithm","com.opensymphony.oscache.base.algorithm.LRUCache");  // cache algorithm
                cacheProps.setProperty("cache.blocking","false");  // block on stale cache entries?
                cacheProps.setProperty("cache.unlimited.disk","true");  // assume unlimited disk persistence space
                // specify disk persistence Java class
                cacheProps.setProperty("cache.persistence.class",
                "com.opensymphony.oscache.plugins.diskpersistence.DiskPersistenceListener");
                cacheProps.setProperty("cache.path",cacheDir+"/" + group);  // specify the base cache directory with group subdirectory
                cacheProps.setProperty("cache.persistence.overflow.only","false");  // persist always, not just on overflow
                cacheProps.setProperty("cache.key",group);  // base tag for disk cache
                //
                // establish an instance of cache administrator
                cacheAdmin = new GeneralCacheAdministrator(cacheProps);
                if (loadPrevious) {
                    // if loadPrevious, then access the disk cache entries
                    // automatically drawn out by Object Container iteration -- set entrySet() below
                } else {
                    // if not loadPrevious, then destroy any existing disk entries
                    cacheAdmin.getCache().getPersistenceListener().clear();
                }
            } else {
                nonCacheHashMap = new HashMap();  // non-caching case - create in-memory hash map
            }
            //System.err.println("DEBUG: " + toString());
        }
        
        public void put(Object key, Object value) {
            if (isCaching) {
                //cacheAdmin.putInCache(key.toString(),value,group);
                cacheAdmin.putInCache(key.toString(),value);
            } else {
                nonCacheHashMap.put(key,value);
            }
            //System.err.println("DEBUG: Map put() group " + group + " with key " + key.toString() +
            //        " and value " + value.toString());
        }
        
        public Object get(Object key) {
            //System.err.println("DEBUG: CacheMap get() " + key);
            if (isCaching) {
                tmpValue = null;
                try {
                    tmpValue = cacheAdmin.getFromCache(key.toString());
                } catch (NeedsRefreshException e) {
                    System.err.println("CachedHashMap.get(): NeedsRefreshException encountered for " + toString() + ": " + e);
                }
                return tmpValue;
            } else {
                return nonCacheHashMap.get(key);
            }
        }
        
        // remove from memory and disk persistence
        public void remove(Object key) {
            if (isCaching) {
                cacheAdmin.getCache().removeCacheEntry(key.toString());
            } else {
                nonCacheHashMap.remove(key);
            }
        }
        
        // push all from memory to disk persistence -- clear memory references
        public void flush() {
            if (isCaching) {
                cacheAdmin.getCache().flushAll();
            }
        }
        
        // check to see if a key is already mapped
        public boolean containsKey(Object key) {
            if (isCaching) {
                return (get(key.toString()) != null);
            } else {
                return nonCacheHashMap.containsKey(key);
            }
        }
        
        // return a collection of key-value pairs
        //public Set entrySet() {
        public Collection entrySet() {
            if (isCaching) {
                //System.err.println("DEBUG: return entrySet from group " + group);
                return cacheAdmin.getCache().entrySet();
            } else {
                return nonCacheHashMap.entrySet();
            }
        }
        
        public String toString() {
            return "CachedHashMap for group " + group + " with caching set to " + isCaching;
        }
        
        private Properties cacheProps = null;
        private GeneralCacheAdministrator cacheAdmin = null;
        private boolean isCaching = false;
        private String group = null;
        private Object tmpValue = null;
        private HashMap nonCacheHashMap = null;
        
    }
    
    // instance variables
    
    private Vector activeHashMaps = null;  // new cached hash maps that are instantiated will be listed here
    
    // these HashMaps represent the storage lattice for incoming blockettes
    // the keys are the location ID's of the Blockettes themselves
    // each category is listed here
    private CachedHashMap volume = null;     // category 1: volume header blockettes
    private CachedHashMap dictionary = null; // category 2: dictionary header blockettes
    private CachedHashMap station = null;    // category 3: station header blockettes
    private CachedHashMap timespan = null;   // category 4: time span header blockettes
    private CachedHashMap data = null;       // category 5: FSDH pseudo-blockettes, with attached data blockettes and waveforms
    
    private Blockette locateRegister = null;    // holding place for locate()'d Blockettes
    private Blockette addRegister = null;       // holding place for add()'d Blockettes
    private BlocketteVector rankLattice = null; // holding place for Blockette parent-child association
    private Iterator currentIterator = null;    // holding place for a series of Blockettes to be iterated over
    
    // OSCache persistence
//  private Properties cacheProps = null;
//  private GeneralCacheAdministrator cacheAdmin = null;
    private boolean isCaching = false;
    // keep tabs on the base directory of the disk persistence cache, should it be made available
    private String cacheDir = null;
    // the maximum number of entities allowed in cache memory at one time
    private static final int defaultCacheSize = 50;    // default cache size
    private int cacheSize = defaultCacheSize;    // maximum number of elements in caching queue
    
    // in the case of a build filter being used, indicates this rank value and higher is rejected
    private int filterRank = 0;
    
    // stores a map of EncodeKeys to data encoding string identifier
    private CachedHashMap encodingMap = null;
    
    // stores mapping of child lookupId to its parent lookupId.  This will help in cases
    // where we want to locate a specific child blockette.  Iterating backward
    // through the parent-child mapping step by step will help establish the depth
    // or rank of the child.  The topmost parent will be mapped to a value of
    // zero.
    private CachedHashMap parentChildMap = null;
    
    // keep track of highest ID number assigned to this container, per category
    // number (category, lookupID)
    private int[] lastIdMap = new int[30];
    
    // track the current volume number that is in active use in the add() method
    private int currentVolumeNumber = 0;
    
    //  sets to true if init preloading from a disk cache (see iterate())
    private boolean cachePreload = false;
    
    
}
