#include <bios.h>
#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <malloc.h>
#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "screen.h"
#include "extkeys.h"

extern struct vcfg cfg;


/* write a character and attribute during horizontal retrace for
   CGA
   Input ES:DI has destination buffer position
         DS:SI has source buffer position
*/
void retrace( int count )
{
   _asm {
        mov     cx, WORD PTR count
        mov     dx, 03DAh       ; horizontal status port on CGA
wait0:
        lodsw                   ; load character AX
        mov     bx, ax          ; save character in bx
wait1:
        in      al, dx          ; wait till horizontal active
        test    al, 1
        jnz     wait1
        cli
wait2:
        in      al, dx          ; wait till horizontal inactive (retrace)
        test    al, 1
        jz      wait2

        xchg    ax, bx          ; restore character and
        stosw                   ;   move to video memory
        sti
        loop    wait0           ; next character and attribute
   }
}


/*
 * Notes:   Uses the BIOS to read the next keyboard character.  Service
 *          0 is keyboard read.  Service 0x10 is the extended keyboard read.
 */
int  getkey( void )
{
unsigned key;
register scan;
register unsigned lo;

   /*
    *  _bios_keybrd == int 16.  It returns the scan code in ah, hi part of key,
    *  and the ascii key code in al, lo part of key.  If the character was
    *  entered via ALT-xxx, the scan code, hi part of key, is 0.
    */
   key = _bios_keybrd( 0 );
   scan = (key & 0xff00) >> 8;
   lo = key & 0x00ff;
   if (lo == 0)
      lo = scan | 0x100;
   return( lo );
}


void write_to_window( struct screen *p, int start_col, int start_row,
                      int attribute )
{
   while (p->text) {
      fast_write( p->col+start_col, p->row+start_row, p->text, attribute );
      p++;
   }
}


void hlight_line( int x, int y, int lgth, int attr )
{
int off, far *pointer;

   pointer = cfg.videomem;
   off = y * 160 + 2 * x + 1;  /* add one - so it points to attribute byte */
   _asm {
        push    es              ; save es

        mov     cx, lgth        ; number of characters to change color
        mov     ax, attr        ; attribute

        les     di, pointer             ; get destination - video memory
        add     di, off                 ; add offset
        cld                             ; clear direction flag - low to high
        cmp     cfg.rescan, FALSE       ; is this CGA?
        je      notcga

        mov     dx, 03DAh       ; horizontal status port on CGA
        ALIGN   2
wait0:
        mov     bx, ax          ; save character in bx
        ALIGN   2
wait1:
        in      al, dx          ; wait till horizontal active
        test    al, 1
        jnz     wait1
        cli
        ALIGN   2
wait2:
        in      al, dx          ; wait till horizontal inactive (retrace)
        test    al, 1
        jz      wait2

        xchg    ax, bx          ; restore character and
        stosb                   ;   move a BYTE to video memory
        inc     di              ; skip over character to next attribute
        sti
        loop    wait0           ; next attribute
        jmp     SHORT getout
        ALIGN   2
notcga:
        stosb                   ; store a BYTE
        inc     di              ; skip over character to next attribute
        loop    notcga          ; change next attribute
getout:
        pop     es              ; restore es
   }
}


void show_file_position( struct file_position pos, int col_offset )
{
char r[20], c[8];
int i;

   itoa( pos.f_row, r, 10 );
   itoa( pos.f_col+col_offset, c, 10 );
   strcat( r, ":" );
   strcat( r, c );
   for (i=strlen( r ); i<8; i++)
      insert( r, 0, ' ' );
   fast_write( pos.d_col, pos.d_row, r, pos.color );
}


void set_cur( int row, int col )
{
   _asm {
        mov     ah, 2           ; set cursor position
        mov     bh, 0           ; always use page 0
        mov     dh, BYTE PTR row
        mov     dl, BYTE PTR col
        int     VIDEO_IO
   }
}


void c_off( void )
{
   _asm {
        mov     ah, 1
        mov     ch, 16
        mov     cl, 0
        int     VIDEO_IO
   }
}


void c_on( void )
{
int cursor;

   cursor = cfg.insert_cursor;
   _asm {
        mov     cx, WORD PTR cursor
        mov     ah, 1           ; function 1 - set cursor size
        int     VIDEO_IO
   }
}


void write_cur( char chr )
{
   _asm {
        mov     ah, 10          ; write character w/o moving
        mov     al, chr
        mov     bh, 0           ; always use page 0
        mov     cx, 1           ; no of copies
        int     VIDEO_IO
   }
}


void insert_cursor( void )
{
int cursor;

   cursor = cfg.overw_cursor;
   _asm {
        mov     cx, WORD PTR cursor
        mov     ah, 1           ; function 1 - set cursor size
        int     VIDEO_IO
   }
}


/*
void insert( char *t, int col, int c )
{
int i,temp;

   for (i=0; c; i++) {
      temp = *(t + col + i);
      *(t +col + i) = c;
      c = temp;
    }
    *(t + col + i) = '\0';
}
*/

void insert( char far *t, int col, int c )
{
   _asm {
        push    ds

        xor     dx, dx          ; use dx to test for '\0'
        mov     bx, col         ; use bx as index into string
        lds     si, DWORD PTR t ; load address of far pointer t
        mov     ax, c           ; use ah to hold temp variable
        mov     ah, al
        ALIGN   2
a1:
        mov     al,BYTE PTR [si+bx]     ; get character from string
        cmp     dl, al                  ; are we at end of string?
        je      getout                  ; if so, get out
        mov     BYTE PTR [si+bx], ah    ; put new letter in
        mov     ah, al                  ; old letter becomes new letter
        inc     bx                      ; increment index
        jmp     SHORT a1
getout:
        mov     BYTE PTR [si+bx], ah    ; store last character in array
        inc     bx                      ; inc base
        mov     BYTE PTR [si+bx], al    ; store '\0' in last position

        pop     ds
   }
}


/*
void delete( char *t, int col )
{
int i;

   for (i=0; *(t + col +i); i++)
      *(t + col +i) = *(t + col + i + 1);
}
*/

void delete( char far *t, int col )
{
   _asm {
        push    ds

        xor     dx, dx
;        mov     si, t
        lds     si, DWORD PTR t
        mov     bx, col
        mov     al, BYTE PTR [si+bx]
        ALIGN   2
a1:
        cmp     al, dl
        je      getout
        mov     al, BYTE PTR [si+bx+1]
        mov     BYTE PTR [si+bx], al
        inc     bx
        jmp     SHORT a1
getout:
        pop     ds
   }
}


void ftoa( float num, int places, char *t )
{
int dec, sgn;

   strcpy( t, fcvt( (double)num, places, &dec, &sgn ) );
   if (dec < 0)
      for (; dec < 0; dec++)
         insert( t, 0, '0' );
   insert( t, dec, '.' );
   if (sgn != 0)
      insert( t, 0, '-' );
}


void scroll_window( int lines, int r1, int c1, int r2, int c2, int attr )
{
char rah, ral;
   _asm {
        mov     ax, lines
        cmp     ax, 0           ; if line < 0  - scroll window down
        jge     a1

        neg     ax                      ; make lines positive
        mov     BYTE PTR ral, al        ; save number of lines
        mov     BYTE PTR rah, 7         ; function 7 -  scroll window down
        dec     r2                      ; decrement row 2
        jmp     SHORT a2
a1:
        mov     BYTE PTR ral, al        ; number of lines to scroll
        mov     BYTE PTR rah, 6         ; function 6 - scroll window up
a2:
        mov     ax, WORD PTR r1         ; get starting row
        mov     ch, al                  ; put it in ch
        mov     ax, WORD PTR c1         ; get starting column
        mov     cl, al                  ; put it in cl
        mov     ax, WORD PTR r2         ; get ending row
        mov     dh, al                  ; put it in dh
        mov     ax, WORD PTR c2         ; get ending column
        mov     dl, al                  ; put it in cl
        mov     ax, WORD PTR attr       ; get attribute
        mov     bh, al                  ; put it in bh
        mov     ah, BYTE PTR rah        ; get function number
        mov     al, BYTE PTR ral        ; get number of lines
        push    bp                      ; *** in some early versions of
        int     VIDEO_IO                ; *** DOS, u must save bp
        pop     bp
   }
}


void cls( void )
{
   scroll_window( 0, 0, 0, 24, 79, NORMAL );
}


/*
 *   Video BIOS Data Areas
 *   The BIOS routines maintain several dynamic variables in an area of
 *   memory called the Video Display Data Area.  The following contains a
 *   summary of these variables' addresses, their symbolic names, and
 *   their contents.  All addresses are relative to the 0x0000h segment.
 *   From the IBM Technical Reference and other sources.
 *
 *   Address  Name           Type   Description
 *   0x0449   CRT_MODE       Byte   Current BIOS video number
 *   0x044a   CRT_COLS       Word   Number of displayed character columns
 *   0x044c   CRT_LEN        Word   Size of video buffer in bytes
 *   0x044e   CRT_START      Word   Offset of start of video buffer
 *   0x0450   CURSOR_POSN    Word   Array of eight words containing the cursor
 *                                    position for each of eight possible
 *                                    video pages.  The high-order byte of
 *                                    each word contains the character row,
 *                                    the low-order byte the character column
 *   0x0460   CURSOR_MODE    Word   Starting and ending lines for alphanumeric
 *                                    cursor.  The high-order byte contains
 *                                    the starting (top) line; the low-order
 *                                    byte contains the ending (bottom) line
 *   0x0462   ACTIVE_PAGE    Byte   Currently displayed video page number
 *   0x0463   ADDR_6845      Word   I/O port address of CRT Controller's
 *                                    Address register (3B4h for mono;
 *                                    3D4h for color)
 *   0x0465   CRT_MODE_SET   Byte   Current value for Mode Control register
 *                                    (3B8h on MDA, 3D8h on CGA).  On the
 *                                    EGA and VGA, the value emulates those
 *                                    used on the MDA and CGA.
 *   0x0466   CRT_PALETTE    Byte   Current value for the CGA Color Select
 *                                    register (3D9h).  On the EGA and VGA,
 *                                    the value emulates those used on the
 *                                    MDA and CGA.
 *   0x0467   io_rom_init    Word   Pointer to optional i/o rom init routine
 *   0x0469   io_rom_seg     Word   Pointer to io rom segment
 *   0x046b   intr_flag      Byte   Flag to indicate an interrupt happened
 *   0x046c   timer_low      Word   Low word of timer count
 *   0x046e   timer_high     Word   High word of timer count
 *   0x0470   timer_ofl      Byte   Timer has rolled over since last count
 *   0x0471   bios_break     Byte   Bit 7 = 1 if Break Key has been hit
 *   0x0472   reset_flag     Word   Word = 1234h if keyboard reset underway
 *   0x0484   ROWS           Byte   Number of displayed character rows - 1
 *   0x0485   POINTS         Word   Height of character matrix
 *   0x0487   INFO           Byte   EGA and VGA display data
 *   0x0488   INFO_3         Byte   Configuration switches for EGA and VGA
 *   0x0489   flags          Byte   Miscellaneous flags
 *   0x0496   kb_flag_3      Byte   Additional keyboard flag
 *   0x048A   DCC            Byte   Display Combination Code
 *   0x04A8   SAVE_PTR       Dword  Pointer to BIOS save area
 *
*/
void get_vcfg( void )
{
#pragma pack( 1 )    /* Use pragma to force packing on byte boundaries. */

struct LOWMEMVID
{
   char     vidmode;           /* 0x449 */
   unsigned scrwid;            /* 0x44A */
   unsigned scrlen;            /* 0x44C */
   unsigned scroff;            /* 0x44E */
   struct   LOCATE
   {
      unsigned char col;
      unsigned char row;
   } csrpos[8];                /* 0x450 */
   struct   CURSIZE
   {
      unsigned char end;
      unsigned char start;
   } csrsize;                  /* 0x460 */
   char      page;             /* 0x462 */
   unsigned  addr_6845;        /* 0x463 */
   char      crt_mode_set;     /* 0x465 */
   char      crt_palette[30];  /* 0x466 */
   char      rows;             /* 0x484 */
   unsigned  points;           /* 0x485 */
   char      ega_info;         /* 0x487 */
   char      info_3;           /* 0x488 */
   char      skip[13];         /* 0x489 */
   char      kb_flag_3;        /* 0x496 */
} vid;
struct LOWMEMVID _far *pvid = &vid;
#pragma pack( )    /* revert to previously defined pack pragma. */

union REGS in, out;
unsigned char temp, active_display;

   /*
    * Move system information into our video structure.
    */
   _fmemmove( pvid, (void far *)0x00400049l, sizeof( vid ) );

   cfg.rescan = FALSE;
   in.x.ax =  0x1a00;
   int86( VIDEO_IO, &in, &out );
   temp = out.h.al;
   active_display = out.h.bl;
   if (temp == 0x1a && (active_display == 7 || active_display == 8))
      cfg.adapter = VGA;
   else {
      in.h.ah =  0x12;
      in.h.bl =  0x10;
      int86( VIDEO_IO, &in, &out );
      if (out.h.bl != 0x10) {         /* EGA */
         if (vid.ega_info & 0x08)
            cfg.adapter = (vid.addr_6845 == 0x3d4) ? CGA : MDA;
         else
            cfg.adapter = EGA;
      } else if (vid.addr_6845 == 0x3d4)
         cfg.adapter = CGA;
      else
         cfg.adapter = MDA;
   }

   if (cfg.adapter == CGA || cfg.adapter == EGA) {
      if (cfg.adapter == CGA)
         cfg.rescan = TRUE;
      cfg.insert_cursor = 0x0607;
      cfg.overw_cursor = 0x0407;
   } else {
      cfg.insert_cursor = 0x0b0c;
      cfg.overw_cursor = 0x070b;
   }

   cfg.mode = vid.vidmode;
   if (vid.addr_6845 == 0x3D4) {
      cfg.color = TRUE;
      cfg.hi_i  = PRIMARY_FLD;
      cfg.lo_i  = SECOND_FLD;
      cfg.f_bar = F_KEY_BAR;
      cfg.videomem = (void far *)0xb8000000l;
   } else {
      cfg.color = FALSE;
      cfg.hi_i  = REVERSE;
      cfg.f_bar = REVERSE;
      cfg.lo_i  = HI_INTEN;
      cfg.videomem = (void far *)0xb0000000l;
   }
}


void fast_write( int x, int y, char far *strng, int attr )
{
int i, off, far *pointer;
static unsigned char temp[200];
unsigned char *p;


   /*  each character has an attribute byte and an ASCII byte
       video offset = row * (2 bytes * 80 columns = 160) + column * (2 bytes)
                    = row * 160 + col * 2
      procedure :
          a. read each character from string
          b. write character and attribute to screen
   */
   pointer = cfg.videomem;
   off = y*160 + 2*x;
/*
   i=0;
   while (*strng) {
      temp[i++] = *strng++;
      temp[i++] = attr;
   }
   p = temp;
   movedata(FP_SEG(p), FP_OFF(p), FP_SEG(pointer), off, i);
*/

   _asm {
        push    es              ; save es
        push    ds              ; save ds

        cld                     ; direction flag, from low to hi memory
;        mov     si, strng       ; get address of source string
        lds     si, strng       ; get address of source string
        les     di, pointer     ; get destination - video memory
        add     di, off         ; add offset to screen pointer
        mov     ax, attr        ; get attribute
        mov     ah, al          ; put attribute in ah
        mov     dx, 03DAh       ; horizontal status port on CGA
        cmp     WORD PTR cfg.rescan, FALSE   ; do we need to wait for rescan?
        je      notcga
a1:
        ALIGN   2
        lodsb                   ; get character
        or      al, al          ; is this '\0'
        je      getout          ; if yes, we're done
        mov     bx, ax          ; save character in bx
        ALIGN   2
wait1:
        in      al, dx          ; wait till horizontal active
        test    al, 1
        jnz     wait1
        cli
        ALIGN   2
wait2:
        in      al, dx          ; wait till horizontal inactive (retrace)
        test    al, 1
        jz      wait2

        xchg    ax, bx          ; restore character and
        stosw                   ;   move to video memory
        sti                     ; ok for interrupts
        jmp     SHORT a1        ; done for CGA

notcga:
        ALIGN   2
        lodsb                   ; get character
        or      al, al          ; is this '\0'
        je      getout          ; if yes, we're done
        stosw                   ; no need to wait if not CGA
        jmp     SHORT notcga
getout:
        pop     ds
        pop     es
   }
}


void wherexy( int *x, int *y )
{
   _asm {
        mov     ah, 3           ; read cursor position
        mov     bh, 0           ; page zero
        int     VIDEO_IO
        xor     ax, ax
        mov     al, dl
        mov     bx, WORD PTR x
        mov     [bx], ax
        mov     al, dh
        mov     bx, WORD PTR y
        mov     [bx], ax
   }
}


int set_color( int foreground, int background )
{
  return( background<<4 | foreground );
}


#define   U_LEFT    218
#define   U_RIGHT   191
#define   VER_LINE  179
#define   LINE      196
#define   L_LEFT    192
#define   L_RIGHT   217


void show_box( int x, int y, struct screen *p, int attr )
{
   while (p->text) {
      fast_write( p->col+x, p->row+y, p->text, attr );
      p++;
   }
}


/*
 * draws and clears a box
 */
void no_buf_make_window( int col, int row, int width, int height, int attr )
{
   no_buf_box( col++, row++, width, height, attr );  /* DRAW BOX       */
   no_buf_clear_window( col, row, width-2, height-2 );
}


void no_buf_box( int col, int row, int width, int height, int attr )
{
int i, row_count;
char string[82];

   if (height > 0 && width > 0 && height < 25 && width < 81) {
      row_count = 1;
      string[0]= U_LEFT;
      for (i=1; i<width-1; i++)
         string[i] = LINE;
      string[i++] = U_RIGHT; string[i] = '\0';
      fast_write( col, row, string, attr );
      ++row_count;
      ++row;

      if (row_count < height) {
         string[0] = VER_LINE;
         string[1] = '\0';
         for (i=1; i<height-1; i++) {
            fast_write( col, row, string, attr );
            fast_write( col+width-1, row, string, attr );
            ++row;
            ++row_count;
         }
      }

      if (row_count <= height) {
         string[0] = L_LEFT;
         for (i=1; i<width-1; i++)
            string[i] = LINE;
         string[i++] = L_RIGHT; string[i] = '\0';
         fast_write( col, row, string, attr );
      }
   }
}


void no_buf_clear_window( int col, int row, int width, int height )
{
int far *destination;

   destination = cfg.videomem;
   _asm {
        mov     ax, row         ; get row in ax
        mov     bx, 160         ; put 160 in bx
        mul     bl              ; multiply al * bl ==> ax
        mov     cx, ax          ; save ax in cx  cx = row * 160
        mov     ax, col         ; get col in ax
        shl     ax, 1           ; col * 2
        add     ax, cx          ; row * 160 + col * 2
        les     di, destination ; get buffer pointer
        add     di, ax          ; make di point to right row and col
        mov     dx, WORD PTR cfg.rescan         ; get rescan b4 ds is gone
        mov     ah, NORMAL      ; normal attr - white on black background
        mov     al, ' '         ; use space
a1:
        cmp     height, 0
        jbe     out2            ; any more rows left
        mov     cx, width       ; number of columns to blank
        cmp     cx, 0           ; make sure cx is positive
        jbe     a3              ; any more columns left
        cmp     dx, FALSE       ; is this CGA?
        je      notcga          ; move line reguardless of retrace

        push    dx              ; dx still has rescan, save it
        push    di              ; save di so we can go to next screen line
        mov     dx, 03DAh       ; horizontal status port on CGA
wait0:
        mov     bx, ax          ; save character in bx
wait1:
        in      al, dx          ; wait till horizontal active
        test    al, 1
        jnz     wait1
        cli                     ; no interrupts alowed
wait2:
        in      al, dx          ; wait till horizontal inactive (retrace)
        test    al, 1
        jz      wait2

        xchg    ax, bx          ; restore character and
        stosw                   ;   move to video memory
        sti                     ; turn interrupts back on
        loop    wait0           ; cx has implied count

        pop     di              ; get back original row and col
        pop     dx              ; get back rescan
        jmp     SHORT a3        ; done for CGA

notcga:
        push    di              ; save videomem location
        rep     stosw           ; store word
        pop     di              ; get back videomem location
a3:
        add     di, 160         ; make videomem point to next line
        dec     height          ; ready for another row
        jmp     a1
out2:
   }
}


void window_control( SMALL_WINDOW **window_ptr, int action, int col, int row,
                     int width, int height )
{
SMALL_WINDOW  *p;
size_t store_me;

   if (action == SAVE) {                 /* SAVE */
      p = (SMALL_WINDOW *)malloc( sizeof(SMALL_WINDOW) );
      if (p != NULL) {
         p->n = NULL;
         if (*window_ptr != NULL)             /* push a window on stack */
            p->n = *window_ptr;
         *window_ptr = p;
         store_me = (width * height) * sizeof( int );
         p->buf = (int *)malloc( store_me );
         save_window( p->buf, col, row, width, height );
      }
   } else if (action == RESTORE) {                 /* RESTORE */
      if (*window_ptr != NULL) {
         p = *window_ptr;
         restore_window( p->buf, col, row, width, height );
         *window_ptr = p->n;                 /* pop a window off stack */
         free( p->buf );
         free( p );
}  }  }


/*
void save_window( int *destination, int col, int row, int width, int height )
{
int i, j, offset;
int far *pointer;

   pointer = cfg.videomem;
   offset = row * 80 + col;
   pointer += offset;
   for (i=0; i < height; i++) {
      for (j=0; j < width; j++)
         *destination++ = *(pointer + j);
      pointer += 80;
   }
}
*/

void save_window( int far *destination, int col, int row, int width,
                  int height )
{
int far *temp;

   temp = cfg.videomem;    /* temporary variable - save pointer */
   _asm {
        push    es
        push    ds

        mov     ax, WORD PTR row        ; get row in ax
        mov     bx, 160                 ; put 160 in bl
        mul     bl                      ; multiply, al * bl ==> ax
        mov     cx, ax                  ; save in cx

        mov     ax, WORD PTR col        ; get col in ax
        shl     ax, 1                   ; multiply by 2
        add     ax, cx                  ; ax = row * 160 + col * 2

        mov     dx, WORD PTR cfg.rescan         ; get rescan b4 ds is gone
;        mov     di, destination                ; load es:di
        les     di, DWORD PTR destination       ; load es:di
;        lds     si, DWORD PTR cfg.videomem     ; load ds:si - ds is gone
        lds     si, DWORD PTR temp      ; load ds:si - use stack since ds gone
        add     si, ax                  ; go to right row and col
        ALIGN   2
b1:
        cmp     height, 0       ; any more rows left?
        jbe     getout          ; if not, we're done

        mov     cx, width       ; load count register
        cmp     dx, FALSE       ; is this CGA?
        je      notcga          ; if not, forget rescan

        push    dx              ; dx still has rescan, save it
        push    cx              ; save number of words to write, pass to retrace
        call    retrace         ; write during horizontal retrace
        pop     ax              ; get back number of words written
        shl     ax, 1           ; multiply by 2
        add     di, ax          ; di = di + num of bytes
        pop     dx              ; get back rescan
        jmp     SHORT b2        ; done for CGA
        ALIGN   2
notcga:
        push    si              ; save videomem location
        rep     movsw           ; copy to screen with no waiting
        pop     si              ; get back videomem location
        ALIGN   2
b2:
        add     si, 160         ; make videomem point to next line
        dec     height          ; ready for another row
        jmp     SHORT b1
getout:
        pop     ds
        pop     es
   }
}


/*
void restore_window( int *source, int col, int row, int width, int height )
{
int i, j, offset;
int far *pointer;

   pointer = cfg.videomem;
   offset = row * 80 + col;
   pointer += offset;
   for (i=0; i < height; i++) {
      for (j=0; j < width; j++)
         *(pointer + j) = *source++;
      pointer += 80;
   }
}
*/

void restore_window( int far *source, int col, int row, int width, int height )
{
int far *temp, far *video;

   temp = source;
   video = cfg.videomem;
   _asm {
        push    es
        push    ds

        mov     ax, WORD PTR row        ; load ax with row
        mov     bx, 160                 ; load bx with 160 (no. bytes / row )
        mul     bl                      ; multiply, al * bl ==> ax
        mov     cx, ax                  ; store ax in cx

        mov     ax, WORD PTR col        ; get number of columns
        shl     ax, 1                   ; multiply by 2 = number of bytes
        add     ax, cx                  ; ax = row * 160 + col * 2

        mov     dx, WORD PTR cfg.rescan         ; get rescan b4 ds is gone
        les     di, video               ; load es:di with videomem
;        mov     si, source                      ; load ds:si with buffer
;        lds     si, source                      ; load ds:si with buffer
        lds     si, DWORD PTR temp              ; load ds:si with buffer
        add     di, ax                          ; go to right row and col
        ALIGN   2
c1:
        cmp     height, 0       ; any more rows left?
        jbe     getout          ; if not, we're done

        mov     cx, width       ; load count register w/ width
        cmp     dx, FALSE       ; is this CGA?
        je      notcga          ; if not, forget rescan

        push    dx              ; save rescan
        push    cx              ; save number of words to write, pass to retrace
        call    retrace         ; write during horizontal retrace
        pop     ax              ; get number of words written
        shl     ax, 1           ; multiply by 2 to get number of bytes
        add     si, ax          ; add it to screen buffer
        pop     dx              ; get back rescan
        jmp     SHORT c2        ; done for CGA
        ALIGN   2
notcga:
        push    di              ; save video pointer
        rep     movsw           ; copy to screen with no waiting
        pop     di              ; get back video pointer
        ALIGN   2
c2:
        add     di, 160         ; make video pointer point to next row
        dec     height          ; we're done with this row
        jmp     SHORT c1        ; do the next row
getout:
        pop     ds
        pop     es
   }
}
