!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!                                                             
!   glissade_masks.F90 - part of the Community Ice Sheet Model (CISM)  
!                                                              
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!
!   Copyright (C) 2005-2018
!   CISM contributors - see AUTHORS file for list of contributors
!
!   This file is part of CISM.
!
!   CISM is free software: you can redistribute it and/or modify it
!   under the terms of the Lesser GNU General Public License as published
!   by the Free Software Foundation, either version 3 of the License, or
!   (at your option) any later version.
!
!   CISM is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   Lesser GNU General Public License for more details.
!
!   You should have received a copy of the Lesser GNU General Public License
!   along with CISM. If not, see <http://www.gnu.org/licenses/>.
!
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!
! This module contains routines for computing various masks used by the Glissade 
! velocity solver.
!
! Author: William Lipscomb
!         Los Alamos National Laboratory
!         Group T-3, MS B216
!         Los Alamos, NM 87545
!         USA
!         <lipscomb@lanl.gov>
!
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

  module glissade_masks

    use glimmer_global, only: dp
    use glimmer_paramets, only: iulog
    use glimmer_physcon, only: rhoi, rhoo
    use glimmer_log
    use glimmer_utils, only: point_diag
    use glide_types
    use cism_parallel, only: this_rank, main_task, nhalo, parallel_globalindex, &
         parallel_type, parallel_halo, parallel_reduce_sum

    implicit none

    private
    public :: glissade_get_masks, glissade_calving_front_mask,      &
              glissade_marine_cliff_mask, glissade_ice_sheet_mask,  &
              glissade_ocean_connection_mask,                       &
              glissade_marine_connection_mask, glissade_lake_mask,  &
              glissade_extend_mask, glissade_fill, glissade_fill_with_buffer

    public :: initial_color, fill_color, boundary_color

    ! colors for fill subroutines
    integer, parameter :: initial_color = 0   ! initial color, represented by integer
    integer, parameter :: fill_color = 1      ! fill color, represented by integer
    integer, parameter :: boundary_color = -1 ! boundary color, represented by integer

  contains

!****************************************************************************

  subroutine glissade_get_masks(nx,          ny,          &
                                parallel,                 &
                                thck,        topg,        &
                                eus,         thklim,      &
                                ice_mask,                 &
                                floating_mask,            &
                                ocean_mask,               &
                                land_mask,                &
                                grounding_line_mask)

    !TODO: Modify glissade_get_masks so that 'parallel' is not needed
    !----------------------------------------------------------------
    ! Compute various masks for the Glissade dycore.
    !
    ! There are different ways to handle masks.
    ! The approach in Glide was to define an integer cell_mask array,
    !  in which each bit encodes information about whether a cell is ice-covered
    !  or ice-free, floating or grounded, etc.
    ! The approach here is to compute a separate 2D integer array for
    !  each kind of mask. This uses more memory but also is more transparent.
    !
    ! The basic masks used for Glissade dynamic calculations are as follows:
    !
    ! (1) ice_mask = 1 where ice is present (thck > thklim), else = 0
    ! (2) floating_mask = 1 if ice is present (thck > thklim) and floating, else = 0
    ! (3) ocean_mask = 1 if the topography is below sea level (topg < eus) and thk <= thklim, else = 0
    ! (4) land_mask = 1 if the topography is at or above sea level (topg >= eus), else = 0
    ! (5) grounding_line_mask = 1 if a cell is adjacent to the grounding line, else = 0
    !
    ! where thck = ice thickness
    !       thklim = threshold thickness for ice to be dynamically active
    !       topg = bed topography
    !       eus = eustatic sea level (= 0 by default)
    !
    ! Notes:
    ! (1) thck, thklim, topg and eus can either have units of length (e.g., meters)
    !     or be dimensionless, as long as they are defined consistently.
    ! (2) ice_mask is always computed; the other masks are optional.
    ! (3) Thermal calculations may have a different threshold, thklim_temp
    !     (where generally thklim_temp < thklim).
    !     This mask can be computed by replacing thklim with thklim_temp in the subroutine call.
    ! (4) For some calculations it may be useful to call this subroutine with thklim = 0
    !     so as to identify all cells with nonzero ice thickness, not just dynamically active cells.
    !----------------------------------------------------------------
    
    !----------------------------------------------------------------
    ! Input-output arguments
    !----------------------------------------------------------------

    integer, intent(in) ::   &
         nx,  ny                ! number of grid cells in each direction

    type(parallel_type), intent(in) :: &
         parallel               ! info for parallel communication

    ! Default dimensions are meters, but this subroutine will work for
    ! any units as long as thck, topg, eus and thklim have the same units.

    real(dp), dimension(nx,ny), intent(in) ::  &
         thck,                & ! ice thickness (m)
         topg                   ! elevation of topography (m)

    real(dp), intent(in) :: &
         eus,                 & ! eustatic sea level (m), = 0. by default
         thklim                 ! minimum ice thickness for active cells (m)

    integer, dimension(nx,ny), intent(out) ::  &
         ice_mask               ! = 1 if thck > thklim, else = 0  

    integer, dimension(nx,ny), intent(out), optional ::  &
         floating_mask,       & ! = 1 if thck > thklim and ice is floating, else = 0
         ocean_mask,          & ! = 1 if topg is below sea level and thk <= thklim, else = 0
         land_mask,           & ! = 1 if topg is at or above sea level, else = 0
         grounding_line_mask    ! = 1 if a cell is adjacent to the grounding line, else = 0

    !----------------------------------------------------------------
    ! Local arguments
    !----------------------------------------------------------------

    integer :: i, j, ii, jj

    integer, dimension(nx,ny) ::  &
         grounded_mask          ! = 1 if ice is present and grounded, else = 0

    !----------------------------------------------------------------
    ! Compute masks in cells
    !----------------------------------------------------------------

    ice_mask(:,:) = 0

    do j = 1, ny
       do i = 1, nx

          if (thck(i,j) > thklim) then
             ice_mask(i,j) = 1
          else
             ice_mask(i,j) = 0
          endif

          if (present(ocean_mask)) then
             if (topg(i,j) < eus .and. ice_mask(i,j) == 0) then
                ocean_mask(i,j) = 1
             else
                ocean_mask(i,j) = 0
             endif
          endif

          if (present(floating_mask)) then
             if (topg(i,j) - eus < (-rhoi/rhoo)*thck(i,j) .and. ice_mask(i,j) == 1) then
                floating_mask(i,j) = 1
             else
                floating_mask(i,j) = 0
             endif
          endif

          if (present(land_mask)) then
             if (topg(i,j) >= eus) then
                land_mask(i,j) = 1
             else
                land_mask(i,j) = 0
             endif
          endif

       enddo  ! i
    enddo  ! j

    ! halo updates
    ! Note: These are not strictly needed because the above loops include halo cells.
    !       However, they are included in case the user calls this subroutine without
    !        first updating thck in halo cells.
    !TODO: Drop the halo updates and require the user to update thck before the call?

    call parallel_halo(ice_mask, parallel)
    if (present(floating_mask)) call parallel_halo(floating_mask, parallel)

    ! Identify grounded cells; this mask is used in some calculations below
    if (present(floating_mask)) then
       where (ice_mask == 1 .and. floating_mask == 0)
          grounded_mask = 1
       elsewhere
          grounded_mask = 0
       endwhere
    endif

    ! Optionally, compute grounding line mask using grounded_mask, floating_mask and ocean_mask
    !TODO: Move grounding_line_mask to a different subroutine?

    if (present(grounding_line_mask)) then

       if (.not.present(floating_mask) .or. .not.present(ocean_mask)) then
          call write_log('Need floating_mask and ocean_mask to compute grounding_line_mask', GM_FATAL)
       endif

       grounding_line_mask(:,:) = 0

       do j = 2, ny-1
          do i = 2, nx-1

             if (grounded_mask(i,j) == 1) then
                ! check whether one or more neighbors is a floating or ocean cell
                do jj = j-1, j+1
                   do ii = i-1, i+1
                      if (floating_mask(ii,jj) == 1 .or. ocean_mask(ii,jj) == 1) then
                         grounding_line_mask(i,j) = 1
                      endif
                   enddo
                enddo
             elseif (floating_mask(i,j) == 1) then
                ! check whether one or more neighbors is a grounded cell
                do jj = j-1, j+1
                   do ii = i-1, i+1
                      if (grounded_mask(ii,jj) == 1) then
                         grounding_line_mask(i,j) = 1
                      endif
                   enddo
                enddo
             endif   ! grounded_mask or floating_mask

          enddo   ! i
       enddo   ! j

       !TODO: Drop this halo call?
       call parallel_halo(grounding_line_mask, parallel)

    endif   ! present(grounding_line_mask)

    ! Note: Halo calls are not included for the ocean and land masks.
    !       Halo values will still be correct, provided that topg is correct in halo cells.
    !       The reason not to include these calls is that for outflow global BCs,
    !        we may not want to set ocean_mask and land_mask = 0 in halo cells (as would be
    !        done automatically for outflow BCs). Instead, we want to compute ocean_mask
    !        and land_mask in halo cells based on topg (which for outflow BCs is extrapolated
    !        to halo cells from adjacent physical cells). 
    !       In particular, setting ocean_mask = 1 in the global halo ensures that calving_front
    !        cells are treated correctly just inside the global halo.

!!    if (present(ocean_mask)) call parallel_halo(ocean_mask)
!!    if (present(land_mask)) call parallel_halo(land_mask)

  end subroutine glissade_get_masks

!****************************************************************************

  subroutine glissade_calving_front_mask(&
       nx,                     ny,                   &
       which_ho_calving_front,                       &
       parallel,                                     &
       thck,                   topg,                 &
       eus,                                          &
       ice_mask,               floating_mask,        &
       ocean_mask,             land_mask,            &
       calving_front_mask,                           &
       dthck_dx_cf,                                  &
       dx,                     dy,                   &
       thck_effective,         thck_effective_min,   &
       partial_cf_mask,        full_mask,            &
       effective_areafrac)

    ! Compute a calving_front mask, effective calving_front thickness, and related fields.
    ! If using the subgrid calving front scheme, then compute the surface elevation gradient
    ! between each CF cell and its thickest upstream neighbor.
    ! If this gradient is less than a prescribed value, the CF cell is considered to be full.
    ! Otherwise, it is marked as a partial CF cell.

    integer, intent(in) ::   &
         nx,  ny,              &  ! number of grid cells in each direction
         which_ho_calving_front   ! subgrid calving front option

    type(parallel_type), intent(in) :: parallel    !> info for parallel communication

    ! Default dimensions are meters, but this subroutine will work for any units
    !  as long as thck, topg, and eus have the same units.

    real(dp), dimension(nx,ny), intent(in) ::  &
         thck,                  & ! ice thickness (m)
         topg                     ! elevation of topography (m)

    real(dp), intent(in) :: &
         eus                      ! eustatic sea level (m), = 0. by default

    integer, dimension(nx,ny), intent(in) ::  &
         ice_mask,              & ! = 1 if thck > thklim, else = 0
         floating_mask,         & ! = 1 if thck > thklim and ice is floating, else = 0
         ocean_mask,            & ! = 1 if topg is below sea level and thk <= thklim, else = 0
         land_mask                ! = 1 if topg is at or above sea level, else = 0

    integer, dimension(nx,ny), intent(out) ::  &
         calving_front_mask       ! = 1 if ice is floating and borders at least one ocean cell, else = 0

    real(dp), intent(in) :: &
         dthck_dx_cf,           & ! assumed max value of |dH/dx| at the CF for full cells
         dx, dy                   ! grid cell size (m)

    real(dp), dimension(nx,ny), intent(out) :: &
         thck_effective,        & ! effective ice thickness (m) for calving
                                  ! Generally, H_eff > H at the CF, with H_eff = H elsewhere
         effective_areafrac       ! effective ice-covered fraction, in range [0,1]
                                  ! 0 < f < 1 for partial calving-front cells

    real(dp), intent(in) :: &
         thck_effective_min       ! minimum effective thickness for CF cells

    integer, dimension(nx,ny), intent(out) :: &
         partial_cf_mask,       & ! = 1 for partially filled CF cells (thck < thck_effective), else = 0
         full_mask                ! = 1 for ice-filled cells that are not partial_cf cells, else = 0

    !----------------------------------------------------------------
    ! Local arguments
    !----------------------------------------------------------------

    integer :: i, j, ii, jj, ig, jg

    real(dp), dimension(nx,ny) :: &
         thck_flotation,           & ! flotation thickness (m)
         capped_thck                 ! min(thck, thck_flotation)

    real(dp) :: &
         max_neighbor_thck,        & ! max thickness (m) of the four edge neighbors
         distance,                 & ! distance between adjacent cell centers
         dthck_dx                    ! dH/dx between adjacent cells near the CF

    integer, dimension(nx,ny) :: &
         interior_mask               ! = 1 for interior cells (grounded or floating) not at the CF

    character(len=100) :: message

    ! Compute a calving front mask, effective calving front thickness, and related fields.
    ! CF cells are defined as floating cells that border ice-free ocean.

    calving_front_mask = 0
    interior_mask = 0

    ! Identify calving front cells (floating cells that border ice-free ocean)
    ! and floating interior cells (floating cells not at the calving front).
    do j = 2, ny-1
       do i = 2, nx-1
          if (floating_mask(i,j) == 1) then
             if (ocean_mask(i-1,j) == 1 .or. ocean_mask(i+1,j) == 1 .or. &
                 ocean_mask(i,j-1) == 1 .or. ocean_mask(i,j+1) == 1) then
                calving_front_mask(i,j) = 1
              ! Note - The following logic adds some CF cells in regions with thin floating ice.
              ! Commmented out for now because it changes CalvingMIP answers.
             elseif (thck(i,j) < thck_effective_min) then
                ! If two adjacent floating cells have very thin ice, we can think of them as sharing a CF
                if ( (floating_mask(i-1,j) == 1 .and. thck(i-1,j) < thck_effective_min) .or. &
                     (floating_mask(i+1,j) == 1 .and. thck(i+1,j) < thck_effective_min) .or. &
                     (floating_mask(i,j-1) == 1 .and. thck(i,j-1) < thck_effective_min) .or. &
                     (floating_mask(i,j+1) == 1 .and. thck(i,j+1) < thck_effective_min) ) then
                   calving_front_mask(i,j) = 1
                endif
             else
                interior_mask(i,j) = 1
             endif
          endif   ! floating
       enddo
    enddo

    call parallel_halo(calving_front_mask, parallel)
    call parallel_halo(interior_mask, parallel)

    if (which_ho_calving_front == HO_CALVING_FRONT_SUBGRID) then

       ! Initialize thck_effective and masks
       thck_effective = thck
       full_mask = 0
       partial_cf_mask = 0

       ! Identify full cells and partial CF cells.
       ! All ice-covered cells not at the CF are full cells.
       ! For CF cells, compute the max thickness of interior neighbors (capped at the flotation thicknes)..
       ! * If the thickness of the CF cell is close to that of the interior cell,
       !   mark the CF cell as a full cell.
       ! * Otherwise, mark the CF cell as a partial CF cell.
       ! If there are no interior neighbors with nonzero (capped) thickness, then compare
       ! to the thickness of CF neighbors.

       thck_flotation = max(-(rhoo/rhoi) * (topg - eus), 0.0d0)
       capped_thck = min(thck, thck_flotation)

       do j = 2, ny-1
          do i = 2, nx-1
             if (ice_mask(i,j) == 1) then
                if (calving_front_mask(i,j) == 1) then
                   max_neighbor_thck = max(&
                        interior_mask(i-1,j)*capped_thck(i-1,j), interior_mask(i+1,j)*capped_thck(i+1,j), &
                        interior_mask(i,j-1)*capped_thck(i,j-1), interior_mask(i,j+1)*capped_thck(i,j+1))
                   if (max_neighbor_thck > 0.0d0) then
                      distance = sqrt(dx*dy)
                      dthck_dx = (max_neighbor_thck - thck(i,j)) / distance
                      ! If the gradient exceeds a critical value, this is a partial CF cell;
                      !  set thck_effective based on the critical gradient.
                      ! If the gradient is at or below the critical valude, this is a full cell with thck_effective = thck.
                      if (dthck_dx > dthck_dx_cf) then
                         partial_cf_mask(i,j) = 1
                         thck_effective(i,j) = max_neighbor_thck - dthck_dx_cf*distance
                      else
                         full_mask(i,j) = 1
                      endif   ! dthck_dx > dthck_dx_cf
                   else   ! no floating interior neighbors
                      ! Mark as a partial cell, and compute thck_effective from a CF neighbor
                      partial_cf_mask(i,j) = 1
                      max_neighbor_thck = max(&
                           calving_front_mask(i-1,j)*thck(i-1,j), calving_front_mask(i+1,j)*thck(i+1,j), &
                           calving_front_mask(i,j-1)*thck(i,j-1), calving_front_mask(i,j+1)*thck(i,j+1))
                      distance = sqrt(dx*dy)
                      dthck_dx = (max_neighbor_thck - thck(i,j)) / distance
                      if (dthck_dx > dthck_dx_cf) then
                         thck_effective(i,j) = max_neighbor_thck - dthck_dx_cf*distance
                      endif
!!                         call parallel_globalindex(i, j, ig, jg, parallel)
!!                         write(iulog,*) 'No interior neighbor:', ig, jg, thck(i,j)
!!                         write(iulog,*) '   New H_eff:', thck_effective(i,j)
                   endif   ! max_neighbor_thck > 0

                else   ! not a CF cell; thck_effective = thck

                   full_mask(i,j) = 1

                endif   ! calving_front_mask
             endif   ! ice_mask
          enddo   ! i
       enddo   ! j

       ! Limit thck_effective at the CF so as not to exceed the flotation thickness
       where (calving_front_mask == 1)
          thck_effective = min(thck_effective, thck_flotation)
       endwhere

       ! Set a lower limit for thck_effective
       where (calving_front_mask == 1)
          thck_effective = max(thck_effective, thck_effective_min)
       endwhere

       call parallel_halo(thck_effective, parallel)
       call parallel_halo(full_mask, parallel)
       call parallel_halo(partial_cf_mask, parallel)

       ! Use the ratio thck/thck_effective to compute effective_areafrac.

       do j = 1, ny
          do i = 1, nx
             if (calving_front_mask(i,j) == 1) then
                effective_areafrac(i,j) = thck(i,j) / thck_effective(i,j)
                effective_areafrac(i,j) = min(effective_areafrac(i,j), 1.0d0)
             elseif (ocean_mask(i,j) == 1) then
                effective_areafrac(i,j) = 0.0d0
             else  ! non-CF ice-covered cells and/or land cells
                effective_areafrac(i,j) = 1.0d0
             endif
          enddo
       enddo

    else   ! no subgrid calving front scheme

       thck_effective = thck
       partial_cf_mask = 0
       full_mask = ice_mask

       where (ice_mask == 1 .or. land_mask == 1)
          effective_areafrac = 1.0d0
       elsewhere
          effective_areafrac = 0.0d0
       endwhere

    endif  ! which_ho_calving_front

  end subroutine glissade_calving_front_mask

!****************************************************************************

  subroutine glissade_marine_cliff_mask(&
       nx,                     ny,                   &
       ice_mask,               floating_mask,        &
       land_mask,              ocean_mask,           &
       marine_cliff_mask)

    ! Compute a mask to identify marine cliff cells.
    ! These are defined as cells with grounded marine ice, adjacent to ice-free ocean cells.

    integer, intent(in) ::   &
         nx,  ny                  ! number of grid cells in each direction

    integer, dimension(nx,ny), intent(in) ::  &
         ice_mask,              & ! = 1 if thck > thklim, else = 0
         floating_mask,         & ! = 1 if thck > thklim and ice is floating, else = 0
         land_mask,             & ! = 1 if topg is at or above sea level, else = 0
         ocean_mask               ! = 1 if topg is below sea level and thk <= thklim, else = 0

    integer, dimension(nx,ny), intent(out) ::  &
         marine_cliff_mask        ! = 1 if ice is grounded, marine-based and borders at least one ocean cell

    !----------------------------------------------------------------
    ! Local arguments
    !----------------------------------------------------------------

    integer :: i, j


    marine_cliff_mask(:,:) = 0

    do j = 2, ny-1
       do i = 2, nx-1
          if (ice_mask(i,j) == 1 .and. land_mask(i,j) == 0 .and. floating_mask(i,j) == 0) then ! grounded marine-based ice
             if (ocean_mask(i-1,j) == 1 .or. ocean_mask(i+1,j) == 1 .or. &
                 ocean_mask(i,j-1) == 1 .or. ocean_mask(i,j+1) == 1) then
                marine_cliff_mask(i,j) = 1
             endif   ! adjacent to ocean
          endif  ! grounded marine-based ice
       enddo  ! i
    enddo   ! j

    ! Note: parallel halo update at the higher level

  end subroutine glissade_marine_cliff_mask

!****************************************************************************

  subroutine glissade_ice_sheet_mask(nx,            ny,     &
                                     parallel,              &
                                     itest, jtest,  rtest,  &
                                     ice_mask,      thck,   &
                                     ice_sheet_mask,        &
                                     ice_cap_mask)

    ! Define masks that identify the ice sheet as distinct from ice caps.
    ! An ice cap is defined as a patch of ice separate from the main ice sheet.

    ! The algorithm is as follows:
    ! (1) Mark all cells with ice (ice_mask = 1) with the initial color.
    !     Mark other cells with the boundary color.
    ! (2) Seed the fill by giving the fill color to some cells that are definitely
    !     part of the ice sheet (based on thck > minthck_ice_sheet).
    ! (3) Recursively fill all cells that are connected to filled cells by a path
    !     that passes through ice-covered cells only.
    ! (4) Repeat the recursion as necessary to spread the fill to adjacent processors.
    ! (5) Once the fill is done, any cells that still have the initial color and
    !     are on land are considered to be ice caps.
    !
    ! Note: The recursive fill applies to edge neighbors, not corner neighbors.
    !        The path back to ice sheet cells must go through edges, not corners.
    !       The ice sheet seeding criterion can be changed by adjusting minthck_ice_sheet.

    integer, intent(in) :: nx, ny                  !> horizontal grid dimensions

    type(parallel_type), intent(in) :: parallel    !> info for parallel communication

    integer, intent(in) :: itest, jtest, rtest     !> coordinates of diagnostic point

    integer, dimension(nx,ny), intent(in) ::  &
         ice_mask               !> = 1 if ice is present (thck > thklim)

    real(dp), dimension(nx,ny), intent(in) ::  &
         thck                   !> ice thickness (m)

    integer, dimension(nx,ny), intent(out) ::  &
         ice_sheet_mask         !> = 1 for ice sheet cells

    integer, dimension(nx,ny), intent(out) ::  &
         ice_cap_mask           !> = 1 for ice cap cells, separately from the main ice sheet

    real(dp), parameter :: &
         minthck_ice_sheet = 2000.d0  !> thickness threshold (m) for initializing ice sheet cells

    ! local variables

    integer :: i, j, iter

    integer :: &
         max_iter,             & ! max(ewtasks, nstasks)
         local_count,          & ! local counter for filled values
         global_count,         & ! global counter for filled values
         global_count_save       ! globalcounter for filled values from previous iteration

    integer, dimension(nx,ny) ::  &
         color                  !> color variable for the fill

    logical, parameter :: verbose_ice_sheet_mask = .false.

    ! initialize
    ! Note: Ice-covered cells receive the initial color, and ice-free cells receive the boundary color.

    do j = 1, ny
       do i = 1, nx
          if (ice_mask(i,j) == 1) then
             color(i,j) = initial_color
          else
             color(i,j) = boundary_color
          endif
       enddo
    enddo

    ! Loop through cells, identifying cells that are definitely part of the ice sheet
    !  based on a threshold ice thickness. 
    ! Fill these cells and then recursively fill ice-covered neighbors.
    ! We may have to do this several times to incorporate connections between neighboring processors.

    max_iter = max(parallel%ewtasks, parallel%nstasks)
    global_count_save = 0

    do iter = 1, max_iter

       if (iter == 1) then   ! identify ice sheet cells that can seed the fill

          do j = 1, ny
             do i = 1, nx
                if (color(i,j) == initial_color .and. thck(i,j) >= minthck_ice_sheet) then
                   ! assign the fill color to this cell, and recursively fill ice-covered neighbors
                   call glissade_fill(nx,    ny,    &
                                      i,     j,     &
                                      color, ice_mask)
                endif
             enddo
          enddo

       else  ! iter > 1

          ! Check for halo cells that were just filled on neighbor processors
          ! Note: In order for a halo cell to seed the fill on this processor, it must already have the fill color.

          call parallel_halo(color, parallel)

          ! west halo layer
          i = nhalo
          do j = 1, ny
             if (color(i,j) == fill_color) then
                call glissade_fill(nx,    ny,    &
                                   i+1,   j,     &
                                   color, ice_mask)
             endif
          enddo

          ! east halo layers
          i = nx - nhalo + 1
          do j = 1, ny
             if (color(i,j) == fill_color) then
                call glissade_fill(nx,    ny,    &
                                   i-1,   j,     &
                                   color, ice_mask)
             endif
          enddo

          ! south halo layer
          j = nhalo
          do i = nhalo+1, nx-nhalo  ! already checked halo corners above
             if (color(i,j) == fill_color) then
                call glissade_fill(nx,    ny,    &
                                   i,     j+1,   &
                                   color, ice_mask)
             endif
          enddo

          ! north halo layer
          j = ny-nhalo+1
          do i = nhalo+1, nx-nhalo  ! already checked halo corners above
             if (color(i,j) == fill_color) then
                call glissade_fill(nx,    ny,    &
                                   i,     j-1,   &
                                   color, ice_mask)
             endif
          enddo

       endif  ! iter = 1

       ! Count the number of filled cells.  If converged, then exit the loop.

       local_count = 0
       do j = nhalo+1, ny-nhalo
          do i = nhalo+1, nx-nhalo
             if (color(i,j) == fill_color) local_count = local_count + 1
          enddo
       enddo

       global_count = parallel_reduce_sum(local_count)

       if (global_count == global_count_save) then
          if (verbose_ice_sheet_mask .and. main_task) &
               write(iulog,*) 'Fill converged: iter, global_count =', iter, global_count
          exit
       else
          if (verbose_ice_sheet_mask .and. main_task) &
               write(iulog,*) 'Convergence check: iter, global_count =', iter, global_count
          global_count_save = global_count
       endif

    enddo  ! max_iter

    ! Any cells with the fill color are considered to be part of the land-based ice sheet.
    ! Any cells with the initial color are deemed to be ice caps.
 
    ice_sheet_mask(:,:) = 0.0d0
    ice_cap_mask(:,:) = 0.0d0

    do j = 1, ny
       do i = 1, nx
          if (color(i,j) == initial_color) then
             ice_cap_mask(i,j) = 1
          elseif (color(i,j) == fill_color) then
             ice_sheet_mask(i,j) = 1
          endif
       enddo
    enddo

    call parallel_halo(ice_sheet_mask, parallel)
    call parallel_halo(ice_cap_mask, parallel)

  end subroutine glissade_ice_sheet_mask

!****************************************************************************

  subroutine glissade_ocean_connection_mask(nx,            ny,          &
                                            parallel,                   &
                                            itest, jtest,  rtest,       &
                                            thck,          input_mask,  &
                                            ocean_mask,                 &
                                            ocean_connection_mask)

    ! Create a masks consisting of cells with input_mask = 1, which either
    ! are adjacent to the ocean, or are connected to the ocean through other cells
    ! with input_mask = 1.

    ! The algorithm is as follows:
    ! (1) Assign the initial color to cells with input_mask = 1.
    !     Mark other cells with the boundary color.
    ! (2) Seed the fill by giving the fill color to cells that have input_mask = 1
    !     and are adjacent to ocean cells (ocean_mask = 1).
    ! (3) Recursively fill all cells that have input_mask = 1 and are connected
    !     to the ocean by a path that passes through other cells with input_mask = 1.
    ! (4) Repeat the recursion as necessary to spread the fill to adjacent processors.
    ! (5) Once the fill is done, all cells with the fill color are assigned
    !     to ocean_connection_mask.
    !
    ! Note: The logic is general enough that we could replace ocean_mask and
    !       ocean_connection_mask with generic input and output masks.

    integer, intent(in) :: nx, ny                  !> horizontal grid dimensions

    type(parallel_type), intent(in) :: parallel    !> info for parallel communication

    integer, intent(in) :: itest, jtest, rtest     !> coordinates of diagnostic point

    real(dp), dimension(nx,ny), intent(in) ::  &
         thck                   !> ice thickness (m)

    integer, dimension(nx,ny), intent(in) ::  &
         input_mask,          & !> = 1 for cells that meet some criterion specified elsewhere
         ocean_mask             !> = 1 for ice-free cells with topg below sea level

    integer, dimension(nx,ny), intent(out) ::  &
         ocean_connection_mask  !> = 1 for cells with input_mask = 1, connected to the ocean
                                !>   through other such cells

    ! local variables

    integer :: i, j, iter

    integer :: &
         max_iter,             & ! max(ewtasks, nstasks)
         local_count,          & ! local counter for filled values
         global_count,         & ! global counter for filled values
         global_count_save       ! globalcounter for filled values from previous iteration

    integer, dimension(nx,ny) ::  &
         color                   ! color variable for the fill

    logical, parameter :: verbose_ocean_connection_mask = .false.

    ! initialize
    ! Note: Cells with input_mask = 1 receive the initial color, and other cells receive the boundary color.

    where (input_mask == 1)
       color = initial_color
    elsewhere
       color = boundary_color
    endwhere

    ! Identifying cells that have the initial color and are adjacent to the ocean.
    ! Fill these cells and then recursively fill neighbors that have the initial color.
    ! We may have to do this several times to incorporate connections between neighboring processors.

    max_iter = max(parallel%ewtasks, parallel%nstasks)
    global_count_save = 0

    do iter = 1, max_iter

       if (iter == 1) then   ! identify cells adjacent to the ocean, which can seed the fill

          do j = 2, ny-1
             do i = 2, nx-1
                if (color(i,j) == initial_color .and.  &
                     (ocean_mask(i-1,j) == 1 .or. ocean_mask(i+1,j) == 1 .or. &
                      ocean_mask(i,j-1) == 1 .or. ocean_mask(i,j+1) == 1) ) then
                   ! assign the fill color to this cell, and recursively fill neighbors with input_mask = 1
                   call glissade_fill(nx,    ny,    &
                                      i,     j,     &
                                      color, input_mask)
                endif
             enddo
          enddo

       else  ! iter > 1

          ! Check for halo cells that were just filled on neighbor processors
          ! Note: In order for a halo cell to seed the fill on this processor, it must already have the fill color.

          call parallel_halo(color, parallel)

          ! west halo layer
          i = nhalo
          do j = 1, ny
             if (color(i,j) == fill_color) then
                call glissade_fill(nx,    ny,    &
                                   i+1,   j,     &
                                   color, input_mask)
             endif
          enddo

          ! east halo layers
          i = nx - nhalo + 1
          do j = 1, ny
             if (color(i,j) == fill_color) then
                call glissade_fill(nx,    ny,    &
                                   i-1,   j,     &
                                   color, input_mask)
             endif
          enddo

          ! south halo layer
          j = nhalo
          do i = nhalo+1, nx-nhalo  ! already checked halo corners above
             if (color(i,j) == fill_color) then
                call glissade_fill(nx,    ny,    &
                                   i,     j+1,   &
                                   color, input_mask)
             endif
          enddo

          ! north halo layer
          j = ny-nhalo+1
          do i = nhalo+1, nx-nhalo  ! already checked halo corners above
             if (color(i,j) == fill_color) then
                call glissade_fill(nx,    ny,    &
                                   i,     j-1,   &
                                   color, input_mask)
             endif
          enddo

       endif  ! iter = 1

       ! Count the number of filled cells.  If converged, then exit the loop.

       local_count = 0
       do j = nhalo+1, ny-nhalo
          do i = nhalo+1, nx-nhalo
             if (color(i,j) == fill_color) local_count = local_count + 1
          enddo
       enddo

       global_count = parallel_reduce_sum(local_count)

       if (global_count == global_count_save) then
          if (verbose_ocean_connection_mask .and. main_task) &
               write(iulog,*) 'ocean_connection_mask, fill converged: iter, global_count =', iter, global_count
          exit
       else
          if (verbose_ocean_connection_mask .and. main_task) &
               write(iulog,*) 'ocean_connection_mask, convergence check: iter, global_count =', iter, global_count
          global_count_save = global_count
       endif

    enddo  ! max_iter

    ! Any cells with the fill color are deemed to be ocean-connected.
    ocean_connection_mask(:,:) = 0

    where (color == fill_color)
       ocean_connection_mask = 1
    elsewhere
       ocean_connection_mask = 0
    endwhere

    !TODO: Move this update to a higher level?
    call parallel_halo(ocean_connection_mask, parallel)

  end subroutine glissade_ocean_connection_mask

!****************************************************************************

  subroutine glissade_marine_connection_mask(nx,           ny,             &
                                             parallel,                     &
                                             itest, jtest, rtest,          &
                                             thck,         topg,           &
                                             eus,          thklim,         &
                                             marine_connection_mask)

    ! Identify cells that have a marine path to the ocean.
    ! The path can include grounded marine-based ice.
    ! The ocean is identified as ice-free cells with bed elevation below sea level,
    !  or optionally with bed elevation below some threshold value.

    ! subroutine arguments

    integer, intent(in) :: nx, ny                  !> horizontal grid dimensions

    type(parallel_type), intent(in) :: parallel    !> info for parallel communication

    integer, intent(in) :: itest, jtest, rtest     !> coordinates of diagnostic point

    ! Note: Input thck and topg need to be correct in halo cells
    real(dp), dimension(nx,ny), intent(in) ::  &
         thck,                   & !> ice thickness (m)
         topg                      !> elevation of topography (m)

    real(dp), intent(in) :: &
         eus,                    & !> eustatic sea level (m), = 0. by default
         thklim                    !> minimum ice thickness for active cells (m)

    integer, dimension(nx,ny), intent(out) ::  &
         marine_connection_mask    !> = 1 for ocean cells and cells connected to the ocean through marine-based ice

    ! local variables

    integer, dimension(nx,ny) ::  &
         ocean_mask,          &  !> = 1 where topg - eus is below sea level and ice is absent, else = 0
         ocean_mask_temp,     &  !> temporary version of ocean_mask
         marine_mask,         &  !> marine-based cells; topg - eus < 0
         color                   ! integer 'color' mask to mark filled cells

    integer :: i, j, iter

    integer :: &
         max_iter,             & ! max(ewtasks, nstasks)
         local_count,          & ! local counter for filled values
         global_count,         & ! global counter for filled values
         global_count_save       ! globalcounter for filled values from previous iteration

    !Note: could make this a config parameter if different values are desired for different grids
    real(dp), parameter :: &
         ocean_topg_threshold = -500.d0   !> ocean threshold elevation (m) to seed the fill; negative below sea level

    logical, parameter :: verbose_marine_connection = .false.

    ! Compute ocean_mask, which is used to seed the fill.
    ! If ocean_topg_threshold was passed in, then ocean_mask includes only cells
    !  with topg - eus < ocean_topg_threshold.

    where (thck <= thklim .and. topg - eus < ocean_topg_threshold)
       ocean_mask = 1
    elsewhere
       ocean_mask = 0
    endwhere

    ! Occasionally, e.g. in Greenland fjords, a cell could be marked as ice-free ocean even though it is landlocked.
    ! To make this less likely, set ocean_mask = 0 for any cells with non-ocean neighbors.
    ! This logic could be iterated if needed.

    ocean_mask_temp = ocean_mask
    do j = 2, ny-1
       do i = 2, nx-1
          if (ocean_mask_temp(i-1,j) == 0 .or. ocean_mask_temp(i+1,j) == 0 .or.  &
              ocean_mask_temp(i,j-1) == 0 .or. ocean_mask_temp(i,j+1) == 0) then
             ocean_mask(i,j) = 0
          endif
       enddo
    enddo

    call parallel_halo(ocean_mask, parallel)

    ! initialize
    ! Compute a marine mask; = 1 for all cells with topg - eus < 0.
    ! Marine-based cells receive the initial color; land cells receive the boundary color.
    ! Seed the fill with ocean cells.

    where (ocean_mask == 1)
       marine_mask = 1
       color = fill_color
    elsewhere (topg - eus < 0.0d0)  ! marine-based cells
       marine_mask = 1
       color = initial_color
    elsewhere   ! land cells
       marine_mask = 0
       color = boundary_color
    endwhere

    if (verbose_marine_connection) then
       call point_diag(marine_mask, 'marine_mask', itest, jtest, rtest, 7, 7)
    endif

    ! Loop through cells, identifying marine-based cells that border the ocean.
    ! Fill each such cell, and then recursively fill marine-based neighbor cells.
    ! We may have to do this several times to incorporate connections between neighboring processors.
    ! The result is a mask that all marine-based cells connected to the ocean are filled.

    max_iter = max(parallel%ewtasks, parallel%nstasks)
    global_count_save = 0

    do iter = 1, max_iter

       if (iter == 1) then   ! identify marine-based cells adjacent to ocean cells, which can seed the fill

          do j = 2, ny-1
             do i = 2, nx-1
                if (marine_mask(i,j) == 1) then
                   if (ocean_mask(i-1,j) == 1 .or. ocean_mask(i+1,j) == 1 .or.   &
                       ocean_mask(i,j-1) == 1 .or. ocean_mask(i,j+1) == 1) then

                      if (color(i,j) /= boundary_color .and. color(i,j) /= fill_color) then

                         ! assign the fill color to this cell, and recursively fill marine-based neighbor cells
                         call glissade_fill(nx,    ny,    &
                                            i,     j,     &
                                            color, marine_mask)
                      endif
                   endif  ! adjacent to ocean
                endif  ! marine-based
             enddo  ! i
          enddo  ! j

       else  ! iter > 1

          ! Check for halo cells that were just filled on neighbor processors
          ! Note: In order for a halo cell to seed the fill on this processor, it must not only have the fill color,
          !       but also must have marine_mask = 1.

          call parallel_halo(color, parallel)

          ! west halo layer
          i = nhalo
          do j = 1, ny
             if (color(i,j) == fill_color .and. marine_mask(i,j) == 1) then
                call glissade_fill(nx,    ny,    &
                                   i+1,   j,     &
                                   color, marine_mask)
             endif
          enddo

          ! east halo layers
          i = nx - nhalo + 1
          do j = 1, ny
             if (color(i,j) == fill_color .and. marine_mask(i,j) == 1) then
                call glissade_fill(nx,    ny,    &
                                   i-1,   j,     &
                                   color, marine_mask)
             endif
          enddo

          ! south halo layer
          j = nhalo
          do i = nhalo+1, nx-nhalo  ! already checked halo corners above
             if (color(i,j) == fill_color .and. marine_mask(i,j) == 1) then
                call glissade_fill(nx,    ny,    &
                                   i,     j+1,   &
                                   color, marine_mask)
             endif
          enddo

          ! north halo layer
          j = ny-nhalo+1
          do i = nhalo+1, nx-nhalo  ! already checked halo corners above
             if (color(i,j) == fill_color .and. marine_mask(i,j) == 1) then
                call glissade_fill(nx,    ny,    &
                                   i,     j-1,   &
                                   color, marine_mask)
             endif
          enddo

       endif  ! iter = 1

       ! Count the number of filled cells.  If converged, then exit the loop.

       local_count = 0
       do j = nhalo+1, ny-nhalo
          do i = nhalo+1, nx-nhalo
             if (color(i,j) == fill_color) local_count = local_count + 1
          enddo
       enddo

       global_count = parallel_reduce_sum(local_count)

       if (global_count == global_count_save) then
          if (verbose_marine_connection .and. main_task) &
               write(iulog,*) 'Fill converged: iter, global_count =', iter, global_count
          exit
       else
          if (verbose_marine_connection .and. main_task) &
               write(iulog,*) 'Convergence check: iter, global_count =', iter, global_count
          global_count_save = global_count
       endif

    enddo  ! max_iter

    call parallel_halo(color, parallel)

    ! Set the marine connection mask.  This includes:
    ! (1) cells that are already ocean
    ! (2) cells with the fill color, meaning they are marine-based cells connected to the ocean

    marine_connection_mask(:,:) = 0

    do j = 1, ny
       do i = 1, nx
          if (ocean_mask(i,j) == 1 .or. color(i,j) == fill_color) then
             marine_connection_mask(i,j) = 1
          endif
       enddo
    enddo

    call parallel_halo(marine_connection_mask, parallel)

    if (verbose_marine_connection) then
       call point_diag(color, 'color', itest, jtest, rtest, 7, 7)
       call point_diag(marine_connection_mask, 'marine_connection_mask', itest, jtest, rtest, 7, 7)
    endif

  end subroutine glissade_marine_connection_mask

!****************************************************************************

  subroutine glissade_lake_mask(nx,           ny,             &
                                parallel,                     &
                                itest, jtest, rtest,          &
                                floating_mask,                &
                                ocean_mask,                   &
                                lake_mask,                    &
                                ocean_connection_mask)

  ! TODO - Rewrite this subroutine with marine_connection_mask as an input.
  !        Lake cells are just floating cells without a marine connection.

  ! Identify interior lake cells: cells that are floating but are not connected
  !  to the ocean along a path through other floating cells.
  ! Optionally, identify cells with an ocean connection: either ice-free ocean,
  !  or floating and connected through other floating cells to the ocean.
  ! Note: The path to the ocean must pass through edge neighbors, not corner neighbors.

    integer, intent(in) :: nx, ny                  !> horizontal grid dimensions

    type(parallel_type), intent(in) :: parallel    !> info for parallel communication

    integer, intent(in) :: itest, jtest, rtest     !> coordinates of diagnostic point

    ! Note: The calling subroutine decides how to define floating_mask.
    !       E.g., it could be defined based on f_ground_cell with which_ho_ground = 2.
    ! Each input mask needs to be correct in halo cells

    integer, dimension(nx,ny), intent(in) ::  &
         floating_mask,          & !> = 1 where ice is present (thck > thklim) and floating, else = 0
         ocean_mask                !> = 1 where topg - eus is below sea level and ice is absent, else = 0

    integer, dimension(nx,ny), intent(out) ::  &
         lake_mask                 !> = 1 for floating cells disconnected from the ocean, else = 0

    integer, dimension(nx,ny), intent(out), optional ::  &
         ocean_connection_mask     !> = 1 for ocean cells, and cells connected to the ocean through floating ice

    ! local variables

    integer, dimension(nx,ny) ::  &
         color,                  & ! integer 'color' mask to mark filled cells
         border_mask               ! = 1 for grounded marine ice adjacent to ocean-connected cells

    integer :: i, j, iter

    integer :: &
         max_iter,             & ! max(ewtasks, nstasks)
         local_count,          & ! local counter for filled values
         global_count,         & ! global counter for filled values
         global_count_save       ! globalcounter for filled values from previous iteration

    logical, parameter :: verbose_lake = .false.

    integer :: ig, jg

    if (verbose_lake) then
       call point_diag(floating_mask, 'floating_mask', itest, jtest, rtest, 7, 7)
    endif

    ! initialize
    ! Floating cells receive the initial color;
    !  grounded cells and ice-free cells receive the boundary color.

    do j = 1, ny
       do i = 1, nx
          if (floating_mask(i,j) == 1) then
             color(i,j) = initial_color
          else    ! grounded or ice-free
             color(i,j) = boundary_color
          endif
       enddo
    enddo

    ! Loop through cells, identifying floating cells that border the ocean.
    ! Fill each such floating cell, and then recursively fill floating neighbor cells.
    ! We may have to do this several times to incorporate connections between neighboring processors.
    ! The result is a mask that identifies (with the fill color) all floating cells connected to the ocean.

    max_iter = max(parallel%ewtasks, parallel%nstasks)
    global_count_save = 0

    do iter = 1, max_iter

       if (iter == 1) then   ! identify floating cells adjacent to ocean cells, which can seed the fill

          do j = 2, ny-1
             do i = 2, nx-1
                if (floating_mask(i,j) == 1) then
                   if (ocean_mask(i-1,j) == 1 .or. ocean_mask(i+1,j) == 1 .or.   &
                       ocean_mask(i,j-1) == 1 .or. ocean_mask(i,j+1) == 1) then

                      if (color(i,j) /= boundary_color .and. color(i,j) /= fill_color) then

                         ! assign the fill color to this cell, and recursively fill floating neighbor cells
                         call glissade_fill(nx,    ny,    &
                                            i,     j,     &
                                            color, floating_mask)
                      endif
                   endif  ! adjacent to ocean
                endif  ! floating
             enddo  ! i
          enddo  ! j

       else  ! count > 1

          ! Check for halo cells that were just filled on neighbor processors
          ! Note: In order for a halo cell to seed the fill on this processor, it must not only have the fill color,
          !       but also must have floating_mask = 1.

          call parallel_halo(color, parallel)

          ! west halo layer
          i = nhalo
          do j = 1, ny
             if (color(i,j) == fill_color .and. floating_mask(i,j) == 1) then
                call glissade_fill(nx,    ny,    &
                                   i+1,   j,     &
                                   color, floating_mask)
             endif
          enddo

          ! east halo layers
          i = nx - nhalo + 1
          do j = 1, ny
             if (color(i,j) == fill_color .and. floating_mask(i,j) == 1) then
                call glissade_fill(nx,    ny,    &
                                   i-1,   j,     &
                                   color, floating_mask)
             endif
          enddo

          ! south halo layer
          j = nhalo
          do i = nhalo+1, nx-nhalo  ! already checked halo corners above
             if (color(i,j) == fill_color .and. floating_mask(i,j) == 1) then
                call glissade_fill(nx,    ny,    &
                                   i,     j+1,   &
                                   color, floating_mask)
             endif
          enddo

          ! north halo layer
          j = ny-nhalo+1
          do i = nhalo+1, nx-nhalo  ! already checked halo corners above
             if (color(i,j) == fill_color .and. floating_mask(i,j) == 1) then
                call glissade_fill(nx,    ny,    &
                                   i,     j-1,   &
                                   color, floating_mask)
             endif
          enddo

       endif  ! iter = 1

       ! Count the number of filled cells.  If converged, then exit the loop.

       local_count = 0
       do j = nhalo+1, ny-nhalo
          do i = nhalo+1, nx-nhalo
             if (color(i,j) == fill_color) local_count = local_count + 1
          enddo
       enddo

       global_count = parallel_reduce_sum(local_count)

       if (global_count == global_count_save) then
          if (verbose_lake .and. main_task) &
               write(iulog,*) 'Fill converged: iter, global_count =', iter, global_count
          exit
       else
          if (verbose_lake .and. main_task) &
               write(iulog,*) 'Convergence check: iter, global_count =', iter, global_count
          global_count_save = global_count
       endif

    enddo  ! max_iter

    call parallel_halo(color, parallel)

    ! Identify lake cells: floating cells that still have the initial color.

    lake_mask(:,:) = 0

    do j = 1, ny
       do i = 1, nx
          if (color(i,j) == initial_color .and. floating_mask(i,j) == 1) then
             lake_mask(i,j) = 1

             if (verbose_lake .and. this_rank == rtest) then
                call parallel_globalindex(i, j, ig, jg, parallel)
                write(iulog,*) 'Lake cell: task, i, j, ig, jg =', this_rank, i, j, ig, jg
             endif

          endif
       enddo
    enddo

    call parallel_halo(lake_mask, parallel)

    if (verbose_lake) then
       call point_diag(lake_mask, 'lake_mask', itest, jtest, rtest, 7, 7)
    endif

    if (present(ocean_connection_mask)) then

       ! Identify cells connected to the ocean.  This includes:
       ! (1) cells that are already ocean
       ! (2) floating cells with the fill color, meaning they are connected to the ocean via other floating cells

       ocean_connection_mask(:,:) = 0

       do j = 1, ny
          do i = 1, nx
             if (ocean_mask(i,j) == 1 .or. color(i,j) == fill_color) then
                ocean_connection_mask(i,j) = 1
             endif
          enddo
       enddo

       !TODO: Is this halo call needed?
       call parallel_halo(ocean_connection_mask, parallel)

       if (verbose_lake) then
          call point_diag(color, 'color', itest, jtest, rtest, 7, 7)
          call point_diag(ocean_connection_mask, 'ocean_connection_mask', itest, jtest, rtest, 7, 7)
       endif

    endif  ! present(ocean_connection_mask)

  end subroutine glissade_lake_mask

!****************************************************************************

  subroutine glissade_extend_mask(nx,       ny,  &
                                  input_mask,      &
                                  extended_mask)

    ! Compute a mask that includes
    ! (1) cells with input_mask = 1
    ! (2) cells that are adjacent to cells with input_mask = 1
    ! For now, assume that both edge and diagonal neighbors are adjacent.
    ! If needed, could add an option to choose only edge neighbors.

    integer, intent(in) ::   &
         nx,  ny                !> number of grid cells in each direction

    integer, dimension(nx,ny), intent(in) :: &
         input_mask             !> input mask to be extended

    integer, dimension(nx,ny), intent(out) :: &
         extended_mask          !> input mask extended by adding neighbor cells

    ! local variables

    integer :: i, j

    extended_mask(:,:) = 0

    do j = 2, ny-1
       do i = 2, nx-1
          if (input_mask(i,j) == 1) then
             extended_mask(i,j) = 1
          elseif (input_mask(i,j-1)   == 1 .or. input_mask(i,j+1)   == 1   .or.  &
                  input_mask(i-1,j+1) == 1 .or. input_mask(i+1,j+1) == 1 .or.  &
                  input_mask(i-1,j)   == 1 .or. input_mask(i+1,j)   == 1 .or.  &
                  input_mask(i-1,j-1) == 1 .or. input_mask(i+1,j-1) == 1) then
             extended_mask(i,j) = 1
          endif
       enddo
    enddo

    !Note: Halo update moved to higher level
!    call parallel_halo(extended_mask)

  end subroutine glissade_extend_mask

!****************************************************************************

  recursive subroutine glissade_fill(nx,  ny,         &
                                     i,   j,          &
                                     color,           &
                                     fill_mask)

    ! Given a cell (i,j), determine whether it should be given the fill color
    !  and recursively fill neighbor cells.
    ! This subroutine differs from the subroutine below in that cell (i,j) is filled and
    !  the subroutine is called recursively only if fill_mask = 1.
    ! In the subroutine below, cell (i,j) can be filled when fill_mask = 0,
    !  but the subroutine is called recursively only if fill_mask = 1.

    integer, intent(in) :: nx, ny                       !> domain size
    integer, intent(in) :: i, j                         !> horizontal indices of current cell

    integer, dimension(nx,ny), intent(inout) :: &
         color                                          !> color (initial, fill or boundary)

    integer, dimension(nx,ny), intent(in) :: &
         fill_mask                                      !> = 1 if the cell satisfies the fill criterion

    if (color(i,j) /= fill_color .and. color(i,j) /= boundary_color .and. fill_mask(i,j) == 1) then

       ! assign the fill color to this cell
       color(i,j) = fill_color

       ! recursively call this subroutine for each neighbor to see if it should be filled
       !TODO - May want to rewrite this to avoid recursion, which can crash the code when
       !       the recursion stack is very large on fine grids.
       if (i > 1)  call glissade_fill(nx,    ny,  &
                                      i-1,   j,   &
                                      color, fill_mask)

       if (i < nx) call glissade_fill(nx,    ny,  &
                                      i+1,   j,   &
                                      color, fill_mask)

       if (j > 1)  call glissade_fill(nx,    ny, &
                                      i,     j-1, &
                                      color, fill_mask)

       if (j < ny) call glissade_fill(nx,    ny,  &
                                      i,     j+1, &
                                      color, fill_mask)

    endif   ! not fill color or boundary color

  end subroutine glissade_fill

!****************************************************************************

  recursive subroutine glissade_fill_with_buffer(&
       nx,  ny,         &
       i,   j,          &
       color,           &
       fill_mask)

    ! Given a domain with an initial color, a boundary color and a fill color,
    !  recursively assign the fill color to all cells that are connected to cells
    !  with the fill color.  The connection must pass through cell edges (not vertices).
    ! Note: "with_buffer" refers to the idea that we fill not only cells with fill_mask = 1,
    !       but also cells with fill_mask = 0, provided the cell does not already have
    !       the fill color or boundary color.  But only cells with fill_mask = 1 result
    !       in additional filling.
    !       This logic is used if we want to fill not only active cells, but also
    !        a buffer layer of inactive cells.

    integer, intent(in) :: nx, ny                       !> domain size
    integer, intent(in) :: i, j                         !> horizontal indices of current cell

    integer, dimension(nx,ny), intent(inout) :: &
         color                                          !> color (initial, fill or boundary)

    integer, dimension(nx,ny), intent(in) :: &
         fill_mask                                      !> = 1 if the cell satisfies the fill criterion

    if (color(i,j) /= fill_color .and. color(i,j) /= boundary_color) then

       ! assign the fill color to this cell
       color(i,j) = fill_color

       ! If fill_mask = 1, then fill this cell but do not call the subroutine recursively
       if (fill_mask(i,j) == 0) return   ! skip the recursion

       ! recursively call this subroutine for each neighbor to see if it should be filled
       !TODO - May want to rewrite this to avoid recursion, which can crash the code when
       !       the recursion stack is very large on fine grids.
       if (i > 1)  call glissade_fill_with_buffer(nx,    ny,  &
                                                  i-1,   j,   &
                                                  color, fill_mask)

       if (i < nx) call glissade_fill_with_buffer(nx,    ny,  &
                                                  i+1,   j,   &
                                                  color, fill_mask)

       if (j > 1)  call glissade_fill_with_buffer(nx,    ny, &
                                                  i,     j-1, &
                                                  color, fill_mask)

       if (j < ny) call glissade_fill_with_buffer(nx,    ny,  &
                                                  i,     j+1, &
                                                  color, fill_mask)

    endif   ! not fill color or boundary color

  end subroutine glissade_fill_with_buffer

!****************************************************************************

  end module glissade_masks

!****************************************************************************

