!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!
!   glissade_utils.F90 - part of the Community Ice Sheet Model (CISM)
!
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!
!   Copyright (C) 2005-2018
!   CISM contributors - see AUTHORS file for list of contributors
!
!   This file is part of CISM.
!
!   CISM is free software: you can redistribute it and/or modify it
!   under the terms of the Lesser GNU General Public License as published
!   by the Free Software Foundation, either version 3 of the License, or
!   (at your option) any later version.
!
!   CISM is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   Lesser GNU General Public License for more details.
!
!   You should have received a copy of the Lesser GNU General Public License
!   along with CISM. If not, see <http://www.gnu.org/licenses/>.
!
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!
! This module holds some utility subroutines for the Glissade dynamical core
!
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

module glissade_utils

  use glimmer_global, only: dp
  use glimmer_paramets, only: iulog
  use glimmer_log
  use glide_types
  use cism_parallel, only: this_rank, main_task

  implicit none

  private
  public :: glissade_adjust_thickness, glissade_smooth_usrf, &
       glissade_smooth_topography, glissade_adjust_topography, &
       glissade_basin_sum, glissade_basin_average, &
       glissade_usrf_to_thck, glissade_thck_to_usrf, &
       glissade_edge_fluxes, glissade_input_fluxes, &
       glissade_rms_error

contains

!****************************************************************************

  subroutine glissade_adjust_thickness(model)

    ! Optionally, check for spurious surface depressions that could arise in the following case:
    ! (1) usrf, thck, and topg have all been read in.  (Recall that usrf is an optional input.)
    ! (2) There are interior lakes: regions disconnected from the ocean, where (usrf - thck) > topg.
    ! (3) The ice in these interior lake regions is too thick to float.
    ! In this case, the default behavior is to reset usrf = topg + thck, possibly leading to
    !  steep surface depressions and unstable flow.
    ! The alternative is to set thck = usrf - topg in grounded regions, maintaining the observed usrf.
    !
    !TODO: In this and the next two subroutines, we could pass in thck, topg, etc. instead of the model derived type.

    use cism_parallel, only: parallel_reduce_max

    !----------------------------------------------------------------
    ! Input-output arguments
    !----------------------------------------------------------------

    type(glide_global_type), intent(inout) :: model   ! derived type holding ice-sheet info

    ! local variables

    real(dp) :: usrf_max
    real(dp) :: topg
    real(dp) :: thck_flot

    integer :: i, j
    integer :: nx, ny
    integer :: itest, jtest, rtest
    ! The following variables give the boundaries of a box going from itest-3 to itest+3,
    ! and jtest-3 to jtest+3, but limited to stay within the range of the local points
    ! owned by rdiag_local
    integer :: itest_m3, itest_p3, jtest_m3, jtest_p3

    logical, parameter :: verbose_adjust_thickness = .true.

    ! Copy some model variables to local variables

    nx = model%general%ewn
    ny = model%general%nsn

    rtest = -999
    itest = 1
    jtest = 1
    if (this_rank == model%numerics%rdiag_local) then
       rtest = model%numerics%rdiag_local
       itest = model%numerics%idiag_local
       jtest = model%numerics%jdiag_local
       itest_m3 = max(itest-3, 1)
       itest_p3 = min(itest+3, nx)
       jtest_m3 = max(jtest-3, 1)
       jtest_p3 = min(jtest+3, ny)
    endif

    ! Make sure ursf was read in with nonzero values.
    ! Otherwise, we cannot use usrf to adjust the ice thickness.

    usrf_max = maxval(model%geometry%usrf)
    usrf_max = parallel_reduce_max(usrf_max)

    if (usrf_max > tiny(0.0d0)) then

       do j = 1, ny
          do i = 1, nx
             topg = model%geometry%topg(i,j) - model%climate%eus  ! shorthand for relative bed topography
             if (model%geometry%usrf(i,j) - model%geometry%thck(i,j) > topg) then
                thck_flot = -(rhoo/rhoi) * topg
                if (model%geometry%thck(i,j) >= thck_flot) then  ! grounded
                   ! increase thck to remove the sub-ice cavity
                   model%geometry%thck(i,j) = model%geometry%usrf(i,j) - topg
                else   ! floating
                   ! do nothing; keep the existing thickness
                endif
             elseif (model%geometry%usrf(i,j) - model%geometry%thck(i,j) < topg) then
                ! reduce thck so that lsrf = topg
                model%geometry%thck(i,j) = model%geometry%usrf(i,j) - topg
             endif
          enddo
       enddo

    else   ! usrf_max < tiny

       call write_log('Error: Must read in usrf to use adjust_input_thickness option', GM_FATAL)

    endif   ! usrf_max > tiny

  end subroutine glissade_adjust_thickness

!****************************************************************************

  subroutine glissade_smooth_usrf(model, nsmooth)

    ! Use a Laplacian smoother to smooth the upper surface elevation,
    !  and compute a thickness consistent with this new elevation.
    ! This can be useful if the input thickness and topography are inconsistent,
    !  such that their sum has large gradients.

    use glide_thck, only: glide_calclsrf
    use glissade_masks, only: glissade_get_masks
    use glissade_grid_operators, only: glissade_laplacian_smoother
    use cism_parallel, only: parallel_halo

    !----------------------------------------------------------------
    ! Input-output arguments
    !----------------------------------------------------------------

    type(glide_global_type), intent(inout) :: model   ! derived type holding ice-sheet info

    integer, intent(in), optional :: nsmooth     ! number of smoothing passes

    ! local variables

    real(dp), dimension(model%general%ewn, model%general%nsn) :: &
         topg,               &  ! bed topography (m)
         thck,               &  ! thickness (m)
         usrf,               &  ! surface elevation (m)
         usrf_smoothed          ! surface elevation after smoothing

    integer, dimension(model%general%ewn, model%general%nsn) :: &
         ice_mask,           &  ! = 1 if ice is present (thck > 0, else = 0
         floating_mask,      &  ! = 1 if ice is present (thck > 0) and floating, else = 0
         ocean_mask             ! = 1 if topg < 0 and ice is absent, else = 0

    integer :: n_smoothing_passes   ! local version of nsmooth
    integer :: i, j, n
    integer :: nx, ny
    integer :: itest, jtest, rtest

!    logical, parameter :: verbose_smooth_usrf = .false.
    logical, parameter :: verbose_smooth_usrf = .true.

    ! Initialize

    if (present(nsmooth)) then
       n_smoothing_passes = nsmooth
    else
       n_smoothing_passes = 1
    endif

    ! Copy some model variables to local variables

    nx = model%general%ewn
    ny = model%general%nsn

    rtest = -999
    itest = 1
    jtest = 1
    if (this_rank == model%numerics%rdiag_local) then
       rtest = model%numerics%rdiag_local
       itest = model%numerics%idiag_local
       jtest = model%numerics%jdiag_local
    endif

    ! compute the initial upper surface elevation
    call glide_calclsrf(model%geometry%thck, model%geometry%topg, model%climate%eus, model%geometry%lsrf)
    model%geometry%usrf = max(0.d0, model%geometry%thck + model%geometry%lsrf)

    ! Save input fields
    topg = (model%geometry%topg - model%climate%eus)
    thck = model%geometry%thck
    usrf = model%geometry%usrf

    ! compute initial masks
    call glissade_get_masks(nx,                  ny,                    &
                            model%parallel,                             &
                            model%geometry%thck, model%geometry%topg,   &
                            model%climate%eus,   0.0d0,                 &  ! thklim = 0
                            ice_mask,                                   &
                            floating_mask = floating_mask,              &
                            ocean_mask = ocean_mask)

    do n = 1, n_smoothing_passes

       call glissade_laplacian_smoother(nx,     ny,              &
                                        usrf,   usrf_smoothed,   &
                                        npoints_stencil = 9)

       ! Force usrf = topg on ice-free land
       where (topg > 0.0d0 .and. ice_mask == 0) usrf_smoothed = topg

       ! Force usrf = unsmoothed value for floating ice and ice-free ocean, to avoid advancing the calving front
       where (floating_mask == 1 .or. ocean_mask == 1)
          usrf_smoothed = usrf
       endwhere

       ! Force usrf >= topg
       usrf_smoothed = max(usrf_smoothed, topg)

       usrf = usrf_smoothed
       call parallel_halo(usrf, model%parallel)

    enddo

    ! Given the smoothed usrf, adjust the input thickness such that topg is unchanged.
    ! Do this only where ice is present.  Elsewhere, usrf = topg.

    where (usrf > topg)     ! ice is present
       where (topg < 0.0d0)    ! marine-based ice
          where (topg*(1.0d0 - rhoo/rhoi) > usrf)  ! ice is floating
             thck = usrf / (1.0d0 - rhoi/rhoo)
          elsewhere   ! ice is grounded
             thck = usrf - topg
          endwhere
       elsewhere   ! land-based ice
          thck = usrf - topg
       endwhere
    endwhere

    ! Copy the new thickness and usrf to the model derived type
    model%geometry%thck = thck
    model%geometry%usrf = usrf

  end subroutine glissade_smooth_usrf

!****************************************************************************

  subroutine glissade_smooth_topography(model)

    ! Use a Laplacian smoother to smooth the input bed topography
    !TODO - This smoothing needs some more testing.  In particular, it is unclear how best to treat
    !        the ice thickness in regions that transition from grounded to floating
    !        when the topography is smoothed. Is it better to preserve thickness, or to
    !        increase thickness to keep the ice grounded?

    use glide_thck, only: glide_calclsrf
    use glissade_masks, only: glissade_get_masks
    use glissade_grid_operators, only: glissade_laplacian_smoother

    !----------------------------------------------------------------
    ! Input-output arguments
    !----------------------------------------------------------------

    type(glide_global_type), intent(inout) :: model   ! derived type holding ice-sheet info

    ! local variables

    integer, dimension(model%general%ewn, model%general%ewn) :: &
         ice_mask,           &  !
         floating_mask

    real(dp), dimension(model%general%ewn, model%general%nsn) :: &
         topg_smoothed          ! bed topography after smoothing

    integer :: i, j
    integer :: nx, ny
    integer :: itest, jtest, rtest
    ! The following variables give the boundaries of a box going from itest-3 to itest+3,
    ! and jtest-3 to jtest+3, but limited to stay within the range of the local points
    ! owned by rdiag_local
    integer :: itest_m3, itest_p3, jtest_m3, jtest_p3

    logical, parameter :: verbose_smooth_topg = .false.

    ! Copy some model variables to local variables

    nx = model%general%ewn
    ny = model%general%nsn

    rtest = -999
    itest = 1
    jtest = 1
    if (this_rank == model%numerics%rdiag_local) then
       rtest = model%numerics%rdiag_local
       itest = model%numerics%idiag_local
       jtest = model%numerics%jdiag_local
       itest_m3 = max(itest-3, 1)
       itest_p3 = min(itest+3, nx)
       jtest_m3 = max(jtest-3, 1)
       jtest_p3 = min(jtest+3, ny)
    endif

    ! compute the initial upper surface elevation (to be held fixed under smoothing of bed topography)
    call glide_calclsrf(model%geometry%thck, model%geometry%topg, model%climate%eus, model%geometry%lsrf)
    model%geometry%usrf = max(0.d0, model%geometry%thck + model%geometry%lsrf)

    ! compute initial mask
    ! Modify glissade_get_masks so that 'parallel' is not needed
    call glissade_get_masks(nx,                  ny,                    &
                            model%parallel,                             &
                            model%geometry%thck, model%geometry%topg,   &
                            model%climate%eus,   0.0d0,                 &  ! thklim = 0
                            ice_mask,                                   &
                            floating_mask = floating_mask)

    call glissade_laplacian_smoother(model%general%ewn, model%general%nsn,  &
                                     model%geometry%topg, topg_smoothed,    &
                                     npoints_stencil = 5)

    !WHL - debug - Try doing less smoothing than the smoother computes
    model%geometry%topg = 0.50d0 * (model%geometry%topg + topg_smoothed)

    ! Given the smoothed topography, adjust the input thickness such that usrf is unchanged.
    where (model%geometry%topg - model%climate%eus < 0.0d0)  ! marine-based ice
       where (ice_mask == 1 .and. floating_mask == 0)
          ! Ice was grounded before smoothing of topography; assume it is still grounded.
          ! This means that where topg has been lowered, we will thicken the ice.
          model%geometry%thck = model%geometry%usrf - model%geometry%topg
       elsewhere
          ! Ice was floating before smoothing of topography.
          ! It may now be grounded where topg has been raised, in which case we move lsrf up to meet the topography.
          model%geometry%lsrf = max(model%geometry%lsrf, model%geometry%topg)
          model%geometry%thck = model%geometry%usrf - model%geometry%lsrf
       endwhere
    elsewhere   ! land-based ice
       model%geometry%thck = model%geometry%usrf - model%geometry%topg
    endwhere

    !WHL - usrf for debugging only
    call glide_calclsrf(model%geometry%thck, model%geometry%topg, model%climate%eus, model%geometry%lsrf)
    model%geometry%usrf = max(0.d0, model%geometry%thck + model%geometry%lsrf)

  end subroutine glissade_smooth_topography

!****************************************************************************

  subroutine glissade_adjust_topography(model)

    ! Adjust the input bed topography in a specified region.
    ! For example, we may want to raise the topography close to the surface in a region
    !  where the ice is not sufficiently grounded, and the data are not well constrained.
    ! Note: So far, this subroutine has been used to raise eastern Thwaites topography.
    !       It has not been used to lower topography.

    use glide_thck, only: glide_calclsrf  ! TODO - Make this a glissade subroutine (e.g., in this module)

    !----------------------------------------------------------------
    ! Input-output arguments
    !----------------------------------------------------------------

    type(glide_global_type), intent(inout) :: model   ! derived type holding ice-sheet info

    ! local variables

    real(dp), dimension(:,:), allocatable :: &
         topg           ! topography with units of m

    integer :: i, j
    integer :: nx, ny
    integer :: itest, jtest, rtest
    ! The following variables give the boundaries of a box going from itest-3 to itest+3,
    ! and jtest-3 to jtest+3, but limited to stay within the range of the local points
    ! owned by rdiag_local
    integer :: itest_m3, itest_p3, jtest_m3, jtest_p3

    real(dp) :: factor

    real(dp) :: &
         xmin, xmax, ymin, ymax  ! x and y boundaries of adjusted region

    ! Note: There are two ways to use these parameters.
    ! (1) If topg_max_adjust > topg_no_adjust, then we change high topography (topg > topg_max_adjust)
    !     by topg_delta and leave low topography (topg < topg_no_adjust) unchanged.
    ! (2) If topg_max_adjust < topg_no_adjust, then we change low topography (topg < topg_max_adjust)
    !     by topg_delta and leave high topography (topg > topg_no_adjust) unchanged.
    ! Between topg_no_adjust and topg_max_adjust, the adjustment is phased in linearly.

    real(dp) :: &
         topg_no_adjust, &    ! elevation (m) beyond which there is no adjustment
         topg_max_adjust, &   ! elevation (m) beyond which there is full adjustment (by topg_delta)
         topg_delta           ! max change in topography (m); can be either sign

    logical, parameter :: verbose_adjust_topg = .true.

    ! Copy some model variables to local variables

    nx = model%general%ewn
    ny = model%general%nsn

    rtest = -999
    itest = 1
    jtest = 1
    if (this_rank == model%numerics%rdiag_local) then
       rtest = model%numerics%rdiag_local
       itest = model%numerics%idiag_local
       jtest = model%numerics%jdiag_local
       itest_m3 = max(itest-3, 1)
       itest_p3 = min(itest+3, nx)
       jtest_m3 = max(jtest-3, 1)
       jtest_p3 = min(jtest+3, ny)
    endif

    xmin = model%paramets%adjust_topg_xmin
    xmax = model%paramets%adjust_topg_xmax
    ymin = model%paramets%adjust_topg_ymin
    ymax = model%paramets%adjust_topg_ymax
    topg_no_adjust = model%paramets%adjust_topg_no_adjust
    topg_max_adjust = model%paramets%adjust_topg_max_adjust
    topg_delta = model%paramets%adjust_topg_delta

    if (verbose_adjust_topg .and. this_rank == rtest) then
       i = itest
       j = jtest
       write(iulog,*) ' '
       write(iulog,*) 'Adjust input topography, diag point: r, i ,j =', rtest, itest, jtest
       write(iulog,*) 'x1, y1 =', model%general%x1(i), model%general%y1(j)
       write(iulog,*) 'thck, topg =', model%geometry%thck(i,j), model%geometry%topg(i,j)
       write(iulog,*) 'xmin, xmax =', xmin, xmax
       write(iulog,*) 'ymin, ymax =', ymin, ymax
       write(iulog,*) 'topg_no_adjust, topg_max_adjust (m) =', topg_no_adjust, topg_max_adjust
       write(iulog,*) 'topg_delta =', topg_delta
    endif

    ! Compute the lower and upper ice surface before the adjustment
    call glide_calclsrf(model%geometry%thck, model%geometry%topg, model%climate%eus, model%geometry%lsrf)
    model%geometry%usrf = max(0.d0, model%geometry%thck + model%geometry%lsrf)

    !TODO - Use model%geometry%topg - model%climate%eus?
    allocate(topg(model%general%ewn, model%general%nsn))
    topg = model%geometry%topg

    ! Apply the topographic correction.
    ! Case 1: topg_max_adjust > topg_no_adjust; change higher topography and leave lower topography unchanged
    ! Case 2: topg_max_adjust < topg_no_adjust; change lower topography and leave higher topography unchanged

    if (topg_max_adjust > topg_no_adjust) then

       ! Where topg > topg_max_adjust, apply the max correction, topg_delta.
       ! Where topg < topg_no_adjust, apply no correction.
       ! Where topg_no_adjust < topg < topg_max_adjust, phase in the correction linearly.

       do j = 1, ny
          do i = 1, nx
             if (model%general%x1(i) >= xmin .and. model%general%x1(i) <= xmax .and. &
                 model%general%y1(j) >= ymin .and. model%general%y1(j) <= ymax) then
                if (topg(i,j) > topg_no_adjust) then
                   factor = min((topg(i,j) - topg_no_adjust)/(topg_max_adjust - topg_no_adjust), 1.0d0)
                   topg(i,j) = topg(i,j) + factor * topg_delta
                endif
             endif
          enddo
       enddo

    elseif (topg_max_adjust < topg_no_adjust) then

       ! Where topg < topg_max_adjust, apply the max correction, topg_delta.
       ! Where topg > topg_no_adjust, apply no correction.
       ! Where topg_max_adjust < topg < topg_no_adjust, phase in the correction linearly.

       do j = 1, ny
          do i = 1, nx
             if (model%general%x1(i) >= xmin .and. model%general%x1(i) <= xmax .and. &
                 model%general%y1(j) >= ymin .and. model%general%y1(j) <= ymax) then
                if (topg(i,j) < topg_no_adjust) then
                   factor = min((topg_no_adjust - topg(i,j))/(topg_no_adjust - topg_max_adjust), 1.0d0)
                   topg(i,j) = topg(i,j) + factor * topg_delta
                endif
             endif
          enddo
       enddo

    endif

    model%geometry%topg = topg
    deallocate(topg)

    ! In some cells, the new lower surface (usrf - thck) may lie below the topography.
    ! In these cells, reduce the ice thickness such that lsrf = topg, preserving the input value of usrf.

    where (model%geometry%usrf - model%geometry%thck < model%geometry%topg)
       model%geometry%thck = model%geometry%usrf - model%geometry%topg
    endwhere

  end subroutine glissade_adjust_topography

!****************************************************

  subroutine glissade_basin_sum(&
       nx,           ny,            &
       nbasin,       basin_number,  &
       rmask,                       &
       field_2d,                    &
       field_basin_sum)

    ! For a given 2D input field, compute the sum over a basin.
    ! The sum is taken over grid cells with mask = 1.
    ! All cells are weighted equally.

    use cism_parallel, only: parallel_reduce_sum, nhalo

    integer, intent(in) :: &
         nx, ny                    !> number of grid cells in each dimension

    integer, intent(in) :: &
         nbasin                    !> number of basins

    integer, dimension(nx,ny), intent(in) :: &
         basin_number              !> basin ID for each grid cell

    ! Note: For the next two fields, the dimension can be either (nx,ny) or (nx-1,ny-1)
    real(dp), dimension(:,:), intent(in) :: &
         rmask,                 &  !> real mask for weighting the input field
         field_2d                  !> input field to be averaged over basins

    real(dp), dimension(nbasin), intent(out) :: &
         field_basin_sum           !> basin-sum output field

    ! local variables

    integer :: i, j, nb

    !TODO - Replace sumcell with sumarea, and pass in cell area.
    !       Current algorithm assumes all cells with mask = 1 have equal weight.

    real(dp), dimension(nbasin) ::  &
         sumfield_local     ! sum of field on local task

    sumfield_local(:) = 0.0d0

    ! loop over locally owned cells
    do j = nhalo+1, ny-nhalo
       do i = nhalo+1, nx-nhalo
          nb = basin_number(i,j)
          if (nb >= 1) then
             sumfield_local(nb) = sumfield_local(nb) + rmask(i,j)*field_2d(i,j)
          endif
       enddo
    enddo

    field_basin_sum(:) =  parallel_reduce_sum(sumfield_local(:))

  end subroutine glissade_basin_sum

!****************************************************

  subroutine glissade_basin_average(&
       nx,           ny,            &
       nbasin,       basin_number,  &
       rmask,                       &
       field_2d,                    &
       field_basin_avg)

    ! For a given 2D input field, compute the average over a basin.
    ! The average is taken over grid cells with mask = 1.
    ! All cells are weighted equally.
    ! Note: This subroutine assumes an input field located at cell centers

    use cism_parallel, only: parallel_reduce_sum, nhalo

    integer, intent(in) :: &
         nx, ny                    !> number of grid cells in each dimension

    integer, intent(in) :: &
         nbasin                    !> number of basins

    integer, dimension(nx,ny), intent(in) :: &
         basin_number              !> basin ID for each grid cell

    ! Note: For the next two fields, the dimension can be either (nx,ny) or (nx-1,ny-1)
    real(dp), dimension(:,:), intent(in) :: &
         rmask,                  & !> real mask for weighting the value in each cell
         field_2d                  !> input field to be averaged over basins

    real(dp), dimension(nbasin), intent(out) :: &
         field_basin_avg           !> basin-average output field

    ! local variables

    integer :: i, j, nb

    !TODO - Replace sumcell with sumarea, and pass in cell area.
    !       Current algorithm assumes all cells with mask = 1 have equal weight.

    real(dp), dimension(nbasin) ::  &
         summask_local,          & ! sum of mask in each basin on local task
         summask_global,         & ! sum of mask in each basin on full domain
         sumfield_local,         & ! sum of field on local task
         sumfield_global           ! sum of field over full domain

    summask_local(:) = 0.0d0
    sumfield_local(:) = 0.0d0

    ! loop over locally owned cells only
    do j = nhalo+1, ny-nhalo
       do i = nhalo+1, nx-nhalo
          nb = basin_number(i,j)
          if (nb >= 1) then
             summask_local(nb) = summask_local(nb) + rmask(i,j)
             sumfield_local(nb) = sumfield_local(nb) + rmask(i,j)*field_2d(i,j)
          endif
       enddo
    enddo

    summask_global(:)  =  parallel_reduce_sum(summask_local(:))
    sumfield_global(:) =  parallel_reduce_sum(sumfield_local(:))

    do nb = 1, nbasin
       if (summask_global(nb) > tiny(0.0d0)) then
          field_basin_avg(nb) = sumfield_global(nb)/summask_global(nb)
       else
          field_basin_avg(nb) = 0.0d0
       endif
    enddo

  end subroutine glissade_basin_average

!****************************************************************************

  subroutine glissade_rms_error(&
       nx,       ny,          &
       mask,     parallel,    &
       field,    field_ref,   &
       rmse)

    use cism_parallel, only: parallel_global_sum

    ! Compute the root-mean-square error of an input field relative to a reference field.
    ! Typically, the input field would be computed by CISM, with the reference field
    !  based on observations.

    ! input/output arguments

    integer, intent(in) :: &
         nx, ny                  ! number of cells in x and y direction on input grid (global)

    integer, dimension(nx,ny), intent(in) :: &
         mask                    ! = 1 for the domain over which the rmse is computed

    type(parallel_type), intent(in) :: &
         parallel                ! info for parallel communication

    real(dp), dimension(nx,ny), intent(in) :: &
         field,                & ! 2D model field
         field_ref               ! reference field

    real(dp), intent(out) :: &
         rmse                    ! root-mean-square error

    ! local variables

    real(dp), dimension(nx,ny) :: &
         sq_diff              ! |field - field_ref|^2

    real(dp) :: &
         sum_sq_diff,       & ! global sum of sq_diff
         ncells               ! number of global cells with mask = 1

    ncells = parallel_global_sum(mask, parallel)

    sq_diff = (abs(field - field_ref))**2
    sum_sq_diff = parallel_global_sum(sq_diff, parallel, mask)

    if (ncells > 0.0d0) then
       rmse = sqrt(sum_sq_diff/ncells)
    else
       rmse = 0.0d0
    endif

  end subroutine glissade_rms_error

!***********************************************************************

  subroutine glissade_usrf_to_thck(usrf, topg, eus, thck)

    ! Given the bed topography and upper ice surface elevation, compute the ice thickness.
    ! The ice is assumed to satisfy a flotation condition.
    ! That is, if topg - eus < 0 (marine-based ice), and if the upper surface is too close
    !  to sea level to ground the ice, then the ice thickness is chosen to satisfy
    !  rhoi*H = -rhoo*(topg-eus).
    ! Note: usrf, topg, eus and thck must all have the same units (often but not necessarily meters).

    use glimmer_physcon, only : rhoo, rhoi

    real(dp), dimension(:,:), intent(in) :: &
         usrf,           & ! ice upper surface elevation
         topg              ! elevation of bedrock topography

    real(dp), intent(in) :: &
         eus               ! eustatic sea level

    real(dp), dimension(:,:), intent(out) :: &
         thck              ! ice thickness

    ! initialize
    thck(:,:) = 0.0d0

    where (usrf > (topg - eus))   ! ice is present, thck > 0
       where (topg - eus < 0.0d0)   ! marine-based ice
          where ((topg - eus) * (1.0d0 - rhoo/rhoi) > usrf)  ! ice is floating
             thck = usrf / (1.0d0 - rhoi/rhoo)
          elsewhere   ! ice is grounded
             thck = usrf - (topg - eus)
          endwhere
       elsewhere   ! land-based ice
          thck = usrf - (topg - eus)
       endwhere
    endwhere

  end subroutine glissade_usrf_to_thck

!***********************************************************************

  subroutine glissade_thck_to_usrf(thck, topg, eus, usrf)

    ! Given the bed topography and ice thickness, compute the upper surface elevation.
    ! The ice is assumed to satisfy a flotation condition.
    ! That is, if topg - eus < 0 (marine-based ice), and if the ice is too thin to be grounded,
    !  then the upper surface is chosen to satisfy rhoi*H = rhoo*(H - usrf),
    !  or equivalently usrf = (1 - rhoi/rhoo)*H.
    ! Note: usrf, topg, eus and thck must all have the same units (often but not necessarily meters).

    use glimmer_physcon, only : rhoo, rhoi

    real(dp), dimension(:,:), intent(in) :: &
         thck,           & ! ice thickness
         topg              ! elevation of bedrock topography

    real(dp), intent(in) :: &
         eus               ! eustatic sea level

    real(dp), dimension(:,:), intent(out) :: &
         usrf              ! ice upper surface elevation

    where ((topg - eus) < -(rhoi/rhoo)*thck)
       usrf = (1.0d0 - rhoi/rhoo)*thck   ! ice is floating
    elsewhere   ! ice is grounded
       usrf = (topg - eus) + thck
    endwhere

  end subroutine glissade_thck_to_usrf

!***********************************************************************

  subroutine glissade_edge_fluxes(&
        nx,        ny,        &
        dew,       dns,       &
        itest,     jtest,  rtest, &
        thck,                 &
        uvel,      vvel,      &
        flux_e,    flux_n)

    use cism_parallel, only: nhalo

    ! Compute ice volume fluxes across each cell edge

    ! input/output arguments

    integer, intent(in) :: &
         nx, ny,                & ! number of cells in x and y direction on input grid (global)
         itest, jtest, rtest

    real(dp), intent(in) :: &
         dew, dns                 ! cell edge lengths in EW and NS directions (m)

    real(dp), dimension(nx,ny), intent(in) :: &
         thck                     ! ice thickness (m) at cell centers

    real(dp), dimension(nx-1,ny-1), intent(in) :: &
         uvel, vvel               ! vertical mean velocity (m/s) at cell corners

    real(dp), dimension(nx,ny), intent(out) :: &
         flux_e, flux_n           ! ice volume fluxes (m^3/yr) at cell edges

    ! local variables

    integer :: i, j
    real(dp) :: thck_edge, u_edge, v_edge
    logical, parameter :: verbose_edge_fluxes = .false.

    ! loop over locally owned edges
    do j = nhalo+1, ny-nhalo
       do i = nhalo+1, nx-nhalo

          ! east edge volume flux
          thck_edge = 0.5d0 * (thck(i,j) + thck(i+1,j))
          u_edge = 0.5d0 * (uvel(i,j-1) + uvel(i,j))
          flux_e(i,j) = thck_edge * u_edge * dns  ! m^3/yr

          ! north edge volume flux
          thck_edge = 0.5d0 * (thck(i,j) + thck(i,j+1))
          v_edge = 0.5d0 * (vvel(i-1,j) + vvel(i,j))
          flux_n(i,j) = thck_edge * v_edge * dew  ! m^3/yr

          if (verbose_edge_fluxes .and. this_rank == rtest .and. i==itest .and. j==jtest) then
             write(iulog,*) 'East  flux: rank, i, j, H, u, flx =', &
                  rtest, itest, jtest, thck_edge, u_edge, flux_e(i,j)
             write(iulog,*) 'North flux: rank, i, j, H, v, flx =', &
                  rtest, itest, jtest, thck_edge, v_edge, flux_n(i,j)
          endif

       enddo
    enddo

  end subroutine glissade_edge_fluxes

!***********************************************************************

  subroutine glissade_input_fluxes(&
        nx,      ny,            &
        dew,     dns,           &
        itest,   jtest,  rtest, &
        thck,                   &
        uvel,    vvel,          &
        flux_in,                &
        parallel)

    use glimmer_physcon, only: scyr
    use cism_parallel, only: nhalo, parallel_halo, staggered_parallel_halo

    ! Compute ice volume fluxes into a cell from each neighboring cell

    ! input/output arguments

    integer, intent(in) :: &
         nx, ny,                & ! number of cells in x and y direction on input grid (global)
         itest, jtest, rtest

    real(dp), intent(in) :: &
         dew, dns                 ! cell edge lengths in EW and NS directions (m)

    real(dp), dimension(nx,ny), intent(in) :: &
         thck                     ! ice thickness (m) at cell centers

    real(dp), dimension(nx-1,ny-1), intent(in) :: &
         uvel, vvel               ! vertical mean velocity (m/s) at cell corners

    real(dp), dimension(-1:1,-1:1,nx,ny), intent(out) :: &
         flux_in                  ! ice volume fluxes (m^3/yr) into cell from each neighbor cell

    type(parallel_type), intent(in) :: parallel   ! info for parallel communication

    ! local variables

    integer :: i, j, ii, jj

    real(dp) :: &
         u_sw, u_se, u_ne, u_nw,    & ! u velocity components at each vertex
         v_sw, v_se, v_ne, v_nw       ! u velocity components at each vertex

    real(dp) :: &
         area_w, area_s, area_e, area_n,   & ! area flux from each neighbor cell
         area_sw, area_se, area_ne, area_nw

    logical, parameter :: verbose_input_fluxes = .false.

    ! halo updates for thickness and velocity

    call parallel_halo(thck, parallel)
    call staggered_parallel_halo(uvel, parallel)
    call staggered_parallel_halo(vvel, parallel)

    ! initialize
    flux_in(:,:,:,:) = 0.0d0

    ! Estimate the ice volume flux into each cell from each neighbor.
    ! Note: flux_in(0,0,:,:) = 0 since there is no flux from a cell into itself.
    ! The loop includes one row of halo cells.

    do j = nhalo+1, ny-nhalo
       do i = nhalo+1, nx-nhalo

          ! Compute the upwind velocity components at each vertex
          ! Convert from m/s to m/yr for diagnostics
          u_sw = max( uvel(i-1,j-1), 0.0d0)*scyr
          v_sw = max( vvel(i-1,j-1), 0.0d0)*scyr
          u_se = max(-uvel(i,j-1),   0.0d0)*scyr
          v_se = max( vvel(i,j-1),   0.0d0)*scyr
          u_ne = max(-uvel(i,j),     0.0d0)*scyr
          v_ne = max(-vvel(i,j),     0.0d0)*scyr
          u_nw = max( uvel(i-1,j),   0.0d0)*scyr
          v_nw = max(-vvel(i-1,j),   0.0d0)*scyr

          ! Estimate the area fluxes from each edge neighbor
          area_w = 0.5d0*(u_nw + u_sw)*dns - 0.5d0*(u_nw*v_nw + u_sw*v_sw)
          area_s = 0.5d0*(v_sw + v_se)*dew - 0.5d0*(u_sw*v_sw + u_se*v_se)
          area_e = 0.5d0*(u_se + u_ne)*dns - 0.5d0*(u_se*v_se + u_ne*v_ne)
          area_n = 0.5d0*(v_ne + v_nw)*dew - 0.5d0*(u_ne*v_ne + u_nw*v_nw)

          ! Estimate the area fluxes from each diagonal neighbor
          ! Note: The sum is equal to the sum of the terms subtracted from the edge areas above
          area_sw = u_sw*v_sw
          area_se = u_se*v_se
          area_ne = u_ne*v_ne
          area_nw = u_nw*v_nw

          ! Estimate the volume fluxes from each edge neighbor
          flux_in(-1, 0,i,j) = area_w * thck(i-1,j)
          flux_in( 0,-1,i,j) = area_s * thck(i,j-1)
          flux_in( 1, 0,i,j) = area_e * thck(i+1,j)
          flux_in( 0, 1,i,j) = area_n * thck(i,j+1)

          ! Estimate the volume fluxes from each diagonal neighbor
          flux_in(-1,-1,i,j) = area_sw * thck(i-1,j-1)
          flux_in( 1,-1,i,j) = area_se * thck(i+1,j-1)
          flux_in( 1, 1,i,j) = area_ne * thck(i+1,j+1)
          flux_in(-1, 1,i,j) = area_nw * thck(i-1,j+1)

          if (verbose_input_fluxes .and. this_rank == rtest .and. i==itest .and. j==jtest) then
             write(iulog,*) ' '
             write(iulog,*) 'upstream u (m/yr), this_rank, i, j:'
             write(iulog,'(3e12.4)') u_nw, u_ne
             write(iulog,'(3e12.4)') u_sw, u_se
             write(iulog,*) ' '
             write(iulog,*) 'upstream v (m/yr):'
             write(iulog,'(3e12.4)') v_nw, v_ne
             write(iulog,'(3e12.4)') v_sw, v_se
             write(iulog,*) ' '
             write(iulog,*) 'Input area fluxes (m^2/yr):'
             write(iulog,'(3e12.4)') area_nw, area_n, area_ne
             write(iulog,'(3e12.4)') area_w,  0.0d0, area_e
             write(iulog,'(3e12.4)') area_sw, area_s, area_se
             write(iulog,*) ' '
             write(iulog,*) 'Input ice volume fluxes (m^3/yr):'
             do jj = 1,-1,-1
                do ii = -1,1
                   write(iulog,'(e12.4)',advance='no') flux_in(ii,jj,i,j)
                enddo
                write(iulog,*) ' '
             enddo
          endif

       enddo   ! i
    enddo   ! j

    do j = -1, 1
       do i = -1, 1
          call parallel_halo(flux_in, parallel)
       enddo
    enddo

  end subroutine glissade_input_fluxes

!****************************************************************************

!TODO - Other utility subroutines to add here?
!       E.g., tridiag; calclsrf; subroutines to zero out tracers

!****************************************************************************

end module glissade_utils

!****************************************************************************
