/*
 * GNU GPL v3 License
 *
 * Copyright 2018 Niccolo` Tubini
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package monodimensionalProblemTimeDependent;

import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map.Entry;

import oms3.annotations.Author;
import oms3.annotations.Description;
import oms3.annotations.Documentation;
import oms3.annotations.Execute;
import oms3.annotations.Finalize;
import oms3.annotations.In;
import oms3.annotations.Initialize;
import oms3.annotations.Keywords;
import oms3.annotations.License;
import oms3.annotations.Unit;
import ucar.ma2.Array;
import ucar.ma2.ArrayDouble;
import ucar.ma2.ArrayFloat;
import ucar.ma2.ArrayInt;
import ucar.ma2.DataType;
import ucar.ma2.InvalidRangeException;
import ucar.ma2.ArrayDouble.D1;
import ucar.nc2.Attribute;
import ucar.nc2.Dimension;
import ucar.nc2.NetcdfFileWriter;
import ucar.nc2.Variable;

@Description("This class writes a NetCDF with Richards' equation outputs. Before writing, outputs are stored in a buffer writer"
		+ " and as simulation is ended they are written in a NetCDF file.")
@Documentation("")
@Author(name = "Niccolo' Tubini, Francesco Serafin, Riccardo Rigon", contact = "tubini.niccolo@gmail.com")
@Keywords("Hydrology, Richards, Infiltration")
//@Label(JGTConstants.HYDROGEOMORPHOLOGY)
//@Name("shortradbal")
//@Status(Status.CERTIFIED)
@License("General Public License Version 3 (GPLv3)")


public class WriteNetCDFRichards1D2 {

	@Description()
	@In
	@Unit ()
	public String timeUnits = "Seconds since 01/01/1970 00:00:00 UTC";

	@Description()
	@In
	@Unit ()
	public LinkedHashMap<String,ArrayList<double[]>> myVariables; // consider the opportunity to save varibale as float instead of double

	@Description()
	@In
	@Unit ()
	public double[] mySpatialCoordinate;

	@Description()
	@In
	@Unit ()
	public double[] myDualSpatialCoordinate;

	@In
	public int writeFrequency = 1;

	@Description()
	@In
	@Unit ()
	public String fileName;

	@Description("Brief descritpion of the problem")
	@In
	@Unit ()
	public String briefDescritpion;
	@In
	public String topBC = " ";
	@In
	public String bottomBC = " ";
	@In
	public String pathTopBC = " ";
	@In
	public String pathBottomBC = " ";
	@In
	public String pathGrid = " ";
	@In
	public String timeDelta = " ";
	@In
	public String swrcModel = " ";
	@In
	public String soilHydraulicConductivityModel = " ";
	@In
	public String interfaceThermalConductivityModel = " ";


	@Description("Boolean variable to print output file only at the end of the simulation")
	@In
	@Unit ()
	public boolean doProcess;
	
	@Description("Name of the variables to save")
	@In
	@Unit ()
	public String [] outVariables = new String[]{""};

	double[] tempVariable;
	Iterator it;
	DateFormat dateFormat;
	Date date = null;
	String filename;
	NetcdfFileWriter dataFile;
	int KMAX;
	int DUALKMAX;
	int NREC;
	int[] origin;
	int[] dual_origin;
	int[] time_origin;
	int i;
	Dimension kDim;
	Dimension dualKDim;
	Dimension timeDim;
	D1 depth;
	D1 dualDepth;
	Array times;
	String dims;
	List<String> outVariablesList;

	Variable timeVar;
	Variable depthVar;
	Variable dualDepthVar;
	Variable psiVar;
	Variable icVar;
	Variable thetaVar;
	Variable darcyVelocitiesVar;
	Variable darcyVelocitiesCapillaryVar;
	Variable darcyVelocitiesGravityVar;
	Variable poreVelocitiesVar;
	Variable celerityVar;
	Variable kinematicRatioVar;
	Variable errorVar;
	Variable topBCVar;
	Variable bottomBCVar;
	Variable runOffVar;

	ArrayDouble.D1 dataPsiIC;
	ArrayDouble.D1 dataError;
	ArrayDouble.D1 dataTopBC;
	ArrayDouble.D1 dataBottomBC;
	ArrayDouble.D1 dataRunOff;

	ArrayDouble.D2 dataPsi;
	ArrayDouble.D2 dataTheta;
	ArrayDouble.D2 dataDarcyVelocities;
	ArrayDouble.D2 dataDarcyVelocitiesCapillary;
	ArrayDouble.D2 dataDarcyVelocitiesGravity;
	ArrayDouble.D2 dataPoreVelocities;
	ArrayDouble.D2 dataCelerity;
	ArrayDouble.D2 dataKinematicRatio;


	int step = 0;

	@Execute
	public void writeNetCDF() throws IOException {


		/*
		 * Create a new file
		 */
		if(step == 0) {
			outVariablesList = Arrays.asList(outVariables);
			
			//			System.out.println("WriterNetCDF step:" + step);
			KMAX = mySpatialCoordinate.length;
			DUALKMAX = myDualSpatialCoordinate.length;
			//			NREC = myVariables.keySet().size();

			dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm");
			date = null;

			origin = new int[]{0, 0};
			dual_origin = new int[]{0, 0};
			time_origin = new int[]{0};

			dataFile = null;


			try {
				// Create new netcdf-3 file with the given filename
				dataFile = NetcdfFileWriter.createNew(NetcdfFileWriter.Version.netcdf3, fileName);
				// add a general attribute describing the problem and containing other relevant information for the user
				dataFile.addGroupAttribute(null, new Attribute("Description of the problem",briefDescritpion));
				dataFile.addGroupAttribute(null, new Attribute("Top boundary condition",topBC));
				dataFile.addGroupAttribute(null, new Attribute("Bottom boundary condition",bottomBC));
				dataFile.addGroupAttribute(null, new Attribute("path top boundary condition",pathTopBC));
				dataFile.addGroupAttribute(null, new Attribute("path bottom boundary condition",pathBottomBC));
				dataFile.addGroupAttribute(null, new Attribute("path grid",pathGrid));			
				dataFile.addGroupAttribute(null, new Attribute("time delta",timeDelta));
				dataFile.addGroupAttribute(null, new Attribute("swrc model",swrcModel));
				dataFile.addGroupAttribute(null, new Attribute("soil hydraulic conductivity model",soilHydraulicConductivityModel));
				dataFile.addGroupAttribute(null, new Attribute("interface thermal conductivity model",interfaceThermalConductivityModel));

				//add dimensions  where time dimension is unlimit
				// the spatial dimension is defined using just the indexes 
				kDim = dataFile.addDimension(null, "depth", KMAX);
				dualKDim = dataFile.addDimension(null, "dualDepth", DUALKMAX);
				timeDim = dataFile.addUnlimitedDimension("time");

				// Define the coordinate variables.
				depthVar = dataFile.addVariable(null, "depth", DataType.DOUBLE, "depth");
				dualDepthVar = dataFile.addVariable(null, "dualDepth", DataType.DOUBLE, "dualDepth");
				timeVar = dataFile.addVariable(null, "time", DataType.INT, "time");

				// Define units attributes for data variables.
				// Define units attributes for data variables.
				dataFile.addVariableAttribute(timeVar, new Attribute("units", timeUnits));
				dataFile.addVariableAttribute(timeVar, new Attribute("long_name", "Time."));

				dataFile.addVariableAttribute(depthVar, new Attribute("units", "m"));
				dataFile.addVariableAttribute(depthVar, new Attribute("long_name", "Soil depth."));

				dataFile.addVariableAttribute(dualDepthVar, new Attribute("units", "m"));
				dataFile.addVariableAttribute(dualDepthVar, new Attribute("long_name", "Dual soil depth."));

				// Define the netCDF variables and their attributes.
				String dims = "time depth";
				String dualDims = "time dualDepth";

				psiVar = dataFile.addVariable(null, "psi", DataType.DOUBLE, dims);
				dataFile.addVariableAttribute(psiVar, new Attribute("units", "m"));
				dataFile.addVariableAttribute(psiVar, new Attribute("long_name", "Water suction."));
				
				icVar = dataFile.addVariable(null, "psi_ic", DataType.DOUBLE, "depth");
				dataFile.addVariableAttribute(icVar, new Attribute("units", "m"));
				dataFile.addVariableAttribute(icVar, new Attribute("long_name", "Initial condition for water suction."));
				
				thetaVar = dataFile.addVariable(null, "theta", DataType.DOUBLE, dims);
				dataFile.addVariableAttribute(thetaVar, new Attribute("units", " "));
				dataFile.addVariableAttribute(thetaVar, new Attribute("long_name", "theta for within soil and water depth."));
				
				if (outVariablesList.contains("darcy_velocity") || outVariablesList.contains("all")) {
					darcyVelocitiesVar = dataFile.addVariable(null, "darcy_velocity", DataType.DOUBLE, dualDims);
					dataFile.addVariableAttribute(darcyVelocitiesVar, new Attribute("units", "m/s"));
					dataFile.addVariableAttribute(darcyVelocitiesVar, new Attribute("long_name", "Darcy velocity."));
				}
				
				if (outVariablesList.contains("darcy_velocity_capillary") || outVariablesList.contains("all")) {
					darcyVelocitiesCapillaryVar = dataFile.addVariable(null, "darcy_velocity_capillary", DataType.DOUBLE, dualDims);
					dataFile.addVariableAttribute(darcyVelocitiesCapillaryVar, new Attribute("units", "m/s"));
					dataFile.addVariableAttribute(darcyVelocitiesCapillaryVar, new Attribute("long_name", "Darcy velocity due to the gradient of capillary forces."));
				}
				
				if (outVariablesList.contains("darcy_velocity_gravity") || outVariablesList.contains("all")) {
					darcyVelocitiesGravityVar = dataFile.addVariable(null, "darcy_velocity_gravity", DataType.DOUBLE, dualDims);
					dataFile.addVariableAttribute(darcyVelocitiesGravityVar, new Attribute("units", "m/s"));
					dataFile.addVariableAttribute(darcyVelocitiesGravityVar, new Attribute("long_name", "Darcy velocities due to the gradient of gravity."));
				}
				
				if (outVariablesList.contains("pore_velocity") || outVariablesList.contains("all")) {
					poreVelocitiesVar = dataFile.addVariable(null, "pore_velocity", DataType.DOUBLE, dualDims);
					dataFile.addVariableAttribute(poreVelocitiesVar, new Attribute("units", "m/s"));
					dataFile.addVariableAttribute(poreVelocitiesVar, new Attribute("long_name", "Pore velocities, ratio between the Darcy velocities and porosity."));
				}
				
				if (outVariablesList.contains("celerity") || outVariablesList.contains("all")) {
					celerityVar = dataFile.addVariable(null, "celerities", DataType.DOUBLE, dualDims);
					dataFile.addVariableAttribute(celerityVar, new Attribute("units", "m/s"));
					dataFile.addVariableAttribute(celerityVar, new Attribute("long_name", "Celerity of the pressure wave (Rasmussen et al. 2000"));
				}
				
				if (outVariablesList.contains("kinematic_ratio") || outVariablesList.contains("all")) {
					kinematicRatioVar  = dataFile.addVariable(null, "kinematic_ratio", DataType.DOUBLE, dualDims);
					dataFile.addVariableAttribute(kinematicRatioVar, new Attribute("units", "-"));
					dataFile.addVariableAttribute(kinematicRatioVar, new Attribute("long_name", "Kinematic ratio (Rasmussen et al. 2000)"));
				}
				
				errorVar = dataFile.addVariable(null, "error", DataType.DOUBLE, "time");
				dataFile.addVariableAttribute(errorVar, new Attribute("units", "m"));
				dataFile.addVariableAttribute(errorVar, new Attribute("long_name", "volume error at each time step."));
				
				topBCVar  = dataFile.addVariable(null, "top_bc", DataType.DOUBLE, "time");
				dataFile.addVariableAttribute(topBCVar, new Attribute("units", "mm"));                   //?????
				dataFile.addVariableAttribute(topBCVar, new Attribute("long_name", "rainfall heights")); //?????
				
				bottomBCVar = dataFile.addVariable(null, "bottom_bc", DataType.DOUBLE, "time");
				dataFile.addVariableAttribute(bottomBCVar, new Attribute("units", "m"));                 //?????
				dataFile.addVariableAttribute(bottomBCVar, new Attribute("long_name", "water suction")); //?????
								
				runOffVar = dataFile.addVariable(null, "runoff", DataType.DOUBLE, "time");
				dataFile.addVariableAttribute(runOffVar, new Attribute("units", "m/s"));
				dataFile.addVariableAttribute(runOffVar, new Attribute("long_name", "run off"));



				depth = new ArrayDouble.D1(kDim.getLength());
				dualDepth = new ArrayDouble.D1(dualKDim.getLength());

				for (int k = 0; k < kDim.getLength(); k++) {
					depth.set(k, mySpatialCoordinate[k]);
				}

				for (int k = 0; k < dualKDim.getLength(); k++) {
					dualDepth.set(k, myDualSpatialCoordinate[k]);
				}

				//Create the file. At this point the (empty) file will be written to disk
				dataFile.create();
				dataFile.write(depthVar, depth);
				dataFile.write(dualDepthVar, dualDepth);

				System.out.println("\n\t***Created NetCDF " + fileName +"\n\n");
			} catch (InvalidRangeException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} finally {
				if (dataFile != null)
					try {
						dataFile.close();
					} catch (IOException ioe) {
						ioe.printStackTrace();
					}
			}

		}


		/*
		 * Write data
		 */

		if( step%writeFrequency==0 || doProcess == false) {


			try {

				dataFile = NetcdfFileWriter.openExisting(fileName);

				// number of time record that will be saved
				NREC = myVariables.keySet().size();

				times = Array.factory(DataType.INT, new int[] {NREC});

				dataPsi = new ArrayDouble.D2(NREC, dataFile.findVariable("psi").getShape()[1]);
				dataTheta = new ArrayDouble.D2(NREC, dataFile.findVariable("theta").getShape()[1]);
				dataPsiIC = new ArrayDouble.D1( dataFile.findVariable("psi_ic").getShape()[0]);	
				dataError = new ArrayDouble.D1(NREC);
				dataTopBC = new ArrayDouble.D1(NREC);
				dataBottomBC = new ArrayDouble.D1(NREC);
				dataRunOff = new ArrayDouble.D1(NREC);

				if (outVariablesList.contains("darcy_velocity") || outVariablesList.contains("all")) {
					dataDarcyVelocities = new ArrayDouble.D2(NREC, dataFile.findVariable("darcy_velocity").getShape()[1]);
				}
				
				if (outVariablesList.contains("darcy_velocity_capillary") || outVariablesList.contains("all")) {
					dataDarcyVelocitiesCapillary = new ArrayDouble.D2(NREC, dataFile.findVariable("darcy_velocity_capillary").getShape()[1]);
				}
				
				if (outVariablesList.contains("darcy_velocity_gravity") || outVariablesList.contains("all")) {
					dataDarcyVelocitiesGravity = new ArrayDouble.D2(NREC, dataFile.findVariable("darcy_velocity_gravity").getShape()[1]);
				}
				
				if (outVariablesList.contains("pore_velocity") || outVariablesList.contains("all")) {
					dataPoreVelocities = new ArrayDouble.D2(NREC, dataFile.findVariable("pore_velocity").getShape()[1]);
				}
				
				if (outVariablesList.contains("celerity") || outVariablesList.contains("all")) {
					dataCelerity = new ArrayDouble.D2(NREC, dataFile.findVariable("celerity").getShape()[1]);
				}
				
				if (outVariablesList.contains("kinematic_ratio") || outVariablesList.contains("all")) {
					dataKinematicRatio = new ArrayDouble.D2(NREC, dataFile.findVariable("kinematicRatio").getShape()[1]);
				}
				
				

				
				int i=0;
				it = myVariables.entrySet().iterator();
				while (it.hasNext()) {

					@SuppressWarnings("unchecked")
					Entry<String, ArrayList<double[]>> entry = (Entry<String, ArrayList<double[]>>) it.next();

					try {
						date = dateFormat.parse(entry.getKey());
					} catch (ParseException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}

					times.setLong(i, (long) date.getTime()/1000);


					tempVariable =  entry.getValue().get(0);
					for (int k = 0; k < KMAX; k++) {

						dataPsi.set(i, k, tempVariable[k]);

					}


					tempVariable =  entry.getValue().get(1);
					for (int k = 0; k < KMAX; k++) {

						dataTheta.set(i, k, tempVariable[k]);

					}

					if(step==0) {
						tempVariable =  entry.getValue().get(2);
						for (int k = 0; k < KMAX; k++) {

							dataPsiIC.set(k, tempVariable[k]);

						}
					}
					
					if (outVariablesList.contains("darcy_velocity") || outVariablesList.contains("all")) {
						tempVariable =  entry.getValue().get(3);
						for (int k = 0; k < DUALKMAX; k++) {

							dataDarcyVelocities.set(i, k, tempVariable[k]);

						}
					}
					
					if (outVariablesList.contains("darcy_velocity_capillary") || outVariablesList.contains("all")) {
						tempVariable =  entry.getValue().get(4);
						for (int k = 0; k < DUALKMAX; k++) {

							dataDarcyVelocitiesCapillary.set(i, k, tempVariable[k]);

						}
					}

					if (outVariablesList.contains("darcy_velocity_gravity") || outVariablesList.contains("all")) {
						tempVariable =  entry.getValue().get(5);
						for (int k = 0; k < DUALKMAX; k++) {

							dataDarcyVelocitiesGravity.set(i, k, tempVariable[k]);

						}
					}

					if (outVariablesList.contains("pore_velocity") || outVariablesList.contains("all")) {
						tempVariable =  entry.getValue().get(6);
						for (int k = 0; k < DUALKMAX; k++) {

							dataPoreVelocities.set(i,k, tempVariable[k]);

						}
					}

					if (outVariablesList.contains("celerity") || outVariablesList.contains("all")) {
						tempVariable =  entry.getValue().get(7);
						for (int k = 0; k < DUALKMAX; k++) {

							dataCelerity.set(i,k, tempVariable[k]);

						}
					}
					
					if (outVariablesList.contains("kinematic_ratio") || outVariablesList.contains("all")) {
						tempVariable =  entry.getValue().get(8);
						for (int k = 0; k < DUALKMAX; k++) {

							dataKinematicRatio.set(i,k, tempVariable[k]);

						}
					}

					dataError.set(i, entry.getValue().get(9)[0]);

					dataTopBC.set(i, entry.getValue().get(10)[0]);

					dataBottomBC.set(i, entry.getValue().get(11)[0]);

					dataRunOff.set(i, entry.getValue().get(12)[0]);

					i++;
				}				

				// A newly created Java integer array to be initialized to zeros.
				origin[0] = dataFile.findVariable("psi").getShape()[0];
//				dual_origin[0] = dataFile.findVariable("darcy_velocity").getShape()[0];
				time_origin[0] = dataFile.findVariable("time").getShape()[0];

				//				dataFile.write(kIndexVar, kIndex);
				dataFile.write(dataFile.findVariable("time"), time_origin, times);
				dataFile.write(dataFile.findVariable("psi"), origin, dataPsi);
				if(step==0) {
					dataFile.write(dataFile.findVariable("psi_ic"), origin, dataPsiIC);
				}
				dataFile.write(dataFile.findVariable("theta"), origin, dataTheta);
				
				if (outVariablesList.contains("darcy_velocity") || outVariablesList.contains("all")) {
					dataFile.write(dataFile.findVariable("darcy_velocity"), origin, dataDarcyVelocities);
				}
				
				if (outVariablesList.contains("darcy_velocity_capillary") || outVariablesList.contains("all")) {
					dataFile.write(dataFile.findVariable("darcy_velocity_capillary"), origin, dataDarcyVelocitiesCapillary);
				}
				
				if (outVariablesList.contains("darcy_velocity_gravity") || outVariablesList.contains("all")) {
					dataFile.write(dataFile.findVariable("darcy_velocity_gravity"), origin, dataDarcyVelocitiesGravity);
				}
				
				if (outVariablesList.contains("pore_velocity") || outVariablesList.contains("all")) {
					dataFile.write(dataFile.findVariable("pore_velocity"), origin, dataPoreVelocities);
				}
				
				if (outVariablesList.contains("celerity") || outVariablesList.contains("all")) {
					dataFile.write(dataFile.findVariable("celerity"), origin, dataCelerity);
				}
				
				if (outVariablesList.contains("kinematic_ratio") || outVariablesList.contains("all")) {
					dataFile.write(dataFile.findVariable("kinematic_ratio"), origin, dataKinematicRatio);
				}
				
				dataFile.write(dataFile.findVariable("error"), time_origin, dataError);
				dataFile.write(dataFile.findVariable("top_bc"), time_origin, dataTopBC);
				dataFile.write(dataFile.findVariable("bottom_bc"), time_origin, dataBottomBC);
				dataFile.write(dataFile.findVariable("runoff"), time_origin, dataRunOff);

				System.out.println("\t*** " + myVariables.keySet().toArray()[i-1].toString() +", writing output file \n");



			} catch (IOException e) {
				e.printStackTrace(System.err);

			} catch (InvalidRangeException e) {
				e.printStackTrace(System.err);

			} finally {
				if (dataFile != null)
					try {
						dataFile.close();
					} catch (IOException ioe) {
						ioe.printStackTrace();
					}
			}

		}

		step++;
	}


}


