# Copyright (C) 2011, 2014, 2015, 2016 David Maxwell
#
# This file is part of PISM.
#
# PISM is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation; either version 3 of the License, or (at your option) any later
# version.
#
# PISM is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License
# along with PISM; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""PISM's Python bindings and SSA inversions tools."""

# See if we have been imported by sphinx.  If so, we won't import any
# packages that are not needed just to compile documentation.
imported_from_sphinx = False
import inspect
caller_mod = inspect.getmodule(inspect.currentframe().f_back)
if (caller_mod is not None) and caller_mod.__name__.startswith("sphinx"):
    imported_from_sphinx = True  # pragma: no cover

if not imported_from_sphinx:
    import petsc4py

    try:
        # Look if petsc4py has already been initialized
        PETSc = petsc4py.__getattribute__('PETSc')
    except AttributeError:
        # If not, initialize petsc4py with the PETSc that PISM was compiled against.
        import sys
        from PISM.petsc_version import PISM_PETSC_ARCH
        petsc4py.init(sys.argv, arch=PISM_PETSC_ARCH)
    from petsc4py import PETSc

    from PISM.cpp import *
    import PISM.cpp

    try:
        import netCDF4 as netCDF
    except ImportError:         # pragma: no cover
        print "netCDF4 is not installed!"
        sys.exit(1)
else:                           # pragma: no cover
    # The following constants will be imported from 'cpp' if we are not
    # running inside sphinx.  But if we are inside sphinx, then we'll
    # need them to be able to import submodules.
    WITH_GHOSTS = True
    WITHOUT_GHOSTS = False
    SSAFEM = None
    SSAFD = None
    SIAFD = None
    IPDesignVariableParamIdent = None
    IPDesignVariableParamSquare = None
    IPDesignVariableParamExp = None
    IPDesignVariableParamTruncatedIdent = None
    IP_SSATaucForwardProblem = None
    IP_SSAHardavForwardProblem = None

    class IP_SSATaucTaoTikhonovProblemLCLListener(object):
        "Defined to build Sphinx docs."
        pass

    class IP_SSATaucTaoTikhonovProblemListener(object):
        "Defined to build Sphinx docs."
        pass

    class IP_SSAHardavTaoTikhonovProblemListener(object):
        "Defined to build Sphinx docs."
        pass

from PISM.options import *
import PISM.util
import PISM.vec
import PISM.ssa
import PISM.sia
import PISM.logging


class Context(object):

    """Maintains PISM data that needs to exist only once per processor.

    * ``com``    an MPI Comm
    * ``rank``   the MPI rank of the current processor
    * ``size``   the number of processors
    * ``config`` an :cpp:class:`Config`

There is only ever one :class:`Context`. If you make another one, you'll get
the first one.  You obtain the singleton as so::

    context = PISM.Context()
"""

    # Implement a Singleton pattern by overriding __new__
    _instance = None
    ctx = None

    config = None
    unit_system = None
    enthalpy_converter = None
    log = None
    time = None
    com = None
    rank = None
    size = None

    def __new__(cls):
        if cls._instance is None:
            cls._instance = super(Context, cls).__new__(cls)
            cls._instance.__init_once__()
        return cls._instance

    # Since __init__ is always called after __new__, we don't
    # want to put code that only gets run once in __init__.
    def __init_once__(self):
        ctx = PISM.context_from_options(self.com, "python")
        self.ctx = ctx

        self.com = ctx.com()
        self.rank = ctx.rank()
        self.size = ctx.size()

        self.config = ctx.config()
        self.unit_system = ctx.unit_system()
        self.enthalpy_converter = ctx.enthalpy_converter()
        self.log = ctx.log()
        self.time = ctx.time()

class AlgorithmFailureException(Exception):

    """Python exception wrapping a PISM :cpp:class:`TerminationReason`"""

    def __init__(self, reason):
        """:param reason: a :cpp:class:`TerminationReason`"""
        Exception.__init__(self)
        self._reason = reason

    def __str__(self):
        return self._reason

    def reason(self):
        "Return the stored TerminationReason."
        return self._reason


def verbPrintf(verbosity, com, msg, *args):
    """Mimics PISM's :cpp:func:`verbPrintf` but does formatting on the python side."""
    if len(args) > 0:
        msg = msg % args
    log = PISM.Context().log
    log.message(verbosity, msg)
