# Copyright (C) 2011, 2012, 2015 David Maxwell
#
# This file is part of PISM.
#
# PISM is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation; either version 3 of the License, or (at your option) any later
# version.
#
# PISM is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License
# along with PISM; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""Contains the abstract base class PlotListener for listeners that
create plots of vectors at each iteration.

"""

import PISM.logging
import PISM.vec

def pauseListener(*args):
    """Listener that temporarily halts operation at each iteration waiting for a key press."""
    PISM.logging.pause()

class PlotListener(object):

    """Base class for listeners that create plots of vectors at each iteration.
  Provides objects for converting :cpp:class:`IceModelVec`'s to ``numpy`` vectors
  on processor zero, as well as basic ``matplotlib`` figure management."""

    def __init__(self, grid):
        self.grid = grid
        self.tz_scalar = PISM.vec.ToProcZero(grid, dof=1)
        self.tz_vector = PISM.vec.ToProcZero(grid, dof=2)
        self.figs = {}

    def toproczero(self, *args):
        """Returns a ``numpy`` vector on processor zero corresponding to an :cpp:class:`IceModelVec`.
        Takes as input either a single :cpp:class:`IceModelVec` or dictionary of such
        vectors and the name of an entry. Returns ``None`` on other processors."""
        if len(args) == 2:
            data = args[0]
            name = args[1]
            v = data[name]
        else:
            v = args[0]
        if v is None:
            return None
        if v.get_ndof() == 1:
            return self.tz_scalar.communicate(v)
        return self.tz_vector.communicate(v)

    def figure(self, name='default'):
        """Returns a ``matplotlib`` figure based on a string name.  If the instance has not yet
        created a figure with the given name, a new figure is created and associated with the given name."""
        fig = self.figs.get(name)
        if fig is None:
            import matplotlib.pyplot as pp
            fig = pp.figure()
            self.figs[name] = fig
        return fig.number

    def __call__(self, solver, itr, data):
        raise NotImplementedError()
