# Copyright (C) 2011, 2012, 2014, 2015 David Maxwell
#
# This file is part of PISM.
#
# PISM is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation; either version 3 of the License, or (at your option) any later
# version.
#
# PISM is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License
# along with PISM; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""Classes for interfacing between PISM and the ``siple`` library."""

import PISM
import siple
import petsc4py


class PISMLocalVector(siple.linalg.AbstractVector):

    """Implements the :class:`siple.linalg.linalg_abstract.AbstractVector` interface for few kinds of PISM :cpp:class:`IceModelVec`."""

    def __init__(self, u):
        if isinstance(u, PISM.IceModelVec2S):
            self._core = u
            self.dof = 1
        elif isinstance(u, PISM.IceModelVec2V):
            self._core = u
            self.dof = 2
        else:
            raise ValueError("An PISMLocalVector wraps PISM IceModelVec2S or IceModelVec2V: found a %s" % u)
        self.grid = u.get_grid()

    def set(self, rhs):
        self._core.copy_from(rhs._core)

    def acc(self, rhs):
        self._core.add(1., rhs._core)

    def scale(self, t):
        self._core.scale(t)

    def axpy(self, t, v):
        self._core.add(t, v._core)

    def copy(self):
        c = self.vector_like()
        c._core.copy_from(self._core)
        return c

    def vector_like(self):
        if self.dof == 1:
            c = PISM.IceModelVec2S()
        else:
            c = PISM.IceModelVec2V()
        c.create(self.grid, "", PISM.WITH_GHOSTS, self._core.get_stencil_width())
        return PISMLocalVector(c)

    def zero_like(self):
        z = self.vector_like()
        z._core.set(0.)
        return z

    def dim(self):
        grid = self._core.get_grid()
        return self.dof * grid.Mx() * grid.My()

    def core(self):
        return self._core

    def norm(self, name):

        if name == 'linf':
            return self._core.norm(petsc4py.PETSc.NormType.NORM_INFINITY)
        if name == 'l2':
            return self._core.norm(petsc4py.PETSc.NormType.NORM_2)
        if name == 'l1':
            return self._core.norm(petsc4py.PETSc.NormType.NORM_1)

        raise ValueError()


def pism_logger(message, severity):
    """Implements a ``siple`` logger that prints output to the terminal."""
    verbosity = severity
    PISM.logging.log(message + '\n', verbosity)


def pism_pause(message_in=None, message_out=None):
    """Implements a ``siple`` pause callback appropriate for parallel runs."""
    import sys
    import os
    fd = sys.stdin.fileno()
    com = PISM.Context().com
    if os.isatty(fd):
        return siple.reporting.std_pause(message_in, message_out)
    if not message_in is None:
        PISM.verbPrintf(1, com, message_in + "\n")
    _ = sys.stdin.read(1)
    if not message_out is None:
        PISM.verbPrintf(1, com, message_out + "\n")
