# Copyright (C) 2011, 2012, 2014, 2015, 2016, 2017 David Maxwell and Constantine Khroulev
#
# This file is part of PISM.
#
# PISM is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation; either version 3 of the License, or (at your option) any later
# version.
#
# PISM is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License
# along with PISM; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""Module containing helper functions for using the SIA stress balance
model."""

import PISM

def computeSIASurfaceVelocities(modeldata, siasolver=PISM.SIAFD):
    """Generates surface horizontal velocities corresponding to solving
    the SIA with zero basal sliding.

    :param `modeldata`: :class:`PISM.model.ModelData` containing
                        variables and model physics

    :param `siasolver`: specific class used for solving the SIA

    """
    md = modeldata
    grid = md.grid
    sia = siasolver(grid)
    sia.init()

    geometry = PISM.Geometry(grid)

    geometry.cell_area.set(grid.dx() * grid.dy())
    geometry.ice_thickness.copy_from(md.vecs.thk)
    geometry.bed_elevation.copy_from(md.vecs.topg)
    geometry.sea_level_elevation.set(0.0)
    geometry.ice_area_specific_volume.set(0.0)

    geometry.ensure_consistency(md.config.get_double("geometry.ice_free_thickness_standard"))

    inputs = PISM.StressBalanceInputs()

    inputs.geometry = geometry
    inputs.basal_melt_rate = None
    inputs.melange_back_pressure = None
    inputs.basal_yield_stress = None
    inputs.enthalpy = md.vecs.enthalpy
    inputs.age = None

    sliding_velocity = PISM.IceModelVec2V()
    sliding_velocity.create(grid, 'sliding_velocity', False)
    sliding_velocity.set(0.0)

    sia.update(sliding_velocity, inputs, True)
    u = sia.velocity_u()
    v = sia.velocity_v()

    vel_sia = PISM.model.create2dVelocityVec(grid, name="_sia", stencil_width=1)
    tmp = PISM.IceModelVec2S(grid, 'tmp', False)

    u.getSurfaceValues(tmp, md.vecs.thk)
    vel_sia.set_component(0, tmp)

    v.getSurfaceValues(tmp, md.vecs.thk)
    vel_sia.set_component(1, tmp)

    return vel_sia
