/*
 * Decompiled with CFR 0.152.
 */
package gov.usgs.winston.db;

import gov.usgs.util.ConfigFile;
import gov.usgs.util.Log;
import gov.usgs.util.Retriable;
import gov.usgs.util.Util;
import gov.usgs.util.UtilException;
import gov.usgs.winston.db.PreparedStatementCache;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Locale;
import java.util.logging.Level;
import java.util.logging.Logger;

public class WinstonDatabase {
    public static final String WINSTON_TABLE_DATE_FORMAT = "yyyy_MM_dd";
    public static final String CURRENT_SCHEMA_VERSION = "1.1.1";
    private static final String DEFAULT_DATABASE_PREFIX = "W";
    private static final String DEFAULT_CONFIG_FILENAME = "Winston.config";
    private static final int DEFAULT_CACHE_CAPACITY = 100;
    private Connection winstonConnection;
    private Statement winstonStatement;
    private boolean winstonConnected;
    public final String dbDriver;
    public final String dbURL;
    public final int cacheCap;
    public final String databasePrefix;
    public final String tableEngine;
    private Logger logger = Log.getLogger("gov.usgs.winston");
    private PreparedStatementCache preparedStatements;

    public WinstonDatabase(String dbDriver, String dbURL, String databasePrefix) {
        this(dbDriver, dbURL, databasePrefix, 100);
    }

    public WinstonDatabase(String dbDriver, String dbURL, String databasePrefix, int cacheCap) {
        this(dbDriver, dbURL, databasePrefix, null, cacheCap);
    }

    public WinstonDatabase(String dbDriver, String dbURL, String databasePrefix, String tableEngine, int cacheCap) {
        Locale.setDefault(Locale.US);
        this.dbDriver = dbDriver;
        this.dbURL = dbURL;
        this.cacheCap = cacheCap;
        this.databasePrefix = Util.stringToString(databasePrefix, DEFAULT_DATABASE_PREFIX);
        this.tableEngine = tableEngine == null ? "" : " ENGINE = " + tableEngine;
        this.preparedStatements = new PreparedStatementCache(this.cacheCap, true);
        this.connect();
    }

    public Logger getLogger() {
        return this.logger;
    }

    private void connect() {
        this.winstonConnected = false;
        try {
            Class.forName(this.dbDriver).newInstance();
            DriverManager.setLoginTimeout(3);
            this.winstonConnection = DriverManager.getConnection(this.dbURL);
            this.winstonStatement = this.winstonConnection.createStatement();
            this.winstonConnected = true;
            this.preparedStatements.clear();
            this.logger.log(Level.INFO, "Connected to database.");
        }
        catch (ClassNotFoundException e) {
            this.logger.log(Level.SEVERE, "Could not load the database driver, check your CLASSPATH.", Util.getLineNumber(this, e));
            System.exit(-1);
        }
        catch (Exception e) {
            this.winstonConnection = null;
            this.winstonStatement = null;
            this.logger.log(Level.SEVERE, "Could not connect to Winston.", e);
            this.winstonConnected = false;
        }
    }

    public void close() {
        if (!this.checkConnect()) {
            return;
        }
        try {
            this.winstonStatement.close();
            this.winstonConnection.close();
            this.winstonConnected = false;
        }
        catch (Exception e) {
            this.logger.warning("Error closing database.  This is unusual, but not critical.");
        }
    }

    public boolean checkConnect() {
        if (this.winstonConnected) {
            return true;
        }
        try {
            new Retriable<Object>(){

                @Override
                public boolean attempt() throws UtilException {
                    WinstonDatabase.this.connect();
                    return WinstonDatabase.this.winstonConnected;
                }
            }.go();
        }
        catch (UtilException utilException) {
            // empty catch block
        }
        return this.winstonConnected;
    }

    public boolean connected() {
        return this.winstonConnected;
    }

    public Connection getConnection() {
        return this.winstonConnection;
    }

    public Statement getStatement() {
        return this.winstonStatement;
    }

    public Statement getNewStatement() throws SQLException {
        return this.winstonConnection.createStatement();
    }

    public String getSchemaVersion() {
        this.useRootDatabase();
        String sv = null;
        try {
            ResultSet rs = this.winstonStatement.executeQuery("SELECT schemaversion FROM version ORDER BY installtime DESC LIMIT 1");
            rs.next();
            sv = rs.getString(1);
        }
        catch (SQLException e) {
            sv = "1.0.0";
        }
        return sv;
    }

    public boolean execute(String sql) {
        Boolean b = null;
        try {
            b = (Boolean)new Retriable<Boolean>(){

                @Override
                public void attemptFix() {
                    WinstonDatabase.this.close();
                    WinstonDatabase.this.connect();
                }

                @Override
                public boolean attempt() throws UtilException {
                    try {
                        WinstonDatabase.this.winstonStatement.execute(sql);
                        this.result = new Boolean(true);
                        return true;
                    }
                    catch (SQLException e) {
                        WinstonDatabase.this.logger.log(Level.SEVERE, "execute() failed, SQL: " + sql, e);
                        this.result = new Boolean(false);
                        return false;
                    }
                }
            }.go();
        }
        catch (UtilException utilException) {
            // empty catch block
        }
        return b != null && b != false;
    }

    public ResultSet executeQuery(String sql) {
        ResultSet rs = null;
        try {
            rs = (ResultSet)new Retriable<ResultSet>(){

                @Override
                public void attemptFix() {
                    WinstonDatabase.this.close();
                    WinstonDatabase.this.connect();
                }

                @Override
                public boolean attempt() throws UtilException {
                    try {
                        this.result = WinstonDatabase.this.winstonStatement.executeQuery(sql);
                        return true;
                    }
                    catch (SQLException e) {
                        WinstonDatabase.this.logger.log(Level.SEVERE, "executeQuery() failed, SQL: " + sql, e);
                        return false;
                    }
                }
            }.go();
        }
        catch (UtilException utilException) {
            // empty catch block
        }
        return rs;
    }

    private void createTables() {
        try {
            this.getStatement().execute("CREATE TABLE instruments (iid INT PRIMARY KEY AUTO_INCREMENT,name VARCHAR(255) UNIQUE, description VARCHAR(255), lon DOUBLE DEFAULT -999, lat DOUBLE DEFAULT -999, height DOUBLE DEFAULT -999, timezone VARCHAR(128)) " + this.tableEngine);
            this.getStatement().execute("CREATE TABLE channels (sid INT PRIMARY KEY AUTO_INCREMENT, iid INT, code VARCHAR(50), st DOUBLE, et DOUBLE, alias VARCHAR(255), unit VARCHAR(255), linearA DOUBLE DEFAULT 1E300, linearB DOUBLE DEFAULT 1E300) " + this.tableEngine);
            this.getStatement().execute("CREATE TABLE version (schemaversion VARCHAR(10), installtime DATETIME) " + this.tableEngine);
            this.getStatement().execute("INSERT INTO version VALUES ('1.1.1', NOW())");
            this.getStatement().execute("CREATE TABLE grouplinks (glid INT PRIMARY KEY AUTO_INCREMENT, sid INT, nid INT) " + this.tableEngine);
            this.getStatement().execute("CREATE TABLE groupnodes (nid INT PRIMARY KEY AUTO_INCREMENT, parent INT DEFAULT 0, name CHAR(255), open BOOL DEFAULT 0) " + this.tableEngine);
            this.getStatement().execute("CREATE TABLE channelmetadata (sid INT, name VARCHAR(255), value TEXT, PRIMARY KEY (sid, name)) " + this.tableEngine);
            this.getStatement().execute("CREATE TABLE instrumentmeetadata (imid INT PRIMARY KEY AUTO_INCREMENT, iid INT, name VARCHAR(255), value TEXT) " + this.tableEngine);
            this.getStatement().execute("CREATE TABLE supp_data (sdid INT NOT NULL AUTO_INCREMENT, st DOUBLE NOT NULL, et DOUBLE, sdtypeid INT NOT NULL, sd_short VARCHAR(90) NOT NULL, sd TEXT NOT NULL, PRIMARY KEY (sdid)) " + this.tableEngine);
            this.getStatement().execute("CREATE TABLE supp_data_type (sdtypeid INT NOT NULL AUTO_INCREMENT, supp_data_type VARCHAR(20), supp_color VARCHAR(6) NOT NULL, draw_line TINYINT, PRIMARY KEY (sdtypeid), UNIQUE KEY (supp_data_type) ) " + this.tableEngine);
            this.getStatement().execute("CREATE TABLE supp_data_xref ( sdid INT NOT NULL, cid INT NOT NULL, UNIQUE KEY (sdid,cid) ) " + this.tableEngine);
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Could not create tables in WWS database.  Are permissions set properly?", e);
        }
    }

    public boolean useRootDatabase() {
        return this.useDatabase("ROOT");
    }

    public boolean useDatabase(String db) {
        if (!this.checkConnect()) {
            return false;
        }
        try {
            try {
                this.winstonStatement.execute("USE `" + this.databasePrefix + "_" + db + "`");
            }
            catch (SQLException e) {
                this.close();
                this.connect();
            }
            this.winstonStatement.execute("USE `" + this.databasePrefix + "_" + db + "`");
            return true;
        }
        catch (SQLException e) {
            if (e.getMessage().indexOf("Unknown database") != -1) {
                this.logger.log(Level.INFO, "Attempt to use nonexistent database: " + db);
            } else {
                this.logger.log(Level.SEVERE, "Could not use database: " + db, e);
            }
            return false;
        }
    }

    public boolean checkDatabase() {
        if (!this.checkConnect()) {
            return false;
        }
        try {
            boolean failed = false;
            try {
                this.getStatement().execute("USE " + this.databasePrefix + "_ROOT");
            }
            catch (Exception e) {
                failed = true;
            }
            if (failed) {
                this.getStatement().execute("CREATE DATABASE `" + this.databasePrefix + "_ROOT`");
                this.getStatement().execute("USE `" + this.databasePrefix + "_ROOT`");
                this.logger.info("Created new Winston database: " + this.databasePrefix);
                this.createTables();
            }
            return true;
        }
        catch (Exception e) {
            this.logger.severe("Could not locate or create WWS database.  Are permissions set properly?");
            return false;
        }
    }

    public boolean tableExists(String db, String table) {
        try {
            ResultSet rs = this.getStatement().executeQuery(String.format("SELECT COUNT(*) FROM `%s_%s`.%s", this.databasePrefix, db, table));
            boolean result = rs.next();
            rs.close();
            return result;
        }
        catch (Exception exception) {
            return false;
        }
    }

    public PreparedStatement getPreparedStatement(String sql) {
        try {
            PreparedStatement ps = (PreparedStatement)this.preparedStatements.get(sql);
            if (ps == null) {
                ps = this.winstonConnection.prepareStatement(sql);
                this.preparedStatements.put(sql, ps);
                this.logger.finest(String.format("Adding statement to cache(%d/%d): %s", this.preparedStatements.size(), this.preparedStatements.maxSize(), sql));
            }
            return ps;
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Could not prepare statement.", e);
            return null;
        }
    }

    public static WinstonDatabase processWinstonConfigFile() {
        return WinstonDatabase.processWinstonConfigFile(new ConfigFile(DEFAULT_CONFIG_FILENAME));
    }

    public static WinstonDatabase processWinstonConfigFile(ConfigFile cf) {
        String dbDriver = cf.getString("winston.driver");
        String dbURL = cf.getString("winston.url");
        String databasePrefix = cf.getString("winston.prefix");
        String tableEngine = cf.getString("winston.tableEngine");
        int cacheCap = Util.stringToInt(cf.getString("winston.statementCacheCap"), 100);
        return new WinstonDatabase(dbDriver, dbURL, databasePrefix, tableEngine, cacheCap);
    }
}

