MODULE module_geogrid
!%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
!
! This module contains data from geogrid.exe
!
!%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  USE module_commontypes
  USE static_input_module
  USE wrf_parallel_module
  USE wrf_llxy_module

  TYPE TYPE_GEOGRID
    INTEGER             :: dyn_opt
    CHARACTER(LEN=1)    :: grid_type
    REAL                :: dx, dy
    character (len=128) :: mminlu
    INTEGER             :: num_land_cat
    INTEGER             :: iswater, isice, isurban, isoilwater, islake
    INTEGER             :: grid_id, parent_id, parent_grid_ratio
    INTEGER             :: i_parent_start,i_parent_end
    INTEGER             :: j_parent_start,j_parent_end
    INTEGER             :: map_proj
    REAL                :: cen_lat,moad_cen_lat,cen_lon,stand_lon
    REAL                :: truelat1, truelat2, pole_lat, pole_lon
    REAL                :: corner_lats(16), corner_lons(16)

    INTEGER :: west_east_dim, south_north_dim, bottom_top_dim

    INTEGER :: we_dom_s,        we_dom_e,        sn_dom_s,        sn_dom_e
    INTEGER :: we_patch_s,      we_patch_e,      sn_patch_s,      sn_patch_e
    INTEGER :: we_patch_stag_s, we_patch_stag_e, sn_patch_stag_s, sn_patch_stag_e
    INTEGER :: we_mem_s,        we_mem_e,        sn_mem_s,        sn_mem_e
    INTEGER :: we_mem_stag_s,   we_mem_stag_e,   sn_mem_stag_s,   sn_mem_stag_e

    INTEGER :: nfields

    TYPE (type_field), POINTER :: fields(:)

    INTEGER :: xlat_m_no
    INTEGER :: xlong_m_no
    INTEGER :: xlat_u_no
    INTEGER :: xlong_u_no
    INTEGER :: xlat_v_no
    INTEGER :: xlong_v_no
    INTEGER :: landmask_no
    INTEGER :: luindex_no
    INTEGER :: hgt_m_no
    INTEGER :: hgt_u_no
    INTEGER :: hgt_v_no

  END TYPE type_geogrid

  TYPE (type_geogrid) :: geogrid

  LOGICAL :: geo_allocated

  CONTAINS

  SUBROUTINE alloc_geogrid(istatus)
!-----------------------------------------------------------------------
!
!  PURPOSE:
!
!    Allocate arrays in "geogrid" and initialize them to some values
!    (missing).
!
!-----------------------------------------------------------------------

    IMPLICIT NONE
    INTEGER, INTENT(OUT) :: istatus

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

    ALLOCATE(geogrid%fields(MAXFIELDS), STAT = istatus)

    geogrid%nfields = 0

    geogrid%xlat_m_no   = 0
    geogrid%xlong_m_no  = 0
    geogrid%xlat_u_no   = 0
    geogrid%xlong_u_no  = 0
    geogrid%xlat_v_no   = 0
    geogrid%xlong_v_no  = 0
    geogrid%landmask_no = 0
    geogrid%hgt_m_no    = 0
    geogrid%hgt_u_no    = 0
    geogrid%hgt_v_no    = 0

    geo_allocated = .TRUE.

    RETURN
  END SUBROUTINE alloc_geogrid

  SUBROUTINE dealloc_geogrid(istatus)
!-----------------------------------------------------------------------
!
!  PURPOSE:
!
!    Allocate arrays in "geogrid" and initialize them to some values
!    (missing).
!
!-----------------------------------------------------------------------

    IMPLICIT NONE
    INTEGER, INTENT(OUT) :: istatus

    INTEGER :: n
!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

    DO n = 1, geogrid%nfields
      DEALLOCATE(geogrid%fields(n)%rdata_arr, STAT = istatus)
    END DO
    DEALLOCATE(geogrid%fields, STAT = istatus)

    geogrid%nfields = 0

    geo_allocated = .FALSE.

    RETURN
  END SUBROUTINE dealloc_geogrid

  SUBROUTINE read_static_fields(iamroot,input_fname,io_form,coretype,lvldbg,istatus)
!-----------------------------------------------------------------------
!
! PURPOSE:
!   Read in static fileds from file generated by "geogrid".
!   And set up the map project etc for this grid based on information
!   from the file
!
!-----------------------------------------------------------------------
    IMPLICIT NONE
    LOGICAL,                   INTENT(IN)  :: iamroot
    CHARACTER(LEN=MAXFILELEN), INTENT(IN)  :: input_fname
    INTEGER,                   INTENT(IN)  :: io_form
    CHARACTER(LEN=3),          INTENT(IN)  :: coretype
    INTEGER,                   INTENT(IN)  :: lvldbg
    INTEGER,                   INTENT(OUT) :: istatus

!-----------------------------------------------------------------------
!
! Misc. local variables
!
!-----------------------------------------------------------------------

    LOGICAL :: IAMIO_NODE, do_tiled_input

    INTEGER :: lh_mult, rh_mult, bh_mult, th_mult

    character (len=128) :: title
    character (len=128) :: gridtype, datestr
    integer             :: west_east_dim, south_north_dim, bottom_top_dim
    integer             :: we_input_s, we_input_e, sn_input_s, sn_input_e
    integer             :: we_input_stag_s, we_input_stag_e,      &
                           sn_input_stag_s, sn_input_stag_e

    INTEGER             :: sd1, ed1, sd2, ed2, sd3, ed3
    INTEGER             :: sp1, ep1, sp2, ep2, sp3, ep3
    INTEGER             :: sm1, em1, sm2, em2
    character (len=128) :: cname, stagger, cunits, cdesc
    character (len=128), dimension(3) :: dimnames
    character (len=3)   :: memorder

    real, pointer, dimension(:,:,:) :: real_array

    INTEGER :: io_form_local, dim3

!@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@

    IAMIO_NODE     = .FALSE.
    do_tiled_input = .FALSE.

    IF (io_form > 100) THEN
      io_form_local  = MOD(io_form,100)
      do_tiled_input = .TRUE.
      IAMIO_NODE     = .TRUE.
    ELSE
      IF (iamroot) IAMIO_NODE = .TRUE.
      io_form_local  = io_form
    END IF

    CALL input_init(input_fname,io_form_local,IAMIO_NODE,do_tiled_input,&
                    istatus)

    IF (istatus /= 0) CALL arpsstop('ERROR: opening static input file.',1)

!-----------------------------------------------------------------------
!
!  Read global atrributes from the static data input file
!  And set dimensions
!
!-----------------------------------------------------------------------

    CALL read_global_attrs(   &
             title, datestr, gridtype,geogrid%dyn_opt,                  &
             west_east_dim,south_north_dim, bottom_top_dim,             &
             we_input_s, we_input_e,we_input_stag_s,we_input_stag_e,    &
             sn_input_s, sn_input_e,sn_input_stag_s,sn_input_stag_e,    &
             geogrid%map_proj,geogrid%mminlu,geogrid%num_land_cat,      &
             geogrid%iswater,geogrid%isice,geogrid%isurban,             &
             geogrid%isoilwater,                                        &
             geogrid%grid_id,geogrid%parent_id,                         &
             geogrid%i_parent_start,geogrid%j_parent_start,             &
             geogrid%i_parent_end,geogrid%j_parent_end,                 &
             geogrid%dx,geogrid%dy,geogrid%cen_lat,geogrid%moad_cen_lat,&
             geogrid%cen_lon,geogrid%stand_lon,                         &
             geogrid%truelat1, geogrid%truelat2, geogrid%pole_lat,      &
             geogrid%pole_lon, geogrid%parent_grid_ratio,               &
             geogrid%corner_lats, geogrid%corner_lons, istatus)

    geogrid%islake = -1
    geogrid%grid_type(1:1) = gridtype(1:1)

    geogrid%west_east_dim   = west_east_dim
    geogrid%south_north_dim = south_north_dim

    geogrid%we_dom_s = 1
    geogrid%sn_dom_s = 1

    IF (gridtype(1:1) == 'C') THEN
       geogrid%we_dom_e = west_east_dim   - 1
       geogrid%sn_dom_e = south_north_dim - 1
       IF (coretype /= 'ARW') THEN
         WRITE(6,'(1x,5a)') 'You are required to process ',coretype,    &
                  ' core, but find "ARW" in file ',TRIM(input_fname),'.'
         CALL arpsstop('ERROR: Wrong type of WRF core',1)
       END IF
    ELSE IF (gridtype(1:1) == 'E') THEN
       geogrid%we_dom_e = west_east_dim
       geogrid%sn_dom_e = south_north_dim
       IF (coretype /= 'NMM') THEN
         WRITE(6,'(1x,5a)') 'You are required to process ',coretype,    &
                  ' core, but find "NMM" in file ',TRIM(input_fname),'.'
         CALL arpsstop('ERROR: Wrong type of WRF core',1)
       END IF
    END IF

    ! Given the full dimensions of this domain, find out the range of indices
    ! that will be worked on by this processor. This information is given by
    ! my_minx, my_miny, my_maxx, my_maxy in module_wrfparallel
    CALL parallel_get_tile_dims(west_east_dim, south_north_dim)

    ! Must figure out patch dimensions from info in parallel module
    IF ( .NOT. do_tiled_input) then

      geogrid%we_patch_s      = my_minx
      geogrid%we_patch_stag_s = my_minx
      geogrid%sn_patch_s      = my_miny
      geogrid%sn_patch_stag_s = my_miny
      geogrid%we_patch_e      = my_maxx - 1
      geogrid%sn_patch_e      = my_maxy - 1

      IF (gridtype(1:1) == 'C') THEN
        if (my_x /= nxprocs - 1) then
           geogrid%we_patch_e      = geogrid%we_patch_e + 1
           geogrid%we_patch_stag_e = geogrid%we_patch_e
        else
           geogrid%we_patch_stag_e = geogrid%we_patch_e + 1
        end if
        if (my_y /= nyprocs - 1) then
           geogrid%sn_patch_e      = geogrid%sn_patch_e + 1
           geogrid%sn_patch_stag_e = geogrid%sn_patch_e
        else
           geogrid%sn_patch_stag_e = geogrid%sn_patch_e + 1
        end if
      ELSE IF (gridtype(1:1) == 'E') THEN
        geogrid%we_patch_e      = geogrid%we_patch_e + 1
        geogrid%sn_patch_e      = geogrid%sn_patch_e + 1
        geogrid%we_patch_stag_e = geogrid%we_patch_e
        geogrid%sn_patch_stag_e = geogrid%sn_patch_e
      END IF
    ELSE
      geogrid%we_patch_s      = we_input_s
      geogrid%we_patch_e      = we_input_e
      geogrid%we_patch_stag_s = we_input_stag_s
      geogrid%we_patch_stag_e = we_input_stag_e
      geogrid%sn_patch_s      = sn_input_s
      geogrid%sn_patch_e      = sn_input_e
      geogrid%sn_patch_stag_s = sn_input_stag_s
      geogrid%sn_patch_stag_e = sn_input_stag_e
    END IF

    ! Compute multipliers for halo width; these must be 0/1
    lh_mult = 1
    if (my_x == 0) lh_mult = 0

    rh_mult = 1
    if (my_x == (nxprocs-1)) rh_mult = 0

    bh_mult = 1
    if (my_y == 0) bh_mult = 0

    th_mult = 1
    if (my_y == (nyprocs-1)) th_mult = 0

    geogrid%we_mem_s = geogrid%we_patch_s - HALO_WIDTH*lh_mult
    geogrid%we_mem_e = geogrid%we_patch_e + HALO_WIDTH*rh_mult
    geogrid%sn_mem_s = geogrid%sn_patch_s - HALO_WIDTH*bh_mult
    geogrid%sn_mem_e = geogrid%sn_patch_e + HALO_WIDTH*th_mult
    geogrid%we_mem_stag_s = geogrid%we_patch_stag_s - HALO_WIDTH*lh_mult
    geogrid%we_mem_stag_e = geogrid%we_patch_stag_e + HALO_WIDTH*rh_mult
    geogrid%sn_mem_stag_s = geogrid%sn_patch_stag_s - HALO_WIDTH*bh_mult
    geogrid%sn_mem_stag_e = geogrid%sn_patch_stag_e + HALO_WIDTH*th_mult

    !
    ! Note: Should reconsider variable phi and lambda.
    !       I suspect it is not right, but just copied from WPS.
    !       However, we will not use the map project in the program
    !       So it is ignore temporarily.
    !
    CALL set_domain_projection(geogrid%map_proj,geogrid%stand_lon,      &
       geogrid%truelat1,geogrid%truelat2,geogrid%dx,geogrid%dy,         &
       geogrid%dx,geogrid%dy,west_east_dim,south_north_dim,             &
       REAL(west_east_dim)/2.,REAL(south_north_dim)/2.,                 &
       geogrid%cen_lat,geogrid%cen_lon,geogrid%cen_lat,geogrid%cen_lon)

!-----------------------------------------------------------------------
!
! Loop over each variable in the static file
!
!-----------------------------------------------------------------------

    istatus = 0
    DO WHILE (istatus == 0)

      CALL read_next_field(sp1,ep1,sp2,ep2,sp3,ep3,cname,cunits,cdesc,  &
                       memorder, stagger, dimnames, real_array,istatus)
      IF (istatus == 0) THEN
        geogrid%nfields = geogrid%nfields + 1
        IF (lvldbg > 0) WRITE(6,'(1x,a,I3,2a)') 'Read in No (',geogrid%nfields, &
                                ') static field - ',TRIM(cname)
        IF (gridtype(1:1) == 'C') THEN
          IF ( TRIM(stagger) == 'M') THEN
            geogrid%fields(geogrid%nfields)%stagger = M
            geogrid%fields(geogrid%nfields)%dimnames(1) = 'west_east'
            geogrid%fields(geogrid%nfields)%dimnames(2) = 'south_north'
            sd1 = geogrid%we_dom_s
            ed1 = geogrid%we_dom_e
            sd2 = geogrid%sn_dom_s
            ed2 = geogrid%sn_dom_e
            sd3 = sp3
            ed3 = ep3
            sm1 = geogrid%we_mem_s
            em1 = geogrid%we_mem_e
            sm2 = geogrid%sn_mem_s
            em2 = geogrid%sn_mem_e
          ELSE IF ( TRIM(stagger) == 'U') THEN
            geogrid%fields(geogrid%nfields)%stagger = U
            geogrid%fields(geogrid%nfields)%dimnames(1) = 'west_east_stag'
            geogrid%fields(geogrid%nfields)%dimnames(2) = 'south_north'
            sd1 = geogrid%we_dom_s
            ed1 = geogrid%we_dom_e + 1
            sd2 = geogrid%sn_dom_s
            ed2 = geogrid%sn_dom_e
            sd3 = sp3
            ed3 = ep3
            sm1 = geogrid%we_mem_stag_s
            em1 = geogrid%we_mem_stag_e
            sm2 = geogrid%sn_mem_s
            em2 = geogrid%sn_mem_e
          ELSE IF ( TRIM(stagger) == 'V') THEN
            geogrid%fields(geogrid%nfields)%stagger = V
            geogrid%fields(geogrid%nfields)%dimnames(1) = 'west_east'
            geogrid%fields(geogrid%nfields)%dimnames(2) = 'south_north_stag'
            sd1 = geogrid%we_dom_s
            ed1 = geogrid%we_dom_e
            sd2 = geogrid%sn_dom_s
            ed2 = geogrid%sn_dom_e + 1
            sd3 = sp3
            ed3 = ep3
            sm1 = geogrid%we_mem_s
            em1 = geogrid%we_mem_e
            sm2 = geogrid%sn_mem_stag_s
            em2 = geogrid%sn_mem_stag_e
          ELSE
            CALL arpsstop('EORRO: Unknown stagger in read_static_fields.',1)
          END IF
        ELSE IF (gridtype(1:1) == 'E') THEN
          geogrid%fields(geogrid%nfields)%dimnames(1) = 'west_east'
          geogrid%fields(geogrid%nfields)%dimnames(2) = 'south_north'
          IF ( TRIM(stagger) == 'M') THEN
            geogrid%fields(geogrid%nfields)%stagger = HH
          ELSE IF ( TRIM(stagger) == 'V') THEN
            geogrid%fields(geogrid%nfields)%stagger = VV
          ELSE
            CALL arpsstop('EORRO: Unknown stagger in read_static_fields.',1)
          END IF
          sd1 = geogrid%we_dom_s
          ed1 = geogrid%we_dom_e
          sd2 = geogrid%sn_dom_s
          ed2 = geogrid%sn_dom_e
          sd3 = sp3
          ed3 = ep3
          sm1 = geogrid%we_mem_s
          em1 = geogrid%we_mem_e
          sm2 = geogrid%sn_mem_s
          em2 = geogrid%sn_mem_e
        END IF

        IF (ed3 > sd3) THEN
          dim3 = ed3-sd3+1

          SELECT CASE (dim3)
          CASE (12, 16, 20, 24)
            WRITE(geogrid%fields(geogrid%nfields)%dimnames(3),'(a,I4.4)') 'z-dimension',dim3
          CASE DEFAULT
            WRITE(6,'(1x,a,I4)') 'Unsupported thrid dimenion size = ',dim3
            CALL arpsstop('ERROR: 3rd dimension.',1)
          END SELECT
        END IF

        ALLOCATE(geogrid%fields(geogrid%nfields)%rdata_arr(sm1:em1,sm2:em2,sp3:ep3), &
                                                                   STAT = istatus )
        geogrid%fields(geogrid%nfields)%rdata_arr(sp1:ep1,sp2:ep2,sp3:ep3)           &
               = real_array(sp1:ep1,sp2:ep2,sp3:ep3)

        CALL exchange_halo_r(geogrid%fields(geogrid%nfields)%rdata_arr,              &
               sm1, em1, sm2, em2, sp3, ep3, sp1, ep1, sp2, ep2, sp3, ep3)

!        geogrid%fields(geogrid%nfields)%dimnames(:) = dimnames(:)
        geogrid%fields(geogrid%nfields)%fieldname   = cname
        geogrid%fields(geogrid%nfields)%units       = cunits
        geogrid%fields(geogrid%nfields)%descr       = cdesc
        geogrid%fields(geogrid%nfields)%mem_order   = memorder
        geogrid%fields(geogrid%nfields)%dom_start(1)= sd1
        geogrid%fields(geogrid%nfields)%dom_start(2)= sd2
        geogrid%fields(geogrid%nfields)%dom_start(3)= sd3
        geogrid%fields(geogrid%nfields)%dom_end(1)  = ed1
        geogrid%fields(geogrid%nfields)%dom_end(2)  = ed2
        geogrid%fields(geogrid%nfields)%dom_end(3)  = ed3
        geogrid%fields(geogrid%nfields)%mem_start(1)= sm1
        geogrid%fields(geogrid%nfields)%mem_start(2)= sm2
        geogrid%fields(geogrid%nfields)%mem_start(3)= sp3
        geogrid%fields(geogrid%nfields)%mem_end(1)  = em1
        geogrid%fields(geogrid%nfields)%mem_end(2)  = em2
        geogrid%fields(geogrid%nfields)%mem_end(3)  = ep3
        geogrid%fields(geogrid%nfields)%patch_start(1)= sp1
        geogrid%fields(geogrid%nfields)%patch_start(2)= sp2
        geogrid%fields(geogrid%nfields)%patch_start(3)= sp3
        geogrid%fields(geogrid%nfields)%patch_end(1)  = ep1
        geogrid%fields(geogrid%nfields)%patch_end(2)  = ep2
        geogrid%fields(geogrid%nfields)%patch_end(3)  = ep3

        IF (lvldbg > 1) WRITE(6,'(1x,a,6(a,I4),/,2(13x,6(a,I4),/))') 'Dimensions: ',    &
        'Domain - ',sd1,' - ',ed1,', ',sd2,' - ',ed2,', ',sd3,' - ',ed3,&
        'Memory - ',sm1,' - ',em1,', ',sm2,' - ',em2,', ',sp3,' - ',ep3,&
        'Patch  - ',sp1,' - ',ep1,', ',sp2,' - ',ep2,', ',sp3,' - ',ep3

        DEALLOCATE(real_array)   ! Memory leak in WPS

        ! Save special variable location

        IF (INDEX(cname, 'XLAT_M') /= 0 .AND.                           &
            LEN_TRIM(cname) == LEN_TRIM('XLAT_M') ) THEN
          geogrid%xlat_m_no = geogrid%nfields
        ELSE IF (INDEX(cname, 'XLONG_M') /= 0 .AND.                     &
                 LEN_TRIM(cname) == LEN_TRIM('XLONG_M') ) THEN
          geogrid%xlong_m_no = geogrid%nfields
        ELSE IF (INDEX(cname, 'XLAT_U') /= 0 .AND.                      &
                 LEN_TRIM(cname) == LEN_TRIM('XLAT_U') ) THEN
          geogrid%xlat_u_no = geogrid%nfields
        ELSE IF (INDEX(cname, 'XLONG_U') /= 0 .AND.                     &
                 LEN_TRIM(cname) == LEN_TRIM('XLONG_U') ) THEN
          geogrid%xlong_u_no = geogrid%nfields
        ELSE IF (INDEX(cname, 'XLAT_V') /= 0 .AND.                      &
                 LEN_TRIM(cname) == LEN_TRIM('XLAT_V') ) THEN
          geogrid%xlat_v_no = geogrid%nfields
        ELSE IF (INDEX(cname, 'XLONG_V') /= 0 .AND.                     &
                 LEN_TRIM(cname) == LEN_TRIM('XLONG_V') ) THEN
          geogrid%xlong_v_no = geogrid%nfields
        ELSE IF (INDEX(cname, 'LANDMASK') /= 0 .AND.                    &
                 LEN_TRIM(cname) == LEN_TRIM('LANDMASK') ) THEN
          geogrid%landmask_no = geogrid%nfields
        ELSE IF (INDEX(cname, 'LU_INDEX') /= 0 .AND.                    &
                 LEN_TRIM(cname) == LEN_TRIM('LU_INDEX') ) THEN
          geogrid%luindex_no = geogrid%nfields
        ELSE IF (INDEX(cname, 'HGT_M') /= 0 .AND.                    &
                 LEN_TRIM(cname) == LEN_TRIM('HGT_M') ) THEN
          geogrid%hgt_m_no = geogrid%nfields
        ELSE IF (INDEX(cname, 'HGT_U') /= 0 .AND.                    &
                 LEN_TRIM(cname) == LEN_TRIM('HGT_U') ) THEN
          geogrid%hgt_u_no = geogrid%nfields
        ELSE IF (INDEX(cname, 'HGT_V') /= 0 .AND.                    &
                 LEN_TRIM(cname) == LEN_TRIM('HGT_V') ) THEN
          geogrid%hgt_v_no = geogrid%nfields
        END IF

      END IF
    END DO

    CALL flush(6)

    IF (gridtype(1:1) == 'E') THEN     ! E grid does not distinguish U and V
      geogrid%xlat_u_no  = geogrid%xlat_v_no
      geogrid%xlong_u_no = geogrid%xlong_v_no
    END IF

    CALL input_close(istatus)

    RETURN
  END SUBROUTINE read_static_fields

END MODULE module_geogrid
