#!/usr/bin/env bash

#
# generic_init.sh
# A SystemV init style script that can be used for any command.
# 
# Jeremy Fee <jmfee@usgs.gov>
# Version 0.3 2008/11/17
#

if [ $# -lt 3 ]; then
	echo "Usage: $0 action name command [workdir]"
	echo "	action		{start|stop|restart|reload|status}"
	echo "	name		program name"
	echo "	command		program start command"
	echo "	workdir		program working directory, default is CWD"
	echo
	exit 1
fi


# set the umask for any started programs
umask 022


ACTION=$1
NAME=$2
COMMAND=$3
#default working directory is CWD
WORKDIR=${4:-`pwd`}

#exit status is global and defaults to success
# any start/stop failure will adjust status
# and is never reset to 0.
EXIT_STATUS=0


PS_COMMAND="ps"
PS_FLAGS="aux"
PS_GREP_SEARCH="$COMMAND"


#bsd workaround
#ps does not output beyond 80 columns, by default, which prevents ps from matching most commands
export COLUMNS=1024

#solaris workarounds
OS_NAME=`uname`
if [ "$OS_NAME" == "SunOS" ]; then
	if [ -x "/bin/ps" ]; then
		# Solaris ps truncates command output at 80 characters.
		# Commands *MUST* be unique within the first 80 characters.
		PS_COMMAND="/bin/ps"
		PS_FLAGS="-ef"
		if [ ${#PS_GREP_SEARCH} -gt 79 ]; then
			PS_GREP_SEARCH=${PS_GREP_SEARCH:0:79}
		fi
	fi
fi


start() {
	local pid=`get_pid $COMMAND`
	if [ $pid != 0 ]; then
		echo "$NAME already running (pid=$pid)"
	else
		echo -n "Starting $NAME - "

		##run command
		pushd $WORKDIR > /dev/null
		nohup $COMMAND > /dev/null 2>&1 &
		popd > /dev/null

		##wait a second and see if is running
		sleep 1
		pid=`get_pid $COMMAND`
		if [ $pid != 0 ]; then
			#running
			echo "OK (pid=$pid)"
		else
			#not running
			echo "ERROR, unable to start $NAME using command:"
			echo "	$COMMAND"
			EXIT_STATUS=2
		fi
	fi
}

stop() {
	local pid=`get_pid $COMMAND`
	if [ $pid = 0 ]; then
		echo "$NAME not running"
	else
		echo -n "Stopping $NAME (pid=$pid) - "
		kill $pid

		##wait up to 5 seconds for stop
		pid=`get_pid $COMMAND`
		seconds_waited=0
		while [ $pid != 0 ] && [ $seconds_waited -lt 5 ]; do
			sleep 1
			echo -n "."
			let seconds_waited=seconds_waited+1
			pid=`get_pid $COMMAND`
		done

		if [ $pid = 0 ]; then
			#not running
			echo " OK"
		else
			#still running
			echo " ERROR (pid=$pid)"
			EXIT_STATUS=3
		fi
	fi
}

get_pid() {
	## Find PID
	## ps aux appears consisent for both BSD and LINUX
	##     output pid in column 2 (for awk)
	##     and include the command (for greps)
	#
	# list processes
	#     ps aux
	# remove this init script process (has command in arguments)
	#     grep -v 'generic_init'
	# keep processes that match $command
	#     grep $command
	# remove grep processes
	#     grep -v grep"
	# output PID, from column 2
	#     awk '{print $2}'
	
	local pid=`${PS_COMMAND} ${PS_FLAGS} | grep -v 'generic_init' | grep "${PS_GREP_SEARCH}" | grep -v grep | awk '{print \$2}'`
	if [ ! -z "$pid" ]; then
		echo "$pid"
	else
		echo "0"
	fi
}



#init logic
case "$ACTION" in
	start)
		start
		;;

	stop)
		stop
		;;

	restart|reload|condrestart)
		stop
		start
		;;

	status)
		pid=`get_pid`
		if [ $pid = 0 ]; then
			echo "$NAME not running"
			EXIT_STATUS=1
		else
			echo "$NAME running (pid=$pid)"
		fi
		;;

	*)
		echo "Usage: $0 {start|stop|restart|reload|condrestart|status}"
		exit 1
		;;
esac



exit $EXIT_STATUS
