
## Copyright (c) 2001-2012, Scott D. Peckham

## October 2012  (CSDMS Standard Names with BMI)
## January 2009  (converted from IDL)
## May, July, August, October 2009
## May 2010 (changes to initialize() and read_cfg_file()

## NB!  TF expects d8.codes vs. d8.flow_grid

#############################################################
## Note:  channel_base.py calls many of these but then
##        saves the results among its state variables.
#############################################################
## Note:  It might be more clear to refer to flow_grids
##        and flow_codes as d8_grids and d8_codes, etc.
##        (e.g. d8_code_list, d8_code_map, d8_width_grid)
#############################################################

from numpy import *
import numpy

import os, os.path
import time

import BMI_base
import cfg_files as cfg
import pixels
import rtg_files
import tf_utils

from model_output import *

#---------------------------------------------------------------------
#
#   unit_test()
#
#   class d8_base               # (inherits from BMI_base)
#
#       get_attribute()
#       get_input_var_names()
#       get_output_var_names()
#       get_var_names()
#       get_var_units()
#----------------------------------
#       set_constants()
#       initialize()
#       update()
#       read_cfg_file()           ####  not finished yet  ####
#       set_computed_input_vars()
#       initialize_computed_vars()
#----------------------------------
#       get_pixel_dimensions()
#       get_flow_code_list()
#       get_flow_code_list_opps()
#       get_valid_code_map()
#       get_ID_grid()
#       get_parent_inc_map()
#       get_edge_IDs()
#       get_not_edge_grid()
#----------------------------------
#       update_parent_ID_grid()     # (can handle periodic BCs) 
#       update_parent_IDs()         # (used by erosion_base.update_slope_grid())
#       update_non_parent_IDs()     # (not working or needed yet)
#       update_flow_from_IDs()
#       update_flow_to_IDs()
#       update_noflow_IDs()
#----------------------------------
#       read_flow_grid()
#       update_flow_grid()
#          start_new_flow_grid()
#          resolve_array_cycle()
#          get_resolve_array()
#          break_flow_grid_ties()
#          link_flats()
#----------------------------------
#       update_flow_width_grid()
#       update_flow_length_grid()
#       update_area_grid()          # (added on 10/28/09)
#       update_area_grid_OLD()      # (can't handle periodic BCs)
#----------------------------------
#       get_flow_width_grid()       # OBSOLETE ?
#       get_flow_length_grid()      # OBSOLETE ?

#-----------------------------------------------------------------------
def unit_test(SILENT=False, REPORT=True):

    #---------------------------------------------------------------
    # NOTE! The tests will appear to fail if the existing flow
    #       grid used for comparison was computed using a flat
    #       resolution method other than "Iterative linking".
    #
    # The KY_Sub and Beaver DEMs were processed using RiverTools
    # 3.0 using the WGS_1984 ellipsoid model for computing lengths
    # and areas.  The "Iterative linking" method was used for both
    # as the "Flat resolution method", to make them comparable to
    # the ones generated by functions in d8_base.py and
    # fill_pits.py.  Older version of these data sets used other
    # methods and can't be compared directly.
    #
    # Make sure that LINK_FLATS=True, LR_PERIODIC=False, and
    # TB_PERIODIC=False in CFG file.
    #---------------------------------------------------------------
    start = time.time()

    #------------------------------------------------------
    # Example of DEM with fixed-angle pixels (Geographic)
    #     min(da) = 6802.824074169645  [m^2]
    #     max(da) = 6837.699120083246  [m^2]
    #     min(A)  =    0.000000000000  [km^2]
    #     max(A)  =  807.063354492188  [km^2]
    #------------------------------------------------------
##    directory   = '/Applications/Erode/Data/KY_Sub2/'
##    site_prefix = 'KY_Sub'
##    case_prefix = 'Test1'

    #------------------------------------------------
    # Example of DEM with fixed-length pixels (UTM)
    #     min(da) = 900.000  [m^2]
    #     max(da) = 900.000  [m^2]
    #     min(A)  =   0.000000000000  [km^2]
    #     max(A)  = 681.914184570312  [km^2]
    #------------------------------------------------
##    directory   = '/Applications/Erode/Data/Beaver2/'
##    site_prefix = 'Beaver'
##    case_prefix = 'Test1'

    #-----------------------------------------
    # This function adjusts for the platform
    # and can be changed in "tf_utils.py".
    #-------------------------------------------------
    # NOTE: The Treynor_Iowa DEM has no depressions!
    #------------------------------------------------- 
##    in_directory = tf_utils.TF_Test_Directory()

    d8 = d8_component()

    d8.CCA    = False
    d8.DEBUG  = True # (check flow and area grid against existing)
    d8.SILENT = SILENT
    d8.REPORT = REPORT

    cfg_prefix = 'Test1'
    d8.site_prefix = 'KY_Sub'
    
    d8.initialize(cfg_prefix=cfg_prefix, mode="driver",
                  SILENT=SILENT, REPORT=REPORT)

    d8.A_units = 'km^2'  # (override setting in CFG file)
    d8.update(SILENT=SILENT, REPORT=REPORT)
    print 'grid nx =', d8.nx
    print 'grid ny =', d8.ny
    print 'Run time =', (time.time() - start), ' [secs]'
    print 'Finished with unit_test().'
    print ' '
                
#   unit_test()
#---------------------------------------------------------------------
class d8_component(BMI_base.BMI_component):

    #-------------------------------------------------------------------
    _att_map = {
        'model_name':         'd8_component class',  ###########
        'version':            '3.1',
        'author_name':        'Scott D. Peckham',
        'grid_type':          'uniform',
        'time_step_type':     'fixed',
        'step_method':        'explicit',  ##### or 'none' ??
        #------------------------------------------------------
        'comp_name':          'TF_D8_component',
        'model_family':       'TopoFlow',
        'cfg_template_file':  'None',
        'cfg_extension':      '_d8.cfg',
        'cmt_var_prefix':     'None',
        'gui_xml_file':       'None',
        'dialog_title':       'None',
        'time_units':         'seconds' }

    _input_var_names = ['land_surface__elevation']

    #------------------------------------------
    # Maybe use "step" instead of "increment"
    # and/or incorporate "grid_cell".
    #------------------------------------------
    _output_var_names = [
        'land_surface__d8_total_contributing_area',   ## 'A'
        'land_surface__d8_flow_direction_code',       ## 'flow_grid'
        'land_surface__d8_flow_length_increment',     ## 'ds'
        'land_surface__d8_flow_width_increment' ]     ## 'dw'

##        'ID_grid' = "grid_cell__row_major_id_number"
        
##        'ID_grid', 'parent_IDs', 'edge_IDs', 'noflow_IDs',
##        'w1', 'w2', 'w3', 'w4', 'w5', 'w6', 'w7', 'w8',
##        'p1', 'p2', 'p3', 'p4', 'p5', 'p6', 'p7', 'p8'
        ## nx, ny, dx, dy, dd, da

    _var_name_map = {
        'land_surface__elevation':'DEM',
        'land_surface__d8_total_contributing_area':'area_grid', ## CHECK
        'land_surface__d8_flow_direction_code':'flow_grid',
        'land_surface__d8_flow_length_increment':'ds',
        'land_surface__d8_flow_width_increment':'dw' }

    _var_units_map = {
        'land_surface__elevation':'m',
        'land_surface__d8_total_contributing_area':'m2',  ## CHECK
        'land_surface__d8_flow_direction_code':'1',
        'land_surface__d8_flow_length_increment':'m',
        'land_surface__d8_flow_width_increment':'m' }        
    
    #------------------------------------------------    
    # Return NumPy string arrays vs. Python lists ?
    #------------------------------------------------
    ## _input_var_names  = numpy.array( _input_var_names )
    ## _output_var_names = numpy.array( _output_var_names )
    
    #-------------------------------------------------------------------
    def get_attribute(self, att_name):

        #----------------
        # New. 10/26/11
        #----------------
##        map = {'comp_name':          'TF_D8_component',
##               'version':            '3.1',
##               'model_name':         'd8_component class',
##               'model_family':       'TopoFlow',
##               'cfg_template_file':  'None',
##               'cfg_extension':      '_d8.cfg',
##               'cmt_var_prefix':     'None',
##               'gui_xml_file':       'None',
##               'dialog_title':       'None',
##               'time_step_type':     'fixed',
##               'time_units':         'seconds',
##               'grid_type':          'uniform',
##               'author_name':        'Scott Peckham'}

        try:
            return self._att_map[ att_name.lower() ]
        except:
            print '###################################################'
            print ' ERROR: Could not find attribute: ' + att_name
            print '###################################################'
            print ' '

    #   get_attribute()
    #-------------------------------------------------------------------
    def get_input_var_names(self):

        return self._input_var_names

    #   get_input_var_names()
    #-------------------------------------------------------------------
    def get_output_var_names(self):

        return self._output_var_names
    
    #   get_output_var_names()
    #-------------------------------------------------------------------
    def get_var_name(self, long_var_name):

        return self._var_name_map[ long_var_name ]
    
    #   get_var_name()
    #-------------------------------------------------------------------
    def get_var_units(self, long_var_name):

        return self._var_units_map[ long_var_name ]
    
    #   get_var_units()
    #-------------------------------------------------------------------
    def set_constants(self):

        #------------------------
        # Define some constants
        #------------------------
        self.dt     = 1.0   # (needs to be defined)
        self.nodata = -float32(9999)

    #   set_constants()
    #-----------------------------------------------------------------
    def initialize(self, cfg_prefix='Case1', mode="nondriver",
                   SILENT=False, REPORT=True):

        #--------------------------------------------------------
        # Note:  This function calls functions that compute a
        #        variety of D8 variables and saves them in its
        #        state.  This "d8 state" can be embedded within
        #        another class such as the "channels_base" or
        #        "erosion_base" class.
        #--------------------------------------------------------
        if not(SILENT):
            print ' '
            print 'D8 component: Initializing...'
            
        ## start_init = time.time()
        
        self.status = 'initializing'
        self.mode   = mode

        #------------------------------------------------------
        # Note: A run_model() call or a driver's initialize()
        #       call calling initialize_config_vars() will set
        #       CWD to the location of CFG files.
        #------------------------------------------------------
        # Note: If directories and prefixes are not set in
        #       initialize_config_vars(), then they will
        #       default to CWD and cfg_prefix.
        #------------------------------------------------------
        cfg_extension   = self.get_cfg_extension()
        filename        = cfg_prefix + cfg_extension
        self.cfg_file   = os.path.join( os.getcwd(), filename )
        self.cfg_prefix = cfg_prefix
        
        #-----------------------------------------------
        # Load component parameters from a config file
        #-----------------------------------------------
        self.set_constants()
        self.initialize_config_vars() 
        self.read_grid_info() # (also gets & stores self.da)
        #-----------------------------------------
        # This must come before "Disabled" test.
        #-----------------------------------------
        self.initialize_time_vars()
        
        #--------------------------------------------
        # Convert units for da from m^2 to km^2 ??
        #--------------------------------------------
        # Better to do this in "update_area_grid()"
        # See "unit_test()".
        #--------------------------------------------        
##        if ('km' in self.A_units.lower()):
##            self.da = self.da / 1e6
  
        #-----------------------------------------------
        # These return values that don't depend on the
        # flow grid and don't change, so they should
        # simply be stored for subsequent use.
        #-----------------------------------------------
        # Values that do depend on the flow grid are
        # computed by calls in the update() method.
        #-----------------------------------------------        
        self.get_pixel_dimensions(SILENT=SILENT, REPORT=REPORT)
        self.get_flow_code_list()
        self.get_flow_code_list_opps()
        self.get_ID_grid(SILENT=SILENT)
        self.get_parent_inc_map()
        self.get_edge_IDs(SILENT=SILENT)
        self.get_not_edge_grid(SILENT=SILENT)
        #-------------------------------------------------
        self.get_resolve_array(SILENT=SILENT)   ######
        self.get_valid_code_map(SILENT=SILENT)
        
        #-----------------------------
        # Initialize dw, ds, A, etc.
        #-----------------------------
        self.initialize_computed_vars()
        ## self.initialize_required_components(mode)   ##### NOT READY YET ###
        
        self.status = 'initialized'
        
##        finish_init = time.time()
##        run_time = (finish_init - start_init)
##        print 'Run time for initialize =', run_time, ' [secs]'
        
    #   initialize()
    #-------------------------------------------------------------------
    def update(self, time=None, DEM=None,
               SILENT=True, REPORT=False):

        self.status = 'updating'  # (OpenMI 2.0 convention)
        if not(SILENT):
            print 'D8 component: Updating...'
        # if (self.mode == 'driver'):
        #     self.print_time_and_value(self.z_outlet, 'z_out', '[m]')
        
        #-------------------------
        # Update computed values
        #-------------------------
        ## fill_pits.fill_pits_in_dem()   ## pass DEM to here? ## 

        self.update_flow_grid(DEM, SILENT=SILENT, REPORT=REPORT)
        self.update_parent_ID_grid()
        self.update_parent_IDs()     # (needed for gradients)
        self.update_flow_from_IDs()
        self.update_flow_to_IDs()
        self.update_noflow_IDs()

        # OLD_WAY = True   # (Treynor run time = 0.11 seconds)
        OLD_WAY = False    # (Treynor run time = 0.092 seconds)
        if not(OLD_WAY):
            self.update_flow_width_grid(SILENT=SILENT, REPORT=REPORT)
            self.update_flow_length_grid(SILENT=SILENT, REPORT=REPORT)
            self.update_area_grid(SILENT=SILENT, REPORT=REPORT)
        else:
            self.get_flow_width_grid(SILENT=SILENT, REPORT=REPORT)
            self.get_flow_length_grid(SILENT=SILENT, REPORT=REPORT)

        #-------------------------------------------
        # Read from files as needed to update vars 
        #-------------------------------------------
        # if (self.time_index > 0):
        #     self.read_input_files()

        #----------------------------------------------
        # Use this for saving D8 flow and area grids
        #------------------------------------------------
        # Write user-specified data to output files ?
        #------------------------------------------------
##        self.write_output_files( time )

        #------------------------
        # Check computed values
        #------------------------
##        if (OK):
##            self.status = 'updated'  # (OpenMI 2.0 convention)
##        else:
##            self.status = 'failed'
##            self.DONE   = True

        #------------------------
        # Update internal clock
        #------------------------
        self.update_time()
        self.status = 'updated'  # (OpenMI 2.0 convention)
            
    #   update()
    #-------------------------------------------------------------------
    def get_cfg_extension(self):

        return '_d8.cfg'
    
    #   get_cfg_extension() 
    #---------------------------------------------------------------------
    def read_cfg_file(self):

        #------------------------------------------
        # Read parameters from a CFG file that is
        # in the current working directory.
        #------------------------------------------
        print 'D8 component: Reading config file...'
        file_unit = open(self.cfg_file, 'r')
        
        #-------------------------
        # Skip over header lines
        #-------------------------
        cfg.skip_header( file_unit, n_lines=4 )
            
        #------------------------
        # Read the channel vars
        #------------------------
        self.method      = cfg.read_value( file_unit, dtype='int16' )
        self.method_name = cfg.read_value( file_unit, dtype='string' )
        #---------------------------------------------------------------       
        dt_info          = cfg.read_input_option( file_unit )
        # self.dt_type   = dt_info[0]
        self.dt          = dt_info[1]
        #-----------------------------------------------------------
        #  Read:  LR_PERIODIC, TB_PERIODIC, DEM_file_name, etc.
        #-----------------------------------------------------------
        self.DEM_file    = cfg.read_value(file_unit, dtype='string')
        self.A_units     = cfg.read_value(file_unit, dtype='string')
        self.A_units     = self.A_units.lower()
        self.LINK_FLATS  = cfg.read_value(file_unit, dtype='boolean')
        self.LR_PERIODIC = cfg.read_value(file_unit, dtype='boolean')
        self.TB_PERIODIC = cfg.read_value(file_unit, dtype='boolean')
        #---------------------------------------------------------------
        if (self.LR_PERIODIC and self.TB_PERIODIC):
            self.ALL_PERIODIC = True
        #--------------------------------------------------------------------
        save_grid_dt_info             = cfg.read_input_option( file_unit )
        save_area_grids, area_gs_file = cfg.read_output_option( file_unit )
        save_code_grids, code_gs_file = cfg.read_output_option( file_unit )
        save_ds_grids,   ds_gs_file   = cfg.read_output_option( file_unit )
        save_dw_grids,   dw_gs_file   = cfg.read_output_option( file_unit )
        #--------------------------------------------------------------------
        self.save_grid_dt    = save_grid_dt_info[1]
        self.SAVE_AREA_GRIDS = save_area_grids
        self.SAVE_CODE_GRIDS = save_code_grids
        self.SAVE_DS_GRIDS   = save_ds_grids
        self.SAVE_DW_GRIDS   = save_dw_grids
        self.area_gs_file    = area_gs_file
        self.code_gs_file    = code_gs_file
        self.ds_gs_file      = ds_gs_file
        self.dw_gs_file      = dw_gs_file
        #-----------------------------------------------------------------------
        save_pixels_dt_info            = cfg.read_input_option( file_unit )
        save_area_pixels, area_ts_file = cfg.read_output_option( file_unit )
        save_code_pixels, code_ts_file = cfg.read_output_option( file_unit )
        save_ds_pixels,   ds_ts_file   = cfg.read_output_option( file_unit )
        save_dw_pixels,   dw_ts_file   = cfg.read_output_option( file_unit )
        #-----------------------------------------------------------------------
        self.save_pixels_dt    = save_pixels_dt_info[1]
        self.SAVE_AREA_PIXELS  = save_area_pixels
        self.SAVE_CODE_PIXELS  = save_code_pixels
        self.SAVE_DS_PIXELS    = save_ds_pixels
        self.SAVE_DW_PIXELS    = save_dw_pixels
        self.code_ts_file      = code_ts_file
        self.area_ts_file      = area_ts_file
        self.ds_ts_file        = ds_ts_file
        self.dw_ts_file        = dw_ts_file

        #-----------------------
        # Close the config file
        #-----------------------
        file_unit.close()

        #---------------------------------------------------------
        # Make sure that all "save_dts" are larger or equal to
        # the specified process dt.  There is no point in saving
        # results more often than they change.
        # Issue a message to this effect if any are smaller ??
        #---------------------------------------------------------
        self.save_grid_dt   = maximum(self.save_grid_dt,   self.dt)
        self.save_pixels_dt = maximum(self.save_pixels_dt, self.dt)

    #   read_cfg_file()
    #-------------------------------------------------------------------
    def set_computed_input_vars(self):

        self.ALL_PERIODIC = (self.LR_PERIODIC and self.TB_PERIODIC)
            
        #---------------------------------------------------------
        # Make sure that all "save_dts" are larger or equal to
        # the specified process dt.  There is no point in saving
        # results more often than they change.
        # Issue a message to this effect if any are smaller ??
        #---------------------------------------------------------
        self.save_grid_dt   = maximum(self.save_grid_dt,   self.dt)
        self.save_pixels_dt = maximum(self.save_pixels_dt, self.dt)
        
    #   set_computed_input_vars()
    #-------------------------------------------------------------------
    def embed_child_components(self):

        #------------------------------------------------
        # Instantiate and embed "process components"
        # in the place of the CCA ports.
        #------------------------------------------------
        import erosion_base
        self.ep = erosion_base.erosion_component()

    #   embed_child_components()
    #-------------------------------------------------------------------
    def add_child_ports(self):

        #-------------------------------------------------------
        # Note: Erosion component does not have any ports yet.
        #-------------------------------------------------------
        pass

    #   add_child_ports()
    #-------------------------------------------------------------------
    def initialize_ports(self):

        #-------------------------------------------------
        # Initialize the process objects/components
        # This is also where output files are opened.
        #-------------------------------------------------
        DEBUG = True
        if (self.ep.get_status() != 'initialized'):        # erosion vars
            self.ep.initialize( sync_file=self.sync_file )
            if (DEBUG): print '\nD8 component initialized EROSION.'

    #   initialize_ports()            
    #---------------------------------------------------------------------
    def initialize_computed_vars(self, DOUBLE=False):

        nx = self.nx  # (Local synonyms)
        ny = self.ny

        if (DOUBLE):
            self.dw = zeros([ny, nx], dtype='Float64')
            self.ds = zeros([ny, nx], dtype='Float64')
            self.A  = zeros([ny, nx], dtype='Float64')
        else:
            self.dw = zeros([ny, nx], dtype='Float32')
            self.ds = zeros([ny, nx], dtype='Float32')
            self.A  = zeros([ny, nx], dtype='Float32')

    #   initialize_computed_vars() 
    #---------------------------------------------------------------------
    def get_pixel_dimensions(self, DOUBLE=False,
                             SILENT=True, REPORT=False):

        if not(SILENT):
            print 'Computing pixel dimensions...'

        dx, dy, dd = pixels.get_sizes_by_row(self.rti, METERS=True)
        self.dx = dx
        self.dy = dy
        self.dd = dd

        if not(DOUBLE):
            self.dx = float32(self.dx)
            self.dy = float32(self.dy)
            self.dd = float32(self.dd)

        #------------------------------------------------
        # Get grid cell areas, "da", which is either a
        # scalar (if same for all grid cells) or a grid
        #------------------------------------------------
        # self.da = pixels.get_da( self.rti )
        #---------------------------------------
        # Note that "da" is saved into self by
        # self.read_grid_info().
        #---------------------------------------
        if (REPORT):
            w = 8
            dx_str  = str(self.dx.min()).ljust(w) + ', '
            dx_str += str(self.dx.max()).ljust(w) + ' [m]'
            print '    min(dx), max(dx) = ' + dx_str
            #--------------------------------------------------           
            dy_str  = str(self.dy.min()).ljust(w) + ', '
            dy_str += str(self.dy.max()).ljust(w) + ' [m]'
            print '    min(dy), max(dy) = ' + dy_str
            #--------------------------------------------------
            dd_str  = str(self.dd.min()).ljust(w) + ', '
            dd_str += str(self.dd.max()).ljust(w) + ' [m]'
            print '    min(dd), max(dd) = ' + dd_str
            #--------------------------------------------------
            da_str  = str(self.da.min()).ljust(w) + ', '
            da_str += str(self.da.max()).ljust(w) + ' [m^2]'
            print '    min(da), max(da) = ' + da_str
      
##            print '    min(dx) =', self.dx.min(),  ' [m]'
##            print '    max(dx) =', self.dx.max(),  ' [m]'
##            #------------------------------------------------
##            print '    min(dy) =', self.dy.min(),  ' [m]'
##            print '    max(dy) =', self.dy.max(),  ' [m]'
##            #------------------------------------------------
##            print '    min(dd) =', self.dd.min(),  ' [m]'
##            print '    max(dd) =', self.dd.max(),  ' [m]'
##            #------------------------------------------------
##            print '    min(da) =', self.da.min(),  ' [m^2]'
##            print '    max(da) =', self.da.max(),  ' [m^2]'

    #   get_pixel_dimensions()
    #---------------------------------------------------------------------
    def get_flow_code_list(self, ARC=False):

    #-------------------------------------------
    # Notes: RT flow codes  = | 64 128 1 |
    #                         | 32  x  2 |
    #                         | 16  8  4 |

    #        ARC/INFO codes = | 32 64 128 |
    #                         | 16  x   1 |
    #                         |  8  4   2 |
    #-------------------------------------------
        if not(ARC):   
            self.code_list = int16([1, 2, 4, 8, 16, 32, 64, 128])
        else:    
            self.code_list = int16([128, 1, 2, 4, 8, 16, 32, 64])
        
    #   get_flow_code_list()
    #---------------------------------------------------------------------
    def get_flow_code_list_opps(self, ARC=False):

        if not(ARC):    
            self.code_opps = int16([16, 32, 64, 128, 1, 2, 4, 8])
        else:    
            self.code_opps = int16([8, 16, 32, 64, 128, 1, 2, 4])
        
    #   get_flow_code_list_opps()
    #---------------------------------------------------------------------
    def get_valid_code_map(self, SILENT=True):

        #----------------------------------------------------------------
        # Notes: This map is used near the end of update_flow_grid()
        #        to set any invalid flow code to 0, which signifies
        #        the the flow direction for that grid cell is undefined.
        #----------------------------------------------------------------
        self.valid_code_map = zeros([256], dtype='UInt8')
        self.valid_code_map[ self.code_list ] = numpy.uint8( self.code_list )

    #   get_valid_code_map()
    #---------------------------------------------------------------------
    def get_ID_grid(self, SILENT=True):

        #-----------------------------------------------------
        # Get a grid which for each grid cell contains the
        # calendar-style index (or ID) of that grid cell.
        #-----------------------------------------------------
        if not(SILENT):
            print 'Computing pixel IDs...'
        
        nx = self.nx
        ny = self.ny
        self.ID_grid = reshape(arange(nx*ny, dtype='Int32'), [ny, nx])
        
    #   get_ID_grid() 
    #---------------------------------------------------------------------
    def get_parent_inc_map(self):

        #-----------------------------------------
        # Note: parent_ID = ID + incs[flow_code].
        #-----------------------------------------
        nx = self.nx
        incs = int32(array([-nx + 1, 1, nx + 1, nx, nx - 1,
                            -1, -nx - 1, -nx]))
        
        MAP = zeros([129], dtype='Int32')
        MAP[self.code_list] = incs
        
        self.inc_map = MAP
        
    #   get_parent_inc_map()
    #-------------------------------------------------------------------
    def get_edge_IDs(self, SILENT=True):

        if not(SILENT):
            print 'Computing edge pixel IDs...'

        #------------------------------------------
        # Get IDs of edge pixels, making sure not
        # to double-count the corner pixels
        #------------------------------------------
        nx    = self.nx
        ny    = self.ny
        T_IDs = arange(nx, dtype='Int32')
        B_IDs = T_IDs + (ny - 1) * nx
        L_IDs = (1 + arange(ny - 2, dtype='Int32')) * nx
        R_IDs = L_IDs + (nx - 1)
        edge_IDs = concatenate([T_IDs, B_IDs, L_IDs, R_IDs])

        #-------------------------------------------
        # Save IDs as a tuple of row indices and
        # calendar indices, "numpy.where" style
        #-------------------------------------------        
        self.edge_IDs = (edge_IDs / nx, edge_IDs % nx)   ##  NB! (row, col)

        #------------------------------------------
        # Save IDs as a 1D array of long-integer,
        # calendar-style indices
        #------------------------------------------
        # self.edge_IDs = edge_IDs
        
    #   get_edge_IDs()
    #---------------------------------------------------------------------
    def get_not_edge_grid(self, SILENT=True):

        if not(SILENT):
            print 'Computing "not_edge" grid...'

        self.not_edge_grid = numpy.ones([self.ny, self.nx],
                                        dtype='UInt8')
        self.not_edge_grid[ self.edge_IDs ] = 0
        
##        self.not_edge_grid[:, 0]      = 0
##        self.not_edge_grid[:, nx - 1] = 0
##        self.not_edge_grid[0, :]      = 0
##        self.not_edge_grid[ny - 1, :] = 0

    #   get_not_edge_grid()
    #---------------------------------------------------------------------
    def update_parent_ID_grid(self, SILENT=True):

        #-----------------------------------------------------
        # Get a grid which for each grid cell contains the
        # calendar-style index (or ID) of the grid cell that
        # its D8 code says it flows to.
        #-----------------------------------------------------
        # Note: This version can handle periodic boundaries,
        #       as can occur in a landscape evolution model.
        #-----------------------------------------------------
        if not(SILENT):
            print 'Finding parent pixel IDs...'
        
        nx = self.nx
        ny = self.ny
        self.parent_ID_grid = self.ID_grid + self.inc_map[self.flow_grid]

        #---------------------------------
        # Iterators for using 1D indices
        #---------------------------------
        dirs = self.flow_grid.flat
        pIDs = self.parent_ID_grid.flat

        #---------------------------------------
        # Get IDs for pixels on the four edges
        #---------------------------------------
        T = arange(nx, dtype='Int32')
        B = T + (nx * (ny - 1))
        L = nx * arange(ny, dtype='Int32')
        R = L + (nx - 1)

        #------------------------------------------------
        # Remap parent IDs for pixels on left and right
        # edges to support periodic boundary conditions
        #------------------------------------------------
        if (self.LR_PERIODIC):
            w  = where(logical_or(logical_or((dirs[R] == 1), (dirs[R] == 2)), (dirs[R] == 4)))
            if (size(w[0]) != 0):    
                pIDs[R[w]] -= nx  # (subtract nx)
            #-------------------------------------------------------------------------------------
            w = where(logical_or(logical_or((dirs[L] == 16), (dirs[L] == 32)), (dirs[L] == 64)))
            if (size(w[0]) != 0):    
                pIDs[L[w]] += nx
        else:
            pIDs[R] = 0
            pIDs[L] = 0

        #------------------------------------------------
        # Remap parent IDs for pixels on top and bottom
        # edges to support periodic boundary conditions
        #------------------------------------------------
        if (self.TB_PERIODIC):
            w = where(logical_or(logical_or((dirs[T] == 1), (dirs[T] == 64)), (dirs[T] == 128)))
            if (size(w[0]) != 0):    
                pIDs[T[w]] += self.rti.n_pixels   ## (DOUBLE CHECK THIS)
            #-------------------------------------------------------------------------------------
            w = where(logical_or(logical_or((dirs[B] == 4), (dirs[B] == 8)), (dirs[B] == 16)))
            if (size(w[0]) != 0):    
                pIDs[B[w]] -= self.rti.n_pixels  # (subtract n_pixels)
        else:
            pIDs[T] = 0
            pIDs[B] = 0
            
        #---------------------------------------
        # Pixels with invalid flow directions,
        # like edges, get assigned a pID of 0.
        #---------------------------------------
        wbad = where(self.flow_grid <= 0)
        nw   = size(wbad[0])
        if (nw != 0):    
            self.parent_ID_grid[wbad] = 0

    #   update_parent_ID_grid()
    #---------------------------------------------------------------------
    def update_parent_IDs(self):

        #---------------------------------------------------------
        # Notes: This version cannot handle periodic boundaries,
        #        and requires that D8 flow codes be set to zero
        #        on the four edges.  This can be done at the end
        #        of the update_flow_grid() function.
        #---------------------------------------------------------
        # NB!  The use of 0's here is important.
        #      If iterating, pID[0]=0.
        #---------------------------------------------------------
       
        #-----------------------------------------------------
        # Get a grid which for each grid cell contains the
        # calendar-style index (or ID) of the grid cell that
        # its D8 code says it flows to.
        #-----------------------------------------------------
        pID_grid = self.parent_ID_grid

        #-------------------------------------------
        # Save IDs as a tuple of row indices and
        # calendar indices, "numpy.where" style
        #------------------------------------------- 
        self.parent_IDs = (pID_grid / self.nx, pID_grid % self.nx)
  
    #   update_parent_IDs()
    #-------------------------------------------------------------------
##    def update_non_parent_IDs(parent_IDs, flow_grid, rti):
##
##        #---------------------------
##        # Get flow grid dimensions
##        #---------------------------
##        nx = rti.ncols
##        ny = rti.nrows
##        
##        #---------------------------------------
##        # Return the IDs of non-parent pixels,
##        # such as ridges, but exclude pixels
##        # with flow code of 0, such as edges
##        # and nodata pixels.
##        #--------------------------------------- 
##        base = zeros([ny, nx], dtype='UInt8')
##        base[parent_IDs] = 1
##
##        wbad = no_flow_IDs(flow_grid, rti)
##        nbad = size(wbad[0]])
##        if (nbad > 0):
##            base[wbad] = 1   ##########  Should be 1 or 0 ??
##        
##        wnot = where(base == 0)
##        nnot = size(wnot[0])
##        if (nnot != 0):    
##            non_parent_IDs = wnot
##        else:    
##            non_parent_IDs = -int32(1)
##
##        return non_parent_IDs
##
##    #   update_non_parent_IDs()
    #---------------------------------------------------------------------
    def update_flow_from_IDs(self):

        #----------------------------------------------------------
        # Notes:  This function returns the 4-byte long-integer
        #         array IDs of pixels that flow in a particular
        #         direction.  RiverTools flow codes are assumed.
        #----------------------------------------------------------
        # Notes:  Later, rename w1 to w_NE, n1 to n_NE, then
        #         use self.code_list[0] vs. 1, etc..  This will
        #         then provide support for ARC flow codes, etc.
        #----------------------------------------------------------   
        self.w1 = where(self.flow_grid == 1)
        self.n1 = size(self.w1[0])   # (northeast)
        
        self.w2 = where(self.flow_grid == 2)
        self.n2 = size(self.w2[0])   # (east)
        
        self.w3 = where(self.flow_grid == 4)
        self.n3 = size(self.w3[0])   # (southeast)
        
        self.w4 = where(self.flow_grid == 8)
        self.n4 = size(self.w4[0])   # (south)
        
        self.w5 = where(self.flow_grid == 16)
        self.n5 = size(self.w5[0])   # (southwest)
        
        self.w6 = where(self.flow_grid == 32)
        self.n6 = size(self.w6[0])   # (west)
        
        self.w7 = where(self.flow_grid == 64)
        self.n7 = size(self.w7[0])   # (northwest)
        
        self.w8 = where(self.flow_grid == 128)
        self.n8 = size(self.w8[0])   # (north)

        ##### Same as noflow_IDs  ####################
##        self.w0 = where(self.flow_grid <= 0)
##        self.n0 = size(self.w0[0])   #(undefined)

        # print 'n1 = ' + str(n1)
                
    #   update_flow_from_IDs()
    #---------------------------------------------------------------------
    def update_flow_to_IDs(self):

        nx = self.nx          
       
        #-------------------------------------------------
        # Get IDs of "parent cells" that are downstream
        # of pixels that flow in a given direction.
        #-------------------------------------------------
        if (self.n1 != 0):    # northeast
            p1_IDs  = self.parent_ID_grid[self.w1]
            self.p1 = (p1_IDs / nx, p1_IDs % nx)
        else:
            self.p1 = (-1, -1)
        #-------------------------------------------------
        if (self.n2 != 0):     # east
            p2_IDs  = self.parent_ID_grid[self.w2]
            self.p2 = (p2_IDs / nx, p2_IDs % nx)
        else:
            self.p2 = (-1, -1)
        #-------------------------------------------------
        if (self.n3 != 0):     # southeast
            p3_IDs  = self.parent_ID_grid[self.w3]
            self.p3 = (p3_IDs / nx, p3_IDs % nx)
        else:
            self.p3 = (-1, -1)
        #-------------------------------------------------
        if (self.n4 != 0):     # south 
            p4_IDs  = self.parent_ID_grid[self.w4]
            self.p4 = (p4_IDs / nx, p4_IDs % nx)
        else:
            self.p4 = (-1, -1)
        #-------------------------------------------------
        if (self.n5 != 0):     # southwest
            p5_IDs  = self.parent_ID_grid[self.w5]
            self.p5 = (p5_IDs / nx, p5_IDs % nx)
        else:   
            self.p5 = (-1, -1)
        #-------------------------------------------------
        if (self.n6 != 0):     # west 
            p6_IDs  = self.parent_ID_grid[self.w6]
            self.p6 = (p6_IDs / nx, p6_IDs % nx)
        else:
            self.p6 = (-1, -1)
        #-------------------------------------------------
        if (self.n7 != 0):     # northwest  
            p7_IDs  = self.parent_ID_grid[self.w7]
            self.p7 = (p7_IDs / nx, p7_IDs % nx)
        else:
            self.p7 = (-1, -1)
        #-------------------------------------------------
        if (self.n8 != 0):     # north 
            p8_IDs  = self.parent_ID_grid[self.w8]
            self.p8 = (p8_IDs / nx, p8_IDs % nx)
        else:
            self.p8 = (-1, -1)
        #-------------------------------------------------
        ##    print 'p1.shape, size(p1)       =', p1.shape, size(p1)
        ##    print 'w1[0].shape, size(w1[0]) =', w1[0].shape, size(w1[0])      
        
        #-------------------------------------
        # Some flow directions may not occur
        #-------------------------------------
        self.p1_OK = (self.p1[0][0] != -1)
        self.p2_OK = (self.p2[0][0] != -1)
        self.p3_OK = (self.p3[0][0] != -1)
        self.p4_OK = (self.p4[0][0] != -1)
        self.p5_OK = (self.p5[0][0] != -1)
        self.p6_OK = (self.p6[0][0] != -1)
        self.p7_OK = (self.p7[0][0] != -1)
        self.p8_OK = (self.p8[0][0] != -1)

    #   update_flow_to_IDs()
    #-------------------------------------------------------------------
    def update_noflow_IDs(self):
        
        #--------------------------------------------------
        # 1/19/07.  Need to set d and u to zero at any ID
        # where flow terminates.  This includes pixels on
        # the edges, those with unresolved flow direction
        # and those where elevation is nodata or NaN.
        # A RiverTools flow grid will have a flow code of
        # zero at all of these places.
        #--------------------------------------------------
        noflow_IDs = where(self.flow_grid <= 0)
        num_IDs    = size(noflow_IDs[0])
        
        if (num_IDs != 0):
            self.noflow_IDs = noflow_IDs
        else:
            #----------------------------
            # Return IDs of edge pixels
            #----------------------------
            ## self.get_edge_IDs()  # (called by initialize())
            self.noflow_IDs = self.edge_IDs()

    #   update_noflow_IDs()
    #-------------------------------------------------------------------
    def read_flow_grid(self, SILENT=True):

        #----------------------------------------------------
        # Read a grid of D8 flow codes, same size as DEM.
        #----------------------------------------------------
        if not(SILENT):
            print 'Reading D8 flow grid...'
        code_file = (self.in_directory +
                     self.site_prefix + '_flow.rtg')
        self.flow_grid = rtg_files.read_grid(code_file, self.rti,
                                             RTG_type='BYTE')

    #   read_flow_grid()
    #-------------------------------------------------------------------
    def update_flow_grid(self, DEM=None, SILENT=True, REPORT=False):

        #--------------------------------------------------------
        # This can be used to test whether "update_area_grid()"
        # is working, even if there is a problem with the
        # "update_flow_grid()" function.
        #--------------------------------------------------------
##        print '### TEST:  Reading existing flow grid instead'
##        print '###        of computing one.'
##        print '################################################'
##        self.read_flow_grid()
##        return
    
        #------------------------------------------------------
        # NOTES:  Direction codes are given by:  |64  128  1|
        #                                        |32   x   2|
        #                                        |16   8   4|

        #         A RiverTools resolve array is returned by:
        #            self.get_resolve_array().
        #------------------------------------------------------
        if not(SILENT):    
            print 'Updating flow grid...'
        
        #----------------------------------------
        # Assign directions where well-defined,
        # and special codes otherwise
        #----------------------------------------
        self.start_new_flow_grid(DEM, SILENT=SILENT, REPORT=REPORT)
        
        #----------------------------------
        # Break ties with "resolve" array
        #----------------------------------
        self.break_flow_grid_ties(SILENT=SILENT)
        
        #-------------------
        # Link the flats ?
        #-------------------
        if (self.LINK_FLATS): self.link_flats(SILENT=SILENT)

        #-------------------------------------
        # Assign zeroes along 4 edges of DEM
        #-------------------------------------------------------
        # NB! This doesn't allow periodic boundary conditions.
        #-------------------------------------------------------
        #### self.flow_grid[ self.edge_IDs ] = 0   ##############
        
        #--------------------------------------------------
        # Change data type from 2-byte to 1-byte (NEW WAY)
        #--------------------------------------------------
        w = where( logical_or(self.flow_grid < 0, self.flow_grid > 128) )
        if (size(w[0]) > 0):
            self.flow_grid[w] = 0
        self.flow_grid = self.valid_code_map[ self.flow_grid ]

        #--------------------------------------------
        # This doesn't work because flow_grid is 2D
        #--------------------------------------------
##        w = where( setmember1d(self.flow_grid, self.code_list) )
##        if (size(w[0]) > 0):
##            self.flow_grid[w] = 0
    
        #--------------------
        # This doesn't work
        #--------------------
##        w = where( self.flow_grid not in self.code_list )
##        ## if (size(w) > 0):
##        if (size(w[0]) > 0):
##            self.flow_grid[w] = 0
            
        if not(SILENT):
            print '   min(codes), max(codes) =', \
                  self.flow_grid.min(), self.flow_grid.max()
            print 'flow grid ='
            print self.flow_grid

        #-------------------------------------------------
        # Compare saved flow grid to one just computed
        #-------------------------------------------------
        # Note that unless self.LR_PERIODIC = False and
        # self.TB_PERIODIC = False, the flow grids won't
        # agree on the edges.  This is because we don't
        # set them to zero when using periodic boundary
        # conditions.
        #-------------------------------------------------
        if (self.DEBUG):
            code_file = (self.in_directory +
                         self.site_prefix + '_flow.rtg')
            saved_flow_grid = rtg_files.read_grid(code_file, self.rti,
                                                  RTG_type='BYTE')
            w = where( saved_flow_grid != uint8(self.flow_grid) )
            if (size(w[0]) == 0):
                print '##### SUCCESS! Flow grids are identical.'
            else:
                print '##### FAILURE. Flow grids differ at:'
                print 'number of pixels =', size(w[0])

        #---------------------------------------------------
        # Change data type from 2-byte to 1-byte (OLD WAY)
        #---------------------------------------------------
##        dirs = self.flow_grid
##        w = where(logical_and(logical_and(logical_and(logical_and( \
##                  logical_and(logical_and(logical_and(logical_and( \
##                   (dirs != 1), (dirs != 2)), (dirs != 4)), \
##                   (dirs != 8)), (dirs != 16)), (dirs != 32)), \
##                   (dirs != 64)), (dirs != 128)), (dirs != 0)))
##        n_bad = size(w[0])
##        if (n_bad > 0):    
##            dirs[w] = int16(0)
##        self.flow_grid = numpy.uint8( dirs )
        ### dirs = idl_func.byte(dirs)
        
        #----------------------------
        # Save dir values to file ?
        #----------------------------
##        if (file is not None):    
##            file_unit = open(file, 'wb')
##            I2PY_SWAP_ENDIAN = False
##            if (I2PY_SWAP_ENDIAN):
##                array(dirs, copy=0).byteswap(True)
##            dirs.tofile(file_unit)
##            file_unit.close()

    
    #   update_flow_grid()
    #-------------------------------------------------------------------    
    def start_new_flow_grid(self, DEM=None, SILENT=True, REPORT=False):
        
        #--------------------------------------------------------------
        # Notes: In caller, modified so that DEM array has
        #        type INTEGER when DEM has type BYTE.  Need a signed
        #        type to compute slopes correctly here.  For example,
        #        (100b - 200b) = 156b.

        #        Use of NumPy's ROLL induces periodic boundaries.
        #--------------------------------------------------------------
        if not(SILENT):
            print '   update_flow_grid(): Initializing grid...'
            ## print '   Starting new flow grid...'
            ## print '   Initializing grid in update_flow_grid()...'

        #-----------------------------
        # Define some local synonyms
        #-----------------------------
        dx = self.dx[0]
        dy = self.dy[0]
        dd = self.dd[0]  #################

        #-----------------------------------------------------
        # Get a depression-filled DEM via the "erosion" port
        #-----------------------------------------------------
        ## DEM = self.get_port_data('DEM',  self.ep, 'EROSION')
        
        #------------------------------------------------
        # Read DEM from file.  Used by unit_test() now.
        #---------------------------------------------------------
        # NOTE! This DEM should have already had the depressions
        #       filled, e.g. by the tool in fill_pits.py.
        #---------------------------------------------------------
        if (DEM == None):
            dp = (self.in_directory + self.site_prefix)
            DEM_file = (dp + '_2D-z0.rtg')
            if not(os.path.exists(DEM_file)):
                DEM_file = (dp + '_DEM.rtg')
            if not(os.path.exists(DEM_file)):
                print 'ERROR: Could not find DEM file.'
                return
            DEM = rtg_files.read_grid( DEM_file, self.rti, SILENT=False )
            if not(SILENT):
                print '   min(DEM), max(DEM) =', DEM.min(), DEM.max()

        #------------------------------
        # Slopes to 8 neighbor pixels
        #------------------------------
        s1 = (DEM - numpy.roll(numpy.roll(DEM, 1, axis=0), -1, axis=1)) / dd   # (upper-right)
        s2 = (DEM - numpy.roll(DEM, -1, axis=1)) / dx                          # (right)
        s3 = (DEM - numpy.roll(numpy.roll(DEM, -1, axis=0), -1, axis=1)) / dd  # (lower-right)
        s4 = (DEM - numpy.roll(DEM, -1, axis=0)) / dy                          # (bottom)
        s5 = (DEM - numpy.roll(numpy.roll(DEM, -1, axis=0), 1, axis=1)) / dd   # (lower-left)
        s6 = (DEM - numpy.roll(DEM, 1, axis=1)) / dx                           # (left)
        s7 = (DEM - numpy.roll(numpy.roll(DEM, 1, axis=0), 1, axis=1)) / dd    # (upper-left)
        s8 = (DEM - numpy.roll(DEM, 1, axis=0)) / dy                           # (top)

        #--------------------------
        # Find the steepest slope
        #--------------------------
        max_slope = numpy.maximum(s1, s2)
        max_slope = numpy.maximum(max_slope, s3)
        max_slope = numpy.maximum(max_slope, s4)
        max_slope = numpy.maximum(max_slope, s5)
        max_slope = numpy.maximum(max_slope, s6)
        max_slope = numpy.maximum(max_slope, s7)
        max_slope = numpy.maximum(max_slope, s8)
        #-----------------------------------------
        # Faster way using 3rd, "out" argument ?
        #-----------------------------------------
##        numpy.maximum(s1, s2, max_slope)
##        numpy.maximum(max_slope, s3, max_slope)
##        numpy.max_slope = maximum(max_slope, s4)
##        numpy.max_slope = maximum(max_slope, s5)
##        numpy.max_slope = maximum(max_slope, s6)
##        numpy.max_slope = maximum(max_slope, s7)
##        numpy.max_slope = maximum(max_slope, s8)

        
        ## max_slope = (maximum(maximum(maximum(maximum(maximum(maximum(maximum(s1, s2), s3), s4), s5), s6), s7), s8))

        g    = self.code_list
        dirs = (s1 == max_slope) * g[0] + \
               (s2 == max_slope) * g[1] + \
               (s3 == max_slope) * g[2] + \
               (s4 == max_slope) * g[3] + \
               (s5 == max_slope) * g[4] + \
               (s6 == max_slope) * g[5] + \
               (s7 == max_slope) * g[6] + \
               (s8 == max_slope) * g[7]
        
        #------------------------------------------
        # Assign negative codes to flats and pits
        #------------------------------------------
        FLATS = where(max_slope == 0)
        n_flats = size(FLATS[0])
        if (n_flats != 0):    
            dirs[ FLATS ] = (-1 * dirs[ FLATS ])
        
        #---------------------------------------------
        # There shouldn't be any of these left since
        # they were filled by fill_pits.fill_pits().
        #---------------------------------------------
        PITS = where(max_slope < 0)
        n_pits = size(PITS[0])
        if (n_pits != 0):
            dirs[ PITS ] = -300
        
        #---------------------------------------------
        # Assign code of zero to NODATA & NaN pixels
        # Don't use NOT wrapped around FINITE. (254)
        #---------------------------------------------
        # Also assign code of zero to pixels
        # that are marked with RT closed-basin code?
        # Streamlines can end at either place.
        #----------------------------------------------
        w = where(logical_or((DEM <= self.nodata), (isfinite(DEM) != 1)))
        #### or (DEM eq self.closed_basin_code)
        n_bad = size(w[0])
        if (n_bad != 0):    
            dirs[w] = 0
        
        #-----------------------------------------------
        # Set left & right flow grid borders to zero ?
        #-----------------------------------------------
        if not(self.LR_PERIODIC):
            dirs[:, 0]          = 0
            dirs[:,self.nx - 1] = 0
        
        #-----------------------------------------------
        # Set top & bottom flow grid borders to zero ?
        #-----------------------------------------------
        if not(self.TB_PERIODIC):
            dirs[0, :]          = 0
            dirs[self.ny - 1,:] = 0

        self.flow_grid = dirs  ##############

        if (REPORT):
            print '   --------------------------------------------'
            print '   Data type of flow grid at start =', dirs.dtype
            print '   Number of flats         =', n_flats
            print '   Number of 1-pixel pits  =', n_pits
            print '   Number of nodata/NaN    =', n_bad
            print '   min(codes), max(codes) =', dirs.min(), dirs.max()
            print '   --------------------------------------------'

    #   start_new_flow_grid()
    #---------------------------------------------------------------------
    def resolve_array_cycle(self, w, t):

        #--------------------------------------------------------
        #NOTES:  This function takes a set of "byte-coded ties"
        #        and returns all of the "byte-coded ties" that are
        #        equivalent up to a rotation.  This ensures that
        #        consistent tie-breaker directions are assigned
        #        in producing the array called "resolve".
        #        w is the array to be "Cycled."
        #        t is the number of cycles.
        #        v is the "Cycled" ARRAY.
        #--------------------------------------------------------
        v = w
        for j in xrange(1, t+1):
            v = (v * 2)
            v += (v > 255)
            v = numpy.bitwise_and(v, 255)
            ##### v = numpy.logical_and(v, 255)   
        return v
        
    #   resolve_array_cycle()
    #---------------------------------------------------------------------
    def get_resolve_array(self, SILENT=True):

        #------------------------------------------------------
        # NOTES:  RT/SJ D8 flow direction codes are given by:

        #       |64  128  1|
        #       |32   x   2|
        #       |16   8   4|
        #------------------------------------------------------
        if not(SILENT):
            print 'Computing "resolve array"...'
            
        resolve = zeros([256], dtype='UInt8')
        
        #-----------------------------------
        # High-symmetry groups are:
        # {17,34,68,136},{51,102,204,153},
        # {119,238,193,187},{85,170},{255}
        #-----------------------------------
        resolve[[17, 85]]                  = 1
        resolve[[34, 102, 119, 238, 170]]  = 2
        resolve[[136, 153, 187, 221, 255]] = 8
        resolve[68]  = 4
        resolve[51]  = 32
        resolve[204] = 128
        
        #-----------------------
        # Resolve corner cases
        #-----------------------
        resolve[[1, 5, 69]]   = 1
        resolve[[4, 20, 21]]  = 4
        resolve[[16, 80, 84]] = 16
        resolve[[64, 65, 81]] = 64
        
        w = uint8([2, 3, 6, 7, 10, 11, 14, 15, 18, 19, 22, 23])              #(12)
        w = concatenate((w,[27,31,35,38,39,43,47,54,55,66,67,70]))           #(12)
        w = concatenate((w,[71,74,75,78,79,82,83,86,87,91,95,103]))          #(12)
        w = concatenate((w,[107,111,134,135,138,139,142,143,150,151,155]))   #(11)
        w = concatenate((w,[159,166,167,171,175,183,191,206,207,223]))       #(10)
        
        #----------------------------------------
        # Resolve rotationally equivalent cases
        #----------------------------------------
        resolve[w] = 2
        resolve[self.resolve_array_cycle(w, 2)] = 8
        resolve[self.resolve_array_cycle(w, 4)] = 32
        resolve[self.resolve_array_cycle(w, 6)] = 128

        self.resolve = resolve

        #--------------
        # For testing
        #--------------
##        print 'resolve ='
##        print reshape(resolve, (16,16))
##        print 'sum(resolve) =', numpy.sum(resolve)
##        print ' '
        
    #   get_resolve_array()
    #-------------------------------------------------------------------    
    def break_flow_grid_ties(self, SILENT=True):

        #----------------------------------------------------------
        # Notes: This routine resolves all non-flat ties, using
        #        a "tie-breaker array" called "resolve".  Note
        #        that "resolve" always returns one of the eight
        #        D8 flow codes.  Note that "resolve" maps valid
        #        D8 flow codes to themselves.
        #----------------------------------------------------------
        if not(SILENT):
            print '   update_flow_grid(): Breaking ties...'
            ## print '   Breaking ties in update_flow_grid():...'

        dirs = self.flow_grid  # (local synonym)
        
        w = where(dirs > 0)
        if (size(w[0]) != 0):    
            self.flow_grid[w] = self.resolve[ dirs[w] ]
            # Next line should also work ??
            ## dirs[w] = self.resolve[ dirs[w] ]        

    #   break_flow_grid_ties()
    #-------------------------------------------------------------------    
    def link_flats(self, SILENT=True):

        #--------------------------------------------------------
        # Notes: This procedure uses NumPy array operations to
        #        eliminate all loops and to increase speed.
        #        However, it is not equivalent to LinkUp2,
        #        because the sequential processing of lines
        #        there makes it possible to be linked to a
        #        just-resolved left neighbor.

        #       DIRECTION CODES:      SUBSCRIPTS:
        #       ----------------      -----------
        #          |64 128 1|          |6  7  0|    (i1)
        #          |32  x  2|          |5  x  1|    (i2)
        #          |16  8  4|          |4  3  2|    (i3)

        #       Note that resolve[code] always returns one
        #       of the component directions that made "code";
        #       it never "splits the difference".

        #       Need to make sure edges are not set to zero
        #       yet, but also need to exclude them from
        #       being counted as FLATS.

        #       Can't use the line:
        #           dirs(0,:)=0  &  dirs(NS-1,:)=0

        #       And can't use the line:
        #           FLATS=where((dirs(1:NS-1,i2) < 0) AND $
        #                       (dirs(1:NS-1,i2) != -300), NUM)

        #       Current solution is to use a NOTEDGE array.
        #--------------------------------------------------------
        if not(SILENT):
            print '   update_flow_grid(): Linking flats...'
            ## print '   Linking flats in update_flow_grid()...'

        #-----------------        
        # Local synonyms
        #-----------------
        nx = self.nx
        ny = self.ny
        g  = self.code_list   # (RT D8 flow codes)
        h  = self.code_opps   # (opposites of RT D8 flow codes)
        #** g = [1, 2, 4, 8, 16, 32, 64, 128]
        #** h = [16, 32, 64, 128, 1, 2, 4, 8]
        dirs     = self.flow_grid.flat
        NOT_EDGE = self.not_edge_grid.flat
        
        total_flats = int64(0)
        n_reps      = int64(0)
        
        while (True):
            n_reps += 1
            STILL_ACTIVE = False
            flats = where(logical_and(logical_and((dirs < 0), \
                                                  (dirs != -300)), \
                                                  (NOT_EDGE == 1)))
            # n_flats = size(flats)

            n_flats = size(flats[0])
            flats   = flats[0]   ################
            
            if (n_flats != 0):
                #----------------------------------
                # Flow codes of 8 neighbor pixels
                # (Doesn't work for edge pixels.)
                #----------------------------------
                d0 = dirs[flats - nx + 1]   # (upper-right)
                d1 = dirs[flats + 1]        # (right) 
                d2 = dirs[flats + nx + 1]   # (lower-right)
                d3 = dirs[flats + nx]       # (bottom)
                d4 = dirs[flats + nx - 1]   # (lower-left)
                d5 = dirs[flats - 1]        # (left)
                d6 = dirs[flats - nx - 1]   # (upper-left)
                d7 = dirs[flats - nx]       # (top)
                
                #--------------------------------------------
                # A direction is VALID if:
                # (1) the neighbor in this direction has
                #     a well-defined (gt 0) direction, and
                #
                # (2) this neighbor's direction is not back
                #     towards the center pixel (BACKFLOW)
                #
                #     Later use "READY" idea?
                #     What is the fastest way?
                #--------------------------------------------
                VALID0 = logical_and((d0 > 0), (d0 != h[0]))
                VALID1 = logical_and((d1 > 0), (d1 != h[1]))
                VALID2 = logical_and((d2 > 0), (d2 != h[2]))
                VALID3 = logical_and((d3 > 0), (d3 != h[3]))
                VALID4 = logical_and((d4 > 0), (d4 != h[4]))
                VALID5 = logical_and((d5 > 0), (d5 != h[5]))
                VALID6 = logical_and((d6 > 0), (d6 != h[6]))
                VALID7 = logical_and((d7 > 0), (d7 != h[7]))
                
                #--------------------------------------------
                # A direction is allowed only if it was one
                # of the directions that was summed in
                # producing the value of code.
                #--------------------------------------------
                codes = (-1 * dirs[flats])
                #---------------------------
                ALLOWED7 = (codes >= g[7])
                codes    = (codes - (g[7] * ALLOWED7))
                ALLOWED6 = (codes >= g[6])
                codes    = (codes - (g[6] * ALLOWED6))
                ALLOWED5 = (codes >= g[5])
                codes    = (codes - (g[5] * ALLOWED5)) 
                ALLOWED4 = (codes >= g[4])
                codes    = (codes - (g[4] * ALLOWED4))
                ALLOWED3 = (codes >= g[3])
                codes    = (codes - (g[3] * ALLOWED3)) 
                ALLOWED2 = (codes >= g[2])
                codes    = (codes - (g[2] * ALLOWED2))  
                ALLOWED1 = (codes >= g[1])
                codes    = (codes - (g[1] * ALLOWED1))
                ALLOWED0 = (codes >= g[0])
                ###########################################
                ###  (11/5/09) Should we add next line ??
                ###  It is not in the original IDL code.
                ###########################################
                # codes    = (codes - (g[0] * ALLOWED0))
                ###########################################                
                READY = ((logical_and(VALID0, ALLOWED0)) * g[0] + \
                         (logical_and(VALID1, ALLOWED1)) * g[1] + \
                         (logical_and(VALID2, ALLOWED2)) * g[2] + \
                         (logical_and(VALID3, ALLOWED3)) * g[3] + \
                         (logical_and(VALID4, ALLOWED4)) * g[4] + \
                         (logical_and(VALID5, ALLOWED5)) * g[5] + \
                         (logical_and(VALID6, ALLOWED6)) * g[6] + \
                         (logical_and(VALID7, ALLOWED7)) * g[7])
                
                FIXABLE   = where(READY)
                n_fixable = size(FIXABLE[0])
                FIXABLE   = FIXABLE[0]  ######## (see above)
                
                if (n_fixable != 0):    
                    #-----------------------------------
                    # Assign flow codes to READY flats
                    #-----------------------------------
                    total_flats += n_fixable
                    dirs[flats[FIXABLE]] = self.resolve[READY[FIXABLE]]
                    
                    ### links = self.resolve.flat[FIXABLE]   #################################
                    ### dirs[FIXABLE] = links
    ##                links = resolve[READY[FIXABLE]]
    ##                dirs[FLATS[FIXABLE]] = links
                    STILL_ACTIVE = True

                #-----------------------------------------------
                # We're finished if we "fixed" everything that
                # was "fixable", even if some flats remain.
                #-----------------------------------------------
                FINISHED = (n_fixable == n_flats)
            else:
                #-----------------
                # (n_flats == 0)
                #-----------------
                FINISHED = True
            if (FINISHED) or not(STILL_ACTIVE):
                break
            
                #------------------------
                # Are we finished yet ?
                #------------------------
##                if (n_flats != 0) and (n_fixable != n_flats):    
##                    UNFINISHED = True
##                else:    
##                    UNFINISHED = False               
##            else:    
##                UNFINISHED = False   ######
##            if not(UNFINISHED) or not(STILL_ACTIVE):
##                break
            
        self.total_flats = total_flats

        if not(SILENT):
            print '   Number of iterations =', n_reps, ' (in link_flats())'
            
        #-------------------------------------
        # This not necessary, and also dirs
        # is an iterator now (.flat)
        #-------------------------------------
        ## self.flow_grid = dirs
        
    #   link_flats()    
    #-------------------------------------------------------------------    
    def update_flow_width_grid(self, DOUBLE=False, METHOD2=False,
                               SILENT=True, REPORT=False):

        #-------------------------------------------------------------
        # NOTES: This routine returns the flow widths for each
        #        pixel in the DEM.  The extra width of flowing to a
        #        diagonal pixel is taken into account, as well as the
        #        lat/lon-dependence for DEMs with fixed-angle pixels
        #        (Geographic lat/lon).

        #        METHOD2 version ensures that the sum of all flow
        #        widths around a pixel is equal to 2*(dx + dy), but
        #        is incorrect for case of a plane and others.

        #        Flow widths are zero where (flow grid eq 0).
        
        #        Flow widths are for entire pixel and are appropriate
        #        for overland or subsurface flow.
        #------------------------------------------------------------- 
        #        Is this only used by Seepage function now ???
        #-------------------------------------------------------------
        # NB!    numpy.where returns a 2-tuple, but with "empty"
        #        second part when applied to a 1D array.  This means
        #        that nw = size(w[0]) still works.
        #-------------------------------------------------------------
        if not(SILENT):
            print 'Updating flow width grid...'

        fg = self.flow_grid   # (local synonym)

        #----------------
        # Diagonal flow
        #----------------
        wd = where(logical_or(logical_or(logical_or((fg == 1), (fg == 4)), \
                                                    (fg == 16)), (fg == 64)))
        nwd = size(wd[0])
        if (nwd != 0):
            rows = wd[0]
            if not(METHOD2):
                self.dw[wd] = self.dd[rows]
            else:    
                self.dw[wd] = (self.dx[rows] + self.dy[rows]) / 4

        #---------------------
        # East and west flow
        #---------------------
        wh  = where(logical_or((fg == 2), (fg == 32)))
        nwh = size(wh[0])
        if (nwh != 0):
            self.dw[wh] = self.dy[ wh[0] ] # (wh[0] = rows)
            if (METHOD2):    
                self.dw[wh] = self.dw[wh] / 2

        #-----------------------
        # North and south flow
        #-----------------------
        wv  = where(logical_or((fg == 8), (fg == 128)))
        nwv = size(wv[0])
        if (nwv != 0):
            self.dw[wv] = self.dx[ wv[0] ]  # (wv[0] = rows)
            if (METHOD2):    
                self.dw[wv] = self.dw[wv] / 2

        #---------------------------
        # Undefined flow direction
        #---------------------------
        wb = where(fg == 0)
        nwb = size(wb[0])
        if (nwb != 0):
            #--------------------------------------
            # This prevents divide by zero errors
            #--------------------------------------
            self.dw[wb] = self.dx[ wb[0] ] # (wb[0] = rows)

        #------------------
        # Optional report
        #------------------
        if (REPORT):
            dw_str = str(self.dw.min()) + ', ' + str(self.dw.max())
            print '    min(dw), max(dw) = ' + dw_str + ' [m]'
            
##            print '    min(dw) = ' + str(self.dw.min()) + '  [m]'
##            print '    max(dw) = ' + str(self.dw.max()) + '  [m]'

    #   update_flow_width_grid()
    #-------------------------------------------------------------------
    def update_flow_length_grid(self, DOUBLE=False,
                                SILENT=True, REPORT=False):

        #-------------------------------------------------------------
        # NOTES: This routine returns the flow lengths for each
        #        pixel in the DEM.  The extra length of flowing to a
        #        diagonal pixel is taken into account, as well as the
        #        latitude-dependence for DEMs with fixed-angle pixels
        #        (Geographic lat/lon).

        #        The only difference between this and the Flow_Widths
        #        function is that roles of dx and dy are switched.

        #        Flow lengths are set to dx[0] for the pixels where
        #       (flow grid eq 0), such as on the edges of the DEM.
        #-------------------------------------------------------------
        if not(SILENT):
            print 'Updating flow length grid...'

        fg = self.flow_grid  # (local synonym)
              
        #----------------
        # Diagonal flow
        #----------------
        wd = where(logical_or(logical_or(logical_or((fg == 1), (fg == 4)), \
                                                    (fg == 16)), (fg == 64)))
        nwd = size(wd[0])
        if (nwd != 0):
            self.ds[wd] = self.dd[ wd[0] ] # (wd[0] = rows)
        
        #---------------------
        # East and west flow
        #---------------------
        wh  = where(logical_or((fg == 2), (fg == 32)))
        nwh = size(wh[0])
        if (nwh != 0):    
            self.ds[wh] = self.dx[ wh[0] ] # (wh[0] = rows)
        
        #-----------------------
        # North and south flow
        #-----------------------
        wv  = where(logical_or((fg == 8), (fg == 128)))
        nwv = size(wv[0])
        if (nwv != 0):    
            self.ds[wv] = self.dy[ wv[0] ] # (wv[0] = rows)

        #---------------------------
        # Undefined flow direction
        #---------------------------
        wb = where(fg == 0)
        nwb = size(wb[0])
        if (nwb != 0):
            #--------------------------------------
            # This prevents divide by zero errors
            #--------------------------------------
            self.ds[wb] = self.dx[ wb[0] ] # (wb[0] = rows)
            
        #------------------
        # Optional report
        #------------------
        if (REPORT):
            ds_str = str(self.ds.min()) + ', ' + str(self.ds.max())
            print '    min(ds), max(ds) = ' + ds_str + ' [m]'
            
##            print '    min(ds) = ' + str(self.ds.min()) + '  [m]'
##            print '    max(ds) = ' + str(self.ds.max()) + '  [m]'

    #   update_flow_length_grid()
    #-------------------------------------------------------------------
    def update_area_grid(self, SILENT=True, REPORT=False):

        #------------------------------------------------------
        # Notes: Idea is to find the pixels whose area has
        #        not yet been assigned, and to recursively
        #        assign areas to those for which all children
        #        have been assigned areas.

        #        g = [1, 2, 4, 8, 16, 32, 64, 128]
        #        h = [16, 32, 64, 128, 1, 2, 4, 8]

        #        Notice that the first and last pixel of each
        #        line are assigned an area of zero.

        #        DIRECTION CODES:      SUBSCRIPTS:
        #        ----------------      -----------
        #           |64 128 1|          |6  7  0|    (i1)
        #           |32  x  2|          |5  x  1|    (i2)
        #           |16  8  4|          |4  3  2|    (i3)
        #------------------------------------------------------
        if not(SILENT):    
            print 'Updating upstream area grid...'
        
        #------------------
        # Initialize vars
        #------------------------------------------------
        # initialize_computed_vars() initializes self.A
        #------------------------------------------------
        self.A = minimum(self.A, 0)   # (reset to all zeros)
        n_reps = int32(0)

        #-----------------        
        # Local synonyms
        #-----------------
        nx = self.nx
        ny = self.ny
        h  = self.code_opps
        # g  = self.code_list  # (not used here)

        #--------------------------------------------
        # Convert units for da from m^2 to km^2 ??
        #--------------------------------------------
        # da was stored by self.read_grid_info()
        # Units for da are specified in '_d8.cfg"
        # file as either 'm^2' or 'km^2'
        #-----------------------------------------
        if ('km' in self.A_units.lower()):
            pixel_area = self.da / 1e6
        else:
            pixel_area = self.da
    
        while (True):
        
            STILL_ACTIVE = False
            n_reps += 1
            
            #-------------------------------------------------------
            # Added test for (dirs ne 0), so that edge
            # and nodata pixels won't be considered "READY".
            #-------------------------------------------------------
            # This led to an infinite loop bug because the edge
            # pixels were "ready", then PASS_CHANGE=1b, then their
            # areas were changed back to zero in caller which made
            # them "ready" again.
            #-------------------------------------------------------
            unknown   = where(logical_and((self.A == 0), (self.flow_grid != 0)))
            rows      = unknown[0]
            cols      = unknown[1]
            n_unknown = size( rows )
            
            if (n_unknown != 0):    
                #-----------------------------------------
                # Initialize all unknown pixels as READY
                #-----------------------------------------
                READY = numpy.ones([n_unknown], dtype='UInt8')
            
                #-------------------------------------------------
                # Upstream areas of 8 neighbor pixels (periodic)
                # but only for the "unknown" pixels.
                #-------------------------------------------------
                a0 = self.A[(rows-1) % ny, (cols+1) % nx]   # (upper-right)
                a1 = self.A[rows,          (cols+1) % nx]   # (right)
                a2 = self.A[(rows+1) % ny, (cols+1) % nx]   # (lower-right)
                a3 = self.A[(rows+1) % ny,  cols]           # (bottom)
                a4 = self.A[(rows+1) % ny, (cols-1) % nx]   # (lower-left)
                a5 = self.A[rows,          (cols-1) % nx]   # (left)
                a6 = self.A[(rows-1) % ny, (cols-1) % nx]   # (upper-left)
                a7 = self.A[(rows-1) % ny,  cols]           # (top)
        
                #--------------------------------------
                # Upstream areas of 8 neighbor pixels
                # for all pixels in the grid
                #--------------------------------------
##                a0 = numpy.roll(numpy.roll(self.A, 1, axis=0), -1, axis=1)  # (upper-right)
##                a1 = numpy.roll(self.A, -1, axis=1)                         # (right)
##                a2 = numpy.roll(numpy.roll(self.A, -1, axis=0), -1, axis=1) # (lower-right)
##                a3 = numpy.roll(self.A, -1, axis=0)                         # (bottom)
##                a4 = numpy.roll(numpy.roll(self.A, -1, axis=0), 1, axis=1)  # (lower-left)
##                a5 = numpy.roll(self.A, 1, axis=1)                          # (left)
##                a6 = numpy.roll(numpy.roll(self.A, 1, axis=0), 1, axis=1)   # (upper-left)
##                a7 = numpy.roll(self.A, 1, axis=0)                          # (top)
                
                #---------------------------------------------
                # Flow codes of 8 neighbor pixels (periodic)
                # but only for the "unknown" pixels.
                #---------------------------------------------
                d0 = self.flow_grid[(rows-1) % ny, (cols+1) % nx]   # (upper-right)
                d1 = self.flow_grid[rows,          (cols+1) % nx]   # (right)
                d2 = self.flow_grid[(rows+1) % ny, (cols+1) % nx]   # (lower-right)
                d3 = self.flow_grid[(rows+1) % ny,  cols]           # (bottom)
                d4 = self.flow_grid[(rows+1) % ny, (cols-1) % nx]   # (lower-left)
                d5 = self.flow_grid[rows,          (cols-1) % nx]   # (left)
                d6 = self.flow_grid[(rows-1) % ny, (cols-1) % nx]   # (upper-left)
                d7 = self.flow_grid[(rows-1) % ny,  cols]           # (top)
         
                #----------------------------------
                # A pixel is not READY if any of
                # it's children have unknown area
                #----------------------------------
                # Pixels w/ no children are READY
                # unless they are "edge" pixels
                #----------------------------------
                w0 = where(logical_and((d0 == h[0]), (a0 == 0)))
                if (size(w0[0]) != 0):
                    READY[w0] = 0
                #---------------------------------------------------
                w1 = where(logical_and((d1 == h[1]), (a1 == 0)))
                if (size(w1[0]) != 0):    
                    READY[w1] = 0
                #---------------------------------------------------
                w2 = where(logical_and((d2 == h[2]), (a2 == 0)))
                if (size(w2[0]) != 0):    
                    READY[w2] = 0
                #---------------------------------------------------
                w3 = where(logical_and((d3 == h[3]), (a3 == 0)))
                if (size(w3[0]) != 0):    
                    READY[w3] = 0
                #---------------------------------------------------
                w4 = where(logical_and((d4 == h[4]), (a4 == 0)))
                if (size(w4[0]) != 0):    
                    READY[w4] = 0
                #---------------------------------------------------
                w5 = where(logical_and((d5 == h[5]), (a5 == 0)))
                if (size(w5[0]) != 0):    
                    READY[w5] = 0
                #---------------------------------------------------
                w6 = where(logical_and((d6 == h[6]), (a6 == 0)))
                if (size(w6[0]) != 0):    
                    READY[w6] = 0
                #---------------------------------------------------
                w7 = where(logical_and((d7 == h[7]), (a7 == 0)))
                if (size(w7[0]) != 0):    
                    READY[w7] = 0
                
                #----------------------------------------
                # If a pixel is my child and I'm READY,
                # then add it's value to mine.
                #----------------------------------------
                WR      = where(READY)
                n_ready = size(WR[0])
                WR = WR[0]  ################
                
                if (n_ready != 0):
                    #---------------------------
                    # This also assigns areas
                    # to pixels w/ no children
                    #---------------------------
                    STILL_ACTIVE = True
                    rows = rows[WR]
                    cols = cols[WR]
                    if (size(pixel_area) == 1):
                        self.A[rows, cols] = pixel_area
                    else:
                        self.A[rows, cols] = pixel_area[rows, cols]
                    #---------------------------------------------                    
                    w0 = where(d0[WR] == h[0])
                    if (size(w0[0]) != 0):
                        self.A[rows[w0], cols[w0]] += a0[WR[w0]]
                    #---------------------------------------------
                    w1 = where(d1[WR] == h[1])
                    if (size(w1[0]) != 0):
                        self.A[rows[w1], cols[w1]] += a1[WR[w1]]
                    #---------------------------------------------
                    w2  = where(d2[WR] == h[2])
                    if (size(w2[0]) != 0):
                        self.A[rows[w2], cols[w2]] += a2[WR[w2]]
                    #---------------------------------------------
                    w3 = where(d3[WR] == h[3])
                    if (size(w3[0]) != 0):
                        self.A[rows[w3], cols[w3]] += a3[WR[w3]]
                    #---------------------------------------------
                    w4 = where(d4[WR] == h[4])
                    if (size(w4[0]) != 0):
                        self.A[rows[w4], cols[w4]] += a4[WR[w4]]
                    #---------------------------------------------
                    w5 = where(d5[WR] == h[5])
                    if (size(w5[0]) != 0):
                        self.A[rows[w5], cols[w5]] += a5[WR[w5]]
                    #---------------------------------------------
                    w6 = where(d6[WR] == h[6])
                    if (size(w6[0]) != 0):
                        self.A[rows[w6], cols[w6]] += a6[WR[w6]]
                    #---------------------------------------------
                    w7 = where(d7[WR] == h[7])
                    if (size(w7[0]) != 0):
                        self.A[rows[w7], cols[w7]] += a7[WR[w7]]
                
                #------------------------
                # Are we finished now ?
                #------------------------
                if (n_unknown != 0) and (n_ready != n_unknown):   
                    UNFINISHED = True
                else:    
                    UNFINISHED = False
                
            else:    
                UNFINISHED = False

            if not(UNFINISHED) or not(STILL_ACTIVE):
                break
        
        if (UNFINISHED):    
            print 'Upstream area not defined for all pixels.'

        #-----------------------------
        # Save area grid for testing
        #-----------------------------
        SAVE_TEST = False
        if (SAVE_TEST):
            file_unit = open('00_AREA_GRID_TEST.rtg', 'wb')
            if (self.rti.SWAP_ENDIAN):
                grid = self.A.copy()
                grid.byteswap(True)
                grid.tofile(file_unit)
            else:
                self.A.tofile( file_unit )
            file_unit.close()
        
        #------------------
        # Optional report
        #------------------
        if (REPORT):
            if ('km' in self.A_units.lower()):
                unit_str = ' [km^2]'
            else:
                unit_str = ' [m^2]'
            A_str = str(self.A.min()) + ', ' + str(self.A.max())
            print '    min(A), max(A) = ' + A_str + unit_str
            print '    Number of iterations = ' + str(n_reps)

        #-------------------------------------------------
        # Compare saved area grid to one just computed
        #-------------------------------------------------
        # Note that unless self.LR_PERIODIC = False and
        # self.TB_PERIODIC = False, the area grids won't
        # agree on the edges.  This is because we don't
        # set them to zero when using periodic boundary
        # conditions.
        #-------------------------------------------------
        if (self.DEBUG):
            area_file = (self.in_directory +
                         self.site_prefix + '_area.rtg')
            saved_area_grid = rtg_files.read_grid(area_file, self.rti,
                                                  RTG_type='FLOAT')
            w = where( saved_area_grid != float32(self.A) )
            if (size(w[0]) == 0):
                print '##### SUCCESS! Area grids are identical.'
            else:
                print '##### FAILURE. Area grids differ at:'
                print '  Number of pixels =', size(w[0])
                diff = absolute(saved_area_grid - self.A)
                print '  Maximum difference =', diff.max()
                print ' '
                
    #   update_area_grid()
    #-------------------------------------------------------------------
    def update_area_grid_OLD(self, SILENT=True, REPORT=False):

        #------------------------------------------------------
        # Notes: This version cannot handle periodic boundary
        #        conditions.
        #------------------------------------------------------
        # Notes: Idea is to find the pixels whose area has
        #        not yet been assigned, and to recursively
        #        assign areas to those for which all children
        #        have been assigned areas.

        #        g = [1, 2, 4, 8, 16, 32, 64, 128]
        #        h = [16, 32, 64, 128, 1, 2, 4, 8]

        #        Notice that the first and last pixel of each
        #        line are assigned an area of zero.

        #        DIRECTION CODES:      SUBSCRIPTS:
        #        ----------------      -----------
        #           |64 128 1|          |6  7  0|    (i1)
        #           |32  x  2|          |5  x  1|    (i2)
        #           |16  8  4|          |4  3  2|    (i3)
        #------------------------------------------------------
        if not(SILENT):    
            print 'Updating upstream area grid...'
        
        #------------------
        # Initialize vars
        #------------------------------------------------
        # initialize_computed_vars() initializes self.A
        #------------------------------------------------
        self.A = minimum(self.A, 0)   # (reset to all zeros)
        n_reps = int32(0)

        #-----------------        
        # Local synonyms
        #-----------------
        nx = self.nx
        ny = self.ny
        h  = self.code_opps
        # g  = self.code_list  # (not used here)\
        
        #-----------------------------------------
        # da was stored by self.read_grid_info()
        #-----------------------------------------
        pixel_area = (self.da  / 1e6) # [m^2 -> km^2]
        
        #-------------------------------------------------
        # Use "flattened versions" of grids for indexing
        #-------------------------------------------------
        A    = self.A.flat
        dirs = self.flow_grid.flat  # (these are "iterators")
        
        while (True):
        
            STILL_ACTIVE = False
            n_reps += 1
            
            #-------------------------------------------------------
            # Added test for (dirs ne 0), so that edge
            # and nodata pixels won't be considered "READY".
            #-------------------------------------------------------
            # This led to an infinite loop bug because the edge
            # pixels were "ready", then LINECHANGE=1b, then their
            # areas were changed back to zero in caller which made
            # them "ready" again.
            #-------------------------------------------------------
            unknown   = where(logical_and((A == 0), (dirs != 0)))           
            IDs       = unknown[0]
            n_unknown = size( IDs )
            
            if (n_unknown != 0):    
                #-----------------------------------------
                # Initialize all unknown pixels as READY
                #-----------------------------------------
                READY = zeros([n_unknown], dtype='UInt8') + 1
                
                #--------------------------------------
                # Upstream areas of 8 neighbor pixels
                #--------------------------------------
                a0 = A[IDs - nx + 1]   #(upper-right)
                a1 = A[IDs + 1]        #(right)
                a2 = A[IDs + nx + 1]   #(lower-right)
                a3 = A[IDs + nx]       #(bottom)
                a4 = A[IDs + nx - 1]   #(lower-left)
                a5 = A[IDs - 1]        #(left)
                a6 = A[IDs - nx - 1]   #(upper-left)
                a7 = A[IDs - nx]       #(top)
                
                #----------------------------------
                # Flow codes of 8 neighbor pixels
                #----------------------------------
                d0 = dirs[IDs - nx + 1]   #(upper-right)
                d1 = dirs[IDs + 1]        #(right)
                d2 = dirs[IDs + nx + 1]   #(lower-right)
                d3 = dirs[IDs + nx]       #(bottom)
                d4 = dirs[IDs + nx - 1]   #(lower-left)
                d5 = dirs[IDs - 1]        #(left)
                d6 = dirs[IDs - nx - 1]   #(upper-left)
                d7 = dirs[IDs - nx]       #(top)
                
                #----------------------------------
                # A pixel is not READY if any of
                # it's children have unknown area
                #----------------------------------
                # Pixels w/ no children are READY
                # unless they are "edge" pixels
                #----------------------------------
                w0 = where(logical_and((d0 == h[0]), (a0 <= 0)))
                nw0 = size(w0[0])
                if (nw0 != 0):    
                    READY[w0] = uint8(0)
                w1 = where(logical_and((d1 == h[1]), (a1 <= 0)))
                nw1 = size(w1[0])
                if (nw1 != 0):    
                    READY[w1] = uint8(0)
                w2 = where(logical_and((d2 == h[2]), (a2 <= 0)))
                nw2 = size(w2[0])
                if (nw2 != 0):    
                    READY[w2] = uint8(0)
                w3 = where(logical_and((d3 == h[3]), (a3 <= 0)))
                nw3 = size(w3[0])
                if (nw3 != 0):    
                    READY[w3] = uint8(0)
                w4 = where(logical_and((d4 == h[4]), (a4 <= 0)))
                nw4 = size(w4[0])
                if (nw4 != 0):    
                    READY[w4] = uint8(0)
                w5 = where(logical_and((d5 == h[5]), (a5 <= 0)))
                nw5 = size(w5[0])
                if (nw5 != 0):    
                    READY[w5] = uint8(0)
                w6 = where(logical_and((d6 == h[6]), (a6 <= 0)))
                nw6 = size(w6[0])
                if (nw6 != 0):    
                    READY[w6] = uint8(0)
                w7 = where(logical_and((d7 == h[7]), (a7 <= 0)))
                nw7 = size(w7[0])
                if (nw7 != 0):    
                    READY[w7] = uint8(0)
                
                #----------------------------------------
                # If a pixel is my child and I'm READY,
                # then add it's value to mine.
                #----------------------------------------
                WR      = where(READY)
                n_ready = size(WR[0])
                WR      = WR[0]   ##### (Need this) #####
                
                if (n_ready != 0):    
                    #---------------------------
                    # This also assigns areas
                    # to pixels w/ no children
                    #---------------------------
                    ### PASS_CHANGE  = True
                    STILL_ACTIVE = True
                    UWR = IDs[WR]         #####
                    A[UWR] = pixel_area
                    
                    w0 = where(d0[WR] == h[0])
                    nw0 = size(w0[0])
                    if (nw0 != 0):    
                        A[UWR[w0]] += a0[WR[w0]]
                    w1 = where(d1[WR] == h[1])
                    nw1 = size(w1[0])
                    if (nw1 != 0):    
                        A[UWR[w1]] += a1[WR[w1]]
                    w2 = where(d2[WR] == h[2])
                    nw2 = size(w2[0])
                    if (nw2 != 0):    
                        A[UWR[w2]] += a2[WR[w2]]
                    w3 = where(d3[WR] == h[3])
                    nw3 = size(w3[0])
                    if (nw3 != 0):    
                        A[UWR[w3]] += a3[WR[w3]]
                    w4 = where(d4[WR] == h[4])
                    nw4 = size(w4[0])
                    if (nw4 != 0):    
                        A[UWR[w4]] += a4[WR[w4]]
                    w5 = where(d5[WR] == h[5])
                    nw5 = size(w5[0])
                    if (nw5 != 0):    
                        A[UWR[w5]] += a5[WR[w5]]
                    w6 = where(d6[WR] == h[6])
                    nw6 = size(w6[0])
                    if (nw6 != 0):    
                        A[UWR[w6]] += a6[WR[w6]]
                    w7 = where(d7[WR] == h[7])
                    nw7 = size(w7[0])
                    if (nw7 != 0):    
                        A[UWR[w7]] += a7[WR[w7]]
                
                #------------------------
                # Are we finished now ?
                #------------------------
                if (n_unknown != 0) and (n_ready != n_unknown):   
                    UNFINISHED = True
                else:    
                    UNFINISHED = False
                
            else:    
                UNFINISHED = False

            if not(UNFINISHED) or not(STILL_ACTIVE):
                break
        
        if (UNFINISHED):    
            print 'Upstream area not defined for all pixels.'

        #-----------------------------
        # Save area grid for testing
        #-----------------------------
        SAVE_TEST = False
        if (SAVE_TEST):
            file_unit = open('00_AREA_GRID_TEST.rtg', 'wb')
            if (self.rti.SWAP_ENDIAN):
                grid = self.A.copy()
                grid.byteswap(True)
                grid.tofile(file_unit)
            else:
                self.A.tofile( file_unit )
            file_unit.close()
        
        #------------------
        # Optional report
        #------------------
        if (REPORT):
            A_str = str(self.A.min()) + ', ' + str(self.A.max())
            print '    min(A), max(A) = ' + A_str + ' [km^2]'
            print '    Number of iterations = ' + str(n_reps)
    
    #   update_area_grid_OLD()
    #-------------------------------------------------------------------
    #-------------------------------------------------------------------    
    def get_flow_width_grid(self, DOUBLE=False, METHOD2=False,
                            SILENT=True, REPORT=False):

        #-------------------------------------------------------------
        # NOTES: This routine returns the flow widths for each
        #        pixel in the DEM.  The extra width of flowing to a
        #        diagonal pixel is taken into account, as well as the
        #        lat/lon-dependence for DEMs with fixed-angle pixels
        #        (Geographic lat/lon).

        #        METHOD2 version ensures that the sum of all flow
        #        widths around a pixel is equal to 2*(dx + dy), but
        #        is incorrect for case of a plane and others.

        #        Flow widths are zero where (flow grid eq 0).
        
        #        Flow widths are for entire pixel and are appropriate
        #        for overland or subsurface flow.
        #------------------------------------------------------------- 
        #        Is this only used by Seepage function now ???
        #-------------------------------------------------------------
        # NB!    numpy.where returns a 2-tuple, but with "empty"
        #        second part when applied to a 1D array.  This means
        #        that nw = size(w[0]) still works.
        #-------------------------------------------------------------        
        if not(SILENT):
            print 'Computing flow width grid...'

        #-----------------------
        # Get pixel dimensions
        #-----------------------
        dx, dy, dd = pixels.get_sizes_by_row(self.rti, METERS=True)

        #-------------------------
        # Double or Float type ?
        #-------------------------
        if (DOUBLE):    
            dw = zeros([self.ny, self.nx], dtype='Float64')
        else:    
            dw = zeros([self.ny, self.nx], dtype='Float32')
            dx = float32(dx)
            dy = float32(dy)
            dd = float32(dd)

        #----------------------------------------------
        # Initialize to default value that is used
        # for pixels with flow code of zero.  This is
        # done to avoid "divide by zero" errors.
        #----------------------------------------------
        dw += dx[0]
        ## dw = dw + dx.min()
        
        for row in xrange(self.ny):
            g = self.flow_grid[row,:]
            
            #----------------
            # Diagonal flow
            #----------------
            wd  = where(logical_or(logical_or(logical_or((g == 1), (g == 4)), \
                                              (g == 16)), (g == 64)))
            nwd = size(wd[0])
            if (nwd != 0):    
                if not(METHOD2):    
                    dw[row, wd] = dd[row]
                else:    
                    dw[row, wd] = (dx[row] + dy[row]) / 4
            
            #---------------------
            # East and west flow
            #---------------------
            wh  = where(logical_or((g == 2), (g == 32)))
            nwh = size(wh[0])
            if (nwh != 0):    
                dw[row, wh] = dy[row]
                if (METHOD2):    
                    dw[row, wh] = dw[row,wh] / 2
            
            #-----------------------
            # North and south flow
            #-----------------------
            wv  = where(logical_or((g == 8), (g == 128)))
            nwv = size(wv[0])
            if (nwv != 0):    
                dw[row, wv] = dx[row]
                if (METHOD2):    
                    dw[row, wv] = dw[row, wv] / 2

        #------------------
        # Optional report
        #------------------
        if (REPORT):
            print '    min(dw) = ' + str(dw.min()) + '  [m]'
            print '    max(dw) = ' + str(dw.max()) + '  [m]'

        self.dw = dw

    #   get_flow_width_grid()
    #-------------------------------------------------------------------
    def get_flow_length_grid(self, DOUBLE=False,
                             SILENT=True, REPORT=False):

        #-------------------------------------------------------------
        # NOTES: This routine returns the flow lengths for each
        #        pixel in the DEM.  The extra length of flowing to a
        #        diagonal pixel is taken into account, as well as the
        #        latitude-dependence for DEMs with fixed-angle pixels
        #        (Geographic lat/lon).

        #        The only difference between this and the Flow_Widths
        #        function is that roles of dx and dy are switched.

        #        Flow lengths are set to dx[0] for the pixels where
        #       (flow grid eq 0), such as on the edges of the DEM.
        #-------------------------------------------------------------
        if not(SILENT):
            print 'Computing flow length grid...'

        #-----------------------
        # Get pixel dimensions
        #-----------------------
        dx, dy, dd = pixels.get_sizes_by_row(self.rti, METERS=True)
            
        #-------------------------
        # Double or Float type ?
        #-------------------------
        if (DOUBLE):    
            ds = zeros([self.ny, self.nx], dtype='Float64')
        else:    
            ds = zeros([self.ny, self.nx], dtype='Float32')
            dx = float32(dx)
            dy = float32(dy)
            dd = float32(dd)
        
        #----------------------------------------------
        # Initialize to default value that is used
        # for pixels with flow code of zero.  This is
        # done to avoid "divide by zero" errors.
        #----------------------------------------------
        ds += dx[0]
        ## ds += dx.min()
        
        for row in xrange(self.ny):
            g = self.flow_grid[row,:]
            
            #----------------
            # Diagonal flow
            #----------------
            wd  = where(logical_or(logical_or(logical_or((g == 1), (g == 4)), \
                                              (g == 16)), (g == 64)))
            nwd = size(wd[0])
            if (nwd != 0):    
                ds[row, wd] = dd[row]
            
            #---------------------
            # East and west flow
            #---------------------
            wh  = where(logical_or((g == 2), (g == 32)))
            nwh = size(wh[0])
            if (nwh != 0):    
                ds[row, wh] = dx[row]
            
            #-----------------------
            # North and south flow
            #-----------------------
            wv  = where(logical_or((g == 8), (g == 128)))
            nwv = size(wv[0])
            if (nwv != 0):    
                ds[row, wv] = dy[row]

        #------------------
        # Optional report
        #------------------
        if (REPORT):
            print '    min(ds) = ' + str(ds.min()) + '  [m]'
            print '    max(ds) = ' + str(ds.max()) + '  [m]'

        self.ds = ds

    #   get_flow_length_grid()
    #-------------------------------------------------------------------

