/*! \file Success.c
\brief <b>Dynamic equations:</b>  Vertical solutions of spatial habitat-type succession.

This source file determines the switching of habitats (succession). \n
  
Note: documented with Doxygen, which expects specific syntax within special comments. \n
	
The Everglades Landscape Model (ELM). \n
last updated: Jan 2005 \n
*/
	
/* General notes on revisions to this source file. 
       Nov/Dec 2004 v2.3.2: documentation upgrade 
       		- Doxygen tags added 
	
*/

#include "success.h"


/*! \brief Transfers habitat-specific parameters into data struct, allocate memory. 
 */
void HabSwitch_Init(void)
{
  int ii;
  
	/* put these habitat-specific parameters into a concise data struct for use here */
  for( ii = 0; ii < habNumTot; ii++) {
	Habi[ii].Water.Llo = HP_SfDepthLo[ii+1];
	Habi[ii].Water.Lhi = HP_SfDepthHi[ii+1];
	Habi[ii].Water.Pin = HP_SfDepthInt[ii+1];
	Habi[ii].Nutrient.Llo = HP_PhosLo[ii+1];
	Habi[ii].Nutrient.Lhi = HP_PhosHi[ii+1];
	Habi[ii].Nutrient.Pin = HP_PhosInt[ii+1];
	Habi[ii].PFin = HP_FireInt[ii+1];
  }
   
   alloc_hab_hist( );
   sprintf (msgStr, "Succession ON, module OK for %d habitats...", ii ); usrErr(msgStr); WriteMsg(msgStr,1);

   return; 
}


/*! \brief Switches habitats (the option that is currently used) 

	Returns the number of habitat to switch to based on the length of the weekly
	(or other) averaged period of favorable conditions.
	In this case the period of habitat favorable conditions is calculated as the number of
	weeks (or other time intervals) during which the habitat conditions matched the
	specified for a period longer than the given one - SW_TIME_TH. The switching occurs as soon
	as the number of such successfull weeks exceeds the time specified in Pin
	\param ix Model domain row 
	\param iy Model domain column 
	\param Water Current water depth data array 
	\param Nutrient Current nutrient conc. data array 
	\param Fire Fire data array (unused, no fire implemented)
	\param HAB Habitat-type data array 
	\return habitat type of cell */
unsigned char HabSwitch(int ix, int iy, float *Water, float *Nutrient, int *Fire, unsigned char *HAB)
{
 int StateW, StateN, i;
 int HabW[MAX_NHAB], HabN[MAX_NHAB], DW, DN;
 int cell =  T(ix,iy);
 int hab = HAB[cell];	/* current habitat in the cell */

 /* define the habitat type that matches the existing water conditions */
 
 /* HabHist is an array of integers : nnNssS, where nn are the two digits for the number of
 	weeks being in the nutrient favorable conditions, N the number of days in those conditions 
 	during the current week; similarly ss are the two digits for the number of weeks in the
 	water level favorable conditions and S is the number of days during the current week in
 	that state. The switching occurs when both of these histories exceed the habitat specific 
 	periods  Pin.
 */
 for ( i = 0; i < habNumTot; i++ )
	{
	 DW = HabHist[cell*habNumTot +i]%10;
	 HabW[i] = (HabHist[cell*habNumTot +i]/10)%100;
	 DN = (HabHist[cell*habNumTot +i]/1000)%10;
	 HabN[i] = HabHist[cell*habNumTot +i]/10000;
	 
 	 /* when the averaging time elapses, if #favorable days exceeds a threshold (using 4 for 7 day AV_PER), increment the period (weeks) history */
 	 if ((int)SimTime.TIME%AV_PER == 0)
 	   { 
 	 	if (DW > SW_TIME_TH_W) HabHist[cell*habNumTot +i] = HabHist[cell*habNumTot +i] - DW + 100;
 	 	else HabHist[cell*habNumTot +i] = 0;
 	 	if (DN > SW_TIME_TH_N) HabHist[cell*habNumTot +i] = HabHist[cell*habNumTot +i] - DN*1000 + 10000;
 	 	else HabHist[cell*habNumTot +i] = 0;
	   }	

 	/* check what habitat type the existing conditions match; increment the day# if conditions are favorable */
  	 if ( InHab (Water[cell], Habi[i].Water) )  HabHist[cell*habNumTot + i]++;
 	 if ( InHab (Nutrient[cell]*conv_kgTOmg, Habi[i].Nutrient) )  
 	 		HabHist[cell*habNumTot + i] = HabHist[cell*habNumTot + i] + 1000;
 	} 
   
	 /* check if the historical conditions for switching hold */
 for ( i = 0; i < habNumTot; i++ )  
	if ( HabW[i] >= Habi[i].Water.Pin && HabN[i] >= Habi[i].Nutrient.Pin ) 
		{ HabHist[cell*habNumTot +i] = 0;
		  return (i+1); /* returns new hab, HAB[] is offset by 1 from these array IDs */
		}
		
 return hab;	
}

/*! \brief Defines the suitability of current habitat relative to existing conditions.
	\param Var The model variable being considered
	\param Params Struct of habitat switching parameters  
	\return true/false */
int InHab(float Var, struct HabData Params )
{
  if ( (Var <= Params.Lhi || Params.Lhi >= MAX_SW) && Var >= Params.Llo ) 
 	  return 1;
  else return 0;
}


/*! \brief Allocate memory for habitat history info */
void alloc_hab_hist( void )
{
  HabHist = (unsigned long *) nalloc(sizeof(unsigned long)*(s0+2)*(s1+2)*(habNumTot),"HabHist");
  init_pvar(HabHist,NULL,'i',0);
}


 /**********************/
 