/*! \file MultiRun.c
\brief Multiple simulation runs per program execution.

Currently used in an integrated sensitivity analysis of model.  Anticipate uses
may include formal uncertainty analyses and stochastic ecological 
(e.g., fire) processes.

This source file contains several functions to acquire and set up a multi-run execution. \n

Note: documented with Doxygen, which expects specific syntax within special comments. \n
	
The Everglades Landscape Model (ELM). \n
last updated: Feb 2005 \n
*/

/* General notes on revisions to this source file. 
       Feb 2005 v2.4.0 initial development
	
*/


#include "multi_run.h"


/*!\brief Determine the number of simulation runs per program execution.
	\param SParmNum the number of parameters to be evaluated in a sensitivity analysis
	\param NumSRunsPerSet The number of simulation runs per set of parameter-evaluations (currently using one low-parameter, one high-parameter value)
	\return numberRuns The number of simulation runs per program execution

*/

int numRuns(int SParmNum, int NumSRunsPerSet)
{
   int numberRuns;
      
   numberRuns = (1+NumSRunsPerSet*SParmNum);
   	
   return numberRuns;
}

/*!\brief Get the parameter list for a multi-parameter sensitivity analysis.

	This does not check to ensure that a parameter in the list is
	actually a parameter found in the model.  If there is a typo
	in a parameter in this input list, the program will merely
	make a set of runs with all-nominal values of the intended parameter.  Therefore: use 
	the list of parameters that is generated from the GlobalParms.xls
	and HabParms_defs.xls (OpenOffice) workbooks.
	Yes, we can/will provide some checking at some point.
	
	\param sparm_name Name of the sensitivity parameter being varied for this run
	\return SParmNum the number of parameters to be evaluated in a sensitivity analysis

*/
int SensiParm_list(char* sparm_name) 
{
   FILE *SParm_listFile;
   char filename[20];
   char modelFileName[300];
   char ss[422], *line;
   int SParmNum = 0;
   
   if (strcmp(sparm_name,"ALL")!=0) return(1) ;
   
   sprintf(filename,"SensiParm_list"); 
   sprintf( modelFileName, "%s/%s/Data/%s", ModelPath, ProjName, filename );

/* Open file with list of parameters for Sensitivity Analysis */
 if ( ( SParm_listFile = fopen( modelFileName, "r" ) ) ==  NULL )
 {
     printf( "Can't open %s file!\n", modelFileName ) ;
     exit(-1) ;
 }

    sprintf(msgStr, "Sensitivity Analysis Parameters: ") ;
    WriteMsg(msgStr,1); 
    usrErr(msgStr);
   
    fgets( ss, 420, SParm_listFile );
    fgets( ss, 420, SParm_listFile ); 
    fgets( ss, 420, SParm_listFile ); /* skip header lines */
   
    while ( fgets( ss, 420, SParm_listFile ) != NULL && !feof( SParm_listFile ) )
    {
		/* Allocate memory  for next parm set (initially, we have already gotten the info for the nominal run into the struct w/ SParmNum = 0) */
    if ( (ProgExec = ( ProgAttr *) malloc( (size_t) sizeof( ProgAttr ))) == NULL ) {
        sprintf(msgStr, "Failed to allocate memory for next struct defining multiple simulation runs.\n ") ;
        usrErr(msgStr);
        exit( -2 ) ;
    }
        line = ss;
        sscanf (line, "%s", &ProgExec->S_ParmName);

        SParmNum++; /* count the number of parms to be evaluated in sensi */
        RunList[SParmNum] = ProgExec;
        ProgExec->S_ParmRelVal = 0; /* init to 0 for all */
        
        sprintf(msgStr, "%d: %s", SParmNum, ProgExec->S_ParmName) ;
        WriteMsg(msgStr,1); 
        usrErr(msgStr);
    

     
    } /* end of reading basin lines */
    
    fclose(SParm_listFile);
    return (SParmNum);

}


/*!\brief Allocate memory for data structs used in multiple-run sensitivity analyses.
	\param SParmNum the number of parameters to be evaluated in a sensitivity analysis

*/
void alloc_mem_runs(int SParmNum) 
{
    if ( (ProgExec = ( ProgAttr *) malloc( (size_t) sizeof( ProgAttr ))) == NULL ) {
        sprintf(msgStr, "Failed to allocate memory for first struct defining (possibly multiple) simulation run(s).\n ") ;
        usrErr(msgStr);
        exit( -2 ) ;
    }
        /* allocate memory for array of pointers to simulation run attributes */
    if ( (RunList = ( ProgAttr **) malloc( (size_t) sizeof( ProgAttr *) * (SParmNum+1) ) ) == NULL )
    {
        printf( "Failed to allocate memory for RunList of simulation attributes \n " ); 
        exit( -2 );
    }


}

/****************************************************************************/

