/*! \file gridmap.c 
\brief Re-map data from one grid resolution to that needed in ELM.  

This uses the SFWMD "grid_io" functions to acquire/process grid_io data at a 
scale that is coarser than ELM.  These input data are used
as boundary conditions for variables such as stage/depth, rainfall & potential ET.

Note: documented with Doxygen, which expects specific syntax within special comments. \n
	
The Everglades Landscape Model (ELM). \n
last updated: Jan 2005 \n
*/

/* ### version history: ###
   v2.2.0 first used in ELM code
   v2.2.1 changes:
          provided exit instead of return on critical errors
          has basic functionality
   v2.2.2
          (need to?) reforumlate console messages for consistency
          need to clean up old code that is commented out when possible/desirable
          fixed bug, putting dayCntr, recCntr, recRead, step, day, and skip into data structure
   v2.3.1
          fixed documentation, modified vars, added doxy tags
*/


#include "gridmap.h"

int NOT_ELM_NOT_GRIO = -1;    /* index = -1 and elmID = -1, data pointer from grid_io (values) not to be incremented */ 
int NOT_ELM_IN_GRIO = -2;     /* index = -2 and elmID = -1,inc. values */
int IN_ELM_NOT_GRIO = -3;     /* index = -3 and elmID = some value, no inc. values */ 

#define A_NEG_VAL -90		/* use unsure */

/*! \brief Do some simple formatting of numeric date to text.

	The grid_io tag has a strict alphanumeric date format.
	\param sd starting date requested */
void formatDate(char* sd) {

  char month[12],yr[5],mo[3],da[3], dig1[2], dig2[2], day_str[5] = "";
  char sdate[GRID_TAG_LENGTH];
  
  sscanf(initDateRead, "%4s/%2s/%2s",yr,mo,da); /* initDateRead fmt= 1999/01/01 */
  if (atoi(mo) == 1) strcpy(month,"January ");
  else if (atoi(mo) == 2) strcpy(month,"February ");
  else if (atoi(mo) == 3) strcpy(month,"March ");
  else if (atoi(mo) == 4) strcpy(month,"April ");
  else if (atoi(mo) == 5) strcpy(month,"May ");
  else if (atoi(mo) == 6) strcpy(month,"June ");
  else if (atoi(mo) == 7) strcpy(month,"July ");
  else if (atoi(mo) == 8) strcpy(month,"August ");
  else if (atoi(mo) == 9) strcpy(month,"September ");
  else if (atoi(mo) == 10) strcpy(month,"October ");
  else if (atoi(mo) == 11) strcpy(month,"November ");
  else if (atoi(mo) == 12) strcpy(month,"December ");

  strcpy(sdate,month);

  sscanf(da, "%1s%1s\n",dig1,dig2);
  if(strcmp(dig1,"0") != 0 )
    strcpy(day_str,dig1);
  else strcat(day_str," "); /* need a blank space for days<10 */
  strcat(day_str,dig2);
  strcat(day_str,", ");     /* put comma and space after day */
  strcat(sdate,day_str);
  strcat(sdate,yr);         /* tag in grid_io file is "January  1, 1965", "March 11, 1965" */
  strcpy(sd,sdate);  
}


/*! \brief Populate the elm_OG_map grid struct with values.

	The data are read from grmap pre-processor program output file (gridmapping.txt), located in 
	the standard ELM input-data directory.
	\param gridmapfilename filename of the pre-processed grid-mapping indices
	\return success or exit program  */
int mapGrids(char *gridmapfilename) {

  int i;
  int success = 1, fail = -1;
  char line[80];
  	/*! \em gridmapfile_ptr Pointer to the text file that maps the relationship between ELM and the other grid */
  FILE *gridmapfile_ptr;

  /*! \em ogid Other-Grid cell ID read from input grid-mapping file */
  int ogid=0; 
  /*! \em ogr \em ogc Other-Grid cell row (ogr), column (ogc), read from input grid-mapping file */
  int ogr=0, ogc=0;  
  /*! \em eid ELM cell ID read from input grid-mapping file */
  int eid=0;  
	/*! \em ii Counter that is same as OGidCnt - it is just local to this function */
  int ii=0;
	/*! \em jj Counter for eid */
  int jj=0; 
  
	/*! \em eINog The maximum number of smaller (elm) cells contained IN the bigger (other-grid) cell read from input grid-mapping file */
  int eINog=0;   
  for(i=0; i<80; i++)
    line[i]=' ';
  
  /* open the file */
  if((gridmapfile_ptr = fopen(gridmapfilename,"r")) == NULL) {
    printf( "Can't open the grid-map definition file %s!\n ",gridmapfilename ) ;
      exit (fail);
  }


  /* skip the (5) header lines in the input file, replicated here: 

     This is coarser grid ids mapped to finer grid ids
     Note that the mapped id = -1 means no cell intersect between 2 grids.
     ID      ROW     COL     MAPPED_ID
     ==      ===     ===     =========
     LINE BELOW IS RESERVED FOR APPLICATION USE, DEFAULT VALUE:
   */

  fgets(line,80,gridmapfile_ptr); /* datafile header line 1 */
  fgets(line,80,gridmapfile_ptr); /* datafile header line 2 */
  fgets(line,80,gridmapfile_ptr); /* datafile header line 3 */
  fgets(line,80,gridmapfile_ptr); /* datafile header line 4 */
  fgets(line,80,gridmapfile_ptr); /* datafile header line 5 */

  /* read the cell size mapping, row and col information for initializing arrays */
  fgets(line,80,gridmapfile_ptr);
  sscanf(line,"%d %d %d %d",&eINog, &OGrow_tot, &OGcol_tot, &gridio_batch_len);

  /* after reading the line, initialize array of struct of size OGrow_tot * OGcol_tot */
  if(elm_OG_map==NULL) elm_OG_map = (struct eOGMap*) malloc(sizeof(struct eOGMap ) * OGrow_tot * OGcol_tot);
  if(elm_OG_map==NULL) {
    printf("\nFailed to allocate memory for elm_OG_map\n");
    exit (fail);
  }
    
  /* read the cell size mapping to initialize the elmID pointer */
  for(ii=0; ii<OGrow_tot*OGcol_tot; ii++) {
  /* memory issue? */
    /* if(elm_OG_map[ii].elmID==NULL) */ elm_OG_map[ii].elmID = (int*)malloc(sizeof(int)*eINog);
    if(elm_OG_map[ii].elmID==NULL) {
      printf("\nFailed to allocate memory for grid-map elmID\n");
      exit (fail);
    }
  }
  for(ii=0; ii<OGrow_tot*OGcol_tot; ii++) {
    elm_OG_map[ii].eidCnt=0;
    elm_OG_map[ii].OGID=0;
    elm_OG_map[ii].OGROW=0;
    elm_OG_map[ii].OGCOL=0;
    for(jj=0; jj<eINog; jj++) {
      elm_OG_map[ii].elmID[jj]= A_NEG_VAL; /* struct has memory issue? */
    }
  }  
  ii=0, jj=0, OGidCnt = 0;  /* this count may be smaller or larger than (OGrow_tot*OGcol_tot) 
                          * because it contains Other-Grid cells
                          * up to elm boundary (i.e., 1-2209 in case of SFWMM grid) 
                          */

  /* Not all the OG cells are included in the gridmapping file.
     Only from OG cell 1 to the upper limit of elm.
     i.e., in case of SFWMM cell is 2209.
   */
	/* this is the first line of data of the cell-by-cell mapping attributes */
  fgets(line,80,gridmapfile_ptr);
  sscanf(line,"%d %d %d %d",&elm_OG_map[ii].OGID,&elm_OG_map[ii].OGROW,
                            &elm_OG_map[ii].OGCOL,&elm_OG_map[ii].elmID[jj]);

  elm_OG_map[ii].index = ii;  /* init the first index to 0 (value of ii at this point) */
  elm_OG_map[ii].eidCnt++;   /* increment to show we've read in one more (actually, just one at this point) elmID */
  OGidCnt++;                /* increment to show we've read in one more (actually, just one at this point) OGID */
  ii++;                     /* increment to show we've read in one more (actually, just one at this point) elm_OG_map struct */

  while(fgets(line,80,gridmapfile_ptr) != NULL) {
    sscanf(line,"%d %d %d %d",&ogid,&ogr,&ogc,&eid);  
    if(ogid == elm_OG_map[ii-1].OGID) { 
      /* when OGID is same as next line OGID, the structure
         only must be updated for the elmID and eidCnt, we do repeat though! 
         The index will be modified later on according
         to how elm cell falls in OG cell. This index
         is same as ii and OGidCnt in this loop */
      jj++; /* counter for eid */
      elm_OG_map[ii-1].elmID[jj] = eid;
      elm_OG_map[ii-1].OGROW = ogr; 
      elm_OG_map[ii-1].OGCOL = ogc;
      elm_OG_map[ii-1].eidCnt++;
      elm_OG_map[ii-1].index = ii-1;    
    }
    else {
      jj=0;
      elm_OG_map[ii].OGID = ogid;
      elm_OG_map[ii].elmID[jj] = eid;
      elm_OG_map[ii].OGROW = ogr;
      elm_OG_map[ii].OGCOL = ogc;
      elm_OG_map[ii].index = ii;
      elm_OG_map[ii].eidCnt++;

      OGidCnt++;  /* counts number of OG cell id's, used as a global variable for later on */
      ii++;       /* ii is same as OGidCnt - it is just local to this function */
    }
    
  } /* end of while-reading of data line/records */
  fclose(gridmapfile_ptr);
  return success;
}

/*! \brief Initialize data structures.

	This function at time=0 initializes the data structures for ELM values 
   of size (s0xs1) and 2d([gridio_batch_len][OGrow_tot*OGcol_tot]) array of grid_io data for holding Other-Grid 
   data values. Also reads an grid_io binary file and moves the file pointer
   to the initial date on the record.
   
   \param binfilename file name of grid_io data
   \param this_struct struct of the grid information
   \return success or exit from program
 */
int initDataStruct(char* binfilename, applicationStruct *this_struct) {

  char sdate[GRID_TAG_LENGTH];
  int comp;

  int i, j; /* short loop counters */
  int fail = -1, success = 1;
  int pos; /* not used except in assignment */
  

  this_struct->day = this_struct->recRead = this_struct->step = 0;

  if(this_struct->dataELM==NULL) this_struct->dataELM = (float*)malloc(sizeof(float)*s0*s1);
  if(this_struct->dataELM==NULL) {
    printf("\nFailed to allocate memory for grid-map dataELM\n");
    exit (fail);
  }
  for(i=0;i<s0*s1;i++) {
    this_struct->dataELM[i]=0.0;
  }


  /* initialize the variable for holding multiday data values in GRID grid */
  /* initialize array of pointers; each points to  OGrow_tot*OGcol_tot (columns) floats */
  if(this_struct->dataAnyGrid==NULL) this_struct->dataAnyGrid = (float**)malloc( gridio_batch_len * sizeof(float*) );
  if(this_struct->dataAnyGrid==NULL) {
    printf("\nFailed to allocate memory for grid-map dataAnyGrid\n");
    exit (fail);
  }

  for(i=0; i<gridio_batch_len; i++) {
    if(this_struct->dataAnyGrid[i]==NULL) this_struct->dataAnyGrid[i] = (float*) malloc(sizeof(float) * (OGrow_tot * OGcol_tot) ) ;
    if(this_struct->dataAnyGrid[i]==NULL) {
      printf("\nFailed to allocate memory for grid-map gridio_batch_len[%d]\n",i);
      exit (fail);
    }
    for (j=0; j<OGrow_tot*OGcol_tot; j++) {
      this_struct->dataAnyGrid[i][j]=0.0;
    }
  }


  /* open the grid_io data file */
  if((this_struct->binfile_ptr = fopen(binfilename,"rb")) == NULL) {
    printf( "Can't open the grid_io binary data %s file!\n ", binfilename) ;
    exit (fail);
  }


  /* read the header and config part of the grid */
  if(grid_read_header(this_struct->binfile_ptr, &this_struct->grid) != 0) {
    fprintf(stdout, "Unable to read grid header from grid_io input file %s\n", binfilename);
    exit (fail);
  }
  /* initialize the variable for holding the data values */
  /* this is used by first come first served client. is just a temp storage
     that is initialized */
  if(this_struct->values==NULL) this_struct->values = (float*)nalloc(this_struct->grid.header.number_of_nodes*sizeof(float),"values");
  if(this_struct->values==NULL) {
    printf("\nFailed to allocate memory for grid_io values array\n");
    exit (fail);
  }
  for(i=0; i< this_struct->grid.header.number_of_nodes; ++i) {
    this_struct->values[i] = 0.0;
  }

  /* reformat the starting date for comparison with grid_io tag format*/
  formatDate(sdate);


  /* keep comparing the dates in binary file and the requested date
     by the model. if found, grid_skip to the beginning. */
  comp = 1;
  while(comp != 0) {
    if(grid_read(this_struct->binfile_ptr, &this_struct->grid, this_struct->tag, this_struct->values ) == -1 ) {
      fprintf(stdout, "Couldn't match starting date for date = %s in %s\n",sdate, binfilename);
      exit (fail);
    }
    comp = strncmp(this_struct->tag, sdate, strlen(sdate));

    if(comp == 0) {
      pos = grid_skip(this_struct->binfile_ptr, &this_struct->grid, -1);
    }

    this_struct->skip++; /* number of records skipped to get to starting date */

  }  /* end of while */

  return success;
}


/*! \brief Process/acquire the grid_io data (at coarse scale).

	Checks the number of days simulated vs the
	number of records (arrays) read so far, and puts the data
	into memory, awaiting for returning the data for use.
   \param binfilename file name of grid_io data
   \param this_struct struct of the grid information
   \return success or exit from program
 */
int processData(char* binfilename, applicationStruct *this_struct) {

  int kk=0, ii=0, jj=0;
  int jstart = 0; /* starting row of grid_io data */
  int jend = 0;  /* ending row of grid_io data */
  int ogid = 0; /* Other-Grid number of cells within elm boundary*/
  int cntr = 0; /* grid_io data node counter*/
  int this_sz = 0; /* only used in assignment statement */
  int row_bound = 0;
  
  float val = 0.0;

  int fail = -1, success = 1;

 /* TODO: determine what comment below means */
 /* don't use the logic "if(SimTime.TIME>0) condition", does not work if
     gridio_batch_len is smaller than N_Itr, and also, SimTime.TIME
     gets undefined with very large values after many
     iteration when using cc compiler.
   */
  if (this_struct->step >= gridio_batch_len) {   /* read more records */

    /* reopen this file and skip to the appropriate record */
    if((this_struct->binfile_ptr = fopen(binfilename, "rb")) == NULL) {
      printf("Can't open the rainfall binary %s file!\n",binfilename ) ;
      exit (fail);
    }

    if(grid_read_header(this_struct->binfile_ptr, &this_struct->grid) != 0) {
      fprintf(stdout, "Unable to read grid header from input file %s\n",binfilename);
      exit (fail);
    }
 
    this_sz = grid_skip(this_struct->binfile_ptr, &this_struct->grid, (this_struct->skip+this_struct->recRead-1));

  } 

  this_struct->step = 0;
 
  /* recRead and PORnumday (global, number of days in sim Period of Record) and gridio_batch_len happens once
   * per batch reading. This while populates the other grid structure
   * with available data read from grid_io. Those Other-Grid (OG) cells that are
   * not within grid_io boundary are filled with the boundary cells values.
   */

  /* PORnumday is established by user by input to Driver.parm file */
  while(this_struct->recRead < PORnumday && this_struct->step < gridio_batch_len ) {
    if(grid_read(this_struct->binfile_ptr, &this_struct->grid, this_struct->tag, this_struct->values ) != 0 ) {
      fprintf(stdout, "Unable to read grid value from input file %s\n",binfilename);
      exit (fail);
    }

    ogid = 0;    
    cntr = 0;   
    val = 0.0;

    /* binary grid_io contains X rows, we need up to
     * elm boundary which is number of OG cells within
     * elm boundary. Although the elm_OG_map
     * is initialized by OGrow_tot*OGcol_tot, by reading the data
     * from the gridmapping.txt file, we read only as many as
     * OG rows that are within elm boundary. This is
     * represented by the row_bound, calculated below.
     */

    row_bound = OGidCnt / OGcol_tot;
    for(ii=0; ii< row_bound; ++ii) {
      jstart =  *(this_struct->grid.config.xstart+ii);
      jend =  *(this_struct->grid.config.xend+ii);
      val = this_struct->values[cntr];

      /* fill the OG cells not in grid_io (missings) with the value in jstart
         we do this to make sure every elm cell that does not map to grid_io but
         maps to OG cells, contains value
       */
      for(kk=1; kk<jstart; kk++, ogid++)
        this_struct->dataAnyGrid[this_struct->step][ogid] = val;

      /* because using the last grid_io cell boundary value
       * for the OG adjacent empty cell, we can use 'jj<jend'
       * instead if 'jj<=jend'
       */
      for(jj=jstart; jj<jend; jj++, ogid++, cntr++) {
        this_struct->dataAnyGrid[this_struct->step][ogid] = this_struct->values[cntr];
      }

      /* fill the missings with jend value */
      for(kk=jend; kk<OGcol_tot+1; kk++, ogid++) {
        this_struct->dataAnyGrid[this_struct->step][ogid] = this_struct->values[cntr];
      }
       
      cntr++;
    }
    this_struct->recRead = this_struct->recRead + 1;    /* increment the total number of records read */
    this_struct->step = this_struct->step +1; /* increment the number of steps taken in reading this batch of records */
  } /* end while for reading records */
 
  if(this_struct->step > 0 )
    fclose (this_struct->binfile_ptr);    /* close the binary file after each group of files read */
 
  this_struct->step = 0;              /* set the step to 0, so that returnData will start at the beginning */
}



/*! \brief Map the coarse-scale data from Other-Grid to the ELM grid scale.

	At this time, we have read enough records (data arrays) into memory,
	and this function remaps the Other-Grid data for the day to the
	ELM/SME grid data structure (passing it to the caller function in UnitMod.c).
	This function is called daily; however the data records are
	already processed  up through gridio_batch_len.
	
	\param dataSME Data array at the ELM/SME resolution
	\param this_struct struct of the grid information  */
void returnData(float* dataSME, applicationStruct *this_struct) {

  int i=0, j=0;
  int ii=0;
  int index=0;
/*  int eid=0; */

  this_struct->dayCntr += 1;                /* for printing purpose only */
  this_struct->recCntr = this_struct->recRead - this_struct->dayCntr;
 

  /* init for the bottom part of elm that does not have data is done in initData */
 
  /* map wmm data to elm data */
  for(ii=0; ii<OGidCnt; ++ii) {

    if(debug > 4) { 
      index = elm_OG_map[ii].index;
      printf("%d %d %d %d %d %3.3f\n",index, elm_OG_map[ii].OGID, elm_OG_map[ii].OGROW,
                                      elm_OG_map[ii].OGCOL ,elm_OG_map[ii].elmID[0],
                                      this_struct->dataAnyGrid[this_struct->step][ii]);
    }

    if(elm_OG_map[ii].elmID[0] == NOT_ELM_NOT_GRIO)
      continue;

    for(i=0; i<elm_OG_map[ii].eidCnt; i++) {
      j = elm_OG_map[ii].elmID[i];

      this_struct->dataELM[j] = this_struct->dataAnyGrid[this_struct->step][ii];   /* * 254; */
    }
  }


  /* put in applxx_inp it is working 
  for(i = 0; i < s0; i++) {
    for(j = 0; j < s1; j++) {
      eid = i*s1+j;
      dataSME[T((i+1),(j+1))] = (int)(this_struct->dataELM[eid]);
    }
  } */

  this_struct->step = this_struct->step + 1; /* global, reset for the next run */
  this_struct->day = this_struct->day + 1;
}



/**********************
	EVERYTHING FROM HERE ON BELOW IS COMMENTED OUT OR UNUSED
*********************/

/* TODO: remove this function when sure is no longer needed */
/*! \brief UNUSED. Re-sample "grid_io" data for input to the ELM grid.

	Used for all the ELM cells that fall within some other (coarser-scale) rectangular grid of grid_io-formatted data. 
	Our source of data is the grid_io binary data
	that usually covers only the clipped area of South Florida Water Management Model (SFWMM) and not the full domain of the SFWMM
	(or any Other-Grid of grid_io data). Therefore, if ELM cells are within
	the Other-Grid, but there is no grid_io
	data available at that ELM grid location (outside of grid_io data domain), we need to estimate data for that
	portion of the Other-Grid cells as well.
	This function maps ELM cells to the Other-Grid cells and their data, if
	data are availabile. 
	The domain of both the ELM and the Other-Grid are rectangular, and the
	grid_io data is within the Other-Grid boundary area.
	
	\remarks The grid_io (usually, SFWMM-derived now) grid is constructed from a lower left origin,
	and cells are numbered starting at 1, increasing to the north and to the east.
	The ELM grid array is constructed from a upper left origin,
	and cells are numbered starting at 0, increasing to the south and to the east.
	
	\note All the columns in ELMv2.3 are within the SFWMM v5.4
	rectangular grid. Howeve, ELM rows from 175 to 184 are
	out of the SFWMM domain.
	In the (usual) case of the SFWMM, all grid_io cells are within the SFWMM rectangular grid.
	To map the 3 grids, the following conditions are being
	considered:
	\li 1) An Other-Grid cell does not map to ELM cell and does map to grid_io data.
	   It means:  data is available, but ELM does not need data.
	\li 2) An Other-Grid cell does not map to ELM cell and does not map to grid_io data.
	   It means:  data is not available, but ELM does not need data.
	\li 3) An Other-Grid cell does map to ELM cell and does map to grid_io
	   It means:  data is available and ELM does need data.
	\li 4) An Other-Grid cell does map to ELM cell and does not map to grid_io data.
	   It means:  data is not available, but ELM does need some data estimate.

	In this program we use the Other-Grid boundary cell data for the data-estimates needed by ELM.
	Later on we can estimate the data by other means.
 */

void mapELM2Grid_io(applicationStruct *this_struct) {

  int ii=0, tmp_id=0, jstart=0, jend=0, diff=0;
  int this_index=0, kk=0,j=0, jj=0;

  for(ii=0; ii<OGrow_tot ;ii++) {
    jstart =  *(this_struct->grid.config.xstart+ii);
    jend =  *(this_struct->grid.config.xend+ii);
    tmp_id = ii * OGcol_tot + jstart;  /* running cell id of grid_io wmm grid */
    diff = jend - jstart + 1;     /* number of cells in this row */

    this_index = tmp_id - 1;

    for(kk=0;kk<jstart-1;kk++){
      j=ii*OGcol_tot+kk;
      /* elm cell exists, grid_io does not have data for it we have to patch data */
      if(elm_OG_map[j].elmID[0] == NOT_ELM_NOT_GRIO)
        elm_OG_map[j].index = NOT_ELM_NOT_GRIO;      /*out of elm boundary*/
      else /* in elm not in grid_io */
        elm_OG_map[j].index = IN_ELM_NOT_GRIO;
    }

    for(kk=0;kk<diff;++kk) {   /* finds corresponding cell ids of wmm  to grid_io cell ids for this row */
      if(elm_OG_map[this_index+kk].OGID == tmp_id+kk) {   /* found the OG cell in grid_io file */
        if(elm_OG_map[this_index+kk].elmID[0]==NOT_ELM_NOT_GRIO) {
          elm_OG_map[this_index+kk].index = NOT_ELM_IN_GRIO;
        }
        /* found the elm cell in grid_io, keep the index */
      }
      /* else within grid_io boundary , leave the default index */
    } /* end of for kk */

    for(kk=jend;kk<OGcol_tot;kk++) {
      if(elm_OG_map[ii*OGcol_tot+kk].elmID[0] == NOT_ELM_NOT_GRIO)
         elm_OG_map[ii*OGcol_tot+kk].index = NOT_ELM_NOT_GRIO;
      else
        elm_OG_map[ii*OGcol_tot+kk].index = IN_ELM_NOT_GRIO;
    }
  } /* end of for ii */
}


/* \brief Prints debug data (lots!!) to console at high debug flag level.

	Will remove (?) when gridmap functionality is verified.  */

void printGridMap() {

  int i,j;

  printf("\ngridmap ==> Printing GridMap .......\n");
  printf("GRID_ID  GRID_ROW  GRID_COL INDEX  ELM_ID  ELM_CNT\n");
  printf("-----------------------------------------------\n");
  for(i=0;i<OGrow_tot*OGcol_tot;i++) {
    printf("%d\t%d\t%d\t%d\t%d\t%d\n",elm_OG_map[i].OGID,elm_OG_map[i].OGROW,
            elm_OG_map[i].OGCOL,elm_OG_map[i].index,elm_OG_map[i].elmID[0],elm_OG_map[i].eidCnt);
    for(j=1;j<elm_OG_map[i].eidCnt;j++)
      printf("\t\t\t\t%d\n",elm_OG_map[i].elmID[j]);
  }
}



/* TODO: remove this function when sure is no longer needed */
/* void printELM2Grid_io() {
  int ii=0;
  printf("\ngridmap==> Printing 3 grids .......\n");
  printf(" -1    ==> No data needed\n");
  printf(" -2    ==> Data available, but not needed\n");
  printf(" -3    ==> Needs data estimation\n");
  printf(" other ==> Data available and in use\n");
  printf("index  OGID   elmID  elmcnt\n");
  printf("============================\n");
  for(ii=0; ii< OGidCnt; ++ii)
    printf("%d\t%d\t%d\t%d\n", elm_OG_map[ii].index, elm_OG_map[ii].OGID,
                               elm_OG_map[ii].elmID[0],elm_OG_map[ii].eidCnt);
  sprintf(msgStr,"gridmap==> Finished mapping to grid_io");
  usrErr(msgStr);
} */


/* TODO: remove this function when sure is no longer needed */
/*  void drawELM2Grid_io() {
  int ii=0,j=0,jj=0,kk=0;
 
  printf("\ngridmap==> Drawing three grids .......\n");
  printf("LEGEND:\n");
  printf("Scaled in 2x2 mile\n");
  printf("o : Out of ELM Grid, No grid_io data available, No data needed\n");
  printf("+ : In ELM Grid, No grid_io data available, Needs data\n");
  printf("x : Out of ELM Grid, grid_io data available, No data needed,\n");
  printf("* : In ELM Grid, grid_io data available, Used grid_io data\n");
  j = OGidCnt / OGcol_tot;  
  for(ii=j; ii>0;--ii) {
    if(kk<0)
      break;
    kk = ii * OGcol_tot - OGcol_tot;    
    for(jj=0; jj<OGcol_tot; ++jj) {
      if(elm_OG_map[kk+jj].index == NOT_ELM_IN_GRIO) {
        printf("%s","x");
      }
      else if(elm_OG_map[kk+jj].index == IN_ELM_NOT_GRIO) {
        printf("%s","+");
      }
      else if(elm_OG_map[kk+jj].index == NOT_ELM_NOT_GRIO) {
        printf("%s","o");
      }
      else {
        printf("%s","*");
      }
    }
    printf("\n");
  }
} */

/* TODO: remove this function when sure is no longer needed */
/* In case of any error or at the end of program
 * clean up the pointes and exit.
 */
void cleanUp (float** dataAnyGrid, float* dataELM) {
  int i;

  printf("\ngridmap ==> Cleaning up !\n");
  for(i=0;i<OGrow_tot*OGcol_tot;i++) {
    if(elm_OG_map[i].elmID != NULL)
      free(elm_OG_map[i].elmID);
  }
  if(elm_OG_map != NULL)
    free(elm_OG_map); 
			 
  if(dataELM != NULL)
   free(dataELM);
  
  for(i=0;i<gridio_batch_len;i++) {
    if(dataAnyGrid[i] != NULL)
      free(dataAnyGrid[i]);
  }
  if(dataAnyGrid != NULL)
    free(dataAnyGrid);

//  if(values != NULL)
//    free(values);
} 


/* TODO: remove this function when sure is no longer needed */
/*
void printBatchData(float **value,int days, int size) {
  int k=0, ogid=1, j=0, i=0, ii=0;   
  int roGcol=0,io_row=1,io_col=1,step=0;
  int jstart,jend;

  printf("\ngridmap ==> Printing batched data .......\n");
  printf("W_ID  W_ROW  W_COL  IO_ROW  IO_COL  INDEX    ELM_CNT   VALUE      ELM_ID\n");
  printf("--------------------------------------------------------------------------------\n");
  for(step=0;step<days;step++) {
    if(i >= OGidCnt)
      continue;
    for(ii=0; ii< grid.header.number_of_rows; ++ii) {
      if(i >= OGidCnt)
        continue;
      jstart = *(grid.config.xstart+ii);
      jend =  *(grid.config.xend+ii);
      for(io_col=1; io_col<jstart; io_col++,roGcol++,i++) {
        printf("%d\t%d\t%d\t%d\t%d\t%d\t%d\t%3.2f\t   %d\n",
                elm_OG_map[i].OGID,elm_OG_map[i].OGROW,elm_OG_map[i].OGCOL,ii+1,io_col, 
	        elm_OG_map[i].index,elm_OG_map[i].eidCnt,*(*(value+step)+roGcol),elm_OG_map[i].elmID[0]);
        for(j=1;j<elm_OG_map[i].eidCnt;j++)
          printf("\t\t\t\t\t\t\t\t   %d\n",elm_OG_map[i].elmID[j]);
      }

      for(io_col=jstart; io_col<jend; io_col++,roGcol++,i++) {
        printf("%d\t%d\t%d\t%d\t%d\t%d\t%d\t%3.2f\t   %d\n",
               elm_OG_map[i].OGID,elm_OG_map[i].OGROW,elm_OG_map[i].OGCOL,ii+1,io_col,
               elm_OG_map[i].index,elm_OG_map[i].eidCnt,*(*(value+step)+roGcol),elm_OG_map[i].elmID[0]);
        for(j=1;j<elm_OG_map[i].eidCnt;j++)
          printf("\t\t\t\t\t\t\t\t   %d\n",elm_OG_map[i].elmID[j]);
      }
    
      for(io_col=jend; io_col<OGcol_tot+1; io_col++,roGcol++,i++) {
        printf("%d\t%d\t%d\t%d\t%d\t%d\t%d\t%3.2f\t   %d\n",
               elm_OG_map[i].OGID,elm_OG_map[i].OGROW,elm_OG_map[i].OGCOL,ii+1,io_col,
               elm_OG_map[i].index,elm_OG_map[i].eidCnt,*(*(value+step)+roGcol),elm_OG_map[i].elmID[0]);
        for(j=1;j<elm_OG_map[i].eidCnt;j++)
          printf("\t\t\t\t\t\t\t\t   %d\n",elm_OG_map[i].elmID[j]);
      }
    }  end of rows 
  }    end of days 
}      end of print 
*/

/*************************/
