/*! \file BudgStats.c
\brief Budget and statistical summaries of model variables.

This source file calculates:
\li Temporal budget summaries of water and phosphorus across spatially
distributed hydrologic Basins and/or Indicator-Regions (BIRs);

\li Temporal mean summaries of principal Performance Measure variables across BIRs;

\li Temporal mean summaries of selected model variables without spatial aggregation.  \n

There are no calculations in this source code that affect the model results;
the code here only summarizes the model results. \n

Mass balances of water and phosphorus are verified at high levels of precision in all budget files. \n
The water budgets are output in units of height per BIR and in units of volume per BIR.  \n
The phosphorus budgets are output in units of:
\li P (total phosphorus) mass per BIR;
\li P (total phosphorus) mass/area per BIR;
\li P "live" (contained in live plants) mass/area per BIR;
\li P "dead" (contained in soils) mass/area per BIR;
\li P "water-borne" (contained in water) mass/area per BIR;

The mean values of principal Performance Measure variables are ouput for each BIR. \n
The mean values of selected model variables are calculated (for output in gen_output function). \n

Note: documented with Doxygen, which expects specific syntax within special comments. \n
	
The Everglades Landscape Model (ELM). \n
last updated: Jan 2005 \n

\remarks TODO:  Yes, there is way too much replicated code here - its a pretty quick fix, but in a hurry as usual (Jan 2005) */

#include "budgstats.h"
#include "unitmod_vars.h"  
#include "budgstats_birvars.h"  
#include "budgstats_statvars.h"  


/******/
/*! \brief Calling function for budget and summary stats

	Called every time step, this calls functions to summarize variables across time 
	At the time interval for output, this calls functions to make the final summaries,
	and then resets the data for new interval.

*/
/* uses Basin/Indicator-Region -specific (BIR-specific) arrays for budget, with basin 0 the entire system */	
/* we also call cell-specific and BIR- specific means in this module */

void stats(int step)
{
    CellAvg();  /* Generate sums, means (avgs), on cell-by-cell basis (not by Basin/IRegion here) */
        
    BIRstats_sum();  
    BIRbudg_sum();  

    if ( (SimTime.IsBIRavgEnd) )  {
        if (!SimTime.IsDay0) BIRstats_date(); /* skip TIME 0 of simulation */
		BIRstats_sumFinal();
        BIRstats_reset();
    } 
    if ( (SimTime.IsBudgEnd) )  {
        if (!SimTime.IsBudgFirst) BIRbudg_date(); /* skip first budget interval to accumulate info for mass-balance budget check */
		BIRbudg_sumFinal();
        BIRbudg_reset();
    } 
} 


/*******/
/*! \brief Generate variable sums/means for Basins/Indicator-Regions (non-budget) stats.

*/
void BIRstats_sum(void)
{
    int ix, iy, cellLoc;
    float cellSiz_DT = CELL_SIZE * DT;
   
    for(ix=1; ix<=s0; ix++) 
        for(iy=1; iy<=s1; iy++) 

            if(ON_MAP[cellLoc= T(ix,iy)])  {
/* get the (non-budget) Performance Measure summary values for cells within basins/indicator regions */
                    Sfwat_avg[basn[cellLoc]] +=  SURFACE_WAT[cellLoc];
                    Sfwat_avg[0]             +=  SURFACE_WAT[cellLoc];
                    Unsat_avg[basn[cellLoc]] +=  UNSAT_DEPTH[cellLoc];
                    Unsat_avg[0]             +=  UNSAT_DEPTH[cellLoc];
                    TPsf_avg[basn[cellLoc]] +=  TP_SFWT_CONC_MG[cellLoc];
                    TPsf_avg[0]             +=  TP_SFWT_CONC_MG[cellLoc];
                    TPpore_avg[basn[cellLoc]] +=  TP_SEDWT_CONCACTMG[cellLoc];
                    TPpore_avg[0]             +=  TP_SEDWT_CONCACTMG[cellLoc];
                    TPsoil_avg[basn[cellLoc]] +=  TPtoSOIL_rep[cellLoc];
                    TPsoil_avg[0]             +=  TPtoSOIL_rep[cellLoc];
                    NCperi_avg[basn[cellLoc]] +=  NC_ALG[cellLoc];
                    NCperi_avg[0]             +=  NC_ALG[cellLoc];
                    Cperi_avg[basn[cellLoc]] +=  C_ALG[cellLoc];
                    Cperi_avg[0]             +=  C_ALG[cellLoc];
                    Mac_avg[basn[cellLoc]] +=  MAC_TOT_BIOM[cellLoc];
                    Mac_avg[0]             +=  MAC_TOT_BIOM[cellLoc];
                    Elev_avg[basn[cellLoc]] +=  SED_ELEV[cellLoc];
                    Elev_avg[0]             +=  SED_ELEV[cellLoc];
            } /* end spatial loop for summations across domain*/
} 


/*******/
/*! \brief Generate (every time step) budget summations for Basins/Indicator-Regions.

*/
void BIRbudg_sum(void)
{
    int ix, iy, cellLoc;
    float cellSiz_DT = CELL_SIZE * DT;
   
    for(ix=1; ix<=s0; ix++) 
        for(iy=1; iy<=s1; iy++) 

            if(ON_MAP[cellLoc= T(ix,iy)])  {

/* calculate sums of inputs/outputs, then sum the storages */
/* the summations here are ONLY relating to vertical flows in this source file */
/* horizontal cell<->cell, cell<->canal, and cell<->external_system flows in WatMgmt.c and Fluxes.c source files */
/****/
/*  sums of water inflow/outflow (m) */
                    /*  vertical system inputs to cells: rain  */
                RAIN[basn[cellLoc]] += (double) SF_WT_FROM_RAIN[cellLoc]*cellSiz_DT ; 
                RAIN[0]             += (double) SF_WT_FROM_RAIN[cellLoc]*cellSiz_DT; 

                    /*  vertical system outputs from cells: evap, transp, recharge */
                EVAP[basn[cellLoc]] += (double) (SF_WT_EVAP[cellLoc])*cellSiz_DT; 
                EVAP[0]             += (double) (SF_WT_EVAP[cellLoc])*cellSiz_DT; 
                TRANSP[basn[cellLoc]] += (double) (SAT_WT_TRANSP[cellLoc] 
                                                   + UNSAT_TRANSP[cellLoc])*cellSiz_DT; 
                TRANSP[0]             += (double) (SAT_WT_TRANSP[cellLoc] 
                                                   + UNSAT_TRANSP[cellLoc])*cellSiz_DT; 

                    /* Important NOTE: recharge is 0 in elm, not now linked to solute fluxes */
                RCHG[basn[cellLoc]] += (double) (SAT_WT_RECHG[cellLoc])*cellSiz_DT; 
                RCHG[0]             += (double) (SAT_WT_RECHG[cellLoc])*cellSiz_DT; 


                    /* mass (kg P) sums of P inflow/outflow */
			
                P_RAIN[basn[cellLoc]] += (double) (TP_FR_RAIN[cellLoc]* DT); 
                P_RAIN[0]             += (double) (TP_FR_RAIN[cellLoc]* DT); 
                P_settl[basn[cellLoc]] += (double) TP_settl[cellLoc]*DT;  /* kgP */
                P_settl[0]             += (double) TP_settl[cellLoc]*DT;  /* kgP */
			
                if (!ESPmodeON) { /* not calc'd if running in ESP (Everglades Settling-of Phosphorus, i.e., EWQModel emulation) mode */


                    Calg_GPP[basn[cellLoc]] += (double) ( (C_ALG_GPP_P[cellLoc]  ) 
                                                          * conv_gTOkg * cellSiz_DT);/* gP/m2 => kgP */
                    NCalg_GPP[basn[cellLoc]] += (double) ( (NC_ALG_GPP_P[cellLoc] ) 
                                                           * conv_gTOkg * cellSiz_DT);/* gP/m2 => kgP */
                    Calg_mort[basn[cellLoc]] += (double) ( (C_ALG_MORT_P[cellLoc]) 
                                                           * conv_gTOkg * cellSiz_DT);/* gP/m2 => kgP */
                    NCalg_mort[basn[cellLoc]] += (double) ( (NC_ALG_MORT_P[cellLoc]) 
                                                            * conv_gTOkg * cellSiz_DT);/* gP/m2 => kgP */
                    dop_macIn[basn[cellLoc]] += (double) DOP_nphBio[cellLoc] 
                        * cellSiz_DT;  /* kgP/m2 => kgP */
                    dop_sorbIn[basn[cellLoc]] += (double) (TP_SORBTION[cellLoc]>0.0) ? (TP_SORBTION[cellLoc] * DT) : (0.0); /* kgP */
                    dop_decomp[basn[cellLoc]] += (double)  DOP_DECOMP[cellLoc] 
                        * cellSiz_DT;  /* kgP/m2 => kgP */
                    dop_desorb[basn[cellLoc]] += (double)  (TP_SORBTION[cellLoc]<=0.0) ? (-TP_SORBTION[cellLoc] * DT) : (0.0) ; /* kgP */
                    
                    floc_decomp[basn[cellLoc]] += (double)  FlocP_DECOMP[cellLoc]
                        * cellSiz_DT;  /* kgP/m2 => kgP */
                    floc_In[basn[cellLoc]] += (double)  (FlocP_FR_ALGAE[cellLoc] + FlocP_PhBio[cellLoc])
                        * cellSiz_DT;  /* kgP/m2 => kgP */

                    mac_NPP[basn[cellLoc]] += (double) phbio_npp_P[cellLoc] 
                        * cellSiz_DT;  /* kgP/m2 => kgP */
                    mac_mort[basn[cellLoc]] += (double) ( phbio_mort_P[cellLoc]+ nphbio_mort_P[cellLoc]) 
                        * cellSiz_DT;  /* kgP/m2 => kgP */
                    wat_sfMiner[basn[cellLoc]] += (double) TP_SFWT_MINER[cellLoc]*DT;  /* kgP */
                    wat_sedMiner[basn[cellLoc]] += (double) TP_SED_MINER[cellLoc]*DT;  /* kgP */
                    wat_sfUpt[basn[cellLoc]] += (double) TP_SFWT_UPTAK[cellLoc] * DT;  /* kgP */
                    wat_sedUpt[basn[cellLoc]] += (double) TP_SEDWT_UPTAKE[cellLoc]*DT;  /* kgP */
			
			
                    Calg_GPP[0] += (double) ( (C_ALG_GPP_P[cellLoc]  ) 
                                              * conv_gTOkg * cellSiz_DT);/* gP/m2 => kgP */
                    NCalg_GPP[0] += (double) ( (NC_ALG_GPP_P[cellLoc] ) 
                                               * conv_gTOkg * cellSiz_DT);/* gP/m2 => kgP */
                    Calg_mort[0] += (double) ( (C_ALG_MORT_P[cellLoc]) 
                                               * conv_gTOkg * cellSiz_DT);/* gP/m2 => kgP */
                    NCalg_mort[0] += (double) ( (NC_ALG_MORT_P[cellLoc]) 
                                                * conv_gTOkg * cellSiz_DT);/* gP/m2 => kgP */
                    dop_macIn[0] += (double) DOP_nphBio[cellLoc] 
                        * cellSiz_DT;/* kgP/m2 => kgP */
                    dop_sorbIn[0] += (double) (TP_SORBTION[cellLoc]>0.0) ? (TP_SORBTION[cellLoc] * DT) : (0.0); /* kgP */
                    dop_decomp[0] += (double) DOP_DECOMP[cellLoc] 
                        * cellSiz_DT;/* kgP/m2 => kgP */
                    dop_desorb[0] += (double)  (TP_SORBTION[cellLoc]<=0.0) ? (-TP_SORBTION[cellLoc] * DT) : (0.0) ; /* kgP */
                    floc_decomp[0] += (double)  FlocP_DECOMP[cellLoc]
                        * cellSiz_DT;/* kgP/m2 => kgP */
                    floc_In[0] += (double)  (FlocP_FR_ALGAE[cellLoc] + FlocP_PhBio[cellLoc])
                        * cellSiz_DT;/* kgP/m2 => kgP */
                    mac_NPP[0] += (double) phbio_npp_P[cellLoc] 
                        * cellSiz_DT;/* kgP/m2 => kgP */
                    mac_mort[0] += (double) ( phbio_mort_P[cellLoc]+ nphbio_mort_P[cellLoc]) 
                        * cellSiz_DT;/* kgP/m2 => kgP */
                    wat_sfMiner[0] += (double) TP_SFWT_MINER[cellLoc]*DT;  /* kgP */
                    wat_sedMiner[0] += (double) TP_SED_MINER[cellLoc]*DT;  /* kgP */
                    wat_sfUpt[0] += (double) TP_SFWT_UPTAK[cellLoc] * DT;  /* kgP */
                    wat_sedUpt[0] += (double) TP_SEDWT_UPTAKE[cellLoc]*DT;  /* kgP */
                }
			
                    /* no salt inflow/outflow to/from system in vertical direction */

                    /* sum the storages (budgets) in IndRegions at the desired interval  */
                if ( (SimTime.IsBudgEnd ) ) { 
				/* water (m * m2) */
                    SUMSF[basn[cellLoc]] += (double) SURFACE_WAT[cellLoc]*CELL_SIZE;
                    SUMSF[0]             += (double) SURFACE_WAT[cellLoc]*CELL_SIZE; 
                    SUMUW[basn[cellLoc]] += (double) UNSAT_WATER[cellLoc]*CELL_SIZE;
                    SUMUW[0]             += (double) UNSAT_WATER[cellLoc]*CELL_SIZE; 
                    SUMGW[basn[cellLoc]] += (double) SAT_WATER[cellLoc]*CELL_SIZE;
                    SUMGW[0]             += (double) SAT_WATER[cellLoc]*CELL_SIZE;  

                      /* most of this not needed when only running in ESP mode (ESPmodeOn), not bothering to clean up now */
				/* phosphorus (units converted to kg P)*/
                        P_Calg[basn[cellLoc]] += (double) (C_ALG_P[cellLoc] ) 
                            * conv_gTOkg * CELL_SIZE; /* gP/m2 => kgP */
                        P_Calg[0] += (double) (C_ALG_P[cellLoc] ) 
                            * conv_gTOkg * CELL_SIZE; /* gP/m2 => kgP */
                        P_NCalg[basn[cellLoc]] += (double) (NC_ALG_P[cellLoc]) 
                            * conv_gTOkg * CELL_SIZE; /* gP/m2 => kgP */
                        P_NCalg[0] += (double) (NC_ALG_P[cellLoc]) 
                            * conv_gTOkg * CELL_SIZE; /* gP/m2 => kgP */
                        P_MAC[basn[cellLoc]] += (double) (mac_nph_P[cellLoc] + mac_ph_P[cellLoc] )  
                            * CELL_SIZE; /* kgP/m2 => kgP */
                        P_MAC[0] += (double) (mac_nph_P[cellLoc] + mac_ph_P[cellLoc] )  
                            * CELL_SIZE; /* kgP/m2 => kgP */

                        P_LIVE_CELL[basn[cellLoc]] = (P_Calg[basn[cellLoc]] + P_NCalg[basn[cellLoc]] + P_MAC[basn[cellLoc]]);
                        P_LIVE_CELL[0] 	       = (P_Calg[0] + P_NCalg[0] + P_MAC[0]);

                        P_DEAD_CELL[basn[cellLoc]] += (double) ( TP_SORB[cellLoc] +
                                                                ( FlocP[cellLoc] + DOP[cellLoc]  ) /* kgP/m2 */ * CELL_SIZE );
                        P_DEAD_CELL[0] 	       += (double) ( TP_SORB[cellLoc] +
                                                                ( FlocP[cellLoc] + DOP[cellLoc] ) /* kgP/m2 */ * CELL_SIZE );

                        P_WAT_CELL[basn[cellLoc]] += (double) (TP_SED_WT[cellLoc] + TP_SF_WT[cellLoc] ); /*kgP*/
                        P_WAT_CELL[0] 		+= (double) (TP_SED_WT[cellLoc] + TP_SF_WT[cellLoc] ); /*kgP*/
				
                        P_CELL[basn[cellLoc]] = (double) (P_LIVE_CELL[basn[cellLoc]] + P_DEAD_CELL[basn[cellLoc]] + P_WAT_CELL[basn[cellLoc]]);
                        P_CELL[0] 		  = (double) (P_LIVE_CELL[0] + P_DEAD_CELL[0] + P_WAT_CELL[0]);
                    
                    
                    
				/* salt (kgS) */
                    TOT_S_CELL[basn[cellLoc]] += (double) (SALT_SED_WT[cellLoc] + SALT_SURF_WT[cellLoc] ); 
                    TOT_S_CELL[0] +=             (double) (SALT_SED_WT[cellLoc] + SALT_SURF_WT[cellLoc] ); 
                }
            } /* end spatial loop for summations across domain*/

} /* end BIRbudg_sum() */


/*! \brief Assemble the final sums for end-of-period summary statistics

	Assemble the sums for parent hydrologic Basins from their children Indicator Regions.
	\note Please remember that the struct "basins" is generic to include hydrologic Basins 
	and Indicator Regions. 

*/
void BIRstats_sumFinal(void)
{
    int ibas, ii; 
        
        for (ibas = numBasn; ibas >= 0; ibas--) {
            basins = basn_list[ibas];

            for (ii=0; ii<basins->numIR; ii++) { 
/* to a parent basin, add the (non-budget) Performance Measure summation values within any child- Indicator Regions */
				Sfwat_avg[ibas] +=  Sfwat_avg[basins->IR[ii]];
                Unsat_avg[ibas] +=  Unsat_avg[basins->IR[ii]];
                TPsf_avg[ibas] +=  TPsf_avg[basins->IR[ii]];
                TPpore_avg[ibas] +=  TPpore_avg[basins->IR[ii]];
                TPsoil_avg[ibas] +=  TPsoil_avg[basins->IR[ii]];
                NCperi_avg[ibas] +=  NCperi_avg[basins->IR[ii]];
                Cperi_avg[ibas] +=  Cperi_avg[basins->IR[ii]];
                Mac_avg[ibas] +=  Mac_avg[basins->IR[ii]];
                Elev_avg[ibas] +=  Elev_avg[basins->IR[ii]];
            }

        /* now time for output of data */
            if (!SimTime.IsDay0) BIRstats_print(ibas); /* skip TIME 0 of simulation */
        }
}


/*! \brief Assemble the final sums for end-of-period budget

	Assemble the sums for parent hydrologic Basins from their children Indicator Regions,
	and assemble the grand sums of inflow/outflow to/from all Basins/Indicator Regions.
	\note Please remember that the struct "basins" is generic to include hydrologic Basins 
	and Indicator Regions. 

*/
void BIRbudg_sumFinal(void)
{
    int ibas, ii; 
        
        for (ibas = numBasn; ibas >= 0; ibas--) {
            basins = basn_list[ibas];

            for (ii=0; ii<basins->numIR; ii++) { 
                    /* sum up all the I/O to child- Indicator Regions within a parent hydrologic basin,
                       adding that to the parent (the horizontal flows are treated within the flow calcs) */

                    /* flows */
                RAIN[ibas]+=RAIN[basins->IR[ii]];  
                EVAP[ibas]+=EVAP[basins->IR[ii]];
                TRANSP[ibas]+=TRANSP[basins->IR[ii]]; 
                RCHG[ibas]+=RCHG[basins->IR[ii]];

                VOL_IN_STR[ibas]+=VOL_IN_STR[basins->IR[ii]]; 
                VOL_OUT_STR[ibas]+=VOL_OUT_STR[basins->IR[ii]];
                S_IN_STR[ibas]+=S_IN_STR[basins->IR[ii]]; 
                S_OUT_STR[ibas]+=S_OUT_STR[basins->IR[ii]];
                P_IN_STR[ibas]+=P_IN_STR[basins->IR[ii]];
                P_OUT_STR[ibas]+=P_OUT_STR[basins->IR[ii]];

                P_RAIN[ibas]+=P_RAIN[basins->IR[ii]];
                Calg_GPP[ibas]+=Calg_GPP[basins->IR[ii]];  
                NCalg_GPP[ibas]+=NCalg_GPP[basins->IR[ii]];
                Calg_mort[ibas]+=Calg_mort[basins->IR[ii]]; 
                NCalg_mort[ibas]+=NCalg_mort[basins->IR[ii]];
                mac_NPP[ibas]+=mac_NPP[basins->IR[ii]]; 
                mac_mort[ibas]+=mac_mort[basins->IR[ii]]; 
                dop_macIn[ibas]+=dop_macIn[basins->IR[ii]]; 
                dop_sorbIn[ibas]+=dop_sorbIn[basins->IR[ii]];
                floc_decomp[ibas]+=floc_decomp[basins->IR[ii]]; 
                floc_In[ibas]+=floc_In[basins->IR[ii]];
                 P_settl[ibas]+=P_settl[basins->IR[ii]]; 
                dop_decomp[ibas]+=dop_decomp[basins->IR[ii]]; 
                dop_desorb[ibas]+=dop_desorb[basins->IR[ii]]; 
                wat_sfMiner[ibas]+=wat_sfMiner[basins->IR[ii]]; 
                wat_sedMiner[ibas]+=wat_sedMiner[basins->IR[ii]];
                wat_sfUpt[ibas]+=wat_sfUpt[basins->IR[ii]];  
                wat_sedUpt[ibas]+=wat_sedUpt[basins->IR[ii]];

                    /* stocks */
                SUMSF[ibas]+=SUMSF[basins->IR[ii]];  
                SUMUW[ibas]+=SUMUW[basins->IR[ii]];
                SUMGW[ibas]+=SUMGW[basins->IR[ii]];  
                TOT_VOL_CAN[ibas]+=TOT_VOL_CAN[basins->IR[ii]];

                TOT_S_CELL[ibas]+=TOT_S_CELL[basins->IR[ii]];  
                TOT_S_CAN[ibas]+=TOT_S_CAN[basins->IR[ii]];
                
                P_CELL[ibas]+=P_CELL[basins->IR[ii]];  
                TOT_P_CAN[ibas]+=TOT_P_CAN[basins->IR[ii]];
                P_LIVE_CELL[ibas]+=P_LIVE_CELL[basins->IR[ii]];  
                P_DEAD_CELL[ibas]+=P_DEAD_CELL[basins->IR[ii]];  
                P_WAT_CELL[ibas]+=P_WAT_CELL[basins->IR[ii]];  

            } /* end of IR summation */
            
                /*  inputs  */
            VOL_IN[ibas] = RAIN[ibas] + VOL_IN_STR[ibas] + VOL_IN_OVL[ibas] + VOL_IN_SPG[ibas] + VOL_IN_GW[ibas]; 
            VOL_IN_SUM[ibas] += VOL_IN[ibas];
            S_IN[ibas] = S_IN_STR[ibas] + S_IN_OVL[ibas] + S_IN_SPG[ibas] + S_IN_GW[ibas]; 
            S_IN_SUM[ibas] += S_IN[ibas];
            P_IN[ibas] = P_RAIN[ibas] + P_IN_STR[ibas] + P_IN_OVL[ibas] + P_IN_SPG[ibas] + P_IN_GW[ibas]; 
            P_IN_SUM[ibas] += P_IN[ibas];
            

                /*  outputs */
            VOL_OUT[ibas] = EVAP[ibas] + TRANSP[ibas] + RCHG[ibas] + VOL_OUT_STR[ibas] + VOL_OUT_OVL[ibas] + VOL_OUT_SPG[ibas] + VOL_OUT_GW[ibas]; 
            VOL_OUT_SUM[ibas] += VOL_OUT[ibas];
            S_OUT[ibas] = S_OUT_STR[ibas] + S_OUT_OVL[ibas] + S_OUT_SPG[ibas] + S_OUT_GW[ibas]; 
            S_OUT_SUM[ibas] += S_OUT[ibas];
            P_OUT[ibas] = P_OUT_STR[ibas] + P_OUT_OVL[ibas] + P_OUT_SPG[ibas] + P_OUT_GW[ibas]; 
            if (ESPmodeON) P_OUT[ibas] += P_settl[ibas]; /* only if running ESP (Everglades Settling-of Phosphorus, i.e., EWQModel emulation) with settling losses from system */
            P_OUT_SUM[ibas] += P_OUT[ibas];
            

                /* actual new volume/mass = sum of the current storages */
            TOT_VOL[ibas] = SUMSF[ibas] + SUMUW[ibas] + SUMGW[ibas] + TOT_VOL_CAN[ibas]; 
            TOT_S[ibas] = TOT_S_CELL[ibas] + TOT_S_CAN[ibas]; 
            P[ibas] = P_CELL[ibas] + TOT_P_CAN[ibas]; 
            
            if (!ESPmodeON) { /* do same for diff P fractions when running full model */
                    /*  inputs  */
                P_LIVE_IN[ibas] = Calg_GPP[ibas] + NCalg_GPP[ibas] + mac_NPP[ibas];
                P_LIVE_IN_SUM[ibas] += P_LIVE_IN[ibas];
                P_DEAD_IN[ibas] = dop_macIn[ibas]  + dop_sorbIn[ibas] + P_settl[ibas] + floc_In[ibas];
                P_DEAD_IN_SUM[ibas] += P_DEAD_IN[ibas];
                P_WAT_IN[ibas] =  P_RAIN[ibas] + dop_decomp[ibas] + floc_decomp[ibas] + dop_desorb[ibas] 
                    + P_IN_STR[ibas] + P_IN_OVL[ibas] + P_IN_SPG[ibas] + P_IN_GW[ibas] ;
                P_WAT_IN_SUM[ibas] += P_WAT_IN[ibas] ;
                    /*  outputs */
                P_LIVE_OUT[ibas] =  Calg_mort[ibas] + NCalg_mort[ibas] + mac_mort[ibas];
                P_LIVE_OUT_SUM[ibas] += P_LIVE_OUT[ibas];
                P_DEAD_OUT[ibas] = dop_decomp[ibas] + dop_desorb[ibas] + floc_decomp[ibas];
                P_DEAD_OUT_SUM[ibas] += P_DEAD_OUT[ibas];
                P_WAT_OUT[ibas] = Calg_GPP[ibas] + NCalg_GPP[ibas] + mac_NPP[ibas] + dop_sorbIn[ibas] + P_settl[ibas]
                    + P_OUT_STR[ibas] + P_OUT_OVL[ibas] + P_OUT_SPG[ibas] + P_OUT_GW[ibas] ;
                P_WAT_OUT_SUM[ibas] += P_WAT_OUT[ibas];
                    /* actual new volume/mass = sum of the current storages */
                P_LIVE[ibas] = P_LIVE_CELL[ibas]; 
                P_DEAD[ibas] = P_DEAD_CELL[ibas]; 
                P_WAT[ibas] = P_WAT_CELL[ibas] + TOT_P_CAN[ibas]; 
            } /* end of specific P-fraction summations */
           
        /* now time for output of data */
            if (!SimTime.IsBudgFirst) BIRbudg_print(ibas); /* skip first budget interval to accumulate info for mass-balance budget check */
        }
}


/*******/
/*! \brief Print out values of BIRavg Performance Measure data.
	\param ibas Basin number

*/
void BIRstats_print(int ibas)
{
    int Fnum;
    float BIRavgStep = BIRavg_Intvl / DT;
            
    basins = basn_list[ibas];

    Fnum = ((ibas>=52)?(5):
            ((ibas>=39)?(4):
             ((ibas>=26)?(3):
              ((ibas>=13)?(2):
               (1) ) )) );
    
/* non-budget, Basin/Indicator-Region mean values are printed */
    fprintf(   ((Fnum==5)?(BIRavg5):
                ((Fnum==4)?(BIRavg4):
                   ((Fnum==3)?(BIRavg3):
                    ((Fnum==2)?(BIRavg2):
                     (BIRavg1) ) ) ) ),
               "%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t",
               Sfwat_avg[ibas]/numCells[ibas]/BIRavgStep, 
               Unsat_avg[ibas]/numCells[ibas]/BIRavgStep,
               TPsf_avg[ibas]/numCells[ibas]/BIRavgStep, 
               TPpore_avg[ibas]/numCells[ibas]/BIRavgStep, 
               TPsoil_avg[ibas]/numCells[ibas]/BIRavgStep,
               NCperi_avg[ibas]/numCells[ibas]/BIRavgStep, 
               Cperi_avg[ibas]/numCells[ibas]/BIRavgStep, 
               Mac_avg[ibas]/numCells[ibas]/BIRavgStep, 
               Elev_avg[ibas]/numCells[ibas]/BIRavgStep);

}

/*******/
/*! \brief Calculate the mass balance errors and the in/out averages, then print out values of BIR budget data.
	\param ibas Basin number

*/
void BIRbudg_print(int ibas)
{
    int Fnum;
    float budgStep = budg_Intvl / DT;
            
    basins = basn_list[ibas];

    Fnum = ((ibas>=52)?(5):
            ((ibas>=39)?(4):
             ((ibas>=26)?(3):
              ((ibas>=13)?(2):
               (1) ) )) );
    
        /* error  = newVol - (oldVol+inputs-outputs) */
    TOT_VOL_ERR[ibas] = TOT_VOL[ibas] - (TOT_VOL_OLD[ibas]+VOL_IN[ibas]-VOL_OUT[ibas] ); 
    TOT_S_ERR[ibas] = TOT_S[ibas] - (TOT_S_OLD[ibas]+S_IN[ibas]-S_OUT[ibas] ); 
    P_ERR[ibas] = P[ibas] - (P_OLD[ibas]+P_IN[ibas]-P_OUT[ibas] ); 
                
        /* cumulative error  */
    TOT_VOL_CUM_ERR[ibas] += TOT_VOL_ERR[ibas]; /* cumulative net error */
    S_ERR_CUM[ibas] += TOT_S_ERR[ibas];
    P_ERR_CUM[ibas] += P_ERR[ibas]; 
                
        /* average inputs and outputs */
    VOL_IN_AVG[ibas] = VOL_IN_SUM[ibas]/(SimTime.TIME/budg_Intvl);
    VOL_OUT_AVG[ibas] = VOL_OUT_SUM[ibas]/(SimTime.TIME/budg_Intvl); 
    S_IN_AVG[ibas] = S_IN_SUM[ibas]/(SimTime.TIME/budg_Intvl);
    S_OUT_AVG[ibas] = S_OUT_SUM[ibas]/(SimTime.TIME/budg_Intvl); 
    P_IN_AVG[ibas] = P_IN_SUM[ibas]/(SimTime.TIME/budg_Intvl);
    P_OUT_AVG[ibas] = P_OUT_SUM[ibas]/(SimTime.TIME/budg_Intvl); 

        /* do same for diff P fractions when running full model */
    if (!ESPmodeON) { 
        P_LIVE_ERR[ibas] = P_LIVE[ibas] - (P_LIVE_OLD[ibas]+P_LIVE_IN[ibas]-P_LIVE_OUT[ibas] ); 
        P_DEAD_ERR[ibas] = P_DEAD[ibas] - (P_DEAD_OLD[ibas]+P_DEAD_IN[ibas]-P_DEAD_OUT[ibas] ); 
        P_WAT_ERR[ibas] = P_WAT[ibas] - (P_WAT_OLD[ibas]+P_WAT_IN[ibas]-P_WAT_OUT[ibas] ); 
        P_LIVE_ERR_CUM[ibas] += P_LIVE_ERR[ibas];  
        P_DEAD_ERR_CUM[ibas] += P_DEAD_ERR[ibas]; 
        P_WAT_ERR_CUM[ibas] += P_WAT_ERR[ibas]; 
        P_LIVE_IN_AVG[ibas] = P_LIVE_IN_SUM[ibas]/(SimTime.TIME/budg_Intvl);
        P_LIVE_OUT_AVG[ibas] = P_LIVE_OUT_SUM[ibas]/(SimTime.TIME/budg_Intvl); 
        P_DEAD_IN_AVG[ibas] = P_DEAD_IN_SUM[ibas]/(SimTime.TIME/budg_Intvl);
        P_DEAD_OUT_AVG[ibas] = P_DEAD_OUT_SUM[ibas]/(SimTime.TIME/budg_Intvl); 
        P_WAT_IN_AVG[ibas] = P_WAT_IN_SUM[ibas]/(SimTime.TIME/budg_Intvl);
        P_WAT_OUT_AVG[ibas] = P_WAT_OUT_SUM[ibas]/(SimTime.TIME/budg_Intvl); 
    }
                

/********** PRINTING DATA 
 **********/

/* now all the various budget files are printed */

/* hydrology */
        /* print two files, one with volumes and inputs/outputs in units of thousands acre-feet, the other in units of cm across basin */
        /* HYDRO: units used in calcs =  m^3, */
        /* OUTPUT of flows, storages are in thousands of acre-ft and in cm across basin (for SFWMM comp) */
        /* OUTPUT of error analyses use mm height across the basin */
/* hydrology - volume units */
    fprintf(  ((Fnum==5)?(budget_Wacr5):
               ((Fnum==4)?(budget_Wacr4):
                   ((Fnum==3)?(budget_Wacr3):
                    ((Fnum==2)?(budget_Wacr2):
                     (budget_Wacr1) ) ) ) ),
               "%9.2f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.2f\t", 
               TOT_VOL_OLD[ibas]*conv_m3TOthousacrft, 
               RAIN[ibas]*conv_m3TOthousacrft, 
               EVAP[ibas]*conv_m3TOthousacrft, 
               TRANSP[ibas]*conv_m3TOthousacrft,
               RCHG[ibas]*conv_m3TOthousacrft, 
               VOL_IN_STR[ibas]*conv_m3TOthousacrft, 
               VOL_OUT_STR[ibas]*conv_m3TOthousacrft, 
               VOL_IN_OVL[ibas]*conv_m3TOthousacrft, 
               VOL_OUT_OVL[ibas]*conv_m3TOthousacrft, 
               VOL_IN_SPG[ibas]*conv_m3TOthousacrft, 
               VOL_OUT_SPG[ibas]*conv_m3TOthousacrft, 
               VOL_IN_GW[ibas]*conv_m3TOthousacrft,  
               VOL_OUT_GW[ibas]*conv_m3TOthousacrft,
               (TOT_VOL_OLD[ibas]+VOL_IN[ibas]-VOL_OUT[ibas])*conv_m3TOthousacrft );
        /* the new volume, error (mm in basin),  error/inflow, avg Total vol In and Out */
    fprintf(   ((Fnum==5)?(budget_Wacr5):
                ((Fnum==4)?(budget_Wacr4):
                   ((Fnum==3)?(budget_Wacr3):
                    ((Fnum==2)?(budget_Wacr2):
                     (budget_Wacr1) ) ) ) ),
               "%9.2f\t%9.4f\t%9.3f\t%9.3f\t%9.3f\t", 
               TOT_VOL[ibas]*conv_m3TOthousacrft, 
               TOT_VOL_ERR[ibas]*conv_cmTOmm*basins->conv_m3TOcm, 
               TOT_VOL_CUM_ERR[ibas]*conv_cmTOmm*basins->conv_m3TOcm, 
               VOL_IN_AVG[ibas]*conv_m3TOthousacrft, 
               VOL_OUT_AVG[ibas]*conv_m3TOthousacrft );  
    
/* hydrology - height units */
    fprintf(   ((Fnum==5)?(budget_Wcm5):
                ((Fnum==4)?(budget_Wcm4):
                   ((Fnum==3)?(budget_Wcm3):
                    ((Fnum==2)?(budget_Wcm2):
                     (budget_Wcm1) ) ) ) ),
               "%9.2f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.3f\t%9.2f\t", 
               TOT_VOL_OLD[ibas]*basins->conv_m3TOcm, 
               RAIN[ibas]*basins->conv_m3TOcm, 
               EVAP[ibas]*basins->conv_m3TOcm, 
               TRANSP[ibas]*basins->conv_m3TOcm, 
               RCHG[ibas]*basins->conv_m3TOcm, 
               VOL_IN_STR[ibas]*basins->conv_m3TOcm, 
               VOL_OUT_STR[ibas]*basins->conv_m3TOcm, 
               VOL_IN_OVL[ibas]*basins->conv_m3TOcm, 
               VOL_OUT_OVL[ibas]*basins->conv_m3TOcm, 
               VOL_IN_SPG[ibas]*basins->conv_m3TOcm, 
               VOL_OUT_SPG[ibas]*basins->conv_m3TOcm, 
               VOL_IN_GW[ibas]*basins->conv_m3TOcm,  
               VOL_OUT_GW[ibas]*basins->conv_m3TOcm,
               (TOT_VOL_OLD[ibas]+VOL_IN[ibas]-VOL_OUT[ibas])*basins->conv_m3TOcm );
        /* the new volume, error (mm in basin),  error/inflow, avg Total vol In and Out */
    fprintf(   ((Fnum==5)?(budget_Wcm5):
                ((Fnum==4)?(budget_Wcm4):
                   ((Fnum==3)?(budget_Wcm3):
                    ((Fnum==2)?(budget_Wcm2):
                     (budget_Wcm1) ) ) ) ),
               "%9.2f\t%9.4f\t%9.3f\t%9.3f\t%9.3f\t", 
               TOT_VOL[ibas]*basins->conv_m3TOcm, 
               TOT_VOL_ERR[ibas]*conv_cmTOmm*basins->conv_m3TOcm, 
               TOT_VOL_CUM_ERR[ibas]*conv_cmTOmm*basins->conv_m3TOcm, 
               VOL_IN_AVG[ibas]*basins->conv_m3TOcm, 
               VOL_OUT_AVG[ibas]*basins->conv_m3TOcm );  

/* salt/tracer - mass units */
        /* mass and inputs/outputs in metric tons (Mg) */
    fprintf(  ((Fnum==5)?(budget_S5):
               ((Fnum==4)?(budget_S4):
                   ((Fnum==3)?(budget_S3):
                    ((Fnum==2)?(budget_S2):
                     (budget_S1) ) ) ) ),
               "%9.3f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.3f\t", 
               TOT_S_OLD[ibas]*conv_kgTO_Mg, 
               S_IN_STR[ibas]*conv_kgTO_Mg, 
               S_OUT_STR[ibas]*conv_kgTO_Mg, 
               S_IN_OVL[ibas]*conv_kgTO_Mg, 
               S_OUT_OVL[ibas]*conv_kgTO_Mg, 
               S_IN_SPG[ibas]*conv_kgTO_Mg, 
               S_OUT_SPG[ibas]*conv_kgTO_Mg, 
               S_IN_GW[ibas]*conv_kgTO_Mg,  
               S_OUT_GW[ibas]*conv_kgTO_Mg,
               (TOT_S_OLD[ibas]+S_IN[ibas]-S_OUT[ibas])*conv_kgTO_Mg );
        /* the new mass (Mg), error & cumulative error (ug/m2 of basin), cumulative avg Total mass In and Out */
    fprintf(   ((Fnum==5)?(budget_S5):
                ((Fnum==4)?(budget_S4):
                   ((Fnum==3)?(budget_S3):
                    ((Fnum==2)?(budget_S2):
                     (budget_S1) ) ) ) ),
               "%9.3f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t", 
               TOT_S[ibas]*conv_kgTO_Mg, 
               TOT_S_ERR[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
               S_ERR_CUM[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
               S_IN_AVG[ibas]*conv_kgTO_Mg, 
               S_OUT_AVG[ibas]*conv_kgTO_Mg );  
                
/* phosphorus */
/* phosphorus: total, all fractions of P */
        /* we print to two files, one with units of Mg per basin, the other in mg/m2 per basin */
    if (ESPmodeON) {
        fprintf(   ((Fnum==5)?(budget_P5):
                    ((Fnum==4)?(budget_P4):
                   ((Fnum==3)?(budget_P3):
                    ((Fnum==2)?(budget_P2):
                     (budget_P1) ) ) ) ),
                   "%9.3f\t%9.4f\t%9.4f\t", 
                   P_OLD[ibas]*conv_kgTO_Mg,
                   P_RAIN[ibas]*conv_kgTO_Mg, 
                   P_settl[ibas]*conv_kgTO_Mg);
                   
        fprintf(  ((Fnum==5)?(budget_Par5):
                   ((Fnum==4)?(budget_Par4):
                   ((Fnum==3)?(budget_Par3):
                    ((Fnum==2)?(budget_Par2):
                     (budget_Par1) ) ) ) ),
                   "%9.3f\t%9.4f\t%9.4f\t", 
                   P_OLD[ibas]*basins->conv_kgTOmgm2,
                   P_RAIN[ibas]*basins->conv_kgTOmgm2, 
                   P_settl[ibas]*basins->conv_kgTOmgm2);
    }
    
    else {
        fprintf(  ((Fnum==5)?(budget_P5):
                   ((Fnum==4)?(budget_P4):
                   ((Fnum==3)?(budget_P3):
                    ((Fnum==2)?(budget_P2):
                     (budget_P1) ) ) ) ),
                   "%9.3f\t%9.4f\t", 
                   P_OLD[ibas]*conv_kgTO_Mg,
                   P_RAIN[ibas]*conv_kgTO_Mg);
                   
        fprintf(   ((Fnum==5)?(budget_Par5):
                    ((Fnum==4)?(budget_Par4):
                     ((Fnum==3)?(budget_Par3):
                      ((Fnum==2)?(budget_Par2):
                       (budget_Par1) ) ) ) ),
                   "%9.3f\t%9.4f\t", 
                   P_OLD[ibas]*basins->conv_kgTOmgm2,
                   P_RAIN[ibas]*basins->conv_kgTOmgm2);
    }
    
    fprintf(   ((Fnum==5)?(budget_P5):
                ((Fnum==4)?(budget_P4):
                   ((Fnum==3)?(budget_P3):
                    ((Fnum==2)?(budget_P2):
                     (budget_P1) ) ) ) ),
               "%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.3f\t", 
               P_IN_STR[ibas]*conv_kgTO_Mg, 
               P_OUT_STR[ibas]*conv_kgTO_Mg, 
               P_IN_OVL[ibas]*conv_kgTO_Mg, 
               P_OUT_OVL[ibas]*conv_kgTO_Mg, 
               P_IN_SPG[ibas]*conv_kgTO_Mg, 
               P_OUT_SPG[ibas]*conv_kgTO_Mg, 
               P_IN_GW[ibas]*conv_kgTO_Mg,  
               P_OUT_GW[ibas]*conv_kgTO_Mg,
               (P_OLD[ibas]+P_IN[ibas]-P_OUT[ibas])*conv_kgTO_Mg );
        /* the new mass (Mg), error & cumulative error (ug/m2 of basin), cumulative avg Total mass In and Out */
    fprintf(   ((Fnum==5)?(budget_P5):
                ((Fnum==4)?(budget_P4):
                   ((Fnum==3)?(budget_P3):
                    ((Fnum==2)?(budget_P2):
                     (budget_P1) ) ) ) ),
               "%9.3f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t", 
               P[ibas]*conv_kgTO_Mg, 
               P_ERR[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
               P_ERR_CUM[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
               P_IN_AVG[ibas]*conv_kgTO_Mg, 
               P_OUT_AVG[ibas]*conv_kgTO_Mg );  
    
    fprintf(   ((Fnum==5)?(budget_Par5):
                ((Fnum==4)?(budget_Par4):
                   ((Fnum==3)?(budget_Par3):
                    ((Fnum==2)?(budget_Par2):
                     (budget_Par1) ) ) ) ),
               "%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.3f\t", 
               P_IN_STR[ibas]*basins->conv_kgTOmgm2, 
               P_OUT_STR[ibas]*basins->conv_kgTOmgm2, 
               P_IN_OVL[ibas]*basins->conv_kgTOmgm2, 
               P_OUT_OVL[ibas]*basins->conv_kgTOmgm2, 
               P_IN_SPG[ibas]*basins->conv_kgTOmgm2, 
               P_OUT_SPG[ibas]*basins->conv_kgTOmgm2, 
               P_IN_GW[ibas]*basins->conv_kgTOmgm2,  
               P_OUT_GW[ibas]*basins->conv_kgTOmgm2,
               (P_OLD[ibas]+P_IN[ibas]-P_OUT[ibas])*basins->conv_kgTOmgm2 );
        /* the new mass (mg/m^2), error & cumulative error (ug/m2 of basin), cumulative avg Total mass In and Out */
    fprintf(   ((Fnum==5)?(budget_Par5):
                ((Fnum==4)?(budget_Par4):
                   ((Fnum==3)?(budget_Par3):
                    ((Fnum==2)?(budget_Par2):
                     (budget_Par1) ) ) ) ),
               "%9.3f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t", 
               P[ibas]*basins->conv_kgTOmgm2, 
               P_ERR[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
               P_ERR_CUM[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
               P_IN_AVG[ibas]*basins->conv_kgTOmgm2, 
               P_OUT_AVG[ibas]*basins->conv_kgTOmgm2 );
    
/* phosphorus: separate (live, dead, water-borne) fractions of P */
/* don't print for ESPmodeON, which does not include live, dead, water-borne fractions */
    if (!ESPmodeON) { 
/* phosphorus: live P fraction */
            /* mass and inputs/outputs in mg/m2 */
        fprintf(   ((Fnum==5)?(budget_Plive5):
                    ((Fnum==4)?(budget_Plive4):
                       ((Fnum==3)?(budget_Plive3):
                        ((Fnum==2)?(budget_Plive2):
                         (budget_Plive1) ) ) ) ),
                   "%9.3f\t%9.3f\t%9.3f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.3f\t", 
                   P_MAC[ibas]*basins->conv_kgTOmgm2, 
                   P_Calg[ibas]*basins->conv_kgTOmgm2, 
                   P_NCalg[ibas]*basins->conv_kgTOmgm2,
                   mac_NPP[ibas]*basins->conv_kgTOmgm2, 
                   Calg_GPP[ibas]*basins->conv_kgTOmgm2, 
                   NCalg_GPP[ibas]*basins->conv_kgTOmgm2,
                   mac_mort[ibas]*basins->conv_kgTOmgm2, 
                   Calg_mort[ibas]*basins->conv_kgTOmgm2, 
                   NCalg_mort[ibas]*basins->conv_kgTOmgm2,
                   (P_LIVE_OLD[ibas]+P_LIVE_IN[ibas]-P_LIVE_OUT[ibas])*basins->conv_kgTOmgm2 );
            /* the new mass (mg/m2), error & cumulative error (ug/m2 of basin), cumulative avg Total mass In and Out */
        fprintf(   ((Fnum==5)?(budget_Plive5):
                    ((Fnum==4)?(budget_Plive4):
                       ((Fnum==3)?(budget_Plive3):
                        ((Fnum==2)?(budget_Plive2):
                         (budget_Plive1) ) ) ) ),
                   "%9.3f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t", 
                   P_LIVE[ibas]*basins->conv_kgTOmgm2, 
                   P_LIVE_ERR[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
                   P_LIVE_ERR_CUM[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
                   P_LIVE_IN_AVG[ibas]*basins->conv_kgTOmgm2, 
                   P_LIVE_OUT_AVG[ibas]*basins->conv_kgTOmgm2 );  
        
/* phosphorus: dead P fraction */
            /* mass and inputs/outputs in mg/m2 */
        fprintf( ((Fnum==5)?(budget_Pdead5):
                  ((Fnum==4)?(budget_Pdead4):
                       ((Fnum==3)?(budget_Pdead3):
                        ((Fnum==2)?(budget_Pdead2):
                         (budget_Pdead1) ) ) ) ),
                 "%9.3f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.3f\t", 
                 P_DEAD_OLD[ibas]*basins->conv_kgTOmgm2, 
                 dop_macIn[ibas]*basins->conv_kgTOmgm2, 
                 floc_In[ibas]*basins->conv_kgTOmgm2, 
                 (dop_decomp[ibas]+floc_decomp[ibas])*basins->conv_kgTOmgm2, 
                 P_settl[ibas]*basins->conv_kgTOmgm2, 
                 dop_sorbIn[ibas]*basins->conv_kgTOmgm2, 
                 dop_desorb[ibas]*basins->conv_kgTOmgm2, 
                 (P_DEAD_OLD[ibas]+P_DEAD_IN[ibas]-P_DEAD_OUT[ibas])*basins->conv_kgTOmgm2 );
            /* the new mass (mg/m2), error & cumulative error (ug/m2 of basin), cumulative avg Total mass In and Out */
        fprintf(  ((Fnum==5)?(budget_Pdead5):
                   ((Fnum==4)?(budget_Pdead4):
                       ((Fnum==3)?(budget_Pdead3):
                        ((Fnum==2)?(budget_Pdead2):
                         (budget_Pdead1) ) ) ) ),
                 "%9.3f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t", 
                 P_DEAD[ibas]*basins->conv_kgTOmgm2, 
                 P_DEAD_ERR[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
                 P_DEAD_ERR_CUM[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
                 P_DEAD_IN_AVG[ibas]*basins->conv_kgTOmgm2, 
                 P_DEAD_OUT_AVG[ibas]*basins->conv_kgTOmgm2 );  

/* phosphorus: water-borne P fraction */
            /* mass and inputs/outputs in mg/m2 */ /* there is a blank column to match the old budget summary template */
        fprintf(  ((Fnum==5)?(budget_Pwat5):
                   ((Fnum==4)?(budget_Pwat4):
                       ((Fnum==3)?(budget_Pwat3):
                        ((Fnum==2)?(budget_Pwat2):
                         (budget_Pwat1) ) ) ) ),
                 "%9.3f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t%9.3f\t", 
                 P_WAT_OLD[ibas]*basins->conv_kgTOmgm2,  
                 P_RAIN[ibas]*basins->conv_kgTOmgm2, 
                 P_settl[ibas]*basins->conv_kgTOmgm2,
                 wat_sfMiner[ibas]*basins->conv_kgTOmgm2, 
                 wat_sedMiner[ibas]*basins->conv_kgTOmgm2, 
                 wat_sfUpt[ibas]*basins->conv_kgTOmgm2,
                 wat_sedUpt[ibas]*basins->conv_kgTOmgm2, 
                 dop_desorb[ibas]*basins->conv_kgTOmgm2, 
                 dop_sorbIn[ibas]*basins->conv_kgTOmgm2,
                 P_IN_STR[ibas]*basins->conv_kgTOmgm2, 
                 P_OUT_STR[ibas]*basins->conv_kgTOmgm2,
                 (P_WAT_OLD[ibas]+P_WAT_IN[ibas]-P_WAT_OUT[ibas])*basins->conv_kgTOmgm2 );
            /* the new mass (mg/m2), error & cumulative error (ug/m2 of basin), cumulative avg Total mass In and Out */
        fprintf(  ((Fnum==5)?(budget_Pwat5):
                   ((Fnum==4)?(budget_Pwat4):
                       ((Fnum==3)?(budget_Pwat3):
                        ((Fnum==2)?(budget_Pwat2):
                         (budget_Pwat1) ) ) ) ),
                 "%9.3f\t%9.4f\t%9.4f\t%9.4f\t%9.4f\t", 
                 P_WAT[ibas]*basins->conv_kgTOmgm2, 
                 P_WAT_ERR[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
                 P_WAT_ERR_CUM[ibas]*conv_mgTOug*basins->conv_kgTOmgm2, 
                 P_WAT_IN_AVG[ibas]*basins->conv_kgTOmgm2, 
                 P_WAT_OUT_AVG[ibas]*basins->conv_kgTOmgm2 );  
    } /* end of non-ESPmodeON prints */

} /* end of BIRbudg_print() */

/*******/
/*! \brief Print date stamp on current BIRavg output record.

*/
void BIRstats_date(void)
{
    int Fileset;
    int FileNum;
    extern ProgAttr *ProgExec;

/* TODO:  yes, yes all of the output in BIR stuff is absurdly crude/repetitive */

    if (numBasn>=52) { /* with more than 58 (plus whole-system basin == 59) basins: */
        Fileset=5;              /* we have five sets of basin files */
    }
    else if (numBasn>=39) { /* with more than 38 (plus whole-system basin == 39) basins: */
        Fileset=4;              /* we have four sets of basin files */
    }
    else if (numBasn>=26) { /* with more than 25 (plus whole-system basin == 26) basins: */
        Fileset=3;              /* we have three sets of basin files */
    }
    else if (numBasn>=13) { /* with more than 12 (plus whole-system basin == 13) basins: */
        Fileset=2;              /* we have two sets of basin files */
    }
    else {
        Fileset=1;       /* otherwise, just one set of files */
    }
    
   /* for (FileNum = 1; FileNum <= Fileset; FileNum++) */
   
/* date stamp for each record in the output files, including (feb05) which parameter being evaluated in sensitivity analysis (or "NONE" for standard, nominal run)  */
                     fprintf(BIRavg1, "\n%s\t%f\t%d/%d/%d\t",ProgExec->S_ParmName,ProgExec->S_ParmVal,SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=13) fprintf(BIRavg2, "\n%s\t%f\t%d/%d/%d\t",ProgExec->S_ParmName,ProgExec->S_ParmVal,SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
    if (numBasn>=26) fprintf(BIRavg3, "\n%s\t%f\t%d/%d/%d\t",ProgExec->S_ParmName,ProgExec->S_ParmVal,SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
    if (numBasn>=39) fprintf(BIRavg4, "\n%s\t%f\t%d/%d/%d\t",ProgExec->S_ParmName,ProgExec->S_ParmVal,SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
    if (numBasn>=52) fprintf(BIRavg5, "\n%s\t%f\t%d/%d/%d\t",ProgExec->S_ParmName,ProgExec->S_ParmVal,SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
      
}


/*******/
/*! \brief Print date stamp on current BIR budget output record.

*/
void BIRbudg_date(void)
{

/* date stamp for each record in the budget output files (TODO: not labeled (feb05) for sensi analysis) */
    
    fprintf(budget_Wacr1, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=13) fprintf(budget_Wacr2, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=26) fprintf(budget_Wacr3, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=39) fprintf(budget_Wacr4, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=52) fprintf(budget_Wacr5, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 

    fprintf(budget_Wcm1, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=13) fprintf(budget_Wcm2, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=26) fprintf(budget_Wcm3, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=39) fprintf(budget_Wcm4, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=52) fprintf(budget_Wcm5, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 

    fprintf(budget_P1, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
    if (numBasn>=13) fprintf(budget_P2, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=26) fprintf(budget_P3, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=39) fprintf(budget_P4, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=52) fprintf(budget_P5, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 

    fprintf(budget_Par1, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=13) fprintf(budget_Par2, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=26) fprintf(budget_Par3, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=39) fprintf(budget_Par4, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    if (numBasn>=52) fprintf(budget_Par5, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 

    fprintf(budget_S1, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
    if (numBasn>=13) fprintf(budget_S2, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
    if (numBasn>=26) fprintf(budget_S3, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
    if (numBasn>=39) fprintf(budget_S4, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
    if (numBasn>=52) fprintf(budget_S5, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );

    if (!ESPmodeON) {
        fprintf(budget_Plive1, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=13) fprintf(budget_Plive2, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=26) fprintf(budget_Plive3, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=39) fprintf(budget_Plive4, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=52) fprintf(budget_Plive5, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 

        fprintf(budget_Pdead1, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=13) fprintf(budget_Pdead2, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=26) fprintf(budget_Pdead3, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=39) fprintf(budget_Pdead4, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=52) fprintf(budget_Pdead5, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 

        fprintf(budget_Pwat1, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=13) fprintf(budget_Pwat2, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=26) fprintf(budget_Pwat3, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=39) fprintf(budget_Pwat4, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
        if (numBasn>=52) fprintf(budget_Pwat5, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
    }
} 

/*******/
/*! \brief Reset BIR (non-budget) statistics summations to zero.

	\param ibas Basin/Indicator-Region ID number
*/
void BIRstats_reset(void)
{
    int ibas;
    
        for (ibas = numBasn; ibas >= 0; ibas--) {
            Sfwat_avg[ibas]=Unsat_avg[ibas]=TPsf_avg[ibas]=TPpore_avg[ibas]=TPsoil_avg[ibas]=
                NCperi_avg[ibas]=Cperi_avg[ibas]=Mac_avg[ibas]=Elev_avg[ibas]=0.0;
        }
        fflush (BIRavg1); 
        if (numBasn>=13) fflush (BIRavg2);
        if (numBasn>=26) fflush (BIRavg3); 
        if (numBasn>=39) fflush (BIRavg4); 
        if (numBasn>=52) fflush (BIRavg5); 

}

/*******/
/*! \brief Reset BIR budget summations to zero.

	For ALL budgets, store the old total volume in array TOT_xxx_OLD and reset any summations and BIRavgs to 0.
	\param ibas Basin/Indicator-Region ID number
*/
void BIRbudg_reset(void)
{
    int ibas;
    
        for (ibas = numBasn; ibas >= 0; ibas--) {
            TOT_VOL_OLD[ibas] = TOT_VOL[ibas]; 
            TOT_S_OLD[ibas] = TOT_S[ibas]; 
            P_OLD[ibas] = P[ibas]; 
            if (!ESPmodeON) {
                P_LIVE_OLD[ibas] = P_LIVE[ibas]; 
                P_DEAD_OLD[ibas] = P_DEAD[ibas]; 
                P_WAT_OLD[ibas] = P_WAT[ibas];
            }
         
            SUMSF[ibas] = SUMGW[ibas] = SUMUW[ibas] = TOT_VOL[ibas] = TOT_VOL_CAN[ibas] =  0.0;
            RAIN[ibas] = VOL_IN_STR[ibas] = VOL_IN_OVL[ibas] = VOL_IN_SPG[ibas] = VOL_IN_GW[ibas] =  0.0;
            EVAP[ibas] = TRANSP[ibas] = RCHG[ibas] = VOL_OUT_STR[ibas] = VOL_OUT_OVL[ibas] = VOL_OUT_SPG[ibas] = VOL_OUT_GW[ibas] =  0.0;

            TOT_S[ibas] = TOT_S_CELL[ibas] = TOT_S_CAN[ibas] =  0.0;
            S_IN_STR[ibas] = S_IN_OVL[ibas] = S_IN_SPG[ibas] = S_IN_GW[ibas] =  0.0;
            S_OUT_STR[ibas] = S_OUT_OVL[ibas] = S_OUT_SPG[ibas] = S_OUT_GW[ibas] =  0.0;

            P[ibas] = P_CELL[ibas] = TOT_P_CAN[ibas] =  0.0;
            P_RAIN[ibas] = P_IN_STR[ibas] = P_IN_OVL[ibas] = P_IN_SPG[ibas] = P_IN_GW[ibas] =  0.0;
            P_OUT_STR[ibas] = P_OUT_OVL[ibas] = P_OUT_SPG[ibas] = P_OUT_GW[ibas] =  0.0;

            P_LIVE[ibas] = P_LIVE_CELL[ibas] = P_MAC[ibas] = P_Calg[ibas] = P_NCalg[ibas] = 0.0;
            Calg_GPP[ibas] = NCalg_GPP[ibas] = Calg_mort[ibas] = NCalg_mort[ibas] = 0.0;
            mac_NPP[ibas] = mac_mort[ibas] = 0.0;

            P_DEAD[ibas] = P_DEAD_CELL[ibas] = 0.0;
            dop_macIn[ibas] =  dop_sorbIn[ibas] = floc_In[ibas] = 0.0;
            dop_decomp[ibas] = dop_desorb[ibas] = floc_decomp[ibas] = 0.0;

            P_WAT[ibas] = P_WAT_CELL[ibas] = 0.0;
            wat_sfMiner[ibas] = wat_sedMiner[ibas] = wat_sfUpt[ibas] = wat_sedUpt[ibas] =  P_settl[ibas] = 0.0;

            
        }


        fflush (budget_Wacr1); 
        if (numBasn>=13) fflush (budget_Wacr2);
        if (numBasn>=26) fflush (budget_Wacr3); 
        if (numBasn>=39) fflush (budget_Wacr4); 
        if (numBasn>=52) fflush (budget_Wacr5); 

        fflush (budget_Wcm1); 
        if (numBasn>=13) fflush (budget_Wcm2); 
        if (numBasn>=26) fflush (budget_Wcm3); 
        if (numBasn>=39) fflush (budget_Wcm4); 
        if (numBasn>=52) fflush (budget_Wcm5); 

        fflush (budget_S1); 
        if (numBasn>=13) fflush (budget_S2); 
        if (numBasn>=26) fflush (budget_S3); 
        if (numBasn>=39) fflush (budget_S4); 
        if (numBasn>=52) fflush (budget_S5); 

        fflush (budget_P1); 
        if (numBasn>=13) fflush (budget_P2); 
        if (numBasn>=26) fflush (budget_P3); 
        if (numBasn>=39) fflush (budget_P4); 
        if (numBasn>=52) fflush (budget_P5); 

        fflush (budget_Par1); 
        if (numBasn>=13) fflush (budget_Par2); 
        if (numBasn>=26) fflush (budget_Par3); 
        if (numBasn>=39) fflush (budget_Par4); 
        if (numBasn>=52) fflush (budget_Par5); 

        if (!ESPmodeON) { 
            fflush (budget_Plive1);  
            if (numBasn>=13) fflush (budget_Plive2);  
            if (numBasn>=26) fflush (budget_Plive3); 
            if (numBasn>=39) fflush (budget_Plive4); 
            if (numBasn>=52) fflush (budget_Plive5); 

            fflush (budget_Pdead1);  
            if (numBasn>=13) fflush (budget_Pdead2);  
            if (numBasn>=26) fflush (budget_Pdead3); 
            if (numBasn>=39) fflush (budget_Pdead4); 
            if (numBasn>=52) fflush (budget_Pdead5); 

            fflush (budget_Pwat1);  
            if (numBasn>=13) fflush (budget_Pwat2);  
            if (numBasn>=26) fflush (budget_Pwat3); 
            if (numBasn>=39) fflush (budget_Pwat4); 
            if (numBasn>=52) fflush (budget_Pwat5); 
        } 
} /* end of BIRbudg_reset() */



/*******/
/*! \brief Set up the Basin & Indicator Region (BIR) linkages/inheritances

*/
void BIRinit(void)
{
    int ix,iy,cellLoc,ibas;
    int ii, jj, basnID;
    int basnCnt=-1; /* the whole system basin-0 in the basinIR file is not considered part of this count of the number of Basin/IRs */
    char ss[222], *line, modnam[20], boundIR[3];

    sprintf( modelFileName, "%s/%s/Data/basinIR", ModelPath, ProjName );
/* Open file with basin linkage/inheritance data */
    if ( ( basInFile = fopen( modelFileName, "r" ) ) ==  NULL )
    {
        sprintf( msgStr,"Can't open %s basin definition input file! ",modelFileName ) ; usrErr(msgStr);
        exit(-1) ;
    }

    fgets( ss, 220, basInFile );fgets( ss, 220, basInFile ); /* skip 2 header lines */
    fgets( ss, 220, basInFile ); sscanf( ss,"%s", &modnam); 
    if (strcmp(modnam,modelName) != 0) {
        sprintf(msgStr, "The model name (%s) found in the %s file doesn't match the one (%s) you asked for in Driver.parm!",
                modnam, modelFileName, modelName); usrErr(msgStr);
        exit(-1);
    }
        /* Allocate memory for first basin */
    if ( (basins = ( basnDef *) malloc( (size_t) sizeof( basnDef ))) == NULL ) {
        printf( "Failed to allocate memory for first basin (%s)\n ", basins->basnTxt ) ;
        exit( -2 ) ;
    }
        /* allocate memory for array of pointers to basin atts */
    if ( (basn_list = 
          ( basnDef **) malloc( (size_t) sizeof( basnDef *) * (numBasn+1))) == NULL )
    {
        printf( "Failed to allocate memory for basin_list\n " ) ;
        exit( -2 ) ;
    };


    fgets( ss, 220, basInFile ); /* skip the column name header */
    
    while ( fgets( ss, 220, basInFile ) != NULL && !feof( basInFile ) )
    {
        line = ss;
        sscanf (line, "%d\t%s\t%d",&basnID, &basins->basnTxt,&basins->numIR);
        line = Scip( line, '\t' );line = Scip( line, '\t' );line = Scip( line, '\t' );
/* the indicator regions within a hydrologic basin are subscripted starting at 1,
   with the 0'th indicator region being the hydrologic basin itself.  Flok is
   used to check for allowable flows among basins/indicator-regions, to determine if user configured
   BIRs are OK (NOT used to constrain calculated flows, but only used to verify that no unallowed interbasin
   overland flow is occuring via some "leak" in the vector topology) */
        for (ii=0; ii<basins->numIR; ii++) {
            sscanf (line, "%d%", &basins->IR[ii]);
            basins->FLok[ii] = basins->IR[ii]; /* flow is allowed among IRegions in a particular hydro basin (family) */
            basins->numFLok++;
            line = Scip( line, ',' );
        }
        basnCnt++; /* count the number of basins&IRs to check that the # in the file is same as # defined on the map */

/* if there are any FlowOk Basin/Indicator-Region attached to this Basin/Indicator-Region, read them */
        while (1) {
            while ( *line != '#' && *line != '\0' ) line++;
            if(*line != '\0') {
                ++line;
                sscanf (line, "%d%", &basins->FLok[ii]);
                basins->numFLok++;
                ii++;
            }
            else break;
        }

        basn_list[basnID] = basins;
    
            /* Allocate memory for next basin */
        if ( ( basins = ( basnDef *) malloc( (size_t) sizeof(  basnDef ))) == NULL )
        {
            printf( "Failed to allocate memory for next basin (%s)\n ", basins->basnTxt ) ;
            exit( -2 ) ;
        }

     
    } /* end of reading basin lines */
    
    free ((char *)basins);
    fclose(basInFile);

    if (basnCnt != numBasn) { sprintf(msgStr, "Error - the %d basins read from basinIR does not match the %d basins in the basins map! Please fix the basinIR file.\n",
                                      basnCnt,numBasn); usrErr(msgStr); exit (-1); }
 
/* count the number of cells in the Basins/Indicator-Regions */
    for (ibas=numBasn;ibas>=0;ibas--) numCells[ibas] = 0;
    for(ix=0; ix<=s0+1; ix++) 
        for(iy=0; iy<=s1+1; iy++)
            if (ON_MAP[cellLoc= T(ix,iy)]) { 
                numCells[basn[cellLoc]]++; /* count the number of cells in each basin */
                numCells[0]++; /* count the number of cells in each basin */
            }
        /* whole system (basin 0) is calc'd independently, while each of the hydro basins are
           sums of their respective Indicator Regions */
    for (ibas=numBasn;ibas>=0;ibas--) {
        basins =  basn_list[ibas];
        basins->family = ibas; /* set the basin to initially be it's own family; below it may find that it is a child of a diff family */
        basins->parent = ibas; /* set the basin to initially be a childless parent */
        for (ii=0; ii<basins->numIR; ii++) {
        
            basn_list[basins->IR[ii]]->family = ibas; /* the hydrologic basin family name of this IRegion */
            basn_list[basins->IR[ii]]->parent = 0; /* this IR is a child, not a parent */
            numCells[ibas] += numCells[basins->IR[ii]]; /* add the IRegion cells to the parent basin #cells */
        }
        basins->conv_m3TOcm = conv_mTOcm/(numCells[ibas]*CELL_SIZE);
        basins->conv_kgTOmgm2 = conv_kgTOmg/(numCells[ibas]*CELL_SIZE);

    }
} /* end of BIRinit() */



/*******/
/*! \brief Open files and create headers for BIR output

*/
void BIRoutfiles(void)
{
    int Fnum,Fsets,ibasLow,ibasHigh,ibas;


    
/********/
/* For the budget & BIRavg files, we can have multiple sets of files, with
   >1 sets needed if a large (>12) number of Basins/Indicator-Regions are defined.  First,
   we open the files, then go to write headers for them */

    {/*Open the primary files for all of the budgets & BIRavgs */
    
/* Indicator region avgs */
        sprintf( modelFileName, "%s/%s/Output/Budget/BIRavg1", OutputPath, ProjName ); 
        if ( ( BIRavg1 = fopen( modelFileName, "w" ) ) ==  NULL ) 
        {printf( "Can't open BIRavg1 file! " );exit(-1) ;} 
/* water budget/massCheck info - units in acre-feet */
        sprintf( modelFileName, "%s/%s/Output/Budget/budg_Wacr1", OutputPath, ProjName ); 
        if ( ( budget_Wacr1 = fopen( modelFileName, "w" ) ) ==  NULL ) 
        {printf( "Can't open budg_Wacr1 file! " );exit(-1) ;} 
/* water budget/massCheck info - units in cm height */
        sprintf( modelFileName, "%s/%s/Output/Budget/budg_Wcm1", OutputPath, ProjName ); 
        if ( ( budget_Wcm1 = fopen( modelFileName, "w" ) ) ==  NULL ) 
        {printf( "Can't open budg_Wcm1 file! " );exit(-1) ;}
/* Phosph budget/massCheck info for all P fractions (units=Mg) */
        sprintf( modelFileName, "%s/%s/Output/Budget/budg_P1", OutputPath, ProjName ); 
        if ( ( budget_P1 = fopen( modelFileName, "w" ) ) ==  NULL ) 
        {printf( "Can't open budg_P1 file! " );exit(-1) ;} 
/* Phosph budget/massCheck info for all P fractions (units=mg/m2) */
        sprintf( modelFileName, "%s/%s/Output/Budget/budg_Par1", OutputPath, ProjName ); 
        if ( ( budget_Par1 = fopen( modelFileName, "w" ) ) ==  NULL ) 
        {printf( "Can't open budg_Par1 file! " );exit(-1) ;} 
/* Salt budget/massCheck info */
        sprintf( modelFileName, "%s/%s/Output/Budget/budg_S1", OutputPath, ProjName ); 
        if ( ( budget_S1 = fopen( modelFileName, "w" ) ) ==  NULL ) 
        {printf( "Can't open budg_S1 file! " );exit(-1) ;} 
        if (!ESPmodeON) {
/* Phosph budget/massCheck info for live P fraction */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pliv1", OutputPath, ProjName ); 
            if ( ( budget_Plive1 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Pliv1 file! " );exit(-1) ;}
/* Phosph budget/massCheck info for dead P fraction */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pded1", OutputPath, ProjName ); 
            if ( ( budget_Pdead1 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Pded1 file! " );exit(-1) ;} 
/* Phosph budget/massCheck info for water-borne P fraction */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pwat1", OutputPath, ProjName ); 
            if ( ( budget_Pwat1 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Pwat1 file! " );exit(-1) ;}
        }


        if (numBasn>=13) {
/*Open the secondary files for all of the budgets & BIRavgs if we have more than 12 basins */
/* Indicator region avgs */
            sprintf( modelFileName, "%s/%s/Output/Budget/BIRavg2", OutputPath, ProjName ); 
            if ( ( BIRavg2 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            { printf( "Can't open BIRavg2 file! " ); exit(-1) ;} 
/* water budget/massCheck info - units in acre-feet */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Wacr2", OutputPath, ProjName ); 
            if ( ( budget_Wacr2 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            { printf( "Can't open budg_Wacr2 file! " ); exit(-1) ;} 
/* water budget/massCheck info - units in cm height */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Wcm2", OutputPath, ProjName ); 
            if ( ( budget_Wcm2 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Wcm2 file! " ); exit(-1) ;} 
/* Phosph budget/massCheck info for all P fractions (units=Mg) */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_P2", OutputPath, ProjName ); 
            if ( ( budget_P2 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_P2 file! " );exit(-1) ;} 
/* Phosph budget/massCheck info for all P fractions (units=mg/m2) */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Par2", OutputPath, ProjName ); 
            if ( ( budget_Par2 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Par2 file! " );exit(-1) ;} 
/* Salt budget/massCheck info */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_S2", OutputPath, ProjName ); 
            if ( ( budget_S2 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_S2 file! " );exit(-1) ;} 

            if (!ESPmodeON) {
/* Phosph budget/massCheck info for live P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pliv2", OutputPath, ProjName ); 
                if ( ( budget_Plive2 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pliv2 file! " );exit(-1) ;}
/* Phosph budget/massCheck info for dead P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pded2", OutputPath, ProjName ); 
                if ( ( budget_Pdead2 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pded2 file! " );exit(-1) ;} 
/* Phosph budget/massCheck info for water-borne P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pwat2", OutputPath, ProjName ); 
                if ( ( budget_Pwat2 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pwat2 file! " );exit(-1) ;} 
            }/* end of !ESPmodeON */
    
        } /* end of opening secondary files */

        
        if (numBasn>=26) {
/*Open the tertiary files for all of the budgets if we have more than 25 basins */
/* Indicator region avgs */
            sprintf( modelFileName, "%s/%s/Output/Budget/BIRavg3", OutputPath, ProjName ); 
            if ( ( BIRavg3 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            { printf( "Can't open BIRavg3 file! " ); exit(-1) ;} 
/* water budget/massCheck info - units in acre-feet */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Wacr3", OutputPath, ProjName ); 
            if ( ( budget_Wacr3 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            { printf( "Can't open budg_Wacr3 file! " ); exit(-1) ;} 
/* water budget/massCheck info - units in cm height */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Wcm3", OutputPath, ProjName ); 
            if ( ( budget_Wcm3 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Wcm3 file! " ); exit(-1) ;} 
/* Phosph budget/massCheck info for all P fractions (units=Mg) */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_P3", OutputPath, ProjName ); 
            if ( ( budget_P3 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_P3 file! " );exit(-1) ;} 
/* Phosph budget/massCheck info for all P fractions (units=mg/m2) */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Par3", OutputPath, ProjName ); 
            if ( ( budget_Par3 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Par3 file! " );exit(-1) ;} 
/* Salt budget/massCheck info */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_S3", OutputPath, ProjName ); 
            if ( ( budget_S3 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_S3 file! " );exit(-1) ;} 

            if (!ESPmodeON) {
/* Phosph budget/massCheck info for live P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pliv3", OutputPath, ProjName ); 
                if ( ( budget_Plive3 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pliv3 file! " );exit(-1) ;}
/* Phosph budget/massCheck info for dead P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pded3", OutputPath, ProjName ); 
                if ( ( budget_Pdead3 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pded3 file! " );exit(-1) ;} 
/* Phosph budget/massCheck info for water-borne P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pwat3", OutputPath, ProjName ); 
                if ( ( budget_Pwat3 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pwat3 file! " );exit(-1) ;} 
            }/* end of !ESPmodeON */
    
        } /* end of opening tertiary files */
        
        if (numBasn>=39) {
/*Open the fourth set of files for all of the budgets if we have more than 38 basins */
/* Indicator region avgs */
            sprintf( modelFileName, "%s/%s/Output/Budget/BIRavg4", OutputPath, ProjName ); 
            if ( ( BIRavg4 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            { printf( "Can't open BIRavg4 file! " ); exit(-1) ;} 
/* water budget/massCheck info - units in acre-feet */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Wacr4", OutputPath, ProjName ); 
            if ( ( budget_Wacr4 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            { printf( "Can't open budg_Wacr4 file! " ); exit(-1) ;} 
/* water budget/massCheck info - units in cm height */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Wcm4", OutputPath, ProjName ); 
            if ( ( budget_Wcm4 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Wcm4 file! " ); exit(-1) ;} 
/* Phosph budget/massCheck info for all P fractions (units=Mg) */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_P4", OutputPath, ProjName ); 
            if ( ( budget_P4 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_P4 file! " );exit(-1) ;} 
/* Phosph budget/massCheck info for all P fractions (units=mg/m2) */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Par4", OutputPath, ProjName ); 
            if ( ( budget_Par4 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Par4 file! " );exit(-1) ;} 
/* Salt budget/massCheck info */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_S4", OutputPath, ProjName ); 
            if ( ( budget_S4 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_S4 file! " );exit(-1) ;} 

            if (!ESPmodeON) {
/* Phosph budget/massCheck info for live P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pliv4", OutputPath, ProjName ); 
                if ( ( budget_Plive4 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pliv4 file! " );exit(-1) ;}
/* Phosph budget/massCheck info for dead P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pded4", OutputPath, ProjName ); 
                if ( ( budget_Pdead4 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pded4 file! " );exit(-1) ;} 
/* Phosph budget/massCheck info for water-borne P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pwat4", OutputPath, ProjName ); 
                if ( ( budget_Pwat4 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pwat4 file! " );exit(-1) ;} 
            }/* end of !ESPmodeON */
    
        } /* end of opening fourth set of files */
        
        if (numBasn>=52) {
/*Open the fifth set of files for all of the budgets if we have more than 51 basins */
/* Indicator region avgs */
            sprintf( modelFileName, "%s/%s/Output/Budget/BIRavg5", OutputPath, ProjName ); 
            if ( ( BIRavg5 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            { printf( "Can't open BIRavg5 file! " ); exit(-1) ;} 
/* water budget/massCheck info - units in acre-feet */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Wacr5", OutputPath, ProjName ); 
            if ( ( budget_Wacr5 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            { printf( "Can't open budg_Wacr5 file! " ); exit(-1) ;} 
/* water budget/massCheck info - units in cm height */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Wcm5", OutputPath, ProjName ); 
            if ( ( budget_Wcm5 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Wcm5 file! " ); exit(-1) ;} 
/* Phosph budget/massCheck info for all P fractions (units=Mg) */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_P5", OutputPath, ProjName ); 
            if ( ( budget_P5 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_P5 file! " );exit(-1) ;} 
/* Phosph budget/massCheck info for all P fractions (units=mg/m2) */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_Par5", OutputPath, ProjName ); 
            if ( ( budget_Par5 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_Par5 file! " );exit(-1) ;} 
/* Salt budget/massCheck info */
            sprintf( modelFileName, "%s/%s/Output/Budget/budg_S5", OutputPath, ProjName ); 
            if ( ( budget_S5 = fopen( modelFileName, "w" ) ) ==  NULL ) 
            {printf( "Can't open budg_S5 file! " );exit(-1) ;} 

            if (!ESPmodeON) {
/* Phosph budget/massCheck info for live P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pliv5", OutputPath, ProjName ); 
                if ( ( budget_Plive5 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pliv5 file! " );exit(-1) ;}
/* Phosph budget/massCheck info for dead P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pded5", OutputPath, ProjName ); 
                if ( ( budget_Pdead5 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pded5 file! " );exit(-1) ;} 
/* Phosph budget/massCheck info for water-borne P fraction */
                sprintf( modelFileName, "%s/%s/Output/Budget/budg_Pwat5", OutputPath, ProjName ); 
                if ( ( budget_Pwat5 = fopen( modelFileName, "w" ) ) ==  NULL ) 
                {printf( "Can't open budg_Pwat5 file! " );exit(-1) ;} 
            }/* end of !ESPmodeON */
    
        } /* end of opening fifth set of files */
        
    } /* end of opening all files */



/********/
/* Now we need to print the headers to each set(s) of basin files */
/* First, figure out the number of files sets, and then loop thru that */
        /* there are 13 total basins printed per file (basin 0 is whole system,
           not counted in numBasn, but always printed out in first file) */
    if (numBasn>=52) { /* with more than 58 (plus whole-system basin == 59) basins: */
        Fsets=5;              /* we have five sets of basin files */
    }
    else if (numBasn>=39) { /* with more than 38 (plus whole-system basin == 39) basins: */
        Fsets=4;              /* we have four sets of basin files */
    }
    else if (numBasn>=26) { /* with more than 25 (plus whole-system basin == 26) basins: */
        Fsets=3;              /* we have three sets of basin files */
    }
    else if (numBasn>=13) { /* with more than 12 (plus whole-system basin == 13) basins: */
        Fsets=2;              /* we have two sets of basin files */
    }
    else {
        Fsets=1;       /* otherwise, just one set of files */
    }

    for (Fnum = 1; Fnum <= Fsets; Fnum++) { 
    
        if (Fnum == 1) { /* first file set */
            if (Fsets > 1) {
                ibasLow = 0;        
                ibasHigh = 12;
            }
            else {
                ibasLow = 0;
                ibasHigh = Min(12,numBasn);
            }
        }

        if (Fnum == 2) { /* second file set */
            if (Fsets > 2) {
                ibasLow = 13;        
                ibasHigh = 25;
            }
            else {
                ibasLow = 13;
                ibasHigh = Min(25,numBasn);
            }
        }

        if (Fnum == 3) { /* third file set */
            if (Fsets > 3) {
                ibasLow = 26;        
                ibasHigh = 38;
            }
            else {
                ibasLow = 26;
                ibasHigh = Min(38,numBasn);
            }
        }
        
        if (Fnum == 4) { /* fourth file set */
            if (Fsets > 4) {
                ibasLow = 39;        
                ibasHigh = 51;
            }
            else {
                ibasLow = 39;
                ibasHigh = Min(51,numBasn);
            }
        }
        

        else if (Fnum == 5) {
            ibasLow = 52;        
            ibasHigh = numBasn;
        }

/*****/
/* now print the 4 header lines for all of the budget & BIRavg files */

/* LINE 1&2: first header line, providing scenario ID; second line is units, etc. */
/* feb05 added two tabs at end of BIRavg header to accomodate sensitivity parm label & value */
        fprintf ( ((Fnum==5)?(BIRavg5):
                   ((Fnum==4)?(BIRavg4):
                   ((Fnum==3)?(BIRavg3):
                    ((Fnum==2)?(BIRavg2):
                     (BIRavg1) ) ) ) ), 
                  "%s %s %s %s scenario: Basins/Indicator-Regions (BIR): daily mean attributes. \nUnits = Hydro depths== m, TP surface & pore water==mg/L, TP soil==mg/kg, noncalc&calc periph==gC/m2, mac==kgC/m2, LandElevation= m NGVD'29+DATUM_DISTANCE(6m in v2.3).\n          \t\t\t ", &modelName, &modelVers, &SimAlt, &SimModif ); 
        fprintf ( ((Fnum==5)?(budget_Wacr5):
                   ((Fnum==4)?(budget_Wacr4):
                   ((Fnum==3)?(budget_Wacr3):
                    ((Fnum==2)?(budget_Wacr2):
                     (budget_Wacr1) ) ) ) ), 
                  "%s %s %s %s scenario: Basins/Indicator-Regions (BIR): water budget and mass balance check. \nUnits = thousands of acre-feet per basin, with error (and cumulative net error) reporting in mm height across basin.\n          \t ", &modelName, &modelVers, &SimAlt, &SimModif ); 
        fprintf ( ((Fnum==5)?(budget_Wcm5):
                   ((Fnum==4)?(budget_Wcm4):
                   ((Fnum==3)?(budget_Wcm3):
                    ((Fnum==2)?(budget_Wcm2):
                     (budget_Wcm1) ) ) ) ), 
                  "%s %s %s %s scenario: Basins/Indicator-Regions (BIR): water budget and mass balance check. \nUnits = cm per basin, with error (and cumulative net error) reporting in mm height across basin.\n          \t ", &modelName, &modelVers, &SimAlt, &SimModif ); 
        fprintf ( ((Fnum==5)?(budget_S5):
                   ((Fnum==4)?(budget_S4):
                   ((Fnum==3)?(budget_S3):
                    ((Fnum==2)?(budget_S2):
                     (budget_S1) ) ) ) ), 
                  "%s %s %s %s scenario: Basins/Indicator-Regions (BIR): conservative tracer budget and mass balance check. \nUnits = metric tons (Mg) per basin, with error (and cumulative net error) reporting in ug/m2 in basin.\n          \t ", &modelName, &modelVers, &SimAlt, &SimModif ); 
        fprintf ( ((Fnum==5)?(budget_P5):
                   ((Fnum==4)?(budget_P4):
                   ((Fnum==3)?(budget_P3):
                    ((Fnum==2)?(budget_P2):
                     (budget_P1) ) ) ) ), 
                  "%s %s %s %s scenario: Basins/Indicator-Regions (BIR): Total P budget and mass balance check for all fractions. \nUnits = metric tons (Mg) per basin, with error (and cumulative net error) reporting in ug/m2 in basin.\n          \t ", &modelName, &modelVers, &SimAlt, &SimModif ); 
        fprintf ( ((Fnum==5)?(budget_Par5):
                   ((Fnum==4)?(budget_Par4):
                   ((Fnum==3)?(budget_Par3):
                    ((Fnum==2)?(budget_Par2):
                     (budget_Par1) ) ) ) ), 
                  "%s %s %s %s scenario: Basins/Indicator-Regions (BIR): Total P budget and mass balance check for all fractions. \nUnits = mg/m2 in basin, with error (and cumulative net error) reporting in ug/m2 in basin.\n          \t ", &modelName, &modelVers, &SimAlt, &SimModif ); 
        if (!ESPmodeON) {

            fprintf ( ((Fnum==5)?(budget_Plive5):
                       ((Fnum==4)?(budget_Plive4):
                       ((Fnum==3)?(budget_Plive3):
                        ((Fnum==2)?(budget_Plive2):
                         (budget_Plive1) ) ) ) ), 
                      "%s %s %s %s scenario: Basins/Indicator-Regions (BIR): Total P budget and mass balance check for live fraction. \nUnits =  mg/m2 in basin, with error (and cumulative net error) ug/m2.\n          \t ", &modelName, &modelVers, &SimAlt, &SimModif ); 
            fprintf ( ((Fnum==5)?(budget_Pdead5):
                       ((Fnum==4)?(budget_Pdead4):
                       ((Fnum==3)?(budget_Pdead3):
                        ((Fnum==2)?(budget_Pdead2):
                         (budget_Pdead1) ) ) ) ), 
                      "%s %s %s %s scenario: Basins/Indicator-Regions (BIR): Total P budget and mass balance check for dead, non-water-borne fraction. \nUnits =  mg/m2 in basin, with error (and cumulative net error) ug/m2.\n          \t ", &modelName, &modelVers, &SimAlt, &SimModif ); 
            fprintf ( ((Fnum==5)?(budget_Pwat5):
                       ((Fnum==4)?(budget_Pwat4):
                       ((Fnum==3)?(budget_Pwat3):
                        ((Fnum==2)?(budget_Pwat2):
                         (budget_Pwat1) ) ) ) ), 
                      "%s %s %s %s scenario: Basins/Indicator-Regions (BIR): Total P budget and mass balance check for water-borne fraction (some horiz I/O not shown - see budget for all combined fractions). \nUnits =  mg/m2 in basin, with error (and cumulative net error) ug/m2.\n          \t ", &modelName, &modelVers, &SimAlt, &SimModif ); 
        }

/* LINE 3: third header line, looping thru basins, providing basin ID and size */
        for (ibas = ibasHigh; ibas >= ibasLow; ibas--) 
        {
        fprintf ( ((Fnum==5)?(BIRavg5):
                  ((Fnum==4)?(BIRavg4):
                   ((Fnum==3)?(BIRavg3):
                    ((Fnum==2)?(BIRavg2):
                     (BIRavg1) ) ) ) ), 
                      "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t         \t         \t         \t         \t         \t",
                      ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) ); 
        fprintf ( ((Fnum==5)?(budget_Wacr5):
                   ((Fnum==4)?(budget_Wacr4):
                   ((Fnum==3)?(budget_Wacr3):
                    ((Fnum==2)?(budget_Wacr2):
                     (budget_Wacr1) ) ) ) ), 
                      "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t",
                      ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) ); 
        fprintf ( ((Fnum==5)?(budget_Wcm5):
                   ((Fnum==4)?(budget_Wcm4):
                   ((Fnum==3)?(budget_Wcm3):
                    ((Fnum==2)?(budget_Wcm2):
                     (budget_Wcm1) ) ) ) ), 
                      "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t",
                      ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) ); 
        fprintf ( ((Fnum==5)?(budget_S5):
                   ((Fnum==4)?(budget_S4):
                   ((Fnum==3)?(budget_S3):
                    ((Fnum==2)?(budget_S2):
                     (budget_S1) ) ) ) ), 
                      "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t",
                      ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) ); 
            if (!ESPmodeON) {
                fprintf ( ((Fnum==5)?(budget_P5):
                          ((Fnum==4)?(budget_P4):
                           ((Fnum==3)?(budget_P3):
                            ((Fnum==2)?(budget_P2):
                             (budget_P1) ) ) ) ), 
                          "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t",
                          ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) );
                fprintf ( ((Fnum==5)?(budget_Par5):
                           ((Fnum==4)?(budget_Par4):
                           ((Fnum==3)?(budget_Par3):
                            ((Fnum==2)?(budget_Par2):
                             (budget_Par1) ) ) ) ), 
                      "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t",
                          ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) );
                fprintf ( ((Fnum==5)?(budget_Plive5):
                           ((Fnum==4)?(budget_Plive4):
                            ((Fnum==3)?(budget_Plive3):
                             ((Fnum==2)?(budget_Plive2):
                              (budget_Plive1) ) ) ) ), 
                      "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t",
                          ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) ); 
                fprintf ( ((Fnum==5)?(budget_Pdead5):
                           ((Fnum==4)?(budget_Pdead4):
                            ((Fnum==3)?(budget_Pdead3):
                             ((Fnum==2)?(budget_Pdead2):
                              (budget_Pdead1) ) ) ) ), 
                      "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t          \t        \t         \t         \t         \t         \t         \t         \t         \t",
                          ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) ); 
                fprintf ( ((Fnum==5)?(budget_Pwat5):
                           ((Fnum==4)?(budget_Pwat4):
                            ((Fnum==3)?(budget_Pwat3):
                             ((Fnum==2)?(budget_Pwat2):
                              (budget_Pwat1) ) ) ) ), 
                      "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t",
                          ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) ); 
            }
    
            else {
                fprintf ( ((Fnum==5)?(budget_P5):
                           ((Fnum==4)?(budget_P4):
                           ((Fnum==3)?(budget_P3):
                            ((Fnum==2)?(budget_P2):
                             (budget_P1) ) ) ) ), 
                      "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t",
                          ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) );
                fprintf ( ((Fnum==5)?(budget_Par5):
                           ((Fnum==4)?(budget_Par4):
                           ((Fnum==3)?(budget_Par3):
                            ((Fnum==2)?(budget_Par2):
                             (budget_Par1) ) ) ) ), 
                      "  BIR %2d =\t%6.1fmi^2,\t%6.1fkm^2\t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t         \t",
                          ibas,(numCells[ibas]*CELL_SIZE*3.8610039e-7),(numCells[ibas]*CELL_SIZE*1.0e-6) );
            }
        } /*end of loop for second header line */

/* LINE 4: beginning of fourth header line */
/* feb05 added two tabs prior to "Date" of BIRavg header to accomodate sensitivity parm label & value */
        fprintf ( ((Fnum==5)?(BIRavg5):
                   ((Fnum==4)?(BIRavg4):
                   ((Fnum==3)?(BIRavg3):
                    ((Fnum==2)?(BIRavg2):
                     (BIRavg1) ) ) ) ), "\nSParm\tSParmValu\t    Date\t" ); 
        fprintf ( ((Fnum==5)?(budget_Wacr5):
                   ((Fnum==4)?(budget_Wacr4):
                   ((Fnum==3)?(budget_Wacr3):
                    ((Fnum==2)?(budget_Wacr2):
                     (budget_Wacr1) ) ) ) ), "\n    Date\t" ); 
        fprintf ( ((Fnum==5)?(budget_Wcm5):
                   ((Fnum==4)?(budget_Wcm4):
                   ((Fnum==3)?(budget_Wcm3):
                    ((Fnum==2)?(budget_Wcm2):
                     (budget_Wcm1) ) ) ) ), "\n    Date\t" ); 
        fprintf ( ((Fnum==5)?(budget_S5):
                   ((Fnum==4)?(budget_S4):
                   ((Fnum==3)?(budget_S3):
                    ((Fnum==2)?(budget_S2):
                     (budget_S1) ) ) ) ), "\n    Date\t" ); 
        fprintf ( ((Fnum==5)?(budget_P5):
                   ((Fnum==4)?(budget_P4):
                   ((Fnum==3)?(budget_P3):
                    ((Fnum==2)?(budget_P2):
                     (budget_P1) ) ) ) ), "\n    Date\t" ); 
        fprintf ( ((Fnum==5)?(budget_Par5):
                   ((Fnum==4)?(budget_Par4):
                   ((Fnum==3)?(budget_Par3):
                    ((Fnum==2)?(budget_Par2):
                     (budget_Par1) ) ) ) ), "\n    Date\t" ); 
        if (!ESPmodeON) {
            fprintf ( ((Fnum==5)?(budget_Plive5):
                       ((Fnum==4)?(budget_Plive4):
                       ((Fnum==3)?(budget_Plive3):
                        ((Fnum==2)?(budget_Plive2):
                         (budget_Plive1) ) ) ) ), "\n    Date\t" ); 
            fprintf ( ((Fnum==5)?(budget_Pdead5):
                       ((Fnum==4)?(budget_Pdead4):
                       ((Fnum==3)?(budget_Pdead3):
                        ((Fnum==2)?(budget_Pdead2):
                         (budget_Pdead1) ) ) ) ), "\n    Date\t" ); 
            fprintf ( ((Fnum==5)?(budget_Pwat5):
                       ((Fnum==4)?(budget_Pwat4):
                       ((Fnum==3)?(budget_Pwat3):
                        ((Fnum==2)?(budget_Pwat2):
                         (budget_Pwat1) ) ) ) ), "\n    Date\t" ); 
        }

/* LINE 4: remainder of fourth header line, looping thru basins, providing column/variable ID attributes */
        for (ibas = ibasHigh; ibas >= ibasLow; ibas--) 
        {
        
        fprintf ( ((Fnum==5)?(BIRavg5):
                  ((Fnum==4)?(BIRavg4):
                   ((Fnum==3)?(BIRavg3):
                    ((Fnum==2)?(BIRavg2):
                     (BIRavg1) ) ) ) ), 
                      "   SfWat_%d\t   Unsat_%d\t    TPsf_%d\t  TPpore_%d\t  TPsoil_%d\t  NCperi_%d\t   Cperi_%d\t     Mac_%d\t    Elev_%d\t",
                      ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas); 
        fprintf ( ((Fnum==5)?(budget_Wacr5):
                   ((Fnum==4)?(budget_Wacr4):
                   ((Fnum==3)?(budget_Wacr3):
                    ((Fnum==2)?(budget_Wacr2):
                     (budget_Wacr1) ) ) ) ), 
                      "  VolOld_%d\t    Rain_%d\t    Evap_%d\t  Transp_%d\t    Rchg_%d\t   StrIn_%d\t  StrOut_%d\t   OvlIn_%d\t  OvlOut_%d\t   SpgIn_%d\t  SpgOut_%d\t    GWin_%d\t   GWout_%d\t VolTarg_%d\t  VolNew_%d\t     Err_%d\t  SumErr_%d\t   InAvg_%d\t  OutAvg_%d\t",
                      ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas); 
        fprintf ( ((Fnum==5)?(budget_Wcm5):
                   ((Fnum==4)?(budget_Wcm4):
                   ((Fnum==3)?(budget_Wcm3):
                    ((Fnum==2)?(budget_Wcm2):
                     (budget_Wcm1) ) ) ) ), 
                      "  VolOld_%d\t    Rain_%d\t    Evap_%d\t  Transp_%d\t    Rchg_%d\t   StrIn_%d\t  StrOut_%d\t   OvlIn_%d\t  OvlOut_%d\t   SpgIn_%d\t  SpgOut_%d\t    GWin_%d\t   GWout_%d\t VolTarg_%d\t  VolNew_%d\t     Err_%d\t  SumErr_%d\t   InAvg_%d\t  OutAvg_%d\t",
                      ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas); 
        fprintf ( ((Fnum==5)?(budget_S5):
                   ((Fnum==4)?(budget_S4):
                   ((Fnum==3)?(budget_S3):
                    ((Fnum==2)?(budget_S2):
                     (budget_S1) ) ) ) ), 
                      "  MasOld_%d\t   StrIn_%d\t  StrOut_%d\t   OvlIn_%d\t  OvlOut_%d\t   SpgIn_%d\t  SpgOut_%d\t    GWin_%d\t   GWout_%d\t MasTarg_%d\t  MasNew_%d\t     Err_%d\t  SumErr_%d\t   InAvg_%d\t  OutAvg_%d\t",
                      ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas); 
            if (!ESPmodeON) {
                fprintf ( ((Fnum==5)?(budget_P5):
                          ((Fnum==4)?(budget_P4):
                           ((Fnum==3)?(budget_P3):
                            ((Fnum==2)?(budget_P2):
                             (budget_P1) ) ) ) ), 
                      "  MasOld_%d\t    Rain_%d\t   StrIn_%d\t  StrOut_%d\t   OvlIn_%d\t  OvlOut_%d\t   SpgIn_%d\t  SpgOut_%d\t    GWin_%d\t   GWout_%d\t MasTarg_%d\t  MasNew_%d\t     Err_%d\t  SumErr_%d\t   InAvg_%d\t  OutAvg_%d\t",
                          ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas);
                fprintf ( ((Fnum==5)?(budget_Par5):
                           ((Fnum==4)?(budget_Par4):
                           ((Fnum==3)?(budget_Par3):
                            ((Fnum==2)?(budget_Par2):
                             (budget_Par1) ) ) ) ), 
                      "  MasOld_%d\t    Rain_%d\t   StrIn_%d\t  StrOut_%d\t   OvlIn_%d\t  OvlOut_%d\t   SpgIn_%d\t  SpgOut_%d\t    GWin_%d\t   GWout_%d\t MasTarg_%d\t  MasNew_%d\t     Err_%d\t  SumErr_%d\t   InAvg_%d\t  OutAvg_%d\t",
                          ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas);
                fprintf ( ((Fnum==5)?(budget_Plive5):
                           ((Fnum==4)?(budget_Plive4):
                            ((Fnum==3)?(budget_Plive3):
                             ((Fnum==2)?(budget_Plive2):
                              (budget_Plive1) ) ) ) ), 
                      "  MacBio_%d\t CPerBio_%d\t NCPerBio_%d\t  MacNPP_%d\t CPerGPP_%d\tNCPerGPP_%d\t MacMort_%d\t CalgMort_%d\t NCalgMort_%d\t MasTarg_%d\t  MasNew_%d\t     Err_%d\t  SumErr_%d\t   InAvg_%d\t  OutAvg_%d\t",
                          ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas); 
                fprintf ( ((Fnum==5)?(budget_Pdead5):
                           ((Fnum==4)?(budget_Pdead4):
                            ((Fnum==3)?(budget_Pdead3):
                             ((Fnum==2)?(budget_Pdead2):
                              (budget_Pdead1) ) ) ) ), 
                      "  MasOld_%d\tMac-soil_%d\tPer-soil_%d\t Decomp_%d\tSettl_to_%d\t Sorb_to_%d\t  Desorb_%d\t MasTarg_%d\t  MasNew_%d\t     Err_%d\t  SumErr_%d\t   InAvg_%d\t  OutAvg_%d\t",
                          ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas); 
                fprintf ( ((Fnum==5)?(budget_Pwat5):
                           ((Fnum==4)?(budget_Pwat4):
                            ((Fnum==3)?(budget_Pwat3):
                             ((Fnum==2)?(budget_Pwat2):
                              (budget_Pwat1) ) ) ) ), 
                      "  MasOld_%d\t  P_rain_%d\tSettl_fr_%d\t SfMiner_%d\tSedMiner_%d\t  AlgUpt_%d\t  MacUpt_%d\t  DeSorb_%d\t Sorb_to_%d\t   StrIn_%d\t  StrOut_%d\t MasTarg_%d\t  MasNew_%d\t     Err_%d\t  SumErr_%d\t   InAvg_%d\t  OutAvg_%d\t",
                          ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas); 
            }
            else {
                fprintf ( ((Fnum==5)?(budget_P5):
                           ((Fnum==4)?(budget_P4):
                           ((Fnum==3)?(budget_P3):
                            ((Fnum==2)?(budget_P2):
                             (budget_P1) ) ) ) ), 
                      "  MasOld_%d\t    Rain_%d\t   Settl_%d\t   StrIn_%d\t  StrOut_%d\t   OvlIn_%d\t  OvlOut_%d\t   SpgIn_%d\t  SpgOut_%d\t    GWin_%d\t   GWout_%d\t MasTarg_%d\t  MasNew_%d\t     Err_%d\t  SumErr_%d\t   InAvg_%d\t  OutAvg_%d\t",
                          ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas);
                fprintf ( ((Fnum==5)?(budget_Par5):
                           ((Fnum==4)?(budget_Par4):
                           ((Fnum==3)?(budget_Par3):
                            ((Fnum==2)?(budget_Par2):
                             (budget_Par1) ) ) ) ), 
                      "  MasOld_%d\t    Rain_%d\t   Settl_%d\t   StrIn_%d\t  StrOut_%d\t   OvlIn_%d\t  OvlOut_%d\t   SpgIn_%d\t  SpgOut_%d\t    GWin_%d\t   GWout_%d\t MasTarg_%d\t  MasNew_%d\t     Err_%d\t  SumErr_%d\t   InAvg_%d\t  OutAvg_%d\t",
                          ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas,ibas);
            }
        } /* end of loop for third header line */

    } /*end of printing all files' headers */

        
} /* end of BIRoutfiles */


/*******/
/*! \brief Generate sums, means (avgs), on cell-by-cell basis (not by Basin/IRegion here).

*/
void CellAvg(void)
{
    int ix, iy, cellLoc;
    float avgStep = avg_Intvl / DT;
    
/* sums and average (& hydroperiod) calcs, not budget related */
    for(ix=1; ix<=s0; ix++) {
        for(iy=1; iy<=s1; iy++) {

            if(ON_MAP[cellLoc= T(ix,iy)])  {
                    /* annual hydroperiod on Oct 1 = JulDay ~273 (TODO: use true date fctn) */
                if ( FMOD(DAYJUL, 273.0) ) {
                    if (SURFACE_WAT[cellLoc] > 0.01 )  HydPerAnn[cellLoc] +=  DT; 
                }
                else {
                    HydPerAnn[cellLoc] += DT; /* TODO: (?) remove this line */
                }
                    
                    SfWatAvg[cellLoc] += SURFACE_WAT[cellLoc]; 
                    TotHeadAvg[cellLoc] += HydTotHd[cellLoc]; 
                    UnsatZavg[cellLoc] += UNSAT_DEPTH[cellLoc]; 
                    UnsatMoistAvg[cellLoc] += UNSAT_MOIST_PRP[cellLoc]; 
                    RainAvg[cellLoc]  += SF_WT_FROM_RAIN[cellLoc];
                    EvapAvg[cellLoc]  += SF_WT_EVAP[cellLoc];
                    TranspAvg[cellLoc]  += HYD_TRANSP[cellLoc];
                    ETAvg[cellLoc]  += HYD_ET[cellLoc];

                    TPSfWatAvg[cellLoc] += TP_SFWT_CONC_MG[cellLoc]; 
                    TPSedWatAvg[cellLoc] += TP_SEDWT_CONCACTMG[cellLoc]; 
                    TPSfUptAvg[cellLoc] += TP_SFWT_UPTAK[cellLoc]; 
                    TPSfMinAvg[cellLoc] += TP_SFWT_MINER[cellLoc]; 
                    TP_settlAvg[cellLoc] += TP_settl[cellLoc]; 
                    TPSedUptAvg[cellLoc] += TP_SEDWT_UPTAKE[cellLoc]; 
                    TPSedMinAvg[cellLoc] += TP_SED_MINER[cellLoc];
                    TPSorbAvg[cellLoc] += TP_SORBCONC_rep[cellLoc];
                    TPtoVOLAvg[cellLoc] +=  TPtoVOL_rep[cellLoc];
                    TPtoSOILAvg[cellLoc] +=  TPtoSOIL_rep[cellLoc];
                    SaltSfAvg[cellLoc] += SAL_SF_WT[cellLoc];
                    SaltSedAvg[cellLoc] += SAL_SED_WT[cellLoc];

                    Floc_fr_phBioAvg[cellLoc]  += Floc_fr_phBio[cellLoc];
                    
                    NC_PeriAvg[cellLoc] += NC_ALG[cellLoc];
                    NC_Peri_nppAvg[cellLoc]  += NC_ALG_NPP[cellLoc];
                    NC_Peri_mortAvg[cellLoc]  += NC_ALG_MORT[cellLoc];
                    NC_PeriRespAvg[cellLoc]  += NC_ALG_RESP[cellLoc];
                    NC_PeriNutCFAvg[cellLoc]  += NC_ALG_NUT_CF[cellLoc];
                    NC_Peri_PCAvg[cellLoc]  += NC_ALG_PCrep[cellLoc];
                    C_PeriAvg[cellLoc] += C_ALG[cellLoc]; 
                    C_Peri_nppAvg[cellLoc]  += C_ALG_NPP[cellLoc];
                    C_Peri_mortAvg[cellLoc]  += C_ALG_MORT[cellLoc];
                    C_PeriRespAvg[cellLoc]  += C_ALG_RESP[cellLoc];
                    C_PeriNutCFAvg[cellLoc]  += C_ALG_NUT_CF[cellLoc];
                    C_Peri_PCAvg[cellLoc]  += C_ALG_PCrep[cellLoc];
                    PeriLiteCFAvg[cellLoc]  += ALG_LIGHT_CF[cellLoc];
                    PeriAvg[cellLoc]  += ALG_TOT[cellLoc];

                    Mac_nphBioAvg[cellLoc]  += MAC_NOPH_BIOMAS[cellLoc];
                    Mac_phBioAvg[cellLoc]  += MAC_PH_BIOMAS[cellLoc];
                    Mac_nppAvg[cellLoc]  += PHBIO_NPP[cellLoc];
                    Mac_nphMortAvg[cellLoc]  += NPHBIO_MORT[cellLoc];
                    Mac_phMortAvg[cellLoc]  += PHBIO_MORT[cellLoc];
                    LAI_effAvg[cellLoc]  += LAI_eff[cellLoc];
                    Manning_nAvg[cellLoc]  += HYD_MANNINGS_N[cellLoc];
                    MacNutCfAvg[cellLoc]  += MAC_NUT_CF[cellLoc];
                    MacWatCfAvg[cellLoc]  += MAC_WATER_CF[cellLoc];
                    mac_nph_PCAvg[cellLoc]  += mac_nph_PC_rep[cellLoc];
                    mac_ph_PCAvg[cellLoc]  += mac_ph_PC_rep[cellLoc];
                    Mac_totBioAvg[cellLoc]  += MAC_TOT_BIOM[cellLoc];

                    SedElevAvg[cellLoc]  += SED_ELEV[cellLoc];
              
                if (avgPrint ) /* get the mean value */
                        { 
                            SfWatAvg[cellLoc] /= avgStep;
                            TotHeadAvg[cellLoc] /= avgStep;
                            UnsatZavg[cellLoc] /= avgStep;
                            UnsatMoistAvg[cellLoc] /= avgStep;
                            RainAvg[cellLoc]  /= avgStep ;
                            EvapAvg[cellLoc]  /= avgStep ;
                            TranspAvg[cellLoc] /= avgStep ;
                            ETAvg[cellLoc]  /= avgStep;

                            TPSfWatAvg[cellLoc] /= avgStep;
                            TPSedWatAvg[cellLoc] /= avgStep;
                            TPSfUptAvg[cellLoc] /= avgStep;
                            TPSfMinAvg[cellLoc] /= avgStep;
                            TP_settlAvg[cellLoc] /= avgStep;
                            TPSedUptAvg[cellLoc] /= avgStep;
                            TPSedMinAvg[cellLoc] /= avgStep;
                            TPSorbAvg[cellLoc] /= avgStep; 
                            TPtoVOLAvg[cellLoc] /= avgStep;  
                            TPtoSOILAvg[cellLoc] /= avgStep; 
                            SaltSfAvg[cellLoc] /= avgStep;
                            SaltSedAvg[cellLoc] /= avgStep;

                            Floc_fr_phBioAvg[cellLoc] /= avgStep ;

                            NC_PeriAvg[cellLoc] /= avgStep;
                            NC_Peri_nppAvg[cellLoc] /= avgStep;
                            NC_Peri_mortAvg[cellLoc] /= avgStep;
                            NC_PeriRespAvg[cellLoc] /= avgStep;
                            NC_PeriNutCFAvg[cellLoc] /= avgStep;
                            NC_Peri_PCAvg[cellLoc] /= avgStep;
                            C_PeriAvg[cellLoc] /= avgStep;
                            C_Peri_nppAvg[cellLoc] /= avgStep;
                            C_Peri_mortAvg[cellLoc] /= avgStep;
                            C_PeriRespAvg[cellLoc] /= avgStep;
                            C_PeriNutCFAvg[cellLoc] /= avgStep;
                            C_Peri_PCAvg[cellLoc] /= avgStep;
                            PeriLiteCFAvg[cellLoc] /= avgStep;
                            PeriAvg[cellLoc] /= avgStep;
                    	
                            Mac_nphBioAvg[cellLoc]  /= avgStep ;
                            Mac_phBioAvg[cellLoc]  /= avgStep ;
                            Mac_nppAvg[cellLoc]  /= avgStep ;
                            Mac_nphMortAvg[cellLoc]  /= avgStep ;
                            Mac_phMortAvg[cellLoc]  /= avgStep ;
                            LAI_effAvg[cellLoc]  /= avgStep ;
                            Manning_nAvg[cellLoc]  /= avgStep ;
                            MacNutCfAvg[cellLoc]  /= avgStep ;
                            MacWatCfAvg[cellLoc]  /= avgStep ;
                            mac_nph_PCAvg[cellLoc]  /= avgStep ;
                            mac_ph_PCAvg[cellLoc]  /= avgStep ;
                            Mac_totBioAvg[cellLoc]  /= avgStep;

                            SedElevAvg[cellLoc]  /= avgStep ;
                        }
            }
        }
    } /* end spatial loop for averages */
} /*end of CellAvg() */



/*******/
/*! \brief Zero the array holding hydroperiod data in cells (after printing).

*/
void Cell_reset_hydper(void) 
{
    int ix, iy, cellLoc; 

        for(ix=1; ix<=s0; ix++) 
            for(iy=1; iy<=s1; iy++) 
                if(ON_MAP[cellLoc= T(ix,iy)]) {
                    HydPerAnn[cellLoc] = 0.0;
                }
} 


/*******/
/*! \brief Zero the arrays holding selected variable averages in cells (after printing)/
******/
void Cell_reset_avg(void) 
{
    int ix, iy, cellLoc; 

    for(ix=1; ix<=s0; ix++) 
        for(iy=1; iy<=s1; iy++) 
            if(ON_MAP[cellLoc= T(ix,iy)]) {
                TotHeadAvg[cellLoc] = 0.0;
                SfWatAvg[cellLoc] = 0.0;
                UnsatZavg[cellLoc] = 0.0;
                UnsatMoistAvg[cellLoc] = 0.0;
                RainAvg[cellLoc] = 0.0;
                EvapAvg[cellLoc] = 0.0;
                TranspAvg[cellLoc] = 0.0;
                ETAvg[cellLoc] = 0.0;

                TPSfWatAvg[cellLoc] = 0.0;
                TPSfUptAvg[cellLoc] = 0.0;
                TPSfMinAvg[cellLoc] = 0.0;
                TP_settlAvg[cellLoc] = 0.0;
                TPSedWatAvg[cellLoc] = 0.0;
                TPSedUptAvg[cellLoc] = 0.0;
                TPSedMinAvg[cellLoc] = 0.0;
                TPSorbAvg[cellLoc] = 0.0;
                TPtoVOLAvg[cellLoc] = 0.0;
                TPtoSOILAvg[cellLoc] = 0.0;

                SaltSfAvg[cellLoc] = 0.0;
                SaltSedAvg[cellLoc] = 0.0;

                Floc_fr_phBioAvg[cellLoc]  = 0.0;

                NC_PeriAvg[cellLoc] = 0.0;
                NC_Peri_nppAvg[cellLoc] = 0.0;
                NC_Peri_mortAvg[cellLoc] = 0.0;
                NC_PeriRespAvg[cellLoc] = 0.0;
                NC_PeriNutCFAvg[cellLoc] = 0.0;
                C_Peri_PCAvg[cellLoc] = 0.0;
                C_PeriAvg[cellLoc] = 0.0;
                C_Peri_nppAvg[cellLoc] = 0.0;
                C_Peri_mortAvg[cellLoc] = 0.0;
                C_PeriRespAvg[cellLoc] = 0.0;
                C_PeriNutCFAvg[cellLoc] = 0.0;
                C_Peri_PCAvg[cellLoc] = 0.0;
                PeriAvg[cellLoc] = 0.0;
                PeriLiteCFAvg[cellLoc] = 0.0;

                Mac_nphBioAvg[cellLoc]  = 0.0;
                Mac_phBioAvg[cellLoc]  = 0.0;
                Mac_totBioAvg[cellLoc]  = 0.0;
                Mac_nppAvg[cellLoc]  = 0.0;
                Mac_nphMortAvg[cellLoc]  = 0.0;
                Mac_phMortAvg[cellLoc]  = 0.0;
                LAI_effAvg[cellLoc]  = 0.0;
                Manning_nAvg[cellLoc]  = 0.0;
                MacNutCfAvg[cellLoc]  = 0.0;
                MacWatCfAvg[cellLoc]  = 0.0;
                mac_nph_PCAvg[cellLoc]  = 0.0;
                mac_ph_PCAvg[cellLoc]  = 0.0;

                SedElevAvg[cellLoc]  = 0.0;
            }   
} /* end of Cell_reset_avg() */


/*******/
/*! \brief Allocate memory for the BIR-based and cell-based stats variables.

*/
void alloc_mem_stats (void)
{
    int ix,iy,cellLoc, numBasnMax=0;


    for(ix=0; ix<=s0+1; ix++) 
        for(iy=0; iy<=s1+1; iy++)
            if (ON_MAP[cellLoc= T(ix,iy)]) { 
                if (basn[cellLoc] > numBasnMax) numBasnMax = basn[cellLoc]; 
            }
    numBasn = numBasnMax; /* total number of basins in domain */
    
    sprintf (msgStr," %d Basins/Indicator-Regions...",numBasn); usrErr0(msgStr);
    
/* basin budget arrays (number of actual basins plus basin 0, which is whole system) */
    numCells = (int *) nalloc (sizeof(int)*(numBasn+1),"numCells"); /* number of cells in a basin */

/* Indicator region averages (non-budget) */
	Sfwat_avg = (double *) nalloc(sizeof(double)*(numBasn+1),"Sfwat_avg");
	init_pvar(Sfwat_avg,NULL,'b',0.0);
	Unsat_avg = (double *) nalloc(sizeof(double)*(numBasn+1),"Unsat_avg");
	init_pvar(Unsat_avg,NULL,'b',0.0);
	TPsf_avg = (double *) nalloc(sizeof(double)*(numBasn+1),"TPsf_avg");
	init_pvar(TPsf_avg,NULL,'b',0.0);
	TPpore_avg = (double *) nalloc(sizeof(double)*(numBasn+1),"TPpore_avg");
	init_pvar(TPpore_avg,NULL,'b',0.0);
	TPsoil_avg = (double *) nalloc(sizeof(double)*(numBasn+1),"TPsoil_avg");
	init_pvar(TPsoil_avg,NULL,'b',0.0);
	NCperi_avg = (double *) nalloc(sizeof(double)*(numBasn+1),"NCperi_avg");
	init_pvar(NCperi_avg,NULL,'b',0.0);
	Cperi_avg = (double *) nalloc(sizeof(double)*(numBasn+1),"Cperi_avg");
	init_pvar(Cperi_avg,NULL,'b',0.0);
	Mac_avg = (double *) nalloc(sizeof(double)*(numBasn+1),"Mac_avg");
	init_pvar(Mac_avg,NULL,'b',0.0);
	Elev_avg = (double *) nalloc(sizeof(double)*(numBasn+1),"Elev_avg");
	init_pvar(Elev_avg,NULL,'b',0.0);
/* hydro mass bal vars */
	SUMSF = (double *) nalloc(sizeof(double)*(numBasn+1),"SUMSF");
	init_pvar(SUMSF,NULL,'b',0.0);
	SUMGW = (double *) nalloc(sizeof(double)*(numBasn+1),"SUMGW");
	init_pvar(SUMGW,NULL,'b',0.0);
	SUMUW = (double *) nalloc(sizeof(double)*(numBasn+1),"SUMUW");
	init_pvar(SUMUW,NULL,'b',0.0);
	RAIN = (double *) nalloc(sizeof(double)*(numBasn+1),"RAIN");
	init_pvar(RAIN,NULL,'b',0.0);
	EVAP = (double *) nalloc(sizeof(double)*(numBasn+1),"EVAP");
	init_pvar(EVAP,NULL,'b',0.0);
	TRANSP = (double *) nalloc(sizeof(double)*(numBasn+1),"TRANSP");
	init_pvar(TRANSP,NULL,'b',0.0);
	RCHG = (double *) nalloc(sizeof(double)*(numBasn+1),"RCHG");
	init_pvar(RCHG,NULL,'b',0.0);

	TOT_VOL = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_VOL");
	init_pvar(TOT_VOL,NULL,'b',0.0);
	TOT_VOL_OLD = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_VOL_OLD");
	init_pvar(TOT_VOL_OLD,NULL,'b',0.0);
	VOL_IN = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_IN");
	init_pvar(VOL_IN,NULL,'b',0.0);
	VOL_OUT = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_OUT");
	init_pvar(VOL_OUT,NULL,'b',0.0);
	TOT_VOL_ERR = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_VOL_ERR");
	init_pvar(TOT_VOL_ERR,NULL,'b',0.0);
	TOT_VOL_CUM_ERR = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_VOL_CUM_ERR");
	init_pvar(TOT_VOL_CUM_ERR,NULL,'b',0.0);
	TOT_VOL_AVG_ERR = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_VOL_AVG_ERR");
	init_pvar(TOT_VOL_AVG_ERR,NULL,'b',0.0);
	VOL_ERR_PER_INFL = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_ERR_PER_INFL");
	init_pvar(VOL_ERR_PER_INFL,NULL,'b',0.0);

	VOL_IN_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_IN_SUM");
	init_pvar(VOL_IN_SUM,NULL,'b',0.0);
	VOL_OUT_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_OUT_SUM");
	init_pvar(VOL_OUT_SUM,NULL,'b',0.0);
	VOL_IN_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_IN_AVG");
	init_pvar(VOL_IN_AVG,NULL,'b',0.0);
	VOL_OUT_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_OUT_AVG");
	init_pvar(VOL_OUT_AVG,NULL,'b',0.0);

	TOT_VOL_CAN = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_VOL_CAN");
	init_pvar(TOT_VOL_CAN,NULL,'b',0.0);

	VOL_IN_STR = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_IN_STR");
	init_pvar(VOL_IN_STR,NULL,'b',0.0);
	VOL_IN_OVL = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_IN_OVL");
	init_pvar(VOL_IN_OVL,NULL,'b',0.0);
	VOL_IN_SPG = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_IN_SPG");
	init_pvar(VOL_IN_SPG,NULL,'b',0.0);
	VOL_IN_GW = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_IN_GW");
	init_pvar(VOL_IN_GW,NULL,'b',0.0);

	VOL_OUT_STR = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_OUT_STR");
	init_pvar(VOL_OUT_STR,NULL,'b',0.0);
	VOL_OUT_OVL = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_OUT_OVL");
	init_pvar(VOL_OUT_OVL,NULL,'b',0.0);
	VOL_OUT_SPG = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_OUT_SPG");
	init_pvar(VOL_OUT_SPG,NULL,'b',0.0);
	VOL_OUT_GW = (double *) nalloc(sizeof(double)*(numBasn+1),"VOL_OUT_GW");
	init_pvar(VOL_OUT_GW,NULL,'b',0.0);

/* phosph mass bal vars for all storage fractions */
	P_CELL = (double *) nalloc(sizeof(double)*(numBasn+1),"P_CELL");
	init_pvar(P_CELL,NULL,'b',0.0);
	P_RAIN = (double *) nalloc(sizeof(double)*(numBasn+1),"P_RAIN");
	init_pvar(P_RAIN,NULL,'b',0.0);

	P = (double *) nalloc(sizeof(double)*(numBasn+1),"P");
	init_pvar(P,NULL,'b',0.0);
	P_OLD = (double *) nalloc(sizeof(double)*(numBasn+1),"P_OLD");
	init_pvar(P_OLD,NULL,'b',0.0);
	P_IN = (double *) nalloc(sizeof(double)*(numBasn+1),"P_IN");
	init_pvar(P_IN,NULL,'b',0.0);
	P_OUT = (double *) nalloc(sizeof(double)*(numBasn+1),"P_OUT");
	init_pvar(P_OUT,NULL,'b',0.0);
	P_ERR = (double *) nalloc(sizeof(double)*(numBasn+1),"P_ERR");
	init_pvar(P_ERR,NULL,'b',0.0);
	P_ERR_CUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_ERR_CUM");
	init_pvar(P_ERR_CUM,NULL,'b',0.0);

	P_IN_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_IN_SUM");
	init_pvar(P_IN_SUM,NULL,'b',0.0);
	P_OUT_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_OUT_SUM");
	init_pvar(P_OUT_SUM,NULL,'b',0.0);
	P_IN_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"P_IN_AVG");
	init_pvar(P_IN_AVG,NULL,'b',0.0);
	P_OUT_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"P_OUT_AVG");
	init_pvar(P_OUT_AVG,NULL,'b',0.0);

	TOT_P_CAN = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_P_CAN");
	init_pvar(TOT_P_CAN,NULL,'b',0.0);

	P_IN_STR = (double *) nalloc(sizeof(double)*(numBasn+1),"P_IN_STR");
	init_pvar(P_IN_STR,NULL,'b',0.0);
	P_IN_OVL = (double *) nalloc(sizeof(double)*(numBasn+1),"P_IN_OVL");
	init_pvar(P_IN_OVL,NULL,'b',0.0);
	P_IN_SPG = (double *) nalloc(sizeof(double)*(numBasn+1),"P_IN_SPG");
	init_pvar(P_IN_SPG,NULL,'b',0.0);
	P_IN_GW = (double *) nalloc(sizeof(double)*(numBasn+1),"P_IN_GW");
	init_pvar(P_IN_GW,NULL,'b',0.0);

	P_OUT_STR = (double *) nalloc(sizeof(double)*(numBasn+1),"P_OUT_STR");
	init_pvar(P_OUT_STR,NULL,'b',0.0);
	P_OUT_OVL = (double *) nalloc(sizeof(double)*(numBasn+1),"P_OUT_OVL");
	init_pvar(P_OUT_OVL,NULL,'b',0.0);
	P_OUT_SPG = (double *) nalloc(sizeof(double)*(numBasn+1),"P_OUT_SPG");
	init_pvar(P_OUT_SPG,NULL,'b',0.0);
	P_OUT_GW = (double *) nalloc(sizeof(double)*(numBasn+1),"P_OUT_GW");
	init_pvar(P_OUT_GW,NULL,'b',0.0);

/* phosph mass bal vars for live fraction */
	P_LIVE_CELL = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE_CELL");
	init_pvar(P_LIVE_CELL,NULL,'b',0.0);
	P_MAC = (double *) nalloc(sizeof(double)*(numBasn+1),"P_MAC");
	init_pvar(P_MAC,NULL,'b',0.0);
	P_Calg = (double *) nalloc(sizeof(double)*(numBasn+1),"P_Calg");
	init_pvar(P_Calg,NULL,'b',0.0);
	P_NCalg = (double *) nalloc(sizeof(double)*(numBasn+1),"P_NCalg");
	init_pvar(P_NCalg,NULL,'b',0.0);
	Calg_GPP = (double *) nalloc(sizeof(double)*(numBasn+1),"Calg_GPP");
	init_pvar(Calg_GPP,NULL,'b',0.0);
	NCalg_GPP = (double *) nalloc(sizeof(double)*(numBasn+1),"NCalg_GPP");
	init_pvar(NCalg_GPP,NULL,'b',0.0);
	Calg_mort = (double *) nalloc(sizeof(double)*(numBasn+1),"Calg_mort");
	init_pvar(Calg_mort,NULL,'b',0.0);
	NCalg_mort = (double *) nalloc(sizeof(double)*(numBasn+1),"NCalg_mort");
	init_pvar(NCalg_mort,NULL,'b',0.0);

	mac_NPP = (double *) nalloc(sizeof(double)*(numBasn+1),"mac_NPP");
	init_pvar(mac_NPP,NULL,'b',0.0);
	mac_mort = (double *) nalloc(sizeof(double)*(numBasn+1),"mac_mort");
	init_pvar(mac_mort,NULL,'b',0.0);

	P_LIVE = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE");
	init_pvar(P_LIVE,NULL,'b',0.0);

	P_LIVE_OLD = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE_OLD");
	init_pvar(P_LIVE_OLD,NULL,'b',0.0);
	P_LIVE_IN = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE_IN");
	init_pvar(P_LIVE_IN,NULL,'b',0.0);
	P_LIVE_OUT = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE_OUT");
	init_pvar(P_LIVE_OUT,NULL,'b',0.0);
	P_LIVE_ERR = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE_ERR");
	init_pvar(P_LIVE_ERR,NULL,'b',0.0);
	P_LIVE_ERR_CUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE_ERR_CUM");
	init_pvar(P_LIVE_ERR_CUM,NULL,'b',0.0);

	P_LIVE_IN_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE_IN_SUM");
	init_pvar(P_LIVE_IN_SUM,NULL,'b',0.0);
	P_LIVE_OUT_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE_OUT_SUM");
	init_pvar(P_LIVE_OUT_SUM,NULL,'b',0.0);
	P_LIVE_IN_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE_IN_AVG");
	init_pvar(P_LIVE_IN_AVG,NULL,'b',0.0);
	P_LIVE_OUT_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"P_LIVE_OUT_AVG");
	init_pvar(P_LIVE_OUT_AVG,NULL,'b',0.0);

/* phosph mass bal vars for dead fraction */
	P_DEAD_CELL = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD_CELL");
	init_pvar(P_DEAD_CELL,NULL,'b',0.0);
	dop_macIn = (double *) nalloc(sizeof(double)*(numBasn+1),"dop_macIn");
	init_pvar(dop_macIn,NULL,'b',0.0);
	dop_sorbIn = (double *) nalloc(sizeof(double)*(numBasn+1),"dop_sorbIn");
	init_pvar(dop_sorbIn,NULL,'b',0.0);
	dop_decomp = (double *) nalloc(sizeof(double)*(numBasn+1),"dop_decomp");
	init_pvar(dop_decomp,NULL,'b',0.0);
	dop_desorb = (double *) nalloc(sizeof(double)*(numBasn+1),"dop_desorb");
	init_pvar(dop_desorb,NULL,'b',0.0);
	floc_decomp = (double *) nalloc(sizeof(double)*(numBasn+1),"floc_decomp");
	init_pvar(floc_decomp,NULL,'b',0.0);
	floc_In = (double *) nalloc(sizeof(double)*(numBasn+1),"floc_In");
	init_pvar(floc_In,NULL,'b',0.0);

	P_DEAD = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD");
	init_pvar(P_DEAD,NULL,'b',0.0);

	P_DEAD_OLD = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD_OLD");
	init_pvar(P_DEAD_OLD,NULL,'b',0.0);
	P_DEAD_IN = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD_IN");
	init_pvar(P_DEAD_IN,NULL,'b',0.0);
	P_DEAD_OUT = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD_OUT");
	init_pvar(P_DEAD_OUT,NULL,'b',0.0);
	P_DEAD_ERR = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD_ERR");
	init_pvar(P_DEAD_ERR,NULL,'b',0.0);
	P_DEAD_ERR_CUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD_ERR_CUM");
	init_pvar(P_DEAD_ERR_CUM,NULL,'b',0.0);

	P_DEAD_IN_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD_IN_SUM");
	init_pvar(P_DEAD_IN_SUM,NULL,'b',0.0);
	P_DEAD_OUT_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD_OUT_SUM");
	init_pvar(P_DEAD_OUT_SUM,NULL,'b',0.0);
	P_DEAD_IN_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD_IN_AVG");
	init_pvar(P_DEAD_IN_AVG,NULL,'b',0.0);
	P_DEAD_OUT_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"P_DEAD_OUT_AVG");
	init_pvar(P_DEAD_OUT_AVG,NULL,'b',0.0);

/* phosph mass bal vars for water-borne fraction */
	P_WAT_CELL = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT_CELL");
	init_pvar(P_WAT_CELL,NULL,'b',0.0);
	wat_sfMiner = (double *) nalloc(sizeof(double)*(numBasn+1),"wat_sfMiner");
	init_pvar(wat_sfMiner,NULL,'b',0.0);
	wat_sedMiner = (double *) nalloc(sizeof(double)*(numBasn+1),"wat_sedMiner");
	init_pvar(wat_sedMiner,NULL,'b',0.0);
	wat_sfUpt = (double *) nalloc(sizeof(double)*(numBasn+1),"wat_sfUpt");
	init_pvar(wat_sfUpt,NULL,'b',0.0);
	P_settl = (double *) nalloc(sizeof(double)*(numBasn+1),"P_settl");
	init_pvar(P_settl,NULL,'b',0.0);
	wat_sedUpt = (double *) nalloc(sizeof(double)*(numBasn+1),"wat_sedUpt");
	init_pvar(wat_sedUpt,NULL,'b',0.0);

	P_WAT = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT");
	init_pvar(P_WAT,NULL,'b',0.0);

	P_WAT_OLD = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT_OLD");
	init_pvar(P_WAT_OLD,NULL,'b',0.0);
	P_WAT_IN = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT_IN");
	init_pvar(P_WAT_IN,NULL,'b',0.0);
	P_WAT_OUT = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT_OUT");
	init_pvar(P_WAT_OUT,NULL,'b',0.0);
	P_WAT_ERR = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT_ERR");
	init_pvar(P_WAT_ERR,NULL,'b',0.0);
	P_WAT_ERR_CUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT_ERR_CUM");
	init_pvar(P_WAT_ERR_CUM,NULL,'b',0.0);

	P_WAT_IN_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT_IN_SUM");
	init_pvar(P_WAT_IN_SUM,NULL,'b',0.0);
	P_WAT_OUT_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT_OUT_SUM");
	init_pvar(P_WAT_OUT_SUM,NULL,'b',0.0);
	P_WAT_IN_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT_IN_AVG");
	init_pvar(P_WAT_IN_AVG,NULL,'b',0.0);
	P_WAT_OUT_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"P_WAT_OUT_AVG");
	init_pvar(P_WAT_OUT_AVG,NULL,'b',0.0);

/* salt mass bal vars */
	TOT_S_CELL = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_S_CELL");
	init_pvar(TOT_S_CELL,NULL,'b',0.0);

	TOT_S = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_S");
	init_pvar(TOT_S,NULL,'b',0.0);
	TOT_S_OLD = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_S_OLD");
	init_pvar(TOT_S_OLD,NULL,'b',0.0);
	S_IN = (double *) nalloc(sizeof(double)*(numBasn+1),"S_IN");
	init_pvar(S_IN,NULL,'b',0.0);
	S_OUT = (double *) nalloc(sizeof(double)*(numBasn+1),"S_OUT");
	init_pvar(S_OUT,NULL,'b',0.0);
	TOT_S_ERR = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_S_ERR");
	init_pvar(TOT_S_ERR,NULL,'b',0.0);
	S_ERR_CUM = (double *) nalloc(sizeof(double)*(numBasn+1),"S_ERR_CUM");
	init_pvar(S_ERR_CUM,NULL,'b',0.0);

	S_IN_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"S_IN_SUM");
	init_pvar(S_IN_SUM,NULL,'b',0.0);
	S_OUT_SUM = (double *) nalloc(sizeof(double)*(numBasn+1),"S_OUT_SUM");
	init_pvar(S_OUT_SUM,NULL,'b',0.0);
	S_IN_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"S_IN_AVG");
	init_pvar(S_IN_AVG,NULL,'b',0.0);
	S_OUT_AVG = (double *) nalloc(sizeof(double)*(numBasn+1),"S_OUT_AVG");
	init_pvar(S_OUT_AVG,NULL,'b',0.0);

	TOT_S_CAN = (double *) nalloc(sizeof(double)*(numBasn+1),"TOT_S_CAN");
	init_pvar(TOT_S_CAN,NULL,'b',0.0);

	S_IN_STR = (double *) nalloc(sizeof(double)*(numBasn+1),"S_IN_STR");
	init_pvar(S_IN_STR,NULL,'b',0.0);
	S_IN_OVL = (double *) nalloc(sizeof(double)*(numBasn+1),"S_IN_OVL");
	init_pvar(S_IN_OVL,NULL,'b',0.0);
	S_IN_SPG = (double *) nalloc(sizeof(double)*(numBasn+1),"S_IN_SPG");
	init_pvar(S_IN_SPG,NULL,'b',0.0);
	S_IN_GW = (double *) nalloc(sizeof(double)*(numBasn+1),"S_IN_GW");
	init_pvar(S_IN_GW,NULL,'b',0.0);

	S_OUT_STR = (double *) nalloc(sizeof(double)*(numBasn+1),"S_OUT_STR");
	init_pvar(S_OUT_STR,NULL,'b',0.0);
	S_OUT_OVL = (double *) nalloc(sizeof(double)*(numBasn+1),"S_OUT_OVL");
	init_pvar(S_OUT_OVL,NULL,'b',0.0);
	S_OUT_SPG = (double *) nalloc(sizeof(double)*(numBasn+1),"S_OUT_SPG");
	init_pvar(S_OUT_SPG,NULL,'b',0.0);
	S_OUT_GW = (double *) nalloc(sizeof(double)*(numBasn+1),"S_OUT_GW");
	init_pvar(S_OUT_GW,NULL,'b',0.0);

/* map arrays with summary stats */
	HydPerAnn = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"HydPerAnn");
	init_pvar(HydPerAnn,NULL,'f',0.0);
	SfWatAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SfWatAvg");
	init_pvar(SfWatAvg,NULL,'f',0.0);
	TotHeadAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TotHeadAvg");
	init_pvar(TotHeadAvg,NULL,'f',0.0);
	UnsatZavg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UnsatZavg");
	init_pvar(UnsatZavg,NULL,'f',0.0);

	RainAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"RainAvg");
	init_pvar(RainAvg,NULL,'f',0.0);
	ETAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"ETAvg"); 
	init_pvar(ETAvg,NULL,'f',0.0);
	EvapAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"EvapAvg");
	init_pvar(EvapAvg,NULL,'f',0.0);
	TranspAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TranspAvg");
	init_pvar(TranspAvg,NULL,'f',0.0);
	UnsatMoistAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"UnsatMoistAvg");
	init_pvar(UnsatMoistAvg,NULL,'f',0.0);

	TPSfWatAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TPSfWatAvg");
	init_pvar(TPSfWatAvg,NULL,'f',0.0);
	TPSfUptAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TPSfUptAvg");
	init_pvar(TPSfUptAvg,NULL,'f',0.0);
	TPSfMinAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TPSfMinAvg");
	init_pvar(TPSfMinAvg,NULL,'f',0.0);
	TP_settlAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TP_settlAvg");
	init_pvar(TP_settlAvg,NULL,'f',0.0);
	TPSedWatAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TPSedWatAvg");
	init_pvar(TPSedWatAvg,NULL,'f',0.0);
	TPSedUptAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TPSedUptAvg");
	init_pvar(TPSedUptAvg,NULL,'f',0.0);
	TPSedMinAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TPSedMinAvg");
	init_pvar(TPSedMinAvg,NULL,'f',0.0);
	TPSorbAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TPSorbAvg");
	init_pvar(TPSorbAvg,NULL,'f',0.0);
	TPtoVOLAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TPtoVOLAvg");
	init_pvar(TPtoVOLAvg,NULL,'f',0.0);
	TPtoSOILAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"TPtoSOILAvg");
	init_pvar(TPtoSOILAvg,NULL,'f',0.0);

	SaltSfAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SaltSfAvg");
	init_pvar(SaltSfAvg,NULL,'f',0.0);
	SaltSedAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SaltSedAvg");
	init_pvar(SaltSedAvg,NULL,'f',0.0);

	Floc_fr_phBioAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"Floc_fr_phBioAvg");
	init_pvar(Floc_fr_phBioAvg,NULL,'f',0.0);

	NC_PeriAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_PeriAvg");
	init_pvar(NC_PeriAvg,NULL,'f',0.0);
	NC_Peri_nppAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_Peri_nppAvg");
	init_pvar(NC_Peri_nppAvg,NULL,'f',0.0);
	NC_Peri_mortAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_Peri_mortAvg");
	init_pvar(NC_Peri_mortAvg,NULL,'f',0.0);
	NC_PeriRespAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_PeriRespAvg");
	init_pvar(NC_PeriRespAvg,NULL,'f',0.0);
	NC_PeriNutCFAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_PeriNutCFAvg");
	init_pvar(NC_PeriNutCFAvg,NULL,'f',0.0);
	NC_Peri_PCAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"NC_Peri_PCAvg");
	init_pvar(NC_Peri_PCAvg,NULL,'f',0.0);
	C_PeriAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_PeriAvg");
	init_pvar(C_PeriAvg,NULL,'f',0.0);
	C_Peri_nppAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_Peri_nppAvg");
	init_pvar(C_Peri_nppAvg,NULL,'f',0.0);
	C_Peri_mortAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_Peri_mortAvg");
	init_pvar(C_Peri_mortAvg,NULL,'f',0.0);
	C_PeriRespAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_PeriRespAvg");
	init_pvar(C_PeriRespAvg,NULL,'f',0.0);
	C_PeriNutCFAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_PeriNutCFAvg");
	init_pvar(C_PeriNutCFAvg,NULL,'f',0.0);
	C_Peri_PCAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"C_Peri_PCAvg");
	init_pvar(C_Peri_PCAvg,NULL,'f',0.0);
	PeriAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PeriAvg");
	init_pvar(PeriAvg,NULL,'f',0.0);
	PeriLiteCFAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"PeriLiteCFAvg");
	init_pvar(PeriLiteCFAvg,NULL,'f',0.0);

	Mac_nphBioAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"Mac_nphBioAvg");
	init_pvar(Mac_nphBioAvg,NULL,'f',0.0);
	Mac_phBioAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"Mac_phBioAvg");
	init_pvar(Mac_phBioAvg,NULL,'f',0.0);
	Mac_totBioAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"Mac_totBioAvg");
	init_pvar(Mac_totBioAvg,NULL,'f',0.0);
	Mac_nppAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"Mac_nppAvg");
	init_pvar(Mac_nppAvg,NULL,'f',0.0);
	Mac_nphMortAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"Mac_nphMortAvg");
	init_pvar(Mac_nphMortAvg,NULL,'f',0.0);
	Mac_phMortAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"Mac_phMortAvg");
	init_pvar(Mac_phMortAvg,NULL,'f',0.0);
	LAI_effAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"LAI_effAvg");
	init_pvar(LAI_effAvg,NULL,'f',0.0);
	Manning_nAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"Manning_nAvg");
	init_pvar(Manning_nAvg,NULL,'f',0.0);
	MacNutCfAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MacNutCfAvg");
	init_pvar(MacNutCfAvg,NULL,'f',0.0);
	MacWatCfAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MacWatCfAvg");
	init_pvar(MacWatCfAvg,NULL,'f',0.0);
	mac_nph_PCAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"mac_nph_PCAvg");
	init_pvar(mac_nph_PCAvg,NULL,'f',0.0);
	mac_ph_PCAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"mac_ph_PCAvg");
	init_pvar(mac_ph_PCAvg,NULL,'f',0.0);

	SedElevAvg = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"SedElevAvg");
	init_pvar(SedElevAvg,NULL,'f',0.0);

} /* end of alloc_mem_stats */

/********************************/
