/***************************************************************************
                          cell.cpp  -  description
                             -------------------
    begin                : Mon Feb 24 2003
    copyright            : (C) 2003 by Beheen Trimble
    email                : btrimble@sfwmd.gov
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cell.h"
#include "point.h"
#include "consolemsg.h"

#include <math.h>
#include <map>

Cell::Cell() : console(new ConsoleMsg("Cell")) {
  LowerLeft = 0.0,0.0;
  LowerRight = 0.0,0.0;
  UpperLeft = 0.0,0.0;
  UpperRight = 0.0,0.0;
  
}

Cell::Cell(int id) : console(new ConsoleMsg("Cell")) {
  ID = id;
  MappedID = -1;   
}

Cell::Cell(int id, Point p1,Point p2, Point p3, Point p4) :
          LowerLeft(p1), LowerRight(p2),
          UpperLeft(p3), UpperRight(p4) {
  ID = id;
  console = new ConsoleMsg("Cell");
}

Cell::~Cell() {
  #ifdef DEBUG
  cout << "Cell ==> Deleting Cell" << endl;
  #endif
  
  delete console;
  console = NULL;
 
}

/* This method saves the coordinates of 4 corner
 * of a cell in its data members LowerLeft, LowerRight
 * UpperLeft, and UpperRight
 */
void Cell::setCoords(Point& ll,Point& lr, Point& ur, Point& ul) {
  LowerLeft = ll;
  LowerRight = lr;
  UpperLeft = ul ;
  UpperRight = ur;
}


Point Cell::getLowerLeft() { return LowerLeft; }

Point Cell::getLowerRight() { return LowerRight; }

Point Cell::getUpperLeft() { return UpperLeft; }

Point Cell::getUpperRight() { return UpperRight; }


// although is square grid, assum is a rectangle 
double Cell::calcArea() {

  double a = LowerRight.getX() - LowerLeft.getX();  // same y
  double b = UpperRight.getY() - LowerRight.getY(); // same x

  Area = a * b;

  return Area;
}


void Cell::print() {
  int status = console->xstatus;
  
  console->Msg("Cell ==> Cell Number : ", ID, status);
  cout << "Row : " <<  Row << "  " << "Col : "  << Col << endl;
  cout << "================================================" << endl;
  //console->Msg("Cell ==> Centroid    : ", "", status);
  Centroid.print();

  for(unsigned int i=0; i<pointList.size(); i++) {
    pointList[i]->print();
  }
}

  

/*
  Points in convex polygons

  A convex polygon is just one without any indentations. It can also be defined formally
  having the property that any two points inside the polygon can be connected by a line
  segment that doesn't cross the polygons.

  point-in-polygon:

  Let P(i) be the leftmost point of the polygon, and P(j) be the rightmost point.
  These two points divide the polygon into two parts, the upper chain and the lower
  chain. Any line or ray crosses the polygon at most twice, and any vertical line
  or ray crosses each chain at most once.

  So a lot of the work of the previous point-in-polygon algorithm is wasted --
  we go through each segment checking whether it gives a crossing, but the answer
  will be yes in at most two cases.

  How can we find these two cases more quickly? Binary search.
  Just search for x in the first coordinates of points in the two chains.
  If it is in the chain, you have found a crossing through a vertex
  (and you don't have to be as careful to tell what kind of crossing, either).

  If x is not the coordinate of a vertex in the chain, the two nearest values to it
  tell you which segment the ray from (x,y) might cross.
  So we can test whether a point is in a convex polygon in time O(log n).
*/
int Cell::pointInConvexPolygon(const Point pt) {

  int status = console->xstatus;

  int n = 1;              // n the number of points in cell we care about
  int crossings = 0, on_boundary = 1, inside = 2, outside = 3;
  double px0,py0,px1,py1; // polygon's coordinates of point.
  double point_onthe_line;
  double x,y;             // point in polygon
  double ycrossing;

  x   = pt.getX();
  y   = pt.getY();

  //cout << "Cell ==> Is " << pt << endl;
  
  for(int i = 0; i < n; i++) {
    
    px0 = this->pointList[i]->getX();   // cell numbering is: 3---2
    py0 = this->pointList[i]->getY();   //                    0---1
                               // we just check the line between 0 and 1
    px1 = this->pointList[i+2]->getX();
    py1 = this->pointList[i+2]->getY();
    
    //cout << "Cell ==>  " << px0 << "," << py0 << "  " << px1 << "," << py1 << endl;
    /*
      If this is to be between (x0,y0) and (x1,y1) then x should be between x0 and x1;
      either (x0<x and x<x1) or (x0>x and x>x1). We can write this more succinctly as
      (x0-x)(x1-x)<0 but this might actually slower since it involves a high-precision
      multiplication.
     */

    if((px0 <= x && x < px1) || (px0 > x && x >= px1)) {
        
      /*
        Now suppose x is between x0 and x1 where is the crossing point?
        (tx0 + (1-t)x1, ty0 + (1-t)y1) where different values of t give different
        points on the line. To find the second coordinate of the crossing, let's use
        the known value of the first coordinate to solve for t:
        x = tx0 + (1-t)x1
        t = (x - x1) / (x0 - x1)
        crossing = (x, ty0 + (1-t)y1)
       */
      point_onthe_line = (x - px1) / (px0 - px1);
      ycrossing = (point_onthe_line * py0) + ((1 - point_onthe_line) * py1);

      /*
        Note that we can also tell if (x,y) is exactly on the line segment by testing
        whether y=crossing.
        

        Here's a little problem: this formula might involve a division by zero.
        But in that case x1=x2 so x couldn't be between the two. As long as we only
        compute t when x is between x1 and x2, we're safe.
       */
      if(y == ycrossing) {
        #ifdef DEBUG
        console->DebugMsg("Cell ==> Boundary point ",on_boundary,status);
        #endif
        return (on_boundary);
        
      }
      else if (y > ycrossing) {
        crossings++;
        //console->DebugMsg("Cell ==> Crossing ",crossings,status);
      }
      else if ((py0 <= y && y < py1) || (py0 > y && y >= py1)) {
        #ifdef DEBUG
        cout << "Cell ==> " << ID << " " << px0 << "," << py0 << "  " << px1 << "," << py1 << endl;
        #endif
        crossings++;
      }
    }  //end of if
    /*
      What if the ray from (x,y) passes exactly through a vertex of P?
      Sometimes this should count as a crossing, but sometimes the ray only "grazes"
      P and shouldn't count as a crossing: move the points slightly so these special
      cases don't happen. For instance, we could move the point (x,y) just a tiny
      amount to the right; this won't change whether it's inside or outside the polygon
      but will change whether the ray crosses through any vertices. In fact, we can
      perform this perturbation only in our minds, and let it guide us in thinking
      about solving the problem, without actually moving any points.
      Suppose the ray would cross a vertex before it was perturbed. After the
      perturbation, which edges would it cross? Just the ones that go to the right of
      the vertex. We can test the two rays out of the vertex and see which of them go
      rightwards, and adjust the count accordingly.
     */
    if(px0 == x && py0 <= y) {
      if(py0 == y) {
        #ifdef DEBUG
        console->DebugMsg("Cell ==> Boundary point ",on_boundary,status);
        #endif
        return (on_boundary);

      }

      if(px1 == x) {
        if((py0 <= y && y <= py1) || (py0 >= y && y >= py1)) {
          #ifdef DEBUG
          console->DebugMsg("Cell ==> Boundary point ",on_boundary,status);
          #endif
          return (on_boundary);

        }
      }
      else if(px1 > x)
        crossings++;

    } // end of if

  } //end of for

  if((crossings % 2) == 1) {
    #ifdef DEBUG
    console->DebugMsg("Cell ==> Inside point ",inside,status);
    #endif
    return (inside);

  }
  else {
    //console->DebugMsg("Cell ==> Outside point ",outside,status);
    return (outside);
  }
}

int Cell::getRow() { return Row; }

int Cell::getCol() { return Col;}

void Cell::setRow(int r) { Row = r ;}

void Cell::setCol(int c) {Col = c; }

int Cell::getMappedID() {
  return MappedID;
}


void Cell::setMappedID(int id) {
    MappedID = id;
}


int Cell::pointInCell(const Point pt) {

  int n = 1;              // n the number of points in cell we care about
  int inside = 2, outside = 3;
  double px0,py0,px1,py1; // polygon's coordinates of point.
  double x,y;             // point in polygon

  
  x   = pt.getX();
  y   = pt.getY();

  //cout << "Cell ==> Is " << pt << endl;

  for(int i = 0; i < n; i++) {

    px0 = this->pointList[i]->getX();   // cell numbering is: 3---2
    py0 = this->pointList[i]->getY();   //                    0---1
                               // we just check the line between 0 and 1
    px1 = this->pointList[i+2]->getX();
    py1 = this->pointList[i+2]->getY();

    //cout << "Cell ==>  " << px0 << "," << py0 << "  " << px1 << "," << py1 << endl;
    /*
      If this is to be between (x0,y0) and (x1,y1) then x should be between x0 and x1;
      either (x0<x and x<x1) or (x0>x and x>x1). We can write this more succinctly as
      (x0-x)(x1-x)<0 but this might actually slower since it involves a high-precision
      multiplication.
     */

    // in our case px0 is always < px1
    if((px0 < x && x <= px1) || (px0 <= x && x < px1) ) {
      if ( (py0 < y && y <= py1) || (py0 <= y && y < py1)) {
        /*cout << "Cell ==> " << ID << " " << px0 << "," << py0 << "  "
                                         << x << "," << y << " "
                                         << px1 << "," << py1 << endl;  */
        return inside;
      }
    }  //end of if
  }

  return outside;
}
