/***************************************************************************
                          grid.cpp  -  description
                             -------------------
    begin                : Thu Feb 20 2003
    copyright            : (C) 2003 by Beheen Trimble
    email                : btrimble@sfwmd.gov
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

using namespace std;
 
#include "grid.h"
#include <stdio.h>
#include <fstream>

// to be initialized!
Grid::Grid() : console(new ConsoleMsg("Grid")) {
  LowerLeftCorner = new Point(0.0,0.0);
  UpperRightCorner = new Point(0.0,0.0);
}

Grid::Grid(string n) : console(new ConsoleMsg(n)) {
  LowerLeftCorner = new Point(0.0,0.0);
  UpperRightCorner = new Point(0.0,0.0);
}


Grid::~Grid(){
  #ifdef DEBUG
  cout << "Deleting Grid" << endl;
  #endif

  delete console;
  console = NULL;

  delete LowerLeftCorner;
  delete UpperRightCorner;
  LowerLeftCorner = NULL;
  UpperRightCorner = NULL;

}

//void Grid::setFileFlag(bool v) { FileFlag = v; }
void Grid::setName(string n) { GridName = n;}
void Grid::setX_Origin(double x) { X_Origin = x; }
void Grid::setY_Origin(double y) { Y_Origin = y; }
void Grid::setX_Direction(string d) {X_Direction = d; }
void Grid::setY_Direction(string d) {Y_Direction = d; }
void Grid::setNumRow(int n) {NumRow = n;}
void Grid::setNumCol(int n) {NumCol = n;}
void Grid::setTimeStep(float t) {TimeStep = t;}
void Grid::setNumbering(int i) { Numbering = i; }

/* After the grid object is populated and its data members
 * are set, this method creates the lower left and upper
 * right/left corners of the grid.
 * Keep ll and ur as reference always. However when user
 * enters the origin, if the grid is constructed from
 * top to down, the origin is not the lower left anymore. 
 */
void Grid::setCorners() {


  double xo = getX_Origin();
  double yo = getY_Origin();

  // make sure to calculate ll and ur
  if(X_Direction == "RIGHT" && Y_Direction == "UP") {
    // cell numbering same for all direction
    // 3    2
    // 0    1
    double xll = xo;
    double yll = yo;
    setLowerLeftCorner(xll,yll);
    double xur = xll + getTimeStep() * getNumCol();
    double yur = yll + getTimeStep() * getNumRow();
    setUpperRightCorner(xur,yur);
    double xlr = xur;
    double ylr = yll;
    setLowerRightCorner(xlr,ylr);
    double xul = xll;
    double yul = yur;
    setUpperLeftCorner(xul,yul);
  }
  
  if(X_Direction == "RIGHT" && Y_Direction == "DOWN") {
    // cell numbering
    // 3    2
    // 0    1
    double xul = xo;
    double yul = yo;
    setUpperLeftCorner(xul,yul);
    double xur = xul + getTimeStep() * getNumCol();
    double yur = yul ;
    setUpperRightCorner(xur,yur);
    double xlr = xur;
    double ylr = yul - getTimeStep() * getNumRow(); ;
    setLowerRightCorner(xlr,ylr);
    double xll = xul;
    double yll = ylr;
    setLowerLeftCorner(xll,yll);

  }
  
}

void Grid::setLowerLeftCorner(double x, double y) {
  LowerLeftCorner = new Point(x,y);
}

void Grid::setUpperRightCorner(double x, double y) {
  UpperRightCorner = new Point(x,y);
}

void Grid::setLowerRightCorner(double x, double y) {
  LowerRightCorner = new Point(x,y);
}

void Grid::setUpperLeftCorner(double x, double y) {
  UpperLeftCorner = new Point(x,y);
}

Point* Grid::getLowerLeftCorner() {return LowerLeftCorner;}

Point* Grid::getUpperRightCorner() {return UpperRightCorner;}

Point* Grid::getLowerRightCorner() {return LowerRightCorner;}

Point* Grid::getUpperLeftCorner() {return UpperLeftCorner;}

double Grid::getX_Origin() { return X_Origin; }

double Grid::getY_Origin() { return Y_Origin; }

string Grid::getX_Direction() {return X_Direction; }

string Grid::getY_Direction() { return Y_Direction; }

int Grid::getNumCol() { return NumCol;}

int Grid::getNumRow() { return NumRow; }

int Grid::getNumbering() { return Numbering; }

float Grid::getTimeStep() { return TimeStep;}

string Grid::getGridName() { return GridName; }

// I am sure we can do this method in one step!!
void Grid::drawCell(vector<int> ids, FILE* fptr) {
  
  // file must be opened and closed in calling method
  static char space = '+', endln = '\n', cross = 'x';
  //, star = '*';

  // create a complete grid with no end of line
  int pos = ftello(fptr);
  
  for(int i=0; i< this->NumRow ; ++i) {
    for(int j=0; j< this->NumCol; ++j) {
      fputc(space,fptr);
    }
    //fputc('\n',fptr); don't use it here
  }
  pos = ftello(fptr);
  fprintf(fptr,"\n");
  cout << pos << endl;
  // substitute the ids in their position in grid system
  fseeko(fptr,0,0);  //go to the beginning of the file
  
  for(unsigned int i = 0; i< ids.size(); ++i) {
   
    Cell* c = cellList[ids[i]];
    int row = c->getRow();
    int col = c->getCol();
    int pos = row * NumCol + col ;
   
    fseeko(fptr,pos,0);  /* 0 Seek from beginning of file.  */
    fputc(cross,fptr);
  }
 
  // put the end of line at the end of each col
  int endcolpos = this->NumCol-1;
  fseeko(fptr,0,0);  //go to the beginning of the file
  for(int i=0; i< this->NumRow ; ++i) {
    fseeko(fptr,endcolpos,1);  /* 1 Seek from current position.  */
    fputc(endln,fptr);
    /* 2 Seek from end of file.  */
  }            
}

/* This method creates NumCol*NumRow numbers of cells for
 * this grid and identify each cell by numbering according
 * to user defined numbering.
 * For example, If numbering is 0, the row and columns start
 * at 0 and the first cell id is 0. If numbering is 1, the
 * first row is row=1, col=1, and cell id = 1.
 * Other feature of a cell such as the area, the coordinates
 * for each corner, and the centroid of each cell is calculated.
 *
 */
int Grid::defineCells() {

  int statusx = console->xstatus;
  int statuse = console->estatus;
  int status = statusx;

  #ifdef DEBUG
  console->DebugMsg("Grid ==> Defining cells for ",this->getGridName(),statusx);
  #endif
  
  // check when the cell is not square, and is not
  // in lower left, or things of this nature!
  // Note: this has to be implemented when the grid
  // system is different with this prototype.
  //status = checkConstraint();

  // Create all the cells and their ids.
  for(int ii=Numbering; ii<(NumCol*NumRow+Numbering); ii++) {
    Cell* c = new Cell(ii);
    cellList.push_back(c); // the index of this is always 0
  }
  
  
  // calculate the coordinates of 4 corners of each cell
  // and the coordinate of its center
  Point *start0,centroid; //for all direction
  int r = 0;              // counter to go in different direction

  // calculate the area. Since this is a square grid
  // do it here and once. For rectangular cell must
  // do adjustment.
  double area = TimeStep * TimeStep;

  // Get the coordinate of the first cell on its origin
  // as defined by the user.
    
  start0 = new Point(X_Origin,Y_Origin);
  Cell *cell;

  for(unsigned int i = 0; i< cellList.size(); ) {
    for(int c=0; c<NumCol; ++c, ++i) {
      cell = cellList[i];

      cell->setCol(c+Numbering);
      cell->setRow(r+Numbering);

      // this method creates the corner of cells with respect
      // to the direction given by the user
      Point ll, lr, ur, ul;
      status = setVectorPoint(start0,ll,lr,ur,ul);  // vector here means direction
      if(status == statuse)
        return status; 

      // Note: points are numbered counter clock-wise(ll=0,lr=1,ur=2,ul=3)
      // The order affects the centroid method
      cell->setCoords(ll,lr,ur,ul);
    
      // It is handy to keep a list of
      // 4 corners and the centroid points of each cell
      vector<Point*> pntlist;
      Point *p = new Point(ll);
      pntlist.push_back(p);
      p = new Point(lr);
      pntlist.push_back(p);
      p = new Point(ur);
      pntlist.push_back(p);
      p = new Point(ul);
      pntlist.push_back(p);

      cell->setPointList(pntlist);

      // For other shape cells, area must be
      // calculated somewhere around here!
      // double area = cell->calcArea();

      // given the 4 corners of a cell and the area,
      // calculate the centroid.
      cell->centroid(pntlist,centroid, &area);
      // cell->print();

      // reset and calculate the coord of the next cell
      // at the end of each row, this is the coord of the
      // last cell that must be adjusted for the next row,
      // because the y is different. start0 is adjusted to
      // the origin of the next cell
      resetVectorPoint(start0, ll, lr, ur, ul);
      
    }  //end of cells of one row

    // process the cells for the next row, upward.
    // the coordinate of the lower right of the
    // first cell of the previous row is the coord
    // of the first cell of the next row.
    //
  
    int index = r * NumCol;                 // The index of the first cell
    Cell* cellr = cellList[index];          // of the previous row in the
                                            // cellList.

    // start0 is the origin of the next
    // cell in the next row
    setVectorCell(start0,cellr);
    
    r++;   // next row index

    // terminate when
    if((r - Numbering)==NumRow)
      break;

  } //end of col

  delete start0;

  //else { fixConstraint(); }

  return status;
}

/* this method finds the origin of the next first cell
 * in the next row relative to the cell above or below
 * it, dependes on the direction
 */
void Grid::setVectorCell(Point*& start0, Cell* cellr) {

  // upper left coord of the first cell of the previous
  // row is the lower left of the first cell of next row
  if(X_Direction == "RIGHT" && Y_Direction == "UP") 
    start0 = new Point(cellr->getUpperLeft());
  else if(X_Direction == "RIGHT" && Y_Direction == "DOWN")
    start0 = new Point(cellr->getLowerLeft());
}

        
/* this method moves from the previous cell to the next
 * based on the direction
 */
void Grid::resetVectorPoint(Point*& start0, Point& ll, Point& lr, Point& ur, Point& ul) {    

  if(X_Direction == "RIGHT" && Y_Direction == "UP") {
    // same as (lr.getX(),lr.getY())
    start0 = new Point(ll.getX() + TimeStep,ll.getY());
  }
  else if(X_Direction == "RIGHT" && Y_Direction == "DOWN")
    start0 = new Point(ul.getX() + TimeStep, ul.getY());
}

      
/* this method construct one cell with respect to
 * the direction defined in the configuration file
 * by the user.
 */
int Grid::setVectorPoint(Point* start0, Point& ll, Point& lr, Point& ur, Point& ul) {

  int statusx = console->xstatus;
  int statuse = console->estatus;
  
  // construct the coord of origin, start0 is memory
  if(X_Direction == "RIGHT" && Y_Direction == "UP") {
    Point LL(start0->getX(),start0->getY());
    ll = LL;
    
    Point LR(ll.getX() + TimeStep,ll.getY());
    lr = LR;
    
    Point UR(lr.getX(),lr.getY()+TimeStep);
    ur = UR;
    
    Point UL(ll.getX(),ur.getY());
    ul = UL;

  }

 
  else if(X_Direction == "RIGHT" && Y_Direction == "DOWN") {
    Point UL(start0->getX(),start0->getY());
    ul = UL;
    
    Point UR(ul.getX()+TimeStep,ul.getY());
    ur = UR;
    
    Point LR(ur.getX(),ur.getY()-TimeStep);
    lr = LR;
    
    Point LL(ul.getX(),lr.getY());
    ll = LL;
  }

  else {
    console->ErrMsg("Grid ==> erro in grid direction ", "check the config file.", statuse);
    return statuse;
  }

  return statusx;
}

void Grid::print() {
  for(unsigned int i = 0; i< cellList.size(); i++) {
    cellList[i]->print();
  }
}


void Grid::printCorners() {
  cout << "Grid ==> Lower left coordinates: " << endl;
  LowerLeftCorner->print();
  cout << "Grid ==> Lower right coordinates: " << endl;
  LowerRightCorner->print();
  cout << "Grid ==> Upper right coordinates: " << endl;
  UpperRightCorner->print();
  cout << "Grid ==> Upper left coordinates: " << endl;
  UpperLeftCorner->print();
}


vector<Cell*>& Grid::getCellList() { return cellList; }

int Grid::getCellNumbers() {
  
  vector<Cell*>& grCells = getCellList();
  return grCells.size(); 
}


Cell* Grid::findCellByID(int id) {

  Cell* c = NULL;
  return(c=cellList[id-Numbering]);
}


Cell* Grid::findCellByRowCol(int row, int col) {
  for(unsigned int i=0; i<cellList.size(); ++i) {
    Cell* c = cellList[i];
    if(c->getRow() == row) {
      if(c->getCol() == col) {
        return c;
      }
    }
  }
  return NULL;
}

/* given a point, finds if this point belongs
   to a cell. When found, returns the cell ID.
 */
int Grid::findCellNumber(Point& p) {

  int status = console->estatus;
  int inside = 2;
  
  vector<Cell*> celllist = this->getCellList();
  Cell* c;

  // processing the row from zero row not row = 1
  // if cause problem must change in few places!
  for(unsigned int i=0; i<celllist.size(); i++) {
    c = celllist[i];
//    if(c->getID() == 2 || c->getID() == 36 || c->getID() == 2175 || c->getID() == 2209 ||
//       c->getID() == 1081 || c->getID() == 1188 || c->getID() == 19765 || c->getID() == 19872) {
    status = c->pointInCell(p);
//    }
    if(status == inside)
      break;
  }
  int cellNo = c->getID();
  return cellNo;
}

double Grid::getCellX(int cellNo) {
                                                                                          
  int status = console->estatus;
                                                                                          
  vector<Cell*> celllist = this->getCellList();
                                                                                          
  // processing the row from zero row not row = 1
  // if cause problem must change in few places!
  for(unsigned int i=0; i<celllist.size(); i++) {
    Cell* c = celllist[i];
                                                                                          
    if(cellNo == c->getID())  {
      Point p =  c->getLowerLeft();
      return p.getX();
    }
  }
  return (double)status;
}

double Grid::getCellY(int cellNo) {
  
  int status = console->estatus;
 
  vector<Cell*> celllist = this->getCellList();

  // processing the row from zero row not row = 1
  // if cause problem must change in few places!
  for(unsigned int i=0; i<celllist.size(); i++) {
    Cell* c = celllist[i];
  
    if(cellNo == c->getID())  {
      Point p =  c->getLowerLeft();
      return p.getY();
    }
  }
  return (double)status;
}


/* This method writes all the populated
 * information about the grid into the
 * file. The information are per cell.
 */
int Grid::writeGrid(string outfile,vector<string> header) {
       
  ofstream out;

  int status = console->xstatus;    // Error code variable

  // Open the output file for writting
  out.open(outfile.c_str(), ios::out);
  if (!out) {
    status = console->estatus;
    console->MsgFail("Grid","Open File");
    return status;
  }

  for(unsigned int i=0; i<header.size(); i++)
    out << header[i] << endl;

  /*
  out << "This is finer grid ids mapped to coarser grid ids" << endl;
  out << "Note that the mapped id = -1 means no cell intersect between 2 grids." << endl;
  out << "ID" << '\t' << "ROW" << '\t' << "COL" << '\t' << "MAPPED_ID" << endl;
  out << "==" << '\t' << "===" << '\t' << "===" << '\t' << "=========" << endl;
  */

  for(unsigned int i=0; i<cellList.size(); ++i) {
    out << cellList[i]->getID() << '\t'
               << cellList[i]->getRow() << '\t'
               << cellList[i]->getCol() << '\t'
               << cellList[i]->getMappedID()
               << endl;
  }

  out.close();
  
  return status;
}

