/***************************************************************************
                          polygon.cpp  -  description
                             -------------------
    begin                : Tue Feb 25 2003
    copyright            : (C) 2003 by Beheen Trimble
    email                : btrimble@sfwmd.gov
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "polygon.h"
#include "consolemsg.h"

Polygon::Polygon() : console(new ConsoleMsg("Polygon")) {}

Polygon::~Polygon() {
  #ifdef DEBUG
  cout << "Polygon ==> Deleting Polygon"<< endl;
  #endif
}

Point* Polygon::getCentroid() { return &Centroid; }

vector<Point*>& Polygon::getPointList(void) {return this->pointList; }


/* This method saves the coords of the corners of a
 * polygon grid in its data member pointList
 */
void Polygon::setPointList(vector<Point*> pntlist) {
  for(unsigned int i=0; i< pntlist.size(); i++)  {
    pointList.push_back(pntlist[i]);
  }
}

/*
  Testing if a point is in a polygon (the Jordan curve theorem) that any polygon cuts the
  plane into exactly two connected pieces: the inside and the outside. (The inside always
  has some finite size, while the outside contains points arbitrarily far from the polygon.)
  Given a point (x,y) and a polygon P (represented by its sequence of vertices),
  is (x,y) in P, on the boundary, or outside?

  Draw a ray (portion of a line extending infinitely in one direction) down
  (or in any other direction) from (x,y):
  Pseudo-code for this problem:

  int crossings = 0
  for (each line segment of the polygon)
    if (ray down from (x,y) crosses segment)
      crossings++;
    if (crossings is odd)
      return (inside);
    else return (outside);
*/
int Polygon::pointInPolygon(const Point pt) {

  int status = console->xstatus;
  
  vector<Point*>& ptlist = this->getPointList();
  int n = ptlist.size();  // n the number of points in the polygon
  int ii = 0;             // (i mod n)
  int crossings = 0, on_boundary = 1, inside = 2, outside = 3;
  double px0,py0,px1,py1; // polygon's coordinates of point (i mod n).
  double point_onthe_line;
  double x,y;             // point in polygon
  double ycrossing;

  x   = pt.getX();
  y   = pt.getY();

  //cout << "Polygon ==> Is " << pt << endl;
  //cout << "n is " << n << endl;                
  for(int i = 0; i < n; i++) {
    ii = i % n;

    px0 = ptlist[ii]->getX();
    py0 = ptlist[ii]->getY();
    
    if((ii+1)==n) {
      px1 = ptlist[0]->getX();
      py1 = ptlist[0]->getY();
    }
    else {
      px1 = ptlist[ii+1]->getX();
      py1 = ptlist[ii+1]->getY();
    }
    
    //cout << "Polygon ==> " << px0 << "," << py0 << "  " << px1 << "," << py1 << endl;
    /*
      If this is to be between (x1,y1) and (x2,y2) then x should be between x1 and x2;
      either (x1<x and x<x2) or (x1>x and x>x2). We can write this more succinctly as
      (x1-x)(x2-x)<0 but this might actually slower since it involves a high-precision
      multiplication. 
     */
     
    //if(px0 < x && x < px1 || px0 > x && x > px1) {
    if(px0 <= x && x < px1 || px0 >= x && x > px1) {
      point_onthe_line = (x - px1) / (px0 - px1);
      
      /*
        Now suppose x is between x1 and x2 where is the crossing point?
        (t x1 + (1-t)x2, t y1 + (1-t)y2) where different values of t give different
        points on the line. To find the second coordinate of the crossing, let's use
        the known value of the first coordinate to solve for t:
        x = t x1 + (1-t)x2
        t = (x - x2) / (x1 - x2)
       */
      ycrossing = (point_onthe_line * py0) + ((1 - point_onthe_line) * py1);

      /*
        Note that we can also tell if (x,y) is exactly on the line segment by testing
        whether y=crossing.
        crossing = (x, t y1 + (1-t)y2)

        Here's a little problem: this formula might involve a division by zero.
        But in that case x1=x2 so x couldn't be between the two. As long as we only
        compute t when x is between x1 and x2, we're safe.
       */
      if(y == ycrossing) {

        #ifdef DEBUG
        console->DebugMsg("Polygon ==> Boundary point ",on_boundary,status);
        #endif
        
        return (on_boundary);
      }
      else if (y > ycrossing) {
        crossings++;

        #ifdef DEBUG
        console->DebugMsg("Polygon ==> Crossing ",crossings,status);
        #endif
      }
    }  //end of if
    /*
      What if the ray from (x,y) passes exactly through a vertex of P?
      Sometimes this should count as a crossing, but sometimes the ray only "grazes"
      P and shouldn't count as a crossing: move the points slightly so these special
      cases don't happen. For instance, we could move the point (x,y) just a tiny
      amount to the right; this won't change whether it's inside or outside the polygon
      but will change whether the ray crosses through any vertices. In fact, we can
      perform this perturbation only in our minds, and let it guide us in thinking
      about solving the problem, without actually moving any points.
      Suppose the ray would cross a vertex before it was perturbed. After the
      perturbation, which edges would it cross? Just the ones that go to the right of
      the vertex. We can test the two rays out of the vertex and see which of them go
      rightwards, and adjust the count accordingly.
     */
    if(px0 == x && py0 <= y) {
      if(py0 == y) {
        #ifdef DEBUG
        console->DebugMsg("Polygon ==> Boundary point ",on_boundary,status);
        #endif
        
        return (on_boundary);
      }
      
      if(px1 == x) {
        if((py0 <= y && y <= py1) || (py0 >= y && y >= py1)) {
          #ifdef DEBUG
          console->DebugMsg("Polygon ==> Boundary point ",on_boundary,status);
          #endif
          
          return (on_boundary);
        }
      } else if(px1 > x) crossings++;
      double px_1 = ptlist[ii-1]->getX();
      if (px_1 > x) crossings++;
    } // end of if
    
  } //end of for

  if((crossings % 2) == 1) {
    #ifdef DEBUG
    console->DebugMsg("Polygon ==> Inside point ",inside,status);
    #endif
    return (inside);
  }
  else {
    #ifdef DEBUG
    console->DebugMsg("Polygon ==> Outside point ",outside,status);
    #endif
    
    return (outside);
  }

}



int Polygon::getID() { return ID; }

double Polygon::calcArea(const std::vector<Point*>& ptlist){

  int num = ptlist.size();
  
  Area = 0.0;
  
  for(int i=0; i < num-1; i++) {
    Area += (ptlist[i]->getX() - ptlist[i+1]->getX()) *
            (ptlist[i+1]->getY() + ptlist[i]->getY());
  }
  Area += (ptlist[num-1]->getX() - ptlist[0]->getX()) *
          (ptlist[0]->getY() + ptlist[num-1]->getY());
  Area = Area * 0.5;
  
  return Area;
}

/*
 This method is coppied from point.h file of HSE code (HSM Division) for
 consistency.
 The algorithm obtained from comp.graphics.algorithms-faq
 This algorithm computes the "True" area centroid.  This is not just an
 average of all the vertices around the polygon as many "centroids" are
 commonly calculated (although they are approximately the same for fairly
 "regular" polygons and exactly the same for polygons such as rectanglar
 quadrilaterals and all triangles.)
 "ptlist" contains a ccw ordered list of points.  The centroid is returned
 in "centroid".  "area" has a default value of 0 so if nothing is passed in it
 will not be filled with a value.
*/
int Polygon::centroid(const std::vector<Point*> &ptlist,
                           Point &centroid, double *area) {

  int n = ptlist.size();
  int i, j;
  double ai, atmp = 0, xtmp = 0, ytmp = 0;
  
  if (n < 3) {
    // I could do a quickie centroid for special case of triangles, (linear
    // average of 3 vertices), but I need a quick area computation for
    // triangles as well.  For some reason this function has the side effect of
    // computing area...
    Point avg(0,0);
    for (i=0; i<n; i++) {
      avg += *(ptlist[i]);
    }
    centroid = avg * (1.0 / n);
    //centroid.print();
    if (area) {
      *area = 0.0;
    }
    return 1;      // degenerate polygon.
  }
  
//  for (i = n-1, j = 0; j < n; i = j, j++) {
//    return 1;      // degenerate polygon.
//  }

    
  for (i = n-1, j = 0; j < n; i = j, j++) {
    ai = ptlist[i]->getX() * ptlist[j]->getY() - ptlist[j]->getX() * ptlist[i]->getY();
//    cout << "i: " << i << " " << ptlist[i]->getX() << endl;
//    cout << "j: " << j << " " << ptlist[j]->getY() << endl;
//    cout << "j: " << j << " " << ptlist[j]->getX() << endl;
//    cout << "i: " << i << " " << ptlist[i]->getY() << endl;
    atmp += ai;
    // cout << "atmp: " << atmp << " " << "ai: " << ai << endl;
    xtmp += (ptlist[j]->getX() + ptlist[i]->getX()) * ai;
    ytmp += (ptlist[j]->getY() + ptlist[i]->getY()) * ai;
  }
  if (area) {
    *area = atmp / 2;
  }
  if (atmp != 0) {
    centroid.setX(xtmp / (3 * atmp));
    centroid.setY(ytmp / (3 * atmp));
    //cout << "x center " <<  xtmp / (3 * atmp) << " " <<  ytmp / (3 * atmp) << endl;
    Centroid = centroid;  //new for later use
    
    return 0;      // everything calculated ok.
  }
  return 2;        // zero area polygon
}

