/*! \file success.h
\brief Header file for habitat succession. 

This defines or declares variables & functions that are global to Success.c.  \n

Note: documented with Doxygen, which expects specific syntax within special comments.  \n
	
The Everglades Landscape Model (ELM). \n
last updated: Jan 2005 \n
*/

#ifndef SUCCESS_H
#define SUCCESS_H

#include "globals.h"

	
#define MAX_SW 9999  /*!< A maximum value for high-end of a switching parameter (suitable conditions re. that variable are always met if actual parm exceeds it) */
#define AV_PER 7  /*!< Number of days that constitutes the period of averaging -  use weekly (7 d); should be <100 */
#define SW_TIME_TH_W 4  /*!< Switching time threshold for water - use 4 d for weekly AV_PER (i.e., majority of week) */
#define SW_TIME_TH_N 4  /*!< Switching time threshold for nutrients - use 4 d for weekly AV_PER (i.e., majority of week) */
	

	/*! \em HabHist is an array of integers : nnNssS, where nn are the two digits for the number of
 	weeks being in the nutrient favorable conditions, N the number of days in those conditions 
 	during the current week; similarly ss are the two digits for the number of weeks in the
 	water level favorable conditions and S is the number of days during the current week in
 	that state. The switching occurs when both of these histories exceed the habitat specific 
 	periods  Pin.  */
unsigned long *HabHist;         /* maximum is 4294967296 */

	/*! Threshold parameters that define habitat switching */
struct HabData
{
  float Lhi;             /*!< upper parameter value (Level high) for the habitat */
  float Llo;             /*!< lower parameter value (Level low) for the habitat */
  float Pin;             /*!< length of time needed to stay within (Period in) suitable values to switch to this habitat type */
};

	/*! Habitat parameter types that define habitat switching */
struct Habitat
{
  struct HabData Water;		/*!< Hydrologic threshold parameters that define habitat switching */
  struct HabData Nutrient;	/*!< Nutrient threshold parameters that define habitat switching */
  float  PFin;   /*!< (no fire, UNUSED) length of time needed to stay without a fire (Fireless Period in) to switch to this habitat type */
};
struct Habitat Habi[MAX_NHAB];

void HabSwitch_Init( void );
void alloc_hab_hist( void );
unsigned char HabSwitch(int ix, int iy, float *Water, float *Nutrient, 
		        int *Fire, unsigned char *HAB);
int InHab(float Var, struct HabData Params);

/* externs */
/* from driver_utilities.h */
extern void init_pvar(VOIDP Map, UCHAR* mask, unsigned char Mtype,float iv);
extern VOIDP nalloc(unsigned mem_size, const char var_name[]);

/* from unitmod_habparms.h */
extern float *HP_SfDepthLo; 
extern float *HP_SfDepthHi; 
extern float *HP_SfDepthInt; 
extern float *HP_PhosLo; 
extern float *HP_PhosHi; 
extern float *HP_PhosInt; 
extern float *HP_FireInt; 

/* from serial.h */
extern int habNumTot;

#endif  /* SUCCESS_H */
