/*! \file Driver_Utilities.c
\brief Most Input/Output utility & related functions.

This source file contains a large number of functions used in "utility" tasks
that generally relate to I/O, set-up of the simulation environment, etc. \n

Major categories of functions are (roughly) grouped in this order:  \n
  1) Output \n
  2) Parameter and other data input \n
  3) Spatial interpolation \n
  4) Routine tasks (parsing text, values, error conditions, etc) \n
  5) [effectively unused] Calls/definitions for parallel (not serial) implementation \n
  
Note: documented with Doxygen, which expects specific syntax within special comments. \n
	
The Everglades Landscape Model (ELM). \n
last updated: Dec 2004 \n
*/
	
/* General notes on revisions to this source file. 
       Nov/Dec 2004 v2.3.2: documentation upgrade 
       		- Unused functions removed 
       		- Re-organized, clarified scopes, Doxygen tags added 
       		- Functionality same as v2.1/v2.2 
	
*/

#include "driver_utilities.h"


/*! \brief Call functions to read model parameters from datafiles 
	\param s_parm_name Name of the sensitivity parameter being varied for this run
	\param s_parm_relval Indicator of current value of relative range in sensitivity parm values: nominal (0), low (1), or high (2)  
*/
void read_model_parameters( char* s_parm_name, int s_parm_relval) {

    sprintf(msgStr,"Reading global parameters..."); 
    WriteMsg(msgStr,1); 
    usrErr0(msgStr);

	ReadGlobalParms(s_parm_name, s_parm_relval);
    sprintf(msgStr,"done."); 
    WriteMsg(msgStr,1); 
    usrErr(msgStr);
	
    sprintf(msgStr,"Reading habitat-specific parameters..."); 
    WriteMsg(msgStr,1); 
    usrErr0(msgStr);

	ReadHabParms(s_parm_name, s_parm_relval);
    sprintf(msgStr,"done."); 
    WriteMsg(msgStr,1); 
    usrErr(msgStr);
  
}

/*! \brief Call the read_output_parms function 
	\return pointer to struct of ViewParm */
ViewParm* read_output_parms() {
/* need to change the Outlist_size (#defined in header) if adding outlist variables to the data struct in gen_output of UnitMod.c */
    ViewParm* v = readOutlist(Outlist_size);
    return v;
} 

/*! \brief Get the file to configure model output
	\param size Outlist_size, the number of variables in model outlist
	\return pointer to struct of ViewParm */
ViewParm* readOutlist(int size)
{
  int Npt;
  ViewParm* vp;
  FILE   *cnfgFile;
  
  if(H_OPSYS==UNIX) sprintf(modelFileName,"%s/%s/RunParms/Model.outList",ModelPath,ProjName);
  else sprintf(modelFileName,"%s%s:RunParms:Model.outList",ModelPath,ProjName);
  cnfgFile = fopen(modelFileName, "r");
  if (cnfgFile == NULL) {
      sprintf(msgStr, "Error, can't open file %s", modelFileName);
      WriteMsg(msgStr, 1); exit(-1);}
  
  Npt = readViewParms(cnfgFile,size, &vp);
  return vp;

}
	

/*! \brief Read/process the command-parameters to configure model output

	The configuration file (Model.outList) is read to process the type 
	(e.g., spatial maps, point time series, debug windows) and
	frequency (constant intervals of time) of model output.
	
	\param vpFile Pointer to the Model.outList configuration file
	\param size Outlist_size, the number of variables in model outlist
	\param ppview Struct of ViewParm output configuration data
	\return value of Outlist_size
*/
int readViewParms(FILE *vpFile, int size, ViewParm **ppview)
{
  char test;
  char cmd;
  int index=0, ix, iy, iz, i, nArgs, precision, format=0, Ocmd;
  float x,y;
  ViewParm *vp, *view;

  format = init_config_file(vpFile, '#', '*', '@', '~');
  view = (ViewParm *)nalloc(size * sizeof(ViewParm), "view");
  *ppview = view;
  
  for (i = 0; i < size; i++) {
    vp = view+i;
    vp->scale.s = 1.0;
    vp->scale.o = 0.0;
    vp->step = 0;
    Ocmd=0;
    vp->nPoints = 0;
    vp->points = NULL;
    vp->maxPoints = 0;
    vp->mapType = 'N';
    precision = 1;
    setPrecision(vp,precision);
  }

  while(1) {
    index = parse_packet(vpFile, &nArgs, &test);/*fix*/
    if( index == -1)  break; 
    if( index == -3)  break; 
    if(index >= size) { 
      fprintf(stderr,"\n Read index Error in configuration: index out of range: %d ", index); 
      break; 
    }
    if( index == -2) break;
    else {
      vp = view+index;
      strncpy(vp->fileName,gCArg[0],23);
      vp->fileName[23]='\0';
      if( test == '*' ) setFlag(vp,ISARRAY,1); 
      else if (test == '@') setFlag(vp,ISARRAY,0);
      else { 
        sprintf(msgStr," %s: Syntax Error in configuration, test char = %c ",gCArg[0],test); 
        usrErr(msgStr);  
        break; 
      }
      if(debug) { 
        sprintf(msgStr,"\nReading Output Config for %s: Nargs = %d, CMD0 = %c index = %d",
                gCArg[0],nArgs,gCArg[1][0],index);	
	WriteMsg(msgStr,1);  
      }				
      cmd = gCArg[1][0]; 
      switch (cmd) {	
          case 'O':
              if( isInteger(gCArg[2])  ) { ix = atoi(gCArg[2]);  }
              else  fprintf(stderr," %s: Syntax Error in configuration ",gCArg[0]);	
              if(debug) { 
                  sprintf(msgStr,"\n%s Output Config: O(%d)",gCArg[0],ix);	
                  WriteMsg(msgStr,1); 
              }				
              vp->step = ix; Ocmd=1;
              /* TODO: develop flexible output of calendar-based and julian-based outsteps (jan 11, 2005) */
              /* kluge here in checking use of Model.outList outstep relative to Driver.parm avg_Intvl */
              if (vp->step == CalMonOut && avg_Intvl != 0) {
                  sprintf(msgStr,"\n***ERROR in Model.outList & Driver.parm integration: Driver.parm's avg_Intvl must be 0 for calendar-based Outstep of %s.",gCArg[0]);	
                  usrErr(msgStr); 
                  exit(-1);
              }
              
              break;
          case 'A':
              if( isInteger(gCArg[2]) && nArgs > 2 ) { ix = atoi(gCArg[2]);  }
              else  { 
                  fprintf(stderr," %s: Syntax Error in configuration ",gCArg[0]);  
                  WriteMsg(msgStr,1); 
              }	
              if(debug) { 
                  sprintf(msgStr,"\n%s Output Config: A()",gCArg[0]); 
                  WriteMsg(msgStr,1); 
              }					
              vp->mapType = 'H';
              if( Ocmd == 0 ) vp->step = 1;
              break;
          case 'G':
              if ( nArgs > 2 )  { 
                  if( isInteger(gCArg[2])  ) { 
                      vp->mapType = atoi(gCArg[2]); 
                  } 
                  else { vp->mapType =gCArg[2][0]; } 
              }
              if ( nArgs > 3 ) 
                  if( isInteger(gCArg[3])  ) { 
                      precision = atoi(gCArg[3]); 
                  }
              if ( nArgs > 4 ) strcpy(vp->fileName,gCArg[4]);
              if( precision > 1 && precision < 5 ) setPrecision(vp,precision);
              if( Ocmd == 0 ) vp->step = 1;
              break;
          case 'B':
              if( isFloat(gCArg[2]) && isFloat(gCArg[3])  && nArgs > 3 ) { 
                  vp->bounds.Vmax = atof(gCArg[2]); 
                  vp->bounds.Vmin = atof(gCArg[3]); 
              }
              if(debug) { 
                  sprintf(msgStr,"\n Output Config: G(%.1f,%.1f)",
                          vp->bounds.Vmax, vp->bounds.Vmin); 
                  WriteMsg(msgStr,1); 
              } 
              break;
          case 'P':
              if( isInteger(gCArg[2]) &&  isInteger(gCArg[3]) && nArgs > 3 )  { 	
                  if( (i = (++(vp->nPoints)-1)) >= (vp->maxPoints)) make_more_points(vp,7);
                  ix = atoi(gCArg[2]); iy = atoi(gCArg[3]);  
              }
              else  fprintf(stderr," %s: Syntax Error in configuration ",gCArg[0]);
              if(debug) { 
                  sprintf(msgStr,"\n%s Output Config: P(%d,%d), index = %d",gCArg[0],ix,iy,i); 
                  WriteMsg(msgStr,1);  
              }				
              vp->points[i].x = ix; 
              vp->points[i].y = iy;
              vp->points[i].type = 'p';
              /* TODO: develop flexible output of calendar-based and julian-based outsteps (jan 11, 2005) */
              /* kluge here in preventing use of point time series output using calendar-based outstep  */
              if (vp->step == CalMonOut) {
                  sprintf(msgStr,"\n***ERROR in Model.outList: sorry, but you can't have point-time series output with calendar-based Outstep of %s.  Go yell at Carl!",gCArg[0]);	
                  usrErr(msgStr); 
                  exit(-1);
              }
              break;
          case 'W':
              if( isInteger(gCArg[2]) &&  isInteger(gCArg[3]) &&  isInteger(gCArg[4]) && nArgs > 4 ) {
                  if( (i = (++(vp->nPoints)-1)) >= (vp->maxPoints)) make_more_points(vp,7);
                  ix = atoi(gCArg[2]); iy = atoi(gCArg[3]); iz = atoi(gCArg[4]);  
              }
              else  fprintf(stderr," %s: Syntax Error in configuration ",gCArg[0]);
              if( nArgs > 5 )  vp->points[i].format = gCArg[5][0];
              vp->points[i].type = 'w';
              vp->points[i].x = ix;  	
              vp->points[i].y = iy;  	
              vp->points[i].z = iz;
              if(debug) { 
                  sprintf(msgStr,"\n%s Output Config: W(%d,%d,%d,%c), index = %d",
                          gCArg[0],ix,iy,iz,vp->points[i].format,i);	
                  WriteMsg(msgStr,1);  
              }				
              if( Ocmd == 0 ) vp->step = 1;
              break;
          case 'S': case 'C':
              if( gCArg[1][1] == 'm' || gCArg[1][0] == 'C' ) {     
                  if( isInteger(gCArg[2]) &&  isInteger(gCArg[3]) &&  isInteger(gCArg[4]) && nArgs > 5 ) {
                      if( (i = (++(vp->nPoints)-1)) >= (vp->maxPoints)) make_more_points(vp,7);
                      ix = atoi(gCArg[2]); iy = atoi(gCArg[3]); iz = atoi(gCArg[4]); 
                      vp->points[i].type =  gCArg[5][0];
                  }
                  else  fprintf(stderr," %s: Syntax Error in configuration ",gCArg[0]);
                  vp->points[i].x = ix;  	
                  vp->points[i].y = iy;  	
                  vp->points[i].z = iz;
                  if(debug) { 
                      sprintf(msgStr,"\n%s Output Config: Sm(%d,%d,%d,%c), index = %d",
                              gCArg[0],ix,iy,iz,gCArg[4][0],i);  
                      WriteMsg(msgStr,1); 
                  }				
                  if( Ocmd == 0 ) vp->step = 1;	
              } 
              else {								
                  if( nArgs > 2 ) {
                      if( isFloat(gCArg[2]) ) { x = atof(gCArg[2]); }
                      else  fprintf(stderr," %s: Syntax Error in configuration ",gCArg[0]);
                  }
                  if( nArgs > 3 ) {
                      if( isFloat(gCArg[3]) ) { y = atof(gCArg[3]); }
                      else  fprintf(stderr," %s: Syntax Error in configuration ",gCArg[0]);
                  }
                  if(debug) { 
                      sprintf(msgStr,"\n%s Output Config: S(%f,%f)",gCArg[0],x,y); 
                      WriteMsg(msgStr,1); 
                  }				
                  vp->scale.s = x; 
                  vp->scale.o = y;
              }
              break;
      } /* end of switch */
    } /* end of else */
  } /* end of while */
  return size;
}


/*! \brief Allocate memory & set up another set of point-locations for output

	For each model variable to be output (info in ViewParm struct), increment the number of point-locations
	(cells) based on the number requested in the model output configuration file. 
	
	\param vp Pointer to struct of ViewParm
	\param ptIncr number of points to increment
*/
void make_more_points(ViewParm *vp, int ptIncr)
{
	Point3D *new_pts;
	int i;
	
	new_pts = (Point3D *) malloc( (vp->maxPoints+ptIncr) * sizeof(Point3D) );
	for(i=0; i < vp->maxPoints; i++) {
		new_pts[i].x = vp->points[i].x;
		new_pts[i].y = vp->points[i].y;
		new_pts[i].z = vp->points[i].z;
		new_pts[i].type = vp->points[i].type;
	}
	if( vp->points != NULL ) { free((char*)vp->points); }
	vp->points = new_pts;
	vp->maxPoints += ptIncr;
	if(debug) {
		sprintf(msgStr,"Made %d more points for %s for %d total points",ptIncr,vp->fileName,vp->maxPoints); 
		WriteMsg(msgStr,1);
	}
}


/*! \brief Calculate maximum & minimum values in model output arrays

	For each model variable to be output (info in ViewParm struct), calculate
	the maximum and minimum values across the variable's spatial array.   
	\param vp struct of ViewParm data
	\param Map Model variable array data
	\param Mtype General output format type
	\param mName Variable name
	\param step The current iteration number
	*/
void calc_maxmin(ViewParm *vp, void* Map, char Mtype, char* mName, int step) 
{
  int ix, iy;
  float vout[4], ftmp, fmax = -1000, fmin = 1000;
  
  switch(Mtype) {
  case 'f' :	
    for(ix=1; ix<=s0; ix++) 
      for(iy=1; iy<=s1; iy++) {
	if( ON_MAP[T(ix,iy)] ) {
	  if( (ftmp = ((float*)Map)[T(ix,iy)]) > fmax ) {
	    fmax = ftmp; 
	  }
	  if( ftmp < fmin ) fmin = ftmp;
	}
      }
    break;
  case 'i' :	case 'd' :	
    for(ix=1; ix<=s0; ix++) 
      for(iy=1; iy<=s1; iy++) {
	if( ON_MAP[T(ix,iy)] ) {
	  if((ftmp = ((int*)Map)[T(ix,iy)]) > fmax ) fmax = ftmp;
	  if( ftmp < fmin ) fmin = ftmp;
	}
      }
    break;
  case 'c' :	
    for(ix=1; ix<=s0; ix++) 
      for(iy=1; iy<=s1; iy++) {
	if( ON_MAP[T(ix,iy)] ) {
	  if( (ftmp = (float)((unsigned char*)Map)[T(ix,iy)]) > fmax ) fmax = ftmp;
	  if( ftmp < fmin ) fmin = ftmp;
	}
      } 
    break;
  }
  if(step==0) {
    vp->gScale.Vmax = fmax;
    vp->gScale.Vmin = fmin;
  } else {
    if( fmax > vp->gScale.Vmax ) vp->gScale.Vmax = fmax;
    if( fmin < vp->gScale.Vmin ) vp->gScale.Vmin = fmin;
  }
  vout[0] = fmax; vout[1] = vp->gScale.Vmax; vout[2] = fmin; vout[3] = vp->gScale.Vmin;
  Combine(vout,mName,4,kMAXMIN,step); /* Iterative and cumulative, maximum and minimum values for all window cells at outstep iterations */
}

/*! \brief Provide the 2D array size for the model domain */
void setup_grid() {
  
  exgridsize(procnum,gbl_size,lcl_size,lcl_start); /* effectively unused in serial (non-parallel) implementation */

  /*  s0 and s1 remain unchanged in serial (non-parallel) implementation */
  s0 = lcl_size[0];
  s1 = lcl_size[1];
  
  gridSize = (s0+2)*(s1+2); /* the gridSize is increased by 2 in each dimension for buffer strips around processor domain(s) (functional mainly to parallel implementation) */
  
  sprintf(msgStr,"\nGRID DATA::[ gsize: (%d, %d), lstart: (%d, %d), lend: (%d, %d), lsize: (%d, %d) ]\n",
	  gbl_size[0], gbl_size[1], lcl_start[0], 
	  lcl_start[1], lcl_start[0]+lcl_size[0]-1, 
	  lcl_start[1]+lcl_size[1]-1, lcl_size[0], lcl_size[1] );
  WriteMsg(msgStr,1);  
  sprintf(msgStr,"\nVP DATA:: size: (%d), Variable sizes: float: %d, int: %d, long: %d,  double: %d\n", 
	    sizeof(ViewParm), sizeof(float),sizeof(int) ,sizeof(long) ,sizeof(double)   );
  WriteMsg(msgStr,1);

  gTempSize = gridSize*8;
  gTemp = (UCHAR*) nalloc(gTempSize,"gTemp");  
}


/*! \brief Determine which functions to call for model output 
	\param index Count index of the output iteration for this variable (may be > step)
	\param vp A struct of ViewParm, outlist configuration
	\param Map Model variable array data
	\param filename Output file name
	\param Mtype General output format type
	\param step The current iteration number
	*/
void write_output(int index, ViewParm* vp, void* Map,
		  const char* filename, char Mtype, int step)
{
	int i; Point3D point;
	

        if(vp->mapType != 'N' ) {
			write_map_file((char*)filename,Map,Mtype,index,vp->scale.s,vp->scale.o,getPrecision(vp),vp->mapType);
			if (debug > 3) calc_maxmin(vp,Map,Mtype,(char*)filename,step);
	}
	for(i=0; i< (vp->nPoints); i++ ) {
		point = vp->points[i];
	    if (debug > 3) { sprintf(msgStr,"\nwriting Out: %s(%d): %c(), index = %d, step=%d\n", filename, i, point.type, index, step ); 
            WriteMsg(msgStr,1); 
            }
            
		switch( point.type ) {	
                    case 'm': case 'M': calc_maxmin( vp,Map,Mtype,(char*)filename,step); break;	
                    case 'w': quick_look( Map,(char*) filename, Mtype, point.x, point.y, point.z, point.format ); break;
                    case 'a': case 's': case 'A': case 'S': print_loc_ave( &point, Map, Mtype, (char*)filename, step ); break;
                    case 'p': print_point( vp, Map, Mtype, (char*)filename, step, i ); break;
		}
	}
}
						
/*! \brief Prepare the file and the data for writing output of map (array) 
	\param filename Output file name
	\param Map Model variable array data
	\param Mtype General output format type
	\param index Count index of the output iteration for this variable (may be > step)
	\param scale_value Variable-scaling multiplier
	\param offset_value Variable-scaling offset
	\param bSize The byte size of the output array
	\param type The output map storage type 
*/
int write_map_file(const char* filename, VOIDP Map, char Mtype,
		   int index, float scale_value, float offset_value,
		   int bSize, unsigned char type)
{	
  int i, j, pathType=0;
  char /* ftype[7] */ftype[10], gSize;
  float ftmp; 
  SLONG itmp, imax, imin;
  UCHAR *mPtr;
  char s_mo[3], s_da[3];
  
/* below provides old (v1.0) increasing integer, non-date, appendage to filenames */
/*   ftype[0] = '0' + (index / 100); */
/*   ftype[1] = '0' + ((index%100) / 10); */
/*   ftype[2] = '0' + (index % 10); */
 /*  ftype[8] = '\0'; */

/*  append the date (yyyymmdd) to the root filename*/
  if (SimTime.mo[0]<10) sprintf(s_mo,"0%d", SimTime.mo[0]);
  else sprintf(s_mo,"%d", SimTime.mo[0]);
  if (SimTime.da[0]<10) sprintf(s_da,"0%d", SimTime.da[0]);
  else sprintf(s_da,"%d", SimTime.da[0]);
  sprintf(ftype,"%d%s%s\0",SimTime.yr[0],s_mo,s_da);
  
  if( HDF && type=='H' ) sprintf(ftype,".hdf");
   
  gSize = gridSize*bSize + 1;
  if( gSize > gTempSize ) { 
    if(gTempSize) free((char*)gTemp); 
    gTemp = (UCHAR*) nalloc( gSize, "gTemp" );	
    gTempSize = gSize; 
  }
  
  if( H_OPSYS == UNIX ){ 
    if(check_for((char*)filename, "/", 0, CASE, END ) >= 0 ) { 
      if(filename[0] != '/' ) pathType = 1; 
      else pathType = 2; 
    }
  }
  else { if(check_for ((char*)filename, ":", 0, CASE, END ) >= 0 ) { if( filename[0] == ':' ) pathType = 1; else pathType = 2; }}
  
  if(type=='H') {                         /* HDF format */
    switch(pathType) {
    case 0:	/* No Path */
      if(H_OPSYS==UNIX) sprintf(modelFileName,"%s/%s/Output/Animation/%s%s\0",OutputPath,ProjName,filename,ftype);
      else sprintf(modelFileName,"%s%s:Output:Animation:%s%s\0",OutputPath,ProjName,filename,ftype);
      break;
    case 1:	/* Relative Path */
      if(H_OPSYS==UNIX) sprintf(modelFileName,"%s/%s/Output/Animation/%s%s\0",OutputPath,ProjName,filename,ftype);
      else sprintf(modelFileName,"%s%s:Output:Animation:%s%s\0",OutputPath,ProjName,filename,ftype);
      break;
    case 2:	/* Full Path */
      sprintf(modelFileName,"%s.hdf\0",filename);
      break;
    }
  }
  else if(type=='M') {	
    switch(pathType) {                   /* Map II (float, binary) Format */
    case 0:								/* No Path */
      if(H_OPSYS==UNIX) sprintf(modelFileName,"%s/%s/Output/Maps/%s%s\0",OutputPath,ProjName,filename,ftype);
      else sprintf(modelFileName,"%s%s:Output:Maps:%s%s\0",OutputPath,ProjName,filename,ftype);
      break;
    case 1:	/* Relative Path */
      if(H_OPSYS==UNIX) sprintf(modelFileName,"%s/%s/Output/Maps/%s%s\0",OutputPath,ProjName,filename,ftype);
      else sprintf(modelFileName,"%s%s:Output:Maps:%s%s\0",OutputPath,ProjName,filename,ftype);
      break;
    case 2:	/* Full Path */
      sprintf(modelFileName,"%s%s\0",filename,ftype);
      break;
    }
  }
  else {
  	bSize = 1;
    switch(pathType) {                   /* Generic Binary Format */
    case 0:								/* No Path */
      if(H_OPSYS==UNIX) sprintf(modelFileName,"%s/%s/Output/Animation%d/%s%s\0",OutputPath,ProjName,type,filename,ftype);
      else sprintf(modelFileName,"%s%s:Output:Animation%d:%s%s\0",OutputPath,ProjName,type,filename,ftype);
      break;
    case 1:	/* Relative Path */
      if(H_OPSYS==UNIX) sprintf(modelFileName,"%s/%s/DriverOutput/Animation%d/%s%s\0",OutputPath,ProjName,type,filename,ftype);
      else sprintf(modelFileName,"%s%s:Output:Animation%d:%s%s\0",OutputPath,ProjName,type,filename,ftype);
      break;
    case 2:	/* Full Path */
      sprintf(modelFileName,"%s%s\0",filename,ftype);
      break;
    }
  }
  
  bSize = (bSize < 1) ? 1 : bSize;
  bSize = (bSize > 4) ? 4 : bSize;
  if( Mtype == 'c' ) bSize = 1;
  if(bSize == 1) { imax = 255; imin = 0; }
  else if(bSize == 2) { imax = 255*128; imin = -imax; }
  else if(bSize == 3) { imax = 256*256*128; imin = -imax; }
  mPtr = gTemp;
  
  if  ( debug >3) {  sprintf(msgStr,"\n\nWriting Map %s: scale_value = %f, offset_value = %f, index = %d, filename = %s, type = %c, bSize = %d, pathType = %d, type = %d\n",
			   filename,scale_value,offset_value,index,modelFileName,Mtype,bSize,pathType,type);   WriteMsg(msgStr,1); }
  
  for (i=0; i<s0; i++) {
    for (j=0; j<s1; j++) {
      if( ON_MAP[T(i+1,j+1)] ) {
	switch(Mtype) {
	case 'l' :
      ftmp = ((( (double*) Map)[T(i+1,j+1)] - offset_value) / scale_value); /* v2.3.2 added double (casting to float ftmp) */
	  itmp = (int) ftmp;
	  itmp = (itmp > imax-1) ? imax-1 : itmp;
	  itmp = (itmp < imin) ? imin : itmp;
	  break;
	case 'f' :    			
	  ftmp = ((( (float*) Map)[T(i+1,j+1)] - offset_value) / scale_value); 
	  itmp = (int) ftmp;
	  itmp = (itmp > imax-1) ? imax-1 : itmp;
	  itmp = (itmp < imin) ? imin : itmp;
	  break;
	case 'd' : case 'i' :  	
	  ftmp = ((( (int*) Map)[T(i+1,j+1)] - offset_value) / scale_value); 
	  itmp = (int) ftmp;
	  itmp = (itmp > imax-1) ? imax-1 : itmp;
	  itmp = (itmp < imin) ? imin : itmp;
	  break;
	case 'c' : 				
	  itmp = ((unsigned char*) Map)[T(i+1,j+1)];
	  itmp = (itmp > imax-1) ? imax-1 : itmp;
	  itmp = (itmp < imin) ? imin : itmp;
	  break;
	}
      }
      else  itmp = imax;
      
      enc_Nb(&mPtr,itmp,bSize);
    }
  }	
  writeMap(modelFileName,gTemp,bSize,type,index);
  if(debug) quick_look(Map, (char*)filename, Mtype, dbgPt.x, dbgPt.y, 2,'E');
  return(0);
}


/*! \brief Get the file and read/convert the data of map (array) 

	\param filename Input file name
	\param Map Model variable array data
	\param Mtype General data format type
	\param scale_value Variable-scaling multiplier
	\param offset_value Variable-scaling offset
	\return byte size of data values
*/
int read_map_file(const char *filename, VOIDP Map,
		  unsigned char Mtype, float scale_value,
		  float	offset_value)
{
  int i, j, k0, size;
  unsigned temp;
  UCHAR *tmpPtr, Dsize;
  
gridSize = (s0+2)*(s1+2);
gTempSize = gridSize*8;


  size = gridSize*4 +1;
  if( size > gTempSize ) { 
    if(gTemp) free((char*)gTemp); 
    gTemp = (UCHAR*)nalloc( size, "gTemp" );	
    gTempSize = size; 
  }
  
  if(debug>2) { 
    sprintf(msgStr,"Reading %s\n",filename);  
    usrErr(msgStr); 
    WriteMsg(msgStr,1); 
  } 
  
  Dsize = readMap(filename,gTemp);
  
  if(debug) { 
    sprintf(msgStr,"name = %s, proc = %d, scale_value = %f, offset_value = %f, size = %x\n ",
	    filename,procnum,scale_value,offset_value,Dsize);  
    WriteMsg(msgStr,1); 
  } 
  for (i=1; i<=s0; i++) {
    for (j=1; j<=s1; j++) {
      k0 = Dsize*((i-1)*lcl_size[1] + (j-1));
      tmpPtr = gTemp+k0;
      switch(Dsize) {
        case 1: temp = gTemp[k0]; break;
        case 2: temp = gTemp[k0]*256 + gTemp[k0+1]; break;
        case 3: temp = gTemp[k0]*65536 + gTemp[k0+1]*256 + gTemp[k0+2]; break;
        case 4: temp = gTemp[k0]*16777216 + gTemp[k0+1]*65536 + gTemp[k0+2]*256 + gTemp[k0+3]; break;
        default: fprintf(stderr,"ERROR, illegal size: %x\n",Dsize); temp = 0;
      }						
      switch (Mtype) {
        case 'f' :
	  ((float*)Map)[T(i,j)] = scale_value*((float)temp)+offset_value; 
 	break;
        case 'd' : case 'i' :
	  ((int*)Map)[T(i,j)] = (int)(scale_value * (float)temp + offset_value); 
	break;
        case 'c' :
	  ((unsigned char*)Map)[T(i,j)] = (int)(scale_value * (unsigned char)temp); 
	break;
      }
    }
  }
  if(debug) quick_look(Map, (char*)filename, Mtype, dbgPt.x, dbgPt.y, 2,'E');
  link_edges(Map,Mtype);
  fflush(stderr);
  return Dsize;
}



/************************************************************************/
/*									*/
/*	Routine:	enc_Nb					*/
/*									*/
/*	Place a 1,2,3, or 4 (N) byte signed value into the specified buffer	*/
/*	and advance the buffer pointer past the placed object.		*/
/*									*/
/*	Parameters:							*/
/*		"sptr"	A pointer to pointer to UCHAR (the buffer).	*/
/*		"value"	The value to place in the buffer.		*/
/*									*/
/*	Returns:	None.						*/
/*									*/
/************************************************************************/

/*! \brief Place data in bytes into a buffer to assemble a binary array 

	Place a 1,2,3, or 4 (N) byte signed value into the specified buffer
	and advance the buffer pointer past the placed object.
	\param sptr a pointer to pointer to UCHAR (the buffer)
	\param value The value to place in the buffer
	\param bytes Number of bytes in data
*/
void enc_Nb(UCHAR** sptr,SLONG value,int bytes)
{
  UCHAR tmp[4];
  switch(bytes) {
  case 1:
    *(*sptr)++ = value;
    break;
  case 2:
    *(*sptr)++ = (value >> 8);
    *(*sptr)++ = value;
    break;
  case 3:
    tmp[0] = value;
    tmp[1] = (value >>= 8);
    tmp[2] = (value >> 8);
    *(*sptr)++ = tmp[2];
    *(*sptr)++ = tmp[1];
    *(*sptr)++ = tmp[0];
    break;
  case 4:
    tmp[0] = value;
    tmp[1] = (value >>= 8);
    tmp[2] = (value >>= 8);
    tmp[3] = (value >> 8);
    *(*sptr)++ = tmp[3];
    *(*sptr)++ = tmp[2];
    *(*sptr)++ = tmp[1];
    *(*sptr)++ = tmp[0];
    break;
  }
}

/*! \brief Prepare data and write heading for writing a data window in debug file 
	\param Map Model variable array data
	\param name Model variable name
	\param Mtype General output format type
	\param xc Point location x (row!)
	\param yc Point location y (col!)
	\param range Size range (cells) of window
	\param format Numeric format type
*/
void quick_look(void* Map, char* name, unsigned char Mtype, 
		int xc, int yc, int range, unsigned char format)
{
  int ymin, ymax, xmin, xmax, x, y, N0, N1;
  char* namePtr;

  if (!on_this_proc(xc,yc)) return;
  x = xc - lcl_start[0] + 1;
  y = yc - lcl_start[1] + 1;
  xmin = x - range;
  xmax = x + range + 1;
  ymin = y - range;
  ymax = y + range + 1;
  xmax = iclip( xmax, 0, s0+2);
  xmin = iclip( xmin, 0, s0+2);
  ymax = iclip( ymax, 0, s1+2);
  ymin = iclip( ymin, 0, s1+2);
  N0 = xmax-xmin;
  N1 = ymax-ymin;
  namePtr = name_from_path(name);
  if (debug >3) {
    sprintf(msgStr,"Window to Map %s, s=%d, proc0 = %d, proc1 = %d; Corners(gbl): (%d,%d) (%d,%d)\n",
	    namePtr,range,recpnum[0],recpnum[1],xc-range,yc-range,xc+range+1,yc+range+1);
    writeWindow(Map,namePtr,msgStr,xmin,ymin,N0,N1,istep,Mtype,format);
  }
  
}


/*! \brief Prepare a local data set for writing a local data window in debug file 
	\param fValue Model variable array data
	\param vName Model variable name
	\param desc Message describing the output
	\param x0 Lower x (row!) value
	\param y0 Lower y (col!) value
	\param N0 Range in x (row!) values
	\param N1 Range in y (col!) values
	\param index Current iteration number
	\param Mtype General output format type
	\param format Numeric format type
*/
void writeWindow(void* fValue, char* vName, char* desc,
		 int x0, int y0, int N0, int N1, int index,
		 UCHAR Mtype, UCHAR format)
{
  int mSize, size;
  char ctemp[200];
  
  switch(Mtype) {
  case 'l':		                size = sizeof(double); break; /* v2.3.2 added */
  case 'f': case 'L': case 'E':		size = sizeof(float); break;
  case 'd': case 'i':	                size = sizeof(int);  break;
  case 'c':				size = sizeof(char); break;
  }
  if( (mSize=size*N0*N1) > gTempSize ) {
    if(gTemp != NULL) free((char*)gTemp);
    gTemp = (UCHAR*)malloc(gTempSize=mSize);
  }
  
  Copy(((UCHAR*)fValue)+T(x0,y0)*size, gTemp, N1*size, N0, (s1+2)*size, N1*size);
  sprintf(ctemp,"(%d)WIND: %s",index,vName);
  writeSeries(gTemp,ctemp,desc,N0,N1,Mtype,format);
}

/*! \brief Constrain x0 value within a min and a max 
	\param x0 number to work on
	\param min the minimum
	\param max the maximum
	\return result
*/
int iclip ( int x0, int min, int max )
{
  int rv;
  rv = ( x0 > max) ?  max : x0;
  rv = ( rv < min) ?  min : rv;
  return rv;
}


/*! \brief Memory-copy data from large (global) domain to local domain
	\param src The source of info
	\param dst The destination of info
	\param w The number of characters
	\param n The number of (row) repetitions
	\param sw The source size in col direction
	\param dw The destination size in col direction
	\return resulting (destination) copy */
void Copy(void *src, void *dst, int w, int n, int sw, int dw)
{
  int i;
  for(i=0; i<n; i++) 
    memcpy( ((UCHAR*)dst)+i*dw, ((UCHAR*)src)+i*sw, w ); /* copy w chars from source to dest, return destination */
}



 
/*! \brief Acquire necessary environment variables */
void set_env_vars(void) {
  
  FILE *infile;
  char filename[100], ch;
  static long start;
  int i, maxLen =200;
  
    ModelPath = getenv("ModelPath"); 
    ProjName = getenv("ProjName");
    DriverPath = getenv("DriverPath");
    OS_TYPE = getenv("OSTYPE"); /* OSTYPE not used in code, only here for informational output */

	sprintf(msgStr,"OS type = %s ",OS_TYPE); 
    usrErr(msgStr);
    
	sprintf(msgStr,"Project Name = %s ",ProjName); 
    usrErr(msgStr);
 
}


/*! \brief Print data at point locations to memory (not to file/disk yet) 
	\param vp struct of ViewParm output configuration
	\param Map Array data of variable
	\param Mtype  General output format type
	\param mName The name of the variable
	\param tIndex Current model iteration number
	\param vpindex Index of the current point number
*/
void print_point(ViewParm *vp, void* Map, char Mtype, 
		char* mName, int tIndex, int vpindex) 
{
  	int x, y;
	Point3D *pt;
  
	pt = vp->points + vpindex;
    x = pt->x - lcl_start[0];
    y = pt->y - lcl_start[1];
    if(  (x >= 0) && (x < s0)  &&  (y >= 0) && (y < s1)  ) {
      if(tIndex==0) { pt->z = PListIndex; pSeries[PListIndex].Length=0; 
    }
    else PListIndex = pt->z; 
    if(tIndex==0) { 			
              /* TODO: develop flexible output of calendar-based and julian-based outsteps (jan 11, 2005) */
	  if(PListIndex >= MAX_PTSERIES ) fatal("Too many Point series.");
	  else if( pSeries[PListIndex].data == NULL ) pSeries[PListIndex].data = (float*) malloc( (N_iter*sizeof(float)/vp->step)+2 );
      if( pSeries[PListIndex].data == NULL ) usrErr("out of Memory for Spatial Timeseries.");
	  pSeries[PListIndex].Loc.x = pt->x;
	  pSeries[PListIndex].Loc.y = pt->y;
	  pSeries[PListIndex].outstep = vp->step; 
	  strcpy(pSeries[PListIndex].name,mName);
	  if(debug >2) { 
			sprintf(msgStr,"\nSetup Pointlist %d for %s(%d), step=%d, point=(%d,%d)\n", PListIndex, mName, vpindex, vp->step, x, y ); 
			WriteMsg(msgStr,1); 
	    }
     }
      switch(Mtype) {
		  case 'f' :  pSeries[PListIndex].data[ pSeries[PListIndex].Length++ ] = ((float*)Map)[T(x+1,y+1)];	break;		
		  case 'i' :	pSeries[PListIndex].data[ pSeries[PListIndex].Length++ ] = ((int*)Map)[T(x+1,y+1)]; break;
		  case 'c' :  pSeries[PListIndex].data[ pSeries[PListIndex].Length++ ] = ((unsigned char*)Map)[T(x+1,y+1)];
      }
	  if(debug >3) { 
		  sprintf(msgStr,"\nWriting Point %d for %s(%d), point=(%d,%d), value = %f, index = %d\n", 
							PListIndex, mName, vpindex, x, y, ((float*)Map)[T(x+1,y+1)], pSeries[PListIndex].Length-1 ); 
		  WriteMsg(msgStr,1); 
	  }
      PListIndex++;
    }
}

/*! \brief Establish the number of rows and columns of the model

	Read the map dimensions of the global model array,
	establishing #rows=s0 and #columns=s1
	\param filename character string of the representative map (usually "Elevation") */
void read_map_dims(const char* filename)
{
  FILE *file;
  
  if (debug>3) { sprintf(msgStr,"Getting map dims: %s",filename); usrErr(msgStr);} 
  if(Lprocnum == 1) {
    sprintf(modelFileName,"%s/%s/Data/Map_head/%s",ModelPath,ProjName,filename);
    file = fopen(modelFileName,"r");
    if(file==NULL) { 
      fprintf(stderr,"Unable to open map header file %s.\n",modelFileName); 
      fflush(stderr); 
      Exit(0);
    }
    scan_forward(file,"cols:");
    fscanf(file,"%d",&gbl_size[1]);
    sprintf(msgStr,"cols = %d\n",gbl_size[1]); WriteMsg(msgStr,1);
    scan_forward(file,"rows:");
    fscanf(file,"%d",&gbl_size[0]);
    sprintf(msgStr,"rows = %d\n",gbl_size[0]); WriteMsg(msgStr,1);
    fclose(file);
  }
  broadcastInt(gbl_size);   /* does nothing in serial (non-parallel) implementation */
  broadcastInt(gbl_size+1); /* does nothing in serial (non-parallel) implementation */
  s0 = gbl_size[0];
  s1 = gbl_size[1];
}

/*! \brief Initialize a variable to a value

	\param Map array of data 
	\param mask data mask
	\param Mtype the data type of the map data 
	\param iv the value used to initialize variable */
void init_pvar(VOIDP Map, UCHAR* mask, unsigned char Mtype,float iv) 
{
  int i0, i1;
  
  switch(Mtype) {
  case 'b' :	/* added double for (non-map) basin (b) array budget calcs */
    for(i0=0; i0<=numBasn; i0++) {
	((double*)Map)[i0] = iv;
      }
    break;
  case 'l' :	/* added double (l == letter "ell" ) for map arrays */
    for(i0=0; i0<=s0+1; i0++) 
      for(i1=0; i1<=s1+1; i1++) {
	if(mask==NULL) ((double*)Map)[T(i0,i1)] = iv;
	else if ( mask[T(i0,i1)] == 0 ) ((double*)Map)[T(i0,i1)] = 0;
	else ((double*)Map)[T(i0,i1)] = iv;
      }
    break;
  case 'f' :	
    for(i0=0; i0<=s0+1; i0++) 
      for(i1=0; i1<=s1+1; i1++) {
	if(mask==NULL) ((float*)Map)[T(i0,i1)] = iv;
	else if ( mask[T(i0,i1)] == 0 ) ((float*)Map)[T(i0,i1)] = 0;
	else ((float*)Map)[T(i0,i1)] = iv;
      }
    break;
  case 'i' :	case 'd' :	
    for(i0=0; i0<=s0+1; i0++) 
      for(i1=0; i1<=s1+1; i1++) {
	if(mask==NULL) ((int*)Map)[T(i0,i1)] = (int)iv;
	else if ( mask[T(i0,i1)] == 0 ) ((int*)Map)[T(i0,i1)] = 0;
	else ((int*)Map)[T(i0,i1)] = (int)iv;
      }
    break;
  case 'c' :	
    for(i0=0; i0<=s0+1; i0++) 
      for(i1=0; i1<=s1+1; i1++) {
	if(mask==NULL) ((unsigned char*)Map)[T(i0,i1)] = (unsigned char) '\0' + (int) iv;
	else if ( mask[T(i0,i1)] == 0 ) ((unsigned char*)Map)[T(i0,i1)] = '\0';
	else ((unsigned char*)Map)[T(i0,i1)] = (unsigned char) '\0' + (int) iv;
      } 
    break;
 
   default :
    printf(" in default case\n");
   break;
  }
}

/*! \brief Calculate summary statistics to determine local average 
	\param vt struct of cell points
	\param Map Model variable data array
	\param Mtype the data type of the map data
	\param mName variable name
	\param tIndex the current iteration number

*/
void print_loc_ave(Point3D *vt, void* Map, char Mtype, char* mName, int tIndex) 
{
  int ix, iy, x, y, x0, y0, range, xmax, ymax, xmin, ymin, type = 11;
  float vout[4];
  double sum = 0, normCoef = 0;
  
  if(Mtype != 'f') { fprintf(stderr,"Warning: Wrong type in Loc Ave: %s(%c)\n",mName,Mtype); return; }	
  x0 = vt->x;
  y0 = vt->y;
  range = vt->z;
  x = x0 - lcl_start[0]; 
  y = y0 - lcl_start[1];
  xmin = x - range;
  xmax = x + range;
  ymin = y - range;
  ymax = y + range;
  xmax = (xmax > (s0)) ? (s0) : xmax;
  xmin = (xmin <  0  ) ?   0  : xmin;
  ymax = (ymax > (s1)) ? (s1) : ymax;
  ymin = (ymin <  0  ) ?   0  : ymin;
  
  for(ix=xmin; ix<xmax; ix++)  {
    for(iy=ymin; iy<ymax; iy++) { 
      if( ON_MAP[T(ix+1,iy+1)] ) sum += ((float*)Map)[T(ix+1,iy+1)];			
      normCoef += 1.0;
    }
  }
  
  vout[0] = sum; vout[1] = normCoef;
  
  switch (vt->type) {
     case 's' :  Combine(vout,mName,1,kSUM,tIndex); break; /* Sum for all window cells at outStep iterations */
     case 'a' :  Combine(vout,mName,2,kAVE,tIndex); break; /* Average for all window cells at outStep iterations */
     case 'A' :  Combine(vout,mName,2,kAVECUM,tIndex); break; /* Cumulative average for all window cells over all outStep iterations */
     case 'S' :  Combine(vout,mName,1,kSUMCUM,tIndex); break; /* Cumulative sum for all window cells over all outStep iterations */
  }  
}


/*! \brief Point Time Series interpolation (unused): set up the fields */
void PTS_SetFields (PTSeries* thisStruct, int ix, int iy, 
		int index, int format, int col)
{
  float ptstep = 1.0, *value; 
  int n0;
  
  thisStruct->fX = ix;
  thisStruct->fY = iy;
  if(thisStruct->fValue) free((char*)thisStruct->fValue); 

  thisStruct->fValue = readSeriesCol(modelFileName,format,index,&n0,&ptstep,col);
  
  gNPtTs = thisStruct->fNpt = n0;
  gTS = thisStruct->fTS = ptstep;
  value = thisStruct->fValue;
  if  ( debug>2 ) {  sprintf(msgStr,"\nPTS_SetFields: TSVals: %f %f %f, TS: %f, NPt: %d \n",value[0],value[1],value[2],ptstep,n0);  WriteMsg(msgStr,1); }
}

/*! \brief Point Time Series interpolation (unused): free up memory */
void PTS_Free(PTSeries* thisStruct) 
{
  if(thisStruct->fValue) free((char*)thisStruct->fValue);
}


/*! \brief Point Time Series interpolation (unused): copy fields */
void PTS_CopyFields (PTSeries* thisStruct, PTSeries pV) 
{
  thisStruct->fX = pV.fX;
  thisStruct->fY = pV.fY;
  thisStruct->fValue = pV.fValue;
}

/*! \brief Point Time Series interpolation (unused): initialize structure */
PTSeriesList* PTSL_Init(int nSlots, int format) 
{
  int i;
  PTSeriesList* thisStruct;
  thisStruct = (PTSeriesList*) malloc(sizeof(PTSeriesList));
  thisStruct->fList = (PTSeries*) malloc(sizeof(PTSeries)*nSlots);
  for(i=0; i<nSlots; i++) thisStruct->fList[i].fValue = NULL;
  thisStruct->fNSlots = nSlots;
  thisStruct->fNSlotsUsed = 0;
  thisStruct->fNptTS = 0;
  thisStruct->fFormat = format;
  return(thisStruct);
}

/*! \brief Point Time Series interpolation (unused): add to the series */
void PTSL_AddpTSeries(PTSeriesList* thisStruct, int x, int y, int index, int seriesNum, int col ) 
{
  PTSeries* valListtemp;
  int i, newSlots;
  
  if( seriesNum >= thisStruct->fNSlots ) {
    newSlots = Max(20,(int)(thisStruct->fNSlots * 0.2));
    thisStruct->fNSlots += newSlots;
    valListtemp = (PTSeries*) malloc(sizeof(PTSeries)*(thisStruct->fNSlots));
    for(i=0; i<thisStruct->fNSlotsUsed; i++) {
      valListtemp[i].fNpt = thisStruct->fNptTS;
      PTS_CopyFields(valListtemp+i,thisStruct->fList[i]);
    }
    for(i=thisStruct->fNSlotsUsed; i<thisStruct->fNSlots; i++) valListtemp[i].fValue = NULL;
    if( thisStruct->fList != NULL ) free((char*)thisStruct->fList);
    thisStruct->fList = valListtemp;
  }
  
  PTS_SetFields( thisStruct->fList + seriesNum, x, y, index, thisStruct->fFormat, col );
  thisStruct->fNSlotsUsed = seriesNum;
}

/*! \brief Point Time Series interpolation (unused): free up memory */
void PTSL_Free(PTSeriesList* thisStruct) 
{
  int i;
  if(thisStruct == NULL) return;
  for(i=0; i<thisStruct->fNSlotsUsed; i++) PTS_Free( thisStruct->fList +i );
  if( thisStruct->fList != NULL ) free((char*)thisStruct->fList);
  free((char*)thisStruct);
  thisStruct = NULL;
}

/*! \brief Point Time Series interpolation (unused): at point, calculate inverse distance squared interpolation (restricted, not using pow() function) */
float PTSL_GetInterpolatedValue0(PTSeriesList* thisStruct, int x, int y, int step)  
{
  int i,  dx, dy, my_debug;
  PTSeries pV;
  float  wpow;
  float weight, InterpValue=0.0, distance=0.0; /* was double in v1.0 */
  int r;
  
  wpow = GP_IDW_pow; /* GP_IDW_pow is "adjustable" parameter in global parm input file */

  
  for(i=0; i<thisStruct->fNSlotsUsed; i++) {
    pV = thisStruct->fList[i];
    dx = (pV.fX-(x+lcl_start[0]));
    dy = (pV.fY-(y+lcl_start[1]));
    r = (dx*dx + dy*dy);
    if( r == 0 ) return pV.fValue[step];
    weight = (wpow == 1.0) ? (1.0)/r : 1.0/(r*r);
        /* the pow() and double was incredibly slow (signif slowed simualtion), so removed it
           and gave the choice of only inverse distance or inverse distance**2 weighting */
    /* weight = (wpow == 1.0) ? ( (double)1.0)/r : pow(r,-wpow) ; */
    InterpValue += pV.fValue[step]*weight;
    distance += weight;
  }
  if (distance > 0) InterpValue /= distance;
  else InterpValue = (thisStruct->fList[0]).fValue[step];
  return (float) InterpValue;
}

/*! \brief Point Time Series interpolation (unused): read raw point data */
void PTSL_ReadLists(PTSeriesList* thisStruct, const char *ptsFileName, 
		    int index, float* timeStep, int* nPtTS, int col ) 
{
  char pathname[150], infilename[60], ss[201], ret = '\n';
  FILE* cfile;
  int ix, iy, tst, sCnt=0;
  if( Lprocnum == 1 ) {
    if(H_OPSYS==UNIX) sprintf(modelFileName,"%s/%s/Data/%s.pts",ModelPath,ProjName,ptsFileName);
    else sprintf(modelFileName,"%s%s:Data:%s.pts",ModelPath,ProjName,ptsFileName); 
    cfile = fopen(modelFileName,"r");
    if(cfile==NULL) {fprintf(stdout,"\nERROR: Unable to open timeseries file %s\n",modelFileName); Exit(1); }
    else { sprintf(msgStr,"\nReading file %s\n",modelFileName); WriteMsg(msgStr,1); } 
    if (debug > 2) {sprintf(msgStr,"Reading %s timeseries, column %d",modelFileName, col); usrErr(msgStr);}
    
  

    fgets(ss,200,cfile); /* skip header line */
  }
  while(1) {
    if( Lprocnum == 1 ) {
      tst = fscanf(cfile,"%d",&ix);
      if( tst > 0  ) {
	fscanf(cfile,"%d",&iy); 
	tst = fscanf(cfile,"%s",infilename); 
	sprintf(modelFileName,"%s/%s/Data/%s",ModelPath,ProjName,infilename);
      }
    }
    broadcastInt(&tst); 
    if(tst<1) break;
    broadcastInt(&ix); 
    broadcastInt(&iy); 
    PTSL_AddpTSeries(thisStruct, ix, iy, index, sCnt, col);
    sCnt++;
  }
  if( Lprocnum == 1 ) fclose(cfile);  
  *nPtTS = gNPtTs;
  *timeStep = gTS;
} 

/*! \brief Point Time Series interpolation (unused): generate interpolated spatial (map) data */
void PTSL_CreatePointMap(PTSeriesList* pList,void* Map, unsigned char Mtype,
		         int step, float scale)
{
  int i0, i1;
  
  switch(Mtype) {
  case 'f' :	
    for(i0=0; i0<=s0+1; i0++) 
      for(i1=0; i1<=s1+1; i1++) 
	((float*)Map)[T(i0,i1)] = (ON_MAP[T(i0,i1)]) ? PTSL_GetInterpolatedValue0(pList,i0,i1,step)*scale : 0.0 /*was this value (v0.5beta) -0.012 */; 
    break;
  case 'i' :	case 'd' :	
    for(i0=0; i0<=s0+1; i0++) 
      for(i1=0; i1<=s1+1; i1++) 
	((int*)Map)[T(i0,i1)] = (int)  ( (ON_MAP[T(i0,i1)]) ? PTSL_GetInterpolatedValue0(pList,i0,i1,step)*scale : 0 );
    break;
  case 'c' :	
    for(i0=0; i0<=s0+1; i0++) 
      for(i1=0; i1<=s1+1; i1++) 
	((unsigned char*)Map)[T(i0,i1)] = (unsigned char) '\0' + (int)  ( (ON_MAP[T(i0,i1)]) ? PTSL_GetInterpolatedValue0(pList,i0,i1,step)*scale : 0 );
    break;
  }
}

/*! \brief Point Time Series interpolation (unused): at point, calculate inverse distance interpolation (unrestricted power of distance) */
float PTSL_GetInterpolatedValue1(PTSeriesList* thisStruct, int x, int y, int step)  
{
  int i,  dx, dy;
  PTSeries pV;
  RPointList *pList;
  float  wpow;
  double weight, InterpValue=0.0, distance=0.0;
  long r;
  
  pList = RPL_Init( 20 );
  wpow = GP_IDW_pow; /* GP_IDW_pow is "adjustable" parameter in global parm input file */
  for(i=0; i<thisStruct->fNSlotsUsed; i++) {
    pV = thisStruct->fList[i];
    dx = (pV.fX-x);
    dy = (pV.fY-y);
    r = dx*dx + dy*dy;
    if( r == 0 ) return pV.fValue[step];
    RPL_AddrPoint(pList, dx, dy, r, pV.fValue[step]);
    weight = (wpow == 1.0) ? ((double)1.0)/r : pow(r,-wpow);
    InterpValue += pV.fValue[step]*weight;
    distance += weight;
  }
  RPL_Sort(pList);
  RPL_Free(pList);
  return (float) InterpValue;
}

/*! \brief Point Time Series interpolation (unused):  */
void RP_SetFields (RPoint* thisStruct, int ix, int iy, float r, float value) 
{
  thisStruct->fX = ix;
  thisStruct->fY = iy;
  thisStruct->fr = r;
  thisStruct->fValue = value;	
}

/*! \brief Point Time Series interpolation (unused):  */
void RP_CopyFields (RPoint* thisStruct, RPoint* that) 
{
  thisStruct->fX = that->fX;
  thisStruct->fY = that->fY;
  thisStruct->fr = that->fr;
  thisStruct->fValue = that->fValue;
}

/*! \brief Point Time Series interpolation (unused):  */
void RP_SwapFields (RPoint* thisStruct, RPoint* that) 
{
  RPoint temp;
  RP_CopyFields (&temp, thisStruct);
  RP_CopyFields (thisStruct, that);
  RP_CopyFields (that, &temp);
}

/*! \brief Point Time Series interpolation (unused):  */
RPointList * RPL_Init(int nPoints) 
{
  RPointList *thisStruct;
  thisStruct = (RPointList*) malloc(sizeof(RPointList));
  thisStruct->fList = (RPoint*) malloc(sizeof(RPoint)*nPoints);
  thisStruct->fNPt = nPoints;
  thisStruct->fNPtUsed = 0;
  return thisStruct;
}

/*! \brief Point Time Series interpolation (unused):  */
void RPL_AddrPoint(RPointList *thisStruct, int x, int y, float r, float value) 
{
  RPoint* pointListtemp;
  int i, newPoints;
  
  if( thisStruct->fNPtUsed >= thisStruct->fNPt ) {
    newPoints = 20;
    thisStruct->fNPt += newPoints;
    pointListtemp = (RPoint*) malloc(sizeof(RPoint)*(thisStruct->fNPt));
    for(i=0; i<thisStruct->fNPtUsed; i++) {
      RP_CopyFields(pointListtemp+i,thisStruct->fList+i);
    }
    if( thisStruct->fList != NULL ) free((char*)thisStruct->fList);
    thisStruct->fList = pointListtemp;
  }
  
  RP_SetFields( thisStruct->fList + thisStruct->fNPtUsed, x, y, r, value);
  thisStruct->fNPtUsed++;
}

/*! \brief Point Time Series interpolation (unused):  */
void RPL_Sort(RPointList *thisStruct)
{
  int i, go=1;
  
  while(go) {
    go = 0;
    for(i=1; i<thisStruct->fNPtUsed; i++) if(thisStruct->fList[i-1].fr > thisStruct->fList[i].fr) {
      RP_SwapFields (thisStruct->fList+i-1, thisStruct->fList+i);
      go = 1;
    }
  }		
}


/*! \brief Point Time Series interpolation (unused):  */
void RPL_Free(RPointList* thisStruct) 
{
  if(thisStruct == NULL) return;
   if( thisStruct->fList != NULL ) free((char*)thisStruct->fList);
  free((char*)thisStruct);
  thisStruct = NULL;
}


/*! \brief (Effectively unused). From Model.outList data, set a flag indicating array or not

	In reading the Model.outList, each line pertain to either a 2D array (variable) 
	or a non-array varible/parameter.  This function pertains to very early
	version of ELM (translating Stella models), and Model.outList always
	refers to variables that are arrays.
	
	\param vp struct of ViewParm w/ output configuration
	\param mask array mask
	\param value is always = 1 (is an array)
	*/
void setFlag(ViewParm* vp, UINT mask, int value )
 { 
  if(value) vp->flags |= mask; else vp->flags &= ~mask;
}

/*! \brief Unused (never called) function */
int getFlag(ViewParm* vp, UINT mask) 
{ 
  if( vp->flags & mask ) return 1; else return 0;
}

/*!  \brief (Effectively unused).  */
void setPrecision(ViewParm* vp, int value )
{
  if(value/3) vp->flags |= PMASK2; else vp->flags &= ~PMASK2;
  if((value-1)%2) vp->flags |= PMASK1; else vp->flags &= ~PMASK1;
}

/*!  \brief (Effectively unused).  */
int getPrecision(ViewParm* vp)
 { 
  int rv = 1;
  if(vp->flags & PMASK2) rv +=2;
  if(vp->flags & PMASK1) rv +=1;
  return rv;
}

/*!  \brief Check for occurrences of string s1 in string s0 after position start.  

	After position start, check for occurrences of string s1 in string s0.
	
	\param s0 String array 
	\param s1 String array 
	\param start Starting position
	\param cs Case sensitivity 
	\param rp Return pointer to beginning or not
	\return location in string
	*/
int check_for(char *s0, const char *s1, int start, int  cs, int rp)
{
  /*  Check for occurrences of string s1 in string s0 	*/
  /*  after position start. Return -1 if not found.  		*/
  /* if cs = CASE -> case sensitive, cs = NOCASE, not case sens. 	*/
  /* if rp = BEG -> return position of beginning of s1, 	*/
  /* 			otherwise return position of (next char following the) end of s1 		*/
  int i, j=0, k=-1, Len1 = strlen(s1), Len0 = strlen(s0);
  char t1, t2;
  
  while(k<0) {
    k=0;
    for(i=start; i< Len0; ++i) { 
      t1 = s0[i];
      t2 = s1[j];
      if(cs==NOCASE) { t1 = tolower(t1); t2 = tolower(t2); }   
      if (t1 == t2)  j++;
      else {j=0; k=0;}
      if(j==Len1) { k=1; break; }
    }
  }
  if(k<=0) 		return(-1);
  else if(rp==BEG) 	return(i-Len1+1);
  else      		return(i+1);
}


/*!  \brief Determine if an integer is present in string.  

	\param target_str Target string 
	\return true or false (found int or not)
	*/
int isInteger (char *target_str) {

	int i=-1,first_num=0;
	char ch;

	while( (ch=target_str[++i]) != '\0' ) { 
			if( isdigit(ch) ) first_num=1;
			if( (ch=='-' || ch=='+') && first_num ) return(0);  
			if( !( isspace(ch) || isdigit(ch) || ch=='-' || ch=='+') ) return(0);  
	}
	return(1);	  	
}

/*!  \brief Determine if a float is present in string.  

	\param target_str Target string 
	\return true or false (found float or not)
	*/
int isFloat (char *target_str) {

	int i=-1,first_num=0;
	char ch;

	while( (ch=target_str[++i]) != '\0' ) { 
			if( isdigit(ch) ) first_num=1;
			if( (ch=='-' || ch=='+') && first_num ) return(0);  
			if( !( isspace(ch) || isdigit(ch) || ch=='-' || ch=='+' || ch=='.' || toupper(ch) == 'E') ) return(0);  
	}
	return(1);	  	
}


/*! \brief Send a message to a debug file or to the console

	Depending on when this function is called, messages are sent to one of two debug-related 
	files, or to the console.  During initial model set-up, the first (Driver0.out) debug-info file 
	is used to print confirmation  information on items such as init/end dates of simulation, 
	the model parameter values that were read, etc. During later access to this function while
	the model is iterating, the second (Driver1.out) file is opened and accessed here,
	followed by the X'th for multi-run sensitivity analyses.
	
	\param msg A string message for printing 
	\param wh Unused
	*/
void WriteMsg(const char* msg, int wh) {
  wh = 1;
  if(Driver_outfile) fprintf(Driver_outfile,"%s\n",msg);
  else fprintf(stdout,"%s\n",msg);
  fflush(stdout);
}

/*! \brief Send a message to the console

	The message text is not followed by a line feed. 
	\param dString A string message to print */
void usrErr0(const char* dString)
{
  fprintf(stderr,"%s", dString); 
  fflush(stderr);
}

/*! \brief Send a message to the console

	The message text is followed by a line feed. 
	\param dString A string message to print */
void usrErr(const char* dString)
{
  fprintf(stderr,"%s\n", dString); 
  fflush(stderr);
}


/*! \brief Standard exit from program */
void Exit(int code) { exit(code); }

/*! \brief Exit (code=9) from program after sending a message */
void fatal(const char *msg)
{
  printf("%s",msg);
  fflush(stdout);
  Exit(9);
}

/* The two julday & calcdate functions for going to/from julianDay/CalendarDate
   were (unmodified) from SFWMD HSM's /vol/hsm/src/libs/xmgr_julday/ directory.
   */

/*! \brief Determine the Julian calendar day from a Gregorian date
	\return julian day + hms as a real number

	Takes a date, and returns a Julian day. A Julian day is the number of
	days since some base date  (in the very distant past).
	Handy for getting date of x number of days after a given Julian date
	(use jdate to get that from the Gregorian date).
	Author: Robert G. Tantzen, translator: Nat Howard
	Translated from the algol original in Collected Algorithms of CACM
	(This and jdate are algorithm 199). All taken (unmodified) from SFWMD 
	HSM's /vol/hsm/src/libs/xmgr_julday/ directory.
	
	\param mon Month
	\param day Day
	\param year Year
	\param h Hour
	\param mi Minute
	\param se Second
	\return Julian day
	
  */
double julday(int mon, int day, int year, int h, int mi, double se)
{
  long m = mon, d = day, y = year;
  long c, ya, j;
  double seconds = h * 3600.0 + mi * 60 + se;

  if (m > 2)
    m -= 3;
  else {
    m += 9;
    --y;
  }
  c = y / 100L;
  ya = y - (100L * c);
  j = (146097L * c) / 4L + (1461L * ya) / 4L + (153L * m + 2L) / 5L + d + 1721119L;
  if (seconds < 12 * 3600.0) {
    j--;
    seconds += 12.0 * 3600.0;
  } 
  else {
    seconds = seconds - 12.0 * 3600.0;
  }
  return (j + (seconds / 3600.0) / 24.0);
}

/*! \brief Determine the Gregorian date from a Julian calendar day

	Julian date converter. Takes a julian date (the number of days since
	some distant epoch or other), and returns an int pointer to static space. 
	ip[0] = month; 
	ip[1] = day of month; 
	ip[2] = year (actual year, like 1977, not 77 unless it was  77 a.d.); 
	ip[3] = day of week (0->Sunday to 6->Saturday) 
	These are Gregorian.
	Copied from Algorithm 199 in Collected algorithms of the CACM
	Author: Robert G. Tantzen, Translator: Nat Howard.  All taken (unmodified) from SFWMD 
	HSM's /vol/hsm/src/libs/xmgr_julday/ directory.
	
	\param jd Julian day
	\param m Month
	\param d Day
	\param y Year
	\param h Hour
	\param mi Minute
	\param sec Second
	
  */
void calcdate(double jd, int *m, int *d, int *y, int *h, int *mi, double *sec)
{
  static int ret[4];

  long j = (long)jd;
  double tmp, frac = jd - j;

  if (frac >= 0.5) {
    frac = frac - 0.5;
    j++;
  } 
  else {
    frac = frac + 0.5;
  }

  ret[3] = (j + 1L) % 7L;
  j -= 1721119L;
  *y  = (4L * j - 1L) / 146097L;
  j = 4L * j - 1L - 146097L * *y;
  *d  = j / 4L;
  j = (4L * *d + 3L) / 1461L;
  *d = 4L * *d + 3L - 1461L * j;
  *d = (*d + 4L) / 4L;
  *m = (5L * *d - 3L) / 153L;
  *d = 5L * *d - 3 - 153L * *m;
  *d = (*d + 5L) / 5L;
  *y = 100L * *y + j;
  if (*m < 10)
    *m += 3;
  else {
    *m -= 9;
    *y += 1;
  }
  tmp = 3600.0 * (frac * 24.0);
  *h = (int) (tmp / 3600.0);
  tmp = tmp - *h * 3600.0;

  *mi = (int) (tmp / 60.0);
  *sec = tmp - *mi * 60.0;
}

/*! \brief Modulus of a pair of (double) arguments
	\param x Numerator
	\param y Denominator
	\return modulus (float) result */
float FMOD(float x, float y) { 
return (float)fmod((double)x, (double)y); 
} 

/*! \brief Get a float value following a specific string
	\param inFile File that is open
	\param lString The string being read
	\param fValPtr The float number found */
void getFloat(FILE* inFile, const char* lString, float* fValPtr)
{
  int test, fSize;
  UCHAR iEx=0;
  if(lString)  scan_forward(inFile,lString);
  test = fscanf(inFile,"%f",fValPtr);
  if(test != 1) {
    fprintf(stderr,"Read Error (%d):",test);
    if(lString) fprintf(stderr,"%s\n",lString);
    *fValPtr = 0.0;
    iEx = 1;
  }
  if (iEx) exit(0);
}

/*! \brief Get a character following a specific string
	\param inFile File that is open
	\param lString The string being read
	\param cValPtr The char value found */
void getChar(FILE* inFile, const char* lString, char* cValPtr)
{
  int test;
  UCHAR iEx=0;
  if(lString)  scan_forward(inFile,lString);
  test = fscanf(inFile,"%c",cValPtr);
  if(test != 1) {
    fprintf(stderr,"Read Error (%d):",test);
    if(lString) fprintf(stderr,"%s\n",lString);
    iEx = 1;
    *cValPtr = '\0';
  }
  if (iEx) exit(0);
}


/*! \brief Get a string following a specific string
	\param inFile File that is open
	\param lString The string being read
	\param inString The string found */
void getString(FILE* inFile, const char *lString, char *inString)
{
  int test;
  UCHAR iEx=0;
  if(lString)  scan_forward(inFile,lString);
  test = fscanf(inFile,"%s",inString);
  if(test != 1) {
    fprintf(stderr,"Read Error (%d):",test);
    if(lString) fprintf(stderr,"%s\n",lString);
    iEx = 1;
  }
  if (iEx) exit(0);
}

/*! \brief Get an integer following a specific string
	\param inFile File that is open
	\param lString The string being read
	\param iValPtr The char value found */
void getInt(FILE* inFile, const char* lString, int* iValPtr)
{
  int test;
  UCHAR iEx=0;
  if(lString)  
    scan_forward(inFile,lString);
  test = fscanf(inFile,"%d",iValPtr);
  if(test != 1) {
    fprintf(stderr,"Read Error (%d):",test);
    if(lString) fprintf(stderr,"%s\n",lString);
    *iValPtr = 0;
    iEx = 1;
  }
  if (iEx) exit(0);
}

/*! \brief Find a character  
	\param infile The file that is open
	\param tchar The character being sought
	\return true/false (0 or 1) */
int find_char(FILE *infile, char tchar)
{
  char test = '1', cchar = '#';
  int in_c=0;

  while( test != EOF ) {
    test = fgetc(infile);
    if(test == tchar && in_c==0 ) { return 1; }
    if(test == cchar ) in_c=1;
    if(test == '\n' )  in_c=0;
  }
  return(0);
}

/*! \brief truncate (not rounding) to an integer  
	\param x the value being operated on
	\return result */
int Round(float x)
{ 
  int i = (int)x;
  return (i);
}

/*! \brief Skip ahead in a string until next field  
	\param s A character array
	\param SYM The specific symbol being used to skip over
	\return Location in string
	*/
char *Scip(char *s, char SYM ) 
{
  if(*s == SYM ) return ++s;
  while(*s != SYM && *s != EOS ) s++;
  if(*s != EOS) return ++s;
  else  return s;
}


/*! \brief Skip white space(s) in a file
	\param infile
	\return success/failure */
int skip_white(FILE *infile)
{
  int ch;
  
  while( isspace(ch=fgetc(infile)) ) {;}
  if(ch==EOF) return 0;
  ungetc(ch,infile);
  return 1;		
}  

/*! \brief Scan forward until a particular string is found  
	\param infile The file being read
	\param tstring The sought-after string 
	\return Success/failure */
int scan_forward ( FILE *infile, const char *tstring) 
{
  int sLen, i, cnt=0;
  char Input_string[100], test;
  
  sLen = strlen(tstring);
  while( ( test = fgetc(infile) ) != EOF ) {
    for(i=0; i<(sLen-1); i++) 
	Input_string[i] = Input_string[i+1];
    Input_string[sLen-1] = test;
    Input_string[sLen] = '\0';
    if(++cnt >= sLen) {
      test =  strcmp(Input_string,tstring);
      if( test == 0 ) return 1;
    }
  }
  return(-1);
}


/*! \brief Get the var name from a longer path/filename  
	\param name The var name & path
	\return Pointer to var name alone */
char* name_from_path(char* name)
{
  char* namePtr; int i, slen;
  char dirCh;
  
  namePtr = name;
  dirCh = '/';
  slen = strlen(name);
  
  for(i=0; i<slen; i++) {
  	if( name[slen-i-1] == dirCh ) { namePtr = name+slen-i; break; }
  }
  return namePtr;  
}


/*! \brief Allocate memory for a variable 
	\param mem_size The size of memory space
	\param var_name The variable's name
	\return Pointer to that memory */
VOIDP nalloc(unsigned mem_size, const char var_name[])
{
  VOIDP rp;

  
  if(mem_size == 0) return(NULL);
  rp = (VOIDP)malloc( mem_size );
  total_memory += mem_size;
  fasync(stderr);
  if( rp == NULL ) {
    fprintf(stderr,"Sorry, out of memory(%d): %s\n",mem_size,var_name);
    Exit(0);
  }
  fmulti(stderr);
  return(rp);
}

/*! \brief Unused, never called. Normal distribution */
float Normal(float mean,float sd)
{
    int table_loc ;
    double rand_num ;
    float sign ;
    float interval = .1 ;
    int n_interval = 40 ;
    float high, low ;
    float return_val = 0.0;

    sign = SMDRAND(0.0, 1.0) ;
    sign = (sign < .5 ? -1 : 1) ;
    rand_num = SMDRAND(.5, 1.0);
    low = gRTable[0] ;
    for (table_loc=1; table_loc<n_interval; table_loc++)
    {
        high = gRTable[table_loc] ;
        if (high > rand_num + .5)
        {
            return_val = mean + sd * (sign  * interval * (table_loc - 1 +
                                                          (rand_num+.5-low)/(high-low))) ;
            return(return_val) ;
        }
        low = high ;
    }
    return(return_val) ;
}

/*! \brief Unused, never called. Poisson distribution */
int Poisson(float mu) 
{
  int ix;
  float f0, r, Lp = 1, xf = 1, p=0;
  
  p = f0 = Exp(-mu);
  r = SMDRAND(0.0,1.0);
  
  for( ix=0; ix<500; ix++) {
    if( r < p ) return ix;
    Lp *= mu;
    if(ix>0) xf *= ix;
    p += (Lp*f0)/xf;
  }
  return ix;    
}



/*! \brief Effectively unused in serial (non-parallel) implementation */
void link_edges(VOIDP Map, unsigned char Mtype) 
{
  switch(Mtype) {
  case 'f' :	
    exchange_borders((byte*)Map,sizeof(float)); break;
  case 'd' : case 'i' :	
    exchange_borders((byte*)Map,sizeof(int)); break;
  case 'c' :	
    exchange_borders((byte*)Map,sizeof(UCHAR)); break;
  }	
}

/*! \brief Effectively unused in serial (non-parallel) implementation */
void setup_platform() {
 
  exparam(&env);
  procnum = env.procnum;
  exgridsplit(env.nprocs, 2, nprocs);
  if( exgridinit(2, nprocs) < 0) { usrErr("Failed to Setup Grid"); exit(0); }
  exgridcoord(procnum, recpnum);
}


/****************************************************/

