/*! \file Generic_Driver.c
\brief The main model-driving functions. Vroom vroom.

This source file contains the main calling function to drive the model, including initial set-up: \n
	main:  Main program loop \n
	setup: Controller to call functions to set up & initialize the model \n
	get_parmf: Get user's run-time parameters from configuration file \n
	track_time: Keep track of simulation time \n

Note: documented with Doxygen, which expects specific syntax within special comments. \n
	
The Everglades Landscape Model (ELM). \n
last updated: Jan 2005 \n
*/

/* The Open Source ELM uses the Spatial Modeling Environment (SME) code developed by Tom Maxwell - see the ELM web site for latest contact information.
   General notes on revisions to this source file. 
       Nov/Dec 2004 v2.3.2: documentation upgrade 
       		- Re-organized (incl. header), clarified scopes, Doxygen tags added 
       		- Functionality same as v2.1/v2.2 
       			a) Moved handful of model parameter inputs to GlobalParms (read in from UnitMod.c) 
       			b) Corrected simulation real-time clocking 
       Jan 2005 v2.4.0: multi-run capability 
       Jul 2006 ELM code and data application release, v2.5.2 (no change to Generic_Driver.c since v2.4.0)
	
*/

/*! \mainpage ELM code: Main page

<b><em>For developers:</em></b>\n 
These pages on source code documentation 
show how the code works in detail.  This detail is primarily 
intended for developers who would like to further contribute to the 
OpenSource ELM project for advances in Everglades restoration. \n

<b><em>For users:</em></b>\n
Users of the ELM, who may primarily want to understand the science behind ELM 
and apply it, may be more interested in the Algorithm Documentation and the User's Guide, both found 
in the ELM Home web site's 
<A HREF="http://my.sfwmd.gov/elm" target="new">Model Development</a> 
section. Other sections of the 
<A HREF="http://my.sfwmd.gov/elm" target="new">ELM Home web site</a> 
describe the model Objectives, Results, and other relevant 
information on landscape ecology and modeling in the Everglades.

*/

#include "generic_driver.h"

/*!\brief Main program loop.

	The main controlling function used by model, which
	sets up the program and executes the driving temporal loop.
	
   \param argc number of command line arguments 
   \param argv command line argument
   \return 0
*/
int main(int argc, char* argv[])
{
    /*! \par \b Variables local to function
         \em runSeq An increasing counter of number of new simulations per program execution (sensitivity analysis)
       \n\em RunNum The number of complete simulations runs during program execution (decremented w/ new simulations)
       \n\em totRuns total number of simulations per program execution (sensitivity analysis)
       \n\em NumSRunsPerSet number of simulation runs per set of parameter-evaluations (sensitivity analysis)
       \n\em SParmNum the number of parameters to be evaluated in a sensitivity analysis
       \n\em parmID The array/list identifier of the parameters to be evaluated in a sensitivity analysis
       \n\em view A struct of ViewParm output command configuration
   	   \n\em dynERRORnum counter of cumulative number of important ERRORs in dynamic calculations (used to halt sim)
   	   \n\em begin_sim real-time beginning time of simulation
   	   \n\em end_sim real-time ending time of simulation
   	   \n\em local_begin_time struct holding real-time calendar time from standard time.h
    */
    
    int i;
    int runSeq;
    int RunNum;
    int totRuns;
    int NumSRunsPerSet = 2; /* TODO: for now its always 2 (a low and a high value of parm(s)... code not set up for diff value, though */
    int SParmNum=0; 
    int parmID;

    ViewParm  *view;
    dynERRORnum = 0;
    time_t begin_sim, end_sim;
    struct tm *local_begin_time; /* struct holding real-time calendar time from standard time.h */
    
    /* a couple of items that use the standard time.h to clock the simulation runtime*/
    begin_sim = time(NULL); 
    local_begin_time = localtime(&begin_sim);

    local_setup(argc, argv); /* not much here for serial implementation; establish processor attributes (parallel)*/
    setup();    
    alloc_memory(); 

    read_model_parameters( ProgExec->S_ParmName, ProgExec->S_ParmRelVal );
    if (SensiOn) SParmNum = SensiParm_list(ProgExec->S_ParmName);
    RunNum = numRuns(SParmNum, NumSRunsPerSet);

    init_static_data(); 

    totRuns = RunNum;
    
    /* new simulations can be started within this while ( ) (for sensitivity analyses, Position Analysis, other in future) */
    while (RunNum) {
      RunNum--; /* decrement the number of complete simulations runs during program execution (pertinent to multiple sensitivity analysis runs) */
      runSeq= totRuns - RunNum;

      sync_processors(); /* parallel only */

      open_debug_outFile(runSeq); /* the file Driver1.out (and DriverX.out for X'th subsequent sensitivity run) is opened for writing general messages
                                (warnings, errors, and data depending on debug level) */

      if (runSeq == 1) {
         usrErr0("Reading output requests...");
         view = read_output_parms();  /* reading model.outlist */
         usrErr("Done output requests.");
      }

	  if (SensiOn) { 
	     parmID = (int) (runSeq)/NumSRunsPerSet;
         ProgExec = RunList[parmID]; 

         if (runSeq > 1) {
            ProgExec->S_ParmRelVal = ProgExec->S_ParmRelVal+1 ;
            read_model_parameters( ProgExec->S_ParmName, ProgExec->S_ParmRelVal );
            reinitBIR(); /* re-init the Basin/Ind. Region datav */
         }
	     sprintf(msgStr,"\nSENSITIVITY ANALYSIS: Run %d of %d, relative-value parm ID= %d in %s set.", 
	     	runSeq, totRuns, ProgExec->S_ParmRelVal,  ProgExec->S_ParmName ); 
	     usrErr(msgStr); 
	     WriteMsg(msgStr,1); 
	     
	   }

      init_dynam_data();
      init_eqns();
      if (WatMgmtOn) init_canals(runSeq);
      if (HabSwitchOn) init_succession();

      usrErr0("Calling dynamic ecological modules:\n\t");
      for (i=0; i<NSector; i++) {
          sprintf(msgStr, "%d, ", iSector[i]);
	      usrErr0(msgStr);
      }
       
      /* for Position Analysis, where the model is re-initialized at a recurring interval */
      if(PositAnalOn)  { 
          sprintf(msgStr, "\n\t***running %2d/%2d Position Analysis*** ",mo_R_in,da_R_in); 
	      usrErr0(msgStr);
      }

	  /*  ESP (Everglades Settling-of Phosphorus, i.e., EWQModel emulation) mode.  */
      if (ESPmodeON)  usrErr0("\n\t***running ESP*** ");
      
      if (WatMgmtOn)  usrErr0("\n\twith spatial water managment 'on' ");
      else usrErr0("\n\twith spatial water managment 'off' ");

      if (HabSwitchOn)  usrErr("and succession 'on'.");
      else usrErr("and succession 'off'.");
                
      sprintf(msgStr,"\n*** Running simulation from %d/%d/%d to %d/%d/%d ***",
             yr_in,mo_in,da_in, yr_end,mo_end,da_end ); 
      usrErr(msgStr);
      
		/***************/
		/* temporal loop for driving model */
      for(istep=0; istep<N_iter; istep++) {

        track_time(istep);
        
    	/* when testing/debugging new code/data, allow critical ERRORs to accumulate to aid in identifying problem - then get out.
    		This exit is particularly important with new users, who may configure data sets improperly, resulting in a model
    		that runs merrily along, but accumulates a massive DriverX.out (for X'th simulation run) file w/ errors (exceeding file size limit of OS) */
    	if (dynERRORnum > 500) {
    		sprintf(msgStr,"\n\n***Sorry! Exiting because of many (%d) ERROR and/or capacityERR messages.\n***See the %s/%s/Debug/DriverX.out (for X'th simulation run) file to determine how to fix things.",dynERRORnum, OutputPath, ProjName);
    		usrErr(msgStr);
    		exit(-1);
    		}
        
        if (PositAnalOn)  { /* for Position Analysis, where the model is re-initialized at a recurring interval */
          if ( (SimTime.mo[0] == mo_R_in) && (SimTime.da[0] == da_R_in) ) {
            usrErr0("\nPOSITION ANALYSIS: Re-initializing...");
            init_dynam_data(); 
            init_eqns(); 
            reinitCanals();
            reinitBIR(); /* missing init_succession */
            usrErr("Done re-initializing.");
          }
        }    
        if(debug >3) { /* long console output at this high debug level */
            sprintf(msgStr,"---------------------------------------Iteration = %d\n",istep);
            WriteMsg(msgStr,1); usrErr(msgStr); 
        }
        else
          {  /* print the year/month/day to console - this will be appended with horizontal iteration info later */
            sprintf(msgStr,"\r%d/%2d/%2d:   ",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); 
	          usrErr0(msgStr); /* calendar date */
        }

        /* call dynamic calculations */
        for(i=0; i<NSector; i++) { /* for each dynamic ecological (including hydrology) sector(module) */
          
          if(debug >3) { /* long console output at this high debug level */
              sprintf(msgStr,"Running cell dyn %d of %d\n",iSector[i],NSector); 
              WriteMsg(msgStr,1); 
              usrErr(msgStr); 
          }
          call_cell_dyn(iSector[i],istep); /* invoke the UnitMod.c calling function for dynamic ecological modules */
        }
        
        /* generate output (incl. point time series) */
        if((istep >= 0 && istep <= N_iter) ) {
            if(debug >3) { /* long console output at this high debug level */
              usrErr(""); /* carriage return following hydstep iterations */
              sprintf(msgStr,"(%d)Generating Output",procnum); WriteMsg(msgStr,1); usrErr(msgStr); 
            }
          gen_output(istep, view);  /* the generates all spatial output */
        }
            
        /* send point time series output to disk at this interval */
        if ( FMOD(SimTime.TIME, 180.0) == 0 ) { 
	        if (istep == 0) open_point_lists(pSeries,MAX_PTSERIES);  /* at start of sim, open point timeseries files */
		    send_point_lists2(pSeries,MAX_PTSERIES); 
        }
      }  /* end of temporal loop for driving model */
		/***************/

		/* send the remaining time accumulated of point time series output to disk at end of simulation */
      send_point_lists2(pSeries,MAX_PTSERIES); 
      
      end_sim = time(NULL);
      /* print the simulation start date:time, and the elapsed cpu time, to DriverX.out (for X'th simulation run) file (& elapsed to console) */
      sprintf(msgStr,"########\nThe simulation(s) started on: %s", asctime(local_begin_time) ); WriteMsg(msgStr,1);
      sprintf(msgStr,"\nEND.  The simulation(s) took %8.3f minutes to run using your %s OS box.\n########",( (end_sim-begin_sim)/60.0), OS_TYPE );  WriteMsg(msgStr,1); usrErr(msgStr);

    } /* end of the while (RunNum) and thus end of simulation(s) (plural if running sensitivity) */
    
return 0;
} /* end of main and thus program execution */

/*****************************************************************************/
/*!\brief Controller to call functions to set up & initialize the model.

	Several function calls here effectively do nothing in serial (i.e.,
	non-parallel) implementations.  */
void setup()
{
  int i;
  for(i=0; i<MAX_PTSERIES; i++)  pSeries[i].data = NULL; /* initialize Point Time Series arrays */
  setup_platform(); /* effectively unused in serial (non-parallel) implementation */
  set_env_vars();   /* get necessary enviroment variables */ 
  get_parmf();      /* get the run-time parameters  */ 
  get_map_dims();   /* get the row-column dimensions of the global model array size */
  setup_grid();     /* set up a grid size used in memory allocations (other stuff for parallel) */
}

/****************************************************************************/

/*!\brief Get user's run-time parameters from configuration file.

	This is where the application is set up by the user's edits to
	the Driver.parm text configuration file, providing the simulation
	start and end dates, grid cell size, debug level, model scenario name, etc. */

void get_parmf () 
{
  /*!	\par \b Variables local to function
     	\n\em stop boolean flag to stop reading sector (celldyn module) numbers
     	\n\em ch character used in delineating the reading of sector (celldyn module) numbers/syntax
     	\n\em infile an input file pointer representing the Driver.parm file
     	\n\em filename the filename, including path, of the Driver.parm file
     	\n\em ss a string being parsed to read model configuration information from the Driver.parm file
     	\n\em Re_initDateRead data of simulation re-initialization (Position Analysis mode)
     	\n\em endDateRead date of end of simulation
  */

    int i=0, stop=0;
    unsigned char ch;
    FILE *infile;
    char filename[120], ss[120]; 
	char Re_initDateRead[15], endDateRead[15]; 
	int SParmNum_TEMP = 150; /* TODO: will get this from reading the senitivity-parameter list */
	
    sprintf(filename,"%s/%s/RunParms/Driver.parm",ModelPath,ProjName);
    infile = fopen(filename,"r");
    if(infile==NULL) { fprintf(stderr,"Error, can't open file %s",filename); exit(0); }

     
    fgets(ss,120,infile);
    sscanf(ss,"%s",outpath);    
    OutputPath = (char*)&outpath; /* path for all output */
    open_debug_outFile(0); /* opens the Driver0.out debug file */

    /* start date  */
    fgets(ss,120,infile);
    sscanf(ss,"%s",initDateRead); /* read in as yyyy/mm/dd format */

    sprintf(msgStr,"initDateRead=%s\n",initDateRead); WriteMsg(msgStr,1); 
    sscanf(initDateRead, "%4d/%2d/%2d,",&yr_in,&mo_in,&da_in); /* starting date of sim */

    /* julian day, returns a double (args = month, day, year, hour, minute, second) */
    Jdate_init = julday(mo_in, da_in, yr_in, hr_in, mi_in, se_in); 


     /* end date  */
    fgets(ss,120,infile);
    sscanf(ss,"%s",endDateRead); /* read in as yyyy/mm/dd format */

    sprintf(msgStr,"endDateRead=%s\n",endDateRead); WriteMsg(msgStr,1); 
    sscanf(endDateRead, "%4d/%2d/%2d,",&yr_end,&mo_end,&da_end); /* ending date of sim */
 
 	/* julian day, returns a double (args = month, day, year, hour, minute, second) */
    Jdate_end = julday(mo_end, da_end, yr_end, hr_in, mi_in, se_in); 

    /* re-initialization date  */
    fgets(ss,120,infile);
    sscanf(ss,"%s",Re_initDateRead); /* read in as mm/dd format */

    sprintf(msgStr,"Re_initDateRead=%s\n",Re_initDateRead); WriteMsg(msgStr,1); 
    sscanf(Re_initDateRead, "%2d/%2d,",&mo_R_in,&da_R_in); /* re-initialize month/day */

    if (mo_R_in>0) {
    	PositAnalOn=True;   /* in position analysis mode, we re-initialize every year on same month/day */
        sprintf(msgStr,"\n *** WARNING: Position Analysis capabilities are NOT VERIFIED FOR accuracy/consistency in ELMv2.3 - the Avg's, Basin/Indicator-Region output may not be reliable ***\n"); 
        WriteMsg(msgStr,1); 
        usrErr(msgStr); 
	}	/* TODO: fix the budget/stats re-inits for Position Analysis mode */ 

    fgets(ss,120,infile);
    sscanf(ss,"%s",modelName);    
    sprintf(msgStr, "Model Name= %s", modelName); usrErr(msgStr); WriteMsg(msgStr,1);
    getString(infile,"Model version=",modelVers); /* model version number (eg., v.2.1) */
    sprintf(msgStr,"Model version=%s\n",modelVers); WriteMsg(msgStr,1);  

    getFloat(infile,"CellArea=",&CELL_SIZE);
    sprintf(msgStr,"CellArea=%f\n",CELL_SIZE); WriteMsg(msgStr,1); 
    celWid = sqrt(CELL_SIZE); /* cell width, used in number of hydro calcs */
    sq_celWid = sqrt(celWid); /* square root of cell width, used in number of hydro calcs */
    

    getFloat(infile,"budg_Intvl=",&budg_Intvl);
    sprintf(msgStr,"budg_Intvl=%f\n",budg_Intvl); WriteMsg(msgStr,1); 
    budgCalendar = ( budg_Intvl == 0.0 ) ? (True) : (False); /* true/false flag signifying use of gregorian calendar for budget calcs
    										a budg_Intvl of 0 days defaults to gregorian calendar-month
    										intervals ( = #days/month, diff among months/years) 
    										In this case, the true/false budgCalendar flag is set to True */

    getFloat(infile,"avg_Intvl=",&avg_Intvl);
    sprintf(msgStr,"avg_Intvl=%f\n",avg_Intvl); WriteMsg(msgStr,1); 
    if ( avg_Intvl == 0 ) avgCalendar = True; /* true/false flag signifying use of gregorian calendar for recurring averaging calcs
    										an avg_Intvl of 0 days defaults to gregorian calendar-month
    										intervals ( = #days/month, diff among months/years) 
    										In this case, the true/false avgCalendar flag is set to True */

    getFloat(infile,"can_Intvl=",&can_Intvl);
    sprintf(msgStr,"can_Intvl=%f\n",can_Intvl); WriteMsg(msgStr,1); 
    if ( can_Intvl == 0 ) canalCalendar = True; /* true/false flag signifying use of gregorian calendar for recurring canal (watmanage) calcs
    										a can_Intvl of 0 days defaults to gregorian calendar-month
    										intervals ( = #days/month, diff among months/years) 
    										In this case, the true/false can_Intvl flag is set to True */

    getInt(infile,"seed=",&seed);
    sprintf(msgStr,"seed=%d\n",seed); WriteMsg(msgStr,1);  
    getFloat(infile,"dt=",&dt);
    sprintf(msgStr,"dt=%f\n",(double)dt); WriteMsg(msgStr,1);  
    getInt(infile,"hyd_iter=",&hyd_iter); 
    sprintf(msgStr,"hyd_iter=%d\n",hyd_iter); WriteMsg(msgStr,1); 

    sfstep = dt/hyd_iter;     /* time step for cell-cell overland flows */
    gwstep = dt/(hyd_iter/2); /* time step for cell-cell ground water flows (twice as long as surface water)*/
    canstep = dt/hyd_iter;             /* time step for canal flows */
    step_Cell = sq_celWid * sfstep/CELL_SIZE * sec_per_day; /* constant used in horizontal surface water raster flux equations */

    N_iter= (Jdate_end - Jdate_init + 1)/dt;
	sprintf(msgStr,"N_iter=%d\n",N_iter); WriteMsg(msgStr,1); 
    PORnumday = (int)(Jdate_end - Jdate_init + 1);   /* number of days of simulation Period Of Record */

    getInt(infile,"debug=",&debug);
    sprintf(msgStr,"debug=%d\n",debug); WriteMsg(msgStr,1);  
    getInt(infile,"debug_point=",&dbgPt.x);
    getInt(infile,NULL,&dbgPt.y);
    sprintf(msgStr,"debug point= (%d,%d)\n",dbgPt.x,dbgPt.y); WriteMsg(msgStr,1);  
	
	alloc_mem_runs(SParmNum_TEMP);
	RunList[0] = ProgExec; /* nominal parameter set */
	
    getString(infile,"S_ParmName=",ProgExec->S_ParmName); /* value can be: 
    					1) "NONE", for no sensitivity analysis (nominal parameter set);
    					2) the name of a global or habitat-specific parameter for a single-parm sensitivity analysis; 
    					3) "ALL", leading to multi-parameter sensitivity analyses on 
    					   all parameters, to be read from an input list */
    sprintf(msgStr,"S_ParmName=%s\n",ProgExec->S_ParmName); WriteMsg(msgStr,1); 

    SensiOn = (strcmp(ProgExec->S_ParmName,"NONE")!=0) ? (True) : (False);
    ProgExec->S_ParmRelVal = 0; /* first run of program is ALWAYS the nominal (no parameter-changes) run */

	getInt(infile,"HabSwitchOn=",&HabSwitchOn); /* flag to turn on habitat switching */
    sprintf(msgStr,"HabSwitchOn=%d\n",HabSwitchOn); WriteMsg(msgStr,1); 
    getInt(infile,"WatMgmtOn=",&WatMgmtOn); /* flag to turn on canal/water management */
    sprintf(msgStr,"WatMgmtOn=%d\n",WatMgmtOn); WriteMsg(msgStr,1); 

    getString(infile,"Scenario=",SimAlt); /* simulation's scenario/alternative */
    sprintf(msgStr,"SimAlt=%s\n",SimAlt); WriteMsg(msgStr,1);  
    getString(infile,"Scenario modifier=",SimModif); /* simulation's scenario/alternative & modifier */
    sprintf(msgStr,"SimModif=%s\n",SimModif); WriteMsg(msgStr,1); 

  	scan_forward(infile,"Sectors="); i=0; stop=0;
        while(1) {
	    skip_white(infile);
            ch=fgetc(infile);
            if( isdigit(ch) ) ungetc(ch,infile);
            else {
                switch (ch) {
                    case ';':  NSector = i; stop=1; break;
                    case ',':  skip_white(infile); break;
                }
            }
            if(stop) break;
            else fscanf(infile,"%d",&iSector[i++]);
                /* sector 13 is the ESP (Everglades Settling-of Phosphorus, i.e., EWQModel emulation) mode - this sets the ESPmodeON flag*/
            if (iSector[i-1] == 13) ESPmodeON=1;
            
	}
	fclose(infile);  
    broadcastInt(&NSector);
    sprintf(msgStr,"\n(%d) NSector = %d: Sectors=",procnum,NSector); WriteMsg(msgStr,1); 
    for(i=0; i<NSector; i++) {
  	broadcastInt(&iSector[i]);
	sprintf(msgStr,"(%d:%d)",i,iSector[i]); WriteMsg(msgStr,1);  
    }

    for(i=0; i<MAX_PTSERIES; i++) if(pSeries[i].data) { free((char*)pSeries[i].data);  pSeries[i].data = NULL; }
}

/*!\brief Keep track of simulation time.

	The elapsed days, the gregorian and julian calendars,
	and several end-of calendar- month, year etc flags are
	updated with each iteration.  
	
   \param istep counter for number of model time iterations
*/
void track_time(int istep)
{
    int yr_tom[2];			/* calendar year of tomorrow's model time */
    int mo_tom[2];				/* calendar month of tomorrow's  model time */
    int da_tom[2];				/* calendar day of tomorrow's  model time */
    int hr_tom[2];				/* calendar hour of tomorrow's  model time (unused in model calcs) */
    int mi_tom[2];				/* calendar minute of tomorrow's  model time (unused in model calcs) */
    double se_tom[2];			/* calendar second of tomorrow's  model time (unused in model calcs) */
     
    SimTime.TIME = istep*dt;
    SimTime.Jdate = Jdate_init+SimTime.TIME; /*  increment the julian day counter */
        
        /* use calendar functions to determine the end-of status of today's date */
        calcdate( (SimTime.Jdate+1), mo_tom, da_tom, yr_tom, hr_tom, mi_tom, se_tom); /* for TOMORROW, get the calendar date info from the julian date */
        calcdate( SimTime.Jdate, SimTime.mo, SimTime.da, SimTime.yr, SimTime.hr, SimTime.mi, SimTime.se); /* for TODAY, get the calendar date info from the julian date */

		SimTime.IsMonthEnd = ( SimTime.mo[0] != mo_tom[0] ) ? (True) : (False); /* true/false flag signifying end of a gregorian calendar month */
		SimTime.IsYearEnd =  ( SimTime.yr[0] != yr_tom[0] ) ? (True)  : (False); /* true/false flag signifying end of a gregorian calendar year */
		SimTime.IsWetSeasEnd = (( SimTime.mo[0] == wetEndMon ) && (SimTime.da[0] == wetEndDay))  ? (True) : (False); /* true/false flag signifying end of a gregorian calendar-based wet season */
		SimTime.IsDrySeasEnd = (( SimTime.mo[0] == dryEndMon ) && (SimTime.da[0] == dryEndDay))  ? (True) : (False); /* true/false flag signifying end of a gregorian calendar-based dry season */
		SimTime.IsSimulationEnd = ((PORnumday-1)==istep) ? (True) : (False);

        /* determine status of budget calc/printing flag (interval based on gregorian calendar or constant julian-day) */
        if (!budgCalendar) { 
        	if ( fmod(SimTime.TIME, budg_Intvl) ==0) {
        	/* not using gregorian calendar: at end of (constant) number of days, model time is end of budget interval */
        		SimTime.IsBudgEnd = True; 
         	} 
        	else { SimTime.IsBudgEnd = False; }
        }
        else {
       		if ( SimTime.IsMonthEnd ) {
        	/* using gregorian calendar: at calendar-month's end, model time is end of budget interval */
        		SimTime.IsBudgEnd = True; 
         		budg_Intvl= (float)SimTime.da[0]; /* dynamic interval becomes number of days in current gregorian-calendar month */
         	} 
       	 	else { SimTime.IsBudgEnd = False; }
        }

        /* determine status of cell-specific averages calc/printing flag (interval based on gregorian calendar or constant julian-day) */
        if (!avgCalendar) {
        	if ( fmod(SimTime.TIME, avg_Intvl) ==0) {
        	/* not using gregorian calendar: check for time (day) to finalize and print budgets for current interval */
        		avgPrint = True; }
        	else { avgPrint = False; }
        }
        else { 
       		if ( SimTime.IsMonthEnd ) {
        	/* using gregorian calendar: at month's end, it is time (day) to finalize and print budgets for current interval */
       			avgPrint = True;
         		avg_Intvl= (float)SimTime.da[0]; /* interval becomes number of days in current gregorian-calendar month */
       	 	}
       	 	else { avgPrint = False; }
        }

        /* determine status of canal data calc/printing flag (interval based on gregorian calendar or constant julian-day) */
        if (!canalCalendar) {
        	if ( fmod(SimTime.TIME, can_Intvl) ==0) {
        	/* not using gregorian calendar: check for time (day) to finalize and print canal data for current interval */
        		canPrint = True; }
        	else { canPrint = False; }
        }
        else { 
       		if ( SimTime.IsMonthEnd ) {
        	/* using gregorian calendar: at month's end, it is time (day) to finalize and print canal data for current interval */
       			canPrint = True;
         		can_Intvl= (float)SimTime.da[0]; /* interval becomes number of days in current gregorian-calendar month */
       	 	}
       	 	else { canPrint = False; }
        }
        
        SimTime.IsBudgFirst = ( (SimTime.TIME/budg_Intvl) > 1 ) ? (False) : (True);
        SimTime.IsDay0 = (SimTime.TIME==0) ? (True) : (False);
        
        /* TODO: this is to be generalized in MultiRun.c */
        if (SensiOn) {
           BIRavg_Intvl = N_iter - 1;
           SimTime.IsBIRavgEnd = ( fmod(SimTime.TIME, BIRavg_Intvl) ==0) ? (True) : (False);
           
           budg_Intvl = N_iter - 1;
           SimTime.IsBudgEnd = ( fmod(SimTime.TIME, budg_Intvl) ==0) ? (True) : (False);
        }
        else {
           BIRavg_Intvl = budg_Intvl;
           SimTime.IsBIRavgEnd = SimTime.IsBudgEnd;
        }

}
/****************************************************************************/

