/*! \file WatMgmt.c
\brief <b>Dynamic equations:</b>  Horizontal solutions of spatial raster/vector water management in canal network.

This source file contains the raster-vector solutions associated w/ management infrastructure, 
fluxing water and constituents in the canal, levee, and water control structure network. \n
The major functions in the sequence in which they are found in this file: \n 
    Canal_Network_Init (main calling function to initialize canal network) \n
    CanalReInit (position analysis mode, re-initializing canals depth/concentrations) \n
    Run_Canal_Network (main calling function for the canal-levee-structure dynamics) \n
    ReadChanStruct (read in data to define canal/levee network) \n
    ReadStructures (read in data to define water control structure attributes and structure flows) \n
    Read_schedule (read time series data for regulation schedules) \n
    Channel_configure (configure the arrays of cells associated with canals/levees) \n
    MarkCell (store canal-interacting cell coordinates and their flow attributes relative to levees) \n
    FluxChannel (dynamic fluxes of water and solutes between canal vectors and cells) \n
    f_Manning (overland flow equation as modified for canal-cell flows etc) \n
    f_Ground (groundwater flow equation as modified for canal-cell flows etc) \n
    Flows_in_Structures (dynamic fluxes of water and solutes through structures) \n
    misc small utility functions \n

Note: documented with Doxygen, which expects specific syntax within special comments. \n
	
The Everglades Landscape Model (ELM). \n
last updated: May 2006 \n
*/




/* General NOTES on revisions to this source file:
        april 00 VERSION 2.1 - output available for application for water quality performance
        july 02 VERSIOIN 2.1a - first widely-available public release of code -
              added some more code documentation, etc.
        July 03 VERSION 2.2.0 - moved declarations into header file, other similar changes
              to help reviewers better understand organization of codes.
              -Some minor code changes that are found tagged with the comment "v2.2" 
              -verified that "re-organized" v2.2.0 code is functionally the
        	  same as ELMv2.1a code
        June 04 VERSION 2.2.1 - some additions to code documentation
              -Tagged some new comments w/ "v2.2.1note" 
        Aug 2004 VERSION 2.3.0 - added dynamic boundary conditions 
        	- used new spatial time series data input for dynamic boundary conditions
        Oct 2004 VERSION 2.3.1 - internal release 
        	- checked that results reasonable, not fully checked
        Nov 2004 VERSION 2.3.2 - documentation upgrade
        	- added Doxygen tags in comments
        	- fmod, fabs
        Apr 2005 v2.4.4: added (actually redid) flexible canal output
        Aug 2005 v2.5.0: added getCanalElev function 
        	-  calculate slope of elev from the two end points of a reach (instead
        	of local elev following raster by raster elev)
        May 2006 v2.5.1: added edgeMann parameter to struct Chan
        	- allows Manning's n associated w/ edge of canal, to accomodate topographic
        	lip/berm and/or denser veg along canal length

*/
   

#include "watmgmt.h"


/****************************************************************************************/
/*! \brief Initialize the water managment network topology.

	Initialize/configure the network of canals/levees and and water control structures.  
	The canal information stored in the global array Chan_list	
	
	\param baseDatum \ref GP_DATUM_DISTANCE
	\param elev \ref SED_ELEV
*/

void Canal_Network_Init(float baseDatum, float *elev )

{ struct Structure *structs;
  int i,j;
  char filename[30];

  sprintf(filename,"CanalData");

/* Input the canals geometry file, return pointer to first vector	 	*/
/* Configure the canals, marking the cells that interact with them 		*/ 
/* and storing the pointers to the cells for each of the canals in Chan_list array */
  Channel_configure(elev, ReadChanStruct (filename));  

  for( i = 0; i < num_chan; i++ )
  {
      Chan_list[i]->P = Chan_list[i]->ic_P_con * Chan_list[i]->area * Chan_list[i]->ic_depth; /* kg/m3 * m2 * m */
      Chan_list[i]->N = Chan_list[i]->ic_N_con * Chan_list[i]->area * Chan_list[i]->ic_depth; /* kg/m3 * m2 * m */
      Chan_list[i]->S = Chan_list[i]->ic_S_con * Chan_list[i]->area * Chan_list[i]->ic_depth; /* kg/m3 * m2 * m */
  }

  usrErr ("\tCanal geometry configured OK");
  
/* Read data on the water control structures, return pointer to the first structure */
  ReadStructures  (filename, baseDatum);

  ON_MAP[T(2,2)] = 0;
  
  MCopen = (float *) nalloc(sizeof(float)*(s0+2)*(s1+2),"MCopen"); /* open-water manning's coef, used in structure cell-cell bridge flow */
  init_pvar(MCopen,NULL,'f',0.05); /* this is a hard-wired parameter - but do not do anything much with it now for the bridge flows */
  
 
 if ( debug > 0 ) 	/* Output the canal/levee modifications to ON_MAP */
 {
   	sprintf( modelFileName, "%s/%s/Output/Debug/ON_MAP_CANAL.txt", OutputPath, ProjName );
   

  if ( ( ChanInFile = fopen( modelFileName, "w" ) ) ==  NULL )
  {
     printf( "Can't open the %s canal/levee debug file!\n", modelFileName );
     exit(-1) ;
  }
  fprintf ( ChanInFile, "ROWS=%d\nCOLUMNS=%d\nFORMAT=text\nINFO=\"Debug data: Overland flow restrictions due to levees. \"\nMISSING=0\n", s0, s1 );

  for ( i = 1; i <= s0 ; i++ ) 
    { for ( j = 1 ; j <= s1 ; j++ )
         fprintf( ChanInFile, "%4d\t", ON_MAP[T(i,j)]) ;
      fprintf ( ChanInFile, "\n" );
    }   
  fclose( ChanInFile ) ;
 }


/* Open file for writing structure flows */
    { if(H_OPSYS==UNIX) 
   		sprintf( modelFileName, "%s/%s/Output/Canal/structsOut", OutputPath, ProjName );
      else sprintf( modelFileName, "%s%s:Output:structsOut", OutputPath, ProjName );
    }

	/* Open file to print structure flows to */
    if ( ( WstructOutFile = fopen( modelFileName, "w" ) ) ==  NULL )
	{
       printf( "Can't open %s file!\n", modelFileName );
       exit(-1) ;
    }
/* Print The Header Line For This File */
    fprintf ( WstructOutFile,
              "%s %s %s %s scenario: Sum of water flows at structures (ft^3/sec)\n     Date\t", &modelName, &modelVers, &SimAlt, &SimModif );
    structs = struct_first;
    while ( structs != NULL )
    {   fprintf( WstructOutFile, "%7s\t", structs->S_nam); 	 
	    structs = structs->next_in_list;
    }

/* Open file for writing structure P concs */
    { if(H_OPSYS==UNIX) 
   		sprintf( modelFileName, "%s/%s/Output/Canal/structsOut_P", OutputPath, ProjName );
      else sprintf( modelFileName, "%s%s:Output:structsOut_P", OutputPath, ProjName );
    }

	/*  Open file to print structure flows to */
    if ( ( WstructOutFile_P = fopen( modelFileName, "w" ) ) ==  NULL )
	{
       printf( "Can't open %s file!\n", modelFileName );
       exit(-1) ;
    }
/* Print The Header Line For This File */
    fprintf ( WstructOutFile_P,
              "%s %s %s %s scenario: Flow weighted mean TP conc at structures ( mg/L) \n     Date\t", &modelName, &modelVers, &SimAlt, &SimModif );
    structs = struct_first;
    while ( structs != NULL ) 
    {   fprintf( WstructOutFile_P, "%7s\t", structs->S_nam); 	  
	    structs = structs->next_in_list; 
    }

/* Open file for writing structure Salin concs */
    { if(H_OPSYS==UNIX) 
   		sprintf( modelFileName, "%s/%s/Output/Canal/structsOut_S", OutputPath, ProjName );
      else sprintf( modelFileName, "%s%s:Output:structsOut_S", OutputPath, ProjName );
    }

	/*  Open file to print structure flows to */
    if ( ( WstructOutFile_S = fopen( modelFileName, "w" ) ) ==  NULL )
	{
       printf( "Can't open %s file!\n", modelFileName );
       exit(-1) ;
    }
/* Print The Header Line For This File */
    fprintf ( WstructOutFile_S,
              "%s %s %s %s scenario: Flow weighted mean tracer concentration at structures ( g/L) \n     Date\t", &modelName, &modelVers, &SimAlt, &SimModif );
    structs = struct_first;
    while ( structs != NULL ) 
    {   fprintf( WstructOutFile_S, "%7s\t", structs->S_nam); 	  
	    structs = structs->next_in_list; 
    }

/****/
    
	/* Open file to print canal stage info to */
    { if(H_OPSYS==UNIX) 
   		sprintf( modelFileName, "%s/%s/Output/Canal/CanalOut", OutputPath, ProjName );
      else sprintf( modelFileName, "%s%s:Output:CanalOut", OutputPath, ProjName );
    }

    if ( ( CanalOutFile = fopen( modelFileName, "w" ) ) ==  NULL )
	{
       printf( "Can't open %s file!\n", modelFileName );
       exit(-1) ;
    }

    fprintf ( CanalOutFile, "%s %s %s %s scenario: Instantaneous water depth (meters, from bottom of canal) in canal Reaches\n    Date\t", &modelName, &modelVers, &SimAlt, &SimModif ); /* print the header line for this file */
    /*  channels with negative widths are reserved for new canals */
    /* v2.2 put a "R_" prefix in front of the canal ID# */
    for( i = 0; i < num_chan; i++ ) {if (Chan_list[i]->width > 0) fprintf ( CanalOutFile, "R_%d\t",Chan_list[i]->number);}
    

	/* Open file to print canal phosph info to */
    { if(H_OPSYS==UNIX) 
   		sprintf( modelFileName, "%s/%s/Output/Canal/CanalOut_P", OutputPath, ProjName );
      else sprintf( modelFileName, "%s%s:Output:CanalOut_P", OutputPath, ProjName );
    }

    if ( ( CanalOutFile_P = fopen( modelFileName, "w" ) ) ==  NULL )
	{
       printf( "Can't open %s file!\n", modelFileName );
       exit(-1) ;
    }

    fprintf ( CanalOutFile_P, "%s %s %s %s scenario: Instantaneous TP conc (mg/L) in canal Reaches\n    Date\t", &modelName, &modelVers, &SimAlt, &SimModif ); /* print the header line for this file */
    /* v2.2 put a "R_" prefix in front of the canal ID# */
    for( i = 0; i < num_chan; i++ ) {if (Chan_list[i]->width > 0) fprintf ( CanalOutFile_P, "R_%d\t",Chan_list[i]->number);}
    
	/* Open file to print canal salinity/tracer info to */
    { if(H_OPSYS==UNIX) 
   		sprintf( modelFileName, "%s/%s/Output/Canal/CanalOut_S", OutputPath, ProjName );
      else sprintf( modelFileName, "%s%s:Output:CanalOut_S", OutputPath, ProjName );
    }

    if ( ( CanalOutFile_S = fopen( modelFileName, "w" ) ) ==  NULL )
	{
       printf( "Can't open %s file!\n", modelFileName );
       exit(-1) ;
    }

    fprintf ( CanalOutFile_S, "%s %s %s %s scenario: Instantaneous tracer conc (g/L) in canal Reaches\n    Date\t", &modelName, &modelVers, &SimAlt, &SimModif ); /* print the header line for this file */
    /* v2.2 put a "R_" prefix in front of the canal ID# */
    for( i = 0; i < num_chan; i++ ) {if (Chan_list[i]->width > 0) fprintf ( CanalOutFile_S, "R_%d\t",Chan_list[i]->number);}
    
 

/* Open file for canal-cell flux debugging purposes */
  if( debug > 0 ) 
  { 
     if(H_OPSYS==UNIX) 
   		sprintf( modelFileName, "%s/%s/Output/Debug/Can_debug", OutputPath, ProjName );
      else sprintf( modelFileName, "%s%s:Output:Can_debug", OutputPath, ProjName );
    

	/* Open file for debug */
    if ( ( canDebugFile = fopen( modelFileName, "w" ) ) ==  NULL )
	{
       printf( "Can't open %s file!\n", modelFileName );
       exit(-1) ;
    }
  
     fprintf ( canDebugFile, "Depending on level of the debug parameter, data on canal-cell flows printed here. \n" ); /* v2.5 made use of this file pointer */
     fflush(canDebugFile); 
    
  }
  
  return;

}

/********************************************************************************************/
/*! \brief Re-initialize the depths and concentrations in canals under the Positional Analysis mode */

void CanalReInit()
{
    int i;
    for( i = 0; i < num_chan; i++ )

    {
        Chan_list[i]->wat_depth = Chan_list[i]->ic_depth;
        Chan_list[i]->P = Chan_list[i]->ic_P_con * Chan_list[i]->area * Chan_list[i]->ic_depth; /* kg/m3 * m2 * m */
        Chan_list[i]->N = Chan_list[i]->ic_N_con * Chan_list[i]->area * Chan_list[i]->ic_depth; /* kg/m3 * m2 * m */
        Chan_list[i]->S = Chan_list[i]->ic_S_con * Chan_list[i]->area * Chan_list[i]->ic_depth; /* kg/m3 * m2 * m */
         
    }
}




/********************************************************************************************/
/*! \brief Runs the water management network.

	Invoke the calls to water control structure flows and the canal-cell interactions.
	\param SWH \ref SURFACE_WAT
	\param Elevation \ref SED_ELEV
	\param MC \ref HYD_MANNINGS_N
	\param GW \ref SAT_WATER
	\param poros \ref HP_HYD_POROSITY
	\param GWcond \ref HYD_RCCONDUCT
	\param NA \ref DINdummy
	\param PA \ref TP_SF_WT
	\param SA \ref SALT_SURF_WT
	\param GNA \ref DINdummy
	\param GPA \ref TP_SED_WT
	\param GSA \ref SALT_SED_WT
	\param Unsat \ref UNSAT_WATER
	\param sp_yield \ref HP_HYD_SPEC_YIELD

*/

void Run_Canal_Network ( float *SWH, float *Elevation, float *MC, float *GW, float *poros, 
	                 float *GWcond, double *NA, double *PA, double *SA, double *GNA, double *GPA, double *GSA,
                         float *Unsat, float *sp_yield) 

{ int i;
 float CH_vol;

 if ( canPrint && (( N_c_iter++ % hyd_iter ) == 0.0) ) {
 	if (SensiOn) { 
 	    printchan = (SimTime.IsSimulationEnd) ? (1) : (0); /* flag to indicate to print canal/struct data */
 	 }
 	 else printchan = 1;
 }
 else  printchan = 0;

 if (printchan == 1) {
     
     fprintf( WstructOutFile, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
     fprintf( WstructOutFile_P, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
/*      fprintf( WstructOutFile_N, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); */
     fprintf( WstructOutFile_S, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );

     fprintf (CanalOutFile, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
     fprintf (CanalOutFile_P, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
/*      fprintf (CanalOutFile_N, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] ); */
     fprintf (CanalOutFile_S, "\n%d/%d/%d\t",SimTime.yr[0],SimTime.mo[0],SimTime.da[0] );
 }
    
  
 for ( i = 0; i < num_chan; i++ )
 {   /* set the sum of historical/other-data flows to/from a canal per iteration to 0 */	
     Chan_list[i]->sumHistIn = Chan_list[i]->sumHistOut = 0.0;
     Chan_list[i]->sumRuleIn = Chan_list[i]->sumRuleOut = 0.0;
 }

/* calculate flows through water control structures */
Flows_in_Structures ( SWH, Elevation, MC, NA, PA, SA ); 
  
/* output  header for canal-cell flux output canal debug file, at relatively high debug levels (writes a LOT of stuff) */
 if( debug > 3 ) 
 {
     fprintf( canDebugFile, "N%3d  WatDepth     Pconc     Sconc        flux_S        flux_G        flux_L           Qin          Qout #_iter      Area    error \n", N_c_iter );
      
 }

/* Flux the water along a canal using the relaxation procedure */
  
 for( i = 0; i < num_chan; i++ )
     if (Chan_list[i]->width > 0) /* a neg width canal is skipped */

     { 
         FluxChannel ( i, SWH, Elevation, MC, GW, poros, GWcond, NA, PA, SA, GNA, GPA, GSA, Unsat, sp_yield );
             /* calculate total canal volume after canal fluxes */
         CH_vol = Chan_list[i]->area*Chan_list[i]->wat_depth; 

             /* sum the storages on iteration that performs budget checks */
	 if ( !( FMOD(N_c_iter,(1/canstep) )) && (SimTime.IsBudgEnd ) ) { 
             TOT_VOL_CAN[Chan_list[i]->basin] += CH_vol;
             TOT_VOL_CAN[0] += CH_vol;
             TOT_P_CAN[Chan_list[i]->basin] += Chan_list[i]->P;
             TOT_P_CAN[0] += Chan_list[i]->P;
             TOT_S_CAN[Chan_list[i]->basin] += Chan_list[i]->S;
             TOT_S_CAN[0] += Chan_list[i]->S;
         }
                                                                                               
     
         if ( printchan )
         {
                 /* report N&P concentration in mg/L (kg/m3==g/L * 1000 = mg/L) */
             Chan_list[i]->P_con = (CH_vol > 0) ? (Chan_list[i]->P/CH_vol*1000.0) : 0.0;
/*		Chan_list[i]->N_con = (CH_vol > 0) ? (Chan_list[i]->N/CH_vol*1000.0) : 0.0; */
             Chan_list[i]->S_con = (CH_vol > 0) ? (Chan_list[i]->S/CH_vol       ) : 0.0;
 
             /* v2.2 increased decimal places in output (from 6.3f to 7.4f) */
             fprintf( CanalOutFile, "%6.2f\t", Chan_list[i]->wat_depth );
             fprintf( CanalOutFile_P, "%7.4f\t", Chan_list[i]->P_con );
/*          fprintf( CanalOutFile_N, "%7.4f\t", Chan_list[i]->N_con ); */
             fprintf( CanalOutFile_S, "%7.4f\t", Chan_list[i]->S_con ); 
         }
     } 
 if ( printchan )
 {
     fflush(CanalOutFile);
     fflush(CanalOutFile_P);
/*       fflush(CanalOutFile_N); */
     fflush(CanalOutFile_S);
 }
  
 return;
}

	          

/********************************************************************************************/
/*! \brief Reads the information about canals from data file.

	\param filename Canal/levee attributes data file.
	\return chan_first Pointer to first canal data struct

*/
struct Chan *ReadChanStruct ( char* filename )
{ 
 struct Chan *channels, *chan_first, *chan_last;
 struct Chan_reach *C_reach, *last_reach, *next_reach;
 int i, index, firstReach, firstCanal = 1;
 char ss[422], scenario[20], modnam[20], bas_txt[8];
 float C_x, C_y, C_x1, C_y1;

 
 int ibas, foundBasn;
 

 { if(H_OPSYS==UNIX) 
     sprintf( modelFileName, "%s/%s/Data/%s.chan", ModelPath, ProjName, filename );
 else sprintf( modelFileName, "%s%s:Data:%s.chan", ModelPath, ProjName, filename );
 }

/* Open file with canals data */
 if ( ( ChanInFile = fopen( modelFileName, "r" ) ) ==  NULL )
 {
     sprintf( msgStr,"Can't open %s file! ",modelFileName ) ; usrErr(msgStr);
     exit(-1) ;
 }
  
/* Allocate memory for canal reach structure. A canal reach is the straight canal segment */
/* defined in the canal data file by the coordinates of its firstReaching and ending points.   */
/* Each canal can be made of several canal reaches		*/
 if ( (C_reach = (struct Chan_reach *) malloc( (size_t) sizeof( struct Chan_reach ))) == NULL )
 {
     usrErr( "No memory for first canal reach\n " ) ;
     exit( -2 ) ;
 };

 num_chan = 0;
   
/* Read the general canal network information */ 
 fgets( ss, 420, ChanInFile ); sscanf( ss, "%d", &UTM_EOrigin );
 fgets( ss, 420, ChanInFile ); sscanf( ss, "%d", &UTM_NOrigin );
 fgets( ss, 420, ChanInFile );  /* skip comment line */
 fgets( ss, 420, ChanInFile ); sscanf( ss,"%s %s %d %d", &modnam, &scenario, &CHo, &CHANNEL_MAX_ITER );
 if (strcmp(scenario,SimAlt) != 0) {
     sprintf(msgStr, "The scenario (%s) found in the %s file doesn't match the one (%s) you asked for in Driver.parm!",
             scenario, modelFileName, SimAlt); usrErr(msgStr);
             exit(-1);
 }
 if (strcmp(modnam,modelName) != 0) {
     sprintf(msgStr, "The model name (%s) found in the %s file doesn't match the one (%s) you asked for in Driver.parm!",
             modnam, modelFileName, modelName); usrErr(msgStr);
             exit(-1);
 }
  
 fgets( ss, 420, ChanInFile );   /* skip comment line */
 fgets( ss, 420, ChanInFile ); sscanf ( ss, "%f %f %f ", &F_ERROR, &MINDEPTH, &C_F );
     /* the C_F is used only for sensitivity experiments */
 fgets( ss, 420, ChanInFile );   /* skip comment line */
   
/* Read canal descriptors until none left */   
 while ( fgets( ss, 420, ChanInFile ) != NULL && !feof( ChanInFile ) )
 { 
     if ( *ss == '#' ) 	/* "#" identifies each new canal in the data file */
     { /* Allocate memory for canal structure */
         if ( (channels = (struct Chan *) malloc( (size_t) sizeof( struct Chan ))) == NULL )
         {
             printf( "No memory for first canal\n " ) ;
             exit( -2 ) ;
         };
                
             /* Read canal information:  - canal number, 
                - levee marker (1 - left levee, -1 - right, 0 - none, 2 - both sides),
                - ranges of cell interaction to the left and to the right - functionality removed, should ALWAYS=1,
                - canal depth and width,
                - seepage coefficient through the levee,
                - initial S and P concentrations (doubles) in the canal water,
                - initial water depth in canal (can be >, = or < than canal depth 
                - Manning's n associated w/ edge of canal, to accomodate topographic lip/berm and/or denser veg along canal length
                - hydrologic basin that canal exchanges surface water with */
         i = sscanf( ss+1, "%d\t%d\t%d\t%d\t%f\t%f\t%f\t%f\t%f\t%f\t%f\t%s",
                     &channels->number, 
                     &channels->levee,
                     &channels->roil,   &channels->roir, 
                     &channels->depth,  &channels->width,  
                     &channels->cond,   
                     &channels->ic_S_con,  &channels->ic_P_con,  
                     &channels->wat_depth, &channels->edgeMann, &bas_txt);

         foundBasn=0;
         for (ibas=numBasn;ibas>0;ibas--) {
             basins = basn_list[ibas];
             if(strcmp(bas_txt,basins->basnTxt)==0) {
                 channels->basin = ibas;
                 foundBasn=1;
                 channels->family = basn_list[ibas]->family;
                 channels->parent = basn_list[ibas]->parent;
                     /* canals may ONLY belong to parent hydro basins, not to indicator regions */
                 if (!channels->parent) {
                     printf ("Sorry - %s's canal %d's basin is not a parent hydrologic basin! Please assign to a hydro basin in the CanalData.chan file.\n",
                            modelName,channels->number); exit (-1);}     
             }
         }
         if (foundBasn==0) { printf ("Sorry - %s's canal %d's basin does not exist! Please define the basin in the basinIR file.\n",
                            modelName,channels->number); exit (-1);}                

         channels->ic_P_con *= 0.001; /* initial input value in mg/L, convert to g/L==kg/m3 */
         channels->ic_N_con = 0.00; /* v2.2 "removed" N */ /* initial input value in mg/L, convert to g/L==kg/m3 */
         channels->ic_S_con *= 1.0; /* initial input value in g/L, no conversion */
             /* store the initial depth for re-initializing under Positional Analysis mode */
         channels->ic_depth = channels->wat_depth;
         

         channels->reaches = C_reach; 
   	 		
         if (i < 10) 	/* check for the number of input fields read */
         {printf ( " Error in CanalData.chan: reach input %d\n", (num_chan+1) ); exit (-1);}
                        
                            
         num_chan++;		/*count the number of canals (not canal reaches) */
         firstReach = 1;
   	 		
         if (firstCanal) 
         {
             firstCanal = 0;  
             chan_first = channels;
             chan_last = channels;
             continue; 
         }

         last_reach->next_reach = NULL;
         chan_last->next_in_list = channels;
         chan_last = channels;     
     }
     else  
     { /* read the pairs of coordinates for the consecutive canal reaches */
         i = sscanf ( ss, "%f %f", &C_x1, &C_y1 );
         if (i < 2) 	/* check for the number of input fields read */
         {printf ( " Error in CanalData.chan coords: %d'th reach input \n", (num_chan+1) ); exit (-1);}
                
             /* Convert from UTM to km units with 0,0 origin */
         C_x = UTM2kmx (C_x1);  C_y = UTM2kmy (C_y1);           
         C_reach->x1 = C_x;  C_reach->y1 = C_y;
			
         if ( firstReach )	   	     
         {  C_reach->x0 = C_x; C_reach->y0 = C_y; 
         firstReach = 0;
         }
         else 
         {  
				/* Allocate memory for canal reach structure */
             if ( (next_reach = (struct Chan_reach *) malloc( (size_t) sizeof( struct Chan_reach ))) == NULL )
             {	printf( "No memory for first canal reach\n " ) ;
             exit( -2 ) ;
             };
             C_reach->next_reach = next_reach;
             last_reach = C_reach;
             C_reach = next_reach;
             C_reach->x0 = C_x; C_reach->y0 = C_y;
         }   	 		 
     }	
 }/* end loop for reading */
  
 chan_last->next_in_list = NULL;
 last_reach->next_reach = NULL;
 free ((char *) C_reach);
 fclose (ChanInFile);
   
 usrErr( "\tCanal locs/attributes read OK" );
   
 return (chan_first);
}

/****************************************************************************************/
/*! \brief Read attribute data on water control structures

	\param filename Water control structures attributes file name
	\param BASE_DATUM \ref GP_DATUM_DISTANCE
*/

void ReadStructures  ( char* filename, float BASE_DATUM )
{
    /*! \par \b Variables local to function
         \em disAggNum Total # of disaggregated water control structures
       \n\em IDhist ID of historal data set
       \n\em lincnt Flow data input file record # starting at the point where the model reads values
       \n\em structName Water control structure name in time series file 
       \n\em histTP Input data field that either has TP conc (ug/L=ppb) that is constant across time or has (any) string to indicate the use of TimeSeries data from a datafile
       \n\em histTS Input data field that either has TS (salt) conc (g/L=ppt) that is constant across time or has (any) string to indicate the use of TimeSeries data from a datafile
*/

struct Structure *structs, *struct_next, *struct_last;

 char ss[322], *s;
 char sched_name[20];
 int i, j, k;
 int disAggNum=0; 
 int IDhist[MAX_H_STRUCT]; 
 int lincnt; 
 double Jdate_read;
 char lline[2001], *line;
 int yyyy, mm, dd;
 char structName[20]; 
 char scenario[20], s_modname[20];
 char histTP[5]; 
 char histTS[5]; 
 
 num_struct_hist = 0;
 numTPser = numTSser = 0;
 
 { if(H_OPSYS==UNIX) 
     sprintf( modelFileName, "%s/%s/Data/%s.struct", ModelPath, ProjName, filename );
 else sprintf( modelFileName, "%s%s:Data:%s.struct", ModelPath, ProjName, filename );
 }

/* Open file with structures data */
 if ( ( ChanInFile = fopen( modelFileName, "r" ) ) ==  NULL )
 {
     printf( "Can't open %s file!\n", modelFileName ) ;
     exit(-1) ;
 }
    
/* Allocate memory for structures */
 if ( (structs = (struct Structure *) malloc( (size_t) sizeof( struct Structure ))) == NULL )
 {
     printf( "No memory for first structure\n " ) ;
     exit( -2 ) ;
 }
 struct_first = structs;

 fgets( ss, 320, ChanInFile );
 sscanf(ss, "%s\t%s\t%s\t%s",&scenario,&scenario,&s_modname,&scenario); /*  2 junk strings followed by the model name and actual scenario ID */
 if (strcmp(scenario,SimAlt) != 0) {
         sprintf(msgStr, "The scenario (%s) found in the %s file doesn't match the one (%s) you asked for in Driver.parm!",
                 scenario, modelFileName, SimAlt); usrErr(msgStr);
     exit(-1);
  }
 if (strcmp(s_modname,modelName) != 0) {
         sprintf(msgStr, "The model name (%s) found in the %s file doesn't match the one (%s) in the canal data file!",
                 s_modname, modelFileName, modelName); usrErr(msgStr);
     exit(-1);
  }
 
/* Skip 2'nd header line */
 fgets( ss, 320, ChanInFile );


/* Read structure descriptors until none left */   
 while ( fgets( ss, 320, ChanInFile ) != NULL && !feof( ChanInFile ) )
 { 
     s = ss; 
   	 
   	 
     sscanf( s, "%d", &structs->flag );	
/* structure flag: <0 - skip structure info, structure not operating; */
/*             	   >0 - historical/other-data data array, flag=1 normally in this case, but is >1 for aggregated SFWMM structures; */
/*                  0 - rule-based structure   */
     if ( structs->flag < 0 ) continue;
     if ( structs->flag > 0 && structs->flag <10 ) {
         structs->histID = ++num_struct_hist;
         Hist_data[structs->histID-1].flag = structs->flag;
     }
     
         
/*    assign integer to the historical data ID, and increment the counter for structures with historical/other-data data to be read from datafile. */
/*    flags >= 20 represent structures (such as the S-11s) that were aggregated in SFWMM output */
/*    and need to be assigned disaggregated flows for separate ELM structures */
/*    Ex: SFWMM structure name S11 may have flag = 2 in CanalData.struct.  That S11 flow holds the sum of S-11A-C flows. */
/*    The flags on S-11A, S-11B, and S-11C would all have been assigned 20, 20, and 20 in the CanalData.struct file */

/*           NO LONGER NECESSARY: as a temporary measure until the SFWMM output .dss file is fixed, the structures that have calculated "other" partitions */
/*             (S140,S7,S8) are assigned historical flags of 9 (they are in the dss file read by a SAS routine, the flag */
/*             of 9 allows them to remain in the file, but the structure is turned off because it has flag >1)  */
         
     
     s = Scip( s, TAB );
	 
     if ( *s != TAB ) {
             sscanf ( s, "%s", &structs->S_nam );
             if ( structs->flag >0 && structs->flag <10) 
                 sscanf ( s, "%s", &Hist_data[structs->histID-1].S_nam );
             
                 /* for solute concentration of flows thru structures, */
                 /* data use flag XXser: -1== simulated, 0==static historical, 1= time-varying historical */
             s = Scip( s, TAB );
             if ( *s == TAB ) structs->TPser = -1; 
             else {
                 sscanf ( s, "%s", &histTP ); /* read the field that either has TP conc (ug/L=ppb) that is constant across time
                                                   or has (any) string to indicate the use of TimeSeries data from a datafile (read later)*/
                 if (isInteger(histTP) ) {
                      structs->TPser = 0; 
                      structs->TP = atof(histTP)*1.0e-6;  /* conc (ug/L=>g/L) that is constant across time */
                 }
                 else { structs->TPser = 1; numTPser++;} /* otherwise, we will be reading time series data later */
             }
             
             s = Scip( s, TAB );
             if ( *s == TAB ) structs->TNser = -1; /* value to indicate use of simulated, not historical, data */
             else {
                 sscanf ( s, "%f", &structs->TN );
                 structs->TNser = 0;
                 structs->TN *= 1.0e-6;/* read the TN conc (ug/L=>g/L) that is constant across time */
             }
             

             s = Scip( s, TAB );
             if ( *s == TAB ) structs->TSser = -1; /* value to indicate use of simulated, not historical, data */
             /* v2.2 added code to allow use of TimeSeries input data in addition to old method of a constant concentration */
             else {
                  sscanf ( s, "%s", &histTS ); /* read the field that either has TS (salt) conc (g/L=ppt) that is constant across time
                                                   or has (any) string to indicate the use of TimeSeries data from a datafile (read later)*/
                 if (isFloat(histTS) ) {
                      structs->TSser = 0; 
                      structs->TS = atof(histTS)*1.0;  /* conc (no conversion, g/L) that is constant across time */
                 }
                 else { structs->TSser = 1; numTSser++;} /* otherwise, we will be reading time series data later */

             }
             
             s = Scip( s, TAB );
             if ( *s == TAB ) structs->str_cell_i = 0;	/* non 0 if elevation at the structure location is desired */
             else {
                 sscanf( s, "%d", &structs->str_cell_i );
                 structs->str_cell_i++; /* this converts the data map coords to model coords (map+1) */
             }
             
             s = Scip( s, TAB );

             if ( *s == TAB ) structs->str_cell_j = 0;	/* non 0 if elevation at the structure location is desired */
             else {
                 sscanf( s, "%d", &structs->str_cell_j );
                 structs->str_cell_j++;
             }

             if ( structs->str_cell_i != 0 && !ON_MAP[T(structs->str_cell_i, structs->str_cell_j)] )
             {  printf ( "Location of water control structure %s is off-map.\n", structs->S_nam );
             exit (-2);			/*check that cells obtained are within on-map boundaries */
             }
             
             
     }
         
     s = Scip( s, TAB );

     if ( *s == TAB ) structs->canal_fr  = 0;	/* non 0 if the structure involves a donor canal */
     else sscanf( s, "%d", &structs->canal_fr  );
     s = Scip( s, TAB );
     if ( *s == TAB) structs->canal_to = 0;		/* non 0 if the structure involves a receiving canal */
     else sscanf( s, "%d", &structs->canal_to );
     s = Scip( s, TAB );

     if ( structs->canal_fr  - CHo >= num_chan || structs->canal_to - CHo >= num_chan )
     {  printf ( "Canal from/to doesn't exist for water control structure %s\n", structs->S_nam );
     exit (-2);			/* check that canals read are consistent with canals data */
     }   	 
/*  check to ensure canals associated with structures are valid */
     if ( structs->canal_fr != 0 && Chan_list[structs->canal_fr- CHo]->width <= 0.1 )
     { printf ( "The donor canal %d has been turned off for water control structure %s\n", structs->canal_fr, structs->S_nam );
     exit (-2);			
     }   	 
     if ( structs->canal_to != 0 && Chan_list[structs->canal_to- CHo]->width <= 0.1 )
     {  printf ( "The recipient canal %d has been turned off for water control structure %s\n", structs->canal_to, structs->S_nam );
     exit (-2);			
     }   	 
	 
     if (*s == TAB) structs->cell_i_fr = 0;  /* non 0 if the structure involves a donor cell */
     else 
     { sscanf( s, "%d", &structs->cell_i_fr );
     structs->cell_i_fr++;
     }	 
     s = Scip( s, TAB );
     if (*s == TAB) structs->cell_j_fr = 0;
     else 
     { sscanf( s, "%d", &structs->cell_j_fr );
     structs->cell_j_fr++;
     }
     s = Scip( s, TAB );
   	 

     if ( structs->cell_i_fr > 2 && !ON_MAP[T(structs->cell_i_fr, structs->cell_j_fr)] )
     {  printf ( "Donor cell for water control structure %s is off-map.\n", structs->S_nam );
     exit (-2);			/*check that cells obtained are within on-map boundaries */
     }

   	 /* if valid donor cell, mark it on ON_MAP */
     if ( structs->cell_j_fr ) ON_MAP[T(structs->cell_i_fr, structs->cell_j_fr)] += 100;
   	 
     if (*s == TAB) structs->cell_i_to = 0;  /* non 0 if the structure involves a recipient cell */
     else 
     { sscanf( s, "%d", &structs->cell_i_to );
     structs->cell_i_to++;
     }	 
   	 
     s = Scip( s, TAB );
     if (*s == TAB) structs->cell_j_to = 0;
     else
     { sscanf( s, "%d", &structs->cell_j_to );
     structs->cell_j_to++;
     }	 


     if ( structs->cell_i_to > 2 && !ON_MAP[T(structs->cell_i_to, structs->cell_j_to)] )
     {  printf ( "Recipient cell for water control structure %s is off-map.\n", structs->S_nam );
     exit (-2);			/*check that cells obtained are within on-map boundaries */
     }

   	 /* if valid recieving cell, mark it on ON_MAP */
         /* this used to check for cell_j_to > 2, not putting recipient LEC cells ON_MAP */
     if ( structs->cell_j_to ) ON_MAP[T(structs->cell_i_to, structs->cell_j_to)] += 100;
   	 
     s = Scip( s, TAB );		/* read the targeted water levels */
     if (*s == TAB) structs->HW_stage = 0;
     else if ( isalpha (*s) ) {  /* if there is a graph for the schedule */
         sscanf( s, "%s", sched_name );
         structs->HW_graph = Read_schedule ( sched_name, filename, BASE_DATUM );
         structs->HW_stage = -99;  
     } 
     else   { sscanf( s, "%f", &structs->HW_stage );
     structs->HW_stage += BASE_DATUM;
     }
   	 		
     s = Scip( s, TAB );
     if (*s == TAB) structs->TW_stage = 0;
     else if ( isalpha (*s) ) {  /* if there is a graph for the schedule */
         sscanf( s, "%s", sched_name );
         structs->TW_graph = Read_schedule ( sched_name, filename, BASE_DATUM );
         structs->TW_stage = -99;
     } 
     else 	{ sscanf( s, "%f", &structs->TW_stage );
     structs->TW_stage -= BASE_DATUM; 
     }
     s = Scip( s, TAB );
   	 
     if (*s == TAB) structs->cell_i_HW = 0;
     else  		/* read coordinates of the reference cell for head water */
     { sscanf( s, "%d", &structs->cell_i_HW );
     structs->cell_i_HW++;
     }	 
     s = Scip( s, TAB );
     if (*s == TAB) structs->cell_j_HW = 0;
     else 
     { sscanf( s, "%d", &structs->cell_j_HW );
     structs->cell_j_HW++;
     }	 
     s = Scip( s, TAB );

     if ( structs->cell_i_HW > s0 || structs->cell_j_HW > s1 )
     {  printf ( "Error in definition of water control structure %s", structs->S_nam );
     exit (-2);	/*check that cell obtained is within array boundaries */
     }

     if (*s == TAB) structs->cell_i_TW = 0;
     else 		/* read coordinates of the reference cell for tail water */
     { sscanf( s, "%d", &structs->cell_i_TW );
     structs->cell_i_TW++;
     }	 
     s = Scip( s, TAB );
     if (*s == TAB) structs->cell_j_TW = 0;
     else 
     { sscanf( s, "%d", &structs->cell_j_TW );
     structs->cell_j_TW++;
     }	 
     s = Scip( s, TAB );

     if ( structs->cell_i_TW > s0 || structs->cell_j_TW > s1 )
     {  printf ( "Error in definition of water control structure %s", structs->S_nam );
     exit (-2);	/*check that cell obtained is within array boundaries */
     }

     if (*s == TAB || *s == '?') structs->w_coef = 0;
     else sscanf( s, "%f", &structs->w_coef );
         /* the next field after the coef is the date the file was exported, ignored here */
   	
     struct_last = structs;
          
         /* Allocate memory for next structure */
     if ( (struct_next = (struct Structure *) malloc( (size_t) sizeof( struct Structure ))) == NULL )
     {
         printf( "Failed to allocate memory for next structure (%s)\n ", structs->S_nam ) ;
         exit( -2 ) ;
     };
     
     /* initialize the sum of structure  flows and constit solute in flows */
     structs->SumFlow = 0.0; 
     structs->Sum_P = 0.0; 
     structs->Sum_N = 0.0; 
     structs->Sum_S = 0.0; 

     structs->next_in_list = struct_next;
     structs = struct_next;
 }
   
 struct_last->next_in_list = NULL;
 free ((char *)structs);
 fclose (ChanInFile);
   

 usrErr( "\tWater control structure locs/attributes read OK" );
 usrErr0 ( "\tControl structures' water flow time series...");
 
   
/**********************/
/* read structure flow input file(s) */
     { if(H_OPSYS==UNIX) 
         sprintf( modelFileName, "%s/%s/Data/%s.struct_wat", ModelPath, ProjName, filename );
     else sprintf( modelFileName, "%s%s:Data:%s.struct_wat", ModelPath, ProjName, filename );
     }

/* Open file with structure flow data */
     if ( ( F_struct_wat = fopen( modelFileName, "r" ) ) ==  NULL )
     {
         printf( "Can't open %s file!\n", modelFileName ) ;
         exit(-1) ;
     }

     fgets(lline, 2000, F_struct_wat); /* first header line with the alternative's name */
     line=lline;
     sscanf(line, "%s",&scenario);
     if (strcmp(scenario,SimAlt) != 0) {
         sprintf(msgStr, "The scenario (%s) found in the %s file doesn't match the one (%s) you asked for in Driver.parm!", scenario, modelFileName, SimAlt); usrErr(msgStr);
         exit(-1);
     }

     fgets(lline, 2000, F_struct_wat); /* read 2'nd header line with column names */
     line=lline;
     line = Scip( line, TAB );
     line = Scip( line, TAB );
     line = Scip( line, TAB ); /*read past the three date columns, ready to read the variable names */


     /* loop through all the control structure columns */
     for ( i = 0; i < num_struct_hist; i++ ) { 

         Hist_data[i].arrayWat = (float*) nalloc( ( (PORnumday+2)*sizeof(float) ), "timeseries temp" );
         Hist_data[i].arrayP =   (float*) nalloc( ( (PORnumday+2)*sizeof(float) ), "timeseries temp" );
          Hist_data[i].arrayN =  (float*) nalloc( ( (PORnumday+2)*sizeof(float) ), "timeseries temp" ); 
          Hist_data[i].arrayS =  (float*) nalloc( ( (PORnumday+2)*sizeof(float) ), "timeseries temp" ); 

/*  allocate memory for dissaggregated structures and provide a couple of attributes */
         if ( Hist_data[i].flag > 1 ) {
             structs = struct_first;  /* go through all structures to pull out the disaggregated ones */
             while ( structs != NULL ) 
             {
                 /* Ex: the SFWMM aggregated S10's flag is 2, S10AB&C's flags are 20 */
                 if ( structs->flag == 10 * Hist_data[i].flag  )  
                 { 
                 	/* disAggNum counts the total number of disaggregated structures for simulation */
                     disAggNum++;
                     Hist_data[num_struct_hist+disAggNum-1].arrayWat = 
			     (float*) nalloc( ( (PORnumday+2)*sizeof(float) ), "timeseries temp" );
                     Hist_data[num_struct_hist+disAggNum-1].arrayP    = 
			     (float*) nalloc( ( (PORnumday+2)*sizeof(float) ), "timeseries temp" );
                     Hist_data[num_struct_hist+disAggNum-1].arrayN    = 
			     (float*) nalloc( ( (PORnumday+2)*sizeof(float) ), "timeseries temp" ); 
                     Hist_data[num_struct_hist+disAggNum-1].arrayS    = 
			     (float*) nalloc( ( (PORnumday+2)*sizeof(float) ), "timeseries temp" ); 

                     Hist_data[num_struct_hist+disAggNum-1].aggID = i; /* the aggregated structure ID is attached to the dissagregated one */
                     strcpy( Hist_data[num_struct_hist+disAggNum-1].S_nam, structs->S_nam ); /* the disaggregated structure's name is put into the HistData structure */
                     Hist_data[i].aggCnt++; /* increment a count of the number of disagg structs associated w/ each agg struct */
                     structs->flag = 1; /* set the disagg struct's processing flag to 1, the only flag used for simulation structure flows (new input style) */
                     structs->histID = num_struct_hist+disAggNum; /* store the historical ID value for the disagg structures */
                     
                 } 
   
                 structs = structs->next_in_list;
             }             
         }
         
         sscanf ( line,"%s\t",&structName); /* read the structure names in the header */
         if (strcmp(Hist_data[i].S_nam , structName) != 0) {
             printf( "variable name  %s in input file does not match order of CanalData.structs.\n", structName);
             exit(-1); 
             }
         line = Scip( line, TAB );
     }


     lincnt = 0; /*  historical/other-data data input file record #  starting at the point where the model reads values */
     while ( fgets(lline, 2000, F_struct_wat) != NULL && !feof( F_struct_wat ) )
     {
         line = lline;
         sscanf( line, "%d %d %d",&yyyy,&mm,&dd); /* read the date of the current record */
         
        /* julian day, returns a double (args = mon, day, year, hours, minutes, seconds) */
         Jdate_read = julday( mm,dd,yyyy,0,0,0.0 ); 
         if ((lincnt == 0) && (Jdate_read > Jdate_init) ) {
             printf (" \n***Error\nStructure flow data (%s) init date > simulation start date\n", modelFileName); exit (-1);
         }
         if ((Jdate_read >= Jdate_init) && (Jdate_read <= Jdate_end))
         {
             line = Scip( line, TAB );
             line = Scip( line, TAB );

             for ( i = 0; i < num_struct_hist; i++ ) { /* loop through all the control structure columns in the input file */
                 line = Scip( line, TAB );
                 if (*line == TAB) Hist_data[i].arrayWat[((int)(lincnt))] = 0;
                 else sscanf( line, "%f", &Hist_data[i].arrayWat[((int)(lincnt))] );
                 Hist_data[i].arrayWat[((int)(lincnt))] *= 2446.848; /* cfs => m^3/d (0.02832*60*60*24)  */
             }
         
         for (i = num_struct_hist; i < num_struct_hist+disAggNum; i++) { /* dissaggregate all agg structs */
             k = Hist_data[i].aggID; /* use the aggregated historical data WCstructure */
             Hist_data[i].arrayWat[((int)(lincnt))] =
                 Hist_data[k].arrayWat[((int)(lincnt))] / (Hist_data[k].aggCnt) ;

         }
         
              
         lincnt++; /* increment the number of daily data records */
     }

} /* end of reading structure flow records */
     
if (lincnt == 0) {
    printf (" \n***Error\nStructure flow data (%s) has date problem (0 records read)\n", modelFileName); exit (-1);
}
else if (lincnt != PORnumday ) {
    printf (" \n***Error\nStructure flow data (%s) has problem (%d records read, %d expected)\n", modelFileName, lincnt, PORnumday); exit (-1);
}
     
sprintf (msgStr, "done, found %d records.",lincnt); usrErr(msgStr);
fclose (F_struct_wat);

/**********************/
/* read time series structure P concentration input file(s) */
/* there can be situations when we have no time series TP data (numTPser==0) */
 if (numTPser>0) {
     usrErr0 ( "\tControl structures' TP concen. time series...");

     { if(H_OPSYS==UNIX) 
         sprintf( modelFileName, "%s/%s/Data/%s.struct_TP", ModelPath, ProjName, filename );
     else sprintf( modelFileName, "%s%s:Data:%s.struct_TP", ModelPath, ProjName, filename );
     }

/* Open file with structure concentration data */
     if ( ( F_struct_TP = fopen( modelFileName, "r" ) ) ==  NULL )
     {
         printf( "Can't open %s file!\n", modelFileName ) ;
         exit(-1) ;
     }

     fgets(lline, 2000, F_struct_TP); /* first header line with the alternative's name */
     line=lline;
     sscanf(line, "%s",&scenario);
     if (strcmp(scenario,SimAlt) != 0) {
         sprintf(msgStr, "The scenario (%s) found in the %s file doesn't match the one (%s) you asked for in Driver.parm!", scenario, modelFileName, SimAlt); usrErr(msgStr);
         exit(-1);
     }
     fgets(lline, 2000, F_struct_TP); /* read 2'nd header line with column names */
     line=lline;
     line = Scip( line, TAB );
     line = Scip( line, TAB );
     line = Scip( line, TAB );

/* now we're past the three date columns, ready to read the variable names */
     for ( j = 0; j < numTPser; j++ ) { /* loop through all the data columns */
         IDhist[j] = -1;
         strcpy(structName,"NULL"); /* a string structName read previously is retained - this NULL assignment will take care of case
                                   where nothing was found/read in the time series file, and thus cause an error */
         sscanf ( line,"%s\t",&structName); /* read the structure names in the header */
         for ( i = 0; i < num_struct_hist+disAggNum; i++ ) { 
             if (strcmp(Hist_data[i].S_nam , structName) == 0) {
                 IDhist[j] = i;
                 line = Scip( line, TAB );
             }
         }
         if (IDhist[j] == -1 ) { printf( "variable name %s in %s not found in CanalData.structs. (if 'NULL', a variable in CanalData.structs defined as having a time series column was not found in %s)\n", structName, modelFileName, modelFileName);
         exit(-1); 
         }
     }

     lincnt = 0; /*  input file record #  starting at the point where the model reads values */

     while ( fgets(lline, 2000, F_struct_TP) != NULL && !feof( F_struct_TP ) )
     {
         line = lline;
         sscanf( line, "%d %d %d",&yyyy,&mm,&dd); /* read the date of the current record */
         
        /* julian day, returns a double (args = mon, day, year, hours, minutes, seconds) */
         Jdate_read = julday( mm,dd,yyyy,0,0,0.0 ); 
         if ((lincnt == 0) && (Jdate_read > Jdate_init) ) {
             printf (" \n***Error\nStructure TP data (%s) init date > simulation start date\n", modelFileName); exit (-1);
         }
         if ((Jdate_read >= Jdate_init) && (Jdate_read <= Jdate_end))
         {
             line = Scip( line, TAB );
             line = Scip( line, TAB );
             for ( j = 0; j < numTPser; j++ ) { /* loop through all the control structure columns in the input file */
                 line = Scip( line, TAB );
                 if (*line == TAB) Hist_data[IDhist[j]].arrayP[((int)(lincnt))] = 0;
                 else sscanf( line, "%f", &Hist_data[IDhist[j]].arrayP[((int)(lincnt))] );
                 Hist_data[IDhist[j]].arrayP[((int)(lincnt))] *= 1.0E-6; /* data read in ug/L, convert to g/L=kg/m3 */
                 
             }
              
             lincnt++; /* increment the number of daily data records */
         }

     } /* end of reading TP conc data records */
     
     if (lincnt == 0 ) {
             printf (" \n***Error\nTP Time series data (%s) has date problem (0 records read)\n", modelFileName); exit (-1);
     }
     else if (lincnt != PORnumday ) {
             printf (" \n***Error\nTP Time series data (%s) has problem (%d records read, %d expected)\n", modelFileName, lincnt, PORnumday); exit (-1);
     }
     sprintf (msgStr, "done, found %d records.",lincnt); usrErr(msgStr);
     
     fclose (F_struct_TP);
 } /* end of reading TP time series data (if needed) */
 
 
 
/**********************/
/* read time series structure tracer (salt) concentration input file(s) */
/* there can be situations when we have no time series TS (salt) data (numTSser==0) */
 if (numTSser>0) {
     usrErr0 ( "\tControl structures' Tracer concen. time series...");

     { if(H_OPSYS==UNIX) 
         sprintf( modelFileName, "%s/%s/Data/%s.struct_TS", ModelPath, ProjName, filename );
     else sprintf( modelFileName, "%s%s:Data:%s.struct_TS", ModelPath, ProjName, filename );
     }

/* Open file with structure concentration data */
     if ( ( F_struct_TS = fopen( modelFileName, "r" ) ) ==  NULL )
     {
         printf( "Can't open %s file!\n", modelFileName ) ;
         exit(-1) ;
     }

     fgets(lline, 2000, F_struct_TS); /* first header line with the alternative's name */
     line=lline;
     sscanf(line, "%s",&scenario);
     if (strcmp(scenario,SimAlt) != 0) {
         sprintf(msgStr, "The scenario (%s) found in the %s file doesn't match the one (%s) you asked for in Driver.parm!", scenario, modelFileName, SimAlt); usrErr(msgStr);
         exit(-1);
     }
     fgets(lline, 2000, F_struct_TS); /* read 2'nd header line with column names */
     line=lline;
     line = Scip( line, TAB );
     line = Scip( line, TAB );
     line = Scip( line, TAB );

/* now we're past the three date columns, ready to read the variable names */
     for ( j = 0; j < numTSser; j++ ) { /* loop through all the data columns */
         IDhist[j] = -1;
         strcpy(structName,"NULL"); /* a string structName read previously is retained - this NULL assignment will take care of case
                                   where nothing was found/read in the time series file, and thus cause an error */
         sscanf ( line,"%s\t",&structName); /* read the structure names in the header */
         for ( i = 0; i < num_struct_hist+disAggNum; i++ ) { 
             if (strcmp(Hist_data[i].S_nam , structName) == 0) {
                 IDhist[j] = i;
                 line = Scip( line, TAB );
             }
         }
         if (IDhist[j] == -1 ) { printf( "variable name %s in %s not found in CanalData.structs. (if 'NULL', a variable in CanalData.structs defined as having a time series column was not found in %s)\n", structName, modelFileName, modelFileName);
         exit(-1); 
         }
     }

     lincnt = 0; /*  input file record #  starting at the point where the model reads values */

     while ( fgets(lline, 2000, F_struct_TS) != NULL && !feof( F_struct_TS ) )
     {
         line = lline;
         sscanf( line, "%d %d %d",&yyyy,&mm,&dd); /* read the date of the current record */
         
        /* julian day, returns a double (args = mon, day, year, hours, minutes, seconds) */
         Jdate_read = julday( mm,dd,yyyy,0,0,0.0 ); 
         if ((lincnt == 0) && (Jdate_read > Jdate_init) ) {
             printf (" \n***Error\nStructure Tracer data (%s) init date > simulation start date\n", modelFileName); exit (-1);
         }
         if ((Jdate_read >= Jdate_init) && (Jdate_read <= Jdate_end))
         {
             line = Scip( line, TAB );
             line = Scip( line, TAB );
             for ( j = 0; j < numTSser; j++ ) { /* loop through all the control structure columns in the input file */
                 line = Scip( line, TAB );
                 if (*line == TAB) Hist_data[IDhist[j]].arrayS[((int)(lincnt))] = 0;
                 else sscanf( line, "%f", &Hist_data[IDhist[j]].arrayS[((int)(lincnt))] );
                 Hist_data[IDhist[j]].arrayS[((int)(lincnt))] *= 1.0; /* data read in g/L=kg/m3, no conversion*/
                 
             }
              
             lincnt++; /* increment the number of daily data records */
         }

     } /* end of reading tracer (salt) conc data records */
     
     if (lincnt == 0 ) {
             printf (" \n***Error\nTracer Time series data (%s) has date problem (0 records read)\n", modelFileName); exit (-1);
     }
     else if (lincnt != PORnumday ) {
             printf (" \n***Error\nTracer Time series data (%s) has problem (%d records read, %d expected)\n", modelFileName, lincnt, PORnumday); exit (-1);
     }
     sprintf (msgStr, "done, found %d records.",lincnt); usrErr(msgStr);
     
     fclose (F_struct_TS);
 } /* end of reading trace (salt) time series data (if needed) */
 

 
 sprintf(msgStr, "\t**** Evaluate scenario: %s %s ****", &SimAlt, &SimModif ); usrErr(msgStr); WriteMsg(msgStr,1);

 return;
}


/****************************************************************************************/
/*! \brief Read target stage schedule as a recurring time-series graph

	\param sched_name The name of the target stage location
	\param filename Input file name for graph
	\param BASE_DATUM \ref GP_DATUM_DISTANCE
	\return Sce_Graph struct of the time series schedule graph

*/

struct Schedule *Read_schedule ( char *sched_name, char *filename, float BASE_DATUM )
{ 
char ss[301], *s, scenario[20], modnam[20];
int i, count = 1, sched_found=0;
struct Schedule *Sce_Graph;

   { if(H_OPSYS==UNIX) 
   		sprintf( modelFileName, "%s/%s/Data/%s.graph", ModelPath, ProjName, filename );
     else sprintf( modelFileName, "%s%s:Data:%s.graph", ModelPath, ProjName, filename );
   }

/* Open file with schedule graphs */
  if ( ( schedFile = fopen( modelFileName, "r" ) ) ==  NULL )
	{
       printf( "Can't open %s file!\n", modelFileName ) ;
       exit(-1) ;
    }

 fgets( ss, 300, schedFile );  /* skip comment line */
 fgets( ss, 300, schedFile );  /* skip comment line */
 fgets( ss, 300, schedFile );  /* skip comment line */
 fgets( ss, 300, schedFile ); sscanf( ss,"%s %s", &modnam, &scenario );
 if (strcmp(scenario,SimAlt) != 0) {
     sprintf(msgStr, "The scenario (%s) found in the %s file doesn't match the one (%s) you asked for in Driver.parm!",
             scenario, modelFileName, SimAlt); usrErr(msgStr);
             exit(-1);
 }
 if (strcmp(modnam,modelName) != 0) {
     sprintf(msgStr, "The model name (%s) found in the %s file doesn't match the one (%s) you asked for in Driver.parm!",
             modnam, modelFileName, modelName); usrErr(msgStr);
             exit(-1);
 }

/* Read schedule descriptors until none left */   
  while ( fgets( ss, 300, schedFile ) != NULL && !feof( schedFile ) )
	{ if ( !Wrdcmp ( sched_name, ss ) ) continue;
	  sched_found = True;
	  
	  fgets( ss, 300, schedFile );  
	  s = ss;
	  /* count the number of pairs for the graph */
	  while ( *s++ != EOL ) if ( *s == '(' ) count++;    

 	  /* Allocate memory for next schedule */
      if ( (Sce_Graph = (struct Schedule *) malloc( (size_t) sizeof( struct Schedule ))) == NULL )
       {
         printf( "Failed to allocate memory for Managed flow Schedule graph in (%s)\n ", sched_name ) ;
         exit( -2 ) ;
       };
     
 	  /* Allocate memory for next schedule graph containing 'count' points*/
      if ( (Sce_Graph->graph_points = 
     		(struct Points *) malloc( (size_t) sizeof( struct Points ) * count)) == NULL )
       {
         printf( "Failed to allocate memory for Managed flow Schedule graph in (%s)\n ", sched_name ) ;
         exit( -2 ) ;
       };
	  
	  Sce_Graph->num_point = count;
	  
 	  /* Read the graph points */
	  s = ss;
	  for ( i = 0; i < count && *s!=EOS; i++ )
	  { 
		s = Scip( s,'(');	
		sscanf( s, "%f", &Sce_Graph->graph_points[i].time ); 
		s = Scip ( s, ',');
	 	sscanf( s, "%f", &Sce_Graph->graph_points[i].value );
	 	Sce_Graph->graph_points[i].value += BASE_DATUM;
	  }
	} /* end of while */
	/* v2.5 included this error-trap */
	if (!sched_found) {
	     sprintf(msgStr, "The schedule name (%s) was not found in the %s schedule file! (Was named in the CanalData.struct file.)",
             sched_name, modelFileName); usrErr(msgStr);
             exit(-1);
    }

	
  fclose (schedFile);
  return Sce_Graph;
}

/********************************************************************************************/
/*! \brief Configure attributes of grid cells interacting with canal vectors.

	Identifies the cells that are crossed by the canal, calculates the length of canal within 
	each cell and marks the cells directly interacting with the canal. Store the pointers 
	to the adjacent cell arrays.  
	
	\param ElevMap \ref SED_ELEV
	\param channel_first Pointer to first canal data structure
	
	*/

void Channel_configure (  float *ElevMap, struct Chan *channel_first ) 
{ int i, j;							  /*coord. of current cell crossed by the canal */
 int  cellLoc, cellLoc_i, cellLoc_j;
 int HV, k, L_range, R_range;
 float T_length;						/* length of a straight segment of a canal reach (m) */
 float C_length;					  /*  total length of all of the segments of a canal reach (m) */
 float distTot;					      /*  cumulative (along grid cells) distance along a canal reach, from the starting point (m) */
 float A1, B1, A2, B2; 							 /* parameters of the canal equation */
 float L, p_middle, m, i45, x, y, x1, y1, length, RoIL, RoIR;
 float cell_step = 1.0;
   
 float a0, b0, a1, b1;
 int c_num, z;
 struct Cells  *cell; /* addresses for adjacent cell structures */
 struct Chan *channel;
 struct Chan_reach *Reach;
 
 int numChek; /* placeholder for debugging cell elev along canals */
   

 int *marked; /* a "temporary" fix used to determine if a cell belonging to a canal has already been marked */
 marked = (int *) nalloc(sizeof(int)*(s0+2)*(s1+2),"marked");
 init_pvar(marked,NULL,'d',0);

/* Open file to print canal-cell interaction info to */
 sprintf( modelFileName, "%s/%s/Output/Debug/CanalCells_interaction.txt", OutputPath, ProjName );
 if ( ( CanalCellInteractDebug = fopen( modelFileName, "w" ) ) ==  NULL )
 {
     printf( "Can't open %s file!\n", modelFileName ) ;
     exit(-1) ;
 }

     /* allocate memory for array of pointers to Chan structs */
 if ( (Chan_list = 
       (struct Chan **) malloc( (size_t) sizeof( struct Chan *) * num_chan)) == NULL )
 {
     printf( "Failed to allocate memory for next canal (%d)\n ", channel->number ) ;
     exit( -2 ) ;
 };

     /* allocate memory for first cell structure */
 if ( (cell = (struct Cells *) malloc( (size_t) sizeof( struct Cells ))) == NULL )
 { 
     printf( "Failed to allocate memory for cell structure\n " ) ;
     exit( -2 ) ;
 }

     /* store the pointer to first cell in the array */
 channel = channel_first;

     /* loop over all the canal reaches (== channel, defined as an individual canal reach with an ID number) */
 while (channel != NULL)
 {
     Reach = channel->reaches;
     c_num = channel->number;
     z  = channel->levee;
     RoIL = channel->roil;
     RoIR = channel->roir;
	
     Chan_list[c_num - CHo] = channel;
	
     channel->cells = cell;
     distTot = 0;
     C_length = 0;
     C_Mark = 0;
    
         /* a neg width canal is skipped, to SKIPCHAN near end of this cycle along canals */
     if (channel->width < 0)   goto SKIPCHAN; 
    
     fprintf ( CanalCellInteractDebug, "N = %d  levee = %d\n", c_num, z );
     
/* get the elevation at the starting reach segment (may be up or down stream) of the canal */
     cellLoc_i=(int)(Reach->x0+1);
     cellLoc_j=(int)(Reach->y0+1);
     cellLoc = T(cellLoc_i,cellLoc_j );
     Chan_list[c_num - CHo]->elev_start = ElevMap[cellLoc]; 
     if ( !ON_MAP[cellLoc] ) {
     	printf( "Starting elevation of Reach %d is out of the active domain.  Fix the reach location.\n",c_num);
     	exit(-2);
     }
     fprintf ( CanalCellInteractDebug, " %d, %d, StartElev = %f\n", cellLoc_i,cellLoc_j,  Chan_list[c_num - CHo]->elev_start );

	   	
     while (Reach != NULL)  /* loop through canal reach segments */
     {
         a0 = Reach->x0;
         b0 = Reach->y0;
         a1 = Reach->x1;
         b1 = Reach->y1;

         x1 = 0;  y1 = 0;

         T_length = sqrt( (b1 - b0)*(b1 - b0) + (a1 - a0)*(a1 - a0) ); 	/* total reach segment length */
         C_length += T_length;										    /* total canal reach length */

         if ( a1 != a0 && b1 != b0 ) 
         {
             A1 = (b1 - b0)/(a1 - a0);                      /* define the equation: y = A1x + B1 */
             B1 = b0 - A1*a0;
             A2 = 1/A1;  B2 = - B1/A1;                      /* define the equation: x = A2y + B2 */
         } 
    
         x = a0; y = b0;
         i = Round (x);    j = Round (y);   
              
             /* loop along cells of the canal reach segment */ 
         while ( x1 != a1 || y1 != b1 )
         {
	     if ( a1 == a0 )                                     /* if goes straight horizontally */
             {  x1 = a0; 
             y1 = (j == Round (b1) ) ? b1 : j + STEP(b1 - b0)*cell_step; 
             length = y1 - y;
             }
       
             else if ( b1 == b0 )                                   /* if goes straight vertically */
             { x1 = (i == Round (a1) ) ? a1 : i + STEP(a1 - a0)*cell_step; 
             y1 = b0; 
             length = x1 - x;
             }

             else if ( i == Round (a1) && j == Round (b1) )
             { x1 = a1;  y1 = b1;                                               /* if last cell */ 
             length = T_length * (y1 - y)/(b1 - b0);
             C_Mark = 0;
             }
          
             else
             { y1 = A1*(i + STEP(a1 - a0)*cell_step) + B1;  length = T_length * (y1 - y)/(b1 - b0);
             x1 = A2*(j + STEP(b1 - b0)*cell_step) + B2;  L = T_length * (x1 - x)/(a1 - a0);
    
             if ( length < L )                       /* define the end coordinates and length */
                 x1 = i + STEP(a1 - a0)*cell_step;  /* (the next coordinate is the closest)*/
             else
             {  y1 = j + STEP(b1 - b0)*cell_step;  length = L;
             }
             }

             if ( Abs (a1 - a0) < Abs (b1 - b0) )
                     /* define the direction of canal reach and  */
             { p_middle = (x1 + x)/2;      /* calculate the middle point on the canal reach in cell */
             m = i + cell_step/2;								/* m - the center of the cell */
             HV = 1;  /* to indicate that the canal reach goes closer to the horizontal direction */
             }
             else
             { p_middle = (y1 + y)/2;
             m = j + cell_step/2;
             HV = 0;
             }
             
		 distTot = distTot + length*celWid;  /*  cumulative (along grid cells) distance along a canal reach, from the starting point (m) */
         
                 /* mark cells depending on where the center of canal reach is rel. to the cell center*/

	     if ( Round (a1) == Round (a0) ) /* if the canal reach goes within the range of one */
                     /*cell horizontally, mark the two vertical cells */
             { if (b0 < b1)	/* if going from left to right */
             {	    	/* if the middle point of the canal reach is lower than the */
                     /* cell center, mark the cell as LEFT else as RIGHT */
                 if ( p_middle > m ) 
                 { cell = MarkCell ( i, j, LEFT, length, cell, EAST, z, i, j, c_num, marked, distTot );
                 L_range = (int)(RoIL - 1); R_range = (int)RoIR;
                 }  
                 else 
                 { cell = MarkCell ( i, j, RIGHT, length, cell, EAST, z, i - 1, j, c_num, marked, distTot );
                 L_range = (int) RoIL;  R_range = (int)(RoIR - 1);
                 }  
                 for ( k = 0; k < L_range; k++ )
                     cell = MarkCell ( i - k - 1, j, LEFT, length, cell, ALL, z, i, j, c_num, marked, distTot );
                 for ( k = 0; k < R_range; k++ )
                     cell = MarkCell ( i + k + 1, j, RIGHT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             }
             else  /* vice versa if we go from right to left */
             {     	/* if the middle point of the canal reach is lower than the */
                     /* cell center, mark the cell as RIGHT else as LEFT */
                 if ( p_middle > m ) 
                 { cell = MarkCell ( i, j, RIGHT, length, cell, EAST, z, i, j, c_num, marked, distTot );
                 L_range = (int)RoIL; R_range = (int)(RoIR - 1);
                 } 
                 else 
                 { cell = MarkCell ( i, j, LEFT, length, cell, EAST, z, i - 1, j, c_num, marked, distTot );
                 L_range = (int)(RoIL - 1); R_range = (int)RoIR;
                 }
                 for ( k = 0; k < L_range; k++ )
                     cell = MarkCell ( i + k + 1, j, LEFT, length, cell, ALL, z, i, j, c_num, marked, distTot );
                 for ( k = 0; k < R_range; k++ )
                     cell = MarkCell ( i - k - 1, j, RIGHT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             }
             }
             else if ( Round (b1) == Round (b0) )	/* if the canal reach goes vertically */
             { if (a0 < a1) /* if going from top to bottom */
             {	    /* if the middle point of the canal reach is further to the right than the */
                     /* cell center, mark the cell as RIGHT else as LEFT */
                 if ( p_middle > m ) 
                 { cell = MarkCell ( i, j, RIGHT, length, cell, SOUTH, z, i, j, c_num, marked, distTot );
                 L_range = (int)RoIL; R_range = (int)(RoIR -1);
                 }
                 else 
                 { cell = MarkCell ( i, j, LEFT, length, cell, SOUTH, z, i, j - 1, c_num, marked, distTot );
                 L_range = (int)(RoIL - 1); R_range = (int)RoIR;
                 }
                 for ( k = 0; k < L_range; k++ )
                     cell = MarkCell ( i, j + k + 1, LEFT, length, cell, ALL, z, i, j, c_num, marked, distTot );
                 for ( k = 0; k < R_range; k++ )
                     cell = MarkCell ( i, j - k - 1, RIGHT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             }
             else  /* vice versa if we go from bottom to top */
             {	    /* if the middle point of the canal reach is further to the left than the */
                     /* cell center, mark the cell as LEFT else as RIGHT */
                 if ( p_middle > m ) 
                 { cell = MarkCell ( i, j, LEFT, length, cell, SOUTH, z, i, j, c_num, marked, distTot );
                 L_range = (int)(RoIL - 1); R_range = (int)RoIR;
                 }
                 else 
                 { cell = MarkCell ( i, j, RIGHT, length, cell, SOUTH, z, i, j - 1, c_num, marked, distTot );
                 L_range = (int)RoIL; R_range = (int)(RoIR - 1);
                 }
                 for ( k = 0; k < L_range; k++ )
                     cell = MarkCell ( i, j - k - 1, LEFT, length, cell, ALL, z, i, j, c_num, marked, distTot );
                 for ( k = 0; k < R_range; k++ )
                     cell = MarkCell ( i, j + k + 1, RIGHT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             }
             }
             else
                     /* if the canal reach crosses more than one cell (in both dimensions), mark diagonal cells */
             { if ( A1>0 )                 /* this is an extremely confusing check, but in this */
                 i45 = ( HV ) ? p_middle - m : m - p_middle;  /*form it seems to be working OK */
             else 
                 i45 = p_middle - m;
                 /*  b0>b1, a0>a1 \/ b0<b1, a0>a1 */
                 /*  b0>b1, a0<a1 /\ b0<b1, a0<a1 */
             if ( b0 < b1 )
             { if ( a0 < a1 ) 	  /* when a0<a1 the canal reach goes down in the \ direction */
             { if ( i45 > 0 ) 
             { cell = MarkCell ( i, j, LEFT, length, cell, EAST, z, i, j, c_num, marked, distTot );
             cell = MarkCell ( i + 1, j - 1, RIGHT, length, cell, SOUTH, z, i + 1, j - 1, c_num, marked, distTot );
             }
             else 
             { cell = MarkCell ( i, j, RIGHT, length, cell, SOUTH, z, i, j, c_num, marked, distTot );
             cell = MarkCell ( i - 1, j + 1, LEFT, length, cell, EAST, z, i - 1, j + 1, c_num, marked, distTot );
             }
             for ( k = 0; k < RoIR-1; k++ )
                 cell = MarkCell ( i + k + 1, j - k - 1, RIGHT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             for ( k = 0; k < RoIL-1; k++ )
                 cell = MarkCell ( i - k - 1, j + k + 1, LEFT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             }
             else				  /* in this case the canal reach goes up in the / direction */
             { if ( i45 > 0 ) 
             { cell = MarkCell ( i, j, LEFT, length, cell, NONE, z, i, j, c_num, marked, distTot );
             cell = MarkCell ( i + 1, j + 1, RIGHT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             }
             else 
             { cell = MarkCell ( i, j, RIGHT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             cell = MarkCell ( i - 1, j - 1, LEFT, length, cell, NONE, z, i - 1, j - 1, c_num, marked, distTot );
             }
             for ( k = 0; k < RoIR-1; k++ )
                 cell = MarkCell ( i + k + 1, j + k + 1, RIGHT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             for ( k = 0; k < RoIL-1; k++ )
                 cell = MarkCell ( i - k - 1, j - k - 1, LEFT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             }
             }
                 /* if going in the opposite direction change left and right */            
             else
             { if ( a0 > a1 )	    /* when a0<a1 the canal reach goes up in the \ direction */
             { if ( i45 < 0 ) 
             { cell = MarkCell ( i, j, LEFT, length, cell, SOUTH, z, i, j, c_num, marked, distTot );
             cell = MarkCell ( i - 1, j + 1, RIGHT, length, cell, EAST, z, i - 1, j + 1, c_num, marked, distTot );
             }
             else 
             { cell = MarkCell ( i, j, RIGHT, length, cell, EAST, z, i, j, c_num, marked, distTot );
             cell = MarkCell ( i + 1, j - 1, LEFT, length, cell, SOUTH, z, i + 1, j - 1, c_num, marked, distTot );
             }

             for ( k = 0; k < RoIL-1; k++ )
                 cell = MarkCell ( i + k + 1, j - k - 1, LEFT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             for ( k = 0; k < RoIR-1; k++ )
                 cell = MarkCell ( i - k - 1, j + k + 1, RIGHT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             }
             else				/* in this case the canal reach goes down in the / direction */
             { if ( i45 < 0 ) 
             { cell = MarkCell ( i, j, LEFT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             cell = MarkCell ( i - 1, j - 1, RIGHT, length, cell, NONE, z, i - 1, j - 1, c_num, marked, distTot );
             }
             else 
             { cell = MarkCell ( i, j, RIGHT, length, cell, NONE, z, i, j, c_num, marked, distTot );
             cell = MarkCell ( i + 1, j + 1, LEFT, length, cell, ALL, z, i + 1, j + 1, c_num, marked, distTot );
             }

             for ( k = 0; k < (int)(RoIL-1); k++ )
                 cell = MarkCell ( i + k + 1, j + k + 1, LEFT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             for ( k = 0; k < (int)(RoIR-1); k++ )
                 cell = MarkCell ( i - k - 1, j - k - 1, RIGHT, length, cell, ALL, z, i, j, c_num, marked, distTot );
             }
             }
             }
     	 
             num_cell++;  	/*at this point we're counting the cells only once thinking that all 
                                  cells have the same length of interaction irrelevant of their distance from the canal 
                                  and that this length will be equal to the canal length / number of cells it crosses */

                 /* prepare for the next step in cycle */
             x = x1;  y = y1;
             i = Round (x);     if ( a1 < a0 && i == x ) i--;         /* if we're going upwards, we */
                 /* need to change the rule of Rounding so that we move ahead along cells */
             j = Round (y);     if ( b1 < b0 && j == y ) j--;
             C_Mark = 1;
    

         } /* end loop along cells of the canal reach segment  */
         
         Reach = Reach->next_reach;
     } /* end loop over all reach segments of a canal */

/* get the elevation at the ending reach segment (may be up or down stream) of the canal */
     cellLoc_i=(int)(a1+1);
     cellLoc_j=(int)(b1+1);
     cellLoc = T(cellLoc_i,cellLoc_j );
     Chan_list[c_num - CHo]->elev_end = ElevMap[cellLoc];  

     Chan_list[c_num - CHo]->num_of_cells = num_cell;
     num_cell = 0;
         /* store the canal attributes of:
            length of entire canal, 
            area of entire canal, 
            minimum volume allowed in canal,
            avg length of cell along reach segment,
            overland flow coefficient (not incl. dynamic manning's n, C_F is for sensitivity experiments only (normally=1) ),
            seepage flow coefficient (including GP_calibGWat groundwater flow calibration coef),
            land surface elevation difference between starting and ending points of canal 
            slope of the elevation gradient from start to end points */
     Chan_list[c_num - CHo]->length = C_length*celWid;
     Chan_list[c_num - CHo]->area = Chan_list[c_num - CHo]->width*Chan_list[c_num - CHo]->length;
     Chan_list[c_num - CHo]->minVol = Chan_list[c_num - CHo]->area*MINDEPTH;
     Chan_list[c_num - CHo]->seg_len = Chan_list[c_num - CHo]->length/Chan_list[c_num - CHo]->num_of_cells;
     Chan_list[c_num - CHo]->SW_flow_coef = sqrt(Chan_list[c_num - CHo]->seg_len) * sec_per_day * C_F;
     Chan_list[c_num - CHo]->SPG_flow_coef = Chan_list[c_num - CHo]->cond * GP_calibGWat;
     Chan_list[c_num - CHo]->elev_drop = Chan_list[c_num - CHo]->elev_start - Chan_list[c_num - CHo]->elev_end;
     Chan_list[c_num - CHo]->elev_slope = Chan_list[c_num - CHo]->elev_drop / Chan_list[c_num - CHo]->length ; /* v2.5 calculated slope */
     
     if ( !ON_MAP[cellLoc] ) {
     	printf( "Ending elevation of Reach %d is out of the active domain.  Fix the reach location.\n",c_num);
     	exit(-2);
     }
     fprintf ( CanalCellInteractDebug, " %d, %d, EndingElev = %f, ElevSlope= %f, ReachLength= %f\n", cellLoc_i,cellLoc_j,  
     					Chan_list[c_num - CHo]->elev_end, Chan_list[c_num - CHo]->elev_slope, Chan_list[c_num - CHo]->length );
     
     getCanalElev(c_num - CHo); /* v2.5 */
     
   SKIPCHAN: /* a single goto comes here, skipping a canal with negative width */
     channel = channel->next_in_list;

     cell_last->next_cell = NULL;			/* close the cell list */

 } /* end loop loop over all the canal reaches */
 
 fclose ( CanalCellInteractDebug );
 
 return; 
}

/********************************************************************************************/
/*! \brief Assign elevation to canal using slope & distance from start point

	This function does one thing:	\n
	1. For each grid cell along a canal, we determine the elevation of the top of the canal bank, 
	using the slope of the canal and the distance from the beginning point of the canal. (function new in v2.5)	\n				
	
	\param chan_n Canal reach number (adjusted for the beginning ID number)
	\return void
*/

void getCanalElev (int chan_n)
{
     int i;
         /* cycle along canal cells */
     cell = Chan_list[chan_n]->cells;
     i = T(cell->x, cell->y);
     
     while ( cell != NULL )
     { 
        cell->reachElev = Chan_list[chan_n]->elev_start - cell->reachDistDown * Chan_list[chan_n]->elev_slope;
        fprintf ( CanalCellInteractDebug, " Reach= %d, Cell (%d, %d), ReachElev= %f\n", chan_n+CHo, cell->x, cell->y, cell->reachElev );
        cell = cell->next_cell;
     }
}


/********************************************************************************************/
/*! \brief Mark cell function  

	This function does two things:	\n
	1. It stores the cell coordinates (x, y) in the interaction array, also 			
	identifying the location of this cell with respect to the canal;	\n				
	2. It marks the cell (xm,ym) on the ON_MAP array to prevent horizontal overland		
	fluxes if there is a levee associated with this canal.
	
	\param x Grid cell row
	\param y Grid cell column
	\param c_ind Cell index = 1 for left cells and -1 for right cells
	\param length Length of canal associated with this cell
	\param cell A struct of Cells
	\param direction Direction of interaction: 1 - none, 2 -to the East, 3 - to the South, 4 - all directions 
	\param levee Levee location attribute
	\param xm Grid cell row marked
	\param ym Grid cell column marked
	\param c_num Canal ID number
	\param marked Denote a marked cell
	\param distTot Cumulative (along grid cells) distance along a canal reach, from the starting point (m)
	\return cell Pointer to struct of Cells

*/

struct Cells *MarkCell (int x, int y, int c_ind, float length, struct Cells *cell, int direction, int levee, 
					int xm, int ym, int c_num, int *marked, float distTot )
{ struct Cells *cell_next;
  int N, cellLoc;
  
   if ( x >= s0 || y >= s1 )
     {  printf ( "Error in definition of canal in cell x=%d  y=%d ", x, y );
     	exit (-2);			/*check that cells obtained are within array boundaries */
     }

   N = T(xm+1, ym+1);
   if ( !ON_MAP[N] ) return cell;
   
   if ( levee && direction != ALL )
   ON_MAP[N] = ((ON_MAP[N]-1) & (direction+100)) + 1;	/* modifying ON_MAP to take into account the 
   allowed direction of fluxing: 00 - none, 01=1 - to the East, 10=2 - to the South, 11=3 - all 
   ON_MAP they naturally become 1 - none, 2 -to the East, 3 - to the South, 4 - all directions 
   0 is reserved for cells out of the ON_MAP area*/
   
   if ( !C_Mark ) return cell; /* C_Mark is to make gaps between canals, so that there will be no 
   		canals being connected through cells if it turns out that they are linked to the same cells */
             
       /* a temporary fix used to determine if a cell belonging to a canal has already been marked */
       /***********/
   cellLoc = T(x+1,y+1);
   if (marked[cellLoc] == c_num ) return cell;
   marked[cellLoc] = c_num;
       /***********/
             

   
   cell->x = x+1;				   		/* shift coord to match the other maps */
   cell->y = y+1;
   cell->ind = c_ind;	
   cell->length = length*celWid;	 	/* length of canal associated with this cell * cell width m */
   cell->reachDistDown = distTot; /*  cumulative (by grid cells) distance along a canal reach, from the starting point (m) (v2.5) */
   
   fprintf ( CanalCellInteractDebug, " %d, %d, l/r = %d, distance= %7.1f\n", cell->x, cell->y, c_ind, cell->reachDistDown );
   
   /* allocate memory for next cell structure */
   if ( (cell_next = (struct Cells *) malloc((size_t) sizeof( struct Cells ))) == NULL )
       {
         printf( "Failed to allocate memory for cell structure\n " ) ;
         exit( -2 ) ;
       }
   cell->next_cell = cell_next;
   cell_last = cell;

   return cell_next;
}

/********************************************************************************************/
/*! \brief Runs the iterative algorithm over the network of canals.

	The iterative relaxation routine calculates the new head in the canal and the water 
	exchange with the adjacent cells, including constituent fluxes.
	
	\param chan_n Canal ID number
	\param SWH \ref SURFACE_WAT
	\param ElevMap \ref SED_ELEV
	\param MC \ref HYD_MANNINGS_N
	\param GWH \ref SAT_WATER
	\param poros \ref HP_HYD_POROSITY
	\param GWcond \ref HYD_RCCONDUCT
	\param NA \ref DINdummy
	\param PA \ref TP_SF_WT
	\param SA \ref SALT_SURF_WT
	\param GNA \ref DINdummy
	\param GPA \ref TP_SED_WT
	\param GSA \ref SALT_SED_WT
	\param Unsat \ref UNSAT_WATER
	\param sp_yield \ref HP_HYD_SPEC_YIELD
	

*/


void  FluxChannel ( int chan_n, float *SWH, float *ElevMap, float *MC, 
                    float *GWH, float *poros, float *GWcond,
                    double *NA, double *PA, double *SA, double *GNA, double *GPA, double *GSA,
                    float *Unsat, float *sp_yield)
{ 
		
    /*! \par \b Variables local to function
         \em converged The convergence flag
       \n\em error The error calculated as the diff between the incoming volume and the absorbed volume
       \n\em prev_error The error calculated in the previous iteration 
       \n\em iter The iteration number in the iterative relaxation procedure
       \n\em max_iter The max number of iterations allowed
       \n\em init_factor The depth increment for the initial guess (now same as "factor")
       \n\em factor The depth increment for the new guess (now same as "init_factor")
       \n\em attempt The counter of number of attempts at convergence
       \n\em CanWatDep The water level height in canal
       \n<em> T_flux_S, T_flux_G, T_flux_L </em> The total net fluxes among canal and cells (Surface, Groundwater, Levee seepage)
       \n<em> fluxS, fluxG, fluxL  </em> The flux among canal and cells (Surface, Groundwater, Levee seepage)
       \n<em> fluxO, fluxCk </em> Flow-checks to ensure the canal is not drained below its minimum level   
       \n<em> SW_coef, SPG_coef, GW_coef </em> Aggregated flow coefficients (Surface, Groundwater, Levee seepage)
       \n<em> N_fl, P_fl, S_fl </em> Flux of Nitrogen (UNUSED), Phosphorus, and Salt/tracer among canal and cells
       \n\em I_Length  Average length of cell along reach segment 
       \n\em seg_area Average area of reach segment
       \n\em MIN_VOL Minimum canal volume allowed
       \n\em GW_head Groundwater head
       \n<em> h_GWflow, h_SPflow </em> Water heights for groundwater and seepage flows 
       \n\em dh Difference between hydraulic heads
       \n<em> Qin, Qout, Qout1  </em> Flow volumes
       \n\em SWater \ref SURFACE_WAT
       \n\em tot_head Grid cell water head
       \n\em CH_vol Total volume of canal at the estimated water depth
       \n\em CH_vol_R Total canal volume prior to relaxation procedure
       \n\em CH_bottElev Elevation of bottom of canal at cell location
       \n<em> H_rad_ch, H_rad_cell </em> Hydraulic radii, canal (channel) and cell
       \n<em> can_cell_M, can_cell_A  </em> Temp vars used in reducing excess flows between canal<->cell
       \n\em errorConv  Numerical error after convergence
*/
int converged = 0;						
 float error = 1.0;					
 float prev_error = 1.0;					
 int iter = 0; 							
 int max_iter = CHANNEL_MAX_ITER;		
 float init_factor = 0.5;				
 float factor = init_factor; 			

 int i;
 int attempt=0;							
 double CanWatDep = Chan_list[chan_n]->wat_depth;
 double T_flux_S, T_flux_G, T_flux_L;		
 double fluxO, fluxCk;
 double fluxS, fluxG, fluxL; 
 double SW_coef, SPG_coef, GW_coef; 	
 double N_fl, P_fl, S_fl; 				
 /* double SNfl, GNfl, SPfl, GPfl, SSfl, GSfl; UNUSED */
 double I_Length; 						
 double seg_area; 						
 double MIN_VOL; 						
 double GW_head; 						
 double h_GWflow, h_SPflow;				
 double dh;							
 double Qin, Qout, Qout1;				
 double SWater;							
 double tot_head;						
 double CH_vol;						
 double CH_vol_R;					
 double CH_bottElev;					
 double H_rad_ch, H_rad_cell;			
 double can_cell_M, can_cell_A;			
 float errorConv;						

/* UNUSED below, unimplemented gwater/sfwater integration vars */ 
    int equilib, revers=1;
    float H_pot_don, H_pot_rec;
    
    float fluxTOunsat_don, fluxHoriz; /* vertical and horiz components of the calculated total groundwater Flux */
    float UnsatZ_don, UnsatCap_don, UnsatPot_don, Sat_vs_unsat; /* unsaturated zone capacities, potentials, in the donor cell */
    float UnsatZ_rec, UnsatCap_rec, UnsatPot_rec; /* unsaturated zone capacities, potentials, in the recipient cell */
    float sfTOsat_don, unsatTOsat_don, unsatTOsat_rec, satTOsf_rec, horizTOsat_rec; /* fluxes for vertical integration */
    
    float sedwat_don, sedwat_rec;
    float sed_stuf1fl_rec, sed_stuf2fl_rec, sed_stuf3fl_rec; /* vertical flux of solutes */
    float sf_stuf1fl_don, sf_stuf2fl_don, sf_stuf3fl_don; /* vertical flux of solutes */
    float sed_stuf1fl_horz, sed_stuf2fl_horz, sed_stuf3fl_horz; /* horiz flux of solutes */
/* UNUSED above, unimplemented gwater/sfwater integration vars */ 
 
 
/* look at all the structures connected to the canal and add/subtract flow from/to them */
 Qin = 0; Qout = 0;
 structs = struct_first;
 while ( structs != NULL )
 {
     
     if ( structs->canal_fr  == chan_n+CHo )  
     {
         if ( structs->flow > 0 ) Qout += structs->flow;
         else Qin -= structs->flow;
     } 
     if ( structs->canal_to == chan_n+CHo ) 
     {
         if ( structs->flow > 0 ) Qin += structs->flow;
         else Qout -= structs->flow;
     } 
    
     structs = structs->next_in_list;
 }
  
 CH_vol_R = Chan_list[chan_n]->area * CanWatDep;  /* total canal volume prior to relaxation procedure */
 MIN_VOL =  Chan_list[chan_n]->minVol;  			/* minimum canal volume allowed */
 SPG_coef = Chan_list[chan_n]->SPG_flow_coef;	/* GP_calibGWat already in this */

 I_Length =   Chan_list[chan_n]->seg_len;          /* avg length of cell along reach segment */
 seg_area = I_Length * Chan_list[chan_n]->width;  /* avg area of reach segment */
 can_cell_M = Chan_list[chan_n]->area*CELL_SIZE;  /* used in reducing excess flows between canal<->cell */
 can_cell_A = Chan_list[chan_n]->area+CELL_SIZE;  /* used in reducing excess flows between canal<->cell */
 

 
/* start relaxation procedure */  
 do
 {
     if   ( Abs( error) < F_ERROR ) 
     {
             converged = 1; /* when we've converged, we'll go through the cell list one more time w/o modifying the depth estimate */
                        /* occasionally, we get a modified total flux value, and diff error, a second time with the same depth estimate! */
             errorConv=error;
             factor = 0;
     }
     else if  ( iter >= (max_iter-1) )
     {
         if (attempt == 4) {
             converged = 1; /* haven't actually converged, but stop now */
             sprintf(msgStr,"Day %6.1f: ERROR - couldn't converge on Canal %d, HALTING the iterative relaxation after 4 tries, error = %f.", 
             		SimTime.TIME, Chan_list[chan_n]->number, error); 
             WriteMsg( msgStr,True ); dynERRORnum++;
         }
         else {
             if (debug > 3) { sprintf(msgStr,"Day %6.1f: Warning - couldn't converge on Canal %d, redoing the iterative relaxation, error = %f.", 
             		SimTime.TIME, Chan_list[chan_n]->number, error); 
             WriteMsg( msgStr,True ); }

             attempt++; /* the attempt counter increment */
             iter = 0; /* reset iterations to 0, and double factor & max # iterations on first try */ 
             if (attempt == 1) { factor = init_factor*2.0;  max_iter = CHANNEL_MAX_ITER * 2; }
             else if (attempt == 2) { factor = init_factor*0.5; max_iter = CHANNEL_MAX_ITER * 2; }
             else if (attempt == 3) { factor = init_factor*4.0; max_iter = CHANNEL_MAX_ITER * 4; }
             else if (attempt == 4) { factor = init_factor*0.05; max_iter = CHANNEL_MAX_ITER * 4; }
         }
     }

     else
     {
             /* if error changed sign, we have overshot the target, therefore reverse direction */
         if (error * prev_error < 0 && iter != 1) 
             factor = - sgn(error) *  Abs( factor ) * 0.5; 	/* and reduce the step */
         
             /* if error contiues to grow after first iter.(#0), reverse direction */	
         if (iter == 1 && error > 0) factor = - factor;
         
     }
     
     if (iter != 0 && !converged ) 
     {
         CanWatDep += factor; 
         prev_error = error;
         if ( CanWatDep < MINDEPTH ) 
         { 
             if (debug > 3) { sprintf(msgStr,"Day %6.1f: Warning - estimate is below minimum depth, revising estimate. (Canal %d, depth %f)", 
             		SimTime.TIME, Chan_list[chan_n]->number,CanWatDep); 
             WriteMsg( msgStr,True ); }            

             CanWatDep = MINDEPTH; 
            /* converged = 1; */
            factor = -factor*0.1; /* reverse the direction and decrease for next estimate*/
         }      
     }     
    
     T_flux_G = T_flux_S = T_flux_L = 0.0;           /* at each convergence iter, set the sum flows to 0 */
     CH_vol = Chan_list[chan_n]->area * CanWatDep;  /* volume of canal at the estimated water depth */
    
         /* cycle along canal cells */
     cell = Chan_list[chan_n]->cells;
     
     while ( cell != NULL )
     { 
         i = T(cell->x, cell->y);
         if ( !ON_MAP[i] )  	 /* check that cell is ON_MAP, else we do nothing */
                    /* return to beginning of do loop                */
         {  cell = cell->next_cell;  continue;
         }
         if (basn[i] == 0) {
             sprintf(msgStr,"ERROR - Cell (%d, %d) is out-of-system.", cell->x, cell->y); 
             WriteMsg( msgStr,True );  dynERRORnum++; }

         fluxS = 0.;  fluxG = 0.; fluxL = 0.; /* overland, subsurface, and seepage cell-canal fluxes set to 0 */
	   /* v2.5 In the case of some canal reaches, there is a lip or berm, and often associated vegetation, that is 
	   along the edge of a canal, decreasing flow rates between canal & marsh.  If user provides a non-zero roughness
	   parameter in the CanalData.chan input file, the the mean roughness of this berm and the cell is used for 
	   surface exchanges */
         SW_coef = ( MC[i] == 0.0 ) ? 0 : Chan_list[chan_n]->SW_flow_coef / ( (Chan_list[chan_n]->edgeMann > 0.0) ?  (MC[i] + Chan_list[chan_n]->edgeMann)/2 : MC[i] );
         GW_coef = GWcond[i] * poros[HAB[i]];	
         SWater = SWH[i];

         if (debug>0 && (SWater < -GP_MinCheck) ) {
         sprintf(msgStr,"Day %6.1f: capacityERR - neg sfwater along canal %d, candepth %f, cell (%d, %d), celldepth %f", 
                 SimTime.TIME,Chan_list[chan_n]->number,CanWatDep, cell->x, cell->y, SWater); 
         WriteMsg( msgStr,True );  dynERRORnum++; }
         
         GW_head = GWH[i]/poros[HAB[i]];
         tot_head = SWater + ElevMap[i]; 
         CH_bottElev = cell->reachElev - Chan_list[chan_n]->depth;  /* elev of bottom of canal at cell location (v2.5 is from fixed vector slope instead of actual grid cell elev) */
             
         dh = ( CH_bottElev + CanWatDep ) - tot_head;

             /* hydraulic radius of canal for overland flow out of canal */
         H_rad_ch =   Max ( ( seg_area * ramp(CanWatDep - Chan_list[chan_n]->depth) +
                        SWater * (CELL_SIZE-seg_area) ) / CELL_SIZE, 0.0);   /* v2.5 contrain to >= 0 */
            /* hydraulic radius of cell for overland flow into canal (same eqn right now) */
         H_rad_cell = Max ( ( seg_area * ramp(CanWatDep - Chan_list[chan_n]->depth) +
                        SWater * (CELL_SIZE-seg_area) ) / CELL_SIZE, 0.0);     /* v2.5 contrain to >= 0 */

         	/* determine the heights of the water cross sections associated with the subsurf and seep flows */
         if (dh > 0.0) { /* from canal */
         	h_GWflow = Min(Chan_list[chan_n]->depth, CanWatDep);
         	h_SPflow = Max(CH_bottElev + CanWatDep - ElevMap[i], 0.0);
         }
         else if (dh < 0.0) { /* from cell */
         	h_GWflow = Max(GW_head-CH_bottElev, 0.0);
         	h_SPflow = Max(tot_head-ElevMap[i], 0.0);
         }

	  	 
         switch( Chan_list[chan_n]->levee )
         {	
         	
             case 2:	/* if levees are on both sides of the canal */
                         /*seepage and subsurface flow on both sides */
                 fluxL =  (h_SPflow > 0.0) ? (f_Ground ( dh, h_SPflow, SPG_coef, I_Length) ) : (0.0);
                 fluxG =  (h_GWflow > 0.0) ? (f_Ground ( dh, h_GWflow, GW_coef, I_Length) ) : (0.0); 
                 break;
             case 0: /* if there are no levees */
                     /*overland flow either side */
                 if ( dh > 0 ) {
                         /* flux from canal to cell  */
                     fluxS = f_Manning ( dh, H_rad_ch, SW_coef);
                 }
                 else if (SWater>GP_DetentZ) {
                     fluxS =  f_Manning ( dh, H_rad_cell, SW_coef) ; 
                    /* fluxS =  ( (-fluxS) > (SWater-GP_DetentZ) *CELL_SIZE )  ? ( -(SWater-GP_DetentZ)*CELL_SIZE ) : (fluxS);*/
                    if (-fluxS > (SWater-GP_DetentZ) *CELL_SIZE )
                    	fluxS = -(SWater-GP_DetentZ)*CELL_SIZE;
                 }
	  
	        		/* subsurface groundwater flow both sides */
                 fluxG =  (h_GWflow > 0.0) ? (f_Ground ( dh, h_GWflow, GW_coef, I_Length) ) : (0.0); 
                 break;
             case 1:  /* if levee is on the left */ 
                 if ( cell->ind < 0  )
                 { /*overland flow */
                     if ( dh > 0 ) {
                             /* flux from canal to cell  */
                         fluxS = f_Manning ( dh, H_rad_ch, SW_coef);
                     }
                     else if (SWater>GP_DetentZ) {
                         fluxS = f_Manning ( dh, H_rad_cell, SW_coef) ; 
                        /* fluxS =  ( (-fluxS) > (SWater-GP_DetentZ) *CELL_SIZE )  ? ( -(SWater-GP_DetentZ)*CELL_SIZE ) : (fluxS);*/
                    if (-fluxS > (SWater-GP_DetentZ) *CELL_SIZE )
                    	fluxS = -(SWater-GP_DetentZ)*CELL_SIZE;
                     }
	  
                 }	
                 else 
                         /*seepage on the levee side */
                     fluxL =  (h_SPflow > 0.0) ? (f_Ground ( dh, h_SPflow, SPG_coef, I_Length) ) : (0.0);
                     
 	        		/* subsurface groundwater flow both sides */
                 fluxG =  (h_GWflow > 0.0) ? (f_Ground ( dh, h_GWflow, GW_coef, I_Length) ) : (0.0); 
                 break;
             case -1: /* if levee is on the right */
                 if ( cell->ind > 0 )
                 { /*overland flow */
                     if ( dh > 0 ) {
                             /* flux from canal to cell */
                         fluxS = f_Manning( dh, H_rad_ch, SW_coef);
                     }
                     else if (SWater>GP_DetentZ) {
                         fluxS = f_Manning ( dh, H_rad_cell, SW_coef); 
                         /*fluxS =  ( (-fluxS) > (SWater-GP_DetentZ) *CELL_SIZE )  ? ( -(SWater-GP_DetentZ)*CELL_SIZE ) : (fluxS);*/
                    if (-fluxS > (SWater-GP_DetentZ) *CELL_SIZE )
                    	fluxS = -(SWater-GP_DetentZ)*CELL_SIZE;
                     }
                 }	
                 else
                         /*seepage on the levee side */
                     fluxL =  (h_SPflow > 0.0) ? (f_Ground ( dh, h_SPflow, SPG_coef, I_Length) ) : (0.0);

 	        		/* subsurface groundwater flow both sides */
                 fluxG =  (h_GWflow > 0.0) ? (f_Ground ( dh, h_GWflow, GW_coef, I_Length) ) : (0.0); 
                 break;  
             default:
                 printf ("Error in levee location: Channel N %d", chan_n+CHo );
                 exit (-1);
                 break;
         } /*end switch */
	  


             /* can_cell_M and can_cell_A are the areas of the canal and the cell multiplied (M) and added (A) together, respectively */
             /* the flux eqn is the same as the virtual weir eqn, equilibrating the heads across two unequal area regions */

             /* Surface flows: making sure that the cell head is not higher than the canal head */
         if (fluxS>0 && ( (fluxCk = tot_head + fluxS/CELL_SIZE - (CH_bottElev + CanWatDep - fluxS/Chan_list[chan_n]->area) ) > 0.0 ) ) {
             fluxS = (can_cell_M*(CH_bottElev + CanWatDep)/can_cell_A - can_cell_M*tot_head/can_cell_A );
             fluxCk = tot_head + fluxS/CELL_SIZE - (CH_bottElev + CanWatDep - fluxS/Chan_list[chan_n]->area);
             if (fluxCk > F_ERROR && debug>3) {
                 sprintf(msgStr,"Day %6.1f: Warning - excessive head in cell after flux from Canal %d (%f m diff)", 
                         SimTime.TIME, Chan_list[chan_n]->number,fluxCk); 
                 WriteMsg( msgStr,True );  
             }
         }
         
             /* Surface flows: making sure that the canal water level is not flooded much higher than the water level in the cell */
         if ( fluxS<0 && ( (fluxCk = (CH_bottElev + CanWatDep - fluxS/Chan_list[chan_n]->area) - (tot_head + fluxS/CELL_SIZE) ) > 0.0 ) ) {
             fluxS = (can_cell_M*(CH_bottElev + CanWatDep)/can_cell_A - can_cell_M*tot_head/can_cell_A );
             fluxCk =  (CH_bottElev + CanWatDep - fluxS/Chan_list[chan_n]->area) - (tot_head + fluxS/CELL_SIZE);
             if (fluxCk > F_ERROR && debug>3) {
                 sprintf(msgStr,"Day %6.1f: Warning - excessive head in Canal %d after flux from cell (%f m diff)", 
                         SimTime.TIME, Chan_list[chan_n]->number,fluxCk); 
                 WriteMsg( msgStr,True );  
             }
         }

             /* now ensure the canal is not drained below its minimum level */ 
         if ( fluxS > 0 || fluxG > 0 || fluxL > 0) /* check gw, seep, and sfwat */
         {
             fluxO  = (fluxS > 0) ? fluxS : 0;
             fluxO += (fluxG > 0) ? fluxG : 0;
             fluxO += (fluxL > 0) ? fluxL : 0;
    
                 /* using vol associated with new estimate (before subtracting losses) */
             if ( (fluxCk = CH_vol - MIN_VOL - fluxS - fluxG - fluxL) < 0.0 )
             { fluxCk =  (fluxO>0) ? ( ( fluxO + fluxCk )/fluxO) : (0.0);	/* v2.5 removed a divide by zero, but shouldn't be needed */
             if (fluxS>0) fluxS *= /* (double) */fluxCk; /* reduce all pos outflows by this proportion */
             if (fluxG>0) fluxG *= fluxCk;
             if (fluxL>0) fluxL *= fluxCk;
             if ( (fluxCk = (CH_vol - MIN_VOL - fluxS - fluxG - fluxL)/Chan_list[chan_n]->area) < -GP_MinCheck ) {
                 sprintf(msgStr,"Day %6.1f: capacityERR - excessive draining from Canal %d (%f m below min)", SimTime.TIME,Chan_list[chan_n]->number,fluxCk); 
                 WriteMsg( msgStr,True ); dynERRORnum++; }
             }
         }

/******** UNUSED *********************/
    if (debug>99 && fluxG != 0.0) do {
        /* This is a part of the routine used to integrate surface/groundwater
         fluxes in the cell-cell Fluxes.c.  It only modifies the allocations if
         a cell is a recipient of groundwater flux (does not deal with cells as
         donor, which is handled in Fluxes.c).
         NOTE:****This routine is NOT fully integrated into the canal iterative solution,
         and thus is NOT used for now - get to it some day if it is determined to be needed.  */
       fluxHoriz =  fluxG;


/**** recipient cell's **pre-flux** capacities */
        UnsatZ_rec  =  (fluxG>0) ? ( ElevMap[i] - GWH[i] / poros[HAB[i]] ) : (0.0) ; /* unsat zone depth */
        if (debug>2 && (UnsatZ_rec < -0.001 ) ) {
            sprintf(msgStr,"Day %6.1f: capacityERR - neg unsat depth (%f m) in recipient cell (%d,%d) in canal-cell gw fluxes!", 
                    SimTime.TIME, UnsatZ_rec, cell->x, cell->y ); WriteMsg( msgStr,True ); dynERRORnum++; }
            
        UnsatCap_rec =  (fluxG>0) ? (UnsatZ_rec * poros[HAB[i]]) : (0.0); /* maximum pore space capacity (UnsatCap)
                                                              in the depth (UnsatZ) of new unsaturated zone */
        UnsatPot_rec  = (fluxG>0) ? (UnsatCap_rec - Unsat[i] ) : (0.0); /* (height of) the volume of pore space (soil "removed")
                                                           that is unoccupied in the unsat zone */
     /* sf-unsat-sat fluxes */
        horizTOsat_rec = (fluxG>0) ? (fluxHoriz) : (0.0); /* lateral inflow to soil into saturated storage */
        satTOsf_rec = (fluxG>0) ? (Max(fluxHoriz - UnsatPot_rec, 0.0) ) : (0.0); /* upwelling beyond soil capacity */
        if (fluxG>0) unsatTOsat_rec = (UnsatZ_rec > 0.0) ? /* incorporation of unsat moisture into sat storage with
                                                 rising water table due to horiz inflow */
            ( ((horizTOsat_rec-satTOsf_rec)/poros[HAB[i]] ) / UnsatZ_rec * Unsat[i] ) :
            (0.0);
        else
            unsatTOsat_rec = 0.0;
        
/**** potential new head in recipient cell will be ... */
        if (fluxG>0)  H_pot_rec = (GWH[i] + horizTOsat_rec + unsatTOsat_rec - satTOsf_rec)
            / poros[HAB[i]] + (SWater + satTOsf_rec);


        equilib = 1;
        
    } while  (!equilib); /* end of ***incomplete** routine for integrating groundwater-surface water and preventing head reversals */
/******** UNUSED *********************/

   /*  if (fluxG>0) fluxG=fluxHoriz; */ /* unused, part of unimplemented sf/gwater routine */
    
         CH_vol -= ( fluxS + fluxG + fluxL); /* subtract flows (pos == loss) from volume associated with new estimate */
         
         if ( converged )
         { 	/* the nutrient mass fluxes (pre-flux volumes) (mass=kg, vol=m3)*/
                 /* remember pos flux is from canal */
                 /* (CH_vol_R is vol prior to in/out fluxes) */
	/* surface water canal-cell flux */	          
             if ( fluxS != 0 ) {
             if ( fluxS > 0 ) /* surface water flux from canal to cell */
             {  
                 /* this is where Disp_Calc comes in */ /* hcf TEST to see about reducing dispersion, halfing the flux of salt (i.e., ca. 500m grid dispersion) */
                 N_fl = (CH_vol_R > 0) ? (Chan_list[chan_n]->N/CH_vol_R*fluxS) : (0.0);
                 P_fl = (CH_vol_R > 0) ? (Chan_list[chan_n]->P/CH_vol_R*fluxS) : (0.0);
                 S_fl = (CH_vol_R > 0) ? (Chan_list[chan_n]->S/CH_vol_R*fluxS) : (0.0);

                 if (Chan_list[chan_n]->basin != basn[i] && debug > -999 ) {
                     if ( (Chan_list[chan_n]->family !=  basn_list[basn[i]]->family) && (debug > 0) ) {        /* if the flow is not within the family... */
              		 sprintf(msgStr,"Day %6.1f: ERROR - no basin-basin canal-cell overland flow! (Reach %d, basn %d, with cell %d,%d, basin %d)", 
                                 SimTime.TIME,Chan_list[chan_n]->number, Chan_list[chan_n]->basin, cell->x, cell->y, basn[i]); 
             		 WriteMsg( msgStr,True );  dynERRORnum++;
                     }
                     else {    /* if it's flow within a family, just keep
                                  track of what the children do among themselves */            
                         if  ( !Chan_list[chan_n]->parent  ) { 
                         /* then find out about the flow for the family's sake */
                             VOL_OUT_OVL[Chan_list[chan_n]->basin] += fluxS;
                             P_OUT_OVL[Chan_list[chan_n]->basin] += P_fl;
                             S_OUT_OVL[Chan_list[chan_n]->basin] += S_fl;
                         }
                         if ( !basn_list[basn[i]]->parent ) {                 
                                 /* then find out about the flow for the family's sake */
                             VOL_IN_OVL[basn[i]] += fluxS; 
                             P_IN_OVL[basn[i]] += P_fl; 
                             S_IN_OVL[basn[i]] += S_fl; 
                         }
                     }
                 }                
             }
             else /* surface water flux from cell to canal */
             {
                 /* this is where Disp_Calc comes in */
                 N_fl = ( SWH[i] > 0 ) ? NA[i]*fluxS/CELL_SIZE/SWH[i] : 0.;
                 P_fl = ( SWH[i] > 0 ) ? PA[i]*fluxS/CELL_SIZE/SWH[i] : 0.;
                 S_fl = ( SWH[i] > 0 ) ? SA[i]*fluxS/CELL_SIZE/SWH[i] : 0.;
                 if (Chan_list[chan_n]->basin != basn[i] && debug > -999 ) {

                     if ( (Chan_list[chan_n]->family !=  basn_list[basn[i]]->family) && (debug > 0) ) {        /* if the flow is not within the family... */
             		 sprintf(msgStr,"Day %6.1f: ERROR - no basin-basin canal-cell overland flow! (Reach %d, basn %d, with cell %d,%d, basin %d)", 
             		 		SimTime.TIME,Chan_list[chan_n]->number, Chan_list[chan_n]->basin, cell->x, cell->y, basn[i]); 
             		 WriteMsg( msgStr,True );  dynERRORnum++;
                 }
                 else {    /* if it's flow within a family, just keep
                          track of what the children do among themselves */            
                     if  ( !basn_list[basn[i]]->parent  ) { 
                         /* then find out about the flow for the family's sake */
                         VOL_OUT_OVL[basn[i]] -= fluxS;
                         P_OUT_OVL[basn[i]] -= P_fl;
                         S_OUT_OVL[basn[i]] -= S_fl;
                     }
                     if ( !Chan_list[chan_n]->parent ) {                 
                         /* then find out about the flow for the family's sake */
                         VOL_IN_OVL[Chan_list[chan_n]->basin] -= fluxS; 
                         P_IN_OVL[Chan_list[chan_n]->basin] -= P_fl; 
                         S_IN_OVL[Chan_list[chan_n]->basin] -= S_fl; 
                     }
                 }
                 }


             } 
                 /* now pass that mass flux to/from the cell/canal */
             NA[i] += N_fl;
             PA[i] += P_fl;
             SA[i] += S_fl;
             Chan_list[chan_n]->N -= N_fl;
             Chan_list[chan_n]->P -= P_fl;
             Chan_list[chan_n]->S -= S_fl;
             }

	/* groundwater canal-cell flux */	          
             if ( fluxG != 0 ) {
             if ( fluxG > 0 ) /* ground water flux from canal to cell */
             {  
                 N_fl = (CH_vol_R > 0) ? (Chan_list[chan_n]->N/CH_vol_R*fluxG) : (0.0);
                 P_fl = (CH_vol_R > 0) ? (Chan_list[chan_n]->P/CH_vol_R*fluxG) : (0.0);
                 S_fl = (CH_vol_R > 0) ? (Chan_list[chan_n]->S/CH_vol_R*fluxG) : (0.0);

                 if (Chan_list[chan_n]->basin != basn[i] && debug > -999 ) {
                     if (Chan_list[chan_n]->family !=  
                         basn_list[basn[i]]->family ) {        /* if the flow is not within the family... */
                         if  ( !Chan_list[chan_n]->parent  ) { /* and if the donor or recipient is a child... */
                         /* then find out about the flow for the family's sake */
                             VOL_OUT_GW[Chan_list[chan_n]->family] += fluxG;
                             P_OUT_GW[Chan_list[chan_n]->family] += P_fl;
                             S_OUT_GW[Chan_list[chan_n]->family] += S_fl;
                         }
                         if ( !basn_list[basn[i]]->parent ) {                 
                           /* then find out about the flow for the family's sake */
                             VOL_IN_GW[basn_list[basn[i]]->family] += fluxG; 
                             P_IN_GW[basn_list[basn[i]]->family] += P_fl; 
                             S_IN_GW[basn_list[basn[i]]->family] += S_fl; 
                         }
                     VOL_OUT_GW[Chan_list[chan_n]->basin] += fluxG;
                     VOL_IN_GW[basn[i]] += fluxG; 
                     P_OUT_GW[Chan_list[chan_n]->basin] += P_fl;
                     P_IN_GW[basn[i]] += P_fl; 
                     S_OUT_GW[Chan_list[chan_n]->basin] += S_fl;
                     S_IN_GW[basn[i]] += S_fl; 
                 }
                 else {    /* if it's flow within a family, just keep
                          track of what the children do among themselves */            
                     if  ( !Chan_list[chan_n]->parent  ) { 
                         /* then find out about the flow for the family's sake */
                         VOL_OUT_GW[Chan_list[chan_n]->basin] += fluxG;
                         P_OUT_GW[Chan_list[chan_n]->basin] += P_fl;
                         S_OUT_GW[Chan_list[chan_n]->basin] += S_fl;
                     }
                     if ( !basn_list[basn[i]]->parent ) {                 
                         /* then find out about the flow for the family's sake */
                         VOL_IN_GW[basn[i]] += fluxG; 
                         P_IN_GW[basn[i]] += P_fl; 
                         S_IN_GW[basn[i]] += S_fl; 
                     }
                 }
                 }
             } /* end of positive flow evaluations */
             
             
             else  /*  ground water flux from cell to canal */
             {
                 N_fl = ( GWH[i] > 0 ) ? GNA[i]*fluxG/CELL_SIZE/GWH[i] : 0.;
                 P_fl = ( GWH[i] > 0 ) ? GPA[i]*fluxG/CELL_SIZE/GWH[i] : 0.;
                 S_fl = ( GWH[i] > 0 ) ? GSA[i]*fluxG/CELL_SIZE/GWH[i] : 0.;
                 if (Chan_list[chan_n]->basin != basn[i] && debug > -999) {
                     if (Chan_list[chan_n]->family !=  
                         basn_list[basn[i]]->family ) {        /* if the flow is not within the family... */
                         if  ( !basn_list[basn[i]]->parent  ) { /* and if the donor or recipient is a child... */
                         /* then find out about the flow for the family's sake */
                             VOL_OUT_GW[basn_list[basn[i]]->family] -= fluxG;
                             P_OUT_GW[basn_list[basn[i]]->family] -= P_fl;
                             S_OUT_GW[basn_list[basn[i]]->family] -= S_fl;
                         }
                         if ( !Chan_list[chan_n]->parent ) {                 
                           /* then find out about the flow for the family's sake */
                             VOL_IN_GW[Chan_list[chan_n]->family] -= fluxG; 
                             P_IN_GW[Chan_list[chan_n]->family] -= P_fl; 
                             S_IN_GW[Chan_list[chan_n]->family] -= S_fl; 
                         }
                     VOL_IN_GW[Chan_list[chan_n]->basin] -= fluxG;
                     VOL_OUT_GW[basn[i]] -= fluxG;
                     P_IN_GW[Chan_list[chan_n]->basin] -= P_fl;
                     P_OUT_GW[basn[i]] -= P_fl;
                     S_IN_GW[Chan_list[chan_n]->basin] -= S_fl;
                     S_OUT_GW[basn[i]] -= S_fl;
                 }
                 else {    /* if it's flow within a family, just keep
                          track of what the children do among themselves */            
                     if  ( !basn_list[basn[i]]->parent  ) { 
                         /* then find out about the flow for the family's sake */
                         VOL_OUT_GW[basn[i]] -= fluxG;
                         P_OUT_GW[basn[i]] -= P_fl;
                         S_OUT_GW[basn[i]] -= S_fl;
                     }
                     if ( !Chan_list[chan_n]->parent ) {                 
                         /* then find out about the flow for the family's sake */
                         VOL_IN_GW[Chan_list[chan_n]->basin] -= fluxG; 
                         P_IN_GW[Chan_list[chan_n]->basin] -= P_fl; 
                         S_IN_GW[Chan_list[chan_n]->basin] -= S_fl; 
                     }
                 }
                 }
             } 
                 /* now pass that mass flux to/from the cell/canal
                  this includes allocating nuts to surface water thru upflow (unimplemented) */
             /* unused, part of unimplemented sf/gwater routine */
/*              SNfl = Chan_list[chan_n]->N/CH_vol_R*satTOsf_rec ; */
/*              GNfl = N_fl - SNfl; */
/*              SPfl = Chan_list[chan_n]->P/CH_vol_R*satTOsf_rec ; */
/*              GPfl = P_fl - SPfl; */
/*              SSfl = Chan_list[chan_n]->S/CH_vol_R*satTOsf_rec ; */
/*              GSfl = S_fl - SSfl; */
/*              GNA[i] += GNfl; */
/*              GPA[i] += GPfl; */
/*              GSA[i] += GSfl; */
             GNA[i] += N_fl;
             GPA[i] += P_fl;
             GSA[i] += S_fl;
             Chan_list[chan_n]->N -= N_fl;
             Chan_list[chan_n]->P -= P_fl;
             Chan_list[chan_n]->S -= S_fl;
                 /* upflow into surface water nuts (this is actually canal conc., but what the heck) */
             /* unused, part of unimplemented sf/gwater routine */
/*              NA[i] += SNfl; */
/*              PA[i] += SPfl; */
/*              SA[i] += SSfl; */
             }

	/* seepage canal-cell flux */	          
             if ( fluxL != 0 ) {
             if ( fluxL > 0 ) /* seepage flux from canal to cell */
             {  
                 N_fl = (CH_vol_R > 0) ? (Chan_list[chan_n]->N/CH_vol_R*fluxL) : (0.0);
                 P_fl = (CH_vol_R > 0) ? (Chan_list[chan_n]->P/CH_vol_R*fluxL) : (0.0);
                 S_fl = (CH_vol_R > 0) ? (Chan_list[chan_n]->S/CH_vol_R*fluxL) : (0.0);
                 if (Chan_list[chan_n]->basin != basn[i] && debug > -999 ) {
                     if (Chan_list[chan_n]->family !=  
                         basn_list[basn[i]]->family ) {        /* if the flow is not within the family... */
                         if  ( !Chan_list[chan_n]->parent  ) { /* and if the donor or recipient is a child... */
                         /* then find out about the flow for the family's sake */
                             VOL_OUT_SPG[Chan_list[chan_n]->family] += fluxL;
                             P_OUT_SPG[Chan_list[chan_n]->family] += P_fl;
                             S_OUT_SPG[Chan_list[chan_n]->family] += S_fl;
                         }
                         if ( !basn_list[basn[i]]->parent ) {                 
                           /* then find out about the flow for the family's sake */
                             VOL_IN_SPG[basn_list[basn[i]]->family] += fluxL; 
                             P_IN_SPG[basn_list[basn[i]]->family] += P_fl; 
                             S_IN_SPG[basn_list[basn[i]]->family] += S_fl; 
                         }
                     VOL_OUT_SPG[Chan_list[chan_n]->basin] += fluxL;
                     VOL_IN_SPG[basn[i]] += fluxL; 
                     P_OUT_SPG[Chan_list[chan_n]->basin] += P_fl;
                     P_IN_SPG[basn[i]] += P_fl; 
                     S_OUT_SPG[Chan_list[chan_n]->basin] += S_fl;
                     S_IN_SPG[basn[i]] += S_fl; 
                 }
                 else {    /* if it's flow within a family, just keep
                          track of what the children do among themselves */            
                     if  ( !Chan_list[chan_n]->parent  ) { 
                         /* then find out about the flow for the family's sake */
                         VOL_OUT_SPG[Chan_list[chan_n]->basin] += fluxL;
                         P_OUT_SPG[Chan_list[chan_n]->basin] += P_fl;
                         S_OUT_SPG[Chan_list[chan_n]->basin] += S_fl;
                     }
                     if ( !basn_list[basn[i]]->parent ) {                 
                         /* then find out about the flow for the family's sake */
                         VOL_IN_SPG[basn[i]] += fluxL; 
                         P_IN_SPG[basn[i]] += P_fl; 
                         S_IN_SPG[basn[i]] += S_fl; 
                     }
                 }
                 }
                     if ( tot_head > GW_head ) { /* to cell surface water */
             			NA[i] += N_fl;
             			PA[i] += P_fl;
             			SA[i] += S_fl;
             			}
             		 else { /* to cell groundwater */
             			GNA[i] += N_fl;
             			GPA[i] += P_fl;
             			GSA[i] += S_fl;
             		}
             }
             else if (fluxL < 0) /*  seepage flux from cell to canal */
             {
                 if ( tot_head > GW_head ) { /* cell's tot head > its groundwater head */
                 	N_fl = ( SWH[i] > 0 ) ? NA[i]*fluxL/CELL_SIZE/SWH[i] : 0.;
                 	P_fl = ( SWH[i] > 0 ) ? PA[i]*fluxL/CELL_SIZE/SWH[i] : 0.;
                 	S_fl = ( SWH[i] > 0 ) ? SA[i]*fluxL/CELL_SIZE/SWH[i] : 0.;
             		NA[i] += N_fl;
             		PA[i] += P_fl;
             		SA[i] += S_fl;
                 }
                 else {
                  	N_fl = ( GWH[i] > 0 ) ? GNA[i]*fluxL/CELL_SIZE/GWH[i] : 0.;
                 	P_fl = ( GWH[i] > 0 ) ? GPA[i]*fluxL/CELL_SIZE/GWH[i] : 0.;
                 	S_fl = ( GWH[i] > 0 ) ? GSA[i]*fluxL/CELL_SIZE/GWH[i] : 0.;
             		GNA[i] += N_fl;
             		GPA[i] += P_fl;
             		GSA[i] += S_fl;
                }
                 if (Chan_list[chan_n]->basin != basn[i] && debug > -999) {
                     if (Chan_list[chan_n]->family !=  
                         basn_list[basn[i]]->family ) {        /* if the flow is not within the family... */
                         if  ( !basn_list[basn[i]]->parent  ) { /* and if the donor or recipient is a child... */
                         /* then find out about the flow for the family's sake */
                             VOL_OUT_SPG[basn_list[basn[i]]->family] -= fluxL;
                             P_OUT_SPG[basn_list[basn[i]]->family] -= P_fl;
                             S_OUT_SPG[basn_list[basn[i]]->family] -= S_fl;
                         }
                         if ( !Chan_list[chan_n]->parent ) {                 
                           /* then find out about the flow for the family's sake */
                             VOL_IN_SPG[Chan_list[chan_n]->family] -= fluxL; 
                             P_IN_SPG[Chan_list[chan_n]->family] -= P_fl; 
                             S_IN_SPG[Chan_list[chan_n]->family] -= S_fl; 
                         }
                     VOL_IN_SPG[Chan_list[chan_n]->basin] -= fluxL;
                     VOL_OUT_SPG[basn[i]] -= fluxL;
                     P_IN_SPG[Chan_list[chan_n]->basin] -= P_fl;
                     P_OUT_SPG[basn[i]] -= P_fl;
                     S_IN_SPG[Chan_list[chan_n]->basin] -= S_fl;
                     S_OUT_SPG[basn[i]] -= S_fl;
                 }
                 else {    /* if it's flow within a family, just keep
                          track of what the children do among themselves */            
                     if  ( !basn_list[basn[i]]->parent  ) { 
                         /* then find out about the flow for the family's sake */
                         VOL_OUT_SPG[basn[i]] -= fluxL;
                         P_OUT_SPG[basn[i]] -= P_fl;
                         S_OUT_SPG[basn[i]] -= S_fl;
                     }
                     if ( !Chan_list[chan_n]->parent ) {                 
                         /* then find out about the flow for the family's sake */
                         VOL_IN_SPG[Chan_list[chan_n]->basin] -= fluxL; 
                         P_IN_SPG[Chan_list[chan_n]->basin] -= P_fl; 
                         S_IN_SPG[Chan_list[chan_n]->basin] -= S_fl; 
                     }
                 }
                 }
             } 
                 /* now pass that mass flux to/from the canal */
             Chan_list[chan_n]->N -= N_fl;
             Chan_list[chan_n]->P -= P_fl;
             Chan_list[chan_n]->S -= S_fl;
             }
				/* now we recalculate the depths of SFWater and Gwater in cell */
/*              SWH[i] += (fluxS+satTOsf_rec)/CELL_SIZE;  */
/*              GWH[i] += (fluxG-satTOsf_rec)/CELL_SIZE;  */
             SWH[i] += (fluxS)/CELL_SIZE; 
             GWH[i] += (fluxG)/CELL_SIZE; 
             if ( tot_head > GW_head ) SWH[i] += fluxL/CELL_SIZE; /* whether to add seepage to surface or */
             else GWH[i] += fluxL/CELL_SIZE;                      /* groundwater is not critical, as vertical updates will follow */
				

         } /* that's all we do when finally converged */
		    
         T_flux_S += fluxS;
         T_flux_G += fluxG;
         T_flux_L += fluxL;
	  	  
         cell = cell->next_cell;
     } /*end cycle along canal cells */

         
     error = (CanWatDep - Chan_list[chan_n]->wat_depth) - (Qin - Qout - T_flux_S - T_flux_G - T_flux_L)/ Chan_list[chan_n]->area;
     if (converged && error != errorConv ) {
         /* error the second time around should be close to that of the first */
         if (fabs(error-errorConv)>F_ERROR ) { 
             sprintf(msgStr,"Day %6.1f: ERROR - converg error != to error after cell-canal fluxes (Chan %d, est= %5.3f, last depth= %5.3f m)", 
         		SimTime.TIME, Chan_list[chan_n]->number, CanWatDep, Chan_list[chan_n]->wat_depth); 
             WriteMsg( msgStr,True );  dynERRORnum++;
         }
         
         if (CanWatDep>Chan_list[chan_n]->depth*3.0 || CanWatDep < 0.0) {
         	CanWatDep = MINDEPTH; /* lost the mass balance, but let it go for now */
         	sprintf(msgStr,"           Reset Chan %d depth to minimum allowed (%5.3f m)", 
         			Chan_list[chan_n]->number, MINDEPTH); 
         	WriteMsg( msgStr,True );  }
         	 
         	 /*errorConv is the error upon first reaching convergence criteria; */
             /* we then recalc the fluxes in the same loop thru cells (w/ same depth est.), */
             /* and should have the same fluxes and error as before */
         if (debug>4) { /* this here to halt program with high debug */
             printf("\nDay %6.1f: ERROR - converg error != to error after cell-canal fluxes (Chan %d, est= %5.3f m)\n", 
             		SimTime.TIME, Chan_list[chan_n]->number, CanWatDep);
             exit (-1);

         }
         
     }
     
     ++iter;

 } while (!converged );  /* end relaxation procedure */

 if ( factor != 0 && Qout > 0) /* if relaxation was abnormally stopped because of hitting the bottom */
 { 
     sprintf(msgStr,"Day %6.1f: ERROR - hit bottom of Canal %d, stopped the iterative relaxation", SimTime.TIME, Chan_list[chan_n]->number); 
     WriteMsg( msgStr,True );     
     dynERRORnum++;         
            /*#######*******########  this old code should not be reached
             the code is questionable - an ERROR message printed
             to file will indicate signif problem due to entrance to this */
     Qout1 = (Chan_list[chan_n]->wat_depth - CanWatDep)*Chan_list[chan_n]->area + Qin - T_flux_S - T_flux_G - T_flux_L;
     structs = struct_first;
     while ( structs != NULL )	 /* recalculate the outflow in the data for possible use in the next canal inflow */
     {	
         if ( structs->canal_fr  == chan_n+CHo )   structs->flow *= Qout1/Qout;
    
         structs = structs->next_in_list;
     }
     Qout = Qout1;
 } 

 Chan_list[chan_n]->wat_depth = CanWatDep;      /* new canal water depth */
 CH_vol =  CanWatDep * Chan_list[chan_n]->area; /* new canal volume */

 if( debug > 3 ) 
 {
         /* concentration in mg/L (kg/m3==g/L, convert P (not S) to mg/L) */
     Chan_list[chan_n]->P_con = (CH_vol > 0) ? (Chan_list[chan_n]->P/CH_vol*1000.0) : 0.0;
     Chan_list[chan_n]->S_con = (CH_vol > 0) ? (Chan_list[chan_n]->S/CH_vol) : 0.0;
     fprintf( canDebugFile, 
              "%4d  %8.3f  %8.4f  %8.4f  %12.1f  %12.1f  %12.1f  %12.1f  %12.1f  %5d  %9.1f  %7.5f \n",
              Chan_list[chan_n]->number, Chan_list[chan_n]->wat_depth, Chan_list[chan_n]->P_con,  
              Chan_list[chan_n]->S_con, T_flux_S, T_flux_G, T_flux_L, Qin, Qout, iter, Chan_list[chan_n]->area, error );
 
 }

 return;
}

/***************************************************************************************/
/*! \brief Surface water exchange between cell and canal 	

	\param delta Head difference between cell and canal (m)
	\param Water Hydraulic radius (m) 
	\param SW_coef Aggregated flow coefficient (m^0.5 * sec)/m^(1/3)
	\return m^3
*/

float f_Manning( float delta, float Water, float SW_coef )
{	float a_delta;
    
	a_delta = Abs (delta);
	return ( sgn( delta ) * SW_coef * pow(Water, GP_mannDepthPow) * sqrt(a_delta) * canstep);

}
/****************************************************************************************/
/*! \brief Subsurface ground water exchange between cell and canal 

	\param dh Head difference between cell and canal (m)
	\param height Height of water (m) 
	\param GW_coef Aggregated flow coefficient (m/d)
	\param I_Length Average length of cell along reach segment
	\return m^3
*/

float f_Ground ( float dh, float height, float GW_coef, float I_Length )
{	
	float  w;

/* porosity already part of GW_coef */
	w = dh * I_Length * GW_coef / (0.5*celWid) * height * canstep; 

	return ( w);

}

/****************************************************************************/
/*! \brief Determine flows through water control structures 				
	\param SWH \ref SURFACE_WAT
	\param Elev \ref SED_ELEV
	\param MC \ref HYD_MANNINGS_N
	\param NA \ref DINdummy
	\param PA \ref TP_SF_WT
	\param SA \ref SALT_SURF_WT

\remarks Database (Structs_attr) formatting notes/requirements:  
\li 1) No particular sequence is necessary, but the sequence of the structures in the 
 CanalData.struct and CanalData.struct_wat files must be the same (and the prog will
 catch and whip you if you try otherwise). 
\li 2) Due to the cycle that checks for multiple historical/other-data outflows from a canal,
 be sure that the database (to CanalData.struct) does not erroneously have 2 types
 (canal and cell) of recipients for a given structure.   
\li 3) There need to be 2 header lines (as described in the dbase: run-name and col IDs). 
\li 4) Flow is expected to ALWAYS be positive, even though there remain checks for reverse 
 flows in the code.  For a two-way structure, designate a separate structure for the
 reverse flow. 
\li 5) As labeled in dbase, the concentration of P and N in structure flows are ug/L (ppb),
 and S is in g/L (ppt).  (The code converts and deals with kg, m^3, and kg/m^3 (==g/L) ).  
\li 6) The water flows in the input data are cfs, converted and used here as m^3/d. 
\li 7) For rule based structures, don't forget to provide the cell location of the structure
 itself so that the elevation at a canal-canal flow structure is available. 
\li 8) ******* You MAY NOT have multiple historical/other-data demands from a particular cell -
 only canals are allowed such multiple demands. 
\li 9) If you want to apply a concentration to an internal structure that is one of multiple
 outflows from a canal, you need to apply that conc to all structures (actually the first
 in the sequence of the database) draining that canal. (This would only be for
 tracer studies etc., all internal model structure concentrations are calc'd). \n 

 Flow to/from the on-map, within-model-domain area, can only occur by
a flow to/from a "cell" mapped at (1,1).  That's based on our standard (0,0) upper left cell: 
the code uses a (1,1) origin - so that (1,1) from a map => (2,2) in the code! 
*/


void Flows_in_Structures (  float *SWH, float *Elev, float *MC, double *NA, double *PA, double *SA )
{
    /*! \par \b Variables local to function
         <em> cell_to, cell_fr, can_to, can_fr </em> The cell or canal the flow goes to, or comes from
       \n\em cell_struct The grid cell location of the water control structure
       \n<em> TWf, HWf </em> The flag to indicate a schedule for the TailWater, HeadWater
       \n<em> HeadH, HeadT, HeadH2, HeadT2 </em> Head in Headwater or Tailwater
       \n<em> CH_vol, cell_vol </em> Canal, cell total water volume 
       \n<em> Nconc, Pconc, Sconc </em> Nitrogen (UNUSED), Phosphorus, Salt/tracer concentration
       \n<em> N_fl, P_fl, S_fl </em> Nitrogen (UNUSED), Phosphorus, Salt/tracer flows
       \n\em chan_over Ratio of canal volume to historical/other-data demands
       \n\em flow_rev Flow reversal check not invoked in current version
       \n\em ChanHistOut Sum of the multiple historical/other-data outflows from canal
       \n<em> HeadH_drop, HeadT_drop </em> Elevation difference between begin & end in (Headwater, Tailwater) canals
       \n\em FFlux Flow (m) calculated between two cells
*/
	struct Structure *structs, *struct_last, *struct_hist_start;
    int cell_to, cell_fr, can_to, can_fr;
    int cell_struct;
    int TWf, HWf;
    double HeadH, HeadT, HeadH2, HeadT2;
    double CH_vol, cell_vol;
    double Nconc, Pconc, Sconc;
    double N_fl, P_fl, S_fl; 
    double chan_over;
    double flow_rev; 
    double ChanHistOut;
    double HeadH_drop, HeadT_drop;
    float FFlux; 

    /* from unitmod.h */
    extern float* boundcond_depth; /* depth in external cells */



    TWf = 0; HWf = 0; /* 0 when there is no schedule graphs for tail or headwater */
    
    structs = struct_first; 

    while ( structs != NULL )
    {       	
        if ( structs->flag < 0 || (structs->flag >1) ) {
            structs->flow = 0.0; /* used in summing up flows for Qin and Qout later */
            structs->conc_P = 0.0;
            structs->conc_N = 0.0;
            structs->conc_S = 0.0;
            goto OUT; 
        } /* nothing to do when structure inactive (neg or >1 flag) */
        
	/*! \note v2.3note re-activated used of head and tail-water stage schedules, for tidal boundary conditions */
        if ( structs->TW_stage == -99 ) /* this means that there is a graph for TW schedules */
        { 
        structs->TW_stage = GetGraph ( structs->TW_graph, ( FMOD(SimTime.TIME,365.0) >0.0 ) ? ( FMOD(SimTime.TIME,365.0) ) : ( 0.0) ); 
        /* add sea level rise */
        structs->TW_stage = structs->TW_stage + GP_SLRise * (SimTime.TIME/365);
        TWf = 1; 
        }
	  
        if ( structs->HW_stage == -99 ) /* this means that there is a graph for HW schedules */
        { 
        structs->HW_stage = GetGraph ( structs->HW_graph, ( FMOD(SimTime.TIME,365.0) >0.0 ) ? ( FMOD(SimTime.TIME,365.0) ) : ( 365.0) ); 
        /* add sea level rise */
        structs->HW_stage = structs->HW_stage + GP_SLRise * (SimTime.TIME/365);
        HWf = 1; }
	
	  
/* CASE 1 */
/*check that this is a canal-to-canal structure */
/*  these flows are always internal to model domain */
        if ( structs->canal_fr  != 0 && structs->canal_to != 0 ) {
            can_fr = structs->canal_fr - CHo;
            can_to = structs->canal_to - CHo;
            cell_struct = T(structs->str_cell_i,structs->str_cell_j); 
     		
                /* first check for historical/sfwmm flow */
            if ( structs->flag == 1 ) {
                if (structs->multiOut == 1) goto OUT; /* this structure has already been processed as a multiple outflow from a canal */
                    /* FLOW */
                    /* don't do much with zero or negative flows */           
                structs->flow = Hist_data[structs->histID-1].arrayWat[((int)(SimTime.TIME))]*canstep;
                if (structs->flow == 0.0) { 
                    structs->conc_P = 0.0;
                    structs->conc_N = 0.0;
                    structs->conc_S = 0.0;
                    goto OUT; 
                }   /* zero the reported struct flow conc (for output info only) */
                else if (structs->flow < 0.0) {
                    sprintf( msgStr, "Day %6.1f: ERROR IN ASSUMPTION OF NO NEG HIST FLOW from chan %d",
                             SimTime.TIME, Chan_list[can_to]->number );
                    WriteMsg( msgStr,True );
                    dynERRORnum++;
                    goto OUT;
                }

                    /* for multiple historical/other-data outflows from a canal, ensure mass balance (sum of outflows !> avail volume) */
                    /* all flows from the canal associated with each set must be non-negative, which is currently valid */
                    /* the below cycle thru the structures will also process a canal->cell flow structure */

                struct_hist_start = structs; /* this is the first structure w/ historical info for a particular canal */
                ChanHistOut = 0.0; /* sum of all historical/other-data outflows from a canal */

                while ( structs != NULL )   
                {  
                    if  ( structs->flag == 1 && struct_hist_start->canal_fr  == structs->canal_fr   ) { /* look for outflows from same canal */
                        structs->flow = Hist_data[structs->histID-1].arrayWat[((int)(SimTime.TIME))]*canstep;
                        ChanHistOut += structs->flow; /* sum the multiple historical/other-data outflows from canal */
                    }
                        
                    structs = structs->next_in_list; /* increment to next structure */
                } /* end cycle thru list of remaining structures in total list */
                    
                structs = struct_hist_start; /* now we are pointing back to the first (or only) water control structure draining the canal */
                         
                    /* concentration in donor canal for structure flow, g/L = kg/m^3 */
                    /* this is based on previous depth/volume calc'd in FluxChannel */
                    /* or use historical conc */

                    /* CONC */
           			/* canal TOTAL volume before new structure flows */
                CH_vol = Chan_list[can_fr]->area*Chan_list[can_fr]->wat_depth; 
                    /* concentrations applied to all structures draining this canal */
                Pconc = (structs->TPser == -1) ? 
                    ( (CH_vol>0) ? (Chan_list[can_fr]->P/CH_vol) : (0.0) ) :
                    ( (structs->TPser == 0) ? (structs->TP) : ( Hist_data[structs->histID-1].arrayP[((int)(SimTime.TIME))] ) );
                Nconc = (structs->TNser == -1) ? 
                    ( (CH_vol>0) ? (Chan_list[can_fr]->N/CH_vol) : (0.0) ) :
                    ( (structs->TNser == 0) ? (structs->TN) : ( Hist_data[structs->histID-1].arrayN[((int)(SimTime.TIME))] ) );
                Sconc = (structs->TSser == -1) ? 
                    ( (CH_vol>0) ? (Chan_list[can_fr]->S/CH_vol) : (0.0) ) :
                    ( (structs->TSser == 0) ? (structs->TS) : ( Hist_data[structs->histID-1].arrayS[((int)(SimTime.TIME))] ) );


                    /* vol avail for fluxing */
                CH_vol =  Chan_list[can_fr]->area*(ramp(Chan_list[can_fr]->wat_depth-MINDEPTH) );
                chan_over = (ChanHistOut > 0.0 ) ? ( CH_vol/ChanHistOut ) : 1.0;

                do /* loop again through the remaining sequence of structures to correct any outflows for the canal */
                {   
                    if (structs->flag == 1 &&  struct_hist_start->canal_fr  == structs->canal_fr   ) { 

                        structs->multiOut = 1; /* set the flag to indicate this structure has been processed */
                            
                            /* revised FLOW */
                        if ( chan_over < 1.0 ) { /* if hist/other-data demand > canal vol */
                            if( debug > 0 ) { 
                                fprintf( canDebugFile, "Day \t%6.1f\t %s:\tTotDemand= \t%7.0f\t; exceeded chan \t%d\t avail vol= \t%7.0f \tby \t%7.0f \tm3 (\t%5.3f \tm deep).\n",
                                         SimTime.TIME, structs->S_nam, ChanHistOut, Chan_list[can_fr]->number,
                                         CH_vol, (ChanHistOut - CH_vol), Chan_list[can_fr]->wat_depth );
                                
                            }  
                                /* decrement the individual structure outflows by it's proportional contrib. */
                            structs->flow = (structs->flow) * chan_over ;
                        } 
                            /* sum of all realized (corrected for excess demand) historical/other-data outflows from a canal during an iteration */
                            /* used to determine any allowable rule-based canal flows after the historical/other-data demands */
                        Chan_list[can_fr]->sumHistOut += structs->flow;	        

                            /* the recipient changes as we cycle thru the structures (may be another canal or a cell) */
                        can_to = structs->canal_to - CHo;
                        cell_to = T(structs->cell_i_to,structs->cell_j_to); 

                        structs->conc_P = Pconc;/* concentrations applied to all structures draining this canal */
                        structs->conc_N = Nconc;
                        structs->conc_S = (structs->TSser == -1) ?
                            (Sconc) :
                            ( (structs->TSser == 0) ? (structs->TS) :
                              ( Hist_data[structs->histID-1].arrayS[((int)(SimTime.TIME))] ) ); /* see note below on tracer conc. application */
                        
                        P_fl = structs->flow * structs->conc_P; 
                        N_fl = structs->flow * structs->conc_N;
                        S_fl = structs->flow * structs->conc_S;
                                /* change in nut mass in fr canal (kg) */
                        Chan_list[can_fr]->N -=  N_fl ;
                        Chan_list[can_fr]->P -=  P_fl ;
                        Chan_list[can_fr]->S -=  (structs->TSser == -1) ? (S_fl) : (0);
                            /* for the salt (S), we sometimes want to apply a tracer to internal canals/cells.
                               Thus, in that case, we don't subtract salt from the canal because this is "introduced" salt
                               that was not calculated from the available water and salt (i.e., the conc. was from the
                               CanalData.struct data file, either fixed or a time series) */
                            
                        if (can_to > 0) {    /* IF this is a canal recip, change in nut mass in to canal (kg) */
                                /* sum of all realized (corrected for excess demand) historical/other-data outflows into a canal during an iteration */
                                /* used to determine any allowable rule-based canal flows after the historical/other-data demands */
                            Chan_list[can_to]->sumHistIn += structs->flow;	        

                            Chan_list[can_to]->N +=  N_fl ;
                            Chan_list[can_to]->P +=  P_fl ;
                            Chan_list[can_to]->S +=  S_fl ;
                                /* mass balance in fr and to canals */
                                /* canals themselves always belong to a parent hydro basin,
                                   thus canal-canal struct flows do not keep track of child-parent, child-child flows */
                            if (Chan_list[can_fr]->basin != Chan_list[can_to]->basin && debug > -999 ) {
                                VOL_OUT_STR[Chan_list[can_fr]->basin] += structs->flow;
                                VOL_IN_STR[Chan_list[can_to]->basin] += structs->flow; 
                                P_OUT_STR[Chan_list[can_fr]->basin] += P_fl;
                                P_IN_STR[Chan_list[can_to]->basin] += P_fl; 
                                S_OUT_STR[Chan_list[can_fr]->basin] += (structs->TSser == -1) ? (S_fl) : (0); /* see note above on tracer conc. application */
                                S_IN_STR[Chan_list [can_to]->basin] += S_fl; 
                            }


                                             
                        }
                        else if (cell_to > 0) {
                                /* OR change in nut mass in downstream internal cell (kg) */
 
                            if (structs->cell_i_to > 2 ) { /* to an on-map cell */
                                SWH[cell_to] += structs->flow/CELL_SIZE;
                                PA[cell_to] += P_fl; 
                                NA[cell_to] += N_fl; 
                                SA[cell_to] += S_fl; 
                                    /* mass balance in fr canal and to cell */
                                if (Chan_list[can_fr]->basin != basn[cell_to] && debug > -999 ) {
                                    VOL_OUT_STR[Chan_list[can_fr]->basin] += structs->flow;
                                    VOL_IN_STR[basn[cell_to]] += structs->flow; 
                                    P_OUT_STR[Chan_list[can_fr]->basin] += P_fl;
                                    P_IN_STR[basn[cell_to]] += P_fl; 
                                    S_OUT_STR[Chan_list[can_fr]->basin] += (structs->TSser == -1) ? (S_fl) : (0); /* see note above on tracer conc. application */
                                    S_IN_STR[basn[cell_to]] += S_fl; 

                                        /* all structure flows, EXCEPT canal-canal, must be between parent (family) hydro basins */
                                    if ( (Chan_list[can_fr]->family ==  basn_list[basn[cell_to]]->family) && (debug > 0) ) {
                                        sprintf( msgStr, "Day %6.1f: ERROR in basin configuration for structure from chan %d to cell (%d,%d): flow must be between diff hydrologic basins",
                                                 SimTime.TIME, Chan_list[can_fr]->number,structs->cell_i_to,structs->cell_j_to );
                                        WriteMsg( msgStr,True );
                                        dynERRORnum++;
                                    }
                                }
                                


                            }
                            else if (structs->cell_i_to == 2 && debug > -999 ) { /* to off-map cell, mass balance check */
                                VOL_OUT_STR[Chan_list[can_fr]->basin] += structs->flow; 
                                VOL_OUT_STR[0] += structs->flow; 
                                P_OUT_STR[Chan_list[can_fr]->basin] += P_fl; 
                                P_OUT_STR[0] += P_fl; 
                                S_OUT_STR[Chan_list[can_fr]->basin] += (structs->TSser == -1) ? (S_fl) : (0); /* see note above on tracer conc. application */
                                S_OUT_STR[0] += (structs->TSser == -1) ? (S_fl) : (0); /* see note above on tracer conc. application */
                            }
                        }
                            /* don't do much with zero or negative flows */           
                        if (structs->flow == 0.0) {
                            structs->conc_P = 0.0;
                            structs->conc_N = 0.0;
                            structs->conc_S = 0.0;
                        } /* for output purposes only, zero the reported conc */
                        else if (structs->flow < 0.0) {
                            sprintf( msgStr, "Day %6.1f: ERROR IN ASSUMPTION OF NO NEG HIST FLOW from another cell/chan into chan %d",
                                     SimTime.TIME,Chan_list[can_fr]->number );
                            WriteMsg( msgStr,True );
                            dynERRORnum++;
                        }
                    }

                    structs = structs->next_in_list;
                } while (structs != NULL); /*  cycling to the end of the list again !!!! */

                structs = struct_hist_start; /* now pointing back to the first historical outflow structure that we dealt with */

                goto OUT;
            } /* finished check for multiple (positive) outflows exceeding canal volume */

            else  { /* Rule- based flow calcs */

                HeadH_drop = 0.5 * Abs(Chan_list[can_fr]->elev_drop); /* elevation difference between begin&end of canal */
                HeadT_drop = 0.5 * Abs(Chan_list[can_to]->elev_drop);
                
                HeadH = HeadH_drop + /* add portion of the elevation drop along the canal length */
                    Elev[cell_struct] - Chan_list[can_fr]->depth + Chan_list[can_fr]->wat_depth 
                    + (Chan_list[can_fr]->sumHistIn - Chan_list[can_fr]->sumHistOut)/Chan_list[can_fr]->area; 
				/* in both head and tail, add net historical/other-data flows before determining hydraulic potential */
                    /* in tailwater only, check to see if other virtual struct has added water already, add to head */
                HeadT = -HeadT_drop + /* subtract portion of the elevation drop along the canal length */
                    Elev [cell_struct] - Chan_list[can_to]->depth + Chan_list[can_to]->wat_depth
                    + ( Chan_list[can_to]->sumRuleIn + Chan_list[can_to]->sumHistIn - Chan_list[can_to]->sumHistOut)/Chan_list[can_to]->area;

                    /* FLOW */
                if ( structs->w_coef == 0.0)  {/* check for a virtual structure, flow only in pos direction */
                    if ( HeadH>HeadT ) {	 
                        structs->flow =
                            Chan_list[can_fr]->area*Chan_list[can_to]->area/
                            (Chan_list[can_fr]->area+Chan_list[can_to]->area) * (HeadH-HeadT) ;
                        if (debug >3) {
                            HeadH2 = HeadH - structs->flow/Chan_list[can_fr]->area; 
                            HeadT2 = HeadT + structs->flow/Chan_list[can_to]->area;
                            sprintf( msgStr, "Day %6.1f:  virtual flow from chan %d (%f m) to chan %d, New headH-headT=%f, Orig HeadH-HeadT=%f",
                                     SimTime.TIME,Chan_list[can_fr]->number,HeadH-Elev[cell_struct]+Chan_list[can_fr]->depth,
                                     Chan_list[can_to]->number,(HeadH2-HeadT2),(HeadH-HeadT) );
                            WriteMsg( msgStr,True ); 
                            if (Abs(HeadH2 - HeadT2) > GP_MinCheck) /* virtual flows should exactly equilibrate the two canals */
                            { sprintf( msgStr, "Day %6.1f: ERROR virtual flow from chan %d to chan %d, New headH-headT=%f, Orig HeadH-HeadT=%f",
                                       SimTime.TIME,Chan_list[can_fr]->number,Chan_list[can_to]->number,(HeadH2-HeadT2),(HeadH-HeadT) );
                            WriteMsg( msgStr,True ); dynERRORnum++; }

                        }
                    	
                        CH_vol =  Chan_list[can_fr]->area* ramp((HeadH-HeadH_drop)-Elev[cell_struct]+Chan_list[can_fr]->depth-MINDEPTH); /* avail flow volume */
                        if (structs->flow > CH_vol) {
                            structs->flow =  CH_vol;
                            HeadH2 = HeadH - structs->flow/Chan_list[can_fr]->area; 
                            HeadT2 = HeadT + structs->flow/Chan_list[can_to]->area;
                            if (debug>3 && Abs(HeadH2 - HeadT2) > GP_MinCheck) {
                                sprintf( msgStr, "Day %6.1f: Warning: stage didn't equilibrate in modified virtual struct flow from chan %d (%f m) to chan %d, New headH-headT=%f, Orig HeadH-HeadT=%f",
                                         SimTime.TIME,Chan_list[can_fr]->number,CH_vol/Chan_list[can_fr]->area,Chan_list[can_to]->number,(HeadH2-HeadT2),(HeadH-HeadT) );
                                WriteMsg( msgStr,True ); }
                        }
                        
                            /* revise volume to be TOTAL volume for conc calc */
                        CH_vol = CH_vol + MINDEPTH*Chan_list[can_fr]->area;
                            /* conc in donor (fr) canal (kg/m3) */
                        structs->conc_P = (CH_vol>0) ? (Chan_list[can_fr]->P/CH_vol) : (0.0);
                        structs->conc_N = (CH_vol>0) ? (Chan_list[can_fr]->N/CH_vol) : (0.0);
                        structs->conc_S = (CH_vol>0) ? (Chan_list[can_fr]->S/CH_vol) : (0.0);
                    }
                    else  structs->flow = 0.0;
                    
                }
                
                else
                {
                    printf ("\nVirtual structs are the only rule-based canal-canal flows allowed in this version!\n"); exit(-1);
                        /* the code that was here is in older version,
                           not fully implemented (ELM driven by historical/other-data flows, only canal-canal virtual structs allowed) */
                    
                } /* end of rule based flow calcs */
 
                    /* don't do much with zero or negative rule-based flows  */           
                if (structs->flow == 0.0) {
                    structs->conc_P = 0.0;
                    structs->conc_N = 0.0;
                    structs->conc_S = 0.0;
                    goto OUT;
                } /* for output purposes only, zero the reported conc */
                else if (structs->flow < 0.0 && structs->w_coef != 0) { 		
                    sprintf( msgStr, "Day %6.1f: ERROR IN ASSUMPTION OF NO NEG RULE-BASED FLOW from chan %d",
                             SimTime.TIME,Chan_list[can_to]->number);
                    WriteMsg( msgStr,True );
                    goto OUT;
                } 
								

                    /* increment the sum of rule-based inflows/outflows to canals
                       for potential multiple virtual inflows into some canals */
                Chan_list[can_fr]->sumRuleOut += structs->flow;	        
                Chan_list[can_to]->sumRuleIn += structs->flow;	        

                    /* nut flows */
                P_fl = structs->flow * structs->conc_P; 
                N_fl = structs->flow * structs->conc_N;
                S_fl = structs->flow * structs->conc_S;
                    /* change in nut mass in fr canal (kg) */
                Chan_list[can_fr]->N -=  N_fl ;
                Chan_list[can_fr]->P -=  P_fl ;
                Chan_list[can_fr]->S -=  S_fl ;
                    /* change in nut mass in to canal (kg) */
                Chan_list[can_to]->N +=  N_fl ;
                Chan_list[can_to]->P +=  P_fl ;
                Chan_list[can_to]->S +=  S_fl ;
                    /* mass balance in fr and to canals */
                    /* virtual structure flows can not (by definition) establish flow among basins */
                if (Chan_list[can_fr]->basin != Chan_list[can_to]->basin && debug > 0 ) {
                    sprintf( msgStr, "Day %6.1f: ERROR - no hydrologic basin-basin flows in virtual structure, canals %d - %d",
                             SimTime.TIME, Chan_list[can_to]->number, Chan_list[can_fr]->number);
                    WriteMsg( msgStr,True ); dynERRORnum++;
                }
         
                goto OUT;
            }
        }

/* CASE 2 */
/*check that this is a structure between a cell and a cell*/
        if ( structs->cell_i_to != 0 && structs->cell_j_to != 0  
             && structs->cell_i_fr != 0 && structs->cell_j_fr != 0 ) { 

            cell_to = T(structs->cell_i_to,structs->cell_j_to); 
            cell_fr = T(structs->cell_i_fr,structs->cell_j_fr);

                /* first check for historical (or other data) flow */
            if ( structs->flag == 1 ) {
                    /* FLOW */
                    /* don't do much with zero or negative flows */
                structs->flow = Hist_data[structs->histID-1].arrayWat[((int)(SimTime.TIME))]*canstep;
                if (structs->flow == 0.0) {
                    structs->conc_P = 0.0;
                    structs->conc_N = 0.0;
                    structs->conc_S = 0.0;
                    goto OUT;    /* for output purposes only, zero the reported conc */
                }
                else if (structs->flow < 0.0) { /* all flows should be positive, but check */
                    sprintf( msgStr, "Day %6.1f: ERROR IN ASSUMPTION OF NO NEG HIST FLOW from cell (%d,%d)",
                             SimTime.TIME, structs->cell_i_to, structs->cell_j_to);
                    WriteMsg( msgStr,True );
                    goto OUT;
                }
                    /* for internal cells, check for excessive historical/model demand */
                if (structs->cell_i_fr != 2 ) 
                {

                    cell_vol = SWH[cell_fr] * CELL_SIZE;
                    if (structs->flow > cell_vol) { /* if hist/other-data demand > cell vol */
                        if( debug > 0 ) { 
                            fprintf( canDebugFile, "Day \t%6.1f\t %s:\tTotDemand= \t%7.0f\t; exceeded cell \t(%d,%d)\t vol= \t%7.0f \tby \t%7.0f\t m3.\n",
                                     SimTime.TIME, structs->S_nam, structs->flow, structs->cell_i_fr, structs->cell_j_fr, cell_vol, (structs->flow - cell_vol) );
                           
                        }
                            
                        structs->flow = cell_vol;
                    }}
                    /* CONCENTRATION */
                    /* now calculate the nutrient conc in the donor cell */
                    /* nut conc in flow from outside system using historical data or 0.0 */
                if (structs->cell_i_fr == 2) { 
                    structs->conc_P = (structs->TPser == 1)  ?
                        Hist_data[structs->histID-1].arrayP[((int)(SimTime.TIME))] :
                        ((structs->TPser == 0) ? (structs->TP) : (0.0) );
                    structs->conc_N = (structs->TNser == 1)  ?
                        Hist_data[structs->histID-1].arrayN[((int)(SimTime.TIME))] :
                        ((structs->TNser == 0) ? (structs->TN) : (0.0) );
                    structs->conc_S = (structs->TSser == 1)  ?
                        Hist_data[structs->histID-1].arrayS[((int)(SimTime.TIME))] :
                        ((structs->TSser == 0) ? (structs->TS) : (0.0) );
                }
                else
                { /* nut conc in flow within system - check to see if there are historical concs with the historical flow */
                    structs->conc_P = (structs->TPser == -1)  ?
                        ( (SWH[cell_fr] > 0) ? PA[cell_fr]/CELL_SIZE/SWH[cell_fr] : 0.0 ) :
                        ( (structs->TPser == 0) ? (structs->TP) : (Hist_data[structs->histID-1].arrayP[((int)(SimTime.TIME))] ) );
                    structs->conc_N = ( structs->TNser == -1)  ?
                        ( (SWH[cell_fr] > 0) ? NA[cell_fr]/CELL_SIZE/SWH[cell_fr] : 0.0 ) :
                        ( (structs->TNser == 0) ? (structs->TN) : (Hist_data[structs->histID-1].arrayN[((int)(SimTime.TIME))] ) );
                    structs->conc_S = ( structs->TSser == -1)  ?
                        ( (SWH[cell_fr] > 0) ? SA[cell_fr]/CELL_SIZE/SWH[cell_fr] : 0.0 ) :
                        ( (structs->TSser == 0) ? (structs->TS) : (Hist_data[structs->histID-1].arrayS[((int)(SimTime.TIME))] ) );
                }
                    /* the nut and water update done after the rule-based flow calcs */
/* determine mass transfer */
                    
                P_fl = structs->flow * structs->conc_P;
                N_fl = structs->flow * structs->conc_N;
                S_fl = structs->flow * structs->conc_S;
                
                
                if (structs->cell_i_to == 2) { /* "to" cell is outside system */
                    if (debug > -999) {/* m3 and kg nuts outflow out from the system */
                        VOL_OUT_STR[basn[cell_fr]] += structs->flow; /* mass balance */
                        VOL_OUT_STR[0] += structs->flow; 
                        P_OUT_STR[basn[cell_fr]] += P_fl; 
                        P_OUT_STR[0] += P_fl; 
                        S_OUT_STR[basn[cell_fr]] += (structs->TSser == -1) ? (S_fl) : (0); /* see below note on tracer application */
                        S_OUT_STR[0] += (structs->TSser == -1) ? (S_fl) : (0); /* see below note on tracer application */
                    } 
                    PA[cell_fr] -= P_fl;
                    NA[cell_fr] -= N_fl; 
                    SA[cell_fr] -= (structs->TSser == -1) ? (S_fl) : (0);
                        /* for the salt (S), we sometimes want to apply a tracer to internal canals/cells.
                           Thus, in that case, we don't subtract salt from the canal because this is "introduced" salt
                           that was not calculated from the available water and salt (i.e., the conc. was from the
                           CanalData.struct data file, either fixed or a time series) */
                        /* recalculate the surface water depth in the "from" cell */
                    SWH[cell_fr] -= structs->flow/CELL_SIZE;
                }
                else if (structs->cell_i_fr == 2) {  /* "from" cell is outside system */
                    if (debug > -999) {/* m3 and kg nuts inflow to the system */
                        VOL_IN_STR[basn[cell_to]] += structs->flow; 
                        VOL_IN_STR[0] += structs->flow; 
                        P_IN_STR[basn[cell_to]] += P_fl; 
                        P_IN_STR[0] += P_fl; 
                        S_IN_STR[basn[cell_to]] += S_fl; 
                        S_IN_STR[0] += S_fl; 
                    } 
                    PA[cell_to] += P_fl;
                    NA[cell_to] += N_fl;
                    SA[cell_to] += S_fl;
                        /* recalculate the surface water depth in the "to" cell */
                    SWH[cell_to] += structs->flow/CELL_SIZE;
                }
                else { /* internal cell-cell m3 and kg nuts flow */
                    PA[cell_fr] -= P_fl;
                    NA[cell_fr] -= N_fl; 	
                    SA[cell_fr] -= (structs->TSser == -1) ? (S_fl) : (0);
                        /* for the salt (S), we sometimes want to apply a tracer to internal canals/cells.
                           Thus, in that case, we don't subtract salt from the cell because this is "introduced" salt
                           that was not calculated from the available water and salt (i.e., the conc. was from the
                           CanalData.struct data file, either fixed or a time series) */
                    
                    PA[cell_to] += P_fl;
                    NA[cell_to] += N_fl;
                    SA[cell_to] += S_fl;
                        /* recalculate the surface water depth in the interacting cells */
                    SWH[cell_fr] -= structs->flow/CELL_SIZE;
                    SWH[cell_to] += structs->flow/CELL_SIZE;
                    if (basn[cell_to] != basn[cell_fr] && debug > -999 ) {
                        VOL_OUT_STR[basn[cell_fr]] += structs->flow; 
                        VOL_IN_STR[basn[cell_to]] += structs->flow; 
                        P_OUT_STR[basn[cell_fr]] += P_fl; 
                        P_IN_STR[basn[cell_to]] += P_fl; 
                        S_OUT_STR[basn[cell_fr]] += (structs->TSser == -1) ? (S_fl) : (0); /* see above note on tracer application */
                        S_IN_STR[basn[cell_to]] += S_fl; 

                            /* all structure flows, EXCEPT canal-canal, must be between parent (family) hydro basins */
                        if ( (basn_list[basn[cell_fr]]->family ==  basn_list[basn[cell_to]]->family) && (debug > 0) ) {
                            sprintf( msgStr, "Day %6.1f: ERROR in basin configuration for structure from cell (%d,%d) to cell (%d,%d): flow must be between diff hydrologic basins",
                                     SimTime.TIME, structs->cell_i_fr,structs->cell_j_fr,structs->cell_i_to,structs->cell_j_to );
                            WriteMsg( msgStr,True ); dynERRORnum++;
                        }
                    }
                    
                }
                goto OUT;
            }
            
                /**** Rule- based flows */
            else {  
                if ( structs->w_coef != 0.0)  {/* check for a virtual structure  */
                    printf ("\nVirtual structs are the only rule-based cell-cell flows in this version!\n"); exit(-1);
                }
                else {
                        /* calc'd flow, cell-cell, using the functions defined in Fluxes.c for overland flow, bridge-flow only app right now */
                        /* MCopen is a (whole array) of Manning's coefs that are initialized at an open-water value */
                    FFlux = Flux_SWcells(structs->cell_i_fr,structs->cell_j_fr,
                                      structs->cell_i_to,structs->cell_j_to,SWH,Elev,MCopen);
                        /* flow may be negative */
 				/* FFlux units = m */
                    structs->flow = FFlux*CELL_SIZE;
                    structs->conc_P = (SWH[cell_fr] > 0) ? PA[cell_fr]/CELL_SIZE/SWH[cell_fr] : 0.0 ;
/*                     structs->conc_N = (SWH[cell_fr] > 0) ? NA[cell_fr]/CELL_SIZE/SWH[cell_fr] : 0.0 ; */
                    structs->conc_S = (SWH[cell_fr] > 0) ? SA[cell_fr]/CELL_SIZE/SWH[cell_fr] : 0.0 ;

                    if (FFlux != 0.0) Flux_SWstuff ( structs->cell_i_fr,structs->cell_j_fr,
                                                  structs->cell_i_to,structs->cell_j_to,FFlux,SWH,SA,NA,PA);
                    
                    
                }
                goto OUT;
            } /* end of rule-based flow calc */
            
        }

/* CASE 3 */
/*check that this is a structure between a canal and a cell*/
        if ( structs->canal_fr  != 0 && structs->cell_i_to != 0 && structs->cell_j_to != 0 ) { 
            can_fr = structs->canal_fr  - CHo;
            cell_to = T(structs->cell_i_to,structs->cell_j_to); 	  
            cell_struct = T(structs->str_cell_i,structs->str_cell_j);

                /* first check for historical (or other data) flow */
            if ( structs->flag == 1 ) {

                if (structs->multiOut == 1) goto OUT; /* this structure has already been processed as a multiple outflow from a canal */
            	
                structs->flow = Hist_data[structs->histID-1].arrayWat[((int)(SimTime.TIME))]*canstep;
                    /* don't do much with zero or negative flows */           
                if (structs->flow == 0.0) {/* zero the reported struct flow conc (for output info only) */
                    structs->conc_P = 0.0;
                    structs->conc_N = 0.0;
                    structs->conc_S = 0.0;
                    goto OUT; 
                }
                else if (structs->flow < 0.0) { 
                    sprintf( msgStr, "Day %6.1f: ERROR IN ASSUMPTION OF NO NEG HIST FLOW from cell (%d,%d)",
                             SimTime.TIME, structs->cell_i_to, structs->cell_j_to );
                    WriteMsg( msgStr,True ); dynERRORnum++;
                    goto OUT;
                }
                
                    /* for multiple historical/other-data outflows from a canal, ensure mass balance (sum of outflows !> avail volume) */
                    /* all flows from the canal associated with each set must be non-negative, which is currently valid */

                struct_hist_start = structs; /* this is the first structure w/ historical info for a particular canal */
                ChanHistOut = 0.0; /* sum of all historical outflows from a canal */

                while ( structs != NULL )   
                {  
                    if  ( structs->flag == 1 && struct_hist_start->canal_fr  == structs->canal_fr   ) { /* look for outflows from same canal */
                        structs->flow = Hist_data[structs->histID-1].arrayWat[((int)(SimTime.TIME))]*canstep;
                        ChanHistOut += structs->flow; /* sum the multiple historical outflows from canal */
                    }
                        
                    structs = structs->next_in_list; /* increment to next structure */
                } /* end cycle thru list of remaining structures in total list */
                    
                structs = struct_hist_start; /* now we are pointing back to the first (or only) water control structure draining the canal */                 

                    /* concentration in donor canal for structure flow, g/L = kg/m^3 */
                    /* this is based on previous depth/volume calc'd in FluxChannel */
                    /* or use historical conc */
                        /* canal TOTAL volume before new structure flows */
                CH_vol = Chan_list[can_fr]->area*Chan_list[can_fr]->wat_depth; 
                    /* concentrations applied to all structures draining this canal */
                Pconc = (structs->TPser == -1) ? 
                    ( (CH_vol>0) ? (Chan_list[can_fr]->P/CH_vol) : (0.0) ) :
                    ( (structs->TPser == 0) ? (structs->TP) : (Hist_data[structs->histID-1].arrayP[((int)(SimTime.TIME))] ) ) ;
                Nconc = (structs->TNser == -1) ? 
                    ( (CH_vol>0) ? (Chan_list[can_fr]->N/CH_vol) : (0.0) ) :
                    ( (structs->TNser == 0) ? (structs->TN) : (Hist_data[structs->histID-1].arrayN[((int)(SimTime.TIME))] ) ) ;
                Sconc = (structs->TSser == -1) ? 
                    ( (CH_vol>0) ? (Chan_list[can_fr]->S/CH_vol) : (0.0) ) :
                    ( (structs->TSser == 0) ? (structs->TS) : (Hist_data[structs->histID-1].arrayS[((int)(SimTime.TIME))] ) ) ;

                   /* vol avail for fluxing */
                CH_vol =  Chan_list[can_fr]->area*(ramp(Chan_list[can_fr]->wat_depth-MINDEPTH) );
                chan_over = (ChanHistOut > 0.0 ) ? ( CH_vol/ChanHistOut ) : 1.0;

                do /* loop again through the remaining sequence of structures to correct any outflows for the canal */
                {   
                    if ( structs->flag == 1 && struct_hist_start->canal_fr  == structs->canal_fr   ) { 

                        structs->multiOut = 1; /* flag to indicate this structure has been processed (below) */

                            /* revised FLOW */
                        if ( chan_over < 1.0 ) { /* if hist/other-data demand > canal vol */
                            if( debug > 0 ) { 
                                fprintf( canDebugFile, "Day \t%6.1f\t %s:\tTotDemand= \t%7.0f\t; exceeded chan \t%d\t avail vol= \t%7.0f\t by \t%7.0f\t m3 (\t%5.3f\t m deep).\n",
                                         SimTime.TIME, structs->S_nam, ChanHistOut, Chan_list[can_fr]->number,
                                         CH_vol, (ChanHistOut - CH_vol), Chan_list[can_fr]->wat_depth );
                                
                            }  
                                /* decrement the individual structure outflows by it's proportional contrib. */
                            structs->flow = (structs->flow) * chan_over ;
                        } 
                            /* sum of all realized (corrected for excess demand) historical/other-data outflows from a canal during an iteration */
                            /* used to determine any allowable rule-based canal flows after the historical/other-data demands */
                        Chan_list[can_fr]->sumHistOut += structs->flow;	        

                            /* the recipient changes as we cycle thru the structures (may be another cell or a canal) */
                        can_to = structs->canal_to - CHo;
                        cell_to = T(structs->cell_i_to,structs->cell_j_to); 

                        structs->conc_P = Pconc;/* concentrations applied to all structures draining this canal */
                        structs->conc_N = Nconc;
                        structs->conc_S = (structs->TSser == -1) ?
                            (Sconc) :
                            ( (structs->TSser == 0) ? (structs->TS) :
                            ( Hist_data[structs->histID-1].arrayS[((int)(SimTime.TIME))] ) ); /* see note below on tracer conc. application */
                         
                        P_fl = structs->flow * structs->conc_P; 
                        N_fl = structs->flow * structs->conc_N;
                        S_fl = structs->flow * structs->conc_S;
                                /* change in nut mass in fr canal (kg) */
                        Chan_list[can_fr]->N -=  N_fl ;
                        Chan_list[can_fr]->P -=  P_fl ;
                        Chan_list[can_fr]->S -=  (structs->TSser == -1) ? (S_fl) : (0);
                            /* for the salt (S), we sometimes want to apply a tracer to internal canals/cells.
                               Thus, in that case, we don't subtract salt from the canal because this is "introduced" salt
                               that was not calculated from the available water and salt (i.e., the conc. was from the
                               CanalData.struct data file, either fixed or a time series) */
  
                        if (can_to > 0) {    /* IF this is a canal recip, change in nut mass in to canal (kg) */
                            /* sum of all realized (corrected for excess demand) historical/other-data outflows into a canal during an iteration */
                            /* used to determine any allowable rule-based canal flows after the historical/other-data demands */
                        	Chan_list[can_to]->sumHistIn += structs->flow;	        

                            Chan_list[can_to]->N +=  N_fl ;
                            Chan_list[can_to]->P +=  P_fl ;
                            Chan_list[can_to]->S +=  S_fl ;
                                /* mass balance in fr and to canals */
                            if (Chan_list[can_fr]->basin != Chan_list[can_to]->basin && debug > -999 ) {
                                VOL_OUT_STR[Chan_list[can_fr]->basin] += structs->flow;
                                VOL_IN_STR[Chan_list [can_to]->basin] += structs->flow; 
                                P_OUT_STR[Chan_list[can_fr]->basin] += P_fl;
                                P_IN_STR[Chan_list [can_to]->basin] += P_fl; 
                                S_OUT_STR[Chan_list[can_fr]->basin] += (structs->TSser == -1) ? (S_fl) : (0); /* see above note on tracer application */
                                S_IN_STR[Chan_list [can_to]->basin] += S_fl; 
                            }
                        }
                        else if (cell_to > 0) {
                                /* OR change in nut mass in downstream cell (kg) */
 
                            if (structs->cell_i_to > 2 ) { /* to an on-map cell */
                                SWH[cell_to] += structs->flow/CELL_SIZE;
                                PA[cell_to] += P_fl; 
                                NA[cell_to] += N_fl; 
                                SA[cell_to] += S_fl; 
                                    /* mass balance in fr canal and to cell */
                                if (Chan_list[can_fr]->basin != basn[cell_to] && debug > -999 ) {
                                    VOL_OUT_STR[Chan_list[can_fr]->basin] += structs->flow;
                                    VOL_IN_STR[basn[cell_to]] += structs->flow; 
                                    P_OUT_STR[Chan_list[can_fr]->basin] += P_fl;
                                    P_IN_STR[basn[cell_to]] += P_fl; 
                                    S_OUT_STR[Chan_list[can_fr]->basin] += (structs->TSser == -1) ? (S_fl) : (0); /* see above note on tracer application */
                                    S_IN_STR[basn[cell_to]] += S_fl; 

                                        /* all structure flows, EXCEPT canal-canal, must be between parent (family) hydro basins */
                                    if ( (Chan_list[can_fr]->family ==  basn_list[basn[cell_to]]->family) && (debug > 0) ) {
                                        sprintf( msgStr, "Day %6.1f: ERROR in basin configuration for structure from chan %d to cell (%d,%d): flow must be between diff hydrologic basins",
                                                 SimTime.TIME, Chan_list[can_fr]->number,structs->cell_i_to,structs->cell_j_to );
                                        WriteMsg( msgStr,True ); dynERRORnum++;
                                    }
                                }
                            }
                            else if (structs->cell_i_to == 2 && debug > -999 ) { /* to off-map cell, mass balance check */
                                VOL_OUT_STR[Chan_list[can_fr]->basin] += structs->flow; 
                                VOL_OUT_STR[0] += structs->flow; 
                                P_OUT_STR[Chan_list[can_fr]->basin] += P_fl; 
                                P_OUT_STR[0] += P_fl; 
                                S_OUT_STR[Chan_list[can_fr]->basin] += (structs->TSser == -1) ? (S_fl) : (0); /* see above note on tracer application */
                                S_OUT_STR[0] += (structs->TSser == -1) ? (S_fl) : (0); /* see above note on tracer application */
                            }
                        }
                            /* don't do much with zero or negative flows */           
                        if (structs->flow == 0.0) {
                            structs->conc_P = 0.0;
                            structs->conc_N = 0.0;
                            structs->conc_S = 0.0;
                        } /* for output purposes only, zero the reported conc */
                        else if (structs->flow < 0.0) {
                            sprintf( msgStr, "Day %6.1f: ERROR IN ASSUMPTION OF NO NEG HIST FLOW from another chan/cell into chan %d",
                                     SimTime.TIME, Chan_list[can_fr]->number );
                            WriteMsg( msgStr,True ); dynERRORnum++;
                        }
                    }

                    structs = structs->next_in_list;
                } while (structs != NULL); /*  cycling to the end of the list again !!!! */

                structs = struct_hist_start; /* now pointing back to the first historical outflow structure that we dealt with */
    

                goto OUT;
                    /* finished check for multiple (positive) outflows exceeding canal volume */
            }
                /* FLOW */
                /**** Rule- based flows */
            else { 
                HeadH = Elev [cell_struct] - Chan_list[can_fr]->depth + Chan_list[can_fr]->wat_depth
                		+ (Chan_list[can_fr]->sumHistIn - Chan_list[can_fr]->sumHistOut)/Chan_list[can_fr]->area; 
	  /* in v2.3, the rule-based canal->cell flows use either tide data or SFWMM/other model data for boundary conditions */
                
               /* v2.2 and previous eqn: HeadT = ( ( structs->cell_i_to == 2 ) ? (Elev[cell_struct] - 0.1) : (SWH[cell_to]+Elev[cell_to]) ); */
                /* Head at Tailwater location: check that destination is out of model domain (always in v2.3 and prior versions)
                	The TWf (TailWater flag) indicates use of annually-recurring tidal data, with Sea Level Rise added.
                	If the TWf is false, use relative depth data from the SFWMM or other model output (boundcond_depth data is NOT stage, but depth relative to elev NGVD'29) */
                	/* v2.5 use cell_i_TW, cell_j_TW in cell_to location */
                HeadT = ( ( structs->cell_i_to == 2 ) ? 
                	( (TWf) ? (structs->TW_stage) : ( boundcond_depth[T(structs->cell_i_TW,structs->cell_j_TW)]  + Elev[cell_struct]) ): 
                	(SWH[cell_to]+Elev[cell_to]) );
				/* volume available for fluxing - right now, the only structures here drain border seepage canals, leave 30 cm water */
                CH_vol =  Chan_list[can_fr]->area* ramp(HeadH-Elev[cell_struct]+Chan_list[can_fr]->depth-0.3); /* avail flow volume */
            
                if (HeadH>HeadT) 
                {
                    /* this ****DOES NOT**** work when the HW and TW checks are being made at a remote cell */
                    /* ok for now, just fix when driving ELM with rules and checking remote cells */
                    structs->flow =  Min(structs->w_coef * ( HeadH - HeadT ) * canstep, CH_vol );                   
                        /* all rule based instances of canal->cell flow  go
                           to external cells, so this reversal check not invoked */
                    flow_rev = ( structs->cell_i_to != 2 ) ? 
                    	(HeadH - structs->flow/Chan_list[can_fr]->area) - (HeadT + structs->flow/CELL_SIZE) :
                    	(0.0);
             		/* the flux eqn is the same as the (new Jul98) virtual weir eqn, equilibrating the heads across two unequal area regions */
                    /* flow is only in positive directioin */
                    if (flow_rev < 0.0) {  structs->flow = 
                        ramp( Chan_list[can_fr]->area*CELL_SIZE*HeadH/(Chan_list[can_fr]->area+CELL_SIZE) -
                        Chan_list[can_fr]->area*CELL_SIZE*HeadT/(Chan_list[can_fr]->area+CELL_SIZE) );
                          }
               }
                else structs->flow = 0.0; /* undefined flow if heads are equal, thus this stmt added */
                
            
            
                    /* don't do much with zero or negative flows */           
                if (structs->flow == 0.0) {
                    structs->conc_P = 0.0;
                    structs->conc_N = 0.0;
                    structs->conc_S = 0.0;
                    goto OUT;
                } /* for output purposes only, zero the reported conc */
                else if (structs->flow < 0) {  
                    sprintf( msgStr, "Day %6.1f: ERROR IN ASSUMPTION OF NO NEG FLOW from cell (%d,%d)",
                             SimTime.TIME, structs->cell_i_to, structs->cell_j_to );
                    WriteMsg( msgStr,True ); dynERRORnum++;
                    goto OUT;
                }
          
                    /* rule-based flows:  calculate the nutrient concentrations */
                    /* conc in canal -> cell flow  */
                CH_vol =  Chan_list[can_fr]->area* ramp(HeadH-Elev[cell_struct]+Chan_list[can_fr]->depth); /* TOTAL volume */
                    /* conc in donor (fr) canal (kg/m3) */
                structs->conc_P = (CH_vol>0) ? (Chan_list[can_fr]->P/CH_vol) : (0.0);
                structs->conc_N = (CH_vol>0) ? (Chan_list[can_fr]->N/CH_vol) : (0.0);
                structs->conc_S = (CH_vol>0) ? (Chan_list[can_fr]->S/CH_vol) : (0.0);

                P_fl = structs->flow * structs->conc_P; 
                N_fl = structs->flow * structs->conc_N;
                S_fl = structs->flow * structs->conc_S;
            	  
                    /* increment the sum of rule-based flows to from canals
                       for potential multiple virtual flows associated with some canals */
                Chan_list[can_fr]->sumRuleOut += structs->flow;	        
                    /* change in nut mass in downstream cell (kg) */
                if (structs->cell_i_to > 2 ) { /* to an on-map cell */
                    SWH[cell_to] += structs->flow/CELL_SIZE;
                    PA[cell_to] += P_fl; 
                    NA[cell_to] += N_fl; 
                    SA[cell_to] += S_fl; 
                        /* mass balance in fr canal and to cell */
                    if (Chan_list[can_fr]->basin != basn[cell_to] && debug > -999 ) {
                        VOL_OUT_STR[Chan_list[can_fr]->basin] += structs->flow;
                        VOL_IN_STR[basn[cell_to]] += structs->flow; 
                        P_OUT_STR[Chan_list[can_fr]->basin] += P_fl;
                        P_IN_STR[basn[cell_to]] += P_fl; 
                        S_OUT_STR[Chan_list[can_fr]->basin] += S_fl;
                        S_IN_STR[basn[cell_to]] += S_fl; 

                            /* all structure flows, EXCEPT canal-canal, must be between parent (family) hydro basins */
                        if ( (Chan_list[can_fr]->family ==  basn_list[basn[cell_to]]->family) && (debug > 0) ) {
                            sprintf( msgStr, "Day %6.1f: ERROR in basin configuration for structure from chan %d to cell (%d,%d): flow must be between diff hydrologic basins",
                                     SimTime.TIME, Chan_list[can_fr]->number,structs->cell_i_to,structs->cell_j_to );
                            WriteMsg( msgStr,True ); dynERRORnum++;
                        }
                    }
                }
                else if (structs->cell_i_to == 2 && debug > -999 ) { /* to off-map cell, mass balance check */
                    VOL_OUT_STR[Chan_list[can_fr]->basin] += structs->flow; 
                    VOL_OUT_STR[0] += structs->flow; 
                    P_OUT_STR[Chan_list[can_fr]->basin] += P_fl; 
                    P_OUT_STR[0] += P_fl; 
                    S_OUT_STR[Chan_list[can_fr]->basin] += S_fl; 
                    S_OUT_STR[0] += S_fl; 
                }

            
                    /* calc the change in mass in  canal (kg) */
                Chan_list[can_fr]->N -=  N_fl ;
                Chan_list[can_fr]->P -=  P_fl ;
                Chan_list[can_fr]->S -=  S_fl ;

                goto OUT;
            }
        }
        
    

/* CASE 4 */
/*check that this is a structure between a cell and a canal*/
            /* if from 1,1 to canal, always will require historical data */
        if ( structs->canal_to != 0 && structs->cell_i_fr != 0 && structs->cell_j_fr != 0 ) { 
            can_to = structs->canal_to - CHo;
            cell_fr = T(structs->cell_i_fr,structs->cell_j_fr); 	  
            cell_struct = T(structs->str_cell_i,structs->str_cell_j);

                /* first check for historical (or other data) flow */
            if ( structs->flag == 1 ) {
  
                if (structs->multiOut == 1) {
                    sprintf( msgStr, "Day %6.1f: ERROR IN ASSUMPTION OF NO NEG HIST FLOW from chan %d",
                             SimTime.TIME, Chan_list[can_to]->number );
                    WriteMsg( msgStr,True ); dynERRORnum++;
                    goto OUT; /* this structure has already been processed as a multiple outflow from a canal */
                }
            	
                structs->flow = Hist_data[structs->histID-1].arrayWat[((int)(SimTime.TIME))]*canstep;
                    /* don't do much with zero or negative flows */           
                if (structs->flow == 0.0) {
                    structs->conc_P = 0.0;
                    structs->conc_N = 0.0;
                    structs->conc_S = 0.0;
                    goto OUT; 
                }
                else if (structs->flow <0) { 
                    sprintf( msgStr, "Day %6.1f: ERROR IN ASSUMPTION OF NO NEG HIST FLOW from chan %d",
                             SimTime.TIME, Chan_list[can_to]->number );
                    WriteMsg( msgStr,True ); dynERRORnum++;
                    goto OUT;
                } 

                if (structs->cell_i_fr != 2 ) /* cell (2,2) == (1,1) in maps, not in model domain */
                {

                    cell_vol = SWH[cell_fr] * CELL_SIZE;
                    if (structs->flow > cell_vol) {
                        if( debug > 0 ) { 
                            fprintf( canDebugFile, "Day \t%6.1f\t %s:\tTotDemand= \t%7.0f\t; exceeded cell \t(%d,%d)\t vol= \t%7.0f\t by \t%7.0f\t m3.\n",
                                     SimTime.TIME, structs->S_nam, structs->flow, structs->cell_i_fr, structs->cell_j_fr, cell_vol, (structs->flow - cell_vol) );
                            
                        }
                        structs->flow = cell_vol;
                    }}
                    /* calculate the nutrient conc in the donor cell */
                    /* nut conc in flow from outside system using historical data or 0.0 */
                if (structs->cell_i_fr == 2) { 
                    structs->conc_P = (structs->TPser == 1)  ?
                        Hist_data[structs->histID-1].arrayP[((int)(SimTime.TIME))] :
                        ((structs->TPser == 0) ? (structs->TP) : (0.0) );
                    structs->conc_N = (structs->TNser == 1)  ?
                        Hist_data[structs->histID-1].arrayN[((int)(SimTime.TIME))] :
                        ((structs->TNser == 0) ? (structs->TN) : (0.0) );
                    structs->conc_S = (structs->TSser == 1)  ?
                        Hist_data[structs->histID-1].arrayS[((int)(SimTime.TIME))] :
                        ((structs->TSser == 0) ? (structs->TS) : (0.0) );
                }
                else
                { /* nut conc in flow within system - check to see if there are historical concs with the historical flow */
                    structs->conc_P = (structs->TPser == -1)  ?
                        ( (SWH[cell_fr] > 0) ? PA[cell_fr]/CELL_SIZE/SWH[cell_fr] : 0.0 ) :
                        ( (structs->TPser == 0) ? (structs->TP) : (Hist_data[structs->histID-1].arrayP[((int)(SimTime.TIME))] ) );
                    structs->conc_N = ( structs->TNser == -1)  ?
                        ( (SWH[cell_fr] > 0) ? NA[cell_fr]/CELL_SIZE/SWH[cell_fr] : 0.0 ) :
                        ( (structs->TNser == 0) ? (structs->TN) : (Hist_data[structs->histID-1].arrayN[((int)(SimTime.TIME))] ) );
                    structs->conc_S = ( structs->TSser == -1)  ?
                        ( (SWH[cell_fr] > 0) ? SA[cell_fr]/CELL_SIZE/SWH[cell_fr] : 0.0 ) :
                        ( (structs->TSser == 0) ? (structs->TS) : (Hist_data[structs->histID-1].arrayS[((int)(SimTime.TIME))] ) );
                }
                    /* now determine the nutrient mass (kg) flow from cell to canal */
                    /* 1000 is to convert g/m3 to kg */	
                P_fl = structs->flow * structs->conc_P;
                N_fl = structs->flow * structs->conc_N;
                S_fl = structs->flow * structs->conc_S;
                if (structs->cell_i_fr == 2 ) {/* "from" cell is outside system */
                    if (debug > -999) {  
                        VOL_IN_STR[Chan_list[can_to]->basin] += structs->flow; /* mass balance */
                        VOL_IN_STR[0] += structs->flow; 
                        P_IN_STR[Chan_list[can_to]->basin] += P_fl; 
                        P_IN_STR[0] += P_fl; 
                        S_IN_STR[Chan_list[can_to]->basin] += S_fl; 
                        S_IN_STR[0] += S_fl; 
                    } }
                else { /* internal cell-> canal flow */
                    SWH[cell_fr] -= structs->flow/CELL_SIZE;
                    PA[cell_fr] -= P_fl; 
                    NA[cell_fr] -= N_fl; 
                    SA[cell_fr] -= (structs->TSser == -1) ? (S_fl) : (0);
                            /* for the salt (S), we sometimes want to apply a tracer to internal canals/cells.
                               Thus, in that case, we don't subtract salt from the canal because this is "introduced" salt
                               that was not calculated from the available water and salt (i.e., the conc. was from the
                               CanalData.struct data file, either fixed or a time series) */
                        /* mass balance in fr cell and to canal */
                    if (basn[cell_fr] != Chan_list[can_to]->basin && debug > -999 ) {
                        VOL_OUT_STR[basn[cell_fr]] += structs->flow;
                        VOL_IN_STR[Chan_list[can_to]->basin] += structs->flow; 
                        P_OUT_STR[basn[cell_fr]] += P_fl;
                        P_IN_STR[Chan_list[can_to]->basin] += P_fl; 
                        S_OUT_STR[basn[cell_fr]] += (structs->TSser == -1) ? (S_fl) : (0); /* see above note on tracer application */
                        S_IN_STR[Chan_list[can_to]->basin] += S_fl; 

                            /* all structure flows, EXCEPT canal-canal, must be between parent (family) hydro basins */
                        if ( (Chan_list[can_to]->family ==  basn_list[basn[cell_fr]]->family) && (debug > 0) ) {
                            sprintf( msgStr, "Day %6.1f: ERROR in basin configuration for structure from cell (%d,%d) to chan %d: flow must be between diff hydrologic basins",
                                     SimTime.TIME, structs->cell_i_fr, structs->cell_j_fr, Chan_list[can_to]->number );
                            WriteMsg( msgStr,True ); dynERRORnum++;
                        }
                    }
                }
            	
                  /* sum of all realized (corrected for excess demand) historical/other-data outflows into a canal during an iteration */
                  /* used to determine any allowable rule-based canal flows after the historical/other-data demands */
                Chan_list[can_to]->sumHistIn += structs->flow;	        

                   /* change in nut mass in "to" canal (kg) */
                Chan_list[can_to]->N +=  N_fl ;
                Chan_list[can_to]->P +=  P_fl ;
                Chan_list[can_to]->S +=  S_fl ;

                goto OUT;
            }

            else { /* rule based flow calc */
                if (structs->cell_i_fr != 2  ) {
                    sprintf( msgStr, "Day %6.1f: ERROR in flow configuration for structure from cell (%d,%d) to chan %d: this rule-based flow is restricted to flow from outside model domain.",
                             SimTime.TIME, structs->cell_i_fr, structs->cell_j_fr, Chan_list[can_to]->number );
                    WriteMsg( msgStr,True );   dynERRORnum++;    
                }
                HeadH = (structs->HW_stage) ; 
	  /* in v2.3, the rule-based cell->canal flows use only tide data for boundary conditions */
                HeadT = Elev [cell_struct] - Chan_list[can_to]->depth + Chan_list[can_to]->wat_depth
                    + (Chan_list[can_to]->sumHistIn - Chan_list[can_to]->sumHistOut)/Chan_list[can_to]->area;
            
                if (HeadH>HeadT) 
                {
                    structs->flow =  Max(structs->w_coef * ( HeadH - HeadT ) * canstep, 0 );                   
                }
                else structs->flow = 0.0; /* undefined flow if heads are equal, thus this stmt added */
                
                /* don't do much with zero or negative flows */           
            if (structs->flow == 0.0) {
                structs->conc_P = 0.0;
                structs->conc_N = 0.0;
                structs->conc_S = 0.0;
                goto OUT;
            } /* for output purposes only, zero the reported conc */

                /*  rule-based: calculate the nutrients - this is restricted to calculated tidal boundary condition flows  from outside of system */
                /* concentration in structure flows, g/L = kg/m^3; mass of stock in kg */

                if (structs->cell_i_fr == 2) { 
                    structs->conc_P = (structs->TPser == 1)  ?
                        Hist_data[structs->histID-1].arrayP[((int)(SimTime.TIME))] :
                        ((structs->TPser == 0) ? (structs->TP) : (0.0) );
                    structs->conc_N = (structs->TNser == 1)  ?
                        Hist_data[structs->histID-1].arrayN[((int)(SimTime.TIME))] :
                        ((structs->TNser == 0) ? (structs->TN) : (0.0) );
                    structs->conc_S = (structs->TSser == 1)  ?
                        Hist_data[structs->histID-1].arrayS[((int)(SimTime.TIME))] :
                        ((structs->TSser == 0) ? (structs->TS) : (0.0) );
                }
            P_fl = structs->flow * structs->conc_P; 
            N_fl = structs->flow * structs->conc_N;
            S_fl = structs->flow * structs->conc_S;
             
                /* this calculated flow is restricted to an outside-system cell, which does not need updating */
                /* these are nutrient stocks, not concentrations */  
                /* mass balance in fr cell and to canal */
            if (structs->cell_i_fr == 2  ) {
                VOL_IN_STR[Chan_list[can_to]->basin] += structs->flow; 
                VOL_IN_STR[0] += structs->flow; 
                P_IN_STR[Chan_list[can_to]->basin] += P_fl; 
                P_IN_STR[0] += P_fl; 
                S_IN_STR[Chan_list[can_to]->basin] += S_fl; 
                S_IN_STR[0] += S_fl; 
            }
            else 	{  
                sprintf( msgStr, "Day %6.1f: ERROR in flow to chan %d: cannot have calculated cell->canal flows except from outside of model domain. ",
                         SimTime.TIME, Chan_list[can_to]->number);
                WriteMsg( msgStr,True ); dynERRORnum++;
                goto OUT; 
            }
            

                /* change in nut mass in canal (kg) */
            Chan_list[can_to]->N +=  N_fl ;
            Chan_list[can_to]->P +=  P_fl ;
            Chan_list[can_to]->S +=  S_fl ;
            
            goto OUT;
            }
        }
        
 
/* CASE 5 */
/* if none of the above structure flow conditions */
        printf ("\nError in data for structure N %s: impossible condition! \n", structs->S_nam );
        exit (-1);
	
      OUT:
        if (TWf) { structs->TW_stage = -99; TWf = 0; }
        if (HWf) { structs->HW_stage = -99; HWf = 0; }

            /* sum of flows and solute masses thru structs over the can_Intvl canal summary interval */
        structs->SumFlow += structs->flow; 
        structs->Sum_P += structs->conc_P * structs->flow; 
/*        structs->Sum_N += structs->conc_N * structs->flow; */
        structs->Sum_S += structs->conc_S * structs->flow; 

        structs->multiOut = 0; /* reset the multi-outflow flag to zero after done */
        
            /* print sum of flows and the flow weighted mean conc thru structures */
        if ( printchan ) { 
            fprintf( WstructOutFile, "%7.0f\t", (structs->SumFlow)/(2446.848) ); /* convert to cfs (0.02832*60*60*24) */
            fprintf( WstructOutFile_P, "%7.4f\t", 
                     (structs->SumFlow > 0.0) ? (structs->Sum_P/structs->SumFlow*1000.0) : (0.0) ); /* convert g/L to mg/L */
/*            fprintf( WstructOutFile_N, "%7.4f\t", 
              (structs->SumFlow > 0.0) ? (structs->Sum_N/structs->SumFlow*1000.0) : (0.0) );*/ /* convert g/L to mg/L */
            fprintf( WstructOutFile_S, "%7.4f\t", 
                     (structs->SumFlow > 0.0) ? (structs->Sum_S/structs->SumFlow) : (0.0) ); /* g/L, no conversion */

            structs->SumFlow = 0.0; 
            structs->Sum_P = 0.0;
/*            structs->Sum_N = 0.0; */
            structs->Sum_S = 0.0;
        }
        
            /* increment to the next structure in the list */
        structs = structs->next_in_list;
    }

    if (printchan) {
        fflush( WstructOutFile );
        fflush( WstructOutFile_P );
/*         fflush( WstructOutFile_N ); */
        fflush( WstructOutFile_S );
    }
 
    return;
    
}

/****************************************************************************************/
/*! \brief Graph to read the time dependent schedules.

	\param graph A struct of Schedule
	\param x Time, julian day within recurring year
	\return value The target stage (m)
*/


float GetGraph (struct Schedule *graph, float x)
{
    float s;
    int ig = 0, Np;
	
    Np = graph->num_point;

    while(1) 
    {
	if (x <= graph->graph_points[ig].time) 
        { if ( ig > 0 ) ig--; 
        else return ( graph->graph_points[0].value );
        }
	else if (x > graph->graph_points[ig].time && x <= graph->graph_points[ig+1].time) 
        {
            s = 	( graph->graph_points[ig+1].value - graph->graph_points[ig].value )/
                ( graph->graph_points[ig+1].time - graph->graph_points[ig].time );
            return ( s * ( x - graph->graph_points[ig].time ) + graph->graph_points[ig].value ); 
        }
	else if (x > graph->graph_points[ig+1].time) 
        { if ( ig < Np ) ig++; 
        else return ( graph->graph_points[Np].value );
        }
    }
}


/********************************************************************************************/
/*! \brief Converter from meters in UTM to grid cell column location (zero origin)	

	\param UTM Geographic coordinate (m, ELM uses UTM zone17, NAD'27, but not important here)
	\return Grid cell column number
*/
float UTM2kmy ( double UTM )
{ return ( Abs(UTM - UTM_EOrigin)/celWid );  
}

/*! \brief Converter from meters in UTM to grid cell row location (zero origin)	

	\param UTM Geographic coordinate (m, ELM uses UTM zone17, NAD'27, but not important here)
	\return Grid cell row number
*/
float UTM2kmx ( double UTM )
{ return ( Abs(UTM - UTM_NOrigin)/celWid );	   	 
}

/****************************************************************************************/
/*! \brief Compare two words

	\param s A string
	\param t Another string
	\return True/false
*/
int Wrdcmp ( char *s, char *t )
{ 
  for ( ; *s == *t; s++, t++ )
  	 if ( isspace (*(s+1)) || *(s+1) == EOS ) return 1;
  	
  return 0;
  
}

/******/
