/*! \file globals.h
\brief  Header file for global data/functions.

This defines or declares variables & functions that are global to the model. \n

Note: documented with Doxygen, which expects specific syntax within special comments.  \n
	
The Everglades Landscape Model (ELM). \n
last updated: Feb 2005 \n
*/

/* General notes on revisions to this source file. 
       Nov/Dec 2004 v2.3.2: documentation upgrade 
       		- Re-organized, clarified scopes, Doxygen tags added 
       Apr 2005 v2.4.4 - final version of all code for internal review
	
*/

#ifndef GLOBALS_H
#define GLOBALS_H

#include <ctype.h> 
#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h> 

typedef int             BOOL;
typedef FILE*           STREAM; 
typedef char*           STRING; 
typedef unsigned char   UCHAR;
typedef unsigned char   byte;
typedef unsigned int    UINT;
typedef unsigned short  USHORT;
typedef long            SLONG;
typedef unsigned long   ULONG;

/*! Program attributes, with info on a the type of simulation run(s) within an execution of main program.

	This is used for standard runs, sensitivity analyses, and others (e.g., stochastic processes) in future */
typedef struct prog_attr {
	char S_ParmName[30]; 	/*!< Sensitivity: Name of the sensitivity parameter being varied for this run
						(must exactly match a model parameter in either the global or habitat-specific parameter dbases) */		
	int S_ParmRelVal; 	/*!< Sensitivity: indicator of current value of relative range in sensitivity parm values: nominal (0), low (1), or high (2)  */		
	float S_ParmVal; 	/*!< Sensitivity: current value of parameter (used only for output)  */		
} ProgAttr;

int SensiOn;		/*!< boolean flag to signifiy sensitivity analysis is on/off */

/*! Current time data of simulation */
struct simTime {
  float TIME;   			/*!< counter of the relative day(s) since initialization, starting at TIME=0 (not 1). 
  								TIME includes fractions of days if the vertical dt<1.0, but there is no need to have 
  								a **vertical** dt of less than 1 day */     
  double Jdate;				/*!< julian day (since epochs ago) of the current model time */
  int yr[2];				/*!< calendar year of current model time */
  int mo[2];				/*!< calendar month of current model time */
  int da[2];				/*!< calendar day of current model time */
  int hr[2];				/*!< calendar hour of current model time (unused in model calcs) */
  int mi[2];				/*!< calendar minute of current model time (unused in model calcs) */
  double se[2];				/*!< calendar second of current model time (unused in model calcs) */
  int IsSimulationEnd; 		/*!< boolean flag to signify ending iteration of simulation at current model time */
  int IsMonthEnd; 			/*!< boolean flag to signify ending day of a gregorian calendar month at current model time */
  int IsYearEnd; 			/*!< boolean flag to signify ending day of a gregorian calendar year at current model time */
  int IsWetSeasEnd; 		/*!< boolean flag to signify ending day of a gregorian calendar- based wet season at current model time */
  int IsDrySeasEnd; 		/*!< boolean flag to signify ending day of a gregorian calendar- based dry season at current model time */
  int IsBudgEnd; 			/*!< boolean flag to indicate if interval for budget output (Basin/Indicator-Region) has ended at current model time */
  int IsBIRavgEnd; 			/*!< boolean flag to indicate if interval for BIR stats output (Basin/Indicator-Region) has ended at current model time */
  int IsBudgFirst; 			/*!< boolean flag to indicate if current interval is the first budget interval (Basin/Indicator-Region budget) of the simulation */
  int IsDay0; 				/*!< boolean flag to signify current model time is the first day (day 0) of the simulation */
} SimTime; 

/*! Point time series spatial interpolation: unused ELMv2.3 (last used in v2.1) */
typedef struct pTSeries {
  int fX;
  int fY;
  int fNpt;
  float fTS;
  float *fValue;
} PTSeries;

/*! Point time series spatial interpolation: unused ELMv2.3 (last used in v2.1) */
typedef struct pTSeriesList {
  PTSeries* fList;
  int fNSlots;
  int fNSlotsUsed;
  int fNptTS;
  int fFormat;
  int fIndex;
} PTSeriesList;

/*! Point time series spatial interpolation: unused ELMv2.3 (last used in v2.1) */
typedef struct rPoint {
  float fr;
  int fX;
  int fY;
  float falpha;
  float fValue;
} RPoint;

/*! Point time series spatial interpolation: unused ELMv2.3 (last used in v2.1) */
typedef struct rPointList {
  RPoint* fList;
  int fNPt;
  int fNPtUsed;
} RPointList;


/*! Model output attributes */
typedef struct {
  float **pfvar;		/*!< Model variable data array.  Note that this variable is not necessarily of type float; 
  					the model arrays are floats, doubles, and unsigned chars, and cast as such within the tgen array as needed; 
  					TODO: (jan 18, 2005) should modify this struct to better reflect this */
  const char  *pname;	/*!< Name of model variable for output */
  char  ctype;			/*!< Output numeric format type */
} outVar_struct;

/*! Basin/Indicator-Region attributes */
typedef struct basndef {
  char basnTxt[12];  	/*!< name of Basin/Indicator-Region */
  int numIR;   			/*!< number of indicator regions in this hydrologic basin */     
  int IR[30];    		/*!< max of 30 indicator regions within a hydrologic basin */   
  int FLok[30];   		/*!< hydrologic basins with which the current basin may have overland flow */  
  int numFLok;   		/*!< number of FLok basins attached to this one */   
  int parent;     		/*!< ID if this is a main (parent) hydro basin */  
  int family;      		/*!< the family name (parent basin & IRegion children) of hydrologic basin */ 
  double conv_m3TOcm;  		/*!< for a basin, convert a m^3 volume to a cm height across the basin */
  double conv_kgTOmgm2;		/*!< for a basin, convert a kg mass to a mg/m2 mass-per-area across the basin */
} basnDef; 


/*! Point time series output, point attributes */
typedef struct point3D {
  int x;						/*!< the x (row!) location */
  int y;						/*!< the y (col!) location */
  int z;						/*!< the index (?) number of point locs */
  unsigned char type;			/*!< output type (p or w, point or window?) */
  unsigned char format;			/*!< numeric format */
} Point3D;

/*! The grid location of a point */
typedef struct point2D {
  int x;						/*!< the x (row!) location */
  int y;						/*!< the y (col!) location */
} Point2D;

/*! Model output configuration, scaling data */
typedef struct scale1 {
  float s;						/*!< variable-scaling multiplier */
  float o;						/*!< variable-scaling offset */
} Scale1;

/*! Model output configuration, scaling data (unused currently) */
typedef struct scale2 {
  float Vmin;						/*!< minimum */
  float Vmax;						/*!< maximum */
} Scale2;

/*! Model output configuration data */
typedef struct viewParm {
  Scale1 scale;					/*!< struct with variable-scaling multiplier and offset */
  Scale2 gScale;				/*!< struct with min and max scalars */
  Scale2 bounds;				/*!< struct with min and max scalars */
  Point3D *points;				/*!< struct with attributes of the point location */
  unsigned char nPoints;		/*!< number of point locations */
  unsigned char maxPoints;		/*!< max NPoints currently */
  int step;						/*!< output time step interval */
  unsigned char mapType;		/*!< output map storage type (integer= animation (model variable value=0-254) directory #, M= map integer, restricted only by byte size, H= HDF (not re-implemented, v2.3) */
  unsigned int flags;			/*!< effectively unused */
  char fileName[24];			/*!< variable name, used as output file name */
} ViewParm;

/*! Point time series output configuration data */
typedef struct seriesParm {
  float* data;					/*!< Array data of variable */
  Point2D Loc;					/*!< struct with the grid location of a point */
  char name[24];				/*!< name of the variable */
  int Length;					/*!< Time length in number of variable's outsteps taken */
  int outstep;					/*!< output time step interval */
  int laststep; 				/*!< last (i.e., most recent) temporal outstep for this file */
} SeriesParm;

/*! Unused, Parallel implementation */
typedef struct nodenv {
  int procnum;            /*!< Processor number of calling program */
  int nprocs;             /*!< Number of nodes in this processor group */
  int groupid;            /*!< Group identifier for this processor group */
  int taskid;             /*!< Identifies particular task on a node */
} nodenv;


#define VOIDP void*
		
#define HDF 0 /*!< boolean flag to indicate use of (compile/link to) NCSA's Hierarchical Data Format.  
		NOTE: (Nov 2004): do not #define HDF to true until hdf header sources are updated */
#define UNIX	1    /*!< (unused) UNIX boolean flag to indicate a unix operating system.  */
#define H_OPSYS UNIX /*!< define the operating system as UNIX   
		NOTE: ELM is always in unix (used to run on Macintosh transputers) - code using this can be removed */

#define kMAX 0		/*!< only used in case stmt in Combine (Serial.c) */
#define kMIN 1		/*!< only used in case stmt in Combine (Serial.c) */
#define kMAXMIN 2		/*!< only used in case stmt in Combine (Serial.c) */
#define kSUM 3		/*!< only used in case stmt in Combine (Serial.c) */
#define kAVE 4		/*!< only used in case stmt in Combine (Serial.c) */
#define kSUMCUM 5		/*!< only used in case stmt in Combine (Serial.c) */
#define kAVECUM 6		/*!< only used in case stmt in Combine (Serial.c) */

		
#define MAX_PTSERIES  800  /*!< maximum number of grid cells for Point Time Series output */
#define MAX_TS_FILES  50  /*!< max # files (=variables) for writing Point Time Series output */
#define MAX_MSG_SIZE 1200	/*!< maximum size of msg (message) string for messages to console or debug file */

#define ISARRAY         0x8000          /* TRUE/ is Array       */
#define PMASK2          0x0200          /* TRUE/        */
#define PMASK1          0x0100          /* TRUE/        */
		
#define kCArgDepth      10	/*!< Sizes array of arguments to parse the output configuration file */
#define kCArgWidth      50	/*!< Sizes array of arguments to parse the output configuration file */


#define EOL '\n'
#define EOS '\0'
#define TAB '\t'
#define POUND  ((UCHAR)0x23)
#define True 1
#define False 0
#define CASE 0
#define NOCASE 1
#define BEG 0
#define END 1

#define DT dt	/*!< allow upper case definition of dt (primary model delta-time time step (d) ) */
#define T(x,y) ((int)((x)*(s1+2)+(y))) 	/*!< define location of 2D row(x), col(y) data stored in 1D array */
#define sec_per_day 86400.0 	/*!< seconds per day, used in Manning's eqn */
#define PI 3.1415927
#define wetEndMon 9   /*!< gregorian-calendar month of end of wet season */
#define wetEndDay 30  /*!< gregorian-calendar day of end of wet season */
#define dryEndMon 4   /*!< gregorian-calendar month of end of dry season */
#define dryEndDay 30  /*!< gregorian-calendar day of end of dry season */

#define Abs(x) (((x)>=0) ? (x) : -(x))
#define Cos(x) cos((double)x)
#define Sin(x) sin((double)x)
#define Tan(x) tan((double)x)
#define Arctan(x) atan((double)x)
#define Exp(x) exp((double)x)
#define Max(x,y) (((x)>(y))?(x):(y))
#define Min(x,y) (((y)>(x))?(x):(y))
#define ramp(x) (((x)>0)?(x):0)
#define sgn(x) (((x)>=0) ? 1.0 : -1.0)

#define conv_mgTOg 1.0e-3  /*!< conversion from mg to g */
#define conv_gTOkg 1.0e-3  /*!< conversion from g to kg */
#define conv_kgTO_Mg 1.0e-3  /*!< conversion from kilograms to Megagrams */

#define conv_mgTOug 1.0e3  /*!< conversion from milligrams to micrograms */
#define conv_gTOmg 1.0e3  /*!< conversion from g to mg */
#define conv_kgTOg 1.0e3  /*!< conversion from kg to g */
#define conv_kgTOmg 1.0e6  /*!< conversion from kg to mg */

#define conv_inTOtenths_mm 254.0  /*!< conversion from inches to tenths of a mm */
#define conv_ftTOm 0.3048  /*!< conversion from feet to meters */

#define conv_mTOcm 100.0  /*!< conversion from meters to cm */
#define conv_cmTOmm 10.0  /*!< conversion from cm to mm */
#define conv_mTOkm 0.001  /*!< conversion from m to km */

#define MAX_NHAB 255 	/*!< maximum number of habitat (HAB) types */
#define MAX_SECTOR  30	/*!< maximum number of cell_dyn sectors (ecological modules) */

char  msgStr[MAX_MSG_SIZE];	/*!< a string for messages to file/console */

	/*! \anchor timeSteps
		\par Variables associated with the model time and time steps
 	   \n\em dt primary model delta-time time step (d)
 	   \n\em sfstep time step (d) for horizontal surface water flows
 	   \n\em gwstep time step (d) for horizontal ground water flows
 	   \n\em canstep time step (d) for horizontal canal water flows 
 	    */
float dt, sfstep, gwstep, canstep;
int hyd_iter;        /*!< max number of horiz iterations per model primary (vertical) time step (dt) */

unsigned char *ON_MAP; 	/*!< the map that defines the model domain; is modified by WatMgmt.c function for levee interaction */

	/*! \anchor cellData
		\par Variables associated with the grid cell size
 	   \n\em CELL_SIZE square grid cell size, m^2
 	   \n\em celWid cell width, m
 	   \n\em sq_celWid square root of cell width, m */
float CELL_SIZE, celWid, sq_celWid;       
int s0; 	/*!< number of rows in array of model domain */
int s1; 	/*!< number of columns in array of model domain */

		/*! \em debug Value (0 - 5) indicates the debug level for warning/error checking & writing to console and debug files. \n

		At debug >=0 (i.e., always), print critical ERROR violations to DriverX.out (for X'th simulation run) debug file. 
		An increasing amount of information is printed with increasing debug level: 
		\li debug =0 Echo short console info on iteration# etc, print critical error/warning info. USE WITH CAUTION. 
		\li debug =1 Report mis-configured basin flows. Currently same level as debug=2.  
		\li debug =2 DEFAULT for general use, more warnings etc.  
		\li debug =3 Echo long console output, prints additional (non-critical) errors/warnings to DriverX.out (for X'th simulation run) file 
		\li debug =4 Prints details of cell vertical and/or horizontal flux data, and details of indiv canal fluxes, to DriverX.out (for X'th simulation run) 
		\li debug =5 Prints grid_map information, and prints to another canal debugging file for special purposes  
		*/
int debug; 
int dynERRORnum; /*!< A counter of the cumulative number of ERRORS in dynamic calculations - used to abort (after allowing enough to understand problem) */

void WriteMsg(const char* msg, int wh);
void usrErr(const char* dString);
void usrErr0(const char* dString);


#endif /* GLOBALS_H */
