/*! \file grid_io.h
\brief Header file for grid_io utility functions.

This defines or declares variables & functions that are global to grid_io.c. \n

Note: documented with Doxygen, which expects specific syntax within special comments.  \n
	
The Everglades Landscape Model (ELM). \n
last updated: Jan 2005 \n
*/

/* General notes on revisions to this source file. 
       Nov/Dec 2004 v2.3.2: documentation upgrade 
       		- Doxygen tags added 
	
*/

/*
SCCS ID: "@(#)grid_io.h	1.10 05/22/00 SFWMD Research and Planning Departments"

Copyright, 1995, South Florida Water Management District

DISCLAIMER:

ANY INFORMATION, INCLUDING BUT NOT LIMITED TO SOFTWARE AND DATA,
RECEIVED FROM THE SOUTH FLORIDA WATER MANAGEMENT DISTRICT ("DISTRICT")
IN FULFILLMENT OF A PUBLIC RECORDS REQUEST IS PROVIDED "AS IS" WITHOUT
WARRANTY OF ANY KIND, AND THE DISTRICT EXPRESSLY DISCLAIMS ALL EXPRESS
AND IMPLIED WARRANTIES, INCLUDING BUT NOT LIMITED TO THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
THE DISTRICT DOES NOT WARRANT, GUARANTEE, OR MAKE ANY REPRESENTATIONS
REGARDING THE USE, OR THE RESULTS OF THE USE, OF THE INFORMATION
PROVIDED TO YOU BY THE DISTRICT IN TERMS OF CORRECTNESS, ACCURACY,
RELIABILITY, TIMELINESS OR OTHERWISE.  THE ENTIRE RISK AS TO THE
RESULTS AND PERFORMANCE OF ANY INFORMATION OBTAINED FROM THE DISTRICT
IS ENTIRELY ASSUMED BY THE RECIPIENT.

*/

/* -------------------------------------------------------------
   header grid_io.h
   This header contains definitions for SFWMM data grid and 
   io routines
   ------------------------------------------------------------- */
#ifndef GRID_IO_H
#define GRID_IO_H
#include "stdlib.h"
#include "stdio.h"

/* grid definition record - contains all information necessary to
   define the areal grid */

#ifndef MAX_GRID_ROWS
#define MAX_GRID_ROWS 66		/*!< maximum number of data array rows */
#endif

#define GRID_TITLE_LENGTH 80		/*!< maximum string length of grid_io header title */
#define GRID_TAG_LENGTH 80			/*!< the length of the tag identifying a grid-data snapshot */

/*! Header info of grid_io data */
typedef struct {
	char title[GRID_TITLE_LENGTH];		/*!< title of grid_io data in header */
	int number_of_rows;					/*!< number of rows of grid_io data, found in header */
	int number_of_nodes;				/*!< number of nodes (grid cells) of grid_io data, found in header */
	struct { float x, y; } size;		/*!< size (row by col) of grid_io data, found in header */ /* TODO: verify, ELMchange, original had "struct { float x, y; } size", no semicolon after size */
} GR_HEADER;

/*! Grid size/config info of grid_io data */
typedef struct {
  int *xstart;							/*!< starting row of grid_io data */
  int *xend;							/*!< ending row of grid_io data */
  int *cum_node_count;					/*!< cumulative count of number of nodes in grid_io data */
} GR_CONFIG;

/*! All info needed on grid_io data */
typedef struct {
	GR_HEADER header;						/*!< struct of GR_HEADER */
	GR_CONFIG config;						/*!< struct of GR_CONFIG */
} GRID;

/* fortran interface for files */

struct GR_FORTRAN_FILE_STRUC {
       FILE *fptr;
       int unit_number;
       struct GR_FORTRAN_FILE_STRUC *next;
};
typedef struct GR_FORTRAN_FILE_STRUC GR_FORTRAN_FILE;

/* available routines */
#ifdef __cplusplus
extern "C" {
#endif
int grid_write_header(FILE *file, GRID *grid);
#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
extern "C" {
#endif
int grid_read_header(FILE *file, GRID *grid);
#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
extern "C" {
#endif
int grid_write(FILE *file, GRID *grid, char *tag, float *values);
#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
extern "C" {
#endif
int grid_read(FILE *file, GRID *grid, char *tag, float *values);
#ifdef __cplusplus
}
#endif


/*int grid_find_tag(FILE *file, GRID *grid);*/
#ifdef __cplusplus
extern "C" {
#endif
int grid_skip(FILE *file, GRID *grid, int num);
#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
extern "C" {
#endif
int grid_tag_search(FILE *file, GRID *grid, char *string);
#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
extern "C" {
#endif
int grid_top(FILE *file, GRID *grid);
#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
extern "C" {
#endif
int grid_bottom(FILE *file, GRID *grid);
#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
extern "C" {
#endif
int grid_count_snapshots(FILE *file, GRID *grid);
#ifdef __cplusplus
}
#endif

#ifdef __cplusplus
extern "C" {
#endif
int grid_free(GRID *grid);
#ifdef __cplusplus
}
#endif
/* end of header */

#endif /* GRID_IO_H */
