/*! \file gridmap.h
\brief Header file for re-mapping grid data among grid scales. 

This defines or declares variables & functions that are global to gridmap.c.  \n

Note: documented with Doxygen, which expects specific syntax within special comments.  \n
	
The Everglades Landscape Model (ELM). \n
last updated: Jan 2005 \n
*/

#ifndef GRIDMAP_H
#define GRIDMAP_H

#include "grid_io.h"
#include "globals.h"

	
int OGrow_tot; /*!< total number of rows in the non-ELM grid */
int OGcol_tot; /*!< total number of columns in the non-ELM grid */
int OGidCnt;   /*!< count of the (unique) grid IDs in the non-ELM grid; can be greater or less than the grid_io data file size OGrow_tot * OGcol_tot, depending on the degree to which the ELM and the non-ELM grids overlap in domain extent */
	
int gridio_batch_len;   /*!< the (number of records) length of the batch of data records (maps) read at one time.  Generally will be less than the simulation Period of Record length to keep memory needs down. */
	

/*! This data structure is specific to the output of the (C++) "grmap" pre-processor program.
	That pre-processor program must be run (once) for each new combination of
	ELM-grid-data (grain/extent) and data utilizing another grid.  The resulting
	ELM-input data file is expected to reside in the standard ELM-input data directory.

   Since gridmap can map any data-grid (for now, coarser in resolution only) to the 
   ELM grid, we call the name of struct ELM+AnyOtherGrid+Map (eOGMap) to
   be generic enough. The data-input application that uses this struct can be
   called to reflect the applied grid; i.e struct eOGMap *elm_wmm_map
   This is global and can be used by applications. Only one copy is enough. */
struct eOGMap {                           
  int OGID;         /*!< each Other-Grid ID (OGID) holds one or more elmID in its array */
  int* elmID;       /*!< at most one Other-Grid (OG) cell maps into how many elm cells,
                         elmID=-1 means there is no elm cell in this OGID. This is used in a loop */
  int eidCnt;       /*!< exact elmID array */
  int index;        /*!< the index of elm_OG_map[OGrow_tot*OGcol_tot] where elm cell
                         falls in Other-Grid (OG) cell.  Otherwise the index number gets
                         different values according to different cases:
                    \li NOT_ELM_NOT_GRIO: 
                        outside of ELM domain, and Other-Grid (OG) cell does not have data (does not fall in grid_io (GRIO) data domain).
                        index= -1, no data for elm, no grid_io increment, elmID= -1
                    \li NOT_ELM_IN_GRIO:
                        outside of ELM domain, and Other-Grid (OG) cell is in grid_io (GRIO) data domain, but not needed.
                        index= -2, no data for elm, grid_io increment values, elmID= -1 
                    \li IN_ELM_NOT_GRIO: 
                        ELM cell within Other-Grid (OG) cell, and OG cell does not have data (OG cell does not fall in grid_io (GRIO) data domain).
                        index= -3, data estimate needed for elm, no grid_io increment, elmID= x
                    \li case4: 
                        ELM cell within Other-Grid (OG) cell, and OG cell does have data (falls in grid_io (GRIO) data domain).  
                        index=elm_OG_map[i] where i = {0,1,...,OGrow_tot*OGcol_tot}, data avai. for elm, grid_io increment, elmID= x    */

  int OGROW;          /*!< row ID of Other-Grid (OG) cell */
  int OGCOL;          /*!< column ID of Other-Grid (OG) cell */
} *elm_OG_map;        /*!< array of struct of size OGrow_tot * OGcol_tot (Other-Grid) */
struct eOGMap *elm_wmm_map ;  /* all grid_io data inputs are using the same eOGMap.
								If in future they need their own copy, can make it like applicationStruct! */

/*! An application-specific data structure for use by variety of data-input apps. */
typedef struct {

  FILE* binfile_ptr;                /*!<  grid_io data file pointer, stays open */
  float *dataELM;                   /*!<  array of data at ELM grid dimension */
  float **dataAnyGrid;              /*!<  array of non-ELM-grid-dim grid_io data (gridio_batch_len * OGrow_tot * OGcol_tot) */
  char tag[GRID_TAG_LENGTH];        /*!<  holds date field tag of the grid_io data set */
  float *values;                    /*!<  holds array of values of grid_io data */
  GRID grid;                        /*!<  grid_io data structure that holds the model grid information. this is declared in grid_io.h */
  int dayCntr;                    	/*!<  control variable, must be in application */
  int recCntr;                    	/*!<  control variable, must be in application */
  int day;         /*!< This variable keeps track of the number of records returned to
                      UnitMod, calling program. It is always one batch behind the recRead
                      and it gets added until program ends. */

  int step;        /*!< This variable keeps track of the records in batch of data read in 
                      sofar. When batch of data all is returned to UnitMod, this variable 
                      gets reset per batch */

  int recRead;     /*!< This variable keeps track of the total number of records read so far from the
                      binary grid_io file. It increments until it equals PORnumday (# days in sim POR) */

  int skip;        /*!< The user can ask for any starting date for the simulation.
                      This is the number of records skipped to get to starting date in the grid_io
                      binary file.  The skip gets added to recRead-1 every time a new batch of binary  
                      data is being read. This tells grid_io which record from the beginning of
                      file go to for the next batch. Gets reset per program execution. */
} applicationStruct ;


void formatDate(char*);
int mapGrids(char*);
int initDataStruct(char*,applicationStruct*);
int processData(char*,applicationStruct*);
void returnData(float*,applicationStruct*);
void mapELM2Grid_io(applicationStruct*);
void printGridMap();
/*void printBatchData(float **value,int days, int size); */ /* TODO: remove this function when sure is no longer needed */

/* externs */
/* from generic_driver.h */
extern char initDateRead[15]; /* date of initialization of model */
extern int PORnumday;

#endif  /* GRIDMAP_H */
