/*! \file unitmod.h
\brief Header file for the ecological Unit Model. 

This defines or declares variables & functions that are global to Unit_Mod.c.  \n

Note: documented with Doxygen, which expects specific syntax within special comments.  \n
	
The Everglades Landscape Model (ELM). \n
last updated: May 2006 \n
*/

/* General notes on revisions to this source file. 
       Jan 2005 v2.3.3: documentation upgrade 
       		- Re-organized, clarified scopes, Doxygen tags added 
       		- Created quasi-automated unitmod_*.h header file generation for vars and parms 
        Apr 2005 v2.4.4: changed init maps: icMult removed, icMacBio added
        May 2006 v2.5: added some debug-level output for boundary condition stage
	
*/
#ifndef UNITMOD_H
#define UNITMOD_H

#include "globals.h"

/* non-spatial variables */

float DAYJUL; /*!< A "julian" day counter within a 365 day "year" */

/*! \anchor solarRadVariables
	\par Non-spatial solar radiation intermediate variables.  
	These (unmodified) complex calculations
	were developed, tested, and published for global applications by Nikolov & Zeller (1992).
	\n\em LATRAD Regional latitude calculated in radians
	\n\em SOLALPHA Intermediate calculation
	\n\em SOLALTCORR Intermediate calculation (altitude correction)
	\n\em SOLBETA Intermediate calculation
	\n\em SOLCOSDEC Intermediate calculation
	\n\em SOLDEC Intermediate calculation
	\n\em SOLDEC1 Intermediate calculation
	\n\em SOLELEV_SINE Intermediate calculation
	\n\em SOLRADATMOS Solar radiation in upper atmosphere (cal/cm2/d)
	\n\em SOLRISSET_HA Intermediate calculation
	\n\em SOLRISSET_HA1 Intermediate calculation
	*/
float LATRAD, SOLALPHA, SOLALTCORR, SOLBETA, SOLCOSDEC, SOLDEC, SOLDEC1, SOLELEV_SINE, SOLRADATMOS,
      SOLRISSET_HA, SOLRISSET_HA1; 
/* float AMPL, DAYLENGTH;*/ /* unused AMPL in unused DAYLENGTH */

float dispParm_scaled;/* v2.1b, v2.4: dispersion adjustment parameter,  scaled for model grid (=0 for no adjustment)  */

/* data for graph functions that interpolate the data */
    /*! \em g7 data for graph7 - empirical data of a (0-1) control function, the proportion (dependent var, second of each pair) of maximum vertical water infiltration rate through soil as a function of soil moisture proportion (0-1) (independent var, first of each pair) */
float g7[11][2] = { {0.0000000,0.0000000},{0.1000000,0.0200000},{0.2000000,0.0400000},{0.3000000,0.0700000},
    {0.4000000,0.1250000},{0.5000000,0.2150000},{0.6000000,0.3450000},{0.7000000,0.5750000},
    {0.8000000,0.8550000},{0.9000000,0.9850000},{1.0000000,1.0000000} };
    /*! \em g8 data for graph8 - empirical data of a (0-1) control function, the proportion (dependent var, second of each pair) of water available to plants as a function of proportion (0-1) of water available upper soil profile (independent var, first of each pair) (generally, simply 1:1 relationship)  */
float g8[11][2] = { {0.0000000,0.0050000},{0.1000000,0.0100000},{0.2000000,0.0400000},{0.3000000,0.1000000},
    {0.4000000,0.2800000},{0.5000000,0.7150000},{0.6000000,0.8650000},{0.7000000,0.9350000},
    {0.8000000,0.9750000},{0.9000000,0.9950000},{1.0000000,1.0000000} };


/* functions */
int call_cell_dyn(int sector,int step);

void cell_dyn1(int step);
void cell_dyn2(int step);
void cell_dyn4(int step);
void horizFlow(int step);
void cell_dyn7(int step);
void cell_dyn8(int step);
void cell_dyn9(int step);
void cell_dyn10(int step);
void cell_dyn12(int step);
void cell_dyn13(int step);

float tempCF(int form, float c1, float topt, float tmin, float tmax, float tempC);

void init_static_data(void);
void init_dynam_data(void);
void init_eqns(void);
void init_canals(int irun);
void init_succession(void);
void reinitBIR(void);
void reinitCanals(void);

void gen_output(int step, ViewParm *view);

void ReadGlobalParms(char* s_parm_name, int s_parm_relval);
void ReadHabParms(char* s_parm_name, int s_parm_relval);

void get_map_dims(void);
void alloc_memory();
float graph7(unsigned char y, float x);
float graph8(unsigned char y, float x);


/* **** externs **** */

/* from success.h */
extern void HabSwitch_Init( void );
extern unsigned char HabSwitch(int ix, int iy, float *Water, float *Nutrient,
                               int *Fire, unsigned char *HAB);

/* from watmgmt.h */
extern void Run_Canal_Network(float *SWH, float *ElevMap, float *MC,
		              float *GWV, float *poros, float *GWcond,
                              double *NA, double *PA, double *SA,
		              double *GNA, double *GPA, double *GSA,
		              float *Unsat, float *sp_yield);
extern void Canal_Network_Init(float baseDatum, float *elev ); 
extern void CanalReInit();


/* from driver_utilities.h */
extern int CalMonOut;
extern void getInt(FILE* inFile, const char* lString, int* iValPtr);
extern void read_map_dims(const char* filename);
extern void write_output(int index, ViewParm* vp, void* Map,
                         const char* filename, char Mtype, int step);
extern void getChar(FILE* inFile, const char* lString, char* cValPtr);
extern void init_pvar(VOIDP Map, UCHAR* mask, unsigned char Mtype,float iv);
extern int read_map_file(const char* filename, VOIDP Map,
	                 unsigned char Mtype, float scale_value, float offset_value);
extern int scan_forward (FILE* infile, const char* tstring);
//extern float* get_DBase_parmOLD(const char *filename, int i0, int i1, int index);
//extern float* get_DBase_parm(const char *filename, float *dArray);
extern VOIDP nalloc(unsigned mem_size, const char var_name[]);
extern float FMOD( float x, float y);

/* from budgstats.h */
extern basnDef **basn_list;
extern basnDef *basins;

extern void stats(int step);
extern void alloc_mem_stats(void);
extern void BIRinit(void);
extern void BIRoutfiles(void);
extern void BIRstats_reset(void);
extern void BIRbudg_reset(void);
extern void Cell_reset_avg(void);
extern void Cell_reset_hydper(void);

/* from fluxes.h */
extern void Flux_SWater(int it, float *SURFACE_WAT,float *SED_ELEV,float *HYD_MANNINGS_N,
	           double *STUF1, double *STUF2, double *STUF3);
extern void Flux_GWater(int it, float *SatWat, float *Unsat, float *SfWat,
		        float *rate, float *poros,
		        float *sp_yield, float *elev,
		        double *gwSTUF1, double *gwSTUF2, double *gwSTUF3,
		        double *swSTUF1, double *swSTUF2, double *swSTUF3);

/* from generic_driver.h */
extern int WatMgmtOn, ESPmodeON, HabSwitchOn;
extern int avgPrint; /* v2.2.1note true/false flag to indicate whether it is the time (day) to finalize 
					and print averages for the current interval  */

/* from driver_utilities.h */
extern char *OutputPath, *ProjName;

/* from serial.h */
extern float* get_hab_parm(char* s_parm_name, int s_parm_relval, char* parmName);
extern float get_global_parm(char* s_parm_name, int s_parm_relval, char* parmName );

/* from stage.h, rain.h, evap.h */
extern int stage_data_wmm(float*);
extern int rain_data_wmm(float*);
extern int evap_data_wmm(float*);


#endif /* UNITMOD_H */
